#!/usr/bin/perl -w

use strict;
use Getopt::Long;

&Main();

exit 0;

sub Main()
{
    my $inputFile;
    my $theoremPrefix = "th_";
    my $outputFile;
    my $help;

    GetOptions('inputFile|i=s'     => \$inputFile,
               'theoremPrefix|t=s' => \$theoremPrefix,
               'outputFile|o=s'    => \$outputFile,
               'help|h|?'          => \$help)
        || &Usage();

    &Usage() if ($help || !defined $inputFile || !defined $outputFile);

    &ExpandAxioms($inputFile, $theoremPrefix, $outputFile);
}


# Expands data from the input file to produce the output file.
sub ExpandAxioms
{
    my ($inputFile, $theoremPrefix, $outputFile) = @_;

    open(IN, "<$inputFile") || die "Couldn't open $inputFile for reading: $!\n";
    open(OUT, ">$outputFile") || die "Couldn't open $outputFile for writing: $!\n";

    my %leftVariablesSet;
    my $allLeftVariables = "";
    my $combinedRightSide = "";

    while (<IN>)
    {
        chomp;

        my $currLineNumber = $.;
        my $left;
        my $right;

        # Skip empty lines.
        if (/^\s*$/)
        {
            next;
        }

        # Skip comments.
        if (/^%/)
        {
            next;
        }

	s/^assume /!/;
	s/^let/?/;
	s/^have //;
	
        # Lines that have both left and right side, e.g. ?[X] : (A).
        if (/^[!?]\s*\[([^\]]+)\]\s*:\s*\((.+)\)\s*$/)
        {
            $left = $1;
            $right = $2;
        }
        # Lines that have only the right side, e.g. (A).
        elsif (/^\s*\((.+)\)\s*$/)
        {
            $right = $1;
        }
        else
        {
            die "Couldn't parse line number $currLineNumber: $_\n";
        }

        # Assume that the first line always starts with ! (for each), so only print output for other lines.
        if ($currLineNumber > 1)
        {
	    #regular theorem number
            my $theoremNumber = $currLineNumber - 1;
	    #theorem number with 0 in front of one digit numbers
	    my $paddedTheoremNumber = sprintf("%02u", $theoremNumber);
            my $implication;

            if (defined $left)
            {
                $implication = "($_)";
            }
            else
            {
                $implication = $_;
            }

            print OUT "fof($theoremPrefix$paddedTheoremNumber, axiom, (! [$allLeftVariables] : (($combinedRightSide) => $implication))).\n\n"
        }

        # Get all variables from the left side.
        if (defined $left)
        {
            if ($allLeftVariables ne "")
            {
                $allLeftVariables .= ",$left";
            }
            else
            {
                $allLeftVariables = $left;
            }
        }

        # Append the current right side to previous right sides.
        if ($combinedRightSide ne "")
        {
            $combinedRightSide .= " & $right";
        }
        else
        {
            $combinedRightSide = $right;
        }
    }

    close(OUT);
    close(IN);
}

sub Usage()
{
    print "Unknown option: @_\n" if (@_);

    print "Usage: perl ExpandAxioms.pl -i inputFile [-t theoremPrefix] -o outputFile\n\n";

    exit 0;
}
