(*    Title:              CNF.thy
      ID:                 $Id: CNF.thy$
      Author:             Filip Maric
*)

header {* Theory describing formulae in Conjunctive Normal Form (CNF) *}
theory CNF
imports Main MyList
begin



(********************************************************************)
subsection{* Syntax *}
(********************************************************************)

types    Variable  = nat

datatype Literal   = Pos Variable | Neg Variable

types    Clause    = "Literal list"

types    Formula   = "Clause list"


text{* The variable of a literal *}

consts var      :: "Literal => Variable"
primrec 
"var (Pos v) = v"
"var (Neg v) = v"

text{* The opposite of a given literal *}
consts opposite :: "Literal => Literal"
primrec
"opposite (Pos v) = (Neg v)"
"opposite (Neg v) = (Pos v)"

text{* Clause obtained by negating all literals of a clause *}
consts
oppositeClause :: "Clause => Clause"
primrec
"(oppositeClause []) = []"
"(oppositeClause (h#t)) = ((opposite h) # (oppositeClause t))"

text{* Set of variables of a given clause, formula or valuation *}
consts variableSet :: "(Literal list) => (Variable set)"
primrec
"variableSet [] = {}"
"variableSet (literal # list) = {(var literal)} Un (variableSet list)"

consts vars           :: "'a => Variable set"
defs (overloaded)
vars_def [simp]: "vars (clause::Clause) == variableSet clause"
primrec
"vars [] = {}"
"vars ((c::Clause) # (f::Formula)) = (variableSet c) Un (vars f)"

text{* Check if the literal is member of a clause, 
                    clause is a member of a formula or the 
                    literal is a member of a formula *}
consts member  :: "'a => 'b => bool" (infixl "el" 55)
defs (overloaded)
literalElClause_def [simp]: "((literal::Literal) el (clause::Clause)) == literal mem clause"
defs (overloaded)
clauseElFormula_def [simp]: "((clause::Clause) el (formula::Formula)) == clause mem formula"
primrec
"(literal::Literal) el ([]::Formula) = False"
"((literal::Literal) el ((clause # formula)::Formula)) = ((literal el clause) | (literal el formula))"



(*--------------------------------------------------------------------------------*)
lemma oppositeIdempotency [simp]:
  fixes literal::Literal
  shows "opposite (opposite literal) = literal"
by (induct literal) auto

lemma oppositeSymmetry [simp]:
  fixes literal1::Literal and literal2::Literal
  shows "(opposite literal1 = literal2) = (opposite literal2 = literal1)"
by auto

lemma oppositeUniqueness [simp]:
  fixes literal1::Literal and literal2::Literal
  shows "(opposite literal1 = opposite literal2) = (literal1 = literal2)"
proof
  assume "opposite literal1 = opposite literal2"
  then have "opposite (opposite literal1) = opposite (opposite literal2)" 
    by simp
  then show "literal1 = literal2" 
    by simp 
qed simp

lemma oppositeIsDifferentFromLiteral [simp]:
  fixes literal::Literal
  shows "opposite literal ~= literal"
by (cases literal) auto

lemma literalIsDifferentFromOpposite [simp]:
  fixes literal::Literal
  shows "literal ~= opposite literal"
by (cases literal) auto

lemma oppositeLiteralsHaveSameVariable [simp]:
  fixes literal::Literal
  shows "var (opposite literal) = var literal"
by (induct literal) auto


lemma literalsWithSameVariableAreEqualOrOpposite:
"(var literal1 = var literal2) = (literal1 = literal2 | opposite literal1 = literal2)"
proof
assume "var literal1 = var literal2"
show "literal1 = literal2 | opposite literal1 = literal2"
proof (cases literal1)
  case "Pos"
  show ?thesis proof (cases literal2)
    case "Pos"
    from prems show ?thesis by auto 
  next
    case "Neg"
    from prems show ?thesis by auto
  qed
next
  case "Neg"
  show ?thesis proof (cases literal2)
    case "Pos"
    from prems show ?thesis by auto
  next
    case "Neg"
    from prems show ?thesis by auto
  qed
qed
next
  assume "literal1 = literal2 | opposite literal1 = literal2"
  then show "var literal1 = var literal2" by auto
qed

(*--------------------------------------------------------------------------------*)
lemma literalElFormulaCharacterization:
  fixes literal :: Literal and formula :: Formula
  shows "(literal el formula) = (? clause. clause mem formula & literal mem clause)"
by (induct formula) auto

lemma literalListContainsItsLiteralsVariable: 
  fixes literal :: Literal and literalList :: "Literal list"
  assumes "literal el literalList"
  shows "var literal : vars literalList"
using assms
by (induct literalList) (auto iff: mem_iff)

lemma formulaContainsItsClausesVariables:
  fixes clause :: Clause and formula :: Formula
  assumes "clause el formula"
  shows "vars clause <= vars formula"
using assms
by (induct formula) (auto iff:mem_iff)

lemma variableDefinedImpliesLiteralDefined:
  fixes literal :: Literal and literalList :: "Literal list"
  shows "var literal : vars literalList = (literal el literalList | (opposite literal) el literalList)"
    (is "(?lhs literalList) = (?rhs literalList)")
proof
  assume "?rhs literalList"
  then show "?lhs literalList" 
  proof
    assume "literal el literalList"
    then show ?thesis
      using literalListContainsItsLiteralsVariable[of "literal" "literalList"] 
      by simp
  next
    assume "(opposite literal) el literalList"
    then show ?thesis
      using literalListContainsItsLiteralsVariable[of "opposite literal" "literalList"]
      by simp
  qed
next
  assume "?lhs literalList" 
  then show "?rhs literalList"
  proof (induct literalList)
    case Nil
    then show ?case 
      by simp
  next
    case (Cons literal' literalList')
    note ih=this
    show ?case
    proof (cases "var literal : vars literalList'")
      case True
      with ih show "?rhs (literal' # literalList')" 
	by simp
    next
      case False
      with ih have "var literal' = var literal" 
	by simp
      then have "literal' = literal | opposite literal' = literal"
	by (simp add:literalsWithSameVariableAreEqualOrOpposite)
      then show "?rhs (literal' # literalList')" 
	by auto
    qed
  qed
qed


lemma variableDefinedInFormulaImpliesLiteralDefinedInFormula:
  fixes literal :: Literal and formula :: Formula
  assumes "var literal : vars formula" 
  shows "literal el formula | (opposite literal) el formula"
using assms
proof (induct formula)
  case (Cons clause formula')
  thus ?case
  proof (cases "var literal : vars clause")
    case True
    with variableDefinedImpliesLiteralDefined [of "literal" "clause"]
    show ?thesis
      by auto
  next
    case False
    with `var literal : vars (clause # formula')`
    have "var literal : vars formula'"
      by simp
    with Cons
    show ?thesis
      by auto
  qed
qed simp

(********************************************************************)
subsection{* Semantics *}
(********************************************************************)

types    Valuation = "Literal list"

consts inconsistent   :: "Valuation => bool"
primrec
"inconsistent [] = False"
"inconsistent (literal # valuation) = (((opposite literal) el valuation) | (inconsistent valuation))"

text{* Checks if the valuation contains both a literal and its opposite *}
definition [simp]: "consistent valuation == ~inconsistent valuation"

text{* Checks if the literal is contained in the given valuation *}
consts literalTrue     :: "Literal => Valuation => bool"
text{* Checks if the opposite literal is contained in the given valuation *}
consts literalFalse    :: "Literal => Valuation => bool"
text{* Checks if there is a literal from the clause which is true in the given valuation *}
consts clauseTrue      :: "Clause => Valuation => bool"
text{* Checks if all the literals from the clause are false in the given valuation *}
consts clauseFalse     :: "Clause => Valuation => bool"
text{* Checks if there is a clause from the formula which is false in the given valuation *}
consts formulaFalse    :: "Formula => Valuation => bool"
text{* Checks if all the clauses from the formula are false in the given valuation *}
consts formulaTrue     :: "Formula => Valuation => bool"

defs
literalTrue_def [simp]: "(literalTrue literal valuation) == literal el valuation"
literalFalse_def [simp] : "(literalFalse literal valuation) == (opposite literal) el valuation"

primrec
"clauseTrue [] valuation = False"
"clauseTrue (literal # clause) valuation = ((literalTrue literal valuation) | (clauseTrue clause valuation))"

primrec
"clauseFalse [] valuation = True"
"clauseFalse (literal # clause) valuation = ((literalFalse literal valuation) & (clauseFalse clause valuation))"

primrec
"formulaTrue [] valuation = True"
"formulaTrue (clause # formula) valuation = ((clauseTrue clause valuation) & (formulaTrue formula valuation))"

primrec
"formulaFalse [] valuation = False"
"formulaFalse (clause # formula) valuation = ((clauseFalse clause valuation) | (formulaFalse formula valuation))"


text{* Model of a formula is a consistent valuation under which formula/clause is true*}
consts model :: "Valuation => 'a => bool"
defs (overloaded)
modelFormula_def [simp]: "(model valuation (formula::Formula)) == consistent valuation & (formulaTrue formula valuation)"
modelClause_def [simp]: "(model valuation (clause::Clause)) == consistent valuation & (clauseTrue clause valuation)"

text{* Checks if a formula has a model *}
consts satisfiable :: "Formula => bool"
defs
satisfiable_def:
"(satisfiable formula) == (? valuation. (model valuation formula))"

(********************************************************************)

(*--------------------------------------------------------------------------------*)
lemma clauseTrueIffContainsTrueLiteral: 
  fixes clause :: Clause and valuation :: Valuation  
  shows "clauseTrue clause valuation = (? literal. literal el clause & literalTrue literal valuation)"
by (induct clause) auto

lemma clauseFalseIffAllLiteralsAreFalse:
  fixes clause :: Clause and valuation :: Valuation  
  shows "clauseFalse clause valuation = (! literal. literal el clause --> literalFalse literal valuation)"
by (induct clause) auto

lemma formulaTrueIffAllClausesAreTrue: 
  fixes formula :: Formula and valuation :: Valuation
  shows "formulaTrue formula valuation = (! clause. clause el formula --> clauseTrue clause valuation)"
by (induct formula) auto

lemma clauseTrueRemove:
  assumes "clauseTrue (remove literal clause) valuation"
  shows "clauseTrue clause valuation"
proof-
  from `clauseTrue (remove literal clause) valuation`
  obtain literal'::Literal where
    "literal' el (remove literal clause)" and "literalTrue literal' valuation"
    using clauseTrueIffContainsTrueLiteral [of "remove literal clause" "valuation"]
    by auto
  from `literal' el (remove literal clause)`
  have "literal' el clause"
    using memRemoveImpliesMemList[of "literal'" "literal" "clause"]
    by simp
  with `literalTrue literal' valuation`
  show ?thesis
    by (auto simp add: clauseTrueIffContainsTrueLiteral)
qed

(*--------------------------------------------------------------------------------*)
lemma inconsistentCharacterization: 
  fixes valuation :: Valuation
  shows "inconsistent valuation = (? literal. literalTrue literal valuation & literalFalse literal valuation)"
by (induct valuation) auto

lemma emptyValuationIsConsistent: "~inconsistent []"
by auto

lemma inconsistentRemove:
  assumes "inconsistent (remove literal valuation)"
  shows "inconsistent valuation"
using assms
proof (induct valuation)
  case (Cons literal' valuation')
  show ?case
    proof (cases "literal = literal'")
      case True
      thus ?thesis
	using Cons
	by auto
    next
      case False
      thus ?thesis
	using Cons
	using memRemoveImpliesMemList[of "opposite literal'" "literal" "valuation'"]
	by auto
    qed
qed simp

(*--------------------------------------------------------------------------------*)
lemma formulaRemainsTrueWhenItsNonMemberLiteralIsRemovedFromValuation:
  assumes "~ var literal : vars formula" and "formulaTrue formula valuation"
  shows "formulaTrue formula (remove literal (remove (opposite literal) valuation))" (is "formulaTrue formula ?valuation'")
proof-
{
  fix clause :: Clause
  assume "clause el formula"
  with `~var literal : vars formula` 
  have "~var literal : vars clause"
    using formulaContainsItsClausesVariables[of "clause" "formula"]
    by auto
  hence "~literal el clause" and "~(opposite literal) el clause"
    using variableDefinedImpliesLiteralDefined[of "literal" "clause"]
    by auto

  from `formulaTrue formula valuation` `clause el formula`
  have "clauseTrue clause valuation"
    by (simp add: formulaTrueIffAllClausesAreTrue)
  then obtain literal'::Literal
    where "literal' el clause" and "literalTrue literal' valuation"
    by (auto simp add: clauseTrueIffContainsTrueLiteral)
  
  from `literal' el clause` `~literal el clause`
  have "literal' ~= literal"
    by auto
  moreover
  from `literal' el clause` `~(opposite literal) el clause`
  have "literal' ~= (opposite literal)"
    by auto
  ultimately
  have "literalTrue literal' ?valuation'"
    using `literalTrue literal' valuation`
    using memListImpliesMemRemoveOrRemoved [of "literal'" "valuation" "opposite literal"]
    using memListImpliesMemRemoveOrRemoved [of "literal'" "(remove (opposite literal) valuation)" "literal"]
    by simp
  with `literal' el clause`
  have "clauseTrue clause ?valuation'"
    by (auto simp add: clauseTrueIffContainsTrueLiteral)
}
thus ?thesis
  by (simp add: formulaTrueIffAllClausesAreTrue)
qed

lemma formulaTrueDependsOnlyOnItsVariables:
  assumes "~vbl : vars formula" and "satisfiable formula"
  obtains valuation'::Valuation
  where "~vbl : vars valuation'" "model valuation' formula"
proof-
  from `satisfiable formula`
  obtain valuation::Valuation
    where "consistent valuation" and "formulaTrue formula valuation"
    unfolding satisfiable_def
    by auto
  let ?valuation' = "remove (Pos vbl) (remove (Neg vbl) valuation)"
  from `~vbl : vars formula` `formulaTrue formula valuation`
  have "formulaTrue formula ?valuation'"
    using formulaRemainsTrueWhenItsNonMemberLiteralIsRemovedFromValuation[of "Pos vbl" "formula" "valuation"]
    by simp
  moreover
  have "consistent ?valuation'"
    using `consistent valuation`
    using inconsistentRemove [of "Pos vbl" "remove (Neg vbl) valuation"]
    using inconsistentRemove [of "Neg vbl" "valuation"]
    by auto
  moreover
  have "~vbl : vars ?valuation'"
  proof-
    have "~(Pos vbl) mem ?valuation'"
      by (simp add: removedNotMemRemove)
    moreover
    have "~(Neg vbl) mem ?valuation'"
    proof-
      have "~(Neg vbl) mem (remove (Neg vbl) valuation)"
	by (simp add: removedNotMemRemove)
      thus ?thesis
	using memRemoveImpliesMemList [of "Neg vbl" "Pos vbl" "remove (Neg vbl) valuation"]
	by auto
    qed
    ultimately
    show ?thesis
      using variableDefinedImpliesLiteralDefined[of "Pos vbl" "?valuation'"]
      by simp
  qed
  ultimately
  obtain valuation' 
    where "~vbl : vars valuation'" and "model valuation' formula"
    by auto
  thus ?thesis
    ..
qed

(*--------------------------------------------------------------------------------*)
lemma satisfiableSubset: 
  fixes formula0 :: Formula and formula :: Formula
  assumes subset: "! (clause::Clause). clause el formula0 --> clause el formula"
  shows  "satisfiable formula --> satisfiable formula0"
proof
  assume "satisfiable formula"
  show "satisfiable formula0"
  proof -
    from `satisfiable formula` obtain valuation :: Valuation
      where "model valuation formula" by (auto simp add: satisfiable_def)
    {
      fix clause :: Clause
      assume "clause el formula0"
      with subset have "clause el formula" by simp
      with `model valuation formula` have "clauseTrue clause valuation" by (simp add: formulaTrueIffAllClausesAreTrue)
    } then have "formulaTrue formula0 valuation" by (simp add: formulaTrueIffAllClausesAreTrue)
    with `model valuation formula` have "model valuation formula0" by simp
    then show ?thesis by (auto simp add: satisfiable_def)
  qed
qed

lemma emptyFormulaIsSatisfiable:
"satisfiable []"
proof-
  have "formulaTrue ([]::Formula) ([]::Valuation)"
    by simp
  with emptyValuationIsConsistent
  have "model ([]::Valuation) ([]::Formula)"
    by auto
  thus ?thesis
    unfolding satisfiable_def
    by rule
qed

lemma formulaWithEmptyClauseIsUnsatisfiable:
  assumes "[] mem formula"
  shows "~satisfiable formula"
using assms
unfolding satisfiable_def
by (auto simp add: formulaTrueIffAllClausesAreTrue)

end