(*    Title:              ClassicDPLL.thy
      ID:                 $Id: ClassicDPLL.thy$
      Author:             Filip Maric
*)

theory ClassicDPLL
imports CNF
begin

text{* Substitutes given literal with True, substitutes its opposite with False in formula
       and simplifies the formula afterwards. All clauses that contain True are removed from formula, 
       and all constants False are removed from clauses. *}
consts setLiteralTrue :: "Literal => Formula => Formula"
primrec
"setLiteralTrue literal [] = []"
"setLiteralTrue literal (clause # formula) = 
  (if (literal mem clause) then 
      (setLiteralTrue literal formula)
  else (if ((opposite literal) mem clause) then
      ((remove (opposite literal) clause) # (setLiteralTrue literal formula))
  else
      (clause # (setLiteralTrue literal formula))
  )
)"


lemma setLiteralTrueEliminatesVariable: 
  shows "~var literal : vars (setLiteralTrue literal formula)"
proof (induct formula)
  case (Cons clause formula')
  show ?case
  proof (cases "literal mem clause")
    case True
    with Cons
    show ?thesis
      by simp
  next
    case False
    show ?thesis
    proof (cases "opposite literal mem clause")
      case True
      from `~literal mem clause` 
      have "~literal mem (remove (opposite literal) clause)"
	using memRemoveImpliesMemList[of "literal" "opposite literal""clause"]
	by auto
      hence "~var literal : vars (remove (opposite literal) clause)"
	using removedNotMemRemove[of "opposite literal" "clause"]
	using variableDefinedImpliesLiteralDefined[of "literal" "remove (opposite literal) clause"]
	by simp
      with Cons True
      show ?thesis
	by simp
    next
      case False
      from `~literal mem clause` `~opposite literal mem clause` 
      have "~var literal : vars clause"
	using variableDefinedImpliesLiteralDefined[of "literal" "clause"]
	by simp
      with Cons False
      show ?thesis
	by simp
    qed
  qed
qed simp


lemma setLiteralTrueEliminatesLiteral:
  fixes literal::Literal and formula::Formula
  shows "~literal el (setLiteralTrue literal formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  show ?case
    proof (cases "literal mem clause")
      case True
      with Cons
      show ?thesis
	by simp
    next
      case False
      show ?thesis
      proof (cases "(opposite literal) mem clause")
	from `~literal mem clause` 
	have "~literal mem (remove (opposite literal) clause)"
	  using memRemoveImpliesMemList[of "literal" "opposite literal""clause"]
	  by auto
	with Cons
	show ?thesis
	  by simp
      next
	case False
	with Cons
	show ?thesis
	  by simp
      qed
    qed
qed simp

lemma setLiteralTrueNonMemberLiteralDoesNotChangeTheFormula:
  assumes "~var literal : vars formula"
  shows "setLiteralTrue literal formula = formula"
using assms
proof (induct formula)
  case (Cons clause formula')
  then have "var literal ~: vars clause" and "var literal ~: vars formula'"
    by auto
  from `var literal ~: vars clause`
  have "~literal mem clause" and "~(opposite literal) mem clause"
    using variableDefinedImpliesLiteralDefined [of "literal" "clause"]
    by auto
  with Cons
  show ?case
    by simp
qed simp

lemma literalNotDefinedInFormulaImpliesSetTrueLiteralEqualsSetTrueOppositeLiteral:
  assumes "~literal el formula" and "~(opposite literal) el formula"
  shows "setLiteralTrue literal formula = setLiteralTrue (opposite literal) formula"
proof-
  from assms 
  have "~var literal : vars formula"
    using variableDefinedInFormulaImpliesLiteralDefinedInFormula[of "literal" "formula"]
    by auto
  from `~var literal : vars formula` 
  have "setLiteralTrue literal formula = formula"
    by (simp add: setLiteralTrueNonMemberLiteralDoesNotChangeTheFormula)
  moreover
  from `~var literal : vars formula` 
  have "setLiteralTrue (opposite literal) formula = formula"
    by (simp add: setLiteralTrueNonMemberLiteralDoesNotChangeTheFormula)
  ultimately
  show ?thesis
    by simp
qed


lemma setLiteralTrueLiteralTrueInSatisfyingValuation:
  fixes valuation::Valuation and formula::Formula and literal::Literal
  assumes "model valuation formula" and "literalTrue literal valuation" 
  shows "model valuation (setLiteralTrue literal formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  from `model valuation (clause # formula')`
  have "consistent valuation" and "clauseTrue clause valuation" and "formulaTrue formula' valuation"
    by auto
  show ?case
  proof (cases "literal el clause")
    case True
    with `consistent valuation` `clauseTrue clause valuation` `formulaTrue formula' valuation` Cons
    show ?thesis
      by auto
  next
    case False
    show ?thesis
    proof (cases "(opposite literal) el clause")
      case True
      have "clauseTrue (remove (opposite literal) clause) valuation"
      proof-
	from `clauseTrue clause valuation` 
	obtain literal'::Literal
	  where "literal' el clause" and "literalTrue literal' valuation"
	  by (auto simp add: clauseTrueIffContainsTrueLiteral)
	show ?thesis
	proof (cases "literal' = (opposite literal)")
	  case False
	  with `literal' el clause`
	  have "literal' el (remove (opposite literal) clause)"
	    by (induct clause) auto
	  with `literalTrue literal' valuation` show ?thesis
	    by (auto simp add: clauseTrueIffContainsTrueLiteral)
	next
	  case True
	  with `literalTrue literal' valuation` 
	  have "literalFalse literal valuation"
	    by simp
	  with `literalTrue literal valuation`
	  have "inconsistent valuation"
	    by (auto simp add: inconsistentCharacterization)
	  with `consistent valuation`
	  have False
	    by simp
	  thus ?thesis
	    by simp
	qed
      qed
      with Cons True
      show ?thesis
	by auto
    next
      case False
      with Cons `clauseTrue clause valuation`
      show ?thesis
	by auto
    qed
  qed
qed simp


lemma setLiteralTrueLiteralUndefinedInSatisfyingValuation:
  assumes "model valuation formula" and "var literal ~: vars valuation"
  shows "model valuation (setLiteralTrue literal formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  from `model valuation (clause # formula')`
  have "consistent valuation" and "clauseTrue clause valuation" and "formulaTrue formula' valuation"
    by auto
  show ?case
  proof (cases "literal el clause")
    case True
    with `consistent valuation` `clauseTrue clause valuation` `formulaTrue formula' valuation` Cons
    show ?thesis
      by auto
  next
    case False
    show ?thesis
    proof (cases "(opposite literal) el clause")
      case True
      have "clauseTrue (remove (opposite literal) clause) valuation"
      proof-
	from `clauseTrue clause valuation` 
	obtain literal'::Literal
	  where "literal' el clause" and "literalTrue literal' valuation"
	  by (auto simp add: clauseTrueIffContainsTrueLiteral)
	from `var literal ~: vars valuation`
	have "~literalTrue literal valuation" and "~literalFalse literal valuation"
	  using variableDefinedImpliesLiteralDefined[of "literal" "valuation"]
	  by auto
	with `literalTrue literal' valuation`
	have "literal' ~= (opposite literal)"
	  by auto
	with `literal' el clause`
	have "literal' el (remove (opposite literal) clause)"
	  by (induct clause) auto
	with `literalTrue literal' valuation`
	show ?thesis
	  by (auto simp add: clauseTrueIffContainsTrueLiteral)
      qed
      with Cons
      show ?thesis
	by simp
    next
      case False
      with Cons `clauseTrue clause valuation`
      show ?thesis
	by auto
    qed
  qed
qed simp

lemma extendValuationWithSetLiteralTrueLiteral: 
  assumes "formulaTrue (setLiteralTrue literal formula) valuation" 
  shows "formulaTrue formula (literal # valuation)"
using assms
proof (induct formula)
  case (Cons clause formula')
  show ?case
  proof (cases "literal el clause")
    case True
    hence "clauseTrue clause (literal # valuation)"
      by (auto simp add: clauseTrueIffContainsTrueLiteral)
    with Cons True
    show ?thesis
      by simp
  next
    case False
    show ?thesis
    proof (cases "opposite literal el clause")
      case True
      with False Cons(2)
      have "clauseTrue (remove (opposite literal) clause) valuation"
	by auto
      hence "clauseTrue clause valuation"
	using clauseTrueRemove[of "opposite literal" "clause" "valuation"]
	by simp
      hence "clauseTrue clause (literal # valuation)"
	by (auto simp add: clauseTrueIffContainsTrueLiteral)
      with `~literal el clause` True Cons
      show ?thesis
	by auto
    next
      case False
      with `~literal el clause` Cons
      show ?thesis
	by (auto simp add: clauseTrueIffContainsTrueLiteral)
    qed
  qed
qed simp

lemma satisfiableSetLiteralTrueImpliesSatisfiableFormula:
  assumes "satisfiable (setLiteralTrue literal formula)"
  shows "satisfiable formula"
proof-
  from assms obtain valuation::Valuation
    where "consistent valuation" "formulaTrue (setLiteralTrue literal formula) valuation"
    unfolding satisfiable_def
    by auto
  then obtain valuation'::Valuation
    where "model valuation' (setLiteralTrue literal formula)" "var literal ~: vars valuation'"
    using setLiteralTrueEliminatesVariable[of "literal" "formula"]
    using formulaTrueDependsOnlyOnItsVariables[of "var literal" "setLiteralTrue literal formula"]
    unfolding satisfiable_def
    by auto
  hence "consistent valuation'" and "formulaTrue (setLiteralTrue literal formula) valuation'"
    by auto
  hence "formulaTrue formula (literal # valuation')"
    using extendValuationWithSetLiteralTrueLiteral[of "literal" "formula" "valuation'"]
    by simp
  moreover
  from `var literal ~: vars valuation'` 
  have "~(opposite literal) el valuation'"
    using variableDefinedImpliesLiteralDefined[of "literal" "valuation'"]
    by simp
  with `consistent valuation'` 
  have "consistent (literal # valuation')"
    by auto
  ultimately
  show ?thesis
    unfolding satisfiable_def
    by (auto simp del: inconsistent.simps)
qed

lemma split_rule:
  "satisfiable formula = 
  (satisfiable (setLiteralTrue literal formula) | satisfiable (setLiteralTrue (opposite literal) formula))" 
  (is "?lhs = ?rhs")
proof
  assume "?lhs"
  show "?rhs"
  proof-
    from `satisfiable formula`
    obtain valuation::Valuation
      where "consistent valuation" and "formulaTrue formula valuation"
      unfolding satisfiable_def
      by auto
    show ?thesis
    proof (cases "var literal : vars valuation")
      case True
      hence "literalTrue literal valuation | literalFalse literal valuation"
	using variableDefinedImpliesLiteralDefined [of "literal" "valuation"]
	by simp
      moreover
      {
	assume "literalTrue literal valuation"
	with `consistent valuation` `formulaTrue formula valuation`
	have "model valuation (setLiteralTrue literal formula)"
	  using setLiteralTrueLiteralTrueInSatisfyingValuation[of "valuation" "formula" "literal"]
	  by simp
	hence "satisfiable (setLiteralTrue literal formula)"
	  unfolding satisfiable_def
	  by auto
      }
      moreover
      {
	assume "literalFalse literal valuation"
	with `consistent valuation` `formulaTrue formula valuation`
	have "model valuation (setLiteralTrue (opposite literal) formula)"
	  using setLiteralTrueLiteralTrueInSatisfyingValuation[of "valuation" "formula" "opposite literal"]
	  by simp
	hence "satisfiable (setLiteralTrue (opposite literal) formula)"
	  unfolding satisfiable_def
	  by auto
      }
      ultimately
      show ?thesis
	by auto
    next
      case False
      with `consistent valuation` `formulaTrue formula valuation`
      show ?thesis
	using setLiteralTrueLiteralUndefinedInSatisfyingValuation [of "valuation" "formula" "literal"]
	unfolding satisfiable_def
	by auto
    qed
  qed
next
  assume "?rhs"
  moreover
  {
    assume "satisfiable (setLiteralTrue literal formula)"
    then have "satisfiable formula"
      using satisfiableSetLiteralTrueImpliesSatisfiableFormula[of "literal" "formula"]
      by simp
  }
  moreover
  {
    assume "satisfiable (setLiteralTrue (opposite literal) formula)"
    then have "satisfiable formula"
      using satisfiableSetLiteralTrueImpliesSatisfiableFormula[of "opposite literal" "formula"]
      by simp
  }
  ultimately
  show "?lhs"
    by auto
qed

lemma setLiteralTrueUnitLiteral:
  assumes "[literal] mem formula"
  shows "[] mem (setLiteralTrue (opposite literal) formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  show ?case
  proof (cases "clause = [literal]")
    case True
    thus ?thesis
      by auto
  next
    case False
    with Cons
    show ?thesis
      by simp
  qed
qed simp

lemma unitLiteralRule:
  assumes "[literal] mem formula"
  shows "satisfiable formula = satisfiable (setLiteralTrue literal formula)" (is "?lhs = ?rhs")
proof
  assume "?lhs"
  from `[literal] mem formula` 
  have "[] mem (setLiteralTrue (opposite literal) formula)"
    by (simp add: setLiteralTrueUnitLiteral)
  hence "~satisfiable (setLiteralTrue (opposite literal) formula)"
    by (simp add: formulaWithEmptyClauseIsUnsatisfiable)
  thus "?rhs"
    using `?lhs` split_rule[of "formula" "literal"]
    by auto
next
  assume "?rhs"
  thus "?lhs"
    using split_rule[of "formula" "literal"]
    by auto
qed

lemma pureLiteralRuleSubset:
  fixes literal::Literal and clause::Clause and formula::Formula
  assumes "~opposite literal el formula" and
  "clause el (setLiteralTrue literal formula)" 
  shows "clause el (setLiteralTrue (opposite literal) formula)"
using assms
proof (induct formula)
  case (Cons clause' formula')
  show ?case
  proof (cases "literal mem clause'")
    case True
    with Cons
    show ?thesis
      by auto
  next
    case False
    with Cons
    show ?thesis
      by auto
  qed
qed simp

lemma pureLiteralRule:
  assumes "~opposite literal el formula" 
  shows "satisfiable formula = satisfiable (setLiteralTrue literal formula)" (is "?lhs = ?rhs")
proof
  assume "?lhs"
  hence "satisfiable (setLiteralTrue literal formula) | satisfiable (setLiteralTrue (opposite literal) formula)"
    using split_rule[of "formula" "literal"]
    by simp
  moreover
  {
    assume "satisfiable (setLiteralTrue (opposite literal) formula)"
    then obtain valuation::Valuation
      where "consistent valuation" and "formulaTrue (setLiteralTrue (opposite literal) formula) valuation"
      unfolding satisfiable_def
      by auto
    have "satisfiable (setLiteralTrue literal formula)"
    proof-
      {
	fix clause::Clause
	assume "clause el (setLiteralTrue literal formula)"
	with `~opposite literal el formula`
	have "clause el (setLiteralTrue (opposite literal) formula)"
	  using pureLiteralRuleSubset[of "literal" "formula" "clause"]
	  by simp
      }
      with `formulaTrue (setLiteralTrue (opposite literal) formula) valuation` 
      have "formulaTrue (setLiteralTrue literal formula) valuation"
	using formulaTrueIffAllClausesAreTrue
	by simp
      with `consistent valuation`
      show ?thesis
	unfolding satisfiable_def
	by auto
    qed
  }
  ultimately
  show "?rhs"
    by auto
next
  assume "?rhs"
  thus "?lhs"
    using split_rule[of "formula" "literal"]
    by auto
qed

(*************************************************************)
(*             SPLIT LITERAL SELECTION STRATEGY              *)
(*************************************************************)
constdefs selectLiteral::"Formula => Literal"
"(selectLiteral formula) == (hd (hd formula))"

lemma selectLiteralElFormula: 
  assumes "formula ~= []" and "~[] mem formula" 
  shows "(selectLiteral formula) el formula"
using assms
unfolding selectLiteral_def
by (induct formula) (auto simp add:headNonEmptyListMemList split: split_if_asm)

(*************************************************************)
(*                 UNIT LITERAL DETECTION                    *)
(*************************************************************)
constdefs hasUnitLiteral :: "Formula => bool"
"hasUnitLiteral formula == (? literal. [literal] mem formula)"

consts getUnitLiteral :: "Formula => Literal"
primrec
"getUnitLiteral (clause # formula) = 
(if (? literal. clause = [literal]) then
  (hd clause)
else (getUnitLiteral formula))"

lemma getUnitLiteralIsUnit:
  assumes "hasUnitLiteral formula"
  shows "[getUnitLiteral formula] mem formula"
using assms
unfolding hasUnitLiteral_def
by (induct formula) (auto split: split_if_asm)


(*************************************************************)
(*                 PURE LITERAL DETECTION                    *)
(*************************************************************)
consts hasPureLiteral :: "Formula => bool"
consts getPureLiteral :: "Formula => Literal"
consts getLiterals :: "Formula => (Literal list)"

consts getPureLiteralHelper :: "(Literal list) => (Literal list) => Literal"
consts hasPureLiteralHelper :: "Literal list => Literal list => bool"


primrec 
"getLiterals [] = []"
"getLiterals (clause#formula) = clause@(getLiterals formula)"

defs
hasPureLiteral_def: "hasPureLiteral formula == (? literal. literal el formula & ~(opposite literal) el formula)"

primrec
"(getPureLiteralHelper (literal # rest) literals) = 
 (if ~(opposite literal) mem literals then 
      literal 
  else 
     (getPureLiteralHelper rest literals)
 )"

defs
getPureLiteral_def: "getPureLiteral formula == (getPureLiteralHelper (getLiterals formula) (getLiterals formula))"

defs
hasPureLiteralHelper_def: "(hasPureLiteralHelper l s) == (? literal. literal mem l & ~(opposite literal) mem s)"

lemma hasPureLiteralHelperMem: 
  assumes "hasPureLiteralHelper l s"
  shows "(getPureLiteralHelper l s) mem l"
using assms
unfolding hasPureLiteralHelper_def
proof (induct l)
  case (Cons a l')
  then obtain literal::Literal
    where "literal mem (a # l')" and "~(opposite literal) mem s"
    by auto
  show ?case
  proof (cases "literal = a")
    case True
    with Cons `~(opposite literal) mem s`
    show ?thesis
      by simp
  next
    case False
    with Cons `literal mem (a # l')`
    show ?thesis
      by (auto split: split_if_asm)
  qed
qed simp

lemma hasPureLiteralHelperNotOppositeMem: 
  assumes "hasPureLiteralHelper l s"
  shows "~opposite (getPureLiteralHelper l s) mem s"
using assms
unfolding hasPureLiteralHelper_def
proof (induct l)
  case (Cons a l')
  then obtain literal::Literal
    where "literal mem (a # l')" and "~(opposite literal) mem s"
    by auto
  show ?case
  proof (cases "literal = a")
    case True
    with Cons `~(opposite literal) mem s`
    show ?thesis
      by simp
  next
    case False
    with Cons `literal mem (a # l')`
    show ?thesis
      by (auto split: split_if_asm)
  qed
qed simp


lemma memGetLiterals: 
  shows "literal el formula = literal mem (getLiterals formula)"
  by (induct formula) (auto simp add: memAppend)

lemma hasPureLiteralHasPureLiteralHelper:
  shows "hasPureLiteral formula = hasPureLiteralHelper (getLiterals formula) (getLiterals formula)"
unfolding hasPureLiteralHelper_def
unfolding hasPureLiteral_def
by (auto simp add: memGetLiterals)

lemma getPureLiteralIsPure:
  assumes "hasPureLiteral formula"
  shows "getPureLiteral formula el formula" and "~opposite (getPureLiteral formula) el formula"
proof-
  from `hasPureLiteral formula`
  have *: "hasPureLiteralHelper (getLiterals formula) (getLiterals formula)"
    using hasPureLiteralHasPureLiteralHelper
    by simp
  from * have "getPureLiteralHelper (getLiterals formula) (getLiterals formula) mem (getLiterals formula)"
    using hasPureLiteralHelperMem
    by simp
  then show "getPureLiteral formula el formula"
    unfolding getPureLiteral_def
    by (auto simp add: memGetLiterals)
  from * have "~opposite (getPureLiteralHelper (getLiterals formula) (getLiterals formula)) mem (getLiterals formula)"
    using hasPureLiteralHelperNotOppositeMem
    by simp
  then show "~opposite (getPureLiteral formula) el formula"
    unfolding getPureLiteral_def
    by (auto simp add: memGetLiterals)
qed

(*************************************************************)
(*                 AXIOMATIC APPROACH                        *)
(*************************************************************)
(*
consts selectLiteral::"Formula => Literal"
axioms selectLiteralElFormula:
  "formula ~= [] & ~[] mem formula --> (selectLiteral formula) el formula"

consts hasUnitLiteral :: "Formula => bool"
consts getUnitLiteral :: "Formula => Literal"
axioms getUnitLiteralIsUnit: 
  "hasUnitLiteral formula --> [getUnitLiteral formula] mem formula"

consts hasPureLiteral :: "Formula => bool"
consts getPureLiteral :: "Formula => Literal"
axioms getPureLiteralIsPure: 
  "hasPureLiteral formula --> getPureLiteral formula el formula & ~opposite (getPureLiteral formula) el formula"
*)

lemma getUnitLiteralElFormula:
  assumes "hasUnitLiteral formula"
  shows "getUnitLiteral formula el formula"
proof-
  from assms
  have "[getUnitLiteral formula] mem formula"
    using getUnitLiteralIsUnit[of "formula"]
    by simp
  thus ?thesis
    using literalElFormulaCharacterization [of "getUnitLiteral formula" "formula"]
    by auto
qed

(*************************************************************)
(*                  TERMINATION  MEASURE                     *)
(*************************************************************)
consts numLiterals::"Formula => nat"
primrec
"numLiterals [] = 0"
"numLiterals (clause # formula) = (length clause) + (numLiterals formula)"

lemma setLiteralTrueNumLiteralsLEQ:
  "numLiterals (setLiteralTrue literal formula) <= numLiterals formula"
proof (induct formula)
  case (Cons clause formula')
  thus ?case
    using lengthRemove[of "opposite literal" "clause"]
    by auto
qed simp

lemma numLiteralsSetLiteralTrueMemberLiteral: 
  assumes "literal el formula" 
  shows "numLiterals formula > numLiterals (setLiteralTrue literal formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  show ?case
  proof (cases "literal mem clause")
    case True
    hence "length clause > 0"
      by auto
    with setLiteralTrueNumLiteralsLEQ [of "literal" "formula'"]
    have "numLiterals (setLiteralTrue literal formula') < length clause + numLiterals formula'"
      by arith
    with True
    show ?thesis
      by (auto split: split_if_asm)
  next
    case False
    show ?thesis
      proof (cases "opposite literal mem clause")
	case True
	hence "length (remove (opposite literal) clause) < length clause"
	  using lengthRemoveMember[of "opposite literal" "clause"]
	  by simp
	with setLiteralTrueNumLiteralsLEQ [of "literal" "formula'"]
	have "length (remove (opposite literal) clause) + numLiterals (setLiteralTrue literal formula') < 
	  length clause + numLiterals formula'"
	  by arith
	with `~literal mem clause` `opposite literal mem clause`
	show ?thesis
	  by (auto split: split_if_asm)
      next
	case False
	with Cons `~literal mem clause`
	show ?thesis
	  by auto
      qed
    qed
qed simp

lemma numLiteralsSetLiteralTrueOppositeMemberLiteral: 
  assumes "literal el formula"
  shows "numLiterals formula > numLiterals (setLiteralTrue (opposite literal) formula)"
using assms
proof (induct formula)
  case (Cons clause formula')
  show ?case
  proof (cases "(opposite literal) mem clause")
    case True
    hence "length clause > 0"
      by auto
    with setLiteralTrueNumLiteralsLEQ [of "opposite literal" "formula'"]
    have "numLiterals (setLiteralTrue (opposite literal) formula') < length clause + numLiterals formula'"
      by arith
    with True
    show ?thesis
      by (auto split: split_if_asm)
  next
    case False
    show ?thesis
      proof (cases "literal mem clause")
	case True
	hence "length (remove literal clause) < length clause"
	  using lengthRemoveMember[of "literal" "clause"]
	  by simp
	with setLiteralTrueNumLiteralsLEQ [of "opposite literal" "formula'"]
	have "length (remove literal clause) + numLiterals (setLiteralTrue (opposite literal) formula') < 
	  length clause + numLiterals formula'"
	  by arith
	with `literal mem clause` `~opposite literal mem clause`
	show ?thesis
	  by (auto split: split_if_asm)
      next
	case False
	with Cons 
	show ?thesis
	  by auto
      qed
    qed
qed simp

(*************************************************************)
(*                  TERMINATION  CONDITIONS                  *)
(*************************************************************)

lemma dpllTermination_1: 
  assumes "formula ~= []" and "~[] mem formula" 
  shows "numLiterals formula > numLiterals (setLiteralTrue (selectLiteral formula) formula)"
proof-
  from assms 
  have "(selectLiteral formula) el formula"
    by (simp add: selectLiteralElFormula)
  thus ?thesis
    by (rule numLiteralsSetLiteralTrueMemberLiteral)
qed

lemma dpllTermination_2: 
  assumes "formula ~= []" and "~[] mem formula" 
  shows "numLiterals formula > numLiterals (setLiteralTrue (opposite (selectLiteral formula)) formula)"
proof-
  from assms 
  have "(selectLiteral formula) el formula"
    by (simp add: selectLiteralElFormula)
  thus ?thesis
    by (rule numLiteralsSetLiteralTrueOppositeMemberLiteral)
qed

lemma dpllTermination_3:
  assumes "hasUnitLiteral formula"
  shows "numLiterals formula > numLiterals (setLiteralTrue (getUnitLiteral formula) formula)"
proof-
  from assms
  have "getUnitLiteral formula el formula"
    by (rule getUnitLiteralElFormula)
  thus ?thesis
    by (rule numLiteralsSetLiteralTrueMemberLiteral)
qed

lemma dpllTermination_4:
  assumes "hasPureLiteral formula"
  shows "numLiterals formula > numLiterals (setLiteralTrue (getPureLiteral formula) formula)"
proof-
  from assms
  have "getPureLiteral formula el formula"
    by (simp add: getPureLiteralIsPure)
  thus ?thesis
    by (rule numLiteralsSetLiteralTrueMemberLiteral)
qed

(*************************************************************)
(*           DAVIS-PUTNAM-LONGEMAN-LOVELLAND                 *)
(*************************************************************)

function dpll::"Formula => bool" 
where
"(dpll formula) = 
 (if (formula = []) then 
       True
  else if ([] mem formula) then 
       False
  else if (hasPureLiteral formula) then
      (dpll (setLiteralTrue (getPureLiteral formula) formula))
  else if (hasUnitLiteral formula) then
      (dpll (setLiteralTrue (getUnitLiteral formula) formula))
  else if (dpll (setLiteralTrue (selectLiteral formula) formula)) then 
       True 
  else
      (dpll (setLiteralTrue (opposite (selectLiteral formula)) formula))
 )"
by pat_completeness auto
termination
by (relation "measure (% formula. (numLiterals formula))") 
   (auto simp add: dpllTermination_1 dpllTermination_2 dpllTermination_3 dpllTermination_4)

(*************************************************************)
(*                      DPLL Correctness                     *)
(*************************************************************)

lemma dpllCorrectness: "(dpll F) = (satisfiable F)"
proof (induct F rule: dpll.induct)
  case (1 formula)
  note ih = this
  show ?case
  proof (cases "formula = []")
    case True
    thus ?thesis
      by (simp add:emptyFormulaIsSatisfiable)
  next
    case False
    show ?thesis
    proof (cases "[] mem formula")
      case True
      with `formula ~= []` 
      show ?thesis
	by (simp add:formulaWithEmptyClauseIsUnsatisfiable)
    next
      case False
      show ?thesis
      proof (cases "hasPureLiteral formula")
	case True
	let ?pl = "getPureLiteral formula"
	from True have "?pl el formula" "~opposite ?pl el formula"
	  by (auto simp add: getPureLiteralIsPure)
	with True `formula ~= []` `~[] mem formula` ih
	  pureLiteralRule[of "?pl" "formula"]
	show ?thesis
	  by auto
      next 
	case False
	show ?thesis
	proof (cases "hasUnitLiteral formula")
	  case True
	  let ?ul = "getUnitLiteral formula"
	  from True have "[?ul] mem formula"
	    by (simp add: getUnitLiteralIsUnit)
	  with True `formula ~= []` `~[] mem formula` `~hasPureLiteral formula` ih
	    unitLiteralRule[of "?ul" "formula"]
	  show ?thesis
	    by auto
	next
	  case False
	  with `formula ~= []` `~[] mem formula` `~hasPureLiteral formula` `~hasUnitLiteral formula` ih
	    split_rule[of "formula" "selectLiteral formula"]
	  show ?thesis
	    by auto
	qed
      qed
    qed
  qed
qed

end