section{* Executable implementations of set families *}

subsection{* Abstract representation of sets and families *}

theory FamilyImpl
imports Family
        Frankl (* TODO: eliminate dependency by migrating frankl_fun to FranklImpl *)
        More.MoreSet ListNat
        "HOL-Library.List_lexord"
        "HOL-Library.Code_Target_Nat"
begin

abbreviation map where "map \<equiv> List.map"

text{* Set operations are specified in the following locale. Families are
always represented as (distinct) lists of sets. *}

(* ************************************************************************ *)
locale SetImpl = 
  fixes to_set :: "'s \<Rightarrow> 'a set"
  fixes inv :: "'s \<Rightarrow> bool"
  assumes to_set_inj: "\<lbrakk>inv s1; inv s2; to_set s1 = to_set s2\<rbrakk> \<Longrightarrow> s1 = s2"
  assumes to_set_ex: "finite a \<Longrightarrow> \<exists> s. a = to_set s \<and> inv s"
  assumes to_set_finite: "finite (to_set s)"
begin

definition f_to_set :: "'s list \<Rightarrow> 'a set set" where
  [simp]: "f_to_set F = set (map to_set F)"

abbreviation fs_to_set :: "'s list list \<Rightarrow> 'a set set set" ("\<circle>") where
  "\<circle> FF \<equiv> set (map f_to_set FF)"

lemma to_set_inj_on:
  shows "\<forall> a \<in> set A. inv a \<Longrightarrow> inj_on to_set (set A)"
unfolding inj_on_def
by (simp add: to_set_inj)

lemma set_set: 
  "\<lbrakk>\<forall>a\<in>set A. inv a; inv h\<rbrakk> \<Longrightarrow> 
     h \<in> set A \<longleftrightarrow> to_set h \<in> f_to_set A"
using to_set_inj
by auto

lemma f_to_set_ex:
  assumes "finite (\<Union> F)"
  shows "\<exists> Fl. F = f_to_set Fl \<and> (\<forall> A \<in> set Fl. inv A)"
proof-
  from assms
  have "finite F" "\<forall> A \<in> F. finite A"
    by (auto simp add: finiteUn_iff)
  thus ?thesis
  proof (induct F)
    case empty
    thus ?case
      by auto
  next
    case (insert A F)
    then obtain Fl Al where "F = f_to_set Fl \<and> (\<forall> A \<in> set Fl. inv A)" "A = to_set Al \<and> inv Al"
      using to_set_ex[of A]
      by auto
    thus ?case
      by (rule_tac x="Al # Fl" in exI) auto
  qed
qed

abbreviation contains :: "'a \<Rightarrow> 's \<Rightarrow> bool" where
  "contains a S \<equiv> a \<in> to_set S"

definition count :: "'a \<Rightarrow> 's list \<Rightarrow> nat" where
  "count a F = length (filter (contains a) F)"

lemma count_set:
  assumes "distinct F" "\<forall> x \<in> set F. inv x" 
  shows "count a F = Family.count a (f_to_set F)"
proof-
  have "to_set ` {x \<in> set F. a \<in> to_set x} = {S \<in> to_set ` set F. a \<in> S}"
    by auto
  moreover
  have "inj_on to_set {x \<in> set F. a \<in> to_set x}"
    using to_set_inj `\<forall> x \<in> set F. inv x`
    unfolding inj_on_def
    by auto
  ultimately
  have "card {x \<in> set F. a \<in> to_set x} = card {S \<in> to_set ` set F. a \<in> S}"
    using card_image[of to_set "{x \<in> set F. a \<in> to_set x}"]
    by auto
  thus ?thesis
    using assms distinct_card[of "filter (contains a) F", symmetric]
    unfolding count_def Family.count_def contains_def
    by simp
qed

(* TODO: migrate to FranklImp.thy *)

definition frankl_fun :: "'a \<Rightarrow> 's list \<Rightarrow> int" where 
  "frankl_fun x F \<equiv> 2 * int (count x F) - int (length F)"

lemma frankl_fun_set: 
  assumes "distinct Fs" "\<forall> A \<in> set Fs. inv A"
  shows "frankl_fun a Fs = Frankl.frankl_fun a (f_to_set Fs)"
proof-
  have "inj_on to_set (set Fs)"
    using assms to_set_inj
    by (auto simp add: inj_on_def)
  hence "card (to_set ` set Fs) = card (set Fs)"
    by (rule card_image)
  thus ?thesis
    using distinct_card[OF assms(1)] count_set[OF assms, of a]
    unfolding frankl_fun_def
    by auto
qed

end (* Locale *)

(* ----------------------------------------------------------------------- *)
subsubsection{* Implementation of sets by sorted and distinct lists *}
(* ----------------------------------------------------------------------- *)


abbreviation sd where
  "sd A \<equiv> sorted A \<and> distinct A"
abbreviation sdf where
  "sdf F \<equiv> \<forall> A \<in> set F. sd A"
abbreviation sdff where
  "sdff \<F> \<equiv> \<forall> F \<in> set \<F>. sdf F"

text{* For example, the family @{text "{{1, 2, 3}, {2, 3, 4}"} is represented
by @{text "[[1, 2, 3], [2, 3, 4]]"}. *}

(* FIXME: must fix nat type because of NonIsomorphicFamilies *)

global_interpretation SetImpl_lists: SetImpl "set :: nat list \<Rightarrow> nat set" "sd"
  defines f_to_set_l = "SetImpl_lists.f_to_set" and
          count_l = "SetImpl_lists.count" and
          frankl_fun_l = "SetImpl_lists.frankl_fun"
proof (unfold_locales)
  fix a :: "nat set"
  assume "finite a"
  thus "\<exists>s. a = set s \<and> sd s"
    apply (rule_tac x="sorted_list_of_set a" in exI)
    by simp
qed (auto simp add: sorted_distinct_set_unique)

abbreviation dm where
  "dm F n \<equiv> \<Union> f_to_set_l F \<subseteq> {0..<n}"
abbreviation dmf where 
  "dmf FF n \<equiv> \<forall> F \<in> set FF. dm F n"

abbreviation fs_to_set_l where
 "fs_to_set_l F \<equiv> set (map f_to_set_l F)"

(* ----------------------------------------------------------------------- *)
subsubsection{* Implementation of sets of nats by nats *}
(* ----------------------------------------------------------------------- *)

text{* For example, the family @{text "{{1, 2, 3}, {2, 3, 4}"} is represented
by @{text "[14, 28]"}. Encoding and decoding funcions are defined in 
the theory ListNat. *}

global_interpretation SetImpl_nats: SetImpl "set \<circ> nat2list" "\<lambda> n. True"
  defines f_to_set_n = "SetImpl_nats.f_to_set" and
          count_n = "SetImpl_nats.count" and
          frankl_fun_n = "SetImpl_nats.frankl_fun"
proof (unfold_locales)
  fix s1 s2
  assume "(set \<circ> nat2list) s1 = (set \<circ> nat2list) s2"
  thus "s1 = s2"
    using sorted_nat2list[of s1] sorted_nat2list[of s2]
    using distinct_nat2list[of s1] distinct_nat2list[of s2]
    using sorted_distinct_set_unique[of "nat2list s1" "nat2list s2"]
    using inj_nat2list
    by (simp add: inj_on_def)
next
  fix a :: "nat set"
  assume "finite a"
  thus "\<exists> s. a = (set \<circ> nat2list) s \<and> True"
    by (rule_tac x="list2nat (sorted_list_of_set a)" in exI) (simp add: nat2list_list2nat)
qed simp_all

ML{*

(* Converts a term that represents a family (set of sets) to a corresponding list of lists. If the
term represents a list of families, then the the transformation is applied to each family in the
list and the term that represents the corresponding list of lists of lists is returned. *)

fun sets_to_lists F =
  let
    val c1 = Const ("List.list.Cons", @{typ "nat \<Rightarrow> nat list \<Rightarrow> nat list"})
    val c2 = Const ("Set.insert", @{typ "nat \<Rightarrow> nat set \<Rightarrow> nat set"})
    val c1' = Const ("List.list.Cons", @{typ "nat list \<Rightarrow> nat list list \<Rightarrow> nat list list"})
    val c2' = Const ("Set.insert", @{typ "nat set \<Rightarrow> nat set set \<Rightarrow> nat set set"})
    val c3 = Const ("List.list.Cons", @{typ "nat set set \<Rightarrow> nat set set list \<Rightarrow> nat set set list"})
    val c3' = Const ("List.list.Cons", @{typ "nat list list \<Rightarrow> nat list list list \<Rightarrow> nat list list list"})
    val sub1 = (c2, c1)
    val sub2 = (c2', c1')
    val sub3 = (c3, c3')
    val sub4 = (@{term "{}::nat set"}, @{term "[]::nat list"})
    val sub5 = (@{term "{}::nat set set"}, @{term "[]::nat list list"})
    val sub6 = (@{term "[]::nat set set list"}, @{term "[]::nat list list list"})
  in
    subst_free [sub1, sub2, sub3, sub4, sub5, sub6] F
  end;

(* Given a term that represents a family returns a theorem stating that it is obtained
by applying f_to_set_l to the corresponding list of list representation. 
E.g. f_to_set_thm @{term "{{1}, {1, 2}}::nat set set"} returns
     "{{1}, {1, 2}} = f_to_set_l [[1], [1, 2]]": thm
*)

fun f_to_set_l_thm F =
let
    fun mk_f_to_set_l_thm F =
    let 
       val term_eq = @{term "HOL.eq :: nat set set \<Rightarrow>  nat set set \<Rightarrow> bool"}
       val term_f_to_set = @{term "f_to_set_l :: nat list list \<Rightarrow> nat set set"}
    in
       HOLogic.mk_Trueprop (list_comb (term_eq, [F, term_f_to_set $ sets_to_lists F]))
    end
in
  Goal.prove @{context} [] [] (mk_f_to_set_l_thm F) (fn _ => (simp_tac @{context} 1))
end

(* Similar as the previous one, except f_to_set_l is applied to every family i the given
  list.
  E.g. list_f_to_set_thm @{term "[{{1}, {2}}, {{1, 2}, {1, 3}}]::nat set set list"} returns
   "[{{1}, {2}}, {{1, 2}, {1, 3}}] = map f_to_set_l [[[1], [2]], [[1, 2], [1, 3]]]" : thm
*)

fun list_f_to_set_l_thm F =
let
  fun mk_list_f_to_set_l_thm FF =
  let
     val term_eq = @{term "HOL.eq :: nat set set list \<Rightarrow>  nat set set list \<Rightarrow> bool"}
     val term_map_f_to_set_l = @{term "map f_to_set_l :: nat list list list \<Rightarrow> nat set set list"}
  in
     HOLogic.mk_Trueprop (list_comb (term_eq, [FF, term_map_f_to_set_l $ sets_to_lists FF]))
  end
in
  Goal.prove @{context} [] [] (mk_list_f_to_set_l_thm F) (fn _ => (simp_tac @{context} 1))
end

(* Converts a term that represents a family (set of sets) to a corresponding list of nats. *)

fun sets_to_nats t = 
  let
    val cons_list = Const ("List.list.Cons", @{typ "nat \<Rightarrow> nat list \<Rightarrow> nat list"})
    val insert_list = Const ("Set.insert", @{typ "nat set \<Rightarrow> nat set set \<Rightarrow> nat set set"})
  in
  case t of insert_list $ S $ L => 
               cons_list $ (@{term "list2nat"} $ sets_to_lists S) $ sets_to_nats L 
          | @{term "{}::nat set set"} => @{term "[]:: nat list"}
          | _ => raise Fail ("pattern not matched")
  end

(* Given a term that represents a family returns a theorem stating that it is obtained
by applying f_to_set_n to the corresponding list of nats representation. 
E.g. f_to_set_n_thm @{term "{{1}, {1, 2}}::nat set set"} returns
     "{{1}, {1, 2}} = f_to_set_n [list2nat [1], list2nat [1, 2]]": thm
*)

fun f_to_set_n_thm F =
let
    fun mk_f_to_set_n_thm F =
    let 
       val term_eq = @{term "HOL.eq :: nat set set \<Rightarrow>  nat set set \<Rightarrow> bool"}
       val term_f_to_set = @{term "f_to_set_n :: nat list  \<Rightarrow> nat set set"}
    in
       HOLogic.mk_Trueprop (list_comb (term_eq, [F, term_f_to_set $ sets_to_nats F]))
    end
    val ss = put_simpset HOL_ss @{context} addsimps 
              ([@{thm nat2list_list2nat}, @{thm SetImpl_nats.f_to_set_def}] @
               @{thms List.list.map} @ 
               [@{thm comp_apply}] @ 
               @{thms distinct.simps} @
               @{thms List.list.set} @
               [@{thm empty_iff}, @{thm insert_iff}] @
               @{thms eq_numeral_simps} @ [@{thm zero_neq_one}] @
               @{thms Num.num.distinct} @ @{thms num.inject} @
               [@{thm sorted.Nil}, @{thm sorted_single}, @{thm sorted_many}] @
               @{thms le_numeral_simps} @ @{thms le_num_simps} @ @{thms less_num_simps} @ [@{thm le0}])
in
  Goal.prove @{context} [] [] (mk_f_to_set_n_thm F) (fn _ => (asm_full_simp_tac ss 1))
end
*}

end
