(* ************************************************************************** *)
section{* Union closed families *}
(* ************************************************************************** *)

theory UnionClosed
imports Main
begin

(* ************************************************************************** *)
subsection{* Union closed families *}
(* ************************************************************************** *)

definition sum_family (infixl "\<uplus>" 100) where
  "A \<uplus> B = (\<lambda> (a, b). a \<union> b) ` {(a, b) . a \<in> A \<and> b \<in> B}"

definition union_closed :: "'a set set \<Rightarrow> bool" where
  "union_closed F \<equiv> \<forall> A B. A \<in> F \<and> B \<in> F \<longrightarrow> A \<union> B \<in> F"

lemma "union_closed F \<longleftrightarrow> F \<uplus> F = F"
unfolding union_closed_def sum_family_def
by auto
  
abbreviation finite_union_closed where
  "finite_union_closed F \<equiv> union_closed F \<and> finite (\<Union> F)"

lemma union_closed_n:
  assumes "finite F'" and "F' \<noteq> {}" and "F' \<subseteq> F"
  assumes "union_closed F"
  shows "\<Union> F' \<in> F"
using assms
proof (induct F')
  case empty
  thus ?case
    by simp
next
  case (insert a F'')
  thus ?case
    using insert
    by (cases "F'' = {}") (auto simp add: union_closed_def)
qed

lemma union_closed_insert:
  assumes "union_closed F" and "\<forall> B \<in> F. A \<union> B \<in> F \<union> {A}"
  shows "union_closed (F \<union> {A})"
using assms
unfolding union_closed_def
by (auto simp add: Un_commute)


(* ************************************************************************** *)
subsubsection{* Closure -- minimal union closed family containing F *}
(* ************************************************************************** *)

definition closure where
  "closure F = Union ` (Pow F - {{}})"

lemma closure_closed: 
  "union_closed (closure F)"
unfolding union_closed_def
proof ((rule allI)+ , rule impI, erule conjE)
  fix A B :: "'a set"
  assume "A \<in> closure F" "B \<in> closure F"
  then obtain xa xb where
    "A = \<Union> xa" "xa \<in> Pow F - {{}}"
    "B = \<Union> xb" "xb \<in> Pow F - {{}}"
    by (auto simp add: closure_def)
  have "\<Union>xa \<union> \<Union>xb = \<Union> (xa \<union> xb)"
    by auto
  moreover
  have "xa \<union> xb \<in> Pow F - {{}}"
    using `xa \<in> Pow F - {{}}` `xb \<in> Pow F - {{}}`
    by auto
  ultimately
  show "A \<union> B \<in> closure F"
    using `A = \<Union> xa` `B = \<Union> xb`
    unfolding closure_def
    by blast
qed

lemma closure_min_closed:
  assumes "finite F" and "F \<subseteq> F'" and "union_closed F'"
  shows "closure F \<subseteq> F'"
proof
  fix x
  assume "x \<in> closure F"
  then obtain S where "x = \<Union> S" "S \<in> Pow F - {{}}"
    unfolding closure_def
    by auto
  thus "x \<in> F'"
    using union_closed_n[of S F'] `union_closed F'` `finite F` `F \<subseteq> F'`
    using finite_subset
    by auto
qed

lemma closure_union_closed_id:
  assumes "finite F" and "union_closed F"
  shows "closure F = F"
using assms closure_min_closed[of F F]
by (auto simp add: closure_def)

lemma [simp]:
  shows "\<Union> (closure F) = \<Union> F"
unfolding closure_def
by auto

lemma finite_closure:
  assumes "finite (\<Union> F)"
  shows "finite (\<Union> (closure F))"
using assms
by simp

lemma closure_mono:
  assumes "F \<subseteq> F'"
  shows "closure F \<subseteq> closure F'"
using assms
unfolding closure_def
by auto
  
text{* Iterative closure: insert set to closed family and close. *}

abbreviation insert_and_close :: "'a set \<Rightarrow> 'a set set \<Rightarrow> 'a set set" where
  "insert_and_close A F \<equiv> F \<union> {A} \<union> (op \<union> A) ` F"

lemma union_closed_insert_and_close:
  assumes "union_closed F"
  shows "union_closed (insert_and_close A F)"
using assms
unfolding union_closed_def
by auto

lemma  closure_insert:
  assumes "finite F"
  shows "closure (F \<union> {A}) = insert_and_close A (closure F)" (is "?lhs = ?rhs")
proof
  show "?lhs \<subseteq> ?rhs"
    using assms union_closed_insert_and_close[of "closure F" A]
    using closure_closed[of F]
    using closure_min_closed[of "F \<union> {A}" ?rhs]
    by (auto simp add: closure_def)
next
  show "?rhs \<subseteq> ?lhs"
  proof
    fix x
    assume "x \<in> ?rhs"
    hence "x \<in> closure F \<union> {A} \<or> x \<in> op \<union> A ` (closure F)"
      by simp
    thus "x \<in> ?lhs"
    proof
      assume "x \<in> closure F \<union> {A}"
      thus "x \<in> closure (F \<union> {A})"
        by (auto simp add: closure_def)
    next
      assume "x \<in> op \<union> A ` (closure F)"
      then obtain y where "x = y \<union> A" "y \<in> closure F"
        by auto
      then obtain k where "k \<in> Pow F" "y = \<Union> k" "k \<noteq> {}"
        unfolding closure_def
        by auto

      show ?thesis
        unfolding closure_def
        apply (rule rev_image_eqI[of "k \<union> {A}"])
        using `x = y \<union> A` `y = \<Union> k` `k \<noteq> {}` `k \<in> Pow F`
        by auto
    qed
  qed
qed

lemma closure_subset:
  shows "F \<subseteq> closure F"
unfolding closure_def
by auto

lemma closure_empty: 
  shows "{} \<in> closure F \<longleftrightarrow> {} \<in> F"
using closure_subset[of F]
unfolding closure_def
by force

lemma insert_and_close_empty:
  shows "insert_and_close {} F = F \<union> {{}}" 
by auto

lemma closure_remove_empty:
  assumes "finite F"
  shows "closure (F - {{}}) = closure F - {{}}"
proof (cases "{} \<in> F")
  case True
  hence "F = (F - {{}}) \<union> {{}}"
    by auto
  have "closure F = closure (F - {{}}) \<union> {{}}"
    apply (subst `F = (F - {{}}) \<union> {{}}`)
    apply (subst closure_insert[of "F - {{}}" "{}"])
    apply (simp add: assms)
    apply (subst insert_and_close_empty)
    by simp
  thus ?thesis
    using True closure_empty[of "F - {{}}"]
    by simp
next
  case False
  thus ?thesis
    using closure_empty[of F]
    by simp
qed

lemma insert_and_close_closure:
  assumes "finite F" and "union_closed F"
  shows "closure (F \<union> {A}) = insert_and_close A F" (is "?lhs = ?rhs")
using assms
using closure_insert[of F A]
using closure_union_closed_id[of F]
by simp

(* ************************************************************************** *)
subsection{* Union closed families closed to unions with an additional family *}
(* ************************************************************************** *)

definition union_closed_additional where 
  [simp]: "union_closed_additional F Fc \<equiv>
           union_closed F \<and> (\<forall> A \<in> F. (op \<union> A) ` Fc \<subseteq> F)"

lemma
  shows "union_closed_additional F Fc \<longleftrightarrow> F \<uplus> F = F \<and> F \<uplus> Fc \<subseteq> F"
unfolding union_closed_additional_def union_closed_def sum_family_def
by auto force
  
lemma union_closed_additional: 
  shows "(\<forall> A \<in> F. \<forall> Ai \<in> Fc. A \<union> Ai \<in> F) \<longleftrightarrow>
         (\<forall> A \<in> F. (op \<union> A) ` Fc \<subseteq> F)"
by auto

abbreviation insert_and_close_additional where
  "insert_and_close_additional A F Fc \<equiv> F \<union> {A} \<union> ((op \<union> A) ` F) \<union> ((op \<union> A) ` Fc)"

lemma closure_additional_set:
  assumes "finite F" and "A \<in> closure F" and
  "A' \<in> F'" and "union_closed F'" and "\<forall> A' \<in> F'. op \<union> A' ` F \<subseteq> F'"
  shows "A \<union> A' \<in> F'"
proof-
  have "op \<union> A' ` F \<subseteq> F'"
    using `\<forall>A'\<in>F'. image (op \<union> A') F \<subseteq> F'` `A' \<in> F'`
    by simp
  obtain X where "A = \<Union> X" "X \<subseteq> F" "X \<noteq> {}"
    using `A \<in> closure F`
    unfolding closure_def
    by auto
  hence "finite X"
    using `finite F`
    by (auto simp add: finite_subset)
  
  have "A \<union> A' = \<Union> ((op \<union> A') ` X)"
    using `A = \<Union> X` `X \<noteq> {}`
    by auto

  show ?thesis
  proof (subst `A \<union> A' = \<Union> image (op \<union> A') X`, rule union_closed_n)
    show "finite (op \<union> A' ` X)"
      using `finite X`
      by simp
  next
    show "op \<union> A' ` X \<noteq> {}"
      using `X \<noteq> {}`
      by simp
  next
    show "op \<union> A' ` X \<subseteq> F'"
      using `image (op \<union> A') F \<subseteq> F'`
      using `X \<subseteq> F`
      by auto
  next
    show "union_closed F'"
      by fact
  qed
qed

lemma union_closed_additional_closure:
  assumes "finite Fc" and "union_closed_additional F Fc"
  shows "union_closed_additional F (closure Fc)"
unfolding union_closed_additional_def
proof (safe)
  fix x y
  assume "x \<in> F" "y \<in> closure Fc"
  thus "x \<union> y \<in> F"
    using closure_additional_set[of Fc y x F]
    using `union_closed_additional F Fc` `finite Fc`
    by (auto simp add: Un_commute)
next
  show "union_closed F"
    using `union_closed_additional F Fc`
    by simp
qed

(* ************************************************************************** *)
subsubsection{* Union closed extensions *}
(* ************************************************************************** *)

definition union_closed_extensions where
  [simp]: "union_closed_extensions Fc \<equiv>
           {F. F \<subseteq> Pow (\<Union> Fc) \<and> union_closed_additional F Fc}"

syntax
  "_union_closed_extensions" :: "'a set set \<Rightarrow> 'a set set set"    ("\<lbrace>_\<rbrace>")
translations 
  "\<lbrace>Fc\<rbrace>" == "CONST union_closed_extensions Fc"

end
