theory UnionClosedImpl
imports UnionClosed FamilyImpl
begin

(* ************************************************************************ *)
subsection{* Abstract representation of union closed families *}
(* ************************************************************************ *)

text{* Sets that support empty set, unions and powerset. *}

locale SetUnionImpl = SetImpl to_set for to_set :: "'s \<Rightarrow> 'a set" +
  fixes empty :: "'s"
  assumes empty_set: "to_set empty = {}"
  assumes empty_inv: "inv empty"

  fixes union :: "'s \<Rightarrow> 's \<Rightarrow> 's" (infixl "\<squnion>" 100)
  assumes union_set: "to_set (s1 \<squnion> s2) = to_set s1 \<union> to_set s2"
  assumes union_inv: "\<lbrakk>inv s1; inv s2\<rbrakk> \<Longrightarrow> inv (s1 \<squnion> s2)"

  fixes pow :: "'s \<Rightarrow> 's list"
  assumes pow_set: "f_to_set (pow s) = Pow (to_set s)"
  assumes pow_inv: "inv s \<Longrightarrow> \<forall> A \<in> set (pow s). inv A"
  assumes pow_distinct: "inv s \<Longrightarrow> distinct (pow s)"
begin

(* -------------------------------------------------------------------------- *)

lemma subset_in_pow:
  assumes "inv A" and "inv S" and "to_set A \<subseteq> to_set S"
  shows "A \<in> set (pow S)"
proof-
  have "to_set A \<in> f_to_set (pow S)"
    using `to_set A \<subseteq> to_set S` pow_set[of S]
    by simp
  thus ?thesis
    using `inv A` `inv S`
    using pow_inv to_set_inj
    by auto
qed

(* -------------------------------------------------------------------------- *)

definition Union :: "'s list \<Rightarrow> 's" where
"Union F = foldl union empty F"

lemma Union_set:
  shows "to_set (Union F) = \<Union> (f_to_set F)"
by (induct F rule: rev_induct) (auto simp add: Union_def empty_set union_set)

lemma Union_inv:
  shows "\<forall> A \<in> set F. inv A \<Longrightarrow> inv (Union F)"
unfolding Union_def
by (induct F rule: rev_induct) (auto simp add: empty_inv union_inv)

(* -------------------------------------------------------------------------- *)

abbreviation union_with_all :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  "union_with_all A F \<equiv> (map (op \<squnion> A) F)"

lemma union_with_all_set:
  shows "map to_set (union_with_all A F) = map (op \<union> (to_set A) \<circ> to_set) F"
by (simp add: comp_def union_set)

lemma union_with_all_set':
  shows "(to_set ` (op \<squnion> A ` set F)) = (op \<union> (to_set A) ` to_set ` set F)"
proof-
  have "set (map to_set (union_with_all A F)) = set (map (op \<union> (to_set A) \<circ> to_set) F)"
    using union_with_all_set[of A F]
    by (auto simp add: comp_def)
  thus ?thesis
    by (simp add: image_comp[symmetric])
qed

(* -------------------------------------------------------------------------- *)

definition insert_set :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_set A F \<equiv> (if A \<in> set F then F else A # F)"

lemma insert_set_set [simp]: 
  shows "set (insert_set A F) = {A} \<union> set F"
by (auto simp add: insert_set_def)

definition insert_sets :: "'s list \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_sets l F = foldl (\<lambda> F' a. insert_set a F') F l"

lemma insert_sets_set [simp]:
  shows "set (insert_sets l F) = set l \<union> set F"
unfolding insert_sets_def
by (induct l rule: rev_induct) auto

lemma insert_sets_remdups:
  shows "distinct F \<Longrightarrow> insert_sets l F = remdups (rev l @ F)"
unfolding insert_sets_def
by (induct l rule: rev_induct) (auto simp add: insert_set_def distinct_remdups_id)

lemma insert_sets_distinct:
  shows "distinct F \<Longrightarrow> distinct (insert_sets l F)"
using insert_sets_remdups[of F l]
by auto

(* -------------------------------------------------------------------------- *)

definition insert_and_close :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  [simp]: "insert_and_close A F \<equiv> insert_sets ([A] @ union_with_all A F) F"

lemma insert_and_close_set:
  assumes "distinct F"
  shows "f_to_set (insert_and_close h F) = UnionClosed.insert_and_close (to_set h) (f_to_set F)"
  using assms
  using insert_sets_remdups
  using union_with_all_set'[of h F, THEN sym]
  by auto

lemma  insert_and_close_inv:
  assumes "\<forall> l \<in> set F. inv l" and "inv h" and "distinct F"
  shows "\<forall> l \<in> set (insert_and_close h F). inv l"
using assms
by (auto simp add: insert_sets_remdups union_inv)

lemma  insert_and_close_subset:
  assumes "\<forall> l \<in> set F. to_set l \<subseteq> S" and "to_set h \<subseteq> S" and "distinct F"
  shows "\<forall> l \<in> set (insert_and_close h F). to_set l \<subseteq> S"
proof (safe)
  fix x l
  assume "l \<in> set (insert_and_close h F)" "x \<in> to_set l"
  hence "x \<in> \<Union> (f_to_set (insert_and_close h F))"
    by auto
  hence "x \<in> \<Union> (UnionClosed.insert_and_close (to_set h) (f_to_set F))"
    using `distinct F`
    apply (subst insert_and_close_set[of F h, THEN sym])
    by simp_all
  thus "x \<in> S"
    using assms
    by (auto split: if_split_asm)
qed

(* -------------------------------------------------------------------------- *)

definition insert_and_close_additional :: "'s \<Rightarrow> 's list \<Rightarrow> 's list \<Rightarrow> 's list" where
  [simp]: "insert_and_close_additional A F Fc \<equiv> insert_sets ([A] @ union_with_all A F @ union_with_all A Fc) F"

lemma insert_and_close_additional_set:
  assumes "distinct F"
  shows "f_to_set (insert_and_close_additional A F Fc) = UnionClosed.insert_and_close_additional (to_set A) (f_to_set F) (f_to_set Fc)"
  using assms
  using insert_sets_remdups
  using union_with_all_set'[of A F, THEN sym]
  using union_with_all_set'[of A Fc, THEN sym]
  by auto

lemma  insert_and_close_additional_subset:
  assumes "to_set A \<subseteq> S" and
          "\<forall> l \<in> set F. to_set l \<subseteq> S" and
          "\<forall> l \<in> set Fc. to_set l \<subseteq> S"
  assumes "distinct F"
  shows "\<forall> l \<in> set (insert_and_close_additional A F Fc). to_set l \<subseteq> S"
proof (safe)
  fix x l
  assume "l \<in> set (insert_and_close_additional A F Fc)" "x \<in> to_set l"
  hence "x \<in> \<Union> (f_to_set (insert_and_close_additional A F Fc))"
    by auto
  hence "x \<in> \<Union> (UnionClosed.insert_and_close_additional (to_set A) (f_to_set F) (f_to_set Fc))"
    using `distinct F`
    apply (subst insert_and_close_additional_set[of F A Fc, THEN sym])
    by simp_all
  thus "x \<in> S"
    using assms
    by (auto split: if_split_asm)
qed

lemma  insert_and_close_additional_inv:
  assumes "inv A" and "\<forall> l \<in> set F. inv l" and "\<forall> l \<in> set Fc. inv l" 
  assumes "distinct F"
  shows "\<forall> l \<in> set (insert_and_close_additional A F Fc). inv l"
using assms
by (auto simp add: insert_sets_remdups union_inv)

lemma insert_and_close_additional_cong:
  assumes "set F = set F'" and "distinct F" and "distinct F'"
  shows "set (insert_and_close_additional A F Fc) =
         set (insert_and_close_additional A F' Fc)"
  using assms
  by (simp add: insert_sets_remdups)

(* -------------------------------------------------------------------------- *)

definition close :: "'s list \<Rightarrow> 's list" where
  "close F = foldl (\<lambda> F A. insert_and_close A F) [] F"

lemma close_snoc:
  shows "close (F @ [A]) = insert_and_close A (close F)"
unfolding close_def
by simp

lemma close_distinct: 
  shows "distinct (close A)"
unfolding close_def
by (induct A rule: rev_induct) (auto simp add: insert_sets_distinct)

lemma close_set: 
  shows "f_to_set (close A) = UnionClosed.closure (f_to_set A)"
proof (induct A rule: rev_induct)
  case (snoc l a)
  let ?L = "\<lambda> x. foldl (\<lambda>F A. local.insert_and_close A F) [] x"
  show ?case
    using close_distinct[of a]
    using insert_and_close_set[of "?L a"]
    using closure_insert[of "f_to_set a" "to_set l"]
    using snoc(1)
    unfolding close_def
    by simp
next
  case Nil
  thus ?case
    by (simp add: close_def closure_def)
qed

lemma close_completeness:
  assumes "inv A" and "\<forall> a \<in> set F. inv a"
  assumes "to_set A \<in> UnionClosed.closure (f_to_set F)"
  shows "A \<in> set (close F)"
using assms
proof (induct F arbitrary: A rule: rev_induct)
  case Nil
  thus ?case
    by (simp add: closure_def)
next
  case (snoc a l)
  from `to_set A \<in> closure (f_to_set (l @ [a]))`
  have "to_set A = to_set a \<or> to_set A \<in> closure (f_to_set l) \<or> to_set A \<in> op \<union> (to_set a) ` closure (to_set ` set l)"
    using closure_insert[of "f_to_set l" "to_set a"]
    by simp
  thus ?case
  proof
    assume "to_set A = to_set a"
    thus ?thesis
      using snoc(2) snoc(3)
      by (simp add: to_set_inj close_snoc)
  next
    assume "to_set A \<in> closure (f_to_set l) \<or>
            to_set A \<in> op \<union> (to_set a) ` closure (to_set ` set l)"
    thus ?thesis
    proof
      assume "to_set A \<in> closure (f_to_set l)"
      thus ?thesis
        using snoc(1) snoc(2) snoc(3)
        by (simp add: close_snoc)
    next
      assume "to_set A \<in> op \<union> (to_set a) ` closure (to_set ` set l)"
      then obtain X where "X \<in> closure (f_to_set l)" "to_set A = to_set a \<union> X"
        by auto
      hence "finite X"
        using finiteUn_iff[of "closure (f_to_set l)"]
        using finite_closure[of "f_to_set l"]
        by (auto simp add: to_set_finite)
      then obtain Y where "X = to_set Y" "inv Y"
        using to_set_ex[of X] 
        by auto
      hence "Y \<in> set (close l)"
        using snoc(1)[of Y] `X \<in> closure (f_to_set l)` snoc(3)
        by simp
      hence "A = a \<squnion> Y"
        using `to_set A = to_set a \<union> X` `X = to_set Y`
        using snoc(2) snoc(3) `inv Y`
        by (auto simp add: union_set union_inv to_set_inj)
      hence "A \<in> op \<squnion> a ` set (close l)"
        using `Y \<in> set (close l)`
        by simp
      thus ?thesis
        by (simp add: close_snoc)
    qed
  qed
qed

lemma close_inv:
  assumes "\<forall> l \<in> set F. inv l"
  shows "\<forall> l \<in> set (close F). inv l"
using assms
proof (induct F rule: rev_induct)
  case (snoc a l)
  thus ?case
    using close_distinct[of l]
    by (auto simp add: close_def insert_sets_remdups union_inv)
qed (simp add: close_def)

lemma close_subset:
  assumes "\<forall> l \<in> set F. to_set l \<subseteq> S"
  shows "\<forall> l \<in> set (close F). to_set l \<subseteq> S"
using assms
proof (induct F rule: rev_induct)
  case (snoc a l)
  thus ?case
    using insert_and_close_subset[of "close l" S a]
    using close_distinct[of l]
    unfolding close_def
    by simp
qed (simp add: close_def)

definition close_and_insert_empty :: "'s list \<Rightarrow> 's list" where
  "close_and_insert_empty F \<equiv>
     let X = close F
      in (if empty \<in> set X then X else empty # X)"

lemma close_and_insert_empty_set':
  shows "set (close_and_insert_empty F) = set (close F) \<union> {empty}"
by (auto simp add: Let_def close_and_insert_empty_def)

lemma close_and_insert_empty_set:
  shows "f_to_set (close_and_insert_empty F) =
         (UnionClosed.closure (f_to_set F)) \<union> {{}}"
proof-
  have "to_set ` set (close F) = closure (to_set ` set F)"
    by (metis f_to_set_def close_set image_set)
  thus ?thesis
    unfolding close_and_insert_empty_def
    by (auto simp add: empty_set Let_def) (metis empty_set image_eqI)
qed

primrec union_closed :: "'s list \<Rightarrow> bool" where
  "union_closed [] = True"
| "union_closed (h # t) = (list_all (\<lambda> x. h \<squnion> x \<in> set (h # t)) t \<and> union_closed t)"

lemma union_closed_set:
  shows "union_closed F \<Longrightarrow> UnionClosed.union_closed (f_to_set F)"
proof (induct F)
  case Nil
  thus ?case
    by (simp add: UnionClosed.union_closed_def)
next
  case (Cons h t)
  thus ?case
    using union_closed_insert[of "f_to_set t" "to_set h"]
    by (auto simp add: list_all_iff) (metis imageI union_set)
qed

primrec union_closed_additional' :: "'s list \<Rightarrow> 's list \<Rightarrow> bool" where
  "union_closed_additional' F [] = True"
| "union_closed_additional' F (h # t)  = (list_all (\<lambda> x. h \<squnion> x \<in> set F) F \<and> union_closed_additional' F t)"

lemma union_closed_additional'_set: 
  assumes "union_closed_additional' F Fc"
  shows "\<forall> A \<in> f_to_set Fc. \<forall> B \<in> (f_to_set F). A \<union> B \<in> (f_to_set F)"
using assms
proof (induct Fc)
  case Nil
  thus ?case
    by simp
next
  case (Cons h t)
  thus ?case
    by (simp add: list_all_iff) (metis union_set imageI)
qed

definition union_closed_additional :: "'s list \<Rightarrow> 's list \<Rightarrow> bool" where
  "union_closed_additional F Fc \<longleftrightarrow> union_closed F \<and> union_closed_additional' F Fc"

lemma union_closed_additional_set:
  assumes "union_closed_additional F Fc"
  shows "UnionClosed.union_closed_additional (f_to_set F) (f_to_set Fc)"
using assms
unfolding union_closed_additional_def
using union_closed_set[of F] union_closed_additional'_set[of F Fc]
by (auto) (metis sup_commute)

end (* Locale *)

(* ************************************************************************ *)

(* ----------------------------------------------------------------------- *)
subsubsection{* Implementation by sorted and distinct lists *}
(* ----------------------------------------------------------------------- *)

text{* Power set of a list *}

primrec Pow_l :: "'a list \<Rightarrow> 'a list list" where
  "Pow_l [] = [[]]"
| "Pow_l (h # t) = 
     (let X = Pow_l t
       in X @ map (op # h) X)"

lemma Pow_l_set:
  shows "set (map set (Pow_l X)) = Pow (set X)"
proof (induct X)
  case Nil
  thus ?case
    by simp
next
  case (Cons a l')
  thus ?case
    using Cons[THEN sym]
    by (auto simp add: Let_def comp_def Pow_insert)
qed

lemma Pow_l_distinct:
  shows "distinct A \<Longrightarrow> distinct (Pow_l A)"
proof (induct A)
  case Nil
  thus ?case
    by simp
next
  case (Cons a A)
  thus ?case
  proof (auto simp add: Let_def distinct_map inj_on_def)
    fix x
    assume "a \<notin> set A"
    assume "a # x \<in> set (Pow_l A)"
    hence "set (a # x) \<in> Pow (set A)"
      using Pow_l_set[of A]
      by (auto simp del: set_simps)
    thus False
      using `a \<notin> set A`
      by simp
  qed
qed

lemma Pow_l_elems_distinct:
  assumes "distinct A" and "x \<in> set (Pow_l A)"
  shows "distinct x"
using assms
proof (induct A arbitrary: x)
  case Nil
  thus ?case
    by simp
next
  case (Cons h t)
  thus ?case
  proof (cases "x \<in> set (Pow_l t)")
    case True
    thus ?thesis
      using Cons
      by simp
  next
    case False
    then obtain y where "y \<in> set (Pow_l t)" "x = h # y"
      using Cons(3)
      by (auto simp add: Let_def)
    thus ?thesis
      using Cons(2) Cons(1)
      using Pow_l_set[of t]
      by auto
  qed
qed

lemma Pow_l_elems_sorted:
  assumes "sorted A" and "x \<in> set (Pow_l A)"
  shows "sorted x"
using assms
proof (induct A arbitrary: x)
  case Nil
  thus ?case
    by simp
next
  case (Cons t h)
  show ?case
  proof (cases "x \<in> set (Pow_l t)")
    case True
    thus ?thesis
      using Cons
      by simp
  next
    case False
    then obtain y where "y \<in> set (Pow_l t)" "x = h # y"
      using Cons(4)
      by (auto simp add: Let_def)
    thus ?thesis
      using Cons(1) Cons(2) Cons(3)
      using Pow_l_set[of t]
      using sorted_Cons[of h y]
      by auto
  qed
qed

lemma Pow_l_no_perms:
  assumes "distinct A" and "sorted A"
  assumes "x \<in> set (Pow_l A)" and "y \<in> set (Pow_l A)"
  assumes "set x = set y"
  shows "x = y"
using assms
using Pow_l_elems_sorted[of A x]
using Pow_l_elems_distinct[of A x]
using Pow_l_elems_sorted[of A y]
using Pow_l_elems_distinct[of A y]
by (simp add: sorted_distinct_set_unique)

text{* Merging two sorted lists *}

fun merge :: "'a::linorder list \<Rightarrow> 'a::linorder list \<Rightarrow> 'a::linorder list" (infixl "\<squnion>" 100) where
  "merge [] l = l"
| "merge l [] = l"
| "merge (h1 # t1) (h2 # t2) = 
     (if h1 < h2 then 
          h1 # merge t1 (h2 # t2)
      else if h1 > h2 then
          h2 # merge (h1 # t1) t2
      else
          h1 # merge t1 t2)"

lemma merge_set:
  shows "set (l1 \<squnion> l2) = set l1 \<union> set l2"
by (induct l1 l2 rule: merge.induct) auto

lemma merge_sorted:
  assumes "sorted l1" and "sorted l2"
  shows "sorted (l1 \<squnion> l2)"
using assms
by (induct l1 l2 rule: merge.induct) (auto simp add: sorted_Cons merge_set)

lemma merge_distinct:
  assumes "distinct l1" and "distinct l2" and "sorted l1" and "sorted l2"
  shows "distinct (l1 \<squnion> l2)"
using assms
by (induct l1 l2 rule: merge.induct) (auto simp add: sorted_Cons merge_set)

global_interpretation SetUnionImpl_lists: SetUnionImpl "\<lambda> (l::nat list). sorted l \<and> distinct l" set "[]" merge Pow_l
  defines
  close_l = "SetUnionImpl_lists.close" and
  insert_set_l = "SetUnionImpl_lists.insert_set" and
  insert_sets_l = "SetUnionImpl_lists.insert_sets" and
  insert_and_close_l = "SetUnionImpl_lists.insert_and_close" and
  Union_l = "SetUnionImpl_lists.Union" and
  close_insert_empty_l = "SetUnionImpl_lists.close_and_insert_empty" and
  union_closed_l = "SetUnionImpl_lists.union_closed" and
  union_closed_additional'_l = "SetUnionImpl_lists.union_closed_additional'" and
  union_closed_additional_l = "SetUnionImpl_lists.union_closed_additional"
proof (unfold_locales)
  fix s :: "nat list"
  show "SetImpl.f_to_set set (Pow_l s) = Pow (set s)"
    unfolding f_to_set_l_def[symmetric]
    unfolding SetImpl_lists.f_to_set_def[of "Pow_l s"]
    by (rule Pow_l_set)
qed (auto simp add: Pow_l_elems_sorted Pow_l_elems_distinct merge_set merge_sorted merge_distinct sorted_distinct_set_unique Pow_l_distinct)

(* ----------------------------------------------------------------------- *)
subsubsection{* Implementation by sets represented as natural numbers *}
(* ----------------------------------------------------------------------- *)

text{* For example, the family @{text "{{0, 1, 2}, {1, 2, 3}}"} is represented
as @{text "[7, 14]"}. *}

text{* Powerset *}
definition pow_n :: "nat \<Rightarrow> nat list" where
"pow_n n = map list2nat (Pow_l (nat2list n))"

text{* Union is obtained by bitwise disjunction -- this is a naive implementation *}

function bitor :: "nat \<Rightarrow> nat \<Rightarrow> nat" where
 "bitor x y = 
  (if x = 0 then y
   else if y = 0 then x else
   let (xd, xm) = Divides.divmod_nat x 2;
       (yd, ym) = Divides.divmod_nat y 2 in
   if (xm = 1 | ym = 1)
   then 1 + 2 * bitor xd yd
   else 2 * bitor xd yd)"
by pat_completeness auto
termination
by  (relation "measure (\<lambda> (x, _). x)", auto simp add: divmod_nat_div_mod)
declare bitor.simps[simp del]

lemma bitor_set:
  shows "set (nat2list (bitor x y)) = set (nat2list x) \<union> set (nat2list y)"
proof (induct x y rule: bitor.induct)
  case (1 x' y')
  show ?case
  proof (cases "x' = 0")
    case True
    thus ?thesis
      by (simp add: bitor.simps nat2list_def)
  next
    case False
    show ?thesis
    proof (cases "y' = 0")
      case True
      thus ?thesis
        using `x' \<noteq> 0`
        by (simp add: bitor.simps nat2list_def)
    next
      case False
      show ?thesis
      proof (cases "x' mod 2 \<noteq> Suc 0 \<and> y' mod 2 \<noteq> Suc 0")
        case True
        hence "bitor x' y' = 2 * bitor (x' div 2) (y' div 2)"
          using `x' \<noteq> 0` `y' \<noteq> 0`
          using bitor.simps[of x' y']
          by (simp add: split_def Let_def div_nat_def[THEN sym] mod_nat_def[THEN sym] split: if_split_asm)
        hence "nat2list (bitor x' y') = map (op + 1) (nat2list (bitor (x' div 2) (y' div 2)))"
          by (simp add: nat2list_even)
        moreover
        have "set (nat2list (bitor (x' div 2) (y' div 2))) = set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))"
          using 1(2)[of "Divides.divmod_nat x' 2" "x' div 2" "x' mod 2"
                        "Divides.divmod_nat y' 2" "y' div 2" "y' mod 2"]
          using `x' \<noteq> 0` `y' \<noteq> 0` True
          unfolding div_nat_def mod_nat_def
          by (simp add: divmod_nat_div_mod)
        moreover
        from True
        have "x' mod 2 = 0" "y' mod 2 = 0"
          by auto
        hence "x' = 2 * (x' div 2)" "y' = 2 * (y' div 2)"
          by  auto
        hence "nat2list x' = map (op +1) (nat2list (x' div 2))"
              "nat2list y' = map (op +1) (nat2list (y' div 2))"
          using nat2list_even[of "x' div 2"] nat2list_even[of "y' div 2"]
          by auto
        ultimately
        show ?thesis
          by auto
      next
        case False
        hence "bitor x' y' = 2 * bitor (x' div 2) (y' div 2) + 1"
          using `x' \<noteq> 0` `y' \<noteq> 0`
          using bitor.simps[of x' y']
          by (simp add: split_def Let_def div_nat_def[THEN sym] mod_nat_def[THEN sym] split: if_split_asm)
        hence "nat2list (bitor x' y') = 0 # map (op + 1) (nat2list (bitor (x' div 2) (y' div 2)))"
          using nat2list_odd
          by simp
        moreover
        have "set (nat2list (bitor (x' div 2) (y' div 2))) = set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))"
          using 1(1)[of "Divides.divmod_nat x' 2" "x' div 2" "x' mod 2"
                        "Divides.divmod_nat y' 2" "y' div 2" "y' mod 2"]
          using `x' \<noteq> 0` `y' \<noteq> 0` False
          by simp (simp add: divmod_nat_div_mod)
        moreover
        from False
        have *: "x' mod 2 = 0 \<and> y' mod 2 = 1 \<or> x' mod 2 = 1 \<and> y' mod 2 = 0 \<or> x' mod 2 = 1 \<and> y' mod 2 = 1"
          by auto
        have "set (nat2list x') \<union> set (nat2list y') = {0} \<union> (op +1 ` (set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))))"
        proof-
          {
            assume "x' mod 2 = 0 \<and> y' mod 2 = 1"
            hence "x' = 2 * (x' div 2) \<and> y' = 2 * (y' div 2) + 1"
              using mod_mult_div_eq[of y' 2, THEN sym] mod_mult_div_eq[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_even[of "x' div 2"] nat2list_odd[of "y' div 2"]
              by auto
          }
          moreover
          {
            assume "x' mod 2 = 1 \<and> y' mod 2 = 0"
            hence "x' = 2 * (x' div 2) + 1 \<and> y' = 2 * (y' div 2)"
              using mod_mult_div_eq[of y' 2, THEN sym] mod_mult_div_eq[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_even[of "y' div 2"] nat2list_odd[of "x' div 2"]
              by auto
          }
          moreover
          {
            assume "x' mod 2 = 1 \<and> y' mod 2 = 1"
            hence "x' = 2 * (x' div 2) + 1 \<and> y' = 2 * (y' div 2) + 1"
              using mod_mult_div_eq[of y' 2, THEN sym] mod_mult_div_eq[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_odd[of "x' div 2"] nat2list_odd[of "y' div 2"]
              by auto
          }
          ultimately
          show ?thesis
            using *
            by blast
        qed
        ultimately
        show ?thesis
          by simp
      qed
    qed
  qed
qed


global_interpretation SetUnionImpl_nats: SetUnionImpl "\<lambda> n. True" "set \<circ> nat2list" 0 bitor pow_n 
  defines
  close_n = "SetUnionImpl_nats.close" and
  insert_set_n = "SetUnionImpl_nats.insert_set" and
  insert_sets_n = "SetUnionImpl_nats.insert_sets" and
  insert_and_close_n = "SetUnionImpl_nats.insert_and_close" and
  Union_n = "SetUnionImpl_nats.Union" and
  close_and_insert_empty_n = "SetUnionImpl_nats.close_and_insert_empty" and
  union_closed_n = "SetUnionImpl_nats.union_closed" and
  union_closed_additional'_n = "SetUnionImpl_nats.union_closed_additional'" and
  union_closed_additional_n = "SetUnionImpl_nats.union_closed_additional"
proof (unfold_locales)
next
  show "(set \<circ> nat2list) 0 = {}"
    by (simp add: nat2list_def)
next
  fix s1 s2
  show "(set \<circ> nat2list) (bitor s1 s2) = (set \<circ> nat2list) s1 \<union> (set \<circ> nat2list) s2"
    using bitor_set
    by simp
next
  fix s
  show "SetImpl.f_to_set (set \<circ> nat2list) (pow_n s) = Pow ((set \<circ> nat2list) s)"
  proof-
    have "(set \<circ> nat2list \<circ> list2nat) ` set (Pow_l (nat2list s)) = set ` set (Pow_l (nat2list s))"
    proof (rule image_cong, simp)
      fix x
      assume "x \<in> set (Pow_l (nat2list s))"
      thus "(set \<circ> nat2list \<circ> list2nat) x = set x"
        using sorted_nat2list[of s] distinct_nat2list[of s]
        using nat2list_list2nat[of x] Pow_l_elems_sorted[of "nat2list s" x] Pow_l_elems_distinct[of "nat2list s" x]
        by simp
    qed
    thus ?thesis
      unfolding f_to_set_n_def[symmetric]
      unfolding SetImpl_nats.f_to_set_def
      unfolding pow_n_def
      using Pow_l_set[of "nat2list s"]
      by simp
  qed
next
  fix s
  show "distinct (pow_n s)"
    unfolding pow_n_def
    by (auto simp add: distinct_map distinct_nat2list Pow_l_distinct 
        inj_on_def inj_list2nat 
        Pow_l_elems_sorted[of "nat2list s"] sorted_nat2list
        Pow_l_elems_distinct[of "nat2list s"])
qed simp_all

end