theory Combinatorics
imports Main
  "~~/src/HOL/Library/List_lexord"
  "~~/src/HOL/Library/Permutation"
begin

(* ----------------------------------------------------------------- *)

fun combine_aux :: "'a list \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> 'a list list" where
 "combine_aux elems n k = 
     (if k = 0 then [[]] else
      if k = n then [elems] else
      (case elems of
          [] \<Rightarrow> []
      | (h # t) \<Rightarrow>
             (map (\<lambda> l'. h # l') (combine_aux t (n-(1::nat)) (k-(1::nat)))) @
                   combine_aux t (n-(1::nat)) k))"
declare combine_aux.simps[simp del]

definition combine :: "'a list \<Rightarrow> nat \<Rightarrow> 'a list list" where
  "combine elems k = combine_aux elems (length elems) k"

primrec interleave :: "'a \<Rightarrow> 'a list \<Rightarrow> 'a list list" where
  "interleave x [] = [[x]]"
| "interleave x (h # t) = (x # (h # t)) # (map (\<lambda> l. h # l) (interleave x t))"

text{* For example, @{lemma "interleave (1::nat) [2, 3, 4] = [[1, 2, 3, 4], [2, 1, 3, 4], [2, 3, 1, 4], [2, 3, 4, 1]]" by simp}. *}

primrec permute :: "'a list \<Rightarrow> 'a list list" where
  "permute [] = [[]]"
| "permute (h # t) = concat (map (\<lambda> l. interleave h l) (permute t))"

text{* For example, @{lemma "permute [1::nat, 2, 3] = [[1, 2, 3], [2, 1, 3], [2, 3, 1], [1, 3, 2], [3, 1, 2], [3, 2, 1]]" by simp}. *}

(* Implementation of variations using permutations and combinations *)  
definition variations' where 
  "variations' l k = concat (map permute (combine l k))"

(* Direct implementation of variations  *)  
fun variations_aux where
  "variations_aux len elems suffix = 
    (if len = (0::nat) then [suffix]
     else 
       let cs = (filter (\<lambda> x. x \<notin> set suffix) elems) in
         (concat (map (\<lambda> x. variations_aux (len-1) elems (x # suffix)) cs)) 
    )"
declare variations_aux.simps [simp del]

definition variations where
  "variations elems len = variations_aux len elems []"
  
(* variations where the first element must be less than others *)  
definition min_hd :: "nat list \<Rightarrow> bool" where
  "min_hd p \<longleftrightarrow> (\<forall> x \<in> set (tl p). hd p < x)"

lemma min_hd [iff]: "min_hd x \<longleftrightarrow> list_all (op < (hd x)) (tl x)"
unfolding min_hd_def
by (auto simp add: list_all_iff)
  
definition variations_min_hd where
  "variations_min_hd len n = concat (map (\<lambda> x. map (\<lambda> l. x # l) (variations [x+1..<n] (len-1))) [0..<n])"
  
(* ----------------------------------------------------------------- *)
  
lemma combine_aux_induct:
  assumes 
  "\<And> l n. P [[]] l n 0"
  "\<And> l n. 0 < n \<Longrightarrow> P [l] l n n" 
  "\<And> k n. \<lbrakk>0 < k; k \<noteq> n\<rbrakk> \<Longrightarrow> P [] [] n k"
  "\<And> h t n k. \<lbrakk>
     P (combine_aux t (n-(1::nat)) (k-(1::nat))) t (n-(1::nat)) (k-(1::nat));
     P (combine_aux t (n-(1::nat)) k) t (n-(1::nat)) k; 
     k > 0\<rbrakk> \<Longrightarrow> 
     P (map (op # h) (combine_aux t (n-(1::nat)) (k-(1::nat))) @
          combine_aux t (n-(1::nat)) k) (h # t) n k"
  shows "P (combine_aux l n k) l n k"
using assms
proof (induct l n k rule: combine_aux.induct)
  case (1 l n k)
  show ?case
  proof (cases "k=0")
    case True
    thus ?thesis
      using 1(3)
      by (simp add: combine_aux.simps)
  next
    case False
    show ?thesis
    proof (cases "k = n")
      case True
      thus ?thesis
      using `k \<noteq> 0`
        using 1(4)
        by (simp add: combine_aux.simps)
    next
      case False
      show ?thesis
      proof (cases "l = []")
        case True
        thus ?thesis
        using `k \<noteq> 0` `k \<noteq> n` 1(5)
          by (simp add: combine_aux.simps)
      next
        case False
        then obtain h t where "l = h # t" by (auto simp add: neq_Nil_conv)
        let ?l1 = "combine_aux t (n-(1::nat)) (k-(1::nat))"
        let ?l2 = "combine_aux t (n-(1::nat)) k"
        have "P ?l1 t (n-(1::nat)) (k-(1::nat))"
        using 1(1)[of h t] 1(3-6)  `k \<noteq> 0` `k \<noteq> n` `l = h # t`
          by auto
        moreover
        have "P ?l2 t (n-(1::nat)) k"
        using 1(2)[of h t] 1(3-6)  `k \<noteq> 0` `k \<noteq> n` `l = h # t`
          by auto
        ultimately
        show ?thesis
        using `l = h # t` `k \<noteq> 0` `k \<noteq> n`
          using combine_aux.simps[of l n k] 1(6)
          by simp
      qed
    qed
  qed
qed

lemma combine_aux_subset:
shows "\<forall> A. A \<in> set (combine_aux l n k) \<longrightarrow> set A \<subseteq> set l"
by (rule combine_aux_induct) (auto, force)

lemma combine_subset:
assumes "l \<in> set (combine elems len)"
shows "set l \<subseteq> set elems"
using assms
using combine_aux_subset[rule_format, of l elems "length elems" len]
unfolding combine_def
by simp

lemma combine_aux_distinct:
shows "\<forall> A. A \<in> set (combine_aux elems n k) \<and> distinct elems \<longrightarrow> distinct A"
proof (rule combine_aux_induct, simp, simp, simp, safe)
  fix h t n k and A :: "'a list"
  assume ind1: "\<forall>A. A \<in> set (combine_aux t (n - 1) (k - 1)) \<and> distinct t \<longrightarrow> distinct A" and
         ind2: "\<forall>A. A \<in> set (combine_aux t (n - 1) k) \<and> distinct t \<longrightarrow> distinct A" and
         "A \<in> set (map (op # h) (combine_aux t (n - 1) (k - 1)) @ combine_aux t (n - 1) k)" 
         "distinct (h # t)"
  have "(\<exists> A' \<in> set (combine_aux t (n - 1) (k - 1)). A = h # A') \<or> A \<in> set (combine_aux t (n - 1) k)"
    using `A \<in> set (map (op # h) (combine_aux t (n - 1) (k - 1)) @ combine_aux t (n - 1) k)`
    by auto
  thus "distinct A"
  proof
    assume "A \<in> set (combine_aux t (n - 1) k)"
    thus ?thesis
      using ind2 `distinct (h # t)`
      by simp
  next
    assume "\<exists>A'\<in>set (combine_aux t (n - 1) (k - 1)). A = h # A'"
    then obtain A' where "A' \<in> set (combine_aux t (n - 1) (k - 1))" "A = h # A'"
      by auto
    hence "distinct A'" "set A' \<subseteq> set t"
      using ind1 `distinct (h # t)` 
      using combine_aux_subset[of t "n-1" "k-1"]
      by auto
    thus ?thesis
      using  `distinct (h # t)` `A = h # A'`
      by auto
  qed
qed

lemma combine_distinct:
assumes "l \<in> set (combine elems len)" "distinct elems"
shows "distinct l"
using assms combine_aux_distinct
unfolding combine_def
by auto

lemma combine_aux_length:
shows "\<forall> A. A \<in> set (combine_aux elems n k) \<and> n = length elems \<longrightarrow> length A = k" 
proof (rule combine_aux_induct[where P = "\<lambda> c l n k. \<forall> A. A \<in> set c \<and> n = length l \<longrightarrow> length A = k"], simp, simp, simp, safe)
  fix h t n k and A :: "'a list"
  assume ind1: "\<forall>A. A \<in> set (combine_aux t (length (h # t) - 1) (k - 1)) \<and> length (h # t) - 1 = length t \<longrightarrow> length A = k - 1" and
         ind2: "\<forall>A. A \<in> set (combine_aux t (length (h # t) - 1) k) \<and> length (h # t) - 1 = length t \<longrightarrow> length A = k" and
         *: "A \<in> set (map (op # h) (combine_aux t (length (h # t) - 1) (k - 1)) @ combine_aux t (length (h # t) - 1) k)" and
         "k > 0"
  have "(\<exists> A' \<in> set (combine_aux t (length (h # t) - 1) (k - 1)). A = h # A') \<or> A \<in> set (combine_aux t (length (h # t) - 1) k)"
    using *
    by auto
  thus "length A = k"
  proof
    have "length (h # t) - 1 = length t"
      by simp
    moreover
    assume "A \<in> set (combine_aux t (length (h # t) - 1) k)"
    ultimately
    show ?thesis
      using ind2
      by blast
  next
    assume "\<exists>A'\<in>set (combine_aux t (length (h # t) - 1) (k - 1)). A = h # A'"
    then obtain A' where "A' \<in> set (combine_aux t (length (h # t) - 1) (k - 1))" "A = h # A'"
      by auto
    moreover
    have "length (h # t) - 1 = length t"
      by simp
    moreover
    ultimately
    have "length A' = k - 1"
      using ind1
      by blast
    thus ?thesis
      using `A = h # A'` `k > 0`
      by simp
  qed
qed

lemma combine_length:
assumes "l \<in> set (combine elems len)" 
shows "length l = len"
using assms combine_aux_length[of elems "length elems" len]
unfolding combine_def
by blast

lemma set_combine: 
assumes "l \<in> set (combine elems len)" "distinct elems"
shows "distinct l" "length l = len" "set l \<subseteq> set elems"
using assms combine_subset combine_distinct combine_length
by blast+

lemma combine_aux_preserves_order:
shows "\<forall> l i j i' j'. l \<in> set (combine_aux elems n k) \<and> distinct elems \<and> l ! i = elems ! i' \<and> l ! j = elems ! j' \<and> i < j \<and> j < length l \<and> i' < length elems \<and> j' < length elems \<longrightarrow> i' < j'"
proof (rule combine_aux_induct[where P="\<lambda> c e n k. \<forall> l i j i' j'. l \<in> set c \<and> distinct e \<and> l ! i = e ! i' \<and> l ! j = e ! j' \<and> i < j \<and> j < length l \<and> i' < length e \<and> j' < length e \<longrightarrow> i' < j'"], 
      simp, force simp add: nth_eq_iff_index_eq, simp, safe)
  fix h n k i j i' j' and t l ::"'a list"
  assume ind1: "\<forall>l i j i' j'. l \<in> set (combine_aux t (n - 1) (k - 1)) \<and> distinct t \<and> l ! i = t ! i' \<and> l ! j = t ! j' \<and> i < j \<and> j < length l \<and> i' < length t \<and> j' < length t \<longrightarrow>  i' < j'" and
         ind2: "\<forall>l i j i' j'. l \<in> set (combine_aux t (n - 1) k) \<and> distinct t \<and> l ! i = t ! i' \<and> l ! j = t ! j' \<and> i < j \<and> j < length l \<and> i' < length t \<and> j' < length t \<longrightarrow> i' < j'" and
               "0 < k" and
               "l \<in> set (map (op # h) (combine_aux t (n - 1) (k - 1)) @ combine_aux t (n - 1) k)" "distinct (h # t)" and
            *: "l ! i = (h # t) ! i'" "l ! j = (h # t) ! j'" "i < j" "j < length l" "i' < length (h # t)" "j' < length (h # t)"
       
  have "l \<in> set (combine_aux t (n - 1) k) \<or> (\<exists> l' \<in> set (combine_aux t (n - 1) (k - 1)). l = h # l')" 
    using `l \<in> set (map (op # h) (combine_aux t (n - 1) (k - 1)) @ combine_aux t (n - 1) k)`
    by auto
  thus "i' < j'"
  proof
    assume "l \<in> set (combine_aux t (n - 1) k)"
    hence "distinct l" "set l \<subseteq> set t"
      using combine_aux_distinct[of t "n-1" k] combine_aux_subset[of t "n-1" k] `distinct (h # t)` 
      by auto
    hence "i' > 0 \<or> j' > 0"
      using * nth_eq_iff_index_eq[of l i j]
      by - (rule ccontr, auto)
    show ?thesis
    proof (cases "i' = 0")
      case True
      thus ?thesis
        using `i' > 0 \<or> j' > 0`
        by simp
    next
      case False
      show ?thesis
      proof (cases "j' = 0")
        case True
        thus ?thesis
          using * `i' > 0 \<or> j' > 0` `distinct (h # t)` `set l \<subseteq> set t`
          by force
      next
        case False
        have "i' - 1 < length t \<and> j' - 1 < length t"
          using * `i' \<noteq> 0` `j' \<noteq> 0`
          by auto
        hence "i' - 1 < j' - 1"
          using ind2[rule_format, of l i "i' - 1" j "j' - 1"] * `distinct (h # t)` `i' \<noteq> 0` `j' \<noteq> 0`
          using `l \<in> set (combine_aux t (n - 1) k)`
          by auto
        thus ?thesis
          using `i' \<noteq> 0` `j' \<noteq> 0`
          by simp
      qed
    qed
  next
    assume "\<exists>l'\<in>set (combine_aux t (n - 1) (k - 1)). l = h # l'"
    then obtain l' where "l' \<in> set (combine_aux t (n - 1) (k - 1))" "l = h # l'"
      by auto
    hence "distinct l'" "set l' \<subseteq> set t"
      using combine_aux_distinct[of t "n-1" "k-1"] combine_aux_subset[of t "n-1" "k-1"] `distinct (h # t)`
      by auto
    hence "distinct l"
       using `l = h # l'` `distinct (h # t)`
       by auto
      
    have "i' > 0 \<or> j' > 0"
      using * `distinct l` nth_eq_iff_index_eq[of l i j]
      by - (rule ccontr, auto)

    show ?thesis
    proof (cases "i' = 0")
      case True
      thus ?thesis
        using `i' > 0 \<or> j' > 0`
        by simp
    next
      case False
      show ?thesis
      proof (cases "j' = 0")
        case True
        thus ?thesis
          using * `i' > 0 \<or> j' > 0` `l = h # l'` `distinct (h # t)` `set l' \<subseteq> set t`
          by force
      next
        case False
        show ?thesis
        proof (cases "i = 0")
          case True
          thus ?thesis
            using * `i' \<noteq> 0` `j' \<noteq> 0` `distinct (h # t)` `l = h # l'`
            by auto
        next
          case False
          hence "i > 0" "j > 0" "i' > 0" "j' > 0"
            using `i' \<noteq> 0` `j' \<noteq> 0` `i < j`
            by auto
          moreover
          hence "i - 1 < j - 1 \<and> j - 1 < length l' \<and> i' - 1 < length t \<and> j' - 1 < length t"
                "l' ! (i - 1) = t ! (i' - 1) \<and> l' ! (j - 1) = t ! (j' - 1)"
            using * `l = h # l'`
            by auto
          ultimately
          show ?thesis
            using ind1[rule_format, of l' "i-1" "i'-1" "j-1" "j'-1"] * `distinct (h # t)`
            using `l' \<in> set (combine_aux t (n - 1) (k - 1))`
            by auto          
        qed
      qed
    qed
  qed
qed

lemma combine_preserves_order:
  assumes "l \<in> set (combine elems len)" "distinct elems" "l ! i = elems ! i'" "l ! j = elems ! j'" "i < j" "j < length l" "i' < length elems" "j' < length elems"
  shows "i' < j'"
using assms combine_aux_preserves_order[of elems "length elems" len]
unfolding combine_def
by blast

(* ----------------------------------------------------------------- *)

lemma multiset_interleave:
  "p \<in> set (interleave h a) \<Longrightarrow> multiset_of p = multiset_of a + {#h#}"
proof (induct a arbitrary: p)
  case Nil
  thus ?case
    by simp
next
  case (Cons h' t')
  thus ?case
    using add.commute[of "{#h#}" "{#h'#}"]
    using add.assoc [of "multiset_of t'" "{#h#}" "{#h'#}"]
    using add.assoc [of "multiset_of t'" "{#h'#}" "{#h#}"]
    by auto
qed

(* ----------------------------------------------------------------- *)

lemma isPermutation_permute:
  "p \<in> set (permute l) \<Longrightarrow> p <~~> l"
proof (induct l arbitrary: p)
  case Nil
  thus ?case
    by (simp add: multiset_of_eq_perm)
next
  case (Cons h t)
  thus ?case
    unfolding multiset_of_eq_perm[THEN sym]
    by (auto simp add: multiset_interleave)
qed

lemma permute_bij:
  assumes "p \<in> set (permute l)"
  shows "\<exists> f. bij_betw f {..<length p} {..<length l} \<and>
    (\<forall> i<length p. p ! i = l ! (f i))"
  using assms
  using isPermutation_permute[of p l]
  using permutation_Ex_bij[of p l]
  by auto

(* ----------------------------------------------------------------- *)
  
lemma set_variations':
assumes "distinct elems"
shows "set (variations' elems len) \<subseteq> {l. length l = len \<and> set l \<subseteq> set elems \<and> distinct l}"
proof
    fix x
    assume "x \<in> set (variations' elems len)"
    then obtain x' where "x' \<in> set (combine elems len)" "x \<in> set (permute x')"
      unfolding variations'_def
      by auto
    thus "x \<in> {l. length l = len \<and> set l \<subseteq> set elems \<and> distinct l}"
      using isPermutation_permute[of x x'] set_combine[of x' elems len] assms
      by (auto simp add: perm_distinct_iff perm_length perm_set_eq)
qed

(* ----------------------------------------------------------------- *)

lemma set_variations_aux:
  "set (variations_aux len elems suffix) = {l. \<exists> prefix. length prefix = len \<and> distinct prefix \<and> set prefix \<subseteq> set elems - set suffix \<and> l = prefix @ suffix}"
proof (induct len elems suffix rule: variations_aux.induct)
  case (1 len elems suffix)
  show ?case
  proof (cases "len=0")
    case True
    thus ?thesis
      by (simp add: variations_aux.simps[of 0 elems suffix])
  next
    case False
    have "set (variations_aux len elems suffix) =  (\<Union>x\<in>{x. x \<in> set elems - set suffix}. set (variations_aux (len - 1) elems (x # suffix)))"
      using `len \<noteq> 0`
      by (simp add: variations_aux.simps[of len elems suffix])
    also have "... = {l. \<exists> prefix. length prefix = len \<and> distinct prefix \<and> set prefix \<subseteq> set elems - set suffix \<and> l = prefix @ suffix}"
    proof safe
      fix l x
      assume "l \<in> set (variations_aux (len - 1) elems (x # suffix))" "x \<in> set elems" "x \<notin> set suffix"
      then obtain prefix' where  "length prefix' = len - 1" "distinct prefix'" "set prefix' \<subseteq> set elems" "set prefix' \<inter> set (x # suffix) = {}" "l = prefix' @ x # suffix" 
        using 1(1)[OF `len \<noteq> 0`, of "[x\<leftarrow>elems . x \<notin> set suffix]" x]
        by auto
      thus "\<exists> prefix. length prefix = len \<and> distinct prefix \<and> set prefix \<subseteq> set elems - set suffix \<and> l = prefix @ suffix"
        using `x \<in> set elems` `x \<notin> set suffix` `len \<noteq> 0`
        by (rule_tac x="prefix' @ [x]" in exI, auto)
    next
      fix x prefix
      assume *: "len = length prefix" "distinct prefix" "set prefix \<subseteq> set elems - set suffix"
      let ?x = "last prefix" and ?prefix' = "butlast prefix"
      have "length ?prefix' = len - 1 \<and> distinct ?prefix' \<and> set ?prefix' \<subseteq> set elems - set suffix \<and> prefix @ suffix = ?prefix' @ ?x # suffix"
        using `len \<noteq> 0` *
        using in_set_butlastD[of _ prefix]
        by (auto simp add: distinct_butlast)
      moreover
      have "?x \<in> set prefix" "?x \<notin> set ?prefix'"
        using `len = length prefix` `len \<noteq> 0` `distinct prefix`
        using distinct_append[of ?prefix' "[?x]"]
        by auto
      hence "?x \<in> set elems - set suffix" "?x \<notin> set ?prefix'"
        using *
        by auto
      ultimately
      have "prefix @ suffix \<in> set (variations_aux (len - 1) elems (?x # suffix))"
        using `len \<noteq> 0` *
        by (subst 1(1), auto) force
      thus "prefix @ suffix \<in> (\<Union>x\<in>{x. x \<in> set elems - set suffix}. set (variations_aux (length prefix - 1) elems (x # suffix)))"
        using `?x \<in> set elems - set suffix` `len = length prefix`
        by auto
    qed
    finally
    show ?thesis
      .
  qed
qed

lemma set_variations:
  "set (variations elems len) = {l. length l = len \<and> distinct l \<and> set l \<subseteq> set elems}"
unfolding variations_def
by (subst set_variations_aux, auto)

(* ----------------------------------------------------------------- *)
  
lemma set_variations_min_hd:
assumes "len > 0"
shows "set (variations_min_hd len n) = {l. length l = len \<and> min_hd l \<and> distinct l \<and> set l \<subseteq> {0..<n}}"
proof safe
  fix l
  assume "l \<in> set (variations_min_hd len n)"
  then obtain x l' where *: "x \<in> set [0..<n]" "l = x # l'" "l' \<in> set (variations [x+1..<n] (len-1))"
    unfolding variations_min_hd_def
    by auto
  thus "length l = len" "distinct l" "list_all (op < (hd l)) (tl l)"
    using set_variations[of "[x+1..<n]" "len-1"] `len > 0`
    by (auto simp add: list_all_iff)
  fix x' assume "x' \<in> set l"
  thus "x' \<in> {0..<n}"
    using * set_variations[of "[x+1..<n]" "len-1"] `len > 0`
    by auto                                    
next
  fix l
  assume *: "len = length l" "list_all (op < (hd l)) (tl l)" "distinct l" "set l \<subseteq> {0..<n}"
  hence "tl l \<in> set (variations [(hd l)+1..<n] (length l - 1))"
    using set_variations[of "[hd l + 1..<n]" "length l - 1"]
    using list.set_sel(2)[of l]
    by (force simp add: distinct_tl list_all_iff)
  thus "l \<in> set (variations_min_hd (length l) n)"
    using * `len > 0`
    unfolding variations_min_hd_def
    by auto (rule_tac x="hd l" in bexI, force+)
qed

end