theory convex
imports Main Real "~~/src/HOL/Library/Product_Vector" "~~/src/HOL/Library/Code_Target_Nat" "~~/src/HOL/Library/Code_Target_Int"  "~~/src/HOL/Library/List_lexord" "Combinatorics"
begin

(* ----------------------------------------------------------------- *)

definition And where "And l = foldl (op \<and>) True l" 

lemma And: "And l \<longleftrightarrow> (\<forall> x \<in> set l. x)"
by (induct l rule: rev_induct) (auto simp add: And_def)

definition Or where "Or l = foldl (op \<or>) False l"

lemma Or: "Or l \<longleftrightarrow> (\<exists> x \<in> set l. x)"
unfolding Or_def
by (induct l rule: rev_induct) auto

(* ----------------------------------------------------------------- *)

lemma length1:
assumes "length l = 1" 
shows "\<exists> l0. l = [l0]"
using assms
by (cases l) auto

lemma length2:
assumes "length l = 2" 
shows "\<exists> l0 l1. l = [l0, l1]"
using assms
by (metis One_nat_def Suc_length_conv length1 numeral_2_eq_2)

lemma length3:
assumes "length l = 3" 
shows "\<exists> l0 l1 l2. l = [l0, l1, l2]"
using assms
by (metis Suc_length_conv length2 numeral_2_eq_2 numeral_3_eq_3)

lemma numeral_4_eq_4: "4 = Suc (Suc (Suc (Suc 0)))"
by auto

lemma length4:
assumes "length l = 4" 
shows "\<exists> l0 l1 l2 l3. l = [l0, l1, l2, l3]"
using assms
by (metis Suc_length_conv length3 numeral_3_eq_3 numeral_4_eq_4)

lemma numeral_5_eq_5: "5 = Suc (Suc (Suc (Suc (Suc 0))))"
by auto

lemma length5:
assumes "length l = 5" 
shows "\<exists> l0 l1 l2 l3 l4. l = [l0, l1, l2, l3, l4]"
using assms
by (metis Suc_length_conv length4 numeral_4_eq_4 numeral_5_eq_5)

(* ----------------------------------------------------------------- *)

(* Knuth's axioms *)
locale convex =
   fixes ccw :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool" and
         col :: "'p \<Rightarrow> 'p \<Rightarrow> 'p \<Rightarrow> bool" 
   assumes ax0: "ccw p q r \<Longrightarrow> p \<noteq> q \<and> p \<noteq> r \<and> q \<noteq> r \<and> \<not> col p q r" and
           ax1: "ccw p q r \<Longrightarrow> ccw q r p" and
           ax2: "ccw p q r \<Longrightarrow> \<not> ccw p r q" and
           ax3: "\<lbrakk>p \<noteq> q;  p \<noteq> r; q \<noteq> r; \<not> col p q r\<rbrakk> \<Longrightarrow> ccw p q r \<or> ccw p r q" and
           ax4: "\<lbrakk>ccw t q r; ccw p t r;  ccw p q t\<rbrakk> \<Longrightarrow> ccw p q r" and
           ax5: "\<lbrakk>ccw t s p; ccw t s q; ccw t s r; ccw t p q; ccw t q r\<rbrakk> \<Longrightarrow> ccw t p r" and
           ax5':"\<lbrakk>ccw s t p; ccw s t q; ccw s t r; ccw t p q; ccw t q r\<rbrakk> \<Longrightarrow> ccw t p r" and

           col1: "col a b c \<Longrightarrow> col b a c" and
           col2: "col a b c \<Longrightarrow> col c a b"
begin

definition in_general_position :: "'p set \<Rightarrow> bool" where
  "in_general_position S \<longleftrightarrow> \<not> (\<exists> A B C. A \<in> S \<and> B \<in> S \<and> C \<in> S \<and> A \<noteq> B \<and> A \<noteq> C \<and> B \<noteq> C \<and> col A B C)"

lemma in_general_position_mono:
assumes "A \<subseteq> B" "in_general_position B"
shows "in_general_position A"
using assms
unfolding in_general_position_def
by blast

lemma ax0': "\<not> ccw p p r" "\<not> ccw p q q" "\<not> ccw p q p"
using ax0
by blast+

lemma ax2':
assumes "\<not> ccw p q r" "p \<noteq> r \<and> p \<noteq> q \<and> r \<noteq> q \<and> \<not> col p r q"
shows "ccw p r q"
using assms
using ax3 ax2
by blast

lemma ax_pos:
 shows "ccw p q r \<longrightarrow> ccw q r p \<and> ccw r p q \<and> \<not> ccw q p r \<and> \<not> ccw p r q \<and> \<not> ccw r q p"
using ax1 ax2
by metis

lemma ax_neg:
 assumes "in_general_position {p, q, r}" "distinct [p, q, r]"
 shows "\<not> ccw p q r \<longrightarrow> \<not> ccw q r p \<and> \<not> ccw r p q \<and> ccw q p r \<and> ccw p r q \<and> ccw r q p"
apply auto
using assms ax1 ax2 ax2'
unfolding in_general_position_def
by (smt distinct_length_2_or_more insertI1 insert_commute)+

lemma ax5_list:
  assumes "\<forall> i < length l. ccw t s (l ! i)" "\<forall> i < length l - 1. ccw t (l ! i) (l ! (i+1))" "length l \<ge> 3"
  shows "ccw t (l ! 0) (l ! (length l - 1))"
using assms
proof (induct l)
  case Nil
  thus ?case
    by simp
next
  case (Cons p l')
  show ?case
  proof (cases "length l' = 2")
    case True
    hence "ccw t s ((p # l') ! 0)" "ccw t s ((p # l') ! 1)"  "ccw t s ((p # l') ! 2)"
      using Cons(2)
      apply auto
      apply (erule_tac x="0" in allE, simp)
      apply (erule_tac x="1" in allE, simp)
      apply (erule_tac x="2" in allE, simp)
      done
    moreover
    have "ccw t p (l' ! 0)"  "ccw t (l' ! 0) (l' ! 1)"
      using Cons(3) `length l' = 2`
      apply auto
      apply (erule_tac x="0" in allE, simp)
      apply (erule_tac x="1" in allE, simp)
      done
    ultimately  
    show ?thesis
      using ax5[of t s p "l' ! 0" "l' ! 1"] `length l' = 2`
      by auto
  next
    case False
    hence "length l' \<ge> 3"
      using Cons(4)
      by auto
    have "(p # l') ! length l' = l' ! (length l' - 1)"
      using `length l' \<ge> 3`
      by (cases "length l'") auto
    have "\<forall>i<length l'. ccw t s (l' ! i)" "\<forall>i<length l' - 1. ccw t (l' ! i) (l' ! (i + 1))"
      using Cons(2-) `length l' \<ge> 3`
      apply auto
      apply (erule_tac x="i+1" in allE, auto)
      apply (erule_tac x="i+1" in allE, auto)
      done
    hence "ccw t (l' ! 0) (l' ! (length l' - 1))"
      using Cons(1) `length l' \<ge> 3` 
      by auto
    moreover
    have "ccw t p (l' ! 0)"
      using Cons(3) `length l' \<ge> 3`
      by auto (erule_tac x="0" in allE, auto)
    moreover
    have "ccw t s p" "ccw t s (l' ! 0)" "ccw t s (l' ! (length l' - 1))"
      using Cons(2) `length l' \<ge> 3` `(p # l') ! length l' = l' ! (length l' - 1)`
      by auto (erule_tac x="1" in allE, force)
    ultimately
    show ?thesis
      using ax5[of t s p "l' ! 0" "l' ! (length l' - 1)"] `(p # l') ! length l' = l' ! (length l' - 1)`
      by simp
  qed
qed

(* Convex polygons *)

definition convex_polygon :: "'p list \<Rightarrow> bool" where
"convex_polygon p \<longleftrightarrow> 
   length p \<ge> 3 \<and> 
   (\<forall> i j k. 0 \<le> i \<and> i < j \<and> j < k \<and> k < length p \<longrightarrow> ccw (p ! i) (p ! j) (p ! k))"

lemma convex_polygon_distinct:
  assumes "convex_polygon p"
  shows "distinct p"
using assms
unfolding distinct_conv_nth
proof safe
  fix i j
  assume "convex_polygon p" "i < length p" "j < length p" "i \<noteq> j" "p ! i = p ! j"
  
  have "length p \<ge> 3" 
    using `convex_polygon p`
    unfolding convex_polygon_def
    by simp
  
  obtain k where "k \<noteq> i" "k \<noteq> j" "k < length p"
    using `length p \<ge> 3`
    by (metis Suc_leD Suc_lessI Suc_n_not_le_n length_greater_0_conv list.size(3) n_not_Suc_n numeral_3_eq_3 old.nat.distinct(2))
  
  have "p ! i \<noteq> p ! j"
    using `convex_polygon p` `i \<noteq> j` `k \<noteq> i` `k \<noteq> j` `i < length p` `j < length p` `k < length p` 
    using ax0
    unfolding convex_polygon_def
    apply (cases "i < j")
    apply (cases "j < k")
    apply force
    apply (cases "i < k")
    apply (simp, metis linorder_neqE_nat)
    apply (simp, metis linorder_neqE_nat)
    apply (cases "i < k")
    apply (simp, metis linorder_neqE_nat)
    apply (cases "j < k")
    apply (simp, metis linorder_neqE_nat)
    apply (simp, metis linorder_neqE_nat)
    done
  thus False
    using `p ! i = p ! j`
    by simp
qed

lemma convex_polygon_in_general_position:
  assumes "convex_polygon p"
  shows "in_general_position (set p)"
using assms
unfolding in_general_position_def
proof safe
  fix A B C
  assume "convex_polygon p" "col A B C" and *: "A \<in> set p" "B \<in> set p" "C \<in> set p" "A \<noteq> B" "A \<noteq> C" "B \<noteq> C"

  obtain i j k where **: "A = p ! i" "B = p ! j" "C = p ! k" "i < length p" "j < length p" "k < length p" 
    using * in_set_conv_nth[of _ p]
    by auto
  have "i \<noteq> j" "j \<noteq> k" "i \<noteq> k"
    using * ** convex_polygon_distinct[OF `convex_polygon p`] nth_eq_iff_index_eq[of p]
    by auto
  have "\<not> col A B C"
    using `convex_polygon p` `i \<noteq> j` `i \<noteq> k` `j \<noteq> k` ** 
    using ax0
    unfolding convex_polygon_def
    apply (cases "i < j")
    apply (cases "j < k")
    apply force
    apply (cases "i < k")
    apply simp
    apply (metis linorder_neqE_nat col1 col2)
    apply (simp, metis linorder_neqE_nat col2)
    apply (simp, metis linorder_neqE_nat col1 col2)
    done
  thus False
    using `col A B C`
    by simp
qed

definition cyclic_perm :: "'p list \<Rightarrow> 'p list \<Rightarrow> bool" where
  "cyclic_perm p p' \<longleftrightarrow> p = p' \<or> (\<exists> n < length p. p' = drop n p @ take n p)"

lemma cyclic_perm_refl [simp]: "cyclic_perm p p"
unfolding cyclic_perm_def
by auto

lemma cyclic_perm_sym: 
assumes "cyclic_perm p p'"
shows "cyclic_perm p' p"
using assms
unfolding cyclic_perm_def
apply (auto simp add: min_def)
apply (case_tac "n=0")
apply (erule_tac x="0" in allE, simp)
apply (erule_tac x="length p - n" in allE, simp)
done

lemma cyclic_perm_trans:
 assumes "cyclic_perm p q" "cyclic_perm q r" 
 shows "cyclic_perm p r"
proof (cases "p = q")
  case True
  thus ?thesis
    using assms
    unfolding cyclic_perm_def
    by auto
next  
  case False
  show ?thesis
  proof (cases "q = r")
    case True
    thus ?thesis
      using assms
      unfolding cyclic_perm_def
      by auto
  next
    case False
    obtain n1 n2 where *: "n1 < length p" "q = drop n1 p @ take n1 p" "n2 < length q" "r = drop n2 q @ take n2 q"
      using assms `p \<noteq> q` `q \<noteq> r `
      unfolding cyclic_perm_def
      by auto
    have "n2 < length p"
      using *
      by auto
    show ?thesis
    proof (cases "n2 + n1 < length p")
      case True
      hence "\<exists>n<length p. r = drop n p @ take n p"
        using *
        unfolding cyclic_perm_def
        by (rule_tac x="n1+n2" in exI) (auto simp add: take_add add.commute)
      thus ?thesis
        using `p \<noteq> q` `q \<noteq> r`
        unfolding cyclic_perm_def
        by simp
    next
      case False
      hence "drop (n2 + n1 - length p) p = drop (n2 + n1 - length p) ((take n1 p) @ (drop n1 p))"
        by simp
      also have "... = drop (n2 + n1 - length p) (take n1 p) @ drop n1 p"
        using `n1 < length p` `n2 < length p`
        by (subst drop_append) (simp add: min_def)
      finally
      have "\<exists>n<length p. r = drop n p @ take n p"
        using * False
        unfolding cyclic_perm_def
        by (rule_tac x="n2 + n1 - length p" in exI, auto simp add: min_def)
      thus ?thesis
        using `p \<noteq> q` `q \<noteq> r`
        unfolding cyclic_perm_def
        by simp
    qed
  qed
qed

lemma cyclic_perm_set:
 assumes "cyclic_perm p p'"
 shows "set p' = set p"
proof (cases "p = p'")
  case True
  thus ?thesis
    using assms
    unfolding cyclic_perm_def
    by auto
next
  case False
  with `cyclic_perm p p'` obtain n where
    "n < length p" "p' = drop n p @ take n p"
    unfolding cyclic_perm_def
    by auto
  thus ?thesis
     apply (subst append_take_drop_id[of n p, symmetric])
     apply (subst `p' = drop n p @ take n p`) 
     apply (subst set_append)+
     by auto
qed

lemma cyclic_perm_length:
assumes "cyclic_perm p p'"
shows "length p' = length p"
using assms
unfolding cyclic_perm_def
by (auto split: split_if_asm)

lemma cyclic_perm_distinct:
assumes "cyclic_perm p p'" 
shows "distinct p = distinct p'"
proof (cases "p = p'")
  case True
  thus ?thesis
    using assms
    unfolding cyclic_perm_def
    by auto
next    
  case False
  with `cyclic_perm p p'` obtain n where
    "n < length p" "p' = drop n p @ take n p"
    unfolding cyclic_perm_def
    by auto
  show ?thesis
     apply (subst append_take_drop_id[of n p, symmetric])
     apply (subst `p' = drop n p @ take n p`)
     apply (subst distinct_append[of "drop n p" "take n p"])
     apply (subst distinct_append[of "take n p" "drop n p"])
     by blast
qed

lemma cyclic_perm_shift: 
assumes "p' = drop n p @ take n p" "i < length p" "n < length p"
shows "p' ! i = (p ! ((i + n) mod length p))"
using assms
by (auto simp add: nth_append add.commute less_diff_conv mod_if)

lemma convex_polygon_cyclic_perm: 
 assumes "convex_polygon p" "cyclic_perm p p'"
 shows "convex_polygon p'"
proof (cases "p = p'")
  case True
  thus ?thesis
    using assms
    by auto
next    
  case False
  with `cyclic_perm p p'` obtain n where
    "n < length p" "p' = drop n p @ take n p"
    unfolding cyclic_perm_def
    by auto
  have "set p' = set p"
    by (rule cyclic_perm_set[OF assms(2)])
     
  have "length p' = length p"
    by (rule cyclic_perm_length[OF assms(2)])
     
  have "length p \<ge> 3" and *: "\<forall>i j k. 0 \<le> i \<and> i < j \<and> j < k \<and> k < length p \<longrightarrow> ccw (p ! i) (p ! j) (p ! k)"
     using `convex_polygon p`
     unfolding convex_polygon_def
     by auto
     
  show ?thesis
  unfolding convex_polygon_def
  proof safe
    show "length p' \<ge> 3" 
      using `length p' = length p` `length p \<ge> 3`
      by simp
  next
    fix i j k
    assume "i < j" "j < k" "k < length p'"
    thus "ccw (p' ! i) (p' ! j) (p' ! k)"
       using cyclic_perm_shift[OF `p' = drop n p @ take n p` _ `n < length p`]
      apply (subst  (asm) `length p' = length p`)
      apply (subst `p' = drop n p @ take n p`)+
      apply (auto simp add: nth_append *)
    proof-
      assume "i < j" "k < length p" "j < length p - n" "\<not> k < length p - n"
      hence "k + n - length p < n + i" "n + i < n + j" "n + j < length p" 
        using `n < length p`
        by auto
      hence "ccw (p ! (k + n - length p)) (p ! (n + i)) (p ! (n + j)) "
        using *
        by simp
      thus "ccw (p ! (n + i)) (p ! (n + j)) (p ! (k + n - length p))"
        by (rule ax1)
    next
      assume "j < k" "k < length p" "i < length p - n" "\<not> j < length p - n"
      hence "j + n - length p < k + n - length p" "k + n - length p < n + i" "n + i < length p"
        by auto
      hence "ccw (p ! (j + n - length p)) (p ! (k + n - length p)) (p ! (n + i))"
        using *
        by simp
      thus "ccw (p ! (n + i)) (p ! (j + n - length p)) (p ! (k + n - length p))"
        using ax1
        by blast
    next
       assume "i < j" "j < k" "k < length p" "\<not> i < length p - n"
       hence "i - (length p - n) < j - (length p - n)" "j - (length p - n) < k - (length p - n)" "k - (length p - n) < length p"
         using `n < length p`
         by auto
       thus "ccw (p ! (i - (length p - n))) (p ! (j - (length p - n))) (p ! (k - (length p - n)))"
         using *
         by simp
    qed
  qed
qed

(* set contains convex n-polygon *)

definition contains_convex_polygon where
 "contains_convex_polygon n S \<longleftrightarrow> (\<exists> p. set p \<subseteq> S \<and> length p = n \<and> convex_polygon p)"

lemma contains_convex_polygon_mono:
assumes "contains_convex_polygon n S" "k \<le> n" "k \<ge> 3"
shows "contains_convex_polygon k S"
using assms
unfolding contains_convex_polygon_def convex_polygon_def
apply auto
apply (rule_tac x="take k p" in exI)
by simp (meson in_general_position_mono less_imp_le_nat min.absorb2 order_trans set_take_subset)

lemma contains_convex_polygon_mono_set:
assumes "contains_convex_polygon n S" "S \<subseteq> S'"
shows "contains_convex_polygon n S'"
using assms
unfolding contains_convex_polygon_def
by auto

(* proof automation *)

definition "ax1_formula l = And (map (\<lambda> x. let p = x!0; q = x!1; r = x!2 in ccw p q r \<longrightarrow> ccw q r p) (variations' l 3))"
definition "ax2_formula l = And (map (\<lambda> x. let p = x!0; q = x!1; r = x!2 in ccw p q r \<longrightarrow> \<not> ccw p r q) (variations' l 3))"
definition "ax3_formula l = And (map (\<lambda> x. let p = x!0; q = x!1; r = x!2 in ccw p q r \<or> ccw p r q) (variations' l 3))"
definition "ax4_formula l = And (map (\<lambda> x. let p = x!0; q = x!1; r = x!2; t=x!3 in ccw t q r \<and> ccw p t r \<and> ccw p q t  \<longrightarrow> ccw p q r ) (variations' l 4))"
definition "ax5_formula l = And (map (\<lambda> x. let p = x!0; q = x!1; r = x!2; s=x!3; t=x!4 in ccw t s p \<and> ccw t s q \<and> ccw t s r \<and> ccw t p q \<and> ccw t q r \<longrightarrow> ccw t p r) (variations' l 5))"

lemma ax1_formula:
"ax1_formula l"
unfolding ax1_formula_def
apply (subst And)
apply (auto simp add: Let_def)
using ax1
by blast

lemma ax2_formula:
"ax2_formula l"
unfolding ax2_formula_def
apply (subst And)
apply (auto simp add: Let_def)
using ax2
by blast

lemma ax3_formula:
assumes "in_general_position (set l)" "distinct l"
shows "ax3_formula l"
unfolding ax3_formula_def
proof (subst And, safe)
  fix y
  assume "y \<in> set (map (\<lambda>x. let p = x ! 0; q = x ! 1; r = x ! 2 in ccw p q r \<or> ccw p r q) (variations' l 3))"
  then obtain x where "x \<in> set (variations' l 3)" "y = (let p = x ! 0; q = x ! 1; r = x ! 2 in ccw p q r \<or> ccw p r q)"
    by auto
  hence "set x \<subseteq> set l" "length x = 3" "distinct x"
    using set_variations'[OF `distinct l`]
    by auto
  hence *: "x ! 0 \<noteq> x ! 1" "x ! 0 \<noteq> x ! 2" "x ! 1 \<noteq> x ! 2"
    by (auto simp add: nth_eq_iff_index_eq)
  hence "x ! 0 \<in> set l" "x ! 1 \<in> set l" "x ! 2 \<in> set l"
    using `set x \<subseteq> set l` `length x = 3`
    by auto
  hence "\<not> col (x ! 0) (x ! 1) (x ! 2)"
    using `in_general_position (set l)` *
    unfolding in_general_position_def
    by blast
  hence "let p = x ! 0; q = x ! 1; r = x ! 2 in ccw p q r \<or> ccw p r q"
    using * ax3
    unfolding Let_def
    by simp
  thus y
    using `y = (let p = x ! 0; q = x ! 1; r = x ! 2 in ccw p q r \<or> ccw p r q)`
    by simp
qed

lemma ax4_formula:
"ax4_formula l"
unfolding ax4_formula_def
apply (subst And)
apply (auto simp add: Let_def)
using ax4
by blast

lemma ax5_formula:
"ax5_formula l"
unfolding ax5_formula_def
apply (subst And)
apply (auto simp add: Let_def)
using ax5
by blast

definition convex_polygon_formula :: "'p list \<Rightarrow> bool" where
  "convex_polygon_formula l = And (map (\<lambda> x. ccw (x ! 0) (x ! 1) (x ! 2)) (combine l 3))"
    
lemma convex_polygon_formula:
assumes "convex_polygon l"
shows "convex_polygon_formula l"
unfolding convex_polygon_formula_def
proof (subst And, simp, safe)
  have "distinct l" 
    using `convex_polygon l`
    by (simp add: convex_polygon_distinct)
  fix x
  assume *: "x \<in> set (combine l 3)"
  hence "length x = 3" "distinct x" "set x \<subseteq> set l"
    using set_combine[OF _ `distinct l`]
    by auto
  hence "x ! 0 \<in> set l" "x ! 1 \<in> set l" "x ! 2 \<in> set l"
    by auto
  then obtain i j k where **: "i < length l" "j < length l" "k < length l" "x ! 0 = l ! i" "x ! 1 = l ! j" "x ! 2 = l ! k"
    using in_set_conv_nth[of "x!0" l] using in_set_conv_nth[of "x!1" l] using in_set_conv_nth[of "x!2" l]
    by auto
  moreover
  hence "i < j" "j < k" "k < length l"
    using combine_preserves_order[OF * `distinct l`, of 0 i 1 j] combine_preserves_order[OF * `distinct l`, of 1 j 2 k] `length x = 3`
    by auto
  ultimately
  show "ccw (x ! 0) (x ! Suc 0) (x ! 2)"
    using `convex_polygon l`
    unfolding convex_polygon_def
    by auto
qed

(* convex hull *)

definition convex_hull :: "'p list \<Rightarrow> 'p set \<Rightarrow> bool" where 
  "convex_hull p S \<longleftrightarrow> (S \<noteq> {} \<longrightarrow> p \<noteq> []) \<and> set p \<subseteq> S \<and> distinct p \<and> 
    (let n = length p 
      in \<forall> i. 0 \<le> i \<and> i < n \<longrightarrow> 
         (let s = p ! i; 
              t = p ! ((i + 1) mod n) 
           in (\<forall> r \<in> S. r \<noteq> s \<and> r \<noteq> t \<longrightarrow> ccw s t r)))"

(* empty set has an empty a convex hull *)           
lemma convex_hull_0: "convex_hull [] {}"
unfolding convex_hull_def
by auto

lemma convex_hull_empty:
assumes "convex_hull p S"
shows "p = [] \<longleftrightarrow> S = {}"
using assms
by (auto simp add: convex_hull_def)


(* convex hull of a single point *)
lemma convex_hull_1: "convex_hull [A] {A}"
unfolding convex_hull_def
by auto

lemma convex_hull_1_point:
assumes "convex_hull p {A}"
shows "p = [A]"
using assms
unfolding convex_hull_def
by (metis (no_types, lifting) List.finite_set card.empty card.insert distinct.simps(2) distinct_card distinct_singleton in_set_conv_nth length_Cons less_Suc0 list_eq_iff_nth_eq nth_Cons_0 set_empty subset_empty subset_insert_iff subset_singletonD)

(* convex hull of two points *)
lemma convex_hull_2: "A \<noteq> B \<Longrightarrow> convex_hull [A, B] {A, B}"
unfolding convex_hull_def
apply (auto simp add: Let_def)
apply (case_tac "i = 0", simp, case_tac "i = 1", simp, simp)
apply (case_tac "i = 0", simp, case_tac "i = 1", simp, simp)
done

lemma convex_hull_2_points:
assumes "A \<noteq> B" "convex_hull p {A, B}"
shows "p = [A, B] \<or> p = [B, A]"
proof-
  have "set p \<subseteq> {A, B}" "distinct p" "p \<noteq> []"
    using assms
    unfolding convex_hull_def
    by auto
  hence "length p \<le> 2"
    by (smt List.finite_set One_nat_def add.commute add.right_neutral add_Suc_right antisym_conv assms(1) card.empty card.insert contra_subsetD distinct_card dual_order.trans empty_set empty_subsetI eq_iff insert_absorb insert_commute insert_not_empty insert_subsetI le_cases le_cases list.simps(15) list.size(3) list.size(4) numeral_One numeral_le_iff one_add_one order_refl order_trans semiring_norm(69) set_empty set_eq_subset singleton_insert_inj_eq' subset_antisym subset_insert_iff subset_singletonD)
  hence "length p = 1 \<or> length p = 2"
    using `p \<noteq> []`
    by (metis One_nat_def dual_order.antisym length_0_conv less_2_cases not_le)
  thus ?thesis
  proof
    assume "length p = 1"
    have False
    proof-
      show ?thesis
        using `convex_hull p {A, B}` `length p = 1` `A \<noteq> B`
        unfolding convex_hull_def
        by (auto simp add: Let_def ax0')
    qed
    thus ?thesis
      by simp
  next
    assume "length p = 2"
    thus ?thesis
      using `set p \<subseteq> {A, B}`
      apply (cases p, simp, simp)
      by (smt Suc_length_conv `distinct p` distinct.simps(2) insert_absorb insert_not_empty length_0_conv set_ConsD set_empty subset_empty subset_insert_iff)
  qed
qed

lemma card1: 
  "card S = 1 \<longleftrightarrow> (\<exists> A. S = {A})"
by (simp add: card_Suc_eq)

lemma card2:
  "card S = 2 \<longleftrightarrow> (\<exists> A B. A \<noteq> B \<and> S = {A, B})"
using card_Suc_eq[of S 1]  
by  auto (metis One_nat_def card1 singletonI)

lemma convex_hull_card_lt_3:
assumes "finite S" "card S < 3" "convex_hull p S"
shows "length p = card S"
proof-
  have "card S = 0 \<or> card S = 1 \<or> card S = 2"
    using assms(2)
    by auto
  thus ?thesis
  proof
    assume "card S = 0"
    thus ?thesis
      using assms convex_hull_empty[of p S]
      by auto
  next
    assume "card S = 1 \<or> card S = 2"
    thus ?thesis
    proof
      assume "card S = 1"
      then obtain A where "S = {A}"
        using card1[of S]
        by auto
      thus ?thesis
        using convex_hull_1_point[of p A] `convex_hull p S`
        by auto
    next
      assume "card S = 2"
      then obtain A B where "S = {A, B}" "A \<noteq> B"
        using card2[of S]
        by auto
      thus ?thesis
        using convex_hull_2_points[of A B p] `convex_hull p S`
        by auto
    qed
  qed
qed

(* convex hull of more than two points must be at least a triangle *)
lemma convex_hull_card_gt_2:
assumes "convex_hull p S" "card S > 2"
shows "length p > 2"
proof (rule ccontr)
  assume "\<not> ?thesis"
  show False
  proof (cases "length p = 2")
    case True
    then obtain A B where "p = [A, B]"
      by (smt One_nat_def Suc_diff_Suc Suc_length_conv diff_zero length_0_conv lessI less_2_cases numeral_One numeral_eq_iff pos2 semiring_norm(85))
    hence "A \<in> S" "B \<in> S" "A \<noteq> B"
      using `convex_hull p S`
      unfolding convex_hull_def
      by auto
    then obtain C where "C \<noteq> A" "C \<noteq> B" "C \<in> S"
      using `card S > 2` card2[of S]
      by auto
    have "\<forall>q\<in>S. q \<noteq> A \<and> q \<noteq> B \<longrightarrow> ccw A B q" "\<forall>q\<in>S. q \<noteq> B \<and> q \<noteq> A \<longrightarrow> ccw B A q"
      using `convex_hull p S` `p = [A, B]`
      unfolding convex_hull_def
      by auto
    hence "ccw A B C" "ccw B A C"
      using `C \<in> S` `C \<noteq> A` `C \<noteq> B`
      by auto
    thus False
      using ax1 ax2
      by blast
  next
    assume "length p \<noteq> 2"
    hence "length p < 2"
      using `\<not> length p > 2`
      by auto
    hence "length p = 1 \<or> length p = 0"
      using less_2_cases by auto
    thus False
    proof
       assume "length p = 0"
       thus False
         using convex_hull_empty[OF `convex_hull p S`]
         using `card S > 2`
         by auto
    next
       assume "length p = 1"
       then obtain A where "p = [A]" 
         by (cases p) auto
       hence "A \<in> S"
         using `convex_hull p S`
         unfolding convex_hull_def
         by auto
       then obtain B where "B \<in> S" "A \<noteq> B"
           using `card S > 2` card1[of S]
           by auto
       hence "ccw A A B"
         using `p = [A]` `convex_hull p S`
         unfolding convex_hull_def
         by simp
       thus False
         using ax0 
         by blast
    qed
  qed
qed

(* convex hull of a convex polygon is itself *)
lemma convex_hull_polygon:
assumes "convex_polygon p"
shows "convex_hull p (set p)"
unfolding convex_hull_def Let_def
proof safe
  show "distinct p"
    using assms
    using convex_polygon_distinct
    by auto
next
  fix i and q :: 'p
  let ?X = "p ! i" and ?Y = "p ! ((i + 1) mod length p)" 
  assume "q \<in> set p" "i < length p" "q \<noteq> ?X" "q \<noteq> ?Y"
  
  have "distinct p"
    using assms convex_polygon_distinct
    by simp
  
  obtain j where "p ! j = q" "j < length p"
    using `q \<in> set p` in_set_conv_nth[of q p]
    by auto
  have "j \<noteq> i" "j \<noteq> (i + 1) mod length p"
    using `distinct p` `p ! j = q` `q \<noteq> ?X` `q \<noteq> ?Y`
    by auto

  show "ccw ?X ?Y q"
  proof (cases "i < length p - 1")
    case True
    hence *: "(i + 1) mod length p = i + 1"
      by auto
    hence "j < i \<or> j > i + 1"
      using `j \<noteq> i` `j \<noteq> (i + 1) mod length p`
      by auto
    thus ?thesis
    proof
      assume "j < i"
      hence "ccw q ?X ?Y"
        using `convex_polygon p` `p ! j = q`[symmetric] `i < length p - 1` *
        unfolding convex_polygon_def
        by auto
      thus ?thesis
        using ax1
        by blast
    next
      assume "j > i + 1"
      thus ?thesis
        using `j < length p`
        using `convex_polygon p` `p ! j = q`[symmetric]
        unfolding convex_polygon_def
        by auto
    qed
  next
    have "?X \<in> set p"
      using `i < length p`
      by auto
    hence "length p > 1" 
      using `q \<in> set p` `q \<noteq> ?X`
      by (cases p) auto
    assume "\<not> i < length p - 1"
    hence "i = length p - 1"
      using `i < length p`
      by auto
    hence *: "(i + 1) mod (length p) = 0"
      using `length p > 1`
      using le_add_diff_inverse[of 1 "length p"] 
      by fastforce
    have "i > 0"
      using `length p > 1` `i = length p - 1`
      by auto
    have "0 < j \<and> j < i"
      using `j \<noteq> i` `j \<noteq> (i + 1) mod length p` `j < length p` `i = length p - 1` *
      by auto
    hence "ccw ?Y q ?X"
      using `convex_polygon p` `p ! j = q`[symmetric] `i < length p` *
      unfolding convex_polygon_def
      by auto
    thus "ccw ?X ?Y q"
      using ax1
      by blast
    qed
qed

lemma convex_hull_is_convex_polygon': (* auxiliary lemma *) 
 fixes n::nat
 assumes "n = k - i" "0 \<le> i" "i < j" "j < k" "k < length p" "convex_hull p S"
 shows "ccw (p ! i) (p ! j) (p ! k)"
using assms
proof (induct n arbitrary: k i)
  case 0
  thus ?case
  by simp
next
  case (Suc n)
  have "distinct p" "set p \<subseteq> S"
     using `convex_hull p S`
     unfolding convex_hull_def
     by auto
  
  hence "p ! i \<in> S" "p ! j \<in> S" "p ! k \<in> S" 
     using Suc(3-6) `set p \<subseteq> S`
     by auto
  have "p ! k \<noteq> p ! i" "p ! k \<noteq> p ! j"  "p ! i \<noteq> p ! j" "p ! k \<noteq> p ! (i+1)" "p ! i \<noteq> p ! (k-1)"
    using nth_eq_iff_index_eq 
    using Suc(3-6) `distinct p`
    by fastforce+

  have "ccw (p ! i) (p ! (i + 1)) (p ! k)"
       using `convex_hull p S` Suc(3-6) `p ! k \<in> S` `p ! k \<noteq> p ! i` `p ! k \<noteq> p ! (i + 1)`
       unfolding convex_hull_def
       by (auto simp add: Let_def) (erule_tac x=i in allE, auto)

  have "ccw (p ! (k - 1)) (p ! k) (p ! i)"
       using `convex_hull p S` Suc(3-6) `p ! i \<in> S` `p ! k \<noteq> p ! i` `p ! i \<noteq> p ! (k - 1)`
       unfolding convex_hull_def
       by (auto simp add: Let_def) (erule_tac x="k - 1" in allE, auto)
  hence "ccw (p ! i) (p ! (k - 1)) (p ! k)"
       by (subst ax1, simp_all)+
       
  show "ccw (p ! i) (p ! j) (p ! k)"
  proof (cases "j = i+1")
    case True
    thus ?thesis
      using `ccw (p ! i) (p ! (i+1)) (p ! k)`
      by simp
  next
    case False
    show ?thesis
    proof (cases "j = k - 1")
      case True
      thus ?thesis
        using `ccw (p ! i) (p ! (k - 1)) (p ! k)`
        by simp
    next
      case False
      have "ccw (p ! (i + 1)) (p ! j) (p ! k)"
        using Suc(1)[of k "i+1"]
        using Suc(2-7) `j \<noteq> i + 1`
        by simp
      moreover
      have "ccw (p ! i) (p ! j) (p ! (k-1))"
        using Suc(1)[of "k-1" i] 
        using Suc(2-7) `j \<noteq> k - 1`
        by simp
      moreover
      have "p ! j \<noteq> p ! (i + 1)"
        using nth_eq_iff_index_eq `j \<noteq> i + 1` 
        using Suc(3-6) `distinct p`
        by fastforce+
      have "ccw (p ! i) (p ! (i + 1)) (p ! j)"
       using `convex_hull p S` Suc(3-6) `p ! j \<in> S` `p ! i \<noteq> p ! j` `p ! j \<noteq> p ! (i + 1)`
       unfolding convex_hull_def
       by (auto simp add: Let_def) (erule_tac x=i in allE, auto)
      moreover
      have "i + 1 \<noteq> k - 1" "i \<noteq> k - 1"
        using `j \<noteq> k - 1` `j \<noteq> i + 1` Suc(3-5)
        by auto
      hence "p ! (i+1) \<noteq> p ! (k-1)" "p ! i \<noteq> p ! (k - 1)"
        using nth_eq_iff_index_eq[of p "i+1" "k-1"] nth_eq_iff_index_eq[of p "i" "k-1"] 
        using Suc(3-6) `distinct p`
        by auto
      hence "ccw (p ! i) (p ! (i + 1)) (p ! (k - 1))"
       using `convex_hull p S` Suc(3-6) `p ! j \<in> S` `p ! i \<noteq> p ! j` 
       unfolding convex_hull_def
       by (auto simp add: Let_def) (erule_tac x=i in allE, force)
      ultimately
        show ?thesis
        using `ccw (p ! i) (p ! (i + 1)) (p ! k)` 
        using `ccw (p ! i) (p ! (k - 1)) (p ! k)` 
        using ax5[of "p ! i" "p ! (i+1)" "p ! j" "p ! (k-1)" "p ! k"]
        by simp
      qed
   qed
qed

lemma convex_hull_is_convex_polygon:
 assumes "in_general_position S" "convex_hull p S" "length p \<ge> 3"
 shows "convex_polygon p"
unfolding convex_polygon_def
proof safe
  show "length p \<ge> 3"
    by fact
next
  fix i j k
  assume *: "0 \<le> i" "i < j" "j < k" "k < length p"
  show "ccw (p ! i) (p ! j) (p ! k)"
    using convex_hull_is_convex_polygon'[of "k-i" k i j p S] * `convex_hull p S`
    by simp
qed

lemma cyclic_perm_convex_hull:
  assumes "cyclic_perm p p'" "convex_hull p S"
  shows "convex_hull p' S"
proof (cases "p = p'")
  case True
  thus ?thesis
    using assms
    by auto
next
  case False
  show ?thesis 
  proof (cases "p = []")
    case True
    thus ?thesis
      using cyclic_perm_set[OF assms(1)] `p \<noteq> p'`
      by auto
  next
    case False
    hence "p' \<noteq> []"
      using cyclic_perm_set[OF assms(1)] `p \<noteq> p'`
      by auto
    
    show ?thesis
      using `p \<noteq> p'` `p \<noteq> []` `p' \<noteq> []`
      unfolding convex_hull_def Let_def
    proof safe
      fix x
      assume "x \<in> set p'"
      thus "x \<in> S"
        using cyclic_perm_set[OF assms(1)]
        using `convex_hull p S`
        unfolding convex_hull_def
        by auto
    next
      show "distinct p'"
        using cyclic_perm_distinct[OF assms(1)]
        using `convex_hull p S`
        unfolding convex_hull_def
        by auto
    next
      fix i q
      let ?j = "(i + 1) mod length p'"
      assume "p \<noteq> p'" "p \<noteq> []" "p' \<noteq> []"
      then obtain n where "p' = (drop n p) @ (take n p)" "n < length p"
        using assms(1)
        unfolding cyclic_perm_def
        by auto

      have "length p' = length p"
        by (rule cyclic_perm_length[OF assms(1)])
     
      have "distinct p" and
        *: " \<forall>i. 0 \<le> i \<and> i < length p \<longrightarrow> 
              (\<forall>q\<in>S. q \<noteq> p ! i \<and> q \<noteq> p ! ((i + 1) mod length p) \<longrightarrow> 
                ccw (p ! i) (p ! ((i + 1) mod length p)) q)"
        using `convex_hull p S`
        unfolding convex_hull_def Let_def
        by auto

      have "?j < length p'"
        by (simp add: `length p' = length p` `p \<noteq> []`)
     
      assume ++: "q \<in> S" "i < length p'" "q \<noteq> p' ! i" "q \<noteq> p' ! ?j"
   
      have "p' ! i = p ! ((i + n) mod length p)"
        using cyclic_perm_shift[of p' n p i] `p' = (drop n p) @ (take n p)` `n < length p` `i < length p'` `length p' = length p`
        by simp
   
      have "p' ! ?j = p ! ((?j + n) mod length p)" 
        using cyclic_perm_shift[of p' n p ?j] `p' = (drop n p) @ (take n p)` `n < length p` `?j < length p'` `length p' = length p`
        by metis
      hence "p' ! ?j = p ! ((i + 1 + n) mod length p)"
        by (metis `length p' = length p` mod_add_eq mod_add_right_eq)
      hence "p' ! ?j = p ! ((i + n + 1) mod length p)"
        by simp
   
      have "((i + n) mod length p + 1) mod length p = (i + 1 + n) mod length p"
        by (metis One_nat_def add.right_neutral add_Suc_right add_diff_cancel_right' add_eq_if add_is_0 mod_add_eq mod_add_right_eq)
      hence "ccw (p ! ((i + n) mod length p)) (p ! ((i + n + 1) mod length p)) q"
        using *[rule_format, of "(i + n) mod length p" q] ++ `p' ! i = p ! ((i + n) mod length p)` `p' ! ?j = p ! ((i + n + 1) mod length p)` `p \<noteq> []`
        by simp
      thus "ccw (p' ! i) (p' ! ?j) q"
        using `p' ! i = p ! ((i + n) mod length p)` `p' ! ?j = p ! ((i + n + 1) mod length p)`
        by simp
      qed
   qed
qed

(* if there is no n-gon in a set of points, then its convex hull must have less than n points *)
lemma convex_hull_length_ub:
assumes "in_general_position S" "convex_hull p S" "\<not> contains_convex_polygon n S" "n \<ge> 3" 
shows "length p < n"
proof (rule ccontr)
 assume "\<not> ?thesis"
 have "set p \<subseteq> S"
   using `convex_hull p S`
   by (simp add: convex_hull_def)
 hence "contains_convex_polygon (length p) S"
   using `convex_hull p S` `in_general_position S` `\<not> length p < n` `n \<ge> 3`
   using convex_hull_is_convex_polygon[of S p]
   unfolding contains_convex_polygon_def
   by auto
 thus False
   using contains_convex_polygon_mono[of "length p" S n] `\<not> length p < n` `\<not> contains_convex_polygon n S` `n \<ge> 3`
   by force
qed

lemma blue_consecutive:
  assumes "convex_polygon l" "ccw (l ! 0) (l ! 1) M" "ccw (l ! (length l - 2)) (l ! (length l - 1)) M" "ccw (l ! 0) (l ! (length l - 1)) M"
  shows "\<forall> i < length l - 1. ccw (l ! i) (l ! (i + 1)) M"
using assms
proof (induct l rule: rev_induct)
  case Nil
  thus ?case
    by simp
next
  case (snoc A l')
  show ?case
  proof (cases "length l' = 1")
    case True
    thus ?thesis
      using snoc
      by (simp add: nth_append)
  next
    case False
    hence "length l' \<ge> 2"
      using `convex_polygon (l' @ [A])`
      unfolding convex_polygon_def
      by auto
    show ?thesis
    proof (cases "length l' = 2")
      case True
      thus ?thesis
        using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! 1) M` `ccw ((l' @ [A]) ! (length (l' @ [A]) - 2)) ((l' @ [A]) ! (length (l' @ [A]) - 1)) M`
        by (auto simp add: nth_append, case_tac "i=1", auto)
    next
      case False
      hence "length l' \<ge> 3"
        using `length l' \<ge> 2`
        by auto
      show ?thesis
      proof safe
        fix i
        assume "i < length (l' @ [A]) - 1"
        hence "i < length l'"
          by simp
        show "ccw ((l' @ [A]) ! i) ((l' @ [A]) ! (i + 1)) M"
        proof (cases "i < length l' - 1")
          case False
          hence "i = length l' - 1"
            using `i < length l'`
            by auto
          have "ccw (l' ! (length l' - 1)) A M"
            using `ccw ((l' @ [A]) ! (length (l' @ [A]) - 2)) ((l' @ [A]) ! (length (l' @ [A]) - 1)) M`
            using `length l' \<ge> 2`
            by (simp add: nth_append split: split_if_asm)
          thus ?thesis
            using `length l' \<ge> 2` `i = length l' - 1`
            by (auto simp add: nth_append)
        next
          case True
          have "ccw (l' ! i) (l' ! (i+1)) M"
          proof (rule snoc(1)[rule_format, of i])
            show "convex_polygon l'"
              using `convex_polygon (l' @ [A])` `length l' \<ge> 3`
              unfolding convex_polygon_def
              by (auto simp add: nth_append split: split_if_asm)

            show "ccw (l' ! 0) (l' ! 1) M"
              using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! 1) M` `length l' \<ge> 2`
              by (auto simp add: nth_append split: split_if_asm)

            show "ccw (l' ! 0) (l' ! (length l' - 1)) M"
            proof (cases "length l' = 2")
              case True
              thus ?thesis
                using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! 1) M` `length l' \<ge> 2`
                by (auto simp add: nth_append split: split_if_asm)
            next
              case False
              hence "length l' > 2"
                using `length l' \<ge> 2`
                by auto
              show ?thesis
              proof (rule ax5)
                show "ccw (l' ! 0) (l' ! 1) (l' ! (length l' - 1))"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm)
              next
                show "ccw (l' ! 0) (l' ! 1) A"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm) (erule_tac x=0 in allE, erule_tac x=1 in allE, erule_tac x="length l'" in allE, simp)
              next
                show "ccw (l' ! 0) (l' ! 1) M"
                  using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! 1) M` `length l' \<ge> 2`
                  by (auto simp add: nth_append split: split_if_asm)
              next
                show "ccw (l' ! 0) (l' ! (length l' - 1)) A"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm) (erule_tac x=0 in allE, erule_tac x="length l' - 1" in allE, erule_tac x="length l'" in allE, simp)
              next
                show "ccw (l' ! 0) A M"
                  using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! (length (l' @ [A]) - 1)) M` `length l' \<ge> 2`
                  by (auto simp add: nth_append split: split_if_asm)
              qed
            qed

            show "ccw (l' ! (length l' - 2)) (l' ! (length l' - 1)) M"
            proof (cases "length l' = 2")
              case True
              thus ?thesis
                using `ccw ((l' @ [A]) ! 0) ((l' @ [A]) ! 1) M` `length l' \<ge> 2`
                by (auto simp add: nth_append split: split_if_asm)
            next
              case False
              hence "length l' > 2"
                using `length l' \<ge> 2`
                by simp
              have "ccw (l' ! (length l' - 1)) M (l' ! (length l' - 2))"
              proof (rule ax5)
                show "ccw (l' ! (length l' - 1)) A M"
                  using ` ccw ((l' @ [A]) ! (length (l' @ [A]) - 2)) ((l' @ [A]) ! (length (l' @ [A]) - 1)) M` `length l' > 2`
                  by (simp add: nth_append split: split_if_asm)
              next
                have "ccw (l' ! 0) (l' ! (length l' - 1)) A"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm)  (erule_tac x=0 in allE, erule_tac x="length l' - 1" in allE, erule_tac x="length l'" in allE, simp)
                thus "ccw (l' ! (length l' - 1)) A (l' ! 0)"
                  using ax1
                  by blast
              next
                have "ccw (l' ! (length l' - 2)) (l' ! (length l' - 1)) A"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm) (erule_tac x="length l' - 2" in allE, erule_tac x="length l' - 1" in allE, erule_tac x="length l'" in allE, simp)                
                thus "ccw (l' ! (length l' - 1)) A (l' ! (length l' - 2))"
                  using ax1
                  by blast
              next
                have "ccw  (l' ! 0) (l' ! (length l' - 1)) M"
                  by fact
                thus "ccw (l' ! (length l' - 1)) M (l' ! 0)"
                  using ax1
                  by blast
              next
                have "ccw (l' ! 0) (l' ! (length l' - 2)) (l' ! (length l' - 1))"
                  using `convex_polygon (l' @ [A])` `length l' > 2`
                  unfolding convex_polygon_def
                  by (auto simp add: nth_append split: split_if_asm)
                thus "ccw (l' ! (length l' - 1)) (l' ! 0) (l' ! (length l' - 2)) "
                  using ax1
                  by blast
              qed
              thus ?thesis
                using ax1
                by blast
            qed
          next
            show  "i < length l' - 1"
              by fact
          qed
          thus ?thesis
            using `length l' \<ge> 2` `i < length l' - 1`
            by (auto simp add: nth_append)
        qed
      qed
    qed
  qed
qed

lemma red_consecutive:
assumes "convex_polygon l" "\<forall> i < length l - 1. ccw (l ! i) M (l ! (i+1))"
shows "ccw (l ! 0) M (l ! (length l - 1))"
using assms
proof (induct l)
  case Nil
  thus ?case
    unfolding convex_polygon_def
    by simp
next
  case (Cons A l')
  show ?case
  proof (cases "length l' = 1")
    case True
    thus ?thesis
      using Cons
      by simp
  next
    case False
    hence "length l' \<ge> 2"
      using `convex_polygon (A # l')`
      unfolding convex_polygon_def
      by auto
      
    have "ccw A M (l'! (length l' - 1))"
    proof (rule ax4)
      show "ccw (l'!0) M (l' ! (length l' - 1))"
      proof (cases "length l' = 2")
        case True
        thus ?thesis
          using Cons(3)
          by auto (erule_tac x="1" in allE, auto)
      next
        case False
        hence "length l' \<ge> 3"
          using `length l' \<ge> 2`
          by auto
        show ?thesis
        proof (rule Cons(1))
          show "convex_polygon l'"
            using `convex_polygon (A # l')` `length l' \<ge> 3`
            unfolding convex_polygon_def
            by auto (erule_tac x="i+1" in allE, erule_tac x="j+1" in allE, erule_tac x="k+1" in allE, auto)
        next
          show "\<forall>i<length l' - 1. ccw (l' ! i) M (l' ! (i + 1))"
            using `\<forall>i<length (A # l') - 1. ccw ((A # l') ! i) M ((A # l') ! (i + 1))` `length l' \<ge> 2`
            by auto (erule_tac x="i+1" in allE, auto)
        qed
      qed
    next
      show "ccw A (l' ! 0) (l' ! (length l' - 1))"
        using `convex_polygon (A # l')` `length l' \<ge> 2`
        unfolding convex_polygon_def
        by (cases "length l'", auto) (erule_tac x="0" in allE, erule_tac x="1" in allE, erule_tac x="length l'" in allE, auto)
    next
      show "ccw A M (l'!0)"
        using `\<forall>i<length (A # l') - 1. ccw ((A # l') ! i) M ((A # l') ! (i + 1))` `length l' \<ge> 2`
        by (cases l') auto
    qed
    thus ?thesis
      using `length l' \<ge> 2`
      by (cases "length l'") (auto simp add: nth_Cons)
  qed
qed

lemma ex_convex_hull': (* auxiliary lemma *)
  assumes "convex_hull p S" "M \<notin> S" "length p \<ge> 3" "in_general_position (S \<union> {M})"
  "\<not> ccw (p ! (length p - 1)) (p ! 0) M"
  "ccw (p ! (length p - 2)) (p ! (length p - 1)) M"
  shows "\<exists> p. convex_hull p (S \<union> {M})"
proof-
  let ?t = "p ! (length p - 1)"
  let ?tk = "p ! (length p - 2)"
         
  let ?blue = "{i. i < length p - 1 \<and> ccw (p ! i) (p ! (i+1)) M}"
  let ?start = "Min ?blue"
  obtain start where "start = ?start" by auto
  have "Suc (length p - 2) = length p - 1"
   using `length p \<ge> 3`
   by auto
  hence "length p - 2 \<in> ?blue"
    using `ccw ?tk ?t M`
    by auto
  hence "start \<le> length p - 2" "ccw (p ! start) (p ! (start+1)) M"
    using `length p \<ge> 3` Min_in[of ?blue] `start = ?start`
    by auto
      
  have "\<forall> i < start. \<not> ccw (p ! i) (p ! (i+1)) M"
  proof safe
    fix i
    assume "i < start" "ccw (p ! i) (p ! (i+1)) M"
    hence "i \<in> ?blue"
      using `start \<le> length p - 2`
      by simp
    hence "i \<ge> start"
      using `length p \<ge> 3` Min_le[of ?blue] `start = ?start`
      by auto
    thus False
      using `i < start`
      by simp
  qed
  
  let ?p = "M # (drop start p)"
  have "convex_hull ?p (S \<union> {M})"
    unfolding convex_hull_def Let_def
  proof safe
     fix x
     assume "x \<in> set ?p" "x \<notin> S"
     thus "x = M"
       using `convex_hull p S` set_drop_subset[of start p]
       unfolding convex_hull_def
       by auto
  next
     show "distinct ?p"
       using `convex_hull p S` set_drop_subset[of start p] `M \<notin> S`
       unfolding convex_hull_def
       by auto
  next
     fix i r
     assume *: "i < length ?p" "r \<noteq> ?p ! i" "r \<noteq> ?p ! ((i + 1) mod length ?p)" "r \<in> S"
     show "ccw (?p ! i) (?p ! ((i + 1) mod length ?p)) r"
     proof (cases "i > 0 \<and> i < length ?p - 1")
       case True
       hence "i > 0" "i < length ?p - 1"
         by auto
       have pi: "?p ! i = p ! (start + i - 1)"
         using nth_drop[of start "i - 1" p]
         using `start \<le> length p - 2` `i < length ?p` `i > 0`
         by simp
       
       have "start + i < length p"
         using `start \<le> length p - 2` `i < length ?p - 1`
         by auto
         
       have "(i + 1) mod length ?p = i + 1"
         using `i < length ?p - 1`
         by auto
       hence "?p ! ((i + 1) mod length ?p) = ?p ! (i + 1)"
         by simp
       also have "... = p ! (start + i)"
         using nth_drop[of start i p]
         using `start + i < length p`
         by simp
       finally have pi1: "?p ! ((i + 1) mod length ?p) = p ! (start + i)"
        .
        
       have "r \<noteq> p ! (start + i)" "r \<noteq> p ! (start + i - 1)"
          using `r \<noteq> ?p ! i` `r \<noteq> ?p ! ((i + 1) mod length ?p)`
          using pi pi1
          by auto
       
       hence "ccw (p ! (start + i - 1)) (p ! (start + i)) r"
         using `convex_hull p S` `r \<in> S` `start + i < length p` `i > 0`
         unfolding convex_hull_def Let_def
         by auto (erule_tac x="start + i - 1" in allE, auto)
         
       thus ?thesis
         using pi pi1
         by simp
     next
       case False
       show ?thesis
       proof (cases "i = 0")
         case True
         have "length p - start + 1 \<ge> 3"
           using `start \<le> length p - 2` `length p \<ge> 3`
           by auto
         hence "(i + 1) mod length ?p = 1"
           using `i = 0`
           by simp
         hence "?p ! ((i + 1) mod length ?p) = ?p ! 1"
           by simp
         also have "... = p ! start "
           using nth_drop[of start i p]
           using `i = 0` `start \<le> length p - 2`
           by simp
         finally have pi1: "?p ! ((i + 1) mod length ?p) = p ! start"
           .
           
         let ?t' = "p ! start"
         let ?tk' = "p ! (start + 1)"
         have "ccw M ?t' r"
         proof (cases "r = ?tk'")
           case True
           thus ?thesis
             using `ccw (p ! start) (p ! (start+1)) M` ax1
             by blast
         next
           case False
           let ?q = "if start > 0 then p ! (start - 1) else ?t"
           have "?q \<noteq> ?t'" "?q \<in> S" "?t' \<in> S"
             using `start \<le> length p - 2` `length p \<ge> 3` nth_eq_iff_index_eq[of p]
             using `convex_hull p S`
             unfolding convex_hull_def
             by auto
           hence "?t' \<noteq> M" "?q \<noteq> M"
             using `M \<notin> S`
             by auto
           hence "\<not> col ?q M ?t'"
             using `in_general_position (S \<union> {M})` `?q \<noteq> ?t'` `?q \<in> S` `?t' \<in> S`
             unfolding in_general_position_def
             by blast
           show ?thesis
           proof (cases "r = ?q")
             case True
             show ?thesis
             proof (cases "start > 0")
               case False
               thus ?thesis
                 using `r = ?q` `\<not> ccw ?t (p ! 0) M`
                 using `?t' \<noteq> M` `?q \<noteq> M` `?q \<noteq> ?t'` `\<not> col ?q M ?t'`
                 using ax1[OF ax2'[of ?q ?t' M]]
                 by simp
             next
               case True
               hence "\<not> (ccw ?q ?t' M)"
                 using `\<forall> i < start. \<not> (ccw (p ! i) (p ! (i + 1)) M)`
                 by - (erule_tac x="start-1" in allE, simp)
               thus ?thesis
                 using `r = ?q` `start > 0` `?q \<noteq> M` `?t' \<noteq> M` `?q \<noteq> ?t'` `\<not> col ?q M ?t'`
                 using ax1[OF ax2'[of ?q ?t' M]]
                 by simp
             qed
           next
             case False
             have "r \<noteq> ?t'" "r \<noteq> M"
               using `r \<noteq> ?p ! i` `r \<noteq> ?p ! ((i + 1) mod length ?p)` pi1 `i = 0`
               by simp_all
             
             hence "ccw ?t' ?tk' r"
               using `convex_hull p S` `r \<in> S` `length p \<ge> 3` `r \<noteq> ?tk'` `start \<le> length p - 2`
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="start" in allE, simp)
             moreover
             have "?q \<in> S" "?q \<noteq> ?tk'" "?q \<noteq> ?t'"
               using `convex_hull p S` `length p \<ge> 3` nth_eq_iff_index_eq[of p]
               using `start \<le> length p - 2`
               unfolding convex_hull_def
               by auto
             hence "ccw ?t' ?tk' ?q"
               using `convex_hull p S` `length p \<ge> 3` `start \<le> length p - 2`
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="start" in allE, simp)
             moreover
             have "ccw ?t' ?tk' M"
               by fact
             moreover
             have "ccw ?q ?t' r"
               using `convex_hull p S` `length p \<ge> 3` `r \<noteq> ?t'` `r \<noteq> ?q` `r \<in> S` `start \<le> length p - 2` 
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="start-1" in allE, auto, erule_tac x="length p -1" in allE, auto)
             hence "ccw ?t' r ?q"
               using ax1
               by blast
             moreover
             have "\<not> ccw ?q ?t' M"
               using `\<not> ccw ?t (p ! 0) M`
               using `\<forall> i < start. \<not> (ccw (p ! i) (p ! (i + 1)) M)`
               by - (erule_tac x="start-1" in allE, simp)
             hence "ccw ?t' ?q M"
               using ax1[OF ax1[OF ax2'[of ?q ?t' M]]] `?q \<noteq> ?t'` `?t' \<noteq> M` `?q \<noteq> M` `\<not> col ?q M ?t'`
               by simp
             ultimately
             have "ccw ?t' r M"
               using ax5
               by blast
             thus ?thesis
               using ax1
               by blast
           qed
         qed
         thus ?thesis
           using pi1 `i = 0`
           by simp
       next
         case False
         with `\<not> (0 < i \<and> i < length ?p - 1)` `i < length ?p`
         have "i = length ?p - 1"
           by auto
         hence pi1: "?p ! ((i + 1) mod length ?p) = M"
           by simp
         have "length p - start > 0"
           using `start \<le> length p - 2` `length p \<ge> 3`
           by auto
         hence pi: "?p ! i = p ! (length p - 1)"
           using `i = length ?p - 1` 
           by auto
           
         have "ccw ?t M r"
         proof (cases "r = ?tk")
           case True
           thus ?thesis
             using `ccw ?tk ?t M`
             using ax1
             by blast
         next
           case False
           let ?q = "p ! 0"
           
           have "?t \<in> S" "?q \<in> S"  "?t \<noteq> ?q"
             using `length p \<ge> 3` `convex_hull p S` nth_eq_iff_index_eq[of p]
             unfolding convex_hull_def
             by auto
           hence "?t \<noteq> M" "?q \<noteq> M"
             using `M \<notin> S`
             by auto
           have "\<not> col ?t M ?q"
             using `in_general_position (S \<union> {M})` `?t \<in> S` `?q \<in> S` `?t \<noteq> ?q` `?t \<noteq> M` `?q \<noteq> M` 
             unfolding in_general_position_def
             by blast
               
           show ?thesis
           proof (cases "r = ?q")
             case True
             thus ?thesis
               using `\<not> ccw ?t ?q M` ax2'[of ?t ?q M] `?t \<noteq> ?q` `?t \<noteq> M` `?q \<noteq> M` `r = ?q` `\<not> col ?t M ?q`
               by simp
           next
             case False
             have "Suc (length p - 2) = length p - 1"
               using `length p \<ge> 3`
               by simp
             have "r \<noteq> ?t" "r \<noteq> M"
               using `r \<noteq> ?p ! i` `r \<noteq> ?p ! ((i + 1) mod length ?p)` pi pi1
               by simp_all
             hence "ccw ?tk ?t r"
               using `convex_hull p S` `r \<in> S` `length p \<ge> 3` `r \<noteq> ?tk`
               using `Suc (length p - 2) = length p - 1`
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="length p - 2" in allE, simp)
             moreover
             have "?q \<in> S" "?q \<noteq> ?tk" "?q \<noteq> ?t"
               using `convex_hull p S` `length p \<ge> 3` nth_eq_iff_index_eq[of p]
               unfolding convex_hull_def
               by auto
             hence "ccw ?tk ?t ?q"
               using `convex_hull p S` `length p \<ge> 3`
               using `Suc (length p - 2) = length p - 1`
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="length p - 2" in allE, simp)
             moreover
             have "ccw ?t ?q r"
             using `convex_hull p S` `length p \<ge> 3` `r \<noteq> ?t` `r \<noteq> ?q` `r \<in> S` 
               unfolding convex_hull_def Let_def
               by auto (erule_tac x="length p - 1" in allE, simp)
             moreover
             have "ccw ?t M ?q"
               using `\<not> ccw ?t ?q M`
               using ax2'[of ?t ?q M] `?t \<noteq> ?q` `?t \<noteq> M` `?q \<noteq> M` `\<not> col ?t M ?q`
               by simp
             moreover
             have "ccw ?tk ?t M"
               by fact
             ultimately
             show ?thesis
               using ax5'
               by blast
           qed
         qed
         thus ?thesis
           using pi pi1
           by simp
       qed
     qed
  next
    fix i q
    assume "0 \<le> i" "i < length ?p" "M \<noteq> ?p ! i" "M \<noteq> ?p ! ((i + 1) mod length ?p)"
    have "i > 0"
      using `M \<noteq> ?p ! i`
      by - (rule ccontr, auto)

    have "length p - start > 0"
      using `start \<le> length p - 2` `length p \<ge> 3`
      by auto

    have "(i + 1) mod length ?p \<noteq> 0"
      using `M \<noteq> ?p ! ((i + 1) mod length ?p)`
      by - (rule ccontr, auto)
    hence "i \<noteq> length ?p - 1"
      by auto
    hence "i < length p - start"
      using `i < length ?p`
      by auto

    have pi: "?p ! i = p ! (start + i - 1)"
      using nth_drop[of start "i - 1" p]
      using `start \<le> length p - 2` `i < length ?p` `i > 0`
      by simp

    have "(i + 1) mod length ?p = i + 1"
      using `i < length p - start`
      by auto
    hence "?p ! ((i + 1) mod length ?p) = ?p ! (i + 1)"
      by simp
    also have "... = p ! (start + i)"
      using nth_drop[of start i p]
      using `i < length p - start`
      by simp
    finally have pi1: "?p ! ((i + 1) mod length ?p) = p ! (start + i)"
      .
    
    have "S \<subseteq> S \<union> {M}"
      by auto
    hence "convex_polygon p"
      using convex_hull_is_convex_polygon[OF in_general_position_mono[OF `S \<subseteq> S \<union> {M}`  `in_general_position (S \<union> {M})`] `convex_hull p S`] 
      using `length p \<ge> 3`
      unfolding convex_polygon_def
      by auto 
      
    let ?blue = "drop start p"
    have "ccw (?blue ! (i - 1)) (?blue ! ((i - 1) + 1)) M"
    proof (cases "length ?blue = 2")
      case True
      thus ?thesis
        using `i < length p - start` `i > 0` `ccw (p ! start) (p ! (start+1)) M` 
        by simp
    next
      case False
      hence "length ?blue \<ge> 3"
        using `start \<le> length p - 2` `length p \<ge> 3`
        by auto
      show ?thesis
      proof (rule blue_consecutive[rule_format])
        show "convex_polygon ?blue"
          using `convex_polygon p` `length ?blue \<ge> 3`
          unfolding convex_polygon_def
          by auto
      next
        show "ccw (?blue ! 0) (?blue ! 1) M"
          using `i < length p - start` `i > 0` `ccw (p ! start) (p ! (start + 1)) M`
          by simp
      next
        show "ccw (?blue ! (length ?blue - 2)) (?blue ! (length ?blue - 1)) M"
          using `i < length p - start` `i > 0` `ccw ?tk ?t M`
          by simp
      next
        show "i - 1 < length ?blue - 1"
          using `i < length p - start` `i > 0` 
          by simp
      next
        show "ccw (?blue ! 0) (?blue ! (length ?blue - 1)) M"
        proof-
          have "?t \<in> S" "(p ! 0) \<in> S"  "?t \<noteq> p ! 0"
            using `length p \<ge> 3` `convex_hull p S` nth_eq_iff_index_eq[of p]
            unfolding convex_hull_def
            by auto
          hence "?t \<noteq> M" "p ! 0 \<noteq> M"
            using `M \<notin> S`
            by auto
          have "\<not> col ?t M (p ! 0)"
            using `in_general_position (S \<union> {M})` `?t \<in> S` `p ! 0 \<in> S` `?t \<noteq> p ! 0` `?t \<noteq> M` `p ! 0 \<noteq> M` 
            unfolding in_general_position_def
            by blast      
      
          have "ccw ?t M (p ! start)"
          proof (cases "start = 0")
            case True
            thus ?thesis
              using `\<not> ccw ?t (p ! 0) M` `?t \<noteq> p ! 0` `?t \<noteq> M` `p ! 0 \<noteq> M` `start = 0` `\<not> col ?t M (p ! 0)`
              using ax2'[of ?t "(p ! 0)" M]
              by simp
          next
            case False
            let ?red = "?t # take (start+1) p"
            have "ccw (?red ! 0) M (?red ! (length ?red - 1))"
            proof (cases "length ?red = 2")
              case True
              have "ccw (p ! (length p - 1)) M (p ! 0)"
              proof (rule ax2')
                show "\<not> ccw (p ! (length p - 1)) (p ! 0) M"
                  by fact
              next
                have *: "p ! (length p - 1) \<in> S" "p ! 0 \<in> S" "p ! (length p - 1) \<noteq> p ! 0"
                  using `convex_hull p S` nth_eq_iff_index_eq[of p] `length p \<ge> 3`
                  unfolding convex_hull_def
                  by auto
                moreover
                hence **: "p ! (length p - 1) \<noteq> M" "p ! 0 \<noteq> M"
                  using `M \<notin> S`
                  by auto
                ultimately
                have "\<not> col (p ! (length p - 1)) M (p ! 0)"
                  using `in_general_position (S \<union> {M})`
                  unfolding in_general_position_def
                  by blast
                thus "p ! (length p - 1) \<noteq> M \<and> p ! (length p - 1) \<noteq> p ! 0 \<and> M \<noteq> p ! 0 \<and> \<not> col (p ! (length p - 1)) M (p ! 0)"
                  using * **
                  by simp
              qed
              thus ?thesis
                using `start \<le> length p - 2` `length p \<ge> 3` `length ?red = 2`
                by (simp add: min_def split: split_if_asm)
            next
              case False
              hence "length ?red \<ge> 3"
                using `length p \<ge> 3`
                by simp
              show ?thesis
              proof (rule red_consecutive)
                show "convex_polygon ?red"
                unfolding convex_polygon_def
                proof safe
                  fix i j k
                  assume "i < j" "j < k" "k < length ?red"
                  show "ccw (?red ! i) (?red ! j) (?red ! k)"
                  proof (cases "i = 0")
                    case True
                    have "ccw (p ! (j - 1)) (p ! (k - 1)) (p ! (length p - 1))"
                      using `i < j` `j < k` `k < length ?red` `start \<le> length p - 2` `length p \<ge> 3`
                      using `convex_polygon p`
                      unfolding convex_polygon_def
                      by (auto simp add: min_def split: split_if_asm) (erule_tac x="j-1" in allE, erule_tac x="k-1" in allE, erule_tac x="length p - 1" in allE, auto)
                    hence "ccw (?red ! j) (?red ! k) (?red ! i)"
                      using `i < j` `j < k` `k < length ?red` `i = 0`
                      by simp
                    thus ?thesis
                      using ax1 
                      by blast
                  next
                    case False
                    have "ccw  (p ! (i-1)) (p ! (j - 1)) (p ! (k - 1))"
                      using `i \<noteq> 0` `i < j` `j < k` `k < length ?red`
                      using `convex_polygon p`
                      unfolding convex_polygon_def
                      by auto (erule_tac x="i-1" in allE, erule_tac x="j-1" in allE, erule_tac x="k-1" in allE, auto)
                    thus ?thesis
                      using `i \<noteq> 0` `i < j` `j < k` `k < length ?red` `start \<le> length p - 2` `length p \<ge> 3`
                      by simp
                    qed
                  next
                    show "length ?red \<ge> 3"
                      by fact
                  qed
                next
                  show "\<forall>i<length ?red - 1.  ccw (?red ! i) M (?red ! (i + 1))"
                  proof safe
                    fix i
                    assume "i < length ?red - 1"
                    hence "i < length p" "i < start + 1"
                      by simp_all
                    show "ccw (?red ! i) M (?red ! (i + 1))"
                    proof (cases "i = 0")
                      case True
                      have "ccw ?t M (p ! 0)"
                        using `\<not> ccw ?t (p ! 0) M` `?t \<noteq> p ! 0` `?t \<noteq> M` `p ! 0 \<noteq> M` `\<not> col ?t M (p ! 0)`
                        using ax2'[of ?t "(p ! 0)" M]
                        by simp
                      thus ?thesis
                        using `i = 0`
                        by simp
                    next
                      case False
                      have "p ! (i-1) \<in> S" "p ! i \<in> S" "p ! i \<noteq> p ! (i-1)"
                        using `convex_hull p S` nth_eq_iff_index_eq[of p] `i \<noteq> 0` `i < length p`
                        unfolding convex_hull_def
                        by auto
                      moreover
                      hence "p ! (i-1) \<noteq> M" "p ! i \<noteq> M"
                        using `M \<notin> S`
                        by auto
                      ultimately
                      have "\<not> col (p ! (i-1)) M (p ! i)"
                        using `in_general_position (S \<union> {M})`
                        unfolding in_general_position_def
                        using singletonI by force
                      have "\<not> ccw (p ! (i - 1)) (p ! i) M"
                        using `\<forall> i < start. \<not> ccw (p ! i) (p ! (i+1)) M` `i < start + 1` `i \<noteq> 0`
                        by auto (erule_tac x="i-1" in allE, auto)
                      hence "ccw (p ! (i - 1)) M (p ! i)"
                        using ax2'[of "p ! (i - 1)" "p ! i" M]
                        using `p ! (i - 1) \<noteq> M` `p ! i \<noteq> p ! (i-1)` `p ! i \<noteq> M` `\<not> col (p ! (i - 1)) M (p ! i)`
                        by simp
                      thus ?thesis
                        using `start \<le> length p - 2` `length p \<ge> 3` `i < length p` `i < start + 1` `i \<noteq> 0`
                        by auto
                    qed
                  qed
                qed
              qed
              thus "ccw ?t M (p ! start)"
                using `length p \<ge> 3` `start \<le> length p - 2` `start \<noteq> 0`
                by (simp add: min_def split: split_if_asm)         
            qed
            hence "ccw (p ! start) ?t M"
              using ax1
              by blast
            thus ?thesis
              using `i < length p - start` `i > 0` 
              by simp
          qed
       qed
    qed
    hence "ccw (p ! (start + i - 1)) (p ! (start + i)) M"
      using `i < length p - start` `i > 0`
      by simp
    thus "ccw (?p ! i) (?p ! ((i + 1) mod length ?p)) M"
      using pi pi1 ax1
      by simp
  qed
  thus ?thesis
    by blast
qed

lemma ex_switch:
assumes "P (0::nat)" "\<exists> i < n. \<not> P i" "n > 0"
  shows "\<exists> k < n - 1. P k \<and> \<not> P (k+1)"
proof-
  let ?K = "{k. k < n \<and> \<not> P k}"
  let ?k = "Min ?K"

  obtain i where "i < n" "\<not> P i" "i > 0"
    using assms
    by auto
  
  hence "i \<in> ?K"
    by auto
  hence "?K \<noteq> {}"
    by auto
  hence "?k \<in> ?K"
    using Min_in[OF _ `?K \<noteq> {}`]
    by simp

  have "0 \<notin> ?K"
    using `P 0`
    by auto
  hence "?k > 0"
    using `?k \<in> ?K`
    by - (rule ccontr, simp)

  have "?k < n" 
    using Min_le[OF _ `i \<in> ?K`] `i < n`
    by auto
    
  have "?k - 1 \<notin> ?K"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "?k - 1 \<in> ?K"
      by simp
    hence "?k \<le> ?k - 1" 
      using Min_le[of ?K "?k - 1"]
      by auto
    thus False
     using `?k > 0`
     by simp
  qed
    
  show ?thesis
    using `?k \<in> ?K` `?k > 0` `?k < n` `n > 0` `?k - 1 \<notin> ?K`
    by (rule_tac x="?k-1" in exI) auto
qed

lemma ex_convex_hull:
assumes "finite S" "in_general_position S"
shows "\<exists> p. convex_hull p S"
using assms
proof (induct S)
  case empty
  thus ?case
    using convex_hull_empty[of "[]" "{}"] convex_hull_0
    by auto
next
  case (insert M S')
  show ?case
  proof (cases "S' = {}")
    case True
    thus ?thesis
     by (rule_tac x="[M]" in exI) (simp add: convex_hull_1)
  next
    case False
    
    have "card S' > 0"
      using `S' \<noteq> {}` `finite S'`
      using card_eq_0_iff[of S']
      by auto
    
    obtain p' where "convex_hull p' S'"
      using insert `S' \<noteq> {}` in_general_position_mono[of S' "S' \<union> {M}"]
      by auto
    show ?thesis
    proof (cases "\<forall> i. i < length p' \<longrightarrow> ccw (p' ! i ) (p' ! ((i + 1) mod (length p'))) M")
      case True
      hence "convex_hull p' (S' \<union> {M})"
        using `convex_hull p' S'` `S' \<noteq> {}`
        unfolding convex_hull_def
        by auto
      thus ?thesis
        by auto
    next
      case False
      then obtain i where "i < length p'" "\<not> (ccw (p' ! i) (p' ! ((i + 1) mod length p')) M)"
        by auto
      show ?thesis
      proof (cases "length p' \<ge> 3")
         case True
         show ?thesis
         proof-
           have "\<exists> j < length p'. ccw (p' ! j) (p' ! ((j + 1) mod length p')) M"
           proof (rule ccontr)
             assume "\<not> ?thesis"
             hence *: "\<forall> j < length p'. \<not> ccw (p' ! j) (p' ! ((j + 1) mod length p')) M"
               by simp
             have *: "\<forall> j < length p'. ccw (p' ! ((j + 1) mod length p')) (p' ! j) M"
             proof safe
               fix j
               assume "j < length p'"
               let ?pj = "p' ! j" and ?pj1 = "p' ! ((j + 1) mod length p')"
               have "\<not> ccw ?pj ?pj1 M"
                 using * `j < length p'`
                 by simp
               show "ccw ?pj1 ?pj M"
               proof (rule ax1[OF ax1[OF ax2'[of ?pj ?pj1 M, OF `\<not> ccw ?pj ?pj1 M`]]])
                 have "j \<noteq> Suc j mod length p'"
                   using `length p' \<ge> 3` `j < length p'`
                   by (metis Divides.mod_less Suc_leD Suc_lessI Suc_n_not_le_n mod_self n_not_Suc_n numeral_3_eq_3)
                 hence "?pj \<noteq> ?pj1" "?pj \<in> S'" "?pj1 \<in> S'"
                   using `convex_hull p' S'` `length p' \<ge> 3` nth_eq_iff_index_eq[of p'] `j < length p'`
                   unfolding convex_hull_def Let_def
                   by auto
                 moreover
                 hence "?pj \<noteq> M" "?pj1 \<noteq> M" 
                   using `M \<notin> S'`
                   by auto
                 ultimately
                 have "\<not> col ?pj M ?pj1"
                   using `in_general_position (insert M S')`
                   unfolding in_general_position_def
                   by blast
                 thus "?pj \<noteq> M \<and> ?pj \<noteq> ?pj1 \<and> M \<noteq> ?pj1 \<and> \<not> col ?pj M ?pj1"
                   using `?pj \<noteq> M` `?pj \<noteq> ?pj1` `?pj1 \<noteq> M`
                   by simp
               qed
             qed
             let ?K = "{k. k < length p' \<and> ccw (p' ! k) (p' ! 0) M}"
             let ?k = "Max ?K"
             
             have "1 \<in> ?K"
               using *[rule_format, of 0] `length p' \<ge> 3`
               by auto (cases p', auto)
             hence "?K \<noteq> {}"
               by auto
             hence "?k \<in> ?K"
               using Max_in[of ?K]
               by simp
             
             have "?k + 1 \<notin> ?K"
             proof (rule ccontr)
               assume "\<not> ?thesis"
               hence "?k + 1 \<le> ?k"
                 using Max_ge[of ?K "?k+1"]
                 by auto
               thus False
                 by simp
             qed
             have "Suc (length p' - Suc 0) = length p'"
               using `length p' \<ge> 3`
               by simp
             hence "length p' - 1 \<notin> ?K"
               using *[rule_format, of "length p' - 1"]
               apply auto
               using ax2 ax1
               by blast
             hence "?k < length p' - 1"
               using `?k \<in> ?K`
               by (cases "?k = length p' - 1") auto
             have "?k \<ge> 1"
               using `1 \<in> ?K` Max_ge[of ?K 1]
               by simp
              have "\<not> ccw (p' ! (?k+1)) (p' ! 0) M"
                using `?k+1 \<notin> ?K` `?k < length p' - 1`
                by simp
              have "ccw (p' ! (?k+1)) M (p' ! 0)"
              proof (rule ax2'[OF `\<not> ccw (p' ! (?k+1)) (p' ! 0) M`])
                obtain k where "k = ?k" "k < length p' - 1" 
                  using `?k < length p' - 1` 
                  by auto 
                have "p' ! 0 \<noteq> p' ! (k+1)" "p' ! 0 \<in> S'" "p' ! (k + 1) \<in> S'"
                  using `convex_hull p' S'` `k < length p' - 1` `length p' \<ge> 3` nth_eq_iff_index_eq[of p']
                  unfolding convex_hull_def
                  by auto
                moreover
                hence "p' ! 0 \<noteq> M" "p' ! (k+1) \<noteq> M"
                  using `M \<notin> S'`
                  by auto
                ultimately
                have "\<not> col (p' ! (k+1)) M (p' ! 0)"
                  using `in_general_position (insert M S')`
                  unfolding in_general_position_def
                  using singletonI by force
                thus "p' ! (?k + 1) \<noteq> M \<and> p' ! (?k + 1) \<noteq> p' ! 0 \<and> M \<noteq> p' ! 0 \<and> \<not> col (p' ! (?k + 1)) M (p' ! 0)"
                  using `k = ?k` `p' ! (k+1) \<noteq> M` `p' ! 0 \<noteq> p' ! (k+1)` `p' ! 0 \<noteq> M`
                  by simp
              qed
              have "ccw (p' ! (?k+1)) (p' ! ?k) (p' ! 0)"
              proof (rule ax4)
                have "ccw (p' ! ?k) (p' ! 0) M"
                  using `?k \<in> ?K`
                  by simp
                thus "ccw M (p' ! ?k) (p' ! 0)"
                  using ax1
                  by blast
              next
                show "ccw (p'!(?k+1)) M (p' ! 0)"
                  by fact
              next
                show "ccw (p'!(?k+1)) (p'!?k) M"
                  using *[rule_format, of ?k] `?k < length p' - 1`
                  by auto
              qed
              hence "ccw  (p' ! 0) (p' ! (?k+1)) (p' ! ?k)"
                using ax1
                by blast
              hence "\<not> ccw  (p' ! 0) (p' ! ?k) (p' ! (?k+1))"
                using ax2
                by blast
              moreover
              have "convex_polygon p'"
                using convex_hull_is_convex_polygon[OF in_general_position_mono[OF _ `in_general_position (insert M S')`] `convex_hull p' S'`] `length p' \<ge> 3`
                by auto
              hence "ccw (p' ! 0) (p' ! ?k) (p' ! (?k+1))"
                unfolding convex_polygon_def
                using `?k \<ge> 1` `?k < length p' - 1`
                by simp
              ultimately
              show False
                by simp
           qed
           then obtain j where "j < length p'" "ccw (p' ! j) (p' ! ((j + 1) mod length p')) M"
             by auto
           let ?ll = "drop j p' @ take j p'"
           let ?P = "\<lambda> k. ccw (?ll ! k) (?ll ! ((k + 1) mod (length ?ll))) M"
           have "\<exists> k < length ?ll - 1. ?P k \<and> \<not> ?P (k+1)"
           proof (rule ex_switch)
             have "Suc (length p' - Suc 0) = length p'"
               using `length p' \<ge> 3`
               by auto
             thus "ccw (?ll ! 0) (?ll ! ((0 + 1) mod length ?ll)) M"
               using `length p' \<ge> 3` `j < length p'` `ccw (p' ! j) (p' ! ((j + 1) mod length p')) M`
               by (auto simp add: nth_append) (cases "j = length p' - 1", auto)
           next
             show "\<exists>i<length ?ll. \<not> ccw (?ll ! i) (?ll ! ((i + 1) mod length ?ll)) M"
             proof (cases "i < j")
               case True
               hence "?ll ! (length p' - j + i) = p' ! i"
                  using `j < length p'` `length p' \<ge> 3` `i < length p'`
                  by (auto simp add: nth_append)
               moreover
               have "Suc (length p' - Suc 0) = length p'"
                 using `length p' \<ge> 3` 
                 by auto
               hence "?ll ! (((length p' - j + i) + 1) mod (length p')) = p' ! ((i + 1) mod (length p'))"
                  using `j < length p'` `length p' \<ge> 3` `i < length p'` `i < j`
                  by (cases "i+1=j") (auto simp add: nth_append)
               moreover
               have "length p' - j + i < length p'" "length ?ll = length p'"
                  using `j < length p'` `length p' \<ge> 3` `i < length p'` `i < j`
                  by auto
               ultimately
               show ?thesis
                 using `\<not> ccw (p' ! i) (p' ! ((i + 1) mod (length p'))) M`
                 by (rule_tac x="length p' - j + i" in exI, simp) 
           next
             case False
             hence "i > j"
               using `\<not> ccw (p' ! i) (p' ! ((i + 1) mod (length p'))) M` `ccw (p' ! j) (p' ! ((j + 1) mod (length p'))) M`
               by (cases "i = j") auto
             have "?ll ! ((i - j) mod (length p')) = p' ! i"
                  using `j < length p'` `length p' \<ge> 3` `i < length p'` `j < i`
                  by (auto simp add: nth_append)
             moreover
             have "Suc (length p' - Suc 0) = length p'" "Suc (length p' - Suc j) = length p' - j"
               using `length p' \<ge> 3` `j < length p'`
               by auto
             hence "?ll ! ((i - j + 1) mod (length p')) = p' ! ((i + 1) mod (length p'))"
                  using `j < length p'` `length p' \<ge> 3` `i < length p'` `j < i`
                  apply (cases "i = length p' - 1")
                  apply (auto simp add: nth_append)
                  apply (cases "j = 0", auto)+
                  done
             moreover     
             have "i - j < length p'" "length ?ll = length p'"
                using `j < length p'` `length p' \<ge> 3` `i < length p'` `j < i`
                by auto
             ultimately
               show ?thesis
                 using `\<not> ccw (p' ! i) (p' ! ((i + 1) mod (length p'))) M`
                 by (rule_tac x="i-j" in exI, simp) 
             qed
           next
             show "length ?ll > 0"
               using `length p' \<ge> 3`
               by auto
           qed
           then obtain k where "k < length ?ll - 1" "?P k \<and> \<not> ?P (k+1)"
             using `length p' \<ge> 3` `j < length p'`
             by blast
             
           have "cyclic_perm p' ?ll"
             using `j < length p'`
             unfolding cyclic_perm_def 
             by auto
             
           have "\<exists> p'' k. cyclic_perm p' p'' \<and> k < length p'' - 2 \<and> ccw (p'' ! k) (p'' ! (k+1)) M \<and> \<not> (ccw (p'' ! (k+1)) (p'' ! (k+2)) M)"
           proof (cases "k < length ?ll - 2")
             case True
             hence "?ll ! ((k+1) mod (length ?ll)) = ?ll ! (k + 1)" "?ll ! ((k+2) mod (length ?ll)) = ?ll ! (k + 2)"
               by simp_all
             thus ?thesis
               using `cyclic_perm p' ?ll` `?P k \<and> \<not> ?P (k+1)` `k < length ?ll - 2` 
               by (rule_tac x="?ll" in exI, simp, rule_tac x="k" in exI, simp)
           next
             case False
             obtain ll where "ll = ?ll"
               by auto
             have "k = length ll - 2"
               using `k < length ?ll - 1` `\<not> k < length ?ll - 2` `ll = ?ll`
               by simp
             have "length ll = length p'"
                using  `length p' \<ge> 3` `ll = ?ll`
                by auto
               
             have "Suc (Suc (length p' - 2)) = length p'"
                using  `length p' \<ge> 3`
                by auto
             hence "ll ! ((k + 1) mod (length ll)) = ll ! (k + 1)" "ll ! ((k + 2) mod (length ll)) = ll ! 0"
               using `j < length p'` `length p' \<ge> 3` `k = length ll - 2` `length ll = length p'`
               by simp_all
             obtain k' pp'' where *: "pp'' = drop 1 ll @ take 1 ll" "k' = length pp'' - 3" 
               by auto
             have "Suc (length p' - Suc (Suc (Suc 0))) = length p' - 2"
               using `length p' \<ge> 3`
               by auto
             hence  "pp'' ! k' = ll ! k"  "pp'' ! (k' + 1) = ll ! (k + 1)" "pp'' ! (k' + 2) = ll ! 0"
               using * `length ll = length p'` `length p' \<ge> 3` `k = length ll - 2`
               by (auto simp add: min_def nth_append numeral_2_eq_2)
               
             hence "ccw (pp'' ! k') (pp'' ! (k'+1)) M" "\<not> ccw (pp'' ! (k'+1)) (pp'' ! (k'+2)) M"
               using `?P k \<and> \<not> ?P (k+1)`
               using `ll ! ((k + 1) mod (length ll)) = ll ! (k + 1)` `ll ! ((k + 2) mod (length ll)) = ll ! 0`
               using `ll = ?ll`
               by auto
             moreover
             have "cyclic_perm ll pp''"
               using `pp'' = drop 1 ll @ take 1 ll` `length ll = length p'` `length p' \<ge> 3` 
               unfolding cyclic_perm_def
               by - (rule disjI2, rule_tac x=1 in exI, simp)
             hence "cyclic_perm p' pp''"
               using `cyclic_perm p' ?ll` `ll = ?ll` cyclic_perm_trans
               by blast
             moreover
             have "k' < length pp'' - 2"
               using `k' = length pp'' - 3` `length ll = length p'` `pp'' = drop 1 ll @ take 1 ll` `length p' \<ge> 3`
               by auto
             ultimately
             show ?thesis
               by force
           qed
           
           then obtain k p'' where "cyclic_perm p' p''" "k < length p'' - 2" "ccw (p'' ! k) (p'' ! (k+1)) M" "\<not> (ccw (p'' ! (k+1)) (p'' ! (k+2)) M)"
             by auto
             
           have "length p'' = length p'"
             using cyclic_perm_length[OF `cyclic_perm p' p''`]
             by auto

           let ?p = "drop (k+2) p'' @ take (k+2) p''"
           have "cyclic_perm p'' ?p"
             using `k < length p'' - 2` `length p' \<ge> 3` `length p'' = length p'`
             unfolding cyclic_perm_def
             by - (rule disjI2, rule_tac x="k+2" in exI, auto)
           have "\<exists>p. convex_hull p (S' \<union> {M})"
           proof (rule ex_convex_hull')
             show "convex_hull ?p S'"
               using `cyclic_perm p'' ?p` cyclic_perm_convex_hull[of p'' ?p S'] 
               using `cyclic_perm p' p''` cyclic_perm_convex_hull[of p' p'' S'] `convex_hull p' S'`
               by simp
           next
             have "length p' + k - Suc (Suc (length p' - 2)) = k"
               using `length p' \<ge> 3`
               by auto
             hence "?p ! (length ?p - 2) = p'' ! k"
               using `k < length p'' - 2` `length p' \<ge> 3` `length p'' = length p'`
               by (auto simp add: min_def nth_append)

             have "?p ! (length ?p - 1) = p'' ! (k+1)"
               using `k < length p'' - 2` `length p' \<ge> 3` `length p'' = length p'`
               by (auto simp add: min_def nth_append)
               
             have "?p ! 0 = p'' ! (k + 2)"
               using `k < length p'' - 2` `length p' \<ge> 3` `length p'' = length p'`
               by (auto simp add: min_def nth_append)

             show "ccw (?p ! (length ?p - 2)) (?p ! (length ?p - 1)) M"
               using `ccw (p'' ! k) (p'' ! (k+1)) M` `?p ! (length ?p - 2) = p'' ! k` `?p ! (length ?p - 1) = p'' ! (k+1)`
               by simp
             
             show "\<not> ccw (?p ! (length ?p - 1)) (?p ! 0) M"
               using `\<not> ccw (p'' ! (k+1)) (p'' ! (k+2)) M`  `?p ! (length ?p - 1) = p'' ! (k+1)`  `?p ! 0 = p'' ! (k+2)`
               by simp
           next
              show "M \<notin> S'" "3 \<le> length (drop (k + 2) p'' @ take (k + 2) p'')" "in_general_position (S' \<union> {M})"
                using `M \<notin> S'` `length p' \<ge> 3` `in_general_position (insert M S')` `length p'' = length p'`
                by auto
           qed
           thus ?thesis
             by auto
         qed
      next
         case False
         show ?thesis
         proof (cases "length p' = 1")
           case True
           then obtain A where "p' = [A]"
             using length1
             by auto
           hence "card S' = 1"
           using `convex_hull p' S'` convex_hull_card_gt_2[of p' S'] convex_hull_card_lt_3[of S' p'] `card S' > 0` `finite S'`
             by (cases "card S' > 2") auto
           hence "S' = {A}"
             using card1[of S']
             using `convex_hull p' S'` `p' = [A]`
             unfolding convex_hull_def
             by auto
           hence "convex_hull (M # p') (S' \<union> {M})"
             using `p' = [A]` convex_hull_2[of M A]  `M \<notin> S'`
             by auto
           thus ?thesis
             by auto
         next
           case False
           have "length p' > 0"
             using `convex_hull p' S'` `S' \<noteq> {}`
             unfolding convex_hull_def
             by auto
           hence "length p' = 2"
             using `\<not> length p' \<ge> 3` `length p' \<noteq> 1`
             by linarith
           then obtain A B where "p' = [A, B]"
             using length2[of p']
             by auto                          
           hence "card S' = 2"
             using `convex_hull p' S'` convex_hull_card_lt_3[of S' p'] convex_hull_card_gt_2[of p' S'] `card S' > 0` `finite S'`
             by (cases "card S' > 2") auto
           hence "S' = {A, B}"
             using card2[of S']
             using `convex_hull p' S'` `p' = [A, B]`
             unfolding convex_hull_def
             by auto
             
           have "i = 0 \<or> i = 1"
             using `i < length p'`  `p' = [A, B]`
             by auto
           hence "\<not> ccw A B M \<or> \<not> ccw B A M"
             using `\<not> (ccw (p' ! i) (p' ! ((i + 1) mod length p')) M)` `p' = [A, B]`
             by auto
             
           have "A \<noteq> B"
             using `p' = [A, B]` `convex_hull p' S'`
             unfolding convex_hull_def
             by auto
           hence "distinct [A, B, M]"   
             using `M \<notin> S'` `S' = {A, B}` `A \<noteq> B`
             by auto

           have "in_general_position (set [M, A, B])"
             using `in_general_position (insert M S')` `S' = {A, B}`
             by simp
           hence "\<not> col B M A" "\<not> col A M B" "A \<noteq> M" "B \<noteq> M" "A \<noteq> B"
             using `distinct [A, B, M]`
             unfolding in_general_position_def
             by auto
           hence "ccw A B M \<or> ccw B A M"
             using ax1[OF ax1[OF ax2'[of A B M]]]
             using ax1[OF ax1[OF ax2'[of B A M]]]
             using `\<not> ccw A B M \<or> \<not> ccw B A M`
             by blast
           hence "convex_polygon [A, B, M] \<or> convex_polygon [B, A, M]"
           proof
             assume "ccw A B M"
             hence "convex_polygon [A, B, M]"
               unfolding convex_polygon_def
             proof safe
               show "length [A, B, M] \<ge> 3"
                 by simp
             next
               fix p q r
               assume "p < q" "q < r" "r < length [A, B, M]"
               hence "p = 0" "q = 1" "r = 2"
                 by auto
               thus "ccw ([A, B, M] ! p) ([A, B, M] ! q) ([A, B, M] ! r)"
                 using `ccw A B M`
                 by auto
             qed
             thus ?thesis
               by simp
           next
             assume "ccw B A M"
             hence "convex_polygon [B, A, M]"
               unfolding convex_polygon_def
             proof safe
               show "length [B, A, M] \<ge> 3"
                 by auto
             next
               fix p q r
               assume "p < q" "q < r" "r < length [B, A, M]"
               hence "p = 0" "q = 1" "r = 2"
                 by auto
               thus "ccw ([B, A, M] ! p) ([B, A, M] ! q) ([B, A, M] ! r)"
                 using `ccw B A M`
                 by auto
             qed
             thus ?thesis
               by simp
           qed
           hence "convex_hull [A, B, M] (S' \<union> {M}) \<or> convex_hull [B, A, M] (S' \<union> {M})"
           using `p' = [A, B]`  `M \<notin> S'`  convex_hull_polygon[of "[A, B, M]"]  convex_hull_polygon[of "[B, A, M]"] `S' = {A, B}`
             by (auto simp add: insert_commute)
           thus ?thesis
             by auto          
         qed
      qed
    qed
  qed
qed


(* nested hulls *)

inductive nested_hulls :: "'p list list \<Rightarrow> bool" where
  base: "nested_hulls []"
| ind: "\<lbrakk>nested_hulls H; p \<noteq> []; set p \<inter> set (concat H) = {}; convex_hull p (set p \<union> set (concat H))\<rbrakk> \<Longrightarrow> nested_hulls (p # H)"

lemma nested_hulls_1: "nested_hulls [[A]]"
proof-
  have "nested_hulls ([A] # [])"
  proof
    show "nested_hulls []"
      using base
      by auto
  next
    show "convex_hull [A] (set [A] \<union> set (concat []))"
      using convex_hull_1
      by auto
  qed auto
  thus ?thesis
    by simp
qed

lemma 
nested_hulls_2: "A \<noteq> B \<Longrightarrow> nested_hulls [[A, B]]"
proof-
  assume "A \<noteq> B"
  have "nested_hulls ([A, B] # [])"
  proof
    show "nested_hulls []"
      by (rule nested_hulls.intros)
  next
    show "convex_hull [A, B] (set [A, B] \<union> set (concat []))"
      using `A \<noteq> B` convex_hull_2
      by auto
  qed auto
  thus ?thesis
    by simp
qed

lemma nested_hulls_convex_polygon:
assumes "convex_polygon p"
shows "nested_hulls [p]"
proof-
  have "nested_hulls (p # [])"
  proof
    show "nested_hulls []"
      by (rule nested_hulls.intros)
  next
    show "convex_hull p (set p \<union> set (concat []))"
      using convex_hull_polygon[OF assms(1)]
      by simp
  next
    show "p \<noteq> []"
      using `convex_polygon p` 
      unfolding convex_polygon_def
      by auto
  qed auto
  thus ?thesis
    by simp
qed

lemma ex_nested_hulls: 
 assumes "finite S" "in_general_position S"
 shows "\<exists> H. nested_hulls H \<and> set (concat H) = S"
proof-
obtain n where "card S = n" by auto
thus ?thesis
using assms
proof (induct n arbitrary: S rule: nat_less_induct)
  case (1 n)
  show ?case
  proof (cases "n = 0")
    case True
    thus ?thesis
      using base `finite S` `card S = n`
      by auto
  next
    case False
    then obtain p where "convex_hull p S"
      using ex_convex_hull[of S] `finite S` `card S = n` `in_general_position S`
      by force
    moreover
    have "set p \<subseteq> S" "p \<noteq> []"
      using `convex_hull p S` `card S = n` `n \<noteq> 0`
      unfolding convex_hull_def
      by auto
    moreover
    have "card (S - set p) < n"
    proof-
      have "finite (set p)"
        using `finite S` finite_subset
        by auto
      hence "card (S - set p) = card S - card (set p)"
        using card_Diff_subset[of "set p" S]  `set p \<subseteq> S`
        by auto
      moreover
      have "card (set p) > 0"
        using `finite (set p)` `p \<noteq> []`
        by (simp add: card_gt_0_iff)
      ultimately
      show ?thesis
        using `card S = n` `\<not> n = 0`
        by auto
    qed
    then obtain H where "nested_hulls H" "set (concat H) = S - set p"
      using 1(1)[rule_format, of "card (S - set p)" "S - set p"]
      using `finite S` finite_subset[of "S - set p" S] `in_general_position S` in_general_position_mono[of "S - set p" S]
      by auto
    ultimately
    show ?thesis
      using ind[of H p] `set p \<subseteq> S`
      by (rule_tac x="p # H" in exI) (simp add: Un_absorb1)
  qed
qed
qed

lemma  nested_hulls_disjoint:
assumes "nested_hulls (p # H)"
shows "set p \<inter> set (concat H) = {}"
using assms
by (cases, auto)

lemma nested_hulls_tl:
assumes "nested_hulls (p # H)"
shows "nested_hulls H"
using assms
by cases simp

lemma nested_hulls_drop: 
assumes "nested_hulls H" "n < length H"
shows "nested_hulls (drop n H)"
using assms
by (induct H arbitrary: n) (auto simp add: drop_Cons' ind)

lemma nested_hulls_distinct:
assumes "nested_hulls H"
shows "distinct (concat H)"
using assms
by (induct H) (auto simp add: convex_hull_def)

lemma nested_hulls_length:
assumes "nested_hulls H"
shows "length (concat H) = card (set (concat H))"
using nested_hulls_distinct[OF assms]
using distinct_card[of "concat H"]
by auto

lemma nested_hulls_are_convex_hulls:
assumes "nested_hulls (p # H)"
shows "convex_hull p (set (concat (p # H)))"
using assms
by cases simp

lemma nested_hulls_are_nested':
assumes "nested_hulls (p # H)"
shows "\<forall> i. i < length p \<longrightarrow> (\<forall> x \<in> set (concat H). ccw (p ! i) (p ! ((i + 1) mod length p)) x)"
using assms
unfolding convex_hull_def
proof (cases, safe)
  fix i x
  assume "nested_hulls H" "set p \<inter> set (concat H) = {}" "convex_hull p (set p \<union> set (concat H))" "i < length p"
         "x \<in> set (concat H)"
  thus "ccw (p ! i) (p ! ((i + 1) mod length p)) x"
     unfolding convex_hull_def
     by (simp add: Let_def) (cases "p = []",  simp, force simp add: Let_def)
qed

lemma nested_hulls_are_nested:
assumes "nested_hulls H" "n < length H"
shows "\<forall> i. i < length (H ! n) \<longrightarrow> (\<forall> x \<in> set (concat (drop (n + 1) H)). ccw (H ! n ! i) (H ! n ! ((i + 1) mod length (H ! n))) x)"
proof-
  have "nested_hulls (drop n H)"
    using assms nested_hulls_drop
    by auto
  thus ?thesis
    using Cons_nth_drop_Suc[of n H] `n < length H`
    using nested_hulls_are_nested'[of "H ! n" "drop (n+1) H"]
    by simp
qed

lemma nested_hulls_are_convex_polygons:
assumes "nested_hulls H" "p \<in> set H" "in_general_position (set (concat H))" "length p \<ge> 3"
shows "convex_polygon p"
using assms
proof (induct H)
  case base
  thus ?case
    by simp
next
  case (ind H p')
  thus ?case
    using in_general_position_mono[of "set (concat H)" "set (concat (p' # H))"]
    using convex_hull_is_convex_polygon
    by auto
qed

lemma nested_hulls_are_convex_polygons':
assumes "nested_hulls H" "p \<in> set H" "in_general_position (set (concat H))"
shows "\<forall> i j k. 0 \<le> i \<and> i < j \<and> j < k \<and> k < length p \<longrightarrow> ccw (p ! i) (p ! j) (p ! k)"
using nested_hulls_are_convex_polygons[OF assms]
unfolding convex_polygon_def
by auto

lemma nested_hulls_no_Nil:
assumes "nested_hulls H"
shows "[] \<notin> set H"
using assms
by induct auto

lemma cyclic_perm_nested_huls:
assumes "nested_hulls (p # H)" "cyclic_perm p p'"
shows "nested_hulls (p' # H)"
proof (cases "p = []")
  case True
  thus ?thesis
    using assms
    unfolding cyclic_perm_def
    by simp
next
  case False
  hence "p' \<noteq> []"
    using assms
    unfolding cyclic_perm_def
    by auto
  show ?thesis
    using assms ind[of H p'] cyclic_perm_set[of p p'] cyclic_perm_convex_hull[of p p' "set p' \<union> set (concat H)"] `p' \<noteq> []`
    by cases auto
qed     

lemma cyclic_perms_nested_hulls:
assumes "length H' = length H" "\<forall> i < length H. cyclic_perm (H ! i) (H' ! i)" "nested_hulls H"
shows "nested_hulls H' \<and> set (concat H') = set (concat H)"
using assms
proof (induct H arbitrary: H')
  case Nil
  thus ?case
    by auto
next
  case (Cons p h)
  from `length H' = length (p # h)` obtain p' h' where "H' = p' # h'"
    by (cases H') auto
  have *: "nested_hulls h' \<and> set (concat h') = set (concat h)"
  proof (rule Cons(1))
    show "length h' = length h"
      using Cons(2) `H' = p' # h'`
      by simp
  next
    show "nested_hulls h"
      using Cons(4)
      by (rule nested_hulls_tl)
  next    
    show "\<forall>i<length h. cyclic_perm (h ! i) (h' ! i)"
      using Cons(3) `H' = p' # h'`
      by auto
  qed
  
  have "cyclic_perm p p'"
    using Cons(3) `H' = p' # h'`
    by auto
  
  have "nested_hulls (p # h')"
  proof (rule ind)
    show "nested_hulls h'"
      using *
      by simp
  next
    show "set p \<inter> set (concat h') = {}"
      using `nested_hulls (p # h)` *
      by cases auto
  next
    show "convex_hull p (set p \<union> set (concat h'))"
      using `nested_hulls (p # h)` *
      by cases auto
  next
    show "p \<noteq> []"
      using `nested_hulls (p # h)` * `cyclic_perm p p'`
      by cases auto
  qed
  moreover
  have "cyclic_perm p p'"
    using Cons(3)  `H' = p' # h'`
    by auto
  ultimately
  show ?case
    using cyclic_perm_nested_huls[of p h' p'] `H' = p' # h'` * cyclic_perm_set[of p p']
    by auto
qed

(* hull structure *)

fun hull_structure  :: "nat \<Rightarrow> nat \<Rightarrow> nat list list" where
"hull_structure m n = 
    (if n = 0 then [[]]
     else if n < 3 then [[n]]
     else concat (map (\<lambda> k. if n \<ge> k then map (op # k) (hull_structure m (n-k)) else []) [3..<m]))"
declare hull_structure.simps [simp del]

lemma hull_structure:
assumes "nested_hulls H" "finite (set (concat H))"  "card (set (concat H)) = n" 
   "m \<ge> 3" "\<not> contains_convex_polygon m (set (concat H))" "in_general_position (set (concat H))"
shows "map length H \<in> set (hull_structure m n)"
using assms
proof (induct m n arbitrary: H rule: hull_structure.induct)
  case (1 m n)
  show ?case
  proof (cases "n = 0")
    case True
    thus ?thesis
      using hull_structure.simps[of m n] 1(2-4) nested_hulls_no_Nil[of H]
      by (cases H, auto)
  next
    case False

    have "H \<noteq> []"
      using `card (set (concat H)) = n` `n \<noteq> 0`
      by auto

    have "H = hd H # tl H"
      using `H \<noteq> []` 
      by auto
    hence "concat H = (hd H) @ concat (tl H)"
      using concat_append[of "[hd H]" "tl H"]
      by auto
    hence "set (concat H) = set (hd H) \<union> set (concat (tl H))"
      by auto        
      
    have "finite (set (concat (tl H)))"
      using `finite (set (concat H))` `H \<noteq> []`
      by simp

    have "convex_hull (hd H) (set (concat H))"
      using nested_hulls_are_convex_hulls[of "hd H" "tl H"] `H \<noteq> []` `nested_hulls H`
      by simp

    hence "length (hd H) = card (set (hd H))"
      using distinct_card[of "hd H"]  
      unfolding convex_hull_def
      by simp
      
    have "card (set (concat (tl H))) = n - length (hd H)"
    proof-
      have "card (set (concat H)) = card (set (hd H)) + card (set (concat (tl H)))"
      proof-
        have "card (set (hd H) \<union> set (concat (tl H))) = card (set (hd H)) + card (set (concat (tl H)))"
        proof (rule card_Un_disjoint)
          show "set (hd H) \<inter> set (concat (tl H)) = {}"
            using `nested_hulls H` `H \<noteq> []`
            using nested_hulls_disjoint[of "hd H" "tl H"]
            by simp
          show "finite (set (hd H))"
            using `finite (set (concat H))`
            using finite_subset[of "set (hd H)" "set (concat H)"]
            by auto
          show "finite (set (concat (tl H)))"
            by fact
        qed
        thus ?thesis
          using `set (concat H) = set (hd H) \<union> set (concat (tl H))` 
          by simp
      qed
      thus ?thesis
        using `length (hd H) = card (set (hd H))` 
        using `card (set (concat H)) = n`
        by simp
    qed

    show ?thesis
    proof (cases "n < 3")
      case True
      hence "length (hd H) = n"
        using convex_hull_card_lt_3[of "set (concat H)" "hd H"] `n \<noteq> 0` `card (set (concat H)) = n`
        using  `convex_hull (hd H) (set (concat H))`
        by simp
      moreover
      have "length H = 1"
      proof (rule ccontr)
        assume "\<not> ?thesis"
        hence "length H > 1"
          using `H \<noteq> []`
          by (cases H, auto)
        hence "tl H \<noteq> []"
          by (cases H, auto)
        moreover
        have "set (concat (tl H)) \<inter> set (hd H) = {}"
          using `nested_hulls H` nested_hulls_disjoint[of "hd H" "tl H"] `H \<noteq> []`
          by auto
        have "card (set (concat (tl H))) = 0"
          using `card (set (concat (tl H))) = n - length (hd H)` `length (hd H) = n`
          by simp
        thus False
          using `tl H \<noteq> []` `finite (set (concat (tl H)))` `H = hd H # tl H`
          using nested_hulls_no_Nil[of H] `nested_hulls H`
          by auto (metis list.set_intros(2) list.set_sel(1))
      qed
      ultimately
      show ?thesis
        using `\<not> n = 0` `n < 3` card_0_eq[of "set (concat H)"]
        using hull_structure.simps[of m n]
        by (cases H, auto)
    next
      case False
    
      have "length (hd H) \<in> set [3..<m]"
      proof-
        have "length (hd H) < m"
          using `in_general_position (set (concat H))`
          using `\<not> contains_convex_polygon m (set (concat H))`
          using `convex_hull (hd H) (set (concat H))` `m \<ge> 3`
          using convex_hull_length_ub[of "set (concat H)" "hd H" m]
          by simp
        moreover
        have "length (hd H) \<ge> 3"
          using `convex_hull (hd H) (set (concat H))`
          using convex_hull_card_gt_2[of "hd H" "set (concat H)"] `\<not> n < 3` `card (set (concat H)) = n`
          by auto
        ultimately
        show ?thesis
          by auto
       qed

       have "length (hd H) \<le> n"
         using `length (hd H) = card (set (hd H))` 
         using `card (set (concat H)) = n`  `H \<noteq> []` `n \<noteq> 0`
         using card_mono[OF `finite (set (concat H))`, of "set (hd H)"]
         by force
        
      have "map length (tl H) \<in> set (hull_structure m (n - length (hd H)))"
      proof (rule 1(1))
        show "n \<noteq> 0" "\<not> n < 3" 
          by fact+
        
        show "length (hd H) \<in> set [3..<m]" "length (hd H) \<le> n"
          by fact+
        
        show "finite (set (concat (tl H)))"
          by fact
          
        show "card (set (concat (tl H))) = n - length (hd H)"
          by fact

        show "nested_hulls (tl H)"
          using nested_hulls_tl[of "hd H" "tl H"] `H \<noteq> []` `nested_hulls H`
          by simp
        show "\<not> contains_convex_polygon m (set (concat (tl H)))"
          using contains_convex_polygon_mono_set[of m "set (concat (tl H))" "set (concat H)"]
          using `\<not> contains_convex_polygon m (set (concat H))`
          using `set (concat H) = set (hd H) \<union> set (concat (tl H))`
          by auto
        show "in_general_position (set (concat (tl H)))"
          using `in_general_position (set (concat H))`
          using in_general_position_mono[of "set (concat (tl H))" "set (concat H)"]
          using `set (concat H) = set (hd H) \<union> set (concat (tl H))`
          by auto
      next
        show "m \<ge> 3"
          by fact
      qed
      moreover
      have "map length H = length (hd H) # map length (tl H)"
        using `H = hd H # tl H`
        by (metis list.simps(9))
      ultimately
      show ?thesis
        unfolding hull_structure.simps[of m n]
        using `\<not> n = 0` `\<not> n < 3` 
        using `length (hd H) \<in> set [3..<m]` `length (hd H) \<le> n`
        using image_eqI[of "map length H" "op # (length (hd H))" "map length (tl H)" "set (hull_structure m (n - length (hd H)))"]
        by simp (rule_tac x="length (hd H)" in bexI, auto)
    qed
  qed
qed

(* ----------------------------------------------------------------- *)

definition "area_3_1 p q r x y \<longleftrightarrow> \<not> ccw x y p \<and> ccw x y q"
definition "area_3_2 p q r x y \<longleftrightarrow> \<not> ccw x y q \<and> ccw x y r"
definition "area_3_3 p q r x y \<longleftrightarrow> \<not> ccw x y r \<and> ccw x y p"

lemma area_3_cases:
  assumes "ccw p q r" "ccw p q x" "ccw q r x" "ccw r p x" "ccw p q y" "ccw q r y" "ccw r p y" "x \<noteq> y" "in_general_position {p, q, r, x, y}"
  shows "area_3_1 p q r x y \<or> area_3_2 p q r x y \<or> area_3_3 p q r x y"
proof-
  have "distinct [p, q, r, x, y]"
    using assms ax0
    by simp
moreover
  have "ax1_formula [p, q, r, x, y]"
    by (rule ax1_formula)
moreover
  have "ax2_formula [p, q, r, x, y]"
    by (rule ax2_formula)
moreover
  have "ax3_formula [p, q, r, x, y]"
    using ax3_formula[of "[p, q, r, x, y]"]
    using `distinct [p, q, r, x, y]` `in_general_position {p, q, r, x, y}`
    by simp
moreover
  have "ax4_formula [p, q, r, x, y]"
    by (rule ax4_formula)
moreover
  have "ax5_formula [p, q, r, x, y]"
    by (rule ax5_formula)
ultimately
  show ?thesis
    using assms(2-8)
    unfolding area_3_1_def area_3_2_def area_3_3_def
    unfolding ax1_formula_def ax2_formula_def ax3_formula_def ax4_formula_def ax5_formula_def
    by normalization sat
qed

lemma area_32:
  assumes "area_3_2 p q r x y"
  shows "area_3_1 q r p x y"
using assms
unfolding area_3_2_def area_3_1_def
by simp

lemma area_33:
  assumes "area_3_3 p q r x y"
  shows "area_3_1 r p q x y"
using assms
unfolding area_3_3_def area_3_1_def
by simp

lemma cyclic_perm_n:
  assumes "l' = drop n l @ take n l" "n < length l"
  shows "cyclic_perm l l'"
  using assms
  unfolding cyclic_perm_def
  by auto

lemma area_3_1:
assumes "ccw p q r" "ccw p q x" "ccw q r x" "ccw r p x" "ccw p q y" "ccw q r y" "ccw r p y" "x \<noteq> y" and gp: "in_general_position {p, q, r, x, y}"
shows "\<exists> p' q' r'. cyclic_perm [p, q, r] [p', q', r'] \<and> area_3_1 p' q' r' x y"
proof-
  have "distinct [p, q, r]"
    using `ccw p q r` ax0
    by simp
  have "x \<notin> set [p, q, r]"
    using assms ax0
    by simp blast
  have "y \<notin> set [p, q, r]"  
    using assms ax0
    by simp blast

  note * = `distinct [p, q, r]` `x \<notin> set [p, q, r]` `y \<notin> set [p, q, r]` `x \<noteq> y`

  {
     assume "area_3_1 p q r x y"
     hence ?thesis
        unfolding cyclic_perm_def
        by force
  }
  moreover
  {
     assume "area_3_2 p q r x y"
     moreover
     have "cyclic_perm [p, q, r] [q, r, p]"
       unfolding cyclic_perm_def
       by auto
     ultimately
     have ?thesis
       using area_32[of p q r x y] *
       by auto
  }
  moreover 
  {
     assume "area_3_3 p q r x y"
     moreover
     have "cyclic_perm [p, q, r] [r, p, q]"
       unfolding cyclic_perm_def
       by auto
     ultimately
     have ?thesis
        using area_33[of p q r x y] *
        by auto
  }
  ultimately
  show ?thesis
    using area_3_cases[OF assms]
    by blast
qed

lemma nested_hulls_3_1:
assumes "nested_hulls ([p1, q1, r1] # (p2 # q2 # pp) # H)" and gp: "in_general_position (set (concat ([p1, q1, r1] # (p2 # q2 # pp) # H)))" (is "in_general_position ?S") 
shows "\<exists> p1' q1' r1'. nested_hulls ([p1', q1', r1'] # (p2 # q2 # pp) # H) \<and> 
        set (concat ([p1, q1, r1] # (p2 # q2 # pp) # H)) = set (concat ([p1', q1', r1'] # (p2 # q2 # pp) # H)) \<and>
        area_3_1 p1' q1' r1' p2 q2"
proof-
  have "convex_polygon [p1, q1, r1]"
    using convex_hull_is_convex_polygon[OF gp nested_hulls_are_convex_hulls[OF assms(1)]]
    by simp
  hence "ccw p1 q1 r1"
    unfolding convex_polygon_def
    by force
  moreover
  have "ccw p1 q1 p2" "ccw q1 r1 p2" "ccw r1 p1 p2"
    using nested_hulls_are_convex_hulls[OF assms(1)]
    using nested_hulls_disjoint[OF assms(1)]
    unfolding convex_hull_def Let_def
    apply simp_all
    apply ((erule conjE)+, erule_tac x=0 in allE, simp)
    apply ((erule conjE)+, erule_tac x=1 in allE, simp)
    apply ((erule conjE)+, erule_tac x=2 in allE, simp)
    done
  moreover
  have "ccw p1 q1 q2" "ccw q1 r1 q2" "ccw r1 p1 q2"
    using nested_hulls_are_convex_hulls[OF assms(1)]
    using nested_hulls_disjoint[OF assms(1)]
    unfolding convex_hull_def Let_def
    apply simp_all
    apply ((erule conjE)+, erule_tac x=0 in allE, simp)
    apply ((erule conjE)+, erule_tac x=1 in allE, simp)
    apply ((erule conjE)+, erule_tac x=2 in allE, simp)
    done
  moreover
  have "in_general_position {p1, q1, r1, p2, q2}"
    by (rule in_general_position_mono[OF _ gp], simp)
  moreover
  have "p2 \<noteq> q2"
    using nested_hulls_are_convex_hulls[OF nested_hulls_tl[OF assms(1)]]
    unfolding convex_hull_def
    by simp
  ultimately
  obtain p1' q1' r1' where
  *: "cyclic_perm [p1, q1, r1] [p1', q1', r1']" and "area_3_1 p1' q1' r1' p2 q2"
    unfolding convex_polygon_def
    using area_3_1[of p1 q1 r1 p2 q2]
    by blast
  thus ?thesis
    using cyclic_perm_nested_huls[OF _ *, OF assms(1)]
    using cyclic_perm_set[OF *]
    by (rule_tac x="p1'" in exI, rule_tac x="q1'" in exI, rule_tac x="r1'" in exI, auto)
qed

(* ----------------------------------------------------------------- *)

definition "area_4_1 p q r s x \<longleftrightarrow> \<not> ccw p r x \<and> ccw q s x"
definition "area_4_2 p q r s x \<longleftrightarrow> \<not> ccw p r x \<and> \<not> ccw q s x"
definition "area_4_3 p q r s x \<longleftrightarrow> ccw p r x \<and> \<not> ccw q s x"
definition "area_4_4 p q r s x \<longleftrightarrow> ccw p r x \<and> ccw q s x"

lemma area_4_cases: "area_4_1 p q r s x \<or> area_4_2 p q r s x \<or> area_4_3 p q r s x \<or> area_4_4 p q r s x"
unfolding area_4_1_def area_4_2_def area_4_3_def area_4_4_def
by auto

lemma area_4_2:
assumes "area_4_2 p q r s x" "distinct [p, q, r, s]" "x \<notin> set [p, q, r, s]" "in_general_position {p, q, r, s, x}"
shows "area_4_1 q r s p x"
proof-
  have "p \<noteq> r" "p \<noteq> x" "r \<noteq> x"
    using assms
    by auto
  hence "\<not> col p x r"
    using `in_general_position {p, q, r, s, x}`
    unfolding in_general_position_def
    by blast
  thus ?thesis
    using assms
    using ax1[OF ax1[OF ax2'[of p r x]]]
    unfolding area_4_2_def area_4_1_def
    by auto
qed

lemma area_4_3:
assumes "area_4_3 p q r s x" "distinct [p, q, r, s]" "x \<notin> set [p, q, r, s]" "in_general_position {p, q, r, s, x}"
shows "area_4_1 r s p q x"
proof-
  have "q \<noteq> s" "q \<noteq> x" "s \<noteq> x"
    using assms
    by auto
  hence "\<not> col q x s"
    using `in_general_position {p, q, r, s, x}`
    unfolding in_general_position_def
    by blast
  thus ?thesis
    using assms
    using ax2[of p r x] ax1[of r p x]
    using ax1[OF ax1[OF ax2'[of q s x]]]
    unfolding area_4_3_def area_4_1_def
    by auto
qed

lemma area_4_4:
assumes "area_4_4 p q r s x" "distinct [p, q, r, s]" "x \<notin> set [p, q, r, s]" "in_general_position {p, q, r, s, x}"
shows "area_4_1 s p q r x"
  using assms
  using ax2[of q s x] ax1[of s q x]
  unfolding area_4_4_def area_4_1_def
  by auto

lemma area_4_1:
assumes "convex_polygon [p, q, r, s]" "ccw p q x" "ccw q r x" "ccw r s x" "ccw s p x" and gp: "in_general_position {p, q, r, s, x}"
shows "\<exists> p' q' r' s'. cyclic_perm [p, q, r, s] [p', q', r', s'] \<and> area_4_1 p' q' r' s' x"
proof-
  have "distinct [p, q, r, s]"
    by (rule convex_polygon_distinct[OF assms(1)])
   
  have "x \<notin> set [p, q, r, s]"
    using assms ax0
    by simp blast

  note * = `distinct [p, q, r, s]` `x \<notin> set [p, q, r, s]`
  
  {
    assume "area_4_1 p q r s x"
    hence ?thesis
      using assms
      by (rule_tac x="p" in exI, 
          rule_tac x="q" in exI, 
          rule_tac x="r" in exI, 
          rule_tac x="s" in exI) auto
  }
  moreover
  {
    assume "area_4_2 p q r s x"
    hence ?thesis
      using area_4_2[OF _ * gp]
      unfolding cyclic_perm_def
      by  (rule_tac x="q" in exI, rule_tac x="r" in exI, rule_tac x="s" in exI, rule_tac x="p" in exI, auto)
   }
   moreover
   {
    assume "area_4_3 p q r s x"
    hence ?thesis
      using area_4_3[OF _ * gp]
      unfolding cyclic_perm_def
      by (rule_tac x="r" in exI, rule_tac x="s" in exI, rule_tac x="p" in exI, rule_tac x="q" in exI) (rule conjI, rule disjI2, force, simp)
   }
   moreover
   {
    assume "area_4_4 p q r s x"
    hence ?thesis
      using area_4_4[OF _ * gp]
      unfolding cyclic_perm_def
      by (rule_tac x="s" in exI, rule_tac x="p" in exI, rule_tac x="q" in exI, rule_tac x="r" in exI) (rule conjI, rule disjI2, force+)
   }
   ultimately
   show ?thesis
     using area_4_cases
     by blast
qed

lemma nested_hulls_4_1:
assumes "nested_hulls ([p1, q1, r1, s1] # (p2 # pp) # H)" and gp: "in_general_position (set (concat ([p1, q1, r1, s1] # (p2 # pp) # H)))" (is "in_general_position ?S") 
shows "\<exists> p1' q1' r1' s1'. nested_hulls ([p1', q1', r1', s1'] # (p2 # pp) # H) \<and> 
        set (concat ([p1, q1, r1, s1] # (p2 # pp) # H)) = set (concat ([p1', q1', r1', s1'] # (p2 # pp) # H)) \<and>
        area_4_1 p1' q1' r1' s1' p2"
proof-        
  have "convex_polygon [p1, q1, r1, s1]"
    using convex_hull_is_convex_polygon[OF gp nested_hulls_are_convex_hulls[OF assms(1)]]
    by simp
  moreover
  have "ccw p1 q1 p2" "ccw q1 r1 p2" "ccw r1 s1 p2" "ccw s1 p1 p2"
    using nested_hulls_are_convex_hulls[OF assms(1)]
    using nested_hulls_disjoint[OF assms(1)]
    unfolding convex_hull_def Let_def
    apply simp_all
    apply ((erule conjE)+, erule_tac x=0 in allE, simp)
    apply ((erule conjE)+, erule_tac x=1 in allE, simp)
    apply ((erule conjE)+, erule_tac x=2 in allE, simp)
    apply ((erule conjE)+, erule_tac x=3 in allE, simp)
    done
  moreover
  have "in_general_position {p1, q1, r1, s1, p2}"
    by (rule in_general_position_mono[OF _ gp], simp) 
  ultimately
  obtain p1' q1' r1' s1' where
  *: "cyclic_perm [p1, q1, r1, s1] [p1', q1', r1', s1']" and "area_4_1 p1' q1' r1' s1' p2"
    using area_4_1[of p1 q1 r1 s1 p2]
    by blast
  thus ?thesis
    using cyclic_perm_nested_huls[OF _ *, OF assms(1)]
    using cyclic_perm_set[OF *]
    by (rule_tac x="p1'" in exI, rule_tac x="q1'" in exI, rule_tac x="r1'" in exI, rule_tac x="s1'" in exI, auto)
qed

(* ----------------------------------------------------------------- *)

definition "area_5_1 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and>   ccw p r x \<and>   ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 
definition "area_5_2 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and> \<not> ccw p r x \<and> \<not> ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 
definition "area_5_3 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and> \<not> ccw q s x \<and>   ccw r t x" 
definition "area_5_4 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and>   ccw q s x \<and> \<not> ccw r t x" 
definition "area_5_5 p q r s t x  \<longleftrightarrow>   ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 
definition "area_5_6 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and> \<not> ccw p r x \<and>   ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 
definition "area_5_7 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and> \<not> ccw p r x \<and> \<not> ccw q t x \<and> \<not> ccw q s x \<and>   ccw r t x" 
definition "area_5_8 p q r s t x  \<longleftrightarrow> \<not> ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and> \<not> ccw q s x \<and> \<not> ccw r t x" 
definition "area_5_9 p q r s t x  \<longleftrightarrow>   ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and>   ccw q s x \<and> \<not> ccw r t x" 
definition "area_5_10 p q r s t x \<longleftrightarrow>   ccw p s x \<and>   ccw p r x \<and>   ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 
definition "area_5_11 p q r s t x \<longleftrightarrow> \<not> ccw p s x \<and>   ccw p r x \<and> \<not> ccw q t x \<and>   ccw q s x \<and>   ccw r t x" 

lemma area_5_cases: 
assumes "convex_polygon [p, q, r, s, t]" "ccw p q x" "ccw q r x" "ccw r s x" "ccw s t x" "ccw t p x" "in_general_position {p, q, r, s, t, x}"
shows "area_5_1 p q r s t x \<or> area_5_2 p q r s t x \<or> area_5_3 p q r s t x \<or> area_5_4 p q r s t x \<or> area_5_5 p q r s t x \<or> area_5_6 p q r s t x \<or> area_5_7 p q r s t x \<or> area_5_8 p q r s t x \<or> area_5_9 p q r s t x \<or> area_5_10 p q r s t x \<or> area_5_11 p q r s t x"
proof-
  have *: "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t"
    using convex_polygon_distinct[OF `convex_polygon [p, q, r, s, t]`]
    by auto
  moreover 
  have **: "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x"
    using assms(2-6) ax0
    by blast+
  ultimately
  have "distinct [p, q, r, s, t, x]"
    by auto
  hence "ax3_formula [p, q, r, s, t, x]" 
    using `in_general_position {p, q, r, s, t, x}`
    using ax3_formula
    by simp
  moreover
  have "ax1_formula [p, q, r, s, t, x]" "ax2_formula [p, q, r, s, t, x]" "ax4_formula [p, q, r, s, t, x]" "ax5_formula [p, q, r, s, t, x]"
    using ax1_formula ax2_formula ax4_formula ax5_formula
    by auto
  moreover
  have "convex_polygon_formula [p, q, r, s, t]"
    by (rule convex_polygon_formula[OF assms(1)])
  ultimately
  show ?thesis
    using assms(2-6)
    unfolding area_5_1_def area_5_2_def area_5_3_def area_5_4_def area_5_5_def area_5_6_def area_5_7_def area_5_8_def area_5_9_def area_5_10_def area_5_11_def
    unfolding ax1_formula_def ax2_formula_def ax3_formula_def ax4_formula_def ax5_formula_def convex_polygon_formula_def
    by normalization sat 
qed

lemma area_5_2:
assumes "area_5_2 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_1 q r s t p x"
proof- 
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence "\<not> col p x r" "\<not> col p x s"
    using `in_general_position {p, q, r, s, t, x}` 
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms ax1[OF ax1[OF ax2'[of p r x]]] ax1[OF ax1[OF ax2'[of p s x]]]
    unfolding area_5_2_def area_5_1_def
    by auto
qed

lemma area_5_3:
assumes "area_5_3 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_1 r s t p q x"
proof- 
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence  "\<not> col q x s" "\<not> col p x s" "\<not> col q x t"
    using `in_general_position {p, q, r, s, t, x}` 
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms
    using ax2[of r p x] ax1[of p r x]
    using ax1[OF ax1[OF ax2'[of q s x]]]
    using ax1[OF ax1[OF ax2'[of p s x]]]
    using ax1[OF ax1[OF ax2'[of q t x]]]
    unfolding area_5_3_def area_5_1_def
    by auto
qed

lemma area_5_4:
assumes "area_5_4 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_1 s t p q r x"
proof-
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence   "\<not> col r x t" "\<not> col p x s" "\<not> col q x t"
    using `in_general_position {p, q, r, s, t, x}`
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms
    unfolding area_5_4_def area_5_1_def
    using ax2[OF ax1[of q s x]]
    using ax1[OF ax1[OF ax2'[of p s x]]]
    using ax1[OF ax1[OF ax2'[of r t x]]]
    using ax1[OF ax1[OF ax2'[of q t x]]]
    by auto
qed

lemma area_5_5:
assumes "area_5_5 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_1 t p q r s x"
proof- 
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence  "\<not> col q x t"
    using `in_general_position {p, q, r, s, t, x}` 
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms
    unfolding area_5_5_def area_5_1_def
    using ax2[OF ax1[of r t x]]
    using ax1[OF ax1[OF ax2'[of q t x]]]
    by auto
qed

lemma area_5_7:
assumes "area_5_7 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_6 q r s t p x"
proof-
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence "\<not> col p x r" "\<not> col p x s"
    using `in_general_position {p, q, r, s, t, x}` 
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms ax1[OF ax1[OF ax2'[of p r x]]] ax1[OF ax1[OF ax2'[of p s x]]]
    unfolding area_5_7_def area_5_6_def
    by auto
qed

lemma area_5_8:
assumes "area_5_8 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_6 r s t p q x"
proof- 
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence "\<not> col q x s" "\<not> col p x s" "\<not> col q x t"
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    using `in_general_position {p, q, r, s, t, x}`
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms
    using ax2[OF ax1[of p r x]]
    using ax1[OF ax1[OF ax2'[of q s x]]]
    using ax1[OF ax1[OF ax2'[of p s x]]]
    using ax1[OF ax1[OF ax2'[of q t x]]]
    unfolding area_5_8_def area_5_6_def
    by auto
qed

lemma area_5_9:
assumes "area_5_9 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_6 s t p q r x"
proof- 
  have "p \<noteq> q" "p \<noteq> r" "p \<noteq> s" "p \<noteq> t" "q \<noteq> r" "q \<noteq> s" "q \<noteq> t" "r \<noteq> s" "r \<noteq> t" "s \<noteq> t" 
       "p \<noteq> x" "q \<noteq> x" "r \<noteq> x" "s \<noteq> x" "t \<noteq> x" 
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    by auto
  hence "\<not> col r x t" "\<not> col q x t"
    using `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    using `in_general_position {p, q, r, s, t, x}`
    by (smt in_general_position_def insertI1 insert_commute)+
  thus ?thesis
    using assms
    using ax2[OF ax1[of q s x]]
    using ax2[OF ax1[of p s x]]
    using ax1[OF ax1[OF ax2'[of r t x]]]
    using ax1[OF ax1[OF ax2'[of q t x]]]
    unfolding area_5_9_def area_5_6_def
    by auto
qed

lemma area_5_10:
assumes "area_5_10 p q r s t x" "distinct [p, q, r, s, t]" "x \<notin> set [p, q, r, s, t]" "in_general_position {p, q, r, s, t, x}"
shows "area_5_6 t p q r s x"
  using assms
  using ax2[OF ax1[of r t x]]
  using ax2[OF ax1[of q t x]]
  unfolding area_5_10_def area_5_6_def
  by simp


definition "area_5_canon p q r s t x \<longleftrightarrow> \<not> ccw p s x \<and> ccw q s x \<and> ccw r t x \<and> (ccw p r x \<or> ccw q t x)"

lemma area_5_canon_iff: "area_5_canon p q r s t x \<longleftrightarrow> area_5_1 p q r s t x \<or> area_5_6 p q r s t x \<or> area_5_11 p q r s t x"
unfolding area_5_canon_def area_5_1_def area_5_6_def area_5_11_def
by auto

lemma area_5_canon:
assumes "convex_polygon [p, q, r, s, t]" "ccw p q x" "ccw q r x" "ccw r s x" "ccw s t x" "ccw t p x" and gp: "in_general_position {p, q, r, s, t, x}"
shows "\<exists> p' q' r' s' t'. cyclic_perm [p, q, r, s, t] [p', q', r', s', t'] \<and> area_5_canon p' q' r' s' t' x"
using assms
proof-
  have "distinct [p, q, r, s, t]"
    using `convex_polygon [p, q, r, s, t]`
    using convex_polygon_distinct
    by blast
  have "x \<notin> set [p, q, r, s, t]"
    using assms ax0
    by simp blast

  note * = `distinct [p, q, r, s, t]` `x \<notin> set [p, q, r, s, t]`
    
  {
     assume "area_5_1 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [p, q, r, s, t] \<and> area_5_canon p q r s t x"
       by (simp add: area_5_canon_iff) 
  }
  moreover
  {
     assume **: "area_5_2 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [q, r, s, t, p] \<and> area_5_canon q r s t p x"
       using area_5_2[OF ** * gp]
       by (auto simp add: area_5_canon_iff cyclic_perm_def)
  }
  moreover
  {
     assume **: "area_5_3 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [r, s, t, p, q] \<and> area_5_canon r s t p q x"
       using area_5_3[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_4 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [s, t, p, q, r] \<and> area_5_canon s t p q r x"
       using area_5_4[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_5 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [t, p, q, r, s] \<and> area_5_canon t p q r s x"
       using area_5_5[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
    assume **: "area_5_6 p q r s t x"
    hence "cyclic_perm [p, q, r, s, t] [p, q, r, s, t] \<and> area_5_canon p q r s t x"
      by (simp add: area_5_canon_iff cyclic_perm_def)
  }
  moreover
  {
     assume **: "area_5_7 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [q, r, s, t, p] \<and> area_5_canon q r s t p x"
       using area_5_7[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_8 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [r, s, t, p, q] \<and> area_5_canon r s t p q x"
       using area_5_8[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_9 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [s, t, p, q, r] \<and> area_5_canon s t p q r x"
       using area_5_9[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_10 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [t, p, q, r, s] \<and> area_5_canon t p q r s x"
       using area_5_10[OF ** * gp]
       by (simp add: area_5_canon_iff cyclic_perm_def) (rule disjI2, force)
  }
  moreover
  {
     assume **: "area_5_11 p q r s t x"
     hence "cyclic_perm [p, q, r, s, t] [p, q, r, s, t] \<and> area_5_canon p q r s t x"
       by (simp add: area_5_canon_iff)
  }
  ultimately
  show ?thesis
    using area_5_cases[OF assms(1-6) gp]
    by metis
qed

lemma nested_hulls_5_canon:
assumes "nested_hulls ([p1, q1, r1, s1, t1] # (p2 # pp) # H)" and gp: "in_general_position (set (concat ([p1, q1, r1, s1, t1] # (p2 # pp) # H)))" (is "in_general_position ?S") 
shows "\<exists> p1' q1' r1' s1' t1'. nested_hulls ([p1', q1', r1', s1', t1'] # (p2 # pp) # H) \<and> 
        set (concat ([p1, q1, r1, s1, t1] # (p2 # pp) # H)) = set (concat ([p1', q1', r1', s1', t1'] # (p2 # pp) # H)) \<and>
        area_5_canon p1' q1' r1' s1' t1' p2"
proof-
  have "convex_polygon [p1, q1, r1, s1, t1]"
    using convex_hull_is_convex_polygon[OF gp nested_hulls_are_convex_hulls[OF assms(1)]]
    by simp
  moreover
  have "ccw p1 q1 p2" "ccw q1 r1 p2" "ccw r1 s1 p2" "ccw s1 t1 p2" "ccw t1 p1 p2"
    using nested_hulls_are_convex_hulls[OF assms(1)]
    using nested_hulls_disjoint[OF assms(1)]
    unfolding convex_hull_def Let_def
    apply simp_all
    apply ((erule conjE)+, erule_tac x=0 in allE, simp)
    apply ((erule conjE)+, erule_tac x=1 in allE, simp)
    apply ((erule conjE)+, erule_tac x=2 in allE, simp)
    apply ((erule conjE)+, erule_tac x=3 in allE, simp)
    apply ((erule conjE)+, erule_tac x=4 in allE, simp)
    done
  moreover
  have "in_general_position {p1, q1, r1, s1, t1, p2}"
    by (rule in_general_position_mono[OF _ gp], simp) 
  ultimately
  obtain p1' q1' r1' s1' t1' where
  *: "cyclic_perm [p1, q1, r1, s1, t1] [p1', q1', r1', s1', t1']" and "area_5_canon p1' q1' r1' s1' t1' p2"
    using area_5_canon[of p1 q1 r1 s1 t1 p2]
    by blast
  thus ?thesis
    using cyclic_perm_nested_huls[OF _ *, OF assms(1)]
    using cyclic_perm_set[OF *]
    by (rule_tac x="p1'" in exI, rule_tac x="q1'" in exI, rule_tac x="r1'" in exI, rule_tac x="s1'" in exI, rule_tac x="t1'" in exI, auto)
qed

(* ----------------------------------------------------------------- *)

definition all_area_3_1 where
  "all_area_3_1 H \<longleftrightarrow> (\<forall> i < length H - 1. length (H ! i) = 3 \<and> length (H ! (i+1)) \<ge> 2 \<longrightarrow> area_3_1  (H ! i ! 0) (H ! i ! 1) (H ! i ! 2) (H ! (i+1) ! 0) (H ! (i+1) ! 1))" 

definition all_area_4_1 where
  "all_area_4_1 H \<longleftrightarrow> (\<forall> i < length H - 1. length (H ! i) = 4 \<longrightarrow> area_4_1 (H ! i ! 0) (H ! i ! 1) (H ! i ! 2) (H ! i ! 3) (H ! (i + 1) ! 0))"

definition all_area_5_canon where
  "all_area_5_canon H \<longleftrightarrow> (\<forall> i < length H - 1. length (H ! i) = 5 \<longrightarrow> area_5_canon (H ! i ! 0) (H ! i ! 1) (H ! i ! 2) (H ! i ! 3) (H ! i ! 4) (H ! (i + 1) ! 0))"



lemma nested_hulls_all_canon:
assumes "nested_hulls H" "in_general_position (set (concat H))"
shows "\<exists> H'. nested_hulls H' \<and> set (concat H') = set (concat H) \<and> length H' = length H \<and> all_area_3_1 H' \<and> all_area_4_1 H' \<and> all_area_5_canon H'"
proof (cases "H = []")
  case True
  thus ?thesis
    using assms
    by (rule_tac x="[]" in exI, simp add: nested_hulls.base all_area_3_1_def all_area_4_1_def all_area_5_canon_def)
next
  case False
  show ?thesis
    using assms `H \<noteq> []`
    proof induct
      case base
      thus ?case
        by (rule_tac x="[]" in exI) (simp add: nested_hulls.base all_area_4_1_def all_area_5_canon_def)
    next
      case (ind H p)
      show ?case
      proof (cases "H = []")
        case True
        hence "nested_hulls [p]"
          using nested_hulls.ind[of "[]" p] ind(5) nested_hulls.base `p \<noteq> []` 
          by simp
        thus ?thesis
          using `H = []`
          by (rule_tac x="[p]" in exI, simp add: all_area_3_1_def all_area_4_1_def all_area_5_canon_def)
      next
        case False
        with ind(2) obtain H' where 
          "nested_hulls H'" "set (concat H') = set (concat H)" "length H' = length H" "all_area_3_1 H'" "all_area_4_1 H'" "all_area_5_canon H'"
          using `in_general_position (set (concat (p # H)))` in_general_position_mono[of "set (concat H)" "set (concat (p # H))"]
          by auto
        have "H' \<noteq> []" 
          using `H \<noteq> []` `length H' = length H`
          by auto

        have "nested_hulls (p # H')"
          using `nested_hulls H'` `set p \<inter> set (concat H) = {}` `convex_hull p (set p \<union> set (concat H))`
          using `set (concat H') = set (concat H)` `p \<noteq> []`
          using nested_hulls.intros(2)
          by simp

        have "set (concat (p # H')) = set (concat (p # H))"
          using `set (concat H') = set (concat H)`
          by simp
          
        have "H' ! 0 \<noteq> []"
          using `nested_hulls H'` nested_hulls_no_Nil[of H'] `H' \<noteq> []` in_set_conv_nth[of "H' ! 0"]
          by auto
        hence *: "H' = (hd (H' ! 0) # tl (H' ! 0)) # tl H'"
          using `H' \<noteq> []`
          using hd_Cons_tl[of "H' ! 0"] hd_conv_nth[of H', symmetric]
          by auto          
          
        show ?thesis
        proof (cases "length p = 4")
          case False
          show ?thesis
          proof (cases "length p = 5")
            case False
            show ?thesis
            proof (cases "length p = 3")
              case False
              have "all_area_3_1 (p # H')"
                using `all_area_3_1 H'` `length p \<noteq> 3`
                unfolding all_area_3_1_def
                apply auto
                apply (case_tac "i = 0", simp)
                apply (erule_tac x="i - 1" in allE, simp)
                done
              moreover
              have "all_area_4_1 (p # H')"
                using `all_area_4_1 H'` `length p \<noteq> 4`
                unfolding all_area_4_1_def
                apply auto
                apply (case_tac "i = 0", simp)
                apply (erule_tac x="i - 1" in allE, simp)
                done
              moreover
              have "all_area_5_canon (p # H')"
                using `all_area_5_canon H'` `length p \<noteq> 5`
                unfolding all_area_5_canon_def
                apply auto
                apply (case_tac "i = 0", simp)
                apply (erule_tac x="i - 1" in allE, simp)
                done
              ultimately
              show ?thesis
                using `length H' = length H` `nested_hulls (p # H')` `set (concat (p # H')) = set (concat (p # H))`
                by (rule_tac x="p # H'" in exI, simp)
            next
            case True
            show ?thesis
            proof (cases "length (H' ! 0) \<ge> 2")
               case True
               hence "H' ! 0 = (H' ! 0) ! 0 # (H' ! 0) ! 1 # drop 2 (H' ! 0)"
                 by (simp add: Cons_nth_drop_Suc `H' ! 0 \<noteq> []` numeral_2_eq_2)
               hence *: "H' = ((H' ! 0) ! 0 #  (H' ! 0) ! 1 # drop 2 (H' ! 0)) # tl H'"
                  using `H' \<noteq> []`
                  using hd_conv_nth[of H', symmetric]
                  by auto          
               obtain a b c where "p = [a, b, c]"
                  using `length p = 3` length3[of p]
                  by auto
               then obtain a' b' c' where "nested_hulls ([a', b', c'] # H')" 
                 "set (concat ([a', b', c'] # H')) = set (concat (p # H))"
                 "area_3_1 a' b' c' ((H' ! 0) ! 0) ((H' ! 0) ! 1)"
                 using nested_hulls_3_1[of a b c "(H' ! 0) ! 0" "(H' ! 0) ! 1" "drop 2 (H' ! 0)" "tl H'"] 
                   `nested_hulls (p # H')` `p = [a, b, c]` `in_general_position (set (concat (p # H)))`
                   `set (concat (p # H')) = set (concat (p # H))`  `p = [a, b, c]` *
                 by auto (smt UN_insert Un_insert_left list.simps(15))
               let ?H' = "[a', b', c'] # H'"
               have "all_area_3_1 ?H'"
                 unfolding all_area_3_1_def
               proof safe
                 fix i
                 assume "i < length ?H' - 1" "length (?H' ! i) = 3" "length (?H' ! (i+1)) \<ge> 2"
                 show "area_3_1 (?H' ! i ! 0) (?H' ! i ! 1) (?H' ! i ! 2) (?H' ! (i + 1) ! 0) (?H' ! (i + 1) ! 1)"
                 proof (cases "i = 0")
                   case True
                   thus ?thesis
                     using `area_3_1 a' b' c' ((H' ! 0) ! 0) ((H' ! 0) ! 1)`
                     by simp
                 next
                   case False
                   thus ?thesis
                     using `all_area_3_1 H'` `H' \<noteq> []` `length (?H' ! i) = 3` `length (?H' ! (i+1)) \<ge> 2`  `i < length (?H') - 1` 
                     unfolding all_area_3_1_def
                     by (erule_tac x="i-1" in allE, auto)
                 qed
               qed
               moreover
               have "all_area_4_1 ?H'"
                 using `all_area_4_1 H'` `H' \<noteq> []`
                 unfolding all_area_4_1_def
                 apply auto
                 apply (case_tac "i = 0", simp)
                 apply (erule_tac x="i-1" in allE, simp)
                 done            
               moreover
               have "all_area_5_canon ?H'"
                 using `all_area_5_canon H'` `H' \<noteq> []`
                 unfolding all_area_5_canon_def
                 apply auto
                 apply (case_tac "i = 0", simp)
                 apply (erule_tac x="i-1" in allE, simp)
                 done
               ultimately
               show ?thesis
                 using `nested_hulls ?H'` `set (concat ?H') = set (concat (p # H))` `length H' = length H`
                 by (rule_tac x="?H'" in exI, simp)
            next
               case False
               hence "all_area_3_1 (p # H')"
                 using `all_area_3_1 H'`
                 unfolding all_area_3_1_def
                 apply auto
                 apply (case_tac "i = 0", simp)
                 apply (erule_tac x="i - 1" in allE, simp)
                 done
               moreover
               have "all_area_4_1 (p # H')"
                 using `all_area_4_1 H'` `length p \<noteq> 4`
                 unfolding all_area_4_1_def
                 apply auto
                 apply (case_tac "i = 0", simp)
                 apply (erule_tac x="i - 1" in allE, simp)
                 done
               moreover
               have "all_area_5_canon (p # H')"
                 using `all_area_5_canon H'` `length p \<noteq> 5`
                 unfolding all_area_5_canon_def
                 apply auto
                 apply (case_tac "i = 0", simp)
                 apply (erule_tac x="i - 1" in allE, simp)
                 done
               ultimately
               show ?thesis
                 using `length H' = length H` `nested_hulls (p # H')` `set (concat (p # H')) = set (concat (p # H))`
                 by (rule_tac x="p # H'" in exI, simp)               
             qed
          qed
          next
            case True
            then obtain a b c d e where "p = [a, b, c, d, e]"
              using length5[of p]
              by auto
            then obtain a' b' c' d' e' where "nested_hulls ([a', b', c', d', e'] # H')" 
              "set (concat ([a', b', c', d', e'] # H')) = set (concat (p # H))"
              "area_5_canon a' b' c' d' e' (hd (H' ! 0))"
              using nested_hulls_5_canon[of a b c d e "hd (H' ! 0)" "tl (H' ! 0)" "tl H'"] 
              `nested_hulls (p # H')` `p = [a, b, c, d, e]` `in_general_position (set (concat (p # H)))`
              `set (concat (p # H')) = set (concat (p # H))` *
              by auto
              
            let ?H' = "[a', b', c', d', e'] # H'"
            have "all_area_5_canon ?H'"
              unfolding all_area_5_canon_def
            proof safe
              fix i
              assume "i < length ?H' - 1" "length (?H' ! i) = 5"
              show "area_5_canon (?H' ! i ! 0) (?H' ! i ! 1) (?H' ! i ! 2) (?H' ! i ! 3) (?H' ! i ! 4) (?H' ! (i + 1) ! 0)"
              proof (cases "i = 0")
                case True
                thus ?thesis
                  using `area_5_canon a' b' c' d' e' (hd (H' ! 0))` hd_conv_nth[of "H' ! 0"] `H' ! 0 \<noteq> []`
                  by simp
              next
                case False
                thus ?thesis
                  using `all_area_5_canon H'` `H' \<noteq> []` `length (?H' ! i) = 5` `i < length (?H') - 1` 
                  unfolding all_area_5_canon_def
                  by (erule_tac x="i-1" in allE, auto)
              qed
            qed
            moreover
            have "all_area_3_1 ?H'"
              using `all_area_3_1 H'` `H' \<noteq> []`
              unfolding all_area_3_1_def
              apply auto
              apply (case_tac "i = 0", simp)
              apply (erule_tac x="i-1" in allE, simp)
              done            
            moreover
            have "all_area_4_1 ?H'"
              using `all_area_4_1 H'` `H' \<noteq> []`
              unfolding all_area_4_1_def
              apply auto
              apply (case_tac "i = 0", simp)
              apply (erule_tac x="i-1" in allE, simp)
              done
            ultimately
            show ?thesis
              using `nested_hulls ?H'` `set (concat ?H') = set (concat (p # H))` `length H' = length H`
              by (rule_tac x="?H'" in exI, simp)            
          qed
        next
          case True
          then obtain a b c d where "p = [a, b, c, d]"
            using length4[of p]
            by auto
          have "H' ! 0 \<noteq> []"
            using `nested_hulls H'` nested_hulls_no_Nil[of H'] `H' \<noteq> []` in_set_conv_nth[of "H' ! 0"]
            by auto
          hence "H' = (hd (H' ! 0) # tl (H' ! 0)) # tl H'"
            using `H' \<noteq> []`
            using hd_Cons_tl[of "H' ! 0"] hd_conv_nth[of H', symmetric]
            by auto
          then obtain a' b' c' d' where "nested_hulls ([a', b', c', d'] # H')" 
              "set (concat ([a', b', c', d'] # H')) = set (concat (p # H))"
              "area_4_1 a' b' c' d' (hd (H' ! 0))"
            using nested_hulls_4_1[of a b c d "hd (H' ! 0)" "tl (H' ! 0)" "tl H'"] 
            `nested_hulls (p # H')` `p = [a, b, c, d]` `in_general_position (set (concat (p # H)))`
            `set (concat (p # H')) = set (concat (p # H))`
            by auto
          let ?H' = "[a', b', c', d'] # H'"
          have "all_area_4_1 ?H'"
          unfolding all_area_4_1_def
          proof safe
            fix i
            assume "i < length ?H' - 1" "length (?H' ! i) = 4"
            show "area_4_1 (?H' ! i ! 0) (?H' ! i ! 1) (?H' ! i ! 2) (?H' ! i ! 3) (?H' ! (i + 1) ! 0)"
            proof (cases "i = 0")
              case True
              thus ?thesis
                using `area_4_1 a' b' c' d' (hd (H' ! 0))` hd_conv_nth[of "H' ! 0"] `H' ! 0 \<noteq> []`
                by simp
            next
              case False
              thus ?thesis
                using `all_area_4_1 H'` `H' \<noteq> []` `length (?H' ! i) = 4` `i < length (?H') - 1` 
                unfolding all_area_4_1_def
                by (erule_tac x="i-1" in allE, auto)
            qed
          qed
          moreover
          have "all_area_3_1 ?H'"
              using `all_area_3_1 H'` `H' \<noteq> []`
              unfolding all_area_3_1_def
              apply auto
              apply (case_tac "i = 0", simp)
              apply (erule_tac x="i-1" in allE, simp)
              done
          moreover
          have "all_area_5_canon ?H'"
              using `all_area_5_canon H'` `H' \<noteq> []`
              unfolding all_area_5_canon_def
              apply auto
              apply (case_tac "i = 0", simp)
              apply (erule_tac x="i-1" in allE, simp)
              done
          ultimately
          show ?thesis
            using `nested_hulls ?H'` `set (concat ?H') = set (concat (p # H))` `length H' = length H`
            by (rule_tac x="?H'" in exI, simp)
       qed
     qed
   qed
qed


end


(* R2 interpretation *)           
type_synonym point = "real \<times> real"

definition sgn_int :: "real \<Rightarrow> int" where
  "sgn_int x = (if x > 0 then 1 else if x = 0 then 0 else -1)"

lemma sgn_int_0_0:
"sgn_int x = 0 \<longleftrightarrow> x = 0"
unfolding sgn_int_def
by auto

definition det3 :: "real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real \<Rightarrow> real" where 
  "det3 a11 a12 a13 
        a21 a22 a23
        a31 a32 a33 = a11*a22*a33 + a12*a23*a31 + a13*a21*a32 - a31*a22*a13 - a32*a23*a11 - a33*a21*a12"

definition vector_product :: "point \<Rightarrow> point \<Rightarrow> real" where
  "vector_product v1 v2 = 
    (let (v1x, v1y) = v1; (v2x, v2y) = v2 
      in v1x*v2y - v1y*v2x)"

definition triangle_orientation :: "point \<Rightarrow> point \<Rightarrow> point \<Rightarrow> real" where
"triangle_orientation p0 p1 p2 = 
  (let (x0, y0) = p0; (x1, y1) = p1; (x2, y2) = p2
    in det3 x0 y0 1
            x1 y1 1
            x2 y2 1)"

lemma "triangle_orientation p q r = vector_product (p - r) (q - r)"
unfolding triangle_orientation_def vector_product_def
by (cases p, cases q, cases r, auto simp add: det3_def field_simps)
            
definition cw_r2 where "cw_r2 A B C \<longleftrightarrow> triangle_orientation A B C < 0"
definition col_r2 where "col_r2 A B C \<longleftrightarrow> triangle_orientation A B C = 0"
definition ccw_r2 where "ccw_r2 A B C \<longleftrightarrow> triangle_orientation A B C > 0"

lemma ax4_lemma: 
 "triangle_orientation p q r = triangle_orientation t q r + triangle_orientation p t r + triangle_orientation p q t"
by (cases p, cases q, cases r, cases t) (simp add: triangle_orientation_def det3_def)

lemma ax5_lemma: "(triangle_orientation s t q) * (triangle_orientation t p r) = 
             (triangle_orientation t q r) * (triangle_orientation s t p) +
             (triangle_orientation t p q) * (triangle_orientation s t r)"
by (cases p, cases q, cases r, cases s, cases t) (simp add: triangle_orientation_def det3_def field_simps)          

interpretation R2: convex ccw_r2 col_r2
proof
 fix p q r
 assume "ccw_r2 p q r" 
 thus "p \<noteq> q \<and> p \<noteq> r \<and> q \<noteq> r \<and> \<not> col_r2 p q r"
    unfolding ccw_r2_def col_r2_def triangle_orientation_def
    by (cases p, cases q, cases r, auto simp add: det3_def)
next
  fix p q r
  assume "ccw_r2 p q r" 
  thus "ccw_r2 q r p"
    unfolding ccw_r2_def triangle_orientation_def
    by (cases p, cases q, cases r, auto simp add: det3_def field_simps)
next
  fix p q r
  assume "ccw_r2 p q r"
  thus "\<not> ccw_r2 p r q"
    unfolding ccw_r2_def triangle_orientation_def
    by (cases p, cases q, cases r, auto simp add: det3_def field_simps)
next
  fix p q r
  assume "p \<noteq> q" "p \<noteq> r" "q \<noteq> r" "\<not> col_r2 p q r"
  thus "ccw_r2 p q r \<or> ccw_r2 p r q"
    unfolding col_r2_def ccw_r2_def triangle_orientation_def
    by (cases p, cases q, cases r, auto simp add: det3_def field_simps)
next
  fix t q r p
  assume "ccw_r2 t q r" "ccw_r2 p t r" "ccw_r2 p q t"
  thus "ccw_r2 p q r"
    using ax4_lemma[of p q r t]
    unfolding ccw_r2_def
    by simp
next
  fix t s p q r
  assume assms: "ccw_r2 t s p" "ccw_r2 t s q" "ccw_r2 t s r" "ccw_r2 t p q" "ccw_r2 t q r" 
  show "ccw_r2 t p r"
  proof-
    obtain tx ty sx sy px py qx qy rx ry where
    *: "t = (tx, ty)" "s = (sx, sy)" "p = (px, py)" "q = (qx, qy)" "r = (rx, ry)"
      by (cases p, cases q, cases r, cases t, cases s, auto)
    have "(triangle_orientation t q r) * (triangle_orientation s t p) < 0"
    proof-
      have "triangle_orientation t q r > 0"
        using assms *
        unfolding ccw_r2_def
        by simp
      moreover
      have "triangle_orientation s t p < 0"
        using assms *
        unfolding ccw_r2_def
        by (simp add: triangle_orientation_def det3_def field_simps)
      ultimately
      show ?thesis
        by (simp add: mult_pos_neg)
    qed
    moreover
    have "(triangle_orientation t p q) * (triangle_orientation s t r) < 0"
    proof-
      have "triangle_orientation t p q > 0"
        using assms *
        unfolding ccw_r2_def
        by simp
      moreover
      have "triangle_orientation s t r < 0"
        using assms *
        unfolding ccw_r2_def
        by (simp add: triangle_orientation_def det3_def field_simps)
      ultimately
      show ?thesis
        by (simp add: mult_pos_neg)
    qed
    ultimately
    have "(triangle_orientation s t q) * (triangle_orientation t p r) < 0" 
      using ax5_lemma[of s t q p r]
      by simp
    moreover 
    have "triangle_orientation s t q < 0"
      using assms *
      unfolding ccw_r2_def
      by (simp add: triangle_orientation_def det3_def field_simps)
    ultimately
    show ?thesis
      unfolding ccw_r2_def
      by (smt mult_nonpos_nonpos)
 qed
next
  fix t s p q r
  assume assms: "ccw_r2 s t p" "ccw_r2 s t q" "ccw_r2 s t r" "ccw_r2 t p q" "ccw_r2 t q r" 
  show "ccw_r2 t p r"
  proof-
    obtain tx ty sx sy px py qx qy rx ry where
    *: "t = (tx, ty)" "s = (sx, sy)" "p = (px, py)" "q = (qx, qy)" "r = (rx, ry)"
      by (cases p, cases q, cases r, cases t, cases s, auto)
    have "(triangle_orientation t q r) * (triangle_orientation s t p) > 0"
    proof-
      have "triangle_orientation t q r > 0"
        using assms *
        unfolding ccw_r2_def
        by simp
      moreover
      have "triangle_orientation s t p > 0"
        using assms *
        unfolding ccw_r2_def
        by (simp add: triangle_orientation_def det3_def field_simps)
      ultimately
      show ?thesis
        by (simp add: mult_pos_neg)
    qed
    moreover
    have "(triangle_orientation t p q) * (triangle_orientation s t r) > 0"
    proof-
      have "triangle_orientation t p q > 0"
        using assms *
        unfolding ccw_r2_def
        by simp
      moreover
      have "triangle_orientation s t r > 0"
        using assms *
        unfolding ccw_r2_def
        by (simp add: triangle_orientation_def det3_def field_simps)
      ultimately
      show ?thesis
        by (simp add: mult_pos_neg)
    qed
    ultimately
    have "(triangle_orientation s t q) * (triangle_orientation t p r) > 0" 
      using ax5_lemma[of s t q p r]
      by simp
    moreover 
    have "triangle_orientation s t q > 0"
      using assms *
      unfolding ccw_r2_def
      by (simp add: triangle_orientation_def det3_def field_simps)
    ultimately
    show ?thesis
      unfolding ccw_r2_def
      using zero_less_mult_pos by blast
 qed
next
 fix a b c
 assume "col_r2 a b c"
 thus "col_r2 b a c"
   unfolding col_r2_def triangle_orientation_def det3_def
   by (cases a, cases b, cases c, auto simp add: field_simps)
next
 fix a b c
 assume "col_r2 a b c"
 thus "col_r2 c a b"
   unfolding col_r2_def triangle_orientation_def det3_def
   by (cases a, cases b, cases c, auto simp add: field_simps)
qed


(* ------------------------- *)

definition sort_triple where
 "sort_triple i j k = 
     (let pos = True;
         (i, j, pos) = if i < j then (i, j, pos) else (j, i, \<not> pos);
         (j, k, pos) = if j < k then (j, k, pos) else (k, j, \<not> pos);
         (i, j, pos) = if i < j then (i, j, pos) else (j, i, \<not> pos)
       in (i, j, k, pos))"

definition sort_triple' where
  "sort_triple' i j k Neg f negop = 
    (let (i, j, k, pos) = sort_triple i j k;
         pos = if Neg then pos else \<not> pos
      in if pos then f i j k else negop (f i j k))"

definition satisfies_lit :: "(nat \<Rightarrow> bool) \<Rightarrow> int \<Rightarrow> bool" where
  "satisfies_lit val lit \<longleftrightarrow> (if lit > 0 then val (nat lit) else \<not> val (nat (-lit)))"
  
definition satisfies_clause :: "(nat \<Rightarrow> bool) \<Rightarrow> int list \<Rightarrow> bool" where
  "satisfies_clause val clause \<longleftrightarrow> (\<exists> lit \<in> set clause. satisfies_lit val lit)"
  
definition satisfies_formula :: "(nat \<Rightarrow> bool) \<Rightarrow> int list list \<Rightarrow> bool" where
  "satisfies_formula val formula \<longleftrightarrow> (\<forall> clause \<in> set formula. satisfies_clause val clause)"
      
definition simplify_formula :: "int list list \<Rightarrow> int list list" where
  "simplify_formula formula = 
     (let units = filter (\<lambda> clause. length clause = 1) formula
       in units @ map (\<lambda> clause. filter (\<lambda> lit. [-lit] \<notin> set units) clause) (filter (\<lambda> clause. list_all (\<lambda> lit. [lit] \<notin> set units) clause) formula))"
       
definition lits_not_zero where
  "lits_not_zero formula \<longleftrightarrow> (\<forall> clause \<in> set formula. \<forall> lit \<in> set clause. lit \<noteq> 0)"
       
lemma satisfies_formula_simplify': 
  assumes "lits_not_zero formula" "satisfies_formula val formula" "satisfies_clause val clause"
  shows "satisfies_clause val (filter (\<lambda> lit. [-lit] \<notin> set formula) clause)" 
proof-
  from `satisfies_clause val clause` obtain lit where "lit \<in> set clause" "satisfies_lit val lit"
    by (auto simp add: satisfies_clause_def)
  moreover
  have "[-lit] \<notin> set formula"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "satisfies_lit val (-lit)" "lit \<noteq> 0"
      using `satisfies_formula val formula` assms
      by (auto simp add: satisfies_formula_def satisfies_clause_def lits_not_zero_def)
    with `satisfies_lit val lit`
    show False
      by (simp add: satisfies_lit_def split: split_if_asm)
  qed
  ultimately
  show ?thesis
    by (auto simp add: satisfies_clause_def)
qed

lemma satisfies_formula_simplify: 
  assumes "lits_not_zero formula"
  shows "satisfies_formula val formula \<longleftrightarrow> satisfies_formula val (simplify_formula formula)"
proof
  assume "satisfies_formula val (simplify_formula formula)"
  show "satisfies_formula val formula"
    unfolding satisfies_formula_def
  proof safe
    fix clause
    assume "clause \<in> set formula"
    show "satisfies_clause val clause"
    proof (cases "length clause = 1 \<or> (\<forall>lit\<in>set clause. [lit] \<notin> set formula)")
      case True
      thus "satisfies_clause val clause"
      proof
        assume "length clause = 1"
        thus ?thesis
          using `satisfies_formula val (simplify_formula formula)` `clause \<in> set formula`
          unfolding satisfies_formula_def simplify_formula_def
          by (auto simp add: list_all_iff)
      next
        assume "\<forall>lit\<in>set clause. [lit] \<notin> set formula"
        hence "satisfies_clause val (filter (\<lambda>lit. [- lit] \<notin> set formula) clause)"
          using `satisfies_formula val (simplify_formula formula)` `clause \<in> set formula`
          unfolding satisfies_formula_def simplify_formula_def
          by (auto simp add: list_all_iff)
        thus "satisfies_clause val clause"
           by (auto simp add: satisfies_clause_def)
      qed
    next
      case False
        then obtain lit where "lit \<in> set clause" "[lit] \<in> set formula"
        by auto
      hence "satisfies_clause val [lit]"
        using `satisfies_formula val (simplify_formula formula)`
        unfolding satisfies_formula_def simplify_formula_def
        by auto
      thus ?thesis
        using `lit \<in> set clause`
        by (auto simp add: satisfies_clause_def)
    qed
  qed
next
  assume *: "satisfies_formula val formula"
  show "satisfies_formula val (simplify_formula formula)"
    using satisfies_formula_simplify'[OF assms *] *
    unfolding simplify_formula_def satisfies_formula_def 
    by auto
qed
       
locale ccwcode = 
  fixes ccwcode :: "nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat"
  assumes ccwcode_inj: "\<lbrakk>p < q; q < r; r < N; p' < q'; q' < r'; r' < N; ccwcode N p q r = ccwcode N p' q' r'\<rbrakk> \<Longrightarrow> p = p' \<and> q = q' \<and> r = r'"
  assumes ccwcode_gt0: "\<lbrakk>p < q; q < r; r < N\<rbrakk> \<Longrightarrow> ccwcode N p q r > 0"
begin

definition ccwcode_var :: "nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> int" where
  "ccwcode_var N i j k = sort_triple' i j k True (\<lambda> i j k. int (ccwcode N i j k)) (\<lambda> b. - b)"
definition ccwcode_var_neg :: "nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> int" where
  "ccwcode_var_neg N i j k = - ccwcode_var N i j k"

definition ccwcode_val where
  "ccwcode_val N i j k val = sort_triple' i j k True (\<lambda> i j k. val (ccwcode N i j k)) (\<lambda> b. \<not> b)"
definition ccwcode_val_neg where
  "ccwcode_val_neg N i j k val = sort_triple' i j k False (\<lambda> i j k. val (ccwcode N i j k)) (\<lambda> b. \<not> b)"

lemma ccwcode_var_not_zero:
assumes "i \<noteq> j" "j \<noteq> k" "i \<noteq> k" "i < N" "j < N" "k < N"
shows "ccwcode_var N i j k \<noteq> 0"
proof-
  obtain i' j' k' pos' where *: "sort_triple i j k = (i', j', k', pos')"
    by (cases "sort_triple i j k", auto)
  hence "i' < j' \<and> j' < k' \<and> k' < N"
    using assms
    by (auto simp add: sort_triple_def split: split_if_asm)
  hence "ccwcode N i' j' k' > 0"
    using ccwcode_gt0
    by simp
  thus ?thesis
    unfolding ccwcode_var_def
    using *
    by (auto simp add: sort_triple'_def)
qed
  
lemma ccwcode_val_satisfies_lit:
assumes "i \<noteq> j" "j \<noteq> k" "i \<noteq> k" "i < N" "j < N" "k < N"
shows "ccwcode_val N i j k val \<longleftrightarrow> satisfies_lit val (ccwcode_var N i j k)"
using ccwcode_var_not_zero[OF assms]
unfolding ccwcode_val_def ccwcode_var_def sort_triple'_def Let_def
by (auto simp add: satisfies_lit_def split: split_if_asm)

lemma ccwcode_val_negpos: "ccwcode_val_neg N p q r val \<longleftrightarrow> \<not> ccwcode_val N p q r val"
unfolding ccwcode_val_neg_def ccwcode_val_def sort_triple'_def
by auto

lemma ccwcode_val_neg_satisfies_lit:
assumes "i \<noteq> j" "j \<noteq> k" "i \<noteq> k" "i < N" "j < N" "k < N"
shows "ccwcode_val_neg N i j k val \<longleftrightarrow> satisfies_lit val (ccwcode_var_neg N i j k)"
unfolding  ccwcode_val_negpos ccwcode_var_neg_def
using ccwcode_val_satisfies_lit[OF assms, of val] ccwcode_var_not_zero[OF assms]
by (simp add: satisfies_lit_def)

lemma ccwcode_val_negpos_perm:
assumes "distinct [p, q, r]"
shows "ccwcode_val_neg N p q r val \<longleftrightarrow> ccwcode_val N p r q val"
using assms
unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
by auto

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_hulls_nested H val \<longleftrightarrow>
   (\<forall> n < length H - 1. let N = listsum H; start = listsum (take n H) 
                         in \<forall> i < H ! n. \<forall> x. x \<ge> start + H ! n \<and> x < listsum H \<longrightarrow> 
                                          ccwcode_val N (start + i) (start + ((i + 1) mod H ! n)) x val)"
definition "formula_hulls_nested H = 
   (concat ((map (\<lambda> n. let N = listsum H; start = listsum (take n H) 
                        in concat (map (\<lambda> i. map (\<lambda> x. [ccwcode_var N (start + i) (start + (i + 1) mod H ! n) x]) 
                             [start + H ! n..<listsum H]) [0..<H!n]))) [0..<length H - 1]))"

lemma formula_hulls_nested_indices:
fixes H :: "nat list"
assumes "n < length H - 1" "i < H ! n" "x \<ge> listsum (take n H) + H ! n" "x < listsum H" "H ! n > 1"
assumes "start = listsum (take n H)" "p = start + i" "q = start + ((i + 1) mod (H ! n))"
shows "p \<noteq> x \<and> p \<noteq> q \<and> q \<noteq> x \<and> x < listsum H \<and> p < listsum H \<and> q < listsum H"
using assms mod_Suc[of i "H ! n"]
by auto

lemma formula_hulls_nested_lits_not_zero:
assumes "\<forall> n < length H - 1. H ! n > 1"
shows "lits_not_zero (formula_hulls_nested H)"
unfolding lits_not_zero_def formula_hulls_nested_def
using assms formula_hulls_nested_indices[where H=H]
using ccwcode_var_not_zero
by (auto simp add: Let_def)

lemma satisfies_formula_hulls_nested:
assumes "\<forall> n < length H - 1. H ! n > 1"
shows "satisfies_formula_hulls_nested H val \<longleftrightarrow> satisfies_formula val (formula_hulls_nested H)"
using assms formula_hulls_nested_indices[where H=H]
unfolding satisfies_formula_hulls_nested_def formula_hulls_nested_def
apply (auto simp add: Let_def  satisfies_formula_def satisfies_clause_def)
apply (erule_tac x="a" in allE, simp, erule_tac x="aa" in allE, simp, erule_tac x="clause" in allE, simp)
apply (subst ccwcode_val_satisfies_lit[symmetric], auto)
apply (erule_tac x="n" in ballE, erule_tac x="i" in ballE, erule_tac x="x" in ballE)
apply (subst ccwcode_val_satisfies_lit, auto)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_hulls_polygons H val \<longleftrightarrow>                                           
   (\<forall> n < length H. let N = listsum H; start = listsum (take n H) 
                     in \<forall> i j k. i < j \<and> j < k \<and> k < H ! n \<longrightarrow> 
                                 ccwcode_val N (start + i) (start + j) (start + k) val)"

definition "formula_hulls_polygons H = 
   concat (map (\<lambda> n. let start = listsum (take n H); N = listsum H 
                      in concat (map (\<lambda> k. concat (map (\<lambda> j. map (\<lambda> i. [ccwcode_var N (start + i) (start + j) (start + k)]) [0..<j]) [1..<k])) [2..<H!n])) [0..<length H])"

lemma listsum_take_n: 
fixes H :: "nat list"
assumes "n < length H" "p < H ! n" 
shows "listsum (take n H) + p < listsum H"
proof-
  have "listsum (take n H) + p  < listsum (take (n + 1) H)"
    using assms
    by (simp add: take_Suc_conv_app_nth)
  moreover
  have "listsum (take (n + 1) H) \<le> listsum ((take (n + 1) H) @ (drop (n+1) H))"
    by (simp only: listsum_append)
  hence "listsum (take (n + 1) H) \<le> listsum H"
    by auto
  ultimately
  show ?thesis
    by simp
qed

lemma formula_hulls_polygons_lits_not_zero:
"lits_not_zero (formula_hulls_polygons H)"
unfolding lits_not_zero_def formula_hulls_polygons_def Let_def
using ccwcode_var_not_zero
by (auto simp add: listsum_take_n)

lemma satisfies_formula_hulls_polygons:
shows "satisfies_formula_hulls_polygons H val \<longleftrightarrow> satisfies_formula val (formula_hulls_polygons H)"
unfolding satisfies_formula_hulls_polygons_def formula_hulls_polygons_def
apply (auto simp add: Let_def  satisfies_formula_def satisfies_clause_def)
apply (erule_tac x="a" in allE, simp, erule_tac x="clause" in allE, erule_tac x="ab" in allE, erule_tac x="aa" in allE)
apply (subst ccwcode_val_satisfies_lit[symmetric], auto simp add: listsum_take_n)
apply (erule_tac x="n" in ballE, erule_tac x="k" in ballE, erule_tac x="j" in ballE, erule_tac x="i" in ballE)
apply (subst ccwcode_val_satisfies_lit, auto simp add: listsum_take_n)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_ax4 n val \<longleftrightarrow>                                  
   (\<forall> p q r t. {p, q, r, t} \<subseteq> set [0..<n] \<and> p < q \<and> q < r \<and> r < t \<longrightarrow> 
               (ccwcode_val_neg n t q r val \<or> ccwcode_val_neg n p t r val \<or> ccwcode_val_neg n p q t val \<or> ccwcode_val n p q r val) \<and> 
               (ccwcode_val_neg n q t r val \<or> ccwcode_val_neg n t p r val \<or> ccwcode_val_neg n q p t val \<or> ccwcode_val n q p r val))"

definition "formula_ax4 n =
     concat (map (\<lambda> p. concat (map (\<lambda> q. concat (map (\<lambda> r. concat (map (\<lambda> t. 
     [[ccwcode_var_neg n t q r, ccwcode_var_neg n p t r, ccwcode_var_neg n p q t, ccwcode_var n p q r],
      [ccwcode_var_neg n q t r, ccwcode_var_neg n t p r, ccwcode_var_neg n q p t, ccwcode_var n q p r]
     ]) [r+1..<n])) [q+1..<n])) [p+1..<n])) [0..<n])"
               
lemma satisfies_formula_ax4_full: 
   "satisfies_formula_ax4 n val \<longleftrightarrow>                                  
   (\<forall> p q r t. {p, q, r, t} \<subseteq> set [0..<n] \<and> distinct [p, q, r, t] \<longrightarrow> 
               (ccwcode_val_neg n t q r val \<or> ccwcode_val_neg n p t r val \<or> ccwcode_val_neg n p q t val \<or> ccwcode_val n p q r val))" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?rhs"
  thus ?lhs
    unfolding satisfies_formula_ax4_def
    by auto (smt leD less_imp_le_nat less_trans)
next
  assume ?lhs
  hence *: "\<forall>p q r t.
       {p, q, r, t} \<subseteq> set [0..<n] \<and> distinct [p, q, r, t] \<and> p < q \<and> q < r \<and> r < t \<longrightarrow>
       (ccwcode_val_neg n t q r val \<or> ccwcode_val_neg n p t r val \<or> ccwcode_val_neg n p q t val \<or> ccwcode_val n p q r val) \<and>
       (ccwcode_val_neg n q t r val \<or> ccwcode_val_neg n t p r val \<or> ccwcode_val_neg n q p t val \<or> ccwcode_val n q p r val)"
         unfolding satisfies_formula_ax4_def
         by simp
  show ?rhs
  proof safe
     fix p q r t
     assume **: "{p, q, r, t} \<subseteq> set [0..<n]" "distinct [p, q, r, t]" 
     "\<not> ccwcode_val_neg n t q r val" "\<not> ccwcode_val_neg n p t r val" "\<not> ccwcode_val n p q r val"
     {
       assume "p < q \<and> q < r \<and> r < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p q r t] **
         by auto
     }
     moreover
     {
       assume "p < q \<and> q < t \<and> t < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p q t r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover     
     {
       assume "p < r \<and> r < q \<and> q < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p r q t] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "p < r \<and> r < t \<and> t < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p r t q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "p < t \<and> t < q \<and> q < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p t q r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "p < t \<and> t < r \<and> r < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of p t r q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }     
     moreover
     {
       assume "q < p \<and> p < r \<and> r < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q p r t] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "q < p \<and> p < t \<and> t < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q p t r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "q < r \<and> r < p \<and> p < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q r p t] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }     
     moreover
     {
       assume "q < r \<and> r < t \<and> t < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q r t p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "q < t \<and> t < p \<and> p < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q t p r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "q < t \<and> t < r \<and> r < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of q t r p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "r < p \<and> p < q \<and> q < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r p q t] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "r < p \<and> p < t \<and> t < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r p t q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "r < q \<and> q < p \<and> p < t"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r q p t] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }     
     moreover
     {
       assume "r < q \<and> q < t \<and> t < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r q t p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "r < t \<and> t < p \<and> p < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r t p q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "r < t \<and> t < q \<and> q < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of r t q p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "t < p \<and> p < q \<and> q < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t p q r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "t < p \<and> p < r \<and> r < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t p r q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "t < q \<and> q < p \<and> p < r"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t q p r] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }     
     moreover
     {
       assume "t < q \<and> q < r \<and> r < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t q r p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "t < r \<and> r < p \<and> p < q"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t r p q] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     {
       assume "t < r \<and> r < q \<and> q < p"
       hence "ccwcode_val_neg n p q t val"
         using *[rule_format, of t r q p] **
         unfolding ccwcode_val_def ccwcode_val_neg_def sort_triple'_def sort_triple_def
         by auto
     }
     moreover
     have "p \<noteq> q" "p \<noteq> r" "p \<noteq> t" "q \<noteq> r" "q \<noteq> t" "r \<noteq> t"
       using `distinct [p, q, r, t]`
       by auto
     ultimately
     show "ccwcode_val_neg n p q t val"
       by (metis nat_neq_iff)
     qed
qed

lemma formula_ax4_lits_not_zero:
shows "lits_not_zero (formula_ax4 n)"
using ccwcode_var_not_zero
unfolding lits_not_zero_def formula_ax4_def Let_def ccwcode_var_neg_def
by simp

lemma satisfies_formula_ax4: 
"satisfies_formula_ax4 n val \<longleftrightarrow> satisfies_formula val (formula_ax4 n)"
unfolding satisfies_formula_ax4_def formula_ax4_def
apply (simp add: Let_def satisfies_formula_def satisfies_clause_def)
apply (auto simp add: ccwcode_val_satisfies_lit ccwcode_val_neg_satisfies_lit)
apply (erule_tac x=a in allE, erule_tac x=aa in allE, erule_tac x=ab in allE, erule_tac x=ac in allE, simp)
apply (erule_tac x=a in allE, erule_tac x=aa in allE, erule_tac x=ab in allE, erule_tac x=ac in allE, simp)
apply (erule_tac x=p in ballE, erule_tac x=q in ballE, erule_tac x=r in ballE, erule_tac x=t in ballE, simp_all)
apply (erule_tac x=p in ballE, erule_tac x=q in ballE, erule_tac x=r in ballE, erule_tac x=t in ballE, simp_all)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_ax5 n val \<longleftrightarrow>                                  
   (\<forall> p q r s t. {p, q, r, s, t} \<subseteq> set [0..<n] \<and> distinct [p, q, r, s, t] \<and> p < q \<and> q < r \<longrightarrow> 
     (ccwcode_val_neg n t s p val \<or> ccwcode_val_neg n t s q val \<or> ccwcode_val_neg n t s r val \<or> ccwcode_val_neg n t p q val \<or> ccwcode_val_neg n t q r val \<or> ccwcode_val n t p r val) \<and>
     (ccwcode_val_neg n t s p val \<or> ccwcode_val_neg n t s q val \<or> ccwcode_val_neg n t s r val \<or> ccwcode_val_neg n t q p val \<or> ccwcode_val_neg n t r q val \<or> ccwcode_val n t r p val)
     )" 
     
definition "formula_ax5 n =
     concat (map (\<lambda> p. concat (map (\<lambda> q. concat (map (\<lambda> r. concat (map (\<lambda> s. concat (map (\<lambda> t. 
       [[ccwcode_var_neg n t s p, ccwcode_var_neg n t s q, ccwcode_var_neg n t s r, ccwcode_var_neg n t p q, ccwcode_var_neg n t q r, ccwcode_var n t p r],
        [ccwcode_var_neg n t s p, ccwcode_var_neg n t s q, ccwcode_var_neg n t s r, ccwcode_var_neg n t q p, ccwcode_var_neg n t r q, ccwcode_var n t r p]
       ]) 
       (filter (\<lambda> x. x \<noteq> p \<and> x \<noteq> q \<and> x \<noteq> r \<and> x \<noteq> s) [0..<n]))) (filter (\<lambda> x. x \<noteq> p \<and> x \<noteq> q \<and> x \<noteq> r) [0..<n]))) ([q+1..<n]))) [p+1..<n])) [0..<n])"
       
lemma satisfies_formula_ax5_full: "satisfies_formula_ax5 n val \<longleftrightarrow>                                  
   (\<forall> p q r s t. {p, q, r, s, t} \<subseteq> set [0..<n] \<and> distinct [p, q, r, s, t] \<longrightarrow> 
     ccwcode_val_neg n t s p val \<or> ccwcode_val_neg n t s q val \<or> ccwcode_val_neg n t s r val \<or>
     ccwcode_val_neg n t p q val \<or> ccwcode_val_neg n t q r val \<or> ccwcode_val n t p r val)" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume ?rhs
  thus ?lhs
    unfolding satisfies_formula_ax5_def
  proof safe
    fix p q r s t
    assume "?rhs" and
    *: "{p, q, r, s, t} \<subseteq> set [0..<n]" "distinct [p, q, r, s, t]" "p < q" "q < r" and
    ass: "\<not> ccwcode_val_neg n t s p val" "\<not> ccwcode_val_neg n t s q val" "\<not> ccwcode_val_neg n t s r val"
    "\<not> ccwcode_val_neg n t q p val" "\<not> ccwcode_val n t r p val"
    
    have "{q, p, r, s, t} \<subseteq> set [0..<n] \<and> distinct [q, p, r, s, t] \<and> distinct [t, r, q] \<and> distinct [t, p, r]"
      using *(1-2)
      by auto
    thus "ccwcode_val_neg n t r q val"
      using `?rhs`[rule_format, of q p r s t] 
      using ass
      using ccwcode_val_negpos_perm[of t p r n val]
      using ccwcode_val_negpos_perm[of t r q n val]
      by blast
  qed blast
next
  assume "?lhs"
  hence *: " \<forall>p q r s t.
     {p, q, r, s, t} \<subseteq> set [0..<n] \<and> distinct [p, q, r, s, t] \<and> p < q \<and> q < r \<longrightarrow>
     (ccwcode_val_neg n t s p val \<or>
      ccwcode_val_neg n t s q val \<or>
      ccwcode_val_neg n t s r val \<or> ccwcode_val_neg n t p q val \<or> ccwcode_val_neg n t q r val \<or> ccwcode_val n t p r val) \<and>
     (ccwcode_val_neg n t s p val \<or>
      ccwcode_val_neg n t s q val \<or>
      ccwcode_val_neg n t s r val \<or> ccwcode_val_neg n t q p val \<or> ccwcode_val_neg n t r q val \<or> ccwcode_val n t r p val)"
    unfolding satisfies_formula_ax5_def
    by auto
  show "?rhs"
  proof safe
    fix p q r s t
    assume **: "{p, q, r, s, t} \<subseteq> set [0..<n]" "distinct [p, q, r, s, t]" 
    "\<not> ccwcode_val_neg n t s p val" "\<not> ccwcode_val_neg n t s q val" "\<not> ccwcode_val_neg n t s r val" "\<not> ccwcode_val_neg n t p q val"
    "\<not> ccwcode_val n t p r val"
    hence ***: "distinct [t, p, q]" "distinct [t, q, p]" "distinct [t, r, q]" "distinct [t, q, r]" "distinct [t, p, r]" "distinct [t, r, p]"
      by auto
    {
      assume "p < q" "q < r"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of p q r s t] **
        by simp
    }
    moreover
    {
      assume "p < r" "r < q"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of p r q s t] **
        using ccwcode_val_negpos_perm[of t r p n val, OF ***(6)]
        using ccwcode_val_negpos_perm[of t p q n val, OF ***(1)]
        by simp
    }
    moreover
    {
      assume "q < p" "p < r"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of q p r s t] **
        using ccwcode_val_negpos_perm[of t q r n val, OF ***(4)]
        using ccwcode_val_negpos_perm[of t r p n val, OF ***(6)]
        by simp
    }
    moreover
    {
      assume "q < r" "r < p"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of q r p s t] **
        using ccwcode_val_negpos_perm[of t p q n val, OF ***(1)]
        using ccwcode_val_negpos_perm[of t r p n val, OF ***(6)]
        by simp      
    }
    moreover
    {
      assume "r < p" "p < q"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of r p q s t] **
        using ccwcode_val_negpos_perm[of t q r n val, OF ***(4)]
        using ccwcode_val_negpos_perm[of t r p n val, OF ***(6)]
        by simp
    }
    moreover
    {
      assume "r < q" "q < p"
      hence "ccwcode_val_neg n t q r val"
        using *[rule_format, of r q p s t] **
        using ccwcode_val_negpos_perm[of t q r n val, OF ***(4)]
        using ccwcode_val_negpos_perm[of t r p n val, OF ***(6)]
        by simp
    }
    moreover
    have "p \<noteq> q" "p \<noteq> r" "q \<noteq> r"
      using **
      by auto
    ultimately
    show "ccwcode_val_neg n t q r val"
       by (metis nat_neq_iff)
  qed
qed

lemma formula_ax5_lits_not_zero:
shows "lits_not_zero (formula_ax5 n)"
unfolding lits_not_zero_def formula_ax5_def Let_def ccwcode_var_neg_def
using ccwcode_var_not_zero
by auto

lemma satisfies_formula_ax5: 
  "satisfies_formula_ax5 n val \<longleftrightarrow> satisfies_formula val (formula_ax5 n)"
unfolding satisfies_formula_ax5_def formula_ax5_def
apply (simp add: satisfies_formula_def satisfies_clause_def Let_def)
apply safe
apply (erule_tac x=a in allE, erule_tac x=aa in allE, erule_tac x=ab in allE, erule_tac x=ac in allE, erule_tac x=ad in allE, simp add: ccwcode_val_satisfies_lit ccwcode_val_neg_satisfies_lit)
apply (erule_tac x=a in allE, erule_tac x=aa in allE, erule_tac x=ab in allE, erule_tac x=ac in allE, erule_tac x=ad in allE, simp add: ccwcode_val_satisfies_lit ccwcode_val_neg_satisfies_lit)
apply (erule_tac x=p in ballE, erule_tac x=q in ballE, erule_tac x=r in ballE, erule_tac x=s in allE, simp, erule_tac x=t in allE, simp add: ccwcode_val_satisfies_lit ccwcode_val_neg_satisfies_lit, simp_all)
apply (erule_tac x=p in ballE, erule_tac x=q in ballE, erule_tac x=r in ballE, erule_tac x=s in allE, simp, erule_tac x=t in allE, simp add: ccwcode_val_satisfies_lit ccwcode_val_neg_satisfies_lit, simp_all)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_no_polygon m n val \<longleftrightarrow>
  (\<forall> p. length p = m \<and> set p \<subseteq> set [0..<n] \<and> distinct p \<and> min_hd p \<longrightarrow> 
         (\<exists> i j k. i < j \<and> j < k \<and> k < length p \<and> ccwcode_val_neg n (p ! i) (p ! j) (p ! k) val))"

definition "formula_no_polygon m n = 
  map (\<lambda> p. concat (map (\<lambda> k. concat (map (\<lambda> j. map (\<lambda> i. 
       ccwcode_var_neg n (p ! i) (p ! j) (p ! k)) [0..<j]) [1..<k])) [2..<m])) (variations_min_hd m n)"
  
lemma formula_no_polygon_lits_not_zero:
assumes "m > 0"
shows "lits_not_zero (formula_no_polygon m n)"
unfolding lits_not_zero_def
proof (simp add: formula_no_polygon_def satisfies_clause_def ccwcode_var_neg_def, safe)
  fix clause p q r
  assume *: "clause \<in> set (variations_min_hd m n)" "r \<in> {2..<m}" "q \<in> {Suc 0..<r}" "p \<in> {0..<q}"
  "ccwcode_var n (clause ! p) (clause ! q) (clause ! r) = 0"
  hence "length clause = m" "set clause \<subseteq> {0..<n}" "distinct clause"
    using set_variations_min_hd[of m n, OF assms]
    by auto
  moreover
  have "clause ! p \<in> set clause" "clause ! q \<in> set clause" "clause ! r \<in> set clause"
     using in_set_conv_nth[of "clause ! p" clause]
     using in_set_conv_nth[of "clause ! q" clause]
     using in_set_conv_nth[of "clause ! r" clause]
     using *(2-4) `length clause = m`
     by auto
  ultimately
  have "clause ! p \<noteq> clause ! q" "clause ! q \<noteq> clause ! r" "clause ! p \<noteq> clause ! r" 
        "clause ! p < n" "clause ! q < n" "clause ! r < n"
     using nth_eq_iff_index_eq[of clause] `set clause \<subseteq> {0..<n}` `distinct clause` *
     by - force+
  thus False
     using ccwcode_var_not_zero[of "clause ! p" "clause ! q" "clause ! r" n] *
     by simp
qed
       
lemma satisfies_formula_no_polygon:
assumes "m > 0"
shows "satisfies_formula_no_polygon m n val \<longleftrightarrow> satisfies_formula val (formula_no_polygon m n)"
proof
  assume "satisfies_formula_no_polygon m n val"
  show "satisfies_formula val (formula_no_polygon m n)"
  proof (simp add: formula_no_polygon_def satisfies_formula_def satisfies_clause_def, safe)
    fix clause
    assume "clause \<in> set (variations_min_hd m n)"
    hence *: "distinct clause \<and> min_hd clause \<and> length clause = m \<and> set clause \<subseteq> set [0..<n]"
      using set_variations_min_hd[of m n] `m > 0`
      by auto
    then obtain p q r where "p < q \<and> q < r \<and> r < length clause" "ccwcode_val_neg n (clause ! p) (clause ! q) (clause ! r) val"      
      using `satisfies_formula_no_polygon m n val`
      unfolding satisfies_formula_no_polygon_def
      by auto
    hence "satisfies_lit val (ccwcode_var_neg n (clause ! p) (clause ! q) (clause ! r))"
      using *
      apply (subst ccwcode_val_neg_satisfies_lit[symmetric])
      apply (auto simp add: distinct_conv_nth)
      apply (metis atLeast0LessThan lessThan_iff less_trans nth_mem subsetCE)+
      done
    thus  "\<exists>r\<in>{2..<m}.
              \<exists>q\<in>{Suc 0..<r}.
                 \<exists>p\<in>{0..<q}. satisfies_lit val (ccwcode_var_neg n (clause ! p) (clause ! q) (clause ! r)) "
      using `p < q \<and> q < r \<and> r < length clause` *
      by force
  qed
  next
    assume "satisfies_formula val (formula_no_polygon m n)"
    show "satisfies_formula_no_polygon m n val"
      unfolding satisfies_formula_no_polygon_def
    proof safe
      fix p
      assume "m = length p" "set p \<subseteq> set [0..<n]" "distinct p" "list_all (op < (hd p)) (tl p)"
      hence "p \<in> set (variations_min_hd m n)"
        using set_variations_min_hd[of m n] `m > 0`
        by auto
      then obtain i j k where
      "satisfies_lit val (ccwcode_var_neg n (p ! i) (p ! j) (p ! k))" "k \<in> {2..<m}" "j \<in> {1..<k}" "i \<in> {0..<j}"
        using `satisfies_formula val (formula_no_polygon m n)` `list_all (op < (hd p)) (tl p)`
        by (auto simp add: satisfies_formula_def satisfies_clause_def formula_no_polygon_def)
      hence "ccwcode_val_neg n (p ! i) (p ! j) (p ! k) val"
        using `distinct p` `set p \<subseteq> set [0..<n]` `m = length p`
        apply (subst ccwcode_val_neg_satisfies_lit)
        apply (auto simp add: distinct_conv_nth)
        using contra_subsetD apply fastforce
        using contra_subsetD apply fastforce
        using contra_subsetD apply fastforce
        done
      thus "\<exists>i j k. i < j \<and> j < k \<and> k < length p \<and> ccwcode_val_neg n (p ! i) (p ! j) (p ! k) val"
        using `m = length p` `k \<in> {2..<m}` `j \<in> {1..<k}` `i \<in> {0..<j}`
        by simp blast
    qed
qed
     
(* ----------------------------------------------------------------- *)

lemma listsum_take_n':
fixes H :: "nat list"
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0" "n < length H - 1" "p \<le> H ! n" 
shows "listsum (take n H) + p < listsum H" 
proof-
  have "listsum (take (n + 2) H) = listsum (take n H) + H ! n + H ! (n + 1)"
    using `n < length H - 1`
    by (simp add: take_Suc_conv_app_nth)
moreover
  have "listsum (take (n + 2) H) \<le> listsum ((take (n + 2) H) @ (drop (n+2) H))"
    by (simp only: listsum_append)
  hence "listsum (take (n + 2) H) \<le> listsum H"
    by auto
moreover
  have "H ! (n + 1) > 0"
    using `\<forall> i < length H. H ! i > 0` `n < length H - 1` `H \<noteq> []`
    by auto
ultimately
  show ?thesis
    using `p \<le> H ! n`
    by auto
qed

lemma listsum_take_n'':
fixes H :: "nat list"
assumes "H \<noteq> []" "n < length H - 1" "p < H ! n + H ! (n+1)" 
shows "listsum (take n H) + p < listsum H" 
proof-
  have "listsum (take (n + 2) H) = listsum (take n H) + H ! n + H ! (n + 1)"
    using `n < length H - 1`
    by (simp add: take_Suc_conv_app_nth)
moreover
  have "listsum (take (n + 2) H) \<le> listsum ((take (n + 2) H) @ (drop (n+2) H))"
    by (simp only: listsum_append)
  hence "listsum (take (n + 2) H) \<le> listsum H"
    by auto
ultimately
  show ?thesis
    using `p < H ! n + H ! (n+1)`
    by auto
qed

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_all_3_1 H val \<longleftrightarrow>
  (\<forall> i < length H - 1. H ! i = 3 \<and> H ! (i+1) \<ge> 2 \<longrightarrow>
      (let n = listsum H; start = listsum (take i H)
        in ccwcode_val_neg n (start + 3) (start + 4) (start + 0) val \<and> 
           ccwcode_val n (start + 3) (start + 4) (start + 1) val
      )
  )
"
definition "formula_all_3_1 H = 
  concat (map (\<lambda> i. if H ! i = 3 \<and> H ! (i+1) \<ge> 2 then let n = listsum H; start = listsum (take i H) in 
    [[ccwcode_var_neg n (start + 3) (start + 4) (start + 0)], 
     [ccwcode_var n (start + 3) (start + 4) (start + 1)]] else [])  [0..<length H - 1])"

lemma formula_all_3_1_lits_not_zero:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "lits_not_zero(formula_all_3_1 H)"
 unfolding lits_not_zero_def
 using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n''[OF assms(1), of _ 4]
 using ccwcode_var_not_zero
 unfolding formula_all_3_1_def
 by (auto simp add: Let_def ccwcode_var_neg_def)
     
lemma satisfies_formula_all_3_1:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "satisfies_formula_all_3_1 H val \<longleftrightarrow> satisfies_formula val (formula_all_3_1 H)"
unfolding satisfies_formula_all_3_1_def formula_all_3_1_def
using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n''[OF assms(1), of _ 4]
apply (auto simp add: satisfies_formula_def satisfies_clause_def Let_def)
apply (subst ccwcode_val_neg_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_neg_satisfies_lit, simp_all)
apply (subst ccwcode_val_satisfies_lit, simp_all)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_all_4_1 H val \<longleftrightarrow> 
   (\<forall> i < length H - 1.  H ! i = 4 \<longrightarrow> 
       (let n = listsum H; start = listsum (take i H) 
         in ccwcode_val_neg n start (start + 2) (start + 4) val \<and> 
            ccwcode_val n (start + 1) (start + 3) (start + 4) val))"

definition "formula_all_4_1 H = 
  concat (map (\<lambda> i. if H ! i = 4 then let n = listsum H; start = listsum (take i H) in 
    [[ccwcode_var_neg n start (start + 2) (start + 4)], 
     [ccwcode_var n (start + 1) (start + 3) (start + 4)]] else [])  [0..<length H - 1])"

lemma formula_all_4_1_lits_not_zero:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "lits_not_zero(formula_all_4_1 H)"
 unfolding lits_not_zero_def
 using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n'[OF assms, of _ 4]
 using ccwcode_var_not_zero
 unfolding formula_all_4_1_def
 by (auto simp add: Let_def ccwcode_var_neg_def)
     
lemma satisfies_formula_all_4_1:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "satisfies_formula_all_4_1 H val \<longleftrightarrow> satisfies_formula val (formula_all_4_1 H)"
unfolding satisfies_formula_all_4_1_def formula_all_4_1_def
using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n'[OF assms, of _ 4]
apply (auto simp add: satisfies_formula_def satisfies_clause_def Let_def)
apply (subst ccwcode_val_neg_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_neg_satisfies_lit, simp_all)
apply (subst ccwcode_val_satisfies_lit, simp_all)
done

(* ----------------------------------------------------------------- *)

definition "satisfies_formula_all_5_canon H val \<longleftrightarrow> 
   (\<forall> i < length H - 1.  H ! i = 5 \<longrightarrow> 
       (let n = listsum H; start = listsum (take i H) 
         in ccwcode_val_neg n (start + 0) (start + 3) (start + 5) val \<and> 
            ccwcode_val n (start + 1) (start + 3) (start + 5) val \<and> 
            ccwcode_val n (start + 2) (start + 4) (start + 5) val \<and>
           (ccwcode_val n (start + 0) (start + 2) (start + 5) val \<or> 
            ccwcode_val n (start + 1) (start + 4) (start + 5) val)))"

definition "formula_all_5_canon H = 
  concat (map (\<lambda> i. if H ! i = 5 then let n = listsum H; start = listsum (take i H) in 
     [[ccwcode_var_neg n (start + 0) (start + 3) (start + 5)], 
      [ccwcode_var n (start + 1) (start + 3) (start + 5)],
      [ccwcode_var n (start + 2) (start + 4) (start + 5)],
      [ccwcode_var n (start + 0) (start + 2) (start + 5), ccwcode_var n (start + 1) (start + 4) (start + 5)]
     ] else [])  [0..<length H - 1])"

lemma formula_all_5_canon_lits_not_zero:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "lits_not_zero (formula_all_5_canon H)"
 unfolding lits_not_zero_def
 using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n'[OF assms, of _ 4] listsum_take_n'[OF assms, of _ 5]
 using ccwcode_var_not_zero
 unfolding formula_all_5_canon_def
 by (auto simp add: Let_def ccwcode_var_neg_def)
     
lemma satisfies_formula_all_5_canon:
assumes "H \<noteq> []" "\<forall> i < length H. H ! i > 0"
shows "satisfies_formula_all_5_canon H val \<longleftrightarrow> satisfies_formula val (formula_all_5_canon H)"
unfolding satisfies_formula_all_5_canon_def formula_all_5_canon_def
using listsum_take_n'[OF assms, of _ 0] listsum_take_n'[OF assms, of _ 1] listsum_take_n'[OF assms, of _ 2] listsum_take_n'[OF assms, of _ 3] listsum_take_n'[OF assms, of _ 4] listsum_take_n'[OF assms, of _ 5]
apply (auto simp add: satisfies_formula_def satisfies_clause_def Let_def)
apply (subst ccwcode_val_neg_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_satisfies_lit[symmetric], simp_all)
apply (subst ccwcode_val_satisfies_lit[symmetric], simp_all)
apply (subst (asm) ccwcode_val_satisfies_lit[symmetric], simp_all, force)
apply (subst ccwcode_val_neg_satisfies_lit, simp_all)
apply (subst ccwcode_val_satisfies_lit, simp_all)
apply (subst ccwcode_val_satisfies_lit, simp_all)
apply (subst ccwcode_val_satisfies_lit, simp_all)
apply (subst (asm) ccwcode_val_satisfies_lit, simp_all, force)
done

(* ----------------------------------------------------------------- *)

definition "satisfies H m val \<longleftrightarrow> 
               satisfies_formula_hulls_nested H val \<and>
               satisfies_formula_hulls_polygons H val \<and>
               satisfies_formula_ax4 (listsum H) val \<and>
               satisfies_formula_ax5 (listsum H) val \<and>
               satisfies_formula_no_polygon m (listsum H) val \<and>
               satisfies_formula_all_3_1 H val \<and> 
               satisfies_formula_all_4_1 H val \<and>
               satisfies_formula_all_5_canon H val" 

definition formula where
"formula m H = 
   (formula_hulls_nested H) @ 
   (formula_hulls_polygons H) @ 
   (formula_ax4 (listsum H)) @
   (formula_ax5 (listsum H)) @
   (formula_no_polygon m (listsum H)) @
   (formula_all_3_1 H) @ 
   (formula_all_4_1 H) @ 
   (formula_all_5_canon H)"
   
definition "simplified_formula m H = simplify_formula (formula m H)"   

lemma lits_not_zero_simplify_formula:
assumes "lits_not_zero fml"
shows "lits_not_zero (simplify_formula fml)"
using assms
unfolding simplify_formula_def lits_not_zero_def
by force

lemma lits_not_zero_append:
shows "lits_not_zero (f1 @ f2) \<longleftrightarrow> lits_not_zero f1 \<and> lits_not_zero f2"
unfolding lits_not_zero_def
by auto
   
lemma formula_lits_not_zero:
  assumes  "H \<noteq> []" "\<forall>n < length H - 1. H ! n > 1" "H ! (length H - 1) > 0" "m > 0"
  shows "lits_not_zero (formula m H)"
proof-
  have *: "\<forall>n < length H. H ! n > 0"
    using assms
    by (metis One_nat_def Suc_pred gr0I length_greater_0_conv less_antisym not_one_less_zero)
  show ?thesis
    unfolding formula_def
    using formula_hulls_nested_lits_not_zero[of H, OF assms(2)]
    using formula_hulls_polygons_lits_not_zero[of H]
    using formula_ax4_lits_not_zero[of "listsum H"]
    using formula_ax5_lits_not_zero[of "listsum H"]
    using formula_no_polygon_lits_not_zero[of m "listsum H", OF assms(4)]
    using formula_all_3_1_lits_not_zero[of H, OF assms(1) *]
    using formula_all_4_1_lits_not_zero[of H, OF assms(1) *]
    using formula_all_5_canon_lits_not_zero[of H, OF assms(1) *]
    by (auto simp add: lits_not_zero_append)
qed  
  
lemma satisfies_formula_append: 
"satisfies_formula val (f1 @ f2) \<longleftrightarrow> satisfies_formula val f1 \<and> satisfies_formula val f2"
by (auto simp add: satisfies_formula_def)
   
lemma satisfies:
  assumes  "H \<noteq> []" "\<forall>n < length H - 1. H ! n > 1" "H ! (length H - 1) > 0" "m > 0"
  shows "satisfies H m val \<longleftrightarrow> satisfies_formula val (simplified_formula m H)"
proof-
  have "\<forall>n < length H. H ! n > 0"
    using assms
    by (metis One_nat_def Suc_pred gr0I length_greater_0_conv less_antisym not_one_less_zero)
  show ?thesis
    unfolding simplified_formula_def
    using formula_lits_not_zero[OF assms]
    unfolding satisfies_def formula_def
    using satisfies_formula_hulls_nested[of H val, OF assms(2)]
    using satisfies_formula_hulls_polygons[of H val]
    using satisfies_formula_ax4[of "listsum H" val]
    using satisfies_formula_ax5[of "listsum H" val]
    using satisfies_formula_no_polygon[of m "listsum H" val, OF `m > 0`]
    using satisfies_formula_all_3_1[of H val, OF assms(1) `\<forall>n < length H. H ! n > 0`]
    using satisfies_formula_all_4_1[of H val, OF assms(1) `\<forall>n < length H. H ! n > 0`]
    using satisfies_formula_all_5_canon[of H val, OF assms(1) `\<forall>n < length H. H ! n > 0`]
    by (subst satisfies_formula_simplify[symmetric]) (simp_all add: satisfies_formula_append)
qed

end

definition triple_index where
  "triple_index N i j k =
     listsum (map (\<lambda> n. ((N - n) * (N - n - 1)) div 2) [1 ..< i+1]) +
     (j - i - 1) * (2 * N - j - i - 2) div 2 + 
     k - j"
     
lemma triple_index_mono_k: 
  assumes "i < j" "j < k" "k < k'" "k' < N"
  shows "triple_index N i j k' > triple_index N i j k"
using assms  
unfolding triple_index_def
by simp

lemma triple_index_j_inc:
assumes "i < j" "j + 2 < N"
shows "triple_index N i j (N-1) + 1 = triple_index N i (j + 1) (j + 2)"
proof-
  have "(j - i - 1) * (2 * N - j - i - 2) div 2 + (N - 1) - j + 1 =
        (j + 1 - i - 1) * (2 * N - (j + 1) - i - 2) div 2 + (j + 2) - (j + 1)" (is "?lhs = ?rhs")
  proof-
    have "?lhs = (j - i - 1) * (2 * N - j - i - 2) div 2 + (N - j)"
      using assms
      by simp
    also have "... = (j - i) * (2 * N - j - i - 3) div 2 + 1"
    proof-
      have "(j - i - 1) * (2 * N - j - i - 2) + 2*(N - j) = (j - i) * (2 * N - j - i - 3) + 2"
      proof (subst int_int_eq[symmetric])
        have "(int j - int i - 1) * (2 * int N - int j - int i - 2) + 2*(int N - int j) = (int j - int i) * (2 * int N - int j - int i - 3) + 2"
          by (simp add: int_distrib)
        thus "int ((j - i - 1) * (2 * N - j - i - 2) + 2 * (N - j)) = int ((j - i) * (2 * N - j - i - 3) + 2)"
          using assms zdiff_int[of "Suc i" j, symmetric] zdiff_int[of j N, symmetric] zdiff_int[of i j, symmetric] zdiff_int[of "j+i+2" "2*N", symmetric] zdiff_int[of "j+i+3" "2*N", symmetric]
          by (simp add: int_mult int_distrib)
      qed
      moreover
      have "2 dvd (j - i - 1) * (2 * N - j - i - 2)"
        using assms
        by auto
      moreover
      have "2 dvd (j - i) * (2 * N - j - i - 3)"
        using assms
        by auto
      ultimately
      show ?thesis
        using div_add[of 2 "(j - i - 1) * (2 * N - j - i - 2)" "2*(N-j)"]
        using div_add[of 2 "(j - i) * (2 * N - j - i - 3)" "2"]
        by simp
    qed
    also have "... = ?rhs"
      using assms
      by (simp add: numeral_3_eq_3)
    finally show ?thesis
      .
  qed
  moreover 
  have "(\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + (j - i - 1) * (2 * N - j - i - 2) div 2 + (N - 1) - j + 1
   = (\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + ((j - i - 1) * (2 * N - j - i - 2) div 2 + (N - 1) - j + 1)"
     using assms
     by simp
  moreover
  have " (\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + (j + 1 - i - 1) * (2 * N - (j + 1) - i - 2) div 2 + (j + 2) -
    (j + 1) =  (\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + ((j + 1 - i - 1) * (2 * N - (j + 1) - i - 2) div 2 + (j + 2) -
    (j + 1))"
     using assms
     by simp
  ultimately
  show ?thesis
    using assms
    unfolding triple_index_def
    by simp
qed

lemma triple_index_mono_j: 
  assumes "i < j" "j < k" "k < N" 
          "j < j'" 
          "i < j'" "j' < k'" "k' < N"
  shows "triple_index N i j' k' > triple_index N i j k"
using assms  
proof (induct "j' - j" arbitrary: j k k')
  case 0
  thus ?case
    by simp
next
  case (Suc d')
    have "k = N - 1 \<or> k < N - 1"
      using Suc
      by auto
    hence "triple_index N i j k \<le> triple_index N i j (N - 1)"
      using triple_index_mono_k[of i j k "N-1" N] Suc
      by auto
    moreover
    have "triple_index N i j (N - 1) < triple_index N i (j + 1) (j + 2)"
      using triple_index_j_inc[of i j N] Suc
      by simp
    moreover
    have "triple_index N i (j + 1) (j + 2) \<le> triple_index N i (j + 1) k'"
      using triple_index_mono_k[of i "j+1" "j+2" k' N] Suc
      by (cases "k' = j + 2") auto
    ultimately
    have *: "triple_index N i j k < triple_index N i (j + 1) k'"
      by simp
    show ?case
    proof (cases "j' = j + 1")
      case True
      thus ?thesis
        using *
        by simp
    next
      case False
      hence "triple_index N i (j + 1) k' < triple_index N i j' k'"
        using Suc(1)[of "j+1" k' k'] Suc(2-9) 
        by auto
      thus ?thesis
        using *
        by simp
  qed
qed

lemma triple_index_i_inc:
assumes "i + 3 < N"
shows "triple_index N i (N-2) (N-1) + 1 = triple_index N (i + 1) (i + 2) (i + 3)"
proof-
  have "triple_index N i (N-2) (N-1) + 1 = 
        (\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + 
        ((N - i - 3) * (N - i) div 2 + 2)"
  proof-
    have "N - 2 - i - 1 = N - i - 3"
       using assms
       by simp
    thus ?thesis
       using assms
       unfolding triple_index_def
       by simp
  qed
  moreover
  have "triple_index N (i + 1) (i + 2) (i + 3) = 
    (\<Sum>n\<leftarrow>[1..<i + 1]. (N - n) * (N - n - 1) div 2) + ((N - i - 1) * (N - i - 2) div 2 + 1)"
    unfolding triple_index_def
    by simp
  moreover
  have "(N - i - 3) * (N - i) div 2 + 2 = (N - i - 1) * (N - i - 2) div 2 + 1"
  proof-
     have "(N - i - 3) * (N - i) + 4 = (N - i - 1) * (N - i - 2) + 2" (is "?lhs = ?rhs")
     proof (subst int_int_eq[symmetric])
        have "N - i \<ge> 3"
          using assms
          by auto
        have "(int N - int i - 3) * (int N - int i) + 4 = (int N - int i - 1) * (int N - int i - 2) + 2"
          by (simp add: int_distrib)
        thus "int ((N - i - 3) * (N - i) + 4) = int ((N - i - 1) * (N - i - 2) + 2)"
          using assms `N - i \<ge> 3` zdiff_int[of 3 "N - i", symmetric] zdiff_int[of i N, symmetric] zdiff_int[of "i+1" N, symmetric] zdiff_int[of "i+2" N, symmetric]
          by (auto simp add: int_mult int_distrib)
     qed

     hence "((N - i - 3) * (N - i) + 4) div 2 = ((N - i - 1) * (N - i - 2) + 2) div 2"
       by (rule arg_cong)
     moreover
     have "2 dvd (N - i - 3) * (N - i)"
       using assms
       by auto
     moreover
     have "2 dvd (N - i - 1) * (N - i - 2)"
       using assms
       by auto
     ultimately
     show ?thesis
       using div_add[of 2 "(N - i - 3) * (N - i)" 4]
       using div_add[of 2 "(N - i - 1) * (N - i - 2)" 2]
       by simp
  qed
  ultimately
  show ?thesis
    by simp
qed

lemma triple_index_mono_i:
  assumes "i < j" "j < k" "k < N"
          "i < i'"
          "i' < j'" "j' < k'" "k' < N"
  shows "triple_index N i j k < triple_index N i' j' k'"
using assms  
proof (induct "i' - i" arbitrary: i j j' k k')
  case 0
  thus ?case
    by simp
next
  case (Suc d')
  have "j = N - 2 \<or> j < N - 2" "k = N - 1 \<or> k < N - 1" "N - 2 < N - 1"
    using Suc
    by auto
  hence "triple_index N i j k \<le> triple_index N i (N-2) (N-1)"
    using triple_index_mono_k[of i "N-2" k "N-1" N]
    using triple_index_mono_j[of i j k N "N-2" "N-1"] Suc(2-9)
    by auto
  moreover
  have "triple_index N i (N-2) (N-1) < triple_index N (i+1) (i+2) (i+3)"
    using triple_index_i_inc[of i N] Suc(2-9)
    by auto
  moreover 
  have "j' = i + 2 \<or> j' > i + 2" "k' = i + 3 \<or> k' > i + 3"
    using Suc(2-9)
    by auto
  hence "triple_index N (i+1) (i+2) (i+3) \<le> triple_index N (i+1) j' k'"
    using triple_index_mono_k[of "i+1" "i+2" "i+3" k' N]
    using triple_index_mono_j[of "i+1" "i+2" "i+3" N j' k'] Suc(2-9)
    by auto
  ultimately
  have *: "triple_index N i j k  < triple_index N (i+1) j' k'"
    by simp
  show ?case
  proof (cases "i' = i+1")
    case True
    thus ?thesis
      using *
      by simp
  next
    case False
    have "d' = i' - Suc i"
      using Suc
      by auto
    thus ?thesis
      using *
      using Suc(1)[of "i+1" j' k' j' k'] Suc(2-9) False
      by auto
  qed
qed
          
definition ccwcode_var' where
 [code del]: "ccwcode_var' = ccwcode.ccwcode_var triple_index"

definition ccwcode_var_neg' where
 [code del]: "ccwcode_var_neg' = ccwcode.ccwcode_var_neg triple_index"

definition formula_ax4' where
 [code del]: "formula_ax4' = ccwcode.formula_ax4 triple_index"

definition formula_ax5' where
 [code del]: "formula_ax5' = ccwcode.formula_ax5 triple_index"

definition formula_hulls_nested' where
 [code del]: "formula_hulls_nested' = ccwcode.formula_hulls_nested triple_index"
 
definition formula_hulls_polygons' where
 [code del]: "formula_hulls_polygons' = ccwcode.formula_hulls_polygons triple_index"

definition formula_no_polygon' where 
 [code del]: "formula_no_polygon' = ccwcode.formula_no_polygon triple_index"

definition formula_all_3_1' where 
 [code del]: "formula_all_3_1' = ccwcode.formula_all_3_1 triple_index"

 definition formula_all_4_1' where 
 [code del]: "formula_all_4_1' = ccwcode.formula_all_4_1 triple_index"

definition formula_all_5_canon' where 
 [code del]: "formula_all_5_canon' = ccwcode.formula_all_5_canon triple_index"

definition formula' where 
 [code del]: "formula' = ccwcode.formula triple_index" 

definition simplified_formula' where 
 [code del]: "simplified_formula' = ccwcode.simplified_formula triple_index" 
 
definition satisfies' where
 [code del]: "satisfies' = ccwcode.satisfies triple_index"
 
interpretation ccwcode triple_index where
  "ccwcode.ccwcode_var triple_index = ccwcode_var'" and
  "ccwcode.ccwcode_var_neg triple_index = ccwcode_var_neg'" and
  "ccwcode.formula_ax4 triple_index = formula_ax4'" and
  "ccwcode.formula_ax5 triple_index = formula_ax5'" and
  "ccwcode.formula_hulls_nested triple_index = formula_hulls_nested'" and
  "ccwcode.formula_hulls_polygons triple_index = formula_hulls_polygons'" and
  "ccwcode.formula_no_polygon triple_index = formula_no_polygon'" and
  "ccwcode.formula_all_3_1 triple_index = formula_all_3_1'" and
  "ccwcode.formula_all_4_1 triple_index = formula_all_4_1'" and
  "ccwcode.formula_all_5_canon triple_index = formula_all_5_canon'" and
  "ccwcode.formula triple_index = formula'" and 
  "ccwcode.simplified_formula triple_index = simplified_formula'" and 
  "ccwcode.satisfies triple_index = satisfies'"
proof unfold_locales
fix p q r N p' q' r'
assume *: "p < q" "q < r" "r < N" "p' < q'" "q' < r'" "r' < N" 
"triple_index N p q r = triple_index N p' q' r'"
thus "p = p' \<and> q = q' \<and> r = r'"
proof (cases "p > p'")
  case True
  thus ?thesis
   using *
   using triple_index_mono_i[of p' q' r' N p q r]
   by simp
next
  case False
  show ?thesis
  proof (cases "p < p'")
    case True
    thus ?thesis
      using *
      using triple_index_mono_i[of p q r N p' q' r']
      by simp
  next
    case False
    hence "p = p'"
      using `\<not> (p' <  p)`
      by simp
    show ?thesis
    proof (cases "q' < q")
      case True
      thus ?thesis
        using *
        using triple_index_mono_j[of p' q' r' N q r] `p = p'`
        by simp
    next
      case False
      show ?thesis
      proof (cases "q < q'")
        case True
        thus ?thesis
          using *
          using triple_index_mono_j[of p q r N q' r'] `p = p'`
          by simp
      next
        case False
        hence "q = q'"
          using `\<not> (q' < q)`
          by simp
        show ?thesis
        proof (cases "r' < r")
          case True
          thus ?thesis
            using * `p = p'` `q = q'`
            using triple_index_mono_k[of p' q' r' r N]
            by simp
        next
          case False
          show ?thesis
          proof (cases "r < r'")
             case True
             thus ?thesis
               using * `p = p'` `q = q'`
               using triple_index_mono_k[of p q r r' N]
               by simp
          next
             case False
             hence "r = r'"
               using `\<not> (r' < r)`
               by simp
             thus ?thesis
               using  `p = p'` `q = q'`
               by simp
          qed
        qed
      qed
    qed
  qed
qed
next
  fix p q r N :: nat
  assume "p < q" "q < r" "r < N"
  thus "0 < triple_index N p q r"
    unfolding triple_index_def
    by simp
qed (simp_all add: ccwcode_var'_def ccwcode_var_neg'_def formula_ax4'_def formula_ax5'_def formula_hulls_nested'_def formula_hulls_polygons'_def formula_no_polygon'_def formula_all_3_1'_def formula_all_4_1'_def formula_all_5_canon'_def formula'_def simplified_formula'_def satisfies'_def)

definition numvars :: "nat \<Rightarrow> nat" where 
  "numvars n = n * (n - 1) * (n - 2) div 6" 

definition formula_invariant_part where
  "formula_invariant_part m n = formula_no_polygon' m n @ formula_ax4' n @ formula_ax5' n"
definition formula_variant_part where
  "formula_variant_part H =
     formula_hulls_nested' H @ formula_hulls_polygons' H @ 
     formula_all_3_1' H @ formula_all_4_1' H @ formula_all_5_canon' H"

term simplify_formula     
     
ML{*
fun listsum x = 
  if null x then 0
  else hd x + listsum (tl x);

fun int2str i = 
  if i > 0 then PolyML.makestring i ^ " " 
  else if i < 0 then ("-" ^ PolyML.makestring (~ i) ^ " ") 
  else "0\n";

val iofi = @{code integer_of_int};
val nofi = @{code nat_of_integer};
val iofn = @{code integer_of_nat};

fun formula_invariant_part m n = @{code formula_invariant_part} (nofi m) (nofi n);

fun simplified_formula H fip =  ((@{code formula_variant_part} H) @ fip);

val fip617 = formula_invariant_part 6 17;

fun export_cnf H =
let 
  val ll = map (fn cl => (map iofi cl) @ [0]) (simplified_formula (map nofi H) fip617);
  val num_clauses = List.length ll;
  val num_vars = iofn (@{code numvars} (nofi (listsum H)));
  val ll = "p cnf " ^ (PolyML.makestring num_vars) ^ " " ^ (PolyML.makestring num_clauses) ^ "\n" ::
           List.concat (map (map int2str) ll)
in 
  File.write_list (Path.make ["_formula" ^ (fold (fn s => fn t => t ^ "_" ^ s) (map PolyML.makestring H) "") ^ ".cnf"]) ll
end;

val m = 6;
val n = 17;
val Hs = (map (map iofn)) (@{code R2.hull_structure} (nofi m) (nofi n));

(*
map (export_cnf m) Hs;
*)

*}

method_setup rawsat = {*
  Scan.succeed (SIMPLE_METHOD' o SAT.rawsat_tac)
*} "SAT solver (no preprocessing)"

(* ------------------------------ *)

lemma length_concat_take:
assumes "n < length H" "i < length (H ! n)" 
shows "length (concat (take n H)) + i < length (concat H)"
proof-
  have "length (concat (take n H)) + i < length (concat (take n H) @ (H ! n))"
    using assms
    by auto
  moreover
  have "length (concat (take n H) @ (H ! n)) = length (concat (take (n + 1) H))"
     using take_Suc_conv_app_nth[of n H] `n < length H`
     by simp
  ultimately
  have "length (concat (take n H)) + i < length (concat (take (n + 1) H))"
    by simp
  moreover 
  have "length (concat (take (n + 1) H)) \<le> length (concat H)"
    using append_take_drop_id[of "n+1" H, symmetric]
    using concat_append[of "take (n + 1) H" "drop (n + 1) H"]
    by auto
  ultimately
  show ?thesis
    by simp
qed

lemma concat:
assumes "i < length (H ! n)" "n < length H"
shows "(concat H) ! (length (concat (take n H)) + i) =  H ! n ! i"
proof-
  let ?l = "length (concat (take n H))"
  let ?k = "?l + i"
  have "(concat H) ! ?k = ((concat (take n H)) @ (concat (drop n H))) ! ?k"
    apply (subst append_take_drop_id[of n H, symmetric])
    apply (subst concat_append)
    by simp
  also have "... = (concat (drop n H)) ! (?k - ?l)"
    using nth_append
    by simp
  also have "... = (concat (drop n H)) ! i"
    by simp
  also have "... = concat ([H!n] @ drop (n+1) H) ! i"
    using assms Cons_nth_drop_Suc[of n H]
    by simp
  also have "... = ( H!n @ concat (drop (n+1) H)) ! i"
    by simp
  also have "... = H ! n ! i"
    using nth_append[of "H!n" "concat (drop (n+1) H)"] assms
    by auto
  finally
    show ?thesis
    .
qed


locale convexccwcode = convex + ccwcode
begin
value "hull_structure 4 5"

lemma satisfies_:
  assumes "in_general_position S" "card S > 0"
  assumes "m \<ge> 3" "\<not> contains_convex_polygon m S"
  assumes "nested_hulls H" "set (concat H) = S"  "all_area_3_1 H" "all_area_4_1 H" "all_area_5_canon H"
  shows "\<exists> val. satisfies (map length H) m val"
proof-
   obtain n where "card S = n" by auto
   
  let ?H = "map length H"
  have "?H \<in> set (hull_structure m (card S))"
    using hull_structure[OF `nested_hulls H`_ _ `m \<ge> 3`] `in_general_position S` `\<not> contains_convex_polygon m S` `set (concat H) = S`
    by auto

   let ?g = "\<lambda> i. (concat H) ! i"
   let ?f = "inv_into {0..< card S} ?g"

   have "length H > 0"
     using `set (concat H) = S` `card S > 0`
     by (cases H) auto
   hence "H \<noteq> []"
     by auto

   have "length (concat H) = listsum ?H"
     by (simp add: length_concat)
    
   have "length (concat H) = card S"
     using `nested_hulls H` `set (concat H) = S`
     using nested_hulls_length
     by simp
   moreover
   have "distinct (concat H)"
     using `nested_hulls H`
     using nested_hulls_distinct
     by simp
   ultimately  
   have "bij_betw ?g {0..<card S} S"
     using `set (concat H) = S` inj_on_nth[of "concat H" "{0..<card S}"] set_conv_nth[of "concat H"] `card S = n`
     unfolding bij_betw_def
     by auto
     
   hence "bij_betw ?f S {0..<card S}"
     using bij_betw_inv_into
     by blast
   have gf: "\<forall> x \<in> S. ?g (?f x) = x"
     using `bij_betw ?g {0..<card S} S`
     by (meson bij_betw_inv_into_right)
   have fg: "\<forall> x \<in> {0..<card S}. ?f (?g x) = x"
     using `bij_betw ?g {0..<card S} S`
     by (meson bij_betw_inv_into_left)
                                                         
   let ?val = "\<lambda> n. \<exists> p q r. p < q \<and> q < r \<and> r < card S \<and> n = ccwcode (card S) p q r \<and> ccw (?g p) (?g q) (?g r)"
   obtain val where "val = ?val" by auto
   have ccw: "\<forall> p q r. p < q \<and> q < r \<and> {p, q, r} \<subseteq> {0..<card S} \<longrightarrow>
                       (val (ccwcode (card S) p q r) \<longleftrightarrow> ccw (?g p) (?g q) (?g r))"
   proof ((rule allI)+, rule impI, (erule conjE)+)
     fix p q r
     assume *: "p < q" "q < r" "{p, q, r} \<subseteq> {0..<card S}"

     show "val (ccwcode (card S) p q r) \<longleftrightarrow> ccw (?g p) (?g q) (?g r)"
     proof
       assume "ccw (concat H ! p) (concat H ! q) (concat H ! r)"
       thus "val (ccwcode (card S) p q r)"
         using `val = ?val` *
         by auto
     next
       assume "val (ccwcode (card S) p q r)"
       then obtain p' q' r' where "p' < q'" "q' < r'" "r' < card S" "ccw (?g p') (?g q') (?g r')" "ccwcode (card S) p q r = ccwcode (card S) p' q' r'"
         using `val = ?val` *
         by auto
       thus "ccw (?g p) (?g q) (?g r)"
         using ccwcode_inj[of p' q' r' "card S" p q r] *
         by auto
     qed
   qed
   
   have ccwcode_val: "\<forall> p q r. distinct [p, q, r] \<and> {p, q, r} \<subseteq> {0..<card S} \<longrightarrow> 
     (ccwcode_val (card S) p q r val \<longleftrightarrow> ccw (?g p) (?g q) (?g r))"
   proof ((rule allI)+, rule impI, erule conjE)
     fix p q r
     assume as: "distinct [p, q, r]" "{p, q, r} \<subseteq> {0..<card S}"
     hence "{?g p, ?g q, ?g r} \<subseteq> S"
       using in_set_conv_nth[of "?g p" "concat H"] in_set_conv_nth[of "?g q" "concat H"] in_set_conv_nth[of "?g r" "concat H"]
       using `set (concat H) = S` `length (concat H) = card S`
       by auto
     hence *: "in_general_position {?g p, ?g q, ?g r}"
       using in_general_position_mono[of "{?g p, ?g q, ?g r}", OF _ `in_general_position S`]
       by simp
     
     have **: "distinct [?g p, ?g q, ?g r]"
       using `distinct [p, q, r]`  `length (concat H) = card S` `{p, q, r} \<subseteq> {0..<card S}`
       using nth_eq_iff_index_eq[OF `distinct (concat H)`]
       by auto
     
     show "ccwcode_val (card S) p q r val \<longleftrightarrow> ccw (?g p) (?g q) (?g r)"
       using as ccw 
       unfolding ccwcode_val_def sort_triple'_def sort_triple_def
       apply auto
       using ax_pos ax_neg * **
       by blast+
   qed
     
   hence ccwcode_val_neg: "\<forall> p q r. distinct [p, q, r] \<and> {p, q, r} \<subseteq> {0..<card S} \<longrightarrow> 
     (ccwcode_val_neg (card S) p q r val \<longleftrightarrow> \<not> ccw (?g p) (?g q) (?g r))"
     using ccwcode_val ccwcode_val_negpos 
     by simp

   let ?n = "listsum ?H"
   have "card S = ?n"
     using `length (concat H) = ?n` `length (concat H) = card S`
     by simp
     
   have "satisfies ?H m val"
   unfolding Let_def satisfies_def
   proof safe
     show "satisfies_formula_ax4 ?n val"
     unfolding satisfies_formula_ax4_full
     proof safe
       fix p q r t
       assume *: "{p, q, r, t} \<subseteq> set [0..<?n]" "distinct [p, q, r, t]" and
       "\<not> ccwcode_val_neg ?n t q r val" "\<not> ccwcode_val_neg ?n p t r val" "\<not> ccwcode_val ?n p q r val"
     
       hence "ccw (?g t) (?g q) (?g r)"  "ccw (?g p) (?g t) (?g r)"  "\<not> ccw (?g p) (?g q) (?g r)"
         using ccwcode_val_neg ccwcode_val `card S = ?n` 
         by auto
       hence "\<not> ccw (?g p) (?g q) (?g t)"
         using ax4 
         by blast
       thus "ccwcode_val_neg (listsum ?H) p q t val"
         using ccwcode_val_neg * `card S = ?n`
         by auto
     qed
   next
     show "satisfies_formula_ax5 ?n val"
     unfolding satisfies_formula_ax5_full
     proof safe
       fix p q r s t
       assume *: "{p, q, r, s, t} \<subseteq> set [0..<?n]" "distinct [p, q, r, s, t]" and
       "\<not> ccwcode_val_neg ?n t s p val" "\<not> ccwcode_val_neg ?n t s q val" "\<not> ccwcode_val_neg ?n t s r val" 
       "\<not> ccwcode_val_neg ?n t p q val" "\<not> ccwcode_val ?n t p r val"
     
       hence "ccw (?g t) (?g s) (?g p)"  "ccw (?g t) (?g s) (?g q)"  "ccw (?g t) (?g s) (?g r)"
             "ccw (?g t) (?g p) (?g q)" "\<not> ccw (?g t) (?g p) (?g r)"
         using ccwcode_val_neg ccwcode_val `card S = ?n`
         by auto
       hence "\<not> ccw (?g t) (?g q) (?g r)"
         using ax5
         by blast
       thus "ccwcode_val_neg ?n t q r val"
         using ccwcode_val_neg * `card S = ?n`
         by auto
     qed   
   next
     show "satisfies_formula_no_polygon m ?n val"
     unfolding satisfies_formula_no_polygon_def
     proof safe
       fix p
       assume "m = length p" "set p \<subseteq> set [0..<?n]" "distinct p"
       hence "length (map ?g p) = m" "set (map ?g p) \<subseteq> S" "distinct (map ?g p)"
         using `set (concat H) = S` `bij_betw ?g {0..<card S} S` subset_inj_on[of ?g "{0..<card S}"] `card S = ?n`
         by (auto simp add: distinct_map bij_betw_def)
       moreover
       have "in_general_position (set (map ?g p))"
         by (rule in_general_position_mono[OF `set (map ?g p) \<subseteq> S` `in_general_position S`])
       ultimately
       obtain i j k where "i < j" "j < k" "k < length p" "\<not> ccw (?g (p ! i)) (?g (p ! j)) (?g (p ! k))"
         using `\<not> (contains_convex_polygon m S)` `m \<ge> 3`
         unfolding contains_convex_polygon_def convex_polygon_def
         by simp (erule_tac x="map ?g p" in allE, auto)
       thus "\<exists>i j k. i < j \<and> j < k \<and> k < length p \<and> ccwcode_val_neg ?n (p ! i) (p ! j) (p ! k) val"
         using ccwcode_val_neg `distinct p` `set p \<subseteq> set [0..<?n]` nth_eq_iff_index_eq[of p] `card S = ?n`
         apply -
         apply (rule_tac x="i" in exI, rule_tac x="j" in exI, simp, rule_tac x="k" in exI, simp)
         apply (erule_tac x="p!i" in allE, erule_tac x="p!j" in allE, erule_tac x="p!k" in allE)
         by force
     qed
   next
     show "satisfies_formula_hulls_polygons ?H val"
     unfolding satisfies_formula_hulls_polygons_def Let_def
     proof safe
       fix n i j k
       assume "n < length ?H" "i < j" "j < k" "k < ?H ! n"
       let ?start = "listsum (take n (map length H))"
       
       have "listsum (take n ?H) = length (concat (take n H))"
         by (simp add: length_concat take_map)
         
       have ub: "?start + i < length (concat H)" "?start + j < length (concat H)" "?start + k < length (concat H)"
         using `i < j` `j < k` `k < ?H ! n` `n < length ?H` 
         using length_concat_take[of n H] `listsum (take n ?H) = length (concat (take n H))`
         by auto
       hence g: "?g (?start + i) = (H ! n) ! i" "?g (?start + j) = (H ! n) ! j" "?g (?start + k) = (H ! n) ! k"
         using `i < j` `j < k` `k < ?H ! n` `n < length ?H` `length (concat H) = card S` concat[of _ H n] `listsum (take n ?H) = length (concat (take n H))`
         by auto

       have "ccwcode_val (card S) (?start + i) (?start + j) (?start + k) val"
       proof (subst ccwcode_val[rule_format])
         show "ccw (concat H ! (?start + i)) (concat H ! (?start + j)) (concat H ! (?start + k))"
         proof-
           have "ccw (H ! n ! i) (H ! n ! j) (H ! n ! k)"
             using `nested_hulls H` `set (concat H) = S` `in_general_position S`
             using nested_hulls_are_convex_polygons'[of "H"] `n < length ?H` `i < j` `j < k` `k < ?H ! n`
             by simp
           thus ?thesis
             using g
             by simp
         qed
       next
         show "distinct [?start+i, ?start+j, ?start+k] \<and> {?start+i, ?start+j, ?start+k} \<subseteq> {0..<card S}"
           using `n < length ?H` `i < j` `j < k` `k < ?H ! n` ub `length (concat H) = card S`
           by simp
       qed
       thus "ccwcode_val (listsum ?H) (?start + i) (?start + j) (?start + k) val"
         using `length (concat H) = listsum ?H` `length (concat H) = card S`
         by simp
     qed
   next
     show "satisfies_formula_hulls_nested ?H val"
     unfolding satisfies_formula_hulls_nested_def Let_def
     proof safe
       fix n i x
       assume "n < length (map length H) - 1"
        
       let ?start = "listsum (take n ?H)"
       let ?j = "(i + 1) mod (?H ! n)"

       assume "i < ?H ! n" "?start + ?H ! n \<le> x" "x < listsum ?H"

       hence *: "listsum (take n (map length H)) = length (concat (take n H))"
         using `length (concat H) = listsum ?H`
         by (simp add: length_concat take_map)
       hence "x \<ge> length (concat (take (Suc n) H))"
         using take_Suc_conv_app_nth[of n H] `n < length ?H - 1` `?start + ?H ! n \<le> x`
         by simp
        
       have "length (H ! n) > 0"
         using nested_hulls_no_Nil[OF `nested_hulls H`] in_set_conv_nth[of "[]" H] `n < length ?H - 1`
         by auto
       hence "?j < ?H ! n" "?j < length (H ! n)"
         using mod_less_divisor[of "length (H ! n)" "i+1"] `n < length ?H - 1`
         by auto
        
       have ub: "?start + i < length (concat H)" "?start + ?j < length (concat H)" "x < length (concat H)"
         using `x < listsum ?H` `length (concat H) = listsum ?H` 
         using length_concat_take[of n H] `n < length ?H - 1` `i < ?H ! n` `?j < ?H ! n` *
         by auto
          
       have *: "?g (?start + i) = (H ! n) ! i" "?g (?start + ?j) = (H ! n) ! ?j" "?g x = concat H ! x" 
         using `i < ?H ! n` `?j < length (H ! n)` `n < length ?H - 1` concat[of i H n] concat[of ?j H n] *
         by auto
       
       have ccw: "ccw (H ! n ! i) (H ! n ! ?j) (concat H ! x)"
       proof-
         have "concat H ! x \<in> set (concat (drop (n + 1) H))"
         proof-
           have "concat H ! x = concat (drop (n + 1) H) ! (x - length (concat (take (n + 1) H)))"
             using `x \<ge> length (concat (take (Suc n) H))`
             apply (subst append_take_drop_id[of "n+1" H, symmetric])
             apply (subst concat_append)
             apply (subst nth_append)
             by auto
           moreover
           have "x < length (concat (drop (n + 1) H)) + length (concat (take (n + 1) H))"
             using `x < listsum ?H`
             apply (subst (asm) `length (concat H) = listsum ?H`[symmetric])
             apply (subst (asm) append_take_drop_id[of "n+1" H, symmetric])
             apply (subst (asm) concat_append)
             by simp
           hence "x - length (concat (take (n + 1) H)) < length (concat (drop (n + 1) H))"
             using `x \<ge> length (concat (take (Suc n) H))`
             by simp
           ultimately
           show ?thesis
             using in_set_conv_nth[of "concat H ! x" "concat (drop (n + 1) H)"]
             by auto
         qed
         thus ?thesis
         using nested_hulls_are_nested[of H n] `nested_hulls H` `n < length ?H - 1` `i < ?H ! n`
         by auto
       qed
       
       hence "H ! n ! i \<noteq> H ! n ! ?j"
         using ax0
         by auto
       hence "i \<noteq> ?j"
         using * `distinct (concat H)`
         by auto
       
       have "ccwcode_val (card S) (?start + i) (?start + ?j) x val"
       proof (subst ccwcode_val[rule_format])
         show "ccw (?g (?start + i)) (?g (?start + ?j)) (?g x)"
           using ccw *
           by simp
       next
         have "?start + i < x \<and> ?start + ?j < x"
           using `?j < ?H ! n` `i < ?H ! n` `x \<ge> ?start + ?H ! n`
           by linarith
         thus "distinct [?start + i, ?start + ?j, x] \<and> {?start + i, ?start + ?j, x} \<subseteq> {0..<card S}"
           using `length (concat H) = card S` `length (H ! n) > 0` ub `i < ?H ! n` `i \<noteq> ?j`
           by auto
       qed
       thus "ccwcode_val (listsum ?H) (?start + i) (?start + ?j) x val"
         using `length (concat H) = listsum ?H` `length (concat H) = card S`
         by simp
     qed
   next
     show "satisfies_formula_all_4_1 (map length H) val"
       unfolding satisfies_formula_all_4_1_def Let_def
     proof (rule allI, rule impI, rule impI) 
       fix i
       assume "i < length (map length H) - 1" "map length H ! i = 4"
       hence "i < length H - 1" "length (H ! i) = 4" "H \<noteq> []"
         using nth_map[of 0 "H" length]
         by auto
       
       hence "i + 1 < length H"
         by auto
         
       hence "H ! (i + 1) \<noteq> []"
         using nested_hulls_no_Nil[OF `nested_hulls H`] in_set_conv_nth[of "[]" H]
         by auto
         
       let ?start = "listsum (take i ?H)"
       
       have "listsum (take i ?H) = length (concat (take i H))"
         by (simp add: length_concat take_map)
         
       have *: "H ! i ! 0 = ?g ?start" "H ! i ! 1 = ?g (?start + 1)" "H ! i ! 2 = ?g (?start + 2)" "H ! i ! 3 = ?g (?start + 3)" "H ! (i + 1) ! 0 = ?g (?start + 4)"
         using `listsum (take i ?H) = length (concat (take i H))`
         using concat[of 0 H i] concat[of 1 H i] concat[of 2 H i] concat[of 3 H i] concat[of 0 H "i+1"] `length (H ! i) = 4` `i < length H - 1` `length H > 0`
         using `H ! (i + 1) \<noteq> []` `i + 1 < length H`
         apply auto
         apply (subst (asm) take_Suc_conv_app_nth, simp)
         apply (subst (asm) concat_append, simp)
         done
                
       have ub: "?start < length (concat H)" "?start + 1 < length (concat H)" "?start + 2 < length (concat H)" "?start + 3 < length (concat H)" "?start + 4 < length (concat H)"
          using `length (concat H) = listsum ?H` `listsum (take i ?H) = length (concat (take i H))`  
          using length_concat_take[of i H 0] length_concat_take[of i H 1] length_concat_take[of i H 2] length_concat_take[of i H 3]
          using length_concat_take[of "i+1" H 0] `H ! (i + 1) \<noteq> []`
          using `i + 1 < length H` `length (H ! i) = 4`
          apply auto
          apply (subst (asm) take_Suc_conv_app_nth, simp)
          apply (subst (asm) concat_append, simp)
          done

       have ccw: "\<not> ccw (H ! i ! 0) (H ! i ! 2) (H ! (i + 1) ! 0)" "ccw (H ! i ! 1) (H ! i ! 3) (H ! (i + 1) ! 0)"  
         using `all_area_4_1 H`
         using  `length (H ! i) = 4` `i < length H - 1`
         unfolding all_area_4_1_def area_4_1_def
         by auto
          
       have "ccwcode_val_neg (card S) ?start (?start + 2) (?start + 4) val"
       proof (subst ccwcode_val_neg)
         show "distinct [?start, (?start + 2), (?start + 4)] \<and> {?start, (?start + 2), (?start + 4)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "\<not> ccw (?g ?start) (?g (?start + 2)) (?g (?start + 4))"
           using * ccw
           by auto
       qed
       moreover
       have "ccwcode_val (card S) (?start + 1) (?start + 3)  (?start + 4) val"
       proof (subst ccwcode_val)
         show "distinct [(?start + 1), (?start + 3), (?start + 4)] \<and> {?start + 1, ?start + 3, ?start + 4} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "ccw (?g (?start + 1)) (?g (?start + 3)) (?g (?start + 4))"
           using * ccw
           by auto
       qed
       ultimately       
       show "ccwcode_val_neg (listsum ?H) ?start (?start + 2) (?start + 4) val \<and>
             ccwcode_val (listsum ?H) (?start + 1) (?start + 3)  (?start + 4) val"
         using `length (concat H) = listsum ?H` `length (concat H) = card S`
         by simp
     qed
   next
     show "satisfies_formula_all_3_1 (map length H) val"
       unfolding satisfies_formula_all_3_1_def Let_def
     proof (rule allI, rule impI, rule impI, erule conjE) 
       fix i
       assume "i < length ?H - 1" "?H ! i = 3" "?H ! (i + 1) \<ge> 2"
       hence "i < length H - 1" "length (H ! i) = 3" "H \<noteq> []"
         using nth_map[of 0 "H" length]
         by auto
       
       hence "i + 1 < length H"
         by auto
         
       hence "length (H ! (i + 1)) \<ge> 2"
         using `?H ! (i + 1) \<ge> 2`
         by auto
       hence "H ! (i + 1) \<noteq> []"
         by auto
         
       let ?start = "listsum (take i ?H)"
       
       have "listsum (take i ?H) = length (concat (take i H))"
         by (simp add: length_concat take_map)
         
       have *: "H ! i ! 0 = ?g ?start" "H ! i ! 1 = ?g (?start + 1)" "H ! i ! 2 = ?g (?start + 2)" "H ! (i + 1) ! 0 = ?g (?start + 3)" "H ! (i + 1) ! 1 = ?g (?start + 4)"
         using `listsum (take i ?H) = length (concat (take i H))`
         using concat[of 0 H i] concat[of 1 H i] concat[of 2 H i] concat[of 0 H "i+1"]  concat[of 1 H "i+1"] `length (H ! i) = 3` 
         using `i < length H - 1` `length H > 0` `length (H ! (i + 1)) \<ge> 2` `i + 1 < length H` `H ! (i + 1) \<noteq> []`
         apply auto
         apply (subst (asm) take_Suc_conv_app_nth, simp)
         apply (subst (asm) concat_append, simp)
         apply (subst (asm) take_Suc_conv_app_nth)
         back
         apply simp
         apply (subst (asm) concat_append, auto simp add: add.commute)
         done
                
       have ub: "?start < length (concat H)" "?start + 1 < length (concat H)" "?start + 2 < length (concat H)" "?start + 3 < length (concat H)" "?start + 4 < length (concat H)"
          using `length (concat H) = listsum ?H` `listsum (take i ?H) = length (concat (take i H))`  
          using length_concat_take[of i H 0] length_concat_take[of i H 1] length_concat_take[of i H 2]
          using length_concat_take[of "i+1" H 0] length_concat_take[of "i+1" H 1] `H ! (i + 1) \<noteq> []`
          using `i + 1 < length H` `length (H ! i) = 3` `length (H ! (i + 1)) \<ge> 2`
          apply auto
          apply (subst (asm) take_Suc_conv_app_nth, simp)
          apply (subst (asm) concat_append, simp)
          apply (subst (asm) take_Suc_conv_app_nth, auto)
          done

       have ccw: " \<not> ccw (H ! (i + 1) ! 0) (H ! (i + 1) ! 1) (H ! i ! 0)" "ccw (H ! (i + 1) ! 0) (H ! (i + 1) ! 1) (H ! i ! 1)"  
         using `all_area_3_1 H`
         using  `length (H ! i) = 3` `length (H ! (i+1)) \<ge> 2` `i < length H - 1`
         unfolding all_area_3_1_def area_3_1_def
         by auto
          
       have "ccwcode_val_neg (card S) (?start + 3) (?start + 4) (?start + 0) val"
       proof (subst ccwcode_val_neg)
         show "distinct [(?start + 3), (?start + 4), (?start + 0)] \<and> {(?start + 3), (?start + 4), (?start + 0)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "\<not> ccw (?g (?start + 3)) (?g (?start + 4)) (?g (?start + 0))"
           using * ccw
           by (auto simp add: add.commute)
       qed
       moreover
       have "ccwcode_val (card S) (?start + 3) (?start + 4) (?start + 1) val"
       proof (subst ccwcode_val)
         show "distinct [(?start + 3), (?start + 4), (?start + 1)] \<and> {?start + 3, ?start + 4, ?start + 1} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "ccw (?g (?start + 3)) (?g (?start + 4)) (?g (?start + 1))"
           using * ccw
           by auto
       qed
       ultimately
       show "ccwcode_val_neg (listsum ?H) (?start + 3) (?start + 4) (?start + 0) val \<and>
             ccwcode_val (listsum ?H) (?start + 3) (?start + 4) (?start + 1) val"
         using `length (concat H) = listsum ?H` `length (concat H) = card S`
         by simp
     qed
   next
     show "satisfies_formula_all_5_canon (map length H) val"
       unfolding satisfies_formula_all_5_canon_def Let_def
     proof (rule allI, rule impI, rule impI)
       fix i
       assume "i < length (map length H) - 1" "map length H ! i = 5"
       hence "i < length H - 1" "length (H ! i) = 5" "H \<noteq> []"
         using nth_map[of 0 "H" length]
         by auto
       
       hence "i + 1 < length H"
         by auto
         
       hence "H ! (i + 1) \<noteq> []"
         using nested_hulls_no_Nil[OF `nested_hulls H`] in_set_conv_nth[of "[]" H]
         by auto
         
       let ?start = "listsum (take i ?H)"
       
       have "listsum (take i ?H) = length (concat (take i H))"
         by (simp add: length_concat take_map)
         
       have *: "H ! i ! 0 = ?g ?start" "H ! i ! 1 = ?g (?start + 1)" "H ! i ! 2 = ?g (?start + 2)" "H ! i ! 3 = ?g (?start + 3)" "H ! i ! 4 = ?g (?start + 4)" "H ! (i + 1) ! 0 = ?g (?start + 5)"
         using `listsum (take i ?H) = length (concat (take i H))`
         using concat[of 0 H i] concat[of 1 H i] concat[of 2 H i] concat[of 3 H i] concat[of 4 H i] concat[of 0 H "i+1"] `length (H ! i) = 5` `i < length H - 1` `length H > 0`
         using `H ! (i + 1) \<noteq> []` `i + 1 < length H`
         apply auto
         apply (subst (asm) take_Suc_conv_app_nth, simp)
         apply (subst (asm) concat_append, simp)
         done

       have ub: "?start < length (concat H)" "?start + 1 < length (concat H)" "?start + 2 < length (concat H)" "?start + 3 < length (concat H)" "?start + 4 < length (concat H)" "?start + 5 < length (concat H)"
          using `length (concat H) = listsum ?H` `listsum (take i ?H) = length (concat (take i H))`  
          using length_concat_take[of i H 0] length_concat_take[of i H 1] length_concat_take[of i H 2] length_concat_take[of i H 3] length_concat_take[of i H 4]
          using length_concat_take[of "i+1" H 0] `H ! (i + 1) \<noteq> []`
          using `i + 1 < length H` `length (H ! i) = 5`
          apply auto
          apply (subst (asm) take_Suc_conv_app_nth, simp)
          apply (subst (asm) concat_append, simp)
          done

       have ccw: "\<not> ccw (H ! i ! 0) (H ! i ! 3) (H ! (i + 1) ! 0)"
                 "ccw (H ! i ! 1) (H ! i ! 3) (H ! (i + 1) ! 0)"
                 "ccw (H ! i ! 2) (H ! i ! 4) (H ! (i + 1) ! 0)"
                 "ccw (H ! i ! 0) (H ! i ! 2) (H ! (i + 1) ! 0) \<or> ccw (H ! i ! 1) (H ! i ! 4) (H ! (i + 1) ! 0)"  
         using `all_area_5_canon H`
         using  `length (H ! i) = 5` `i < length H - 1`
         unfolding all_area_5_canon_def area_5_canon_def
         by auto         
       
       have "ccwcode_val_neg (card S) ?start (?start + 3) (?start + 5) val"
       proof (subst ccwcode_val_neg)
         show "distinct [?start, (?start + 3), (?start + 5)] \<and> {?start, (?start + 3), (?start + 5)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "\<not> ccw (?g ?start) (?g (?start + 3)) (?g (?start + 5))"
           using * ccw
           by auto
       qed
       moreover
       have "ccwcode_val (card S) (?start + 1) (?start + 3) (?start + 5) val"
       proof (subst ccwcode_val)
         show "distinct [(?start + 1), (?start + 3), (?start + 5)] \<and> {(?start + 1), (?start + 3), (?start + 5)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "ccw (?g (?start + 1)) (?g (?start + 3)) (?g (?start + 5))"
           using * ccw
           by auto
       qed
       moreover
       have "ccwcode_val (card S) (?start + 2) (?start + 4) (?start + 5) val"
       proof (subst ccwcode_val)
         show "distinct [(?start + 2), (?start + 4), (?start + 5)] \<and> {(?start + 2), (?start + 4), (?start + 5)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
       next
         show "ccw (?g (?start + 2)) (?g (?start + 4)) (?g (?start + 5))"
           using * ccw
           by auto
       qed
       moreover 
       have "ccwcode_val (card S) (?start + 0) (?start + 2)  (?start + 5) val \<or>
             ccwcode_val (card S) (?start + 1) (?start + 4)  (?start + 5) val"
       proof-
         have "distinct [(?start + 0), (?start + 2), (?start + 5)] \<and> {(?start + 0), (?start + 2), (?start + 5)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
         moreover
         have "distinct [(?start + 1), (?start + 4), (?start + 5)] \<and> {(?start + 1), (?start + 4), (?start + 5)} \<subseteq> {0..<card S}"
           using ub `length (concat H) = card S`
           by auto
         moreover
         have "ccw (?g (?start + 0)) (?g (?start + 2)) (?g (?start + 5)) \<or> ccw (?g (?start + 1)) (?g (?start + 4)) (?g (?start + 5))"
           using * ccw
           by auto         
         ultimately
         show ?thesis
            using ccwcode_val
            by auto
       qed
       ultimately
       show "ccwcode_val_neg (listsum ?H) (?start + 0) (?start + 3) (?start + 5) val \<and>
             ccwcode_val (listsum ?H) (?start + 1) (?start + 3)  (?start + 5) val \<and> 
             ccwcode_val (listsum ?H) (?start + 2) (?start + 4)  (?start + 5) val \<and>
              (ccwcode_val (listsum ?H) (?start + 0) (?start + 2)  (?start + 5) val \<or>
               ccwcode_val (listsum ?H) (?start + 1) (?start + 4)  (?start + 5) val) "
         using `length (concat H) = listsum ?H` `length (concat H) = card S`
         by simp
     qed
   qed
   thus ?thesis
     by auto
qed

 
lemma reduction_to_unsat:
  assumes "in_general_position S" "card S = n" "n > 0" "m \<ge> 3"
          "\<forall>H \<in> set (hull_structure m n). \<not> (\<exists>val. satisfies H m val)"
  shows "contains_convex_polygon m S" 
proof (rule ccontr)
  assume "\<not> ?thesis"

  obtain H0 where "nested_hulls H0" "set (concat H0) = S"
    using ex_nested_hulls[of S] card_ge_0_finite[of S] `card S = n` `n > 0` `in_general_position S` 
    by auto
  
  have "\<exists> H. nested_hulls H \<and> set (concat H) = S \<and> length H = length H0 \<and> all_area_3_1 H \<and> all_area_4_1 H \<and> all_area_5_canon H"
    using nested_hulls_all_canon[OF `nested_hulls H0`] `set (concat H0) = S` `in_general_position S`
    by auto
  then obtain H where *: "nested_hulls H" "set (concat H) = S" "all_area_3_1 H" "all_area_4_1 H" "all_area_5_canon H"
    by blast
    
  let ?H = "map length H"
  
  have "?H \<in> set (hull_structure m n)"
    using hull_structure assms `\<not> contains_convex_polygon m S` `nested_hulls H` `set (concat H) = S`
    by auto
  moreover
  have "\<exists> val. satisfies ?H m val"
    using satisfies_[OF `in_general_position S` _ `m \<ge> 3` `\<not> (contains_convex_polygon m S)`] `n > 0` `card S = n` *
    by auto
  ultimately
  show False
    using `\<forall>H \<in> set (hull_structure m n). \<not> (\<exists>val. satisfies H m val)`
    by auto
qed

lemma reduction_to_unsat_exe:
assumes "in_general_position S" "card S = n" "n > 0" "m \<ge> 3"
        "\<forall> H \<in> set (hull_structure m n). \<not> (\<exists> val. satisfies_formula val (simplified_formula m H))"
  shows "contains_convex_polygon m S" 
proof (rule reduction_to_unsat[OF assms(1-2)])
  show "\<forall>H\<in>set (hull_structure m n). \<not> (\<exists>val. satisfies H m val)"
  proof safe
    fix H val
    assume "H \<in> set (hull_structure m n)"
    hence *: "H \<noteq> [] \<and> (\<forall>n<length H - 1. 1 < H ! n) \<and> 0 < H ! (length H - 1)"
      using `m \<ge> 3` `n > 0`
    proof (induct m n arbitrary: H rule: hull_structure.induct)
       case (1 m n)
       show ?case
       proof (cases "n < 3")
         case True
         thus ?thesis
           using hull_structure.simps[of m n] `n > 0` 1(2)
           by simp
       next
         case False
         then obtain x H' where "3 \<le> x \<and> x < m \<and> x \<le> n" "H' \<in> set (hull_structure m (n - x))" "H = x # H'"
           using hull_structure.simps[of m n] `n > 0` 1(2)
           by force
         thus ?thesis
           using 1(1)[of x H'] 1(2-4)
           by simp (cases "x =  n", auto simp add: hull_structure.simps[of m 0]  nth_Cons')
       qed
    qed
    assume "satisfies H m val"
    thus False
      using satisfies[of H, of m val] `H \<in> set (hull_structure m n)` *
      using assms
      by simp
  qed
next
  show "n > 0"
    using assms
    by simp
next
  show "m \<ge> 3"
    using assms
    by simp
qed

end

context convex
begin

interpretation TripleIndex: convexccwcode ccw col triple_index
by unfold_locales


lemma unsat_3_3: "\<forall> H \<in> set (hull_structure 3 3).  \<not> (\<exists> val. satisfies_formula val (simplified_formula' 3 H))"
proof safe
  fix H
  assume "H \<in> set (hull_structure 3 3)"
  moreover
  have "hull_structure 3 3 = []"
    by eval
  ultimately
  show False
    by simp
qed

theorem ErdosSekeres_3_3:
  assumes "in_general_position S" "card S = 3"
  shows "contains_convex_polygon 3 S"
  using TripleIndex.reduction_to_unsat_exe[OF assms _ _ unsat_3_3]
  by simp

lemma unsat_4_5: "\<forall> H \<in> set (hull_structure 4 5).  \<not> (\<exists> val. satisfies_formula val (simplified_formula' 4 H))"
proof safe
  fix H val
  assume "H \<in> set (hull_structure 4 5)" "satisfies_formula val (simplified_formula' 4 H)"
  moreover
  have "hull_structure 4 5 = [[3, 2]]"
    by eval
  moreover 
  have "\<not> satisfies_formula val (simplified_formula' 4 [3, 2])"
    by normalization sat
  ultimately
  show False
    by auto
qed

theorem ErdosSekeres_4_5:
  assumes "in_general_position S" "card S = 5"
  shows "contains_convex_polygon 4 S"
  using TripleIndex.reduction_to_unsat_exe[OF assms _ _ unsat_4_5]
  by simp


lemma unsat_5_9: "\<forall> H \<in> set (hull_structure 5 9).  \<not> (\<exists> val. satisfies_formula val (simplified_formula' 5 H))"
proof safe
  fix H val
  assume "H \<in> set (hull_structure 5 9)" "satisfies_formula val (simplified_formula' 5 H)"
  moreover
  have "hull_structure 5 9 = [[3, 3, 3], [3, 4, 2], [4, 3, 2], [4, 4, 1]]"
    by eval
  moreover

  have "\<not> satisfies_formula val (simplified_formula' 5 [3, 3, 3])"                  
    by normalization (rule notI, (erule conjE)+, rawsat)

  moreover
  have "\<not> satisfies_formula val (simplified_formula' 5 [3, 4, 2])"
    by normalization (rule notI, (erule conjE)+, rawsat)

  moreover
  have "\<not> satisfies_formula val (simplified_formula' 5 [4, 3, 2])"
    by normalization (rule notI, (erule conjE)+, rawsat)

  moreover
  have "\<not> satisfies_formula val (simplified_formula' 5 [4, 4, 1])"
    by normalization (rule notI, (erule conjE)+, rawsat)
    

  ultimately
  show False
   by auto
qed

theorem ErdosSekeres_5_9:
  assumes "in_general_position S" "card S = 9"
  shows "contains_convex_polygon 5 S"
  using TripleIndex.reduction_to_unsat_exe[OF assms _ _ unsat_5_9]
  by simp

end
end