#include <vector>
#include <map>
#include <set>
#include <iterator>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <ctime>
#include <iomanip>

int ccw_var(int N, int i, int j, int k) {
  int result = 0;
  for (int n = 1; n <= i; n++)
    result += (N - n) * (N - n - 1) / 2;
  result += (j - i - 1) * (2 * N - j - i - 2) / 2;
  result += k - j;
  return result;
}

int ccw_lit(int N, int i, int j, int k) {
  bool pos = true; 
  if (i > j) {
    std::swap(i, j); pos = !pos;
  }
  if (j > k) {
    std::swap(j, k); pos = !pos;
  }
  if (i > j) {
    std::swap(i, j); pos = !pos;
  }
  int v = ccw_var(N, i, j, k);
  return pos ? v : -v;
}

std::set<int> units;

std::ostream* output = &std::cout;

void emit(const std::vector<int>& cl) {
  if (!std::any_of(cl.begin(), cl.end(), [](int l){ return units.find(l) != units.end(); })) {
    copy_if(cl.begin(), cl.end(), std::ostream_iterator<int>(*output, " "), [](int l) { return units.find(-l) == units.end(); });
    (*output) << "0" << std::endl;
  }
  if (cl.size() == 1)
    units.insert(cl[0]);
}

void formula_hulls_nested(const std::vector<int>& H) {
  int N = std::accumulate(H.begin(), H.end(), 0);
  for (int n = 0; n < H.size() - 1; n++) {
    int start = accumulate(H.begin(), H.begin() + n, 0);
    for (int i = 0; i < H[n]; i++)
      for (int x = start + H[n]; x < N; x++)
	emit({ccw_lit(N, start+i, start + (i+1) % (H[n]), x)});
  }
}

void formula_hulls_polygons(const std::vector<int>& H) {
  int N = std::accumulate(H.begin(), H.end(), 0);
  for (int n = 0; n < H.size(); n++) {
    int start = std::accumulate(H.begin(), H.begin() + n, 0);
    for (int k = 2; k < H[n]; k++)
      for (int j = 1; j < k; j++)
	for (int i = 0; i < j; i++)
	  emit({ccw_lit(N, start+i, start+j, start+k)});
  }
}

void formula_ax4(int n) {
  for (int p = 0; p < n; p++)
    for (int q = p+1; q < n; q++)
      for (int r = q+1; r < n; r++)
	for (int t = r+1; t < n; t++) {
	  emit({-ccw_lit(n, t, q, r), -ccw_lit(n, p, t, r), -ccw_lit(n, p, q, t), ccw_lit(n, p, q, r)});
	  emit({-ccw_lit(n, q, t, r), -ccw_lit(n, t, p, r), -ccw_lit(n, q, p, t), ccw_lit(n, q, p, r)});
	}
}

void formula_ax5(int n) {
  for (int p = 0; p < n; p++) {
    for (int q = p+1; q < n; q++) {
      for (int r = q+1; r < n; r++) {
	for (int s = 0; s < n; s++) {
	  if (s == p || s == q || s == r)
	    continue;
	  for (int t = 0; t < n; t++) {
	    if (t == p || t == q || t == r || t == s)
	      continue;
	    emit({-ccw_lit(n, t, s, p), -ccw_lit(n, t, s, q), -ccw_lit(n, t, s, r), -ccw_lit(n, t, p, q), -ccw_lit(n, t, q, r), ccw_lit(n, t, p, r)});
	    emit({-ccw_lit(n, t, s, p), -ccw_lit(n, t, s, q), -ccw_lit(n, t, s, r), -ccw_lit(n, t, q, p), -ccw_lit(n, t, r, q), ccw_lit(n, t, r, p)});
	  }
	}
      }
    }
  }
}

void formula_no_polygon6(int n) {
  for (int p = 0; p < n; p++) {
    for (int q = p; q < n; q++) {
      if (q == p) continue;
      for (int r = p; r < n; r++) {
	if (r == p || r == q) continue;
	for (int s = p; s < n; s++) {
	  if (s == p || s == q || s == r) continue;
	  for (int t = p; t < n; t++) {
	    if (t == p || t == q || t == r || t == s) continue;
	    for (int u = p; u < n; u++) {
	      if (u == p || u == q || u == r || u == s || u == t) continue;
	      emit({-ccw_lit(n, p, q, r), 
		    -ccw_lit(n, p, q, s), 
		    -ccw_lit(n, p, q, t), 
		    -ccw_lit(n, p, q, u), 
		    -ccw_lit(n, p, r, s), 
		    -ccw_lit(n, p, r, t), 
		    -ccw_lit(n, p, r, u), 
		    -ccw_lit(n, p, s, t), 
		    -ccw_lit(n, p, s, u), 
		    -ccw_lit(n, p, t, u), 
		    -ccw_lit(n, q, r, s), 
		    -ccw_lit(n, q, r, t), 
		    -ccw_lit(n, q, r, u), 
		    -ccw_lit(n, q, s, t), 
		    -ccw_lit(n, q, s, u), 
		    -ccw_lit(n, q, t, u), 
		    -ccw_lit(n, r, s, t), 
		    -ccw_lit(n, r, s, u), 
		    -ccw_lit(n, r, t, u), 
		    -ccw_lit(n, s, t, u)});
	    }
	  }
	}
      }
    }
  }
}

void formula_no_polygon7(int n) {
  for (int p = 0; p < n; p++) {
    for (int q = p; q < n; q++) {
      if (q == p) continue;
      for (int r = p; r < n; r++) {
	if (r == p || r == q) continue;
	std::cerr << p << " " << q << " " << r << std::endl;
	for (int s = p; s < n; s++) {
	  if (s == p || s == q || s == r) continue;
	  for (int t = p; t < n; t++) {
	    if (t == p || t == q || t == r || t == s) continue;
	    for (int u = p; u < n; u++) {
	      if (u == p || u == q || u == r || u == s || u == t) continue;
	      for (int v = p; v < n; v++) {
		if (v == p || v == q || v == r || v == s || v == t || v == u) continue;
		emit({-ccw_lit(n, p, q, r), 
		      -ccw_lit(n, p, q, s), 
		      -ccw_lit(n, p, q, t), 
		      -ccw_lit(n, p, q, u),
		      -ccw_lit(n, p, q, v), 
		      -ccw_lit(n, p, r, s), 
		      -ccw_lit(n, p, r, t), 
		      -ccw_lit(n, p, r, u),
		      -ccw_lit(n, p, r, v), 
		      -ccw_lit(n, p, s, t), 
		      -ccw_lit(n, p, s, u), 
		      -ccw_lit(n, p, s, v), 
		      -ccw_lit(n, p, t, u), 
		      -ccw_lit(n, p, t, v), 
		      -ccw_lit(n, p, u, v), 
		      -ccw_lit(n, q, r, s), 
		      -ccw_lit(n, q, r, t), 
		      -ccw_lit(n, q, r, u), 
		      -ccw_lit(n, q, r, v), 
		      -ccw_lit(n, q, s, t), 
		      -ccw_lit(n, q, s, u), 
		      -ccw_lit(n, q, s, v), 
		      -ccw_lit(n, q, t, u), 
		      -ccw_lit(n, q, t, v), 
		      -ccw_lit(n, q, u, v), 
		      -ccw_lit(n, r, s, t), 
		      -ccw_lit(n, r, s, u), 
		      -ccw_lit(n, r, s, v), 
		      -ccw_lit(n, r, t, u), 
		      -ccw_lit(n, r, t, v), 
		      -ccw_lit(n, r, u, v), 
		      -ccw_lit(n, s, t, u),
		      -ccw_lit(n, s, t, v),
		      -ccw_lit(n, s, u, v),
		      -ccw_lit(n, t, u, v)
		      });
	      }
	    }
	  }
	}
      }
    }
  }
}

void formula_all_3_1(const std::vector<int>& H) {
  for (int i = 0; i < H.size() - 1; i++)
    if (H[i] == 3 && H[i+1] >= 2) {
      int n = std::accumulate(H.begin(), H.end(), 0);
      int start = std::accumulate(H.begin(), H.begin() + i, 0);
      emit({-ccw_lit(n, start + 3, start + 4, start + 0)});
      emit({ccw_lit(n, start + 3, start + 4, start + 1)});
    }
}

void formula_all_4_1(const std::vector<int>& H) {
  for (int i = 0; i < H.size() - 1; i++)
    if (H[i] == 4) {
      int n = std::accumulate(H.begin(), H.end(), 0);
      int start = std::accumulate(H.begin(), H.begin() + i, 0);
      emit({-ccw_lit(n, start + 0, start + 2, start + 4)});
      emit({ccw_lit(n, start + 1, start + 3, start + 4)});
    }
}

void formula_all_5_canon(const std::vector<int>& H) {
  for (int i = 0; i < H.size() - 1; i++)
    if (H[i] == 5) {
      int n = std::accumulate(H.begin(), H.end(), 0);
      int start = std::accumulate(H.begin(), H.begin() + i, 0);
      emit({-ccw_lit(n, start + 0, start + 3, start + 5)});
      emit({ccw_lit(n, start + 1, start + 3, start + 5)});
      emit({ccw_lit(n, start + 2, start + 4, start + 5)});
      emit({ccw_lit(n, start + 0, start + 2, start + 5), ccw_lit(n, start + 1, start + 4, start + 5)});
    }
}

void generate(const std::vector<int>& H) {
  units.clear();
  (*output) << "p cnf 0 0" << std::endl;
  formula_hulls_nested(H);
  formula_hulls_polygons(H);
  formula_all_3_1(H);
  formula_all_4_1(H);
  formula_all_5_canon(H);
  formula_ax4(std::accumulate(H.begin(), H.end(), 0));
  formula_ax5(std::accumulate(H.begin(), H.end(), 0));
  formula_no_polygon6(std::accumulate(H.begin(), H.end(), 0));
}

std::string join(const std::vector<int>& v) {
  std::string s;
  for (auto it = v.begin(); it != v.end(); it++) {
    if (it != v.begin())
      s += "_";
    s += std::to_string(*it);
  }
  return s;
}

int main() {
  std::ios::sync_with_stdio(false);
  
  std::string line;
  while (std::getline(std::cin, line)) {
    std::istringstream iss_line(line);
    std::vector<int> H;
    int x;
    while (iss_line >> x)
      H.push_back(x);
    std::ofstream out("formula_" + join(H) + ".cnf");
    output = &out;

    std::cout << join(H) << "\t";
    
    clock_t begin = clock();
    generate(H);
    clock_t end = clock();
    
    double elapsed_secs = double(end - begin) / CLOCKS_PER_SEC;

    std::cout << std::fixed << std::showpoint << std::setprecision(3) << elapsed_secs
	      << std::endl;
  }
}
