header{* Homogeneous coordinates in extended complex plane *}
theory HomogeneousCoordinates
imports MoreComplex Matrices
begin

typedef homo_coords = "{v. v \<noteq> vec_zero}"
by (rule_tac x="(1, 0)" in exI, simp)

lemma obtain_homo_coords:
  fixes x::homo_coords
  obtains A B where
  "Rep_homo_coords x = (A, B)" "A \<noteq> 0 \<or> B \<noteq> 0"
by (cases x) (auto simp add: Abs_homo_coords_inverse)

definition homo_coords_eq :: "homo_coords \<Rightarrow> homo_coords \<Rightarrow> bool" (infix "\<approx>" 50) where
 [simp]: "z1 \<approx> z2 \<longleftrightarrow>
     (let z1 = Rep_homo_coords z1; 
          z2 = Rep_homo_coords z2 
       in (\<exists> k. k \<noteq> (0::complex) \<and> z2 = k *\<^sub>s\<^sub>v z1))"

lemma homo_coords_eq_reflp:
  "reflp homo_coords_eq"
by (auto simp add: reflp_def, rule_tac x="1" in exI, simp)

lemma homo_coords_eq_symp:
  "symp homo_coords_eq"
by (auto simp add: symp_def, rule_tac x="1/k" in exI, simp)

lemma homo_coords_eq_transp:
"transp homo_coords_eq"
by (auto simp add: transp_def, rule_tac x="ka*k" in exI, simp)

lemma homo_coords_eq_equivp:
  "equivp homo_coords_eq"
  by (auto intro: equivpI homo_coords_eq_reflp homo_coords_eq_symp homo_coords_eq_transp)

lemma homo_coords_eq_refl [simp]:
  "z \<approx> z"
using homo_coords_eq_reflp
by (auto simp add: reflp_def refl_on_def)

lemma homo_coords_eq_trans:
  assumes "z1 \<approx> z2"  "z2 \<approx> z3"
  shows "z1 \<approx> z3"
using assms homo_coords_eq_transp
unfolding transp_def
by blast

lemma homo_coords_eq_sym:
  assumes "z1 \<approx> z2"
  shows "z2 \<approx> z1"
using assms homo_coords_eq_symp
unfolding symp_def
by blast

lemma homo_coords_eq_mix:
  assumes  "Rep_homo_coords z1 = (z1', z1'')" "Rep_homo_coords z2 = (z2', z2'')"
  shows "z1 \<approx> z2 \<longleftrightarrow> z2'*z1'' = z1'*z2''"
using assms
proof (cases "z1'' \<noteq> 0 \<or> z2'' \<noteq> 0")
  case False
  thus ?thesis
    using assms using Rep_homo_coords[of z1]  Rep_homo_coords[of z2]
    by auto
next
  case True
  thus ?thesis
    using assms
    apply auto
    apply (rule_tac x="z2''/z1''" in exI)
    using Rep_homo_coords[of z2]
    apply (auto simp add: field_simps)
    apply (rule_tac x="z2''/z1''" in exI)
    using Rep_homo_coords[of z1]
    apply (auto simp add: field_simps)
    done
qed

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (Rep_homo_coords x)) = Rep_homo_coords x"
using Rep_homo_coords[of x]
by (simp add: Abs_homo_coords_inverse)

text {* Quotient of homogeneous coordinates *}
quotient_type 
  complex_homo = homo_coords / "homo_coords_eq"
by (rule homo_coords_eq_equivp)

text {* Infinite point *}
definition inf_homo_rep where [simp]: "inf_homo_rep = Abs_homo_coords (1, 0)"
lift_definition inf_homo :: complex_homo  ("\<infinity>\<^sub>h")  is inf_homo_rep
done

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (1, 0)) = (1, 0)"
by (simp add: Abs_homo_coords_inverse)

lemma [simp]: "Rep_homo_coords inf_homo_rep = (1, 0)"
by simp

lemma inf_snd_0: "z \<approx> inf_homo_rep \<longleftrightarrow> (let (z1, z2) = Rep_homo_coords z in z1 \<noteq> 0 \<and> z2 = 0)"
using Rep_homo_coords[of z]
by auto

lemma not_inf_snd_not0: 
  assumes "\<not> z \<approx> inf_homo_rep" 
  shows "let (z1, z2) = Rep_homo_coords z in z2 \<noteq> 0"
using assms Rep_homo_coords[of z] inf_snd_0[of z]
by auto

text{* Zero *}
definition zero_homo_rep where [simp]: "zero_homo_rep = Abs_homo_coords (0, 1)"
lift_definition zero_homo :: complex_homo ("0\<^sub>h") is zero_homo_rep
done

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (0, 1)) = (0, 1)"
by (simp add: Abs_homo_coords_inverse)

lemma [simp]: "Rep_homo_coords zero_homo_rep = (0, 1)"
by simp

lemma zero_fst_0: "z \<approx> zero_homo_rep \<longleftrightarrow> (let (z1, z2) = Rep_homo_coords z in z1 = 0 \<and> z2 \<noteq> 0)"
using Rep_homo_coords[of z]
by auto

text{* One *}
definition one_homo_rep where [simp]: "one_homo_rep = Abs_homo_coords (1, 1)"
lift_definition one_homo :: complex_homo  ("1\<^sub>h")  is one_homo_rep
done

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (1, 1)) = (1, 1)"
by (simp add: Abs_homo_coords_inverse)

lemma [simp]: "Rep_homo_coords one_homo_rep = (1, 1)"
by simp

lemma [simp]: "1\<^sub>h \<noteq> \<infinity>\<^sub>h" "0\<^sub>h \<noteq> \<infinity>\<^sub>h" "0\<^sub>h \<noteq> 1\<^sub>h" "1\<^sub>h \<noteq> 0\<^sub>h" "\<infinity>\<^sub>h \<noteq> 0\<^sub>h" "\<infinity>\<^sub>h \<noteq> 1\<^sub>h"
by (transfer, auto)+

(* ii *)

definition ii_homo_rep where "ii_homo_rep = Abs_homo_coords (ii, 1)"

lift_definition ii_homo :: "complex_homo"  ("ii\<^sub>h")  is ii_homo_rep
done

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (ii, 1)) = (ii, 1)"
  by (simp add: Abs_homo_coords_inverse)

lemma [simp]: "Rep_homo_coords ii_homo_rep = (ii, 1)"
  by (simp add: ii_homo_rep_def)

(* Three points *)

lemma ex_3_different_points:
  fixes z::complex_homo
  shows "\<exists> z1 z2. z \<noteq> z1 \<and> z1 \<noteq> z2 \<and> z \<noteq> z2"
proof (cases "z \<noteq> 0\<^sub>h \<and> z \<noteq> 1\<^sub>h")
  case True
  thus ?thesis
    by (rule_tac x="0\<^sub>h" in exI, rule_tac x="1\<^sub>h" in exI, auto)
next
  case False
  hence "z = 0\<^sub>h \<or> z = 1\<^sub>h"
    by simp
  thus ?thesis
  proof
    assume "z = 0\<^sub>h"
    thus ?thesis
      by (rule_tac x="\<infinity>\<^sub>h" in exI, rule_tac x="1\<^sub>h" in exI, auto)
  next
    assume "z = 1\<^sub>h"
    thus ?thesis
      by (rule_tac x="\<infinity>\<^sub>h" in exI, rule_tac x="0\<^sub>h" in exI, auto)
  qed
qed

text {* Conversion from complex *}
definition of_complex_coords where
  "of_complex_coords z = Abs_homo_coords (z, 1)"

lemma [simp]: "Rep_homo_coords (of_complex_coords z) = (z, 1)"
by (simp add: of_complex_coords_def Abs_homo_coords_inverse)

lift_definition of_complex :: "complex \<Rightarrow> complex_homo" is of_complex_coords
by (simp del: homo_coords_eq_def)

lemma of_complex_inj:
  assumes "of_complex x = of_complex y"
  shows "x = y"
using assms
by transfer simp

lemma of_complex_image_inj:
  assumes "of_complex ` A = of_complex ` B"
  shows "A = B"
using assms
using of_complex_inj
by auto

lemma [simp]: "of_complex x \<noteq> \<infinity>\<^sub>h"
by transfer simp

lemma [simp]: "\<infinity>\<^sub>h \<noteq> of_complex x"
by transfer simp

lemma inf_homo_or_complex_homo:
   "z = \<infinity>\<^sub>h \<or> (\<exists> x. z = of_complex x)"
proof(transfer)
  fix z
  obtain a b where *: "Rep_homo_coords z = (a, b)"
    by (rule obtain_homo_coords)
  show "z \<approx> inf_homo_rep \<or> (\<exists>x. z \<approx> of_complex_coords x)"
    using * Rep_homo_coords[of z]
    by (cases "b = 0") auto
qed

lemma zero_of_complex [simp]: "of_complex 0 = 0\<^sub>h"
by transfer simp

lemma one_of_complex [simp]: "of_complex 1 = 1\<^sub>h"
by transfer simp

lemma 
  [simp]: "of_complex a = 0\<^sub>h \<longleftrightarrow> a = 0"
by (subst zero_of_complex[symmetric]) (auto simp add: of_complex_inj)

lemma 
  [simp]: "of_complex a = 1\<^sub>h \<longleftrightarrow> a = 1"
by (subst one_of_complex[symmetric]) (auto simp add: of_complex_inj)

text {* Coercion to complex *}
definition to_complex_homo_coords :: "homo_coords \<Rightarrow> complex" where
  "to_complex_homo_coords z = (let (z1, z2) = Rep_homo_coords z in z1/z2)"

lift_definition to_complex :: "complex_homo \<Rightarrow> complex" is to_complex_homo_coords
proof-
  fix x y
  assume "x \<approx> y"
  thus "to_complex_homo_coords x = to_complex_homo_coords y"
    by (auto simp add: to_complex_homo_coords_def split_def Let_def)
qed

lemma [simp]: "to_complex (of_complex z) = z"
by (transfer) (simp add: of_complex_coords_def to_complex_homo_coords_def Abs_homo_coords_inverse)

lemma [simp]: "z \<noteq> \<infinity>\<^sub>h \<Longrightarrow> (of_complex (to_complex z)) = z"
proof (transfer)
  fix z
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  assume "\<not> z \<approx> inf_homo_rep"
  hence "z2 \<noteq> 0"
    using zz Rep_homo_coords[of z]
    by auto (erule_tac x="1/z1" in allE, simp)
  thus "of_complex_coords (to_complex_homo_coords z) \<approx> z"
    using zz
    by (auto simp add: of_complex_coords_def to_complex_homo_coords_def Abs_homo_coords_inverse)
qed

text {* Addition *}
definition add_homo_coords :: "homo_coords \<Rightarrow> homo_coords \<Rightarrow> homo_coords" (infixl "+\<^sub>h\<^sub>c" 100) where 
  "z +\<^sub>h\<^sub>c w = (let (z1, z2) = Rep_homo_coords z; 
                  (w1, w2) = Rep_homo_coords w in 
      Abs_homo_coords (z1*w2 + w1*z2, z2*w2))"

lemma add_homo_coords_Rep:
  assumes "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)" "z2 \<noteq> 0 \<or> w2 \<noteq> 0"
  shows "Rep_homo_coords (z +\<^sub>h\<^sub>c w) = (z1*w2 + w1*z2, z2*w2)"
proof-
  from assms
  have "(z1*w2 + w1*z2, z2*w2) \<noteq> vec_zero"
    using Rep_homo_coords[of z] Rep_homo_coords[of w]
    by auto
  thus ?thesis
    using assms(1-2)
    by (auto simp add: add_homo_coords_def split_def Let_def Abs_homo_coords_inverse)
qed

lemma add_homo_coords_00:
  assumes "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)" "z2 = 0" "w2 = 0"
  shows "z +\<^sub>h\<^sub>c w = Abs_homo_coords (0, 0)"
using assms unfolding add_homo_coords_def
by simp

lemma add_coords_well_defined_lemma:
  assumes "x \<approx> y" "x' \<approx> y'" 
  shows "x +\<^sub>h\<^sub>c x' \<approx> y +\<^sub>h\<^sub>c y'"
using assms
proof-
  obtain Ax Bx where xx: "Rep_homo_coords x = (Ax, Bx)"
    by (rule obtain_homo_coords)
  obtain Ax' Bx' where xx': "Rep_homo_coords x' = (Ax', Bx')"
    by (rule obtain_homo_coords)
  obtain Ay By where yy: "Rep_homo_coords y = (Ay, By)"
    by (rule obtain_homo_coords)
  obtain Ay' By' where yy': "Rep_homo_coords y' = (Ay', By')"
    by (rule obtain_homo_coords)
  from assms obtain k k' where
    *: "k \<noteq> 0" "Ay = k*Ax" "By = k*Bx" "k' \<noteq> 0" "Ay' = k'*Ax'" "By' = k'*Bx'"
    using xx xx' yy yy'
    by auto
  show ?thesis
  proof (cases "Bx = 0 \<and> Bx' = 0")
    case True
    thus ?thesis
      using add_homo_coords_00[of x Ax 0 x' Ax' 0] add_homo_coords_00[of y Ay 0 y' Ay' 0] xx yy xx' yy' *
      by (auto, rule_tac x="1" in exI, simp)
  next
    case False
    thus ?thesis
      using xx xx' yy yy' *
      using Rep_homo_coords[of x] Rep_homo_coords[of x'] `k \<noteq> 0` `k' \<noteq> 0`
      using add_homo_coords_Rep[of x Ax Bx x' Ax' Bx'] add_homo_coords_Rep[of y "k * Ax" "k * Bx" y' "k' * Ax'" "k' * Bx'"]
      by simp (rule_tac x="k*k'" in exI, auto simp add: field_simps)
  qed
qed

lift_definition add_homo :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" (infixl "+\<^sub>h" 100) is add_homo_coords
by (rule add_coords_well_defined_lemma, simp_all)

lemma add_homo_commute: "x +\<^sub>h y = y +\<^sub>h x"
proof (transfer)
  fix x y
  obtain Ax Bx where xx: "Rep_homo_coords x = (Ax, Bx)"
    by (rule obtain_homo_coords)
  obtain Ay By where yy: "Rep_homo_coords y = (Ay, By)"
    by (rule obtain_homo_coords)
  
  show "x +\<^sub>h\<^sub>c y \<approx> y +\<^sub>h\<^sub>c x"
  proof (cases "Bx \<noteq> 0 \<or> By \<noteq> 0")
    case True
    thus ?thesis
      using add_homo_coords_Rep[of x Ax Bx y Ay By, OF xx yy]
      using add_homo_coords_Rep[of y Ay By x Ax Bx, OF yy xx]
      by auto (rule_tac x="1" in exI, simp)+
  next
    case False
    thus ?thesis
      using xx yy add_homo_coords_00
      by (auto, rule_tac x="1" in exI, simp)
  qed
qed

lemma of_complex_add: "(of_complex za) +\<^sub>h (of_complex zb) = of_complex (za + zb)"
proof (transfer)
  fix za zb
  have "Rep_homo_coords (Abs_homo_coords (za, 1)) = (za, 1)"  "Rep_homo_coords (Abs_homo_coords (zb, 1)) = (zb, 1)"
    by (auto simp add: Abs_homo_coords_inverse)
  thus "of_complex_coords za +\<^sub>h\<^sub>c of_complex_coords zb \<approx>  of_complex_coords (za + zb)"
    unfolding of_complex_coords_def
    using add_homo_coords_Rep[of "Abs_homo_coords (za, 1)" za 1 "Abs_homo_coords (zb, 1)" zb 1]
    by (simp add: Abs_homo_coords_inverse)
qed

lemma [simp]: "(of_complex z) +\<^sub>h \<infinity>\<^sub>h = \<infinity>\<^sub>h"
proof (transfer)
  fix z
  show "of_complex_coords z +\<^sub>h\<^sub>c inf_homo_rep \<approx> inf_homo_rep"
    using add_homo_coords_Rep[of "Abs_homo_coords (z, 1)" z 1 "Abs_homo_coords (1, 0)" 1 0]
    unfolding of_complex_coords_def
    by (simp add: Abs_homo_coords_inverse)
qed

lemma [simp]: "\<infinity>\<^sub>h +\<^sub>h (of_complex z) = \<infinity>\<^sub>h"
  by (subst add_homo_commute) simp

lemma add_homo_zero_right [simp]: "z +\<^sub>h 0\<^sub>h = z"
proof (transfer)
  fix z
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  thus "z +\<^sub>h\<^sub>c zero_homo_rep \<approx> z"
    using add_homo_coords_Rep[of z z1 z2 zero_homo_rep 0 1]
    by auto (metis zero_neq_one)
qed

lemma add_homo_zero_left [simp]: "0\<^sub>h +\<^sub>h z = z"
  by (subst add_homo_commute) simp

text {* uminus *}
definition uminus_homo_coords where
  "uminus_homo_coords z = (let (z1, z2) = Rep_homo_coords z in Abs_homo_coords (-z1, z2))"

lemma uminus_homo_coords_Rep [simp]: "Rep_homo_coords (uminus_homo_coords z) = (let (z1, z2) = Rep_homo_coords z in (-z1, z2))"
unfolding uminus_homo_coords_def Let_def
apply (cases "Rep_homo_coords z")
using Rep_homo_coords[of z]
by (auto simp add: Abs_homo_coords_inverse)

lift_definition uminus_homo :: "complex_homo \<Rightarrow> complex_homo" is uminus_homo_coords
by (auto simp add: split_def Let_def) 

lemma of_complex_uminus [simp]: "uminus_homo (of_complex z) = of_complex (-z)"
by (transfer) auto

text {* Subtraction *}
definition minus_homo :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" (infixl "-\<^sub>h" 100) where
  "z1 -\<^sub>h z2 = z1 +\<^sub>h (uminus_homo z2)"

lemma minus_homo_coords_Rep:
  assumes "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)" "z2 \<noteq> 0 \<or> w2 \<noteq> 0"
  shows "Rep_homo_coords (z +\<^sub>h\<^sub>c (uminus_homo_coords w)) = (z1*w2 - w1*z2, z2*w2)"
using assms
using add_homo_coords_Rep[of z z1 z2 "uminus_homo_coords w" "-w1" "w2"] uminus_homo_coords_Rep[of w]
by simp

lemma of_complex_minus:
  "(of_complex z1) -\<^sub>h (of_complex z2) = of_complex (z1 - z2)"
unfolding minus_homo_def complex_diff_def
by (simp add: of_complex_add)

lemma [simp]:
  assumes "z \<noteq> \<infinity>\<^sub>h"
  shows "z -\<^sub>h z = 0\<^sub>h"
proof-
  from assms obtain z' where "z = of_complex z'"
    using inf_homo_or_complex_homo[of z]
    by auto
  thus ?thesis
    by (simp add: of_complex_minus)
qed

lemma diff_zero_homo:
  assumes "z1 -\<^sub>h z2 = 0\<^sub>h" "z1 \<noteq> \<infinity>\<^sub>h \<or> z2 \<noteq> \<infinity>\<^sub>h"
  shows "z1 = z2"
using assms
unfolding minus_homo_def
proof transfer
  fix z w
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  obtain w1 w2 where ww: "Rep_homo_coords w = (w1, w2)"
    by (rule obtain_homo_coords)
  have mww: "Rep_homo_coords (uminus_homo_coords w) = (-w1, w2)"
    using ww
    by simp
  assume *: "z +\<^sub>h\<^sub>c uminus_homo_coords w \<approx> zero_homo_rep"  and
            "\<not> z \<approx> inf_homo_rep \<or> \<not> w \<approx> inf_homo_rep"
  have "z2 \<noteq> 0 \<or> w2 \<noteq> 0"
    using Rep_homo_coords[of z] Rep_homo_coords[of w]
    using `\<not> z \<approx> inf_homo_rep \<or> \<not> w \<approx> inf_homo_rep`
    using inf_snd_0[of z] inf_snd_0[of w] zz ww
    by auto
  thus "z \<approx> w"
    using * zz ww
    apply simp
    apply (subst (asm) minus_homo_coords_Rep[of z z1 z2 w w1 w2])
    apply auto
    apply (rule_tac x="w2/z2" in exI, auto simp add: field_simps)
    apply (rule_tac x="w2/z2" in exI, auto)
    done
qed

text {* Multiplication *}
definition mult_homo_coords :: "homo_coords \<Rightarrow> homo_coords \<Rightarrow> homo_coords" (infixl "*\<^sub>h\<^sub>c" 100) where 
  "x *\<^sub>h\<^sub>c y = (let (x1, y1) = Rep_homo_coords x; 
                  (x2, y2) = Rep_homo_coords y in 
      Abs_homo_coords (x1*x2, y1*y2))"

lemma mult_homo_coords_Rep:
  assumes "Rep_homo_coords x = (Ax, Bx)" "Rep_homo_coords x' = (Ax', Bx')" "(Bx \<noteq> 0 \<or> Ax' \<noteq> 0) \<and> (Bx' \<noteq> 0 \<or> Ax \<noteq> 0)"
  shows "Rep_homo_coords (x *\<^sub>h\<^sub>c x') = (Ax*Ax', Bx*Bx')"
using assms Rep_homo_coords[of x] Rep_homo_coords[of x']
by (auto simp add: mult_homo_coords_def split_def Let_def Abs_homo_coords_inverse)

lemma mult_homo_coords_00:
  assumes "Rep_homo_coords x = (Ax, Bx)" "Rep_homo_coords x' = (Ax', Bx')" "(Bx = 0 \<and> Ax' = 0) \<or> (Bx' = 0 \<and> Ax = 0)"
  shows "x *\<^sub>h\<^sub>c x' = Abs_homo_coords (0, 0)"
using assms unfolding mult_homo_coords_def
by auto

lemma mult_coords_well_defined_lemma:
  assumes "x \<approx> y" "x' \<approx> y'" 
  shows "x *\<^sub>h\<^sub>c x' \<approx> y *\<^sub>h\<^sub>c y'"
proof-
  obtain Ax Bx where xx: "Rep_homo_coords x = (Ax, Bx)"
    by (rule obtain_homo_coords)
  obtain Ax' Bx' where xx': "Rep_homo_coords x' = (Ax', Bx')"
    by (rule obtain_homo_coords)
  obtain Ay By where yy: "Rep_homo_coords y = (Ay, By)"
    by (rule obtain_homo_coords)
  obtain Ay' By' where yy': "Rep_homo_coords y' = (Ay', By')"
    by (rule obtain_homo_coords)
  from assms obtain k k' where
    *: "k \<noteq> 0" "Ay = k*Ax" "By = k*Bx" "k' \<noteq> 0" "Ay' = k'*Ax'" "By' = k'*Bx'"
    using xx xx' yy yy'
    by auto
  show ?thesis
  proof (cases "(Bx \<noteq> 0 \<or> Ax' \<noteq> 0) \<and> (Bx' \<noteq> 0 \<or> Ax \<noteq> 0)")
    case False
    thus ?thesis
      using mult_homo_coords_00[of x Ax Bx x' Ax' Bx'] mult_homo_coords_00[of y Ay By y' Ay' By'] xx yy xx' yy' *
      by auto (rule_tac x="1" in exI, simp)+
  next
    case True
    thus ?thesis
      using xx xx' yy yy' *
      using Rep_homo_coords[of x] Rep_homo_coords[of x'] `k \<noteq> 0` `k' \<noteq> 0`
      using mult_homo_coords_Rep[of x Ax Bx x' Ax' Bx']
            mult_homo_coords_Rep[of y "k * Ax" "k * Bx" y' "k' * Ax'" "k' * Bx'"]
      by simp (rule_tac x="k*k'" in exI, auto simp add: field_simps)
  qed
qed

lift_definition mult_homo :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" (infixl "*\<^sub>h" 100) is mult_homo_coords
by (rule mult_coords_well_defined_lemma, simp_all)

lemma mult_of_complex:
  shows "(of_complex z1) *\<^sub>h (of_complex z2) = of_complex (z1 * z2)"
proof (transfer)
  fix z1 z2
  show "of_complex_coords z1 *\<^sub>h\<^sub>c of_complex_coords z2 \<approx> of_complex_coords (z1 * z2)"
    using mult_homo_coords_Rep[of "of_complex_coords z1" _ _ "of_complex_coords z2"]
    by simp
qed

lemma mult_homo_commute:
  shows "z1 *\<^sub>h z2 = z2 *\<^sub>h z1"
proof transfer
  fix z1 z2
  obtain z11 z12 where z1: "Rep_homo_coords z1 = (z11, z12)"
    by (rule obtain_homo_coords)
  obtain z21 z22 where z2: "Rep_homo_coords z2 = (z21, z22)"
    by (rule obtain_homo_coords)
  show "z1 *\<^sub>h\<^sub>c z2 \<approx> z2 *\<^sub>h\<^sub>c z1"
  proof (cases "(z12 \<noteq> 0 \<or> z21 \<noteq> 0) \<and> (z22 \<noteq> 0 \<or> z11 \<noteq> 0)")
    case True
    thus ?thesis
      using mult_homo_coords_Rep[of z1 z11 z12 z2 z21 z22] z1 z2
      using mult_homo_coords_Rep[of z2 z21 z22 z1 z11 z12]
      by simp (rule_tac x=1 in exI, simp)
  next
    case False
    thus ?thesis
      using mult_homo_coords_00[of z1 z11 z12 z2 z21 z22] z1 z2
      using mult_homo_coords_00[of z2 z21 z22 z1 z11 z12]
      by auto (rule_tac x=1 in exI, simp)+
  qed
qed

lemma mult_homo_zero_left [simp]:
  assumes "z \<noteq> \<infinity>\<^sub>h"
  shows "0\<^sub>h *\<^sub>h z = 0\<^sub>h"
using assms
proof-
  obtain z' where "z = of_complex z'"
    using inf_homo_or_complex_homo[of z] assms
    by auto
  thus ?thesis
    using zero_of_complex
    using mult_of_complex[of 0 z']
    by simp
qed

lemma mult_homo_zero_right [simp]:
  assumes "z \<noteq> \<infinity>\<^sub>h"
  shows "z *\<^sub>h 0\<^sub>h = 0\<^sub>h"
using mult_homo_zero_left[OF assms]
by (simp add: mult_homo_commute)

lemma mult_homo_inf_right [simp]:
  assumes "z \<noteq> 0\<^sub>h"
  shows "z *\<^sub>h \<infinity>\<^sub>h = \<infinity>\<^sub>h"
using assms
proof (transfer)
  fix z
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  assume "\<not> z \<approx> zero_homo_rep"
  hence "z1 \<noteq> 0"
    using Rep_homo_coords[of z] zz
    by auto (metis divide_self_if eq_divide_eq mult_divide_mult_cancel_right)
  thus "z *\<^sub>h\<^sub>c inf_homo_rep \<approx> inf_homo_rep"
    using zz mult_homo_coords_Rep[of z z1 z2 "Abs_homo_coords (1, 0)" 1 0]
    by auto
qed

lemma mult_homo_inf_left [simp]:
  assumes "z \<noteq> 0\<^sub>h"
  shows "\<infinity>\<^sub>h *\<^sub>h z = \<infinity>\<^sub>h"
using mult_homo_inf_right[OF assms]
by (simp add: mult_homo_commute)

lemma mult_homo_one_left [simp]:
  shows "1\<^sub>h *\<^sub>h z = z"
proof (transfer)
  fix z
  obtain z1 z2 where "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  thus "one_homo_rep *\<^sub>h\<^sub>c z \<approx> z"
    using mult_homo_coords_Rep[of "Abs_homo_coords (1, 1)" 1 1 z z1 z2]
    by auto (metis zero_neq_one)
qed

lemma mult_homo_one_right [simp]:
  shows "z *\<^sub>h 1\<^sub>h = z"
using mult_homo_one_left[of z]
by (simp add: mult_homo_commute)

text {* Reciprocal *}
definition reciprocal_homo_coords :: "homo_coords \<Rightarrow> homo_coords" where 
  "reciprocal_homo_coords x = (let (x1, y1) = Rep_homo_coords x in Abs_homo_coords (y1, x1))"

lemma reciprocal_homo_coords_Rep: "Rep_homo_coords (reciprocal_homo_coords x) = (let (x1, y1) = Rep_homo_coords x in (y1, x1))"
apply (cases "Rep_homo_coords x")
unfolding reciprocal_homo_coords_def Let_def
using Rep_homo_coords[of x]
by (auto simp add: Abs_homo_coords_inverse)

lift_definition reciprocal_homo :: "complex_homo \<Rightarrow> complex_homo" is reciprocal_homo_coords
proof-
  fix x y
  assume "x \<approx> y"
  thus "reciprocal_homo_coords x \<approx> reciprocal_homo_coords y"
    by (cases "Rep_homo_coords x", cases "Rep_homo_coords y")  (auto simp add: reciprocal_homo_coords_Rep)
qed

lemma [simp]: "reciprocal_homo_coords (reciprocal_homo_coords z) = z"
  unfolding reciprocal_homo_coords_def[of "reciprocal_homo_coords z"]
  by (cases "Rep_homo_coords z")  (auto simp add: reciprocal_homo_coords_Rep, metis Rep_homo_coords_inverse) 

lemma [simp]: "reciprocal_homo (reciprocal_homo z) = z"
by (transfer) (auto, rule_tac x="1" in exI, simp)

lemma [simp]: "reciprocal_homo 0\<^sub>h = \<infinity>\<^sub>h"
by (transfer) (simp add: reciprocal_homo_coords_Rep)

lemma [simp]: "reciprocal_homo \<infinity>\<^sub>h = 0\<^sub>h"
by (transfer) (simp add: reciprocal_homo_coords_Rep)

lemma [simp]: "reciprocal_homo 1\<^sub>h = 1\<^sub>h"
by (transfer) (simp add: reciprocal_homo_coords_Rep)

text {* Division *}
definition divide_homo :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" (infixl ":\<^sub>h" 100) where
  "x :\<^sub>h y = x *\<^sub>h (reciprocal_homo y)" 

lemma [simp]:
  assumes "z \<noteq> 0\<^sub>h"
  shows "z :\<^sub>h 0\<^sub>h = \<infinity>\<^sub>h"
using assms
unfolding divide_homo_def
by simp

lemma [simp]:
  assumes "z \<noteq> \<infinity>\<^sub>h"
  shows "z :\<^sub>h \<infinity>\<^sub>h = 0\<^sub>h"
using assms
unfolding divide_homo_def
by simp

lemma [simp]: "\<infinity>\<^sub>h :\<^sub>h 0\<^sub>h = \<infinity>\<^sub>h"
unfolding divide_homo_def
by (transfer) (simp add: reciprocal_homo_coords_def mult_homo_coords_def)

lemma [simp]: "0\<^sub>h :\<^sub>h \<infinity>\<^sub>h =  0\<^sub>h"
unfolding divide_homo_def
by (transfer) (simp add: mult_homo_coords_def reciprocal_homo_coords_def)

lemma divide_homo_one [simp]:
  shows "z :\<^sub>h 1\<^sub>h = z"
unfolding divide_homo_def
by simp

lemma of_complex_divide:
  assumes "z2 \<noteq> 0"
  shows "(of_complex z1) :\<^sub>h (of_complex z2) = of_complex (z1 / z2)"
using assms
unfolding divide_homo_def
proof (transfer)
  fix z1 z2  :: complex
  assume "z2 \<noteq> 0"
  thus "of_complex_coords z1 *\<^sub>h\<^sub>c reciprocal_homo_coords (of_complex_coords z2) \<approx>
       of_complex_coords (z1 / z2)"
    by (auto simp add: of_complex_coords_def Abs_homo_coords_inverse mult_homo_coords_def reciprocal_homo_coords_def)
       (rule_tac x="1/z2" in exI,  auto)
qed

lemma divide_homo_coords_Rep [simp]:
  assumes "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)"
          "(z2 \<noteq> 0 \<or> w2 \<noteq> 0) \<and> (w1 \<noteq> 0 \<or> z1 \<noteq> 0)"
  shows "Rep_homo_coords (z *\<^sub>h\<^sub>c (reciprocal_homo_coords w)) = (z1*w2, z2*w1)"
using assms
using mult_homo_coords_Rep[of z z1 z2 "reciprocal_homo_coords w" "w2" "w1"] reciprocal_homo_coords_Rep[of w]
by simp

text {* Conjugate *}
definition cnj_homo_coords where 
  "cnj_homo_coords z = (let (z1, z2) = Rep_homo_coords z in Abs_homo_coords (cnj z1, cnj z2))"

lemma [simp]: "Rep_homo_coords (cnj_homo_coords z) = vec_cnj (Rep_homo_coords z)"
apply (cases "Rep_homo_coords z")
using Rep_homo_coords[of z]
by (simp add: cnj_homo_coords_def Abs_homo_coords_inverse vec_cnj_def)

lift_definition cnj_homo :: "complex_homo \<Rightarrow> complex_homo" is cnj_homo_coords
by auto

lemma "cnj_homo (of_complex z) = of_complex (cnj z)"
by (transfer) (simp add: vec_cnj_def)

lemma "cnj_homo \<infinity>\<^sub>h = \<infinity>\<^sub>h"
by (transfer) (simp add: vec_cnj_def)

lemma cnj_homo_coords_involution [simp]:
  "cnj_homo_coords (cnj_homo_coords z) = z"
unfolding cnj_homo_coords_def[of "cnj_homo_coords z"] Let_def
by (cases "Rep_homo_coords z", auto simp add: Let_def split_def vec_cnj_def) (metis Rep_homo_coords_inverse)

lemma cnj_homo_involution [simp]: "cnj_homo (cnj_homo z) = z"
by (transfer) (auto, rule_tac x="1" in exI, simp)

lemma [simp]: 
  "cnj_homo \<infinity>\<^sub>h = \<infinity>\<^sub>h"
by (transfer) (auto simp add: vec_cnj_def)

lemma [simp]: 
  "cnj_homo 0\<^sub>h = 0\<^sub>h"
by (transfer) (auto simp add: vec_cnj_def)

text {* Inversion *}
definition inversion_homo where
  "inversion_homo = cnj_homo \<circ> reciprocal_homo"

lemma inversion_homo_sym:
  "inversion_homo = reciprocal_homo \<circ> cnj_homo"
unfolding inversion_homo_def
by (rule ext, simp) (transfer, case_tac "Rep_homo_coords x", auto simp add: reciprocal_homo_coords_Rep split_def Let_def vec_cnj_def, metis zero_neq_one)

lemma inversion_homo_involution [simp]: "inversion_homo (inversion_homo z) = z"
proof-
  have *: "cnj_homo \<circ> reciprocal_homo = reciprocal_homo \<circ> cnj_homo"
    using inversion_homo_sym
    by (simp add: inversion_homo_def)
  show ?thesis
    unfolding inversion_homo_def
    by (subst *) simp
qed

lemma [simp]:
  "inversion_homo 0\<^sub>h = \<infinity>\<^sub>h"
by (simp add: inversion_homo_def)

lemma [simp]:
  "inversion_homo \<infinity>\<^sub>h = 0\<^sub>h"
by (simp add: inversion_homo_def)

subsection{* Ratio and crossratio *}

definition ratio_rep where 
  "ratio_rep z1 z2 z3 = 
      (let (z1x, z1y) = Rep_homo_coords z1; 
           (z2x, z2y) = Rep_homo_coords z2;
           (z3x, z3y) = Rep_homo_coords z3 in
        Abs_homo_coords ((z1x*z2y - z2x*z1y)*z3y, (z1x*z3y - z3x*z1y)*z2y))"

lemma ratio_rep_Rep [simp]:
  assumes "(\<not> z1 \<approx> z2 \<and> \<not> z3 \<approx> inf_homo_rep) \<or> (\<not> z1 \<approx> z3 \<and> \<not> z2 \<approx> inf_homo_rep)"
  shows "Rep_homo_coords (ratio_rep z1 z2 z3) = (let (z1x, z1y) = Rep_homo_coords z1; 
           (z2x, z2y) = Rep_homo_coords z2;
           (z3x, z3y) = Rep_homo_coords z3 in ((z1x*z2y - z2x*z1y)*z3y, (z1x*z3y - z3x*z1y)*z2y))"
proof-
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')" 
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')" 
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')" 
    by (rule obtain_homo_coords)
  have "((z1' * z2'' - z2' * z1'') * z3'', (z1' * z3'' - z3' * z1'') * z2'') \<noteq> vec_zero"
    using assms
    using homo_coords_eq_mix[OF zz1 zz2] homo_coords_eq_mix[OF zz3, of inf_homo_rep 1 0]
    using homo_coords_eq_mix[OF zz1 zz3] homo_coords_eq_mix[OF zz2, of inf_homo_rep 1 0]
    by auto
  thus ?thesis
    using zz1 zz2 zz3
    unfolding ratio_rep_def Let_def
    by (simp add: Abs_homo_coords_inverse)
qed

lemma ratio_rep_Rep' [simp]:
  assumes "(z1 \<approx> z2 \<or> z3 \<approx> inf_homo_rep) \<and> (z1 \<approx> z3 \<or> z2 \<approx> inf_homo_rep)"
  shows "ratio_rep z1 z2 z3 = Abs_homo_coords (0, 0)"
using assms
unfolding ratio_rep_def
by (cases "Rep_homo_coords z1", cases "Rep_homo_coords z2", cases "Rep_homo_coords z3") auto

lift_definition ratio :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" is ratio_rep
proof-
  fix z1 z2 z3 w1 w2 w3
  assume *: "z1 \<approx> w1"  "z2 \<approx> w2"  "z3 \<approx> w3" 
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')" 
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')" 
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')" 
    by (rule obtain_homo_coords)
  obtain w1' w1'' where ww1: "Rep_homo_coords w1 = (w1', w1'')" 
    by (rule obtain_homo_coords)
  obtain w2' w2'' where ww2: "Rep_homo_coords w2 = (w2', w2'')" 
    by (rule obtain_homo_coords)
  obtain w3' w3'' where ww3: "Rep_homo_coords w3 = (w3', w3'')" 
    by (rule obtain_homo_coords)

  show "ratio_rep z1 z2 z3 \<approx> ratio_rep w1 w2 w3"
  proof (cases "\<not> z1 \<approx> z2 \<and> \<not> z3 \<approx> inf_homo_rep \<or> \<not> z1 \<approx> z3 \<and> \<not> z2 \<approx> inf_homo_rep")
    case True
    hence "\<not> w1 \<approx> w2 \<and> \<not> w3 \<approx> inf_homo_rep \<or> \<not> w1 \<approx> w3 \<and> \<not> w2 \<approx> inf_homo_rep"
      using * homo_coords_eq_sym homo_coords_eq_trans
      by metis
    thus ?thesis
      apply (subst homo_coords_eq_def, unfold Let_def)
      using ratio_rep_Rep[OF `\<not> z1 \<approx> z2 \<and> \<not> z3 \<approx> inf_homo_rep \<or> \<not> z1 \<approx> z3 \<and> \<not> z2 \<approx> inf_homo_rep`]
      using ratio_rep_Rep[OF `\<not> w1 \<approx> w2 \<and> \<not> w3 \<approx> inf_homo_rep \<or> \<not> w1 \<approx> w3 \<and> \<not> w2 \<approx> inf_homo_rep`]
      using zz1 zz2 zz3 ww1 ww2 ww3 *
      by (simp add: Let_def field_simps, (erule_tac exE)+) (rule_tac x="k*ka*kb" in exI, simp)
  next
    case False
    hence "\<not> (\<not> w1 \<approx> w2 \<and> \<not> w3 \<approx> inf_homo_rep \<or> \<not> w1 \<approx> w3 \<and> \<not> w2 \<approx> inf_homo_rep)"
      using * homo_coords_eq_sym homo_coords_eq_trans
      by metis
    thus ?thesis
      using False
      by (simp del: homo_coords_eq_def)
  qed
qed

lemma ratio_is_ratio:
  assumes "z1 \<noteq> z2 \<or> z1 \<noteq> z3" "z1 \<noteq> \<infinity>\<^sub>h" "z2 \<noteq> \<infinity>\<^sub>h \<or> z3 \<noteq> \<infinity>\<^sub>h"
  shows "ratio z1 z2 z3 = (z1 -\<^sub>h z2) :\<^sub>h (z1 -\<^sub>h z3)"
  unfolding minus_homo_def divide_homo_def
  using assms
proof transfer
  fix z w v
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)" 
    by (rule obtain_homo_coords)
  obtain w1 w2 where ww: "Rep_homo_coords w = (w1, w2)" 
    by (rule obtain_homo_coords)
  obtain v1 v2 where vv: "Rep_homo_coords v = (v1, v2)" 
    by (rule obtain_homo_coords)
  assume *: "\<not> z \<approx> w \<or> \<not> z \<approx> v" "\<not> z \<approx> inf_homo_rep"
            "\<not> w \<approx> inf_homo_rep \<or> \<not> v \<approx> inf_homo_rep"
  hence **: "\<not> z \<approx> w \<and> \<not> v \<approx> inf_homo_rep \<or> \<not> z \<approx> v \<and> \<not> w \<approx> inf_homo_rep"
    by (metis homo_coords_eq_trans)
  have "z2 \<noteq> 0" "w2 \<noteq> 0 \<or> v2 \<noteq> 0" "z1*w2 \<noteq> z2*w1 \<or> z1*v2 \<noteq> z2*v1"
    using zz vv ww not_inf_snd_not0[of v] not_inf_snd_not0[of z] not_inf_snd_not0[of w] homo_coords_eq_mix[of z z1 z2 w w1 w2]  homo_coords_eq_mix[of z z1 z2 v v1 v2] *
    by auto
  thus "ratio_rep z w v \<approx>
    z +\<^sub>h\<^sub>c uminus_homo_coords w *\<^sub>h\<^sub>c
    reciprocal_homo_coords (z +\<^sub>h\<^sub>c uminus_homo_coords v)"
    using zz ww vv **
    using divide_homo_coords_Rep[of "z +\<^sub>h\<^sub>c uminus_homo_coords w" "z1 * w2 + - w1 * z2" "z2 * w2" "(z +\<^sub>h\<^sub>c uminus_homo_coords v)" "z1 * v2 + - v1 * z2" "z2 * v2" ]
    using minus_homo_coords_Rep[of z z1 z2 w w1 w2]
    using minus_homo_coords_Rep[of z z1 z2 v v1 v2]
    by (auto simp add: field_simps)
qed

lemma
  assumes "z2 \<noteq> \<infinity>\<^sub>h" "z3 \<noteq> \<infinity>\<^sub>h"
  shows "ratio \<infinity>\<^sub>h z2 z3 = 1\<^sub>h"
using assms
proof transfer
  fix z2 z3
  obtain z2x z2y where zz2: "Rep_homo_coords z2 = (z2x, z2y)"
    by (rule obtain_homo_coords)
  obtain z3x z3y where zz3: "Rep_homo_coords z3 = (z3x, z3y)"
    by (rule obtain_homo_coords)
  assume "\<not> z2 \<approx> inf_homo_rep" "\<not> z3 \<approx> inf_homo_rep"
  have "z2y \<noteq> 0" "z3y \<noteq> 0"
    using not_inf_snd_not0[OF `\<not> z2 \<approx> inf_homo_rep`] zz2
    using not_inf_snd_not0[OF `\<not> z3 \<approx> inf_homo_rep`] zz3
    by auto
  thus "ratio_rep inf_homo_rep z2 z3 \<approx> one_homo_rep"
    using `\<not> z2 \<approx> inf_homo_rep` `\<not> z3 \<approx> inf_homo_rep` zz2 zz3
    by (subst homo_coords_eq_def, subst ratio_rep_Rep, simp_all) (rule_tac x="1/(z2y*z3y)" in exI, auto)
qed

(* Follows from ratio_is_ratio *)
lemma
  assumes "z1 \<noteq> \<infinity>\<^sub>h" "z3 \<noteq> \<infinity>\<^sub>h"
  shows "ratio z1 \<infinity>\<^sub>h z3 = \<infinity>\<^sub>h"
using assms
proof transfer
  fix z1 z3
  obtain z1x z1y where zz1: "Rep_homo_coords z1 = (z1x, z1y)"
    by (rule obtain_homo_coords)
  obtain z3x z3y where zz3: "Rep_homo_coords z3 = (z3x, z3y)"
    by (rule obtain_homo_coords)
  assume "\<not> z1 \<approx> inf_homo_rep" "\<not> z3 \<approx> inf_homo_rep"
  have "z1y \<noteq> 0" "z3y \<noteq> 0"
    using not_inf_snd_not0[OF `\<not> z1 \<approx> inf_homo_rep`] zz1
    using not_inf_snd_not0[OF `\<not> z3 \<approx> inf_homo_rep`] zz3
    by auto
  thus "ratio_rep z1 inf_homo_rep z3 \<approx> inf_homo_rep"
    using `\<not> z1 \<approx> inf_homo_rep` `\<not> z3 \<approx> inf_homo_rep` zz1 zz3
    by (subst homo_coords_eq_def, subst ratio_rep_Rep, simp_all) (rule_tac x="-1/(z1y*z3y)" in exI, auto)
qed

(* Follows from ratio_is_ratio *)
lemma
  assumes "z1 \<noteq> \<infinity>\<^sub>h" "z2 \<noteq> \<infinity>\<^sub>h"
  shows "ratio z1 z2 \<infinity>\<^sub>h = 0\<^sub>h"
using assms
proof transfer
  fix z1 z2
  obtain z1x z1y where zz1: "Rep_homo_coords z1 = (z1x, z1y)"
    by (rule obtain_homo_coords)
  obtain z2x z2y where zz2: "Rep_homo_coords z2 = (z2x, z2y)"
    by (rule obtain_homo_coords)
  assume "\<not> z1 \<approx> inf_homo_rep" "\<not> z2 \<approx> inf_homo_rep"
  have "z1y \<noteq> 0" "z2y \<noteq> 0"
    using not_inf_snd_not0[OF `\<not> z1 \<approx> inf_homo_rep`] zz1
    using not_inf_snd_not0[OF `\<not> z2 \<approx> inf_homo_rep`] zz2
    by auto
  thus "ratio_rep z1 z2 inf_homo_rep \<approx> zero_homo_rep"
    using `\<not> z1 \<approx> inf_homo_rep` `\<not> z2 \<approx> inf_homo_rep` zz1 zz2
    by (subst homo_coords_eq_def, subst ratio_rep_Rep, simp_all) (rule_tac x="-1/(z1y*z2y)" in exI, auto)
qed

(* Follows from ratio_is_ratio *)
lemma
  assumes "z1 \<noteq> z2" "z1 \<noteq> \<infinity>\<^sub>h"
  shows "ratio z1 z2 z1 = \<infinity>\<^sub>h"
proof-
  have "z1 -\<^sub>h z2 \<noteq> 0\<^sub>h"
    using diff_zero_homo[of z1 z2] `z1 \<noteq> z2` `z1 \<noteq> \<infinity>\<^sub>h`
    by auto
  thus ?thesis
    using assms
    using ratio_is_ratio[of z1 z2 z1]
    by simp
qed

(* -------------------------- *)

definition cross_ratio_rep where
  "cross_ratio_rep z u v w = 
     (let (z', z'') = Rep_homo_coords z; 
          (u', u'') = Rep_homo_coords u; 
          (v', v'') = Rep_homo_coords v;
          (w', w'') = Rep_homo_coords w 
       in Abs_homo_coords ((z'*u'' - u'*z'')*(v'*w'' - w'*v''), 
                           (z'*w'' - w'*z'')*(v'*u'' - u'*v'')))"

lemma cross_ratio_rep_Rep [simp]:
  assumes "(\<not> z1 \<approx> z2 \<and> \<not> z3 \<approx> z4) \<or> (\<not> z1 \<approx> z4 \<and> \<not> z2 \<approx> z3)"
  shows "Rep_homo_coords (cross_ratio_rep z1 z2 z3 z4) = 
     (let (z1', z1'') = Rep_homo_coords z1; 
          (z2', z2'') = Rep_homo_coords z2; 
          (z3', z3'') = Rep_homo_coords z3;
          (z4', z4'') = Rep_homo_coords z4 
       in ((z1'*z2'' - z2'*z1'')*(z3'*z4'' - z4'*z3''), (z1'*z4'' - z4'*z1'')*(z3'*z2'' - z2'*z3'')))"
proof-
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')" 
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')" 
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')" 
    by (rule obtain_homo_coords)
  obtain z4' z4'' where zz4: "Rep_homo_coords z4 = (z4', z4'')" 
    by (rule obtain_homo_coords)
  show ?thesis
    using zz1 zz2 zz3 zz4
    using assms
    unfolding cross_ratio_rep_def Let_def
    using homo_coords_eq_mix[OF zz1 zz2] homo_coords_eq_mix[OF zz3 zz4]
    using homo_coords_eq_mix[OF zz1 zz4] homo_coords_eq_mix[OF zz2 zz3]
    by (auto simp add: Abs_homo_coords_inverse)
qed

lift_definition cross_ratio :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo" is cross_ratio_rep
proof-
  fix z1 z2 z3 z4 w1 w2 w3 w4
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')" 
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')" 
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')" 
    by (rule obtain_homo_coords)
  obtain z4' z4'' where zz4: "Rep_homo_coords z4 = (z4', z4'')" 
    by (rule obtain_homo_coords)
  obtain w1' w1'' where ww1: "Rep_homo_coords w1 = (w1', w1'')" 
    by (rule obtain_homo_coords)
  obtain w2' w2'' where ww2: "Rep_homo_coords w2 = (w2', w2'')" 
    by (rule obtain_homo_coords)
  obtain w3' w3'' where ww3: "Rep_homo_coords w3 = (w3', w3'')" 
    by (rule obtain_homo_coords)
  obtain w4' w4'' where ww4: "Rep_homo_coords w4 = (w4', w4'')" 
    by (rule obtain_homo_coords)
  let ?w12 = "w1' * w2'' - w2' * w1''"
  let ?w34 = "w3' * w4'' - w4' * w3''"
  let ?w14 = "w1' * w4'' - w4' * w1''"
  let ?w32 = "w3' * w2'' - w2' * w3''"
  let ?z12 = "z1' * z2'' - z2' * z1''"
  let ?z34 = "z3' * z4'' - z4' * z3''"
  let ?z14 = "z1' * z4'' - z4' * z1''"
  let ?z32 = "z3' * z2'' - z2' * z3''"

  assume *: "z1 \<approx> w1" "z2 \<approx> w2" "z3 \<approx> w3" "z4 \<approx> w4"
  hence **: 
    "?w12 * ?w34 = 0 \<longleftrightarrow> ?z12 * ?z34 = 0" "?w14 * ?w32 = 0 \<longleftrightarrow> ?z14 * ?z32 = 0"
    using zz1 zz2 zz3 zz4 ww1 ww2 ww3 ww4
    by auto

  show "cross_ratio_rep z1 z2 z3 z4 \<approx> cross_ratio_rep w1 w2 w3 w4"
  proof (cases "?z12 * ?z34 = 0 \<and> ?z14 * ?z32 = 0")
    case True
    thus ?thesis
      using zz1 zz2 zz3 zz4 ww1 ww2 ww3 ww4 **
      by (simp add: cross_ratio_rep_def split_def Let_def) (rule_tac x="1" in exI, auto)
  next
    case False
    have "\<not> z1 \<approx> z2 \<and> \<not> z3 \<approx> z4 \<or> \<not> z1 \<approx> z4 \<and> \<not> z2 \<approx> z3"
      using False
      using homo_coords_eq_mix[OF zz1 zz2] homo_coords_eq_mix[OF zz3 zz4]
      using homo_coords_eq_mix[OF zz1 zz4] homo_coords_eq_mix[OF zz2 zz3]
      by (simp del: homo_coords_eq_def) metis
    moreover
    have "\<not> w1 \<approx> w2 \<and> \<not> w3 \<approx> w4 \<or> \<not> w1 \<approx> w4 \<and> \<not> w2 \<approx> w3"
      using ** False
      using homo_coords_eq_mix[OF ww1 ww2] homo_coords_eq_mix[OF ww3 ww4]
      using homo_coords_eq_mix[OF ww1 ww4] homo_coords_eq_mix[OF ww2 ww3]
      by (simp del: homo_coords_eq_def) metis
    ultimately
    show ?thesis
      using *
      using cross_ratio_rep_Rep[of z1 z2 z3 z4] 
      using cross_ratio_rep_Rep[of w1 w2 w3 w4]
      using zz1 zz2 zz3 zz4 ww1 ww2 ww3 ww4
      apply simp
      apply (erule exE)+
      apply simp
      apply (rule_tac x="k*ka*kb*kc" in exI)
      apply (simp add: field_simps)
      done
  qed
qed

lemma "cross_ratio z 0\<^sub>h 1\<^sub>h \<infinity>\<^sub>h = z"
proof (transfer)
  fix z
  have *: "\<not> z \<approx> zero_homo_rep \<and> \<not> one_homo_rep \<approx> inf_homo_rep \<or> \<not> z \<approx> inf_homo_rep \<and> \<not> zero_homo_rep \<approx> one_homo_rep"
    by (cases "Rep_homo_coords z") auto
  show "cross_ratio_rep z zero_homo_rep one_homo_rep inf_homo_rep \<approx> z"
    using cross_ratio_rep_Rep[OF *]
    by (simp add: split_def Let_def) (rule_tac x="-1" in exI, simp)
qed

lemma cross_ratio_0:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3"
  shows "cross_ratio z1 z1 z2 z3 = 0\<^sub>h"
using assms
proof (transfer)
  fix z1 z2 z3
  let ?z1 = "Rep_homo_coords z1" and ?z2 = "Rep_homo_coords z2" and ?z3 = "Rep_homo_coords z3"
  assume "\<not> z1 \<approx> z2" "\<not> z1 \<approx> z3"
  thus "cross_ratio_rep z1 z1 z2 z3 \<approx> zero_homo_rep"
    using cross_ratio_rep_Rep[of z1 z1 z2 z3]
      homo_coords_eq_mix[of z1 "fst ?z1" "snd ?z1" z2 "fst ?z2" "snd ?z2"]  homo_coords_eq_mix[of z1 "fst ?z1" "snd ?z1" z3 "fst ?z3" "snd ?z3"]
    by (cases ?z1, cases ?z2, cases ?z3, simp add: split_def Let_def) 
qed

lemma cross_ratio_1:
  assumes "z1 \<noteq> z2" "z2 \<noteq> z3"
  shows "cross_ratio z2 z1 z2 z3 = 1\<^sub>h"
using assms
proof (transfer)
  fix z1 z2 z3
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')"
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')"
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')"
    by (rule obtain_homo_coords)
  assume "\<not> z1 \<approx> z2" "\<not> z2 \<approx> z3"
  thus "cross_ratio_rep z2 z1 z2 z3 \<approx> one_homo_rep"
    using zz1 zz2 zz3
    using homo_coords_eq_mix[of z1 z1' z1'' z2 z2' z2'']  homo_coords_eq_mix[of z2 z2' z2'' z3 z3' z3'']
    by (auto simp add: cross_ratio_rep_def split_def Let_def Abs_homo_coords_inverse) (rule_tac x="1 / ((z2' * z3'' - z3' * z2'') * (z2' * z1'' - z1' * z2''))" in exI, simp)
qed

lemma cross_ratio_inf:
  assumes "z1 \<noteq> z3" "z2 \<noteq> z3"
  shows "cross_ratio z3 z1 z2 z3 = \<infinity>\<^sub>h"
using assms
proof (transfer)
  fix z1 z2 z3
  obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')"
    by (rule obtain_homo_coords)
  obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')"
    by (rule obtain_homo_coords)
  obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')"
    by (rule obtain_homo_coords)
  assume "\<not> z1 \<approx> z3" "\<not> z2 \<approx> z3"
  thus "cross_ratio_rep z3 z1 z2 z3 \<approx> inf_homo_rep"
    using zz1 zz2 zz3
    using homo_coords_eq_mix[of z1 z1' z1'' z3 z3' z3''] homo_coords_eq_mix[of z2 z2' z2'' z3 z3' z3'']
    by (auto simp add: cross_ratio_rep_def split_def Let_def Abs_homo_coords_inverse)
qed

lemma
  assumes "(z \<noteq> u \<and> v \<noteq> w) \<or> (z \<noteq> w \<and> u \<noteq> v)" "z \<noteq> \<infinity>\<^sub>h"  "u \<noteq> \<infinity>\<^sub>h"  "v \<noteq> \<infinity>\<^sub>h" "w \<noteq> \<infinity>\<^sub>h"
  shows "cross_ratio z u v w = ((z-\<^sub>hu) *\<^sub>h (v-\<^sub>hw)) :\<^sub>h ((z-\<^sub>hw) *\<^sub>h (v-\<^sub>hu))"
using assms
unfolding minus_homo_def divide_homo_def
proof transfer
  fix z u v w
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)" 
    by (rule obtain_homo_coords)
  obtain u1 u2 where uu: "Rep_homo_coords u = (u1, u2)" 
    by (rule obtain_homo_coords)
  obtain v1 v2 where vv: "Rep_homo_coords v = (v1, v2)" 
    by (rule obtain_homo_coords)
  obtain w1 w2 where ww: "Rep_homo_coords w = (w1, w2)" 
    by (rule obtain_homo_coords)

  assume *: "\<not> z \<approx> u \<and> \<not> v \<approx> w \<or> \<not> z \<approx> w \<and> \<not> u \<approx> v" and
         **: "\<not> z \<approx> inf_homo_rep" "\<not> u \<approx> inf_homo_rep" "\<not> v \<approx> inf_homo_rep" "\<not> w \<approx> inf_homo_rep"
  have "z2 \<noteq> 0" "u2 \<noteq> 0" "v2 \<noteq> 0" "w2 \<noteq> 0"
    using ** zz uu vv ww
    using not_inf_snd_not0[of z] not_inf_snd_not0[of u] not_inf_snd_not0[of v] not_inf_snd_not0[of w]
    by simp_all
  moreover
  have "((z1*u2 - z2*u1 \<noteq> 0) \<and> (v1*w2 - v2*w1 \<noteq> 0)) \<or> ((z1*w2 - z2*w1 \<noteq> 0) \<and> (v1*u2 - v2*u1 \<noteq> 0))"
    using *
    apply (subst (asm) homo_coords_eq_mix[OF zz uu])
    apply (subst (asm) homo_coords_eq_mix[OF vv ww])
    apply (subst (asm) homo_coords_eq_mix[OF zz ww])
    apply (subst (asm) homo_coords_eq_mix[OF uu vv])
    by (auto simp add: field_simps)
  moreover 
  hence "z1 * w2 \<noteq> w1 * z2 \<and> v1 * u2 \<noteq> u1 * v2 \<or> z1 * u2 \<noteq> u1 * z2 \<and> v1 * w2 \<noteq> w1 * v2"
    by auto
  ultimately
  show "cross_ratio_rep z u v w \<approx>
       z +\<^sub>h\<^sub>c uminus_homo_coords u *\<^sub>h\<^sub>c (v +\<^sub>h\<^sub>c uminus_homo_coords w) *\<^sub>h\<^sub>c
       reciprocal_homo_coords
        (z +\<^sub>h\<^sub>c uminus_homo_coords w *\<^sub>h\<^sub>c (v +\<^sub>h\<^sub>c uminus_homo_coords u))"
    using uu vv ww zz *
    apply simp
    apply (subst divide_homo_coords_Rep[of "(z +\<^sub>h\<^sub>c uminus_homo_coords u) *\<^sub>h\<^sub>c (v +\<^sub>h\<^sub>c uminus_homo_coords w)" "(z1 * u2 - u1 * z2) * (v1 * w2 - w1 * v2)"  "z2 * u2 * (v2 * w2)" "(z +\<^sub>h\<^sub>c uminus_homo_coords w) *\<^sub>h\<^sub>c  (v +\<^sub>h\<^sub>c uminus_homo_coords u)" "(z1 * w2 - w1 * z2) * (v1 * u2 - u1 * v2)"
            "z2 * w2 * (v2 * u2)"])
    using mult_homo_coords_Rep[of "z +\<^sub>h\<^sub>c uminus_homo_coords u" "z1 * u2 - u1 * z2" "z2 * u2" "v +\<^sub>h\<^sub>c uminus_homo_coords w" "v1 * w2 - w1 * v2" "v2 * w2"]
    using minus_homo_coords_Rep[of z z1 z2 u u1 u2]
    using minus_homo_coords_Rep[of v v1 v2 w w1 w2]
    using mult_homo_coords_Rep[of "z +\<^sub>h\<^sub>c uminus_homo_coords w" "z1 * w2 - w1 * z2" "z2 * w2" "v +\<^sub>h\<^sub>c uminus_homo_coords u" "v1 * u2 - u1 * v2" "v2 * u2"]
    using minus_homo_coords_Rep[of z z1 z2 w w1 w2]
    using minus_homo_coords_Rep[of v v1 v2 u u1 u2]
    using mult_homo_coords_Rep[of "z +\<^sub>h\<^sub>c uminus_homo_coords u" "z1 * u2 - u1 * z2" "z2 * u2" "v +\<^sub>h\<^sub>c uminus_homo_coords w" "v1 * w2 - w1 * v2" "v2 * w2"]
    using minus_homo_coords_Rep[of z z1 z2 u u1 u2]
    using minus_homo_coords_Rep[of v v1 v2 w w1 w2]
    by simp_all (rule_tac x="z2*u2*(v2*w2)" in exI, simp)
qed


subsection{* Distance *}

definition inprod_homo_rep where 
 "inprod_homo_rep z w = 
    (let (z1, z2) = Rep_homo_coords z;
         (w1, w2) = Rep_homo_coords w 
      in vec_cnj (z1, z2) *\<^sub>v\<^sub>v (w1, w2))"
syntax
  "_inprod_homo_rep" :: "homo_coords \<Rightarrow> homo_coords \<Rightarrow> complex"  ("\<langle>_,_\<rangle>")
translations
  "\<langle>z,w\<rangle>" == "CONST inprod_homo_rep z w"

lemma [simp]: "is_real \<langle>z,z\<rangle>"
  unfolding inprod_homo_rep_def
  by (cases "Rep_homo_coords z", simp add: vec_cnj_def)

lemma [simp]: "Re \<langle>z,z\<rangle> \<ge> 0"
  unfolding inprod_homo_rep_def
  by (cases "Rep_homo_coords z", simp add: vec_cnj_def)

lemma inprod_homo_bilinear1:
  assumes "Rep_homo_coords z' = k *\<^sub>s\<^sub>v Rep_homo_coords z"
  shows "\<langle>z',w\<rangle> = cnj k * \<langle>z,w\<rangle>" 
using assms
unfolding inprod_homo_rep_def Let_def
by (cases "Rep_homo_coords z", cases "Rep_homo_coords z'", cases "Rep_homo_coords w") (simp add: vec_cnj_def complex_cnj field_simps)

lemma inprod_homo_bilinear2:
  assumes "Rep_homo_coords w' = k *\<^sub>s\<^sub>v Rep_homo_coords w"
  shows "\<langle>z,w'\<rangle> = k * \<langle>z,w\<rangle>" 
using assms
unfolding inprod_homo_rep_def Let_def
by (cases "Rep_homo_coords z", cases "Rep_homo_coords z'", cases "Rep_homo_coords w") (simp add: vec_cnj_def complex_cnj field_simps)

definition norm_homo_rep where
  "norm_homo_rep z = sqrt (Re \<langle>z,z\<rangle>)"
syntax
  "_norm_homo_rep" :: "homo_coords \<Rightarrow> complex"  ("\<langle>_\<rangle>")
translations
  "\<langle>z\<rangle>" == "CONST norm_homo_rep z"

lemma
  norm_homo_rep_square: "\<langle>z\<rangle>\<^sup>2 = Re (\<langle>z,z\<rangle>)"
unfolding norm_homo_rep_def
by simp

lemma norm_homo_gt_0: "\<langle>z\<rangle> > 0"
proof-
  obtain z1 z2 where "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  thus ?thesis
    using complex_mult_cnj_cmod[of z1] complex_mult_cnj_cmod[of z2] Rep_homo_coords[of z]
    unfolding norm_homo_rep_def inprod_homo_rep_def
    by (simp add: vec_cnj_def split_def Let_def field_simps power2_eq_square) (metis norm_eq_zero sum_squares_gt_zero_iff)
qed

lemma norm_homo_scale:
  assumes "Rep_homo_coords z' = k *\<^sub>s\<^sub>v Rep_homo_coords z"
  shows "\<langle>z'\<rangle>\<^sup>2 = Re (cnj k * k) * \<langle>z\<rangle>\<^sup>2"
apply (subst norm_homo_rep_square)+
apply (subst inprod_homo_bilinear1[OF assms])
apply (subst inprod_homo_bilinear2[OF assms])
apply (simp add: field_simps)
done

definition dist_homo_rep where 
  "dist_homo_rep z1 z2 = 
     (let (z1x, z1y) = Rep_homo_coords z1; 
          (z2x, z2y) = Rep_homo_coords z2;
          num = (z1x*z2y - z2x*z1y) * (cnj z1x*cnj z2y - cnj z2x*cnj z1y);
          den = (z1x*cnj z1x + z1y*cnj z1y) * (z2x*cnj z2x + z2y*cnj z2y)
       in 2*sqrt(Re num / Re den))"

lemma dist_homo_rep_iff: "dist_homo_rep z w = 2*sqrt(1 - (cmod \<langle>z,w\<rangle>)\<^sup>2 / (\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2))"
proof-
  obtain z1 z2 w1 w2 where *: "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)"
    by (cases "Rep_homo_coords z", cases "Rep_homo_coords w") auto
  have 1: "2*sqrt(1 - (cmod \<langle>z,w\<rangle>)\<^sup>2 / (\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2)) = 2*sqrt((\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2 - (cmod \<langle>z,w\<rangle>)\<^sup>2) / (\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2))"
    using norm_homo_gt_0[of z] norm_homo_gt_0[of w]
    by (simp add: field_simps)

  have 2: "\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2 = Re ((z1*cnj z1 + z2*cnj z2) * (w1*cnj w1 + w2*cnj w2))"
    using *
    by (simp add: norm_homo_rep_def inprod_homo_rep_def vec_cnj_def)

  have 3: "\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2 - (cmod \<langle>z,w\<rangle>)\<^sup>2 = Re ((z1*w2 - w1*z2) * (cnj z1*cnj w2 - cnj w1*cnj z2))"
    apply (subst cmod_square, (subst norm_homo_rep_square)+)
    using *
    by (simp add: inprod_homo_rep_def vec_cnj_def field_simps)

  thus ?thesis
    using 1 2 3
    using *
    unfolding dist_homo_rep_def Let_def
    by simp
qed
    
lift_definition dist_homo :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> real" is dist_homo_rep
proof-
  fix z1 z2 z1' z2'
  obtain z1x z1y z2x z2y z1'x z1'y z2'x z2'y where
    zz: "Rep_homo_coords z1 = (z1x, z1y)" "Rep_homo_coords z2 = (z2x, z2y)" "Rep_homo_coords z1' = (z1'x, z1'y)" "Rep_homo_coords z2' = (z2'x, z2'y)"
    by (cases "Rep_homo_coords z1", cases "Rep_homo_coords z2", cases "Rep_homo_coords z1'", cases "Rep_homo_coords z2'") blast
  
  assume "z1 \<approx> z1'" "z2 \<approx> z2'"
  then obtain k1 k2 where
    *: "k1 \<noteq> 0" "Rep_homo_coords z1' = k1 *\<^sub>s\<^sub>v Rep_homo_coords z1" and
    **: "k2 \<noteq> 0" "Rep_homo_coords z2' = k2 *\<^sub>s\<^sub>v Rep_homo_coords z2"
    by auto
  have "(cmod \<langle>z1,z2\<rangle>)\<^sup>2 / (\<langle>z1\<rangle>\<^sup>2 * \<langle>z2\<rangle>\<^sup>2) = (cmod \<langle>z1',z2'\<rangle>)\<^sup>2 / (\<langle>z1'\<rangle>\<^sup>2 * \<langle>z2'\<rangle>\<^sup>2)"
    using `k1 \<noteq> 0` `k2 \<noteq> 0`
    using cmod_square[symmetric, of k1] cmod_square[symmetric, of k2]
    apply (subst norm_homo_scale[OF *(2)])
    apply (subst norm_homo_scale[OF **(2)])
    apply (subst inprod_homo_bilinear1[OF *(2)])
    apply (subst inprod_homo_bilinear2[OF **(2)])
    by (simp add: power2_eq_square)
  thus "dist_homo_rep z1 z2 = dist_homo_rep z1' z2'"
    by (subst dist_homo_rep_iff)+ simp
qed

lemma dist_homo_finite:
  "dist_homo (of_complex z1) (of_complex z2) = 2 * cmod(z1 - z2) / (sqrt (1+(cmod z1)\<^sup>2) * sqrt (1+(cmod z2)\<^sup>2))"
apply transfer
apply (subst cmod_square)+
apply (simp add: dist_homo_rep_def real_sqrt_divide cmod_def power2_eq_square)
by (smt ab_diff_minus comm_semiring_1_class.normalizing_semiring_rules(24) minus_diff_eq minus_mult_right real_sqrt_mult_distrib2)

lemma dist_homo_infinite1:
  "dist_homo (of_complex z1) \<infinity>\<^sub>h = 2 / sqrt (1+(cmod z1)\<^sup>2)"
by transfer (subst cmod_square, simp add: dist_homo_rep_def real_sqrt_divide)

lemma dist_homo_infinite2:
  "dist_homo \<infinity>\<^sub>h (of_complex z1) = 2 / sqrt (1+(cmod z1)\<^sup>2)"
by transfer (subst cmod_square, simp add: dist_homo_rep_def real_sqrt_divide)

lemma dist_homo_rep_zero: 
  "dist_homo_rep z w = 0 \<longleftrightarrow> (cmod \<langle>z,w\<rangle>)\<^sup>2 = (\<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2)"
using norm_homo_gt_0[of z]  norm_homo_gt_0[of w]
by (subst dist_homo_rep_iff) auto

lemma dist_homo_zero1 [simp]: "dist_homo z z = 0"
by transfer (subst dist_homo_rep_zero, ((subst norm_homo_rep_square)+), subst cmod_square, simp)

lemma dist_homo_zero2 [simp]: 
  assumes "dist_homo z1 z2 = 0"
  shows "z1 = z2"
using assms
proof transfer
  fix z w
  obtain z1 z2 w1 w2 where *: "Rep_homo_coords z = (z1, z2)" "Rep_homo_coords w = (w1, w2)"
    by (cases "Rep_homo_coords z", cases "Rep_homo_coords w", auto)
  let ?x = "(z1*w2 - w1*z2) * (cnj z1*cnj w2 - cnj w1*cnj z2)"
  assume "dist_homo_rep z w = 0"
  hence "(cmod \<langle>z,w\<rangle>)\<^sup>2 = \<langle>z\<rangle>\<^sup>2 * \<langle>w\<rangle>\<^sup>2"
    by (subst (asm) dist_homo_rep_zero)
  hence "Re ?x = 0"
    using *
    by (subst (asm) cmod_square) ((subst (asm) norm_homo_rep_square)+, simp add: inprod_homo_rep_def vec_cnj_def field_simps)
  hence "?x = 0"
    using complex_mult_cnj_cmod[of "z1*w2 - w1*z2"]
    by (subst complex_eq_if_Re_eq[of ?x 0]) (simp add: complex_cnj power2_eq_square, auto)
  thus "z \<approx> w"
    using homo_coords_eq_mix[OF *]
    by (auto simp del: homo_coords_eq_def) (metis complex_cnj_cnj complex_cnj_mult)
qed

lemma dist_homo_sym [simp]: 
  shows "dist_homo z1 z2 = dist_homo z2 z1"
by transfer (simp add: dist_homo_rep_def split_def Let_def field_simps)

text{* Triangle inequality *}
lemma dist_homo_triangle_finite: "cmod(a - b) / (sqrt (1+(cmod a)\<^sup>2) * sqrt (1+(cmod b)\<^sup>2)) \<le> cmod (a - c) / (sqrt (1+(cmod a)\<^sup>2) * sqrt (1+(cmod c)\<^sup>2)) + cmod (c - b) / (sqrt (1+(cmod b)\<^sup>2) * sqrt (1+(cmod c)\<^sup>2))"
proof-
  let ?cc = "1+(cmod c)\<^sup>2" and ?bb = "1+(cmod b)\<^sup>2" and ?aa = "1+(cmod a)\<^sup>2"
  have "sqrt ?cc > 0" "sqrt ?aa > 0" "sqrt ?bb > 0"
    by (auto simp add: power2_eq_square) (metis add_strict_increasing norm_ge_zero norm_mult zero_less_one)+

  have "(a - b)*(1+cnj c*c) = (a-c)*(1+cnj c*b) + (c-b)*(1 + cnj c*a)"
    by (simp add: field_simps)
  moreover
  have "cmod ((a - b)*(1+cnj c*c)) = cmod(a - b) * (1+(cmod c)\<^sup>2)"
    using complex_mult_cnj_cmod[of "cnj c"]
    by (auto simp add: power2_eq_square) (metis abs_add_abs abs_one abs_power2 norm_of_real of_real_1 of_real_add of_real_mult power2_eq_square)
  ultimately
  have "cmod(a - b) * (1+(cmod c)\<^sup>2) \<le> cmod (a-c) * cmod (1+cnj c*b) + cmod (c-b) * cmod(1 + cnj c*a)"
    using complex_mod_triangle_ineq2[of "(a-c)*(1+cnj c*b)" "(c-b)*(1 + cnj c*a)"]
    by simp
  moreover
  have *: "\<And> a b c d b' d'. \<lbrakk>b \<le> b'; d \<le> d'; a \<ge> (0::real); c \<ge> 0\<rbrakk> \<Longrightarrow> a*b + c*d \<le> a*b' + c*d'"
    by (metis add_mono comm_mult_left_mono)
  have "cmod (a-c) * cmod (1+cnj c*b) + cmod (c-b) * cmod(1 + cnj c*a) \<le> cmod (a - c) * (sqrt (1+(cmod c)\<^sup>2) * sqrt (1+(cmod b)\<^sup>2)) + cmod (c - b) * (sqrt (1+(cmod c)\<^sup>2) * sqrt (1+(cmod a)\<^sup>2))"
    using *[OF cmod_1_plus_mult_le[of "cnj c" b] cmod_1_plus_mult_le[of "cnj c" a], of "cmod (a-c)" "cmod (c-b)"]
    by (simp add: field_simps real_sqrt_mult[symmetric])
  ultimately
  have "cmod(a - b) * ?cc \<le> cmod (a - c) * sqrt ?cc * sqrt ?bb + cmod (c - b) * sqrt ?cc * sqrt ?aa"
    by simp
  moreover
  hence "0 \<le> ?cc * sqrt ?aa * sqrt ?bb"
    using mult_right_mono[of 0 "sqrt ?aa"  "sqrt ?bb"]
    using mult_right_mono[of 0 "?cc" "sqrt ?aa * sqrt ?bb"]
    by simp
  moreover
  have "sqrt ?cc / ?cc = 1 / sqrt ?cc"
    using `sqrt ?cc > 0`
    by (simp add: field_simps) (metis abs_of_pos real_sqrt_abs2 real_sqrt_mult_distrib2)
  hence "sqrt ?cc / (?cc * sqrt ?aa) = 1 / (sqrt ?aa * sqrt ?cc)"
    using times_divide_eq_right[of "1/sqrt ?aa" "sqrt ?cc" "?cc"]
    using `sqrt ?aa > 0`
    by simp
  hence "cmod (a - c) * sqrt ?cc / (?cc * sqrt ?aa) = cmod (a - c) / (sqrt ?aa * sqrt ?cc)"
    using times_divide_eq_right[of "cmod (a - c)" "sqrt ?cc" "(?cc * sqrt ?aa)"]
    by simp
  moreover
  have "sqrt ?cc / ?cc = 1 / sqrt ?cc"
    using `sqrt ?cc > 0`
    by (simp add: field_simps) (metis abs_of_pos real_sqrt_abs2 real_sqrt_mult_distrib2)
  hence "sqrt ?cc / (?cc * sqrt ?bb) = 1 / (sqrt ?bb * sqrt ?cc)"
    using times_divide_eq_right[of "1/sqrt ?bb" "sqrt ?cc" "?cc"]
    using `sqrt ?bb > 0`
    by simp
  hence "cmod (c - b) * sqrt ?cc / (?cc * sqrt ?bb) = cmod (c - b) / (sqrt ?bb * sqrt ?cc)"
    using times_divide_eq_right[of "cmod (c - b)" "sqrt ?cc" "?cc * sqrt ?bb"]
    by simp
  ultimately
  show ?thesis
    using divide_right_mono[of "cmod (a - b) * ?cc" "cmod (a - c) * sqrt ?cc * sqrt ?bb + cmod (c - b) * sqrt ?cc * sqrt ?aa" "?cc * sqrt ?aa * sqrt ?bb"] `sqrt ?aa > 0` `sqrt ?bb > 0` `sqrt ?cc > 0`
    by (simp add: add_divide_distrib)
qed

lemma dist_homo_triangle_infinite1: "1 / sqrt(1 + (cmod b)\<^sup>2) \<le> 1 / sqrt(1 + (cmod c)\<^sup>2) + cmod (b - c) / (sqrt(1 + (cmod b)\<^sup>2) * sqrt(1 + (cmod c)\<^sup>2))"
proof-
  let ?bb = "sqrt (1 + (cmod b)\<^sup>2)" and ?cc = "sqrt (1 + (cmod c)\<^sup>2)"
  have "?bb > 0" "?cc > 0"
    by (metis add_strict_increasing real_sqrt_gt_0_iff zero_le_power2 zero_less_one)+
  hence *: "?bb * ?cc \<ge> 0"
    by (metis one_power2 real_sqrt_mult_distrib2 real_sqrt_sum_squares_mult_ge_zero)
  have **: "(?cc - ?bb) / (?bb * ?cc) = 1 / ?bb - 1 / ?cc"
    using `sqrt (1 + (cmod b)\<^sup>2) > 0`  `sqrt (1 + (cmod c)\<^sup>2) > 0`
    by (simp add: field_simps)
  show "1 / ?bb \<le> 1 / ?cc + cmod (b - c) / (?bb * ?cc)"
    using divide_right_mono[OF cmod_diff_ge[of c b] *]
    by (subst (asm) **) (simp add: field_simps norm_minus_commute)
qed

lemma dist_homo_triangle_infinite2: 
  "1 / sqrt(1 + (cmod a)\<^sup>2) \<le> cmod (a - c) / (sqrt (1+(cmod a)\<^sup>2) * sqrt (1+(cmod c)\<^sup>2)) + 1 / sqrt(1 + (cmod c)\<^sup>2)"
using dist_homo_triangle_infinite1[of a c]
by simp

lemma dist_homo_triangle_infinite3:
  "cmod(a - b) / (sqrt (1+(cmod a)\<^sup>2) * sqrt (1+(cmod b)\<^sup>2)) \<le> 1 / sqrt(1 + (cmod a)\<^sup>2) + 1 / sqrt(1 + (cmod b)\<^sup>2)"
proof-
  let ?aa = "sqrt (1 + (cmod a)\<^sup>2)" and ?bb = "sqrt (1 + (cmod b)\<^sup>2)"
  have "?aa > 0" "?bb > 0"
    by (metis add_strict_increasing real_sqrt_gt_0_iff zero_le_power2 zero_less_one)+
  hence *: "?aa * ?bb \<ge> 0"
    by (metis one_power2 real_sqrt_mult_distrib2 real_sqrt_sum_squares_mult_ge_zero)
  have **: "(?aa + ?bb) / (?aa * ?bb) = 1 / ?aa + 1 / ?bb"
    using `?aa > 0` `?bb > 0`
    by (simp add: field_simps)
  show "cmod (a - b) / (?aa * ?bb) \<le> 1 / ?aa + 1 / ?bb"
    using divide_right_mono[OF cmod_diff_le[of a b] *]
    by (subst (asm) **) (simp add: field_simps norm_minus_commute)
qed

lemma dist_homo_triangle:
  shows "dist_homo A B \<le> dist_homo A C + dist_homo C B"
proof (cases "A = \<infinity>\<^sub>h")
  case True
  show ?thesis
  proof (cases "B = \<infinity>\<^sub>h")
    case True
    show ?thesis
    proof (cases "C = \<infinity>\<^sub>h")
      case True
      show ?thesis
        using `A = \<infinity>\<^sub>h` `B = \<infinity>\<^sub>h` `C = \<infinity>\<^sub>h`
        by simp
    next
      case False
      then obtain c where "C = of_complex c"
        using inf_homo_or_complex_homo[of C]
        by auto
      show ?thesis
        using `A = \<infinity>\<^sub>h` `B = \<infinity>\<^sub>h` `C = of_complex c`
        by (simp add: dist_homo_infinite2)
    qed
  next
    case False
    then obtain b where "B = of_complex b"
      using inf_homo_or_complex_homo[of B]
      by auto
    show ?thesis
    proof (cases "C = \<infinity>\<^sub>h")
      case True
      show ?thesis
        using `A = \<infinity>\<^sub>h` `C = \<infinity>\<^sub>h` `B = of_complex b`
        by simp
    next
      case False
      then obtain c where "C = of_complex c"
        using inf_homo_or_complex_homo[of C]
        by auto
      show ?thesis
        using `A = \<infinity>\<^sub>h` `B = of_complex b` `C = of_complex c`
        using mult_left_mono[OF dist_homo_triangle_infinite1[of b c], of 2]
        by (simp add: dist_homo_finite dist_homo_infinite1 dist_homo_infinite2)
    qed
  qed
next
  case False
  then obtain a where "A = of_complex a"
    using inf_homo_or_complex_homo[of A]
    by auto
  show ?thesis
  proof (cases "B = \<infinity>\<^sub>h")
    case True
    show ?thesis
    proof (cases "C = \<infinity>\<^sub>h")
      case True
      show ?thesis
        using `B = \<infinity>\<^sub>h` `C = \<infinity>\<^sub>h` `A = of_complex a`
        by (simp add: dist_homo_infinite2)
    next
      case False
      then obtain c where "C = of_complex c"
        using inf_homo_or_complex_homo[of C]
        by auto
      show ?thesis
        using `B = \<infinity>\<^sub>h` `C = of_complex c` `A = of_complex a`
        using mult_left_mono[OF dist_homo_triangle_infinite2[of a c], of 2]
        by (simp add: dist_homo_finite dist_homo_infinite1 dist_homo_infinite2)
    qed
  next
    case False
    then obtain b where "B = of_complex b"
      using inf_homo_or_complex_homo[of B]
      by auto
    show ?thesis
    proof (cases "C = \<infinity>\<^sub>h")
      case True
      thus ?thesis
        using `C = \<infinity>\<^sub>h` `B = of_complex b` `A = of_complex a`
        using mult_left_mono[OF dist_homo_triangle_infinite3[of a b], of 2]
        by (simp add: dist_homo_finite dist_homo_infinite1 dist_homo_infinite2)
    next
      case False
      then obtain c where "C = of_complex c"
        using inf_homo_or_complex_homo[of C]
        by auto
      show ?thesis
        using `A = of_complex a` `B = of_complex b` `C = of_complex c`
        using mult_left_mono[OF dist_homo_triangle_finite[of a b c], of 2]
        by (simp add: dist_homo_finite norm_minus_commute)
    qed
  qed
qed

instantiation complex_homo :: metric_space
begin
definition "dist_complex_homo = dist_homo"
definition "open_complex_homo S = (\<forall>x\<in>S. \<exists>e>0. \<forall>y. dist_homo y x < e \<longrightarrow> y \<in> S)"
instance
proof
  fix x y :: complex_homo
  show "(dist x y = 0) = (x = y)"
    unfolding dist_complex_homo_def
    using dist_homo_zero1[of x] dist_homo_zero2[of x y]
    by auto
next
  fix S :: "complex_homo set"
  show "open S = (\<forall>x\<in>S. \<exists>e>0. \<forall>y. dist y x < e \<longrightarrow> y \<in> S)"
    unfolding open_complex_homo_def dist_complex_homo_def
    by simp
next
  fix x y z :: complex_homo
  show "dist x y \<le> dist x z + dist y z"
    unfolding dist_complex_homo_def
    using dist_homo_triangle[of x y z]
    by simp
qed
end

end
