header{* Moebius transformations *}
theory Moebius
imports HomogeneousCoordinates
begin

typedef moebius_mat = "{M::complex_mat. mat_det M \<noteq> 0}"
by (rule_tac x="eye" in exI, simp)

definition moebius_mat_eq where 
  [simp]: "moebius_mat_eq A B \<longleftrightarrow> (\<exists> k::complex. k \<noteq> 0 \<and> Rep_moebius_mat B = k *\<^sub>s\<^sub>m (Rep_moebius_mat A))"

lemma [simp]: "moebius_mat_eq x x"
by (simp, rule_tac x=1 in exI, simp)

quotient_type moebius = moebius_mat / moebius_mat_eq
proof (rule equivpI)
  show "reflp moebius_mat_eq"
    by (auto simp add: reflp_def, rule_tac x="1" in exI, simp)
next
  show "symp moebius_mat_eq"
    by (auto simp add: symp_def, rule_tac x="1/k" in exI, simp)
next
  show "transp moebius_mat_eq"
    by (auto simp add: transp_def, rule_tac x="ka*k" in exI, simp)
qed

definition mk_moebius_rep where
  "mk_moebius_rep a b c d = Abs_moebius_mat (a, b, c, d)"

lift_definition mk_moebius :: "complex \<Rightarrow> complex \<Rightarrow> complex \<Rightarrow> complex \<Rightarrow> moebius" is mk_moebius_rep
by (simp del: moebius_mat_eq_def)

lemma mk_moebius_rep_Rep:
  assumes "mat_det (a, b, c, d) \<noteq> 0"
  shows "Rep_moebius_mat (mk_moebius_rep a b c d) = (a, b, c, d)"
using assms
by (simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)

lemma ex_mk_moebius:
  shows "\<exists> a b c d. M = mk_moebius a b c d \<and> mat_det (a, b, c, d) \<noteq> 0"
proof transfer
  fix M
  obtain a b c d where "Rep_moebius_mat M = (a, b, c, d)"
    by (cases "Rep_moebius_mat M") auto
  hence "moebius_mat_eq M (mk_moebius_rep a b c d) \<and> mat_det (a, b, c, d) \<noteq> 0"
    using Rep_moebius_mat[of M]
    by (simp add: mk_moebius_rep_Rep, rule_tac x=1 in exI, simp)
  thus "\<exists>a b c d. moebius_mat_eq M (mk_moebius_rep a b c d) \<and> mat_det (a, b, c, d) \<noteq> 0"
    by blast
qed

subsection{* Action on points *}
definition moebius_pt_rep :: "moebius_mat \<Rightarrow> homo_coords \<Rightarrow> homo_coords"  where 
"moebius_pt_rep M z = 
     (let z = Rep_homo_coords z;
          M = Rep_moebius_mat M
      in Abs_homo_coords (M *\<^sub>m\<^sub>v z))"

lemma [simp]: "Rep_homo_coords (Abs_homo_coords (Rep_moebius_mat M *\<^sub>m\<^sub>v Rep_homo_coords x)) = Rep_moebius_mat M *\<^sub>m\<^sub>v Rep_homo_coords x"
  using Rep_moebius_mat[of M] Rep_homo_coords[of x] mult_mv_nonzero[of "Rep_homo_coords x" "Rep_moebius_mat M"]
by (simp add: Abs_homo_coords_inverse)

lemma [simp]: "Rep_homo_coords (moebius_pt_rep M z) = Rep_moebius_mat M *\<^sub>m\<^sub>v Rep_homo_coords z"
by (simp add: moebius_pt_rep_def)

lift_definition moebius_pt :: "moebius \<Rightarrow> complex_homo \<Rightarrow> complex_homo" is moebius_pt_rep
proof-
  fix M M' x x'
  assume "moebius_mat_eq M M'" "x \<approx> x'"
  thus "moebius_pt_rep M x \<approx> moebius_pt_rep M' x'"
    by (cases "Rep_moebius_mat M", cases "Rep_homo_coords x", auto simp add: field_simps) (rule_tac x="k*ka" in exI, simp)
qed

lemma bij_moebius_pt:
  shows "bij (moebius_pt M)"
unfolding bij_def inj_on_def surj_def
proof (simp, transfer, safe)
  fix M x y
  assume "moebius_pt_rep M x \<approx> moebius_pt_rep M y"
  thus "x \<approx> y"
    using Rep_moebius_mat[of M]
    apply auto
    apply (subst (asm) mult_sv_mv)
    using mult_mv_cancel_l
    by blast
next
  fix M y
  let ?M = "Rep_moebius_mat M" 
  let ?iM = "mat_inv ?M" 
  let ?y = "Rep_homo_coords y"
  show "\<exists> x. y \<approx> moebius_pt_rep M x"
    using Rep_moebius_mat[of M] mat_det_inv[of ?M] Rep_homo_coords[of y] mult_mv_nonzero[of ?y ?iM]
    using mat_inv_r[of ?M] eye_mv_l[of ?y]
    by (auto, rule_tac x="Abs_homo_coords ((mat_inv (Rep_moebius_mat M)) *\<^sub>m\<^sub>v Rep_homo_coords y)" in exI, rule_tac x="1" in exI)
       (auto simp add: Abs_homo_coords_inverse)
qed

definition is_moebius where
  "is_moebius f \<longleftrightarrow> (\<exists> M. f = moebius_pt M)"

text {* Bilinear and linear expressions *}
lemma moebius_bilinear:
  assumes "mat_det (a, b, c, d) \<noteq> 0"
  shows "moebius_pt (mk_moebius a b c d) z = 
            (if z \<noteq> \<infinity>\<^sub>h then 
                 ((of_complex a) *\<^sub>h z +\<^sub>h (of_complex b)) :\<^sub>h 
                 ((of_complex c) *\<^sub>h z +\<^sub>h (of_complex d))
             else
                 (of_complex a) :\<^sub>h 
                 (of_complex c))"
unfolding divide_homo_def
using assms
proof (transfer)
  fix a b c d :: complex and z
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  assume *: "mat_det (a, b, c, d) \<noteq> 0"
  let ?oc = "of_complex_coords"
  show "moebius_pt_rep (mk_moebius_rep a b c d) z \<approx>
       (if \<not> z \<approx> inf_homo_rep
        then ?oc a *\<^sub>h\<^sub>c z +\<^sub>h\<^sub>c ?oc b *\<^sub>h\<^sub>c
             reciprocal_homo_coords (?oc c *\<^sub>h\<^sub>c z +\<^sub>h\<^sub>c ?oc d)
        else ?oc a *\<^sub>h\<^sub>c
             reciprocal_homo_coords (of_complex_coords c))"
  proof (cases "z \<approx> inf_homo_rep")
    case True
    thus ?thesis
      using zz *
      using mult_homo_coords_Rep[of "?oc a" a 1 "reciprocal_homo_coords (?oc c)" 1 c]
      using reciprocal_homo_coords_Rep[of "?oc c"]
      by (force simp add: mk_moebius_rep_Rep field_simps)
  next
    case False
    hence "z2 \<noteq> 0"
      using zz Rep_homo_coords[of z]
      by auto (metis mult.commute complex_divide_def mult_zero_right right_inverse_eq)
    thus ?thesis
      using zz * False
      using regular_homogenous_system[of a d b c z1 z2]
      apply simp
      apply (subst mult_homo_coords_Rep[of "?oc a *\<^sub>h\<^sub>c z +\<^sub>h\<^sub>c ?oc b" "a*z1+b*z2" z2 "reciprocal_homo_coords (?oc c *\<^sub>h\<^sub>c z +\<^sub>h\<^sub>c ?oc d)" z2 "c*z1+d*z2"])
      using add_homo_coords_Rep[of "?oc a *\<^sub>h\<^sub>c z" "a*z1" z2 "?oc b" b 1]
      using mult_homo_coords_Rep[of "?oc a" a 1 z z1 z2]
      using reciprocal_homo_coords_Rep[of "?oc c *\<^sub>h\<^sub>c z +\<^sub>h\<^sub>c ?oc d"]
      using add_homo_coords_Rep[of "?oc c  *\<^sub>h\<^sub>c z" "c*z1" z2 "?oc d" d 1]
      using mult_homo_coords_Rep[of "?oc c" c 1 z z1 z2]
      by (auto simp add: mk_moebius_rep_Rep)
  qed
qed

subsection {* Moebius group *}
definition moebius_inv_rep where 
  "moebius_inv_rep M = 
         (let M = Rep_moebius_mat M 
           in Abs_moebius_mat (mat_inv M))"

lemma [simp]: "Rep_moebius_mat (Abs_moebius_mat (mat_inv (Rep_moebius_mat M))) = mat_inv (Rep_moebius_mat M)"
using Rep_moebius_mat[of M] mat_det_inv[of "Rep_moebius_mat M"]
by (auto simp add: Abs_moebius_mat_inverse)

lemma [simp]: "Rep_moebius_mat (moebius_inv_rep M) = mat_inv (Rep_moebius_mat M)"
by (simp add: moebius_inv_rep_def)

lift_definition moebius_inv :: "moebius \<Rightarrow> moebius" is "moebius_inv_rep"
proof-
  fix x y
  assume "moebius_mat_eq x y"
  thus "moebius_mat_eq (moebius_inv_rep x) (moebius_inv_rep y)"
    by (auto simp add: mat_inv_mult_sm) (rule_tac x="1/k" in exI, simp)
qed

lemma moebius_inv: "moebius_pt (moebius_inv M) = inv (moebius_pt M)"
proof (rule inv_equality[symmetric])
  fix x
  show "moebius_pt (moebius_inv M) (moebius_pt M x) = x"
  proof (transfer)
    fix M x
    show "moebius_pt_rep (moebius_inv_rep M) (moebius_pt_rep M x) \<approx> x"
      using Rep_moebius_mat[of M] Rep_homo_coords[of x]  eye_mv_l
      by (simp add: mat_inv_l) (rule_tac x="1" in exI, simp) 
  qed
next
  fix y
  show "moebius_pt M (moebius_pt (moebius_inv M) y) = y"
  proof (transfer)
    fix M y
    show "moebius_pt_rep M (moebius_pt_rep (moebius_inv_rep M) y) \<approx> y"
      using Rep_moebius_mat[of M] eye_mv_l
      by (simp add: mat_inv_r) (rule_tac x="1" in exI, simp)
  qed
qed

lemma is_moebius_inv:
  assumes "is_moebius m"
  shows "is_moebius (inv m)"
  using assms
  unfolding is_moebius_def
  using moebius_inv[symmetric]
  by auto


definition moebius_comp_rep where 
  "moebius_comp_rep M1 M2 = 
     (let M1 = Rep_moebius_mat M1; 
          M2 = Rep_moebius_mat M2 in 
          Abs_moebius_mat (M1 *\<^sub>m\<^sub>m M2))"

lemma [simp]: "Rep_moebius_mat (Abs_moebius_mat ((Rep_moebius_mat M1) *\<^sub>m\<^sub>m (Rep_moebius_mat M2))) = (Rep_moebius_mat M1) *\<^sub>m\<^sub>m (Rep_moebius_mat M2)"
using Rep_moebius_mat[of M1] Rep_moebius_mat[of M2]
by (simp add: Abs_moebius_mat_inverse)

lemma [simp]: "Rep_moebius_mat (moebius_comp_rep M1 M2) = (Rep_moebius_mat M1) *\<^sub>m\<^sub>m (Rep_moebius_mat M2)"
by (simp add: moebius_comp_rep_def)

lift_definition moebius_comp :: "moebius \<Rightarrow> moebius \<Rightarrow> moebius" is moebius_comp_rep
by auto (rule_tac x="ka*k" in exI, simp)

lemma moebius_comp: "moebius_pt M1 \<circ> moebius_pt M2 = moebius_pt (moebius_comp M1 M2)"
unfolding comp_def
by (rule ext, transfer) (simp, rule_tac x="1" in exI, simp)

lemma is_moebius_comp:
  assumes "is_moebius m1" "is_moebius m2"
  shows "is_moebius (m1 \<circ> m2)"
  using assms
  unfolding is_moebius_def
  using moebius_comp
  by auto

definition [simp]: "id_moebius_rep = Abs_moebius_mat eye"

lift_definition id_moebius :: "moebius" is id_moebius_rep
done

lemma [simp]: "Rep_moebius_mat (Abs_moebius_mat (1, 0, 0, 1)) = eye"
by (simp add: Abs_moebius_mat_inverse)

lemma [simp]: "Rep_moebius_mat (id_moebius_rep) = eye"
by simp

lemma "moebius_pt id_moebius = id"
unfolding id_def
apply (rule ext, transfer)
using eye_mv_l
by simp (rule_tac x="1" in exI, simp)

instantiation moebius :: group_add
begin
definition plus_moebius :: "moebius \<Rightarrow> moebius \<Rightarrow> moebius" where
  [simp]: "plus_moebius = moebius_comp"

definition uminus_moebius :: "moebius \<Rightarrow> moebius" where
  [simp]: "uminus_moebius = moebius_inv"

definition zero_moebius :: "moebius" where
  [simp]: "zero_moebius = id_moebius"

definition minus_moebius :: "moebius \<Rightarrow> moebius \<Rightarrow> moebius" where
  [simp]: "minus_moebius A B = A + (-B)"

instance proof
  fix a b c :: moebius
  show "a + b + c = a + (b + c)"
    unfolding plus_moebius_def
  proof (transfer)
    fix a b c
    show "moebius_mat_eq (moebius_comp_rep (moebius_comp_rep a b) c) (moebius_comp_rep a (moebius_comp_rep b c))"
      using Rep_moebius_mat[of a] Rep_moebius_mat[of b] Rep_moebius_mat[of c]
      by simp (rule_tac x="1" in exI, simp add: mult_mm_assoc)
  qed
next
  fix a :: moebius
  show "a + 0 = a"
    unfolding plus_moebius_def zero_moebius_def
  proof (transfer)
    fix A
    show "moebius_mat_eq (moebius_comp_rep A id_moebius_rep) A"
      using mat_eye_r
      by simp (rule_tac x="1" in exI, simp)
  qed
next
  fix a :: moebius
  show "0 + a = a"
    unfolding plus_moebius_def zero_moebius_def
  proof (transfer)
    fix A
    show "moebius_mat_eq (moebius_comp_rep id_moebius_rep A) A"
      using mat_eye_l 
      by simp (rule_tac x="1" in exI, simp)
  qed
next
  fix a :: moebius
  show "- a + a = 0"
    unfolding plus_moebius_def uminus_moebius_def zero_moebius_def
  proof (transfer)
    fix a
    show "moebius_mat_eq (moebius_comp_rep (moebius_inv_rep a) a) id_moebius_rep"
      using Rep_moebius_mat[of a]
      by (simp add: mat_inv_l)
  qed
next
  fix a b :: moebius
  show "a - b = a + - b"
    unfolding minus_moebius_def
    by simp
qed
end

lemma [simp]: "moebius_comp (moebius_inv M) M = id_moebius"
by (metis left_minus plus_moebius_def uminus_moebius_def zero_moebius_def)

lemma [simp]: "moebius_comp M (moebius_inv M) = id_moebius"
by (metis right_minus plus_moebius_def uminus_moebius_def zero_moebius_def)

lemma moebius_pt_moebius_id [simp]: "moebius_pt (id_moebius) = id"
by (rule ext) (transfer, case_tac "Rep_homo_coords x", auto, rule_tac x="1" in exI, simp)

lemma [simp]: "moebius_pt (moebius_inv M) (moebius_pt M z) = z"
proof-
  have "moebius_pt (moebius_inv M) (moebius_pt M z) = (moebius_pt (moebius_inv M) \<circ> moebius_pt M) z"
    by simp
  thus ?thesis
    using moebius_comp[of "moebius_inv M" M]
    by simp
qed

lemma moebius_pt_invert:
  assumes "w = moebius_pt M z" 
  shows "z = moebius_pt (moebius_inv M) w"
using assms
by auto


subsection{* Special kinds of Moebius transformations *}

text {* Reciprocal (1/z) as a moebius transformation *}
definition reciprocal_moebius :: "moebius" where 
  "reciprocal_moebius = mk_moebius 0 1 1 0"

lemma [simp]: "Rep_moebius_mat (Abs_moebius_mat (0, 1, 1, 0)) = (0, 1, 1, 0)"
by (simp add: Abs_moebius_mat_inverse)

lemma [simp]: "Rep_moebius_mat (mk_moebius_rep 0 1 1 0) = (0, 1, 1, 0)"
by (simp add: mk_moebius_rep_def)

lemma [simp]: "Rep_homo_coords (reciprocal_homo_coords z) = (let (x, y) = Rep_homo_coords z in (y, x))"
unfolding reciprocal_homo_coords_def Let_def 
apply (cases "Rep_homo_coords z")
using Rep_homo_coords[of z]
by (auto simp add: Abs_homo_coords_inverse)

lemma reciprocal_moebius:
  "reciprocal_homo = moebius_pt reciprocal_moebius"
  unfolding reciprocal_moebius_def
by (rule ext, transfer) (auto simp add: split_def Let_def, case_tac "Rep_homo_coords x", rule_tac x="1" in exI, auto)

lemma reciprocal_moebius_inv [simp]:
  "moebius_inv reciprocal_moebius = reciprocal_moebius"
unfolding reciprocal_moebius_def
by transfer simp

lemma reciprocal_homo_only_0_to_inf:
  assumes "reciprocal_homo z = \<infinity>\<^sub>h"
  shows "z = 0\<^sub>h"
using assms
unfolding reciprocal_moebius
using moebius_pt_invert[of "\<infinity>\<^sub>h" reciprocal_moebius z]
by (simp add: reciprocal_moebius[symmetric])

lemma reciprocal_homo_only_inf_to_0:
  assumes "reciprocal_homo z = 0\<^sub>h"
  shows "z = \<infinity>\<^sub>h"
using assms
unfolding reciprocal_moebius
using moebius_pt_invert[of "0\<^sub>h" reciprocal_moebius z]
by (simp add: reciprocal_moebius[symmetric])

text{* Euclidean similarity as a Moebius transform *}
definition similarity_moebius :: "complex \<Rightarrow> complex \<Rightarrow> moebius" where
  "similarity_moebius a b = mk_moebius a b 0 1"

lemma moebius_similarity_linear:
  assumes "a \<noteq> 0"
  shows "moebius_pt (similarity_moebius a b) z = (of_complex a) *\<^sub>h z +\<^sub>h (of_complex b)"
unfolding similarity_moebius_def
using assms
using mult_homo_inf_right[of "of_complex a"]
by (subst moebius_bilinear, auto)

lemma moebius_similarity':
  assumes "a \<noteq> 0"
  shows "moebius_pt (similarity_moebius a b) = (\<lambda> z. (of_complex a) *\<^sub>h z +\<^sub>h (of_complex b))"
using moebius_similarity_linear[OF assms, symmetric]
by simp

lemma is_moebius_similarity':
  assumes "a \<noteq> 0\<^sub>h" "a \<noteq> \<infinity>\<^sub>h" "b \<noteq> \<infinity>\<^sub>h"
  shows "(\<lambda> z. a *\<^sub>h z +\<^sub>h b) = moebius_pt (similarity_moebius (to_complex a) (to_complex b))"
proof-
  obtain ka kb where *: "a = of_complex ka"  "ka \<noteq> 0" "b = of_complex kb"
    using assms
    using inf_homo_or_complex_homo[of a]  inf_homo_or_complex_homo[of b]
    by auto
  thus ?thesis
    unfolding is_moebius_def
    using moebius_similarity'[of ka kb]
    by simp
qed

lemma is_moebius_similarity:
  assumes "a \<noteq> 0\<^sub>h" "a \<noteq> \<infinity>\<^sub>h" "b \<noteq> \<infinity>\<^sub>h"
  shows "is_moebius (\<lambda> z. a *\<^sub>h z +\<^sub>h b)"
using is_moebius_similarity'[OF assms]
unfolding is_moebius_def
by auto

lemma similarity_moebius_comp:
  assumes "a \<noteq> 0" "c \<noteq> 0"
  shows "similarity_moebius a b + similarity_moebius c d = similarity_moebius (a*c) (a*d+b)"
using assms
unfolding similarity_moebius_def plus_moebius_def
by transfer (simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)

lemma similarity_moebius_inv:
  assumes "a \<noteq> 0"
  shows "- similarity_moebius a b = similarity_moebius (1/a) (-b/a)"
using assms
unfolding similarity_moebius_def uminus_moebius_def
by transfer (simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)

lemma similarity_moebius_id: "id_moebius = similarity_moebius 1 0"
unfolding similarity_moebius_def
by transfer (simp add: mk_moebius_rep_def)

lemma similarity_inf_fixed:
  assumes "a \<noteq> 0"
  shows "moebius_pt (similarity_moebius a b) \<infinity>\<^sub>h = \<infinity>\<^sub>h"
using assms
unfolding similarity_moebius_def
by transfer (simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)

lemma similarity_only_inf_to_inf:
  assumes "a \<noteq> 0"  "moebius_pt (similarity_moebius a b) z = \<infinity>\<^sub>h"
  shows "z = \<infinity>\<^sub>h"
using assms moebius_pt_invert[of "\<infinity>\<^sub>h" "similarity_moebius a b" z] similarity_inf_fixed[of "1/a" "-b/a"]
using similarity_moebius_inv[of a b]
by simp

lemma inf_fixed_similarity:
  assumes "moebius_pt M \<infinity>\<^sub>h = \<infinity>\<^sub>h"
  shows "\<exists> a b. a \<noteq> 0 \<and> M = similarity_moebius a b"
using assms
unfolding similarity_moebius_def
proof transfer
  fix M
  obtain a b c d where MM: "Rep_moebius_mat M = (a, b, c, d)"
    by (cases M) (auto simp add: Abs_moebius_mat_inverse)
  assume "moebius_pt_rep M inf_homo_rep \<approx> inf_homo_rep"
  hence "c = 0"
    using MM
    by auto
  hence *: "a \<noteq> 0 \<and> d \<noteq> 0"
    using Rep_moebius_mat[of M] MM
    by auto
  show "\<exists>a b. a \<noteq> 0 \<and> moebius_mat_eq M (mk_moebius_rep a b 0 1)"
  proof (rule_tac x="a/d" in exI, rule_tac x="b/d" in exI)
    show "a/d \<noteq> 0 \<and> moebius_mat_eq M (mk_moebius_rep (a / d) (b / d) 0 1)"
      using MM `c = 0` `a \<noteq> 0 \<and> d \<noteq> 0`
      by simp (rule_tac x="1/d" in exI, simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)
  qed
qed

text{* Translation *}
definition translation_moebius where
  "translation_moebius v = similarity_moebius 1 v"

lemma translation_moebius_comp: 
  "(translation_moebius v1) + (translation_moebius v2) = translation_moebius (v1 + v2)"
unfolding translation_moebius_def similarity_moebius_def plus_moebius_def
by (transfer) (auto simp add: mk_moebius_rep_Rep)

lemma translation_moebius_zero:
  "translation_moebius 0 = id_moebius"
unfolding translation_moebius_def similarity_moebius_def
by (transfer) (auto simp add: mk_moebius_rep_Rep)

lemma moebius_translation_inv:
  "- (translation_moebius v1) = translation_moebius (-v1)"
using translation_moebius_comp[of v1 "-v1"] translation_moebius_zero uminus_moebius_def
using equals_zero_I[of "translation_moebius v1" "translation_moebius (-v1)"]
by simp

lemma moebius_pt_translation [simp]: "moebius_pt (translation_moebius v) (of_complex z) = of_complex (v + z)"
unfolding translation_moebius_def similarity_moebius_def
by transfer (simp add: mk_moebius_rep_Rep)

text{* Rotation *}
definition rotation_moebius where
  "rotation_moebius \<phi> = similarity_moebius (cis \<phi>) 0"

lemma rotation_moebius_comp: 
  "(rotation_moebius \<phi>1) + (rotation_moebius \<phi>2) = rotation_moebius (\<phi>1 + \<phi>2)"
  unfolding rotation_moebius_def similarity_moebius_def plus_moebius_def
  by transfer (simp add: mk_moebius_rep_Rep cis_mult)

lemma rotation_moebius_zero:
  "rotation_moebius 0 = id_moebius"
  unfolding rotation_moebius_def similarity_moebius_def
  by transfer (simp add: mk_moebius_rep_Rep)

lemma rotation_moebius_inverse:
  "- (rotation_moebius \<phi>) = rotation_moebius (- \<phi>)"
using rotation_moebius_comp[of \<phi> "-\<phi>"] rotation_moebius_zero
using equals_zero_I[of "rotation_moebius \<phi>" "rotation_moebius (-\<phi>)"]
by simp

lemma moebius_pt_rotation [simp]: "moebius_pt (rotation_moebius \<phi>) (of_complex z) = of_complex (cis \<phi> * z)"
unfolding rotation_moebius_def similarity_moebius_def
by transfer (simp add: mk_moebius_rep_Rep)

text{* Dilatation *}
definition dilatation_moebius where
  "dilatation_moebius a = similarity_moebius (cor a) 0"

lemma dilatation_moebius_comp: 
  assumes "a1 > 0" "a2 > 0"
  shows "(dilatation_moebius a1) + (dilatation_moebius a2) = dilatation_moebius (a1 * a2)"
using assms
unfolding dilatation_moebius_def similarity_moebius_def plus_moebius_def
by transfer (simp add: mk_moebius_rep_def Abs_moebius_mat_inverse)

lemma dilatation_moebius_zero:
  "dilatation_moebius 1 = id_moebius"
  unfolding dilatation_moebius_def similarity_moebius_def
  by transfer (simp add: mk_moebius_rep_Rep)

lemma dilatation_moebius_inverse:
  assumes "a > 0"
  shows "- (dilatation_moebius a) = dilatation_moebius (1/a)"
using assms
using dilatation_moebius_comp[of a "1/a"] dilatation_moebius_zero
using equals_zero_I[of "dilatation_moebius a" "dilatation_moebius (1/a)"]
by simp

lemma moebius_pt_dilatation [simp]: "a \<noteq> 0 \<Longrightarrow> moebius_pt (dilatation_moebius a) (of_complex z) = of_complex (cor a * z)"
unfolding dilatation_moebius_def similarity_moebius_def
by transfer (simp add: mk_moebius_rep_Rep)

text{* @{term rotation_dilation_moebius} *}

definition rotation_dilatation_moebius where
  "rotation_dilatation_moebius a = similarity_moebius a 0"

lemma rot_dil:
  assumes "a \<noteq> 0"
  shows "rotation_dilatation_moebius a = rotation_moebius (arg a) + dilatation_moebius (cmod a)"
using assms
unfolding rotation_dilatation_moebius_def rotation_moebius_def dilatation_moebius_def similarity_moebius_def plus_moebius_def
by transfer (simp add: mk_moebius_rep_Rep)

subsection{* Decomposition *}

lemma similarity_decomposition:
  assumes "a \<noteq> 0"
  shows "similarity_moebius a b = (translation_moebius b) + (rotation_moebius (arg a)) + (dilatation_moebius (cmod a))"
proof-
  have "similarity_moebius a b = (translation_moebius b) + rotation_dilatation_moebius a"
    unfolding rotation_dilatation_moebius_def translation_moebius_def similarity_moebius_def plus_moebius_def
    using assms
    by transfer (simp add: mk_moebius_rep_Rep)
  thus ?thesis
    using rot_dil[OF assms]
    by (auto simp add: add_assoc simp del: plus_moebius_def)
qed

lemma moebius_decomposition:
  assumes "c \<noteq> 0" "a*d - b*c \<noteq> 0"
  shows "mk_moebius a b c d = 
             translation_moebius (a/c) + 
             rotation_dilatation_moebius ((b*c - a*d)/(c*c)) +
             reciprocal_moebius + 
             translation_moebius (d/c)"
  using assms
  unfolding rotation_dilatation_moebius_def translation_moebius_def similarity_moebius_def plus_moebius_def reciprocal_moebius_def
  by transfer (simp add: mk_moebius_rep_Rep, rule_tac x="1/c" in exI, simp add: field_simps)

lemma wlog_moebius_decomposition:
  assumes
  trans: "\<And> v. P (translation_moebius v)" and rot: "\<And> \<alpha>. P (rotation_moebius \<alpha>)" and dil: "\<And> k. P (dilatation_moebius k)" and recip: "P (reciprocal_moebius)" and
  comp: "\<And> M1 M2. \<lbrakk>P M1; P M2\<rbrakk> \<Longrightarrow> P (M1 + M2)"
  shows "P M"
proof-
    obtain a b c d where "M = mk_moebius a b c d" "mat_det (a, b, c, d) \<noteq> 0"
      using ex_mk_moebius[of M]
      by auto
    show ?thesis
    proof (cases "c = 0")
      case False
      show ?thesis
        using moebius_decomposition[of c a d b] `mat_det (a, b, c, d) \<noteq> 0` `c \<noteq> 0` `M = mk_moebius a b c d`
        using rot_dil[of "(b*c - a*d) / (c*c)"]
        using trans[of "a/c"] rot[of "arg ((b*c - a*d) / (c*c))"] dil[of "cmod ((b*c - a*d) / (c*c))"] recip
        using comp
        by simp (metis trans)
    next
      case True
      hence "M = similarity_moebius (a/d) (b/d)"
        using `M = mk_moebius a b c d` `mat_det (a, b, c, d) \<noteq> 0`
        unfolding similarity_moebius_def
        by transfer (auto simp add: mk_moebius_rep_Rep, rule_tac x="k/d" in exI, case_tac "Rep_moebius_mat M", simp)
      thus ?thesis
        using `c = 0` `mat_det (a, b, c, d) \<noteq> 0`
        using similarity_decomposition[of "a/d" "b/d"]
        using trans[of "b/d"] rot[of "arg (a/d)"] dil[of "cmod (a/d)"] comp
        by simp
    qed
qed

subsection {* Cross ratio and moebius existence *}
lemma is_moebius_cross_ratio:
  assumes "z1 \<noteq> z2" "z2 \<noteq> z3" "z1 \<noteq> z3"
  shows "is_moebius (\<lambda> z. cross_ratio z z1 z2 z3)"
proof-
  have "\<exists> M. \<forall> z. cross_ratio z z1 z2 z3 = moebius_pt M z"
    using assms
  proof (transfer)
    fix z1 z2 z3
    obtain z1' z1'' where zz1: "Rep_homo_coords z1 = (z1', z1'')"
      by (rule obtain_homo_coords)
    obtain z2' z2'' where zz2: "Rep_homo_coords z2 = (z2', z2'')"
      by (rule obtain_homo_coords)
    obtain z3' z3'' where zz3: "Rep_homo_coords z3 = (z3', z3'')"
      by (rule obtain_homo_coords)

    let ?m23 = "z2'*z3''-z3'*z2''"
    let ?m21 = "z2'*z1''-z1'*z2''"
    let ?m13 = "z1'*z3''-z3'*z1''"
    let ?M = "(z1''*?m23, -z1'*?m23, z3''*?m21, -z3'*?m21)"
    assume "\<not> z1 \<approx> z2" "\<not> z2 \<approx> z3" "\<not> z1 \<approx> z3"
    hence *: "?m23 \<noteq> 0" "?m21 \<noteq> 0" "?m13 \<noteq> 0"
      using zz1 zz2 zz3
      using homo_coords_eq_mix[of z1 z1' z1'' z2 z2' z2''] homo_coords_eq_mix[of z1 z1' z1'' z3 z3' z3'']  homo_coords_eq_mix[of z2 z2' z2'' z3 z3' z3'']
      by auto
    have "mat_det ?M = ?m21*?m23*?m13"
      by (simp add: field_simps)
    hence "mat_det ?M \<noteq> 0"
      using *
      by simp

    show "\<exists>M. \<forall>z. cross_ratio_rep z z1 z2 z3 \<approx> moebius_pt_rep M z"
    proof (rule_tac x="Abs_moebius_mat ?M" in exI, rule)
      fix z
      obtain z' z'' where zz: "Rep_homo_coords z = (z', z'')"
        by (rule obtain_homo_coords)

      let ?m01 = "z'*z1''-z1'*z''"
      let ?m03 = "z'*z3''-z3'*z''"

      have "?m01 \<noteq> 0 \<or> ?m03 \<noteq> 0"
        using * Rep_homo_coords[of z] zz
        apply (cases "z'' = 0 \<or> z1'' = 0 \<or> z3'' = 0")
        apply (auto simp add: field_simps)
        apply (subgoal_tac "z1'/z1'' = z3'/z3''")
        by (simp add: field_simps) (metis eq_divide_imp mult_divide_mult_cancel_left times_divide_eq_right times_divide_times_eq)
      note * = * this

      show "cross_ratio_rep z z1 z2 z3 \<approx> moebius_pt_rep (Abs_moebius_mat ?M) z"
        using zz1 zz2 zz3 zz * Rep_homo_coords[of z] mult_mv_nonzero[of "Rep_homo_coords z" ?M] `mat_det ?M \<noteq> 0`
        by (simp add: cross_ratio_rep_def moebius_pt_rep_def split_def Let_def Abs_moebius_mat_inverse Abs_homo_coords_inverse)
           (rule_tac x="1" in exI, simp add: field_simps)
    qed
  qed
  thus ?thesis
    by (auto simp add: is_moebius_def)
qed

lemma ex_moebius_01inf:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"
  shows "\<exists> M. ((moebius_pt M z1 = 0\<^sub>h) \<and> (moebius_pt M z2 = 1\<^sub>h) \<and> (moebius_pt M z3 = \<infinity>\<^sub>h))"
using assms
using is_moebius_cross_ratio[OF `z1 \<noteq> z2` `z2 \<noteq> z3` `z1 \<noteq> z3`]
using cross_ratio_0[OF `z1 \<noteq> z2` `z1 \<noteq> z3`] cross_ratio_1[OF `z1 \<noteq> z2` `z2 \<noteq> z3`] cross_ratio_inf[OF `z1 \<noteq> z3` `z2 \<noteq> z3`]
by (auto simp add: is_moebius_def) metis

lemma ex_moebius:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"  "w1 \<noteq> w2" "w1 \<noteq> w3" "w2 \<noteq> w3"
  shows "\<exists> M. ((moebius_pt M z1 = w1) \<and> (moebius_pt M z2 = w2) \<and> (moebius_pt M z3 = w3))"
proof-
  obtain M1 where *: "moebius_pt M1 z1 = 0\<^sub>h \<and> moebius_pt M1 z2 = 1\<^sub>h \<and> moebius_pt M1 z3 = \<infinity>\<^sub>h"
    using ex_moebius_01inf[OF assms(1-3)]
    by auto
  obtain M2 where **: "moebius_pt M2 w1 = 0\<^sub>h \<and> moebius_pt M2 w2 = 1\<^sub>h \<and> moebius_pt M2 w3 = \<infinity>\<^sub>h"
    using ex_moebius_01inf[OF assms(4-6)]
    by auto
  let ?M = "moebius_comp (moebius_inv M2) M1"
  show ?thesis
    using * ** bij_moebius_pt[of M2]
    by (rule_tac x="?M" in exI, (subst moebius_comp[symmetric])+, (subst moebius_inv)+) (simp add: bij_def inv_f_eq)
qed

lemma ex_moebius_1:
  shows "\<exists> M. moebius_pt M z1 = w1"
proof-
  obtain z2 z3 where "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"
    using ex_3_different_points[of z1]
    by auto
  moreover
  obtain w2 w3 where "w1 \<noteq> w2" "w1 \<noteq> w3" "w2 \<noteq> w3"
    using ex_3_different_points[of w1]
    by auto
  ultimately
  show ?thesis
    using ex_moebius[of z1 z2 z3 w1 w2 w3]
    by auto
qed

lemma wlog_moebius_01inf:
  fixes M::moebius
  assumes "P 0\<^sub>h 1\<^sub>h \<infinity>\<^sub>h" "z1 \<noteq> z2" "z2 \<noteq> z3" "z1 \<noteq> z3" 
   "\<And> M a b c. P a b c \<Longrightarrow> P (moebius_pt M a) (moebius_pt M b) (moebius_pt M c)"
  shows "P z1 z2 z3"
proof-
  from assms obtain M where *:
    "moebius_pt M z1 = 0\<^sub>h"  "moebius_pt M z2 = 1\<^sub>h"   "moebius_pt M z3 = \<infinity>\<^sub>h"
    using ex_moebius_01inf[of z1 z2 z3]
    by auto
  have **: "moebius_pt (moebius_inv M) 0\<^sub>h = z1"  "moebius_pt (moebius_inv M) 1\<^sub>h = z2" "moebius_pt (moebius_inv M) \<infinity>\<^sub>h = z3"
    by (subst *[symmetric], simp)+
  thus ?thesis
    using assms
    by auto
qed

subsection {* Fixed points and moebius uniqueness *}
lemma three_fixed_points_01inf:
  assumes "moebius_pt M 0\<^sub>h = 0\<^sub>h"  "moebius_pt M 1\<^sub>h = 1\<^sub>h" "moebius_pt M \<infinity>\<^sub>h = \<infinity>\<^sub>h"
  shows "M = id_moebius"
using assms
by transfer (case_tac "Rep_moebius_mat M", auto)

lemma three_fixed_points:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"
  assumes "moebius_pt M z1 = z1" "moebius_pt M z2 = z2" "moebius_pt M z3 = z3"
  shows "M = id_moebius"
proof-
  from assms obtain M' where *: "moebius_pt M' z1 = 0\<^sub>h"  "moebius_pt M' z2 = 1\<^sub>h"   "moebius_pt M' z3 = \<infinity>\<^sub>h"
    using ex_moebius_01inf[of z1 z2 z3]
    by auto
  have **: "moebius_pt (moebius_inv M') 0\<^sub>h = z1"  "moebius_pt (moebius_inv M') 1\<^sub>h = z2" "moebius_pt (moebius_inv M') \<infinity>\<^sub>h = z3"
    by (subst *[symmetric], simp)+

  have "M' + M + (-M') = 0"
    unfolding zero_moebius_def
    apply (rule three_fixed_points_01inf)
    using * ** assms
    by (simp add: moebius_comp[symmetric])+
  thus ?thesis
    by (metis eq_neg_iff_add_eq_0 minus_add_cancel zero_moebius_def)
qed

lemma unique_moebius_three_points:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"
  assumes "moebius_pt M1 z1 = w1" " moebius_pt M1 z2 = w2" "moebius_pt M1 z3 = w3"
          "moebius_pt M2 z1 = w1" " moebius_pt M2 z2 = w2" "moebius_pt M2 z3 = w3"
  shows "M1 = M2"
proof-
  let ?M = "moebius_comp (moebius_inv M2) M1"
  have "moebius_pt ?M z1 = z1"
    using `moebius_pt M1 z1 = w1` `moebius_pt M2 z1 = w1`
    using bij_moebius_pt[of M2]
    by (subst moebius_comp[symmetric], subst moebius_inv, simp add: bij_def inv_f_eq)
  moreover
  have "moebius_pt ?M z2 = z2"
    using `moebius_pt M1 z2 = w2` `moebius_pt M2 z2 = w2`
    using bij_moebius_pt[of M2]
    by (subst moebius_comp[symmetric], subst moebius_inv, simp add: bij_def inv_f_eq)
  moreover
  have "moebius_pt ?M z3 = z3"
    using `moebius_pt M1 z3 = w3` `moebius_pt M2 z3 = w3`
    using bij_moebius_pt[of M2]
    by (subst moebius_comp[symmetric], subst moebius_inv, simp add: bij_def inv_f_eq)
  ultimately
  have "?M = id_moebius"
    using assms three_fixed_points
    by auto
  thus ?thesis
    by (metis add_minus_cancel left_minus plus_moebius_def uminus_moebius_def zero_moebius_def)
qed

lemma ex_unique_moebius_three_points:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"  "w1 \<noteq> w2" "w1 \<noteq> w3" "w2 \<noteq> w3"
  shows "\<exists>! M. ((moebius_pt M z1 = w1) \<and> (moebius_pt M z2 = w2) \<and> (moebius_pt M z3 = w3))"
proof-
  obtain M where *: "moebius_pt M z1 = w1 \<and> moebius_pt M z2 = w2 \<and> moebius_pt M z3 = w3"
    using ex_moebius[OF assms]
    by auto
  show ?thesis
    unfolding Ex1_def
  proof (rule_tac x="M" in exI, rule)
    show "\<forall>y. moebius_pt y z1 = w1 \<and> moebius_pt y z2 = w2 \<and> moebius_pt y z3 = w3 \<longrightarrow> y = M"
      using *
      using unique_moebius_three_points[OF assms(1-3)]
      by simp
  qed (simp add: *)
qed

lemma ex_unique_moebius_three_points_fun:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3" "w1 \<noteq> w2" "w1 \<noteq> w3" "w2 \<noteq> w3"
  shows "\<exists>! f. is_moebius f \<and> (f z1 = w1) \<and> (f z2 = w2) \<and> (f z3 = w3)"
proof-
  obtain M where "moebius_pt M z1 = w1" "moebius_pt M z2 = w2" "moebius_pt M z3 = w3"
    using ex_unique_moebius_three_points[OF assms]
    by auto
  thus ?thesis
    using ex_unique_moebius_three_points[OF assms]
    unfolding Ex1_def
    by (rule_tac x="moebius_pt M" in exI) (auto simp add: is_moebius_def)
qed

lemma is_cross_ratio_01inf:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3" "is_moebius f"
  assumes "f z1 = 0\<^sub>h" "f z2 = 1\<^sub>h" "f z3 = \<infinity>\<^sub>h"
  shows "f = (\<lambda> z. cross_ratio z z1 z2 z3)"
  using assms
  using cross_ratio_0[OF `z1 \<noteq> z2` `z1 \<noteq> z3`] cross_ratio_1[OF `z1 \<noteq> z2` `z2 \<noteq> z3`] cross_ratio_inf[OF `z1 \<noteq> z3` `z2 \<noteq> z3`]
  using is_moebius_cross_ratio[OF `z1 \<noteq> z2` `z2 \<noteq> z3` `z1 \<noteq> z3`]
  using ex_unique_moebius_three_points_fun[OF `z1 \<noteq> z2` `z1 \<noteq> z3` `z2 \<noteq> z3`, of "0\<^sub>h" "1\<^sub>h" "\<infinity>\<^sub>h"]
  by auto


lemma moebius_preserve_cross_ratio:
  assumes "z1 \<noteq> z2" "z1 \<noteq> z3" "z2 \<noteq> z3"
  shows "cross_ratio z z1 z2 z3 = cross_ratio (moebius_pt M z) (moebius_pt M z1) (moebius_pt M z2) (moebius_pt M z3)"
proof-
  let ?f = "\<lambda> z. cross_ratio z z1 z2 z3"
  let ?M = "moebius_pt M"
  let ?iM = "inv ?M"
  have "(?f \<circ> ?iM) (?M z1) = 0\<^sub>h"
    using bij_moebius_pt[of M] cross_ratio_0[OF `z1 \<noteq> z2` `z1 \<noteq> z3`]
    by (simp add: bij_def)
  moreover
  have "(?f \<circ> ?iM) (?M z2) = 1\<^sub>h"
    using bij_moebius_pt[of M]  cross_ratio_1[OF `z1 \<noteq> z2` `z2 \<noteq> z3`]
    by (simp add: bij_def)
  moreover
  have "(?f \<circ> ?iM) (?M z3) = \<infinity>\<^sub>h"
    using bij_moebius_pt[of M] cross_ratio_inf[OF `z1 \<noteq> z3` `z2 \<noteq> z3`]
    by (simp add: bij_def)
  moreover
  have "is_moebius (?f \<circ> ?iM)"
    by (rule is_moebius_comp, rule is_moebius_cross_ratio[OF `z1 \<noteq> z2` `z2 \<noteq> z3` `z1 \<noteq> z3`], rule is_moebius_inv, auto simp add: is_moebius_def)
  moreover
  have "?M z1 \<noteq> ?M z2" "?M z1 \<noteq> ?M z3"  "?M z2 \<noteq> ?M z3" 
    using assms
    using bij_moebius_pt[of M]
    unfolding bij_def inj_on_def
    by blast+
  ultimately
  have "?f \<circ> ?iM = (\<lambda> z. cross_ratio z (?M z1) (?M z2) (?M z3))"
    using assms 
    using is_cross_ratio_01inf[of "?M z1" "?M z2" "?M z3" "?f \<circ> ?iM"]
    by simp
  moreover
  have "(?f \<circ> ?iM) (?M z) = cross_ratio z z1 z2 z3"
    using bij_moebius_pt[of M]
    by (simp add: bij_def)
  moreover
  have "(\<lambda> z. cross_ratio z (?M z1) (?M z2) (?M z3)) (?M z) = cross_ratio (?M z) (?M z1) (?M z2) (?M z3)"
    by simp
  ultimately
  show ?thesis
    by simp
qed

lemma fixed_points_0inf':
  assumes "moebius_pt M 0\<^sub>h = 0\<^sub>h" "moebius_pt M \<infinity>\<^sub>h = \<infinity>\<^sub>h"
  shows "\<exists> k::complex_homo. (k \<noteq> 0\<^sub>h \<and> k \<noteq> \<infinity>\<^sub>h) \<and> (\<forall> z. moebius_pt M z = k *\<^sub>h z)"
using assms
proof (transfer)
  fix M
  obtain a b c d where MM: "Rep_moebius_mat M = (a, b, c, d)"
    by (cases M) (auto simp add: Abs_moebius_mat_inverse)
  assume "moebius_pt_rep M zero_homo_rep \<approx> zero_homo_rep" "moebius_pt_rep M inf_homo_rep \<approx> inf_homo_rep"
  hence "b = 0" "c = 0"
    using MM
    by auto
  hence *: "a \<noteq> 0 \<and> d \<noteq> 0"
    using Rep_moebius_mat[of M] MM
    by auto
  show "\<exists>k. (\<not> k \<approx> zero_homo_rep \<and> \<not> k \<approx> inf_homo_rep) \<and> (\<forall>z. moebius_pt_rep M z \<approx> k *\<^sub>h\<^sub>c z)"
  proof (rule_tac x="Abs_homo_coords (a, d)" in exI, rule conjI)
    show "\<not> Abs_homo_coords (a, d) \<approx> zero_homo_rep \<and> \<not> Abs_homo_coords (a, d) \<approx> inf_homo_rep"
      using *
      by (auto simp add: Abs_homo_coords_inverse)
  next
    show "\<forall>z. moebius_pt_rep M z \<approx> Abs_homo_coords (a, d) *\<^sub>h\<^sub>c z"
    proof
      fix z
      obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
        by (rule obtain_homo_coords)
      thus "moebius_pt_rep M z \<approx> Abs_homo_coords (a, d) *\<^sub>h\<^sub>c z"
        using MM * `b = 0` `c = 0` mult_homo_coords_Rep[of "Abs_homo_coords (a, d)" a d z z1 z2] Rep_homo_coords[of z]
        by (simp add: Abs_homo_coords_inverse) (rule_tac x="1" in exI, simp)
    qed
  qed
qed

lemma fixed_points_0inf:
  assumes "moebius_pt M 0\<^sub>h = 0\<^sub>h" "moebius_pt M \<infinity>\<^sub>h = \<infinity>\<^sub>h"
  shows "\<exists> k::complex_homo. (k \<noteq> 0\<^sub>h \<and> k \<noteq> \<infinity>\<^sub>h) \<and> moebius_pt M = (\<lambda> z. k *\<^sub>h z)"
using fixed_points_0inf'[OF assms]
by auto

subsection{* Pole *}

definition is_pole where
  "is_pole M z \<longleftrightarrow> moebius_pt M z = \<infinity>\<^sub>h"

lemma ex1_pole:
  "\<exists>! z. is_pole M z"
using bij_moebius_pt[of M]
unfolding is_pole_def bij_def inj_on_def surj_def
unfolding Ex1_def
by (metis UNIV_I)

definition pole where "pole M = (THE z. is_pole M z)"

lemma pole_mk_moebius:
  assumes "is_pole (mk_moebius a b c d) z" "c \<noteq> 0" "a*d - b*c \<noteq> 0"
  shows "z = of_complex (-d/c)"
proof-
  let ?t1 = "translation_moebius (a / c)"
  let ?rd = "rotation_dilatation_moebius ((b * c - a * d) / (c * c))"
  let ?r = "reciprocal_moebius"
  let ?t2 = "translation_moebius (d / c)"
  have "moebius_pt (?rd + ?r + ?t2) z = \<infinity>\<^sub>h"
    using assms
    unfolding is_pole_def
    apply (subst (asm) moebius_decomposition)  
    apply (auto simp add: moebius_comp[symmetric] translation_moebius_def)
    apply (subst similarity_only_inf_to_inf[of 1 "a/c"], auto)
    done
  hence "moebius_pt (?r + ?t2) z = \<infinity>\<^sub>h"
    using `a*d - b*c \<noteq> 0` `c \<noteq> 0`
    unfolding rotation_dilatation_moebius_def
    apply (simp add: moebius_comp[symmetric])
    apply (subst similarity_only_inf_to_inf[of "(b*c-a*d)/(c*c)" 0], auto)
    done
  hence "moebius_pt ?t2 z = 0\<^sub>h"
    apply (simp add: moebius_comp[symmetric])
    apply (subst (asm) reciprocal_moebius[symmetric])
    apply (subst reciprocal_homo_only_0_to_inf, auto)
    done
  thus ?thesis
    using moebius_pt_invert[of "0\<^sub>h" ?t2 z] moebius_translation_inv[of "d/c"]
    by simp (subst zero_of_complex[symmetric], simp del: zero_of_complex)
qed

lemma pole_similarity:
  assumes "is_pole (similarity_moebius a b) z" "a \<noteq> 0"
  shows "z = \<infinity>\<^sub>h"
using assms
unfolding is_pole_def
using similarity_only_inf_to_inf[of a b z]
by simp

subsection{* Antihomographies *}

definition is_antihomography where
 "is_antihomography f \<longleftrightarrow> (\<exists> f'. is_moebius f' \<and> f = f' \<circ> cnj_homo)"

lemma "is_antihomography inversion_homo"
using reciprocal_moebius
unfolding inversion_homo_sym is_antihomography_def
by (auto simp add: is_moebius_def)

subsection{* Classification *}
lemma similarity_scale_1:
  assumes "k \<noteq> 0"
  shows "similarity (k *\<^sub>s\<^sub>m I) M = similarity I M"
using assms
unfolding similarity_def 
using mat_inv_mult_sm[of k I]
by simp

lemma similarity_scale_2:
  shows "similarity I (k *\<^sub>s\<^sub>m M) = k *\<^sub>s\<^sub>m (similarity I M)"
unfolding similarity_def
by auto

lemma [simp]: "mat_trace (k *\<^sub>s\<^sub>m M) = k * mat_trace M"
by (cases M) (simp add: field_simps)

definition moebius_mb_rep where 
  "moebius_mb_rep I M = Abs_moebius_mat (similarity (Rep_moebius_mat I) (Rep_moebius_mat M))"

lemma moebius_mb_rep_Rep [simp]:
  "Rep_moebius_mat (moebius_mb_rep I M) = similarity (Rep_moebius_mat I) (Rep_moebius_mat M)"
using Rep_moebius_mat[of I] Rep_moebius_mat[of M]
unfolding moebius_mb_rep_def
by (simp add: mat_det_similarity Abs_moebius_mat_inverse)

lift_definition moebius_mb :: "moebius \<Rightarrow> moebius \<Rightarrow> moebius" is moebius_mb_rep
proof-
  fix M M' I I'
  assume "moebius_mat_eq M M'" "moebius_mat_eq I I'"
  thus "moebius_mat_eq (moebius_mb_rep I M) (moebius_mb_rep I' M')"
    by (auto simp add: similarity_scale_1 similarity_scale_2)
qed

definition similarity_invar_rep where
  "similarity_invar_rep M =
    (let M = Rep_moebius_mat M 
      in (mat_trace M)\<^sup>2 / mat_det M - 4)"

lift_definition similarity_invar :: "moebius \<Rightarrow> complex" is similarity_invar_rep
by (auto simp add: similarity_invar_rep_def Let_def power2_eq_square)

lemma
  "similarity_invar (moebius_mb I M) = similarity_invar M"
proof transfer
  fix I M
  show "similarity_invar_rep (moebius_mb_rep I M) = similarity_invar_rep M"
    using Rep_moebius_mat[of I] Rep_moebius_mat[of M]
    by (simp add: similarity_invar_rep_def Let_def mat_trace_similarity mat_det_similarity)
qed

definition similar where
 "similar M1 M2 \<longleftrightarrow> (\<exists> I. moebius_mb I M1 = M2)"

lemma [simp]: "similarity eye M = M"
unfolding similarity_def
by simp (metis eye_def mat_eye_l mat_eye_r)

lemma [simp]: "similarity (1, 0, 0, 1) M = M"
unfolding eye_def[symmetric]
by (simp del: eye_def)

lemma similarity_comp:
  assumes "mat_det I1 \<noteq> 0" "mat_det I2 \<noteq> 0"
  shows "similarity I1 (similarity I2 M) = similarity (I2*\<^sub>m\<^sub>mI1) M"
using assms
unfolding similarity_def
by (simp add: mult_mm_assoc mat_inv_mult_mm)

lemma similarity_inv:
  assumes "similarity I M1 = M2" "mat_det I \<noteq> 0"
  shows "similarity (mat_inv I) M2 = M1"
using assms
unfolding similarity_def
by simp (metis mat_eye_l mult_mm_assoc mult_mm_inv_r)

lemma similar_refl [simp]: "similar M M"
unfolding similar_def
by (rule_tac x="id_moebius" in exI) (transfer, simp, rule_tac x=1 in exI, auto)

lemma similar_sym:
  assumes "similar M1 M2"
  shows "similar M2 M1"
proof-
  from assms obtain I where "M2 = moebius_mb I M1"
    unfolding similar_def
    by auto
  hence "M1 = moebius_mb (moebius_inv I) M2"
  proof transfer
    fix M2 I M1
    assume "moebius_mat_eq M2 (moebius_mb_rep I M1)"
    then obtain k where "k \<noteq> 0" "similarity (Rep_moebius_mat I) (Rep_moebius_mat M1) = k *\<^sub>s\<^sub>m Rep_moebius_mat M2"
      by auto
    thus "moebius_mat_eq M1 (moebius_mb_rep (moebius_inv_rep I) M2)"
      using Rep_moebius_mat[of I] similarity_inv[of "Rep_moebius_mat I" "Rep_moebius_mat M1" "k *\<^sub>s\<^sub>m Rep_moebius_mat M2"]
      by (auto simp add: similarity_scale_2) (rule_tac x="1/k" in exI, simp, metis mult_sm_inv_l)
  qed
  thus ?thesis
    unfolding similar_def
    by auto
qed

lemma similar_trans:
  assumes "similar M1 M2" "similar M2 M3"
  shows "similar M1 M3"
proof-
  obtain I1 I2 where "moebius_mb I1 M1 = M2" "moebius_mb I2 M2 = M3"
    using assms
    by (auto simp add: similar_def)
  thus ?thesis
    unfolding similar_def
  proof (rule_tac x="moebius_comp I1 I2" in exI, transfer)
    fix I1 I2 M1 M2 M3
    assume "moebius_mat_eq (moebius_mb_rep I1 M1) M2"
           "moebius_mat_eq (moebius_mb_rep I2 M2) M3"
    thus "moebius_mat_eq (moebius_mb_rep (moebius_comp_rep I1 I2) M1) M3"
      using Rep_moebius_mat[of I1] Rep_moebius_mat[of I2]
      by (auto simp add: similarity_scale_2 similarity_comp) (rule_tac x="ka*k" in exI, simp)
  qed
qed


end
