theory RiemannSphere
imports HomogeneousCoordinates "~~/src/HOL/Library/Product_Vector"
begin

(* We only need continuous_on_iff so instead of including Topology_Euclidean_Space.thy we only move the next two lemmas *)
lemma Lim_within: "(f ---> l) (at a within S) \<longleftrightarrow>
    (\<forall>e >0. \<exists>d>0. \<forall>x \<in> S. 0 < dist x a \<and> dist x a  < d \<longrightarrow> dist (f x) l < e)"
  by (auto simp add: tendsto_iff eventually_at dist_nz)

lemma continuous_on_iff:
  "continuous_on s f \<longleftrightarrow>
    (\<forall>x\<in>s. \<forall>e>0. \<exists>d>0. \<forall>x'\<in>s. dist x' x < d \<longrightarrow> dist (f x') (f x) < e)"
  unfolding continuous_on_def Lim_within
  apply (intro ball_cong [OF refl] all_cong ex_cong)
  apply (rename_tac y, case_tac "y = x")
  apply simp
  apply (simp add: dist_nz)
  done


(* ---------------------------------------------------------------------------- *)
section {* Riemann sphere *}
(* ---------------------------------------------------------------------------- *)

typedef riemann_sphere = "{(x::real, y::real, z::real). x*x + y*y + z*z = 1}"
by (rule_tac x="(1, 0, 0)" in exI) simp

lemma sphere_bounds':
  assumes "x*x + y*y + z*z = (1::real)"
  shows "-1 \<le> x \<and> x \<le> 1"
proof-
  from assms have "x*x \<le> 1"
    by (smt real_minus_mult_self_le)
  hence "x\<^sup>2 \<le> 1\<^sup>2" "(- x)\<^sup>2 \<le> 1\<^sup>2"
    by (auto simp add: power2_eq_square)
  show "-1 \<le> x \<and> x \<le> 1"
  proof (cases "x \<ge> 0")
    case True
    thus ?thesis
      using square_cancel[OF `x\<^sup>2 \<le> 1\<^sup>2`]
      by simp
  next
    case False
    thus ?thesis
      using square_cancel[OF `(-x)\<^sup>2 \<le> 1\<^sup>2`]
      by simp
  qed
qed

lemma sphere_bounds:
  assumes "x*x + y*y + z*z = (1::real)"
  shows "-1 \<le> x \<and> x \<le> 1"  "-1 \<le> y \<and> y \<le> 1"  "-1 \<le> z \<and> z \<le> 1"
using assms
using sphere_bounds'[of x y z] sphere_bounds'[of y x z] sphere_bounds'[of z x y]
by (auto simp add: field_simps)

text{* Polar coords parametrization *}
lemma sphere_params_on_sphere:
  assumes "x = cos \<alpha> * cos \<beta>" "y = cos \<alpha> * sin \<beta>" "z = sin \<alpha>"
  shows "x*x + y*y + z*z = 1"
proof-
  have "x*x + y*y = (cos \<alpha> * cos \<alpha>) * (cos \<beta> * cos \<beta>) + (cos \<alpha> * cos \<alpha>) * (sin \<beta> * sin \<beta>)"
    using assms
    by simp
  hence "x*x + y*y = cos \<alpha> * cos \<alpha>"
    using sin_cos_squared_add3[of \<beta>]
    by (subst (asm) distrib_left[symmetric]) (simp add: field_simps)
  thus ?thesis
    using assms
    using sin_cos_squared_add3[of \<alpha>]
    by simp
qed

lemma sphere_params:
  assumes "x*x + y*y + z*z = 1"
  shows "x = cos (arcsin z) * cos (atan2 y x) \<and> y = cos (arcsin z) * sin (atan2 y x) \<and> z = sin (arcsin z)"
proof (cases "z=1 \<or> z = -1")
  case True
  hence "x = 0 \<and> y = 0"
    using assms
    by auto
  thus ?thesis
    using `z = 1 \<or> z = -1`
    by (auto simp add: cos_arcsin)
next
  case False
  hence "x \<noteq> 0 \<or> y \<noteq> 0"
    using assms
    by auto (metis minus_one square_eq_1_iff)
  thus ?thesis
    using sphere_bounds[OF assms] assms
    by (auto simp add: cos_arcsin cos_arctan sin_arctan power2_eq_square field_simps real_sqrt_divide atan2_def cos_periodic_pi2 cos_periodic_pi3 sin_periodic_pi3) (smt real_sqrt_abs2)+
qed

lemma ex_sphere_params:
  assumes "x*x + y*y + z*z = 1"
  shows "\<exists> \<alpha> \<beta>. x = cos \<alpha> * cos \<beta> \<and> y = cos \<alpha> * sin \<beta> \<and> z = sin \<alpha> \<and> -pi / 2 \<le> \<alpha> \<and> \<alpha> \<le> pi / 2 \<and> -pi \<le> \<beta> \<and> \<beta> < pi"
using assms arcsin_bounded[of z] sphere_bounds[of x y z]
by (rule_tac x="arcsin z" in exI, rule_tac x="atan2 y x" in exI) (simp add: sphere_params arcsin_bounded atan2_bounded)

text{* Stereographic and inverse stereographic projection *}
definition stereographic_coords :: "riemann_sphere \<Rightarrow> homo_coords"where
"stereographic_coords M = (let (x, y, z) = Rep_riemann_sphere M in 
     (if (x, y, z) \<noteq> (0, 0, 1) then 
           Abs_homo_coords (Complex x y, complex_of_real (1 - z))
      else
           Abs_homo_coords (1, 0)
     ))"

lemma stereographic_coords_rep: 
  "Rep_homo_coords (stereographic_coords M) = (let (x, y, z) = Rep_riemann_sphere M in 
     (if (x, y, z) \<noteq> (0, 0, 1) then 
           (Complex x y, complex_of_real (1 - z))
      else
           (1, 0)
     ))"
proof-
  obtain x y z where MM: "(x, y, z) = Rep_riemann_sphere M"
    by (cases "Rep_riemann_sphere M") auto
  show ?thesis
  proof (cases "(x, y, z) \<noteq> (0, 0, 1) ")
    case True
    thus ?thesis
      using MM[symmetric] Abs_homo_coords_inverse[of "(Complex x y, 1 - cor z)"]
      using Rep_riemann_sphere[of M]
      by (cases "x = 0 \<and> y = 0", cases "z=1") (auto simp add: stereographic_coords_def, metis Complex_eq_1 complex_of_real_def)
  next
    case False
    thus ?thesis
      using MM
      by (simp add: stereographic_coords_def)
  qed
qed

lift_definition stereographic :: "riemann_sphere \<Rightarrow> complex_homo" is stereographic_coords
by (simp del: homo_coords_eq_def)

definition inv_stereographic_coords :: "homo_coords \<Rightarrow> riemann_sphere" where 
  "inv_stereographic_coords z = (
     let (z1, z2) = Rep_homo_coords z 
       in if z2 = 0 then 
              Abs_riemann_sphere (0, 0, 1)
          else
             let z = z1/z2;
                 X = Re (2*z / (1 + z*cnj z));
                 Y = Im (2*z / (1 + z*cnj z));
                 Z = ((cmod z)\<^sup>2 - 1) / (1 + (cmod z)\<^sup>2)
               in Abs_riemann_sphere (X, Y, Z))"

lift_definition inv_stereographic :: "complex_homo \<Rightarrow> riemann_sphere" is inv_stereographic_coords
by (auto simp add: inv_stereographic_coords_def split_def Let_def)

lemma one_plus_square_neq_zero [simp]:
  fixes x :: real
  shows "1 + (cor x)\<^sup>2 \<noteq> 0"
  by (metis (hide_lams, no_types) of_real_1 of_real_add of_real_eq_0_iff of_real_power power_one sum_power2_eq_zero_iff zero_neq_one)

lemma Re_stereographic: "Re (2 * z / (1 + z * cnj z)) = 2 * Re z / (1 + (cmod z)\<^sup>2)"
using one_plus_square_neq_zero
by (subst complex_mult_cnj_cmod, subst Re_divide_real) (auto simp add: power2_eq_square)

lemma Im_stereographic: "Im (2 * z / (1 + z * cnj z)) = 2 * Im z / (1 + (cmod z)\<^sup>2)"
using one_plus_square_neq_zero
by (subst complex_mult_cnj_cmod, subst Im_divide_real) (auto simp add: power2_eq_square)


lemma inv_stereographic_on_sphere:
  assumes "X = Re (2*z / (1 + z*cnj z))" "Y = Im (2*z / (1 + z*cnj z))" "Z = ((cmod z)\<^sup>2 - 1) / (1 + (cmod z)\<^sup>2)"
  shows "X*X + Y*Y + Z*Z = 1"
proof-
  have "1 + (cmod z)\<^sup>2 \<noteq> 0"
    by (metis power_one realpow_two_sum_zero_iff zero_neq_one)
  thus ?thesis
    using assms
    by (simp add: Re_stereographic Im_stereographic) (cases z, simp add: power2_eq_square real_sqrt_mult[symmetric] add_divide_distrib[symmetric], simp add: field_simps)
qed

lemma inv_stereographic_coords_Rep:
  "Rep_riemann_sphere (inv_stereographic_coords z) = 
  (let (z1, z2) = Rep_homo_coords z 
       in if z2 = 0 then 
             (0, 0, 1)
          else
             let z = z1/z2;
                 X = Re (2*z / (1 + z*cnj z));
                 Y = Im (2*z / (1 + z*cnj z));
                 Z = ((cmod z)\<^sup>2 - 1) / (1 + (cmod z)\<^sup>2)
               in (X, Y, Z))"
proof-
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  show ?thesis
    proof (cases "z2 = 0")
      case True
      thus ?thesis
        using zz
        by (simp add: Let_def inv_stereographic_coords_def Abs_riemann_sphere_inverse)
    next
      case False
      thus ?thesis
        using inv_stereographic_on_sphere[of _ "z1/z2"] zz
        by (simp add: Let_def inv_stereographic_coords_def Abs_riemann_sphere_inverse)
    qed
qed

definition [simp]: "North = Abs_riemann_sphere (0, 0, 1)"

lemma stereographic_North: "stereographic x = \<infinity>\<^sub>h \<longleftrightarrow> x = North"
proof (transfer)
  fix x
  show "stereographic_coords x \<approx> inf_homo_rep \<longleftrightarrow> x = North"
  proof
    assume "x = North"
    thus "stereographic_coords x \<approx> inf_homo_rep"
      by (simp add: stereographic_coords_def Abs_riemann_sphere_inverse Abs_homo_coords_inverse)
  next
    assume *: "stereographic_coords x \<approx> inf_homo_rep"
    show "x = North"
    proof (cases "Rep_riemann_sphere x = (0, 0, 1)")
      case True
      thus ?thesis
        by auto (metis Rep_riemann_sphere_inverse)
    next
      case False
      thus ?thesis
        using *
        using Rep_riemann_sphere[of x]
        by (auto simp add: stereographic_coords_def split_def Let_def Abs_homo_coords_inverse complex_of_real_def split: split_if_asm) (metis pair_collapse)
    qed
  qed
qed

lemma stereographic_inv_stereographic':
  assumes 
  z: "z = z1/z2" and "z2 \<noteq> 0" and
  X: "X = Re (2*z / (1 + z*cnj z))" and Y: "Y = Im (2*z / (1 + z*cnj z))" and Z: "Z = ((cmod z)\<^sup>2 - 1) / (1 + (cmod z)\<^sup>2)"
  shows "\<exists> k. k \<noteq> 0 \<and> (Complex X Y, complex_of_real (1 - Z)) = k *\<^sub>s\<^sub>v (z1, z2)"
proof-
  have "1 + (cmod z)\<^sup>2 \<noteq> 0"
    by (metis one_power2 sum_power2_eq_zero_iff zero_neq_one)
  hence "cor (1 - Z) = 2 / cor (1 + (cmod z)\<^sup>2)"
    using Z
    by (simp add: field_simps complex_of_real_def)
  moreover
  have "X = 2 * Re(z) / (1 + (cmod z)\<^sup>2)"
    using X
    by (simp add: Re_stereographic)
  have "Y = 2 * Im(z) / (1 + (cmod z)\<^sup>2)"
    using Y
    by (simp add: Im_stereographic)
  have "Complex X Y = 2 * z / cor (1 + (cmod z)\<^sup>2)"
    using `1 + (cmod z)\<^sup>2 \<noteq> 0`
    by (subst `X = 2*Re(z) / (1 + (cmod z)\<^sup>2)`, subst `Y = 2*Im(z) / (1 + (cmod z)\<^sup>2)`, simp add: Complex_scale4 Complex_scale1 of_real_numeral)
  moreover
  have "1 + (cor (cmod (z1 / z2)))\<^sup>2 \<noteq> 0"
    by (rule one_plus_square_neq_zero)
  ultimately
  show ?thesis
    using `z2 \<noteq> 0` `1 + (cmod z)\<^sup>2 \<noteq> 0`
    by (simp, subst z)+
       (rule_tac x="(2 / (1 + (cor (cmod (z1 / z2)))\<^sup>2)) / z2" in exI, auto)
qed

lemma
  stereographic_inv_stereographic: 
  "stereographic (inv_stereographic z) = z"
proof transfer
  fix z
  obtain z1 z2 where zz: "Rep_homo_coords z = (z1, z2)"
    by (rule obtain_homo_coords)
  have "z \<approx> stereographic_coords (inv_stereographic_coords z)"
  proof (cases "z2 = 0")
    case True
    thus ?thesis
      using zz Rep_homo_coords[of z]
      by (simp add: stereographic_coords_def inv_stereographic_coords_Rep)
  next
    case False
    thus ?thesis
      using zz stereographic_inv_stereographic'[of "z1/z2" z1 z2]
      by (simp add: stereographic_coords_rep inv_stereographic_coords_Rep Let_def)
  qed
  thus "stereographic_coords (inv_stereographic_coords z) \<approx> z"
    by (rule homo_coords_eq_sym)
qed

lemma bij_stereographic: "bij stereographic"
unfolding bij_def inj_on_def surj_def
proof (safe)
  fix x y
  assume "stereographic x = stereographic y"
  thus "x = y"
  proof (transfer)
    fix a b
    assume *: "stereographic_coords a \<approx> stereographic_coords b"
    obtain xa ya za xb yb zb where **: "Rep_riemann_sphere a = (xa, ya, za)" "Rep_riemann_sphere b = (xb, yb, zb)"
      by (metis prod_cases3)

    show "a = b"
    proof (subst Rep_riemann_sphere_inject[symmetric])
      show "Rep_riemann_sphere a = Rep_riemann_sphere b"
      proof (cases "Rep_riemann_sphere a = (0, 0, 1)")
        case True
        thus ?thesis
          using * ** Rep_riemann_sphere[of b]
          unfolding stereographic_coords_def
          by (cases "zb=1") (auto simp add: Abs_homo_coords_inverse complex_of_real_def)
      next
        {
          fix k
          assume "xa * xa + (ya * ya + za * za) = 1" 
                 "zb * zb + (k * (k * (xa * xa)) + k * (k * (ya * ya))) = 1" 
                 "zb \<noteq> 1" "za \<noteq> 1" "k \<noteq> 0" "1 + k * za = k + zb" "k \<noteq> 1"
          hence False
            by algebra
        } note *** = this
        
        case False
        thus ?thesis
          using * ** Rep_riemann_sphere[of a]  Rep_riemann_sphere[of b]
          unfolding stereographic_coords_def
          apply (case_tac[!] "zb = 1", case_tac[!] "za = 1")
          apply (auto simp add: Abs_homo_coords_inverse complex_of_real_def)
          apply (case_tac[!] k)
          using ***
          apply (auto simp add: field_simps)
          apply (case_tac "real1 = 1")
          by auto
      qed
    qed
  qed
next
  fix a
  show "\<exists> b. a = stereographic b"
    by (rule_tac x="inv_stereographic a" in exI) (simp add: stereographic_inv_stereographic)
qed

lemma inv_stereographic_stereographic: 
  "inv_stereographic (stereographic x) = x"
using stereographic_inv_stereographic[of "stereographic x"]
using bij_stereographic
unfolding bij_def inj_on_def
by simp

lemma inv_stereographic_is_inv:
  "inv_stereographic = inv stereographic"
by (rule inv_equality[symmetric], simp_all add: inv_stereographic_stereographic stereographic_inv_stereographic)

text{* Circles on the sphere *}
type_synonym real_vec_4 = "real \<times> real \<times> real \<times> real"

fun mult_sv :: "real \<Rightarrow> real_vec_4 \<Rightarrow> real_vec_4" (infixl "*\<^sub>s\<^sub>v\<^sub>4" 100) where
  "k *\<^sub>s\<^sub>v\<^sub>4 (a, b, c, d) = (k*a, k*b, k*c, k*d)"

typedef plane_vec = "{(a::real, b::real, c::real, d::real). a \<noteq> 0 \<or> b \<noteq> 0 \<or> c \<noteq> 0 \<or> d \<noteq> 0}"
by (rule_tac x="(1, 1, 1, 1)" in exI) simp

definition plane_vec_eq where 
  "plane_vec_eq v1 v2 \<longleftrightarrow> (\<exists> k. k \<noteq> 0 \<and> Rep_plane_vec v2 = k *\<^sub>s\<^sub>v\<^sub>4 Rep_plane_vec v1)"

lemma [simp]: "1 *\<^sub>s\<^sub>v\<^sub>4 x = x"
by (cases x) simp

lemma [simp]: "x *\<^sub>s\<^sub>v\<^sub>4 (y *\<^sub>s\<^sub>v\<^sub>4 v) = (x*y) *\<^sub>s\<^sub>v\<^sub>4 v"
by (cases v) simp

quotient_type plane = plane_vec / plane_vec_eq
proof (rule equivpI)
  show "reflp plane_vec_eq"
    unfolding reflp_def
    by (auto simp add: plane_vec_eq_def) (rule_tac x="1" in exI, simp)
next
  show "symp plane_vec_eq"
    unfolding symp_def
    by (auto simp add: plane_vec_eq_def) (rule_tac x="1/k" in exI, simp)
next
  show "transp plane_vec_eq"
    unfolding transp_def
    by (auto simp add: plane_vec_eq_def) (rule_tac x="ka*k" in exI, simp)
qed

definition on_sphere_circle_rep where 
  "on_sphere_circle_rep \<alpha> A \<longleftrightarrow> 
      (let (X, Y, Z) = Rep_riemann_sphere A;
           (a, b, c, d) = Rep_plane_vec \<alpha>
        in a*X + b*Y + c*Z + d = 0)"

lift_definition on_sphere_circle :: "plane \<Rightarrow> riemann_sphere \<Rightarrow> bool" is on_sphere_circle_rep
proof-
  fix v1 v2
  obtain a1 b1 c1 d1 where vv1: "Rep_plane_vec v1 = (a1, b1, c1, d1)"
    by (cases "Rep_plane_vec v1") auto
  obtain a2 b2 c2 d2 where vv2: "Rep_plane_vec v2 = (a2, b2, c2, d2)"
    by (cases "Rep_plane_vec v2") auto
  assume "plane_vec_eq v1 v2"
  then obtain k where *: "a2 = k*a1" "b2 = k*b1" "c2 = k*c1" "d2 = k*d1" "k \<noteq> 0"
    using vv1 vv2 
    by (auto simp add: plane_vec_eq_def)
  show "on_sphere_circle_rep v1 = on_sphere_circle_rep v2"
  proof (rule ext)
    fix M
    obtain x y z where MM: "Rep_riemann_sphere M = (x, y, z)"
      by (cases "Rep_riemann_sphere M") auto
    have "k * a1 * x + k * b1 * y + k * c1 * z + k * d1 = k*(a1*x + b1*y + c1*z + d1)"
      by (simp add: field_simps)
    thus "on_sphere_circle_rep v1 M = on_sphere_circle_rep v2 M"
      using vv1 vv2 MM *
      by (auto simp add: plane_vec_eq_def on_sphere_circle_rep_def split_def Let_def)
  qed
qed

definition sphere_circle_set where
  "sphere_circle_set \<alpha> = {A. on_sphere_circle \<alpha> A}"

text{* Distance on the Riemann sphere *}
definition dist_riemann_sphere' where
  "dist_riemann_sphere' M1 M2 = 
     (let (x1, y1, z1) = Rep_riemann_sphere M1;
          (x2, y2, z2) = Rep_riemann_sphere M2
       in norm (x1 - x2, y1 - y2, z1 - z2))"

lemma dist_riemann_sphere'_inner:
  "(dist_riemann_sphere' M1 M2)\<^sup>2 = 2 - 2 * inner (Rep_riemann_sphere M1) (Rep_riemann_sphere M2)"
using Rep_riemann_sphere[of M1] Rep_riemann_sphere[of M2]
unfolding dist_riemann_sphere'_def
by (auto simp add: norm_prod_def) (simp add: power2_eq_square field_simps)

lemma xxx [simp]: 
  "Re (2 * m1 / (1 + cor ((cmod m1)\<^sup>2))) = 2 * Re m1 / (1 + (cmod m1)\<^sup>2)"
apply (subst Re_divide_real)
apply (simp add: power2_eq_square)
apply (metis numeral_One of_real_1 of_real_add of_real_eq_0_iff power_one sum_power2_eq_zero_iff zero_neq_numeral)
apply (simp add: power2_eq_square)
done

lemma yyy [simp]: 
  "Im (2 * m1 / (1 + cor ((cmod m1)\<^sup>2))) = 2 * Im m1 / (1 + (cmod m1)\<^sup>2)"
apply (subst Im_divide_real)
apply (simp add: power2_eq_square)
apply (metis numeral_One of_real_1 of_real_add of_real_eq_0_iff power_one sum_power2_eq_zero_iff zero_neq_numeral)
apply (simp add: power2_eq_square)
done

lemma dist_riemann_sphere'_ge_0 [simp]: "dist_riemann_sphere' M1 M2 \<ge> 0"
using norm_ge_zero
unfolding dist_riemann_sphere'_def
by (simp add: split_def Let_def)

lemma dist_homo_stereographic_finite:
  assumes "stereographic M1 = of_complex m1"  "stereographic M2 = of_complex m2"
  shows "dist_riemann_sphere' M1 M2 = 2 * cmod (m1 - m2) / (sqrt (1 + (cmod m1)\<^sup>2) * sqrt (1 + (cmod m2)\<^sup>2))"
proof-
  obtain x1 y1 z1 x2 y2 z2 where MM: "(x1, y1, z1) = Rep_riemann_sphere M1" "(x2, y2, z2) = Rep_riemann_sphere M2"
    by (cases "Rep_riemann_sphere M1", cases "Rep_riemann_sphere M2", auto, blast)
  have *: "M1 = inv_stereographic (of_complex m1)"  "M2 = inv_stereographic (of_complex m2)"
    using inv_stereographic_is_inv assms
    by (metis inv_stereographic_stereographic)+
  have "(1 + (cmod m1)\<^sup>2) \<noteq> 0"  "(1 + (cmod m2)\<^sup>2) \<noteq> 0"
    by (metis power_one realpow_two_sum_zero_iff zero_neq_one)+
  have "(1 + (cmod m1)\<^sup>2) > 0"  "(1 + (cmod m2)\<^sup>2) > 0"
    by (smt realpow_square_minus_le)+
  hence "(1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2) > 0"
    by (metis norm_mult_less norm_zero power2_eq_square zero_power2)
  hence "sqrt ((1 + cmod m1 * cmod m1) * (1 + cmod m2 * cmod m2)) > 0"
    using real_sqrt_gt_0_iff
    by (simp add: power2_eq_square)
  hence **: "(2 * cmod (m1 - m2) / sqrt ((1 + cmod m1 * cmod m1) * (1 + cmod m2 * cmod m2))) \<ge> 0 \<longleftrightarrow> cmod (m1 - m2) \<ge> 0"
    by (metis diff_self divide_nonneg_pos mult_2 norm_ge_zero norm_triangle_ineq4 norm_zero)

  have "(dist_riemann_sphere' M1 M2)\<^sup>2 * (1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2) = 4 * (cmod (m1 - m2))\<^sup>2"
    apply (subst *)+
  proof transfer
    fix m1 m2
    have "(1 + (cmod m1)\<^sup>2) \<noteq> 0"  "(1 + (cmod m2)\<^sup>2) \<noteq> 0"
      by (metis power_one realpow_two_sum_zero_iff zero_neq_one)+
    thus "(dist_riemann_sphere' (inv_stereographic_coords (of_complex_coords m1)) (inv_stereographic_coords (of_complex_coords m2)))\<^sup>2  * (1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2)= 4 * (cmod (m1 - m2))\<^sup>2"
      apply (simp add: dist_riemann_sphere'_inner inv_stereographic_coords_Rep complex_mult_cnj_cmod)
      apply (subst cor_squared)+
      apply (subst xxx)+
      apply (subst yyy)+
      apply (subst left_diff_distrib[of 2])
      apply (subst left_diff_distrib[of "2*(1+(cmod m1)\<^sup>2)"])
      apply (subst distrib_right[of _ _ "(1 + (cmod m1)\<^sup>2)"]) 
      apply (subst distrib_right[of _ _ "(1 + (cmod m1)\<^sup>2)"]) 
      apply (subst distrib_right[of "2 * (2 * Re m1 / (1 + (cmod m1)\<^sup>2) * (2 * Re m2 / (1 + (cmod m2)\<^sup>2))) * (1 + (cmod m1)\<^sup>2)" _ "(1 + (cmod m2)\<^sup>2)"]) 
      apply (subst distrib_right[of "2 * (2 * Im m1 / (1 + (cmod m1)\<^sup>2) * (2 * Im m2 / (1 + (cmod m2)\<^sup>2))) * (1 + (cmod m1)\<^sup>2)" _ "(1 + (cmod m2)\<^sup>2)"])
      apply simp
      apply (subst (asm) cmod_square)+
      apply (subst cmod_square)+
      apply (simp add: field_simps)
      done
  qed
  hence "(dist_riemann_sphere' M1 M2)\<^sup>2 = 4 * (cmod (m1 - m2))\<^sup>2 / ((1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2))"
    using `(1 + (cmod m1)\<^sup>2) \<noteq> 0`  `(1 + (cmod m2)\<^sup>2) \<noteq> 0`
    using eq_divide_imp[of "(1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2)" "(dist_riemann_sphere' M1 M2)\<^sup>2" "4 * (cmod (m1 - m2))\<^sup>2"]
    by simp
  thus "dist_riemann_sphere' M1 M2 = 2 * cmod (m1 - m2) / (sqrt (1 + (cmod m1)\<^sup>2) * sqrt (1 + (cmod m2)\<^sup>2))"
    using power2_eq_iff[of "dist_riemann_sphere' M1 M2" "2 * (cmod (m1 - m2)) / sqrt ((1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2))"]
    using `(1 + (cmod m1)\<^sup>2) * (1 + (cmod m2)\<^sup>2) > 0`  `(1 + (cmod m1)\<^sup>2) > 0` `(1 + (cmod m2)\<^sup>2) > 0`
    apply (auto simp add: power2_eq_square real_sqrt_mult[symmetric])
    using dist_riemann_sphere'_ge_0[of M1 M2] **
    by simp
qed

lemma dist_homo_stereographic_infinite:
  assumes "stereographic M1 = \<infinity>\<^sub>h"  "stereographic M2 = of_complex m2"
  shows "dist_riemann_sphere' M1 M2 = 2 / sqrt (1 + (cmod m2)\<^sup>2)"
proof-
  obtain x2 y2 z2 where MM: "(0, 0, 1) = Rep_riemann_sphere M1" "(x2, y2, z2) = Rep_riemann_sphere M2"
    using `stereographic M1 = \<infinity>\<^sub>h`
    using stereographic_North[of M1]
    by (cases "Rep_riemann_sphere M2", auto simp add: Abs_riemann_sphere_inverse)
  have *: "M1 = inv_stereographic \<infinity>\<^sub>h"  "M2 = inv_stereographic (of_complex m2)"
    using inv_stereographic_is_inv assms
    by (metis inv_stereographic_stereographic)+
  have "(1 + (cmod m2)\<^sup>2) \<noteq> 0"
    by (metis power_one realpow_two_sum_zero_iff zero_neq_one)+
  have "(1 + (cmod m2)\<^sup>2) > 0"
    by (smt realpow_square_minus_le)+
  hence "sqrt (1 + cmod m2 * cmod m2) > 0"
    using real_sqrt_gt_0_iff
    by (simp add: power2_eq_square)
  hence **: "2 / sqrt (1 + cmod m2 * cmod m2) > 0"
    by simp

  have "(dist_riemann_sphere' M1 M2)\<^sup>2 * (1 + (cmod m2)\<^sup>2) = 4"
    apply (subst *)+
  proof transfer
    fix m2
    have "(1 + (cmod m2)\<^sup>2) \<noteq> 0"
      by (metis power_one realpow_two_sum_zero_iff zero_neq_one)
    thus "(dist_riemann_sphere' (inv_stereographic_coords inf_homo_rep) (inv_stereographic_coords (of_complex_coords m2)))\<^sup>2 * (1 + (cmod m2)\<^sup>2) = 4"
      by (simp add: dist_riemann_sphere'_inner inv_stereographic_coords_Rep complex_mult_cnj_cmod)
         (subst left_diff_distrib[of 2], simp)
  qed
  hence "(dist_riemann_sphere' M1 M2)\<^sup>2 = 4 / (1 + (cmod m2)\<^sup>2)"
    using `(1 + (cmod m2)\<^sup>2) \<noteq> 0`
    by (simp add: field_simps)
  thus "dist_riemann_sphere' M1 M2 = 2 / sqrt (1 + (cmod m2)\<^sup>2)"
    using power2_eq_iff[of "dist_riemann_sphere' M1 M2" "2 / sqrt (1 + (cmod m2)\<^sup>2)"]
    using `(1 + (cmod m2)\<^sup>2) > 0`
    apply (auto simp add: power2_eq_square real_sqrt_mult[symmetric])
    using dist_riemann_sphere'_ge_0[of M1 M2] **
    by simp
qed

lemma dist_riemann_sphere'_sym: "dist_riemann_sphere' M1 M2 = dist_riemann_sphere' M2 M1"
proof-
  obtain x1 y1 z1 x2 y2 z2 where MM: "(x1, y1, z1) = Rep_riemann_sphere M1" "(x2, y2, z2) = Rep_riemann_sphere M2"
    by (cases "Rep_riemann_sphere M1", cases "Rep_riemann_sphere M2", auto, blast)
  show ?thesis
    unfolding dist_riemann_sphere'_def 
    using norm_minus_cancel[of "(x1 - x2, y1 - y2, z1 - z2)"] MM[symmetric]
    by simp
qed

lemma dist_homo_stereographic: "dist_riemann_sphere' M1 M2 = dist_homo (stereographic M1) (stereographic M2)"
proof (cases "M1 = North")
  case True
  hence "stereographic M1 = \<infinity>\<^sub>h"
    by (simp add: stereographic_North)
  show ?thesis
  proof (cases "M2 = North")
    case True
    show ?thesis
      using `M1 = North` `M2 = North`
      by (auto simp add: Abs_riemann_sphere_inverse dist_riemann_sphere'_def norm_prod_def)
  next
    case False
    hence "stereographic M2 \<noteq> \<infinity>\<^sub>h"
      using stereographic_North[of M2]
      by simp
    then obtain m2 where "stereographic M2 = of_complex m2"
      using inf_homo_or_complex_homo[of "stereographic M2"]
      by auto
    show ?thesis
      using `stereographic M2 = of_complex m2` `stereographic M1 = \<infinity>\<^sub>h`
      using dist_homo_infinite1 dist_homo_stereographic_infinite
      by simp
  qed
next
  case False
  hence "stereographic M1 \<noteq> \<infinity>\<^sub>h"
    by (simp add: stereographic_North)
  then obtain m1 where "stereographic M1 = of_complex m1"
    using inf_homo_or_complex_homo[of "stereographic M1"]
    by auto
  show ?thesis
  proof (cases "M2 = North")
    case True
    hence "stereographic M2 = \<infinity>\<^sub>h"
      by (simp add: stereographic_North)
    show ?thesis
      using `stereographic M1 = of_complex m1` `stereographic M2 = \<infinity>\<^sub>h`
      using dist_homo_infinite2 dist_homo_stereographic_infinite
      by (subst dist_riemann_sphere'_sym, simp)
  next
    case False
    hence "stereographic M2 \<noteq> \<infinity>\<^sub>h"
      by (simp add: stereographic_North)
    then obtain m2 where "stereographic M2 = of_complex m2"
      using inf_homo_or_complex_homo[of "stereographic M2"]
      by auto
    show ?thesis
      using `stereographic M1 = of_complex m1` `stereographic M2 = of_complex m2`
      using dist_homo_finite dist_homo_stereographic_finite
      by simp
  qed
qed

lemma dist_homo_stereographic':
  "dist_homo A B = dist_riemann_sphere' (inv_stereographic A) (inv_stereographic B)"
by (subst dist_homo_stereographic) (metis stereographic_inv_stereographic)

instantiation riemann_sphere :: metric_space
begin
definition "dist_riemann_sphere = dist_riemann_sphere'"
definition "open_riemann_sphere S = (\<forall>x\<in>S. \<exists>e>0. \<forall>y. dist_riemann_sphere' y x < e \<longrightarrow> y \<in> S)"
instance
proof
  fix x y :: riemann_sphere
  show "(dist x y = 0) = (x = y)"
  proof-
    obtain x1 y1 z1 x2 y2 z2 where MM: "(x1, y1, z1) = Rep_riemann_sphere x" "(x2, y2, z2) = Rep_riemann_sphere y"
      by (cases "Rep_riemann_sphere x", cases "Rep_riemann_sphere y", auto, blast)
    show ?thesis
      unfolding dist_riemann_sphere_def
      using norm_eq_zero[of "(x1 - y2, y1 - y2, z1 - z2)"] MM[symmetric] Rep_riemann_sphere_inject[of x y]
      by (simp add: dist_riemann_sphere'_def) (smt prod.inject zero_prod_def)
  qed
next
  fix S :: "riemann_sphere set"
  show "open S = (\<forall>x\<in>S. \<exists>e>0. \<forall>y. dist y x < e \<longrightarrow> y \<in> S)"
    unfolding open_riemann_sphere_def dist_riemann_sphere_def
    by simp
next
  fix x y z :: riemann_sphere
  show "dist x y \<le> dist x z + dist y z"
  proof-
    obtain x1 y1 z1 x2 y2 z2 x3 y3 z3 where MM: "(x1, y1, z1) = Rep_riemann_sphere x" "(x2, y2, z2) = Rep_riemann_sphere y" "(x3, y3, z3) = Rep_riemann_sphere z"
      by (cases "Rep_riemann_sphere x", cases "Rep_riemann_sphere y", cases "Rep_riemann_sphere z", auto, blast)
    show ?thesis
      unfolding dist_riemann_sphere_def
      using MM[symmetric] norm_minus_cancel[of "(x3 - x2, y3 - y2, z3 - z2)"] norm_triangle_ineq[of "(x1 - x3, y1 - y3, z1 - z3)" "(x3 - x2, y3 - y2, z3 - z2)"]
      by (simp add: dist_riemann_sphere'_def field_simps)
  qed
qed

end

lemma ex_cos_gt':
  assumes "a \<ge> 0" "a < 1" "-pi/2 \<le> \<alpha> \<and> \<alpha> \<le> pi/2"
  shows "\<exists> \<alpha>'. -pi/2 \<le> \<alpha>' \<and> \<alpha>' \<le> pi/2 \<and> \<alpha>' \<noteq> \<alpha> \<and> cos (\<alpha> - \<alpha>') = a"
proof-
  have "arccos a > 0" "arccos a \<le> pi/2"
    using `a \<ge> 0` `a < 1`
    using arccos_lt_bounded arccos_le_pi2
    by auto
    
  show ?thesis
  proof (cases "\<alpha> - arccos a \<ge> - pi/2")
    case True
    thus ?thesis
      using assms `arccos a > 0` `arccos a \<le> pi/2`
      by (rule_tac x = "\<alpha> - arccos a" in exI) auto
  next
    case False
    thus ?thesis
      using assms `arccos a > 0` `arccos a \<le> pi/2`
      by (rule_tac x = "\<alpha> + arccos a" in exI) auto
  qed
qed

lemma ex_cos_gt:
  assumes "a < 1" "-pi/2 \<le> \<alpha> \<and> \<alpha> \<le> pi/2"
  shows "\<exists> \<alpha>'. -pi/2 \<le> \<alpha>' \<and> \<alpha>' \<le> pi/2 \<and> \<alpha>' \<noteq> \<alpha> \<and> cos (\<alpha> - \<alpha>') > a"
proof-
  have "\<exists> a'. a' \<ge> 0 \<and> a' > a \<and> a' < 1"
    using `a < 1`
    using divide_strict_right_mono[of "2*a + (1 - a)" 2 2]
    by (rule_tac x="if a < 0 then 0 else a + (1-a)/2" in exI) (auto simp add: field_simps)
  then obtain a' where "a' \<ge> 0" "a' > a" "a' < 1"
    by auto
  thus ?thesis
    using ex_cos_gt'[of a' \<alpha>] assms
    by auto
qed

instantiation riemann_sphere :: perfect_space
begin
instance proof
  fix M :: riemann_sphere
  obtain x y z where MM: "Rep_riemann_sphere M = (x, y, z)"
    by (cases "Rep_riemann_sphere M") auto
  then obtain \<alpha> \<beta> where *: "x = cos \<alpha> * cos \<beta>" "y = cos \<alpha> * sin \<beta>" "z = sin \<alpha>" "-pi / 2 \<le> \<alpha> \<and> \<alpha> \<le> pi / 2"
    using Rep_riemann_sphere[of M]
    using ex_sphere_params[of x y z]
    by auto
  show "\<not> open {M}"
    unfolding open_riemann_sphere_def
  proof auto
    fix e :: real
    assume "e > 0"
    then obtain \<alpha>' where "1 - (e*e/2) < cos (\<alpha> - \<alpha>')" "\<alpha> \<noteq> \<alpha>'" "-pi/2 \<le> \<alpha>'" "\<alpha>' \<le> pi/2" 
      using ex_cos_gt[of "1 - (e*e/2)" \<alpha>] `- pi / 2 \<le> \<alpha> \<and> \<alpha> \<le> pi / 2`
      by (auto simp add: mult_pos_pos)
    hence "sin \<alpha> \<noteq> sin \<alpha>'"
      using `-pi / 2 \<le> \<alpha> \<and> \<alpha> \<le> pi / 2` sin_inj[of \<alpha> \<alpha>']
      by auto

    have "2 - 2 * cos (\<alpha> - \<alpha>') < e*e"
      using mult_strict_right_mono[OF `1 - (e*e/2) < cos (\<alpha> - \<alpha>')`, of 2]
      by (simp add: field_simps)
    have "2 - 2 * cos (\<alpha> - \<alpha>') \<ge> 0"
      using cos_le_one[of "\<alpha> - \<alpha>'"]
      by (simp add: sign_simps)
    let ?M' = "Abs_riemann_sphere (cos \<alpha>' * cos \<beta>,  cos \<alpha>' * sin \<beta>, sin \<alpha>')"
    have "dist_riemann_sphere' M ?M' = sqrt ((cos \<alpha> - cos \<alpha>')\<^sup>2 + (sin \<alpha> - sin \<alpha>')\<^sup>2)"
      using MM * sphere_params_on_sphere[of _ \<alpha>' \<beta>]
      using sin_cos_squared_add[of \<beta>]
      apply (simp add: dist_riemann_sphere'_def Abs_riemann_sphere_inverse norm_prod_def)
      apply (subst left_diff_distrib[symmetric])+
      apply (subst power_mult_distrib)+
      apply (subst distrib_left[symmetric])
      apply simp
      done
    also have "... = sqrt (2 - 2*cos (\<alpha> - \<alpha>'))"
      by (simp add: power2_eq_square field_simps cos_diff)
    finally
    have "(dist_riemann_sphere' M ?M')\<^sup>2 = 2 - 2*cos (\<alpha> - \<alpha>')"
      using `2 - 2 * cos (\<alpha> - \<alpha>') \<ge> 0`
      by simp
    hence "(dist_riemann_sphere' M ?M')\<^sup>2 < e\<^sup>2"
      using `2 - 2 * cos (\<alpha> - \<alpha>') < e*e`
      by (simp add: power2_eq_square)
    hence "dist_riemann_sphere' M ?M' < e"
      apply (rule power2_less_imp_less)
      using `e > 0`
      by simp
    moreover
    have "M \<noteq> ?M'"
      apply (subst Rep_riemann_sphere_inverse[symmetric])
      using Abs_riemann_sphere_inject[of "Rep_riemann_sphere M" "(cos \<alpha>' * cos \<beta>, cos \<alpha>' * sin \<beta>, sin \<alpha>')" ]
      using MM MM[symmetric] * sphere_params_on_sphere[of _ \<alpha>' \<beta>] Rep_riemann_sphere[of M] `sin \<alpha> \<noteq> sin \<alpha>'`
      by (simp add: Abs_riemann_sphere_inverse)
    ultimately
    show "\<exists>y. dist_riemann_sphere' y M < e \<and> y \<noteq> M"
      by (rule_tac x="?M'" in exI) (simp add: dist_riemann_sphere'_sym)
  qed
qed
end

instantiation complex_homo :: perfect_space
begin
instance proof
  fix x::complex_homo
  show "\<not> open {x}"
    unfolding open_complex_homo_def[of "{x}"]
  proof (auto)
    fix e::real
    assume "e > 0"
    thus "\<exists> y. dist_homo y x < e \<and> y \<noteq> x"
      using not_open_singleton[of "inv_stereographic x"]
      unfolding open_riemann_sphere_def[of "{inv_stereographic x}"]
      apply (subst dist_homo_stereographic', auto)
      apply (erule_tac x=e in allE, auto)
      apply (rule_tac x="stereographic y" in exI, auto simp add: inv_stereographic_stereographic)
      done
  qed
qed

end

lemma "continuous_on UNIV stereographic"
unfolding continuous_on_iff
unfolding dist_complex_homo_def dist_riemann_sphere_def
by (subst dist_homo_stereographic', auto simp add: inv_stereographic_stereographic)

lemma "continuous_on UNIV inv_stereographic"
unfolding continuous_on_iff
unfolding dist_complex_homo_def dist_riemann_sphere_def
by (subst dist_homo_stereographic) (auto simp add: stereographic_inv_stereographic)

end
