section{* Vectors, Matrices *}
theory Matrices
imports MoreComplex LinearSystems Quadratic
begin

(* ---------------------------------------------------------------------------- *)
subsection{* Vectors *}
(* ---------------------------------------------------------------------------- *)

text{* Type of complex vector *}
type_synonym complex_vec = "complex \<times> complex"

definition vec_zero :: "complex_vec" where
  [simp]: "vec_zero = (0, 0)"

text{* Vector scalar multiplication *}
fun mult_sv :: "complex \<Rightarrow> complex_vec \<Rightarrow> complex_vec" (infixl "*\<^sub>s\<^sub>v" 100) where
  "k *\<^sub>s\<^sub>v (x, y) = (k*x, k*y)"

lemma fst_mult_sv [simp]: "fst (k *\<^sub>s\<^sub>v v) = k * fst v"
by (cases v) simp

lemma snd_mult_sv [simp]: "snd (k *\<^sub>s\<^sub>v v) = k * snd v"
by (cases v) simp

lemma mult_sv_mult_sv [simp]: "k1 *\<^sub>s\<^sub>v (k2 *\<^sub>s\<^sub>v v) = (k1*k2) *\<^sub>s\<^sub>v v"
by (cases v) simp

lemma one_mult_sv [simp]: "1 *\<^sub>s\<^sub>v v =  v"
by (cases v) simp

lemma mult_sv_ex_id1 [simp]: "\<exists> k::complex. k \<noteq> 0 \<and> k *\<^sub>s\<^sub>v v = v"
  by (rule_tac x=1 in exI, simp)

lemma mult_sv_ex_id2 [simp]: "\<exists> k::complex. k \<noteq> 0 \<and> v = k *\<^sub>s\<^sub>v v"
  by (rule_tac x=1 in exI, simp)

text{* Multiplication of two vectors *}
fun mult_vv :: "complex \<times> complex \<Rightarrow> complex \<times> complex \<Rightarrow> complex" (infixl "*\<^sub>v\<^sub>v" 100) where
 "(x, y) *\<^sub>v\<^sub>v (a, b) = x*a + y*b"

lemma mult_vv_commute: "v1 *\<^sub>v\<^sub>v v2 = v2 *\<^sub>v\<^sub>v v1"
by (cases v1, cases v2) auto

lemma mult_vv_scale_sv1:
  "(k *\<^sub>s\<^sub>v v1) *\<^sub>v\<^sub>v v2 = k * (v1 *\<^sub>v\<^sub>v v2)"
by (cases v1, cases v2) (auto simp add: field_simps)

lemma mult_vv_scale_sv2:
  "v1 *\<^sub>v\<^sub>v (k *\<^sub>s\<^sub>v v2) = k * (v1 *\<^sub>v\<^sub>v v2)"
by (cases v1, cases v2) (auto simp add: field_simps)

text{* Conjugate vector *}
fun vec_map where
 "vec_map f (x, y) = (f x, f y)"

definition vec_cnj where "vec_cnj = vec_map cnj"

lemma vec_cnj_vec_cnj [simp]: "vec_cnj (vec_cnj v) = v"
by (cases v) (simp add: vec_cnj_def)

lemma cnj_mult_vv: "cnj (v1 *\<^sub>v\<^sub>v v2) = (vec_cnj v1) *\<^sub>v\<^sub>v (vec_cnj v2)"
by (cases v1, cases v2) (simp add: vec_cnj_def)

lemma vec_cnj_sv [simp]: "vec_cnj (k *\<^sub>s\<^sub>v A) = cnj k *\<^sub>s\<^sub>v vec_cnj A"
by (cases A) (auto simp add: vec_cnj_def)


lemma scalsquare_vv_zero:
  "(vec_cnj v) *\<^sub>v\<^sub>v v = 0 \<longleftrightarrow> v = vec_zero"
apply (cases v)
  apply (auto simp add: vec_cnj_def field_simps complex_mult_cnj_cmod power2_eq_square)
  apply (simp only: cor_add[symmetric] cor_mult[symmetric] of_real_eq_0_iff, simp)+
done

(* ---------------------------------------------------------------------------- *)
subsection{* Matrices *}
(* ---------------------------------------------------------------------------- *)

text{* Type of complex matrices *}
type_synonym complex_mat = "complex \<times> complex \<times> complex \<times> complex"

text{* Matrix scalar multiplication *}
fun mult_sm :: "complex \<Rightarrow> complex_mat \<Rightarrow> complex_mat" (infixl "*\<^sub>s\<^sub>m" 100) where
  "k *\<^sub>s\<^sub>m (a, b, c, d) = (k*a, k*b, k*c, k*d)"

lemma [simp]: "k1 *\<^sub>s\<^sub>m (k2 *\<^sub>s\<^sub>m A) = (k1*k2) *\<^sub>s\<^sub>m A"
by (cases A) auto

lemma [simp]: "1 *\<^sub>s\<^sub>m A = A"
  by (cases A) auto

lemma mult_sm_inv_l:
  assumes "k \<noteq> 0" "k *\<^sub>s\<^sub>m A = B"
  shows "A = (1/k) *\<^sub>s\<^sub>m B"
using assms
by auto

lemma mult_sm_ex_id1 [simp]: "\<exists> k::complex. k \<noteq> 0 \<and> k *\<^sub>s\<^sub>m M = M"
  by (rule_tac x=1 in exI, simp)

lemma mult_sm_ex_id2 [simp]: "\<exists> k::complex. k \<noteq> 0 \<and> M = k *\<^sub>s\<^sub>m M"
  by (rule_tac x=1 in exI, simp)

text{* Matrix addition and subtraction *}
definition mat_zero :: "complex_mat" where [simp]: "mat_zero = (0, 0, 0, 0)"

fun mat_plus :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> complex_mat" (infixl "+\<^sub>m\<^sub>m" 100) where
  "mat_plus (a1, b1, c1, d1) (a2, b2, c2, d2) = (a1+a2, b1+b2, c1+c2, d1+d2)"

fun mat_minus :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> complex_mat" (infixl "-\<^sub>m\<^sub>m" 100) where
  "mat_minus (a1, b1, c1, d1) (a2, b2, c2, d2) = (a1-a2, b1-b2, c1-c2, d1-d2)"

fun mat_uminus :: "complex_mat \<Rightarrow> complex_mat" where
  "mat_uminus (a, b, c, d) = (-a, -b, -c, -d)"

(*
interpretation complex_mat_add_group: ab_group_add mat_plus mat_zero mat_minus mat_uminus
by (unfold_locales) (case_tac[!] a, case_tac[!] b, case_tac[!] c, auto)
*)

(*
interpretation vector_space "mult_sm :: complex \<Rightarrow> complex_mat \<Rightarrow> complex_mat"
*)

lemma nonzero_mult_real:
  assumes "A \<noteq> mat_zero" "k \<noteq> 0"
  shows "k *\<^sub>s\<^sub>m A \<noteq> mat_zero"
using assms
by (cases A) simp

text{* Matrix multiplication *}
fun mult_mm :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> complex_mat" (infixl "*\<^sub>m\<^sub>m" 100) where
  "(a1, b1, c1, d1) *\<^sub>m\<^sub>m (a2, b2, c2, d2) =
   (a1*a2 + b1*c2, a1*b2 + b1*d2, c1*a2+d1*c2, c1*b2+d1*d2)"

lemma mult_mm_assoc: "A *\<^sub>m\<^sub>m (B *\<^sub>m\<^sub>m C) = (A *\<^sub>m\<^sub>m B) *\<^sub>m\<^sub>m C"
by (cases A, cases B, cases C) (auto simp add: field_simps)

lemma mult_assoc_5: "A *\<^sub>m\<^sub>m (B *\<^sub>m\<^sub>m C *\<^sub>m\<^sub>m D) *\<^sub>m\<^sub>m E = (A *\<^sub>m\<^sub>m B) *\<^sub>m\<^sub>m C *\<^sub>m\<^sub>m (D *\<^sub>m\<^sub>m E)"
by (simp only: mult_mm_assoc)

lemma mat_zero_r [simp]: "A *\<^sub>m\<^sub>m mat_zero = mat_zero"
  by (cases A) simp

lemma mat_zero_l [simp]: "mat_zero *\<^sub>m\<^sub>m A = mat_zero"
  by (cases A) simp

definition eye :: "complex_mat" where
  [simp]: "eye = (1, 0, 0, 1)"

lemma mat_eye_l:
  "eye *\<^sub>m\<^sub>m A = A"
by (cases A) auto

lemma mat_eye_r:
  "A *\<^sub>m\<^sub>m eye = A"
by (cases A) auto

lemma mult_mm_sm [simp]: "A *\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m B) = k *\<^sub>s\<^sub>m (A *\<^sub>m\<^sub>m B)"
  by (cases A, cases B) (simp add: field_simps)

lemma mult_sm_mm [simp]: "(k *\<^sub>s\<^sub>m A) *\<^sub>m\<^sub>m B = k *\<^sub>s\<^sub>m (A *\<^sub>m\<^sub>m B)"
  by (cases A, cases B) (simp add: field_simps)

lemma mult_sm_eye_mm [simp]: "k *\<^sub>s\<^sub>m eye *\<^sub>m\<^sub>m A = k *\<^sub>s\<^sub>m A"
by (cases A) simp

text{* Matrix determinant *}
fun mat_det where "mat_det (a, b, c, d) = a*d - b*c"

lemma mat_det_mult [simp]: "mat_det (A *\<^sub>m\<^sub>m B) = mat_det A * mat_det B"
by (cases A, cases B) (auto simp add: field_simps)

lemma mat_det_mult_sm [simp]: "mat_det (k *\<^sub>s\<^sub>m A) = (k*k) * mat_det A"
by (cases A) (auto simp add: field_simps)

text{* Matrix inverse *}
fun mat_inv :: "complex_mat \<Rightarrow> complex_mat" where
  "mat_inv (a, b, c, d) = (1/(a*d - b*c)) *\<^sub>s\<^sub>m (d, -b, -c, a)"

lemma mat_inv_r:
  assumes "mat_det A \<noteq> 0"
  shows "A *\<^sub>m\<^sub>m (mat_inv A) = eye"
  using assms
proof (cases A, auto simp add: field_simps)
  fix a b c d :: complex
  assume "a * (a * (d * d)) + b * (b * (c * c)) = a * (b * (c * (d * 2)))"
  hence "(a*d - b*c)*(a*d - b*c) = 0"
    by (auto simp add: field_simps)
  hence *: "a*d - b*c = 0"
    by auto
  assume "a*d \<noteq> b*c"
  with * show False
    by auto
qed

lemma mat_inv_l:
  assumes "mat_det A \<noteq> 0"
  shows "(mat_inv A) *\<^sub>m\<^sub>m A  = eye"
  using assms
proof (cases A, auto simp add: field_simps)
  fix a b c d :: complex
  assume "a * (a * (d * d)) + b * (b * (c * c)) = a * (b * (c * (d * 2)))"
  hence "(a*d - b*c)*(a*d - b*c) = 0"
    by (auto simp add: field_simps)
  hence *: "a*d - b*c = 0"
    by auto
  assume "a*d \<noteq> b*c"
  with * show False
    by auto
qed

lemma mat_det_inv:
  assumes "mat_det A \<noteq> 0"
  shows "mat_det (mat_inv A) = 1 / mat_det A"
proof-
  have "mat_det eye = mat_det A * mat_det (mat_inv A)"
    using mat_inv_l[OF assms, symmetric]
    by simp
  thus ?thesis
    using assms
    by (simp add: field_simps)
qed

lemma mult_mm_inv_l:
  assumes "mat_det A \<noteq> 0" "A *\<^sub>m\<^sub>m B = C"
  shows "B = mat_inv A *\<^sub>m\<^sub>m C"
using assms mat_eye_l[of B]
by (auto simp add: mult_mm_assoc mat_inv_l)

lemma mult_mm_inv_r:
  assumes "mat_det B \<noteq> 0" "A *\<^sub>m\<^sub>m B = C"
  shows "A = C *\<^sub>m\<^sub>m mat_inv B"
using assms mat_eye_r[of A]
by (auto simp add: mult_mm_assoc[symmetric] mat_inv_r)

lemma mult_mm_non_zero_l:
  assumes "mat_det A \<noteq> 0" "B \<noteq> mat_zero"
  shows "A *\<^sub>m\<^sub>m B \<noteq> mat_zero"
using assms mat_zero_r
using mult_mm_inv_l[OF assms(1), of B mat_zero]
by auto

lemma mat_inv_mult_mm:
  assumes "mat_det A \<noteq> 0" "mat_det B \<noteq> 0"
  shows "mat_inv (A *\<^sub>m\<^sub>m B) = mat_inv B *\<^sub>m\<^sub>m mat_inv A"
using assms
proof-
  have "(A *\<^sub>m\<^sub>m B) *\<^sub>m\<^sub>m (mat_inv B *\<^sub>m\<^sub>m mat_inv A) = eye"
    using assms
    by (metis mat_inv_r mult_mm_assoc mult_mm_inv_r)
  thus ?thesis
    using mult_mm_inv_l[of "A *\<^sub>m\<^sub>m B" "mat_inv B *\<^sub>m\<^sub>m mat_inv A" eye] assms mat_eye_r
    by simp
qed

lemma mult_mm_cancel_l:
  assumes "mat_det M \<noteq> 0"  "M *\<^sub>m\<^sub>m A = M *\<^sub>m\<^sub>m B"
  shows "A = B"
using assms
by (metis mult_mm_inv_l)

lemma mult_mm_cancel_r:
  assumes "mat_det M \<noteq> 0"  "A *\<^sub>m\<^sub>m M = B *\<^sub>m\<^sub>m M"
  shows "A = B"
using assms
by (metis mult_mm_inv_r)

lemma mult_mm_non_zero_r:
  assumes "A \<noteq> mat_zero" "mat_det B \<noteq> 0"
  shows "A *\<^sub>m\<^sub>m B \<noteq> mat_zero"
using assms mat_zero_l
using mult_mm_inv_r[OF assms(2), of A mat_zero]
by auto

lemma mat_inv_mult_sm:
  assumes "k \<noteq> 0"
  shows "mat_inv (k *\<^sub>s\<^sub>m A) = (1 / k) *\<^sub>s\<^sub>m mat_inv A"
proof-
  obtain a b c d where "A = (a, b, c, d)"
    by (cases A) auto
  thus ?thesis
    using assms
    by auto (subst mult.assoc[of k a "k*d"], subst mult.assoc[of k b "k*c"], subst right_diff_distrib[of k "a*(k*d)" "b*(k*c)", symmetric], simp, simp add: field_simps)+
qed

lemma mat_inv_inv [simp]:
  assumes "mat_det M \<noteq> 0"
  shows "mat_inv (mat_inv M) = M"
proof-
  have "mat_inv M *\<^sub>m\<^sub>m M = eye"
    using mat_inv_l[OF assms]
    by simp
  thus ?thesis
    using assms mat_det_inv[of M]
    using mult_mm_inv_l[of "mat_inv M" M eye] mat_eye_r
    by (auto simp del: eye_def)
qed

text{* Matrix transpose *}
fun mat_transpose where "mat_transpose (a, b, c, d) = (a, c, b, d)"

lemma [simp]: "mat_transpose (mat_transpose A) = A"
by (cases A) auto

lemma [simp]: "mat_transpose (k *\<^sub>s\<^sub>m A) = k *\<^sub>s\<^sub>m (mat_transpose A)"
by (cases A) simp

lemma [simp]: "mat_transpose (A *\<^sub>m\<^sub>m B) = mat_transpose B *\<^sub>m\<^sub>m mat_transpose A"
by (cases A, cases B) auto

lemma mat_inv_transpose: "mat_transpose (mat_inv M) = mat_inv (mat_transpose M)"
by (cases M) auto

lemma mat_det_transpose:
  fixes M :: "complex_mat"
  shows [simp]: "mat_det (mat_transpose M) = mat_det M"
by (cases M) auto

text{* Diagonal matrices *}

fun mat_diagonal where
 "mat_diagonal (A, B, C, D) = (B = 0 \<and> C = 0)"

text{* Matrix conjugate *}
fun mat_map where
 "mat_map f (a, b, c, d) = (f a, f b, f c, f d)"

definition mat_cnj where "mat_cnj = mat_map cnj"

lemma [simp]: "mat_cnj (mat_cnj A) = A"
unfolding mat_cnj_def
by (cases A) auto

lemma mat_cnj_sm [simp]: "mat_cnj (k *\<^sub>s\<^sub>m A) = cnj k *\<^sub>s\<^sub>m (mat_cnj A)"
by (cases A) (simp add: mat_cnj_def)

lemma mat_det_cnj [simp]: "mat_det (mat_cnj A) = cnj (mat_det A)"
by (cases A) (simp add: mat_cnj_def)

lemma nonzero_mat_cnj: "mat_cnj A = mat_zero \<longleftrightarrow> A = mat_zero"
by (cases A) (auto simp add: mat_cnj_def)

lemma mat_inv_cnj: "mat_cnj (mat_inv M) = mat_inv (mat_cnj M)"
unfolding mat_cnj_def
by (cases M) auto

text{* Matrix adjoint (conjugate *}
definition mat_adj where "mat_adj A = mat_cnj (mat_transpose A)"

lemma mat_adj_mult_mm [simp]: "mat_adj (A *\<^sub>m\<^sub>m B) = mat_adj B *\<^sub>m\<^sub>m mat_adj A"
by (cases A, cases B) (auto simp add: mat_adj_def mat_cnj_def)

lemma mat_adj_mult_sm [simp]: "mat_adj (k *\<^sub>s\<^sub>m A) = cnj k *\<^sub>s\<^sub>m mat_adj A"
  by (cases A) (auto simp add: mat_adj_def mat_cnj_def)

lemma mat_det_adj: "mat_det (mat_adj A) = cnj (mat_det A)"
by (cases A) (auto simp add: mat_adj_def mat_cnj_def)

lemma mat_adj_inv:
  assumes "mat_det M \<noteq> 0"
  shows "mat_adj (mat_inv M) = mat_inv (mat_adj M)"
  by (cases M) (auto simp add: mat_adj_def mat_cnj_def)

lemma mat_transpose_mat_cnj: "mat_transpose (mat_cnj A) = mat_adj A"
by (cases A)  (auto simp add: mat_adj_def mat_cnj_def)

lemma [simp]: "mat_adj (mat_adj A) = A"
unfolding mat_adj_def
by (subst mat_transpose_mat_cnj) (simp add: mat_adj_def)

lemma [simp]: "mat_adj eye = eye"
  by (auto simp add: mat_adj_def mat_cnj_def)

text{* Matrix trace *}
fun mat_trace where
  "mat_trace (a, b, c, d) = a + d"

text{* Multiplication of matrix and a vector *}
fun mult_mv :: "complex_mat \<Rightarrow> complex_vec \<Rightarrow> complex_vec" (infixl "*\<^sub>m\<^sub>v" 100)  where
  "(a, b, c, d) *\<^sub>m\<^sub>v (x, y) = (x*a + y*b, x*c + y*d)"

fun mult_vm :: "complex_vec \<Rightarrow> complex_mat \<Rightarrow> complex_vec" (infixl "*\<^sub>v\<^sub>m" 100) where
  "(x, y) *\<^sub>v\<^sub>m (a, b, c, d)  = (x*a + y*c, x*b + y*d)"

lemma eye_mv_l [simp]: "eye *\<^sub>m\<^sub>v v = v"
by (cases v) simp

lemma mult_mv_mv [simp]: " B *\<^sub>m\<^sub>v (A *\<^sub>m\<^sub>v v) = (B *\<^sub>m\<^sub>m A) *\<^sub>m\<^sub>v v"
by (cases v, cases A, cases B) (auto simp add: field_simps)

lemma mult_vm_vm [simp]: "(v *\<^sub>v\<^sub>m A) *\<^sub>v\<^sub>m B = v *\<^sub>v\<^sub>m (A *\<^sub>m\<^sub>m B)"
by (cases v, cases A, cases B) (auto simp add: field_simps)

lemma mult_mv_inv:
  assumes "x =  A *\<^sub>m\<^sub>v y" "mat_det A \<noteq> 0"
  shows "y = (mat_inv A) *\<^sub>m\<^sub>v x"
using assms
by (cases y) (simp add: mat_inv_l)

lemma mult_vm_inv:
  assumes "x =  y *\<^sub>v\<^sub>m A" "mat_det A \<noteq> 0"
  shows "y = x *\<^sub>v\<^sub>m (mat_inv A) "
using assms
by (cases y) (simp add: mat_inv_r)

lemma mult_mv_cancel_l:
  assumes "mat_det A \<noteq> 0" "A *\<^sub>m\<^sub>v v = A *\<^sub>m\<^sub>v v'"
  shows "v = v'"
using assms
using mult_mv_inv
by blast

lemma mult_vm_cancel_r:
  assumes "mat_det A \<noteq> 0" "v *\<^sub>v\<^sub>m A = v' *\<^sub>v\<^sub>m A"
  shows "v = v'"
using assms
using mult_vm_inv
by blast

lemma  vec_zero_l [simp]:
  "A *\<^sub>m\<^sub>v vec_zero = vec_zero"
by (cases A) simp

lemma  vec_zero_r [simp]:
  "vec_zero *\<^sub>v\<^sub>m A = vec_zero"
by (cases A) simp

lemma mult_mv_nonzero:
  assumes "v \<noteq> vec_zero" "mat_det A \<noteq> 0"
  shows "A *\<^sub>m\<^sub>v v \<noteq> vec_zero"
apply (rule ccontr)
using assms mult_mv_inv[of vec_zero A v] mat_inv_l vec_zero_l
by auto

lemma mult_vm_nonzero:
  assumes "v \<noteq> vec_zero" "mat_det A \<noteq> 0"
  shows "v *\<^sub>v\<^sub>m A \<noteq> vec_zero"
apply (rule ccontr)
using assms mult_vm_inv[of vec_zero v A] mat_inv_r vec_zero_r
by auto

lemma mult_sv_mv: "k *\<^sub>s\<^sub>v (A *\<^sub>m\<^sub>v v) = (A *\<^sub>m\<^sub>v (k *\<^sub>s\<^sub>v v))"
  by (cases A, cases v) (simp add: field_simps)

lemma mult_mv_mult_vm: " A *\<^sub>m\<^sub>v x = x *\<^sub>v\<^sub>m (mat_transpose A)"
by (cases A, cases x) auto

lemma
  mult_mv_vv: "A *\<^sub>m\<^sub>v v1 *\<^sub>v\<^sub>v v2 = v1 *\<^sub>v\<^sub>v (mat_transpose A *\<^sub>m\<^sub>v v2)"
by (cases v1, cases v2, cases A) (auto simp add: field_simps)

lemma mult_vv_mv: "x *\<^sub>v\<^sub>v (A *\<^sub>m\<^sub>v y)  = (x *\<^sub>v\<^sub>m A) *\<^sub>v\<^sub>v y"
by (cases x, cases y, cases A) (auto simp add: field_simps)

lemma vec_cnj_mult_mv:
  shows "vec_cnj (A *\<^sub>m\<^sub>v x) =  (mat_cnj A) *\<^sub>m\<^sub>v (vec_cnj x)"
by (cases A, cases x) (auto simp add: vec_cnj_def mat_cnj_def)

lemma vec_cnj_mult_vm: "vec_cnj (v *\<^sub>v\<^sub>m A) = vec_cnj v *\<^sub>v\<^sub>m mat_cnj A"
unfolding vec_cnj_def mat_cnj_def
by (cases A, cases v, auto)

(* ---------------------------------------------------------------------------- *)
subsection{* Eigenvalues and eigenvectors *}
(* ---------------------------------------------------------------------------- *)

definition eigenpair where
  [simp]: "eigenpair k v H \<longleftrightarrow> v \<noteq> vec_zero \<and> H *\<^sub>m\<^sub>v v = k *\<^sub>s\<^sub>v v"

definition eigenval where
  [simp]: "eigenval k H \<longleftrightarrow> (\<exists> v. v \<noteq> vec_zero \<and> H *\<^sub>m\<^sub>v v = k *\<^sub>s\<^sub>v v)"

lemma eigen_equation:
  shows "eigenval k H \<longleftrightarrow> k\<^sup>2 - mat_trace H * k + mat_det H = 0" (is "?lhs \<longleftrightarrow> ?rhs")
proof-
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  show ?thesis
  proof
    assume ?lhs
    then obtain v where "v \<noteq> vec_zero" "H *\<^sub>m\<^sub>v v = k *\<^sub>s\<^sub>v v"
      unfolding eigenval_def
      by blast
    obtain v1 v2 where vv: "v = (v1, v2)"
      by (cases v) auto
    from `H *\<^sub>m\<^sub>v v = k *\<^sub>s\<^sub>v v` have "(H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) *\<^sub>m\<^sub>v v = vec_zero"
      using HH vv
      by (auto simp add: field_simps)
    hence "mat_det (H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) = 0"
      using `v \<noteq> vec_zero` vv HH
      using regular_homogenous_system[of "A - k" "D - k" B C v1 v2]
      by (auto simp add: field_simps)
    thus ?rhs
      using HH
      by (auto simp add: power2_eq_square field_simps)
  next
    assume ?rhs
    hence *: "mat_det (H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) = 0"
      using HH
      by (auto simp add: field_simps power2_eq_square)
    show ?lhs
    proof (cases "H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye) = mat_zero")
      case True
      thus ?thesis
        using HH
        by (auto) (rule_tac x=1 in exI, simp)
    next
      case False
      hence "(A - k \<noteq> 0 \<or> B \<noteq> 0) \<or> (D - k \<noteq> 0 \<or> C \<noteq> 0)"
        using HH
        by auto
      thus ?thesis
      proof
        assume "A - k \<noteq> 0 \<or> B \<noteq> 0"
        hence "C * B + (D - k) * (k - A) = 0"
          using * singular_system[of "A-k" "D-k" B C "(0, 0)" 0 0  "(B, k-A)"] HH
          by (auto simp add: field_simps)
        hence  "(B, k-A) \<noteq> vec_zero" "(H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) *\<^sub>m\<^sub>v (B, k-A) = vec_zero"
          using HH `A - k \<noteq> 0 \<or> B \<noteq> 0`
          by (auto simp add: field_simps)
        then obtain v where "v \<noteq> vec_zero \<and> (H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) *\<^sub>m\<^sub>v v = vec_zero"
          by blast
        thus ?thesis
          using HH
          unfolding eigenval_def
          by (rule_tac x="v" in exI) (case_tac v, simp add: field_simps)
      next
        assume "D - k \<noteq> 0 \<or> C \<noteq> 0"
        hence "C * B + (D - k) * (k - A) = 0"
          using * singular_system[of "D-k" "A-k" C B "(0, 0)" 0 0  "(C, k-D)"] HH
          by (auto simp add: field_simps)
        hence  "(k-D, C) \<noteq> vec_zero" "(H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) *\<^sub>m\<^sub>v (k-D, C) = vec_zero"
          using HH `D - k \<noteq> 0 \<or> C \<noteq> 0`
          by (auto simp add: field_simps)
        then obtain v where "v \<noteq> vec_zero \<and> (H -\<^sub>m\<^sub>m (k *\<^sub>s\<^sub>m eye)) *\<^sub>m\<^sub>v v = vec_zero"
          by blast
        thus ?thesis
          using HH
          unfolding eigenval_def
          by (rule_tac x="v" in exI) (case_tac v, simp add: field_simps)
      qed
    qed
  qed
qed

(* ---------------------------------------------------------------------------- *)
subsection{* Bilinear and Quadratic forms; Congruence *}
(* ---------------------------------------------------------------------------- *)

text{* Bilinear forms *}
definition bilinear_form where
 [simp]: "bilinear_form v1 v2 H = (vec_cnj v1) *\<^sub>v\<^sub>m H *\<^sub>v\<^sub>v v2"

lemma bilinear_form_scale_m:
  shows "bilinear_form v1 v2 (k *\<^sub>s\<^sub>m H) = k * bilinear_form v1 v2 H"
by (cases v1, cases v2, cases H) (simp add: vec_cnj_def field_simps)

lemma bilinear_form_scale_v1:
  shows "bilinear_form (k *\<^sub>s\<^sub>v v1) v2 H = cnj k * bilinear_form v1 v2 H"
by (cases v1, cases v2, cases H) (simp add: vec_cnj_def field_simps)

lemma bilinear_form_scale_v2:
  shows "bilinear_form  v1 (k *\<^sub>s\<^sub>v v2) H = k * bilinear_form v1 v2 H"
by (cases v1, cases v2, cases H) (simp add: vec_cnj_def field_simps)

text{* Quadratic forms *}
definition quad_form where
 [simp]: "quad_form v H = (vec_cnj v) *\<^sub>v\<^sub>m H *\<^sub>v\<^sub>v v"

lemma "quad_form v H = bilinear_form v v H"
by simp

lemma quad_form_scale_v:
  shows "quad_form (k *\<^sub>s\<^sub>v v) H = cor ((cmod k)\<^sup>2) * quad_form v H"
using bilinear_form_scale_v1 bilinear_form_scale_v2
by (simp add: complex_mult_cnj_cmod field_simps)

lemma quad_form_scale_m:
  shows "quad_form v (k *\<^sub>s\<^sub>m H) = k * quad_form v H"
using bilinear_form_scale_m
by simp

lemma cnj_quad_form [simp]: "cnj (quad_form z H) = quad_form z (mat_adj H)"
by (cases H, cases z) (auto simp add: mat_adj_def mat_cnj_def vec_cnj_def field_simps)

text{* Matrix congruence *}
definition congruence where
  [simp]: "congruence M H \<equiv> mat_adj M *\<^sub>m\<^sub>m H *\<^sub>m\<^sub>m M"

lemma bilinear_form_congruence:
  assumes "mat_det M \<noteq> 0"
  shows "bilinear_form v1 v2 H = bilinear_form (M *\<^sub>m\<^sub>v v1) (M *\<^sub>m\<^sub>v v2) (congruence (mat_inv M) H)"
proof-
  have "mat_det (mat_adj M) \<noteq> 0"
    using assms
    by (simp add: mat_det_adj)
  show ?thesis
    unfolding bilinear_form_def congruence_def
    apply (subst mult_mv_mult_vm)
    apply (subst vec_cnj_mult_vm)
    apply (subst mat_adj_def[symmetric])
    apply (subst mult_vm_vm)
    apply (subst mult_vv_mv)
    apply (subst mult_vm_vm)
    apply (subst mat_adj_inv[OF `mat_det M \<noteq> 0`])
    apply (subst mult_assoc_5)
    apply (subst mat_inv_r[OF `mat_det (mat_adj M) \<noteq> 0`])
    apply (subst mat_inv_l[OF `mat_det M \<noteq> 0`])
    apply (subst mat_eye_l, subst mat_eye_r)
    by simp
qed

lemma quad_form_congruence:
  assumes "mat_det M \<noteq> 0"
  shows "quad_form (M *\<^sub>m\<^sub>v z) (congruence (mat_inv M) H) = quad_form z H"
using bilinear_form_congruence[OF assms]
by simp

lemma congruence_nonzero:
  assumes "H \<noteq> mat_zero" "mat_det M \<noteq> 0"
  shows "congruence M H \<noteq> mat_zero"
  using assms
  unfolding congruence_def
  by (subst mult_mm_non_zero_r, subst mult_mm_non_zero_l) (auto simp add: mat_det_adj)

lemma congruence_congruence:
  shows "congruence M1 (congruence M2 A) = congruence (M2 *\<^sub>m\<^sub>m M1) A"
  unfolding congruence_def
  apply (subst mult_mm_assoc)
  apply (subst mult_mm_assoc)
  apply (subst mat_adj_mult_mm)
  apply (subst mult_mm_assoc)
  by simp

lemma [simp]: "congruence eye A = A"
  by (cases A) (simp add: mat_adj_def mat_cnj_def)


lemma congruence_congruence_inv:
  assumes "mat_det M \<noteq> 0"
  shows "congruence M (congruence (mat_inv M) A) = A"
  using assms congruence_congruence[of M "mat_inv M" A]
  using mat_inv_l[of M] mat_eye_l mat_eye_r
  unfolding congruence_def
  by (simp del: eye_def)

lemma congruence_inv:
  assumes "mat_det M \<noteq> 0" "congruence M A = B"
  shows "congruence (mat_inv M) B = A"
  using assms
  using `mat_det M \<noteq> 0` mult_mm_inv_l[of "mat_adj M" "A *\<^sub>m\<^sub>m M" "B"]
  using mult_mm_inv_r[of M "A" "mat_inv (mat_adj M) *\<^sub>m\<^sub>m B"]
  by (simp add: mat_det_adj mult_mm_assoc mat_adj_inv)

lemma congruence_scale_m:
  shows "congruence A (k *\<^sub>s\<^sub>m B) = k *\<^sub>s\<^sub>m (congruence A B)"
by (cases A, cases B) (auto simp add: mat_adj_def mat_cnj_def field_simps)

lemma inj_congruence:
  assumes "mat_det M \<noteq> 0" "congruence M H = congruence M H'"
  shows "H = H'"
proof-
  have "H *\<^sub>m\<^sub>m M = H' *\<^sub>m\<^sub>m M "
    using assms
    using mult_mm_cancel_l[of "mat_adj M" "H *\<^sub>m\<^sub>m M" "H' *\<^sub>m\<^sub>m M"]
    by (simp add: mat_det_adj mult_mm_assoc)
  thus ?thesis
    using assms
    using mult_mm_cancel_r[of "M" "H" "H'"]
    by simp
qed

definition similarity where "similarity I M = mat_inv I *\<^sub>m\<^sub>m M *\<^sub>m\<^sub>m I"

lemma
  mat_det_similarity [simp]:
  assumes "mat_det I \<noteq> 0"
  shows "mat_det (similarity I M) = mat_det M"
using assms
unfolding similarity_def
by (simp add: mat_det_inv)

lemma mat_trace_similarity [simp]:
  assumes "mat_det I \<noteq> 0"
  shows "mat_trace (similarity I M) = mat_trace M"
proof-
  obtain a b c d where II: "I = (a, b, c, d)"
    by (cases I) auto
  obtain A B C D where MM: "M = (A, B, C, D)"
    by (cases M) auto
  have "A * (a * d) / (a * d - b * c) + D * (a * d) / (a * d - b * c) =
        A + D + A * (b * c) / (a * d - b * c) + D * (b * c) / (a * d - b * c)"
    using assms II
    by (simp add: field_simps)
  thus ?thesis
    using II MM
    by (simp add: field_simps similarity_def)
qed

lemma similarity_eye [simp]: "similarity eye M = M"
unfolding similarity_def
by simp (metis eye_def mat_eye_l mat_eye_r)

lemma similarity_eye' [simp]: "similarity (1, 0, 0, 1) M = M"
unfolding eye_def[symmetric]
by (simp del: eye_def)

lemma similarity_comp [simp]:
  assumes "mat_det I1 \<noteq> 0" "mat_det I2 \<noteq> 0"
  shows "similarity I1 (similarity I2 M) = similarity (I2*\<^sub>m\<^sub>mI1) M"
using assms
unfolding similarity_def
by (simp add: mult_mm_assoc mat_inv_mult_mm)

lemma similarity_inv:
  assumes "similarity I M1 = M2" "mat_det I \<noteq> 0"
  shows "similarity (mat_inv I) M2 = M1"
using assms
unfolding similarity_def
by simp (metis mat_eye_l mult_mm_assoc mult_mm_inv_r)

end