section{* Oriented circlines *}
theory OrientedCirclines
imports Circlines
begin

(* ----------------------------------------------------------------- *)
subsection {* Oriented circlines definition *}
(* ----------------------------------------------------------------- *)

definition ocircline_eq_cmat :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> bool" where
  [simp]: "ocircline_eq_cmat A B \<longleftrightarrow>(\<exists> k::real. k > 0 \<and> B = cor k *\<^sub>s\<^sub>m A)"
lift_definition ocircline_eq_clmat :: "circline_mat \<Rightarrow> circline_mat \<Rightarrow> bool" is ocircline_eq_cmat
  done

lemma [simp]:
  shows "ocircline_eq_cmat H H"
by (simp, rule_tac x=1 in exI, simp)

quotient_type ocircline = circline_mat / ocircline_eq_clmat
proof (rule equivpI)
  show "reflp ocircline_eq_clmat"
    unfolding reflp_def
    by transfer (auto, rule_tac x="1" in exI, simp)
next
  show "symp ocircline_eq_clmat"
    unfolding symp_def
    by transfer (simp only: ocircline_eq_cmat_def, safe, rule_tac x="1/k" in exI, simp)
next
  show "transp ocircline_eq_clmat"
    unfolding transp_def
    by transfer (simp only: ocircline_eq_cmat_def, safe, rule_tac x="k*ka" in exI, simp)
qed

(* ----------------------------------------------------------------- *)
subsection{* Points on oriented circlines *}
(* ----------------------------------------------------------------- *)

lift_definition on_ocircline :: "ocircline \<Rightarrow> complex_homo \<Rightarrow> bool" is on_circline_clmat_hcoords
  by transfer (simp del: quad_form_def, (erule exE)+, simp add: quad_form_scale_m quad_form_scale_v del: quad_form_def)

definition ocircline_set :: "ocircline \<Rightarrow> complex_homo set" where
  "ocircline_set H = {z. on_ocircline H z}"

lemma ocircline_set_I [simp]:
  assumes "on_ocircline H z"
  shows "z \<in> ocircline_set H"
  using assms
  unfolding ocircline_set_def
  by simp

(* ----------------------------------------------------------------- *)
subsection{* Disc and disc complement - in and out points *}
(* ----------------------------------------------------------------- *)

definition in_ocircline_cmat_cvec :: "complex_mat \<Rightarrow> complex_vec \<Rightarrow> bool" where
  [simp]: "in_ocircline_cmat_cvec H z \<longleftrightarrow> Re (quad_form z H) < 0"
lift_definition in_ocircline_clmat_hcoords :: "circline_mat \<Rightarrow> complex_homo_coords \<Rightarrow> bool" is in_ocircline_cmat_cvec
  done
lift_definition in_ocircline :: "ocircline \<Rightarrow> complex_homo \<Rightarrow> bool" is in_ocircline_clmat_hcoords
proof transfer
  fix H H' z z'
  assume hh: "hermitean H \<and> H \<noteq> mat_zero" "hermitean H' \<and> H' \<noteq> mat_zero"
             "z \<noteq> vec_zero" "z' \<noteq> vec_zero"
  assume "ocircline_eq_cmat H H'" "z \<approx>\<^sub>v z'"
  then obtain k k' where
    *: "0 < k" "H' = cor k *\<^sub>s\<^sub>m H" "k' \<noteq> 0" "z' = k' *\<^sub>s\<^sub>v  z"
    by auto
  hence "quad_form z' H' = cor k * cor ((cmod k')\<^sup>2) * quad_form z H"
    by (simp add: quad_form_scale_v quad_form_scale_m del: vec_cnj_sv quad_form_def)
  hence "Re (quad_form z' H') = k * (cmod k')\<^sup>2 * Re (quad_form z H)"
    using hh quad_form_hermitean_real[of H]
    by (simp add: complex_of_real_Re power2_eq_square)
  thus "in_ocircline_cmat_cvec H z = in_ocircline_cmat_cvec H' z'"
    using `k > 0` `k' \<noteq> 0`
    apply auto
    apply (metis mult_pos_neg mult_pos_pos norm_eq_zero zero_less_power2)
    apply (metis semiring_normalization_rules(10) mult_less_cancel_left_pos zero_less_norm_iff zero_less_power)
    done
qed

definition disc where
  "disc H = {z. in_ocircline H z}"

lemma disc_I [simp]:
  assumes "in_ocircline H z"
  shows "z \<in> disc H"
  using assms
  unfolding disc_def
  by simp

definition out_ocircline_cmat_cvec :: "complex_mat \<Rightarrow> complex_vec \<Rightarrow> bool" where
  [simp]: "out_ocircline_cmat_cvec H z \<longleftrightarrow> Re (quad_form z H) > 0"
lift_definition out_ocircline_clmat_hcoords :: "circline_mat \<Rightarrow> complex_homo_coords \<Rightarrow> bool" is out_ocircline_cmat_cvec
  done
lift_definition out_ocircline :: "ocircline \<Rightarrow> complex_homo \<Rightarrow> bool" is out_ocircline_clmat_hcoords
proof transfer
  fix H H' z z'
  assume hh: "hermitean H \<and> H \<noteq> mat_zero" "hermitean H' \<and> H' \<noteq> mat_zero"
             "z \<noteq> vec_zero" "z' \<noteq> vec_zero"
  assume "ocircline_eq_cmat H H'" "z \<approx>\<^sub>v z'"
  then obtain k k' where
    *: "0 < k" "H' = cor k *\<^sub>s\<^sub>m H" "k' \<noteq> 0" "z' = k' *\<^sub>s\<^sub>v  z"
    by auto
  hence "quad_form z' H' = cor k * cor ((cmod k')\<^sup>2) * quad_form z H"
    by (simp add: quad_form_scale_v quad_form_scale_m del: vec_cnj_sv quad_form_def)
  hence "Re (quad_form z' H') = k * (cmod k')\<^sup>2 * Re (quad_form z H)"
    using hh quad_form_hermitean_real[of H]
    by (simp add: complex_of_real_Re power2_eq_square)
  thus "out_ocircline_cmat_cvec H z = out_ocircline_cmat_cvec H' z'"
    using `k > 0` `k' \<noteq> 0`
    apply auto
    apply (metis semiring_normalization_rules(10) mult_less_cancel_left_pos zero_less_norm_iff zero_less_power)
    done
qed

definition disc_compl where
  "disc_compl H = {z. out_ocircline H z}"

lemma disc_compl_I [simp]:
  assumes "out_ocircline H z"
  shows "z \<in> disc_compl H"
  using assms
  unfolding disc_compl_def
  by simp

lemma in_on_out:
  shows "in_ocircline H z \<or> on_ocircline H z \<or> out_ocircline H z"
  apply (transfer, transfer)
  using quad_form_hermitean_real
  using complex_eq_if_Re_eq
  by auto

lemma in_on_out_univ:
  shows "disc H \<union> disc_compl H \<union> ocircline_set H = UNIV"
unfolding disc_def disc_compl_def ocircline_set_def
using in_on_out[of H]
by auto

lemma disc_inter_disc_compl [simp]:
  shows "disc H \<inter> disc_compl H = {}"
  unfolding disc_def disc_compl_def
  by auto (transfer, transfer, simp)

lemma disc_inter_ocircline_set [simp]:
  "disc H \<inter> ocircline_set H = {}"
  unfolding disc_def ocircline_set_def
  by auto (transfer, transfer, simp)

lemma disc_compl_inter_ocircline_set [simp]:
  "disc_compl H \<inter> ocircline_set H = {}"
  unfolding disc_compl_def ocircline_set_def
  by auto (transfer, transfer, simp)

(* ----------------------------------------------------------------- *)
subsection{* Opposite orientation *}
(* ----------------------------------------------------------------- *)

definition opposite_ocircline_cmat :: "complex_mat \<Rightarrow> complex_mat" where
  [simp]: "opposite_ocircline_cmat H = (-1) *\<^sub>s\<^sub>m H"
lift_definition opposite_ocircline_clmat :: "circline_mat \<Rightarrow> circline_mat" is opposite_ocircline_cmat
  by (auto simp add: hermitean_def mat_adj_def mat_cnj_def)
lift_definition opposite_ocircline :: "ocircline \<Rightarrow> ocircline" is opposite_ocircline_clmat
  by transfer auto

lemma opposite_ocircline_involution [simp]:
  shows "opposite_ocircline (opposite_ocircline H) = H"
by (transfer, transfer) (auto, rule_tac x="1" in exI, simp)

lemma on_circline_opposite_ocircline_cmat [simp]:
  assumes "hermitean H \<and> H \<noteq> mat_zero" "z \<noteq> vec_zero"
  shows "on_circline_cmat_cvec (opposite_ocircline_cmat H) z = on_circline_cmat_cvec H z"
  using assms
  by (simp add: quad_form_scale_m del: quad_form_def)

lemma on_circline_opposite_ocircline [simp]:
  shows "on_ocircline (opposite_ocircline H) z \<longleftrightarrow> on_ocircline H z"
  using on_circline_opposite_ocircline_cmat
  by (transfer, transfer, simp)

lemma ocircline_set_opposite_ocircline [simp]:
  shows "ocircline_set (opposite_ocircline H) = ocircline_set H"
  unfolding ocircline_set_def
  by auto

lemma disc_compl_opposite_ocircline [simp]:
  shows "disc_compl (opposite_ocircline H) = disc H"
  unfolding disc_def disc_compl_def
  apply auto
   apply (transfer, transfer)
   apply (auto simp add: quad_form_scale_m simp del: quad_form_def)
  apply (transfer ,transfer)
  apply (auto simp add: quad_form_scale_m simp del: quad_form_def)
  done

lemma disc_opposite_ocircline [simp]:
  shows "disc (opposite_ocircline H) = disc_compl H"
  using disc_compl_opposite_ocircline[of "opposite_ocircline H"]
  by simp

(* ----------------------------------------------------------------- *)
subsection{* Positive orientation. Conversion between unoriented and oriented circlines *}
(* ----------------------------------------------------------------- *)

text{* Remove orientation *}

lift_definition of_ocircline :: "ocircline \<Rightarrow> circline" is "id::circline_mat \<Rightarrow> circline_mat"
  by transfer (simp, erule exE, force)

lemma of_ocircline_opposite_ocircline [simp]:
  shows "of_ocircline (opposite_ocircline H) = of_ocircline H"
  by (transfer, transfer) (simp, erule exE, rule_tac x="-1" in exI, simp)

lemma on_ocircline_of_circline [simp]:
  shows "on_circline (of_ocircline H) z \<longleftrightarrow> on_ocircline H z"
  by (transfer, transfer, simp)

lemma circline_set_of_ocircline [simp]:
  shows "circline_set (of_ocircline H) = ocircline_set H"
  unfolding ocircline_set_def circline_set_def
  by (safe) (transfer, simp)+

lemma inj_of_ocircline:
  assumes "of_ocircline H = of_ocircline H'"
  shows "H = H' \<or> H = opposite_ocircline H'"
using assms
  by (transfer, transfer) (simp, erule exE, simp, metis linorder_neqE_linordered_idom neg_0_less_iff_less of_real_minus)

lemma inj_ocircline_set:
  assumes "ocircline_set H = ocircline_set H'" "ocircline_set H \<noteq> {}"
  shows "H = H' \<or> H = opposite_ocircline H'"
proof-
  from assms 
  have "circline_set (of_ocircline H) = circline_set (of_ocircline H')"
       "circline_set (of_ocircline H') \<noteq> {}"
    by auto
  hence "of_ocircline H = of_ocircline H'"
    by (simp add: inj_circline_set)
  thus ?thesis
    by (rule inj_of_ocircline)
qed

text{* Positive orientation *}

definition pos_oriented_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "pos_oriented_cmat H \<longleftrightarrow>
           (let (A, B, C, D) = H
              in (Re A > 0 \<or> (Re A = 0 \<and> ((B \<noteq> 0 \<and> arg B > 0) \<or> (B = 0 \<and> Re D > 0)))))"
lift_definition pos_oriented_clmat :: "circline_mat \<Rightarrow> bool" is pos_oriented_cmat
  done

lift_definition pos_oriented :: "ocircline \<Rightarrow> bool" is pos_oriented_clmat
  apply transfer
  apply (case_tac circline_mat1, case_tac circline_mat2, simp, erule exE, simp)
  by (metis arg_mult_real_positive mult_pos_pos zero_less_mult_pos)

lemma pos_oriented:
  shows "pos_oriented H \<or> pos_oriented (opposite_ocircline H)"
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  moreover
  hence "Re A = 0 \<and> Re D = 0 \<longrightarrow> B \<noteq> 0"
    using hh hermitean_elems[of A B C D]
    by (cases A, cases D) (auto simp add: Complex_eq)
  moreover
  have "B \<noteq> 0 \<and> \<not> 0 < arg B \<longrightarrow> 0 < arg (- B)"
    using MoreComplex.canon_ang_plus_pi2[of "arg B"] arg_bounded[of B]
    by (auto simp add: arg_uminus)
  ultimately
  show "pos_oriented_cmat H \<or> pos_oriented_cmat (opposite_ocircline_cmat H)"
    by auto
qed

lemma pos_oriented_opposite_ocircline_cmat [simp]:
  assumes "hermitean H \<and> H \<noteq> mat_zero"
  shows  "pos_oriented_cmat (opposite_ocircline_cmat H) \<longleftrightarrow> \<not> pos_oriented_cmat H"
proof-
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  moreover
  hence "Re A = 0 \<and> Re D = 0 \<longrightarrow> B \<noteq> 0"
    using assms hermitean_elems[of A B C D]
    by (cases A, cases D) (auto simp add: Complex_eq)
  moreover
  have "B \<noteq> 0 \<and> \<not> 0 < arg B \<longrightarrow> 0 < arg (- B)"
    using MoreComplex.canon_ang_plus_pi2[of "arg B"] arg_bounded[of B]
    by (auto simp add: arg_uminus)
  moreover
  have "B \<noteq> 0 \<and> 0 < arg B \<longrightarrow> \<not> 0 < arg (- B)"
    using MoreComplex.canon_ang_plus_pi1[of "arg B"] arg_bounded[of B]
    by (auto simp add: arg_uminus)
  ultimately
  show "pos_oriented_cmat (opposite_ocircline_cmat H) = (\<not> pos_oriented_cmat H)"
    by simp (metis not_less_iff_gr_or_eq)
qed

lemma pos_oriented_opposite_ocircline [simp]:
  "pos_oriented (opposite_ocircline H) \<longleftrightarrow> \<not> pos_oriented H"
  using pos_oriented_opposite_ocircline_cmat
  by (transfer, transfer, simp)

lemma pos_oriented_circle_inf:
  assumes "\<infinity>\<^sub>h \<notin> ocircline_set H"
  shows "pos_oriented H \<longleftrightarrow> \<infinity>\<^sub>h \<notin> disc H"
  using assms
  unfolding ocircline_set_def disc_def
  apply simp
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  hence "is_real A"
    using hh hermitean_elems
    by auto
  assume "\<not> on_circline_cmat_cvec H \<infinity>\<^sub>v"
  thus "pos_oriented_cmat H = (\<not> in_ocircline_cmat_cvec H  \<infinity>\<^sub>v)"
    using HH `is_real A`
    by (cases A) (auto simp add: vec_cnj_def Complex_eq)
qed

lemma pos_oriented_euclidean_circle:
  assumes "is_circle (of_ocircline H)"
          "(a, r) = euclidean_circle (of_ocircline H)"
          "circline_type (of_ocircline H) < 0"
  shows "pos_oriented H \<longleftrightarrow> of_complex a \<in> disc H"
using assms
unfolding disc_def
apply simp
proof (transfer, transfer)
  fix H a r
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  hence "is_real A" "is_real D" "C = cnj B"
    using hh hermitean_elems
    by auto

  assume *: "\<not> circline_A0_cmat (id H)" "(a, r) = euclidean_circle_cmat (id H)" "circline_type_cmat (id H) < 0"
  hence "A \<noteq> 0" "Re A \<noteq> 0"
    using HH `is_real A`
    by (case_tac[!] A) (auto simp add: Complex_eq)

  have "Re (A*D - B*C) < 0"
    using `circline_type_cmat (id H) < 0` HH
    by simp

  have **: "(A * (D * cnj A) - B * (C * cnj A)) / (A * cnj A) = (A*D - B*C) / A"
    using `A \<noteq> 0`
    by (simp add: field_simps)
  hence ***: "0 < Re A \<longleftrightarrow> Re ((A * (D * cnj A) - B * (C * cnj A)) / (A * cnj A)) < 0"
    using `is_real A` `A \<noteq> 0` `Re (A*D - B*C) < 0`
    by (auto simp add: Re_divide_real, metis divide_less_0_iff less_iff_diff_less_0, metis divide_less_0_iff less_iff_diff_less_0 mult_neg_neg zero_less_mult_pos)
  show "pos_oriented_cmat H = in_ocircline_cmat_cvec H (of_complex_cvec a)"
    using HH `Re A \<noteq> 0` * `is_real A`
    apply -
    apply (auto simp add: vec_cnj_def)
    apply (smt "**" "***" cnj.simps(1) cnj.simps(2) complex_eq diff_divide_distrib left_diff_distrib' minus_complex.simps(1) mult.commute nonzero_mult_div_cancel_right)
    apply (smt "**" "***" cnj.simps(1) cnj.simps(2) complex_eq diff_divide_distrib left_diff_distrib' minus_complex.simps(1) mult.commute nonzero_mult_div_cancel_left)
    done
qed

text{* Introduce positive orientation *}

definition of_circline_cmat :: "complex_mat \<Rightarrow> complex_mat" where
 [simp]: "of_circline_cmat H = (if pos_oriented_cmat H then H else opposite_ocircline_cmat H)"

lift_definition of_circline_clmat :: "circline_mat \<Rightarrow> circline_mat" is of_circline_cmat
  by (auto simp add: hermitean_def mat_adj_def mat_cnj_def)

lemma of_circline_clmat_def':
  "of_circline_clmat H = (if pos_oriented_clmat H then H else opposite_ocircline_clmat H)"
  by transfer simp

lemma pos_oriented_cmat_mult_positive:
  assumes
    "hermitean H1 \<and> H1 \<noteq> mat_zero"
    "hermitean H2 \<and> H2 \<noteq> mat_zero"
    "\<exists>k. k > 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
  shows "pos_oriented_cmat H1 \<longleftrightarrow> pos_oriented_cmat H2"
  using assms
  apply (cases H1, cases H2)
  apply simp
  apply (erule exE)
  apply simp
  by (metis arg_mult_real_positive mult_pos_pos zero_less_mult_pos)

lemma pos_oriented_cmat_mult_negative:
  assumes
    "hermitean H1 \<and> H1 \<noteq> mat_zero"
    "hermitean H2 \<and> H2 \<noteq> mat_zero"
    "\<exists>k. k < 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
  shows
    "pos_oriented_cmat H1 \<longleftrightarrow> \<not> pos_oriented_cmat H2"
  using assms
proof-
  obtain A B C D A1 B1 C1 D1
    where *: "H1 = (A, B, C, D)" "H2 = (A1, B1, C1, D1)"
    by (cases H1, cases H2) auto
  hence **: "is_real A" "is_real D" "is_real A1" "is_real D1" "B = 0 \<longleftrightarrow> C = 0" "B1 = 0 \<longleftrightarrow> C1 = 0"
    using assms hermitean_elems[of A B C D] hermitean_elems[of A1 B1 C1 D1]
    by auto
  show ?thesis
    using assms * **
    apply -
    apply (rule iffI)
    apply (cases "Re A > 0")
    using mult_neg_pos
    apply fastforce
    apply (cases "B = 0")
    using mult_neg_pos
    apply fastforce
    using arg_uminus_opposite_sign[of B] arg_mult_real_negative
    apply fastforce
    apply (cases "Re A > 0")
    using mult_neg_pos
    apply fastforce
    apply (cases "B = 0")
     apply (simp, erule exE, (erule conjE)+, simp, smt mult_neg_neg complex_of_real_Re zero_complex.simps)
    apply (simp, erule exE, (erule conjE))
    using arg_uminus_opposite_sign[of B]
    apply (case_tac "Re A = 0", simp_all add: arg_mult_real_negative mult_neg_neg)
    done
qed

lift_definition of_circline :: "circline \<Rightarrow> ocircline" is of_circline_clmat
proof transfer
  fix H1 H2
  assume hh:
    "hermitean H1 \<and> H1 \<noteq> mat_zero"
    "hermitean H2 \<and> H2 \<noteq> mat_zero"
  assume "circline_eq_cmat H1 H2"
  then obtain k where *: "k \<noteq> 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  show "ocircline_eq_cmat (of_circline_cmat H1) (of_circline_cmat H2)"
  proof (cases "k > 0")
    case True
    hence "pos_oriented_cmat H1 = pos_oriented_cmat H2"
      using * pos_oriented_cmat_mult_positive[OF hh]
      by blast
    thus ?thesis
      using hh * `k > 0`
      apply (simp del: pos_oriented_cmat_def)
      apply (rule conjI)
       apply (rule impI)
       apply (simp, rule_tac x=k in exI, simp)
      apply (rule impI)
      apply (simp, rule_tac x=k in exI, simp)
      done
  next
    case False
    hence "k < 0"
      using *
      by simp
    hence "pos_oriented_cmat H1 \<longleftrightarrow> \<not> (pos_oriented_cmat H2)"
      using * pos_oriented_cmat_mult_negative[OF hh]
      by blast
    thus ?thesis
      using hh * `k < 0`
      apply (simp del: pos_oriented_cmat_def)
      apply (rule conjI)
       apply (rule impI)
       apply (simp, rule_tac x="-k" in exI, simp)
      apply (rule impI)
      apply (simp, rule_tac x="-k" in exI, simp)
      done
  qed
qed

lemma pos_oriented_of_circline [simp]:
  shows "pos_oriented (of_circline H)"
  using pos_oriented_opposite_ocircline_cmat
  by (transfer, transfer, simp)

lemma of_ocircline_of_circline [simp]:
  shows "of_ocircline (of_circline H) = H"
  apply (transfer, auto simp add: of_circline_clmat_def')
  apply (transfer, simp, rule_tac x="-1" in exI, simp)
  done

lemma of_circline_of_ocircline_pos_oriented [simp]:
  assumes "pos_oriented H"
  shows "of_circline (of_ocircline H) = H"
  using assms
  by (transfer, transfer, simp, rule_tac x=1 in exI, simp)

lemma inj_of_circline:
  assumes "of_circline H = of_circline H'"
  shows "H = H'"
  using assms
proof (transfer, transfer)
  fix H H'
  assume "ocircline_eq_cmat (of_circline_cmat H) (of_circline_cmat H')"
  then obtain k where "k > 0" "of_circline_cmat H' = cor k *\<^sub>s\<^sub>m of_circline_cmat H"
    by auto
  thus "circline_eq_cmat H H'"
    using mult_sm_inv_l[of "-1" "H'" "cor k *\<^sub>s\<^sub>m H"]
    using mult_sm_inv_l[of "-1" "H'" "(- (cor k)) *\<^sub>s\<^sub>m H"]
    apply (simp split: if_split_asm)
    apply (rule_tac x="k" in exI, simp)
    apply (rule_tac x="-k" in exI, simp)
    apply (rule_tac x="-k" in exI, simp)
    apply (rule_tac x="k" in exI, simp)
    done
qed

lemma of_circline_of_ocircline:
  shows "of_circline (of_ocircline H') = H' \<or> 
         of_circline (of_ocircline H') = opposite_ocircline H'"
proof (cases "pos_oriented H'")
  case True
  thus ?thesis
    by auto
next
  case False
  hence "pos_oriented (opposite_ocircline H')"
    using pos_oriented
    by auto
  thus ?thesis
    using of_ocircline_opposite_ocircline[of H']
    using of_circline_of_ocircline_pos_oriented [of "opposite_ocircline H'"]
    by auto
qed

text{* Set of points on oriented and unoriented circlines*}

lemma ocircline_set_of_circline [simp]:
  "ocircline_set (of_circline H) = circline_set H"
  unfolding ocircline_set_def circline_set_def
proof (safe)
  fix z
  assume "on_ocircline (of_circline H) z"
  thus "on_circline H z"
    by (transfer, transfer, simp del: on_circline_cmat_cvec_def opposite_ocircline_cmat_def split: if_split_asm)
next
  fix z
  assume "on_circline H z"
  thus "on_ocircline (of_circline H) z"
    by (transfer, transfer, simp del: on_circline_cmat_cvec_def opposite_ocircline_cmat_def split: if_split_asm)
qed

(* ----------------------------------------------------------------- *)
subsection{* Some special oriented circlines and discs *}
(* ----------------------------------------------------------------- *)

lift_definition mk_ocircline :: "complex \<Rightarrow> complex \<Rightarrow> complex \<Rightarrow> complex \<Rightarrow> ocircline" is mk_circline_clmat
  done

text {* oriented unit circle and unit disc *}

lift_definition ounit_circle :: "ocircline" is unit_circle_clmat
  done

lemma pos_oriented_ounit_circle [simp]: 
  shows "pos_oriented ounit_circle"
  by (transfer, transfer, simp)

lemma of_ocircline_ounit_circle [simp]:
  shows "of_ocircline ounit_circle = unit_circle"
  by (transfer, transfer, simp)

lemma of_circline_unit_circle [simp]:
  shows "of_circline (unit_circle) = ounit_circle"
  by (transfer, transfer, simp)

lemma ocircline_set_ounit_circle [simp]:
  shows "ocircline_set ounit_circle = circline_set unit_circle"
  apply (subst of_circline_unit_circle[symmetric])
  apply (subst ocircline_set_of_circline)
  apply simp
  done

definition unit_disc :: "complex_homo set" where
  "unit_disc = disc ounit_circle"

definition unit_disc_compl :: "complex_homo set" where
  "unit_disc_compl = disc_compl ounit_circle"

definition unit_circle_set :: "complex_homo set" where
  "unit_circle_set = circline_set unit_circle"

lemma zero_in_unit_disc [simp]:
  shows "0\<^sub>h \<in> unit_disc"
  unfolding unit_disc_def disc_def
  by (simp, transfer, transfer) (simp add: Let_def vec_cnj_def)

lemma one_notin_unit_dic [simp]: 
  shows "1\<^sub>h \<notin> unit_disc"
  unfolding unit_disc_def disc_def
  by (simp, transfer, transfer) (simp add: Let_def vec_cnj_def)

lemma inf_notin_unit_disc [simp]:
  shows "\<infinity>\<^sub>h \<notin> unit_disc"
  unfolding unit_disc_def disc_def
  by (simp, transfer, transfer) (simp add: Let_def vec_cnj_def)

lemma unit_disc_iff_cmod_lt_1 [simp]:
  shows "of_complex c \<in> unit_disc \<longleftrightarrow> cmod c < 1"
  unfolding unit_disc_def disc_def
  by (simp, transfer, transfer, simp add: vec_cnj_def cmod_def power2_eq_square)

lemma unit_disc_cmod_square_lt_1 [simp]:
  assumes "z \<in> unit_disc"
  shows "(cmod (to_complex z))\<^sup>2 < 1"
  using assms inf_or_of_complex[of z]
  by (auto simp add: abs_square_less_1)

lemma unit_disc_to_complex_inj:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  assumes "to_complex u = to_complex v"
  shows "u = v"
  using assms
  using inf_or_of_complex[of u] inf_or_of_complex[of v]
  by auto

lemma inversion_unit_disc [simp]: 
  shows "inversion ` unit_disc = unit_disc_compl"
  unfolding unit_disc_def unit_disc_compl_def disc_def disc_compl_def
proof safe
  fix x
  assume "in_ocircline ounit_circle x"
  thus "out_ocircline ounit_circle (inversion x)"
    unfolding inversion_def
    by (transfer, transfer, auto simp add: vec_cnj_def)
next
  fix x
  assume *: "out_ocircline ounit_circle x"
  show "x \<in> inversion ` Collect (in_ocircline ounit_circle)"
  proof (rule image_eqI)
    show "x = inversion (inversion x)"
      by auto
  next
    show "inversion x \<in> Collect (in_ocircline ounit_circle)"
      using *
      unfolding inversion_def
      by (simp, transfer, transfer, auto simp add: vec_cnj_def)
  qed
qed

lemma inversion_unit_disc_compl [simp]: 
  shows "inversion ` unit_disc_compl = unit_disc"
proof-
  have "inversion ` (inversion ` unit_disc) = unit_disc"
    by (auto simp del: inversion_unit_disc simp add: image_iff)
  thus ?thesis
    by simp
qed

lemma inversion_noteq_unit_disc:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "inversion u \<noteq> v"
proof-
  from assms
  have "inversion u \<in> unit_disc_compl"
    by (metis image_eqI inversion_unit_disc)
  thus ?thesis
    using assms
    unfolding unit_disc_def unit_disc_compl_def
    using disc_inter_disc_compl
    by fastforce
qed

lemma in_ocircline_ounit_circle_conjugate [simp]:
  assumes "in_ocircline ounit_circle z"
  shows "in_ocircline ounit_circle (conjugate z)"
  using assms
  by (transfer, transfer, auto simp add: vec_cnj_def)

lemma conjugate_unit_disc [simp]:
  "conjugate ` unit_disc = unit_disc"                
  unfolding unit_disc_def disc_def
  apply (auto simp add: image_iff)
  apply (rule_tac x="conjugate x" in exI, simp)
  done

lemma conjugate_in_unit_disc [simp]:
  assumes "z \<in> unit_disc"
  shows "conjugate z \<in> unit_disc"
  using conjugate_unit_disc
  using assms
  by blast

lemma out_ocircline_ounit_circle_conjugate [simp]:
  assumes "out_ocircline ounit_circle z"
  shows "out_ocircline ounit_circle (conjugate z)"
  using assms
  by (transfer, transfer, auto simp add: vec_cnj_def)

lemma conjugate_unit_disc_compl [simp]:
  "conjugate ` unit_disc_compl = unit_disc_compl"                
  unfolding unit_disc_compl_def disc_compl_def
  apply (auto simp add: image_iff)
  apply (rule_tac x="conjugate x" in exI, simp)
  done

lemma conjugate_in_unit_disc_compl [simp]:
  assumes "z \<in> unit_disc_compl"
  shows "conjugate z \<in> unit_disc_compl"
  using conjugate_unit_disc_compl
  using assms
  by blast

text{* Oriented x axis and lower half plane *}

lift_definition o_x_axis :: "ocircline" is x_axis_clmat
done

lemma o_x_axis_pos_oriented [simp]:
  shows "pos_oriented o_x_axis"
  by (transfer, transfer, simp)

lemma of_ocircline_o_x_axis [simp]: 
  shows "of_ocircline o_x_axis = x_axis"
  by (transfer, transfer, simp)

lemma of_circline_x_axis [simp]:
  shows "of_circline x_axis = o_x_axis"
  using of_circline_of_ocircline_pos_oriented[of o_x_axis]
  using o_x_axis_pos_oriented
  by simp

lemma ocircline_set_circline_set_x_axis [simp]: 
  shows "ocircline_set o_x_axis = circline_set x_axis"
  by (subst of_circline_x_axis[symmetric], subst ocircline_set_of_circline, simp)

lemma ii_in_disc_o_x_axis [simp]: "ii\<^sub>h \<notin> disc o_x_axis"
unfolding disc_def
by simp (transfer, transfer, simp add: Let_def vec_cnj_def)

lemma ii_notin_disc_o_x_axis [simp]: "ii\<^sub>h \<in> disc_compl o_x_axis"
unfolding disc_compl_def
by simp (transfer, transfer, simp add: Let_def vec_cnj_def)

lemma of_complex_in_o_x_axis_disc [simp]:
  shows "of_complex z \<in> disc o_x_axis \<longleftrightarrow> Im z < 0"
  unfolding disc_def
  by auto (transfer, transfer, simp add: vec_cnj_def)+

lemma inf_notin_disc_o_x_axis
  [simp]: "\<infinity>\<^sub>h \<notin> disc o_x_axis"
  unfolding disc_def
  by simp (transfer, transfer, simp add: vec_cnj_def)

lemma disc_o_x_axis:
  "disc o_x_axis = of_complex ` {z. Im z < 0}"
proof-
  {
    fix z
    assume "z \<in> disc o_x_axis"
    hence "\<exists> x. Im x < 0 \<and> z = of_complex x"
      using inf_or_of_complex[of z]
      by auto
  }
  thus ?thesis
    by (auto simp add: image_iff)
qed

text{* Oriented single point circline *}

lift_definition  o_circline_point_0 :: "ocircline" is circline_point_0_clmat
done

lemma of_ocircline_o_circline_point_0 [simp]: 
  shows "of_ocircline o_circline_point_0 = circline_point_0"
  by (transfer, transfer, simp)

text{* ??? *}

lemma "0\<^sub>h \<in> disc_compl (mk_ocircline (-1) (2*\<i>) (-2*\<i>) 1)"
  unfolding disc_compl_def
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)
lemma "\<not> pos_oriented (mk_ocircline (-1) (2*\<i>) (-2*\<i>) 1)"
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)
lemma "circline_type (mk_circline (-1) (2*\<i>) (-2*\<i>) 1) = -1"
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)

lemma "0\<^sub>h \<in> disc_compl (mk_ocircline 1 (2*\<i>) (-2*\<i>) 1)"
  unfolding disc_compl_def
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)
lemma "pos_oriented (mk_ocircline 1 (2*\<i>) (-2*\<i>) 1)"
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)
lemma "circline_type (mk_circline 1 (2*\<i>) (-2*\<i>) 1) = -1"
  by simp (transfer, transfer, simp add: hermitean_def mat_adj_def mat_cnj_def vec_cnj_def)

(* ----------------------------------------------------------------- *)
subsection{* Moebius action on oriented circlines and discs *}
(* ----------------------------------------------------------------- *)

lift_definition moebius_ocircline :: "moebius \<Rightarrow> ocircline \<Rightarrow> ocircline" is moebius_circline_mmat_clmat
  apply (transfer, transfer)
  apply simp
  apply ((erule exE)+, (erule conjE)+)
  apply (simp add: mat_inv_mult_sm)
  apply (rule_tac x="ka / Re (k * cnj k)" in exI, auto simp add: complex_mult_cnj_cmod power2_eq_square)
  done

lemma moebius_circline_ocircline:
  "moebius_circline M H = of_ocircline (moebius_ocircline M (of_circline H))"
  apply (transfer, simp add: of_circline_clmat_def', safe)
  apply (transfer, simp, rule_tac x="-1" in exI, simp)
  done

lemma moebius_ocircline_circline:
  "moebius_ocircline M H = of_circline (moebius_circline M (of_ocircline H)) \<or>
   moebius_ocircline M H = opposite_ocircline (of_circline (moebius_circline M (of_ocircline H)))"
  apply (transfer, simp add: of_circline_clmat_def', safe)
  apply (transfer, simp, rule_tac x="1" in exI, simp)
  apply (transfer, simp, erule_tac x="1" in allE, simp)
  done

lemma inj_moebius_ocircline
  [simp]: "inj (moebius_ocircline M)"
unfolding inj_on_def
proof (safe)
  fix H H'
  assume "moebius_ocircline M H = moebius_ocircline M H'"
  thus "H = H'"
  proof (transfer, transfer)
    fix M H H' :: complex_mat
    assume "mat_det M \<noteq> 0"
    let ?iM = "mat_inv M"
    assume "ocircline_eq_cmat (moebius_circline_cmat_cmat M H) (moebius_circline_cmat_cmat M H')"
    then obtain k where "congruence ?iM H' = congruence ?iM (cor k *\<^sub>s\<^sub>m H)" "k > 0"
      by (auto simp del: congruence_def simp add: congruence_scale_m)
    thus "ocircline_eq_cmat H H'"
      using `mat_det M \<noteq> 0` inj_congruence[of ?iM H' "cor k *\<^sub>s\<^sub>m H"] mat_det_inv[of M]
      by auto
  qed
qed

lemma [simp]:
  "moebius_ocircline id_moebius H = H"
  by (transfer, transfer) (force simp add: mat_adj_def mat_cnj_def)

lemma moebius_ocircline_comp [simp]:
  shows "moebius_ocircline (moebius_comp M1 M2) H = moebius_ocircline M1 (moebius_ocircline M2 H)"
  by (transfer, transfer, simp, rule_tac x=1 in exI, simp add: mat_inv_mult_mm mult_mm_assoc)

lemma moebius_ocircline_comp_inv_left [simp]:
  shows "moebius_ocircline (moebius_inv M) (moebius_ocircline M H) = H"
  by (subst moebius_ocircline_comp[symmetric]) simp

lemma moebius_ocircline_comp_inv_right [simp]:
  shows "moebius_ocircline M (moebius_ocircline (moebius_inv M) H) = H"
  by (subst moebius_ocircline_comp[symmetric]) simp

lemma moebius_ocircline_opposite_ocircline [simp]:
  shows "moebius_ocircline M (opposite_ocircline H) = opposite_ocircline (moebius_ocircline M H)"
  by (transfer, transfer, simp, rule_tac x=1 in exI, simp)

lemma ocircline_set_moebius_ocircline [simp]:
  shows "ocircline_set (moebius_ocircline M H) = moebius_pt M ` ocircline_set H" (is "?lhs = ?rhs")
proof-
  have "?rhs = circline_set (moebius_circline M (of_ocircline H))"
    by simp
  thus ?thesis
    using moebius_ocircline_circline[of M H]
    by auto
qed

lemma disc_moebius_ocircline [simp]:
  shows "disc (moebius_ocircline M H) = moebius_pt M ` (disc H)"
proof (safe)
  fix z
  assume "z \<in> disc H"
  thus "moebius_pt M z \<in> disc (moebius_ocircline M H)"
    unfolding disc_def
  proof (safe)
    assume "in_ocircline H z"
    thus "in_ocircline (moebius_ocircline M H) (moebius_pt M z)"
    proof (transfer, transfer)
      fix H M :: complex_mat and z :: complex_vec
      assume "mat_det M \<noteq> 0"
      assume "in_ocircline_cmat_cvec H z"
      thus "in_ocircline_cmat_cvec (moebius_circline_cmat_cmat M H) (moebius_pt_cmat_cvec M z)"
        using `mat_det M \<noteq> 0` quad_form_congruence[of M z]
        by simp
    qed
  qed
next
  fix z
  assume "z \<in> disc (moebius_ocircline M H)"
  thus "z \<in> moebius_pt M ` disc H"
    unfolding disc_def
  proof(safe)
    assume "in_ocircline (moebius_ocircline M H) z"
    show "z \<in> moebius_pt M ` Collect (in_ocircline H)"
    proof
      show "z = moebius_pt M (moebius_pt (moebius_inv M) z)"
        by simp
    next
      show "moebius_pt (moebius_inv M) z \<in> Collect (in_ocircline H)"
        using `in_ocircline (moebius_ocircline M H) z`
      proof (safe, transfer, transfer)
        fix M H :: complex_mat and z :: complex_vec
        assume "mat_det M \<noteq> 0"
        hence "congruence (mat_inv (mat_inv M)) (congruence (mat_inv M) H) = H"
          by (simp add: congruence_congruence_inv del: congruence_def)
        hence "quad_form z (congruence (mat_inv M) H) = quad_form (mat_inv M *\<^sub>m\<^sub>v z) H"
          using quad_form_congruence[of "mat_inv M" "z" "congruence (mat_inv M) H"]
          using `mat_det M \<noteq> 0` mat_det_inv[of "M"]
          by simp
        moreover
        assume "in_ocircline_cmat_cvec (moebius_circline_cmat_cmat M H) z"
        ultimately
        show "in_ocircline_cmat_cvec H (moebius_pt_cmat_cvec (moebius_inv_cmat M) z)"
          by simp
      qed
    qed
  qed
qed

(* TODO: Could be simplified by reducing to disc and ocircline_set *)
lemma disc_compl_moebius_ocircline [simp]:
  "disc_compl (moebius_ocircline M H) = moebius_pt M ` (disc_compl H)"
proof (safe)
  fix z
  assume "z \<in> disc_compl H"
  thus "moebius_pt M z \<in> disc_compl (moebius_ocircline M H)"
    unfolding disc_compl_def
  proof (safe)
    assume "out_ocircline H z"
    thus "out_ocircline (moebius_ocircline M H) (moebius_pt M z)"
    proof (transfer, transfer)
      fix H M :: complex_mat and z :: complex_vec
      assume "mat_det M \<noteq> 0"
      assume "out_ocircline_cmat_cvec H z"
      thus "out_ocircline_cmat_cvec (moebius_circline_cmat_cmat M H) (moebius_pt_cmat_cvec M z)"
        using `mat_det M \<noteq> 0` quad_form_congruence[of M z]
        by simp
    qed
  qed
next
  fix z
  assume "z \<in> disc_compl (moebius_ocircline M H)"
  thus "z \<in> moebius_pt M ` disc_compl H"
    unfolding disc_compl_def
  proof(safe)
    assume "out_ocircline (moebius_ocircline M H) z"
    show "z \<in> moebius_pt M ` Collect (out_ocircline H)"
    proof
      show "z = moebius_pt M (moebius_pt (moebius_inv M) z)"
        by simp
    next
      show "moebius_pt (moebius_inv M) z \<in> Collect (out_ocircline H)"
        using `out_ocircline (moebius_ocircline M H) z`
      proof (safe, transfer, transfer)
        fix M H :: complex_mat and z :: complex_vec
        assume "mat_det M \<noteq> 0"
        hence "congruence (mat_inv (mat_inv M)) (congruence (mat_inv M) H) = H"
          by (simp add: congruence_congruence_inv del: congruence_def)
        hence "quad_form z (congruence (mat_inv M) H) = quad_form (mat_inv M *\<^sub>m\<^sub>v z) H"
          using quad_form_congruence[of "mat_inv M" "z" "congruence (mat_inv M) H"]
          using `mat_det M \<noteq> 0` mat_det_inv[of "M"]
          by simp
        moreover
        assume "out_ocircline_cmat_cvec (moebius_circline_cmat_cmat M H) z"
        ultimately
        show "out_ocircline_cmat_cvec H (moebius_pt_cmat_cvec (moebius_inv_cmat M) z)"
          by simp
      qed
    qed
  qed
qed

(* ----------------------------------------------------------------- *)
subsection{* Orientation after Moebius transformations *}
(* ----------------------------------------------------------------- *)

lemma moebius_similarity_oriented_lines_to_oriented_lines:
  assumes "a \<noteq> 0"
  shows "\<infinity>\<^sub>h \<in> ocircline_set H \<longleftrightarrow> \<infinity>\<^sub>h \<in> ocircline_set (moebius_ocircline (moebius_similarity a b) H)"
  using moebius_similarity_lines_to_lines[OF `a \<noteq> 0`, of b "of_ocircline H"]
  by simp

lemma moebius_similarity_preserve_orientation':
  assumes "a \<noteq> 0" "\<infinity>\<^sub>h \<notin> ocircline_set H" "pos_oriented H"
  shows "pos_oriented (moebius_ocircline (moebius_similarity a b) H)"
proof-
  let ?M = "moebius_similarity a b"
  let ?H = "moebius_ocircline ?M H"
  have "\<infinity>\<^sub>h \<notin> ocircline_set ?H"
    using `\<infinity>\<^sub>h \<notin> ocircline_set H` moebius_similarity_oriented_lines_to_oriented_lines[OF `a \<noteq> 0`]
    by simp

  have "\<infinity>\<^sub>h \<in> disc_compl H"
    using `\<infinity>\<^sub>h \<notin> ocircline_set H` `pos_oriented H` pos_oriented_circle_inf[of H] in_on_out
    unfolding disc_def disc_compl_def ocircline_set_def
    by auto
  hence "\<infinity>\<^sub>h \<in> disc_compl ?H"
    using moebius_similarity_inf[OF `a \<noteq> 0`, of b]
    by force
  thus "pos_oriented ?H"
    using pos_oriented_circle_inf[of ?H] disc_inter_disc_compl[of ?H] `\<infinity>\<^sub>h \<notin> ocircline_set ?H`
    by auto
qed

lemma moebius_similarity_preserve_orientation:
  assumes "a \<noteq> 0" "\<infinity>\<^sub>h \<notin> ocircline_set H"
  shows "pos_oriented H \<longleftrightarrow> pos_oriented(moebius_ocircline (moebius_similarity a b) H)"
proof-
  let ?M = "moebius_similarity a b"
  let ?H = "moebius_ocircline ?M H"
  have "\<infinity>\<^sub>h \<notin> ocircline_set ?H"
    using `\<infinity>\<^sub>h \<notin> ocircline_set H` moebius_similarity_oriented_lines_to_oriented_lines[OF `a \<noteq> 0`]
    by simp

  have *: "H = moebius_ocircline (- moebius_similarity a b) ?H"
    by simp
  show ?thesis
    using `a \<noteq> 0`
    using moebius_similarity_preserve_orientation' [OF `a \<noteq> 0` `\<infinity>\<^sub>h \<notin> ocircline_set H`]
    using moebius_similarity_preserve_orientation'[OF _   `\<infinity>\<^sub>h \<notin> ocircline_set ?H`, of "1/a" "-b/a"]
    using moebius_similarity_inv[of a b, OF `a \<noteq> 0`]  *
    by auto
qed

lemma reciprocal_preserve_orientation:
  assumes "0\<^sub>h \<in> disc_compl H"
  shows "pos_oriented (moebius_ocircline moebius_reciprocal H)"
proof-
  have "\<infinity>\<^sub>h \<in> disc_compl (moebius_ocircline moebius_reciprocal H)"
    using assms
    by force
  thus "pos_oriented (moebius_ocircline moebius_reciprocal H)"
    using pos_oriented_circle_inf[of "moebius_ocircline moebius_reciprocal H"]
    using disc_inter_disc_compl[of "moebius_ocircline moebius_reciprocal H"]
    using disc_compl_inter_ocircline_set[of "moebius_ocircline moebius_reciprocal H"]
    by auto
qed

lemma reciprocal_not_preserve_orientation:
  assumes "0\<^sub>h \<in> disc H"
  shows "\<not> pos_oriented (moebius_ocircline moebius_reciprocal H)"
proof-
  let ?H = "moebius_ocircline moebius_reciprocal H"
  have "\<infinity>\<^sub>h \<in> disc ?H"
    using assms
    by force
  thus "\<not> pos_oriented ?H"
    using pos_oriented_circle_inf[of ?H] disc_inter_ocircline_set[of ?H]
    by auto
qed

lemma pole_in_disc:
  assumes "M = mk_moebius a b c d" "c \<noteq> 0" "a*d - b*c \<noteq> 0"
  assumes "is_pole M z" "z \<in> disc H"
  shows "\<not> pos_oriented (moebius_ocircline M H)"
proof-
  let ?t1 = "moebius_translation (a / c)"
  let ?rd = "moebius_rotation_dilatation ((b * c - a * d) / (c * c))"
  let ?r =  "moebius_reciprocal"
  let ?t2 = "moebius_translation (d / c)"

  have "0\<^sub>h = moebius_pt (moebius_translation (d/c)) z"
    using pole_mk_moebius[of a b c d z] assms
    by simp

  have "z \<notin> ocircline_set H"
    using `z \<in> disc H` disc_inter_ocircline_set[of H]
    by blast      
                                              
  hence "0\<^sub>h \<notin> ocircline_set (moebius_ocircline ?t2 H)"
    using `0\<^sub>h = moebius_pt ?t2 z`
    using moebius_pt_neq_I[of z _ ?t2]
    by force

  hence *: "\<infinity>\<^sub>h \<notin> ocircline_set (moebius_ocircline (?r + ?t2) H)"
    by auto (metis reciprocal_inf_iff image_eqI)
    
  hence **: "\<infinity>\<^sub>h \<notin> ocircline_set (moebius_ocircline (?rd + ?r + ?t2) H)"
    using `a*d - b*c \<noteq> 0` `c \<noteq> 0`
    unfolding moebius_rotation_dilatation_def
    using moebius_similarity_oriented_lines_to_oriented_lines[of _ "moebius_ocircline (?r + ?t2) H"]
    by simp    

  have "\<not> pos_oriented (moebius_ocircline (?r + ?t2) H)"
    using pole_mk_moebius[of a b c d z] assms
    using reciprocal_not_preserve_orientation
    by force
  hence "\<not> pos_oriented (moebius_ocircline (?rd + ?r + ?t2) H)"
    using *
    using `a*d - b*c \<noteq> 0` `c \<noteq> 0`
    using moebius_similarity_preserve_orientation[of _ "moebius_ocircline (?r + ?t2) H"]
    unfolding moebius_rotation_dilatation_def
    by simp    
  hence "\<not> pos_oriented (moebius_ocircline (?t1 + ?rd + ?r + ?t2) H)"
    using **
    using moebius_similarity_preserve_orientation[of _ "moebius_ocircline (?rd + ?r + ?t2) H"]
    unfolding moebius_translation_def
    by simp

  thus ?thesis
    using assms
    by simp (subst moebius_decomposition, simp_all)
qed

(* TODO: join with the previous lemma to avoid copy-paste proofs *)
lemma pole_in_disc_compl:
  assumes "M = mk_moebius a b c d" "c \<noteq> 0" "a*d - b*c \<noteq> 0"
  assumes "is_pole M z" "z \<in> disc_compl H"
  shows "pos_oriented (moebius_ocircline M H)"
proof-
  let ?t1 = "moebius_translation (a / c)"
  let ?rd = "moebius_rotation_dilatation ((b * c - a * d) / (c * c))"
  let ?r = "moebius_reciprocal"
  let ?t2 = "moebius_translation (d / c)"

  have "0\<^sub>h = moebius_pt (moebius_translation (d/c)) z"
    using pole_mk_moebius[of a b c d z] assms
    by simp

  have "z \<notin> ocircline_set H"
    using `z \<in> disc_compl H` disc_compl_inter_ocircline_set[of H]
    by blast
  hence "0\<^sub>h \<notin> ocircline_set (moebius_ocircline ?t2 H)"
    using `0\<^sub>h = moebius_pt ?t2 z`
    using moebius_pt_neq_I[of z _ ?t2]
    by force
  hence *: "\<infinity>\<^sub>h \<notin> ocircline_set (moebius_ocircline (?r + ?t2) H)"
    by auto (metis reciprocal_inf_iff image_eqI)

  hence **: "\<infinity>\<^sub>h \<notin> ocircline_set (moebius_ocircline (?rd + ?r + ?t2) H)"
    using `a*d - b*c \<noteq> 0` `c \<noteq> 0`
    unfolding moebius_rotation_dilatation_def
    using moebius_similarity_oriented_lines_to_oriented_lines[of _ "moebius_ocircline (?r + ?t2) H"]
    by simp    

  have "pos_oriented (moebius_ocircline (?r + ?t2) H)"
    using pole_mk_moebius[of a b c d z] assms
    using reciprocal_preserve_orientation
    by force
  hence "pos_oriented (moebius_ocircline (?rd + ?r + ?t2) H)"
    using *
    using `a*d - b*c \<noteq> 0` `c \<noteq> 0`
    using moebius_similarity_preserve_orientation[of _ "moebius_ocircline (?r + ?t2) H"]
    unfolding moebius_rotation_dilatation_def
    by simp
  hence "pos_oriented (moebius_ocircline (?t1 + ?rd + ?r + ?t2) H)"
    using **
    using moebius_similarity_preserve_orientation[of _ "moebius_ocircline (?rd + ?r + ?t2) H"]
    unfolding moebius_translation_def
    by simp

  thus ?thesis
    using assms
    by simp (subst moebius_decomposition, simp_all)
qed

(* ----------------------------------------------------------------- *)
subsection{* Oriented circlines uniqueness *}
(* ----------------------------------------------------------------- *)

lemma ocircline_01inf:
  assumes "0\<^sub>h \<in> ocircline_set H \<and> 1\<^sub>h \<in> ocircline_set H \<and> \<infinity>\<^sub>h \<in> ocircline_set H"
  shows "H = o_x_axis \<or> H = opposite_ocircline o_x_axis"
proof-
  have "0\<^sub>h \<in> circline_set (of_ocircline H) \<and> 1\<^sub>h \<in> circline_set (of_ocircline H) \<and> \<infinity>\<^sub>h \<in> circline_set (of_ocircline H)"
    using assms
    by simp
  hence "of_ocircline H = x_axis"
    using unique_circline_01inf'
    by auto
  thus "H = o_x_axis \<or> H = opposite_ocircline o_x_axis"
    by (metis inj_of_ocircline of_ocircline_o_x_axis)
qed

lemma unique_ocircline_01inf:
  "\<exists>! H. 0\<^sub>h \<in> ocircline_set H \<and> 1\<^sub>h \<in> ocircline_set H \<and> \<infinity>\<^sub>h \<in> ocircline_set H \<and> ii\<^sub>h \<notin> disc H"
proof
  show "0\<^sub>h \<in> ocircline_set o_x_axis \<and> 1\<^sub>h \<in> ocircline_set o_x_axis \<and> \<infinity>\<^sub>h \<in> ocircline_set o_x_axis \<and> ii\<^sub>h \<notin> disc o_x_axis"
    by simp
next
  fix H
  assume "0\<^sub>h \<in> ocircline_set H \<and> 1\<^sub>h \<in> ocircline_set H \<and> \<infinity>\<^sub>h \<in> ocircline_set H \<and> ii\<^sub>h \<notin> disc H"
  hence "0\<^sub>h \<in> ocircline_set H \<and> 1\<^sub>h \<in> ocircline_set H \<and> \<infinity>\<^sub>h \<in> ocircline_set H" "ii\<^sub>h \<notin> disc H"
    by auto
  hence "H = o_x_axis \<or> H = opposite_ocircline o_x_axis"
    using ocircline_01inf
    by simp
  thus "H = o_x_axis"           
    using `ii\<^sub>h \<notin> disc H`
    by auto
qed

lemma unique_ocircline_set:
  assumes "A \<noteq> B" "A \<noteq> C" "B \<noteq> C"
  shows "\<exists>! H. pos_oriented H \<and> (A \<in> ocircline_set H \<and> B \<in> ocircline_set H \<and> C \<in> ocircline_set H)"
proof-
  obtain M where *: "moebius_pt M A = 0\<^sub>h"  "moebius_pt M B = 1\<^sub>h" "moebius_pt M C = \<infinity>\<^sub>h"
    using ex_moebius_01inf[OF assms]
    by auto
  let ?iM = "moebius_pt (moebius_inv M)"
  have **: "?iM 0\<^sub>h = A"  "?iM 1\<^sub>h = B"  "?iM \<infinity>\<^sub>h = C"
    using *
    by (auto simp add: moebius_pt_invert)
  let ?H = "moebius_ocircline (moebius_inv M) o_x_axis"
  have 1: "A \<in> ocircline_set ?H" "B \<in> ocircline_set ?H" "C \<in> ocircline_set ?H"
    using **
    by auto
  have 2: "\<And> H'. A \<in> ocircline_set H' \<and> B \<in> ocircline_set H' \<and> C \<in> ocircline_set H' \<Longrightarrow> H' = ?H \<or> H' = opposite_ocircline ?H"
  proof-
    fix H'
    let ?H' = "ocircline_set H'" and ?H'' = "ocircline_set (moebius_ocircline M H')"
    assume "A \<in> ocircline_set H' \<and> B \<in> ocircline_set H' \<and> C \<in> ocircline_set H'"
    hence "moebius_pt M A \<in> ?H''" "moebius_pt M B \<in> ?H''" "moebius_pt M C \<in> ?H''"
      by auto
    hence "0\<^sub>h \<in> ?H''" "1\<^sub>h \<in> ?H''"  "\<infinity>\<^sub>h \<in> ?H''"
      using *
      by auto
    hence "moebius_ocircline M H' = o_x_axis \<or> moebius_ocircline M H' = opposite_ocircline o_x_axis"
      using ocircline_01inf
      by auto
    hence "o_x_axis = moebius_ocircline M H' \<or>  o_x_axis = opposite_ocircline (moebius_ocircline M H')"
      by auto
    thus "H' = ?H \<or> H' = opposite_ocircline ?H"
    proof
      assume *: "o_x_axis = moebius_ocircline M H'"
      show "H' = moebius_ocircline (moebius_inv M) o_x_axis \<or> H' = opposite_ocircline (moebius_ocircline (moebius_inv M) o_x_axis)"
        by (rule disjI1) (subst *, simp)
    next
      assume *: "o_x_axis = opposite_ocircline (moebius_ocircline M H')"
      show "H' = moebius_ocircline (moebius_inv M) o_x_axis \<or> H' = opposite_ocircline (moebius_ocircline (moebius_inv M) o_x_axis)"
        by (rule disjI2) (subst *, simp)
    qed
  qed

  show ?thesis (is "\<exists>! x. ?P x")
  proof (cases "pos_oriented ?H")
    case True
    show ?thesis
    proof
      show "?P ?H"
        using 1 True
        by auto
    next
      fix H
      assume "?P H"
      thus "H = ?H"
        using 1 2[of H] True
        by auto
    qed
  next
    case False
    let ?OH = "opposite_ocircline ?H"
    show ?thesis
    proof
      show "?P ?OH"
        using 1 False
        by auto
    next
      fix H
      assume "?P H"
      thus "H = ?OH"
        using False 2[of H]
        by auto
    qed
  qed
qed

lemma ocircline_set_0h:
  assumes "ocircline_set H = {0\<^sub>h}"
  shows "H = o_circline_point_0 \<or> H = opposite_ocircline (o_circline_point_0)"
proof-
  have "of_ocircline H = circline_point_0"
    using assms
    using unique_circline_type_zero_0' card_eq1_circline_type_zero[of "of_ocircline H"]
    by auto
  thus ?thesis
    by (metis inj_of_ocircline of_ocircline_o_circline_point_0)
qed


(* ----------------------------------------------------------------- *)
subsection{* Angle between circlines *}
(* ----------------------------------------------------------------- *)

fun mat_det_12 :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> complex" where
 "mat_det_12 (A1, B1, C1, D1) (A2, B2, C2, D2) = A1*D2 + A2*D1 - B1*C2 - B2*C1"

lemma mat_det_12_mm_l [simp]: "mat_det_12 (M *\<^sub>m\<^sub>m A) (M *\<^sub>m\<^sub>m B) = mat_det M * mat_det_12 A B"
by (cases M, cases A, cases B) (simp add: field_simps)

lemma mat_det_12_mm_r [simp]: "mat_det_12 (A *\<^sub>m\<^sub>m M) (B *\<^sub>m\<^sub>m M) = mat_det M * mat_det_12 A B"
by (cases M, cases A, cases B) (simp add: field_simps)

lemma mat_det_12_sm_l [simp]: "mat_det_12 (k *\<^sub>s\<^sub>m A) B = k * mat_det_12 A B"
by (cases A, cases B) (simp add: field_simps)

lemma mat_det_12_sm_r [simp]: "mat_det_12 A (k *\<^sub>s\<^sub>m B) = k * mat_det_12 A B"
by (cases A, cases B) (simp add: field_simps)

lemma mat_det_12_congruence [simp]:
  "mat_det_12 (congruence M A) (congruence M B) = (cor ((cmod (mat_det M))\<^sup>2)) * mat_det_12 A B"
  unfolding congruence_def
by ((subst mult_mm_assoc[symmetric])+, subst mat_det_12_mm_l, subst mat_det_12_mm_r, subst mat_det_adj) (auto simp add: field_simps complex_mult_cnj_cmod)


definition cos_angle_cmat :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> real" where
 [simp]: "cos_angle_cmat H1 H2 = - Re (mat_det_12 H1 H2) / (2 * (sqrt (Re (mat_det H1 * mat_det H2))))"

lift_definition cos_angle_clmat :: "circline_mat \<Rightarrow> circline_mat \<Rightarrow> real" is cos_angle_cmat
  done

lemma cos_angle_den_scale [simp]:
  assumes "k1 > 0" "k2 > 0"
  shows "sqrt (Re ((k1\<^sup>2 * mat_det H1) * (k2\<^sup>2 * mat_det H2))) = k1 * k2 * sqrt (Re (mat_det H1 * mat_det H2))"
proof-
  let ?lhs = "(k1\<^sup>2 * mat_det H1) * (k2\<^sup>2 * mat_det H2)"
  let ?rhs = "mat_det H1 * mat_det H2"
  have 1: "?lhs = (k1\<^sup>2*k2\<^sup>2) * ?rhs"
    by simp
  hence "Re ?lhs = (k1\<^sup>2*k2\<^sup>2) * Re ?rhs"
    by (simp add: field_simps)
  thus ?thesis
    using assms
    by (simp add: real_sqrt_mult)
qed

lift_definition cos_angle :: "ocircline \<Rightarrow> ocircline \<Rightarrow> real" is cos_angle_clmat
proof transfer
  fix H1 H2 H1' H2'
  assume "ocircline_eq_cmat H1 H1'" "ocircline_eq_cmat H2 H2'"
  then obtain k1 k2 :: real where
  *:  "k1 > 0" "H1' = cor k1 *\<^sub>s\<^sub>m H1"
      "k2 > 0" "H2' = cor k2 *\<^sub>s\<^sub>m H2"
    by auto
  thus "cos_angle_cmat H1 H2 = cos_angle_cmat H1' H2'"
    unfolding cos_angle_cmat_def
    apply (subst *)+
    apply (subst mat_det_12_sm_l, subst mat_det_12_sm_r)
    apply (subst mat_det_mult_sm)+
    apply (subst power2_eq_square[symmetric])+
    apply (subst cos_angle_den_scale, simp, simp)
    apply simp
    done
qed

lemma cos_angle_opposite1
  [simp]: "cos_angle (opposite_ocircline H) H' = - cos_angle H H'"
  by (transfer, transfer, simp)

lemma cos_angle_opposite2
  [simp]: "cos_angle H (opposite_ocircline H') = - cos_angle H H'"
  by (transfer, transfer, simp)

(* ----------------------------------------------------------------- *)
subsubsection{* Connection with the elementary angle definition between circles *}
(* ----------------------------------------------------------------- *)

lemma Re_sgn:
  assumes "is_real A" "A \<noteq> 0"
  shows "Re (sgn A) = sgn_bool (Re A > 0)"
using assms
using MoreComplex.Re_sgn complex_eq_if_Re_eq
by auto

lemma Re_mult_real3:
  assumes "is_real z1" "is_real z2" "is_real z3"
  shows "Re (z1 * z2 * z3) = Re z1 * Re z2 * Re z3"
using assms
by (metis Re_mult_real mult_reals)

lemma [simp]: "sgn (sqrt x) = sgn x"
  by (simp add: sgn_root sqrt_def)

lemma real_circle_sgn_r:
  assumes "is_circle H" "(a, r) = euclidean_circle H"
  shows "sgn r = - circline_type H"
using assms
proof (transfer, transfer)
  fix H :: complex_mat and a r
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where HH: "H = (A, B, C, D)"
    by (cases H) auto
  hence "is_real A" "is_real D"
    using hermitean_elems hh
    by auto
  assume "\<not> circline_A0_cmat H" "(a, r) = euclidean_circle_cmat H"
  hence "A \<noteq> 0"
    using `\<not> circline_A0_cmat H` HH
    by simp
  hence "Re A * Re A > 0"
    using `is_real A`
    using complex_eq_if_Re_eq not_real_square_gt_zero
    by fastforce
  thus "sgn r = - circline_type_cmat H"
    using HH `(a, r) = euclidean_circle_cmat H` `is_real A` `is_real D` `A \<noteq> 0`
    by (simp add: Re_divide_real sgn_minus[symmetric])
qed

lemma cos_angle_eq_cos_ang_circ:
  assumes
  "is_circle (of_ocircline H1)" "is_circle (of_ocircline H2)"
  "circline_type (of_ocircline H1) < 0" "circline_type (of_ocircline H2) < 0"
  "(a1, r1) = euclidean_circle (of_ocircline H1)" "(a2, r2) = euclidean_circle (of_ocircline H2)"
  "of_complex E \<in> ocircline_set H1 \<inter> ocircline_set H2"
  shows "cos_angle H1 H2 = cos (ang_circ E a1 a2 (pos_oriented H1) (pos_oriented H2))"
proof-
  let ?p1 = "pos_oriented H1" and ?p2 = "pos_oriented H2"
  have "E \<in> circle a1 r1" "E \<in> circle a2 r2"
    using classic_circle[of "of_ocircline H1" a1 r1]  classic_circle[of "of_ocircline H2" a2 r2]
    using assms of_complex_inj
    by auto
  hence *: "cdist E a1 = r1" "cdist E a2 = r2"
    unfolding circle_def
    by (simp_all add: norm_minus_commute)
  have "r1 > 0" "r2 > 0"
    using assms(1-6) real_circle_sgn_r[of "of_ocircline H1" a1 r1]  real_circle_sgn_r[of "of_ocircline H2" a2 r2]
    by auto (metis neg_0_less_iff_less sgn_1_pos sgn_sgn)+
  hence "E \<noteq> a1" "E \<noteq> a2"
    using `cdist E a1 = r1` `cdist E a2 = r2`
    by auto

  let ?k = "sgn_bool (?p1 = ?p2)"
  let ?xx = "?k * (r1\<^sup>2 + r2\<^sup>2 - (cdist a2 a1)\<^sup>2) / (2 * r1 * r2)"

  have "cos (ang_circ E a1 a2 ?p1 ?p2) = ?xx"
    using law_of_cosines[of a2 a1 E] * `r1 > 0` `r2 > 0` cos_ang_circ_simp[OF `E \<noteq> a1` `E \<noteq> a2`]
    by (subst (asm) ang_vec_opposite_opposite'[OF `E \<noteq> a1`[symmetric] `E \<noteq> a2`[symmetric], symmetric]) simp
  moreover
  have "cos_angle H1 H2 = ?xx"
    using `r1 > 0` `r2 > 0`
    using `(a1, r1) = euclidean_circle (of_ocircline H1)` `(a2, r2) = euclidean_circle (of_ocircline H2)`
    using `is_circle (of_ocircline H1)` `is_circle (of_ocircline H2)`
    using `circline_type (of_ocircline H1) < 0` `circline_type (of_ocircline H2) < 0`
  proof (transfer, transfer)
    fix a1 r1 H1 H2 a2 r2
    assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" "hermitean H2 \<and> H2 \<noteq> mat_zero"
    obtain A1 B1 C1 D1 where HH1: "H1 = (A1, B1, C1, D1)"
      by (cases H1) auto
    obtain A2 B2 C2 D2 where HH2: "H2 = (A2, B2, C2, D2)"
      by (cases H2) auto
    have *: "is_real A1" "is_real A2" "is_real D1" "is_real D2" "cnj B1 = C1" "cnj B2 = C2"
      using hh hermitean_elems[of A1 B1 C1 D1] hermitean_elems[of A2 B2 C2 D2] HH1 HH2
      by auto
    have "cnj A1 = A1" "cnj A2 = A2"
      using `is_real A1` `is_real A2`
      by (case_tac[!] A1, case_tac[!] A2, auto simp add: Complex_eq)

    assume "\<not> circline_A0_cmat (id H1)" "\<not> circline_A0_cmat (id H2)"
    hence "A1 \<noteq> 0" "A2 \<noteq> 0"
      using HH1 HH2
      by auto
    hence "Re A1 \<noteq> 0" "Re A2 \<noteq> 0"
      using `is_real A1` `is_real A2`
      using complex.expand
      by auto

    assume "circline_type_cmat (id H1) < 0" "circline_type_cmat (id H2) < 0"
    assume "(a1, r1) = euclidean_circle_cmat (id H1)" "(a2, r2) = euclidean_circle_cmat (id H2)"
    assume "r1 > 0" "r2 > 0"

    let ?D12 = "mat_det_12 H1 H2" and ?D1 = "mat_det H1" and ?D2 = "mat_det H2"
    let ?x1 = "(cdist a2 a1)\<^sup>2 - r1\<^sup>2 - r2\<^sup>2" and ?x2 = "2*r1*r2"
    let ?x = "?x1 / ?x2"
    have *:  "Re (?D12) / (2 * (sqrt (Re (?D1 * ?D2)))) = Re (sgn A1) * Re (sgn A2) * ?x"
    proof-
      let ?M1 = "(A1, B1, C1, D1)" and ?M2 = "(A2, B2, C2, D2)"
      let ?d1 = "B1 * C1 - A1 * D1" and ?d2 = "B2 * C2 - A2 * D2"
      have "Re ?d1 > 0" "Re ?d2 > 0"
        using HH1 HH2 `circline_type_cmat (id H1) < 0`  `circline_type_cmat (id H2) < 0`
        by auto
      hence **: "Re (?d1 / (A1 * A1)) > 0" "Re (?d2 / (A2 * A2)) > 0"
        using `is_real A1` `is_real A2` `A1 \<noteq> 0` `A2 \<noteq> 0`
        by (subst Re_divide_real, simp_all add: complex_neq_0 power2_eq_square)+
      have ***: "is_real (?d1 / (A1 * A1)) \<and> is_real (?d2 / (A2 * A2))"
        using `is_real A1`  `is_real A2` `A1 \<noteq> 0` `A2 \<noteq> 0` `cnj B1 = C1`[symmetric] `cnj B2 = C2`[symmetric] `is_real D1` `is_real D2`
        by (subst div_reals, simp, simp, simp)+

      have "cor ?x = mat_det_12 ?M1 ?M2 / (2 * sgn A1 * sgn A2 * cor (sqrt (Re ?d1) * sqrt (Re ?d2)))"
      proof-
        have "A1*A2*cor ?x1 = mat_det_12 ?M1 ?M2"
        proof-
          have 1: "A1*A2*(cor ((cdist a2 a1)\<^sup>2)) = ((B2*A1 - A2*B1)*(C2*A1 - C1*A2)) / (A1*A2)"
            using `(a1, r1) = euclidean_circle_cmat (id H1)` `(a2, r2) = euclidean_circle_cmat (id H2)`
            unfolding cdist_def cmod_square
            using HH1 HH2 * `A1 \<noteq> 0` `A2 \<noteq> 0` `cnj A1 = A1` `cnj A2 = A2`
            unfolding Let_def
            apply (subst complex_of_real_Re)
            apply (simp add: field_simps)
            apply (simp add: complex_mult_cnj_cmod power2_eq_square)
            apply (simp add: field_simps)
            done
          have 2: "A1*A2*cor (-r1\<^sup>2) = A2*D1 - B1*C1*A2/A1"
            using `(a1, r1) = euclidean_circle_cmat (id H1)`
            using HH1 ** * *** `A1 \<noteq> 0`
            apply (simp add: power2_eq_square)
            apply (subst complex_of_real_Re, simp)
            apply (simp add: field_simps)
            done
          have 3: "A1*A2*cor (-r2\<^sup>2) = A1*D2 - B2*C2*A1/A2"
            using `(a2, r2) = euclidean_circle_cmat (id H2)`
            using HH2 ** * *** `A2 \<noteq> 0`
            apply (simp add: power2_eq_square)
            apply (subst complex_of_real_Re, simp)
            apply (simp add: field_simps)
            done
          have "A1*A2*cor((cdist a2 a1)\<^sup>2) + A1*A2*cor(-r1\<^sup>2) + A1*A2*cor(-r2\<^sup>2) = mat_det_12 ?M1 ?M2"
            using `A1 \<noteq> 0` `A2 \<noteq> 0`
            by (subst 1, subst 2, subst 3) (simp add: field_simps)
          thus ?thesis
            by (simp add: field_simps)
        qed

        moreover

        have "A1 * A2 * cor (?x2) = 2 * sgn A1 * sgn A2 * cor (sqrt (Re ?d1) * sqrt (Re ?d2))"
        proof-
          have 1: "sqrt (Re (?d1/ (A1 * A1))) = sqrt (Re ?d1) / \<bar>Re A1\<bar>"
            using `A1 \<noteq> 0` `is_real A1`
            by (subst Re_divide_real, simp, simp, subst real_sqrt_divide, simp)

          have 2: "sqrt (Re (?d2/ (A2 * A2))) = sqrt (Re ?d2) / \<bar>Re A2\<bar>"
            using `A2 \<noteq> 0` `is_real A2`
            by (subst Re_divide_real, simp, simp, subst real_sqrt_divide, simp)
          have "sgn A1 = A1 / cor \<bar>Re A1\<bar>"
            using `is_real A1`
            unfolding sgn_eq
            by (simp add: cmod_eq_Re)
          moreover
          have "sgn A2 = A2 / cor \<bar>Re A2\<bar>"
            using `is_real A2`
            unfolding sgn_eq
            by (simp add: cmod_eq_Re)
          ultimately
          show ?thesis
            using `(a1, r1) = euclidean_circle_cmat (id H1)` `(a2, r2) = euclidean_circle_cmat (id H2)`  HH1 HH2
            using *** `is_real A1` `is_real A2`
            by simp (subst 1, subst 2, simp)
        qed

        ultimately

        have "(A1 * A2 * cor ?x1) / (A1 * A2 * (cor ?x2)) =
               mat_det_12 ?M1 ?M2 / (2 * sgn A1 * sgn A2 * cor (sqrt (Re ?d1) * sqrt (Re ?d2)))"
          by simp
        thus ?thesis
          using `A1 \<noteq> 0` `A2 \<noteq> 0`
          by simp
      qed
      hence "cor ?x * sgn A1 * sgn A2 = mat_det_12 ?M1 ?M2 / (2 * cor (sqrt (Re ?d1) * sqrt (Re ?d2)))"
        using `A1 \<noteq> 0` `A2 \<noteq> 0`
        by (simp add: sgn_zero_iff)
      moreover
      have "Re (cor ?x * sgn A1 * sgn A2) = Re (sgn A1) * Re (sgn A2) * ?x"
      proof-
        have "is_real (cor ?x)" "is_real (sgn A1)" "is_real (sgn A2)"
          using `is_real A1` `is_real A2` Im_complex_of_real[of ?x]
          by auto
        thus ?thesis
          using Re_complex_of_real[of ?x]
          by (subst Re_mult_real3, auto simp add: field_simps)
      qed
      moreover
      have *: "sqrt (Re ?D1) * sqrt (Re ?D2) = sqrt (Re ?d1) * sqrt (Re ?d2)"
        using HH1 HH2
        by (subst real_sqrt_mult[symmetric])+ (simp add: field_simps)
      have "2 * (sqrt (Re (?D1 * ?D2))) \<noteq> 0"
        using `Re ?d1 > 0`  `Re ?d2 > 0` HH1 HH2 `is_real A1` `is_real A2`  `is_real D1` `is_real D2`
        using hh mat_det_hermitean_real[of "H1"]
        by (subst Re_mult_real, auto)
      hence **: "Re (?D12 / (2 * cor (sqrt (Re (?D1 * ?D2))))) = Re (?D12) / (2 * (sqrt (Re (?D1 * ?D2))))"
        using `Re ?d1 > 0`  `Re ?d2 > 0` HH1 HH2 `is_real A1` `is_real A2`  `is_real D1` `is_real D2`
        by (subst Re_divide_real) auto
      have "Re (mat_det_12 ?M1 ?M2 / (2 * cor (sqrt (Re ?d1) * sqrt (Re ?d2)))) = Re (?D12) / (2 * (sqrt (Re (?D1 * ?D2))))"
        using HH1 HH2 hh mat_det_hermitean_real[of "H1"]
        by (subst **[symmetric], subst Re_mult_real, simp, subst real_sqrt_mult, subst *, simp)
      ultimately
      show ?thesis
        by simp
    qed
    have **: "pos_oriented_cmat H1 \<longleftrightarrow> Re A1 > 0"  "pos_oriented_cmat H2 \<longleftrightarrow> Re A2 > 0"
      using `Re A1 \<noteq> 0` HH1  `Re A2 \<noteq> 0` HH2
      by auto
    show "cos_angle_cmat H1 H2 = sgn_bool (pos_oriented_cmat H1 = pos_oriented_cmat H2) * (r1\<^sup>2 + r2\<^sup>2 - (cdist a2 a1)\<^sup>2) /  (2 * r1 * r2)"
      unfolding Let_def
      using `r1 > 0` `r2 > 0`
      unfolding cos_angle_cmat_def
      apply (subst divide_minus_left)
      apply (subst *)
      apply (subst Re_sgn[OF `is_real A1` `A1 \<noteq> 0`], subst Re_sgn[OF `is_real A2` `A2 \<noteq> 0`])
      apply (subst **, subst **)
      apply (simp add: field_simps)
      done
  qed
  ultimately
  show ?thesis
    by simp
qed

(* ----------------------------------------------------------------- *)
subsection{* Perpendicularity *}
(* ----------------------------------------------------------------- *)

definition perpendicular where
 "perpendicular H1 H2 \<longleftrightarrow> cos_angle (of_circline H1) (of_circline H2) = 0"

lemma perpendicular_sym: "perpendicular H1 H2 \<longleftrightarrow> perpendicular H2 H1"
  unfolding perpendicular_def
  by (transfer, transfer, auto simp add: field_simps)


(* ----------------------------------------------------------------- *)
subsection{* Moebius transforms preserve angles and perpendicularity *}
(* ----------------------------------------------------------------- *)

lemma moebius_preserve_circline_angle [simp]:
  "cos_angle (moebius_ocircline M H1) (moebius_ocircline M H2) = cos_angle H1 H2 "
proof (transfer, transfer)
  fix H1 H2 M :: complex_mat
  assume hh: "mat_det M \<noteq> 0"
  show "cos_angle_cmat (moebius_circline_cmat_cmat M H1) (moebius_circline_cmat_cmat M H2) = cos_angle_cmat H1 H2"
    unfolding cos_angle_cmat_def moebius_circline_cmat_cmat_def
    unfolding Let_def mat_det_12_congruence mat_det_congruence
    using hh mat_det_inv[of M]
    apply (subst cor_squared[symmetric])+
    apply (subst cos_angle_den_scale, simp)
    apply (auto simp add: power2_eq_square real_sqrt_mult field_simps)
    done
qed

lemma perpendicular_moebius [simp]:
  assumes "perpendicular H1 H2"
  shows "perpendicular (moebius_circline M H1) (moebius_circline M H2)"
  using assms
  unfolding perpendicular_def
  using moebius_preserve_circline_angle[of M "of_circline H1" "of_circline H2"]
  using moebius_ocircline_circline[of M "of_circline H1"]
  using moebius_ocircline_circline[of M "of_circline H2"]
  by (auto simp del: moebius_preserve_circline_angle)

end