section{* Poincare disc model *}
theory Poincare
imports OrientedCirclines UnitCirclePreservingMoebius HyperbolicFunctions
begin

(* ------------------------------------------------------------------ *)
subsection{* Poincare lines *}
(* ------------------------------------------------------------------ *)

(* is_poincare_line - technical definition in terms of matrix coefficients *)

definition is_poincare_line_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "is_poincare_line_cmat H \<longleftrightarrow>
             (let (A, B, C, D) = H
               in hermitean (A, B, C, D) \<and> A = D \<and> (cmod B)\<^sup>2 > (cmod A)\<^sup>2)"

lift_definition is_poincare_line_clmat :: "circline_mat \<Rightarrow> bool" is is_poincare_line_cmat
  done

lift_definition is_poincare_line :: "circline \<Rightarrow> bool" is is_poincare_line_clmat
proof (transfer, transfer)
  fix H1 H2 :: complex_mat
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" "hermitean H2 \<and> H2 \<noteq> mat_zero"
  assume "circline_eq_cmat H1 H2"
  thus "is_poincare_line_cmat H1 \<longleftrightarrow> is_poincare_line_cmat H2"
    using hh
    by (cases H1, cases H2) (auto simp add: power_mult_distrib)
qed

lemma is_poincare_line_mk_circline:
  assumes "(A, B, C, D) \<in> hermitean_nonzero"
  shows "is_poincare_line (mk_circline A B C D) \<longleftrightarrow> (cmod B)\<^sup>2 > (cmod A)\<^sup>2 \<and> A = D"
  using assms
  by (transfer, transfer, auto simp add: Let_def)

(* Colinear points *)
definition poincare_colinear :: "complex_homo set \<Rightarrow> bool" where
  "poincare_colinear S \<longleftrightarrow> (\<exists> p. is_poincare_line p \<and> S \<subseteq> circline_set p)"

(* is_poincare_line - abstract characterisation - real circines perpendicular to the unit circle *)
lemma is_poincare_line_iff:
  shows "is_poincare_line H \<longleftrightarrow> circline_type H = -1 \<and> perpendicular H unit_circle"
  unfolding perpendicular_def
proof (simp, transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where *: "H = (A, B, C, D)"
    by (cases H, auto)
  have "is_real A" "is_real D" "C = cnj B"
    using hh * hermitean_elems
    by auto
  thus "is_poincare_line_cmat H \<longleftrightarrow>
         circline_type_cmat H = - 1 \<and> cos_angle_cmat (of_circline_cmat H) unit_circle_cmat = 0"
    using *
    by (simp add: sgn_1_neg complex_eq_if_Re_eq cmod_square power2_eq_square del: pos_oriented_cmat_def)
       (smt cmod_power2 power2_eq_square zero_power2)+
qed

lemma is_poincare_line_x_axis
  [simp]: "is_poincare_line x_axis"
  by (transfer, transfer) (auto simp add: hermitean_def mat_adj_def mat_cnj_def)

lemma not_is_poincare_line_unit_circle
  [simp]: "\<not> is_poincare_line unit_circle"
  by (transfer, transfer, simp)

(* TODO: other direction could also be useful (containing a point and its inverse implies orthogonality) *)
lemma is_poincare_line_inverse_point:
  assumes "is_poincare_line H" "u \<in> circline_set H"
  shows "inversion u \<in> circline_set H"
  using assms
  unfolding is_poincare_line_iff circline_set_def perpendicular_def inversion_def
  apply simp
proof (transfer, transfer)
  fix u H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero" "u \<noteq> vec_zero" and
         aa: "circline_type_cmat H = - 1 \<and> cos_angle_cmat (of_circline_cmat H) unit_circle_cmat = 0" "on_circline_cmat_cvec H u"
  obtain A B C D u1 u2 where *: "H = (A, B, C, D)" "u = (u1, u2)"
    by (cases H, cases u, auto)
  have "is_real A" "is_real D" "C = cnj B"
    using * hh hermitean_elems
    by auto
  moreover
  have "A = D"
    using aa(1) * `is_real A` `is_real D`
    by (auto simp del: pos_oriented_cmat_def simp add: complex.expand split: if_split_asm)
  thus "on_circline_cmat_cvec H (conjugate_cvec (reciprocal_cvec u))"
    using aa(2) *
    by (simp add: vec_cnj_def field_simps)
qed

lemma circline_inversion_poincare_line:
  assumes "is_poincare_line H"
  shows "circline_inversion H = H"
proof-
  obtain u v w where *: "u \<noteq> v" "v \<noteq> w" "u \<noteq> w" "{u, v, w} \<subseteq> circline_set H"
    using assms is_poincare_line_iff[of H]
    using circline_type_neg_card_gt3[of H]
    by auto
  hence "{inversion u, inversion v, inversion w} \<subseteq> circline_set (circline_inversion H)"
        "{inversion u, inversion v, inversion w} \<subseteq> circline_set H"
    using is_poincare_line_inverse_point[OF assms]
    by auto
  thus ?thesis
    using * unique_circline_set[of "inversion u" "inversion v" "inversion w"]
    by (metis insert_subset inversion_involution)
qed

(* The following proofs could probably be simplified by using some other approach *)
lemma ex_is_poincare_line_points':
  assumes i12: "i1 \<in> circline_set H \<inter> unit_circle_set" "i2 \<in> circline_set H \<inter> unit_circle_set" "i1 \<noteq> i2"
  assumes a: "a \<in> circline_set H" "a \<notin> unit_circle_set"
  shows "\<exists> b. b \<noteq> i1 \<and> b \<noteq> i2 \<and> b \<noteq> a \<and> b \<noteq> inversion a \<and> b \<in> circline_set H"
proof-
  have "inversion a \<notin> unit_circle_set"
    using `a \<notin> unit_circle_set` 
    unfolding unit_circle_set_def circline_set_def
    by (simp, metis inversion_id_iff_on_unit_circle inversion_involution)

  have "a \<noteq> inversion a"
    using `a \<notin> unit_circle_set` inversion_id_iff_on_unit_circle[of a]
    unfolding unit_circle_set_def circline_set_def
    by auto

  have "a \<noteq> i1" "a \<noteq> i2" "inversion a \<noteq> i1" "inversion a \<noteq> i2"
    using assms `inversion a \<notin> unit_circle_set`
    by auto

  then obtain b where cr2: "cross_ratio b i1 a i2 = of_complex 2"
    using `i1 \<noteq> i2`
    using ex_cross_ratio[of i1 a i2]
    by blast

  have distinct_b: "b \<noteq> i1" "b \<noteq> i2" "b \<noteq> a"
    using `i1 \<noteq> i2` `a \<noteq> i1` `a \<noteq> i2`
    using ex1_cross_ratio[of i1 a i2]
    using cross_ratio_0[of i1 a i2] cross_ratio_1[of i1 a i2] cross_ratio_inf[of i1 i2 a]
    using cr2
    by auto

  hence "b \<in> circline_set H" 
    using assms four_points_on_circline_iff_cross_ratio_real[of b i1 a i2] cr2
    using unique_circline_set[of i1 i2 a]
    by auto

  moreover

  have "b \<noteq> inversion a"
  proof (rule ccontr)
    assume *: "\<not> ?thesis"
    have "inversion i1 = i1" "inversion i2 = i2"
      using i12
      unfolding unit_circle_set_def
      by auto
    hence "cross_ratio (inversion a) i1 a i2 = cross_ratio a i1 (inversion a) i2"
      using * cross_ratio_inversion[of i1 a i2 b] `a \<noteq> i1` `a \<noteq> i2` `i1 \<noteq> i2` `b \<noteq> i1`
      using four_points_on_circline_iff_cross_ratio_real[of b i1 a i2]
      using i12 distinct_b conjugate_id_iff[of "cross_ratio b i1 a i2"]
      using i12 a `b \<in> circline_set H`            
      by auto
    hence "cross_ratio (inversion a) i1 a i2 \<noteq> of_complex 2"
      using cross_ratio_commute_13[of "inversion a" i1 a i2]
      using reciprocal_id_iff
      apply (auto)
      apply (metis num.distinct(1) of_complex_one_iff one_eq_numeral_iff)
      using of_complex_inj
      by force      
    thus False
      using * cr2
      by simp
  qed

  ultimately
  show ?thesis
    using assms `b \<noteq> i1` `b \<noteq> i2` `b \<noteq> a`
    by auto
qed


lemma ex_is_poincare_line_points:
  assumes "is_poincare_line H"
  shows "\<exists> u v. u \<in> unit_disc \<and> v \<in> unit_disc \<and> u \<noteq> v \<and> {u, v} \<subseteq> circline_set H"
proof-
  obtain u v w where *: "u \<noteq> v" "v \<noteq> w" "u \<noteq> w" "{u, v, w} \<subseteq> circline_set H"
    using assms is_poincare_line_iff[of H]
    using circline_type_neg_card_gt3[of H]
    by auto

  have "\<not> {u, v, w} \<subseteq> unit_circle_set"
    using unique_circline_set[of u v w] *
    apply auto
    using assms not_is_poincare_line_unit_circle unit_circle_set_def
    by blast

  hence "H \<noteq> unit_circle"
    unfolding unit_circle_set_def
    using *
    by auto

  show ?thesis
  proof (cases "(u \<in> unit_disc \<and> v \<in> unit_disc) \<or>
                (u \<in> unit_disc \<and> w \<in> unit_disc) \<or>
                (v \<in> unit_disc \<and> w \<in> unit_disc)")
    case True
    thus ?thesis
      using *
      by auto
  next
    case False

    have "\<exists> a b. a \<noteq> b \<and> a \<noteq> inversion b \<and> a \<in> circline_set H \<and> b \<in> circline_set H \<and> a \<notin> unit_circle_set \<and> b \<notin> unit_circle_set"
    proof (cases "(u \<in> unit_circle_set \<and> v \<in> unit_circle_set) \<or>
                  (u \<in> unit_circle_set \<and> w \<in> unit_circle_set) \<or>
                  (v \<in> unit_circle_set \<and> w \<in> unit_circle_set)")
      case True
      then obtain i1 i2 a where *:
        "i1 \<in> unit_circle_set \<inter> circline_set H" "i2 \<in> unit_circle_set \<inter> circline_set H" 
        "a \<in> circline_set H" "a \<notin> unit_circle_set"
        "i1 \<noteq> i2" "i1 \<noteq> a" "i2 \<noteq> a"
        using * `\<not> {u, v, w} \<subseteq> unit_circle_set`
        by auto
      then obtain b where "b \<in> circline_set H" "b \<noteq> i1" "b \<noteq> i2" "b \<noteq> a" "b \<noteq> inversion a"
        using ex_is_poincare_line_points'[of i1 H i2 a]
        by blast

      hence "b \<notin> unit_circle_set"
        using * `H \<noteq> unit_circle` unique_circline_set[of i1 i2 b]
        unfolding unit_circle_set_def
        by auto
        
      thus ?thesis
        using * `b \<in> circline_set H` `b \<noteq> a` `b \<noteq> inversion a`
        by auto
    next
      case False  
      then obtain f g h where
        *: "f \<noteq> g" "f \<in> circline_set H" "f \<notin> unit_circle_set"  
                    "g \<in> circline_set H" "g \<notin> unit_circle_set"
                    "h \<in> circline_set H" "h \<noteq> f" "h \<noteq> g"
        using *
        by auto
      show ?thesis
      proof (cases "f = inversion g")   
        case False
        thus ?thesis
          using *
          by auto
      next
        case True
        show ?thesis
        proof (cases "h \<in> unit_circle_set")
          case False
          thus ?thesis
            using * `f = inversion g`
            by auto
        next
          case True
          obtain m where cr2: "cross_ratio m h f g = of_complex 2"
            using ex_cross_ratio[of h f g] * `f \<noteq> g` `h \<noteq> f` `h \<noteq> g`
            by auto
          hence "m \<noteq> h" "m \<noteq> f" "m \<noteq> g"
            using `h \<noteq> f` `h \<noteq> g` `f \<noteq> g`
            using ex1_cross_ratio[of h f g]
            using cross_ratio_0[of h f g] cross_ratio_1[of h f g] cross_ratio_inf[of h g f]
            using cr2
            by auto
          hence "m \<in> circline_set H" 
            using four_points_on_circline_iff_cross_ratio_real[of m h f g] cr2
            using `h \<noteq> f` `h \<noteq> g` `f \<noteq> g` *
            using unique_circline_set[of h f g]
            by auto

          show ?thesis
          proof (cases "m \<in> unit_circle_set")
            case False
            thus ?thesis
              using `m \<noteq> f` `m \<noteq> g` `f = inversion g` * `m \<in> circline_set H`
              by auto
          next
            case True
            then obtain n where "n \<noteq> h" "n \<noteq> m" "n \<noteq> f" "n \<noteq> inversion f" "n \<in> circline_set H"
              using ex_is_poincare_line_points'[of h H m f] * `m \<in> circline_set H` `h \<in> unit_circle_set` `m \<noteq> h`
              by auto
            hence "n \<notin> unit_circle_set"
              using * `H \<noteq> unit_circle` unique_circline_set[of m n h] 
              using `m \<noteq> h` `m \<in> unit_circle_set` `h \<in> unit_circle_set` `m \<in> circline_set H`
              unfolding unit_circle_set_def
              by auto
        
            thus ?thesis
              using * `n \<in> circline_set H` `n \<noteq> f` `n \<noteq> inversion f`
              by auto
          qed
        qed
      qed
    qed
    then obtain a b where ab: "a \<noteq> b" "a \<noteq> inversion b" "a \<in> circline_set H" "b \<in> circline_set H" "a \<notin> unit_circle_set" "b \<notin> unit_circle_set"
      by blast
    have "\<forall> x. x \<in> circline_set H \<and> x \<notin> unit_circle_set \<longrightarrow> (\<exists> x'. x' \<in> circline_set H \<inter> unit_disc \<and> (x' = x \<or> x' = inversion x))"
    proof safe
      fix x
      assume x: "x \<in> circline_set H" "x \<notin> unit_circle_set" 
      show "\<exists> x'. x' \<in> circline_set H \<inter> unit_disc \<and> (x' = x \<or> x' = inversion x)"
      proof (cases "x \<in> unit_disc")
        case True
        thus ?thesis
          using x
          by auto
      next
        case False
        hence "x \<in> unit_disc_compl"
          using x  in_on_out_univ[of "ounit_circle"]
          unfolding unit_circle_set_def unit_disc_def unit_disc_compl_def
          by auto
        hence "inversion x \<in> unit_disc"
          using inversion_unit_disc_compl
          by blast
        thus ?thesis
          using is_poincare_line_inverse_point[OF assms, of x] x
          by auto
      qed
    qed
    then obtain a' b' where 
      *: "a' \<in> circline_set H" "a' \<in> unit_disc" "b' \<in> circline_set H" "b' \<in> unit_disc" and
      **: "a' = a \<or> a' = inversion a" "b' = b \<or> b' = inversion b" 
      using ab
      by blast
    have "a' \<noteq> b'"
      using `a \<noteq> b` `a \<noteq> inversion b` ** *
      by auto (metis inversion_involution)
    thus ?thesis
      using *
      by auto
  qed
qed

(* For simplicity we show this only for two different points inside the disk, but that could be relaxed *)
lemma unique_is_poincare_line:
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  assumes pl: "is_poincare_line l1" "is_poincare_line l2"
  assumes on_l: "{u, v} \<subseteq> circline_set l1 \<inter> circline_set l2"
  shows "l1 = l2"
proof-
  have "u \<noteq> inversion u" "v \<noteq> inversion u"
    using in_disc
    using inversion_noteq_unit_disc[of u v]
    using inversion_noteq_unit_disc[of u u]
    by auto
  thus ?thesis
    using on_l
    using unique_circline_set[of u "inversion u" "v"] `u \<noteq> v`
    using is_poincare_line_inverse_point[of l1 u]
    using is_poincare_line_inverse_point[of l2 u]
    using pl
    by auto                                                                            
qed

lemma is_poincare_line_trough_zero_trough_infty [simp]:
  assumes "is_poincare_line l"
  assumes "0\<^sub>h \<in> circline_set l"
  shows "\<infinity>\<^sub>h \<in> circline_set l"
  using is_poincare_line_inverse_point[OF assms]
  by simp

lemma is_poincare_line_trough_zero_is_line:
  assumes "is_poincare_line l" "0\<^sub>h \<in> circline_set l"
  shows "is_line l"
  using assms
  using inf_in_circline_set is_poincare_line_trough_zero_trough_infty
  by blast

lemma is_poincare_line_not_trough_zero_not_trough_infty [simp]:
  assumes "is_poincare_line l"
  assumes "0\<^sub>h \<notin> circline_set l"
  shows "\<infinity>\<^sub>h \<notin> circline_set l"
  using assms
  using is_poincare_line_inverse_point[OF assms(1), of "\<infinity>\<^sub>h"]
  by auto

lemma is_poincare_line_not_trough_zero_is_circle:
  assumes "is_poincare_line l" "0\<^sub>h \<notin> circline_set l"
  shows "is_circle l"
  using assms
  using inf_in_circline_set is_poincare_line_not_trough_zero_not_trough_infty
  by auto

lemma is_poincare_line_0_real_is_x_axis:
  assumes "is_poincare_line l" "0\<^sub>h \<in> circline_set l"
    "x \<in> circline_set l \<inter> circline_set x_axis" "x \<noteq> 0\<^sub>h" "x \<noteq> \<infinity>\<^sub>h"
  shows "l = x_axis"
  using assms
  using is_poincare_line_trough_zero_trough_infty[OF assms(1-2)]
  using unique_circline_set[of x "0\<^sub>h" "\<infinity>\<^sub>h"]
  by auto

lemma is_poincare_line_0_imag_is_y_axis:
  assumes "is_poincare_line l" "0\<^sub>h \<in> circline_set l"
    "y \<in> circline_set l \<inter> circline_set y_axis" "y \<noteq> 0\<^sub>h" "y \<noteq> \<infinity>\<^sub>h"
  shows "l = y_axis"
  using assms
  using is_poincare_line_trough_zero_trough_infty[OF assms(1-2)]
  using unique_circline_set[of y "0\<^sub>h" "\<infinity>\<^sub>h"]
  by auto

(* moebius transformations preserve poincare lines *)
lemma unit_circle_fix_preserve_is_poincare_line [simp]:
  assumes "unit_circle_fix M" "is_poincare_line H"
  shows "is_poincare_line (moebius_circline M H)"
  using assms
  unfolding is_poincare_line_iff
proof (safe)
  let ?H' = "moebius_ocircline M (of_circline H)"
  let ?U' = "moebius_ocircline M ounit_circle"
  assume ++: "unit_circle_fix M" "perpendicular H unit_circle"
  have ounit: "ounit_circle = moebius_ocircline M ounit_circle \<or>
               ounit_circle = moebius_ocircline M (opposite_ocircline ounit_circle)"
    using ++(1) unit_circle_fix_iff[of M]
    by (simp add: inj_of_ocircline moebius_circline_ocircline)

  show "perpendicular (moebius_circline M H) unit_circle"
  proof (cases "pos_oriented ?H'")
    case True
    hence *: "of_circline (of_ocircline ?H') = ?H'"
      using of_circline_of_ocircline_pos_oriented
      by blast
    from ounit show ?thesis
    proof
      assume **: "ounit_circle = moebius_ocircline M ounit_circle"
      show ?thesis
        using ++ 
        unfolding perpendicular_def
        by (simp, subst moebius_circline_ocircline, subst *, subst **) simp
    next
      assume **: "ounit_circle = moebius_ocircline M (opposite_ocircline ounit_circle)"
      show ?thesis
        using ++
        unfolding perpendicular_def
        by (simp, subst moebius_circline_ocircline, subst *, subst **) simp
    qed
  next
    case False
    hence *: "of_circline (of_ocircline ?H') = opposite_ocircline ?H'"
      by (metis of_circline_of_ocircline pos_oriented_of_circline)
    from ounit show ?thesis
    proof
      assume **: "ounit_circle = moebius_ocircline M ounit_circle"
      show ?thesis
        using ++
        unfolding perpendicular_def
        by (simp, subst moebius_circline_ocircline, subst *, subst **) simp
    next
      assume **: "ounit_circle = moebius_ocircline M (opposite_ocircline ounit_circle)"
      show ?thesis
        using ++
        unfolding perpendicular_def
        by (simp, subst moebius_circline_ocircline, subst *, subst **) simp
    qed
  qed
qed simp

lemma unit_circle_fix_preserve_is_poincare_line_iff [simp]:
  assumes "unit_circle_fix M"
  shows "is_poincare_line (moebius_circline M H) \<longleftrightarrow> is_poincare_line H"
  using assms
  using unit_circle_fix_preserve_is_poincare_line[of M H]
  using unit_circle_fix_preserve_is_poincare_line[of "moebius_inv M" "moebius_circline M H"]
  by (auto simp del: unit_circle_fix_preserve_is_poincare_line)

lemma unit_disc_fix_preserve_poincare_colinear [simp]:
  assumes "unit_circle_fix M" "poincare_colinear A"
  shows "poincare_colinear (moebius_pt M ` A)"
  using assms
  unfolding poincare_colinear_def                                                    
  by (auto, rule_tac x="moebius_circline M p" in exI, auto)

lemma unit_disc_fix_preserve_poincare_colinear_iff [simp]:
  assumes "unit_circle_fix M"
  shows "poincare_colinear (moebius_pt M ` A) \<longleftrightarrow> poincare_colinear A"
  using assms
  using unit_disc_fix_preserve_poincare_colinear[of M A]
  using unit_disc_fix_preserve_poincare_colinear[of "moebius_inv M" "moebius_pt M ` A"]
  by (auto simp del: unit_disc_fix_preserve_poincare_colinear)

lemma unit_disc_fix_preserve_poincare_colinear3 [simp]:
  assumes "unit_disc_fix M"
  shows "poincare_colinear {moebius_pt M u, moebius_pt M v, moebius_pt M w} \<longleftrightarrow>
         poincare_colinear {u, v, w}"
  using assms unit_disc_fix_preserve_poincare_colinear_iff[of M "{u, v, w}"]
  by simp

(* conjugation preserves is_poincare_line *)
lemma is_poincare_line_conjugate_circline [simp]:
  assumes "is_poincare_line H"
  shows "is_poincare_line (conjugate_circline H)"
  using assms
  by (transfer, transfer, auto simp add: mat_cnj_def hermitean_def mat_adj_def)

lemma is_poincare_line_conjugate_circline_iff [simp]:
  shows "is_poincare_line (conjugate_circline H) \<longleftrightarrow> is_poincare_line H"
  using is_poincare_line_conjugate_circline[of "conjugate_circline H"]
  by auto
                                  
lemma conjugate_preserve_poincare_colinear [simp]:
  assumes "poincare_colinear A"
  shows "poincare_colinear (conjugate ` A)"
  using assms
  unfolding poincare_colinear_def
  by auto (rule_tac x="conjugate_circline p" in exI, auto)

lemma [simp]: "conjugate ` conjugate ` A = A"
  by (auto simp add: image_iff)

lemma conjugate_preserve_poincare_colinear_iff [simp]:
  shows "poincare_colinear (conjugate ` A) \<longleftrightarrow> poincare_colinear A"
  using conjugate_preserve_poincare_colinear[of "A"]
  using conjugate_preserve_poincare_colinear[of "conjugate ` A"]
  by (auto simp del: conjugate_preserve_poincare_colinear)

lemma ex_unit_disc_fix_is_poincare_line_to_x_axis:
  assumes "is_poincare_line l"
  shows  "\<exists> M. unit_disc_fix M \<and> moebius_circline M l = x_axis"
proof-
  from assms obtain u v where "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc" and "{u, v} \<subseteq> circline_set l"
    using ex_is_poincare_line_points
    by blast
  then obtain M where *: "unit_disc_fix M" "moebius_pt M u = 0\<^sub>h" "moebius_pt M v \<in> positive_x_axis"
    using ex_unit_disc_fix_to_zero_positive_x_axis[of u v]
    by auto
  moreover
  hence "{0\<^sub>h, moebius_pt M v} \<subseteq> circline_set x_axis"
    unfolding positive_x_axis_def
    by auto
  moreover
  have "moebius_pt M v \<noteq> 0\<^sub>h"
    using `u \<noteq> v` *
    by auto (metis moebius_pt_invert)
  moreover
  have "moebius_pt M v \<noteq> \<infinity>\<^sub>h"
    using `unit_disc_fix M` `v \<in> unit_disc`
    using unit_disc_fix_discI
    by fastforce
  ultimately
  show ?thesis
    using `is_poincare_line l` `{u, v} \<subseteq> circline_set l` `unit_disc_fix M`
    using is_poincare_line_0_real_is_x_axis[of "moebius_circline M l" "moebius_pt M v"]
    by (rule_tac x="M" in exI, force) 
qed

lemma wlog_line_x_axis:
  assumes is_line: "is_poincare_line H"
  assumes x_axis: "P x_axis"
  assumes preserves: "\<And> M. \<lbrakk>unit_disc_fix M; P (moebius_circline M H)\<rbrakk> \<Longrightarrow> P H"
  shows "P H"
  using assms
  using ex_unit_disc_fix_is_poincare_line_to_x_axis[of H]
  by auto


(* ------------------------------------------------------------------ *)
subsection{* Poincare lines between the two given points *}
(* ------------------------------------------------------------------ *)

definition mk_poincare_line_cmat :: "real \<Rightarrow> complex \<Rightarrow> complex_mat" where
  [simp]: "mk_poincare_line_cmat A B = (cor A, B, cnj B, cor A)"

lemma mk_poincare_line_cmat_zero_iff:
  "mk_poincare_line_cmat A B = mat_zero \<longleftrightarrow> A = 0 \<and> B = 0"
  by auto

lemma mk_poincare_line_cmat_hermitean
  [simp]:  "hermitean (mk_poincare_line_cmat A B)"
  by simp

lemma mk_poincare_line_cmat_scale:
  "cor k *\<^sub>s\<^sub>m mk_poincare_line_cmat A B = mk_poincare_line_cmat (k * A) (k * B)"
  by simp

(*
   If {z, w} = {0\<^sub>h, \<infinity>\<^sub>h} then there is no unique Poincare line.
   If z and w are mutually inverse, then the standard construction fails (both geometric and algebraic).
   If z and w are different points on the unit circle, then the standard construction fails (only geometric).
   None of this problematic cases occur when z and w are inside the Poincare disc.
*)
definition poincare_line_cvec_cmat :: "complex_vec \<Rightarrow> complex_vec \<Rightarrow> complex_mat" where
  [simp]: "poincare_line_cvec_cmat z w =
            (let (z1, z2) = z;
                 (w1, w2) = w;
                 nom = w1*cnj w2*(z1*cnj z1 + z2*cnj z2) - z1*cnj z2*(w1*cnj w1 + w2*cnj w2);
                 den = z1*cnj z2*cnj w1*w2 - w1*cnj w2*cnj z1*z2
              in if den \<noteq> 0 then
                    mk_poincare_line_cmat (Re(\<i>*den)) (\<i>*nom)
                 else if z1*cnj z2 \<noteq> 0 then
                    mk_poincare_line_cmat 0 (\<i>*z1*cnj z2)
                 else if w1*cnj w2 \<noteq> 0 then
                    mk_poincare_line_cmat 0 (\<i>*w1*cnj w2)
                 else
                    mk_poincare_line_cmat 0 \<i>)"

lemma poincare_line_cvec_cmat_AeqD:
  assumes [simp]: "poincare_line_cvec_cmat z w = (A, B, C, D)"
  shows "A = D"
  using assms
  by (cases z, cases w) (auto split: if_split_asm)

lemma poincare_line_cvec_cmat_hermitean
  [simp]: "hermitean (poincare_line_cvec_cmat z w)"
  by (cases z, cases w) (auto split: if_split_asm simp del: mk_poincare_line_cmat_def)

lemma poincare_line_cvec_cmat_nonzero [simp]:
  assumes "z \<noteq> vec_zero" "w \<noteq> vec_zero"
  shows  "poincare_line_cvec_cmat z w \<noteq> mat_zero"
proof-
  obtain z1 z2 w1 w2 where *: "z = (z1, z2)" "w = (w1, w2)"
    by (cases z, cases w, auto)

  let ?den = "z1*cnj z2*cnj w1*w2 - w1*cnj w2*cnj z1*z2"
  show ?thesis
  proof (cases "?den \<noteq> 0")
    case True
    have "is_real (\<i> * ?den)"
      using eq_cnj_iff_real[of "\<i> *?den"]
      by (simp add: field_simps)
    hence "Re (\<i> * ?den) \<noteq> 0"
      using `?den \<noteq> 0`
      by (metis complex_i_not_zero complex_surj mult_eq_0_iff zero_complex.code)
    thus ?thesis
      using * `?den \<noteq> 0`
      by (simp del: mk_poincare_line_cmat_def mat_zero_def add: mk_poincare_line_cmat_zero_iff)
  next
    case False
    thus ?thesis
      using *
      by (simp del: mk_poincare_line_cmat_def mat_zero_def add: mk_poincare_line_cmat_zero_iff)
  qed
qed

lift_definition poincare_line_hcoords_clmat :: "complex_homo_coords \<Rightarrow> complex_homo_coords \<Rightarrow> circline_mat" is poincare_line_cvec_cmat
  using poincare_line_cvec_cmat_hermitean poincare_line_cvec_cmat_nonzero
  by simp

lift_definition poincare_line :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> circline" is poincare_line_hcoords_clmat
proof transfer
  fix za zb wa wb
  assume "za \<noteq> vec_zero" "zb \<noteq> vec_zero" "wa \<noteq> vec_zero" "wb \<noteq> vec_zero"
  assume "za \<approx>\<^sub>v zb" "wa \<approx>\<^sub>v wb"
  obtain za1 za2 zb1 zb2 wa1 wa2 wb1 wb2 where
  *: "(za1, za2) = za" "(zb1, zb2) = zb"
     "(wa1, wa2) = wa" "(wb1, wb2) = wb"
    by (cases za, cases zb, cases wa, cases wb, auto)
  obtain kz kw where
    **: "kz \<noteq> 0" "kw \<noteq> 0" "zb1 = kz * za1" "zb2 = kz * za2" "wb1 = kw * wa1" "wb2 = kw * wa2"
    using `za \<approx>\<^sub>v zb` `wa \<approx>\<^sub>v wb` *[symmetric]
    by auto

  let ?nom = "\<lambda> z1 z2 w1 w2. w1*cnj w2*(z1*cnj z1 + z2*cnj z2) - z1*cnj z2*(w1*cnj w1 + w2*cnj w2)"
  let ?den = "\<lambda> z1 z2 w1 w2. z1*cnj z2*cnj w1*w2 - w1*cnj w2*cnj z1*z2"

  show "circline_eq_cmat (poincare_line_cvec_cmat za wa)
                         (poincare_line_cvec_cmat zb wb)"
  proof-
    have "\<exists>k. k \<noteq> 0 \<and>
            poincare_line_cvec_cmat (zb1, zb2) (wb1, wb2) = cor k *\<^sub>s\<^sub>m poincare_line_cvec_cmat (za1, za2) (wa1, wa2)"
    proof (cases "?den za1 za2 wa1 wa2 \<noteq> 0")
      case True
      hence "?den zb1 zb2 wb1 wb2 \<noteq> 0"
        using **
        by (simp add: field_simps)

      let ?k = "kz * cnj kz * kw * cnj kw"

      have "?k \<noteq> 0"
        using **
        by simp

      have "is_real ?k"
        using eq_cnj_iff_real[of ?k]
        by auto

      have "cor (Re ?k) = ?k"
        using `is_real ?k`
        using complex_of_real_Re
        by blast

      have "Re ?k \<noteq> 0"
        using `?k \<noteq> 0` `cor (Re ?k) = ?k`
        by (metis of_real_0)

      have arg1: "Re (\<i> * ?den zb1 zb2 wb1 wb2) = Re ?k * Re (\<i> * ?den za1 za2 wa1 wa2)"
        apply (subst **)+
        apply (subst Re_mult_real[symmetric, OF `is_real ?k`])
        apply (rule arg_cong[where f=Re])
        apply (simp add: field_simps)
        done
      have arg2: "\<i> * ?nom zb1 zb2 wb1 wb2 = ?k * \<i> * ?nom za1 za2 wa1 wa2"
        using **
        by (simp add: field_simps)
      have "mk_poincare_line_cmat (Re (\<i>*?den zb1 zb2 wb1 wb2)) (\<i>*?nom zb1 zb2 wb1 wb2) =
            cor (Re ?k) *\<^sub>s\<^sub>m mk_poincare_line_cmat (Re (\<i>*?den za1 za2 wa1 wa2)) (\<i>*?nom za1 za2 wa1 wa2)"
        using `cor (Re ?k) = ?k` `is_real ?k`
        apply (subst mk_poincare_line_cmat_scale)
        apply (subst arg1, subst arg2)
        apply (subst `cor (Re ?k) = ?k`)+
        apply simp
        done
       thus ?thesis
        using `?den za1 za2 wa1 wa2 \<noteq> 0` `?den zb1 zb2 wb1 wb2 \<noteq> 0`
        using `Re ?k \<noteq> 0` `cor (Re ?k) = ?k`
        by (rule_tac x="Re ?k" in exI, simp)
    next
      case False
      hence "?den zb1 zb2 wb1 wb2 = 0"
        using **
        by (simp add: field_simps)
      show ?thesis
      proof (cases "za1*cnj za2 \<noteq> 0")
        case True
        hence "zb1*cnj zb2 \<noteq> 0"
          using **
          by (simp add: field_simps)

        let ?k = "kz * cnj kz"

        have "?k \<noteq> 0" "is_real ?k"
          using **
          using eq_cnj_iff_real[of ?k]
          by auto
        thus ?thesis
          using `za1 * cnj za2 \<noteq> 0` `zb1 * cnj zb2 \<noteq> 0`
          using `\<not> (?den za1 za2 wa1 wa2 \<noteq> 0)` `?den zb1 zb2 wb1 wb2 = 0` **
          by (rule_tac x="Re (kz * cnj kz)" in exI, auto simp add: complex_of_real_Re complex.expand)
      next
        case False
        hence "zb1 * cnj zb2 = 0"
          using **
          by (simp add: field_simps)
        show ?thesis
        proof (cases "wa1 * cnj wa2 \<noteq> 0")
          case True
          hence "wb1*cnj wb2 \<noteq> 0"
            using **
            by (simp add: field_simps)

          let ?k = "kw * cnj kw"

          have "?k \<noteq> 0" "is_real ?k"
            using **
            using eq_cnj_iff_real[of ?k]
            by auto

          thus ?thesis
            using `\<not> (za1 * cnj za2 \<noteq> 0)` 
            using `wa1 * cnj wa2 \<noteq> 0` `wb1 * cnj wb2 \<noteq> 0`
            using `\<not> (?den za1 za2 wa1 wa2 \<noteq> 0)` `?den zb1 zb2 wb1 wb2 = 0` **
            by (rule_tac x="Re (kw * cnj kw)" in exI) 
               (auto simp add: complex_of_real_Re complex.expand)
        next
          case False
          hence "wb1 * cnj wb2 = 0"
            using **
            by (simp add: field_simps)
          thus ?thesis
            using `\<not> (za1 * cnj za2 \<noteq> 0)` `zb1 * cnj zb2 = 0`
            using `\<not> (wa1 * cnj wa2 \<noteq> 0)` `wb1 * cnj wb2 = 0`
            using `\<not> (?den za1 za2 wa1 wa2 \<noteq> 0)` `?den zb1 zb2 wb1 wb2 = 0` **
            by simp
        qed
      qed
    qed
    thus ?thesis
      using *[symmetric]
      by simp
  qed
qed

lemma poincare_line [simp]:
  assumes "z \<noteq> w"
  shows "on_circline (poincare_line z w) z"
        "on_circline (poincare_line z w) w"
proof-
  have "on_circline (poincare_line z w) z \<and> on_circline (poincare_line z w) w"
    using assms
  proof (transfer, transfer)
    fix z w
    assume vz: "z \<noteq> vec_zero" "w \<noteq> vec_zero"
    obtain z1 z2 w1 w2 where
    zw: "(z1, z2) = z" "(w1, w2) = w"
      by (cases z, cases w, auto)

    let ?den = "z1*cnj z2*cnj w1*w2 - w1*cnj w2*cnj z1*z2"
    have *: "cor (Re (\<i> * ?den)) = \<i> * ?den"
    proof-
      have "cnj ?den = -?den"
        by auto
      hence "is_imag ?den"
        using eq_minus_cnj_iff_imag[of ?den]
        by simp
      thus ?thesis
        using complex_of_real_Re[of "\<i> * ?den"]
        by simp
    qed
    show "on_circline_cmat_cvec (poincare_line_cvec_cmat z w) z \<and>
          on_circline_cmat_cvec (poincare_line_cvec_cmat z w) w"
      unfolding poincare_line_cvec_cmat_def mk_poincare_line_cmat_def
      apply (subst zw[symmetric])+
      unfolding Let_def prod.case
      apply (subst *)+
      by (auto simp add: vec_cnj_def field_simps)
  qed
  thus "on_circline (poincare_line z w) z" "on_circline (poincare_line z w) w"
    by auto
qed

lemma poincare_line_circline_set [simp]:
  assumes "z \<noteq> w"
  shows "z \<in> circline_set (poincare_line z w)"
        "w \<in> circline_set (poincare_line z w)"
  using assms
  by (auto simp add: circline_set_def)

lemma poincare_line_type:
  assumes "z \<noteq> w"
  shows "circline_type (poincare_line z w) = -1"
proof-
  have "\<exists> a b. a \<noteq> b \<and> {a, b} \<subseteq> circline_set (poincare_line z w)"
    using poincare_line[of z w] assms
    unfolding circline_set_def
    by (rule_tac x=z in exI, rule_tac x=w in exI, simp)
  thus ?thesis
    using circline_type[of "poincare_line z w"]
    using circline_type_pos_card_eq0[of "poincare_line z w"]
    using circline_type_zero_card_eq1[of "poincare_line z w"]
    by auto
qed

lemma is_poincare_line_poincare_line [simp]:
  assumes "z \<noteq> w"
  shows "is_poincare_line (poincare_line z w)"
  using poincare_line_type[of z w, OF assms]
proof (transfer, transfer)
  fix z w
  assume vz: "z \<noteq> vec_zero" "w \<noteq> vec_zero"
  obtain A B C D where *: "poincare_line_cvec_cmat z w = (A, B, C, D)"
    by (cases "poincare_line_cvec_cmat z w") auto
  assume "circline_type_cmat (poincare_line_cvec_cmat z w) = - 1"
  thus "is_poincare_line_cmat (poincare_line_cvec_cmat z w)"
    using vz *
    using poincare_line_cvec_cmat_hermitean[of z w]
    using poincare_line_cvec_cmat_nonzero[of z w]
    using poincare_line_cvec_cmat_AeqD[of z w A B C D]
    using hermitean_elems[of A B C D]
    using cmod_power2[of D] cmod_power2[of C]
    unfolding is_poincare_line_cmat_def
    by (simp del: poincare_line_cvec_cmat_def add: sgn_1_neg power2_eq_square)
qed

lemma ex_poincare_line_one_point:
  shows "\<exists> l. is_poincare_line l \<and> z \<in> circline_set l"
proof (cases "z = 0\<^sub>h")
  case True
  thus ?thesis
    by (rule_tac x="x_axis" in exI) simp
next
  case False
  thus ?thesis
    by (rule_tac x="poincare_line 0\<^sub>h z" in exI) auto
qed

lemma poincare_colinear_singleton [simp]:
  assumes "u \<in> unit_disc"
  shows "poincare_colinear {u}"
  using assms
  using ex_poincare_line_one_point[of u]
  by (auto simp add: poincare_colinear_def)

lemma ex_poincare_line_two_points:
  assumes "z \<noteq> w"
  shows "\<exists> l. is_poincare_line l \<and> z \<in> circline_set l \<and> w \<in> circline_set l"
  using assms
  by (rule_tac x="poincare_line z w" in exI, simp)

lemma poincare_colinear_doubleton [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_colinear {u, v}"
  using assms
  using ex_poincare_line_one_point[of u]
  using ex_poincare_line_two_points[of u v]
  by (cases "u = v") (simp_all add: poincare_colinear_def)

lemma poincare_line_inversion:
  assumes "z \<noteq> w"
  shows "on_circline (poincare_line z w) (inversion z)"
        "on_circline (poincare_line z w) (inversion w)"
  using assms
  using is_poincare_line_poincare_line[OF `z \<noteq> w`]
  using is_poincare_line_inverse_point
  unfolding circline_set_def
  by auto

(* For simplicity we show this only for two different points inside the disk, but that could be relaxed *)
lemma unique_poincare_line:
  assumes in_disc: "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc"
  assumes on_l: "u \<in> circline_set l" "v \<in> circline_set l" "is_poincare_line l"
  shows "l = poincare_line u v"
  using assms
  using unique_is_poincare_line[of u v l "poincare_line u v"]
  unfolding circline_set_def
  by auto

lemma ex1_poincare_line:
  assumes "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc"
  shows "\<exists>! l. is_poincare_line l \<and> u \<in> circline_set l \<and> v \<in> circline_set l"
proof (rule ex1I)
  let ?l = "poincare_line u v"
  show "is_poincare_line ?l \<and> u \<in> circline_set ?l \<and> v \<in> circline_set ?l"
    using assms
    unfolding circline_set_def
    by auto
next
  fix l
  assume "is_poincare_line l \<and> u \<in> circline_set l \<and> v \<in> circline_set l"
  thus "l = poincare_line u v"
    using unique_poincare_line assms
    by auto
qed

lemma poincare_line_sym:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  shows "poincare_line u v = poincare_line v u"
  using assms
  using unique_poincare_line[of u v "poincare_line v u"]
  by simp

lemma ex_poincare_line_points:
  assumes "is_poincare_line H"
  shows "\<exists> u v. u \<in> unit_disc \<and> v \<in> unit_disc \<and> u \<noteq> v \<and> H = poincare_line u v"
  using assms
  using ex_is_poincare_line_points
  using unique_poincare_line[where l=H]
  by fastforce

lemma poincare_line_0_real_is_x_axis:
  assumes "x \<in> circline_set x_axis" "x \<noteq> 0\<^sub>h" "x \<noteq> \<infinity>\<^sub>h"
  shows "poincare_line 0\<^sub>h x = x_axis"
  using assms
  using is_poincare_line_0_real_is_x_axis[of "poincare_line 0\<^sub>h x" x]
  by auto

lemma poincare_line_0_imag_is_y_axis:
  assumes "y \<in> circline_set y_axis" "y \<noteq> 0\<^sub>h" "y \<noteq> \<infinity>\<^sub>h"
  shows "poincare_line 0\<^sub>h y = y_axis"
  using assms
  using is_poincare_line_0_imag_is_y_axis[of "poincare_line 0\<^sub>h y" y]
  by auto

lemma poincare_line_x_axis:
  assumes "x \<in> unit_disc" "y \<in> unit_disc" "x \<in> circline_set x_axis" "y \<in> circline_set x_axis" "x \<noteq> y"
  shows "poincare_line x y = x_axis"
  using assms
  using unique_poincare_line
  by auto

(* This could be stated in terms of is_poincare_line, and unit disc assumptions could be relaxed. *)
(* Other direction could also be proved. *)
lemma poincare_line_inversion_full':
  assumes "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc" "x \<in> unit_disc"
  assumes "on_circline (poincare_line u v) x"
  shows "on_circline (poincare_line u v) (inversion x)"
proof (cases "u = x \<or> u = v")
  case True
  thus ?thesis
    using assms
    using poincare_line_inversion[of u v]
    by auto
next
  case False
  hence "poincare_line u v = poincare_line u x"
    using unique_poincare_line[of u x "poincare_line u v"]
    using assms
    unfolding circline_set_def
    by simp
  thus ?thesis
    using poincare_line_inversion[of u x] False
    by simp
qed

(* moebius preserve poincare_line construction *)
lemma unit_disc_fix_preserve_poincare_line [simp]:
  assumes "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  shows "poincare_line (moebius_pt M u) (moebius_pt M v) = moebius_circline M (poincare_line u v)"
proof (rule unique_poincare_line[symmetric])
  show "moebius_pt M u \<noteq> moebius_pt M v"
    using `u \<noteq> v` 
    by auto
next
  show "moebius_pt M u \<in> circline_set (moebius_circline M (poincare_line u v))"
       "moebius_pt M v \<in> circline_set (moebius_circline M (poincare_line u v))"
    unfolding circline_set_def
    using moebius_circline[of M "poincare_line u v"] `u \<noteq> v`
    by auto
next
  from assms(1) have "unit_circle_fix M"
    by simp
  thus "is_poincare_line (moebius_circline M (poincare_line u v))"
    using unit_circle_fix_preserve_is_poincare_line assms
    by auto
next
  show "moebius_pt M u \<in> unit_disc" "moebius_pt M v \<in> unit_disc"
    using assms(2-3) unit_disc_fix_iff[OF assms(1)]
    by auto
qed


(* conjugate preserve poincare_line construction *)
lemma conjugate_preserve_poincare_line [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  shows "poincare_line (conjugate u) (conjugate v) = conjugate_circline (poincare_line u v)"
proof-
  have "conjugate u \<noteq> conjugate v"
    using `u \<noteq> v`
    by (auto simp add: conjugate_inj)
  moreover
  have "conjugate u \<in> unit_disc" "conjugate v \<in> unit_disc"
    using assms
    by auto
  moreover
  have "conjugate u \<in> circline_set (conjugate_circline (poincare_line u v))"
       "conjugate v \<in> circline_set (conjugate_circline (poincare_line u v))"
    using `u \<noteq> v`
    by simp_all
  moreover
  have "is_poincare_line (conjugate_circline (poincare_line u v))"
    using is_poincare_line_poincare_line[OF `u \<noteq> v`]
    by simp
  ultimately
  show ?thesis
    using unique_poincare_line[of "conjugate u" "conjugate v" "conjugate_circline (poincare_line u v)"]
    by simp
qed


(* TODO: simplify in terms of 0\<^sub>h \<in> circline_set (poincare_line (of_complex y') (of_complex z')) *)
lemma poincare_colinear_zero_iff:
  assumes "of_complex y' \<in> unit_disc" "of_complex z' \<in> unit_disc" "y' \<noteq> z'" "y' \<noteq> 0" "z' \<noteq> 0"
  shows "poincare_colinear {0\<^sub>h, of_complex y', of_complex z'} \<longleftrightarrow> y'*cnj z' = cnj y'*z'" (is "?lhs \<longleftrightarrow> ?rhs")
proof-
  have "of_complex y' \<noteq> of_complex z'"
    using assms
    using of_complex_inj
    by blast
  show ?thesis
  proof
    assume ?lhs
    hence "0\<^sub>h \<in> circline_set (poincare_line (of_complex y') (of_complex z'))"
      using unique_poincare_line[of "of_complex y'" "of_complex z'"]
      using assms `of_complex y' \<noteq> of_complex z'`
      unfolding poincare_colinear_def
      by auto
    moreover
    let ?mix = "y' * cnj z' - cnj y' * z'"
    have "is_real (\<i> * ?mix)"
      using eq_cnj_iff_real[of ?mix]
      by auto
    hence "y' * cnj z' = cnj y' * z' \<longleftrightarrow> Re (\<i> * ?mix) = 0"
      using complex.expand[of "\<i> * ?mix" 0]
      by (metis complex_i_not_zero eq_iff_diff_eq_0 mult_eq_0_iff zero_complex.simps(1) zero_complex.simps(2))
    ultimately
    show ?rhs
      using `y' \<noteq> z'` `y' \<noteq> 0` `z' \<noteq> 0`
      unfolding circline_set_def
      by simp (transfer, transfer, auto simp add: vec_cnj_def split: if_split_asm, metis Re_complex_of_real Re_mult_real is_real_complex_of_real)
  next
    assume ?rhs
    thus ?lhs
      using assms `of_complex y' \<noteq> of_complex z'`
      unfolding poincare_colinear_def
      unfolding circline_set_def
      apply (rule_tac x="poincare_line (of_complex y') (of_complex z')" in exI)
      apply auto
      apply (transfer, transfer, simp add: vec_cnj_def)
      done
  qed
qed

lemma poincare_colinear_zero_polar_form:
  assumes "poincare_colinear {0\<^sub>h, of_complex x, of_complex y}" "x \<noteq> 0" "y \<noteq> 0" "of_complex x \<in> unit_disc" "of_complex y \<in> unit_disc"
  shows "\<exists> \<phi> rx ry. x = cor rx * cis \<phi> \<and> y = cor ry * cis \<phi> \<and> rx \<noteq> 0 \<and> ry \<noteq> 0"
proof-
  from `x \<noteq> 0` `y \<noteq> 0` obtain \<phi> \<phi>' rx ry where
    polar: "x = cor rx * cis \<phi>" "y = cor ry * cis \<phi>'" and  "\<phi> = arg x" "\<phi>' = arg y"
    by (metis cmod_cis)
  hence "rx \<noteq> 0" "ry \<noteq> 0"
    using `x \<noteq> 0` `y \<noteq> 0`
    by auto
  have "of_complex y \<in> circline_set (poincare_line 0\<^sub>h (of_complex x))"
    using assms
    using unique_poincare_line[of "0\<^sub>h" "of_complex x"]
    unfolding poincare_colinear_def
    unfolding circline_set_def
    using of_complex_zero_iff
    by fastforce
  hence "cnj x * y = x * cnj y"
    using `x \<noteq> 0` `y \<noteq> 0`
    unfolding circline_set_def
    by simp (transfer, transfer, simp add: vec_cnj_def field_simps)
  hence "cis(\<phi>' - \<phi>) =  cis(\<phi> - \<phi>')"
    using polar `rx \<noteq> 0` `ry \<noteq> 0`
    by (simp add: cis_mult)
  hence "sin (\<phi>' - \<phi>) = 0"
    using cis_diff[of "\<phi>' - \<phi>"]
    by simp
  then obtain k :: int where "\<phi>' - \<phi> = k * pi"
    using sin_zero_iff_int2[of "\<phi>' - \<phi>"]
    by auto
  hence *: "\<phi>' = \<phi> + k * pi"
    by simp
  show ?thesis
  proof (cases "even k")
    case True
    then obtain k' where "k = 2*k'"
      using evenE by blast
    hence "cis \<phi> = cis \<phi>'"
      using * cos_periodic_int sin_periodic_int
      by (simp add: cis.ctr field_simps)
    thus ?thesis
      using polar `rx \<noteq> 0` `ry \<noteq> 0`
      by (rule_tac x=\<phi> in exI, rule_tac x=rx in exI, rule_tac x=ry in exI) simp
  next
    case False
    then obtain k' where "k = 2*k' + 1"
      using oddE by blast
    hence "cis \<phi> = - cis \<phi>'"
      using * cos_periodic_int sin_periodic_int
      by (simp add: cis.ctr complex_minus field_simps)
    thus ?thesis
      using polar `rx \<noteq> 0` `ry \<noteq> 0`
      by (rule_tac x=\<phi> in exI, rule_tac x=rx in exI, rule_tac x="-ry" in exI) simp
  qed
qed

(* ------------------------------------------------------------------ *)
subsection{* Ideal points *}
(* ------------------------------------------------------------------ *)

(* TODO: Introduce ideal points for the oriented circline - it would be a list, not a set of two points *)

(*
   Calculation of the two ideal points for a Poincare line:
     calc_ideal_points: two points obtained by algebraic formulas
     ideal_points: THE two points that are the intersections
     we prove that these are the same i.e., that the two ideal points
     are unique and must match with the calcluated ones
*)

definition calc_ideal_point1_cvec :: "complex \<Rightarrow> complex \<Rightarrow> complex_vec" where
 [simp]:  "calc_ideal_point1_cvec A B =
    (let discr = Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2) in
         (B*(-A - \<i>*sqrt(discr)), (cmod B)\<^sup>2))"

definition calc_ideal_point2_cvec :: "complex \<Rightarrow> complex \<Rightarrow> complex_vec" where
  [simp]: "calc_ideal_point2_cvec A B =
    (let discr = Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2) in
         (B*(-A + \<i>*sqrt(discr)), (cmod B)\<^sup>2))"

definition calc_ideal_points_cmat_cvec :: "complex_mat \<Rightarrow> complex_vec set" where
 [simp]:  "calc_ideal_points_cmat_cvec H =
    (if is_poincare_line_cmat H then
        let (A, B, C, D) = H
         in {calc_ideal_point1_cvec A B, calc_ideal_point2_cvec A B}
     else
        {(-1, 1), (1, 1)})"

lift_definition calc_ideal_points_clmat_hcoords :: "circline_mat \<Rightarrow> complex_homo_coords set" is calc_ideal_points_cmat_cvec
  by (auto simp add: Let_def split: if_split_asm)

lift_definition calc_ideal_points :: "circline \<Rightarrow> complex_homo set" is calc_ideal_points_clmat_hcoords
proof transfer
  fix H1 H2
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where k: "k \<noteq> 0" "H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  thus "rel_set (\<approx>\<^sub>v) (calc_ideal_points_cmat_cvec H1) (calc_ideal_points_cmat_cvec H2)"
  proof (cases "is_poincare_line_cmat H1")
    case True
    hence "is_poincare_line_cmat H2"
      using k * hermitean_mult_real[of H1 k] hh
      by (auto simp add: power2_eq_square)
    have **: "sqrt (\<bar>k\<bar> * cmod B1 * (\<bar>k\<bar> * cmod B1) - k * Re D1 * (k * Re D1)) =
         \<bar>k\<bar> * sqrt(cmod B1 * cmod B1 - Re D1 * Re D1)"
    proof-
      have "\<bar>k\<bar> * cmod B1 * (\<bar>k\<bar> * cmod B1) - k * Re D1 * (k * Re D1) =
            k\<^sup>2 * (cmod B1 * cmod B1 - Re D1 * Re D1)"
        by (simp add: power2_eq_square field_simps)
      thus ?thesis
        by (simp add: real_sqrt_mult)
    qed
    show ?thesis
      using `is_poincare_line_cmat H1` `is_poincare_line_cmat H2`
      using * k
      apply (simp add: Let_def)
      apply safe
      apply (simp add: power2_eq_square rel_set_def)
      apply safe
         apply (cases "k > 0")
          apply (rule_tac x="(cor k)\<^sup>2" in exI)
          apply (subst **)
          apply (simp add: power2_eq_square field_simps)
         apply (erule notE, rule_tac x="(cor k)\<^sup>2" in exI)
         apply (subst **)
         apply (simp add: power2_eq_square field_simps)
        apply (cases "k > 0")
        apply (erule notE, rule_tac x="(cor k)\<^sup>2" in exI)
        apply (subst **)
        apply (simp add: power2_eq_square field_simps)
        apply (rule_tac x="(cor k)\<^sup>2" in exI)
        apply (subst **)
        apply (simp add: power2_eq_square field_simps)
       apply (cases "k > 0")
        apply (rule_tac x="(cor k)\<^sup>2" in exI)
        apply (subst **)
        apply (simp add: power2_eq_square field_simps)
       apply (erule notE, rule_tac x="(cor k)\<^sup>2" in exI)
       apply (subst **)
       apply (simp add: power2_eq_square field_simps)
      apply (cases "k > 0")
       apply (erule notE, rule_tac x="(cor k)\<^sup>2" in exI)
       apply (subst **)
       apply (simp add: power2_eq_square field_simps)
      apply (rule_tac x="(cor k)\<^sup>2" in exI)
      apply (subst **)
      apply (simp add: power2_eq_square field_simps)
      done
  next
    case False
    hence "\<not> is_poincare_line_cmat H2"
      using k * hermitean_mult_real[of H1 k] hh
      by (auto simp add: power2_eq_square)
    have "rel_set (\<approx>\<^sub>v) {(- 1, 1), (1, 1)} {(- 1, 1), (1, 1)}"
      by (simp add: rel_set_def)
    thus ?thesis
      using `\<not> is_poincare_line_cmat H1` `\<not> is_poincare_line_cmat H2`
      using *
      apply (simp add: Let_def)
      apply safe
      done
  qed
qed

lemma calc_ideal_point_1_unit:
  assumes "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  assumes "(z1, z2) = calc_ideal_point1_cvec A B"
  shows "z1 * cnj z1 = z2 * cnj z2"
proof-
  let ?discr = "Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2)"
  have "?discr > 0"
    using assms
    by simp (smt cmod_power2 power2_eq_square power_zero_numeral)
  have "(B*(-A - \<i>*sqrt(?discr))) * cnj (B*(-A - \<i>*sqrt(?discr))) = (B * cnj B) * (A\<^sup>2 + cor (abs ?discr))"
    using `is_real A` eq_cnj_iff_real[of A]
    by (simp add: field_simps power2_eq_square)
  also have "... = (B * cnj B) * (cmod B)\<^sup>2"
    using `?discr > 0`
    using assms
    using complex_of_real_Re[of "(cmod B)\<^sup>2 - (Re A)\<^sup>2"] complex_of_real_Re[of A] `is_real A`
    by (simp add: power2_eq_square)
  also have "... = (cmod B)\<^sup>2 * cnj ((cmod B)\<^sup>2)"
    by simp (metis complex_norm_square of_real_power power2_eq_square)
  finally show ?thesis
    using assms
    by simp
qed

lemma calc_ideal_point_2_unit:
  assumes "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  assumes "(z1, z2) = calc_ideal_point2_cvec A B"
  shows "z1 * cnj z1 = z2 * cnj z2"
proof-
  let ?discr = "Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2)"
  have "?discr > 0"
    using assms
    by simp (smt cmod_power2 power2_eq_square power_zero_numeral)
  have "(B*(-A + \<i>*sqrt(?discr))) * cnj (B*(-A + \<i>*sqrt(?discr))) = (B * cnj B) * (A\<^sup>2 + cor (abs ?discr))"
    using `is_real A` eq_cnj_iff_real[of A]
    by (simp add: field_simps power2_eq_square)
  also have "... = (B * cnj B) * (cmod B)\<^sup>2"
    using `?discr > 0`
    using assms
    using complex_of_real_Re[of "(cmod B)\<^sup>2 - (Re A)\<^sup>2"] complex_of_real_Re[of A] `is_real A`
    by (simp add: power2_eq_square)
  also have "... = (cmod B)\<^sup>2 * cnj ((cmod B)\<^sup>2)"
    by simp (metis complex_norm_square of_real_power power2_eq_square)
  finally show ?thesis
    using assms
    by simp
qed

lemma calc_ideal_point1_sq:
  assumes "(z1, z2) = calc_ideal_point1_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "z1 * cnj z1 + z2 * cnj z2 = 2 * (B * cnj B)\<^sup>2"
proof-
  let ?discr = "Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2)"
  have "?discr > 0"
    using assms
    by simp (smt cmod_power2 power2_eq_square power_zero_numeral)
  have "z1 * cnj z1 = (B * cnj B) * (-A + \<i>*sqrt(?discr))*(-A - \<i>*sqrt(?discr))"
    using assms eq_cnj_iff_real[of A]
    by (simp)
  also have "... = (B * cnj B) * (A\<^sup>2 + ?discr)"
    using complex_of_real_Re[of A] `is_real A` `?discr > 0`
    by (simp add: power2_eq_square field_simps)
  finally
  have "z1 * cnj z1 = (B * cnj B)\<^sup>2"
    using complex_of_real_Re[of "(cmod B)\<^sup>2 - (Re A)\<^sup>2"] complex_of_real_Re[of A] `is_real A`
    using complex_mult_cnj_cmod[of B]
    by (simp add: power2_eq_square)
  moreover
  have "z2 * cnj z2 = (B * cnj B)\<^sup>2"
    using assms
    by simp
  ultimately
  show ?thesis
    by simp
qed

lemma calc_ideal_point2_sq:
  assumes "(z1, z2) = calc_ideal_point2_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "z1 * cnj z1 + z2 * cnj z2 = 2 * (B * cnj B)\<^sup>2"
proof-
  let ?discr = "Re ((cmod B)\<^sup>2 - (Re A)\<^sup>2)"
  have "?discr > 0"
    using assms
    by simp (smt cmod_power2 power2_eq_square power_zero_numeral)
  have "z1 * cnj z1 = (B * cnj B) * (-A + \<i>*sqrt(?discr))*(-A - \<i>*sqrt(?discr))"
    using assms eq_cnj_iff_real[of A]
    by (simp)
  also have "... = (B * cnj B) * (A\<^sup>2 + ?discr)"
    using complex_of_real_Re[of A] `is_real A` `?discr > 0`
    by (simp add: power2_eq_square field_simps)
  finally
  have "z1 * cnj z1 = (B * cnj B)\<^sup>2"
    using complex_of_real_Re[of "(cmod B)\<^sup>2 - (Re A)\<^sup>2"] complex_of_real_Re[of A] `is_real A`
    using complex_mult_cnj_cmod[of B]
    by (simp add: power2_eq_square)
  moreover
  have "z2 * cnj z2 = (B * cnj B)\<^sup>2"
    using assms
    by simp
  ultimately
  show ?thesis
    by simp
qed

lemma calc_ideal_point1_mix:
  assumes "(z1, z2) = calc_ideal_point1_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "B * cnj z1 * z2 + cnj B * z1 * cnj z2 = - 2 * A * (B * cnj B)\<^sup>2 "
proof-
  have "B*cnj z1 + cnj B*z1 = -2*A*B*cnj B"
    using assms eq_cnj_iff_real[of A]
    by (simp, simp add: field_simps)
  moreover
  have "cnj z2 = z2"
    using assms
    by simp
  hence "B*cnj z1*z2 + cnj B*z1*cnj z2 = (B*cnj z1 + cnj B*z1)*z2"
    by (simp add: field_simps)
  ultimately
  have "B*cnj z1*z2 + cnj B*z1*cnj z2 = -2*A*(B* cnj B)*z2"
    by simp
  also have "\<dots> = -2*A*(B * cnj B)\<^sup>2"
    using assms
    using complex_mult_cnj_cmod[of B]
    by (simp add: power2_eq_square)
  finally
  show ?thesis
    .
qed

lemma calc_ideal_point2_mix:
  assumes "(z1, z2) = calc_ideal_point2_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "B * cnj z1 * z2 + cnj B * z1 * cnj z2 = - 2 * A * (B * cnj B)\<^sup>2 "
proof-
  have "B*cnj z1 + cnj B*z1 = -2*A*B*cnj B"
    using assms eq_cnj_iff_real[of A]
    by (simp, simp add: field_simps)
  moreover
  have "cnj z2 = z2"
    using assms
    by simp
  hence "B*cnj z1*z2 + cnj B*z1*cnj z2 = (B*cnj z1 + cnj B*z1)*z2"
    by (simp add: field_simps)
  ultimately
  have "B*cnj z1*z2 + cnj B*z1*cnj z2 = -2*A*(B* cnj B)*z2"
    by simp
  also have "\<dots> = -2*A*(B * cnj B)\<^sup>2"
    using assms
    using complex_mult_cnj_cmod[of B]
    by (simp add: power2_eq_square)
  finally
  show ?thesis
    .
qed

lemma calc_ideal_point1_on_circline:
  assumes "(z1, z2) = calc_ideal_point1_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "A*z1*cnj z1 + B*cnj z1*z2 + cnj B*z1*cnj z2 + A*z2*cnj z2 = 0" (is "?lhs = 0")
proof-
  have "?lhs = A * (z1 * cnj z1 + z2 * cnj z2) + (B * cnj z1 * z2 + cnj B * z1 * cnj z2)"
    by (simp add: field_simps)
  also have "... = 2*A*(B*cnj B)\<^sup>2 + (-2*A*(B*cnj B)\<^sup>2)"
    using calc_ideal_point1_sq[OF assms]
    using calc_ideal_point1_mix[OF assms]
    by simp
  finally
  show ?thesis
    by simp
qed

lemma calc_ideal_point2_on_circline:
  assumes "(z1, z2) = calc_ideal_point2_cvec A B" "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
  shows "A*z1*cnj z1 + B*cnj z1*z2 + cnj B*z1*cnj z2 + A*z2*cnj z2 = 0" (is "?lhs = 0")
proof-
  have "?lhs = A * (z1 * cnj z1 + z2 * cnj z2) + (B * cnj z1 * z2 + cnj B * z1 * cnj z2)"
    by (simp add: field_simps)
  also have "... = 2*A*(B*cnj B)\<^sup>2 + (-2*A*(B*cnj B)\<^sup>2)"
    using calc_ideal_point2_sq[OF assms]
    using calc_ideal_point2_mix[OF assms]
    by simp
  finally
  show ?thesis
    by simp
qed

lemma calc_ideal_points_on_unit_circle:
  shows "\<forall> z \<in> calc_ideal_points H. z \<in> circline_set unit_circle"
  unfolding circline_set_def
  apply simp
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where *: "H = (A, B, C, D)"
    by (cases H, auto)
  have "\<forall> (z1, z2) \<in> calc_ideal_points_cmat_cvec H. z1 * cnj z1 = z2 * cnj z2"
    using hermitean_elems[of A B C D]
    unfolding calc_ideal_points_cmat_cvec_def
    using calc_ideal_point_1_unit[of A B]
    using calc_ideal_point_2_unit[of A B]
    using hh *
    apply (cases "calc_ideal_point1_cvec A B", cases "calc_ideal_point2_cvec A B")
    apply (auto simp add: Let_def simp del: calc_ideal_point1_cvec_def calc_ideal_point2_cvec_def)
    done
  thus "Ball (calc_ideal_points_cmat_cvec H) (on_circline_cmat_cvec unit_circle_cmat)"
    using on_circline_cmat_cvec_unit
    by (auto simp del: on_circline_cmat_cvec_def calc_ideal_points_cmat_cvec_def)
qed

lemma calc_ideal_points_on_circline:
  assumes "is_poincare_line H"
  shows "\<forall> z \<in> calc_ideal_points H. z \<in> circline_set H"
  using assms
  unfolding circline_set_def
  apply simp
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero"
  obtain A B C D where *: "H = (A, B, C, D)"
    by (cases H, auto)
  obtain z11 z12 z21 z22 where **: "(z11, z12) = calc_ideal_point1_cvec A B" "(z21, z22) = calc_ideal_point2_cvec A B"
    by (cases "calc_ideal_point1_cvec A B", cases "calc_ideal_point2_cvec A B") auto

  assume "is_poincare_line_cmat H"
  hence "\<forall> (z1, z2) \<in> calc_ideal_points_cmat_cvec H. A*z1*cnj z1 + B*cnj z1*z2 + C*z1*cnj z2 + D*z2*cnj z2 = 0"
    using * ** hh
    using hermitean_elems[of A B C D]
    using calc_ideal_point1_on_circline[of z11 z12 A B]
    using calc_ideal_point2_on_circline[of z21 z22 A B]
    by (auto simp del: calc_ideal_point1_cvec_def calc_ideal_point2_cvec_def)
  thus "Ball (calc_ideal_points_cmat_cvec H) (on_circline_cmat_cvec H)"
    using on_circline_cmat_cvec_circline_equation *
    by (auto simp del: on_circline_cmat_cvec_def calc_ideal_points_cmat_cvec_def simp add: field_simps)
qed

lemma calc_ideal_points_cvec_different [simp]:
  assumes "(cmod B)\<^sup>2 > (cmod A)\<^sup>2" "is_real A"
  shows "\<not> (calc_ideal_point1_cvec A B \<approx>\<^sub>v calc_ideal_point2_cvec A B)"
  using assms
  by (auto) (auto simp add: cmod_def)

lemma calc_ideal_points_different:
  assumes "is_poincare_line H"
  shows  "\<exists> i1 \<in> (calc_ideal_points H). \<exists> i2 \<in> (calc_ideal_points H). i1 \<noteq> i2"
  using assms
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero" "is_poincare_line_cmat H"
  obtain A B C D where *: "H = (A, B, C, D)"
    by (cases H, auto)
  hence "is_real A" using hh hermitean_elems by auto
  thus "\<exists>i1\<in>calc_ideal_points_cmat_cvec H. \<exists>i2\<in>calc_ideal_points_cmat_cvec H. \<not> i1 \<approx>\<^sub>v i2"
    using * hh calc_ideal_points_cvec_different[of A B]
    apply (rule_tac x="calc_ideal_point1_cvec A B" in bexI)
    apply (rule_tac x="calc_ideal_point2_cvec A B" in bexI)
    by auto
qed


lemma two_calc_ideal_points [simp]:
  assumes "is_poincare_line H"
  shows "card (calc_ideal_points H) = 2"
proof-
  have  "\<exists> x \<in> calc_ideal_points H. \<exists> y \<in> calc_ideal_points H. \<forall> z \<in> calc_ideal_points H. z = x \<or> z = y"
    by (transfer, transfer, case_tac H, simp del: calc_ideal_point1_cvec_def calc_ideal_point2_cvec_def)
  then obtain x y where *: "calc_ideal_points H = {x, y}"
    by auto
  moreover
  have "x \<noteq> y"
    using calc_ideal_points_different[OF assms] *
    by auto
  ultimately
  show ?thesis
    by auto
qed

definition ideal_points :: "circline \<Rightarrow> complex_homo set" where
  "ideal_points H = circline_intersection H unit_circle"

lemma two_ideal_points:
  assumes "is_poincare_line H"
  shows "card (ideal_points H) = 2"
proof-
  have "H \<noteq> unit_circle"
    using assms not_is_poincare_line_unit_circle
    by auto
  let ?int = "circline_intersection H unit_circle"
  obtain i1 i2 where "i1 \<in> ?int" "i2 \<in> ?int" "i1 \<noteq> i2"
    using calc_ideal_points_on_circline[OF assms]
    using calc_ideal_points_on_unit_circle[of H]
    using calc_ideal_points_different[OF assms]
    unfolding circline_intersection_def circline_set_def
    by auto
  thus ?thesis
    unfolding ideal_points_def
    using circline_intersection_at_most_2_points[OF `H \<noteq> unit_circle`]
    using card_geq_2_iff_contains_2_elems[of ?int]
    by auto
qed

lemma ideal_points_unique:
  assumes "is_poincare_line H"
  shows "ideal_points H = calc_ideal_points H"
proof-
  have "calc_ideal_points H \<subseteq> ideal_points H"
    using calc_ideal_points_on_circline[OF assms]
    using calc_ideal_points_on_unit_circle[of H]
    unfolding ideal_points_def circline_intersection_def circline_set_def
    by auto
  moreover
  have "H \<noteq> unit_circle"
    using not_is_poincare_line_unit_circle assms
    by auto
  hence "finite (ideal_points H)"
    using circline_intersection_at_most_2_points[of H unit_circle]
    unfolding ideal_points_def
    by auto
  ultimately
  show ?thesis
    using card_subset_eq[of "ideal_points H" "calc_ideal_points H"]
    using two_calc_ideal_points[OF assms]
    using two_ideal_points[OF assms]
    by auto
qed

lemma obtain_ideal_points:
  assumes "is_poincare_line H"
  obtains i1 i2 where "i1 \<noteq> i2" "ideal_points H = {i1, i2}"
  using two_ideal_points[OF assms] card_eq_2_iff_doubleton[of "ideal_points H"]
  by blast

lemma ideal_points_on_unit_circle:
  assumes "is_poincare_line H"
  shows "\<forall> z \<in> ideal_points H. z \<in> circline_set unit_circle"
  using assms
  using calc_ideal_points_on_unit_circle ideal_points_unique by blast

lemma ideal_points_on_circline:
  assumes "is_poincare_line H"
  shows "\<forall> z \<in> ideal_points H. z \<in> circline_set H"
  using assms
  by (simp add: calc_ideal_points_on_circline ideal_points_unique)

lemma ideal_points_different:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  assumes "ideal_points (poincare_line u v) = {i1, i2}"
  shows "i1 \<noteq> i2" "u \<noteq> i1" "u \<noteq> i2" "v \<noteq> i1" "v \<noteq> i2"
proof-
  have "i1 \<in> ocircline_set ounit_circle" "i2 \<in> ocircline_set ounit_circle"
    using assms(3) assms(4) ideal_points_on_unit_circle is_poincare_line_poincare_line
    by fastforce+
  thus "u \<noteq> i1" "u \<noteq> i2" "v \<noteq> i1" "v \<noteq> i2"
    using assms(1-2)
    using disc_inter_ocircline_set[of ounit_circle]
    unfolding unit_disc_def
    by auto
  show "i1 \<noteq> i2"
    using assms
    by (metis doubleton_eq_iff is_poincare_line_poincare_line obtain_ideal_points)
qed

lemma ideal_points_x_axis
  [simp]: "ideal_points x_axis = {of_complex (-1), of_complex 1}"
proof (subst ideal_points_unique, simp)
  have "calc_ideal_points_clmat_hcoords x_axis_clmat = {of_complex_hcoords (- 1), of_complex_hcoords 1}"
    by transfer auto
  thus "calc_ideal_points x_axis = {of_complex (- 1), of_complex 1}"
    by (simp add: calc_ideal_points.abs_eq of_complex.abs_eq x_axis_def)
qed


lemma ideal_points_moebius_circline [simp]:
  assumes  "unit_circle_fix M" "is_poincare_line H"
  shows "ideal_points (moebius_circline M H) = (moebius_pt M) ` (ideal_points H)" (is "?I' = ?M ` ?I")
proof-
  obtain i1 i2 where *: "i1 \<noteq> i2" "?I = {i1, i2}"
    using assms(2)
    by (rule obtain_ideal_points)
  let ?Mi1 = "?M i1" and ?Mi2 = "?M i2"
  have "?Mi1 \<in> ?M ` (circline_set H)"
       "?Mi2 \<in> ?M ` (circline_set H)"
       "?Mi1 \<in> ?M ` (circline_set unit_circle)"
       "?Mi2 \<in> ?M ` (circline_set unit_circle)"
    using *
    unfolding ideal_points_def circline_intersection_def circline_set_def
    by blast+
  hence "?Mi1 \<in> ?I'"
        "?Mi2 \<in> ?I'"
    using unit_circle_fix_iff[of M] assms
    unfolding ideal_points_def circline_intersection_def circline_set_def
    by auto (metis on_circline_moebius_circline_I)+
  moreover
  have "?Mi1 \<noteq> ?Mi2"
    using bij_moebius_pt[of M] *
    using moebius_pt_invert by blast
  moreover
  have "is_poincare_line (moebius_circline M H)"
    using assms unit_circle_fix_preserve_is_poincare_line
    by simp
  ultimately
  have "?I' = {?Mi1, ?Mi2}"
    using two_ideal_points[of "moebius_circline M H"]
    using card_eq_2_doubleton[of ?I' ?Mi1 ?Mi2]
    by simp
  thus ?thesis
    using *(2)
    by auto
qed

lemma ideal_points_poincare_line_moebius [simp]:
  assumes "unit_disc_fix M"  "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  assumes "ideal_points (poincare_line u v) = {i1, i2}"
  shows "ideal_points (poincare_line (moebius_pt M u) (moebius_pt M v)) = {moebius_pt M i1, moebius_pt M i2}"
  using assms
  by auto

lemma ideal_points_conjugate [simp]:
  assumes "is_poincare_line H"
  shows "ideal_points (conjugate_circline H) = conjugate ` (ideal_points H)" (is "?I' = ?M ` ?I")
proof-
  obtain i1 i2 where *: "i1 \<noteq> i2" "?I = {i1, i2}"
    using assms
    by (rule obtain_ideal_points)
  let ?Mi1 = "?M i1" and ?Mi2 = "?M i2"
  have "?Mi1 \<in> ?M ` (circline_set H)"
       "?Mi2 \<in> ?M ` (circline_set H)"
       "?Mi1 \<in> ?M ` (circline_set unit_circle)"
       "?Mi2 \<in> ?M ` (circline_set unit_circle)"
    using *
    unfolding ideal_points_def circline_intersection_def circline_set_def
    by blast+                                   
  hence "?Mi1 \<in> ?I'"
        "?Mi2 \<in> ?I'"
    unfolding ideal_points_def circline_intersection_def circline_set_def
     by auto  (metis conjugate_inj)+
  moreover
  have "?Mi1 \<noteq> ?Mi2"
    using `i1 \<noteq> i2`
    by (auto simp add: conjugate_inj)
  moreover
  have "is_poincare_line (conjugate_circline H)"
    using assms
    by simp
  ultimately
  have "?I' = {?Mi1, ?Mi2}"
    using two_ideal_points[of "conjugate_circline H"]
    using card_eq_2_doubleton[of ?I' ?Mi1 ?Mi2]
    by simp
  thus ?thesis
    using *(2)
    by auto
qed

lemma ideal_points_poincare_line_conjugate [simp]:
  assumes"u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  assumes "ideal_points (poincare_line u v) = {i1, i2}"
  shows "ideal_points (poincare_line (conjugate u) (conjugate v)) = {conjugate i1, conjugate i2}"
  using assms
  by auto

(* ------------------------------------------------------------------ *)
subsection{* Poincare distance *}
(* ------------------------------------------------------------------ *)

abbreviation Re_cross_ratio where "Re_cross_ratio z u v w \<equiv> Re (to_complex (cross_ratio z u v w))"

definition calc_poincare_distance :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> real" where
  [simp]: "calc_poincare_distance u i1 v i2 = abs (ln (Re_cross_ratio u i1 v i2))"

definition poincare_distance_pred :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> real \<Rightarrow> bool" where
  [simp]: "poincare_distance_pred u v d \<longleftrightarrow>
            (u = v \<and> d = 0) \<or> (u \<noteq> v \<and> (\<forall> i1 i2. ideal_points (poincare_line u v) = {i1, i2} \<longrightarrow> d = calc_poincare_distance u i1 v i2))"

definition poincare_distance :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> real" where
  "poincare_distance u v = (THE d. poincare_distance_pred u v d)"

lemma distance_cross_ratio_real_positive:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
  shows "\<forall> i1 i2. ideal_points (poincare_line u v) = {i1, i2} \<longrightarrow> 
                  cross_ratio u i1 v i2 \<noteq> \<infinity>\<^sub>h \<and> is_real (to_complex (cross_ratio u i1 v i2)) \<and> Re_cross_ratio u i1 v i2 > 0" (is "?P u v")
proof (rule wlog_positive_x_axis[OF assms])
  fix x
  assume *: "is_real x" "0 < Re x" "Re x < 1"
  hence "x \<noteq> -1" "x \<noteq> 1"
    by auto
  hence **: "of_complex x \<noteq> \<infinity>\<^sub>h" "of_complex x \<noteq> 0\<^sub>h" "of_complex x \<noteq> of_complex (-1)" "of_complex 1 \<noteq> of_complex x"
        "of_complex x \<in> circline_set x_axis"
    using *
    unfolding circline_set_x_axis
    by (auto simp add: of_complex_inj)

  have ***:  "0\<^sub>h \<noteq> of_complex (-1)" "0\<^sub>h \<noteq> of_complex 1"
    by auto (metis of_complex_inj of_complex_zero zero_neq_neg_one)

  have ****: "- x - 1 \<noteq> 0" "x - 1 \<noteq> 0"
    using `x \<noteq> -1` `x \<noteq> 1`
    by auto (metis add.inverse_inverse)

  have "poincare_line 0\<^sub>h (of_complex x) = x_axis"
    using **
    by (simp add: poincare_line_0_real_is_x_axis)
  thus "?P 0\<^sub>h (of_complex x)"
    using * ** *** ****
    using cross_ratio_not_inf[of "0\<^sub>h" "of_complex 1" "of_complex (-1)" "of_complex x"]
    using cross_ratio_not_inf[of "0\<^sub>h" "of_complex (-1)" "of_complex 1" "of_complex x"]
    using cross_ratio_real[of 0 "-1" x 1] cross_ratio_real[of 0 1 x "-1"]
    apply (auto simp add: poincare_line_0_real_is_x_axis doubleton_eq_iff circline_set_x_axis)
    apply (subst cross_ratio, simp_all, subst Re_complex_div_gt_0, simp, subst mult_neg_neg, simp_all)+
    done
next
  fix M u v
  let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v"
  assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
            "?P ?Mu ?Mv"
  show "?P u v"
  proof safe
    fix i1 i2
    let ?cr = "cross_ratio u i1 v i2"
    assume **: "ideal_points (poincare_line u v) = {i1, i2}"
    have "i1 \<noteq> u" "i1 \<noteq> v" "i2 \<noteq> u" "i2 \<noteq> v" "i1 \<noteq> i2"
      using ideal_points_different[OF *(2-3), of i1 i2] ** `u \<noteq> v`
      by auto
    hence "0 < Re (to_complex ?cr) \<and> is_real (to_complex ?cr) \<and> ?cr \<noteq> \<infinity>\<^sub>h"
      using * **
      apply (erule_tac x="moebius_pt M i1" in allE)
      apply (erule_tac x="moebius_pt M i2" in allE)
      apply (subst (asm) ideal_points_poincare_line_moebius[of M u v i1 i2], simp_all)
      done
    thus "0 < Re (to_complex ?cr)" "is_real (to_complex ?cr)" "?cr = \<infinity>\<^sub>h \<Longrightarrow> False"
      by simp_all
  qed
qed

lemma distance_unique:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "\<exists>! d. poincare_distance_pred u v d"
proof (cases "u = v")
  case True
  thus ?thesis
    by auto
next
  case False
  obtain i1 i2 where *: "i1 \<noteq> i2" "ideal_points (poincare_line u v) = {i1, i2}"
    using obtain_ideal_points[OF is_poincare_line_poincare_line] `u \<noteq> v`
    by blast
  let ?d = "calc_poincare_distance u i1 v i2"
  show ?thesis
  proof (rule ex1I)
    show "poincare_distance_pred u v ?d"
      using * `u \<noteq> v`
    proof (simp del: calc_poincare_distance_def, safe)
      fix i1' i2'
      assume "{i1, i2} = {i1', i2'}"
      hence **: "(i1' = i1 \<and> i2' = i2) \<or> (i1' = i2 \<and> i2' = i1)"
        using doubleton_eq_iff[of i1 i2 i1' i2']
        by blast
      have all_different: "u \<noteq> i1" "u \<noteq> i2" "v \<noteq> i1" "v \<noteq> i2" "u \<noteq> i1'" "u \<noteq> i2'" "v \<noteq> i1'" "v \<noteq> i2'" "i1 \<noteq> i2"
        using ideal_points_different[OF assms, of i1 i2] * ** `u \<noteq> v`
        by auto

      show "calc_poincare_distance u i1 v i2 = calc_poincare_distance u i1' v i2'"
      proof-
        let ?cr = "cross_ratio u i1 v i2"
        let ?cr' = "cross_ratio u i1' v i2'"

        have "Re (to_complex ?cr) > 0" "is_real (to_complex ?cr)"
             "Re (to_complex ?cr') > 0" "is_real (to_complex ?cr')"
          using False distance_cross_ratio_real_positive[OF assms(1-2)] * **
          by auto

        thus ?thesis
          using **
          using cross_ratio_not_zero cross_ratio_not_inf all_different
          by auto (subst cross_ratio_commute_24, subst reciprocal_real, simp_all add: ln_div)
      qed
    qed
  next
    fix d
    assume "poincare_distance_pred u v d"
    thus "d = ?d"
      using * `u \<noteq> v`
      by auto
  qed
qed

lemma poincare_distance_satisfies_pred [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance_pred u v (poincare_distance u v)"
    using distance_unique[OF assms] theI'[of "poincare_distance_pred u v"]
    unfolding poincare_distance_def
    by blast

lemma poincare_distance_I:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v" "ideal_points (poincare_line u v) = {i1, i2}"
  shows "poincare_distance u v = calc_poincare_distance u i1 v i2"
  using assms
  using poincare_distance_satisfies_pred[OF assms(1-2)]
  by simp

lemma poincare_distance_refl [simp]:
  assumes "u \<in> unit_disc"
  shows "poincare_distance u u = 0"
  using assms
  using poincare_distance_satisfies_pred[OF assms assms]
  by simp

lemma unit_disc_fix_preserve_poincare_distance [simp]:
  assumes "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance (moebius_pt M u) (moebius_pt M v) = poincare_distance u v"
proof (cases "u = v")
  case True
  have "moebius_pt M u \<in> unit_disc" "moebius_pt M v \<in> unit_disc"
    using unit_disc_fix_iff[OF assms(1), symmetric] assms
    by blast+
  thus ?thesis
    using assms `u = v`
    by simp
next
  case False
  obtain i1 i2 where *: "ideal_points (poincare_line u v) = {i1, i2}"
    using `u \<noteq> v`
    by (rule obtain_ideal_points[OF is_poincare_line_poincare_line[of u v]])
  let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v" and ?Mi1 = "moebius_pt M i1" and ?Mi2 = "moebius_pt M i2"

  have **: "?Mu \<in> unit_disc" "?Mv \<in> unit_disc"
    using assms
    using unit_disc_fix_iff
    by blast+

  have ***: "?Mu \<noteq> ?Mv"   
    using `u \<noteq> v` 
    by simp

  have "poincare_distance u v = calc_poincare_distance u i1 v i2"
    using poincare_distance_I[OF assms(2-3) `u \<noteq> v` *]
    by auto
  moreover
  have "unit_circle_fix M"
    using assms
    by simp
  hence ++: "ideal_points (poincare_line ?Mu ?Mv) = {?Mi1, ?Mi2}"
    using `u \<noteq> v` assms *
    by simp
  have "poincare_distance ?Mu ?Mv = calc_poincare_distance ?Mu ?Mi1 ?Mv ?Mi2"
    by (rule poincare_distance_I[OF ** *** ++])
  moreover
  have "calc_poincare_distance ?Mu ?Mi1 ?Mv ?Mi2 = calc_poincare_distance u i1 v i2"
    using ideal_points_different[OF assms(2-3) `u \<noteq> v` *]
    unfolding calc_poincare_distance_def
    by (subst moebius_preserve_cross_ratio[symmetric], simp_all)
  ultimately
  show ?thesis
    by simp
qed


text{* Knowing ideal points for x-axis, we can easily explicitly calculate distances *}

lemma poincare_distance_x_axis_x_axis:
  assumes "x \<in> unit_disc" "y \<in> unit_disc" "x \<in> circline_set x_axis" "y \<in> circline_set x_axis"
  shows "poincare_distance x y =
            (let x' = to_complex x; y' = to_complex y
              in abs (ln (Re (((1 + x') * (1 - y')) / ((1 - x') * (1 + y'))))))"
proof-
  obtain x' y' where *: "x = of_complex x'" "y = of_complex y'"
    using inf_or_of_complex[of x] inf_or_of_complex[of y] `x \<in> unit_disc` `y \<in> unit_disc`
    by auto

  have "cmod x' < 1" "cmod y' < 1"
    using `x \<in> unit_disc` `y \<in> unit_disc` *
    by (metis unit_disc_iff_cmod_lt_1)+
  hence **: "x' \<noteq> 1" "x' \<noteq> 1" "y' \<noteq> -1" "y' \<noteq> 1"
    by auto

  show ?thesis
  proof (cases "x = y")
    case True
    thus ?thesis
      using assms(1-2)
      using unit_disc_iff_cmod_lt_1[of "to_complex x"] * **
      by simp (metis add_left_cancel add_neg_numeral_special(7))
  next
    case False
    hence "poincare_line x y = x_axis"
      using poincare_line_x_axis[OF assms]
      by simp
    hence "ideal_points (poincare_line x y) = {of_complex (-1), of_complex 1}"
      by simp
    hence "poincare_distance x y = calc_poincare_distance x (of_complex (-1)) y (of_complex 1)"
      using poincare_distance_I assms `x \<noteq> y`
      by auto
    also have "... = abs (ln (Re (((x' + 1) * (y' - 1)) / ((x' - 1) * (y' + 1)))))"
      using * `cmod x' < 1` `cmod y' < 1`
      by (simp, transfer, transfer, auto)
    finally
    show ?thesis
      using *
      by (metis (no_types, lifting) add.commute minus_diff_eq minus_divide_divide mult_minus_left mult_minus_right to_complex_of_complex)
  qed
qed

lemma poincare_distance_zero_x_axis:
  assumes "x \<in> unit_disc" "x \<in> circline_set x_axis"
  shows "poincare_distance 0\<^sub>h x = (let x' = to_complex x in abs (ln (Re ((1 - x') / (1 + x')))))"
  using assms
  using poincare_distance_x_axis_x_axis[of "0\<^sub>h" x]
  by (simp add: Let_def)

lemma poincare_distance_zero:
  assumes "x \<in> unit_disc"
  shows "poincare_distance 0\<^sub>h x = (let x' = to_complex x in abs (ln (Re ((1 - cmod x') / (1 + cmod x')))))" (is "?P x")
proof (cases "x = 0\<^sub>h")
  case True
  thus ?thesis
    by auto
next
  case False
  show ?thesis
  proof (rule wlog_rotation_to_positive_x_axis)
    show "x \<in> unit_disc" "x \<noteq> 0\<^sub>h" by fact+
  next
    fix \<phi> u
    assume "u \<in> unit_disc" "u \<noteq> 0\<^sub>h" "?P (moebius_pt (moebius_rotation \<phi>) u)"
    thus "?P u"
      using unit_disc_fix_preserve_poincare_distance[of "moebius_rotation \<phi>" "0\<^sub>h" u]
      by (cases "u = \<infinity>\<^sub>h") (simp_all add: Let_def)
  next
    fix x
    assume "is_real x" "0 < Re x" "Re x < 1"
    thus "?P (of_complex x)"
      using poincare_distance_zero_x_axis[of "of_complex x"]
      by simp (auto simp add: circline_set_x_axis cmod_eq_Re complex_is_Real_iff)
  qed
qed

(* TODO: prove this for the whole circle of numbers with the same cmod *)
lemma poincare_distance_zero_opposite [simp]:
  assumes "of_complex z \<in> unit_disc"
  shows "poincare_distance 0\<^sub>h (of_complex (- z)) = poincare_distance 0\<^sub>h (of_complex z)"
proof-
  have *: "of_complex (-z) \<in> unit_disc"
    using assms
    by auto
  show ?thesis
    using poincare_distance_zero[OF assms]
    using poincare_distance_zero[OF *]
    by simp
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Distance explicit formula *}
(* ------------------------------------------------------------------ *)

abbreviation "cosh_dist u v \<equiv> cosh (poincare_distance u v)"

lemma cosh_poincare_distance_cross_ratio_average:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v" "ideal_points (poincare_line u v) = {i1, i2}"
  shows "cosh (poincare_distance u v) =
           ((Re_cross_ratio u i1 v i2) + (Re_cross_ratio v i1 u i2)) / 2"
proof-
  let ?cr = "cross_ratio u i1 v i2"
  let ?crRe = "Re (to_complex ?cr)"
  have "?cr \<noteq> \<infinity>\<^sub>h" "is_real (to_complex ?cr)" "?crRe > 0" 
    using distance_cross_ratio_real_positive[OF assms(1-3)] assms(4)
    by simp_all
  then obtain cr where *: "cross_ratio u i1 v i2 = of_complex cr" "cr \<noteq> 0" "is_real cr" "Re cr > 0"
    using inf_or_of_complex[of "cross_ratio u i1 v i2"]
    by (smt to_complex_of_complex zero_complex.simps(1))
  thus ?thesis
    using *
    using assms cross_ratio_commute_13[of v i1 u i2]
    unfolding poincare_distance_I[OF assms] calc_poincare_distance_def cosh_def
    by (cases "Re cr \<ge> 1")
       (auto simp add: ln_div[of 0] exp_minus field_simps Re_divide power2_eq_square complex.expand)
qed

definition poincare_distance_formula' :: "complex \<Rightarrow> complex \<Rightarrow> real" where
[simp]: "poincare_distance_formula' u v = 1 + 2 * ((cmod (u - v))\<^sup>2 / ((1 - (cmod u)\<^sup>2) * (1 - (cmod v)\<^sup>2)))"

definition poincare_distance_formula :: "complex \<Rightarrow> complex \<Rightarrow> real" where
  [simp]: "poincare_distance_formula u v = acosh (poincare_distance_formula' u v)"

lemma blaschke_preserve_distance_formula [simp]:
  assumes "of_complex k \<in> unit_disc" "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance_formula (to_complex (moebius_pt (blaschke k) u)) (to_complex (moebius_pt (blaschke k) v)) =
         poincare_distance_formula (to_complex u) (to_complex v)"
proof (cases "k = 0")
  case True
  thus ?thesis
    by simp
next
  case False
  obtain u' v' where *: "u' = to_complex u" "v' = to_complex v"
    by auto

  have "cmod u' < 1" "cmod v' < 1" "cmod k < 1"
    using assms *
    using inf_or_of_complex[of u] inf_or_of_complex[of v]
    by auto

  obtain nu du nv dv d kk ddu ddv where
    **: "nu = u' - k" "du = 1 - cnj k *u'" "nv = v' - k" "dv = 1 - cnj k * v'"
        "d = u' - v'" "ddu = 1 - u'*cnj u'" "ddv = 1 - v'*cnj v'" "kk = 1 - k*cnj k"
    by auto

  have d: "nu*dv - nv*du = d*kk"                          
    by (subst **)+ (simp add: field_simps)
  have ddu: "du*cnj du - nu*cnj nu = ddu*kk"
    by (subst **)+ (simp add: field_simps)
  have ddv: "dv*cnj dv - nv*cnj nv = ddv*kk"
    by (subst **)+ (simp add: field_simps)

  have "du \<noteq> 0"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "cmod (1 - cnj k * u') = 0"
      using `du = 1 - cnj k * u'`
      by auto
    hence "cmod (cnj k * u') = 1"
      by auto
    hence "cmod k * cmod u' = 1"
      by auto
    thus False
      using `cmod k < 1` `cmod u' < 1`
      using mult_strict_mono[of "cmod k" 1 "cmod u'" 1]
      by simp
  qed

  have "dv \<noteq> 0"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "cmod (1 - cnj k * v') = 0"
      using `dv = 1 - cnj k * v'`
      by auto
    hence "cmod (cnj k * v') = 1"
      by auto
    hence "cmod k * cmod v' = 1"
      by auto
    thus False
      using `cmod k < 1` `cmod v' < 1`
      using mult_strict_mono[of "cmod k" 1 "cmod v'" 1]
      by simp
  qed

  have "kk \<noteq> 0" 
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "cmod (1 - k * cnj k) = 0"
      using `kk = 1 - k * cnj k`
      by auto
    hence "cmod (k * cnj k) = 1"
      by auto
    hence "cmod k * cmod k = 1"
      by auto
    thus False
      using `cmod k < 1`
      using mult_strict_mono[of "cmod k" 1 "cmod k" 1]
      by simp
  qed

  note nz = `du \<noteq> 0` `dv \<noteq> 0` `kk \<noteq> 0`


  have "nu / du - nv / dv = (nu*dv - nv*du) / (du * dv)"              
    using nz
    by (simp add: field_simps)                               
  hence "(cmod (nu/du - nv/dv))\<^sup>2 = cmod ((d*kk) / (du*dv) * (cnj ((d*kk) / (du*dv))))" (is "?lhs = _")
    unfolding complex_mod_mult_cnj[symmetric]
    by (subst (asm) d) simp
  also have "... = cmod ((d*cnj d*kk*kk) / (du*cnj du*dv*cnj dv))"
    by (simp add: field_simps)
  finally have 1: "?lhs = cmod ((d*cnj d*kk*kk) / (du*cnj du*dv*cnj dv))"
    .                                                                           

  have "(1 - ((cmod nu) / (cmod du))\<^sup>2)*(1 - ((cmod nv) / (cmod dv))\<^sup>2) =
        (1 - cmod((nu * cnj nu) / (du * cnj du)))*(1 - cmod((nv * cnj nv) / (dv * cnj dv)))" (is "?rhs = _")
    by (metis cmod_divide complex_mod_mult_cnj power_divide)
  also have "... = cmod(((du*cnj du - nu*cnj nu) / (du * cnj du)) * ((dv*cnj dv - nv*cnj nv) / (dv * cnj dv)))"
  proof-
    have "u' \<noteq> 1 / cnj k" "v' \<noteq> 1 / cnj k"
      using `cmod u' < 1` `cmod v' < 1` `cmod k < 1`
      by (auto simp add: False)
    moreover
    have "cmod k \<noteq> 1"
      using `cmod k < 1`
      by linarith
    ultimately
    have "cmod (nu/du) < 1" "cmod (nv/dv) < 1"
      using **(1-4)
      using unit_disc_fix_discI[OF blaschke_unit_disc_fix[OF `cmod k < 1`] `u \<in> unit_disc`] `u' = to_complex u`
      using unit_disc_fix_discI[OF blaschke_unit_disc_fix[OF `cmod k < 1`] `v \<in> unit_disc`] `v' = to_complex v`
      using inf_or_of_complex[of u] `u \<in> unit_disc` inf_or_of_complex[of v] `v \<in> unit_disc`
      using moebius_pt_blaschke[of k u'] using moebius_pt_blaschke[of k v'] 
      by auto
    hence "(cmod (nu/du))\<^sup>2 < 1" "(cmod (nv/dv))\<^sup>2 < 1"
      by (simp_all add: cmod_def)
    hence "cmod (nu * cnj nu / (du * cnj du)) < 1"  "cmod (nv * cnj nv / (dv * cnj dv)) < 1"
      by (metis complex_mod_mult_cnj norm_divide power_divide)+
    moreover
    have "is_real (nu * cnj nu / (du * cnj du))" "is_real (nv * cnj nv / (dv * cnj dv))"
      using eq_cnj_iff_real[of "nu * cnj nu / (du * cnj du)"]      
      using eq_cnj_iff_real[of "nv * cnj nv / (dv * cnj dv)"]      
      by (auto simp add: mult.commute)
    moreover          
    have "Re (nu * cnj nu / (du * cnj du)) \<ge> 0"  "Re (nv * cnj nv / (dv * cnj dv)) \<ge> 0"
      using `du \<noteq> 0` `dv \<noteq> 0`
      unfolding complex_mult_cnj_cmod
      by simp_all
    ultimately                           
    have "1 - cmod (nu * cnj nu / (du * cnj du)) = cmod (1 - nu * cnj nu / (du * cnj du))"
         "1 - cmod (nv * cnj nv / (dv * cnj dv)) = cmod (1 - nv * cnj nv / (dv * cnj dv))"     
      by (simp_all add: cmod_def)
    thus ?thesis
      using nz
      apply simp
      apply (subst diff_divide_eq_iff, simp, simp)
      apply (subst diff_divide_eq_iff, simp, simp)
      done
  qed    
  also have "... = cmod(((ddu * kk) / (du * cnj du)) * ((ddv * kk) / (dv * cnj dv)))"
    by (subst ddu, subst ddv, simp)
  also have "... = cmod((ddu*ddv*kk*kk) / (du*cnj du*dv*cnj dv))"
    by (simp add: field_simps)
  finally have 2: "?rhs = cmod((ddu*ddv*kk*kk) / (du*cnj du*dv*cnj dv))"
    .

  have "?lhs / ?rhs =
       cmod ((d*cnj d*kk*kk) / (du*cnj du*dv*cnj dv)) / cmod((ddu*ddv*kk*kk) / (du*cnj du*dv*cnj dv))"
    by (subst 1, subst 2, simp)
  also have "... = cmod ((d*cnj d)/(ddu*ddv))"
    using nz
    by simp
  also have "... = (cmod d)\<^sup>2 / ((1 - (cmod u')\<^sup>2)*(1 - (cmod v')\<^sup>2))"
  proof-
    have "(cmod u')\<^sup>2 < 1" "(cmod v')\<^sup>2 < 1"
      using `cmod u' < 1` `cmod v' < 1`
      by (simp_all add: cmod_def)
    hence "cmod (1 - u' * cnj u') = 1 - (cmod u')\<^sup>2" "cmod (1 - v' * cnj v') = 1 - (cmod v')\<^sup>2"
      by (auto simp add: cmod_eq_Re cmod_power2 power2_eq_square[symmetric])
    thus ?thesis
      using nz
      apply (subst **)+
      unfolding complex_mod_mult_cnj[symmetric]      
      by simp
  qed
  finally
  have 3: "?lhs / ?rhs = (cmod d)\<^sup>2 / ((1 - (cmod u')\<^sup>2)*(1 - (cmod v')\<^sup>2))"
    .

  have "cmod k \<noteq> 1" "u' \<noteq> 1 / cnj k" "v' \<noteq> 1 / cnj k" "u \<noteq> \<infinity>\<^sub>h" "v \<noteq> \<infinity>\<^sub>h"
    using `cmod k < 1` `u \<in> unit_disc` `v \<in> unit_disc` * `k \<noteq> 0` ** `kk \<noteq> 0` nz
    by auto
  thus ?thesis using assms
    using * ** 3
    using moebius_pt_blaschke[of k u']
    using moebius_pt_blaschke[of k v']
    by simp
qed

lemma rotation_preserve_distance_formula [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance_formula (to_complex (moebius_pt (moebius_rotation \<phi>) u)) (to_complex (moebius_pt (moebius_rotation \<phi>) v)) =
         poincare_distance_formula (to_complex u) (to_complex v)"
  using assms
  using inf_or_of_complex[of u] inf_or_of_complex[of v]
  by auto

lemma unit_disc_fix_preserve_distance_formula [simp]:
  assumes "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance_formula (to_complex (moebius_pt M u)) (to_complex (moebius_pt M v)) =
         poincare_distance_formula (to_complex u) (to_complex v)" (is "?P' u v M")
proof-
  have "\<forall> u \<in> unit_disc. \<forall> v \<in> unit_disc. ?P' u v M" (is "?P M")
  proof (rule wlog_unit_disc_fix[OF assms(1)])
    fix k
    assume "cmod k < 1"
    hence "of_complex k \<in> unit_disc"
      by simp
    thus  "?P (blaschke k)"
      using blaschke_preserve_distance_formula
      by simp
  next
    fix \<phi>
    show "?P (moebius_rotation \<phi>)"
      using rotation_preserve_distance_formula
      by simp
  next
    fix M1 M2
    assume *: "?P M1" and **: "?P M2"  and u11: "unit_disc_fix M1" "unit_disc_fix M2"
    thus "?P (M1 + M2)"
      by (auto simp del: poincare_distance_formula_def)
  qed
  thus ?thesis
    using assms
    by simp
qed

lemma poincare_distance_formula:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance u v = poincare_distance_formula (to_complex u) (to_complex v)" (is "?P u v")
proof (rule wlog_x_axis)
  fix x
  assume *: "is_real x" "0 \<le> Re x" "Re x < 1"
  show "?P  0\<^sub>h (of_complex x)" (is "?lhs = ?rhs")
  proof-
    have "of_complex x \<in> unit_disc" "of_complex x \<in> circline_set x_axis" "cmod x < 1"
      using * cmod_eq_Re
      by (auto simp add: circline_set_x_axis)
    hence "?lhs = \<bar>ln (Re ((1 - x) / (1 + x)))\<bar>"
      using poincare_distance_zero_x_axis[of "of_complex x"]
      by simp
    moreover
    have "?rhs = \<bar>ln (Re ((1 - x) / (1 + x)))\<bar>"
    proof-
      let ?x = "1 + 2 * (cmod x)\<^sup>2 / (1 - (cmod x)\<^sup>2)"
      have "?rhs = acosh ?x"
        by simp
      also have "... = ln ((1 + (cmod x)\<^sup>2) / (1 - (cmod x)\<^sup>2) + 2 * (cmod x) / (1 - (cmod x)\<^sup>2))"
      proof-
        have "1 - (cmod x)\<^sup>2 > 0"
          using `cmod x < 1`
          by (smt norm_not_less_zero one_power2 power2_eq_imp_eq power_mono)
        hence 1: "?x = (1 + (cmod x)\<^sup>2) / (1 - (cmod x)\<^sup>2)"
          by (simp add: field_simps)
        have 2: "?x\<^sup>2 - 1 = (4 * (cmod x)\<^sup>2) / (1 - (cmod x)\<^sup>2)\<^sup>2"
          using `1 - (cmod x)\<^sup>2 > 0`       
          apply (subst 1)
          unfolding power_divide
          by (subst divide_diff_eq_iff, simp, simp add: power2_eq_square field_simps)
        show ?thesis
          using `1 - (cmod x)\<^sup>2 > 0`
          unfolding acosh_def
          apply (subst 2)
          apply (subst 1)
          apply (subst real_sqrt_divide)
          apply (subst real_sqrt_mult)
          apply simp
          done
      qed
      also have "... = ln (((1 + (cmod x))\<^sup>2) / (1 - (cmod x)\<^sup>2))"
        apply (subst add_divide_distrib[symmetric])
        apply (simp add: field_simps power2_eq_square)
        done
      also have "... = ln ((1 + cmod x) / (1 - (cmod x)))"
        using `cmod x < 1`      
        using square_diff_square_factored[of 1 "cmod x"]
        by (simp add: power2_eq_square)
      also have "... = \<bar>ln (Re ((1 - x) / (1 + x)))\<bar>"
      proof-
        have *: "Re ((1 - x) / (1 + x)) \<le> 1" "Re ((1 - x) / (1 + x)) > 0"
          using `is_real x` `Re x \<ge> 0` `Re x < 1`
          using complex_is_Real_iff
          by auto
        hence "\<bar>ln (Re ((1 - x) / (1 + x)))\<bar> = - ln (Re ((1 - x) / (1 + x)))"
          by auto
        hence "\<bar>ln (Re ((1 - x) / (1 + x)))\<bar> = ln (Re ((1 + x) / (1 - x)))"
          using ln_div[of 1 "Re ((1 - x)/(1 + x))"] * `is_real x`
          by (simp add: complex_is_Real_iff)
        moreover
        have "ln ((1 + cmod x) / (1 - cmod x)) = ln ((1 + Re x) / (1 - Re x))"
          using `Re x \<ge> 0` `is_real x`
          using cmod_eq_Re by auto
        moreover
        have "(1 + Re x) / (1 - Re x) = Re ((1 + x) / (1 - x))"
          using `is_real x` `Re x < 1`
          by (smt Re_divide_real eq_iff_diff_eq_0 minus_complex.simps one_complex.simps plus_complex.simps)
        ultimately
        show ?thesis
          by simp
      qed
      finally
      show ?thesis
        .
    qed
    ultimately
    show ?thesis
      by simp
  qed
next
  fix M u v
  assume *: "unit_disc_fix M"  "u \<in> unit_disc" "v \<in> unit_disc"
  assume "?P (moebius_pt M u) (moebius_pt M v)"
  thus "?P u v"
    using *(1-3)
    by (simp del: poincare_distance_formula_def)
next
  show "u \<in> unit_disc" "v \<in> unit_disc"
    by fact+
qed

(* Some additional properties proved easily using the distance formula *)

lemma poincare_distance_sym:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance u v = poincare_distance v u"
  using assms
  using poincare_distance_formula[OF assms(1) assms(2)]
  using poincare_distance_formula[OF assms(2) assms(1)]
  by (simp add: mult.commute norm_minus_commute)

lemma poincare_distance_formula'_ge_1:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "1 \<le> poincare_distance_formula' (to_complex u) (to_complex v)"
  using unit_disc_cmod_square_lt_1[OF assms(1)] unit_disc_cmod_square_lt_1[OF assms(2)]
  by auto

lemma poincare_distance_ge0:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance u v \<ge> 0"
  using poincare_distance_formula'_ge_1
  unfolding poincare_distance_formula[OF assms(1) assms(2)]
  unfolding poincare_distance_formula_def
  unfolding poincare_distance_formula'_def
  by (rule acosh_ge_0, simp_all add: assms)

lemma cosh_dist:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "cosh_dist u v = poincare_distance_formula' (to_complex u) (to_complex v)"
  using poincare_distance_formula[OF assms] poincare_distance_formula'_ge_1[OF assms]
  by simp

lemma poincare_distance_eq_0_iff:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance u v = 0 \<longleftrightarrow> u = v"
  using assms
  apply auto
  using poincare_distance_formula'_ge_1[OF assms]
  using unit_disc_cmod_square_lt_1[OF assms(1)] unit_disc_cmod_square_lt_1[OF assms(2)]
  unfolding poincare_distance_formula[OF assms(1) assms(2)]
  unfolding poincare_distance_formula_def
  unfolding poincare_distance_formula'_def
  apply (subst (asm) acosh_eq_0_iff)
  apply assumption
  apply (simp add: unit_disc_to_complex_inj)
  done

lemma conjugate_preserve_poincare_distance [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance (conjugate u) (conjugate v) = poincare_distance u v"
proof-
  obtain u' v' where *: "u = of_complex u'" "v = of_complex v'"
    using assms inf_or_of_complex[of u] inf_or_of_complex[of v]
    by auto

  have **: "conjugate u \<in> unit_disc" "conjugate v \<in> unit_disc"
    using * assms
    by auto

  show ?thesis
    using *
    using poincare_distance_formula[OF assms]
    using poincare_distance_formula[OF **]
    by simp (metis complex_cnj_diff complex_mod_cnj)
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Existence and uniqueness of points with a given distance *}
(* ------------------------------------------------------------------ *)

lemma ex_x_axis_poincare_distance_negative':
  fixes d :: real
  assumes "d \<ge> 0"
  shows "let z = (1 - exp d) / (1 + exp d)
          in is_real z \<and> Re z \<le> 0 \<and> Re z > -1 \<and>
              of_complex z \<in> unit_disc \<and> of_complex z \<in> circline_set x_axis \<and>
              poincare_distance 0\<^sub>h (of_complex z) = d"
proof-
  have "exp d \<ge> 1"
    using assms
    using one_le_exp_iff[of d, symmetric]
    by blast

  hence "1 + exp d \<noteq> 0"
    by linarith

  let ?z = "(1 - exp d) / (1 + exp d)"

  have "?z \<le> 0"
    using `exp d \<ge> 1`
    by (simp add: divide_nonpos_nonneg)

  moreover

  have "?z > -1"
    using exp_gt_zero[of d]
    by (smt divide_less_eq_1_neg nonzero_minus_divide_right)

  moreover

  hence "abs ?z < 1"
    using `?z \<le> 0`
    by simp
  hence "cmod ?z < 1"
    by (metis norm_of_real)
  hence "of_complex ?z \<in> unit_disc"
    by simp

  moreover
  have "of_complex ?z \<in> circline_set x_axis"
    unfolding circline_set_x_axis
    by simp

  moreover
  have "(1 - ?z) / (1 + ?z) = exp d"
  proof-
    have "1 + ?z = 2 / (1 + exp d)"
      using `1 + exp d \<noteq> 0`
      by (subst add_divide_eq_iff, auto)
    moreover
    have "1 - ?z = 2 * exp d / (1 + exp d)"
      using `1 + exp d \<noteq> 0`
      by (subst diff_divide_eq_iff, auto)
    ultimately
    show ?thesis
      using `1 + exp d \<noteq> 0`
      by simp
  qed

  ultimately
  show ?thesis
    using poincare_distance_zero_x_axis[of "of_complex ?z"]
    using `d \<ge> 0` `exp d \<ge> 1`
    by simp (simp add: cmod_eq_Re)
qed

lemma ex_x_axis_poincare_distance_negative:
  assumes "d \<ge> 0"
  shows "\<exists> z. is_real z \<and> Re z \<le> 0 \<and> Re z > -1 \<and>
              of_complex z \<in> unit_disc \<and> of_complex z \<in> circline_set x_axis \<and>
              poincare_distance 0\<^sub>h (of_complex z) = d" (is "\<exists> z. ?P z")
  using ex_x_axis_poincare_distance_negative'[OF assms]
  unfolding Let_def
  by blast

(* the proof might be simplified using uniqueness property of the cross-ratio *)
lemma unique_x_axis_poincare_distance_negative:
  assumes "d \<ge> 0"
  shows "\<exists>! z. is_real z \<and> Re z \<le> 0 \<and> Re z > -1 \<and>
              poincare_distance 0\<^sub>h (of_complex z) = d" (is "\<exists>! z. ?P z")
proof-
  let ?z = "(1 - exp d) / (1 + exp d)"

  have "?P ?z"
    using ex_x_axis_poincare_distance_negative'[OF assms]
    unfolding Let_def
    by blast

  moreover

  have "\<forall> z'. ?P z' \<longrightarrow> z' = ?z"
  proof-
    let ?g = "\<lambda> x'. \<bar>ln (Re ((1 - x') / (1 + x')))\<bar>"
    let ?A = "{x. is_real x \<and> Re x > -1 \<and> Re x \<le> 0}"
    have "inj_on (poincare_distance 0\<^sub>h \<circ> of_complex) ?A"
    proof (rule comp_inj_on)
      show "inj_on of_complex ?A"
        using of_complex_inj
        unfolding inj_on_def
        by blast
    next
      show "inj_on (poincare_distance 0\<^sub>h) (of_complex ` ?A)" (is "inj_on ?f (of_complex ` ?A)")
      proof (subst inj_on_cong)
        have *: "of_complex ` ?A =
                 {z. z \<in> unit_disc \<and> z \<in> circline_set x_axis \<and> Re (to_complex z) \<le> 0}" (is "_ = ?B")
          by (auto simp add: cmod_eq_Re circline_set_x_axis)

        fix x
        assume "x \<in> of_complex ` ?A"
        hence "x \<in> ?B"
          using *
          by simp
        thus "poincare_distance 0\<^sub>h x = (?g \<circ> to_complex) x"
          using poincare_distance_zero_x_axis
          by (simp add: Let_def)
      next
        have *: "to_complex ` of_complex ` ?A = ?A"
          by (auto simp add: image_iff)

        show "inj_on (?g \<circ> to_complex) (of_complex ` ?A)"
        proof (rule comp_inj_on)
          show "inj_on to_complex (of_complex ` ?A)"
            unfolding inj_on_def
            by auto
        next
          have "inj_on ?g ?A"
            unfolding inj_on_def
          proof(safe)
            fix x y
            assume hh: "is_real x" "is_real y" "- 1 < Re x" "Re x \<le> 0"
              "- 1 < Re y" "Re y \<le> 0" "\<bar>ln (Re ((1 - x) / (1 + x)))\<bar> = \<bar>ln (Re ((1 - y) / (1 + y)))\<bar>"

            have "is_real ((1 - x)/(1 + x))"
              using `is_real x` div_reals[of "1-x" "1+x"]
              by auto
            have "is_real ((1 - y)/(1 + y))"
              using `is_real y` div_reals[of "1-y" "1+y"]
              by auto

            have "Re (1 + x) > 0"
              using `- 1 < Re x` by auto
            hence "1 + x \<noteq> 0"
              by force
            have "Re (1 - x) \<ge> 0"
              using `Re x \<le> 0` by auto
            hence "Re ((1 - x)/(1 + x)) > 0"
              using Re_divide_real `0 < Re (1 + x)` complex_eq_if_Re_eq hh(1) hh(4) by auto
            have "Re(1 - x) \<ge> Re ( 1 + x)"
              using hh by auto
            hence "Re ((1 - x)/(1 + x)) \<ge> 1"
              using `Re (1 + x) > 0` `is_real ((1 - x)/(1 + x))`
              by (smt Re_divide_real arg_0_iff hh(1) le_divide_eq_1_pos one_complex.simps(2) plus_complex.simps(2))            

            have "Re (1 + y) > 0"
              using `- 1 < Re y` by auto
            hence "1 + y \<noteq> 0"
              by force
            have "Re (1 - y) \<ge> 0"
              using `Re y \<le> 0` by auto
            hence "Re ((1 - y)/(1 + y)) > 0"
              using Re_divide_real `0 < Re (1 + y)` complex_eq_if_Re_eq hh by auto
            have "Re(1 - y) \<ge> Re ( 1 + y)"
              using hh by auto
            hence "Re ((1 - y)/(1 + y)) \<ge> 1"
              using `Re (1 + y) > 0` `is_real ((1 - y)/(1 + y))`
              by (smt Re_divide_real arg_0_iff hh le_divide_eq_1_pos one_complex.simps(2) plus_complex.simps(2))

            have "ln (Re ((1 - x) / (1 + x))) = ln (Re ((1 - y) / (1 + y)))"
              using `Re ((1 - y)/(1 + y)) \<ge> 1` `Re ((1 - x)/(1 + x)) \<ge> 1` hh
              by auto
            hence "Re ((1 - x) / (1 + x)) = Re ((1 - y) / (1 + y))"
              using `Re ((1 - y)/(1 + y)) > 0` `Re ((1 - x)/(1 + x)) > 0`
              by auto
            hence "(1 - x) / (1 + x) = (1 - y) / (1 + y)"
              using `is_real ((1 - y)/(1 + y))` `is_real ((1 - x)/(1 + x))`
              using complex_eq_if_Re_eq by blast
            hence "(1 - x) * (1 + y) = (1 - y) * (1 + x)"
              using `1 + y \<noteq> 0` `1 + x \<noteq> 0` 
              by (simp add:field_simps)
            thus "x = y"
              by (simp add:field_simps)
          qed            
          thus "inj_on ?g (to_complex ` of_complex ` ?A)"
            using *
            by simp
        qed
      qed
    qed
    thus ?thesis
      using `?P ?z`
      unfolding inj_on_def
      by auto
  qed
  ultimately
  show ?thesis
    by blast
qed

lemma ex_x_axis_poincare_distance_positive:
  assumes "d \<ge> 0"
  shows "\<exists> z. is_real z \<and> Re z \<ge> 0 \<and> Re z < 1 \<and>
              of_complex z \<in> unit_disc \<and> of_complex z \<in> circline_set x_axis \<and>
              poincare_distance 0\<^sub>h (of_complex z) = d" (is "\<exists> z. is_real z \<and> Re z \<ge> 0 \<and> Re z < 1 \<and> ?P z")
proof-
  obtain z where *: "is_real z" "Re z \<le> 0" "Re z > -1" "?P z"
    using ex_x_axis_poincare_distance_negative[OF assms]
    by auto
  hence **: "of_complex z \<in> unit_disc" "of_complex z \<in> circline_set x_axis"
    by (auto simp add: cmod_eq_Re)
  have "is_real (-z) \<and> Re (-z) \<ge> 0 \<and> Re (-z) < 1 \<and> ?P (-z)"
    using * **
    by (simp add: circline_set_x_axis)
  thus ?thesis
    by blast
qed

lemma unique_x_axis_poincare_distance_positive:
  assumes "d \<ge> 0"
  shows "\<exists>! z. is_real z \<and> Re z \<ge> 0 \<and> Re z < 1 \<and>
               poincare_distance 0\<^sub>h (of_complex z) = d" (is "\<exists>! z. is_real z \<and> Re z \<ge> 0 \<and> Re z < 1 \<and> ?P z")
proof-
  obtain z where *: "is_real z" "Re z \<le> 0" "Re z > -1" "?P z"
    using unique_x_axis_poincare_distance_negative[OF assms]
    by auto
  hence **: "of_complex z \<in> unit_disc" "of_complex z \<in> circline_set x_axis"
    by (auto simp add: cmod_eq_Re circline_set_x_axis)
  show ?thesis
  proof
    show "is_real (-z) \<and> Re (-z) \<ge> 0 \<and> Re (-z) < 1 \<and> ?P (-z)"
      using * **
      by simp
  next
    fix z'
    assume "is_real z' \<and> Re z' \<ge> 0 \<and> Re z' < 1 \<and> ?P z'"
    hence "is_real (-z') \<and> Re (-z') \<le> 0 \<and> Re (-z') > -1 \<and> ?P (-z')"
      by (auto simp add: circline_set_x_axis cmod_eq_Re)
    hence "-z' = z"
      using unique_x_axis_poincare_distance_negative[OF assms] *
      by blast
    thus "z' = -z"
      by auto
  qed
qed

text{* Equal distance implies that segments are isometric - this means that congruence could be
defined either by two segments having the same distance or by requiring existence of an isometry
that maps one segment to the other *}

lemma poincare_distance_eq_ex_moebius:
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "u' \<in> unit_disc" "v' \<in> unit_disc"
  assumes "poincare_distance u v = poincare_distance u' v'"
  shows "\<exists> M. unit_disc_fix M \<and> moebius_pt M u = u' \<and> moebius_pt M v = v'" (is "?P' u v u' v'")
proof (cases "u = v")
  case True
  thus ?thesis
    using assms poincare_distance_eq_0_iff[of u' v']
    by (simp add: unit_disc_fix_transitive)
next
  case False
  have "\<forall> u' v'. u \<noteq> v \<and> u' \<in> unit_disc \<and> v' \<in> unit_disc \<and> poincare_distance u v = poincare_distance u' v' \<longrightarrow>
                 ?P' u' v' u v" (is "?P u v")
  proof (rule wlog_positive_x_axis[where P="?P"])
    fix x
    assume "is_real x" "0 < Re x" "Re x < 1"
    hence "of_complex x \<in> unit_disc" "of_complex x \<in> circline_set x_axis"
      unfolding circline_set_x_axis
      by (auto simp add: cmod_eq_Re)

    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix u' v'
      assume "0\<^sub>h \<noteq> of_complex x" and in_disc: "u' \<in> unit_disc" "v' \<in> unit_disc" and
             "poincare_distance 0\<^sub>h (of_complex x) = poincare_distance u' v'"
      hence "u' \<noteq> v'" "poincare_distance u' v' > 0"
        using poincare_distance_eq_0_iff[of "0\<^sub>h" "of_complex x"] `of_complex x \<in> unit_disc`
        using poincare_distance_ge0[of "0\<^sub>h" "of_complex x"]
        by auto
      then obtain M where M: "unit_disc_fix M" "moebius_pt M u' = 0\<^sub>h" "moebius_pt M v' \<in> positive_x_axis"
        using ex_unit_disc_fix_to_zero_positive_x_axis[of u' v'] in_disc
        by auto

      then obtain Mv' where Mv': "moebius_pt M v' = of_complex Mv'"
        using inf_or_of_complex[of "moebius_pt M v'"] in_disc unit_disc_fix_iff[of M]
        by (metis image_eqI inf_notin_unit_disc)

      have "moebius_pt M v' \<in> unit_disc"
        using M(1) `v' \<in> unit_disc`
        by auto

      have "Re Mv' > 0" "is_real Mv'" "Re Mv' < 1"
        using M Mv' of_complex_inj `moebius_pt M v' \<in> unit_disc`
        unfolding positive_x_axis_def circline_set_x_axis
        using cmod_eq_Re
        by auto fastforce

      have "poincare_distance 0\<^sub>h (moebius_pt M v') = poincare_distance u' v'"
        using M(1)
        using in_disc
        by (subst M(2)[symmetric], simp)

      have "Mv' = x"
        using `poincare_distance 0\<^sub>h (moebius_pt M v') = poincare_distance u' v'` Mv'
        using `poincare_distance 0\<^sub>h (of_complex x) = poincare_distance u' v'`
        using unique_x_axis_poincare_distance_positive[of "poincare_distance u' v'"]
          `poincare_distance u' v' > 0`
        using `Re Mv' > 0` `Re Mv' < 1` `is_real Mv'`
        using `is_real x` `Re x > 0` `Re x < 1`
        unfolding positive_x_axis_def
        by auto

      thus "?P' u' v' 0\<^sub>h (of_complex x)"
        using M Mv'
        by auto
    qed
  next
    show "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
      by fact+
  next
    fix M u v
    let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v"
    assume 1: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
    hence 2: "?Mu \<noteq> ?Mv" "?Mu \<in> unit_disc" "?Mv \<in> unit_disc"
      by auto
    assume 3: "?P (moebius_pt M u) (moebius_pt M v)"
    show "?P u v"
    proof safe
      fix u' v'
      assume 4: "u' \<in> unit_disc" "v' \<in> unit_disc" "poincare_distance u v = poincare_distance u' v'"
      hence "poincare_distance ?Mu ?Mv = poincare_distance u v"
        using 1
        by simp
      then obtain M' where 5: "unit_disc_fix M'" "moebius_pt M' u' = ?Mu" "moebius_pt M' v' = ?Mv"
        using 2 3 4
        by auto
      let ?M = "(-M) + M'"
      have "unit_disc_fix ?M \<and> moebius_pt ?M u' = u \<and> moebius_pt ?M v' = v"
        using 5 `unit_disc_fix M`
        using unit_disc_fix_moebius_comp[of "-M" "M'"]
        using unit_disc_fix_moebius_inv[of M]
        by simp
      thus "\<exists>M. unit_disc_fix M \<and> moebius_pt M u' = u \<and> moebius_pt M v' = v"
        by blast
    qed
  qed
  then obtain M where "unit_disc_fix M \<and> moebius_pt M u' = u \<and> moebius_pt M v' = v"
    using assms `u \<noteq> v`
    by blast
  hence "unit_disc_fix (-M) \<and> moebius_pt (-M) u = u' \<and> moebius_pt (-M) v = v'"
    using unit_disc_fix_moebius_inv[of M]
    by auto
  thus ?thesis
    by blast
qed

(* Might be easier to prove after introducing betweenness *)

lemma unique_midpoint_x_axis:
  assumes x: "is_real x" "-1 < Re x" "Re x < 1" and
          y: "is_real y" "-1 < Re y" "Re y < 1" and
          "x \<noteq> y"
  shows "\<exists>! z. -1 < Re z \<and> Re z < 1 \<and> is_real z \<and> poincare_distance (of_complex z) (of_complex x) = poincare_distance (of_complex z) (of_complex y)" (is "\<exists>! z. ?R z (of_complex x) (of_complex y)")
proof-
  let ?x = "of_complex x" and ?y = "of_complex y"
  let ?P = "\<lambda> x y. \<exists>! z. ?R z x y"
  have "\<forall> x. -1 < Re x \<and> Re x < 1 \<and> is_real x \<and> of_complex x \<noteq> ?y \<longrightarrow> ?P (of_complex x) ?y" (is "?Q (of_complex y)")
  proof (rule wlog_real_zero)
    show "?y \<in> unit_disc"
      using y
      by (simp add: cmod_eq_Re)
  next
    show "is_real (to_complex ?y)"
      using y
      by simp
  next
    show "?Q 0\<^sub>h"
    proof (rule allI, rule impI, (erule conjE)+)
      fix x
      assume x: "-1 < Re x" "Re x < 1" "is_real x" 
      let ?x = "of_complex x"
      assume "?x \<noteq> 0\<^sub>h"
      hence "x \<noteq> 0"
        by auto
      hence "Re x \<noteq> 0"
        using x
        using complex_neq_0
        by auto

      have *: "\<forall> a. -1 < a \<and> a < 1 \<longrightarrow> 
                 (poincare_distance (of_complex (cor a)) ?x = poincare_distance (of_complex (cor a)) 0\<^sub>h \<longleftrightarrow>
                 (Re x) * a * a - 2 * a + Re x = 0)"
      proof (rule allI, rule impI)
        fix a :: real
        assume "-1 < a \<and> a < 1"
        hence "of_complex (cor a) \<in> unit_disc"
          by auto
        moreover
        have "(a - Re x)\<^sup>2 / ((1 - a\<^sup>2) * (1 - (Re x)\<^sup>2)) = a\<^sup>2 / (1 - a\<^sup>2) \<longleftrightarrow>
              (Re x) * a * a - 2 * a + Re x = 0" (is "?lhs \<longleftrightarrow> ?rhs")
        proof-
          have "1 - a\<^sup>2 \<noteq> 0"
            using `-1 < a \<and> a < 1`
            by (metis cancel_comm_monoid_add_class.diff_cancel diff_eq_diff_less less_numeral_extra(4) power2_eq_1_iff right_minus_eq)
          hence "?lhs \<longleftrightarrow> (a - Re x)\<^sup>2 / (1 - (Re x)\<^sup>2) = a\<^sup>2"
            by (smt divide_cancel_right divide_divide_eq_left mult.commute)
          also have "... \<longleftrightarrow> (a - Re x)\<^sup>2 = a\<^sup>2 * (1 - (Re x)\<^sup>2)"
          proof-
            have "1 - (Re x)\<^sup>2 \<noteq> 0"
              using x
              by (smt power2_eq_1_iff)
            thus ?thesis
              by (simp add: divide_eq_eq)
          qed
          also have "... \<longleftrightarrow> a\<^sup>2 * (Re x)\<^sup>2 - 2*a*Re x + (Re x)\<^sup>2 = 0"
            by (simp add: power2_diff field_simps)
          also have "... \<longleftrightarrow> Re x * (a\<^sup>2 * Re x - 2 * a + Re x) = 0"
            by (simp add: power2_eq_square field_simps)
          also have "... \<longleftrightarrow> ?rhs"
            using `Re x \<noteq> 0`
            by (simp add: mult.commute mult.left_commute power2_eq_square)
          finally
          show ?thesis
            .
        qed
        moreover 
        have "acosh (1 + 2 * ((a - Re x)\<^sup>2 / ((1 - a\<^sup>2) * (1 - (Re x)\<^sup>2)))) = acosh (1 + 2 * a\<^sup>2 / (1 - a\<^sup>2)) \<longleftrightarrow> ?lhs"
          using `-1 < a \<and> a < 1` x mult_left_cancel[of "2::real" "(a - Re x)\<^sup>2 / ((1 - a\<^sup>2) * (1 - (Re x)\<^sup>2))" "a\<^sup>2 / (1 - a\<^sup>2)"]
          by (subst acosh_eq_iff, simp_all add: square_le_1)
        ultimately
        show "poincare_distance (of_complex (cor a)) (of_complex x) = poincare_distance (of_complex (cor a)) 0\<^sub>h \<longleftrightarrow>
              (Re x) * a * a - 2 * a + Re x = 0"
          using x
          by (auto simp add: poincare_distance_formula cmod_eq_Re)
      qed

      show "?P ?x 0\<^sub>h"
      proof
        let ?a = "(1 - sqrt(1 - (Re x)\<^sup>2)) / (Re x)"        
        let ?b = "(1 + sqrt(1 - (Re x)\<^sup>2)) / (Re x)"

        have "is_real ?a"
          by simp                                                       
        moreover
        have "1 - (Re x)\<^sup>2 > 0"
          using x
          by (smt power2_eq_1_iff square_le_1)
        have "\<bar>?a\<bar> < 1"
        proof (cases "Re x > 0")
          case True
          have "(1 - Re x)\<^sup>2 < 1 - (Re x)\<^sup>2"
            using `Re x > 0` x
            by (simp add: power2_eq_square field_simps)
          hence "1 - Re x < sqrt (1 - (Re x)\<^sup>2)"
            using real_less_rsqrt by fastforce
          thus ?thesis
            using `1 - (Re x)\<^sup>2 > 0` `Re x > 0`
            by simp
        next
          case False
          hence "Re x < 0"
            using `Re x \<noteq> 0`
            by simp

          have "1 + Re x > 0"
            using `Re x > -1`           
            by simp
          hence "2*Re x + 2*Re x*Re x < 0"
            using `Re x < 0`
            by (metis comm_semiring_class.distrib mult.commute mult_2_right mult_less_0_iff one_add_one zero_less_double_add_iff_zero_less_single_add)
          hence "(1 + Re x)\<^sup>2 < 1 - (Re x)\<^sup>2"
            by (simp add: power2_eq_square field_simps)
          hence "1 + Re x < sqrt (1 - (Re x)\<^sup>2)"
            using `1 - (Re x)\<^sup>2 > 0`
            using real_less_rsqrt by blast
          thus ?thesis
            using `Re x < 0`
            by (simp add: field_simps)
        qed
        hence "-1 < ?a" "?a < 1"
          by linarith+
        moreover
        have "(Re x) * ?a * ?a - 2 * ?a + Re x = 0"
          using `Re x \<noteq> 0` `1 - (Re x)\<^sup>2 > 0`
          by (simp add: field_simps power2_eq_square)
        ultimately
        show "-1 < Re (cor ?a) \<and> Re (cor ?a) < 1 \<and> is_real ?a \<and> poincare_distance (of_complex ?a) (of_complex x) = poincare_distance (of_complex ?a) 0\<^sub>h"
          using *
          by auto

        fix z
        assume **: "- 1 < Re z \<and> Re z < 1 \<and> is_real z \<and>
               poincare_distance (of_complex z) (of_complex x) = poincare_distance (of_complex z) 0\<^sub>h"
        hence "Re x * Re z * Re z - 2 * Re z + Re x = 0"
          using *[rule_format, of "Re z"] x
          by (auto simp add: complex_of_real_Re)
        moreover 
        have "sqrt (4 - 4 * Re x * Re x) = 2 * sqrt(1 - Re x * Re x)"
        proof-
          have "sqrt (4 - 4 * Re x * Re x) = sqrt(4 * (1 - Re x * Re x))"
            by simp
          thus ?thesis
            by (simp only: real_sqrt_mult, simp)
        qed
        moreover
        have "(2 - 2 * sqrt (1 - Re x * Re x)) / (2 * Re x) = ?a"
        proof-
          have "(2 - 2 * sqrt (1 - Re x * Re x)) / (2 * Re x) = 
               (2 * (1 - sqrt (1 - Re x * Re x))) / (2 * Re x)"
            by simp
          thus ?thesis
            by (subst (asm) mult_divide_mult_cancel_left) (auto simp add: power2_eq_square)
        qed
        moreover
        have "(2 + 2 * sqrt (1 - Re x * Re x)) / (2 * Re x) = ?b"
        proof-
          have "(2 + 2 * sqrt (1 - Re x * Re x)) / (2 * Re x) = 
               (2 * (1 + sqrt (1 - Re x * Re x))) / (2 * Re x)"
            by simp
          thus ?thesis
            by (subst (asm) mult_divide_mult_cancel_left) (auto simp add: power2_eq_square)
        qed
        ultimately
        have "Re z = ?a \<or> Re z = ?b"
          using real_quadratic_equation_only_two_roots[of "Re x" "Re z" "-2" "Re x"] 
          using `Re x \<noteq> 0` `-1 < Re x` `Re x < 1` `1 - (Re x)\<^sup>2 > 0`
          by (auto simp add: power2_eq_square)
          
          
        have "\<bar>?b\<bar> > 1"
        proof (cases "Re x > 0")
          case True
          have "(Re x - 1)\<^sup>2 < 1 - (Re x)\<^sup>2"
            using `Re x > 0` x
            by (simp add: power2_eq_square field_simps)
          hence "Re x - 1 < sqrt (1 - (Re x)\<^sup>2)"
            using real_less_rsqrt
            by simp
          thus ?thesis
            using `1 - (Re x)\<^sup>2 > 0` `Re x > 0`
            by simp
        next
          case False
          hence "Re x < 0"
            using `Re x \<noteq> 0`
            by simp      
          have "1 + Re x > 0"
            using `Re x > -1`
            by simp
          hence "2*Re x + 2*Re x*Re x < 0"
            using `Re x < 0`
            by (metis comm_semiring_class.distrib mult.commute mult_2_right mult_less_0_iff one_add_one zero_less_double_add_iff_zero_less_single_add)
          hence "1 - (Re x)\<^sup>2 > (- 1 - (Re x))\<^sup>2"
            by (simp add: field_simps power2_eq_square)
          hence "sqrt (1 - (Re x)\<^sup>2) > -1 - Re x"
            using real_less_rsqrt
            by simp
          thus ?thesis
            using `Re x < 0`
            by (simp add: field_simps)
        qed
        hence "?b < -1 \<or> ?b > 1"
          by auto

        hence "Re z = ?a"
          using `Re z = ?a \<or> Re z = ?b` **
          by auto
        thus "z = ?a"
          using ** complex_of_real_Re
          by fastforce
      qed
    qed
  next
    fix a u
    let ?M = "moebius_pt (blaschke a)"
    let ?Mu = "?M u"
    assume "u \<in> unit_disc" "is_real a" "cmod a < 1"
    assume *: "?Q ?Mu"
    show "?Q u"
    proof (rule allI, rule impI, (erule conjE)+)
      fix x                                    
      assume x: "-1 < Re x" "Re x < 1" "is_real x" "of_complex x \<noteq> u"
      let ?Mx = "?M (of_complex x)"
      have "of_complex x \<in> unit_disc"
        using x cmod_eq_Re
        by auto
      hence "?Mx \<in> unit_disc"
        using `is_real a` `cmod a < 1` blaschke_unit_disc_fix[of a]
        using unit_disc_fix_discI
        by blast
      hence "?Mx \<noteq> \<infinity>\<^sub>h"
        by auto
      moreover
      have "of_complex x \<in> circline_set x_axis"
        using x
        by auto
      hence "?Mx \<in> circline_set x_axis"
        using blaschke_real_preserve_x_axis[OF `is_real a` `cmod a < 1`, of "of_complex x"]
        by auto
      hence "-1 < Re (to_complex ?Mx) \<and> Re (to_complex ?Mx) < 1 \<and> is_real (to_complex ?Mx)"
        using `?Mx \<noteq> \<infinity>\<^sub>h` `?Mx \<in> unit_disc`
        unfolding circline_set_x_axis
        by (auto simp add: cmod_eq_Re)
      moreover
      have "?Mx \<noteq> ?Mu"
        using `of_complex x \<noteq> u`
        by simp
      ultimately
      have "?P ?Mx ?Mu"
        using *[rule_format, of "to_complex ?Mx"] `?Mx \<noteq> \<infinity>\<^sub>h`
        by simp
      then obtain Mz where
        "?R Mz ?Mx ?Mu"
        by blast
      have "of_complex Mz \<in> unit_disc" "of_complex Mz \<in> circline_set x_axis"
        using `?R Mz ?Mx ?Mu`
        using cmod_eq_Re 
        by auto

      let ?Minv = "- (blaschke a)"
      let ?z = "moebius_pt ?Minv (of_complex Mz)"
      have "?z \<in> unit_disc"
        using `of_complex Mz \<in> unit_disc` `cmod a < 1`
        by auto
      moreover
      have "?z \<in> circline_set x_axis"
        using `of_complex Mz \<in> circline_set x_axis`
        using blaschke_real_preserve_x_axis `is_real a` `cmod a < 1`
        by fastforce
      ultimately
      have z1: "-1 < Re (to_complex ?z)" "Re (to_complex ?z) < 1" "is_real (to_complex ?z)"
        using inf_or_of_complex[of "?z"]
        unfolding circline_set_x_axis
        by (auto simp add: cmod_eq_Re)
      
      have z2: "poincare_distance ?z (of_complex x) = poincare_distance ?z u"
        using `?R Mz ?Mx ?Mu` `cmod a < 1` `?z \<in> unit_disc` `of_complex x \<in> unit_disc` `u \<in> unit_disc`
        by (metis blaschke_preserve_distance_formula blaschke_unit_disc_fix moebius_pt_comp_inv_right poincare_distance_formula uminus_moebius_def unit_disc_fix_discI unit_disc_iff_cmod_lt_1)
      show "?P (of_complex x) u"
      proof
        show "?R (to_complex ?z) (of_complex x) u"
          using z1 z2 `?z \<in> unit_disc` inf_or_of_complex[of ?z]
          by auto
      next
        fix z'
        assume "?R z' (of_complex x) u"
        hence "of_complex z' \<in> unit_disc" "of_complex z' \<in> circline_set x_axis"
          by (auto simp add: cmod_eq_Re)
        let ?Mz' = "?M (of_complex z')"
        have "?Mz' \<in> unit_disc" "?Mz' \<in> circline_set x_axis"
          using `of_complex z' \<in> unit_disc` `of_complex z' \<in> circline_set x_axis` `cmod a < 1` `is_real a`
          using  blaschke_unit_disc_fix unit_disc_fix_discI
          using blaschke_real_preserve_x_axis circline_set_x_axis
          by blast+
        hence "-1 < Re (to_complex ?Mz')" "Re (to_complex ?Mz') < 1" "is_real (to_complex ?Mz')"
          unfolding circline_set_x_axis
          by (auto simp add: cmod_eq_Re)
        moreover
        have "poincare_distance ?Mz' ?Mx = poincare_distance ?Mz' ?Mu"
          using `?R z' (of_complex x) u`
          using \<open>cmod a < 1\<close> \<open>of_complex x \<in> unit_disc\<close> \<open>of_complex z' \<in> unit_disc\<close> \<open>u \<in> unit_disc\<close>
          by auto
        ultimately
        have "?R (to_complex ?Mz') ?Mx ?Mu"
          using `?Mz' \<in> unit_disc` inf_or_of_complex[of ?Mz']
          by auto
        hence "?Mz' = of_complex Mz"
          using `?P ?Mx ?Mu` `?R Mz ?Mx ?Mu`
          by (metis \<open>moebius_pt (blaschke a) (of_complex z') \<in> unit_disc\<close> \<open>of_complex Mz \<in> unit_disc\<close> to_complex_of_complex unit_disc_to_complex_inj)
        thus "z' = to_complex ?z"
          using moebius_pt_invert by auto
      qed
    qed
  qed
  thus ?thesis
    using assms
    by (metis to_complex_of_complex)
qed
  

(* ------------------------------------------------------------------ *)
subsubsection{* Triangle inequality *}
(* ------------------------------------------------------------------ *)

lemma poincare_distance_formula_zero_sum:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  shows "poincare_distance u 0\<^sub>h + poincare_distance 0\<^sub>h v =
         (let u' = cmod (to_complex u); v' = cmod (to_complex v)
           in acosh (((1 + u'\<^sup>2) * (1 + v'\<^sup>2) + 4 * u' * v') / ((1 - u'\<^sup>2) * (1 - v'\<^sup>2))))"
proof-
  obtain u' v' where uv: "u' = to_complex u" "v' = to_complex v"
    by auto
  have uv': "u = of_complex u'" "v = of_complex v'"
    using uv assms inf_or_of_complex[of u] inf_or_of_complex[of v]
    by auto

  let ?u' = "cmod u'" and ?v' = "cmod v'"

  have disc: "?u'\<^sup>2 < 1" "?v'\<^sup>2 < 1"
    using unit_disc_cmod_square_lt_1[OF `u \<in> unit_disc`]
    using unit_disc_cmod_square_lt_1[OF `v \<in> unit_disc`] uv
    by auto
  thm acosh_add
  have "acosh (1 + 2 * ?u'\<^sup>2 / (1 - ?u'\<^sup>2)) + acosh (1 + 2 * ?v'\<^sup>2 / (1 - ?v'\<^sup>2)) =
        acosh (((1 + ?u'\<^sup>2) * (1 + ?v'\<^sup>2) + 4 * ?u' * ?v') / ((1 - ?u'\<^sup>2) * (1 - ?v'\<^sup>2)))" (is "acosh ?ll + acosh ?rr = acosh ?r")
  proof (subst acosh_add)
    show "?ll \<ge> 1"  "?rr \<ge> 1"
      using disc
      by auto
  next
    show "acosh ((1 + 2 * ?u'\<^sup>2 / (1 - ?u'\<^sup>2)) * (1 + 2 * ?v'\<^sup>2 / (1 - ?v'\<^sup>2)) +
                  sqrt (((1 + 2 * ?u'\<^sup>2 / (1 - ?u'\<^sup>2))\<^sup>2 - 1) * ((1 + 2 * ?v'\<^sup>2 / (1 - ?v'\<^sup>2))\<^sup>2 - 1))) =
          acosh ?r" (is "acosh ?l = _")
    proof-
      have "1 + 2 * ?u'\<^sup>2 / (1 - ?u'\<^sup>2) = (1 + ?u'\<^sup>2) / (1 - ?u'\<^sup>2)"
        using disc
        by (subst add_divide_eq_iff, simp_all)
      moreover
      have "1 + 2 * ?v'\<^sup>2 / (1 - ?v'\<^sup>2) = (1 + ?v'\<^sup>2) / (1 - ?v'\<^sup>2)"
        using disc
        by (subst add_divide_eq_iff, simp_all)
      moreover
      have "sqrt (((1 + 2 * ?u'\<^sup>2 / (1 - ?u'\<^sup>2))\<^sup>2 - 1) * ((1 + 2 * ?v'\<^sup>2 / (1 - ?v'\<^sup>2))\<^sup>2 - 1)) =
               (4  * ?u' * ?v') / ((1 - ?u'\<^sup>2) * (1 - ?v'\<^sup>2))" (is "sqrt ?s = ?t")
      proof-
        have "?s = ?t\<^sup>2"
          using disc
          apply (subst add_divide_eq_iff, simp)+
          apply (subst power_divide)+
          apply simp
          apply (subst divide_diff_eq_iff, simp)+
          apply (simp add: power2_eq_square field_simps)
          done
        thus ?thesis
          using disc
          by simp
      qed
      ultimately
      have "?l = ?r"
        using disc
        by simp (subst add_divide_distrib, simp)
      thus ?thesis
        by simp
    qed
  qed
  thus ?thesis
    using uv' assms
    using poincare_distance_formula
    by (simp add: Let_def)
qed

lemma poincare_distance_triangle_inequality:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  shows "poincare_distance u v + poincare_distance v w \<ge> poincare_distance u w" (is "?P' u v w")
proof-
  have "\<forall> w. w \<in> unit_disc \<longrightarrow> ?P' u v w"  (is "?P v u")
  proof (rule wlog_x_axis[where P="?P"])
    fix x
    assume "is_real x" "0 \<le> Re x" "Re x < 1"
    hence "of_complex x \<in> unit_disc"
      by (simp add: cmod_eq_Re)

    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix w
      assume "w \<in> unit_disc"
      then obtain w' where w: "w = of_complex w'"
        using inf_or_of_complex[of w]
        by auto

      let ?x = "cmod x" and ?w = "cmod w'" and ?xw = "cmod (x - w')"

      have disc: "?x\<^sup>2 < 1" "?w\<^sup>2 < 1"
        using unit_disc_cmod_square_lt_1[OF `of_complex x \<in> unit_disc`]
        using unit_disc_cmod_square_lt_1[OF `w \<in> unit_disc`] w
        by auto

      have "poincare_distance (of_complex x) 0\<^sub>h + poincare_distance 0\<^sub>h w =
           acosh (((1 + ?x\<^sup>2) * (1 + ?w\<^sup>2) + 4 * ?x * ?w) / ((1 - ?x\<^sup>2) * (1 - ?w\<^sup>2)))" (is "_ = acosh ?r1")
        using poincare_distance_formula_zero_sum[OF `of_complex x \<in> unit_disc` `w \<in> unit_disc`] w
        by (simp add: Let_def)
      moreover
      have "poincare_distance (of_complex x) (of_complex w') =
            acosh (((1 - ?x\<^sup>2) * (1 - ?w\<^sup>2) + 2 * ?xw\<^sup>2) / ((1 - ?x\<^sup>2) * (1 - ?w\<^sup>2)))" (is "_ = acosh ?r2")
        using disc
        using poincare_distance_formula[OF `of_complex x \<in> unit_disc` `w \<in> unit_disc`] w
        by (subst add_divide_distrib) simp
      moreover
      have *: "(1 - ?x\<^sup>2) * (1 - ?w\<^sup>2) + 2 * ?xw\<^sup>2 \<le> (1 + ?x\<^sup>2) * (1 + ?w\<^sup>2) + 4 * ?x * ?w"
      proof-
        have "(cmod (x - w'))\<^sup>2 \<le> (cmod x + cmod w')\<^sup>2"
          using norm_triangle_ineq4[of x w']
          by (simp add: power_mono)
        thus ?thesis
          by (simp add: field_simps power2_sum)
      qed
      have "acosh ?r1 \<ge> acosh ?r2"
      proof (subst acosh_mono)
        show "?r1 \<ge> 1"
          using disc
          by simp (smt "*" power2_less_0)
      next
        show "?r2 \<ge> 1"
          using disc
          by simp
      next
        show "?r1 \<ge> ?r2"
          using disc
          using *
          by (subst divide_right_mono, simp_all)
      qed
      ultimately
      show "poincare_distance (of_complex x) w \<le> poincare_distance (of_complex x) 0\<^sub>h + poincare_distance 0\<^sub>h w"
        using `of_complex x \<in> unit_disc` `w \<in> unit_disc` w
        using poincare_distance_formula
        by simp
    qed
  next
    show "v \<in> unit_disc" "u \<in> unit_disc"
      by fact+
  next
    fix M u v
    assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc"
    assume **: "?P (moebius_pt M u) (moebius_pt M v)"
    show "?P u v"
    proof safe
      fix w
      assume "w \<in> unit_disc"
      thus "?P' v u w"
        using * **[rule_format, of "moebius_pt M w"]
        by simp
    qed
  qed
  thus ?thesis
    using assms
    by auto
qed

(* -------------------------------------------------------------------------- *)
subsection{* Circle *}
(* -------------------------------------------------------------------------- *)

definition poincare_circle :: "complex_homo \<Rightarrow> real \<Rightarrow> complex_homo set" where
  "poincare_circle z r = {z'. z' \<in> unit_disc \<and> poincare_distance z z' = r}"

(* Each poincare circle is represented by an euclidean circle in the model -
   the center and radius of that euclidean circle are determined by the following formulas *)
definition poincare_circle_euclidean :: "complex_homo \<Rightarrow> real \<Rightarrow> euclidean_circle" where
  "poincare_circle_euclidean z r =
      (let R = (cosh r - 1) / 2;
           z' = to_complex z;
           cz = 1 - (cmod z')\<^sup>2;
           k = cz * R + 1
        in (z' / k, cz * sqrt(R * (R + 1)) / k))"

(* The euclidean circle is always fully within the disc *)
lemma poincare_circle_in_disc:
  assumes "r > 0" "z \<in> unit_disc" "(ze, re) = poincare_circle_euclidean z r"
  shows "cmod ze < 1" "re > 0" "\<forall> x \<in> circle ze re. cmod x < 1"
proof-
  let ?R = "(cosh r - 1) / 2"
  let ?z' = "to_complex z"
  let ?cz = "1 - (cmod ?z')\<^sup>2"
  let ?k = "?cz * ?R + 1"
  let ?ze = "?z' / ?k"
  let ?re = "?cz * sqrt(?R * (?R + 1)) / ?k"

  from `z \<in> unit_disc`
  obtain z' where z': "z = of_complex z'"
    using inf_or_of_complex[of z]
    by auto

  hence "z' = ?z'"
    by simp

  obtain cz where cz: "cz = (1 - (cmod z')\<^sup>2)"
    by auto

  have "cz > 0" "cz \<le> 1"
    using `z \<in> unit_disc` z' cz
    using unit_disc_cmod_square_lt_1
    by fastforce+

  obtain R where R: "R = ?R"
    by blast

  have "R > 0"
    using cosh_gt_1[of r] `r > 0`
    by (subst R) simp

  obtain k where k: "k = cz * R + 1"
    by auto

  have "k > 1"
    using k `R > 0` `cz > 0`
    by simp

  hence "cmod k = k"
    by simp

  let ?RR = "cz * sqrt(R * (R + 1)) / k"

  have "cmod z' + cz * sqrt(R * (R + 1)) < k"
  proof-
    have "((R+1)-R)\<^sup>2 > 0"
      by simp
    hence "(R+1)\<^sup>2 - 2*R*(R+1) + R\<^sup>2 > 0"
      unfolding power2_diff
      by (simp add: field_simps)
    hence "(R+1)\<^sup>2 + 2*R*(R+1) + R\<^sup>2 - 4*R*(R+1) > 0"
      by simp
    hence "(2*R+1)\<^sup>2 / 4 > R*(R+1)"
      using power2_sum[of "R+1" R]
      by (simp add: field_simps)
    hence "sqrt(R*(R+1)) < (2*R+1) / 2"
      using `R > 0`
      by (smt arith_geo_mean_sqrt linordered_field_class.sign_simps(45) power_divide real_sqrt_four real_sqrt_pow2)
    hence "sqrt(R*(R+1)) - R < 1/2"
      by (simp add: field_simps)
    hence "(1 + (cmod z')) * (sqrt(R*(R+1)) - R) < (1 + (cmod z')) *  (1 / 2)"
      by (subst mult_strict_left_mono, auto, smt norm_not_less_zero)
    also have "... < 1"
      using `z \<in> unit_disc` z'
      by auto
    finally have "(1 - cmod z') * ((1 + cmod z') * (sqrt(R*(R+1)) - R)) < (1 - cmod z') * 1"
      using `z \<in> unit_disc` z'
      by (subst mult_strict_left_mono, simp_all)
    hence "cz * (sqrt (R*(R+1)) - R) < 1 - cmod z'"
      using square_diff_square_factored[of 1 "cmod z'"]
      by (subst cz, subst (asm) mult.assoc[symmetric], simp add: power2_eq_square field_simps)
    hence "cmod z' + cz * sqrt(R*(R+1)) < 1 + R * cz"
      by (simp add: field_simps)
    thus ?thesis
      using k
      by (simp add: field_simps)
  qed
  hence "cmod z' / k + cz * sqrt(R * (R + 1)) / k < 1"
    using `k > 1`
    unfolding add_divide_distrib[symmetric]
    by simp
  hence "cmod (z' / k) + cz * sqrt(R * (R + 1)) / k < 1"
    using `k > 1`
    by simp
  hence "cmod ?ze + ?re < 1"
    using k cz `R = ?R` z'
    by simp

  moreover

  have "cz * sqrt(R * (R + 1)) / k > 0"
    using `cz > 0` `R > 0` `k > 1`
    by auto
  hence "?re > 0"
    using k cz `R = ?R` z'
    by simp

  moreover

  have "cmod ?ze < 1"
    using `cmod ?ze + ?re < 1` `?re > 0`
    by simp

  moreover

  have "ze = ?ze" "re = ?re"
    using `(ze, re) = poincare_circle_euclidean z r`
    unfolding poincare_circle_euclidean_def Let_def
    by simp_all

  moreover

  have "\<forall> x \<in> circle ze re. cmod x \<le> cmod ze + re"
    using norm_triangle_ineq2[of _ ze]
    unfolding circle_def
    by auto smt

  ultimately

  show "cmod ze < 1" "re > 0" "\<forall> x \<in> circle ze re. cmod x < 1"
    by auto
qed


lemma poincare_circle_is_euclidean_circle:
  assumes "z \<in> unit_disc" "r > 0"
  shows  "let (Ze, Re) = poincare_circle_euclidean z r
           in of_complex ` (circle Ze Re) = poincare_circle z r"
proof-
  {
    fix x
    let ?z = "to_complex z"

    from assms obtain z' where z': "z = of_complex z'" "cmod z' < 1"
      using inf_or_of_complex[of z]
      by auto

    have *: "\<And> x. cmod x < 1 \<Longrightarrow> 1 - (cmod x)\<^sup>2 > 0"
      by auto (metis norm_eq_zero pos2 power_0 power_strict_decreasing zero_less_norm_iff zero_power2)

    let ?R = "(cosh r - 1) / 2"
    obtain R where R: "R = ?R"
      by blast

    let ?cx = "1 - (cmod x)\<^sup>2" and ?cz = "1 - (cmod z')\<^sup>2"  and ?czx = "(cmod (z' - x))\<^sup>2"

    let ?k = "1 + R * ?cz"
    obtain k where k: "k = ?k"
      by blast
    have "R > 0"
      using R cosh_gt_1[OF `r > 0`]
      by simp

    hence "k > 1"
      using assms z' k *[of z']
      by auto
    hence **: "cor k \<noteq> 0"
      by (smt of_real_eq_0_iff)


    have "of_complex x \<in> poincare_circle z r \<longleftrightarrow> cmod x < 1 \<and> poincare_distance z (of_complex x) = r"
      unfolding poincare_circle_def
      by auto
    also have "... \<longleftrightarrow> cmod x < 1 \<and> poincare_distance_formula' ?z x = cosh r"
      using poincare_distance_formula[of z "of_complex x"] cosh_dist[of z "of_complex x"]
      unfolding poincare_distance_formula_def
      using assms
      by auto
    also have "... \<longleftrightarrow> cmod x < 1 \<and> ?czx / (?cz * ?cx) = ?R"
      using z'
      by (simp add: field_simps)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> ?czx = ?R * ?cx * ?cz"
      using assms z' *[of z'] *[of x]
      using nonzero_divide_eq_eq[of "(1 - (cmod x)\<^sup>2) * (1 - (cmod z')\<^sup>2)" "(cmod (z' - x))\<^sup>2" ?R]
      by (auto, simp add: field_simps)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (z' - x) * (cnj z' - cnj x) = R * ?cz * (1 - x * cnj x)" (is "_ \<longleftrightarrow> _ \<and> ?l = ?r")
    proof-
      let ?l = "(z' - x) * (cnj z' - cnj x)" and ?r = "R * (1 - Re (z' * cnj z')) * (1 - x * cnj x)"
      have "is_real ?l"
        using eq_cnj_iff_real[of "?l"]
        by simp
      moreover
      have "is_real ?r"
        using eq_cnj_iff_real[of "1 - x * cnj x"]
        using is_real_complex_of_real[of "R * (1 - Re (z' * cnj z'))"]
        by simp
      ultimately
      show ?thesis
        apply (subst R[symmetric])
        apply (subst cmod_square)+
        apply (subst complex_eq_if_Re_eq, simp_all add: field_simps)
        done
    qed
    also have "... \<longleftrightarrow> cmod x < 1 \<and> z' * cnj z' - x * cnj z' - cnj x * z' + x * cnj x = R * ?cz - R * ?cz * x * cnj x"
      unfolding right_diff_distrib left_diff_distrib
      by (simp add: field_simps)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> k * (x * cnj x) - x * cnj z' - cnj x * z' + z' * cnj z' = R * ?cz" (is "_ \<longleftrightarrow> _ \<and> ?lhs = ?rhs")
      by (subst k) (auto simp add: field_simps)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (k * x * cnj x - x * cnj z' - cnj x * z' + z' * cnj z') / k = (R * ?cz) / k"
      using **
      by (auto simp add: Groups.mult_ac(1))
    also have "... \<longleftrightarrow> cmod x < 1 \<and> x * cnj x - x * cnj z' / k - cnj x * z' / k + z' * cnj z' / k = (R * ?cz) / k"
      using **
      unfolding add_divide_distrib diff_divide_distrib
      by auto
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (x - z'/k) * cnj(x - z'/k) = (R * ?cz) / k + (z' / k) * cnj(z' / k) - z' * cnj z' / k"
      by (auto simp add: field_simps diff_divide_distrib)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (cmod (x - z'/k))\<^sup>2 = (R * ?cz) / k + (cmod z')\<^sup>2 / k\<^sup>2 - (cmod z')\<^sup>2 / k"
      apply (subst complex_mult_cnj_cmod)+
      apply (subst complex_eq_if_Re_eq)
      apply (simp_all add: power_divide)
      done
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (cmod (x - z'/k))\<^sup>2 = (R * ?cz * k + (cmod z')\<^sup>2 - (cmod z')\<^sup>2 * k) / k\<^sup>2"
      using **
      unfolding add_divide_distrib diff_divide_distrib
      by (simp add: power2_eq_square)
    also have "... \<longleftrightarrow> cmod x < 1 \<and> (cmod (x - z'/k))\<^sup>2 = ?cz\<^sup>2 * R * (R + 1) / k\<^sup>2" (is "_ \<longleftrightarrow> _ \<and> ?a\<^sup>2 = ?b")
    proof-
      have *: "R * (1 - (cmod z')\<^sup>2) * k + (cmod z')\<^sup>2 - (cmod z')\<^sup>2 * k = (1 - (cmod z')\<^sup>2)\<^sup>2 * R * (R + 1)"
        by (subst k)+ (simp add: field_simps power2_diff)
      thus ?thesis
        by (subst *, simp)
    qed
    also have "... \<longleftrightarrow> cmod x < 1 \<and> cmod (x - z'/k) = ?cz * sqrt (R * (R + 1)) / k"
      using `R > 0` *[of z'] ** `k > 1` `z \<in> unit_disc` z'
      using real_sqrt_unique[of ?a ?b, symmetric]
      by (auto simp add: real_sqrt_divide real_sqrt_mult power_divide power_mult_distrib)
    finally
    have "of_complex x \<in> poincare_circle z r \<longleftrightarrow> cmod x < 1 \<and> x \<in> circle (z'/k) (?cz * sqrt(R * (R+1)) / k)"
      unfolding circle_def z' k R
      by simp
    hence "of_complex x \<in> poincare_circle z r \<longleftrightarrow> (let (Ze, Re) = poincare_circle_euclidean z r in cmod x < 1 \<and> x \<in> circle Ze Re)"
      unfolding poincare_circle_euclidean_def Let_def circle_def
      using z' R k
      by (simp add: field_simps)
    hence "of_complex x \<in> poincare_circle z r \<longleftrightarrow> (let (Ze, Re) = poincare_circle_euclidean z r in x \<in> circle Ze Re)"
      using poincare_circle_in_disc[OF `r > 0` `z \<in> unit_disc`]
      by auto
  } note * = this
  show ?thesis
    unfolding Let_def
  proof safe
    fix Ze Re x
    assume "poincare_circle_euclidean z r = (Ze, Re)" "x \<in> circle Ze Re"
    thus "of_complex x \<in> poincare_circle z r"
      using *[of x]
      by simp
  next
    fix Ze Re x
    assume **: "poincare_circle_euclidean z r = (Ze, Re)" "x \<in> poincare_circle z r"
    then obtain x' where x': "x = of_complex x'"
      unfolding poincare_circle_def
      using inf_or_of_complex[of x]
      by auto
    hence "x' \<in> circle Ze Re"
      using *[of x'] **
      by simp
    thus "x \<in> of_complex ` circle Ze Re"
      using x'
      by auto
  qed
qed

lemma intersect_poincare_circles_x_axis:
  assumes z: "is_real z1" "is_real z2" "r1 > 0" "r2 > 0" "-1 < Re z1" "Re z1 < 1" "-1 < Re z2" "Re z2 < 1" "z1 \<noteq> z2"
  assumes x1: "x1 \<in> poincare_circle (of_complex z1) r1 \<inter> poincare_circle (of_complex z2) r2" and
          x2: "x2 \<in> poincare_circle (of_complex z1) r1 \<inter> poincare_circle (of_complex z2) r2"
  shows "x1 = x2 \<or> x1 = conjugate x2"
proof-
  have in_disc: "of_complex z1 \<in> unit_disc" "of_complex z2 \<in> unit_disc"
    using assms
    by (auto simp add: cmod_eq_Re)

  obtain x1' x2' where x': "x1 = of_complex x1'" "x2 = of_complex x2'"
    using x1 x2
    using inf_or_of_complex[of x1] inf_or_of_complex[of x2]
    unfolding poincare_circle_def
    by auto

  obtain Ze1 Re1 where 1: "(Ze1, Re1) = poincare_circle_euclidean (of_complex z1) r1"
    by (metis poincare_circle_euclidean_def)
  obtain Ze2 Re2 where 2: "(Ze2, Re2) = poincare_circle_euclidean (of_complex z2) r2"
    by (metis poincare_circle_euclidean_def)
  have circle: "x1' \<in> circle Ze1 Re1 \<inter> circle Ze2 Re2"  "x2' \<in> circle Ze1 Re1 \<inter> circle Ze2 Re2"
    using poincare_circle_is_euclidean_circle[of "of_complex z1" r1]
    using poincare_circle_is_euclidean_circle[of "of_complex z2" r2]
    using assms 1 2 `of_complex z1 \<in> unit_disc` `of_complex z2 \<in> unit_disc` x'
    by auto (metis image_iff of_complex_inj)+

  have "is_real Ze1" "is_real Ze2"
    using 1 2 `is_real z1` `is_real z2`
    by (simp_all add: poincare_circle_euclidean_def Let_def)

  have "Re1 > 0" "Re2 > 0"
    using 1 2 in_disc `r1 > 0` `r2 > 0`
    using poincare_circle_in_disc(2)[of r1 "of_complex z1" Ze1 Re1]
    using poincare_circle_in_disc(2)[of r2 "of_complex z2" Ze2 Re2]
    by auto

  have "Ze1 \<noteq> Ze2"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence eq: "Ze1 = Ze2" "Re1 = Re2"
      using circle(1)
      unfolding circle_def
      by auto

    let ?A = "Ze1 - Re1" and ?B = "Ze1 + Re1"
    have "?A \<in> circle Ze1 Re1" "?B \<in> circle Ze1 Re1"
      using `Re1 > 0`
      unfolding circle_def
      by simp_all
    hence "of_complex ?A \<in> poincare_circle (of_complex z1) r1" "of_complex ?B \<in> poincare_circle (of_complex z1) r1"
          "of_complex ?A \<in> poincare_circle (of_complex z2) r2" "of_complex ?B \<in> poincare_circle (of_complex z2) r2"
      using eq
      using poincare_circle_is_euclidean_circle[OF `of_complex z1 \<in> unit_disc` `r1 > 0`]
      using poincare_circle_is_euclidean_circle[OF `of_complex z2 \<in> unit_disc` `r2 > 0`]
      using 1 2
      by auto blast+
    hence "poincare_distance (of_complex z1) (of_complex ?A) = poincare_distance (of_complex z1) (of_complex ?B)"
          "poincare_distance (of_complex z2) (of_complex ?A) = poincare_distance (of_complex z2) (of_complex ?B)"
          "-1 < Re (Ze1 - Re1)" "Re (Ze1 - Re1) < 1" "-1 < Re (Ze1 + Re1)" "Re (Ze1 + Re1) < 1"
      using `is_real Ze1` `is_real Ze2`
      unfolding poincare_circle_def
      by (auto simp add: cmod_eq_Re)
    hence "z1 = z2"
      using unique_midpoint_x_axis[of "Ze1 - Re1" "Ze1 + Re1"]
      using `is_real Ze1` `is_real z1` `is_real z2` `Re1 > 0` `-1 < Re z1` `Re z1 < 1` `-1 < Re z2` `Re z2 < 1`
      by auto
    thus False
      using `z1 \<noteq> z2`
      by simp
  qed

  hence *: "(Re x1')\<^sup>2 + (Im x1')\<^sup>2 - 2 * Re x1' * Ze1 + Ze1 * Ze1 - cor (Re1 * Re1) = 0"
           "(Re x1')\<^sup>2 + (Im x1')\<^sup>2 - 2 * Re x1' * Ze2 + Ze2 * Ze2 - cor (Re2 * Re2) = 0"
           "(Re x2')\<^sup>2 + (Im x2')\<^sup>2 - 2 * Re x2' * Ze1 + Ze1 * Ze1 - cor (Re1 * Re1) = 0"
           "(Re x2')\<^sup>2 + (Im x2')\<^sup>2 - 2 * Re x2' * Ze2 + Ze2 * Ze2 - cor (Re2 * Re2) = 0"
    using circle_equation[of Re1 Ze1] circle_equation[of Re2 Ze2] circle
    using eq_cnj_iff_real[of Ze1] `is_real Ze1` `Re1 > 0`
    using eq_cnj_iff_real[of Ze2] `is_real Ze2` `Re2 > 0`
    using complex_add_cnj[of x1']  complex_add_cnj[of x2']
    by (auto simp add: complex_mult_cnj) (smt comm_semiring_class.distrib complex_of_real_Re diff_diff_add power2_eq_square)+

  hence "- 2 * Re x1' * Ze1 + Ze1 * Ze1 - cor (Re1 * Re1) = - 2 * Re x1' * Ze2 + Ze2 * Ze2 - cor (Re2 * Re2)"
        "- 2 * Re x2' * Ze1 + Ze1 * Ze1 - cor (Re1 * Re1) = - 2 * Re x2' * Ze2 + Ze2 * Ze2 - cor (Re2 * Re2)"
    by (smt add_diff_cancel_right' add_diff_eq eq_iff_diff_eq_0 minus_diff_eq mult_minus_left of_real_minus)+
  hence "2 * Re x1' * (Ze2 - Ze1) =  (Ze2 * Ze2 - cor (Re2 * Re2)) - (Ze1 * Ze1 - cor (Re1 * Re1))"
        "2 * Re x2' * (Ze2 - Ze1) =  (Ze2 * Ze2 - cor (Re2 * Re2)) - (Ze1 * Ze1 - cor (Re1 * Re1))"
    by simp_all (simp add: field_simps)+
  hence "2 * Re x1' * (Ze2 - Ze1) = 2 * Re x2' * (Ze2 - Ze1)"
    by simp
  hence "Re x1' = Re x2'"
    using `Ze1 \<noteq> Ze2`
    by simp
  moreover
  hence "(Im x1')\<^sup>2 = (Im x2')\<^sup>2"
    using *(1) *(3)
    by simp (metis Re_complex_of_real Re_power_real add_diff_cancel_left' diff_add_cancel is_real_complex_of_real)
  hence "Im x1' = Im x2' \<or> Im x1' = -Im x2'"
    using power2_eq_iff
    by blast
  ultimately
  show ?thesis
    using x'
    using complex.expand
    by auto (metis cnj.code complex_surj)
qed

lemma intersect_poincare_circles_conjugate_centers:
  assumes in_disc: "z1 \<in> unit_disc" "z2 \<in> unit_disc" and "z1 \<noteq> z2" "z1 = conjugate z2" "r > 0" and
          u: "u \<in> poincare_circle z1 r \<inter> poincare_circle z2 r"
  shows "is_real (to_complex u)"
proof-
  obtain z1e r1e z2e r2e where
   euclidean: "(z1e, r1e) = poincare_circle_euclidean z1 r"
              "(z2e, r2e) = poincare_circle_euclidean z2 r"
    by (metis poincare_circle_euclidean_def)
  obtain z1' z2' where z': "z1 = of_complex z1'" "z2 = of_complex z2'"
    using inf_or_of_complex[of z1] inf_or_of_complex[of z2] in_disc
    by auto
  obtain u' where u': "u = of_complex u'"
    using u inf_or_of_complex[of u]
    by (auto simp add: poincare_circle_def)
  have "z1' = cnj z2'"
    using `z1 = conjugate z2` z'
    by (auto simp add: of_complex_inj)
  moreover
  let ?cz = "1 - (cmod z2')\<^sup>2"
  let ?den = "?cz * (cosh r - 1) / 2 + 1"
  have "?cz > 0"
    using in_disc z'
    by (simp add: cmod_def)
  hence "?den \<ge> 1"
    using cosh_gt_1[OF `r > 0`]
    by auto
  hence "?den \<noteq> 0"
    by simp
  hence "cor ?den \<noteq> 0"
    using of_real_eq_0_iff
    by blast
  ultimately
  have "r1e = r2e" "z1e = cnj z2e" "z1e \<noteq> z2e"
    using z' euclidean `z1 \<noteq> z2`
    unfolding poincare_circle_euclidean_def Let_def
    by simp_all metis

  hence "u' \<in> circle (cnj z2e) r2e \<inter> circle z2e r2e" "z2e \<noteq> cnj z2e"
    using euclidean u u'
    using poincare_circle_is_euclidean_circle[of z1 r]
    using poincare_circle_is_euclidean_circle[of z2 r]
    using in_disc `r > 0`
    by auto (metis image_iff of_complex_inj)+
  hence "(cmod (u' - z2e))\<^sup>2 = (cmod(u' - cnj z2e))\<^sup>2"
    by (simp add: circle_def)
  hence "(u' - z2e) * (cnj u' - cnj z2e) = (u' - cnj z2e) * (cnj u' - z2e)"
    by (metis complex_cnj_cnj complex_cnj_diff complex_norm_square)
  hence "(z2e - cnj z2e) * (u' - cnj u') = 0"
    by (simp add: field_simps)
  thus ?thesis
    using u' `z2e \<noteq> cnj z2e` eq_cnj_iff_real[of u']
    by simp
qed

lemma unit_disc_fix_f_congruent_triangles:
  assumes
    in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc" and
    in_disc': "u' \<in> unit_disc" "v' \<in> unit_disc" "w' \<in> unit_disc" and d:
    "poincare_distance u v = poincare_distance u' v'"
    "poincare_distance v w = poincare_distance v' w'"
    "poincare_distance u w = poincare_distance u' w'"
  shows
    "\<exists> M. unit_disc_fix_f M \<and> M u = u' \<and> M v = v' \<and> M w = w'"
proof (cases "u = v \<or> u = w \<or> v = w")
  case True
  thus ?thesis
    using assms
    using poincare_distance_eq_0_iff[of u' v']
    using poincare_distance_eq_0_iff[of v' w']
    using poincare_distance_eq_0_iff[of u' w']
    using poincare_distance_eq_ex_moebius[of v w v' w']
    using poincare_distance_eq_ex_moebius[of u w u' w']
    using poincare_distance_eq_ex_moebius[of u v u' v']
    by (metis unit_disc_fix_f_def)
next
  case False

  have "\<forall> w u' v' w'. w \<in> unit_disc \<and> u' \<in> unit_disc \<and> v' \<in> unit_disc \<and> w' \<in> unit_disc \<and> w \<noteq> u \<and> w \<noteq> v \<and>
    poincare_distance u v = poincare_distance u' v' \<and>
    poincare_distance v w = poincare_distance v' w' \<and>
    poincare_distance u w = poincare_distance u' w' \<longrightarrow>
    (\<exists> M. unit_disc_fix_f M \<and> M u = u' \<and> M v = v' \<and> M w = w')" (is "?P u v")
  proof (rule wlog_positive_x_axis[where P="?P"])
    show "v \<in> unit_disc" "u \<in> unit_disc"
      by fact+
  next
    show "u \<noteq> v"
      using False
      by simp
  next
    fix x
    assume x: "is_real x" "0 < Re x" "Re x < 1"

    hence "of_complex x \<noteq> 0\<^sub>h"
      using of_complex_zero_iff[of x]
      by (auto simp add: complex.expand)

    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix w u' v' w'
      assume in_disc: "w \<in> unit_disc" "u' \<in> unit_disc" "v' \<in> unit_disc" "w' \<in> unit_disc"
      assume "poincare_distance 0\<^sub>h (of_complex x) = poincare_distance u' v'"
      then obtain M' where M': "unit_disc_fix M'" "moebius_pt M' u' = 0\<^sub>h" "moebius_pt M' v' = (of_complex x)"
        using poincare_distance_eq_ex_moebius[of u' v' "0\<^sub>h" "of_complex x"] in_disc x
        by (auto simp add: cmod_eq_Re)

      let ?w = "moebius_pt M' w'"
      have "?w \<in> unit_disc"
        using `unit_disc_fix M'` `w' \<in> unit_disc`
        by simp

      assume "w \<noteq> 0\<^sub>h" "w \<noteq> of_complex x"
      hence dist_gt_0: "poincare_distance 0\<^sub>h w > 0" "poincare_distance (of_complex x) w > 0"
        using poincare_distance_eq_0_iff[of "0\<^sub>h" w] in_disc poincare_distance_ge0[of "0\<^sub>h" w]
        using poincare_distance_eq_0_iff[of "of_complex x" w] in_disc poincare_distance_ge0[of "of_complex x" w]
        using x
        by (simp_all add: cmod_eq_Re)

      assume "poincare_distance (of_complex x) w = poincare_distance v' w'"
             "poincare_distance 0\<^sub>h w = poincare_distance u' w'"
      hence "poincare_distance 0\<^sub>h ?w = poincare_distance 0\<^sub>h w"
            "poincare_distance (of_complex x) ?w = poincare_distance (of_complex x) w"
        using M'(1) M'(2)[symmetric] M'(3)[symmetric] in_disc
        using unit_disc_fix_preserve_poincare_distance[of M' u' w']
        using unit_disc_fix_preserve_poincare_distance[of M' v' w']
        by simp_all
      hence "?w \<in> poincare_circle 0\<^sub>h (poincare_distance 0\<^sub>h w) \<inter> poincare_circle (of_complex x) (poincare_distance (of_complex x) w)"
            "w \<in> poincare_circle 0\<^sub>h (poincare_distance 0\<^sub>h w) \<inter> poincare_circle (of_complex x) (poincare_distance (of_complex x) w)"
        using `?w \<in> unit_disc` `w \<in> unit_disc`
        unfolding poincare_circle_def
        by simp_all
      hence "?w = w \<or> ?w = conjugate w"
        using intersect_poincare_circles_x_axis[of 0 x "poincare_distance 0\<^sub>h w" "poincare_distance (of_complex x) w" ?w w] x
        using `of_complex x \<noteq> 0\<^sub>h` dist_gt_0
        using poincare_distance_eq_0_iff
        by simp
      thus "\<exists>M. unit_disc_fix_f M \<and> M 0\<^sub>h = u' \<and> M (of_complex x) = v' \<and> M w = w'"
      proof
        assume "moebius_pt M' w' = w"
        thus ?thesis
          using M'
          using moebius_pt_invert[of M' u' "0\<^sub>h"]
          using moebius_pt_invert[of M' v' "of_complex x"]
          using moebius_pt_invert[of M' w' "w"]
          apply (rule_tac x="moebius_pt (-M')" in exI)
          apply (simp add: unit_disc_fix_f_def)
          apply (rule_tac x="-M'" in exI, simp)
          done
      next
        let ?M = "moebius_pt (-M') \<circ> conjugate"
        assume "moebius_pt M' w' = conjugate w"
        hence "?M w = w'"
          using moebius_pt_invert[of  M' w' "conjugate w"]
          by simp
        moreover
        have "?M 0\<^sub>h = u'" "?M (of_complex x) = v'"
          using moebius_pt_invert[of M' u' "0\<^sub>h"]
          using moebius_pt_invert[of M' v' "of_complex x"]
          using M' `is_real x` eq_cnj_iff_real[of x]
          by simp_all
        moreover
        have "unit_disc_fix_f ?M"
          using `unit_disc_fix M'`
          unfolding unit_disc_fix_f_def
          by (rule_tac x="-M'" in exI, simp)
        ultimately
        show ?thesis
          by blast
      qed
    qed
  next
    fix M u v
    assume 1: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc"
    let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v"
    assume 2: "?P ?Mu ?Mv"
    show "?P u v"
    proof safe
      fix w u' v' w'
      let ?Mw = "moebius_pt M w" and ?Mu' = "moebius_pt M u'" and ?Mv' = "moebius_pt M v'" and ?Mw' = "moebius_pt M w'"
      assume "w \<in> unit_disc" "u' \<in> unit_disc" "v' \<in> unit_disc" "w' \<in> unit_disc" "w \<noteq> u" "w \<noteq> v"
             "poincare_distance u v = poincare_distance u' v'"
             "poincare_distance v w = poincare_distance v' w'"
             "poincare_distance u w = poincare_distance u' w'"
      then obtain M' where M': "unit_disc_fix_f M'" "M' ?Mu = ?Mu'" "M' ?Mv = ?Mv'" "M' ?Mw = ?Mw'"
        using 1 2[rule_format, of ?Mw ?Mu' ?Mv' ?Mw']
        by auto

      let ?M = "moebius_pt (-M) \<circ> M' \<circ> moebius_pt M"
      show "\<exists>M. unit_disc_fix_f M \<and> M u = u' \<and> M v = v' \<and> M w = w'"
      proof (rule_tac x="?M" in exI, safe)
        show "unit_disc_fix_f ?M"
          using M'(1) `unit_disc_fix M`
          by (subst unit_disc_fix_f_comp, subst unit_disc_fix_f_comp, simp_all)
      next
        show "?M u = u'" "?M v = v'" "?M w = w'"
          using M'
          by auto
      qed
    qed
  qed
  thus ?thesis
    using assms False
    by auto
qed


(* ------------------------------------------------------------------ *)
subsection{* Poincare between *}
(* ------------------------------------------------------------------ *)

definition poincare_between :: "complex_homo \<Rightarrow> complex_homo \<Rightarrow> complex_homo \<Rightarrow> bool" where
  "poincare_between u v w \<longleftrightarrow>
         u = v \<or> v = w \<or>
         (let cr = cross_ratio u v w (inversion v)
           in is_real (to_complex cr) \<and> Re (to_complex cr) < 0)"

lemma poincare_between_nonstrict [simp]:
  shows "poincare_between u u v" "poincare_between u v v"
  by (simp_all add: poincare_between_def)                       

lemma poincare_between_sandwich:
  assumes "u \<in> unit_disc" "v \<in> unit_disc"
  assumes "poincare_between u v u"
  shows "u = v"
proof (rule ccontr)
  assume "\<not> ?thesis"
  thus False
    using assms
    using inversion_noteq_unit_disc[of v u]
    using cross_ratio_1[of v u "inversion v"]
    unfolding poincare_between_def Let_def
    by auto
qed

lemma poincare_between_rev:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  shows "poincare_between u v w \<longleftrightarrow> poincare_between w v u"       
  using assms 
  using inversion_noteq_unit_disc[of v w]
  using inversion_noteq_unit_disc[of v u]
  using cross_ratio_commute_13[of u v w "inversion v"]
  using cross_ratio_not_inf[of w "inversion v" v u]
  using cross_ratio_not_zero[of w v u "inversion v"]
  using inf_or_of_complex[of "cross_ratio w v u (inversion v)"]
  unfolding poincare_between_def
  by (auto simp add: Let_def Im_complex_div_eq_0 Re_divide divide_less_0_iff)

lemma poincare_between_poincare_colinear [simp]:       
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  assumes betw: "poincare_between u v w"
  shows "poincare_colinear {u, v, w}"
proof (cases "u = v \<or> v = w")
  case True
  thus ?thesis
    using assms
    by auto
next
  case False
  hence distinct: "distinct [u, v, w, inversion v]"
    using in_disc inversion_noteq_unit_disc[of v v] inversion_noteq_unit_disc[of v u] inversion_noteq_unit_disc[of v w]
    using betw poincare_between_sandwich[of w v]
    by (auto simp add: poincare_between_def Let_def)

  then obtain H where *: "{u, v, w, inversion v} \<subseteq> circline_set H"
    using assms
    unfolding poincare_between_def
    using four_points_on_circline_iff_cross_ratio_real[of u v w "inversion v"]
    by auto
  hence "H = poincare_line u v"
    using assms distinct
    using unique_circline_set[of u v "inversion v"]
    using poincare_line[of u v] poincare_line_inversion[of u v]
    unfolding circline_set_def
    by auto
  thus ?thesis
    using * assms False
    unfolding poincare_colinear_def
    by (rule_tac x="poincare_line u v" in exI) simp
qed

lemma poincare_between_poincare_line_uvz:
  assumes "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc" "z \<in> unit_disc" "poincare_between u v z"
  shows "z \<in> circline_set (poincare_line u v)"
  using assms
  using poincare_between_poincare_colinear[of u v z]
  using unique_poincare_line[OF assms(1-3)]
  unfolding poincare_colinear_def
  by auto

lemma poincare_between_poincare_line_uzv:
  assumes "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc" "z \<in> unit_disc" "poincare_between u z v"
  shows "z \<in> circline_set (poincare_line u v)"
  using assms
  using poincare_between_poincare_colinear[of u z v]
  using unique_poincare_line[OF assms(1-3)]
  unfolding poincare_colinear_def
  by auto

lemma unit_disc_fix_moebius_preserve_poincare_between [simp]:
  assumes "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  shows "poincare_between (moebius_pt M u) (moebius_pt M v) (moebius_pt M w) \<longleftrightarrow>
         poincare_between u v w"
proof (cases "u = v \<or> v = w")
  case True
  thus ?thesis
    using assms
    unfolding poincare_between_def
    by auto
next
  case False
  moreover
  hence "moebius_pt M u \<noteq> moebius_pt M v \<and> moebius_pt M v \<noteq> moebius_pt M w"
    by auto
  moreover
  have "v \<noteq> inversion v" "w \<noteq> inversion v"
    using inversion_noteq_unit_disc[of v w]
    using inversion_noteq_unit_disc[of v v]
    using `v \<in> unit_disc` `w \<in> unit_disc`
    by auto
  ultimately
  show ?thesis
    using assms
    using unit_circle_fix_moebius_pt_inversion[of M v, symmetric]
    unfolding poincare_between_def
    by (simp del: unit_circle_fix_moebius_pt_inversion)
qed

text{* Betweeness on euclidean lines *}

lemma poincare_between_x_axis_u0v:
  assumes "is_real u'" "u' \<noteq> 0" "v' \<noteq> 0"
  shows "poincare_between (of_complex u') 0\<^sub>h (of_complex v') \<longleftrightarrow> is_real v' \<and> Re u' * Re v' < 0"
proof-
  have "Re u' \<noteq> 0"
    using `is_real u'` `u' \<noteq> 0`
    using complex_eq_if_Re_eq
    by auto
  have nz: "of_complex u' \<noteq> 0\<^sub>h" "of_complex v' \<noteq> 0\<^sub>h"
    by (simp_all add: `u' \<noteq> 0` `v' \<noteq> 0`)
  hence "0\<^sub>h \<noteq> of_complex v'"
    by metis

  let ?cr = "cross_ratio (of_complex u') 0\<^sub>h (of_complex v') \<infinity>\<^sub>h"
  have "is_real (to_complex ?cr) \<and> Re (to_complex ?cr) < 0 \<longleftrightarrow> is_real v' \<and> Re u' * Re v' < 0"
    using cross_ratio_0inf[of v' u'] `v' \<noteq> 0` `u' \<noteq> 0` `is_real u'`
    by simp (metis Re_complex_div_lt_0 Re_mult_real complex_cnj_divide divide_cancel_left eq_cnj_iff_real)
  thus ?thesis
    unfolding poincare_between_def inversion_zero
    using `of_complex u' \<noteq> 0\<^sub>h` `0\<^sub>h \<noteq> of_complex v'`
    by simp
qed

lemma poincare_between_u0v:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> 0\<^sub>h" "v \<noteq> 0\<^sub>h"
  shows "poincare_between u 0\<^sub>h v \<longleftrightarrow> (\<exists> k < 0. to_complex u = cor k * to_complex v)" (is "?P u v")
proof (cases "u = v")
  case True
  thus ?thesis
    using assms
    using inf_or_of_complex[of v]
    using poincare_between_sandwich[of u "0\<^sub>h"]      
    by auto
next                                                 
  case False
  have "\<forall> u. u \<in> unit_disc \<and> u \<noteq> 0\<^sub>h \<longrightarrow> ?P u v" (is "?P' v")
  proof (rule wlog_rotation_to_positive_x_axis)
    fix \<phi> v
    let ?M = "moebius_pt (moebius_rotation \<phi>)"
    assume 1: "v \<in> unit_disc" "v \<noteq> 0\<^sub>h"
    assume 2: "?P' (?M v)"
    show "?P' v"
    proof (rule allI, rule impI, (erule conjE)+)
      fix u
      assume 3: "u \<in> unit_disc" "u \<noteq> 0\<^sub>h"  
      have "poincare_between (?M u) 0\<^sub>h (?M v) \<longleftrightarrow> poincare_between u 0\<^sub>h v"
        using `u \<in> unit_disc` `v \<in> unit_disc`
        using unit_disc_fix_moebius_preserve_poincare_between unit_disc_fix_rotation zero_in_unit_disc 
        by fastforce
      thus "?P u v"
        using 1 2[rule_format, of "?M u"] 3
        using inf_or_of_complex[of u] inf_or_of_complex[of v]
        by auto
    qed
  next
    fix x
    assume 1: "is_real x" "0 < Re x" "Re x < 1"
    hence "x \<noteq> 0"
      by auto
    show "?P' (of_complex x)"    
    proof (rule allI, rule impI, (erule conjE)+)
      fix u
      assume 2: "u \<in> unit_disc" "u \<noteq> 0\<^sub>h"
      then obtain u' where "u = of_complex u'"
        using inf_or_of_complex[of u]
        by auto
      show "?P u (of_complex x)"
        using 1 2 `x \<noteq> 0` `u = of_complex u'`
        using poincare_between_rev[of u "0\<^sub>h" "of_complex x"]
        using poincare_between_x_axis_u0v[of x u'] `is_real x`
        apply (auto simp add: cmod_eq_Re)
        apply (rule_tac x="Re u' / Re x" in exI, simp add: complex_of_real_Re divide_neg_pos sign_simps)
        using mult_neg_pos mult_pos_neg
        by blast
    qed
  qed fact+
  thus ?thesis
    using assms
    by auto
qed

lemma poincare_between_u0v_polar_form:
  assumes "x \<in> unit_disc" "y \<in> unit_disc" "x \<noteq> 0\<^sub>h" "y \<noteq> 0\<^sub>h" "to_complex x = cor rx * cis \<phi>" "to_complex y = cor ry * cis \<phi>"
  shows "poincare_between x 0\<^sub>h y \<longleftrightarrow> rx * ry < 0" (is "?P x y rx ry")
proof-
  from assms have "rx \<noteq> 0" "ry \<noteq> 0"
    using inf_or_of_complex[of x] inf_or_of_complex[of y]
    by auto

  have "(\<exists>k<0. cor rx = cor k * cor ry ) = (rx * ry < 0)"
  proof
    assume "\<exists>k<0. cor rx = cor k * cor ry"
    then obtain k where "k < 0" "cor rx = cor k * cor ry"
      by auto
    hence "rx = k * ry"
      using of_real_eq_iff
      by fastforce
    thus "rx * ry < 0" 
      using `k < 0` `rx \<noteq> 0` `ry \<noteq> 0`
      by (smt divisors_zero mult_nonneg_nonpos mult_nonpos_nonpos zero_less_mult_pos2)
  next
    assume "rx * ry < 0"
    hence "rx = (rx/ry)*ry" "rx / ry < 0"
      using `rx \<noteq> 0` `ry \<noteq> 0`
      by (auto simp add: divide_less_0_iff sign_simps)
    thus "\<exists>k<0. cor rx = cor k * cor ry"
      using `rx \<noteq> 0` `ry \<noteq> 0`
      by (rule_tac x="rx / ry" in exI, simp)
  qed
  thus ?thesis
    using assms                                 
    using poincare_between_u0v[OF assms(1-4)]
    by auto
qed


(* unit disc assumption could be relaxed *)
(* one is_real assumption can be moved to the conclusion *)
lemma poincare_between_x_axis_0uv:
  fixes x y :: real
  assumes "-1 < x" "x < 1" "x \<noteq> 0"
  assumes "-1 < y" "y < 1" "y \<noteq> 0"
  shows "poincare_between 0\<^sub>h (of_complex x) (of_complex y) \<longleftrightarrow>
        (x < 0 \<and> y < 0 \<and> y \<le> x) \<or> (x > 0 \<and> y > 0 \<and> x \<le> y)" (is "?lhs \<longleftrightarrow> ?rhs")
proof (cases "x = y")
  case True
  thus ?thesis
    using assms
    unfolding poincare_between_def
    by auto
next
  case False
  let ?x = "of_complex x" and ?y = "of_complex y"

  have "?x \<in> unit_disc" "?y \<in> unit_disc"
    using assms
    by auto

  have distinct: "distinct [0\<^sub>h, ?x, ?y, inversion ?x]"
    using `x \<noteq> 0` `y \<noteq> 0` `x \<noteq> y` `?x \<in> unit_disc` `?y \<in> unit_disc`
    using inversion_noteq_unit_disc[of ?x ?y]
    using inversion_noteq_unit_disc[of ?x ?x]
    using inversion_noteq_unit_disc[of ?x "0\<^sub>h"]
    using of_complex_inj[of x y]
    apply auto
    apply (metis of_complex_zero_iff of_real_eq_0_iff)
    apply (metis of_complex_zero_iff of_real_eq_0_iff)
    apply (metis divide_eq_0_iff of_complex_zero_iff zero_neq_one)
    done

  let ?cr = "cross_ratio 0\<^sub>h ?x ?y (inversion ?x)"
  have "Re (to_complex ?cr) = x\<^sup>2 * (x*y - 1) / (x * (y - x))"
    using `x \<noteq> 0` `x \<noteq> y`
    unfolding inversion_def
    by simp (transfer, transfer, auto simp add: vec_cnj_def power2_eq_square field_simps split: if_split_asm)
  moreover
  have "x\<^sup>2 * (x*y - 1) < 0"
    using assms
    by (smt minus_mult_minus mult_le_cancel_left1 mult_pow2_lt0)
  moreover
  have "x * (y - x) > 0 \<longleftrightarrow> ?rhs"
    using `x \<noteq> 0` `y \<noteq> 0` `x \<noteq> y`
    by (smt mult_le_0_iff)
  ultimately
  have *: "Re (to_complex ?cr) < 0 \<longleftrightarrow> ?rhs"
    by (simp add: divide_less_0_iff)

  show ?thesis
  proof
    assume ?lhs
    have "is_real (to_complex ?cr)" "Re (to_complex ?cr) < 0"
      using `?lhs` distinct
      unfolding poincare_between_def Let_def
      by auto
    thus ?rhs
      using *
      by simp
  next
    assume ?rhs
    hence "Re (to_complex ?cr) < 0"
      using *
      by simp
    moreover
    have "{0\<^sub>h, of_complex (cor x), of_complex (cor y), inversion (of_complex (cor x))} \<subseteq> circline_set x_axis"
      using `x \<noteq> 0` is_real_inversion[of "cor x"]
      using inf_or_of_complex[of "inversion ?x"]
      by (auto simp del: inversion_of_complex)
    hence "is_real (to_complex ?cr)"
      using four_points_on_circline_iff_cross_ratio_real[OF distinct]
      by auto
    ultimately
    show ?lhs
      using distinct
      unfolding poincare_between_def Let_def
      by auto
  qed
qed

lemma poincare_between_0uv:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> 0\<^sub>h" "v \<noteq> 0\<^sub>h"
  shows "poincare_between 0\<^sub>h u v \<longleftrightarrow> (let u' = to_complex u; v' = to_complex v in arg u' = arg v' \<and> cmod u' \<le> cmod v')" (is "?P u v")
proof (cases "u = v")
  case True
  thus ?thesis
    by simp
next
  case False
  have "\<forall> v. v \<in> unit_disc \<and> v \<noteq> 0\<^sub>h \<and> v \<noteq> u \<longrightarrow> (poincare_between 0\<^sub>h u v \<longleftrightarrow> (let u' = to_complex u; v' = to_complex v in arg u' = arg v' \<and> cmod u' \<le> cmod v'))" (is "?P' u")
  proof (rule wlog_rotation_to_positive_x_axis)
    show "u \<in> unit_disc" "u \<noteq> 0\<^sub>h"
      by fact+
  next
    fix x
    assume *: "is_real x" "0 < Re x" "Re x < 1"
    hence "of_complex x \<in> unit_disc" "of_complex x \<noteq> 0\<^sub>h" "of_complex x \<in> circline_set x_axis"
      unfolding circline_set_x_axis
      by (auto simp add: cmod_eq_Re)
    show "?P' (of_complex x)"
    proof safe
      fix v
      assume "v \<in> unit_disc" "v \<noteq> 0\<^sub>h" "v \<noteq> of_complex x" "poincare_between 0\<^sub>h (of_complex x) v"
      hence "v \<in> circline_set x_axis"
        using poincare_between_poincare_line_uvz[of "0\<^sub>h" "of_complex x" v]
        using poincare_line_0_real_is_x_axis[of "of_complex x"]
        using `of_complex x \<noteq> 0\<^sub>h` `v \<noteq> 0\<^sub>h` `v \<noteq> of_complex x` `of_complex x \<in> unit_disc` `of_complex x \<in> circline_set x_axis`
        by auto
      then obtain v' where **: "v = of_complex v'" "-1 < Re v'" "Re v' < 1" "Re v' \<noteq> 0" "is_real v'"
        using `v \<in> unit_disc` `v \<noteq> 0\<^sub>h`
        using inf_or_of_complex[of v]
        unfolding circline_set_x_axis
        by auto (smt cmod_eq_Re complex_surj of_complex_inj of_complex_zero_iff zero_complex.code)
      show "let u' = to_complex (of_complex x); v' = to_complex v in arg u' = arg v' \<and> cmod u' \<le> cmod v'"
        using poincare_between_x_axis_0uv[of "Re x" "Re v'"] * **
        using `poincare_between 0\<^sub>h (of_complex x) v`
        using arg_complex_of_real_positive[of "Re x"] arg_complex_of_real_negative[of "Re x"]
        using arg_complex_of_real_positive[of "Re v'"] arg_complex_of_real_negative[of "Re v'"]
        by (auto simp add: complex_of_real_Re cmod_eq_Re)
    next
      fix v
      assume "v \<in> unit_disc" "v \<noteq> 0\<^sub>h" "v \<noteq> of_complex x"
      then obtain v' where **: "v = of_complex v'" "v' \<noteq> 0" "v' \<noteq> x"
        using inf_or_of_complex[of v]
        by auto blast
      assume "let u' = to_complex (of_complex x); v' = to_complex v in arg u' = arg v' \<and> cmod u' \<le> cmod v'"
      hence ***: "Re x < 0 \<and> Re v' < 0 \<and> Re v' \<le> Re x \<or> 0 < Re x \<and> 0 < Re v' \<and> Re x \<le> Re v'" "is_real v'"
        using arg_pi_iff[of x] arg_pi_iff[of v']
        using arg_0_iff[of x] arg_0_iff[of v']
        using * **
        by (smt cmod_Re_le_iff to_complex_of_complex)+
      have "-1 < Re v'" "Re v' < 1" "Re v' \<noteq> 0" "is_real v'"
        using `v \<in> unit_disc` ** `is_real v'`
        by (auto simp add: cmod_eq_Re complex_eq_if_Re_eq)
      thus "poincare_between 0\<^sub>h (of_complex x) v"
        using poincare_between_x_axis_0uv[of "Re x" "Re v'"] * ** ***
        by (simp add: complex_of_real_Re)
    qed
  next
    fix \<phi> u
    assume "u \<in> unit_disc" "u \<noteq> 0\<^sub>h"
    let ?M = "moebius_rotation \<phi>"
    assume *: "?P' (moebius_pt ?M u)"
    show "?P' u"
    proof (rule allI, rule impI, (erule conjE)+)
      fix v
      assume "v \<in> unit_disc" "v \<noteq> 0\<^sub>h" "v \<noteq> u"
      have "moebius_pt ?M v \<noteq> moebius_pt ?M u"
        using `v \<noteq> u`
        by auto
      obtain u' v' where "v = of_complex v'" "u = of_complex u'" "v' \<noteq> 0" "u' \<noteq> 0"
        using inf_or_of_complex[of u] inf_or_of_complex[of v]
        using `v \<in> unit_disc` `u \<in> unit_disc` `v \<noteq> 0\<^sub>h` `u \<noteq> 0\<^sub>h`
        by auto
      thus "?P u v"
        using *[rule_format, of "moebius_pt ?M v"]
        using `moebius_pt ?M v \<noteq> moebius_pt ?M u`
        using unit_disc_fix_moebius_preserve_poincare_between[of ?M "0\<^sub>h" u v]
        using `v \<in> unit_disc` `u \<in> unit_disc` `v \<noteq> 0\<^sub>h` `u \<noteq> 0\<^sub>h`
        using arg_mult_eq[of "cis \<phi>" u' v']
        by simp (auto simp add: arg_mult)
    qed
  qed
  thus ?thesis
    using assms False
    by auto
qed

(* unit disc assumption could be relaxed *)
(* one is_real assumption can be moved to the conclusion *)
lemma poincare_between_y_axis_0uv:
  fixes x y :: real
  assumes "-1 < x" "x < 1" "x \<noteq> 0"
  assumes "-1 < y" "y < 1" "y \<noteq> 0"
  shows "poincare_between 0\<^sub>h (of_complex (\<i> * x)) (of_complex (\<i> * y)) \<longleftrightarrow>
        (x < 0 \<and> y < 0 \<and> y \<le> x) \<or> (x > 0 \<and> y > 0 \<and> x \<le> y)" (is "?lhs \<longleftrightarrow> ?rhs")
  using assms
  using poincare_between_0uv[of "of_complex (\<i> * x)" "of_complex (\<i> * y)"]
  using arg_pi2_iff[of "\<i> * cor x"] arg_pi2_iff[of "\<i> * cor y"]
  using arg_minus_pi2_iff[of "\<i> * cor x"] arg_minus_pi2_iff[of "\<i> * cor y"]
  apply simp
  apply (cases "x > 0")
  apply (cases "y > 0", simp, simp)
  apply (cases "y > 0")
  apply simp
  using pi_gt_zero apply linarith
  apply simp
  done

lemma poincare_between_x_axis_uvw:
  fixes x y z :: real
  assumes "-1 < x" "x < 1" 
  assumes "-1 < y" "y < 1" "y \<noteq> x"
  assumes "-1 < z" "z < 1" "z \<noteq> x"
  shows "poincare_between (of_complex x) (of_complex y) (of_complex z) \<longleftrightarrow>
        (y < x \<and> z < x \<and> z \<le> y) \<or> (y > x \<and> z > x \<and> y \<le> z)"  (is "?lhs \<longleftrightarrow> ?rhs")
proof (cases "x = 0 \<or> y = 0 \<or> z = 0")
  case True
  thus ?thesis
    apply (cases "x = 0")
    using poincare_between_x_axis_0uv assms
    apply simp
    apply (cases "z = 0")
    using poincare_between_x_axis_0uv assms poincare_between_rev
    apply (smt norm_of_real of_complex_zero of_real_0 poincare_between_nonstrict(2) unit_disc_iff_cmod_lt_1)
  proof (cases "y = 0")
    case False
    assume " x \<noteq> 0" "z \<noteq> 0"
    thus ?thesis
      using `x = 0 \<or> y = 0 \<or> z = 0` False
      by auto
  next 
    case True
    assume "x \<noteq> 0" "z \<noteq> 0"
    hence "poincare_between (of_complex x) 0\<^sub>h (of_complex z) = (is_real z \<and> x * z < 0)"
      using poincare_between_x_axis_u0v 
      by auto
    moreover
    have "x * z < 0 \<longleftrightarrow> ?rhs"
      using True `x \<noteq> 0` `z \<noteq> 0`
      by (smt zero_le_mult_iff)
    ultimately
    show ?thesis
      using True
      by auto
  qed 
next
  case False
  thus ?thesis
  proof (cases "z = y")
    case True
    thus ?thesis
      using assms
      unfolding poincare_between_def
      by auto
  next
    case False
    let ?x = "of_complex x" and ?y = "of_complex y" and ?z = "of_complex z"
  
    have "?x \<in> unit_disc" "?y \<in> unit_disc" "?z \<in> unit_disc"
      using assms
      by auto
  
    have distinct: "distinct [?x, ?y, ?z, inversion ?y]"
      using `y \<noteq> x` `z \<noteq> x` False `?x \<in> unit_disc` `?y \<in> unit_disc` `?z \<in> unit_disc`
      using inversion_noteq_unit_disc[of ?y ?y]
      using inversion_noteq_unit_disc[of ?y ?x]
      using inversion_noteq_unit_disc[of ?y ?z]
      using of_complex_inj[of x y]  of_complex_inj[of y z]  of_complex_inj[of x z]
      by auto

    have "cor y * cor x \<noteq> 1"
      using assms
      by (smt minus_mult_minus mult_less_cancel_left2 mult_less_cancel_right2 of_real_1 of_real_eq_iff of_real_mult)
  
    let ?cr = "cross_ratio ?x ?y ?z (inversion ?y)"
    have "Re (to_complex ?cr) = (x - y) * (z*y - 1)/ ((x*y - 1)*(z - y))"
      using `y \<noteq> x` `z \<noteq> x` False `\<not> (x = 0 \<or> y = 0 \<or> z = 0)`
      using `cor y * cor x \<noteq> 1`
      unfolding inversion_def
      apply simp 
      apply (transfer, transfer, auto simp add: vec_cnj_def power2_eq_square field_simps split: if_split_asm)
      by (metis (no_types, hide_lams) ab_group_add_class.ab_diff_conv_add_uminus distrib_left mult_divide_mult_cancel_left_if mult_minus_right)
    moreover
    have "(x*y - 1) < 0"
      using assms
      by (smt minus_mult_minus mult_less_cancel_right2 zero_less_mult_iff)
    moreover
    have "(z*y - 1) < 0"
      using assms
      by (smt minus_mult_minus mult_less_cancel_right2 zero_less_mult_iff)
    moreover
    have "(x - y) / (z - y) < 0 \<longleftrightarrow> ?rhs"
      using `y \<noteq> x` `z \<noteq> x` False `\<not> (x = 0 \<or> y = 0 \<or> z = 0)`
      by (smt divide_less_cancel divide_nonneg_nonpos divide_nonneg_pos divide_nonpos_nonneg divide_nonpos_nonpos)
    ultimately
    have *: "Re (to_complex ?cr) < 0 \<longleftrightarrow> ?rhs"
      by (smt linordered_field_class.sign_simps(45) minus_divide_left zero_less_divide_iff zero_less_mult_iff)
  
    show ?thesis
    proof
      assume ?lhs
      have "is_real (to_complex ?cr)" "Re (to_complex ?cr) < 0"
        using `?lhs` distinct
        unfolding poincare_between_def Let_def
        by auto
      thus ?rhs
        using *
        by simp
    next
      assume ?rhs
      hence "Re (to_complex ?cr) < 0"
        using *
        by simp
      moreover
      have "{of_complex (cor x), of_complex (cor y), of_complex (cor z), inversion (of_complex (cor y))} \<subseteq> circline_set x_axis"
        using `\<not> (x = 0 \<or> y = 0 \<or> z = 0)` is_real_inversion[of "cor y"]
        using inf_or_of_complex[of "inversion ?y"]
        by (auto simp del: inversion_of_complex)
      hence "is_real (to_complex ?cr)"
        using four_points_on_circline_iff_cross_ratio_real[OF distinct]
        by auto
      ultimately
      show ?lhs
        using distinct
        unfolding poincare_between_def Let_def
        by auto
    qed
  qed
qed

lemma colinear_between:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  assumes "poincare_colinear {u, v, w}"
  shows "poincare_between u v w \<or> poincare_between u w v \<or> poincare_between v u w" (is "?P' u v w")
proof (cases "u=v")
  case True
  thus ?thesis
    using assms
    by auto
next
  case False
  have "\<forall> w. w \<in> unit_disc \<and> poincare_colinear {u, v, w} \<longrightarrow> ?P' u v w" (is "?P u v")
  proof (rule wlog_positive_x_axis[where P="?P"])
    fix x
    assume x: "is_real x" "0 < Re x" "Re x < 1"
    hence "x \<noteq> 0"
      using complex.expand[of x 0]
      by auto
    hence *: "poincare_line 0\<^sub>h (of_complex x) = x_axis"
      using x poincare_line_0_real_is_x_axis[of "of_complex x"]
      unfolding circline_set_x_axis
      by auto
    have "of_complex x \<in> unit_disc"
      using x
      by (auto simp add: cmod_eq_Re)
    have "of_complex x \<noteq> 0\<^sub>h"
      using `x \<noteq> 0`
      by auto
    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix w
      assume "w \<in> unit_disc"
      assume "poincare_colinear {0\<^sub>h, of_complex x, w}"
      hence "w \<in> circline_set x_axis"
        using * unique_poincare_line[of "0\<^sub>h" "of_complex x"] `of_complex x \<in> unit_disc` `x \<noteq> 0` `of_complex x \<noteq> 0\<^sub>h`
        unfolding poincare_colinear_def
        by auto
      then obtain w' where w': "w = of_complex w'" "is_real w'"
        using `w \<in> unit_disc`
        using inf_or_of_complex[of w]
        unfolding circline_set_x_axis
        by auto
      hence "-1 < Re w'" "Re w' < 1"
        using `w \<in> unit_disc`
        by (auto simp add: cmod_eq_Re)
      assume 1: "\<not> poincare_between (of_complex x) 0\<^sub>h w"
      hence "w \<noteq> 0\<^sub>h" "w' \<noteq> 0"
        using w'
        unfolding poincare_between_def
        by auto
      hence "Re w' \<noteq> 0"
        using w' complex.expand[of w' 0]
        by auto

      have "Re w' \<ge> 0"
        using 1 poincare_between_x_axis_u0v[of x w'] `Re x > 0` `is_real x` `x \<noteq> 0` `w' \<noteq> 0` w'
        using mult_pos_neg
        by force

      moreover

      assume "\<not> poincare_between 0\<^sub>h (of_complex x) w"
      hence "Re w' < Re x"
        using poincare_between_x_axis_0uv[of "Re x" "Re w'"]
        using w' x `-1 < Re w'` `Re w' < 1` `Re w' \<noteq> 0`
        by (auto simp add: complex_of_real_Re)

      ultimately
      show "poincare_between 0\<^sub>h w (of_complex x)"
        using poincare_between_x_axis_0uv[of "Re w'" "Re x"]
        using w' x `-1 < Re w'` `Re w' < 1` `Re w' \<noteq> 0`
        by (auto simp add: complex_of_real_Re)
    qed
  next
    show "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
      by fact+
  next
    fix M u v
    assume 1: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
    let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v"
    assume 2: "?P ?Mu ?Mv"
    show "?P u v"
    proof safe
      fix w
      assume "w \<in> unit_disc" "poincare_colinear {u, v, w}" "\<not> poincare_between u v w" "\<not> poincare_between v u w"
      thus "poincare_between u w v"
        using 1 2[rule_format, of "moebius_pt M w"]
        by simp
    qed
  qed
  thus ?thesis
    using assms
    by simp
qed


lemma poincare_between_transitivity:
  assumes "a \<in> unit_disc" "x \<in> unit_disc" "b \<in> unit_disc" "y \<in> unit_disc"
          "poincare_between a x b" "poincare_between a b y"
  shows  "poincare_between x b y"
proof(cases "a = b")
  case True
  thus ?thesis
    using assms
    using poincare_between_sandwich by blast
next
  case False
  have "\<forall> x. \<forall> y. poincare_between a x b \<and> poincare_between a b y \<and> x \<in> unit_disc
                  \<and> y \<in> unit_disc \<longrightarrow> poincare_between x b y" (is "?P a b")
  proof (rule wlog_positive_x_axis[where P="?P"])
    show "a \<in> unit_disc"
      using assms by simp
  next
    show "b \<in> unit_disc"
      using assms by simp
  next
    show "a \<noteq> b"
      using False by simp
  next
    fix M u v
    assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v" 
              "\<forall>x y. poincare_between (moebius_pt M u) x (moebius_pt M v) \<and> 
                  poincare_between (moebius_pt M u) (moebius_pt M v) y \<and>
                  x \<in> unit_disc \<and> y \<in> unit_disc \<longrightarrow>
                  poincare_between x (moebius_pt M v) y"
    show "\<forall>x y. poincare_between u x v \<and> poincare_between u v y \<and> x \<in> unit_disc \<and> y \<in> unit_disc 
                \<longrightarrow> poincare_between x v y"
    proof safe
      fix x y
      assume "poincare_between u x v" "poincare_between u v y" " x \<in> unit_disc" "y \<in> unit_disc"

      have "poincare_between (moebius_pt M u) (moebius_pt M x) (moebius_pt M v)" 
        using `poincare_between u x v` `unit_disc_fix M` `x \<in> unit_disc` `u \<in> unit_disc` `v \<in> unit_disc`
        by simp
      moreover
      have "poincare_between (moebius_pt M u) (moebius_pt M v) (moebius_pt M y)"
        using `poincare_between u v y` `unit_disc_fix M` `y \<in> unit_disc` `u \<in> unit_disc` `v \<in> unit_disc`
        by simp
      moreover
      have "(moebius_pt M x) \<in> unit_disc"
        using `unit_disc_fix M` `x \<in> unit_disc` by simp
      moreover
      have "(moebius_pt M y) \<in> unit_disc"
        using `unit_disc_fix M` `y \<in> unit_disc` by simp
      ultimately
      have "poincare_between (moebius_pt M x) (moebius_pt M v) (moebius_pt M y)"
        using * by blast
      thus "poincare_between x v y"
        using `y \<in> unit_disc` * `x \<in> unit_disc` by simp
    qed
  next
    fix x
    assume xx: "is_real x" "0 < Re x" "Re x < 1"
    hence "of_complex x \<in> unit_disc"
      using cmod_eq_Re by auto
    hence "of_complex x \<noteq> \<infinity>\<^sub>h"
      by simp
    have " of_complex x \<noteq> 0\<^sub>h"
      using xx by auto
    have "of_complex x \<in> circline_set x_axis"
      using xx by simp
    show "\<forall>m n. poincare_between 0\<^sub>h m (of_complex x) \<and> poincare_between 0\<^sub>h (of_complex x) n \<and>
            m \<in> unit_disc \<and> n \<in> unit_disc \<longrightarrow> poincare_between m (of_complex x) n"
    proof safe
      fix m n
      assume **: "poincare_between 0\<^sub>h m (of_complex x)" "poincare_between 0\<^sub>h (of_complex x) n"
                 "m \<in> unit_disc" " n \<in> unit_disc"
      show "poincare_between m (of_complex x) n"
      proof(cases "m = 0\<^sub>h")
        case True
        thus ?thesis
          using ** by auto
      next
        case False
        hence "m \<in> circline_set x_axis"
          using poincare_between_poincare_line_uzv[of "0\<^sub>h" "of_complex x" m]
          using poincare_line_0_real_is_x_axis[of "of_complex x"] 
          using `of_complex x \<in> unit_disc` `of_complex x \<noteq> \<infinity>\<^sub>h` `of_complex x \<noteq> 0\<^sub>h`
          using `of_complex x \<in> circline_set x_axis` `m \<in> unit_disc` **(1)
          by simp
        then obtain m' where "m = of_complex m'" "is_real m'"
          using inf_or_of_complex[of m] `m \<in> unit_disc`
          unfolding circline_set_x_axis
          by auto
        hence "Re m' \<le> Re x"
          using `poincare_between 0\<^sub>h m (of_complex x)` xx `of_complex x \<noteq> 0\<^sub>h`
          using False ** `of_complex x \<in> unit_disc`
          using cmod_Re_le_iff poincare_between_0uv by auto
 
        have "n \<noteq> 0\<^sub>h"
          using **(2, 4) `of_complex x \<noteq> 0\<^sub>h` `of_complex x \<in> unit_disc`
          using poincare_between_sandwich by fastforce
        have "n \<in> circline_set x_axis"
          using poincare_between_poincare_line_uvz[of "0\<^sub>h" "of_complex x" n]
          using poincare_line_0_real_is_x_axis[of "of_complex x"] 
          using `of_complex x \<in> unit_disc` `of_complex x \<noteq> \<infinity>\<^sub>h` `of_complex x \<noteq> 0\<^sub>h`
          using `of_complex x \<in> circline_set x_axis` `n \<in> unit_disc` **(2)
          by simp
        then obtain n' where "n = of_complex n'" "is_real n'"
          using inf_or_of_complex[of n] `n \<in> unit_disc`
          unfolding circline_set_x_axis
          by auto
        hence "Re x \<le> Re n'"
          using `poincare_between 0\<^sub>h (of_complex x) n` xx `of_complex x \<noteq> 0\<^sub>h`
          using False ** `of_complex x \<in> unit_disc` `n \<noteq> 0\<^sub>h`
          using cmod_Re_le_iff poincare_between_0uv
          by (metis Re_complex_of_real arg_0_iff rcis_cmod_arg rcis_zero_arg to_complex_of_complex)
        
        have "poincare_between (of_complex m') (of_complex x) (of_complex n')" 
          using `Re x \<le> Re n'` `Re m' \<le> Re x`
          using poincare_between_x_axis_uvw[of "Re m'" "Re x" "Re n'"]
          using `is_real n'` `is_real m'` `n \<in> unit_disc` `n = of_complex n'`
          using xx `m = of_complex m'` `m \<in> unit_disc`
          by (smt complex_of_real_Re norm_of_real poincare_between_def unit_disc_iff_cmod_lt_1)

        thus ?thesis
          using `n = of_complex n'` `m = of_complex m'`
          by auto
      qed
    qed
  qed 
  thus ?thesis
    using assms
    by blast
qed

(* ------------------------------------------------------------------ *)
subsection{* Poincare between - sum distances *}
(* ------------------------------------------------------------------ *)

lemma poincare_between_sum_distances_x_axis_u0v:
  assumes "of_complex u' \<in> unit_disc" "of_complex v' \<in> unit_disc"
  assumes "is_real u'" "u' \<noteq> 0" "v' \<noteq> 0"
  shows  "poincare_distance (of_complex u') 0\<^sub>h + poincare_distance 0\<^sub>h (of_complex v') = poincare_distance (of_complex u') (of_complex v') \<longleftrightarrow>
          is_real v' \<and> Re u' * Re v' < 0" (is "?P u' v' \<longleftrightarrow> ?Q u' v'")
proof-
  have "Re u' \<noteq> 0"
    using `is_real u'` `u' \<noteq> 0`
    using complex_eq_if_Re_eq
    by simp

  let ?u = "cmod u'" and ?v = "cmod v'" and ?uv = "cmod (u' - v')"
  have disc: "?u\<^sup>2 < 1" "?v\<^sup>2 < 1"
    using unit_disc_cmod_square_lt_1[OF assms(1)]
    using unit_disc_cmod_square_lt_1[OF assms(2)]
    by auto
  have "poincare_distance (of_complex u') 0\<^sub>h + poincare_distance 0\<^sub>h (of_complex v') =
              acosh (((1 + ?u\<^sup>2) * (1 + ?v\<^sup>2) + 4 * ?u * ?v) / ((1 - ?u\<^sup>2) * (1 - ?v\<^sup>2)))" (is "_ = acosh ?r1")
          using poincare_distance_formula_zero_sum[OF assms(1-2)]
          by (simp add: Let_def)
  moreover
  have "poincare_distance (of_complex u') (of_complex v') =
              acosh (((1 - ?u\<^sup>2) * (1 - ?v\<^sup>2) + 2 * ?uv\<^sup>2) / ((1 - ?u\<^sup>2) * (1 - ?v\<^sup>2)))" (is "_ = acosh ?r2")
    using disc
    using poincare_distance_formula[OF assms(1-2)]
    by (subst add_divide_distrib) simp
  moreover
  have "acosh ?r1 = acosh ?r2 \<longleftrightarrow> ?Q u' v'"
  proof
    assume "acosh ?r1 = acosh ?r2"
    hence "?r1 = ?r2"
    proof (subst (asm) acosh_eq_iff)
      show "?r1 \<ge> 1"
      proof-
        have "(1 - ?u\<^sup>2) * (1 - ?v\<^sup>2) \<le> (1 + ?u\<^sup>2) * (1 + ?v\<^sup>2) + 4 * ?u * ?v"
          by (simp add: field_simps)
        thus ?thesis
          using disc
          by simp
      qed
    next
      show "?r2 \<ge> 1"
        using disc
        by simp
    qed
    hence "(1 + ?u\<^sup>2) * (1 + ?v\<^sup>2) + 4 * ?u * ?v = (1 - ?u\<^sup>2) * (1 - ?v\<^sup>2) + 2 * ?uv\<^sup>2"
      using disc
      by auto              
    hence "(cmod (u' - v'))\<^sup>2 = (cmod u' + cmod v')\<^sup>2"
      by (simp add: field_simps power2_eq_square)
    hence *: "Re u' * Re v' + \<bar>Re u'\<bar> * sqrt ((Im v')\<^sup>2 + (Re v')\<^sup>2) = 0"
      using `is_real u'`
      unfolding cmod_power2 cmod_def
      by (simp add: field_simps) (simp add: power2_eq_square field_simps)
    hence "sqrt ((Im v')\<^sup>2 + (Re v')\<^sup>2) = \<bar>Re v'\<bar>"
      using `Re u' \<noteq> 0` `v' \<noteq> 0`
      by (smt complex_neq_0 mult.commute mult_cancel_right mult_minus_left real_sqrt_gt_0_iff)
    hence "Im v' = 0"
      by (smt Im_eq_0 norm_complex_def)
    moreover
    hence "Re u' * Re v' = - \<bar>Re u'\<bar> * \<bar>Re v'\<bar>"
      using *
      by simp
    hence "Re u' * Re v' < 0"
      using `Re u' \<noteq> 0` `v' \<noteq> 0`
      by (simp add: \<open>is_real v'\<close> complex_eq_if_Re_eq)
    ultimately
    show "?Q u' v'"
      by simp
  next
    assume "?Q u' v'"
    hence "is_real v'" "Re u' * Re v' < 0"
      by auto
    have "?r1 = ?r2"
    proof (cases "Re u' > 0")
      case True
      hence "Re v' < 0"
        using `Re u' * Re v' < 0`
        by (smt zero_le_mult_iff)
      show ?thesis
        using disc `is_real u'` `is_real v'`
        using `Re u' > 0` `Re v' < 0`
        unfolding cmod_power2 cmod_def
        by simp (simp add: power2_eq_square field_simps)
    next
      case False
      hence "Re u' < 0"
        using `Re u' \<noteq> 0`
        by simp
      hence "Re v' > 0"
        using `Re u' * Re v' < 0`
        by (smt zero_le_mult_iff)
      show ?thesis
        using disc `is_real u'` `is_real v'`
        using `Re u' < 0` `Re v' > 0`
        unfolding cmod_power2 cmod_def
        by simp (simp add: power2_eq_square field_simps)
    qed
    thus "acosh ?r1 = acosh ?r2"
      by metis
  qed
  ultimately
  show ?thesis
    by simp
qed

(*
  Different proof of the previous theorem relying on the cross-ratio definition, and not the distance formula.
  I suppose that this could be also used to prove the triangle inequality.
*)
lemma poincare_between_sum_distances_x_axis_u0v_different_proof:
  assumes "of_complex u' \<in> unit_disc" "of_complex v' \<in> unit_disc"
  assumes "is_real u'" "u' \<noteq> 0" "v' \<noteq> 0" (* additional condition *) "is_real v'"
  shows  "poincare_distance (of_complex u') 0\<^sub>h + poincare_distance 0\<^sub>h (of_complex v') = poincare_distance (of_complex u') (of_complex v') \<longleftrightarrow>
          Re u' * Re v' < 0" (is "?P u' v' \<longleftrightarrow> ?Q u' v'")
proof-
  have "-1 < Re u'" "Re u' < 1" "Re u' \<noteq> 0"
    using assms
    by (auto simp add: cmod_eq_Re complex_eq_if_Re_eq)
  have "-1 < Re v'" "Re v' < 1" "Re v' \<noteq> 0"
    using assms
    by (auto simp add: cmod_eq_Re complex_eq_if_Re_eq)

  have "\<bar>ln (Re ((1 - u') / (1 + u')))\<bar> + \<bar>ln (Re ((1 - v') / (1 + v')))\<bar> =
        \<bar>ln (Re ((1 + u') * (1 - v') / ((1 - u') * (1 + v'))))\<bar> \<longleftrightarrow> Re u' * Re v' < 0" (is "\<bar>ln ?a1\<bar>  + \<bar>ln ?a2\<bar> = \<bar>ln ?a3\<bar> \<longleftrightarrow> _")
  proof-
    have 1: "0 < ?a1" "ln ?a1 > 0 \<longleftrightarrow> Re u' < 0"
      using `Re u' < 1` `Re u' > -1` `is_real u'`
      using complex_is_Real_iff
      by auto
    have 2: "0 < ?a2" "ln ?a2 > 0 \<longleftrightarrow> Re v' < 0"
      using `Re v' < 1` `Re v' > -1` `is_real v'`
      using complex_is_Real_iff
      by auto
    have 3: "0 < ?a3" "ln ?a3 > 0 \<longleftrightarrow> Re v' < Re u'"
      using `Re u' < 1` `Re u' > -1` `is_real u'`
      using `Re v' < 1` `Re v' > -1` `is_real v'`
      using complex_is_Real_iff
       by auto (simp add: field_simps)+
    show ?thesis
    proof
      assume *: "Re u' * Re v' < 0"
      show "\<bar>ln ?a1\<bar> + \<bar>ln ?a2\<bar> = \<bar>ln ?a3\<bar>"
      proof (cases "Re u' > 0")
        case True
        hence "Re v' < 0"
          using *
          by (smt mult_nonneg_nonneg)
        show ?thesis
          using 1 2 3 `Re u' > 0` `Re v' < 0`
          using `Re u' < 1` `Re u' > -1` `is_real u'`
          using `Re v' < 1` `Re v' > -1` `is_real v'`
          using complex_is_Real_iff
          using ln_div ln_mult
          by simp
      next
        case False
        hence "Re v' > 0" "Re u' < 0"
          using *
          by (smt zero_le_mult_iff)+
        show ?thesis
          using 1 2 3 `Re u' < 0` `Re v' > 0`
          using `Re u' < 1` `Re u' > -1` `is_real u'`
          using `Re v' < 1` `Re v' > -1` `is_real v'`
          using complex_is_Real_iff
          using ln_div ln_mult
          by simp
      qed
    next
      assume *: "\<bar>ln ?a1\<bar> + \<bar>ln ?a2\<bar> = \<bar>ln ?a3\<bar>"
      {
        assume "Re u' > 0" "Re v' > 0"
        hence False
          using * 1 2 3
          using `Re u' < 1` `Re u' > -1` `is_real u'`
          using `Re v' < 1` `Re v' > -1` `is_real v'`
          using complex_is_Real_iff
          using ln_mult ln_div
          by (cases "Re v' < Re u'") auto
      }
      moreover
      {
        assume "Re u' < 0" "Re v' < 0"
        hence False
          using * 1 2 3
          using `Re u' < 1` `Re u' > -1` `is_real u'`
          using `Re v' < 1` `Re v' > -1` `is_real v'`
          using complex_is_Real_iff
          using ln_mult ln_div
          by (cases "Re v' < Re u'") auto
      }
      ultimately
      show "Re u' * Re v' < 0"
        using `Re u' \<noteq> 0` `Re v' \<noteq> 0`
        by (smt divisors_zero mult_le_0_iff)
    qed
  qed
  thus ?thesis
    using assms
    apply (subst poincare_distance_sym, simp, simp)
    apply (subst poincare_distance_zero_x_axis, simp, simp add: circline_set_x_axis)
    apply (subst poincare_distance_zero_x_axis, simp, simp add: circline_set_x_axis)
    apply (subst poincare_distance_x_axis_x_axis, simp, simp, simp add: circline_set_x_axis, simp add: circline_set_x_axis)
    apply simp
    done
qed

lemma poincare_between_sum_distances:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  shows "poincare_between u v w \<longleftrightarrow> poincare_distance u v + poincare_distance v w = poincare_distance u w" (is "?P' u v w")
proof (cases "u = v")
  case True
  thus ?thesis
    using assms
    by simp
next
  case False
  have "\<forall> w. w \<in> unit_disc \<longrightarrow> (poincare_between u v w \<longleftrightarrow> poincare_distance u v + poincare_distance v w = poincare_distance u w)" (is "?P u v")
  proof (rule wlog_positive_x_axis)
    fix x
    assume "is_real x" "0 < Re x" "Re x < 1"
    have "of_complex x \<in> circline_set x_axis"
      using `is_real x`
      by (auto simp add: circline_set_x_axis)

    have "of_complex x \<in> unit_disc"
      using `is_real x` `0 < Re x` `Re x < 1`
      by (simp add: cmod_eq_Re)

    have "x \<noteq> 0"
      using `is_real x` `Re x > 0`
      by auto

    show "?P (of_complex x) 0\<^sub>h"
    proof (rule allI, rule impI)
      fix w
      assume "w \<in> unit_disc"
      then obtain w' where "w = of_complex w'"
        using inf_or_of_complex[of w]
        by auto

      show "?P' (of_complex x) 0\<^sub>h w"
      proof (cases "w = 0\<^sub>h")
        case True
        thus ?thesis
          by simp
      next
        case False
        hence "w' \<noteq> 0"
          using `w = of_complex w'`
          by auto

        show ?thesis
          using `is_real x` `x \<noteq> 0` `w = of_complex w'` `w' \<noteq> 0`
          using `of_complex x \<in> unit_disc` `w \<in> unit_disc`
          apply simp
          apply (subst poincare_between_x_axis_u0v, simp_all)
          apply (subst poincare_between_sum_distances_x_axis_u0v, simp_all)
          done
      qed
    qed
  next
    show "v \<in> unit_disc" "u \<in> unit_disc"
      using assms
      by auto
  next
    show "v \<noteq> u"
      using `u \<noteq> v`
      by simp
  next
    fix M u v
    assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v" and
          **: "?P (moebius_pt M v) (moebius_pt M u)"
    show "?P v u"
    proof (rule allI, rule impI)
      fix w
      assume "w \<in> unit_disc"
      hence "moebius_pt M w \<in> unit_disc"
        using `unit_disc_fix M`
        by auto
      thus "?P' v u w"
        using `u \<in> unit_disc` `v \<in> unit_disc` `w \<in> unit_disc` `unit_disc_fix M`
        using **[rule_format, of "moebius_pt M w"]
        by auto
    qed
  qed
  thus ?thesis
    using assms
    by simp
qed

text{* Some more properties of Poincare_between *}

lemma conjugate_preserve_poincare_between [simp]:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
  shows "poincare_between (conjugate u) (conjugate v) (conjugate w) \<longleftrightarrow> poincare_between u v w"
  using assms
  using poincare_between_sum_distances
  by (metis conjugate_preserve_poincare_distance conjugate_unit_disc image_eqI)

(* todo: include between u y v and u z v *)
lemma unique_poincare_distance_on_ray:
  assumes "d \<ge> 0" "u \<noteq> v" "u \<in> unit_disc" "v \<in> unit_disc"
  assumes "y \<in> unit_disc" "poincare_distance u y = d" "poincare_between u v y"
  assumes "z \<in> unit_disc" "poincare_distance u z = d" "poincare_between u v z"
  shows "y = z"
proof-
  have "\<forall> d y z. d \<ge> 0 \<and>
        y \<in> unit_disc \<and> poincare_distance u y = d \<and> poincare_between u v y \<and>
        z \<in> unit_disc \<and> poincare_distance u z = d \<and> poincare_between u v z \<longrightarrow> y = z" (is "?P u v")
  proof (rule wlog_positive_x_axis[where P="?P"])
    fix x
    assume x: "is_real x" "0 < Re x" "Re x < 1"
    hence "x \<noteq> 0"
      using complex.expand[of x 0]
      by auto
    hence *: "poincare_line 0\<^sub>h (of_complex x) = x_axis"
      using x poincare_line_0_real_is_x_axis[of "of_complex x"]
      unfolding circline_set_x_axis
      by auto
    have "of_complex x \<in> unit_disc"
      using x
      by (auto simp add: cmod_eq_Re)
    have "arg x = 0"
      using x
      using arg_0_iff by blast
    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix y z
      assume "y \<in> unit_disc" "z \<in> unit_disc"
      then obtain y' z' where yz: "y = of_complex y'" "z = of_complex z'"
        using inf_or_of_complex[of y] inf_or_of_complex[of z]
        by auto
      assume betw: "poincare_between 0\<^sub>h (of_complex x) y"  "poincare_between 0\<^sub>h (of_complex x) z"
      hence "y \<noteq> 0\<^sub>h" "z \<noteq> 0\<^sub>h"
        using `x \<noteq> 0` `of_complex x \<in> unit_disc` `y \<in> unit_disc`
        using poincare_between_sandwich[of "0\<^sub>h" "of_complex x"]
        using of_complex_zero_iff[of x]
        by force+

      hence "arg y' = 0" "cmod y' \<ge> cmod x" "arg z' = 0" "cmod z' \<ge> cmod x"
        using poincare_between_0uv[of "of_complex x" y] poincare_between_0uv[of "of_complex x" z]
        using `of_complex x \<in> unit_disc` `x \<noteq> 0` `arg x = 0` `y \<in> unit_disc` `z \<in> unit_disc` betw yz
        by (simp_all add: Let_def)
      hence *: "is_real y'" "is_real z'" "Re y' > 0" "Re z' > 0"
        using arg_0_iff[of y'] arg_0_iff[of z'] x `y \<noteq> 0\<^sub>h` `z \<noteq> 0\<^sub>h` yz
        by auto
      assume "poincare_distance 0\<^sub>h z = poincare_distance 0\<^sub>h y" "0 \<le> poincare_distance 0\<^sub>h y"
      thus "y = z"
        using * yz `y \<in> unit_disc` `z \<in> unit_disc`
        using unique_x_axis_poincare_distance_positive[of "poincare_distance 0\<^sub>h y"]
        by (auto simp add: cmod_eq_Re unit_disc_to_complex_inj)
    qed
  next
    show "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
      by fact+
  next
    fix M u v
    assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
    assume **: "?P (moebius_pt M u) (moebius_pt M v)"
    show "?P u v"
    proof safe
      fix d y z
      assume ***: "0 \<le> poincare_distance u y"
             "y \<in> unit_disc" "poincare_between u v y"
             "z \<in> unit_disc" "poincare_between u v z"
             "poincare_distance u z = poincare_distance u y"
      let ?Mu = "moebius_pt M u" and ?Mv = "moebius_pt M v" and ?My = "moebius_pt M y" and ?Mz = "moebius_pt M z"
      have "?Mu \<in> unit_disc" "?Mv \<in> unit_disc" "?My \<in> unit_disc" "?Mz \<in> unit_disc"
        using `u \<in> unit_disc` `v \<in> unit_disc` `y \<in> unit_disc` `z \<in> unit_disc`
        using `unit_disc_fix M`
        by auto
      hence "?My = ?Mz"
        using * ***
        using **[rule_format, of "poincare_distance ?Mu ?My" ?My ?Mz]
        by simp
      thus "y = z"
        using bij_moebius_pt[of M]
        unfolding bij_def inj_on_def
        by blast
    qed
  qed
  thus ?thesis
    using assms
    by auto
qed

(* ------------------------------------------------------------------ *)
subsection{* Intersection of poincare lines with the x-axis *}
(* ------------------------------------------------------------------ *)

(* ---------------------------------------------------------------- *)
subsubsection{* Betweeness of x-axis intersection *}
(* ---------------------------------------------------------------- *)

lemma poincare_between_x_axis_intersection:
  assumes "u \<in> unit_disc" "v \<in> unit_disc" "z \<in> unit_disc" "u \<noteq> v"
  assumes "u \<notin> circline_set x_axis"  "v \<notin> circline_set x_axis"
  assumes "z \<in> circline_set (poincare_line u v) \<inter> circline_set x_axis"
  shows "poincare_between u z v \<longleftrightarrow> arg (to_complex u) * arg (to_complex v) < 0"
proof-
  have "\<forall> u v. u \<in> unit_disc \<and> v \<in> unit_disc \<and> u \<noteq> v \<and>
       u \<notin> circline_set x_axis \<and> v \<notin> circline_set x_axis \<and> 
       z \<in> circline_set (poincare_line u v) \<inter> circline_set x_axis \<longrightarrow> 
       (poincare_between u z v \<longleftrightarrow> arg (to_complex u) * arg (to_complex v) < 0)" (is "?P z")
  proof (rule wlog_real_zero)
    show "?P 0\<^sub>h"
    proof ((rule allI)+, rule impI, (erule conjE)+)
      fix u v
      assume *: "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v"
                "u \<notin> circline_set x_axis" "v \<notin> circline_set x_axis"
                "0\<^sub>h \<in> circline_set (poincare_line u v) \<inter> circline_set x_axis"
      obtain u' v' where uv: "u = of_complex u'" "v = of_complex v'"
        using * inf_or_of_complex[of u] inf_or_of_complex[of v]
        by auto

       
      hence "u \<noteq> 0\<^sub>h" "v \<noteq> 0\<^sub>h" "u' \<noteq> 0" "v' \<noteq> 0"
        using *
        by auto

      hence "arg u' \<noteq> 0" "arg v' \<noteq> 0"
        using * arg_0_iff[of u'] arg_0_iff[of v']
        unfolding circline_set_x_axis uv
        by auto

      have "poincare_colinear {0\<^sub>h, u, v}"
        using *
        unfolding poincare_colinear_def
        by (rule_tac x="poincare_line u v" in exI, simp)
      have "(\<exists>k<0. u' = cor k * v') \<longleftrightarrow> (arg u' * arg v' < 0)" (is "?lhs \<longleftrightarrow> ?rhs")
      proof
        assume "?lhs"
        then obtain k where "k < 0" "u' = cor k * v'"
          by auto
        thus ?rhs
          using arg_mult_real_negative[of k v'] arg_uminus_opposite_sign[of v']
          using `u' \<noteq> 0` `v' \<noteq> 0` `arg u' \<noteq> 0` `arg v' \<noteq> 0`
          by (auto simp add: mult_neg_pos mult_pos_neg)
      next
        assume ?rhs
        obtain ru rv \<phi> where polar: "u' = cor ru * cis \<phi>" "v' = cor rv * cis \<phi>"
          using `poincare_colinear {0\<^sub>h, u, v}` poincare_colinear_zero_polar_form[of u' v'] uv * `u' \<noteq> 0` `v' \<noteq> 0`
          by auto
        have "ru * rv < 0"
          using polar `?rhs` `u' \<noteq> 0` `v' \<noteq> 0`
          using arg_mult_real_negative[of "ru" "cis \<phi>"] arg_mult_real_positive[of "ru" "cis \<phi>"]
          using arg_mult_real_negative[of "rv" "cis \<phi>"] arg_mult_real_positive[of "rv" "cis \<phi>"]
          apply (cases "ru > 0")
          apply (cases "rv > 0", simp, simp add: mult_pos_neg)
          apply (cases "rv > 0", simp add: mult_neg_pos, simp)
          done
        thus "?lhs"
          using polar     
          by (rule_tac x="ru / rv" in exI, auto simp add: divide_less_0_iff mult_less_0_iff)
      qed
      thus "poincare_between u 0\<^sub>h v = (arg (to_complex u) * arg (to_complex v) < 0)"
        using poincare_between_u0v[of u v] * `u \<noteq> 0\<^sub>h` `v \<noteq> 0\<^sub>h` uv
        by simp
    qed
  next
    fix a z 
    assume 1: "is_real a" "cmod a < 1" "z \<in> unit_disc"
    assume 2: "?P (moebius_pt (blaschke a) z)"
    show "?P z"
    proof ((rule allI)+, rule impI, (erule conjE)+)
      fix u v
      let ?M = "moebius_pt (blaschke a)"
      let ?Mu = "?M u"
      let ?Mv = "?M v"
      assume *: "u \<in> unit_disc" "v \<in> unit_disc" "u \<noteq> v" "u \<notin> circline_set x_axis" "v \<notin> circline_set x_axis"
      hence "u \<noteq> \<infinity>\<^sub>h" "v \<noteq> \<infinity>\<^sub>h"
        by auto

      have **: "\<And> x y :: real. x * y < 0 \<longleftrightarrow> sgn (x * y) < 0"
        by simp

      assume "z \<in> circline_set (poincare_line u v) \<inter> circline_set x_axis"
      thus "poincare_between u z v = (arg (to_complex u) * arg (to_complex v) < 0)"
        using * 1 2[rule_format, of ?Mu ?Mv] `cmod a < 1` `is_real a` blaschke_unit_disc_fix[of a]
        using inversion_noteq_unit_disc[of "of_complex a" u] `u \<noteq> \<infinity>\<^sub>h`
        using inversion_noteq_unit_disc[of "of_complex a" v] `v \<noteq> \<infinity>\<^sub>h`
        apply auto
        apply (subst (asm) **, subst **, subst (asm) sgn_mult, subst sgn_mult, simp)
        apply (subst (asm) **, subst (asm) **, subst (asm) sgn_mult, subst (asm) sgn_mult, simp)
        done
    qed
  next
    show "z \<in> unit_disc" by fact
  next
    show "is_real (to_complex z)"
      using assms inf_or_of_complex[of z]
      by (auto simp add: circline_set_x_axis)
  qed
  thus ?thesis
    using assms
    by simp
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Check if a poincare line intersects the x-axis *}
(* ------------------------------------------------------------------ *)

lemma x_axis_intersection_equation:
  assumes
   "H = mk_circline A B C D"
   "(A, B, C, D) \<in> hermitean_nonzero"
 shows "of_complex z \<in> circline_set x_axis \<inter> circline_set H \<longleftrightarrow>
        A*z\<^sup>2 + 2*Re B*z + D = 0 \<and> is_real z" (is "?lhs \<longleftrightarrow> ?rhs")
proof-
  have "?lhs \<longleftrightarrow> A*z\<^sup>2 + (B + cnj B)*z + D = 0 \<and> z = cnj z"
    using assms
    using circline_equation_x_axis[of z]
    using circline_equation[of H A B C D z]
    using hermitean_elems
    by (auto simp add: power2_eq_square field_simps)
  thus ?thesis
    using eq_cnj_iff_real[of z]
    using hermitean_elems[of A B C D]
    by (simp add: complex_add_cnj complex_eq_if_Re_eq)
qed

text{*
  Check if a poincare line intersects x-axis within the unit disc -
  this could be generalized to checking if an arbitrary circline intersects the x-axis,
  but we do not need that.
*}

definition intersects_x_axis_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "intersects_x_axis_cmat H = (let (A, B, C, D) = H in A = 0 \<or> (Re B)\<^sup>2 > (Re A)\<^sup>2)"

lift_definition intersects_x_axis_clmat :: "circline_mat \<Rightarrow> bool" is intersects_x_axis_cmat
  done

lift_definition intersects_x_axis :: "circline \<Rightarrow> bool" is intersects_x_axis_clmat
proof (transfer)
  fix H1 H2
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" and "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where "k \<noteq> 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  thus "intersects_x_axis_cmat H1 = intersects_x_axis_cmat H2"
    using *
    by simp (smt mult_strict_left_mono power2_eq_square semiring_normalization_rules(13) zero_less_power2)
qed

lemma intersects_x_axis_mk_circline:
  assumes "(A, B, C, D) \<in> hermitean_nonzero"
  shows "intersects_x_axis (mk_circline A B C D) \<longleftrightarrow> A = 0 \<or> (Re B)\<^sup>2 > (Re A)\<^sup>2"
  using assms
  by (transfer, transfer, simp)


lemma intersects_x_axis_iff:
  assumes "is_poincare_line H"
  shows "(\<exists> x \<in> unit_disc. x \<in> circline_set H \<inter> circline_set x_axis) \<longleftrightarrow> intersects_x_axis H"
proof-
  obtain Ac B C Dc where  *: "H = mk_circline Ac B C Dc" "hermitean (Ac, B, C, Dc)" "(Ac, B, C, Dc) \<noteq> mat_zero"
    using ex_mk_circline[of H]
    by auto
  hence "(cmod B)\<^sup>2 > (cmod Ac)\<^sup>2" "Ac = Dc"
    using assms
    using is_poincare_line_mk_circline
    by auto

  hence "H = mk_circline (Re Ac) B (cnj B) (Re Ac)" "hermitean (cor (Re Ac),  B, (cnj B), cor (Re Ac))" "(cor (Re Ac),  B, (cnj B), cor (Re Ac)) \<noteq> mat_zero"
    using hermitean_elems[of Ac B C Dc] *
    by (auto simp add: complex_of_real_Re)
  then obtain A where
    *: "H = mk_circline (cor A) B (cnj B) (cor A)" "(cor A,  B, (cnj B), cor A) \<in> hermitean_nonzero"
    by (auto simp add: complex_of_real_Re)

  show ?thesis
  proof (cases "A = 0")
    case True
    thus ?thesis
      using *
      using x_axis_intersection_equation[OF *(1-2)]
      using intersects_x_axis_mk_circline[OF *(2)]
      using inf_or_of_complex inf_notin_unit_disc
      by auto (metis of_complex_zero of_real_0 unique_circline_01inf' zero_in_unit_disc)
  next
    case False
    show ?thesis
    proof
      assume "\<exists> x \<in> unit_disc. x \<in> circline_set H \<inter> circline_set x_axis"
      then obtain x where **: "of_complex x \<in> unit_disc" "of_complex x \<in> circline_set H \<inter> circline_set x_axis"
        by (metis inf_or_of_complex inf_notin_unit_disc)
      hence "is_real x"
        unfolding circline_set_x_axis
        using of_complex_inj
        by auto
      hence eq: "A * (Re x)\<^sup>2 + 2 * Re B * Re x + A = 0"
        using **
        using x_axis_intersection_equation[OF *(1-2), of "Re x"]
        by (simp add: complex_of_real_Re)
      hence "(2 * Re B)\<^sup>2 - 4 * A * A \<ge> 0"
        using real_quadratic_equation_discriminant[OF False, of "2 * Re B" A]
        by auto
      hence "(Re B)\<^sup>2 \<ge> (Re A)\<^sup>2"
        by (simp add: power2_eq_square)
      moreover
      have "(Re B)\<^sup>2 \<noteq> (Re A)\<^sup>2"
      proof (rule ccontr)
        assume "\<not> ?thesis"
        hence "Re B = Re A \<or> Re B = - Re A"
          using power2_eq_iff by blast
        hence "A * (Re x)\<^sup>2 +  A * 2* Re x + A = 0 \<or> A * (Re x)\<^sup>2 - A * 2 * Re x + A = 0"
          using eq
          by auto
        hence "A * ((Re x)\<^sup>2 +  2* Re x + 1) = 0 \<or> A * ((Re x)\<^sup>2 - 2 * Re x + 1) = 0"
          by (simp add: field_simps)
        hence "(Re x)\<^sup>2 + 2 * Re x + 1 = 0 \<or> (Re x)\<^sup>2 - 2 * Re x + 1 = 0"
          using `A \<noteq> 0`
          by simp
        hence "(Re x + 1)\<^sup>2 = 0 \<or> (Re x - 1)\<^sup>2 = 0"
          by (simp add: power2_sum power2_diff field_simps)
        hence "Re x = -1 \<or> Re x = 1"
          by auto
        thus False
          using `is_real x` `of_complex x \<in> unit_disc`
          by (auto simp add: cmod_eq_Re)
      qed
      ultimately
      show "intersects_x_axis H"
        using intersects_x_axis_mk_circline[OF *(2)]
        using *
        by auto
    next
      assume "intersects_x_axis H"
      hence "(Re B)\<^sup>2 > (Re A)\<^sup>2"
        using * False
        using intersects_x_axis_mk_circline[OF *(2)]
        by simp
      hence discr: "(2 * Re B)\<^sup>2 - 4 * A * A > 0"
        by (simp add: power2_eq_square)
      then obtain x1 x2 where
        eqs: "A * x1\<^sup>2 + 2 * Re B * x1 + A = 0" "A * x2\<^sup>2 + 2 * Re B * x2 + A = 0" "x1 \<noteq> x2"
        using real_quadratic_equation_distinct_solutions[OF `A \<noteq> 0`, of "2 * Re B" A]
        by auto
      hence "x1 * x2 = 1"
        using viette2[OF `A \<noteq> 0`, of "2 * Re B" A x1 x2] discr `A \<noteq> 0`
        by auto
      have "abs x1 \<noteq> 1" "abs x2 \<noteq> 1"
        using eqs discr `x1 * x2 = 1`
        by (auto simp add: abs_if power2_eq_square)
      hence "abs x1 < 1 \<or> abs x2 < 1"
        using `x1 * x2 = 1`
        by (smt mult_le_cancel_left1 mult_minus_right)
      thus "\<exists>x \<in> unit_disc. x \<in> circline_set H \<inter> circline_set x_axis"
        using x_axis_intersection_equation[OF *(1-2), of x1]
        using x_axis_intersection_equation[OF *(1-2), of x2]
        using eqs
        by auto
    qed
  qed
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Check if a poincare line intersects the y-axis *}
(* ------------------------------------------------------------------ *)

definition intersects_y_axis_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "intersects_y_axis_cmat H = (let (A, B, C, D) = H in A = 0 \<or> (Im B)\<^sup>2 > (Re A)\<^sup>2)"

lift_definition intersects_y_axis_clmat :: "circline_mat \<Rightarrow> bool" is intersects_y_axis_cmat
  done

lift_definition intersects_y_axis :: "circline \<Rightarrow> bool" is intersects_y_axis_clmat
proof (transfer)
  fix H1 H2
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" and "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where "k \<noteq> 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  thus "intersects_y_axis_cmat H1 = intersects_y_axis_cmat H2"
    using *
    by simp (smt mult_strict_left_mono power2_eq_square semiring_normalization_rules(13) zero_less_power2)
qed

lemma intersects_x_axis_intersects_y_axis [simp]:
  "intersects_x_axis (moebius_circline (moebius_rotation (pi/2)) H) \<longleftrightarrow> intersects_y_axis H"
  unfolding moebius_rotation_def moebius_similarity_def
  by simp (transfer, transfer, auto simp add: mat_adj_def mat_cnj_def)

lemma intersects_y_axis_iff:
  assumes "is_poincare_line H"
  shows "(\<exists> y \<in> unit_disc. y \<in> circline_set H \<inter> circline_set y_axis) \<longleftrightarrow> intersects_y_axis H" (is "?lhs \<longleftrightarrow> ?rhs")
proof-
  let ?R = "moebius_rotation (pi / 2)"
  let ?H' = "moebius_circline ?R H"
  have 1: "is_poincare_line ?H'"
    using assms
    using unit_circle_fix_preserve_is_poincare_line[OF _ assms, of ?R]
    by simp

  show ?thesis
  proof
    assume "?lhs"
    then obtain y where "y \<in> unit_disc" "y \<in> circline_set H \<inter> circline_set y_axis"
      by auto
    hence "moebius_pt ?R y \<in> unit_disc" "moebius_pt ?R y \<in> circline_set ?H' \<inter> circline_set x_axis"
      using rotation_pi_2_y_axis
      by (auto simp del: rotation_pi_2_y_axis)
         (metis circline_set_moebius_circline_I)
    thus ?rhs
      using intersects_x_axis_iff[OF 1]
      using intersects_x_axis_intersects_y_axis[of H]
      by auto
  next
    assume "intersects_y_axis H"
    hence "intersects_x_axis ?H'"
      using intersects_x_axis_intersects_y_axis[of H]
      by simp
    then obtain x where *: "x \<in> unit_disc" "x \<in> circline_set ?H' \<inter> circline_set x_axis"
      using intersects_x_axis_iff[OF 1]
      by auto
    let ?y = "moebius_pt (-?R) x"
    have "?y \<in> unit_disc" "?y \<in> circline_set H \<inter> circline_set y_axis"
      using * rotation_pi_2_y_axis[symmetric]
      by (auto simp del: rotation_pi_2_y_axis) 
         (metis circline_set_moebius_circline_E)
    thus ?lhs
      by auto
  qed
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Intersection point of a poincare line with the x-axis in the unit disc *}
(* ------------------------------------------------------------------ *)

definition calc_x_axis_intersection_cvec :: "complex \<Rightarrow> complex \<Rightarrow> complex_vec" where
 [simp]:  "calc_x_axis_intersection_cvec A B =
    (let discr = (Re B)\<^sup>2 - (Re A)\<^sup>2 in
         (-Re(B) + sgn (Re B) * sqrt(discr), A))"

(* intersection with the x-axis for poincare lines that are euclidean circles *)
definition calc_x_axis_intersection_cmat_cvec :: "complex_mat \<Rightarrow> complex_vec" where [simp]:
  "calc_x_axis_intersection_cmat_cvec H =
    (let (A, B, C, D) = H in 
       if A \<noteq> 0 then
          calc_x_axis_intersection_cvec A B
       else
          (0, 1)
    )"

lift_definition calc_x_axis_intersection_clmat_hcoords :: "circline_mat \<Rightarrow> complex_homo_coords" is calc_x_axis_intersection_cmat_cvec
  by (auto split: if_split_asm)

lift_definition calc_x_axis_intersection :: "circline \<Rightarrow> complex_homo" is calc_x_axis_intersection_clmat_hcoords
proof transfer
  fix H1 H2
  assume *: "hermitean H1 \<and> H1 \<noteq> mat_zero" "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where hh: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where k: "k \<noteq> 0" "H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto

  have "calc_x_axis_intersection_cvec A1 B1 \<approx>\<^sub>v calc_x_axis_intersection_cvec A2 B2"
    using hh k
    apply simp
    apply (rule_tac x="cor k" in exI)
    apply auto
    apply (simp add: sgn_mult power_mult_distrib)
    apply (subst right_diff_distrib[symmetric])
    apply (subst real_sqrt_mult)
    apply (subst cor_mult)
    by (simp add: real_sgn_eq right_diff_distrib)

  thus "calc_x_axis_intersection_cmat_cvec H1 \<approx>\<^sub>v
        calc_x_axis_intersection_cmat_cvec H2"
    using hh k
    by (auto simp del: calc_x_axis_intersection_cvec_def)
qed


lemma calc_x_axis_intersection_in_unit_disc:
  assumes "is_poincare_line H" "intersects_x_axis H"
  shows "calc_x_axis_intersection H \<in> unit_disc"
proof (cases "is_line H")
  case True
  thus ?thesis
    using assms
    unfolding unit_disc_def disc_def
    by simp (transfer, transfer, auto simp add: vec_cnj_def)
next
  case False
  thus ?thesis
    using assms
    unfolding unit_disc_def disc_def
  proof (simp, transfer, transfer)
    fix H
    assume hh: "hermitean H \<and> H \<noteq> mat_zero"
    then obtain A B D where *: "H = (A, B, cnj B, D)" "is_real A" "is_real D"
      using hermitean_elems
      by (cases H) blast
    assume "is_poincare_line_cmat H"
    hence *: "H = (A, B, cnj B, A)" "is_real A"
      using *
      by auto

    assume "\<not> circline_A0_cmat H"
    hence "A \<noteq> 0"
      using *
      by simp

    assume "intersects_x_axis_cmat H"
    hence "(Re B)\<^sup>2 > (Re A)\<^sup>2"
      using * `A \<noteq> 0`
      by (auto simp add: power2_eq_square complex.expand)

    hence "Re B \<noteq> 0"
      by auto

    have "Re A \<noteq> 0"
      using `is_real A` `A \<noteq> 0`
      by (auto simp add: complex.expand)

    have "sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2) < sqrt((Re B)\<^sup>2)"
      using `Re A \<noteq> 0`
      by (subst real_sqrt_less_iff) auto
    also have "... =  sgn (Re B) * (Re B)"
      by simp (smt mult_minus_right nonzero_eq_divide_eq real_sgn_eq)
    finally
    have 1: "sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2) < sgn (Re B) * (Re B)"
      .

    have 2: "(Re B)\<^sup>2 - (Re A)\<^sup>2 < sgn (Re B) * (Re B) * sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
      using `(Re B)\<^sup>2 > (Re A)\<^sup>2`
      using mult_strict_right_mono[OF 1, of "sqrt ((Re B)\<^sup>2 - (Re A)\<^sup>2)"]
      by simp

    have 3: "(Re B)\<^sup>2 - 2*sgn (Re B)*Re B*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2) + (Re B)\<^sup>2 - (Re A)\<^sup>2 < (Re A)\<^sup>2"
      using mult_strict_left_mono[OF 2, of 2]
      by (simp add: field_simps)

    have "(sgn (Re B))\<^sup>2 = 1"
      using `Re B \<noteq> 0`
      by (simp add: sgn_if)

    hence "(-Re B + sgn (Re B) * sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 < (Re A)\<^sup>2"
      using `(Re B)\<^sup>2 > (Re A)\<^sup>2` 3
      by (simp add: power2_diff field_simps)

    thus "in_ocircline_cmat_cvec unit_circle_cmat (calc_x_axis_intersection_cmat_cvec H)"
      using * `(Re B)\<^sup>2 > (Re A)\<^sup>2`
      by (auto simp add: vec_cnj_def power2_eq_square split: if_split_asm)
  qed
qed


lemma calc_x_axis_intersection:
  assumes "is_poincare_line H" "intersects_x_axis H"
  shows "calc_x_axis_intersection H \<in> circline_set H \<inter> circline_set x_axis"
proof (cases "is_line H")
  case True
  thus ?thesis
    using assms
    unfolding circline_set_def
    by simp (transfer, transfer, auto simp add: vec_cnj_def)
next
  case False
  thus ?thesis
    using assms
    unfolding circline_set_def
  proof (simp, transfer, transfer)
    fix H
    assume hh: "hermitean H \<and> H \<noteq> mat_zero"
    then obtain A B D where *: "H = (A, B, cnj B, D)" "is_real A" "is_real D"
      using hermitean_elems
      by (cases H) blast
    assume "is_poincare_line_cmat H"
    hence *: "H = (A, B, cnj B, A)" "is_real A"
      using *
      by auto
    assume "\<not> circline_A0_cmat H"
    hence "A \<noteq> 0"
      using *
      by auto

    assume "intersects_x_axis_cmat H"
    hence "(Re B)\<^sup>2 > (Re A)\<^sup>2"
      using * `A \<noteq> 0`
      by (auto simp add: power2_eq_square complex.expand)

    hence "Re B \<noteq> 0"
      by auto

    show "on_circline_cmat_cvec H (calc_x_axis_intersection_cmat_cvec H) \<and>
        on_circline_cmat_cvec x_axis_cmat (calc_x_axis_intersection_cmat_cvec H)" (is "?P1 \<and> ?P2")
    proof
      show "on_circline_cmat_cvec H (calc_x_axis_intersection_cmat_cvec H)"
      proof (cases "circline_A0_cmat H")
        case True
        thus ?thesis
          using * `is_poincare_line_cmat H` `intersects_x_axis_cmat H`
          by (simp add: vec_cnj_def)
      next
        case False
        let ?x = "calc_x_axis_intersection_cvec A B"
        let ?nom = "fst ?x" and ?den = "snd ?x"
        have x: "?x = (?nom, ?den)"
          by simp

        hence "on_circline_cmat_cvec H (calc_x_axis_intersection_cvec A B)"
        proof (subst *, subst x, subst on_circline_cmat_cvec_circline_equation)
          have "(sgn(Re B))\<^sup>2 = 1"
            using \<open>Re B \<noteq> 0\<close> sgn_pos zero_less_power2 by fastforce
          have "(sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 = (Re B)\<^sup>2 - (Re A)\<^sup>2"
            using `(Re B)\<^sup>2 > (Re A)\<^sup>2`
            by simp

          have "(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 = 
                (-(Re B))\<^sup>2 + (sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
            by (simp add: power2_diff)
          also have "... = (Re B)*(Re B) + (sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
            by (simp add: power2_eq_square)
          also have "... = (Re B)*(Re B) + (sgn(Re B))\<^sup>2*(sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
            by (simp add: power_mult_distrib)
          also have "... = (Re B)*(Re B) + (Re B)\<^sup>2 - (Re A)\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
            using `(sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 = (Re B)\<^sup>2 - (Re A)\<^sup>2` `(sgn(Re B))\<^sup>2 = 1`
            by simp
          finally have "(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 =
                        (Re B)*(Re B) + (Re B)\<^sup>2 - (Re A)\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)"
            by simp           

          have "is_real ?nom" "is_real ?den"
            using `is_real A`
            by simp+
          hence "cnj (?nom) = ?nom" "cnj (?den) = ?den"
            by (simp add:eq_cnj_iff_real)+      
          hence "A*?nom*(cnj (?nom)) + B*?den*(cnj (?nom)) + (cnj B)*(cnj (?den))*?nom + A*?den*(cnj (?den))
                = A*?nom*?nom + B*?den*?nom + (cnj B)*?den*?nom + A*?den*?den"
            by auto
          also have "... = A*?nom*?nom + (B + (cnj B))*?den*?nom + A*?den*?den"
            by (simp add:field_simps)
          also have "... = A*?nom*?nom + 2*(Re B)*?den*?nom + A*?den*?den"
            by (simp add:complex_add_cnj)
          also have "... = A*?nom\<^sup>2 + 2*(Re B)*?den*?nom + A*?den*?den"
            by (simp add:power2_eq_square)
          also have "... = A*(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2
                           + 2*(Re B)*A*(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)) + A*A*A"
            unfolding calc_x_axis_intersection_cvec_def
            by auto
          also have "... = A*((Re B)*(Re B) + (Re B)\<^sup>2 - (Re A)\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)) 
                     + 2*(Re B)*A*(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)) + A*A*A"
            using `(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2))\<^sup>2 =
                        (Re B)*(Re B) + (Re B)\<^sup>2 - (Re A)\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)`
            by simp
          also have "... = A*((Re B)*(Re B) + (Re B)\<^sup>2 - A\<^sup>2 - 2*(Re B)*sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)) 
                     + 2*(Re B)*A*(-(Re B) + sgn(Re B)*sqrt((Re B)\<^sup>2 - (Re A)\<^sup>2)) + A*A*A"
            using `is_real A`
            by (simp add: complex_of_real_Re)
          also have "... = 0"
            apply (simp add:field_simps)
            by (simp add: power2_eq_square)
          finally have "A*?nom*(cnj (?nom)) + B*?den*(cnj (?nom)) + (cnj B)*(cnj (?den))*?nom + A*?den*(cnj (?den)) = 0"
            by simp       
          thus "circline_equation A B (cnj B) A ?nom ?den"
            by simp
        qed
        thus ?thesis
          using * `is_poincare_line_cmat H` `intersects_x_axis_cmat H`
          by (simp add: vec_cnj_def)
      qed
    next
      show  "on_circline_cmat_cvec x_axis_cmat (calc_x_axis_intersection_cmat_cvec H)"
        using * `is_poincare_line_cmat H` `intersects_x_axis_cmat H` `is_real A`
        using eq_cnj_iff_real[of A]
        by (simp add: vec_cnj_def)
    qed
  qed
qed

lemma unique_calc_x_axis_intersection:
  assumes "is_poincare_line H" "H \<noteq> x_axis"
  assumes "x \<in> unit_disc" "x \<in> circline_set H \<inter> circline_set x_axis"
  shows  "x = calc_x_axis_intersection H"
proof-
  have *: "intersects_x_axis H"
    using assms
    using intersects_x_axis_iff[OF assms(1)]
    by auto
  show "x = calc_x_axis_intersection H"
    using calc_x_axis_intersection[OF assms(1) *]
    using calc_x_axis_intersection_in_unit_disc[OF assms(1) *]
    using assms
    using unique_is_poincare_line[of x "calc_x_axis_intersection H" H x_axis]
    by auto
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Check if a poincare line intersects the positive part of the x-axis *}
(* ------------------------------------------------------------------ *)

definition intersects_x_axis_positive_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "intersects_x_axis_positive_cmat H = (let (A, B, C, D) = H in Re A \<noteq> 0 \<and> Re B / Re A < -1)"

lift_definition intersects_x_axis_positive_clmat :: "circline_mat \<Rightarrow> bool" is intersects_x_axis_positive_cmat
  done

lift_definition intersects_x_axis_positive :: "circline \<Rightarrow> bool" is intersects_x_axis_positive_clmat
proof (transfer)
  fix H1 H2
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" and "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where "k \<noteq> 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  thus "intersects_x_axis_positive_cmat H1 = intersects_x_axis_positive_cmat H2"
    using *
    by simp
qed

lemma intersects_x_axis_positive_intersects_x_axis [simp]:
  assumes "intersects_x_axis_positive H"
  shows "intersects_x_axis H"
  using assms
  apply transfer
  apply transfer
  apply auto
  apply (smt divide_minus_left less_divide_eq_1_pos minus_divide_right real_sqrt_abs real_sqrt_less_iff)+
  done

lemma add_less_abs_positive_iff:
  fixes a b :: real
  assumes "abs b < abs a"
  shows "a + b > 0 \<longleftrightarrow> a > 0"
  using assms
  by auto

lemma calc_x_axis_intersection_positive_abs':
  fixes A B :: real
  assumes "B\<^sup>2 > A\<^sup>2" "A \<noteq> 0"
  shows "abs (sgn(B) * sqrt(B\<^sup>2 - A\<^sup>2) / A) < abs(-B/A)"
proof-
  from assms have "B \<noteq> 0"
    by auto

  have "B\<^sup>2 - A\<^sup>2 < B\<^sup>2"
    using `A \<noteq> 0`
    by auto
  hence "sqrt (B\<^sup>2 - A\<^sup>2) < abs B"
    using real_sqrt_less_iff[of "B\<^sup>2 - A\<^sup>2" "B\<^sup>2"]
    by simp
  thus ?thesis
    using assms `B \<noteq> 0`
    by (simp add: abs_mult divide_strict_right_mono)
qed

lemma calc_intersect_x_axis_positive_lemma:
  assumes "B\<^sup>2 > A\<^sup>2" "A \<noteq> 0"
  shows "(-B + sgn B * sqrt(B\<^sup>2 - A\<^sup>2)) / A > 0 \<longleftrightarrow> -B/A > 1"
proof-
  have "(-B + sgn B * sqrt(B\<^sup>2 - A\<^sup>2)) / A = -B / A + (sgn B * sqrt(B\<^sup>2 - A\<^sup>2)) / A"
    using assms
    by (simp add: field_simps)
  moreover
  have "-B / A + (sgn B * sqrt(B\<^sup>2 - A\<^sup>2)) / A > 0 \<longleftrightarrow> - B / A > 0"
    using add_less_abs_positive_iff[OF calc_x_axis_intersection_positive_abs'[OF assms]]
    by simp
  moreover
  hence "(B/A)\<^sup>2 > 1"
    using assms
    by (simp add: power_divide)
  hence "B/A > 1 \<or> B/A < -1"
    by (smt one_power2 pos2 power2_minus power_0 power_strict_decreasing zero_power2)
  hence "-B / A > 0 \<longleftrightarrow> -B / A > 1"
    by auto
  ultimately
  show ?thesis
    using assms
    by auto
qed

lemma intersects_x_axis_positive_iff':
  assumes "is_poincare_line H"
  shows "intersects_x_axis_positive H \<longleftrightarrow> calc_x_axis_intersection H \<in> unit_disc \<and> calc_x_axis_intersection H \<in> circline_set H \<inter> positive_x_axis" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  let ?x = "calc_x_axis_intersection H"
  assume ?lhs
  hence "?x \<in> circline_set x_axis" "?x \<in> circline_set H" "?x \<in> unit_disc"
    using calc_x_axis_intersection_in_unit_disc[OF assms] calc_x_axis_intersection[OF assms]
    by auto
  moreover
  have "Re (to_complex ?x) > 0"
    using `?lhs` assms
  proof (transfer, transfer)
    fix H
    assume hh: "hermitean H \<and> H \<noteq> mat_zero"
    obtain A B C D where *: "H = (A, B, C, D)"
      by (cases H, auto)
    assume "intersects_x_axis_positive_cmat H"
    hence **: "Re B / Re A < - 1" "Re A \<noteq> 0"
      using *
      by auto
    have "(Re B)\<^sup>2 > (Re A)\<^sup>2"
      using **
      by (smt divide_less_eq_1_neg divide_minus_left less_divide_eq_1_pos real_sqrt_abs real_sqrt_less_iff right_inverse_eq)
    have "is_real A" "A \<noteq> 0"
      using hh hermitean_elems * `Re A \<noteq> 0` complex.expand[of A 0]
      by auto
    have "(cmod B)\<^sup>2 > (cmod A)\<^sup>2"
      using `(Re B)\<^sup>2 > (Re A)\<^sup>2` `is_real A`
      by (smt cmod_power2 power2_less_0 zero_power2)
    have ***: "0 < (- Re B + sgn (Re B) * sqrt ((Re B)\<^sup>2 - (Re A)\<^sup>2)) / Re A"
      using calc_intersect_x_axis_positive_lemma[of "Re A" "Re B"] ** `(Re B)\<^sup>2 > (Re A)\<^sup>2`
      by auto

    assume "is_poincare_line_cmat H"
    hence "A = D"
      using * hh
      by simp

    have "Re ((cor (sgn (Re B)) * cor (sqrt ((Re B)\<^sup>2 - (Re A)\<^sup>2)) - cor (Re B)) / A) = (sgn (Re B) * sqrt ((Re B)\<^sup>2 - (Re A)\<^sup>2) - Re B) / Re D"
      using `is_real A` `A = D`
      by (metis (no_types, lifting) Re_complex_of_real complex_of_real_Re of_real_diff of_real_divide of_real_mult)
    thus "0 < Re (to_complex_cvec (calc_x_axis_intersection_cmat_cvec H))"
      using * hh ** *** `(cmod B)\<^sup>2 > (cmod A)\<^sup>2` `(Re B)\<^sup>2 > (Re A)\<^sup>2` `A \<noteq> 0` `A = D`
      by simp
  qed
  ultimately
  show ?rhs
    unfolding positive_x_axis_def
    by auto
next
  let ?x = "calc_x_axis_intersection H"
  assume ?rhs
  hence "Re (to_complex ?x) > 0"  "?x \<noteq> \<infinity>\<^sub>h" "?x \<in> circline_set x_axis" "?x \<in> unit_disc" "?x \<in> circline_set H"
    unfolding positive_x_axis_def
    by auto
  hence "intersects_x_axis H"
    using intersects_x_axis_iff[OF assms]
    by auto
  thus ?lhs
    using `Re (to_complex ?x) > 0` assms
  proof (transfer, transfer)
    fix H
    assume hh: "hermitean H \<and> H \<noteq> mat_zero"
    obtain A B C D where *: "H = (A, B, C, D)"
      by (cases H, auto)
    assume "0 < Re (to_complex_cvec (calc_x_axis_intersection_cmat_cvec H))" "intersects_x_axis_cmat H" "is_poincare_line_cmat H"
    hence **: "A \<noteq> 0" "0 < Re ((cor (sgn (Re B)) * cor (sqrt ((Re B)\<^sup>2 - (Re A)\<^sup>2)) - cor (Re B)) / A)"  "A = D" "is_real A" "(Re B)\<^sup>2 > (Re A)\<^sup>2"
      using * hh hermitean_elems
      by (auto split: if_split_asm)

    have "Re A \<noteq> 0"
      using complex.expand[of A 0] `A \<noteq> 0` `is_real A`
      by auto

    have "Re ((cor (sgn (Re B)) * cor (sqrt ((Re B)\<^sup>2 - (Re D)\<^sup>2)) - cor (Re B)) / D) = (sgn (Re B) * sqrt ((Re B)\<^sup>2 - (Re D)\<^sup>2) - Re B) / Re D"
      using `is_real A` `A = D`
      by (metis (no_types, lifting) Re_complex_of_real complex_of_real_Re of_real_diff of_real_divide of_real_mult)

    thus "intersects_x_axis_positive_cmat H"
      using * ** `Re A \<noteq> 0`
      using calc_intersect_x_axis_positive_lemma[of "Re A" "Re B"]
      by simp
  qed
qed

lemma intersects_x_axis_positive_iff:
  assumes "is_poincare_line H" "H \<noteq> x_axis"
  shows "intersects_x_axis_positive H \<longleftrightarrow> (\<exists> x. x \<in> unit_disc \<and> x \<in> circline_set H \<inter> positive_x_axis)" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume ?lhs
  thus ?rhs
    using intersects_x_axis_positive_iff'[OF assms(1)]
    by auto
next
  assume ?rhs
  then obtain x where "x \<in> unit_disc" "x \<in> circline_set H \<inter> positive_x_axis"
    by auto
  thus ?lhs
    using unique_calc_x_axis_intersection[OF assms, of x]
    using intersects_x_axis_positive_iff'[OF assms(1)]
    unfolding positive_x_axis_def
    by auto
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Check if a poincare line intersects the positive part of the y-axis *}
(* ------------------------------------------------------------------ *)

definition intersects_y_axis_positive_cmat :: "complex_mat \<Rightarrow> bool" where
  [simp]: "intersects_y_axis_positive_cmat H = (let (A, B, C, D) = H in Re A \<noteq> 0 \<and> Im B / Re A < -1)"

lift_definition intersects_y_axis_positive_clmat :: "circline_mat \<Rightarrow> bool" is intersects_y_axis_positive_cmat
  done

lift_definition intersects_y_axis_positive :: "circline \<Rightarrow> bool" is intersects_y_axis_positive_clmat
proof (transfer)
  fix H1 H2
  assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero" and "hermitean H2 \<and> H2 \<noteq> mat_zero"
  obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
    by (cases H1, cases H2, auto)
  assume "circline_eq_cmat H1 H2"
  then obtain k where "k \<noteq> 0 \<and> H2 = cor k *\<^sub>s\<^sub>m H1"
    by auto
  thus "intersects_y_axis_positive_cmat H1 = intersects_y_axis_positive_cmat H2"
    using *
    by simp
qed

lemma intersects_x_axis_positive_intersects_y_axis_positive [simp]:
  "intersects_x_axis_positive (moebius_circline (moebius_rotation (-pi/2)) H) \<longleftrightarrow> intersects_y_axis_positive H"
  using hermitean_elems
  unfolding moebius_rotation_def moebius_similarity_def
  by simp (transfer, transfer, auto simp add: mat_adj_def mat_cnj_def)

lemma intersects_y_axis_positive_iff:
  assumes "is_poincare_line H" "H \<noteq> y_axis"
  shows "(\<exists> y \<in> unit_disc. y \<in> circline_set H \<inter> positive_y_axis) \<longleftrightarrow> intersects_y_axis_positive H" (is "?lhs \<longleftrightarrow> ?rhs")
proof-
  let ?R = "moebius_rotation (-pi / 2)"
  let ?H' = "moebius_circline ?R H"
  have 1: "is_poincare_line ?H'"
    using assms
    using unit_circle_fix_preserve_is_poincare_line[OF _ assms(1), of ?R]
    by simp

  have 2: "moebius_circline ?R H \<noteq> x_axis"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "H = moebius_circline (moebius_rotation (pi/2)) x_axis"
      using moebius_circline_comp_inv_left[of ?R H]
      by auto
    thus False
      using `H \<noteq> y_axis`
      by auto
  qed

  show ?thesis
  proof
    assume "?lhs"
    then obtain y where "y \<in> unit_disc" "y \<in> circline_set H \<inter> positive_y_axis"
      by auto
    hence "moebius_pt ?R y \<in> unit_disc" "moebius_pt ?R y \<in> circline_set ?H' \<inter> positive_x_axis"
      using rotation_minus_pi_2_positive_y_axis
      by auto
    thus ?rhs
      using intersects_x_axis_positive_iff[OF 1 2]
      using intersects_x_axis_positive_intersects_y_axis_positive[of H]
      by auto
  next
    assume "intersects_y_axis_positive H"
    hence "intersects_x_axis_positive ?H'"
      using intersects_x_axis_positive_intersects_y_axis_positive[of H]
      by simp
    then obtain x where *: "x \<in> unit_disc" "x \<in> circline_set ?H' \<inter> positive_x_axis"
      using intersects_x_axis_positive_iff[OF 1 2]
      by auto
    let ?y = "moebius_pt (-?R) x"
    have "?y \<in> unit_disc" "?y \<in> circline_set H \<inter> positive_y_axis"
      using * rotation_minus_pi_2_positive_y_axis[symmetric]
      by (auto simp del: rotation_minus_pi_2_positive_y_axis)
         (metis image_iff moebius_pt_neq_I)
    thus ?lhs
      by auto
  qed
qed

(* ------------------------------------------------------------------ *)
subsubsection{* Position of the intersection point in the unit disc *}
(* ------------------------------------------------------------------ *)

text{* Check if the intersection point of one poincare line with the x-axis is located more outward
the edge of the disc than the intersection point of another poincare line. *}

definition outward_cmat :: "complex_mat \<Rightarrow> complex_mat \<Rightarrow> bool" where
 [simp]: "outward_cmat H1 H2 = (let (A1, B1, C1, D1) = H1; (A2, B2, C2, D2) = H2
                                 in -Re B1/Re A1 \<le> -Re B2/Re A2)"
lift_definition outward_clmat :: "circline_mat \<Rightarrow> circline_mat \<Rightarrow> bool" is outward_cmat
  done
lift_definition outward :: "circline \<Rightarrow> circline \<Rightarrow> bool" is outward_clmat
  apply transfer
  apply simp
  apply (case_tac circline_mat1, case_tac circline_mat2, case_tac circline_mat3, case_tac circline_mat4)
  apply simp
  apply (erule_tac exE)+
  apply (erule_tac conjE)+
  apply simp
  done

lemma calc_x_axis_intersection_fun_mono:
  fixes x1 x2 :: real
  assumes "x1 > 1" "x2 > x1"
  shows "x1 - sqrt(x1\<^sup>2 - 1) > x2 - sqrt(x2\<^sup>2 - 1)"
  using assms
proof-
  have *: "sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1) > 0"
    using assms
    by (smt one_less_power pos2 real_sqrt_gt_zero)

  have "sqrt(x1\<^sup>2 - 1) < x1"
    using real_sqrt_less_iff[of "x1\<^sup>2 - 1" "x1\<^sup>2"] `x1 > 1`
    by auto
  moreover
  have "sqrt(x2\<^sup>2 - 1) < x2"
    using real_sqrt_less_iff[of "x2\<^sup>2 - 1" "x2\<^sup>2"] `x1 > 1` `x2 > x1`
    by auto
  ultimately
  have "sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1) < x1 + x2"
    by simp
  hence "(x1 + x2) / (sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1)) > 1"
    using *
    using less_divide_eq_1_pos[of "sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1)" "x1 + x2"]
    by simp
  hence "(x2\<^sup>2 - x1\<^sup>2) / (sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1)) > x2 - x1"
    using `x2 > x1`
    using mult_less_cancel_left_pos[of "x2 - x1" 1 "(x2 + x1) / (sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1))"]
    by (simp add: power2_eq_square field_simps)
  moreover
  have "(x2\<^sup>2 - x1\<^sup>2) = (sqrt(x1\<^sup>2 - 1) + sqrt(x2\<^sup>2 - 1)) * ((sqrt(x2\<^sup>2 - 1) - sqrt(x1\<^sup>2 - 1)))"
    using `x1 > 1` `x2 > x1`
    by (simp add: field_simps)
  ultimately
  have "sqrt(x2\<^sup>2 - 1) - sqrt(x1\<^sup>2 - 1) > x2 - x1"
    using *
    by simp
  thus ?thesis
    by simp
qed

lemma calc_x_axis_intersection_mono:
  fixes a1 b1 a2 b2 :: real
  assumes "-b1/a1 > 1" "a1 \<noteq> 0" "-b2/a2 \<ge> -b1/a1" "a2 \<noteq> 0"
  shows "(-b1 + sgn b1 * sqrt(b1\<^sup>2 - a1\<^sup>2)) / a1 \<ge> (-b2 + sgn b2 * sqrt(b2\<^sup>2 - a2\<^sup>2)) / a2" (is "?lhs \<ge> ?rhs")
proof-
  have "?lhs = -b1/a1 - sqrt((-b1/a1)\<^sup>2 - 1)"
  proof (cases "b1 > 0")
    case True
    hence "a1 < 0"
      using assms
      by (smt divide_neg_pos)
    thus ?thesis
      using `b1 > 0` `a1 < 0`
      by (simp add: real_sqrt_divide field_simps)
  next
    case False
    hence "b1 < 0"
      using assms
      by (cases "b1 = 0") auto
    hence "a1 > 0"
      using assms
      by (smt divide_pos_neg)
    thus ?thesis
      using `b1 < 0` `a1 > 0`
      by (simp add: real_sqrt_divide field_simps)
  qed

  moreover

  (* TODO: avoid copy-paste proofs *)
  have "?rhs = -b2/a2 - sqrt((-b2/a2)\<^sup>2 - 1)"
  proof (cases "b2 > 0")
    case True
    hence "a2 < 0"
      using assms
      by (smt divide_neg_pos)
    thus ?thesis
      using `b2 > 0` `a2 < 0`
      by (simp add: real_sqrt_divide field_simps)
  next
    case False
    hence "b2 < 0"
      using assms
      by (cases "b2 = 0") auto
    hence "a2 > 0"
      using assms
      by (smt divide_pos_neg)
    thus ?thesis
      using `b2 < 0` `a2 > 0`
      by (simp add: real_sqrt_divide field_simps)
  qed

  ultimately

  show ?thesis
    using calc_x_axis_intersection_fun_mono[of "-b1/a1" "-b2/a2"]
    using assms
    by (cases "-b1/a1=-b2/a2", auto)
qed

lemma outward:
  assumes "is_poincare_line H1" "is_poincare_line H2"
  assumes "intersects_x_axis_positive H1" "intersects_x_axis_positive H2"
  assumes "outward H1 H2"
  shows "Re (to_complex (calc_x_axis_intersection H1)) \<ge> Re (to_complex (calc_x_axis_intersection H2))"
proof-
  have "intersects_x_axis H1" "intersects_x_axis H2"
    using assms
    by auto
  thus ?thesis
    using assms
  proof (transfer, transfer)
    fix H1 H2
    assume hh: "hermitean H1 \<and> H1 \<noteq> mat_zero"  "hermitean H2 \<and> H2 \<noteq> mat_zero"
    obtain A1 B1 C1 D1 A2 B2 C2 D2 where *: "H1 = (A1, B1, C1, D1)" "H2 = (A2, B2, C2, D2)"
      by (cases H1, cases H2, auto)
    have "is_real A1" "is_real A2"
      using hermitean_elems * hh
      by auto
    assume 1: "intersects_x_axis_positive_cmat H1" "intersects_x_axis_positive_cmat H2"
    assume 2: "intersects_x_axis_cmat H1" "intersects_x_axis_cmat H2"
    assume 3: "is_poincare_line_cmat H1" "is_poincare_line_cmat H2"
    assume 4: "outward_cmat H1 H2"
    have "A1 \<noteq> 0" "A2 \<noteq> 0"
      using * `is_real A1` `is_real A2` 1 complex.expand[of A1 0] complex.expand[of A2 0]
      by auto
    hence "(sgn (Re B2) * sqrt ((Re B2)\<^sup>2 - (Re A2)\<^sup>2) - Re B2) / Re A2
         \<le> (sgn (Re B1) * sqrt ((Re B1)\<^sup>2 - (Re A1)\<^sup>2) - Re B1) / Re A1"
      using calc_x_axis_intersection_mono[of "Re B1" "Re A1" "Re B2" "Re A2"]
      using 1 4 *
      by simp
    moreover
    have "(sgn (Re B2) * sqrt ((Re B2)\<^sup>2 - (Re A2)\<^sup>2) - Re B2) / Re A2 = 
          Re ((cor (sgn (Re B2)) * cor (sqrt ((Re B2)\<^sup>2 - (Re A2)\<^sup>2)) - cor (Re B2)) / A2)"
      using `is_real A2` `A2 \<noteq> 0`
      by (simp add: Re_divide_real)
    moreover
    have "(sgn (Re B1) * sqrt ((Re B1)\<^sup>2 - (Re A1)\<^sup>2) - Re B1) / Re A1 =
           Re ((cor (sgn (Re B1)) * cor (sqrt ((Re B1)\<^sup>2 - (Re A1)\<^sup>2)) - cor (Re B1)) / A1)"
      using `is_real A1` `A1 \<noteq> 0`
      by (simp add: Re_divide_real)
    ultimately
    show "Re (to_complex_cvec (calc_x_axis_intersection_cmat_cvec H2))
          \<le> Re (to_complex_cvec (calc_x_axis_intersection_cmat_cvec H1))"
      using 2 3 `A1 \<noteq> 0` `A2 \<noteq> 0` * `is_real A1` `is_real A2`
      by (simp del: is_poincare_line_cmat_def intersects_x_axis_cmat_def)
  qed
qed


(* ------------------------------------------------------------------ *)
subsection{* Perpendicularity *}
(* ------------------------------------------------------------------ *)

definition perpendicular_to_x_axis_cmat :: "complex_mat \<Rightarrow> bool" where
 [simp]: "perpendicular_to_x_axis_cmat H \<longleftrightarrow> (let (A, B, C, D) = H in is_real B)"

lift_definition perpendicular_to_x_axis_clmat :: "circline_mat \<Rightarrow> bool" is perpendicular_to_x_axis_cmat
  done

lift_definition perpendicular_to_x_axis :: "circline \<Rightarrow> bool" is perpendicular_to_x_axis_clmat
  by transfer auto

lemma perpendicular_to_x_axis:
  assumes "is_poincare_line H"
  shows "perpendicular_to_x_axis H \<longleftrightarrow> perpendicular x_axis H"
  using assms
  unfolding perpendicular_def
proof (transfer, transfer)
  fix H
  assume hh: "hermitean H \<and> H \<noteq> mat_zero" "is_poincare_line_cmat H"
  obtain A B C D where *: "H = (A, B, C, D)"
    by (cases H, auto)
  hence "is_real A" "(cmod B)\<^sup>2 > (cmod A)\<^sup>2" "H = (A, B, cnj B, A)"
    using hermitean_elems[of A B C D] hh
    by auto
  thus "perpendicular_to_x_axis_cmat H =
        (cos_angle_cmat (of_circline_cmat x_axis_cmat) (of_circline_cmat H) = 0)"
    using cmod_square[of B] cmod_square[of A]
    by simp
qed

lemma perpendicular_to_x_axis_y_axis:
  assumes "perpendicular_to_x_axis (poincare_line 0\<^sub>h (of_complex z))" "z \<noteq> 0"
  shows "is_imag z"
  using assms
  by (transfer, transfer, simp)


lemma wlog_perpendicular_axes:
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "z \<in> unit_disc"
  assumes perpendicular: "is_poincare_line H1" "is_poincare_line H2" "perpendicular H1 H2"
  assumes "z \<in> circline_set H1 \<inter> circline_set H2" "u \<in> circline_set H1" "v \<in> circline_set H2"
  assumes axes: "\<And> x y. \<lbrakk>is_real x; 0 \<le> Re x; Re x < 1; is_imag y; 0 \<le> Im y; Im y < 1\<rbrakk> \<Longrightarrow> P 0\<^sub>h (of_complex x) (of_complex y)"
  assumes moebius: "\<And> M u v w. \<lbrakk>unit_disc_fix M; u \<in> unit_disc; v \<in> unit_disc; w \<in> unit_disc; P (moebius_pt M u) (moebius_pt M v) (moebius_pt M w) \<rbrakk> \<Longrightarrow> P u v w"
  assumes conjugate: "\<And> u v w. \<lbrakk>u \<in> unit_disc; v \<in> unit_disc; w \<in> unit_disc; P (conjugate u) (conjugate v) (conjugate w) \<rbrakk> \<Longrightarrow> P u v w"
  shows "P z u v"
proof-
  have "\<forall> v H1 H2. is_poincare_line H1 \<and> is_poincare_line H2 \<and> perpendicular H1 H2 \<and>
                   z \<in> circline_set H1 \<inter> circline_set H2 \<and> u \<in> circline_set H1 \<and> v \<in> circline_set H2 \<and> v \<in> unit_disc \<longrightarrow> P z u v" (is "?P z u")
  proof (rule wlog_x_axis[where P="?P"])
    fix x
    assume x: "is_real x" "Re x \<ge> 0" "Re x < 1"
    have "of_complex x \<in> unit_disc"
      using x
      by (simp add: cmod_eq_Re)

    show "?P 0\<^sub>h (of_complex x)"
    proof safe
      fix v H1 H2
      assume "v \<in> unit_disc"
      then obtain y where y: "v = of_complex y"
        using inf_or_of_complex[of v]
        by auto

      assume 1: "is_poincare_line H1" "is_poincare_line H2" "perpendicular H1 H2"
      assume 2: "0\<^sub>h \<in> circline_set H1" "0\<^sub>h \<in> circline_set H2" "of_complex x \<in> circline_set H1" "v \<in> circline_set H2"

      show "P 0\<^sub>h (of_complex x) v"
      proof (cases "of_complex x = 0\<^sub>h")
        case True
        show "P 0\<^sub>h (of_complex x) v"
        proof (cases "v = 0\<^sub>h")
          case True
          thus ?thesis
            using `of_complex x = 0\<^sub>h`
            using axes[of 0 0]
            by simp
        next
          case False
          show ?thesis
          proof (rule wlog_rotation_to_positive_y_axis)
            show "v \<in> unit_disc" "v \<noteq> 0\<^sub>h"
              by fact+
          next
            fix y
            assume "is_imag y" "0 < Im y" "Im y < 1"
            thus "P 0\<^sub>h (of_complex x) (of_complex y)"
              using x axes[of x y]
              by simp
          next
            fix \<phi> u
            assume "u \<in> unit_disc" "u \<noteq> 0\<^sub>h"
                   "P 0\<^sub>h (of_complex x) (moebius_pt (moebius_rotation \<phi>) u)"
            thus "P 0\<^sub>h (of_complex x) u"
              using `of_complex x = 0\<^sub>h`
              using moebius[of "moebius_rotation \<phi>"  "0\<^sub>h" "0\<^sub>h" u]
              by simp
          qed
        qed
      next
        case False
        hence *: "poincare_line 0\<^sub>h (of_complex x) = x_axis"
          using x poincare_line_0_real_is_x_axis[of "of_complex x"]
          unfolding circline_set_x_axis
          by auto
        hence "H1 = x_axis"
          using unique_poincare_line[of "0\<^sub>h" "of_complex x" H1] 1 2
          using `of_complex x \<in> unit_disc` False
          by simp
        have "is_imag y"
        proof (cases "y = 0")
          case True
          thus ?thesis
            by simp
        next
          case False
          hence "0\<^sub>h \<noteq> of_complex y"
            using of_complex_zero_iff[of y]
            by metis
          hence "H2 = poincare_line 0\<^sub>h (of_complex y)"
            using 1 2 `v \<in> unit_disc`
            using unique_poincare_line[of "0\<^sub>h" "of_complex y" H2] y
            by simp
          thus ?thesis
            using 1 `H1 = x_axis`
            using perpendicular_to_x_axis_y_axis[of y] False
            using perpendicular_to_x_axis[of H2]
            by simp
        qed
        show "P 0\<^sub>h (of_complex x) v"
        proof (cases "Im y \<ge> 0")
          case True
          thus ?thesis
            using axes[of x y] x y `is_imag y` `v \<in> unit_disc`
            by (simp add: cmod_eq_Im)
        next
          case False
          show ?thesis
          proof (rule conjugate)
            have "Im (cnj y) < 1"
              using `v \<in> unit_disc` y `is_imag y` eq_minus_cnj_iff_imag[of y]
              by (simp add: cmod_eq_Im)
            thus "P (conjugate 0\<^sub>h) (conjugate (of_complex x)) (conjugate v)"
              using `is_real x` eq_cnj_iff_real[of x] y `is_imag y`
              using axes[OF x, of "cnj y"] False
              by simp
            show "0\<^sub>h \<in> unit_disc" "of_complex x \<in> unit_disc" "v \<in> unit_disc"
              by (simp, fact+)
          qed
        qed
      qed
    qed
  next
    show "z \<in> unit_disc" "u \<in> unit_disc"
      by fact+
  next
    fix M u v
    assume *: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc"
    assume **: "?P (moebius_pt M u) (moebius_pt M v)"
    show "?P u v"
    proof safe
      fix w H1 H2
      assume ***: "is_poincare_line H1" "is_poincare_line H2" "perpendicular H1 H2"
                  "u \<in> circline_set H1" "u \<in> circline_set H2"
                  "v \<in> circline_set H1" "w \<in> circline_set H2" "w \<in> unit_disc"
      thus "P u v w"
        using moebius[of M u v w] *
        using **[rule_format, of "moebius_circline M H1" "moebius_circline M H2" "moebius_pt M w"]
        by simp
    qed
  qed
  thus ?thesis
    using assms
    by blast
qed

lemma wlog_perpendicular_foot:
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc" "z \<in> unit_disc"
  assumes perpendicular: "u \<noteq> v" "is_poincare_line H" "perpendicular (poincare_line u v) H"
  assumes "z \<in> circline_set (poincare_line u v) \<inter> circline_set H" "w \<in> circline_set H"
  assumes axes: "\<And> u v w. \<lbrakk>is_real u; 0 < Re u; Re u < 1; is_real v; -1 < Re v; Re v < 1; Re u \<noteq> Re v; is_imag w; 0 \<le> Im w; Im w < 1\<rbrakk> \<Longrightarrow> P 0\<^sub>h (of_complex u) (of_complex v) (of_complex w)"
  assumes moebius: "\<And> M z u v w. \<lbrakk>unit_disc_fix M; u \<in> unit_disc; v \<in> unit_disc; w \<in> unit_disc; z \<in> unit_disc; P (moebius_pt M z) (moebius_pt M u) (moebius_pt M v) (moebius_pt M w) \<rbrakk> \<Longrightarrow> P z u v w"
  assumes conjugate: "\<And> z u v w. \<lbrakk>u \<in> unit_disc; v \<in> unit_disc; w \<in> unit_disc; P (conjugate z) (conjugate u) (conjugate v) (conjugate w) \<rbrakk> \<Longrightarrow> P z u v w"
  assumes perm: "P z v u w \<Longrightarrow> P z u v w"
  shows "P z u v w"
proof-
  obtain m n where mn: "m = u \<or> m = v" "n = u \<or> n = v" "m \<noteq> n" "m \<noteq> z"
    using `u \<noteq> v`
    by auto

  have "n \<in> circline_set (poincare_line z m)"
    using `z \<in> circline_set (poincare_line u v) \<inter> circline_set H`
    using mn
    using unique_poincare_line[of z m "poincare_line u v", symmetric] in_disc
    by auto

  have "\<forall> n. n \<in> unit_disc \<and> m \<noteq> n \<and> n \<in> circline_set (poincare_line z m) \<and> m \<noteq> z \<longrightarrow> P z m n w" (is "?Q z m w")
  proof (rule wlog_perpendicular_axes[where P="?Q"])
    show "is_poincare_line (poincare_line u v)"
      using `u \<noteq> v`
      by auto
  next
    show "is_poincare_line H"
      by fact
  next
    show "m \<in> unit_disc" "m \<in> circline_set (poincare_line u v)"
      using mn in_disc
      by auto
  next
    show "w \<in> unit_disc" "z \<in> unit_disc"
      by fact+
  next
    show "z \<in> circline_set (poincare_line u v) \<inter> circline_set H"
      by fact
  next
    show "perpendicular (poincare_line u v) H"
      by fact
  next
    show "w \<in> circline_set H"
      by fact
  next
    fix x y
    assume xy: "is_real x" "0 \<le> Re x" "Re x < 1" "is_imag y" "0 \<le> Im y" "Im y < 1"
    show "?Q 0\<^sub>h (of_complex x) (of_complex y)"
    proof safe
      fix n
      assume "n \<in> unit_disc" "of_complex x \<noteq> n"
      assume "n \<in> circline_set (poincare_line 0\<^sub>h (of_complex x))" "of_complex x \<noteq> 0\<^sub>h"
      hence "n \<in> circline_set x_axis"
        using poincare_line_0_real_is_x_axis[of "of_complex x"] xy
        by (auto simp add: circline_set_x_axis)
      then obtain n' where n': "n = of_complex n'"
        using inf_or_of_complex[of n] `n \<in> unit_disc`
        by auto
      hence "is_real n'"
        using `n \<in> circline_set x_axis`
        using of_complex_inj
        unfolding circline_set_x_axis
        by auto
      hence "-1 < Re n'" "Re n' < 1"
        using `n \<in> unit_disc` n'
        by (auto simp add: cmod_eq_Re)

      have "Re n' \<noteq> Re x"
        using complex.expand[of n' x] `is_real n'` `is_real x` `of_complex x \<noteq> n` n'
        by auto

      have "Re x > 0"
        using xy `of_complex x \<noteq> 0\<^sub>h`
        by (cases "Re x = 0", auto simp add: complex.expand)

      show "P 0\<^sub>h (of_complex x) n (of_complex y)"
        using axes[of x n' y] xy n' `Re x > 0` `is_real n'` `-1 < Re n'` `Re n' < 1` `Re n' \<noteq> Re x`
        by simp
    qed
  next
    fix M u v w
    assume 1: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
    assume 2: "?Q (moebius_pt M u) (moebius_pt M v) (moebius_pt M w)"
    show "?Q u v w"
    proof safe
      fix n
      assume "n \<in> unit_disc" "v \<noteq> n" "n \<in> circline_set (poincare_line u v)" "v \<noteq> u"
      thus "P u v n w"
        using moebius[of M v n w u] 1 2[rule_format, of "moebius_pt M n"]
        by fastforce
    qed
  next
    fix u v w
    assume 1: "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
    assume 2: "?Q (conjugate u) (conjugate v) (conjugate w)"
    show "?Q u v w"
    proof safe
      fix n
      assume "n \<in> unit_disc" "v \<noteq> n" "n \<in> circline_set (poincare_line u v)" "v \<noteq> u"
      thus "P u v n w"
        using conjugate[of v n w u] 1 2[rule_format, of "conjugate n"]
        using conjugate_inj
        by auto
    qed
  qed
  thus ?thesis
    using mn in_disc `n \<in> circline_set (poincare_line z m)` perm
    by auto
qed

lemma perpendicular_to_x_axis_intersects_x_axis:
  assumes "is_poincare_line H" "perpendicular_to_x_axis H"
  shows "intersects_x_axis H"
  using assms hermitean_elems
  by (transfer, transfer, auto simp add: cmod_eq_Re)


lemma perpendicular_intersects:
  assumes "is_poincare_line H1" "is_poincare_line H2"
  assumes "perpendicular H1 H2"
  shows "\<exists> z. z \<in> unit_disc \<and> z \<in> circline_set H1 \<inter> circline_set H2" (is "?P' H1 H2")
proof-
  have "\<forall> H2. is_poincare_line H2 \<and> perpendicular H1 H2 \<longrightarrow> ?P' H1 H2" (is "?P H1")
  proof (rule wlog_line_x_axis)
    show "?P x_axis"
    proof safe
      fix H2
      assume "is_poincare_line H2" "perpendicular x_axis H2"
      thus "\<exists>z. z \<in> unit_disc \<and> z \<in> circline_set x_axis \<inter> circline_set H2"
        using perpendicular_to_x_axis[of H2]
        using perpendicular_to_x_axis_intersects_x_axis[of H2]
        using intersects_x_axis_iff[of H2]
        by auto
    qed
  next
    fix M
    assume "unit_disc_fix M"
    assume *: "?P (moebius_circline M H1)"
    show "?P H1"
    proof safe
      fix H2
      assume "is_poincare_line H2" "perpendicular H1 H2"
      then obtain z where "z \<in> unit_disc" "z \<in> circline_set (moebius_circline M H1) \<and> z \<in> circline_set (moebius_circline M H2)"
        using *[rule_format, of "moebius_circline M H2"] `unit_disc_fix M`
        by auto
      thus "\<exists>z. z \<in> unit_disc \<and> z \<in> circline_set H1 \<inter> circline_set H2"
        using `unit_disc_fix M`
        by (rule_tac x="moebius_pt (-M) z" in exI, auto) (metis moebius_pt_eq_I)
    qed
  next
    show "is_poincare_line H1"
      by fact
  qed
  thus ?thesis
    using assms
    by auto
qed


definition calc_perpendicular_to_x_axis_cmat :: "complex_vec \<Rightarrow> complex_mat" where
 [simp]: "calc_perpendicular_to_x_axis_cmat z =
     (let (z1, z2) = z
       in if z1*cnj z2 + z2*cnj z1 = 0 then
          (0, 1, 1, 0)
       else
          let A = z1*cnj z2 + z2*cnj z1;
              B = -(z1*cnj z1 + z2*cnj z2)
           in (A, B, B, A)
     )"

lift_definition calc_perpendicular_to_x_axis_clmat :: "complex_homo_coords \<Rightarrow> circline_mat" is calc_perpendicular_to_x_axis_cmat
  by (auto simp add: hermitean_def mat_adj_def mat_cnj_def Let_def split: if_split_asm)

lift_definition calc_perpendicular_to_x_axis :: "complex_homo \<Rightarrow> circline" is calc_perpendicular_to_x_axis_clmat
proof (transfer)
  fix z w
  assume "z \<noteq> vec_zero" "w \<noteq> vec_zero"
  obtain z1 z2 w1 w2 where zw: "z = (z1, z2)" "w = (w1, w2)"
    by (cases z, cases w, auto)
  assume "z \<approx>\<^sub>v w"
  then obtain k where *: "k \<noteq> 0" "w1 = k*z1" "w2 = k*z2"
    using zw
    by auto
  have "w1 * cnj w2 + w2 * cnj w1 = (k * cnj k) * (z1 * cnj z2 + z2 * cnj z1)"
    using *
    by (auto simp add: field_simps)
  moreover
  have "w1 * cnj w1 + w2 * cnj w2 = (k * cnj k) * (z1 * cnj z1 + z2 * cnj z2)"
    using *
    by (auto simp add: field_simps)
  ultimately
  show "circline_eq_cmat (calc_perpendicular_to_x_axis_cmat z) (calc_perpendicular_to_x_axis_cmat w)"
    using zw *
    apply (auto simp add: Let_def)
    apply (rule_tac x="Re (k * cnj k)" in exI, auto simp add: complex.expand complex_of_real_Re field_simps)
    done
qed

lemma calc_perpendicular_to_x_axis:
  assumes "z \<noteq> of_complex 1" "z \<noteq> of_complex (-1)"
  shows "z \<in> circline_set (calc_perpendicular_to_x_axis z) \<and>
         is_poincare_line (calc_perpendicular_to_x_axis z) \<and>
         perpendicular_to_x_axis (calc_perpendicular_to_x_axis z)"
  using assms
  unfolding circline_set_def perpendicular_def
proof (simp, transfer, transfer)
  fix z :: complex_vec
  obtain z1 z2 where z: "z = (z1, z2)"
    by (cases z, auto)
  assume **: "\<not> z \<approx>\<^sub>v of_complex_cvec 1" "\<not> z \<approx>\<^sub>v of_complex_cvec (- 1)"
  show "on_circline_cmat_cvec (calc_perpendicular_to_x_axis_cmat z) z \<and>
        is_poincare_line_cmat (calc_perpendicular_to_x_axis_cmat z) \<and>
        perpendicular_to_x_axis_cmat (calc_perpendicular_to_x_axis_cmat z)"
  proof (cases "z1*cnj z2 + z2*cnj z1 = 0")
    case True
    thus ?thesis
      using z
      by (simp add: vec_cnj_def hermitean_def mat_adj_def mat_cnj_def mult.commute)
  next
    case False
    hence "z2 \<noteq> 0"
      using z
      by auto
    hence "Re (z2 * cnj z2) \<noteq> 0"
      using `z2 \<noteq> 0`
      by (auto simp add: complex.expand)

    have "z1 \<noteq> -z2 \<and> z1 \<noteq> z2"
    proof (rule ccontr)
      assume "\<not> ?thesis"
      hence "z \<approx>\<^sub>v of_complex_cvec 1 \<or> z \<approx>\<^sub>v of_complex_cvec (-1)"
        using z `z2 \<noteq> 0`
        by auto
      thus False
        using **
        by auto
    qed

    let ?A = "z1*cnj z2 + z2*cnj z1" and ?B = "-(z1*cnj z1 + z2*cnj z2)"
    have "Re(z1*cnj z1 + z2*cnj z2) \<ge> 0"
      by auto
    hence "Re ?B \<le> 0"
      by (smt uminus_complex.simps(1))
    hence "abs (Re ?B) = - Re ?B"
      by auto
    also have "... = (Re z1)\<^sup>2 + (Im z1)\<^sup>2 + (Re z2)\<^sup>2 + (Im z2)\<^sup>2"
      by (simp add: power2_eq_square[symmetric])
    also have "... > abs (Re ?A)"
    proof (cases "Re ?A \<ge> 0")
      case False
      have "(Re z1 + Re z2)\<^sup>2 + (Im z1 + Im z2)\<^sup>2 > 0"
        using `z1 \<noteq> -z2 \<and> z1 \<noteq> z2`
        by (metis add.commute add.inverse_unique complex_neq_0 plus_complex.code plus_complex.simps)
      thus ?thesis
        using False
        by (simp add: power2_sum power2_eq_square field_simps)
    next
      case True
      have "(Re z1 - Re z2)\<^sup>2 + (Im z1 - Im z2)\<^sup>2 > 0"
        using `z1 \<noteq> -z2 \<and> z1 \<noteq> z2`
        by (meson complex_eq_iff right_minus_eq sum_power2_gt_zero_iff)
      thus ?thesis
        using True
        by (simp add: power2_sum power2_eq_square field_simps)
    qed
    finally
    have "abs (Re ?B) > abs (Re ?A)"
      .
    moreover
    have "cmod ?B = abs (Re ?B)" "cmod ?A = abs (Re ?A)"
      by (simp_all add: cmod_eq_Re)
    ultimately
    have "(cmod ?B)\<^sup>2 > (cmod ?A)\<^sup>2"
      by (smt power2_le_imp_le)
    thus ?thesis
      using z False
      by (simp_all add: Let_def hermitean_def mat_adj_def mat_cnj_def cmod_eq_Re vec_cnj_def field_simps)
  qed
qed

(* TODO: Uniqueness could also be shown *)

lemma ex_perpendicular:
  assumes "is_poincare_line H" "z \<in> unit_disc"
  shows "\<exists> H'. is_poincare_line H' \<and> perpendicular H H' \<and> z \<in> circline_set H'" (is "?P' H z")
proof-
  have "\<forall> z. z \<in> unit_disc \<longrightarrow> ?P' H z" (is "?P H")
  proof (rule wlog_line_x_axis)
    show "?P x_axis"
    proof safe
      fix z
      assume "z \<in> unit_disc"
      then have "z \<noteq> of_complex 1" "z \<noteq> of_complex (-1)"
        by auto
      thus "?P' x_axis z"
        using `z \<in> unit_disc`
        using calc_perpendicular_to_x_axis[of z] perpendicular_to_x_axis
        by (rule_tac x = "calc_perpendicular_to_x_axis z" in exI, auto)
    qed
  next
    fix M
    assume "unit_disc_fix M"
    assume *: "?P (moebius_circline M H)"
    show "?P H"
    proof safe
      fix z
      assume "z \<in> unit_disc"
      hence "moebius_pt M z \<in> unit_disc"
        using `unit_disc_fix M`
        by auto
      then obtain H' where *: "is_poincare_line H'" "perpendicular (moebius_circline M H) H'" "moebius_pt M z \<in> circline_set H'"
        using *
        by auto
      have h: "H = moebius_circline (-M) (moebius_circline M H)"   
        by auto
      show "?P' H z"
        using * `unit_disc_fix M`
        apply (subst h)
        apply (rule_tac x="moebius_circline (-M) H'" in exI)
        apply (simp del: moebius_circline_comp_inv_left)
        done
    qed
  qed fact
  thus ?thesis
    using assms
    by simp
qed

lemma ex_perpendicular_foot:
  assumes "is_poincare_line H" "z \<in> unit_disc"
  shows "\<exists> H'. is_poincare_line H' \<and> z \<in> circline_set H' \<and> perpendicular H H' \<and>
              (\<exists> z' \<in> unit_disc. z' \<in> circline_set H' \<inter> circline_set H)"
  using assms
  using ex_perpendicular[OF assms]
  using perpendicular_intersects[of H]
  by blast

lemma Pythagoras:
  assumes in_disc: "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc" "v \<noteq> w"
  assumes "distinct[u, v, w] \<longrightarrow> perpendicular (poincare_line u v) (poincare_line u w)"
  shows "cosh (poincare_distance v w) = cosh (poincare_distance u v) * cosh (poincare_distance u w)" (is "?P' u v w")
proof (cases "distinct [u, v, w]")
  case False
  thus "?thesis"
    using in_disc
    by (auto simp add: poincare_distance_sym)
next
  case True
  have "distinct [u, v, w] \<longrightarrow> ?P' u v w" (is "?P u v w")
  proof (rule wlog_perpendicular_axes[where P="?P"])
    show "is_poincare_line (poincare_line u v)" "is_poincare_line (poincare_line u w)"
      using `distinct [u, v, w]`
      by simp_all
  next
    show "perpendicular (poincare_line u v) (poincare_line u w)"
      using True assms
      by simp
  next
    show "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
      by fact+
  next
    show "v \<in> circline_set (poincare_line u v)" "w \<in> circline_set (poincare_line u w)"
         "u \<in> circline_set (poincare_line u v) \<inter> circline_set (poincare_line u w)"
      using `distinct [u, v, w]`
      by auto
  next
    fix x y
    assume x: "is_real x" "0 \<le> Re x" "Re x < 1"
    assume y: "is_imag y" "0 \<le> Im y" "Im y < 1"

    have "of_complex x \<in> unit_disc" "of_complex y \<in> unit_disc"
      using x y
      by (simp_all add: cmod_eq_Re cmod_eq_Im)

    show "?P 0\<^sub>h (of_complex x) (of_complex y)"
    proof
      assume "distinct [0\<^sub>h, of_complex x, of_complex y]"
      hence "x \<noteq> 0" "y \<noteq> 0"
        by auto

      let ?den1 = "1 - (cmod x)\<^sup>2" and ?den2 = "1 - (cmod y)\<^sup>2"
      have "?den1 > 0" "?den2 > 0"
        using x y
        by (simp_all add: cmod_eq_Re cmod_eq_Im abs_square_less_1)

      let ?d1 = "1 + 2 * (cmod x)\<^sup>2 / ?den1"
      have "cosh (poincare_distance 0\<^sub>h (of_complex x)) = ?d1"
        using `?den1 > 0`
        using poincare_distance_formula[of "0\<^sub>h" "of_complex x"] `of_complex x \<in> unit_disc`
        by simp

      moreover

      let ?d2 = "1 + 2 * (cmod y)\<^sup>2 / ?den2"
      have "cosh (poincare_distance 0\<^sub>h (of_complex y)) = ?d2"
        using `?den2 > 0` `of_complex y \<in> unit_disc`
        using poincare_distance_formula[of "0\<^sub>h" "of_complex y"]
        by simp

      moreover
      let ?den = "?den1 * ?den2"
      let ?d3 = "1 + 2 * (cmod (x - y))\<^sup>2 / ?den"
      have "cosh (poincare_distance (of_complex x) (of_complex y)) = ?d3"
        using `of_complex x \<in> unit_disc` `of_complex y \<in> unit_disc`
        using `?den1 > 0` `?den2 > 0`
        using poincare_distance_formula[of "of_complex x" "of_complex y"]
        by simp
      moreover
      have "?d1 * ?d2 = ?d3"
      proof-
        have "?d3 = ((1 - (cmod x)\<^sup>2) * (1 - (cmod y)\<^sup>2) + 2 * (cmod (x - y))\<^sup>2) / ?den"
          using `?den1 > 0` `?den2 > 0`
          by (subst add_num_frac, simp, simp)
        also have "... = (Re ((1 - x * cnj x) * (1 - y * cnj y) + 2 * (x - y)*cnj (x - y)) / ?den)"
          using `is_real x` `is_imag y`
          by ((subst cmod_square)+, simp)
        also have "... = Re (1 + x * cnj x * y * cnj y
                               + x * cnj x - 2 * y * cnj x - 2 * x * cnj y + y * cnj y) / ?den"
          by (simp add: field_simps)
        also have "... = Re ((1 + y * cnj y) * (1 + x * cnj x)) / ?den"
          using `is_real x` `is_imag y`
          by (simp add: field_simps)
        finally
        show ?thesis
          using `?den1 > 0` `?den2 > 0`
          apply (subst add_num_frac, simp)
          apply (subst add_num_frac, simp)
          apply simp
          apply (subst cmod_square)+
          apply (simp add: field_simps)
          done
      qed
      ultimately
      show "?P' 0\<^sub>h (of_complex x) (of_complex y)"
        by simp
    qed
  next
    fix M u v w
    assume 1: "unit_disc_fix M" "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
    assume 2: "?P (moebius_pt M u) (moebius_pt M v) (moebius_pt M w)"
    show "?P u v w"
      using 1 2
      by auto
  next
    fix u v w
    assume 1:  "u \<in> unit_disc" "v \<in> unit_disc" "w \<in> unit_disc"
    assume 2: "?P (conjugate u) (conjugate v) (conjugate w)"
    show "?P u v w"
      using 1 2
      by (auto simp add: conjugate_inj)
  qed
  thus ?thesis
    using True
    by simp
qed

(* ------------------------------------------------------------------ *)
section{* Lifting of Poincare *}
(* ------------------------------------------------------------------ *)

typedef p_point = "{z. z \<in> unit_disc}"
  using zero_in_unit_disc
  by (rule_tac x="0\<^sub>h" in exI, simp)

setup_lifting type_definition_p_point

lift_definition p_zero :: "p_point" is "0\<^sub>h"
  by (rule zero_in_unit_disc)

(* Another layer of lifting - can be done at the end *)
typedef p_line = "{H. is_poincare_line H}"
  by (rule_tac x="x_axis" in exI, simp)

setup_lifting type_definition_p_line

lift_definition p_incident :: "p_line \<Rightarrow> p_point \<Rightarrow> bool" is on_circline
  done

definition p_points :: "p_line \<Rightarrow> p_point set" where
  "p_points l = {p. p_incident l p}"

typedef p_isometry = "{f. unit_disc_fix_f f}"
  by (rule_tac x="id" in exI, simp add: unit_disc_fix_f_def, rule_tac x="id_moebius" in exI, simp)

setup_lifting type_definition_p_isometry

lift_definition p_isometry_pt :: "p_isometry \<Rightarrow> p_point \<Rightarrow> p_point" is "\<lambda> f p. f p"
  using unit_disc_fix_f_unit_disc
  by auto

lift_definition p_isometry_line :: "p_isometry \<Rightarrow> p_line \<Rightarrow> p_line" is "\<lambda> f l. unit_disc_fix_f_circline f l"
proof-
  fix f H
  assume "unit_disc_fix_f f" "is_poincare_line H"
  then obtain M where "unit_disc_fix M" and *: "f = moebius_pt M \<or> f = moebius_pt M \<circ> conjugate"
    unfolding unit_disc_fix_f_def
    by auto
  show "is_poincare_line (unit_disc_fix_f_circline f H)"
    using *
  proof
    assume "f = moebius_pt M"
    thus ?thesis
      using `unit_disc_fix M` `is_poincare_line H`
      using unit_disc_fix_f_circline_direct[of M f H]
      by auto
  next
    assume "f = moebius_pt M \<circ> conjugate"
    thus ?thesis
      using `unit_disc_fix M` `is_poincare_line H`
      using unit_disc_fix_f_circline_indirect[of M f H]
      by auto
  qed
qed

lift_definition p_colinear :: "p_point set \<Rightarrow> bool" is poincare_colinear
  done

lemma [simp]: "p_colinear (p_isometry_pt M ` A) \<longleftrightarrow> p_colinear A"
  by transfer (auto simp add: unit_disc_fix_f_def image_comp[symmetric])

lift_definition p_of_complex :: "complex \<Rightarrow> p_point" is "\<lambda> z. if cmod z < 1 then of_complex z else 0\<^sub>h"
  by auto

lift_definition p_x_axis :: "p_line" is x_axis
  by simp

lift_definition p_line :: "p_point \<Rightarrow> p_point \<Rightarrow> p_line" is poincare_line
proof-
  fix u v
  show "is_poincare_line (poincare_line u v)"
  proof (cases "u \<noteq> v")
    case True
    thus ?thesis
      by simp
  next
    (* this branch must work only for formal reasons *)
    case False
    thus ?thesis
      by (transfer, transfer, auto simp add: hermitean_def mat_adj_def mat_cnj_def split: if_split_asm)
  qed
qed

(* Just to test if this works *)
lemma p_on_line:
  assumes "z \<noteq> w"
  shows "p_incident (p_line z w) z"
        "p_incident (p_line z w) w"
  using assms
  by (transfer, simp)+

lemma
  assumes "u \<noteq> v"
  shows "\<exists>! l. {u, v} \<subseteq> p_points l"
  using assms
  apply (rule_tac a="p_line u v" in ex1I, auto simp add: p_points_def p_on_line)
  apply (transfer, simp add: unique_poincare_line)
  done

lemma
  assumes "p_zero \<in> p_points l" "u \<in> p_points l" "u \<noteq> p_zero" "u \<in> p_points p_x_axis"
  shows "l = p_x_axis"
  using assms
  unfolding p_points_def
  apply simp
  apply transfer
  using is_poincare_line_0_real_is_x_axis inf_notin_unit_disc
  unfolding circline_set_def
  by blast

lift_definition p_dist :: "p_point \<Rightarrow> p_point \<Rightarrow> real" is poincare_distance
  done

definition p_congruent :: "p_point \<Rightarrow> p_point \<Rightarrow> p_point \<Rightarrow> p_point \<Rightarrow> bool" where
  [simp]: "p_congruent u v u' v' \<longleftrightarrow> p_dist u v = p_dist u' v'"

lemma
  assumes "p_dist u v = p_dist u' v'"
  assumes "p_dist v w = p_dist v' w'"
  assumes "p_dist u w = p_dist u' w'"
  shows "\<exists> f. p_isometry_pt f u = u' \<and> p_isometry_pt f v = v' \<and> p_isometry_pt f w = w'"
  using assms
  apply transfer
  using unit_disc_fix_f_congruent_triangles
  by auto

instantiation p_point :: metric_space
begin
definition "dist_p_point = p_dist"
definition "(uniformity_p_point :: (p_point \<times> p_point) filter) = (INF e:{0<..}. principal {(x, y). dist_class.dist x y < e})"
definition "open_p_point (U :: p_point set) = (\<forall> x \<in> U. eventually (\<lambda>(x', y). x' = x \<longrightarrow> y \<in> U) uniformity)"
instance
proof
  fix x y :: p_point
  show "(dist_class.dist x y = 0) = (x = y)"
    unfolding dist_p_point_def
    by (transfer, simp add: poincare_distance_eq_0_iff)
next
  fix x y z :: p_point
  show "dist_class.dist x y \<le> dist_class.dist x z + dist_class.dist y z"
    unfolding dist_p_point_def                 
    apply transfer
    using poincare_distance_triangle_inequality poincare_distance_sym
    by metis
qed (simp_all add: open_p_point_def uniformity_p_point_def)
end                                                                       

lift_definition p_between :: "p_point \<Rightarrow> p_point \<Rightarrow> p_point \<Rightarrow> bool" is poincare_between
  done

end                                                                                                  
