(*    Title:              SatSolverVerification/KrsticGoel.thy
      ID:                 $Id: KrsticGoel.thy,v 1.5 2008/11/13 16:09:44 filipmaric Exp $
      Author:             Filip Maric
      Maintainer:         Filip Maric <filip at matf.bg.ac.yu>
*)

header{* Transition system of Krsti\' c and Goel. *}
theory KrsticGoelRF
imports SatSolverVerification
begin

text{* This theory formalizes the transition rule system given by
Krsti\' c and Goel in \cite{KrsticGoel}. Some rules of the system are 
generalized a bit, so that the system can model some more general solvers 
(e.g., SMT solvers). *}

(******************************************************************************)
subsection{* Specification *}
(******************************************************************************)

record State = 
"getFl" :: Formula
"getM" :: LiteralTrail
"getConflictFlag" :: bool
"getC" :: Clause
"getLearntFlag" :: bool

text{* Notice that the given UnitPropagate description is weaker than
in original \cite{KrsticGoel} paper. Namely, propagation can be done
over a clause that is not a member of the formula, but is entailed by
it. The condition imposed on the variable of the unit literal is
necessary to ensure the termination. *}

text{* Notice that Decide requires exhaustive unit propagation with formulae of F *}
definition
appliedDecide:: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"appliedDecide stateA stateB F0 decisionVars == 
  \<exists> l. 
        (var l) \<in> decisionVars \<and> 
        \<not> l el (elements (getM stateA)) \<and> 
        \<not> opposite l el (elements (getM stateA)) \<and>
        \<not> (\<exists> uc. uc el F0 @ (getFl stateA) \<and> (\<exists> ul. isUnitClause uc ul (elements (getM stateA)))) \<and> 

        getFl stateB = getFl stateA \<and>
        getM stateB = getM stateA @ [(l, True)] \<and> 
        getConflictFlag stateB = getConflictFlag stateA \<and>
        getC stateB = getC stateA \<and> 
        getLearntFlag stateB = getLearntFlag stateA
"
definition
applicableDecide :: "State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"applicableDecide state F0 decisionVars == \<exists> state'. appliedDecide state state' F0 decisionVars"

definition
appliedUnitPropagate :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"appliedUnitPropagate stateA stateB F0 decisionVars == 
  \<exists> (uc::Clause) (ul::Literal). 
        uc el F0 @ (getFl stateA) \<and>
        isUnitClause uc ul (elements (getM stateA))  \<and>

       getFl stateB = getFl stateA \<and>
       getM stateB = getM stateA @ [(ul, False)] \<and> 
       getConflictFlag stateB = getConflictFlag stateA \<and>
       getC stateB = getC stateA \<and> 
       getLearntFlag stateB = getLearntFlag stateA
"
definition
applicableUnitPropagate :: "State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"applicableUnitPropagate state F0 decisionVars == \<exists> state'. appliedUnitPropagate state state' F0 decisionVars"


definition
appliedConflict :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> bool"
where
"appliedConflict stateA stateB F0 == 
  \<exists> clause. 
       getConflictFlag stateA = False \<and>
       clause el F0 @ (getFl stateA) \<and> 
       clauseFalse clause (elements (getM stateA)) \<and> 

       getFl stateB = getFl stateA \<and>
       getM stateB = getM stateA \<and> 
       getConflictFlag stateB = True  \<and>
       getC stateB = clause \<and> 
       getLearntFlag stateB = getLearntFlag stateA
  "
definition
applicableConflict :: "State \<Rightarrow> Formula \<Rightarrow> bool"
where
"applicableConflict state F0 == \<exists> state'. appliedConflict state state' F0"

definition
appliedExplain :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> bool"
where
"appliedExplain stateA stateB F0 == 
   \<exists> l reason. 
       getConflictFlag stateA = True \<and>  
       l el getC stateA \<and> 
       reason el F0 @ (getFl stateA) \<and> 
       isReason reason (opposite l) (elements (getM stateA)) \<and> 

       getFl stateB = getFl stateA \<and>
       getM stateB = getM stateA \<and> 
       getConflictFlag stateB = True \<and>
       getC stateB = resolve (getC stateA) reason l \<and> 
       getLearntFlag stateB = getLearntFlag stateA
   "
definition
applicableExplain :: "State \<Rightarrow> Formula \<Rightarrow> bool"
where
"applicableExplain state F0 == \<exists> state'. appliedExplain state state' F0"

definition
appliedBackjumpAndLearn :: "State \<Rightarrow> State \<Rightarrow> bool"
where
"appliedBackjumpAndLearn stateA stateB == 
  \<exists> l level. 
       getConflictFlag stateA = True \<and> 
       isMinimalBackjumpLevel level l (getC stateA) (getM stateA) \<and>

       getFl stateB = getFl stateA @ [getC stateA] \<and>
       getM stateB = prefixToLevel level (getM stateA) @ [(l, False)] \<and>
       getConflictFlag stateB = False \<and>
       getC stateB = [] \<and> 
       getLearntFlag stateB = True
  "
definition
applicableBackjumpAndLearn :: "State \<Rightarrow> bool"
where
"applicableBackjumpAndLearn state == \<exists> state'. appliedBackjumpAndLearn state state'"

definition appliedForget :: "State \<Rightarrow> State \<Rightarrow> bool"
where
"appliedForget stateA stateB ==
  \<exists> Fc. getConflictFlag stateA = False \<and> getLearntFlag stateA = True \<and>
       Fc \<noteq> [] \<and> 
       set Fc \<subseteq> set (getFl stateA) \<and>
       (\<forall> c \<in> set Fc. \<not> (\<exists> l. isReason c l (elements (getM stateA)))) \<and> 
       
       getFl stateB = list_diff (getFl stateA) Fc \<and>
       getM stateB = getM stateA \<and>
       getConflictFlag stateB = getConflictFlag stateA \<and>
       getC stateB = getC stateA \<and> 
       getLearntFlag stateB = False"
definition
applicableForget :: "State \<Rightarrow> bool"
where
"applicableForget state == \<exists> state'. appliedForget state state'"

definition appliedRestart :: "State \<Rightarrow> State \<Rightarrow> bool"
where
"appliedRestart stateA stateB ==
  getConflictFlag stateA = False \<and> 
  getLearntFlag stateA = True \<and> 
  
  getFl stateB = getFl stateA \<and>
  getM stateB = prefixToLevel 0 (getM stateA) \<and>
  getConflictFlag stateB = getConflictFlag stateA \<and>
  getC stateB = getC stateA \<and> 
  getLearntFlag stateB = False"
definition
applicableRestart :: "State \<Rightarrow> bool"
where
"applicableRestart state == \<exists> state'. appliedRestart state state'"


text{* Solving starts with the empty trail, no learnt clauses and in non conflicting state. *}
definition
isInitialState :: "State \<Rightarrow> bool"
where
"isInitialState state == 
      getM state  = [] \<and>
      getFl state = [] \<and>
      getConflictFlag state = False \<and>
      getC state  = [] \<and> 
      getLearntFlag state = False"


text{* We define three different transition relations: 
  (i)   Full system with all the rules
  (ii)  System without the rule Restart
  (iii) System without the rule Forget
*}

text{* The full system *}
definition
transition :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"transition stateA stateB F0 decisionVars== 
     appliedDecide           stateA stateB F0 decisionVars \<or> 
     appliedUnitPropagate    stateA stateB F0 decisionVars \<or> 
     appliedConflict         stateA stateB F0 \<or> 
     appliedExplain          stateA stateB F0 \<or>
     appliedBackjumpAndLearn stateA stateB    \<or>
     appliedForget           stateA stateB    \<or>
     appliedRestart          stateA stateB"

text{* Transition relation is obtained by applying transition rules
iteratively. It is defined using a reflexive-transitive closure. *}
definition
"transitionRelation F0 decisionVars == ({(stateA, stateB). transition stateA stateB F0 decisionVars})^*"

text{* Transition chain *}
definition
isTransitionChain :: "State list \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
[simp]: "isTransitionChain l F0 decisionVars = 
  rel_chain l {(x, y). transition x y F0 decisionVars}"

text{* Final state is one in which no rules apply *}
definition
isFinalState :: "State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"isFinalState state F0 decisionVars == \<not> (\<exists> state'. transition state state' F0 decisionVars)"

text{* The system without the rule Restart *}
definition
transitionNoRestart :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"transitionNoRestart stateA stateB F0 decisionVars== 
     appliedDecide           stateA stateB F0 decisionVars \<or> 
     appliedUnitPropagate    stateA stateB F0 decisionVars \<or> 
     appliedConflict         stateA stateB F0 \<or> 
     appliedExplain          stateA stateB F0 \<or>
     appliedBackjumpAndLearn stateA stateB \<or> 
     appliedForget           stateA stateB"

text{* Transition relation is obtained by applying transition rules
iteratively. It is defined using a reflexive-transitive closure. *}
definition
"transitionRelationNoRestart F0 decisionVars == ({(stateA, stateB). transitionNoRestart stateA stateB F0 decisionVars})^*"

text{* Transition chains *}
definition
isTransitionChainNoRestart :: "State list \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
[simp]: "isTransitionChainNoRestart l F0 decisionVars = 
  rel_chain l {(x, y). transitionNoRestart x y F0 decisionVars}"

text{* The system without the rule forget *}
definition
transitionNoForget :: "State \<Rightarrow> State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"transitionNoForget stateA stateB F0 decisionVars== 
     appliedDecide           stateA stateB F0 decisionVars \<or> 
     appliedUnitPropagate    stateA stateB F0 decisionVars \<or> 
     appliedConflict         stateA stateB F0 \<or> 
     appliedExplain          stateA stateB F0 \<or>
     appliedBackjumpAndLearn stateA stateB \<or> 
     appliedRestart          stateA stateB"

text{* Transition relation is obtained by applying transition rules
iteratively. It is defined using a reflexive-transitive closure. *}
definition
"transitionRelationNoForget F0 decisionVars == ({(stateA, stateB). transitionNoForget stateA stateB F0 decisionVars})^*"

text{* Transition chains *}
definition
isTransitionChainNoForget :: "State list \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
[simp]: "isTransitionChainNoForget l F0 decisionVars = 
  rel_chain l {(x, y). transitionNoForget x y F0 decisionVars}"

    
text{* The following several lemmas establish conditions for applicability of different rules. *}
lemma applicableDecideCharacterization:
  fixes stateA::State
  shows "applicableDecide stateA F0 decisionVars = 
  (\<exists> l. 
        (var l) \<in> decisionVars \<and> 
        \<not> l el (elements (getM stateA)) \<and> 
        \<not> opposite l el (elements (getM stateA)) \<and> 
        \<not> (\<exists> uc. uc el F0 @ (getFl stateA) \<and> (\<exists> ul. isUnitClause uc ul (elements (getM stateA))))) 
  " (is "?lhs = ?rhs")
proof
  assume ?rhs
  then obtain l where 
    *: "(var l) \<in> decisionVars" "\<not> l el (elements (getM stateA))" "\<not> opposite l el (elements (getM stateA))"
    "\<not> (\<exists> uc. uc el F0 @ (getFl stateA) \<and> (\<exists> ul. isUnitClause uc ul (elements (getM stateA))))"
    unfolding applicableDecide_def
    by auto
  let ?stateB = "stateA\<lparr> getM := (getM stateA) @ [(l, True)] \<rparr>"
  from * have "appliedDecide stateA ?stateB F0 decisionVars"
    unfolding appliedDecide_def
    by auto
  thus ?lhs
    unfolding applicableDecide_def
    by auto
next
  assume ?lhs
  then obtain stateB l
    where "(var l) \<in> decisionVars" "\<not> l el (elements (getM stateA))"
    "\<not> opposite l el (elements (getM stateA))"
    "\<not> (\<exists> uc. uc el F0 @ (getFl stateA) \<and> (\<exists> ul. isUnitClause uc ul (elements (getM stateA))))"
    unfolding applicableDecide_def
    unfolding appliedDecide_def
    by auto
  thus ?rhs
    by auto
qed

lemma applicableUnitPropagateCharacterization:
  fixes stateA::State and F0::Formula
  shows "applicableUnitPropagate stateA F0 decisionVars = 
  (\<exists> (uc::Clause) (ul::Literal). 
        uc el F0 @ (getFl stateA) \<and> 
        isUnitClause uc ul (elements (getM stateA)))
  " (is "?lhs = ?rhs")
proof
  assume "?rhs"
  then obtain ul uc 
    where *: 
    "uc el F0 @ (getFl stateA)"
    "isUnitClause uc ul (elements (getM stateA))"
    unfolding applicableUnitPropagate_def
    by auto
  let ?stateB = "stateA\<lparr> getM := getM stateA @ [(ul, False)] \<rparr>"
  from * have "appliedUnitPropagate stateA ?stateB F0 decisionVars" 
    unfolding appliedUnitPropagate_def
    by auto
  thus ?lhs
    unfolding applicableUnitPropagate_def
    by auto
next
  assume ?lhs
  then obtain stateB uc ul
    where
    "uc el F0 @ (getFl stateA)"
    "isUnitClause uc ul (elements (getM stateA))"
    unfolding applicableUnitPropagate_def
    unfolding appliedUnitPropagate_def
    by auto
  thus ?rhs
    by auto
qed

lemma applicableBackjumpAndLearnCharacterization:
  fixes stateA::State
  shows "applicableBackjumpAndLearn stateA = 
     (\<exists> l level. 
         getConflictFlag stateA = True \<and> 
         isMinimalBackjumpLevel level l (getC stateA) (getM stateA)
     )" (is "?lhs = ?rhs")
proof
  assume "?rhs"
  then obtain l level
    where *: 
    "getConflictFlag stateA = True"
    "isMinimalBackjumpLevel level l (getC stateA) (getM stateA)"
    unfolding applicableBackjumpAndLearn_def
    by auto
  let ?stateB = "stateA\<lparr> getM := prefixToLevel level (getM stateA) @ [(l, False)], 
                         getConflictFlag := False, 
                         getFl := (getFl stateA) @ [getC stateA],
                         getC := [], 
                         getLearntFlag := True \<rparr>"
  from * have "appliedBackjumpAndLearn stateA ?stateB"
    unfolding appliedBackjumpAndLearn_def
    by auto
  thus "?lhs"
    unfolding applicableBackjumpAndLearn_def
    by auto
next
  assume "?lhs"
  then obtain stateB l level
    where  "getConflictFlag stateA = True"
    "isMinimalBackjumpLevel level l (getC stateA) (getM stateA)"
    unfolding applicableBackjumpAndLearn_def
    unfolding appliedBackjumpAndLearn_def
    by auto
  thus "?rhs"
    by auto
qed

lemma applicableExplainCharacterization:
  fixes stateA::State
  shows "applicableExplain stateA F0 = 
  (\<exists> l reason. 
       getConflictFlag stateA = True \<and>  
       l el getC stateA \<and> 
       reason el (F0 @ (getFl stateA)) \<and> 
       isReason reason (opposite l) (elements (getM stateA))
  )
  " (is "?lhs = ?rhs")
proof
  assume "?rhs"
  then obtain l reason
    where *: 
    "getConflictFlag stateA = True"
    "l el (getC stateA)" "reason el (F0 @ (getFl stateA))"
    "isReason reason (opposite l) (elements (getM stateA))"
    unfolding applicableExplain_def
    by auto
  let ?stateB = "stateA\<lparr> getC := resolve (getC stateA) reason l \<rparr>"
  from * have "appliedExplain stateA ?stateB F0"
    unfolding appliedExplain_def
    by auto
  thus "?lhs"
    unfolding applicableExplain_def
    by auto
next
  assume "?lhs"
  then obtain stateB l reason
    where
    "getConflictFlag stateA = True"
    "l el getC stateA" "reason el F0 @ (getFl stateA)"
    "isReason reason (opposite l) (elements (getM stateA))"
    unfolding applicableExplain_def
    unfolding appliedExplain_def
    by auto
  thus "?rhs"
    by auto
qed

lemma applicableConflictCharacterization:
  fixes stateA::State and F0::Formula
  shows "applicableConflict stateA F0 = 
    (\<exists> clause. 
       getConflictFlag stateA = False \<and>
       clause el F0 @ (getFl stateA) \<and> 
       clauseFalse clause (elements (getM stateA)))" (is "?lhs = ?rhs")
proof
  assume "?rhs"
  then obtain clause
    where *: 
    "getConflictFlag stateA = False" "clause el F0 @ (getFl stateA)" "clauseFalse clause (elements (getM stateA))"
    unfolding applicableConflict_def
    by auto
  let ?stateB = "stateA\<lparr> getC := clause, 
                         getConflictFlag := True \<rparr>"
  from * have "appliedConflict stateA ?stateB F0"
    unfolding appliedConflict_def
    by auto
  thus "?lhs"
    unfolding applicableConflict_def
    by auto
next
  assume "?lhs"
  then obtain stateB clause
    where
    "getConflictFlag stateA = False"
    "clause el F0 @ (getFl stateA)"
    "clauseFalse clause (elements (getM stateA))"
    unfolding applicableConflict_def
    unfolding appliedConflict_def
    by auto
  thus "?rhs"
    by auto
qed

lemma applicableForgetCharacterization:
  fixes stateA::State
  shows "applicableForget stateA = 
         (getConflictFlag stateA = False \<and> getLearntFlag stateA = True \<and> 
         (\<exists> c. c el (getFl stateA) \<and> \<not> (\<exists> l. isReason c l (elements (getM stateA)))))"
  (is "?lhs = ?rhs")
proof
  assume "?rhs"
  then obtain c
    where *: 
    "\<not> getConflictFlag stateA" "getLearntFlag stateA" "c el (getFl stateA)" "\<not> (\<exists> l. isReason c l (elements (getM stateA)))"
    unfolding applicableForget_def
    by auto
  let ?stateB = "stateA\<lparr> getFl := list_diff (getFl stateA) [c], getLearntFlag := False \<rparr>"
  from * have "appliedForget stateA ?stateB"
    unfolding appliedForget_def
    apply (rule_tac x="[c]" in exI)
    by simp
  thus "?lhs"
    unfolding applicableForget_def
    by auto
next
  assume "?lhs"
  then obtain Fc
    where
    "getConflictFlag stateA = False"
    "getLearntFlag stateA = True"
    "Fc \<noteq> []" "set Fc \<subseteq> set (getFl stateA)"
    "(\<forall>c\<in>set Fc. \<not> (\<exists>l. isReason c l (elements (getM stateA))))"
    unfolding applicableForget_def
    unfolding appliedForget_def
    by auto
  hence "hd Fc \<in> set (getFl stateA)" "\<not> (\<exists>l. isReason (hd Fc) l (elements (getM stateA)))"
    by auto
  thus "?rhs"
    using `getConflictFlag stateA = False` `getLearntFlag stateA = True`
    by auto
qed

lemma applicableRestart:
  fixes stateA::State
  shows "applicableRestart stateA = 
         (getLearntFlag stateA = True \<and> getConflictFlag stateA = False)"
  (is "?lhs = ?rhs")
proof
  let ?stateB = "stateA\<lparr> getM := prefixToLevel 0 (getM stateA), getLearntFlag := False \<rparr>"
  assume "?rhs"
  hence "appliedRestart stateA ?stateB"
    unfolding appliedRestart_def
    by auto
  thus "?lhs"
    unfolding applicableRestart_def
    by auto
next
  assume "?lhs"
  thus "?rhs"
    unfolding applicableRestart_def
    unfolding appliedRestart_def
    by auto
qed

text{* Final states are the ones where no rule is applicable. *}
lemma finalStateNonApplicable: 
  fixes state::State
  shows "isFinalState state F0 decisionVars = 
          (\<not> applicableDecide state F0 decisionVars \<and> 
           \<not> applicableUnitPropagate state F0 decisionVars \<and> 
           \<not> applicableBackjumpAndLearn state \<and> 
           \<not> applicableConflict state F0 \<and> 
           \<not> applicableExplain state F0 \<and> 
           \<not> applicableForget state \<and> 
           \<not> applicableRestart state)"
unfolding isFinalState_def
unfolding transition_def
unfolding applicableDecide_def
unfolding applicableUnitPropagate_def
unfolding applicableBackjumpAndLearn_def
unfolding applicableConflict_def
unfolding applicableExplain_def
unfolding applicableForget_def
unfolding applicableRestart_def
by auto

(******************************************************************************)
subsection{* Invariants *}
(******************************************************************************)
text{* Invariants that are relevant for the rest of correctness proof. *}
definition
invariantsHoldInState :: "State \<Rightarrow> Formula \<Rightarrow> Variable set \<Rightarrow> bool"
where
"invariantsHoldInState state F0 decisionVars == 
    InvariantVarsM (getM state) F0 decisionVars  \<and>
    InvariantVarsF (getFl state) F0 decisionVars  \<and>
    InvariantConsistent (getM state) \<and>
    InvariantUniq (getM state) \<and> 
    InvariantReasonClauses (F0 @ (getFl state)) (getM state) \<and>
    InvariantEquivalent F0 (F0 @ (getFl state)) \<and>
    InvariantCFalse (getConflictFlag state) (getM state) (getC state) \<and>
    InvariantCEntailed (getConflictFlag state) (F0 @ (getFl state)) (getC state)
"

text{* Invariants hold in initial states *}
lemma invariantsHoldInInitialState:
  fixes state :: State and F0 :: Formula
  assumes "isInitialState state" 
  shows "invariantsHoldInState state F0 decisionVars"
using assms
by (auto simp add:
  isInitialState_def 
  invariantsHoldInState_def 
  InvariantVarsM_def
  InvariantVarsF_def
  InvariantConsistent_def
  InvariantUniq_def
  InvariantReasonClauses_def
  InvariantEquivalent_def equivalentFormulae_def
  InvariantCFalse_def
  InvariantCEntailed_def
)


text{* Valid transitions preserve invariants. *}
lemma transitionsPreserveInvariants: 
  fixes stateA::State and stateB::State
  assumes "transition stateA stateB F0 decisionVars" and 
  "invariantsHoldInState stateA F0 decisionVars"
  shows "invariantsHoldInState stateB F0 decisionVars"
proof-
    from `invariantsHoldInState stateA F0 decisionVars`
    have 
      "InvariantVarsM (getM stateA) F0 decisionVars" and
      "InvariantVarsF (getFl stateA) F0 decisionVars" and
      "InvariantConsistent (getM stateA)" and
      "InvariantUniq (getM stateA)" and
      "InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)" and
      "InvariantEquivalent F0 (F0 @ (getFl stateA))" and
      "InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)" and
      "InvariantCEntailed (getConflictFlag stateA) (F0 @ (getFl stateA)) (getC stateA)"
      unfolding invariantsHoldInState_def
      by auto
  {
    assume "appliedDecide stateA stateB F0 decisionVars"
    then obtain l::Literal where
      "(var l) \<in> decisionVars"
      "\<not> literalTrue l (elements (getM stateA))"
      "\<not> literalFalse l (elements (getM stateA))"
      "getM stateB = getM stateA @ [(l, True)]"
      "getFl stateB = getFl stateA"
      "getConflictFlag stateB = getConflictFlag stateA"
      "getC stateB = getC stateA"
      unfolding appliedDecide_def
      by auto

    from `\<not> literalTrue l (elements (getM stateA))` `\<not> literalFalse l (elements (getM stateA))` 
    have *: "var l \<notin> vars (elements (getM stateA))"
      using varInValuationVars[of "l" "elements (getM stateA)"]
      by simp

    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `getFl stateB = getFl stateA`
	`getM stateB = getM stateA @ [(l, True)]` 
	`InvariantVarsM (getM stateA) F0 decisionVars`
	`var l \<in> decisionVars`
	InvariantVarsMAfterDecide [of "getM stateA" "F0" "decisionVars" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `getFl stateB = getFl stateA`
	`InvariantVarsF (getFl stateA) F0 decisionVars`
      by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using `getM stateB = getM stateA @ [(l, True)]` 
	`InvariantConsistent (getM stateA)`
	`var l \<notin> vars (elements (getM stateA))`
	InvariantConsistentAfterDecide[of "getM stateA" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `getM stateB = getM stateA @ [(l, True)]` 
	`InvariantUniq (getM stateA)`
	`var l \<notin> vars (elements (getM stateA))`
	InvariantUniqAfterDecide[of "getM stateA" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantReasonClauses (F0 @ (getFl stateB)) (getM stateB)"
      using `getFl stateB = getFl stateA`
	`getM stateB = getM stateA @ [(l, True)]` 
	`InvariantUniq (getM stateA)`
	`InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)`
      using InvariantReasonClausesAfterDecide[of "F0 @ getFl stateA" "getM stateA" "getM stateB" "l"]
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ (getFl stateB))"
      using `getFl stateB = getFl stateA`
      `InvariantEquivalent F0 (F0 @ (getFl stateA))`
      by simp
    moreover
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using `getM stateB = getM stateA @ [(l, True)]` 
	`getConflictFlag stateB = getConflictFlag stateA`
	`getC stateB = getC stateA`
	`InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)`
	InvariantCFalseAfterDecide[of "getConflictFlag stateA" "getM stateA" "getC stateA" "getM stateB" "l"]
      by simp
    moreover 
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ (getFl stateB)) (getC stateB)"
      using `getFl stateB = getFl stateA`
	`getConflictFlag stateB = getConflictFlag stateA`
	`getC stateB = getC stateA`
	`InvariantCEntailed (getConflictFlag stateA) (F0 @ (getFl stateA)) (getC stateA)`
      by simp
    ultimately
    have ?thesis
      unfolding invariantsHoldInState_def
      by auto
  }
  moreover
  {
    assume "appliedUnitPropagate stateA stateB F0 decisionVars"
    then obtain uc::Clause and ul::Literal where 
      "uc el F0 @ (getFl stateA)"
      "isUnitClause uc ul (elements (getM stateA))"
      "getFl stateB = getFl stateA"
      "getM stateB = getM stateA @ [(ul, False)]"
      "getConflictFlag stateB = getConflictFlag stateA"
      "getC stateB = getC stateA"
      unfolding appliedUnitPropagate_def
      by auto

    from `isUnitClause uc ul (elements (getM stateA))`
    have "ul el uc"
      unfolding isUnitClause_def
      by simp
    with `uc el F0 @ (getFl stateA)` 
    have "var ul \<in> decisionVars \<union> vars F0"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
      unfolding InvariantVarsF_def
      by auto

    from `var ul \<in> decisionVars \<union> vars F0`
    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `getFl stateB = getFl stateA` 
	`InvariantVarsM (getM stateA) F0 decisionVars`
	`getM stateB = getM stateA @ [(ul, False)]`
	InvariantVarsMAfterUnitPropagate[of "getM stateA" "F0" "decisionVars" "ul" "getM stateB"]
      by auto
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `getFl stateB = getFl stateA`
	`InvariantVarsF (getFl stateA) F0 decisionVars`
      by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using `InvariantConsistent (getM stateA)`
	`isUnitClause uc ul (elements (getM stateA))`
	`getM stateB = getM stateA @ [(ul, False)]`
	InvariantConsistentAfterUnitPropagate [of "getM stateA" "uc" "ul" "getM stateB"]
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `InvariantUniq (getM stateA)`
	`isUnitClause uc ul (elements (getM stateA))`
	`getM stateB = getM stateA @ [(ul, False)]`
	InvariantUniqAfterUnitPropagate [of "getM stateA" "uc" "ul" "getM stateB"]
      by simp
    moreover
    have "InvariantReasonClauses (F0 @ (getFl stateB)) (getM stateB)"
      using `getFl stateB = getFl stateA` 
	`InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)`
	`isUnitClause uc ul (elements (getM stateA))`
	`getM stateB = getM stateA @ [(ul, False)]`
	`uc el F0 @ (getFl stateA)`
	InvariantReasonClausesAfterUnitPropagate[of "F0 @ getFl stateA" "getM stateA" "uc" "ul" "getM stateB"]
        formulaEntailsItsClauses[of "uc" "F0 @ (getFl stateA)"]
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ (getFl stateB))"
      using `getFl stateB = getFl stateA` 
      `InvariantEquivalent F0 (F0 @ (getFl stateA))`
      by simp
    moreover 
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using `getM stateB = getM stateA @ [(ul, False)]` 
	`getConflictFlag stateB = getConflictFlag stateA`
	`getC stateB = getC stateA`
	`InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)`
	InvariantCFalseAfterUnitPropagate[of "getConflictFlag stateA" "getM stateA" "getC stateA" "getM stateB" "ul"]
      by simp
    moreover
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ (getFl stateB)) (getC stateB)"
      using `getFl stateB = getFl stateA` 
	`getConflictFlag stateB = getConflictFlag stateA`
	`getC stateB = getC stateA`
	`InvariantCEntailed (getConflictFlag stateA) (F0 @ (getFl stateA)) (getC stateA)`
      by simp
    ultimately
    have ?thesis
      unfolding invariantsHoldInState_def
      by auto
  }
  moreover
  {
    assume "appliedConflict stateA stateB F0"
    then obtain clause::Clause where
      "getConflictFlag stateA = False"
      "clause el F0 @ (getFl stateA)"
      "clauseFalse clause (elements (getM stateA))"
      "getFl stateB = getFl stateA"
      "getM stateB = getM stateA"
      "getConflictFlag stateB = True"
      "getC stateB = clause"
    unfolding appliedConflict_def
    by auto
  
    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
	`getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
	`getFl stateB = getFl stateA`
      by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using `InvariantConsistent (getM stateA)`
	`getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `InvariantUniq (getM stateA)`
	`getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantReasonClauses (F0 @ (getFl stateB)) (getM stateB)"
      using `InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)`
	`getFl stateB = getFl stateA`
	`getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ (getFl stateB))"
      using `InvariantEquivalent F0 (F0 @ (getFl stateA))`
	`getFl stateB = getFl stateA`
      by simp
    moreover
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using
      `clauseFalse clause (elements (getM stateA))`
      `getM stateB = getM stateA`
      `getConflictFlag stateB = True`
      `getC stateB = clause`
      unfolding InvariantCFalse_def
      by simp
    moreover 
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ (getFl stateB)) (getC stateB)"
      unfolding InvariantCEntailed_def
      using
      `getConflictFlag stateB = True`
      `clause el (F0 @ (getFl stateA))`
      `getFl stateB = getFl stateA`
      `getC stateB = clause`
      using formulaEntailsItsClauses[of "clause" "F0 @ (getFl stateA)"]
      by simp
    ultimately
    have ?thesis
      unfolding invariantsHoldInState_def
      by auto
  }
  moreover
  {
    assume "appliedExplain stateA stateB F0"
    then obtain l::Literal and reason::Clause where
	"getConflictFlag stateA = True"
        "l el getC stateA"
	"reason el F0 @ (getFl stateA)"
	"isReason reason (opposite l) (elements (getM stateA))"
	"getFl stateB = getFl stateA"
	"getM stateB = getM stateA"
	"getConflictFlag stateB = True"
	"getC stateB = resolve (getC stateA) reason l"
      unfolding appliedExplain_def
      by auto

    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
	`getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
	`getFl stateB = getFl stateA`
      by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using 
	`getM stateB = getM stateA`
	`InvariantConsistent (getM stateA)`
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using 
	`getM stateB = getM stateA`
	`InvariantUniq (getM stateA)`
      by simp
    moreover
    have "InvariantReasonClauses (F0 @ (getFl stateB)) (getM stateB)"
      using 
	`getFl stateB = getFl stateA`
	`getM stateB = getM stateA`
	`InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)`
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ (getFl stateB))"
      using 
	`getFl stateB = getFl stateA`
	`InvariantEquivalent F0 (F0 @ (getFl stateA))`
      by simp
    moreover 
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using 
	`InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)`
	`l el getC stateA`
	`isReason reason (opposite l) (elements (getM stateA))`
	`getM stateB = getM stateA`
	`getC stateB = resolve (getC stateA) reason l`
	`getConflictFlag stateA = True`
	`getConflictFlag stateB = True`
	InvariantCFalseAfterExplain[of "getConflictFlag stateA" "getM stateA" "getC stateA" "opposite l" "reason" "getC stateB"]
      by simp
    moreover 
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ (getFl stateB)) (getC stateB)"
      using 
	`InvariantCEntailed (getConflictFlag stateA) (F0 @ (getFl stateA)) (getC stateA)`
	`l el getC stateA`
	`isReason reason (opposite l) (elements (getM stateA))`
	`getFl stateB = getFl stateA`
	`getC stateB = resolve (getC stateA) reason l`
	`getConflictFlag stateA = True`
	`getConflictFlag stateB = True`
	`reason el F0 @ (getFl stateA)`
	InvariantCEntailedAfterExplain[of "getConflictFlag stateA" "F0 @ getFl stateA" "getC stateA" "reason" "getC stateB" "opposite l"]
        formulaEntailsItsClauses[of "reason" "F0 @ (getFl stateA)"]
      by simp
    ultimately
    have ?thesis
      unfolding invariantsHoldInState_def
      by auto
  }
  moreover
  {
    assume "appliedBackjumpAndLearn stateA stateB"
    then obtain l level where
      "getConflictFlag stateA = True"
      "isBackjumpLevel level l (getC stateA) (getM stateA)"
      "getFl stateB = getFl stateA @ [getC stateA]"
      "getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]"
      "getConflictFlag stateB = False"
      "getC stateB = []"
      unfolding appliedBackjumpAndLearn_def
      unfolding isMinimalBackjumpLevel_def
      by auto
    with `InvariantConsistent (getM stateA)` `InvariantUniq (getM stateA)`
      `InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)`
    have "isUnitClause (getC stateA) l (elements (prefixToLevel level (getM stateA)))"
      unfolding InvariantUniq_def
      unfolding InvariantConsistent_def
      unfolding InvariantCFalse_def
      using isBackjumpLevelEnsuresIsUnitInPrefix[of "getM stateA" "getC stateA" "level" "l"]
      by simp
    
    from `getConflictFlag stateA = True` `InvariantCEntailed (getConflictFlag stateA) (F0 @ (getFl stateA)) (getC stateA)` 
    have "formulaEntailsClause (F0 @ (getFl stateA)) (getC stateA)"
      unfolding InvariantCEntailed_def
      by simp

    from `isBackjumpLevel level l (getC stateA) (getM stateA)`
    have "isLastAssertedLiteral (opposite l) (oppositeLiteralList (getC stateA)) (elements (getM stateA))"
      unfolding isBackjumpLevel_def
      by simp
    hence "l el getC stateA"
      unfolding isLastAssertedLiteral_def
      using literalElListIffOppositeLiteralElOppositeLiteralList[of "l" "getC stateA"]
      by simp

    have "isPrefix (prefixToLevel level (getM stateA)) (getM stateA)"
      by (simp add:isPrefixPrefixToLevel)

    from `getConflictFlag stateA = True` `InvariantCFalse (getConflictFlag stateA) (getM stateA) (getC stateA)` 
    have "clauseFalse (getC stateA) (elements (getM stateA))"
      unfolding InvariantCFalse_def
      by simp
    hence "vars (getC stateA) \<subseteq> vars (elements (getM stateA))"
      using valuationContainsItsFalseClausesVariables[of "getC stateA" "elements (getM stateA)"]
      by simp
    moreover
    from `l el getC stateA`
    have "var l \<in> vars (getC stateA)"
      by auto
    ultimately
    have "var l \<in> vars F0 \<union> decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
      unfolding InvariantVarsM_def
      by (auto simp del: vars_def_clause)

    from `vars (getC stateA) \<subseteq> vars (elements (getM stateA))`
    `InvariantVarsM (getM stateA) F0 decisionVars`
    have "vars (getC stateA) \<subseteq> vars F0 \<union> decisionVars"
      unfolding InvariantVarsM_def
      by simp
    
    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
	`isUnitClause (getC stateA) l (elements (prefixToLevel level (getM stateA)))`
	`isPrefix (prefixToLevel level (getM stateA)) (getM stateA)`
	`var l \<in> vars F0 \<union> decisionVars`
	`formulaEntailsClause (F0 @ (getFl stateA)) (getC stateA)`
	`getFl stateB = getFl stateA @ [getC stateA]`
	`getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]`
	InvariantVarsMAfterBackjump[of "getM stateA" "F0" "decisionVars" "prefixToLevel level (getM stateA)" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
	`getFl stateB = getFl stateA @ [getC stateA]`
        using `vars (getC stateA) \<subseteq> vars F0 \<union> decisionVars`
	using InvariantVarsFAfterLearn [of "getFl stateA" "F0" "decisionVars" "getC stateA" "getFl stateB"]
	by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using `InvariantConsistent (getM stateA)`
	`isUnitClause (getC stateA) l (elements (prefixToLevel level (getM stateA)))`
	`isPrefix (prefixToLevel level (getM stateA)) (getM stateA)`
	`getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]`
	InvariantConsistentAfterBackjump[of "getM stateA" "prefixToLevel level (getM stateA)" "getC stateA" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `InvariantUniq (getM stateA)`
	`isUnitClause (getC stateA) l (elements (prefixToLevel level (getM stateA)))`
	`isPrefix (prefixToLevel level (getM stateA)) (getM stateA)`
	`getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]`
	InvariantUniqAfterBackjump[of "getM stateA" "prefixToLevel level (getM stateA)" "getC stateA" "l" "getM stateB"]
      by simp
    moreover
    have "InvariantReasonClauses (F0 @ (getFl stateB)) (getM stateB)"
      using `InvariantUniq (getM stateA)` `InvariantReasonClauses (F0 @ (getFl stateA)) (getM stateA)`
	`isUnitClause (getC stateA) l (elements (prefixToLevel level (getM stateA)))`
	`isPrefix (prefixToLevel level (getM stateA)) (getM stateA)`
	`formulaEntailsClause (F0 @ (getFl stateA)) (getC stateA)`
	`getFl stateB = getFl stateA @ [getC stateA]`
	`getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]`
	InvariantReasonClausesAfterBackjump[of "F0 @ (getFl stateA)" "getM stateA"
	"prefixToLevel level (getM stateA)" "getC stateA"  "l" "getM stateB"]
	InvariantReasonClausesAfterLearn[of "F0 @ (getFl stateA)" "prefixToLevel level (getM stateA) @ [(l, False)]" "getC stateA" "F0 @ (getFl stateB)"]
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ (getFl stateB))"
      using
	`InvariantEquivalent F0 (F0 @ (getFl stateA))`
	`formulaEntailsClause (F0 @ (getFl stateA)) (getC stateA)`
	`getFl stateB = getFl stateA @ [getC stateA]`
	InvariantEquivalentAfterLearn[of "F0" "F0 @ getFl stateA" "getC stateA" "F0 @ (getFl stateB)"]
      by simp
    moreover 
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using `getConflictFlag stateB = False`
      unfolding InvariantCFalse_def
      by simp
    moreover 
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ (getFl stateB)) (getC stateB)"
      using `getConflictFlag stateB = False`
      unfolding InvariantCEntailed_def
      by simp
    ultimately
    have ?thesis
      unfolding invariantsHoldInState_def
      by simp
  }
  moreover
  {
    assume "appliedForget stateA stateB"
    then obtain Fc where 
      "getConflictFlag stateA = False" "getLearntFlag stateA = True"
      "set Fc \<subseteq> set (getFl stateA)"
      "\<forall> c \<in> set Fc. (\<not> (\<exists> l. isReason c l (elements (getM stateA))))"
      "getFl stateB = list_diff (getFl stateA) Fc"
      "getM stateB = getM stateA"
      "getConflictFlag stateB = getConflictFlag stateA"
      "getC stateB = getC stateA"
      unfolding appliedForget_def
      by auto

    have "formulaEntailsFormula F0 (getFl stateA)"
      using `InvariantEquivalent F0 (F0 @ (getFl stateA))`
      unfolding InvariantEquivalent_def
      unfolding equivalentFormulae_def
      unfolding formulaEntailsFormula_def
      by (auto simp add: formulaTrueAppend)
    have "\<forall> c \<in> set Fc. formulaEntailsClause F0 c"
    proof-
      {
	fix c
	assume "c \<in> set Fc"
	hence "formulaEntailsClause F0 c"
	  using `formulaEntailsFormula F0 (getFl stateA)`
	  using `set Fc \<subseteq> set (getFl stateA)`
	  using formulaEntailsFormulaThatEntailsClause[of "F0" "getFl stateA" "c"]
	  using formulaEntailsItsClauses[of "c" "getFl stateA"]
	  by auto
      }
      thus ?thesis
	by simp
    qed
    
    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
      using `getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
      using `getFl stateB = list_diff (getFl stateA) Fc`
      unfolding InvariantVarsF_def
      by (auto simp add: listDiffSet)
    moreover
    have "InvariantConsistent (getM stateB)"
      using `InvariantConsistent (getM stateA)`
      using `getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `InvariantUniq (getM stateA)`
      using `getM stateB = getM stateA`
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ getFl stateB)"
      using `InvariantEquivalent F0 (F0 @ getFl stateA)`
      using `getFl stateB = list_diff (getFl stateA) Fc`
      unfolding InvariantEquivalent_def
      unfolding equivalentFormulae_def
      by (auto simp add: formulaTrueIffAllClausesAreTrue listDiffSet)
    moreover
    have "InvariantReasonClauses (F0 @ getFl stateB) (getM stateB)"
    proof-
      {
	fix literal::Literal
	assume 
	  "literal el (elements (getM stateB))" "\<not> literal el (markedElements (getM stateB))"
	have "\<exists>clause. formulaEntailsClause (F0 @ getFl stateB) clause \<and> isReason clause literal (elements (getM stateB))"
	proof-
	  from `literal el (elements (getM stateB))` `\<not> literal el (markedElements (getM stateB))`
	    `InvariantReasonClauses (F0 @ getFl stateA) (getM stateA)`
	    `getM stateB = getM stateA`
	  obtain clause where "formulaEntailsClause (F0 @ getFl stateA) clause" "isReason clause literal (elements (getM stateA))"
	    unfolding InvariantReasonClauses_def
	    by auto

	  have "equivalentFormulae (F0 @ getFl stateA) (F0 @ getFl stateB)"
	    using `InvariantEquivalent F0 (F0 @ getFl stateB)`
	    using `InvariantEquivalent F0 (F0 @ getFl stateA)`
	    unfolding InvariantEquivalent_def
	    using equivalentFormulaeSymmetry[of "F0" "F0 @ getFl stateA"]
	    using equivalentFormulaeTransitivity[of "F0 @ getFl stateA" "F0" "F0 @ getFl stateB"]
	    by simp
	  hence "formulaEntailsClause (F0 @ getFl stateB) clause"
	    using `formulaEntailsClause (F0 @ getFl stateA) clause`
	    using entailsClauseRelpacePartWithEquivalent[of "F0 @ getFl stateA" "F0 @ getFl stateB" "[]" "[]"]
	    by simp
	  with `isReason clause literal (elements (getM stateA))`
	    `getM stateB = getM stateA`
  	    `getFl stateB = list_diff (getFl stateA) Fc`
	  show ?thesis
	    by auto
	qed
      }
      thus ?thesis
	unfolding InvariantReasonClauses_def
	by auto
    qed
    moreover
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using `getConflictFlag stateA = False`
      using `getConflictFlag stateB = getConflictFlag stateA`
      unfolding InvariantCFalse_def
      by simp
    moreover
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ getFl stateB) (getC stateB)"
      using `getConflictFlag stateA = False`
      using `getConflictFlag stateB = getConflictFlag stateA`
      unfolding InvariantCEntailed_def
      by simp
    ultimately
    have "invariantsHoldInState stateB F0 decisionVars"
      unfolding invariantsHoldInState_def
      by simp
  }
  moreover
  {
    assume "appliedRestart stateA stateB"
    hence 
      "getConflictFlag stateA = False"
      "getFl stateB = getFl stateA"
      "getM stateB = prefixToLevel 0 (getM stateA)"
      "getConflictFlag stateB = getConflictFlag stateA"
      "getC stateB = getC stateA"
      unfolding appliedRestart_def
      by auto
    have "InvariantVarsM (getM stateB) F0 decisionVars"
      using `InvariantVarsM (getM stateA) F0 decisionVars`
      using `getM stateB = prefixToLevel 0 (getM stateA)`
      using InvariantVarsMAfterRestart[of "getM stateA" "F0" "decisionVars" "getM stateB"]
      by simp
    moreover
    have "InvariantVarsF (getFl stateB) F0 decisionVars"
      using `InvariantVarsF (getFl stateA) F0 decisionVars`
      using `getFl stateB = getFl stateA`
      by simp
    moreover
    have "InvariantConsistent (getM stateB)"
      using `InvariantConsistent (getM stateA)`
      using `getM stateB = prefixToLevel 0 (getM stateA)`
      using InvariantConsistentAfterRestart[of "getM stateA" "getM stateB"]
      by simp
    moreover
    have "InvariantUniq (getM stateB)"
      using `InvariantUniq (getM stateA)`
      using `getM stateB = prefixToLevel 0 (getM stateA)`
      using InvariantUniqAfterRestart[of "getM stateA" "getM stateB"]
      by simp
    moreover 
    have "InvariantReasonClauses (F0 @ getFl stateB) (getM stateB)"
      using `InvariantUniq (getM stateA)`
      using `InvariantReasonClauses (F0 @ getFl stateA) (getM stateA)`
      using `getM stateB = prefixToLevel 0 (getM stateA)`
      using `getFl stateB = getFl stateA`
      using InvariantReasonClausesAfterRestart[of "F0 @ (getFl stateA)" "getM stateA" "getM stateB"]
      by simp
    moreover
    have "InvariantEquivalent F0 (F0 @ getFl stateB)"
      using `InvariantEquivalent F0 (F0 @ getFl stateA)`
      using `getFl stateB = getFl stateA`
      by simp
    moreover
    have "InvariantCFalse (getConflictFlag stateB) (getM stateB) (getC stateB)"
      using `getConflictFlag stateA = False`
      using `getConflictFlag stateB = getConflictFlag stateA`
      unfolding InvariantCFalse_def
      by simp
    moreover
    have "InvariantCEntailed (getConflictFlag stateB) (F0 @ getFl stateB) (getC stateB)"
      using `getConflictFlag stateA = False`
      using `getConflictFlag stateB = getConflictFlag stateA`
      unfolding InvariantCEntailed_def
      by simp
    ultimately
    have "invariantsHoldInState stateB F0 decisionVars"
      unfolding invariantsHoldInState_def
      by simp
  }
  ultimately
  show ?thesis
    using `transition stateA stateB F0 decisionVars`
    unfolding transition_def
    by auto
qed

text{* The consequence is that invariants hold in all valid runs. *}
lemma invariantsHoldInValidRuns: 
  fixes F0 :: Formula and decisionVars :: "Variable set"
  assumes "invariantsHoldInState stateA F0 decisionVars" and
  "(stateA, stateB) \<in> transitionRelation F0 decisionVars"
  shows "invariantsHoldInState stateB F0 decisionVars"
using assms
using transitionsPreserveInvariants
using rtrancl_induct[of "stateA" "stateB" 
  "{(stateA, stateB). transition stateA stateB F0 decisionVars}" "\<lambda> x. invariantsHoldInState x F0 decisionVars"]
unfolding transitionRelation_def
by auto

lemma invariantsHoldInValidRunsFromInitialState:
  fixes F0 :: Formula and decisionVars :: "Variable set"
  assumes "isInitialState state0" 
  and "(state0, state) \<in> transitionRelation F0 decisionVars"
  shows "invariantsHoldInState state F0 decisionVars"
proof-
  from `isInitialState state0`
  have "invariantsHoldInState state0 F0 decisionVars"
    by (simp add:invariantsHoldInInitialState)
  with assms
  show ?thesis
    using invariantsHoldInValidRuns [of "state0"  "F0" "decisionVars" "state"]
    by simp
qed

text{*
 In the following text we will show that there are two kinds of states:
 \begin{enumerate}
  \item \textit{UNSAT} states where @{term "getConflictFlag state = True"}
  and @{term "getC state = []"}. 
  \item \textit{SAT} states where @{term "getConflictFlag state = False"}, 
  @{term "\<not> formulaFalse F0 (elements (getM state))"} and 
  @{term "vars (elements (getM state)) \<supseteq> decisionVars"}. 
 \end{enumerate}
  
 The soundness theorems claim that if \textit{UNSAT} state is reached
 the formula is unsatisfiable and if \textit{SAT} state is reached,
 the formula is satisfiable.

 Completeness theorems claim that every final state is either
 \textit{UNSAT} or \textit{SAT}. A consequence of this and soundness
 theorems, is that if formula is unsatisfiable the solver will finish
 in an \textit{UNSAT} state, and if the formula is satisfiable the
 solver will finish in a \textit{SAT} state.
*}


(******************************************************************************)
subsection{* Soundness *}
(******************************************************************************)

theorem soundnessForUNSAT:
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes
  "isInitialState state0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars"

  "getConflictFlag state = True" and
  "getC state = []"
  shows "\<not> satisfiable F0"
proof-
  from `isInitialState state0` `(state0, state) \<in> transitionRelation F0 decisionVars`
  have "invariantsHoldInState state F0 decisionVars" 
    using invariantsHoldInValidRunsFromInitialState
    by simp
  hence 
    "InvariantEquivalent F0 (F0 @ (getFl state))"
    "InvariantCEntailed (getConflictFlag state) (F0 @ (getFl state)) (getC state)"
    unfolding invariantsHoldInState_def
    by auto
  with `getConflictFlag state = True` `getC state = []`
  show ?thesis
    by (simp add:unsatReportExtensiveExplain)
qed

theorem soundnessForSAT:
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes 
  "vars F0 \<subseteq> decisionVars" and

  "isInitialState state0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars" and

  "getConflictFlag state = False"
  "\<not> formulaFalse (F0 @ (getFl state)) (elements (getM state))"
  "vars (elements (getM state)) \<supseteq> decisionVars"  
  shows 
  "model (elements (getM state)) F0"
proof-
  from `isInitialState state0` `(state0, state) \<in> transitionRelation F0 decisionVars`
  have "invariantsHoldInState state F0 decisionVars" 
    using invariantsHoldInValidRunsFromInitialState
    by simp
  hence 
    "InvariantConsistent (getM state)" 
    "InvariantEquivalent F0 (F0 @ (getFl state))"
    "InvariantVarsF (getFl state) F0 decisionVars"
    unfolding invariantsHoldInState_def
    by auto
  with assms
  show ?thesis
    using satReport[of "F0" "decisionVars" "F0 @ getFl state" "getM state"]
    unfolding InvariantVarsF_def
    by auto
qed



(**************************************************************************)
(*                          T E R M I N A T I O N                         *)
(**************************************************************************)
subsection{* Termination *}


text{* We want to show that every valid transition decreases a state
  with respect to a suitable ordering. *}

text{* First we show that $Decide$, $UnitPropagate$ and $Backjump$ rule
decrease the trail with respect to the restricted trail ordering
@{term ord_trail_lexDecisionRestricted}. Invariants ensure that trails are indeed
uniq, consistent and with finite variable sets. *}
lemma trailIsDecreasedByDeciedUnitPropagateAndBackjump:
  fixes stateA::State and stateB::State
  assumes "invariantsHoldInState stateA F0 decisionVars" and
  "appliedDecide stateA stateB F0 decisionVars \<or> appliedUnitPropagate stateA stateB F0 decisionVars \<or> appliedBackjumpAndLearn stateA stateB"
  shows "(getM stateB, getM stateA) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
proof-
  from `appliedDecide stateA stateB F0 decisionVars \<or> appliedUnitPropagate stateA stateB F0 decisionVars \<or> appliedBackjumpAndLearn stateA stateB`
    `invariantsHoldInState stateA F0 decisionVars` 
  have "invariantsHoldInState stateB F0 decisionVars"
      using transitionsPreserveInvariants[of "stateA" "stateB" "F0" "decisionVars"]
      unfolding transition_def
      by auto
    from `invariantsHoldInState stateA F0 decisionVars` 
    have *: "uniq (elements (getM stateA))" "vars (elements (getM stateA)) \<subseteq> vars F0 \<union> decisionVars"
      unfolding invariantsHoldInState_def
      unfolding InvariantVarsM_def
      unfolding InvariantUniq_def
      by auto
    from `invariantsHoldInState stateB F0 decisionVars` 
    have **: "uniq (elements (getM stateB))"  "vars (elements (getM stateB)) \<subseteq> vars F0 \<union> decisionVars"
      unfolding invariantsHoldInState_def
      unfolding InvariantVarsM_def
      unfolding InvariantUniq_def
      by auto
  {
    assume "appliedDecide stateA stateB F0 decisionVars"
    hence "(getM stateB, getM stateA) \<in> ord_trail_lexDecision"
      unfolding appliedDecide_def
      by (auto simp add:ord_trail_lexDecision_Append)
    with * ** 
    have "((getM stateB), (getM stateA)) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
      unfolding ord_trail_lexDecisionRestricted_def
      by auto
  }
  moreover
  {
    assume "appliedUnitPropagate stateA stateB F0 decisionVars"
    hence "(getM stateB, getM stateA) \<in> ord_trail_lexDecision"
      unfolding appliedUnitPropagate_def
      by (auto simp add:ord_trail_lexDecision_Append)
    with * ** 
    have "(getM stateB, getM stateA) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
      unfolding ord_trail_lexDecisionRestricted_def
      by auto
  }
  moreover
  {
    assume "appliedBackjumpAndLearn stateA stateB"
    then obtain l::Literal and level::nat
      where 
      "getConflictFlag stateA = True"
      "isBackjumpLevel level l (getC stateA) (getM stateA)"
      "getFl stateB = getFl stateA @ [getC stateA]"
      "getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]"
      "getConflictFlag stateB = False"
      "getC stateB = []"
      unfolding appliedBackjumpAndLearn_def
      unfolding isMinimalBackjumpLevel_def
      by auto
    
    from `isBackjumpLevel level l (getC stateA) (getM stateA)`
    have "isLastAssertedLiteral (opposite l) (oppositeLiteralList (getC stateA)) (elements (getM stateA))"
      unfolding isBackjumpLevel_def
      by simp
    hence "(opposite l) el elements (getM stateA)"
      unfolding isLastAssertedLiteral_def
      by simp
    hence "elementLevel (opposite l) (getM stateA) <= currentLevel (getM stateA)"
      by (simp add: elementLevelLeqCurrentLevel)
    moreover
    from `isBackjumpLevel level l (getC stateA) (getM stateA)`
    have "0 \<le> level" and "level < elementLevel (opposite l) (getM stateA)" 
      unfolding isBackjumpLevel_def 
      using `isLastAssertedLiteral (opposite l) (oppositeLiteralList (getC stateA)) (elements (getM stateA))`
      by auto
    ultimately 
    have "level < currentLevel (getM stateA)" 
      by simp
    with `0 \<le> level` `getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]`
    have "(getM stateB, getM stateA) \<in> ord_trail_lexDecision"
      by (simp add:ord_trail_lexDecision_Backjump)
    with * ** 
    have "(getM stateB, getM stateA) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
      unfolding ord_trail_lexDecisionRestricted_def
      by auto
  }
  ultimately
  show ?thesis
    using assms
    by auto
qed

text{* Next we show that $Conflict$ decreases the conflict flag in the @{term ord_bool_TF} ordering. *}
lemma conflictFlagIsDecreasedByConflict:
  fixes stateA::State and stateB::State
  assumes "appliedConflict stateA stateB F0"
  shows "(getConflictFlag stateB, getConflictFlag stateA) \<in> ord_bool_TF"
using assms
unfolding appliedConflict_def
unfolding ord_bool_TF_def
by auto

text{* Next we show that $Explain$ decreases the conflict clause with
respect to the @{term multLess} clause ordering. *}
lemma conflictClauseIsDecreasedByExplain:
  fixes stateA::State and stateB::State
  assumes "appliedExplain stateA stateB F0"
  shows 
  "(getC stateB, getC stateA) \<in> ord_clause_multTrail (getM stateA)"
proof-
  from `appliedExplain stateA stateB F0`
  obtain l::Literal and reason::Clause where
    "getConflictFlag stateA = True"
    "l el (getC stateA)"
    "isReason reason (opposite l) (elements (getM stateA))"
    "getFl stateB = getFl stateA"
    "getM stateB = getM stateA"
    "getConflictFlag stateB = True"
    "getC stateB = resolve (getC stateA) reason l"
    unfolding appliedExplain_def
    by auto
  thus ?thesis
    using ord_clause_multTrail_Resolve[of "opposite l" "getC stateA" "reason" "getM stateA"]
    by auto
qed

text{* Next, we show that $Forget$ decreases the learnt flag in the @{term ord_bool_FT} boolean ordering. *}
lemma learntFlagIsDecreasedByForget:
  fixes stateA::State and stateB::State
  assumes "appliedForget stateA stateB"
  shows 
  "(getLearntFlag stateB, getLearntFlag stateA) \<in> ord_bool_FT"
using assms
unfolding appliedForget_def
unfolding ord_bool_FT_def
by auto

text{* Next, we show that $Restart$ decreases the learnt flag in the @{term ord_bool_FT} boolean ordering. *}
lemma learntFlagIsDecreasedByRestart:
  fixes stateA::State and stateB::State
  assumes "appliedRestart stateA stateB"
  shows 
  "(getLearntFlag stateB, getLearntFlag stateA) \<in> ord_bool_FT"
using assms
unfolding appliedRestart_def
unfolding ord_bool_FT_def
by auto

text{* Next, we show that $BackjumpAndLearn$ decreases the formula in the @{term ord_formula_inclusion} formula ordering*}

lemma onlyDecideIncreasesLevel:
  assumes "transition stateA stateB F0 decisionVars"
          "currentLevel (getM stateA) < currentLevel (getM stateB)"
  shows "appliedDecide stateA stateB F0 decisionVars"
proof-
  {
    assume "appliedUnitPropagate stateA stateB F0 decisionVars"
    hence False
      unfolding appliedUnitPropagate_def
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
  }
  moreover
  {
    assume "appliedConflict stateA stateB F0"
    hence False
      unfolding appliedConflict_def
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      unfolding currentLevel_def
      by simp
  }
  moreover
  {
    assume "appliedExplain stateA stateB F0"
    hence False
      unfolding appliedExplain_def
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      unfolding currentLevel_def
      by auto
  }
  moreover
  {
    assume "appliedBackjumpAndLearn stateA stateB"
    then obtain l level where 
      "getM stateB = prefixToLevel level (getM stateA) @ [(l, False)]"
      unfolding appliedBackjumpAndLearn_def
      by auto
    hence False
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      using isPrefixPrefixToLevel[of "level" "getM stateA"]
      using currentLevelPrefix[of "prefixToLevel level (getM stateA)" "getM stateA"]
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
  }
  moreover
  {
    assume "appliedForget stateA stateB"
    hence False
      unfolding appliedForget_def
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      unfolding currentLevel_def
      by auto
  }
  moreover
  {
    assume "appliedRestart stateA stateB"
    hence False
      unfolding appliedRestart_def
      using `currentLevel (getM stateA) < currentLevel (getM stateB)`
      using currentLevelPrefixToLevel[of "0" "getM stateA"]
      by auto
  }
  ultimately
  show ?thesis
    using `transition stateA stateB F0 decisionVars`
    unfolding transition_def
    by blast
qed


lemma currentLevelContinuous:
  assumes 
  "l \<noteq> []" 
  "isTransitionChain l F0 decisionVars"
  "currentLevel (getM (hd l)) < level" "level < currentLevel (getM (last l))"
  shows
  "\<exists> m. 0 < m \<and> m < length l - 1 \<and> currentLevel (getM (l ! m)) = level"
using assms
proof (induct l)
  case Nil
  thus ?case
    by simp
next
  case (Cons s l')
  from 
    `currentLevel (getM (hd (s # l'))) < level`
    `level < currentLevel (getM (last (s # l')))`
  have "l' \<noteq> []"
    by auto

  have 
    "transition s (hd l') F0 decisionVars"  
    "isTransitionChain l' F0 decisionVars"
    using `isTransitionChain (s # l') F0 decisionVars`
    using rel_chain_sublist[of "s # l'" "[s]" "l'" "[]" "{(a, b). transition a b F0 decisionVars}"]
    using `l' \<noteq> []`
    by auto

  show ?case
  proof (cases "currentLevel (getM (hd l')) < level")
    case True
    with Cons(1)
      `level < currentLevel (getM (last (s # l')))` `l' \<noteq> []`
      `isTransitionChain l' F0 decisionVars`
    obtain m' where "m' < length l' - 1" "currentLevel (getM (l' ! m')) = level"
      by auto
    hence "Suc m' < length (s # l') - 1" "currentLevel (getM ((s # l') ! (Suc m'))) = level"
      by auto
    thus ?thesis
      by auto
  next
    case False
    hence "appliedDecide s (hd l') F0 decisionVars"
      using `currentLevel (getM (hd (s # l'))) < level`
      using `transition s (hd l') F0 decisionVars`
      using onlyDecideIncreasesLevel[of "s" "hd l'" F0 decisionVars]
      by simp
    hence "currentLevel (getM (hd l')) = currentLevel (getM s) + 1"
      unfolding appliedDecide_def
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
    hence "currentLevel (getM (hd l')) = level"
      using `currentLevel (getM (hd (s # l'))) < level`
      using `\<not> currentLevel (getM (hd l')) < level`
      by simp
    have "length l' > 1"
    proof (rule ccontr)
      assume "\<not> ?thesis"
      hence "length l' = 1"
	using `l' \<noteq> []`
	by (induct l') auto
      hence "last l' = hd l'"
	by (induct l') auto
      thus False
	using `currentLevel (getM (hd l')) = level`
	using `level < currentLevel (getM (last (s # l')))`
	using `l' \<noteq> []`
	by simp
    qed
    hence "1 < length (s # l') - 1" "currentLevel (getM ((s # l') ! 1)) = level"
      using `l' \<noteq> []`
      using `currentLevel (getM (hd l')) = level`
      by (auto simp add: hd_conv_nth)
    thus ?thesis
      by auto
  qed
qed

lemma currentLevelContinuousMiddle:
  assumes 
  "l \<noteq> []" 
  "isTransitionChain l F0 decisionVars"
  "m' < length l" "m' <= m''" "m'' < length l"
  "currentLevel (getM (l ! m')) < level" "level < currentLevel (getM (l ! m''))"
  shows
  "\<exists> m. m' < m \<and> m < m'' \<and> currentLevel (getM (l ! m)) = level"
proof-
  have "m' < m''"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "m' = m''"
      using `m' <= m''`
      by simp
    thus False
      using `currentLevel (getM (l ! m')) < level` 
      using `level < currentLevel (getM (l ! m''))`
      by simp
  qed

  let ?p = "list_before l m'"
  let ?c = "list_from_to l m' m''"
  let ?s = "list_after l m''"
  have "l = ?p @ ?c @ ?s"
    using `m'' > m'`
    using list_before_from_to_after[of "m'" "m''" "l"]
    by simp

  have tr: "isTransitionChain ?c F0 decisionVars"
    using `isTransitionChain l F0 decisionVars`
    using `l = ?p @ ?c @ ?s`
    using rel_chain_sublist(1)[of "l" "?p" "?c" "?s" "{(x, y). transition x y F0 decisionVars}"]
    unfolding isTransitionChain_def
    by blast

  show ?thesis
    using listPropertiesMiddle[of "l" "m'" "m''" "\<lambda> l. isTransitionChain l F0 decisionVars" "\<lambda> s. currentLevel (getM s) < level" "\<lambda> s. currentLevel (getM s) > level" "\<lambda> s. currentLevel (getM s) = level"]
    using `l \<noteq> []` `m' < m''` `m' < length l` `m'' < length l` 
    using `isTransitionChain (list_from_to l m' m'') F0 decisionVars`
    using `currentLevel (getM (l ! m')) < level` `level < currentLevel (getM (l ! m''))`
    using currentLevelContinuous
    by blast
qed

lemma lastVisitOfLevel:
  fixes l :: "State list"
  assumes 
  "l \<noteq> []"
  "isTransitionChain l F0 decisionVars"
  "\<exists> m < length l. currentLevel (getM (l ! m)) = level"
  "currentLevel (getM (last l)) > level"
  shows
  "\<exists> m < length l - 1. 
      currentLevel (getM (l ! m)) = level \<and> 
      (\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) > level)"
proof-
  obtain m 
    where "m < length l" "currentLevel (getM (l ! m)) = level" and
    *: "\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) \<noteq> level"
    using assms
    using lastElementWithAProperty[of "length l" "\<lambda> m. currentLevel (getM (l ! m)) = level"]
    by auto

  have "m < length l - 1"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    with `m < length l`
    have "m = length l - 1"
      by simp
    thus False
      using `currentLevel (getM (l ! m)) = level`
      using `currentLevel (getM (last l)) > level`
      using `l \<noteq> []`
      using last_conv_nth[of "l"]
      by auto
  qed

  have "\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) > level"
  proof-
    {
      fix m'
      assume "m < m'" "m' < length l" 
      hence "m' <= length l - 1"
	by auto
      have "currentLevel (getM (l ! m')) > level"
      proof (rule ccontr)
	assume "\<not> ?thesis"
	with * `m < m'` `m' < length l`
	have "currentLevel (getM (l ! m')) < level"
	  by auto
	then obtain m''
	  where "m' < m''" "m'' < length l - 1" "currentLevel (getM (l ! m'')) = level"
	  using `l \<noteq> []`
	  using `isTransitionChain l F0 decisionVars`
	  using `m' < length l`
	  using `m' <= length l - 1`
	  using `currentLevel (getM (last l)) > level`
	  using last_conv_nth[of "l"]
	  using currentLevelContinuousMiddle[of "l" "F0" "decisionVars" "m'" "length l - 1" "level"]
	  by auto
	with * show False
	  using `m < m'`
	  by auto
      qed
    }
    thus ?thesis
      by auto
  qed
  thus ?thesis
    using `m < length l - 1` `currentLevel (getM (l ! m)) = level`
    by auto
qed

lemma clausesAddedToFAreLearnt:
  assumes 
  "l \<noteq> []" "isTransitionChain l F0 decisionVars"
  "\<not> c el map remdups (F0 @ (getFl (hd l)))" "c el map remdups (F0 @ (getFl (last l)))"
  shows 
  "\<exists> m < length l - 1.  (remdups (getC (l ! m)) = c) \<and>
             (appliedBackjumpAndLearn (l ! m) (l ! (m+1)))"
using assms
proof (induct l)
  case Nil
  thus ?case
    by simp
next
  case (Cons s l')

  have "l' \<noteq> []"
    using `\<not> c el map remdups (F0 @ (getFl (hd (s # l'))))`
    using `c el map remdups (F0 @ (getFl (last (s # l'))))`
    by auto

  hence "transition s (hd l') F0 decisionVars"
    "isTransitionChain l' F0 decisionVars"
    using `isTransitionChain (s # l') F0 decisionVars`
    using rel_chain_sublist[of "s # l'" "[s]" "l'" "[]" "{(x, y). transition x y F0 decisionVars}"]
    by auto

  thus ?case
  proof (cases "appliedBackjumpAndLearn s (hd l') \<and> remdups (getC s) = c")
    case True
    thus ?thesis
      using `l' \<noteq> []`
      by (auto simp add: hd_conv_nth)
  next
    case False
    have "\<not> c el map remdups (F0 @ (getFl (hd l')))"
    proof (cases "appliedBackjumpAndLearn s (hd l')")
      case False
      thus ?thesis
	using `\<not> c el map remdups (F0 @ (getFl (hd (s # l'))))`
	using `transition s (hd l') F0 decisionVars`
	unfolding transition_def
	unfolding appliedDecide_def
	unfolding appliedUnitPropagate_def
	unfolding appliedConflict_def
	unfolding appliedExplain_def
	unfolding appliedForget_def
	unfolding appliedRestart_def
	by (auto simp add: listDiffSet)
    next
      case True
      with `\<not> (appliedBackjumpAndLearn s (hd l') \<and> remdups (getC s) = c)`
      have "remdups (getC s) \<noteq> c"
	by simp
      thus ?thesis
	using True
	using `\<not> c el map remdups (F0 @ (getFl (hd (s # l'))))`
	unfolding appliedBackjumpAndLearn_def
	by auto
    qed
    with `c el map remdups (F0 @ (getFl (last (s # l'))))`
    obtain m
      where "m < length l' - 1"
       "remdups (getC (l' ! m)) = c"
       "appliedBackjumpAndLearn (l' ! m) (l' ! (m + 1))"
      using Cons(1)
      using `l' \<noteq> []`
      using `isTransitionChain l' F0 decisionVars`
      by auto
    hence "m + 1 < length (s # l') - 1"
      "remdups (getC ((s # l') ! (m + 1))) = c"
      "appliedBackjumpAndLearn ((s # l') ! (m + 1)) ((s # l') ! (m + 2))"
      by auto
    thus ?thesis
      by auto
  qed
qed

lemma clausesAddedToFAreLearntMiddle:
  assumes "l \<noteq> []" "isTransitionChain l F0 decisionVars"
          "\<not> clause el map remdups (F0 @ (getFl (l ! m')))"
          "clause el map remdups (F0 @ (getFl (l ! m'')))"
          "m' <= m''" "m' < length l" "m'' < length l"
  shows "\<exists> m. m' <= m \<and> m < m'' \<and> remdups (getC (l ! m)) = clause \<and> 
  appliedBackjumpAndLearn (l ! m) (l ! (m + 1))"
proof-
  have "m' < m''"
    using assms
    by auto
  moreover

  let ?p = "list_before l m'"
  let ?c = "list_from_to l m' m''"
  let ?s = "list_after l m''"
  have "l = ?p @ ?c @ ?s"
    using `m'' > m'`
    using list_before_from_to_after[of "m'" "m''" "l"]
    by simp

  have "isTransitionChain ?c F0 decisionVars"
    using `isTransitionChain l F0 decisionVars`
    using `l = ?p @ ?c @ ?s`
    using rel_chain_sublist(1)[of "l" "?p" "?c" "?s" "{(x, y). transition x y F0 decisionVars}"]
    unfolding isTransitionChain_def
    by blast
  moreover
  have 
    "\<forall>l. l \<noteq> [] \<and>
       \<not> clause el (map remdups (F0 @ getFl (hd l))) \<and>
       clause el (map remdups (F0 @ getFl (last l))) \<and> isTransitionChain l F0 decisionVars \<longrightarrow>
       (\<exists>m\<ge>0. m < length l - 1 \<and>
              remdups (getC (l ! m)) = clause \<and> appliedBackjumpAndLearn (l ! m) (l ! (m + 1)))"
  proof-
    {
      fix l
      assume "l \<noteq> []" "\<not> clause el (map remdups (F0 @ getFl (hd l)))"
        "clause el (map remdups (F0 @ getFl (last l)))" "isTransitionChain l F0 decisionVars"
      then have "\<exists>m\<ge>0. m < length l - 1 \<and>
        remdups (getC (l ! m)) = clause \<and> appliedBackjumpAndLearn (l ! m) (l ! (m + 1))"
	using clausesAddedToFAreLearnt[of "l" "F0" "decisionVars" "clause"]
	by simp
    }
    thus ?thesis
      by auto
  qed
  ultimately
  show ?thesis
    using listRelationsMiddle[of "l" "m'" "m''" "\<lambda> l. isTransitionChain l F0 decisionVars"
    "\<lambda> s. \<not> clause el map remdups (F0 @ (getFl s))" "\<lambda> s. clause el map remdups (F0 @ (getFl s))"
    "\<lambda> x y. remdups (getC x) = clause \<and> appliedBackjumpAndLearn x y"]
    using `l \<noteq> []` `m' < length l` `m'' < length l`
    using `\<not> clause el map remdups (F0 @ (getFl (l ! m')))`
      `clause el map remdups (F0 @ (getFl (l ! m'')))`
    by blast
qed

lemma transitionsRetainPrefixToLowLevels:
  assumes
  "transition stateA stateB F0 decisionVars"
  "currentLevel (getM stateB) > level"
  shows
  "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
proof-
  {
    assume "appliedDecide stateA stateB F0 decisionVars"
    then obtain l where 
      "getM stateB = getM stateA @ [(l, True)]"
      unfolding appliedDecide_def
      by auto
    hence "currentLevel (getM stateB) = currentLevel (getM stateA) + 1"
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
    hence "currentLevel (getM stateA) >= level"
      using `level < currentLevel (getM stateB)`
      by simp
    
    have "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
    proof (cases "currentLevel (getM stateA) > level")
      case True
      thus ?thesis
	using prefixToLevelAppend[of "level" "getM stateA" "[(l, True)]"]
	using `getM stateB = getM stateA @ [(l, True)]`
	by simp
    next
      case False
      hence "currentLevel (getM stateA) = level"
	using `currentLevel (getM stateA) >= level`
	by simp

      hence "prefixToLevel level (getM stateA @ [(l, True)]) = 
	     getM stateA @ prefixToLevel_aux [(l, True)] level (currentLevel (getM stateA))"
	using prefixToLevelAppend[of "level" "getM stateA" "[(l, True)]"]
	by auto
      moreover
      have "prefixToLevel_aux [(l, True)] level (currentLevel (getM stateA)) = []"
	using `currentLevel (getM stateA) = level`
	by auto
      ultimately
      have "prefixToLevel level (getM stateA @ [(l, True)]) = getM stateA"
	by simp
      thus ?thesis
	using `getM stateB = getM stateA @ [(l, True)]`
	using `currentLevel (getM stateA) = level`
	using prefixToLevelCurrentLevel[of "getM stateA"]
	by simp
    qed
  }
  moreover
  {
    assume "appliedUnitPropagate stateA stateB F0 decisionVars"
    then obtain ul where "getM stateB = getM stateA @ [(ul, False)]"
      unfolding appliedUnitPropagate_def
      by auto
    hence "currentLevel (getM stateB) = currentLevel (getM stateA)"
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
    hence "currentLevel (getM stateA) > level"
      using `currentLevel (getM stateB) > level`
      by simp
    
    have "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
      using `getM stateB = getM stateA @ [(ul, False)]`
      using `currentLevel (getM stateA) > level`
      using prefixToLevelAppend[of "level" "getM stateA" "[(ul, False)]"]
      by simp
  }
  moreover
  {
    assume "appliedConflict stateA stateB F0 \<or> appliedExplain stateA stateB F0 \<or> appliedForget stateA stateB"
    hence "getM stateB = getM stateA"
      unfolding appliedConflict_def
      unfolding appliedExplain_def
      unfolding appliedForget_def
      by auto
    hence "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
      by auto
  }
  moreover
  {
    assume "appliedRestart stateA stateB"
    hence "currentLevel (getM stateB) = 0"
      unfolding appliedRestart_def
      using currentLevelPrefixToLevel[of "0" "getM stateA"]
      by auto
    hence "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
      using `level < currentLevel (getM stateB)`
      by auto
  }
  moreover
  {
    assume "appliedBackjumpAndLearn stateA stateB"
    then obtain level' bl where "getM stateB = prefixToLevel level' (getM stateA) @ [(bl, False)]"
      "isMinimalBackjumpLevel level' bl (getC stateA) (getM stateA)"
      unfolding appliedBackjumpAndLearn_def
      by auto
    have "level' < currentLevel (getM stateA)"
      using `isMinimalBackjumpLevel level' bl (getC stateA) (getM stateA)`
      unfolding isMinimalBackjumpLevel_def
      unfolding isBackjumpLevel_def
      using elementLevelLeqCurrentLevel[of "opposite bl" "getM stateA"]
      by auto
    hence "currentLevel (getM stateB) = level'"
      using currentLevelPrefixToLevelEq[of "level'" "getM stateA"]
      using `getM stateB = prefixToLevel level' (getM stateA) @ [(bl, False)]`
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
    hence "level < level'"
      using `level < currentLevel (getM stateB)`
      by simp
    hence "prefixToLevel level (getM stateA) = prefixToLevel level (getM stateB)"
      using `getM stateB = prefixToLevel level' (getM stateA) @ [(bl, False)]`
      using prefixToLevelAppend[of "level" "prefixToLevel level' (getM stateA)" "[(bl, False)]"]
      using currentLevelPrefixToLevelEq[of "level'" "getM stateA"]
      using `level' < currentLevel (getM stateA)`
      using prefixToLevelPrefixToLevelHigherLevel[of "level" "level'" "getM stateA"]
      by auto
  }
  ultimately
  show ?thesis
    using `transition stateA stateB F0 decisionVars`
    unfolding transition_def
    by blast
qed

lemma transitionsRetainPrefixToLowLevelsList:
  assumes
  "l \<noteq> []" "isTransitionChain l F0 decisionVars"
  "currentLevel (getM (l ! m)) = level"
  "\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) > level"
  "m < m'" "m' < length l"
shows
  "getM (l ! m) = prefixToLevel level (getM (l ! m'))"
using assms
proof (induct m')
  case 0
  thus ?case
    by simp
next
  case (Suc m')
  have "m' < length l"
    using `Suc m' < length l`
    by simp

  have "getM (l ! m) = prefixToLevel level (getM (l ! m'))"
  proof (cases "m' = m")
    case False
    hence "m < m'"
      using `m < Suc m'`
      by simp
    thus ?thesis
      using Suc
      using `m < m'` `m' < length l`
      by simp
  next
    case True
    thus ?thesis
      using `currentLevel (getM (l ! m)) = level`
      using prefixToLevelCurrentLevel[of "getM (l ! m')"]
      by simp
  qed
  moreover
  have "prefixToLevel level (getM (l ! m')) = prefixToLevel level (getM (l ! (Suc m')))"
  proof-
    have "transition (l ! m') (l ! (Suc m')) F0 decisionVars"
      using `Suc m' < length l`
      using `isTransitionChain l F0 decisionVars`
      by auto
    moreover
    have "currentLevel (getM (l ! (Suc m'))) > level"
      using `Suc m' < length l`
      using `m < Suc m'`
      using `\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) > level`
      by auto
    ultimately
    show ?thesis
      using transitionsRetainPrefixToLowLevels[of "l ! m'" "l ! Suc m'" "F0" "decisionVars" "level"]
      by simp
  qed
  ultimately
  show ?case
    by simp
qed

lemma backjumpClauseIsNotPresent:
  assumes 
  "isInitialState state0" "(state0, stateA) \<in> transitionRelation F0 decisionVars"
  "appliedBackjumpAndLearn stateA stateB"
  shows "remdups (getC stateA) \<notin> set (map remdups (F0 @ (getFl stateA)))"
proof (rule ccontr)
  assume "\<not> remdups (getC stateA) \<notin> set (map remdups (F0 @ (getFl stateA)))"
  then obtain clause where
    "clause el map remdups (F0 @ (getFl stateA))" "remdups (getC stateA) = clause"
    by auto

  show "False"
  proof-
    have "getConflictFlag stateA"
      using `appliedBackjumpAndLearn stateA stateB`
      unfolding appliedBackjumpAndLearn_def
      by simp
    hence invs: "consistent (elements (getM stateA))" "uniq (elements (getM stateA))"
      "clauseFalse (getC stateA) (elements (getM stateA))"
      using `isInitialState state0` `(state0, stateA) \<in> transitionRelation F0 decisionVars`
      using invariantsHoldInValidRunsFromInitialState[of "state0" "stateA" "F0" "decisionVars"]
      unfolding invariantsHoldInState_def
      unfolding InvariantConsistent_def InvariantUniq_def InvariantCFalse_def
      by auto

    obtain lit level where "isMinimalBackjumpLevel level lit (getC stateA) (getM stateA)"
      using `appliedBackjumpAndLearn stateA stateB`
      unfolding appliedBackjumpAndLearn_def
      by auto

    hence "isUnitClause (getC stateA) lit (elements (prefixToLevel level (getM stateA)))"
      using isBackjumpLevelEnsuresIsUnitInPrefix[of "getM stateA" "getC stateA" "level" "lit"]
      using invs
      unfolding isMinimalBackjumpLevel_def
      by simp
    hence "isUnitClause clause lit (elements (prefixToLevel level (getM stateA)))"
      using `remdups (getC stateA) = clause`
      unfolding isUnitClause_def
      by auto

    have "level < currentLevel (getM stateA)"
      using `isMinimalBackjumpLevel level lit (getC stateA) (getM stateA)`
      unfolding isMinimalBackjumpLevel_def
      unfolding isBackjumpLevel_def
      using elementLevelLeqCurrentLevel[of "opposite lit" "getM stateA"]
      by auto
    hence "currentLevel (getM stateA) > 0"
      by simp
    have "currentLevel (getM state0) = 0"
      using `isInitialState state0`
      unfolding isInitialState_def
      unfolding currentLevel_def
      by simp
    have "currentLevel (getM state0) < currentLevel (getM stateA)"
      using `currentLevel (getM state0) = 0`
      using `currentLevel (getM stateA) > 0`
      by simp
    hence "state0 \<noteq> stateA"
      by auto

    then obtain l where
      "l \<noteq> []" "hd l = state0" "last l = stateA" "isTransitionChain l F0 decisionVars"
      using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
      unfolding transitionRelation_def
      using rtrancl_rel_chain[of "state0" "stateA" "{(stateA, stateB). transition stateA stateB F0 decisionVars}"]
      by auto
    
    have "\<exists>m<length l. currentLevel (getM (l ! m)) = level"
    proof (cases "level = 0")
      case True
      thus ?thesis
	using `currentLevel (getM state0) = 0`
	using `hd l = state0` `l \<noteq> []`
	by (auto simp add: hd_conv_nth)
    next
      case False
      then obtain m where
	"m < length l - 1" "currentLevel (getM (l ! m)) = level"
	using `l \<noteq> []` `isTransitionChain l F0 decisionVars`
	using `hd l = state0` `currentLevel (getM state0) = 0`
	using `last l = stateA` `level < currentLevel (getM stateA)`
	using currentLevelContinuous[of "l" "F0" "decisionVars" "level"]
	by auto
      hence "m < length l"
	by auto
      thus ?thesis
	using `currentLevel (getM (l ! m)) = level`
	by auto
    qed

    obtain m where
      "m < length l - 1" "currentLevel (getM (l ! m)) = level"
      "\<forall> m'. m < m' \<and> m' < length l \<longrightarrow> currentLevel (getM (l ! m')) > level"
      using lastVisitOfLevel[of "l" "F0" "decisionVars" "level"]
      using `l \<noteq> []` `isTransitionChain l F0 decisionVars` `\<exists>m<length l. currentLevel (getM (l ! m)) = level`
      using `level < currentLevel (getM stateA)` `last l = stateA`
      by auto
    
    have "getM (l ! m) = prefixToLevel level (getM stateA)"
      using transitionsRetainPrefixToLowLevelsList[of "l" "F0" "decisionVars" "m" "level" "length l - 1"]
      using `l \<noteq> []` `isTransitionChain l F0 decisionVars` 
      using `currentLevel (getM (l ! m)) = level`
      using `\<forall>m'. m < m' \<and> m' < length l \<longrightarrow> level < currentLevel (getM (l ! m'))`
      using `m < length l - 1`
      using `last l = stateA`
      by (auto simp add: last_conv_nth)

    hence "isUnitClause clause lit (elements (getM (l ! m)))"
      using `isUnitClause clause lit (elements (prefixToLevel level (getM stateA)))`
      by simp
    
    have "appliedDecide (l ! m) (l ! (m + 1)) F0 decisionVars"
      using `isTransitionChain l F0 decisionVars`
      using `m < length l - 1`
      using `currentLevel (getM (l ! m)) = level`
      using `\<forall>m'. m < m' \<and> m' < length l \<longrightarrow> level < currentLevel (getM (l ! m'))`
      using onlyDecideIncreasesLevel[of "l ! m" "l ! (m + 1)" "F0" "decisionVars"]
      by simp

    hence "\<not> (\<exists> uc ul. uc el F0 @ (getFl (l ! m)) \<and> isUnitClause uc ul (elements (getM (l ! m))))"
      unfolding appliedDecide_def
      by auto

    have "\<not> clause el map remdups (F0 @ (getFl (l ! m)))"
    proof (rule ccontr)
      assume "\<not> ?thesis"
      then obtain clause' 
	where "clause' el F0 @ (getFl (l ! m))" "remdups clause' = clause"
	by auto
      hence "isUnitClause clause' lit (elements (getM (l ! m)))"
	using `isUnitClause clause lit (elements (getM (l ! m)))`
	unfolding isUnitClause_def
	by auto
      thus False
	using `\<not> (\<exists> uc ul. uc el F0 @ (getFl (l ! m)) \<and> isUnitClause uc ul (elements (getM (l ! m))))`
	using `clause' el F0 @ (getFl (l ! m))`
	by auto
    qed

    have "clause el map remdups (F0 @ (getFl (l ! (length l - 1))))"
      using `l \<noteq> []`
      using last_conv_nth[of "l"]
      using `last l = stateA`
      using `clause el map remdups (F0 @ (getFl stateA))`
      by simp

      
    obtain m' where 
      "m <= m'" "m' < length l - 1" 
      "remdups (getC (l ! m')) = clause" "appliedBackjumpAndLearn (l ! m') (l ! (m' + 1))"
      using clausesAddedToFAreLearntMiddle[of "l" "F0" "decisionVars" "clause" "m" "length l - 1"]
      using `l \<noteq> []` `isTransitionChain l F0 decisionVars`
      using `\<not> clause el map remdups (F0 @ (getFl (l ! m)))`
      using `clause el map remdups (F0 @ (getFl (l ! (length l - 1))))`
      using `m < length l - 1`
      by auto

    then obtain lit' level' where 
      "isMinimalBackjumpLevel level' lit' (getC (l ! m')) (getM (l ! m'))"
      "getM (l ! (m' + 1)) = prefixToLevel level' (getM (l ! m')) @ [(lit', False)]"
      unfolding appliedBackjumpAndLearn_def
      by auto

    have invsm': "consistent (elements (getM (l ! m'))) \<and> uniq (elements (getM (l ! m'))) \<and> 
      clauseFalse (getC (l ! m')) (elements (getM (l ! m')))"
    proof-
      have "getConflictFlag (l ! m')"
	using `appliedBackjumpAndLearn (l ! m') (l ! (m' + 1))`
	unfolding appliedBackjumpAndLearn_def
	by simp
      have "(hd l, l ! m') \<in> transitionRelation F0 decisionVars"
	using `l \<noteq> []` `m' < length l - 1` `isTransitionChain l F0 decisionVars`
	using rel_chain_hd_rtrancl[of "l" "{(x, y). transition x y F0 decisionVars}" "m'"]
	unfolding transitionRelation_def
	by simp
      hence "invariantsHoldInState (l ! m') F0 decisionVars"
	using `isInitialState state0` `hd l = state0`
	using invariantsHoldInValidRunsFromInitialState[of "state0" "l ! m'" "F0" "decisionVars"]
	by simp
      thus ?thesis
	using `getConflictFlag (l ! m')`
	unfolding invariantsHoldInState_def
	unfolding InvariantCFalse_def
	unfolding InvariantConsistent_def
	unfolding InvariantUniq_def
	by auto
    qed

    have "getM (l ! m) = prefixToLevel level (getM (l ! m'))"
    proof (cases "m = m'")
      case True
      thus ?thesis
	using `currentLevel (getM (l ! m)) = level`
	using prefixToLevelCurrentLevel[of "getM (l ! m)"]
	by auto
    next
      case False
      with `m <= m'`
      have "m < m'"
	by simp
      thus ?thesis
	using transitionsRetainPrefixToLowLevelsList[of "l" "F0" "decisionVars" "m" "level" "m'"]
	using `l \<noteq> []` `isTransitionChain l F0 decisionVars` 
	using `currentLevel (getM (l ! m)) = level`
	using `\<forall>m'. m < m' \<and> m' < length l \<longrightarrow> level < currentLevel (getM (l ! m'))`
	using `m' < length l - 1` `m <= m'`
	by auto
    qed

    hence "isUnitClause clause lit (elements (prefixToLevel level (getM (l ! m'))))"
      using `isUnitClause clause lit (elements (getM (l ! m)))`
      by simp
    hence "isUnitClause (getC  (l ! m')) lit (elements (prefixToLevel level (getM (l ! m'))))"
      using `remdups (getC  (l ! m')) = clause`
      unfolding isUnitClause_def
      by auto
    hence "level' <= level"
      using invsm'
      using `isMinimalBackjumpLevel level' lit' (getC (l ! m')) (getM (l ! m'))`
      using isMinimalBackjumpLevelEnsuresIsNotUnitBeforePrefix[of "getM (l ! m')" "getC (l ! m')" "level'" "lit'" "level"]
      by force

    hence "currentLevel (getM (l ! (m' + 1))) <= level"
      using `getM (l ! (m' + 1)) = prefixToLevel level' (getM (l ! m')) @ [(lit', False)]`
      using currentLevelPrefixToLevel[of "level'" "getM (l ! m')"]
      unfolding currentLevel_def
      by (auto simp add: markedElementsAppend)
    moreover
    have "m < m' + 1" "m' + 1 < length l"
      using `m <= m'` `m' < length l - 1`
      by auto
    ultimately
    show ?thesis
      using `\<forall>m'. m < m' \<and> m' < length l \<longrightarrow> level < currentLevel (getM (l ! m'))`
      by auto
  qed
qed

lemma formulaIsDecreasedByLearn:
  assumes 
  "isInitialState state0" "(state0, stateA) \<in> transitionRelation F0 decisionVars"
  "appliedBackjumpAndLearn stateA stateB"
  shows "(getFl stateB, getFl stateA) \<in> ord_formula_inclusion (decisionVars \<union> (vars F0))"
proof-
  from `appliedBackjumpAndLearn stateA stateB`
  have "getFl stateB = getFl stateA @ [getC stateA]"
    unfolding appliedBackjumpAndLearn_def
    by auto

  have "(state0, stateB) \<in> transitionRelation F0 decisionVars"
    using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
    using `appliedBackjumpAndLearn stateA stateB`
    using rtrancl_into_rtrancl[of "state0" "stateA" "{(stateA, stateB). transition stateA stateB F0 decisionVars}" "stateB"]
    unfolding transitionRelation_def
    unfolding transition_def
    by auto

  from `isInitialState state0` `(state0, stateA) \<in> transitionRelation F0 decisionVars`
  have "invariantsHoldInState stateA F0 decisionVars"
    using invariantsHoldInValidRunsFromInitialState
    by simp

  from `isInitialState state0` `(state0, stateB) \<in> transitionRelation F0 decisionVars`
  have "invariantsHoldInState stateB F0 decisionVars"
    using invariantsHoldInValidRunsFromInitialState
    by simp

  have "vars (getFl stateA) \<subseteq> vars F0 \<union> decisionVars"
    using `invariantsHoldInState stateA F0 decisionVars`
    unfolding invariantsHoldInState_def
    unfolding InvariantVarsF_def
    by simp

  have "vars (getFl stateB) \<subseteq> vars F0 \<union> decisionVars"
    using `invariantsHoldInState stateB F0 decisionVars`
    unfolding invariantsHoldInState_def
    unfolding InvariantVarsF_def
    by simp

  let ?r = "\<lambda> F. remdups (map remdups F)"
  have "(?r (getFl stateB), ?r (getFl stateA)) \<in> ord_formula_inclusionRestricted (decisionVars \<union> vars F0)"
  proof-
    have "set (?r (getFl stateB)) \<supset> set (?r (getFl stateA))"
    proof-
      have "remdups (getC stateA) \<notin> set (?r (getFl stateA))"
	using `isInitialState state0`
	using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
	using `appliedBackjumpAndLearn stateA stateB`
	using backjumpClauseIsNotPresent[of "state0" "stateA" "F0" "decisionVars" "stateB"]
	by auto
      moreover
      have "getFl stateB = getFl stateA @ [getC stateA]"
	using `appliedBackjumpAndLearn stateA stateB`
	unfolding appliedBackjumpAndLearn_def
	by auto
      ultimately
      show ?thesis
	by auto
    qed
    moreover
    have "vars (?r (getFl stateA)) \<subseteq> vars F0 \<union> decisionVars"
      using `vars (getFl stateA) \<subseteq> vars F0 \<union> decisionVars`
      by simp
    moreover
    have "vars (?r (getFl stateB)) \<subseteq> vars F0 \<union> decisionVars"
      using `vars (getFl stateB) \<subseteq> vars F0 \<union> decisionVars`
      by simp
    ultimately
    show ?thesis
      unfolding ord_formula_inclusionRestricted_def
      by (auto simp add: uniqRemdups simp del: vars_def_formula vars_def_clause)
  qed
  thus ?thesis
    unfolding ord_formula_inclusion_def
    by simp
qed


subsubsection{* System without restart *}
text{* We show that the system without restarts terminates *}

definition MCfCLf :: "State \<Rightarrow> ((LiteralTrail \<times> bool) \<times> Clause) \<times> bool"
where
[simp]: "MCfCLf state =  (((getM state, getConflictFlag state), getC state), getLearntFlag state)"

definition ord_MCfCLf_lex :: "Formula \<Rightarrow> Variable set \<Rightarrow> (((LiteralTrail \<times> bool) \<times> Clause) \<times> bool) \<times> (((LiteralTrail \<times> bool) \<times> Clause) \<times> bool) \<Rightarrow> bool"
where
[simp]: "ord_MCfCLf_lex F0 decisionVars = 
((ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars) <*lex*> ord_bool_TF) <*lexp*> (\<lambda> a. ord_clause_multTrail (fst a))) <*lex*> ord_bool_FT"

definition 
ord_state_MCfCLf :: "Formula \<Rightarrow> Variable set \<Rightarrow> State \<times> State \<Rightarrow> bool"
where
"ord_state_MCfCLf F0 decisionVars = 
  {(stateA, stateB). (MCfCLf stateA, MCfCLf stateB) \<in> ord_MCfCLf_lex F0 decisionVars}"

text{* Now we can prove the following key lemma which shows that the termination ordering is well founded. *}
lemma wf_ord_state_MCfCLf:
  assumes "finite decisionVars"
  shows "wf (ord_state_MCfCLf F0 decisionVars)"
proof-
  have "wf (ord_MCfCLf_lex F0 decisionVars)"
  proof-
    have "wf (ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars))"
      using `finite decisionVars`
      using finiteVarsFormula[of "F0"]
      using wf_ord_trail_lexDecisionRestricted[of "vars F0 \<union> decisionVars"]
      by simp
    with wf_ord_bool_TF wf_ord_clause_multTrail wf_ord_bool_FT
    show ?thesis
      by (auto simp add: wf_lex_prod wf_lex_prod_p)
  qed
  thus ?thesis
    unfolding ord_state_MCfCLf_def
    using wf_inv_image[of "ord_MCfCLf_lex F0 decisionVars" "MCfCLf"]
    unfolding inv_image_def
    by simp
qed

lemma transitionRelationNoRestartIsTransitionRelation:
  assumes "(a, b) \<in> transitionRelationNoRestart F0 decisionVars"
  shows "(a, b) \<in> transitionRelation F0 decisionVars"
proof-
  let ?St = "{(stateA, stateB). transition stateA stateB F0 decisionVars}"
  let ?Str = "{(stateA, stateB). transitionNoRestart stateA stateB F0 decisionVars}"
  have "?Str \<subseteq> ?St"
    unfolding transitionNoRestart_def
    unfolding transition_def
    by auto
  thus ?thesis
    using assms
    unfolding transitionRelationNoRestart_def
    unfolding transitionRelation_def
    using rtrancl_mono[of "?Str" "?St"]
    by auto
qed


text{* Now we can prove that every rule application decreases a state
with respect to the constructed termination ordering.*}
lemma stateIsDecreasedByValidTransitionsNoRestart:
  fixes stateA::State and stateB::State
  assumes "isInitialState state0" "(state0, stateA) \<in> transitionRelationNoRestart F0 decisionVars" and 
  "transitionNoRestart stateA stateB F0 decisionVars"
  shows "(stateB, stateA) \<in> ord_state_MCfCLf F0 decisionVars"
proof-

  have "(state0, stateA) \<in> transitionRelation F0 decisionVars"
    using transitionRelationNoRestartIsTransitionRelation[of "state0" "stateA" "F0" "decisionVars"]
    using `(state0, stateA) \<in> transitionRelationNoRestart F0 decisionVars`
    by auto

  have "invariantsHoldInState stateA F0 decisionVars"
    using invariantsHoldInValidRunsFromInitialState[of "state0" "stateA" "F0" "decisionVars"]
    using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
    using `isInitialState state0`
    by simp

  {
    assume "appliedDecide stateA stateB F0 decisionVars \<or> appliedUnitPropagate stateA stateB F0 decisionVars \<or> appliedBackjumpAndLearn stateA stateB"
    with `invariantsHoldInState stateA F0 decisionVars`
    have "(getM stateB, getM stateA) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
      using trailIsDecreasedByDeciedUnitPropagateAndBackjump
      by simp
    hence "(stateB, stateA) \<in> ord_state_MCfCLf F0 decisionVars"
      unfolding ord_state_MCfCLf_def
      unfolding ord_MCfCLf_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedConflict stateA stateB F0"
    hence "getM stateA = getM stateB" "(getConflictFlag stateB, getConflictFlag stateA) \<in> ord_bool_TF"
      using conflictFlagIsDecreasedByConflict[of "stateA" "stateB" "F0"]
      unfolding appliedConflict_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_MCfCLf F0 decisionVars"
      unfolding ord_state_MCfCLf_def
      unfolding ord_MCfCLf_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedExplain stateA stateB F0"
    hence "getM stateA = getM stateB"
      "getConflictFlag stateA = getConflictFlag stateB"
      "(getC stateB, getC stateA) \<in> ord_clause_multTrail (getM stateA)"
      using conflictClauseIsDecreasedByExplain[of "stateA" "stateB" "F0"]
      unfolding appliedExplain_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_MCfCLf F0 decisionVars"
      unfolding ord_state_MCfCLf_def
      unfolding ord_MCfCLf_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedForget stateA stateB"
    hence 
      "getM stateA = getM stateB"
      "getConflictFlag stateA = getConflictFlag stateB"
      "getC stateA = getC stateB"
      "(getLearntFlag stateB, getLearntFlag stateA) \<in> ord_bool_FT"
      using learntFlagIsDecreasedByForget[of "stateA" "stateB"]
      unfolding appliedForget_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_MCfCLf F0 decisionVars"
      unfolding ord_state_MCfCLf_def
      unfolding ord_MCfCLf_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  ultimately
  show ?thesis
    using `transitionNoRestart stateA stateB F0 decisionVars`
    unfolding transitionNoRestart_def
    by auto
qed


text{* Using the termination ordering we show that the transition
 relation is well founded on states reachable from initial state.*}
(*----------------------------------------------------------------------------*)
theorem wfTransitionRelationNoRestart:
  fixes decisionVars :: "Variable set" and F0 :: "Formula"
  assumes "finite decisionVars" and "isInitialState state0"
  shows "wf {(stateB, stateA). 
             (state0, stateA) \<in> transitionRelationNoRestart F0 decisionVars \<and> (transitionNoRestart stateA stateB F0 decisionVars)}"
(*----------------------------------------------------------------------------*)
proof-
  let ?rel = "{(stateB, stateA). 
                  (state0, stateA) \<in> transitionRelationNoRestart F0 decisionVars \<and> (transitionNoRestart stateA stateB F0 decisionVars)}"
  let ?rel'= "ord_state_MCfCLf F0 decisionVars"

  have "\<forall>x y. (x, y) \<in> ?rel \<longrightarrow> (x, y) \<in> ?rel'"
  proof-
    {
      fix stateA::State and stateB::State
      assume "(stateB, stateA) \<in> ?rel"
      hence "(stateB, stateA) \<in> ?rel'"
	using `isInitialState state0`
	using stateIsDecreasedByValidTransitionsNoRestart[of "state0" "stateA" "F0" "decisionVars" "stateB"]
	by simp
    }
    thus ?thesis
      by simp
  qed
  moreover
  have "wf ?rel'"
    using `finite decisionVars`
    by (rule wf_ord_state_MCfCLf)
  ultimately
  show ?thesis
    using wellFoundedEmbed[of "?rel" "?rel'"]
    by simp
qed


subsubsection{* System without forgets *}
text{* We show that the system without forgets terminates *}

definition FLfMCfC :: "State \<Rightarrow> (((Formula \<times> bool) \<times> LiteralTrail) \<times> bool) \<times> Clause"
where
[simp]: "FLfMCfC state =  ((((getFl state, getLearntFlag state), getM state), getConflictFlag state), getC state)"

definition ord_FLfMCfC_lex :: "Formula \<Rightarrow> Variable set \<Rightarrow> ((((Formula \<times> bool) \<times> LiteralTrail) \<times> bool) \<times> Clause) \<times> ((((Formula \<times> bool) \<times> LiteralTrail) \<times> bool) \<times> Clause) \<Rightarrow> bool"
where
[simp]: "ord_FLfMCfC_lex F0 decisionVars = 
(((ord_formula_inclusion (vars F0 \<union> decisionVars) <*lex*> ord_bool_FT) <*lex*> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)) <*lex*> ord_bool_TF) <*lexp*> (\<lambda> a. ord_clause_multTrail (snd (fst a)))"

definition 
ord_state_FLfMCfC :: "Formula \<Rightarrow> Variable set \<Rightarrow> State \<times> State \<Rightarrow> bool"
where
"ord_state_FLfMCfC F0 decisionVars = 
  {(stateA, stateB). (FLfMCfC stateA, FLfMCfC stateB) \<in> ord_FLfMCfC_lex F0 decisionVars}"

text{* Now we can prove the following key lemma which shows that the termination ordering is well founded. *}
lemma wf_ord_state_FLfMCfC:
  assumes "finite decisionVars"
  shows "wf (ord_state_FLfMCfC F0 decisionVars)"
proof-
  have "wf (ord_FLfMCfC_lex F0 decisionVars)"
  proof-
    have "wf (ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars))" 
         "wf (ord_formula_inclusion (vars F0 \<union> decisionVars))"
      using `finite decisionVars`
      using finiteVarsFormula[of "F0"]
      using wf_ord_trail_lexDecisionRestricted[of "vars F0 \<union> decisionVars"]
      using wf_ord_formula_inclusion[of "vars F0 \<union> decisionVars"]
      by auto
    with wf_ord_bool_TF wf_ord_clause_multTrail wf_ord_bool_FT
    show ?thesis
      by (auto simp add: wf_lex_prod wf_lex_prod_p)
  qed
  thus ?thesis
    unfolding ord_state_FLfMCfC_def
    using wf_inv_image[of "ord_FLfMCfC_lex F0 decisionVars" "FLfMCfC"]
    unfolding inv_image_def
    by simp
qed

text{* Now we can prove that every rule application decreases a state
with respect to the constructed termination ordering.*}

lemma transitionRelationNoForgetIsTransitionRelation:
  assumes "(a, b) \<in> transitionRelationNoForget F0 decisionVars"
  shows "(a, b) \<in> transitionRelation F0 decisionVars"
proof-
  let ?St = "{(stateA, stateB). transition stateA stateB F0 decisionVars}"
  let ?Str = "{(stateA, stateB). transitionNoForget stateA stateB F0 decisionVars}"
  have "?Str \<subseteq> ?St"
    unfolding transitionNoForget_def
    unfolding transition_def
    by auto
  thus ?thesis
    using assms
    unfolding transitionRelationNoForget_def
    unfolding transitionRelation_def
    using rtrancl_mono[of "?Str" "?St"]
    by auto
qed

lemma stateIsDecreasedByValidTransitionsNoForget:
  fixes stateA::State and stateB::State
  assumes "isInitialState state0" "(state0, stateA) \<in> transitionRelationNoForget F0 decisionVars" and 
  "transitionNoForget stateA stateB F0 decisionVars"
  shows "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
proof-

  have "(state0, stateA) \<in> transitionRelation F0 decisionVars"
    using transitionRelationNoForgetIsTransitionRelation[of "state0" "stateA" "F0" "decisionVars"]
    using `(state0, stateA) \<in> transitionRelationNoForget F0 decisionVars`
    by auto

  have "invariantsHoldInState stateA F0 decisionVars"
    using invariantsHoldInValidRunsFromInitialState[of "state0" "stateA" "F0" "decisionVars"]
    using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
    using `isInitialState state0`
    by simp

  {
    assume "appliedDecide stateA stateB F0 decisionVars \<or> appliedUnitPropagate stateA stateB F0 decisionVars"
    with `invariantsHoldInState stateA F0 decisionVars`
    have "(getM stateB, getM stateA) \<in> ord_trail_lexDecisionRestricted (vars F0 \<union> decisionVars)"
      using trailIsDecreasedByDeciedUnitPropagateAndBackjump[of "stateA" "F0" "decisionVars" "stateB"]
      by auto
    moreover
    from `appliedDecide stateA stateB F0 decisionVars \<or> appliedUnitPropagate stateA stateB F0 decisionVars`
    have "getFl stateB = getFl stateA" "getLearntFlag stateB = getLearntFlag stateA"
      unfolding appliedDecide_def
      unfolding appliedUnitPropagate_def
      by auto
    ultimately
    have "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
      unfolding ord_state_FLfMCfC_def
      unfolding ord_FLfMCfC_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedBackjumpAndLearn stateA stateB"
    hence "(getFl stateB, getFl stateA) \<in> ord_formula_inclusion (vars F0 \<union> decisionVars)"
      using `isInitialState state0`
      using `(state0, stateA) \<in> transitionRelation F0 decisionVars`
      using formulaIsDecreasedByLearn[of "state0" "stateA" "F0" "decisionVars" "stateB"]
      using Un_commute[of "vars F0" "decisionVars"]
      by simp
    hence "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
      unfolding ord_state_FLfMCfC_def
      unfolding ord_FLfMCfC_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedConflict stateA stateB F0"
    hence "getFl stateA = getFl stateB" "getLearntFlag stateA = getLearntFlag stateB" 
      "getM stateA = getM stateB" "(getConflictFlag stateB, getConflictFlag stateA) \<in> ord_bool_TF"
      using conflictFlagIsDecreasedByConflict[of "stateA" "stateB" "F0"]
      unfolding appliedConflict_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
      unfolding ord_state_FLfMCfC_def
      unfolding ord_FLfMCfC_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedExplain stateA stateB F0"
    hence 
      "getFl stateA = getFl stateB" "getLearntFlag stateA = getLearntFlag stateB" 
      "getM stateA = getM stateB"
      "getConflictFlag stateA = getConflictFlag stateB"
      "(getC stateB, getC stateA) \<in> ord_clause_multTrail (getM stateA)"
      using conflictClauseIsDecreasedByExplain[of "stateA" "stateB" "F0"]
      unfolding appliedExplain_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
      unfolding ord_state_FLfMCfC_def
      unfolding ord_FLfMCfC_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  moreover
  {
    assume "appliedRestart stateA stateB"
    hence 
      "getFl stateB = getFl stateA"
      "(getLearntFlag stateB, getLearntFlag stateA) \<in> ord_bool_FT"
      using learntFlagIsDecreasedByRestart[of "stateA" "stateB"]
      unfolding appliedRestart_def
      by auto
    hence "(stateB, stateA) \<in> ord_state_FLfMCfC F0 decisionVars"
      unfolding ord_state_FLfMCfC_def
      unfolding ord_FLfMCfC_lex_def
      by (auto simp add: lex_prod_def lex_prod_p_def)
  }
  ultimately
  show ?thesis
    using `transitionNoForget stateA stateB F0 decisionVars`
    unfolding transitionNoForget_def
    by auto
qed

text{* Using the termination ordering we show that the transition
 relation is well founded on states reachable from initial state.*}
(*----------------------------------------------------------------------------*)
theorem wfTransitionRelationNoForget:
  fixes decisionVars :: "Variable set" and F0 :: "Formula"
  assumes "finite decisionVars" and "isInitialState state0"
  shows "wf {(stateB, stateA). 
             (state0, stateA) \<in> transitionRelationNoForget F0 decisionVars \<and> (transitionNoForget stateA stateB F0 decisionVars)}"
(*----------------------------------------------------------------------------*)
proof-
  let ?rel = "{(stateB, stateA). 
                  (state0, stateA) \<in> transitionRelationNoForget F0 decisionVars \<and> (transitionNoForget stateA stateB F0 decisionVars)}"
  let ?rel'= "ord_state_FLfMCfC F0 decisionVars"

  have "\<forall>x y. (x, y) \<in> ?rel \<longrightarrow> (x, y) \<in> ?rel'"
  proof-
    {
      fix stateA::State and stateB::State
      assume "(stateB, stateA) \<in> ?rel"
      hence "(stateB, stateA) \<in> ?rel'"
	using `isInitialState state0`
	using stateIsDecreasedByValidTransitionsNoForget[of "state0" "stateA" "F0" "decisionVars" "stateB"]
	by simp
    }
    thus ?thesis
      by simp
  qed
  moreover
  have "wf ?rel'"
    using `finite decisionVars`
    by (rule wf_ord_state_FLfMCfC)
  ultimately
  show ?thesis
    using wellFoundedEmbed[of "?rel" "?rel'"]
    by simp
qed


(*----------------------------------------------------------------------------*)
subsection{* Completeness *}
(*----------------------------------------------------------------------------*)
text{* In this section we will first show that each final state is
either \textit{SAT} or \textit{UNSAT} state. *}


lemma finalNonConflictState: 
  fixes state::State and FO :: Formula
  assumes 
  "getConflictFlag state = False" and
  "\<not> applicableDecide state decisionVars"  and
  "\<not> applicableConflict state"
  shows "\<not> formulaFalse (getF state) (elements (getM state))" and 
  "vars (elements (getM state)) \<supseteq> decisionVars"
proof-
  from `\<not> applicableConflict state` `getConflictFlag state = False`
  show "\<not> formulaFalse (getF state) (elements (getM state))"
    unfolding applicableConflictCharacterization
    by (auto simp add:formulaFalseIffContainsFalseClause formulaEntailsItsClauses)
  show "vars (elements (getM state)) \<supseteq> decisionVars"
  proof
    fix x :: Variable
    let ?l = "Pos x"
    assume "x \<in> decisionVars"
    hence "var ?l = x" and "var ?l \<in> decisionVars" and "var (opposite ?l) \<in> decisionVars"
      by auto
    with `\<not> applicableDecide state decisionVars` 
    have "literalTrue ?l (elements (getM state)) \<or> literalFalse ?l (elements (getM state))"
      unfolding applicableDecideCharacterization
      by force
    with `var ?l = x`
    show "x \<in> vars (elements (getM state))"
      using valuationContainsItsLiteralsVariable[of "?l" "elements (getM state)"]
      using valuationContainsItsLiteralsVariable[of "opposite ?l" "elements (getM state)"]
      by auto
  qed
qed

lemma finalConflictingState: 
  fixes state :: State
  assumes 
  "InvariantUniq (getM state)" and
  "InvariantReasonClauses (getF state) (getM state)" and
  "InvariantCFalse (getConflictFlag state) (getM state) (getC state)" and
  "\<not> applicableExplain state" and
  "\<not> applicableBackjump state" and
  "getConflictFlag state"
  shows
  "getC state = []"
proof (cases "\<forall> l. l el getC state \<longrightarrow> opposite l el decisions (getM state)")
  case True
  {
    assume "getC state \<noteq> []"
    let ?l = "getLastAssertedLiteral (oppositeLiteralList (getC state)) (elements (getM state))"

    from `InvariantUniq (getM state)`
    have "uniq (elements (getM state))"
      unfolding InvariantUniq_def
      .
   
    from `getConflictFlag state` `InvariantCFalse (getConflictFlag state) (getM state) (getC state)`
    have "clauseFalse (getC state) (elements (getM state))"
      unfolding InvariantCFalse_def
      by simp

    with `getC state \<noteq> []`
    `InvariantUniq (getM state)`
    have "isLastAssertedLiteral ?l (oppositeLiteralList (getC state)) (elements (getM state))"
      unfolding InvariantUniq_def
      using getLastAssertedLiteralCharacterization
      by simp

    with True `uniq (elements (getM state))`
    have "\<exists> level. (isBackjumpLevel level (opposite ?l) (getC state) (getM state))"
      using allDecisionsThenExistsBackjumpLevel [of "getM state" "getC state" "opposite ?l"]
      by simp
    then
    obtain level::nat where
      "isBackjumpLevel level (opposite ?l) (getC state) (getM state)"
      by auto
    with `getConflictFlag state`
    have "applicableBackjump state"
      unfolding applicableBackjumpCharacterization
      by auto
    with `\<not> applicableBackjump state`
    have False
      by simp
  }
  thus ?thesis
    by auto
next
  case False
  then obtain literal::Literal where "literal el getC state" "\<not> opposite literal el decisions (getM state)"
    by auto
  with `InvariantReasonClauses (getF state) (getM state)` `InvariantCFalse (getConflictFlag state) (getM state) (getC state)` `getConflictFlag state`
  have "\<exists> c. formulaEntailsClause (getF state) c \<and> isReason c (opposite literal) (elements (getM state))"
    using explainApplicableToEachNonDecision[of "getF state" "getM state" "getConflictFlag state" "getC state" "opposite literal"]
    by auto
  then obtain c::Clause 
    where "formulaEntailsClause (getF state) c" "isReason c (opposite literal) (elements (getM state))"
    by auto
  with `\<not> applicableExplain state` `getConflictFlag state` `literal el (getC state)`
  have "False"
    unfolding applicableExplainCharacterization
    by auto
  thus ?thesis
    by simp
qed
  
lemma finalStateCharacterizationLemma:
  fixes state :: State
  assumes 
  "InvariantUniq (getM state)" and
  "InvariantReasonClauses (getF state) (getM state)" and
  "InvariantCFalse (getConflictFlag state) (getM state) (getC state)" and
  "\<not> applicableDecide state decisionVars"  and
  "\<not> applicableConflict state"
  "\<not> applicableExplain state" and
  "\<not> applicableBackjump state"
  shows
  "(getConflictFlag state = False \<and> 
           \<not>formulaFalse (getF state) (elements (getM state)) \<and> 
           vars (elements (getM state)) \<supseteq> decisionVars) \<or> 
   (getConflictFlag state = True \<and> 
           getC state = [])"
proof (cases "getConflictFlag state")
  case True
  hence "getC state = []"
    using assms
    using finalConflictingState
    by auto
  with True 
  show ?thesis
    by simp
next
  case False
  hence "\<not>formulaFalse (getF state) (elements (getM state))" and "vars (elements (getM state)) \<supseteq> decisionVars"
    using assms
    using finalNonConflictState
    by auto
  with False
  show ?thesis
    by simp
qed

(*----------------------------------------------------------------------------*)
theorem finalStateCharacterization:
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes 
  "isInitialState state0 F0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars" and
  "isFinalState state F0 decisionVars"
  shows 
  "(getConflictFlag state = False \<and> 
      \<not>formulaFalse (getF state) (elements (getM state)) \<and> 
      vars (elements (getM state)) \<supseteq> decisionVars) \<or> 
   (getConflictFlag state = True \<and> 
      getC state = [])"
(*----------------------------------------------------------------------------*)
proof-
  from `isInitialState state0 F0` `(state0, state) \<in> transitionRelation F0 decisionVars`
  have "invariantsHoldInState state F0 decisionVars"
    using invariantsHoldInValidRunsFromInitialState
    by simp
  hence 
    *: "InvariantUniq (getM state)" 
    "InvariantReasonClauses (getF state) (getM state)" 
    "InvariantCFalse (getConflictFlag state) (getM state) (getC state)"
    unfolding invariantsHoldInState_def
    by auto

  from `isFinalState state F0 decisionVars` 
  have **: 
    "\<not> applicableDecide state decisionVars"
    "\<not> applicableConflict state"
    "\<not> applicableExplain  state" 
    "\<not> applicableLearn state" 
    "\<not> applicableBackjump state" 
    unfolding finalStateNonApplicable
    by auto

  from * **
  show ?thesis
    using finalStateCharacterizationLemma[of "state" "decisionVars"]
    by simp
qed

text{* Completeness theorems are easy consequences of this characterization and 
 soundness. *}
(*----------------------------------------------------------------------------*)
theorem completenessForSAT: 
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes 
  "satisfiable F0" and 

  "isInitialState state0 F0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars" and
  "isFinalState state F0 decisionVars"

  shows "getConflictFlag state = False \<and> \<not>formulaFalse (getF state) (elements (getM state)) \<and> 
               vars (elements (getM state)) \<supseteq> decisionVars"
(*----------------------------------------------------------------------------*)
proof-
  from assms
  have *: "(getConflictFlag state = False \<and> 
               \<not>formulaFalse (getF state) (elements (getM state)) \<and> 
               vars (elements (getM state)) \<supseteq> decisionVars) \<or> 
           (getConflictFlag state = True \<and> 
               getC state = [])"
    using finalStateCharacterization[of "state0" "F0" "state" "decisionVars"]
    by auto
  {
    assume "\<not> (getConflictFlag state = False)"
    with * 
    have "getConflictFlag state = True" "getC state = []"
      by auto
    with assms
      have "\<not> satisfiable F0"
      using soundnessForUNSAT
      by simp
    with `satisfiable F0`
    have False
      by simp
  }
  with * show ?thesis
    by auto
qed

(************************************************************************)
theorem completenessForUNSAT: 
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes 
  "vars F0 \<subseteq> decisionVars" and

  "\<not> satisfiable F0" and

  "isInitialState state0 F0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars" and
  "isFinalState state F0 decisionVars"

  shows 
  "getConflictFlag state = True \<and> getC state = []"
(************************************************************************)
proof-
  from assms
  have *: "(getConflictFlag state = False \<and> 
               \<not>formulaFalse (getF state) (elements (getM state)) \<and> 
               vars (elements (getM state)) \<supseteq> decisionVars) \<or> 
           (getConflictFlag state = True \<and> 
               getC state = [])"
    using finalStateCharacterization[of "state0" "F0" "state" "decisionVars"]
    by auto
  {
    assume "\<not> getConflictFlag state = True"
    with *
    have "getConflictFlag state = False \<and> \<not>formulaFalse (getF state) (elements (getM state)) \<and> vars (elements (getM state)) \<supseteq> decisionVars"
      by simp
    with assms
    have "satisfiable F0"
      using soundnessForSAT[of "F0" "decisionVars" "state0" "state"]
      unfolding satisfiable_def
      by auto
    with `\<not> satisfiable F0`
    have False
      by simp
  }
  with * show ?thesis
    by auto
qed

(************************************************************************)
theorem partialCorrectness: 
  fixes F0 :: Formula and decisionVars :: "Variable set" and state0 :: State and state :: State
  assumes 
  "vars F0 \<subseteq> decisionVars" and  

  "isInitialState state0 F0" and
  "(state0, state) \<in> transitionRelation F0 decisionVars" and
  "isFinalState state F0 decisionVars"

  shows 
  "satisfiable F0 = (\<not> getConflictFlag state)"
(************************************************************************)
using assms
using completenessForUNSAT[of "F0" "decisionVars" "state0" "state"]
using completenessForSAT[of "F0" "state0" "state" "decisionVars"]
by auto

end 
