(*<*)
theory AbstractLinearPoly  
imports Main Algebra 
begin
(*>*)

subsection{*Linear Polynomials and Constraints*}

(*<*)
(* ************************************************************************** *)
(* Abstract specification                                                     *)
(* ************************************************************************** *)

type_synonym var = nat

text{* (Infinite) linear polynomials as functions from vars to coeffs *}

definition fun_zero :: "var \<Rightarrow> 'a::zero" where
  [simp]: "fun_zero == \<lambda> v. 0"
definition fun_plus :: "(var \<Rightarrow> 'a) \<Rightarrow> (var \<Rightarrow> 'a) \<Rightarrow> var \<Rightarrow> 'a::plus" where
  [simp]: "fun_plus f1 f2 == \<lambda> v. f1 v + f2 v"
definition fun_scale :: "'a \<Rightarrow> (var \<Rightarrow> 'a) \<Rightarrow> (var \<Rightarrow> 'a::ring)" where
  [simp]: "fun_scale c f == \<lambda> v. c*(f v)"
definition fun_coeff :: "(var \<Rightarrow> 'a) \<Rightarrow> var \<Rightarrow> 'a" where
  [simp]: "fun_coeff f var = f var"
definition fun_vars :: "(var \<Rightarrow> 'a::zero) \<Rightarrow> var set" where
 [simp]: "fun_vars f = {v. f v \<noteq> 0}"
definition fun_vars_list :: "(var \<Rightarrow> 'a::zero) \<Rightarrow> var list" where
 [simp]: "fun_vars_list f = sorted_list_of_set {v. f v \<noteq> 0}"
definition fun_var :: "var \<Rightarrow> (var \<Rightarrow> 'a::{zero,one})" where
 [simp]: "fun_var x = (\<lambda> x'. if x' = x then 1 else 0)"
type_synonym 'a valuation = "var \<Rightarrow> 'a"
definition fun_valuate :: "(var \<Rightarrow> rat) \<Rightarrow> 'a valuation \<Rightarrow> ('a::rational_vector)" where
 [simp]: "fun_valuate lp val = (\<Sum>x\<in>{v. lp v \<noteq> 0}. lp x *R val x)"

text{* Invariant - only finitely many variables *}
definition inv where
 [simp]: "inv c == finite {v. c v \<noteq> 0}"

lemma inv_fun_zero [simp]: 
  "inv fun_zero" by simp

lemma inv_fun_plus [simp]: 
  "\<lbrakk>inv (f1 :: nat \<Rightarrow> 'a::monoid_add); inv f2\<rbrakk> \<Longrightarrow> inv (fun_plus f1 f2)"
proof-
  have *: "{v. f1 v + f2 v \<noteq> (0\<Colon>'a)} \<subseteq> {v. f1 v \<noteq> (0\<Colon>'a)} \<union> {v. f2 v \<noteq> (0\<Colon>'a)}"
    by auto
  assume "inv f1" "inv f2"
  thus ?thesis
    using *
    by (auto simp add: finite_subset)
qed

lemma inv_fun_scale [simp]: 
  "inv (f :: nat \<Rightarrow> 'a::ring) \<Longrightarrow> inv (fun_scale r f)"
proof-
  have *: "{v. r * (f v) \<noteq> 0} \<subseteq> {v. f v \<noteq> 0}" 
    by auto
  assume "inv f"
  thus ?thesis
    using *
    by (auto simp add: finite_subset)
qed

text{* linear_poly type -- rat coeffs *}
(* TODO: change rat to arbitrary ring *)

typedef (open)  linear_poly = "{c :: var \<Rightarrow> rat. inv c}"
by (rule_tac x="\<lambda> v. 0" in exI) auto
(*>*)

text{* Linear polynomials are of the form $a_1 \cdot x_1 + ... + a_n
\cdot x_n$. Their formalization follows the data-refinement approach
of Isabelle/HOL \cite{florian-refinement}. Abstract representation of
polynomials are functions mapping variables to their coefficients,
where only finitely many variables have non-zero
coefficients. Operations on polynomials are defined as operations on
functions. For example, the sum of @{text "p\<^isub>1"} and @{text "p\<^isub>2"} is
defined by @{text "\<lambda> v. p\<^isub>1 v + p\<^isub>2 v"} and the value of a polynomial
@{text "p"} for a valuation @{text "v"} (denoted by @{text "p\<lbrace>v\<rbrace>"}),
is defined by @{text "\<Sum>x\<in>{x. p x \<noteq> 0}. p x \<cdot> v x"}. Executable
representation of polynomials uses RBT mappings instead of functions.
*}

(*<*)
text{* Vector space operations on polynomials *}
instantiation linear_poly :: rational_vector
begin

definition zero_linear_poly :: "linear_poly" where
 "zero_linear_poly = Abs_linear_poly fun_zero"

definition plus_linear_poly :: "linear_poly \<Rightarrow> linear_poly \<Rightarrow> linear_poly" where
 "plus_linear_poly p1 p2 = Abs_linear_poly (fun_plus (Rep_linear_poly p1) (Rep_linear_poly p2))"

definition scaleRat_linear_poly :: "rat \<Rightarrow> linear_poly \<Rightarrow> linear_poly" where
 "scaleRat_linear_poly r p = Abs_linear_poly (fun_scale r (Rep_linear_poly p))"

definition uminus_linear_poly :: "linear_poly \<Rightarrow> linear_poly" where 
  "uminus_linear_poly lp = -1 *R lp"

definition minus_linear_poly :: "linear_poly \<Rightarrow> linear_poly \<Rightarrow> linear_poly" where
  "minus_linear_poly lp1 lp2 = lp1 + (- lp2)"

instance
proof
  fix a b c::linear_poly
  show "a + b + c = a + (b + c)"
  proof-
    let ?ra = "Rep_linear_poly a"
    let ?rb = "Rep_linear_poly b"
    let ?rc = "Rep_linear_poly c"
    
    have "inv (fun_plus ?ra ?rb)"
      using Rep_linear_poly
      by - (rule inv_fun_plus, simp_all)
    moreover
    have "inv (fun_plus ?rb ?rc)"
      using Rep_linear_poly
      by - (rule inv_fun_plus, simp_all)
    ultimately
    show ?thesis
      unfolding plus_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse field_simps)
  qed

next
  fix a b :: linear_poly
  show "a + b = b + a"
  proof-
    let ?ra = "Rep_linear_poly a"
    let ?rb = "Rep_linear_poly b"
    have "inv (fun_plus ?ra ?rb)"
      using Rep_linear_poly
      by - (rule inv_fun_plus, simp_all)
    thus ?thesis
      unfolding plus_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse field_simps)
  qed
next
  fix a::linear_poly
  show "0 + a = a"
    unfolding plus_linear_poly_def zero_linear_poly_def
    by (auto simp add: Abs_linear_poly_inverse Rep_linear_poly_inverse field_simps)
next
  fix a::linear_poly
  show "-a + a = 0"
  proof-
    let ?ra = "Rep_linear_poly a"
    have "inv (fun_scale -1 ?ra)"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    thus ?thesis
      unfolding uminus_linear_poly_def plus_linear_poly_def zero_linear_poly_def scaleRat_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse)
  qed
next
  fix a b :: linear_poly
  show "a - b = a + (- b)"
    unfolding minus_linear_poly_def ..
next
  fix a :: rat and x y :: linear_poly
  show "a *R (x + y) = a *R x + a *R y"
  proof-
    let ?rx = "Rep_linear_poly x"
    let ?ry = "Rep_linear_poly y"
    have "inv (fun_plus ?rx ?ry)"
      using Rep_linear_poly
      by - (rule inv_fun_plus, simp_all)
    moreover
    have "inv (fun_scale a ?rx)"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    moreover
    have "inv (fun_scale a ?ry)"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    ultimately
    show ?thesis
      unfolding scaleRat_linear_poly_def plus_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse field_simps)
  qed
next
  fix a b::rat and x::linear_poly
  show "(a + b) *R x = a *R x + b *R x"
  proof-
    let ?rx = "Rep_linear_poly x"
    have "inv (fun_scale a ?rx)"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    moreover
    have "inv (fun_scale b ?rx)"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    ultimately
    show ?thesis
      unfolding scaleRat_linear_poly_def plus_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse field_simps)
  qed
next
  fix a b :: rat and x::linear_poly
  show "a *R b *R x = (a * b) *R x"
  proof-
    have "inv (fun_scale b (Rep_linear_poly x))"
      using Rep_linear_poly
      by - (rule inv_fun_scale, simp_all)
    thus ?thesis
      unfolding scaleRat_linear_poly_def
      by (auto simp add: Abs_linear_poly_inverse field_simps)
  qed
next
  fix x::linear_poly
  show "1 *R x = x"
    unfolding scaleRat_linear_poly_def
    by (simp add: Rep_linear_poly_inverse)
qed

end

text{* Coefficient *}
definition coeff :: "linear_poly \<Rightarrow> var \<Rightarrow> rat" where
 "coeff lp var = fun_coeff (Rep_linear_poly lp) var"

lemma coeff_plus [simp] : "coeff (lp1 + lp2) var = coeff lp1 var + coeff lp2 var"
unfolding coeff_def plus_linear_poly_def
using inv_fun_plus[of "Rep_linear_poly lp1"] inv_fun_plus[of "Rep_linear_poly lp2"] 
using Abs_linear_poly_inverse[of "\<lambda>v. Rep_linear_poly lp1 v + Rep_linear_poly lp2 v"]
using Rep_linear_poly[of lp1] Rep_linear_poly[of lp2]
by auto

lemma coeff_scaleRat [simp]: "coeff (k *R lp1) var = k * coeff lp1 var"
unfolding coeff_def scaleRat_linear_poly_def
using Abs_linear_poly_inverse[of "\<lambda>v. k * Rep_linear_poly lp1 v"]
using Rep_linear_poly[of lp1]
by simp

lemma coeff_uminus [simp]: "coeff (-lp) var = - coeff lp var"
unfolding uminus_linear_poly_def
by (simp add: coeff_scaleRat)

lemma coeff_minus [simp]: "coeff (lp1 - lp2) var = coeff lp1 var - coeff lp2 var"
  unfolding minus_linear_poly_def
  by (simp add: coeff_uminus coeff_plus)

text{* Set of variables *}

definition vars :: "linear_poly \<Rightarrow> var set" where
 "vars lp = fun_vars (Rep_linear_poly lp)"

lemma linpoly_coeff: 
 "coeff p x \<noteq> 0 \<longleftrightarrow> x \<in> vars p"
unfolding coeff_def vars_def
by simp

lemma finite_vars:
 "finite (vars p)"
unfolding vars_def
using Rep_linear_poly
by simp

text{* List of variables *}
definition vars_list :: "linear_poly \<Rightarrow> var list" where
 "vars_list lp = fun_vars_list (Rep_linear_poly lp)"

lemma set_vars_list: "set (vars_list lp) = vars lp"
unfolding vars_def vars_list_def
using sorted_list_of_set[of "{v. Rep_linear_poly lp v \<noteq> 0}"]
using Rep_linear_poly[of lp]
by simp

text{* Construct single variable polynomial *}
definition Var :: "var \<Rightarrow> linear_poly" where
 "Var x = Abs_linear_poly (fun_var x)"

text{* Value of a polynomial in a given valuation *}
definition valuate :: "linear_poly \<Rightarrow> 'a valuation \<Rightarrow> ('a::rational_vector)" where
  "valuate lp val = fun_valuate (Rep_linear_poly lp) val"
syntax
  "_valuate" :: "linear_poly \<Rightarrow> 'a valuation \<Rightarrow> 'a"    ("_ \<lbrace> _ \<rbrace>")
translations
  "p\<lbrace>v\<rbrace> " == "CONST valuate p v"

lemma 
  valuate_diff: "(p \<lbrace>v1\<rbrace>) - (p \<lbrace>v2\<rbrace>) = p \<lbrace> \<lambda> x. v1 x - v2 x \<rbrace>"
unfolding valuate_def
by (auto simp add: setsum_subtractf[THEN sym] rational_vector.scale_right_diff_distrib)


lemma valuate_opposite_val: 
  shows "p \<lbrace> \<lambda> x. - v x \<rbrace> = - (p \<lbrace> v \<rbrace>)"
using valuate_diff[of p "\<lambda> x. 0" v]
by (auto simp add: valuate_def)

lemma valuate_nonneg:
  fixes v :: "'a::linordered_rational_vector valuation"
  assumes "\<forall> x \<in> vars p. (coeff p x > 0 \<longrightarrow> v x \<ge> 0) \<and> (coeff p x < 0 \<longrightarrow> v x \<le> 0)"
  shows "p \<lbrace> v \<rbrace> \<ge> 0"
unfolding valuate_def fun_valuate_def
proof (rule setsum_nonneg)
  show "\<forall>x\<in>{v. Rep_linear_poly p v \<noteq> 0}. (0\<Colon>'a) \<le> Rep_linear_poly p x *R v x"
  proof
    fix x
    assume "x \<in> {v. Rep_linear_poly p v \<noteq> 0}"
    thus "0 \<le> Rep_linear_poly p x *R v x"
      using assms
      using scaleRat_leq1[of 0 "v x" "coeff p x"]
      using scaleRat_leq2[of "v x" 0 "coeff p x"]
      by (cases "coeff p x > 0") (auto simp add: coeff_def vars_def)
  qed
qed
  
lemma valuate_nonpos:
  fixes v :: "'a::linordered_rational_vector valuation"
  assumes "\<forall> x \<in> vars p. (coeff p x > 0 \<longrightarrow> v x \<le> 0) \<and> (coeff p x < 0 \<longrightarrow> v x \<ge> 0)"
  shows "p \<lbrace> v \<rbrace> \<le> 0"
using assms
using valuate_opposite_val[of p v]
using valuate_nonneg[of p "\<lambda> x. - v x"]
using scaleRat_leq2[of "0::'a" _ "-1"]
using scaleRat_leq2[of _ "0::'a" "-1"]
by force

lemma valuate_uminus: "(-p) \<lbrace>v\<rbrace> = - (p \<lbrace>v\<rbrace>)"
unfolding valuate_def uminus_linear_poly_def scaleRat_linear_poly_def
using Abs_linear_poly_inverse[of "\<lambda>v. - Rep_linear_poly p v"]
using Rep_linear_poly[of p]
using setsum_negf[of  "\<lambda> x. Rep_linear_poly p x *R v x" "{v. Rep_linear_poly p v \<noteq> 0}"]
by simp

lemma valuate_add_lemma:
fixes v :: "'a \<Rightarrow> 'b::rational_vector"
assumes "finite {v. f1 v \<noteq> 0}" "finite {v. f2 v \<noteq> 0}"
shows
"(\<Sum>x\<in>{v. f1 v + f2 v \<noteq> 0}. (f1 x + f2 x) *R v x) =
   (\<Sum>x\<in>{v. f1 v \<noteq> 0}. f1 x *R v x) +  (\<Sum>x\<in>{v. f2 v \<noteq> 0}. f2 x *R v x)"
proof-
  let ?A = "{v. f1 v + f2 v \<noteq> 0} \<union> {v. f1 v + f2 v = 0 \<and> (f1 v \<noteq> 0 \<or> f2 v \<noteq> 0)}"
  have "?A = {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}"
    by auto
  hence
    "finite ?A"
    using assms
    by (subgoal_tac "{v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0} = {v. f1 v \<noteq> 0} \<union> {v. f2 v \<noteq> 0}") auto

  hence "(\<Sum>x\<in>{v. f1 v + f2 v \<noteq> 0}. (f1 x + f2 x) *R v x) = 
    (\<Sum>x\<in>{v. f1 v + f2 v \<noteq> 0} \<union> {v. f1 v + f2 v = 0 \<and> (f1 v \<noteq> 0 \<or> f2 v \<noteq> 0)}. (f1 x + f2 x) *R v x)"
    by (rule setsum_mono_zero_left) auto
  also have "... = (\<Sum>x \<in> {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}. (f1 x + f2 x) *R v x)"
    by (rule setsum_cong) auto
  also have "... = (\<Sum>x \<in> {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}. f1 x *R v x) + 
                   (\<Sum>x \<in> {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}. f2 x *R v x)"
    by (simp add: scaleRat_left_distrib setsum_addf)
  also have "... = (\<Sum>x\<in>{v. f1 v \<noteq> 0}. f1 x *R v x) +  (\<Sum>x\<in>{v. f2 v \<noteq> 0}. f2 x *R v x)"
  proof-
    {
      fix f1 f2::"'a \<Rightarrow> rat"
      assume "finite {v. f1 v \<noteq> 0}" "finite {v. f2 v \<noteq> 0}"
      hence "finite {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0 \<and> f1 v = 0}"
        by (subgoal_tac "{v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0} = {v. f1 v \<noteq> 0} \<union> {v. f2 v \<noteq> 0}") auto
      have "(\<Sum>x\<in>{v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}. f1 x *R v x) = 
        (\<Sum>x\<in>{v. f1 v \<noteq> 0 \<or> (f2 v \<noteq> 0 \<and> f1 v = 0)}. f1 x *R v x)"
        by auto
      also have "... = (\<Sum>x\<in>{v. f1 v \<noteq> 0}. f1 x *R v x)"
        using `finite {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0 \<and> f1 v = 0}`
        by (rule setsum_mono_zero_left[THEN sym]) auto
      ultimately have "(\<Sum>x\<in>{v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}. f1 x *R v x) = 
        (\<Sum>x\<in>{v. f1 v \<noteq> 0}. f1 x *R v x)"
        by simp
    }
    note * = this
    show ?thesis
      using assms
      using *[of f1 f2]
      using *[of f2 f1]
      by (subgoal_tac "{v. f2 v \<noteq> 0 \<or> f1 v \<noteq> 0} = {v. f1 v \<noteq> 0 \<or> f2 v \<noteq> 0}") auto
  qed
  ultimately
  show ?thesis
    by simp
qed

lemma valuate_add:  "(p1 + p2) \<lbrace>v\<rbrace> = (p1 \<lbrace>v\<rbrace>) + (p2 \<lbrace>v\<rbrace>)"
using Rep_linear_poly[of p1]  Rep_linear_poly[of p2]
using Abs_linear_poly_inverse[of "\<lambda>v. Rep_linear_poly p1 v + Rep_linear_poly p2 v"]
using inv_fun_plus[of "Rep_linear_poly p1" "Rep_linear_poly p2"]
by (simp add: valuate_def plus_linear_poly_def) (rule  valuate_add_lemma)

lemma valuate_minus: "(p1 - p2) \<lbrace>v\<rbrace> = (p1 \<lbrace>v\<rbrace>) - (p2 \<lbrace>v\<rbrace>)"
unfolding minus_linear_poly_def
by (simp add: valuate_add valuate_uminus)


lemma valuate_scaleRat:
 "(c *R lp) \<lbrace> v \<rbrace> = c *R ( lp\<lbrace>v\<rbrace> )"
proof (cases "c=0")
  case True
  thus ?thesis
    by (auto simp add: valuate_def zero_linear_poly_def Abs_linear_poly_inverse)
next
  case False
  hence "\<And> v. Rep_linear_poly (c *R lp) v = c * (Rep_linear_poly lp v)"
    unfolding scaleRat_linear_poly_def
    using Abs_linear_poly_inverse[of "\<lambda>v. c * Rep_linear_poly lp v"]
    using Rep_linear_poly
    by auto
  thus ?thesis
    unfolding valuate_def
    using `c \<noteq> 0`
    by auto (subst rational_vector.scale_setsum_right, auto)
qed

lemma valuate_Var: "Var x \<lbrace>v\<rbrace> = v x"
unfolding valuate_def Var_def
using Abs_linear_poly_inverse
by simp

lemma zero_coeff_zero: "p = 0 \<longleftrightarrow> (\<forall> v. coeff p v = 0)"
proof (safe)
  fix v
  show "coeff 0 v = 0"
    unfolding coeff_def zero_linear_poly_def
    using Abs_linear_poly_inverse
    by simp
next
  obtain fp where "p = Abs_linear_poly fp" "inv fp"
    by (cases p) auto

  assume "\<forall> v. coeff p v = 0"
  thus "p = 0"
    unfolding coeff_def zero_linear_poly_def
    using `p = Abs_linear_poly fp` `inv fp`
    using Abs_linear_poly_inject[of fp "\<lambda> _. 0"]
    using Abs_linear_poly_inverse[of fp]
    by (auto simp add: ext)
qed

lemma all_val: 
  assumes "\<forall> (v::var \<Rightarrow> 'a::lrv). \<exists> v'. (\<forall> x \<in> vars p. v' x = v x) \<and> (p \<lbrace>v'\<rbrace> = 0)"
  shows "p = 0"
proof (subst zero_coeff_zero, rule allI)
  fix x
  show "coeff p x = 0"
  proof (cases "x \<in> vars p")
    case False
    thus ?thesis
      using linpoly_coeff[of p x]
      by simp
  next
    case True
    have "(0::'a::lrv) \<noteq> (1::'a)"
      using zero_neq_one
      by auto

    let ?v = "\<lambda> x'. if x = x' then 1 else 0::'a"
    obtain v' where "(\<forall> x \<in> vars p. v' x = ?v x) \<and> p \<lbrace>v'\<rbrace> = 0"
      using assms
      by (erule_tac x="?v" in allE) auto
    hence "\<forall> x' \<in> vars p. v' x' = (if x = x' then 1 else 0)" "p \<lbrace>v'\<rbrace> = 0"
      by auto

    let ?fp = "Rep_linear_poly p"
    have "{x. ?fp x \<noteq> 0 \<and> v' x \<noteq> (0\<Colon>'a)} = {x}"
      using `x \<in> vars p` unfolding vars_def
    proof (safe, simp_all)
      fix x'
      assume "v' x' \<noteq> 0" "Rep_linear_poly p x' \<noteq> 0"
      thus "x' = x"
        using `\<forall> x' \<in> vars p. v' x' = (if x = x' then 1 else 0)`
        unfolding vars_def
        by (erule_tac x="x'" in ballE) (simp_all split: split_if_asm)
    next
      assume "v' x = 0" "Rep_linear_poly p x \<noteq> 0"
      thus False
        using `\<forall> x' \<in> vars p. v' x' = (if x = x' then 1 else 0)`
        using `0 \<noteq> 1`
        unfolding vars_def
        by simp
    qed
    
    have "p \<lbrace>v'\<rbrace> = (\<Sum>x\<in>{v. ?fp v \<noteq> 0}. ?fp x *R v' x)"
      unfolding valuate_def
      by auto
    also have "... = (\<Sum>x\<in>{v. ?fp v \<noteq> 0 \<and> v' v \<noteq> 0}. ?fp x *R v' x)"
      apply (rule setsum_mono_zero_left[THEN sym])
      using Rep_linear_poly[of p]
      by auto
    also have "... = ?fp x *R v' x"
      using `{x. ?fp x \<noteq> 0 \<and> v' x \<noteq> (0\<Colon>'a)} = {x}`
      by simp
    also have "... = ?fp x *R 1"
      using `x \<in> vars p`
      using `\<forall> x' \<in> vars p. v' x' = (if x = x' then 1 else 0)`
      by simp
    ultimately
    have "p \<lbrace>v'\<rbrace> = ?fp x *R 1"
      by simp
    hence "coeff p x *R (1::'a)= 0"
      using `p \<lbrace>v'\<rbrace> = 0`
      unfolding coeff_def
      by simp
    thus ?thesis
      using rational_vector.scale_eq_0_iff
      using `0 \<noteq> 1`
      by simp
  qed
qed

lemma vars_uminus [simp]: "vars (-p) = vars p"
unfolding vars_def uminus_linear_poly_def scaleRat_linear_poly_def
using Abs_linear_poly_inverse Rep_linear_poly
by auto

lemma vars_plus [simp]: "vars (p1 + p2) \<subseteq> vars p1 \<union> vars p2"
unfolding vars_def plus_linear_poly_def
using inv_fun_plus[of "Rep_linear_poly p1" "Rep_linear_poly p2"]
using Abs_linear_poly_inverse Rep_linear_poly
by auto

lemma vars_minus [simp]: "vars (p1 - p2) \<subseteq> vars p1 \<union> vars p2"
unfolding minus_linear_poly_def
using vars_plus[of p1 "-p2"] vars_uminus[of p2]
by simp

lemma vars_scaleRat1: "vars (c *R p) \<subseteq> vars p"
unfolding scaleRat_linear_poly_def vars_def
using inv_fun_scale Abs_linear_poly_inverse Rep_linear_poly
by auto

lemma vars_scaleRat: "c \<noteq> 0 \<Longrightarrow> vars(c *R p) = vars p"
unfolding scaleRat_linear_poly_def vars_def
using inv_fun_scale Abs_linear_poly_inverse Rep_linear_poly
by simp

lemma vars_Var [simp]: "vars (Var x) = {x}"
unfolding Var_def vars_def
using Abs_linear_poly_inverse
by simp

lemma coeff_Var1 [simp]: "coeff (Var x) x = 1"
unfolding Var_def coeff_def
using Abs_linear_poly_inverse
by simp

lemma coeff_Var2: "x \<noteq> y \<Longrightarrow> coeff (Var x) y = 0"
unfolding Var_def coeff_def
using Abs_linear_poly_inverse
by simp

lemma valuate_depend:
  assumes "\<forall> x \<in> vars p. v x = v' x"
  shows "(p \<lbrace>v\<rbrace>) = (p \<lbrace>v'\<rbrace>)"
using assms
unfolding vars_def
unfolding valuate_def
by simp

lemma valuate_update_x_lemma:
  fixes v1 v2 :: "'a::rational_vector valuation"
  assumes
  "\<forall>y. f y \<noteq> 0 \<longrightarrow> y \<noteq> x \<longrightarrow> v1 y = v2 y"
  "finite {v. f v \<noteq> 0}"
  shows
  "(\<Sum>x\<in>{v. f v \<noteq> 0}. f x *R v1 x) + f x *R (v2 x - v1 x) = (\<Sum>x\<in>{v. f v \<noteq> 0}. f x *R v2 x)"
proof (cases "f x = 0")
  case True
  hence "\<forall>y. f y \<noteq> 0 \<longrightarrow> v1 y = v2 y"
    using assms(1)
    by auto
  thus ?thesis
    using `f x = 0`
    by auto
next
  case False
  let ?A = "{v. f v \<noteq> 0}" and ?Ax = "{v. v \<noteq> x \<and> f v \<noteq> 0}"
  have "?A = ?Ax \<union> {x}"
    using `f x \<noteq> 0`
    by auto
  hence "(\<Sum>x\<in>?A. f x *R v1 x) = f x *R v1 x + (\<Sum>x\<in>?Ax. f x *R v1 x)"
        "(\<Sum>x\<in>?A. f x *R v2 x) = f x *R v2 x + (\<Sum>x\<in>?Ax. f x *R v2 x)"
    using assms(2)
    by auto
  moreover
  have "\<forall> y \<in> ?Ax. v1 y = v2 y"
    using assms
    by auto
  moreover
  have "f x *R v1 x + f x *R (v2 x - v1 x) = f x *R v2 x"
    by (subst rational_vector.scale_right_diff_distrib) auto
  ultimately
  show ?thesis
    by simp
qed

lemma valuate_update_x:
  fixes v1 v2 :: "'a::rational_vector valuation"
  assumes "\<forall>y \<in> vars lp. y\<noteq>x \<longrightarrow> v1 y = v2 y"
  shows "lp \<lbrace>v1\<rbrace>  + coeff lp x *R (v2 x - v1 x) = lp \<lbrace>v2\<rbrace>"
  using assms
  unfolding valuate_def vars_def coeff_def
  using valuate_update_x_lemma[of "Rep_linear_poly lp" x v1 v2] Rep_linear_poly
  by auto

lemma coeff_zero:
  "coeff lp v \<noteq> 0 \<equiv> v \<in> vars lp"
  by (auto simp add: coeff_def  vars_def)

lemma vars_zero:
  "vars 0 = {}"
  using zero_coeff_zero
  using coeff_zero
  by auto

lemma vars_empty_zero: 
  "vars lp = {} \<longleftrightarrow> lp = 0"
  using zero_coeff_zero
  using coeff_zero
  by auto

definition max_var:: "linear_poly \<Rightarrow> var" where
"max_var lp \<equiv> if lp = 0 then undefined else Max (vars lp)"

lemma max_var_max:
  assumes "a \<in> vars lp"
  shows "max_var lp \<ge> a"
  using assms
  by (auto simp add: finite_vars max_var_def vars_zero)

lemma [code]: 
  "max_var lp = (let vl = vars_list lp 
                in if vl = [] then undefined else foldl max (hd vl) (tl vl))"
proof (cases "lp = (0::linear_poly)")
  case True
  thus ?thesis
    using set_vars_list[of lp]
    by (auto simp add: max_var_def vars_zero)
next
  case False
  thus ?thesis
    using set_vars_list[of lp, THEN sym]
    using vars_empty_zero[of lp]
    using Max_fin_set_fold[of "hd (vars_list lp)" "tl (vars_list lp)"]
    by (auto simp add: Let_def max_var_def)
qed

definition monom_var:: "linear_poly \<Rightarrow> var" where
"monom_var l = max_var l"

definition monom_coeff:: "linear_poly \<Rightarrow> rat" where
"monom_coeff l = coeff l (monom_var l)"

definition is_monom :: "linear_poly \<Rightarrow> bool" where
  "is_monom l \<longleftrightarrow> length (vars_list l) = 1"

lemma is_monom_vars_not_empty:
  "is_monom l \<Longrightarrow> vars l \<noteq> {}"
  by (auto simp add: is_monom_def vars_list_def) (auto simp add: AbstractLinearPoly.vars_def)

lemma monom_var_in_vars:
  "is_monom l \<Longrightarrow> monom_var l \<in> vars l"
  using vars_zero
  by (auto simp add: monom_var_def max_var_def is_monom_vars_not_empty finite_vars is_monom_def)

lemma is_monom_monom_coeff_not_zero:
  "is_monom l \<Longrightarrow> monom_coeff l \<noteq> 0"
  by (simp add: coeff_zero monom_var_in_vars monom_coeff_def)

lemma list_two_elements:
  "\<lbrakk>y \<in> set l; x \<in> set l; length l = Suc 0; y \<noteq> x\<rbrakk> \<Longrightarrow> False"
  by (induct l) auto

lemma is_monom_vars_monom_var:
  assumes "is_monom l"
  shows "vars l = {monom_var l}"
proof-
  have "\<And>x. \<lbrakk>is_monom l; x \<in> vars l\<rbrakk> \<Longrightarrow> monom_var l = x"
  proof-
    fix x
    assume "is_monom l" "x \<in> vars l"
    hence "x \<in> set (vars_list l)"
      using finite_vars
      by (auto simp add: vars_list_def vars_def)
    show "monom_var l = x"
    proof(rule ccontr)
      assume "monom_var l \<noteq> x"
      hence "\<exists>y. monom_var l = y \<and> y \<noteq> x"
        by simp
      then obtain y where "monom_var l = y" "y \<noteq> x"
        by auto
      hence "Rep_linear_poly l y \<noteq> 0"
        using monom_var_in_vars `is_monom l`
        by (auto simp add: vars_def)
      hence "y \<in> set (vars_list l)"
        using finite_vars
        by (auto simp add: vars_def vars_list_def)
      thus False
        using `x \<in> set (vars_list l)` `is_monom l` `y \<noteq> x`
        using list_two_elements
        by (simp add: is_monom_def)
    qed
  qed
  thus "vars l = {monom_var l}"
    using assms
    by (auto simp add: monom_var_in_vars)
qed

lemma monom_valuate:
  assumes "is_monom m"
  shows "m\<lbrace>v\<rbrace> = (monom_coeff m) *R v (monom_var m)"
  using assms
  using is_monom_vars_monom_var
  by (simp add: vars_def fun_vars_def coeff_def monom_coeff_def fun_valuate_def valuate_def)

end
(*>*)
