section \<open>Generating Basic Combinatorial Objects (Permutations and Combinations)\<close>

theory Combinatorics
  imports Main "HOL-Library.Permutation"
          More_List More_Multiset
begin

subsection \<open>Generating all permutations\<close>

primrec interleave :: "'a \<Rightarrow> 'a list \<Rightarrow> 'a list list" where
  "interleave x [] = [[x]]"
| "interleave x (h # t) = (x # (h # t)) # (map (\<lambda> l. h # l) (interleave x t))"

text\<open>For example, @{lemma "interleave (1::nat) [2, 3, 4] = [[1, 2, 3, 4], [2, 1, 3, 4], [2, 3, 1, 4], [2, 3, 4, 1]]" by simp}.\<close>

primrec permute :: "'a list \<Rightarrow> 'a list list" where
  "permute [] = [[]]"
| "permute (h # t) = concat (map (\<lambda> l. interleave h l) (permute t))"

text \<open>For example, @{lemma "permute [1::nat, 2, 3] = [[1, 2, 3], [2, 1, 3], [2, 3, 1], [1, 3, 2], [3, 1, 2], [3, 2, 1]]" by simp}.\<close>

lemma interleave_multiset [dest]: 
  shows "p \<in> set (interleave h a) \<Longrightarrow> mset p = mset a + {#h#}"
proof (induct a arbitrary: p) 
  case Nil
  thus ?case
    by simp
next
  case (Cons h' t')
  thus ?case
    using add.commute[of "{#h#}" "{#h'#}"]
    using add.assoc [of "mset t'" "{#h#}" "{#h'#}"]
    using add.assoc [of "mset t'" "{#h'#}" "{#h#}"]
    by auto
qed

lemma interleave_not_empty [simp]:
  shows "interleave h t \<noteq> []"
  by (induction t) auto

lemma interleave_hd [simp]:
  shows "hd (interleave h t) = h # t"
  by (induction t) auto

lemma interleave_hd_weak [simp]:
  shows "h # t \<in> set (interleave h t)"
  by (cases "t") auto

lemma interleave_append [simp]: 
  shows "t1 @ [h] @ t2 \<in> set (interleave h (t1 @ t2))"
by (induct t1) auto

lemma interleave_length [simp]:
  shows "length (interleave h t) = length t + 1"
  by (induction t) auto

lemma isPermutation_permute: 
  shows "p \<in> set (permute xs) \<Longrightarrow> p <~~> xs"
proof (induct xs arbitrary: p)
  case Nil
  thus ?case
    by (simp add: mset_eq_perm)
next
  case (Cons h t)
  thus ?case
    unfolding mset_eq_perm[THEN sym]
    by auto
qed

(* TODO: Move to MoreMultiset *)
lemma mset_append:
  assumes "mset p = mset t + {#h#}"
  shows "\<exists> t1 t2. p = t1 @ [h] @ t2 \<and> mset t = mset t1 + mset t2"
  using assms
proof (induct p arbitrary: t)
  case Nil
  thus ?case
    by simp
next
  case (Cons a p')
  show ?case
  proof (cases "a = h")
    case True
    thus ?thesis
      using Cons(2)
      by (rule_tac x="[]" in exI) auto
  next
    case False
    hence "h \<in># mset p'"
      using Cons(2) insert_noteq_member[of a "mset p'"] 
      by auto
    hence "mset p' = mset p' - {#h#} + {#h#}"
      using insert_DiffM2[of h "mset p'"]
      by simp
    thus ?thesis
      using Cons(1)[of "remove1 h p'"] Cons(2)
      by (auto, rule_tac x="a#t1" in exI, auto) 
  qed
qed

lemma permute_isPermutation: 
  shows "p <~~> xs \<Longrightarrow> p \<in> set (permute xs)"
proof (induct xs arbitrary: p)
  case Nil
  thus ?case
    by simp
next
  case (Cons h t)
  obtain t1 t2 where "p = t1 @ [h] @ t2" "mset t = mset t1 + mset t2"
    using Cons(2)
    unfolding mset_eq_perm[THEN sym]
    using mset_append[of p t h]
    by auto
  thus ?case
    using Cons(1)[of "t1 @ t2"] Cons(2)
    using add.assoc[of "mset t1" "mset t2" "{#h#}"]
    using interleave_append[of t1 h t2]
    unfolding mset_eq_perm[symmetric]
    by auto
qed

lemma permute_bij:
  assumes "p \<in> set (permute xs)"
  shows "\<exists> f. bij_betw f {..<length p} {..<length xs} \<and>
       (\<forall> i<length p. p ! i = xs ! (f i))"
  using assms
  using isPermutation_permute[of p xs]
  using permutation_Ex_bij[of p xs]
  by auto

lemma permute_member_set:
  assumes "pm \<in> set (permute xs)"
  shows  "set pm = set xs"
  using assms
  using isPermutation_permute perm_set_eq 
  by blast

lemma permute_member_length:
  assumes  "pm \<in> set (permute xs)"
  shows  "length pm = length xs"
  using assms
  using isPermutation_permute perm_length
  by blast

lemma permute_member_mset:
  assumes "pm \<in> set (permute xs)"
  shows  "mset pm = mset xs"
  using assms
  using isPermutation_permute 
  by (simp add: isPermutation_permute mset_eq_perm)
  
definition product where
  "product xs = foldl (*) 1 xs"
definition fact :: "nat \<Rightarrow> nat" where
  "fact n = product [1..<n+1]"

lemma fact_0 [simp]:
  "fact 0 = 1"
  unfolding fact_def product_def
  by simp

lemma fact_Suc [simp]:
  shows "fact (Suc n) = Suc n * fact n"
  unfolding fact_def product_def
  by auto

lemma fact_gt_0 [simp]:
  shows "fact n > 0"
  by (induction n, auto)
  
lemma permute_length [simp]:
  shows "length (permute xs) = fact (length xs)"
proof (induction xs)
  case Nil
  thus ?case
    by (simp add: fact_def product_def)
next
  case (Cons a l)
  have "(\<Sum>x\<leftarrow>permute l. Suc (length x)) = length (permute l) * Suc (length l)"
    using sum_list_const[of "permute l" "\<lambda> l. Suc (length l)" "Suc (length l)"]
    using permute_member_length[of _ l]
    by auto
  then show ?case
    using Cons
    by (auto simp add: length_concat comp_def)
qed                                                           

lemma permute_hd_weak:
  shows "xs \<in> set (permute xs)"
  using permute_isPermutation[of xs xs]
  by simp

lemma permute_not_empty_hd:
  shows "permute xs \<noteq> [] \<and> hd (permute xs) = xs"
proof (induction xs)
  case Nil
  then show ?case by simp
next
  case (Cons h t)
  let ?M = "map (interleave h) (permute t)"
  let ?C = "concat ?M"  

  have "?M \<noteq> []"  "hd ?M = interleave h t"
    using Cons[THEN conjunct1] Cons[THEN conjunct2]
    by (auto simp add: hd_map)

  have "hd ?M \<noteq> []"  "hd (hd ?M) = h # (hd (permute t))"
    using `?M \<noteq> []` Cons
    by (simp_all add: hd_map)

  have "concat ?M \<noteq> []"
    using `?M \<noteq> []` `hd ?M \<noteq> []`
    by (metis take_Nil take_concat)
    
  hence "hd ?C = h # (hd (permute t))"
    using \<open>hd ?M \<noteq> []\<close> \<open>?M \<noteq> []\<close> \<open>hd (hd ?M) = h # hd (permute t)\<close>
    by auto

  thus ?case 
    using `?C \<noteq> []`
    using Cons
    by simp
qed

lemmas permute_not_empty[simp] = permute_not_empty_hd[THEN conjunct1]
lemmas permute_hd[simp] = permute_not_empty_hd[THEN conjunct2]

(* Action of permutation on lists *)
type_synonym perm = "nat list"

definition permute_list :: "perm \<Rightarrow> 'a list \<Rightarrow> 'a list" where 
  "permute_list p l = map (\<lambda> i. l ! i) p"

lemma permute_list_length [simp]: 
  shows "length (permute_list l p) = length l"
  unfolding permute_list_def
  by simp

lemma permute_list_set [simp]:
  assumes "length p = length l" "set p = {0..<length l}"
  shows "set (permute_list p l) = set l"
  using assms
  unfolding permute_list_def
  by (simp add: nth_image)

lemma permute_list_nth [simp]:
  assumes "i < length pm"
  shows "permute_list pm xs ! i = xs ! (pm ! i)"
  unfolding permute_list_def
  using assms
  by auto

lemma permute_list_mset [simp]:
  assumes "p \<in> set (permute [0..<length xs])"
  shows "mset (permute_list p xs) = mset xs"
proof-
  from assms have "mset p = mset [0..<length xs]"
    using isPermutation_permute mset_eq_perm by blast
  moreover
  have *: "xs = map ((!) xs) [0..<length xs]"
    by (simp add: map_nth)
  hence  "mset xs = image_mset ((!) xs) (mset_set {0..<length xs})"
    by (subst *, subst mset_map, simp)
  ultimately
  show ?thesis
    using assms
    unfolding permute_list_def
    by simp
qed

lemma sum_list_permute_list [simp]:
  fixes xs::"'a::comm_monoid_add list"
  assumes "length xs = n" "p \<in> set (permute [0..<n])"
  shows "sum_list (permute_list p xs) = sum_list xs"
  apply (rule mset_eq_sum_list_eq)
  using assms
  by simp

lemma map_mset_permute_list:
  assumes "p \<in> set (permute [0..<length xs])"
  shows  "map mset (permute_list p xs) = permute_list p (map mset xs)"
  using assms
  unfolding permute_list_def
  by (auto simp add: permute_member_set)

(* Identitity permutation *)
definition perm_id :: "nat \<Rightarrow> perm" where
  "perm_id n = [0..<n]"

lemma perm_id_permute [simp]:
  shows "perm_id n \<in> set (permute [0..<n])"
  using permute_hd_weak[of "[0..<n]"]
  by (simp add: perm_id_def)

lemma permute_list_id [simp]:
  shows "permute_list (perm_id (length xs)) xs = xs"
  unfolding permute_list_def perm_id_def
  by (auto simp add: map_nth)

(* Inverse permutation *)
definition perm_inv :: "perm \<Rightarrow> perm"  where  
  "perm_inv p = map (index_of p) [0..<length p]"

lemma perm_inv_length [simp]:
  shows "length (perm_inv p) = length p"
  unfolding perm_inv_def
  by simp

lemma perm_inv_nth_length [dest]:
  assumes "pm \<in> set (permute [0..<length xs])" "p < length xs"
  shows "perm_inv pm ! p < length pm"
  using assms
  unfolding perm_inv_def
  by (metis add.left_neutral atLeast0LessThan atLeastLessThan_upt length_map length_upt lessThan_iff map_nth nth_map_upt permute_member_set permute_member_length index_of_in_set)
  
lemma perm_inv_perm [simp]:
  assumes "pm \<in> set (permute [0..<length xs])" "p < length xs"
  shows "perm_inv pm ! (pm ! p) = p"
proof-
  have "pm ! p < length pm"
    using assms permute_member_set[of pm "[0..<length xs]"] permute_member_length[of pm "[0..<length xs]"]
    by auto
  moreover
  have "distinct pm"
    using assms permute_isPermutation[of pm "[0..<length xs]"]
    using distinct_upt isPermutation_permute perm_distinct_iff by blast
  ultimately
  show ?thesis
    using assms
    unfolding perm_inv_def
    by (auto simp add: index_of_list_element permute_member_length)
qed

lemma perm_perm_inv [simp]:
  assumes "pm \<in> set (permute [0..<length xs])" "p < length xs"
  shows "pm ! (perm_inv pm ! p) = p"
  using assms
  unfolding perm_inv_def
  by (metis add.left_neutral atLeast_upt length_map length_upt lessThan_iff map_nth nth_map_upt permute_member_set permute_member_length index_of_in_set)

lemma perm_inv_distinct [simp]:
  assumes "distinct p" "set p \<subseteq> {0..<length p}"
  shows "distinct (perm_inv p)"
proof (subst distinct_conv_nth, safe)
  fix i j
  assume "i < length (perm_inv p)" "j < length (perm_inv p)" "i \<noteq> j" "perm_inv p ! i = perm_inv p ! j"
  thus False
    using assms
    unfolding perm_inv_def
    by (metis add.left_neutral atLeast0LessThan card_atLeastLessThan card_subset_eq distinct_card finite_atLeastLessThan index_of_in_set perm_inv_def perm_inv_length length_map length_upt lessThan_iff nth_map_upt)
qed

lemma perm_inv_set_subset [simp]:
  assumes "set p = {0..<n}" "length p = n"
  shows "set (perm_inv p) \<subseteq> {0..<n}"
  using assms
  unfolding perm_inv_def
  by (smt atLeast0LessThan atLeastLessThan_upt image_iff index_of_in_set lessThan_iff set_map subsetI)

lemma perm_inv_set [simp]:
  assumes "distinct p" "set p \<subseteq> {0..<n}" "length p = n"
  shows "set (perm_inv p) = {0..<n}"
  using assms
  by (metis card_atLeastLessThan card_subset_eq diff_zero distinct_card perm_inv_distinct finite_atLeastLessThan perm_inv_length perm_inv_set_subset)
  
lemma perm_inv_mset [simp]:
  assumes "p \<in> set (permute [0..<n])"
  shows "mset (perm_inv p) = mset [0..<n]"
  using assms
  by (metis atLeastLessThan_upt diff_zero perm_inv_distinct distinct_upt perm_inv_set isPermutation_permute permute_member_length length_upt mset_set_set order_refl perm_distinct_iff permute_member_set)
  

lemma perm_inv_permute [simp]:
  assumes "p \<in> set (permute [0..<n])"
  shows "perm_inv p \<in> set (permute [0..<n])"
  using assms
  using perm_inv_mset mset_eq_perm permute_isPermutation
  by blast  

lemma permute_list_perm_inv1 [simp]:
  assumes "p \<in> set (permute [0..<n])" "length p = n"
  shows "permute_list p (perm_inv p) = [0..<n]"
proof (subst list_eq_iff_nth_eq, safe)
  show "length (permute_list p (perm_inv p)) = length [0..<n]"
    using assms
    by simp
next
  fix i
  assume "i < length (permute_list p (perm_inv p))"
  thus "permute_list p (perm_inv p) ! i = [0..<n] ! i"
    unfolding permute_list_def
    using perm_inv_perm[of p p i] assms
    by simp
qed

lemma permute_list_perm_inv2:
  assumes "p \<in> set (permute [0..<n])" "length p = n"
  shows "permute_list (perm_inv p) p = [0..<n]"
proof (subst list_eq_iff_nth_eq, safe)
  show "length (permute_list (perm_inv p) p) = length [0..<n]"
    using assms
    by simp
next
  fix i
  assume "i < length (permute_list (perm_inv p) p)"
  thus "permute_list (perm_inv p) p ! i = [0..<n] ! i"
    unfolding permute_list_def
    using perm_perm_inv[of p p i] assms
    by simp
qed

(* Composition of permutations *)

definition perm_comp :: "nat list \<Rightarrow> nat list \<Rightarrow> nat list" where
  "perm_comp p1 p2 = permute_list p2 p1"

lemma perm_comp_permute [simp]:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])"
  shows  "perm_comp p1 p2 \<in> set (permute [0..<n])"
proof (rule permute_isPermutation)
  show "perm_comp p1 p2 <~~> [0..<n]"
    using isPermutation_permute assms
    unfolding perm_comp_def
    by (metis (mono_tags, lifting) diff_zero permute_member_length length_upt mset_eq_perm permute_list_mset)
qed

lemma permute_list_perm_comp [simp]:
  assumes "p1 \<in> set (permute [0..<length l])" "p2 \<in> set (permute [0..<length l])"
  shows "permute_list (perm_comp p1 p2) l = permute_list p2 (permute_list p1 l)"
  using assms
  unfolding perm_comp_def permute_list_def
  by auto (metis atLeastLessThan_iff atLeastLessThan_upt length_map map_nth nth_map permute_member_length permute_member_set)

lemma permute_list_perm_inv_1 [simp]:
  assumes "p \<in> set (permute [0..<length xs])"
  shows "permute_list p (permute_list (perm_inv p) xs) = xs"
  by (metis assms perm_id_def perm_inv_permute length_map map_nth perm_comp_def permute_list_id permute_list_perm_inv1 permute_list_perm_comp permute_member_length)

lemma permute_list_perm_inv_2 [simp]:
  assumes "p \<in> set (permute [0..<length xs])"
  shows "permute_list (perm_inv p) (permute_list p xs) = xs"
  by (metis assms perm_inv_permute permute_list_perm_inv_1 permute_list_length)

lemma perm_comp_perm_inv_id_1: 
  assumes "p \<in> set (permute [0..<n])"
  shows "perm_comp (perm_inv p) p = [0..<n]"
  using assms permute_member_length permute_list_perm_inv1
  unfolding perm_comp_def 
  by (metis diff_zero length_upt)
  

lemma perm_comp_perm_inv_id_2: 
  assumes "p \<in> set (permute [0..<n])"
  shows "perm_comp p (perm_inv p) = [0..<n]"
  using assms permute_member_length permute_list_perm_inv2
  unfolding perm_comp_def 
  by (metis diff_zero length_upt)
  

(* ************************************************************************** *)
subsection\<open>Generating all combinations\<close>
(* ************************************************************************** *)

fun combine_aux :: "'a list \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> 'a list list" where
 "combine_aux l n k = 
     (if k = 0 then [[]] else 
      if k = n then [l] else
      (case l of 
          [] \<Rightarrow> []
      | (h # t) \<Rightarrow> 
             (map (\<lambda> l'. h # l') (combine_aux t (n-(1::nat)) (k-(1::nat)))) @ 
                   combine_aux t (n-(1::nat)) k))"
declare combine_aux.simps[simp del]

definition combine :: "'a list \<Rightarrow> nat \<Rightarrow> 'a list list" where 
  "combine l k = combine_aux l (length l) k"

lemma combine_aux_induct:
  assumes 
  "\<And> l n. P [[]] l n 0"
  "\<And> l n. 0 < n \<Longrightarrow> P [l] l n n" 
  "\<And> k n. \<lbrakk>0 < k; k \<noteq> n\<rbrakk> \<Longrightarrow> P [] [] n k"
  "\<And> h t n k. \<lbrakk>
     P (combine_aux t (n-(1::nat)) (k-(1::nat))) t (n-(1::nat)) (k-(1::nat));
     P (combine_aux t (n-(1::nat)) k) t (n-(1::nat)) k\<rbrakk> \<Longrightarrow> 
     P (map ((#) h) (combine_aux t (n-(1::nat)) (k-(1::nat))) @
          combine_aux t (n-(1::nat)) k) (h # t) n k"
  shows "P (combine_aux l n k) l n k"
using assms
proof (induct l n k rule: combine_aux.induct)
  case (1 l n k)
  show ?case
  proof (cases "k=0")
    case True
    thus ?thesis
      using 1(3)
      by (simp add: combine_aux.simps)
  next
    case False
    show ?thesis
    proof (cases "k = n")
      case True
      thus ?thesis
        using `k \<noteq> 0`
        using 1(4)
        by (simp add: combine_aux.simps)
    next
      case False
      show ?thesis
      proof (cases "l = []")
        case True
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n` 1(5)
          by (simp add: combine_aux.simps)
      next
        case False
        then obtain h t where "l = h # t" by (auto simp add: neq_Nil_conv)
        let ?l1 = "combine_aux t (n-(1::nat)) (k-(1::nat))"
        let ?l2 = "combine_aux t (n-(1::nat)) k"
        have "P ?l1 t (n-(1::nat)) (k-(1::nat))"
          using 1(1)[of h t] 1(3-6)  `k \<noteq> 0` `k \<noteq> n` `l = h # t`
          by auto
        moreover
        have "P ?l2 t (n-(1::nat)) k"
          using 1(2)[of h t] 1(3-6)  `k \<noteq> 0` `k \<noteq> n` `l = h # t`
          by auto
        ultimately
        show ?thesis
          using `l = h # t` `k \<noteq> 0` `k \<noteq> n`
          using combine_aux.simps[of l n k] 1(6)
          by simp
      qed
    qed
  qed
qed

text\<open>For example, @{lemma "combine [1::nat, 2, 3] 2 = [[1, 2], [1, 3], [2, 3]]" by (simp add: combine_def combine_aux.simps)}. \<close>

lemma combine_aux_subset:
  shows "\<forall> A. A \<in> set (combine_aux l n k) \<longrightarrow> set A \<subseteq> set l"
by (rule combine_aux_induct) (auto, force)

lemma combine_subset:
  assumes "A \<in> set (combine l k)"
  shows "set A \<subseteq> set l"
using assms
using combine_aux_subset[rule_format, of A l "length l" k]
unfolding combine_def
by simp

lemma subset_card_length:
  assumes "A \<subseteq> set l" "length l = card A"
  shows "A = set l"
using assms
by (metis List.finite_set card_length card_seteq)


lemma combine_aux_sublist:
  shows "\<forall> A. A \<subseteq> set L \<and> n = length L \<and> m = card A \<longrightarrow> (\<exists>x\<in>set (combine_aux L n m). A = set x)"
proof (rule combine_aux_induct[where P = "\<lambda> c l n m. (\<forall> A. A \<subseteq> set l \<and> n = length l \<and> m = card A \<longrightarrow> (\<exists>x \<in> set c. A = set x))"])
  fix l :: "'a list" and n :: nat
  show "\<forall>A. A \<subseteq> set l \<and> n = length l \<and> 0 = card A \<longrightarrow> (\<exists>x\<in>set [[]]. A = set x)"
    by (auto simp add: card_eq_0_iff finite_subset)
next
  fix l :: "'a list" and n :: nat
  assume "n > (0::nat)"
  thus "\<forall>A. A \<subseteq> set l \<and> n = length l \<and> n = card A \<longrightarrow> (\<exists>x\<in>set [l]. A = set x)"
    using subset_card_length
    by auto
next
  fix h :: "'a" and t :: "'a list" and n :: nat and k :: nat
  let ?l1 = "combine_aux t (n - 1) (k - 1)" and
      ?l2 = "(combine_aux t (n - 1) k)"
  assume *: "\<forall>A. A \<subseteq> set t \<and> n - 1 = length t \<and> k - 1 = card A \<longrightarrow> (\<exists>x\<in>set ?l1. A = set x)" and
    **: "\<forall>A. A \<subseteq> set t \<and> n - 1 = length t \<and> k = card A \<longrightarrow> (\<exists>x\<in>set ?l2. A = set x)"
  show "\<forall>A. A \<subseteq> set (h # t) \<and> n = length (h # t) \<and> k = card A \<longrightarrow>
              (\<exists>x\<in>set (map ((#) h) (?l1) @ ?l2). A = set x)"
  proof (rule allI, rule impI, (erule conjE)+)
    fix A'
    assume "A' \<subseteq> set (h # t)" "n = length (h # t)" "k = card A'"
    show "\<exists>x\<in>set (map ((#) h) (?l1) @ ?l2). A' = set x"
    proof (cases "h \<in> A'")
      case False
      hence "A' \<subseteq> set t"
        using `A' \<subseteq> set (h # t)`
        by auto
      thus ?thesis
        using **[rule_format, of A'] `n = length (h # t)` `k = card A'`
        by auto
    next
      case True
      hence "\<exists>x\<in>set ?l1. (A' - {h}) = set x"
        using *[rule_format, of "A' - {h}"]  `n = length (h # t)` `k = card A'` `A' \<subseteq> set (h # t)`
        using card_Diff_singleton[of A' h] finite_subset[of A' "set (h # t)"]
        by auto
      then obtain x where "x\<in>set ?l1" "(A' - {h}) = set x"
        by auto
      thus ?thesis
        using `h \<in> A'`
        by (rule_tac x="h # x" in bexI) auto
    qed
  qed
qed simp

lemma combine_sublist:
  assumes "A \<subseteq> set L"
  shows "\<exists>x\<in>set (combine L (card A)). A = set x"
using assms
using combine_aux_sublist[rule_format, of A L "length L" "card A"]
unfolding combine_def
by simp

lemma combine_aux_combines:
  assumes "sorted l" and "distinct l" and "n = length l"
  shows "A \<in> set (combine_aux l n k) \<longleftrightarrow> sorted A \<and> distinct A \<and> length A = k \<and> set A \<subseteq> set l"
using assms 
proof (induct l n k  arbitrary: A rule: combine_aux.induct)
  case (1 l n k)
  show ?case
  proof (cases "k = 0")
    case True
    thus ?thesis
      by (auto simp add: combine_aux.simps)
  next
    case False
    show ?thesis
    proof (cases "k = n")
      case True
      thus ?thesis
        using `k \<noteq> 0` `sorted l` `distinct l` `n = length l`
        using distinct_card[of l] distinct_card[of A]
        using card_seteq[of "set l" "set A"]
        using sorted_distinct_set_unique[of A l]
        by (auto simp add: combine_aux.simps)
    next
      case False
      show ?thesis
      proof (cases "l = []")
        case True
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n`
          by (auto simp add: combine_aux.simps)
      next
        case False
        then obtain h t where "l = h # t" by (auto simp add: neq_Nil_conv)
        let ?l1 = "combine_aux t (n-(1::nat)) (k-(1::nat))"
        let ?l2 = "combine_aux t (n-(1::nat)) k"
        have "sorted t"
          using 1(3) `l = h # t`
          by simp

        have "A \<in> set (combine_aux l n k) = (A \<in> set (map (\<lambda> l'. h # l') ?l1) \<or> A \<in> set ?l2)"
          using `k \<noteq> 0` `k \<noteq> n` `l = h # t`
          by (simp add: combine_aux.simps)
        moreover
        have "A \<in> set ?l2 = (sorted A \<and> distinct A \<and> length A = k \<and> set A \<subseteq> set t)"
          using 1(2)[of h t A]   `k \<noteq> 0` `k \<noteq> n` `l = h # t` `sorted t` 1(4) 1(5)
          by simp
        moreover
        have "A \<in> set (map (\<lambda> l'. h # l') ?l1) = (sorted A \<and> distinct A \<and> length A = k \<and> \<not> set A \<subseteq> set t \<and> set A \<subseteq> set l)"
        proof-
          have "(tl A \<in> set ?l1) =
                (sorted (tl A) \<and> distinct (tl A) \<and> length (tl A) = k - 1 \<and> set (tl A) \<subseteq> set t)"
            using 1(1)[of h t "tl A"] `l = h # t` `sorted t` 1(4) 1(5) `k \<noteq> 0` `k \<noteq> n`
            by simp
          moreover
          have "A \<in> set (map (\<lambda> l'. h # l') ?l1) = (A = h # (tl A) \<and> (tl A \<in> set ?l1))"
            by auto
          moreover
          have "\<forall> x \<in> set t. h \<le> x" "h \<notin> set t"
            using `sorted l` `distinct l` `l = h # t`
            by auto

          have "(A = h # tl A \<and> sorted (tl A) \<and> distinct (tl A) \<and> length A - 1 = k - 1 \<and> set (tl A) \<subseteq> set t) =
                (sorted A \<and> distinct A \<and> length A = k \<and> \<not> set A \<subseteq> set t \<and> set A \<subseteq> insert h (set t))" (is "?lhs = ?rhs")
          proof (rule iffI, (erule_tac[!] conjE)+)
            assume "A = h # tl A" "sorted (tl A)" "distinct (tl A)" "length A - 1 = k - 1" "set (tl A) \<subseteq> set t"
            show ?rhs
            proof (safe)
              show "length A = k"
              proof-
                have "length A \<noteq> 0"
                  using `A = h # tl A`
                  by auto
                show ?thesis
                  using `length A - 1 = k - 1` `k \<noteq> 0` `length A \<noteq> 0`
                  by arith
              qed
            next
              show "sorted A"
                apply (subst `A = h # tl A`, subst sorted.simps(2))
                using `sorted (tl A)` `set (tl A) \<subseteq> set t` `\<forall> x \<in> set t. h \<le> x`
                by auto
            next
              show "distinct A"
                apply (subst `A = h # tl A`)
                using `distinct (tl A)` `h \<notin> set t` `set (tl A) \<subseteq> set t`
                by auto
            next
              assume "set A \<subseteq> set t"
              thus "False"
                apply (subst (asm) `A = h # tl A`) 
                using `h \<notin> set t`
                by auto
            next
              fix x
              assume "x \<in> set A" "x \<notin> set t"
              thus "x = h"
                apply (subst (asm) `A = h # tl A`)
                using `set (tl A) \<subseteq> (set t)`
                by auto
            qed
          next
            assume "sorted A" "distinct A" "length A = k" "\<not> set A \<subseteq> set t" "set A \<subseteq> insert h (set t)"
            show "?lhs"
            proof (safe)
              show "sorted (tl A)"
                using `sorted A`
                by (rule linorder_class.sorted_tl)
            next
              show "distinct (tl A)"
                using `distinct A`
                by (rule distinct_tl)
            next
              show "A = h # tl A"
              proof-
                have "h \<in> set A" "A \<noteq> []"
                  using `\<not> set A \<subseteq> set t` `set A \<subseteq> insert h (set t)`
                  by auto
                have "A = hd A # tl A"
                  using `A \<noteq> []`
                  by simp
                show ?thesis
                proof (cases "hd A = h")
                  case True
                  thus ?thesis
                    using `A \<noteq> []`
                    by auto
                next
                  case False
                  from `h \<in> set A` have "h = hd A \<or> h \<in> set (tl A)"
                    by (subst (asm) `A = hd A # tl A`) auto
                  hence "h \<in> set (tl A)"
                    using `hd A \<noteq> h`
                    by auto
                  have "hd A < h"
                    using `sorted A`
                    apply (subst (asm) `A = hd A # tl A`)
                    using `h \<in> set (tl A)` `hd A \<noteq> h`
                    by auto
                  moreover
                  have "hd A \<in> set t"
                    using `set A \<subseteq> insert h (set t)` 
                    apply (subst (asm) `A = hd A # tl A`)
                    using `hd A \<noteq> h`
                    by simp
                  hence "h \<le> hd A"
                    using `\<forall> x \<in> set t. h \<le> x`
                    by simp
                  ultimately
                  show ?thesis
                    by simp
                qed
              qed

              fix x
              assume "x \<in> set (tl A)"
              show "x \<in> set t"
              proof-
                have "x \<in> set A"
                  using `x \<in> set (tl A)`
                  by (subst `A = h # tl A`) simp
                hence "x \<in> insert h (set t)"
                  using `set A \<subseteq> insert h (set t)`
                  by auto
                have "x \<noteq> h"
                  using `distinct A`
                  apply (subst (asm) `A = h # tl A`) 
                  using  `x \<in> set (tl A)`
                  by auto
                show "x \<in> set t"
                  using `x \<in> insert h (set t)` `x \<noteq> h`
                  by auto
              qed
            next
              show "length A - 1 = k - 1"
                using `length A = k`
                by simp
            qed
          qed
          ultimately
          show ?thesis
            using `l = h # t`
            by simp
        qed
        ultimately
        show ?thesis
          using `l = h # t`
          by auto
      qed
    qed
  qed
qed

lemma combine_combines:
  assumes 
  "sorted l" and "distinct l"
  shows
    "A \<in> set (combine l k)  \<longleftrightarrow> (sorted A \<and> distinct A \<and> length A = k \<and> set A \<subseteq> set l)"
using assms
unfolding combine_def
using combine_aux_combines[of l "length l" A k]
by simp

lemma set_combine:
  shows "set (map set (combine [0..<n] k)) = {A. card A = k \<and> A \<subseteq> {0..<n}}"
  using combine_combines[of "[0..<n]"]
  by (auto simp add: distinct_card) (metis (no_types, lifting) atLeastLessThan_upt combine_sublist image_iff)

lemma combine_aux_length:
  assumes "A \<in> set (combine_aux l n k)" and "length l = n"
  shows "length A = k"
using assms
proof (induct l n k arbitrary: A rule: combine_aux.induct)
  case (1 l n k)
  show ?case
  proof (cases "k = 0")
    case True
    thus ?thesis
      using 1(3)
      by (simp add: combine_aux.simps)
  next
    case False
    show ?thesis
    proof (cases "k = n")
      case True
      thus ?thesis
        using `k \<noteq> 0` 1(3) 1(4)
        by (simp add: combine_aux.simps)
    next
      case False
      show ?thesis
      proof (cases l)
        case Nil
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n` 1(3)
          by (simp add: combine_aux.simps)
      next
        case (Cons h t)
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n` 1(3) 1(4)
          using combine_aux.simps[of l n k]
          using 1(1)[of h t "tl A"] 1(2)[of h t A]
          by auto
      qed
    qed
  qed
qed

lemma combine_length:
  assumes "A \<in> set (combine l k)"
  shows "length A = k"
using assms
using combine_aux_length[of A l "length l" k]
unfolding combine_def
by simp
  
lemma distinct_combine_aux: 
  assumes "n = length l" and "distinct l"
  shows "distinct (combine_aux l n k)"
using assms
proof (induct l n k rule: combine_aux.induct)
  case (1 l n k)
  show ?case
  proof (cases "k = 0")
    case True
    thus ?thesis
      by (simp add: combine_aux.simps)
  next
    case False
    show ?thesis
    proof (cases "k = n")
      case True
      thus ?thesis
        by (simp add: combine_aux.simps)
    next
      case False
      show ?thesis
      proof (cases l)
        case Nil
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n`
          by (simp add: combine_aux.simps)
      next
        case (Cons h t)
        thus ?thesis
          using `k \<noteq> 0` `k \<noteq> n`
          using combine_aux.simps[of l n k]
          using 1(1)[of h t] 1(2)[of h t] 1(3) 1(4)
          apply (auto simp add: distinct_map inj_on_def)
          using combine_aux_subset[rule_format, of _ t "length t" k]
          by force
      qed
    qed
  qed
qed

lemma distinct_combine:
  assumes "distinct l"
  shows "distinct (combine l k)"
using assms
using distinct_combine_aux[of "length l" l k]
unfolding combine_def
by simp

lemma combine_not_empty:
  assumes "k < n" 
  shows "combine [0..<n] k \<noteq> []"
  using assms
  using combine_combines[of "[0..<n]" "[0..<k]" k]
  by auto

(* all subsets *)
definition all_subsets where 
  "all_subsets F = concat (map (\<lambda> k. combine F k) [0..<length F + 1])"

end