section \<open>Cataloguing Union-Closed Families\<close>

subsection \<open>Abstract Specification\<close>

theory FamilyAbs
  imports Main "HOL-Library.List_Lexorder" "HOL-Library.FSet"
    More_List_Lexord Combinatorics Faradzev_Read Canon_Least_Perm
begin

typedef Set = "{ s :: nat set. finite s }" 
  morphisms elements Set
  by auto

lemma finite_elements [simp]:
  shows "finite (elements s)"
  using elements by auto

(* union *)

definition union :: "Set \<Rightarrow> Set \<Rightarrow> Set" where 
  "union s1 s2 = Set (elements s1 \<union> elements s2)"

lemma elements_union [simp]:
  "elements (union A B) = elements A \<union> elements B"
  by (simp add: union_def Set_inverse)

lemma union_finite_Set [simp]:
  assumes "finite A" "finite B"
  shows  "union (Set A) (Set B) = Set (A \<union> B)"
  by (simp add: Set_inverse assms(1) assms(2) union_def)

(* set_card *)

definition set_card :: "Set \<Rightarrow> nat" where
  [simp]: "set_card s = card (elements s)"


(* set_order *)

instantiation Set :: linorder
begin
definition less_Set :: "Set \<Rightarrow> Set \<Rightarrow> bool" where 
  "less_Set s1 s2 \<longleftrightarrow>
    (let n1 = set_card s1; n2 = set_card s2
      in n1 > n2 \<or> (n1 = n2 \<and> rev (sorted_list_of_set (elements s1)) > rev (sorted_list_of_set (elements s2))))"
definition less_eq_Set :: "Set \<Rightarrow> Set \<Rightarrow> bool" where
  "less_eq_Set s1 s2 \<longleftrightarrow>
     s1 = s2 \<or> s1 < s2"

instance
proof
  fix s1 s2 :: Set
  show "s1 < s2 \<longleftrightarrow> (s1 \<le> s2 \<and> \<not> s2 \<le> s1)"
    unfolding less_eq_Set_def less_Set_def Let_def
    by auto
next
  fix s :: Set
  show "s \<le> s"
    unfolding less_eq_Set_def
    by (cases s, auto simp add: Let_def)
next
  fix s1 s2 s3 :: Set
  assume "s1 \<le> s2" "s2 \<le> s3" 
  thus "s1 \<le> s3"
    unfolding less_eq_Set_def less_Set_def
    by (auto simp add: Let_def)
next
  fix s1 s2 :: Set
  assume "s1 \<le> s2" "s2 \<le> s1"
  thus "s1 = s2"
    unfolding less_eq_Set_def less_Set_def
    by (auto simp add: Let_def)
next
  fix s1 s2 :: Set
  obtain s1' s2' where *: "s1 = Set s1'" "s2 = Set s2'" "finite s1'" "finite s2'"
    by (cases s1, cases s2, simp)

  show "s1 \<le> s2 \<or> s2 \<le> s1"
  proof (cases "s1' = s2'")
    case True
    thus ?thesis
      using *
      unfolding less_eq_Set_def
      by simp
  next
    case False
    have "card s1' < card s2' \<or> card s1' > card s2' \<or> card s1' = card s2'"
      by auto
    thus ?thesis
    proof safe
      assume "card s1' < card s2'" "\<not> s2 \<le> s1"
      thus "s1 \<le> s2"
        using *
        unfolding less_eq_Set_def less_Set_def Let_def
        by (metis elements elements_inverse mem_Collect_eq not_less_iff_gr_or_eq rev_swap sorted_list_of_set(1))
    next
      assume "card s1' > card s2'" "\<not> s2 \<le> s1"
      thus "s1 \<le> s2"
        using *
        unfolding less_eq_Set_def less_Set_def Let_def
        by (metis elements elements_inverse mem_Collect_eq not_less_iff_gr_or_eq rev_swap sorted_list_of_set(1))
    next
      assume "card s1' = card s2'" "\<not> s2 \<le> s1"
      thus "s1 \<le> s2"
        using `finite s1'` `finite s2'`
        using * `s1' \<noteq> s2'` sorted_list_of_set_inj[of s1' s2'] `card s1' = card s2'` `\<not> s2 \<le> s1`
        unfolding less_eq_Set_def less_Set_def Let_def
        by (metis elements elements_inverse mem_Collect_eq not_less_iff_gr_or_eq rev_swap sorted_list_of_set(1))
    qed
  qed
qed

end

lemma leq_Set_geq_card:
  assumes "a \<le> b"
  shows "set_card a \<ge> set_card b"
  using assms
  unfolding less_eq_Set_def less_Set_def
  by (auto simp add: Let_def)

(* permute_set *)

definition list_to_fun :: "perm \<Rightarrow> (nat \<Rightarrow> nat)" where
  "list_to_fun p = (\<lambda> i. p ! i)"

definition permute_set :: "perm  \<Rightarrow> Set \<Rightarrow> Set" where
  "permute_set p s = Set ((list_to_fun p) ` elements s)"

lemma permute_set_id:
  assumes "elements s \<subseteq> {0..<n}" 
  shows "permute_set (perm_id n) s = s"
proof-
  obtain s' where "s = Set s'" "finite s'"  "elements s = s'"
    using assms
    by (cases s, auto simp add: Set_inverse)
  thus ?thesis
    using assms
    unfolding permute_set_def perm_id_def list_to_fun_def
    by (simp add: subset_eq)
qed

lemma list_to_fun_inj [simp]:
  assumes "p \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "inj_on (list_to_fun p) (elements s)"
    using assms
    unfolding inj_on_def list_to_fun_def
    by (metis atLeastLessThan_iff card_atLeastLessThan card_distinct diff_zero index_of_list_element permute_member_length length_upt permute_member_set set_upt subset_eq)

lemma permute_set_card [simp]:
  assumes "p \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "set_card (permute_set p s) = set_card s"
  using assms
  unfolding permute_set_def
  by (metis Set_inverse card_image elements finite_imageI list_to_fun_inj mem_Collect_eq set_card_def)

lemma permute_set_elements [simp]:
  assumes "p \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "elements (permute_set p s) \<subseteq> {0..<n}"
  using assms
  unfolding permute_set_def list_to_fun_def
  by (metis Set_inverse diff_zero finite_imageI image_mono length_upt map_nth mem_Collect_eq permute_member_length permute_member_set set_map set_upt subset_eq_atLeast0_lessThan_finite)

lemma permute_set_empty_iff:
  assumes "p \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "permute_set p s = Set {} \<longleftrightarrow> s = Set {}"
  using permute_set_card assms
  by (metis Set_inverse card_0_eq elements_inverse finite.emptyI image_empty mem_Collect_eq permute_set_def set_card_def subset_eq_atLeast0_lessThan_finite)

lemma permute_set_union [simp]:
  shows "permute_set p (union A B) = union (permute_set p A) (permute_set p B)"
  unfolding permute_set_def
  by (simp add: image_Un)

lemma list_to_fun_perm_comp [simp]:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])"
  assumes "x < n"
  shows "list_to_fun (perm_comp p1 p2) x = (list_to_fun p1 (list_to_fun p2 x))"
  using assms
  by (auto simp add: list_to_fun_def perm_comp_def permute_list_def permute_member_length)

lemma permute_set_comp [simp]:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "permute_set (perm_comp p1 p2) s = permute_set p1 (permute_set p2 s)"
proof-
  have "list_to_fun (perm_comp p1 p2) ` elements s =
        list_to_fun p1 ` list_to_fun p2 ` elements s"
    using assms
    using list_to_fun_perm_comp[OF assms(1-2)]
    by (smt atLeastLessThan_iff image_cong image_image subset_iff)
  thus ?thesis
    unfolding permute_set_def
    by (simp add: Set_inverse)
qed
  
lemma permute_set_inv_perm [simp]:
  assumes "p \<in> set (permute [0..<n])" "elements s \<subseteq> {0..<n}"
  shows "permute_set (perm_inv p) (permute_set p s) = s"
  using assms
  by (metis perm_id_def perm_comp_perm_inv_id_1 perm_inv_permute permute_set_comp permute_set_id)

lemma permute_set_inj:
  assumes "p \<in> set (permute [0..<n])"
  assumes "elements s1 \<subseteq> {0..<n}" "elements s2 \<subseteq> {0..<n}"
  assumes "permute_set p s1 = permute_set p s2"
  shows "s1 = s2"
  using assms
  using permute_set_inv_perm
  by fastforce

lemma distinct_map_permute_set:
  assumes "p \<in> set (permute [0..<n])" 
          "\<forall> A \<in> set F. elements A \<subseteq> {0..<n}"
  assumes "distinct F"
  shows "distinct (map (permute_set p) F)"
  using assms permute_set_inj[OF assms(1)]
  by (smt distinct_conv_nth length_map nth_map nth_mem)

lemma list_to_fun_upt_n [simp]:
  assumes "p <~~> [0..<n]"
  shows "list_to_fun p ` {0..<n} = {0..<n}"
  using assms permute_isPermutation[OF assms]
  unfolding list_to_fun_def
  by (metis diff_zero length_upt map_nth permute_member_length permute_member_set set_map set_upt)

(* Family *)

typedef Family = "{ s :: Set set. finite s }"
  morphisms sets Family
  by auto

lemma finite_sets [simp]:
  shows "finite (sets F)"
  using sets by auto

definition add_set :: "Family \<Rightarrow> Set \<Rightarrow> Family" where
  "add_set F s = Family (sets F \<union> {s})"

definition inter :: "Family \<Rightarrow> Family \<Rightarrow> Family" where
  "inter F1 F2 = Family (sets F1 \<inter> sets F2)"

lemma add_set_sets [simp]:
  shows "sets (add_set F s) = sets F \<union> {s}"
  by (simp add: Family_inverse add_set_def)

lemma inter_sets [simp]:
  shows "sets (inter F1 F2) = sets F1 \<inter> sets F2"
  by (simp add: Family_inverse inter_def)

(* union_closed *)

definition union_closed :: "Family \<Rightarrow> bool" where
  "union_closed F \<longleftrightarrow> (\<forall> A \<in> sets F. \<forall> B \<in> sets F. union A B \<in> sets F)"

lemma card_union_gt:
  assumes "finite A" "finite B"
  shows "A \<union> B = A \<or> card (A \<union> B) > card A"
  using assms
  by (metis Un_upper1 card_seteq finite_UnI not_le)

lemma union_closed_add_set:
  assumes "union_closed (add_set F s)"
  assumes "\<forall> A \<in> sets F. \<forall> B \<in> sets F. union A B \<noteq> s"
  shows "union_closed F"
  using assms
  by (auto simp add: union_closed_def add_set_def Family_inverse)

lemma union_closed_add_set_leq_card:
  assumes "\<forall> A \<in> sets F. finite (elements A)"
          "union_closed (add_set F s)"
          "\<forall> s' \<in> sets F. set_card s \<le> set_card  s'" 
  shows "union_closed F"
proof (cases "s \<in> sets F")
  case True
  thus ?thesis
    using assms
    using add_set_def sup_bot.comm_neutral union_closed_def
    by (simp add: Family_inverse insert_absorb)
next
  case False
  show ?thesis
  proof (rule union_closed_add_set)
    show "union_closed (add_set F s)"
      by fact
  next
    show "\<forall> A \<in> sets F. \<forall> B \<in> sets F. union A B \<noteq> s"
    proof safe
      fix A B
      assume "A \<in> sets F" "B \<in> sets F" "s = union A B"
      hence "s \<in> sets F"
        using assms card_union_gt[of "elements A" "elements B"]
        by (auto simp add: union_def Set_inverse elements_inverse)
      thus False
        using `s \<notin> sets F`
        by simp
    qed
  qed
qed

(* family_card *)

definition family_card where
  [simp]: "family_card F = card (sets F)"

(* permute_family  *)

definition permute_family :: "perm \<Rightarrow> Family \<Rightarrow> Family" where
  "permute_family p F = Family (permute_set p ` (sets F))"

lemma permute_family_contains_empty_iff:
  assumes "p \<in> set (permute [0..<n])" 
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "Set {} \<in> sets (permute_family p F) \<longleftrightarrow> Set {} \<in> sets F"
  using assms permute_set_empty_iff[of p n]
  by (cases F, metis Family_inverse finite_imageI imageE image_eqI mem_Collect_eq permute_family_def)

lemma permute_family_contains_upt_n_iff:
  assumes "p \<in> set (permute [0..<n])" 
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "Set {0..<n} \<in> sets (permute_family p F) \<longleftrightarrow> Set {0..<n} \<in> sets F"
  using assms
  unfolding permute_family_def
  by (smt Family_inverse Set_inverse finite_atLeastLessThan finite_imageI finite_sets image_iff perm_inv_permute isPermutation_permute list_to_fun_upt_n mem_Collect_eq permute_set_def permute_set_inv_perm)

lemma permute_family_upt_n: 
  assumes "p <~~> [0..<n]"
  shows "permute_family p (Family {Set {0..<n}}) = Family {Set {0..<n}}"
  using assms
  unfolding permute_family_def
  by (simp add: Family_inverse Set_inverse permute_set_def)

lemma permute_family_elements:
  assumes "p \<in> set (permute [0..<n])"
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "\<forall> s \<in> sets (permute_family p F). elements s \<subseteq> {0..<n}"
  using assms
  by (simp add: Family_inverse permute_family_def)

lemma permute_family_card [simp]:
  assumes "p \<in> set (permute [0..<n])"
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "family_card (permute_family p F) = family_card F"
proof-
  have "sets (permute_family p F) = permute_set p ` sets F"
    unfolding permute_family_def
    by (simp add: Family_inverse)
  moreover
  have "card (permute_set p ` sets F) = family_card F"
    unfolding family_card_def
  proof (rule card_image)
    show "inj_on (permute_set p) (sets F)"
      by (metis assms(1) assms(2) inj_onI permute_set_inv_perm)
  qed
  ultimately
  show ?thesis
    by simp
qed

lemma permute_family_cards [simp]:
  assumes "p \<in> set (permute [0..<n])"
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}" 
  shows "set_card ` sets (permute_family p F) = set_card ` sets F"
  using assms permute_set_card
  using permute_family_def
  by (auto simp add: image_iff Family_inverse)

lemma permute_family_cards_P [simp]:
  assumes "p \<in> set (permute [0..<n])" 
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}"
  shows "{s \<in> sets (permute_family p F). P (set_card s)} =
         (permute_set p) ` {s \<in> sets F. P (set_card s)}"
  using assms permute_set_card
  using permute_family_def
  by (auto simp add: image_iff Family_inverse)

lemma permute_family_id [simp]:
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "permute_family (perm_id n) F = F"
  using assms
  unfolding permute_family_def
  by (cases F) (simp add: permute_set_id Family_inverse)

lemma permute_family_perm_comp [simp]:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])"
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "permute_family (perm_comp p1 p2) F = permute_family p1 (permute_family p2 F)"
  using assms
  unfolding permute_family_def
  by (smt Family_inverse finite_imageI finite_sets image_cong image_image mem_Collect_eq permute_set_comp)

lemma permute_family_inv_perm_1:
  assumes "p \<in> set (permute [0..<n])"
  assumes "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "permute_family (perm_inv p) (permute_family p F) = F"
  using assms
  using permute_family_perm_comp[symmetric] perm_inv_permute
  by (metis perm_id_def perm_comp_perm_inv_id_1 permute_family_id)

lemma permute_family_inv_perm_2:
  assumes "p \<in> set (permute [0..<n])" "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "permute_family p (permute_family (perm_inv p) F) = F"
  using assms
  using permute_family_perm_comp[symmetric] perm_inv_permute
  by (metis perm_id_def perm_inv_permute perm_comp_perm_inv_id_2 permute_family_id)

lemma permute_family_union_closed [simp]:
  assumes "union_closed F"
  shows "union_closed (permute_family p F)"
  using assms
  unfolding union_closed_def permute_family_def
  using permute_set_union[symmetric]
  by (smt Family_inverse finite_imageI finite_sets image_iff mem_Collect_eq)

lemma permute_family_all_perms:
  assumes "p \<in> set (permute [0..<n])"
  assumes "\<forall>s\<in>sets F. elements s \<subseteq> {0..<n}"
  shows "set (map (\<lambda>p'. permute_family p' (permute_family p F)) (permute [0..<n])) = 
         set (map (\<lambda>p. permute_family p F) (permute [0..<n]))" (is "?lhs = ?rhs")
proof safe
  fix x
  assume "x \<in> ?rhs"
  then obtain p' where "p' \<in> set (permute [0..<n])" "x = permute_family p' F"
    by (auto simp add: image_iff)
  hence "perm_comp p' (perm_inv p) \<in> set (permute [0..<n])" 
        "x = permute_family (perm_comp p' (perm_inv p)) (permute_family p F)"
    using assms
    using perm_inv_permute permute_family_elements permute_family_inv_perm_1 permute_family_perm_comp  perm_comp_permute     
    by presburger+
  thus "x \<in> ?lhs"
    using assms
    by (auto simp add: image_iff)
next
  fix x
  assume "x \<in> ?lhs"
  then obtain p' where "p' \<in> set (permute [0..<n])" "x = permute_family p' (permute_family p F)"
    by (auto simp add: image_iff)
  hence "perm_comp p' p \<in>set (permute [0..<n])"
        "permute_family (perm_comp p' p) F = x"
    using assms
    using permute_family_elements permute_family_perm_comp perm_comp_permute     
    by presburger+
  thus "x \<in> ?rhs"
    using assms
    by (auto simp add: image_iff)
qed


(* S *)

definition S :: "nat \<Rightarrow> nat \<Rightarrow> Family set" where
  "S n q = {F. (\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}) \<and> 
                              family_card F = q + 1 \<and>
                              Set {} \<notin> sets F \<and>
                              Set {0..<n} \<in> sets F \<and>
                              union_closed F}"

lemma S_finite:
  assumes "s \<in> S n q"
  shows "finite (sets s)" 
  using assms
  unfolding S_def
  using not_one_le_zero
  by fastforce

lemma S_finite_elements:
  assumes "s \<in> S n q"
  shows "\<forall> x \<in> sets s. finite (elements x)"
  using assms finite_subset
  unfolding S_def
  by fastforce

lemma S_not_empty:
  assumes "s \<in> S n q"
  shows "sets s \<noteq> {}"
  using assms
  unfolding S_def
  by auto

(* equiv *)

global_interpretation EquivFamily: EquivExPerm
  where
    invar = "\<lambda> n F. \<forall>s\<in>sets F. elements s \<subseteq> {0..<n}" and
    permute = "\<lambda> n p F. permute_family p F"
  defines
    equiv = "EquivFamily.equiv"
proof
  fix n a p
  assume "\<forall>s\<in>sets a. elements s \<subseteq> {0..<n}" "p <~~> [0..<n]"
  thus "\<forall>s\<in>sets (permute_family p a). elements s \<subseteq> {0..<n}"
    using permute_family_elements[of p n a]
    using permute_isPermutation by blast
next
  fix n p1 p2 a
  assume "\<forall>s\<in>sets a. elements s \<subseteq> {0..<n}" "p1 <~~> [0..<n]" "p2 <~~> [0..<n]"
  thus "\<exists> p. p <~~> [0..<n] \<and> permute_family p a = permute_family p1 (permute_family p2 a)"
    using permute_family_perm_comp[of p1 n p2 a]
    by (meson isPermutation_permute perm_comp_permute permute_isPermutation)
next
  fix n p a
  assume "\<forall>s\<in>sets a. elements s \<subseteq> {0..<n}" "p <~~> [0..<n]"
  thus "\<exists> p'. p' <~~> [0..<n] \<and> permute_family p' (permute_family p a) = a"
    using permute_family_inv_perm_1 permute_isPermutation
    by (metis isPermutation_permute perm_inv_permute)
next
  fix n a
  show "\<forall>s\<in>sets a. elements s \<subseteq> {0..<n} \<Longrightarrow> permute_family (perm_id n) a = a"
    using permute_family_id[of a n]
    by (simp add: perm_id_def)
qed

lemma equiv_S:
  assumes "equiv N s s'" "s \<in> S N q"
  shows "s' \<in> S N q"
proof-
  obtain p where "p \<in> set (permute [0..<N])" "permute_family p s = s'"
    using assms
    using isPermutation_permute[of _ "[0..<N]"] permute_isPermutation[of _ "[0..<N]"]
    unfolding EquivFamily.equiv_def
    by auto
  thus ?thesis
    using permute_family_elements[of p N s]
    using permute_family_card[of p N s]
    using permute_family_union_closed[of s]
    using permute_family_contains_empty_iff[of p N s]
    using permute_family_contains_upt_n_iff[of p N s]
    using `s \<in> S N q` unfolding S_def
    by auto
qed


(* Family order *)

instantiation Family :: linorder
begin
definition less_Family :: "Family \<Rightarrow> Family \<Rightarrow> bool" where 
  "less_Family F1 F2 \<longleftrightarrow> 
       sorted_list_of_set (sets F1) < sorted_list_of_set (sets F2)"
definition less_eq_Family :: "Family \<Rightarrow> Family \<Rightarrow> bool" where 
  "less_eq_Family F1 F2 \<longleftrightarrow> F1 = F2 \<or> F1 < F2"
instance
proof
  fix F1 F2 :: Family
  show "F1 < F2 \<longleftrightarrow> (F1 \<le> F2 \<and> \<not> F2 \<le> F1)"
    unfolding less_eq_Family_def
    using less_Family_def by auto
next
  fix F :: Family
  show "F \<le> F"
    unfolding less_eq_Family_def
    by (auto simp add: Let_def)
next
  fix F1 F2 F3 :: Family
  assume "F1 \<le> F2" "F2 \<le> F3" 
  thus "F1 \<le> F3"
    unfolding less_eq_Family_def less_Family_def
    by (auto simp add: Let_def)
next
  fix F1 F2 :: Family
  assume "F1 \<le> F2" "F2 \<le> F1"
  thus "F1 = F2"
    unfolding less_eq_Family_def
    by (auto simp add: Let_def less_Family_def)
next
  fix F1 F2 :: Family
  obtain F1' F2' where *: "F1 = Family F1'" "F2 = Family F2'" "finite F1'" "finite F2'"
    by (cases F1, cases F2, simp)

  show "F1 \<le> F2 \<or> F2 \<le> F1"
  proof (cases "F1' = F2'")
    case True
    thus ?thesis
      using *
      unfolding less_eq_Family_def
      by simp
  next
    case False
    show ?thesis
      using * `F1' \<noteq> F2'` sorted_list_of_set_inj[of F1' F2'] `finite F1'` `finite F2'` 
      unfolding less_eq_Family_def
      using Family_inverse less_Family_def by auto
  qed
qed

end

(* family order + add_set *)

lemma less_Family_add_sets:
  assumes "finite (sets F1)" "finite (sets F2)"
          "sets F1 = P1 \<union> S1" "\<forall> x \<in> P1. \<forall> y \<in> S1. x < y" 
          "sets F2 = P2 \<union> S2" "\<forall> x \<in> P2. \<forall> y \<in> S2. x < y"
          "card P1 = card P2"
  assumes "Family P1 < Family P2"
  shows "F1 < F2"
proof-
  have *: "P1 \<inter> S1 = {}" "P2 \<inter> S2 = {}"
    using assms
    by auto
    
  have "sorted_list_of_set (sets F1) = sorted_list_of_set P1 @ sorted_list_of_set S1"
       "sorted_list_of_set (sets F2) = sorted_list_of_set P2 @ sorted_list_of_set S2"
    using assms *
    by (auto simp add: dual_order.strict_implies_order sorted_list_of_set_union)

  moreover
  have "length (sorted_list_of_set P1) = length (sorted_list_of_set P2)"
    using assms
    by (metis distinct_card distinct_sorted_list_of_set finite_Un sorted_list_of_set(1))

  ultimately
  have "Family (sets F1) < Family (sets F2)"
    using `Family P1 < Family P2`
    unfolding less_Family_def
    using Family_inverse assms list_less_right_append
    by fastforce
  thus ?thesis
    by (simp add: sets_inverse)
qed

lemma less_eq_Family_add_set:
  assumes "finite (sets F)"
  assumes "\<forall> f \<in> sets F. f < s1" "\<forall> f \<in> sets F. f < s2" 
  assumes "s1 \<le> s2" 
  shows "add_set F s1 \<le> add_set F s2"
  using assms 
  using sorted_insort_is_snoc[of "sorted_list_of_set (sets F - {s1})" s1]
  using sorted_insort_is_snoc[of "sorted_list_of_set (sets F - {s2})" s2]
  unfolding add_set_def
  unfolding less_eq_Set_def less_eq_Family_def less_Family_def list_less_def
  by (smt Diff_empty Diff_insert0 Family_inverse Un_insert_right case_prodI finite.insertI less_irrefl lexord_append_left_rightI mem_Collect_eq set_sorted_list_of_set sorted_list_of_set.insert_remove sorted_sorted_list_of_set sup_bot.right_neutral)
  
lemma less_Family_add_Max_to_greater:
  assumes "finite (sets F1)" "finite (sets F2)"
  assumes "\<forall> s \<in> sets F2. s < s2" 
  assumes "family_card F1 = family_card F2"
  assumes "F1 < F2"
  shows "add_set F1 s1 < add_set F2 s2"
  using assms 
proof-
  let ?sls = sorted_list_of_set
  have "?sls (sets F2) > ?sls (sets F1)"
    using assms(1-2) `F1 < F2`
    unfolding less_eq_Family_def
    using less_Family_def by blast

  have "length (?sls (sets F1)) = length (?sls (sets F2))"
    using assms
    by (metis family_card_def distinct_card distinct_sorted_list_of_set sorted_list_of_set(1))

  then obtain pf a1 a2 sf1 sf2
    where ps: "?sls (sets F1) = pf @ [a1] @ sf1"
              "?sls (sets F2) = pf @ [a2] @ sf2" "a1 < a2"
    using `?sls (sets F1) < ?sls (sets F2)`
    unfolding list_less_def lexord_def
    by auto

  have 2: "?sls (sets (add_set F2 s2)) = pf @ [a2] @ sf2 @ [s2]"
    using assms ps sorted_list_of_set_insert_Max[of "sets F2" s2]
    by simp    

  show ?thesis
  proof (cases "s1 \<in> sets F1")
    case True
    hence 1: "?sls (sets (add_set F1 s1)) = pf @ [a1] @ sf1"
      using assms ps
      by (simp add: add_set_def sorted_list_of_set.remove Family_inverse)
    thus ?thesis
      using 2 `a2 > a1`
      using \<open>length (sorted_list_of_set (sets F1)) = length (sorted_list_of_set (sets F2))\<close> \<open>sorted_list_of_set (sets F2) > sorted_list_of_set (sets F1)\<close> 
      by (metis append.assoc append_Nil2 less_Family_def list_less_right_append ps(1) ps(2))
  next
    case False

    hence 1: "?sls (sets (add_set F1 s1)) = insort s1 (pf @ [a1] @ sf1)"
      using assms ps
      by (simp add: add_set_def Family_inverse)

    have "pf @ [a2] @ sf2 @ [s2] > insort s1 (pf @ [a1] @ sf1)"
      using insort_append[of "pf @ [a1]" s1 sf1]
    proof
      assume "\<exists>pa sa. sf1 = pa @ sa \<and> insort s1 ((pf @ [a1]) @ sf1) = (pf @ [a1]) @ pa @ [s1] @ sa"
      thus ?thesis
        unfolding list_less_def
        using `a1 < a2` lexord_append_left_rightI[of a1 a2]
        by auto
    next
      assume " \<exists>pa sa. pf @ [a1] = pa @ sa \<and> insort s1 ((pf @ [a1]) @ sf1) = pa @ [s1] @ sa @ sf1"
      then obtain pa sa where "pf @ [a1] = pa @ sa" and
        *: "insort s1 ((pf @ [a1]) @ sf1) = pa @ [s1] @ sa @ sf1"
        by force+

      have "sorted ((pf @ [a2]) @ sf2)"
        using ps(2)
        by (metis append.assoc sorted_sorted_list_of_set)

      have "sorted (pa @ [s1] @ sa @ sf1)"
        using *[symmetric] ps(1)[symmetric] assms(1-2)
        by (simp add: sorted_insort)
      
      obtain us where "pf = pa @ us \<and> us @ [a1] = sa \<or> pf @ us = pa \<and> [a1] = us @ sa"
        using `pf @ [a1] = pa @ sa` append_eq_append_conv2[of pf "[a1]" pa sa]
        by auto
      thus ?thesis
      proof
        assume **: "pf = pa @ us \<and> us @ [a1] = sa"
        show ?thesis
        proof (cases "us = []")
          case True
          have "s1 \<le> a1"
            using `sorted (pa @ [s1] @ sa @ sf1)`
            using ** `us = []`
            by (auto simp add: sorted_append)
          hence "s1 < a2"
            using `a2 > a1`
            by simp
          then show ?thesis
            using * ** `us = []`  lexord_append_left_rightI[of s1 a2 _ pa]
            by (auto simp add:list_less_def)
        next
          case False
          then obtain us1 us2 where "us = us1 # us2"
            by (cases us, auto)
          hence "s1 \<le> us1"
            using * ** `sorted (pa @ [s1] @ sa @ sf1)`
            by (auto simp add: sorted_append)
          moreover
          have "us1 \<in> set (sorted_list_of_set (sets F1))"
            using ps(1) `us = us1 # us2` ** 
            by auto
          hence "s1 \<noteq> us1"
            using `s1 \<notin> sets F1` * assms(1)
            by auto
          ultimately
          show ?thesis
            using * ** `us = us1 # us2` lexord_append_left_rightI[of s1 us1]
            unfolding list_less_def
            by simp
        qed
      next
        assume **: "pf @ us = pa \<and> [a1] = us @ sa"
        show ?thesis
        proof (cases "us = []")
          case True
          have "s1 \<le> a1"
            using `sorted (pa @ [s1] @ sa @ sf1)`
            using ** `us = []`
            by (auto simp add: sorted_append)
          hence "s1 < a2"
            using `a2 > a1`
            by simp
          then show ?thesis
            using * ** `us = []`
            unfolding list_less_def
            by (auto simp add: lexord_append_left_rightI)
        next
          case False
          hence "sa = [] \<and> us = [a1]"
            using **
            by (metis Cons_eq_append_conv append.right_neutral append_is_Nil_conv)
          hence "pa = pf @ [a1]"
            using * **
            by simp
          then show ?thesis
            using * ** `a2 > a1`
            unfolding list_less_def
            by (auto simp add: lexord_append_left_rightI)
        qed
      qed
    qed

    thus "add_set F2 s2 > add_set F1 s1"
      using 1 2
      by (simp add: less_Family_def)
  qed
qed

lemma less_eq_Family_remove_Max1:
  assumes "finite (sets F1)" "finite (sets F2)"
  assumes "\<forall> s \<in> sets F1. s < s1"
  assumes "family_card F1 = family_card F2"
  assumes "add_set F1 s1 \<le> add_set F2 s2"
  shows "F1 \<le> F2"
  using less_Family_add_Max_to_greater
  by (metis assms(1) assms(2) assms(3) assms(4) assms(5) leD leI)

(* augment *)

definition augment_set :: "nat \<Rightarrow> Set \<Rightarrow> Set set" where
  "augment_set n s = {s'. elements s' \<noteq> {} \<and> elements s' \<subseteq> {0..<n} \<and> s' > s}"

definition augment :: "nat \<Rightarrow> Family \<Rightarrow> Family list" where
  "augment n F = 
      (let Fs = {Family ((sets F) \<union> {s}) | s. s \<in> augment_set n (Max (sets F))}
        in sorted_list_of_set {s' \<in> Fs. union_closed s'})"

lemma augment_set_finite:
  shows "finite (augment_set n s)"
proof-
  have "elements ` augment_set n s \<subseteq> Pow {0..<n}"
    unfolding augment_set_def
    by auto
  hence "finite (elements ` augment_set n s)"
    using finite_Pow_iff finite_subset
    by auto       
  moreover
  have "inj_on elements (augment_set n s)"
    unfolding inj_on_def
    by (metis elements_inverse)
  ultimately
  show ?thesis
    using finite_image_iff[of elements "augment_set n s"]
    by simp
qed

lemma augment_set_not_contains_empty [simp]:
  "Set {} \<notin> augment_set n s"
  unfolding augment_set_def
  by (simp add: Set_inverse)

lemma augment_set_not_contains [simp]:
  shows "s \<notin> augment_set n s"
  unfolding augment_set_def
  by auto

lemma augment_set_elements [simp]:
  assumes "s' \<in> augment_set n s"
  shows "elements s' \<subseteq> {0..<n}"
  using assms
  unfolding augment_set_def
  by simp

lemma augment_set_gt:
  assumes "s' \<in> augment_set n s"
  shows "s' > s"
  using assms
  unfolding augment_set_def
  by (metis (mono_tags, lifting) mem_Collect_eq)

lemma augment_set_less_eq_card:
  shows "\<forall> s' \<in> augment_set n s. set_card  s' \<le> set_card s"
proof
  fix s'
  assume "s' \<in> augment_set n s"
  thus "set_card s' \<le> set_card s"
    unfolding augment_set_def
    using leq_Set_geq_card by auto
qed

lemma augment_set_Max_notin: 
  assumes "finite (sets F)"
  assumes "s \<in> augment_set n (Max (sets F))" 
  shows "s \<notin> sets F"
proof (rule ccontr)
  assume "\<not> ?thesis"
  hence "s \<le> Max (sets F)"
    using assms Max_ge[of "sets F" s]
    by simp
  thus False
    using assms leq_Set_geq_card[of s "Max (sets F)"]
    unfolding augment_set_def
    by auto
qed

lemma augment_set_Max_leq_card: 
  assumes "finite (sets F)"
  assumes "s \<in> augment_set n (Max (sets F))" "s' \<in> (sets F)"
  shows "set_card s \<le> set_card s'"
  using assms
  by (meson Max_ge augment_set_less_eq_card le_trans leq_Set_geq_card)

lemma augment_dim:
  assumes "F' \<in> set (augment n F)"
  shows "F \<in> S n q \<longleftrightarrow> F' \<in> S n (q + 1)"
proof-
  from assms obtain s where *:
    "union_closed (Family (insert s (sets F)))" "s \<in> augment_set n (Max (sets F))"
    "F' = Family (insert s (sets F))"
    unfolding augment_def
    by (auto simp add: augment_set_finite)
  show "F \<in> S n q \<longleftrightarrow> F' \<in> S n (q + 1)"
  proof
    assume "F \<in> S n q"
    thus "F' \<in> S n (q + 1)"
      using *
      using S_finite[of F n q]
      using augment_set_elements[of s n "Max (sets F)"] 
      using card_insert_if[of "sets F" s] 
      using augment_set_Max_notin[of F s n]
      unfolding S_def Let_def augment_def
      by (auto simp add: Family_inverse)
  next
    assume "F' \<in> S n (q + 1)"
    thus "F \<in> S n q"
      using *
      using S_finite[of F' n "q+1"] S_finite_elements[of F' n "q+1"]
      using augment_set_elements[of s n "Max (sets F)"] 
      using card_insert_if[of "sets F" s] 
      using augment_set_Max_notin[of F s n]
      using augment_set_Max_leq_card[of F s n]
      using union_closed_add_set_leq_card[of F s]
      unfolding S_def
      by (auto simp add: add_set_def Family_inverse)
         (metis Set_inverse card_Suc_eq card_seteq elements_inverse finite_atLeastLessThan insert_iff mem_Collect_eq)
  qed
qed

(* parent *)

definition parent :: "Family \<Rightarrow> Family" where
  "parent F = Family (sets F - {Max (sets F)})"

lemma add_set_parent_Max [simp]:
  assumes "sets F \<noteq> {}"
  shows "add_set (parent F) (Max (sets F)) = F"
  using assms Max_in
  by (smt finite_sets Family_inverse Set.set_insert add_set_sets finite_insert insert_Diff_single insert_absorb2 insert_is_Un mem_Collect_eq parent_def sets_inject sup_commute)

lemma parent_card [simp]:
  assumes "sets F \<noteq> {}"
  shows "card (sets (parent F)) = card (sets F) - 1"
  using assms
  by (simp add: parent_def Family_inverse)

lemma augmenting_sets_complete:
  assumes "F \<in> S n (q+1)"
  shows "F \<in> set (augment n (parent F))"
proof-
  from assms
  have *: "family_card F > 1" "\<forall> x \<in> sets F. elements x \<subseteq> {0..<n}" "Set {} \<notin> sets F" "union_closed F"
    unfolding S_def
    by auto

  let ?M = "Max (sets F)"

  have "sets F \<noteq> {}" 
    using `family_card F > 1`
    by auto
  have "sets F - {?M} \<noteq> {}"
    using `family_card F > 1`
    unfolding family_card_def
    by (metis Max_in One_nat_def card_empty card_insert_disjoint equals0D finite.emptyI infinite_remove insert_Diff less_numeral_extra(4) not_one_less_zero)

  have "?M \<in> sets F" "finite (sets F)"
    using S_finite \<open>sets F \<noteq> {}\<close> assms
    by auto

  hence "elements ?M \<subseteq> {0..<n}" "elements (Max (sets F)) \<noteq> {}"
    using * 
    by (blast, metis elements_inverse)

  moreover

  have "Max (sets F - {?M}) < ?M"
    using `?M \<in> sets F` Max_mono[of "sets F - {?M}" "sets F"] 
    using `sets F - {?M} \<noteq> {}` `finite (sets F)`
    by (metis Diff_iff Diff_subset Max_in finite_Diff less_eq_Set_def singletonI)

  ultimately

  have "?M \<in> augment_set n (Max (sets F - {?M}))"
    unfolding augment_set_def
    by (metis (mono_tags, lifting) mem_Collect_eq)

  moreover

  have "F = Family (insert ?M (sets F - {?M}))"
    using `?M \<in> sets F`
    by (simp add: insert_absorb sets_inverse)

  ultimately 
                   
  show ?thesis
    using `union_closed F`
    unfolding augment_def Let_def S_def parent_def
    by (force simp add: augment_set_finite image_iff Family_inverse)
qed

lemma augment_parent:
  assumes "finite (sets F)"  "sets F \<noteq> {}"
  assumes "F' \<in> set (augment N F)" 
  shows "F = parent F'"
proof-
  from assms obtain s where
    "s \<in> augment_set N (Max (sets F))" "F' = Family (sets F \<union> {s})"
    using assms
    unfolding augment_def
    by (auto simp add: augment_set_finite)
  hence "s > Max (sets F)"
    using augment_set_def by blast
  hence "s \<notin> sets F"
    using assms
    by auto
  thus ?thesis
    using `s > Max (sets F)` assms
    unfolding parent_def `F' = Family (sets F \<union> {s})`
    by (cases F) (smt Diff_empty Diff_insert0 Family_inverse Max_gr_iff Max_in Un_insert_right finite.insertI insert_Diff_if insert_iff less_irrefl mem_Collect_eq sup_bot.right_neutral)
qed

lemma parent_dim:
  assumes "F \<in> S n (q + 1)"
  shows "parent F \<in> S n q"
proof-
  have "F \<in> set (augment n (parent F))"
    using assms augmenting_sets_complete
    by blast
  thus ?thesis
    using assms augment_dim
    by simp
qed

lemma S_parent_not_empty:
  assumes "F \<in> S N (q + 1)"
  shows "sets (parent F) \<noteq> {}"
  using assms
  unfolding S_def parent_def
  by (metis S_not_empty assms parent_def parent_dim)

lemma S_parent_finite:
  assumes "F \<in> S N (q + 1)"
  shows "finite (sets (parent F))"
  using assms
  unfolding S_def parent_def
  using S_finite assms by auto

lemma parent_leq:
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "finite (sets F')" "sets F' \<noteq> {}"
  assumes "card (sets F) = card (sets F')"
  assumes "F \<le> F'" 
  shows "parent F \<le> parent F'"
proof (rule less_eq_Family_remove_Max1)
  show "\<forall>s\<in>sets (parent F). s < Max (sets F)"
    unfolding parent_def
    using assms
    by (metis DiffD1 DiffE Family_inverse eq_Max_iff finite_Diff finite_sets less_eq_Set_def mem_Collect_eq singleton_iff)
next
  show "add_set (parent F) (Max (sets F)) \<le> add_set (parent F') (Max (sets F'))"
    using assms
    by simp
next
  show "family_card (parent F) = family_card (parent F')"
    using assms
    by simp
next
  show "finite (sets (parent F))" "finite (sets (parent F'))"
    using assms
    by (auto simp add: parent_def)
qed

(* is_canon *)

global_interpretation CanonFamily: CanonLeastPerm where
    invar = "\<lambda> n F. \<forall>s\<in>sets F. elements s \<subseteq> {0..<n}" and
    permute = "\<lambda> n p F. permute_family p F"
  defines
    is_canon = "CanonFamily.is_canon" and
    canon = "CanonFamily.canon"
proof
qed

lemma is_canon_parent:
  assumes "is_canon n F" "F \<in> S n (q + 1)"
  shows "is_canon n (parent F)"
proof-
  let ?perms = "permute [0..<n]"
  let ?FPperms = "map (\<lambda>p. permute_family p (parent F)) ?perms"
  let ?sls = sorted_list_of_set

  have setsF: "sets F \<noteq> {} \<and> finite (sets F)"
    using `F \<in> S n (q + 1)`
    by (simp add: S_finite S_not_empty)
  have elementsF: "\<forall>s\<in>sets F. elements s \<subseteq> {0..<n}"
    using `F \<in> S n (q + 1)`
    by (simp add: S_def)
  have "Max (sets F) \<in> sets F"
    using setsF Max_in[of "sets F"]
    by simp
  hence "sets F = sets (parent F) \<union> {Max (sets F)}"
       "Max (sets F) \<notin> sets (parent F)"
    unfolding parent_def
    by (auto simp add: Family_inverse)

  show ?thesis
    unfolding CanonFamily.is_canon_def
  proof safe
    fix p
    assume "p <~~> [0..<n]"
    hence p: "p \<in> set (permute [0..<n])"
      by (simp add: permute_isPermutation)

    let ?parent_F' = "permute_family p (parent F)"

    let ?M = "Max (sets F)"
    let ?M' = "permute_set p ?M"

    show "parent F \<le> ?parent_F'"
    proof (rule less_eq_Family_remove_Max1)
      show "family_card (parent F) = family_card ?parent_F'"
        using p permute_family_card elementsF
        unfolding parent_def
        by (simp add: Family_inverse)
    next
      show "finite (sets (parent F))"
        using assms setsF
        unfolding parent_def
        by simp
    next
      show "finite (sets ?parent_F')"
        using assms setsF
        by (simp add: permute_family_def parent_def)
    next
      show "\<forall> s \<in> sets (parent F). s < ?M"
        using setsF Max_ge[of "sets F"]
        using less_eq_Set_def parent_def
        by (auto simp add: Family_inverse)
    next
      show "add_set (parent F) (Max (sets F)) \<le> add_set ?parent_F' ?M'"
      proof-
        have *: "add_set (parent F) (Max (sets F)) = F"
          using setsF
          by auto
        moreover
        have "add_set ?parent_F' (permute_set p (Max (sets F))) = permute_family p F"
          using *
          unfolding permute_family_def add_set_def
          by (metis Family_inverse S_parent_finite Un_insert_right \<open>sets F = sets (parent F) \<union> {Max (sets F)}\<close> assms(2) finite_imageI image_insert mem_Collect_eq sup_bot.right_neutral)
        ultimately
        show ?thesis
          using p `is_canon n F`
          by (simp add: CanonFamily.is_canon_def isPermutation_permute)
      qed
    qed
  qed
qed

lemma is_canon_parent_iterate:
  assumes "finite M" "finite F" "F \<noteq> {}"  "\<forall> x \<in> M. \<forall> y \<in> F. x > y"
          "Family (F \<union> M) \<in> S n q" "is_canon n (Family (F \<union> M))"
  shows "is_canon n (Family F)"
  using assms
proof (induction "sorted_list_of_set M" arbitrary: M q rule: rev_induct)
  case Nil
  then show ?case
    using sorted_list_of_set_eq_Nil_iff
    by fastforce
next
  case (snoc x xs)
  let ?M = "sets (parent (Family M))"

  have "x \<in> M"
    using `xs @ [x] = sorted_list_of_set M`
    by (metis Un_iff list.set_intros(1) set_append set_sorted_list_of_set snoc.prems(1))

  hence "M \<noteq> {}"
    by auto

  have "q > 0"
  proof-
    have "x \<notin> F"
      using `x \<in> M` snoc.prems
      by auto
    moreover
    have "card (F \<union> M) = q + 1"
      using S_def[of n q] snoc.prems
      unfolding family_card_def
      by (simp add: Family_inverse)
    ultimately
    show ?thesis
      using `F \<noteq> {}` `x \<in> M`
      by (metis One_nat_def Suc_leI Un_upper1 add.commute card_gt_0_iff card_seteq emptyE finite_Un gr0I le_sup_iff less_one plus_1_eq_Suc)
  qed

  have "Max M \<in> M"
    using \<open>M \<noteq> {}\<close> snoc.prems(1) 
    by auto

  hence "Max M \<notin> F"
    using snoc.prems
    by auto

  have "Max (F \<union> M) = Max M"
    using snoc.prems `M \<noteq> {}`
    by (metis Max_ge Max_in Un_empty Un_iff finite_UnI infinite_growing less_eq_Set_def)

  hence *: "Family (F \<union> ?M) = parent (Family (F \<union> M))"
    using `Max M \<notin> F` `finite M` `finite F`
    by (auto simp add: parent_def Family_inverse Un_Diff)

  show ?case
  proof (rule snoc(1))
    show "xs = sorted_list_of_set ?M"
      using snoc.prems `M \<noteq> {}`
      using snoc(2)[symmetric] sorted_list_of_set_remove_Max[of M]
      by (auto simp add: parent_def Family_inverse)
  next
    show "finite ?M"
      using snoc.prems
      by (simp add: parent_def)
  next
    show "F \<noteq> {}"
      by fact
  next
    show "\<forall> x \<in> sets (parent (Family M)). \<forall> y \<in> F. y < x"
      using snoc.prems
      by (simp add: parent_def Family_inverse)
  next
    show "is_canon n (Family (F \<union> ?M))"
      using * is_canon_parent[of n "Family (M \<union> F)" "q-1"]
      using snoc.prems `q > 0`
      by (auto simp add: Un_commute)
  next
    show "Family (F \<union> ?M) \<in> S n (q - 1)"
      using * parent_dim[of "Family (F \<union> M)" n "q-1"] snoc.prems `q > 0`
      by auto
  next
    show "finite F"
      by fact
  qed
qed

(* is_canon optimization *)

definition min_card :: "Family \<Rightarrow> nat" where
  "min_card F = Min (set_card ` (sets F))"

definition max_card :: "Family \<Rightarrow> nat" where
  "max_card F = Max (set_card ` (sets F))"

definition min_card_above :: "Family \<Rightarrow> nat \<Rightarrow> nat" where
  "min_card_above F c = Min (set_card ` {s \<in> sets F. set_card s > c})"

abbreviation min_card_above_min_card :: "Family \<Rightarrow> nat" where
  "min_card_above_min_card F \<equiv> min_card_above F (min_card F)"

(* min_card *)

lemma ex_min_card:
  assumes "finite (sets F)" "sets F \<noteq> {}"
  shows "\<exists> s \<in> sets F. set_card s = min_card F"
  using assms      
  unfolding min_card_def
  by (metis (mono_tags, lifting) Min_in finite_imageI imageE image_is_empty)

lemma min_card_add_set [simp]:
  assumes "sets F \<noteq> {}"
  shows "min_card (add_set F s) = min (min_card F) (set_card s)"
  using assms
  by (simp add: add_set_def min_card_def Family_inverse)

lemma min_card_insert [simp]:
  assumes "finite F" "F \<noteq> {}"
  shows "min_card (Family (insert s F)) = min (min_card (Family F)) (set_card s)"
  using assms
  by (simp add: add_set_def min_card_def Family_inverse)

lemma min_card_min:
  assumes "sets F \<noteq> {}" "finite (sets F)"
  shows "\<forall> A \<in> sets F. set_card A \<ge> min_card F"
  using assms
  unfolding min_card_def
  by auto

lemma max_card_max:
  assumes "sets F \<noteq> {}" "finite (sets F)"
  shows "\<forall> A \<in> sets F. set_card A \<le> max_card F"
  using assms
  unfolding max_card_def
  by auto

lemma min_card_Max [simp]:
  assumes "finite (sets F)" "sets F \<noteq> {}"
  shows "set_card (Max (sets F)) = min_card F"
  unfolding min_card_def
  by (smt Max_ge Max_in Min_in antisym assms(1) assms(2) finite_imageI image_iff image_is_empty leq_Set_geq_card min_card_def min_card_min)

lemma permute_family_min_card [simp]:
  assumes "p \<in> set (permute [0..<n])"
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}" 
  shows "min_card (permute_family p F) = min_card F"
  using assms permute_family_cards[OF assms]
  by (simp add: min_card_def)

(* min_card_above *)

lemma min_card_above_leq:
  assumes "finite (sets F)"
  assumes "\<exists> s \<in> sets F. set_card s = c" "c > C"
  shows "c \<ge> min_card_above F C"
  using assms
  unfolding min_card_above_def
  by auto

(* min_card_above_min_card *)

lemma min_card_above_min_card_add_set:
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "set_card s < min_card F"
  shows "min_card_above_min_card (add_set F s) = min_card F"
proof-
  have "\<forall> s' \<in> sets F. set_card s' > set_card s"
    using assms
    by (meson leD leI le_trans min_card_min)
  hence "(set_card ` {sa \<in> sets (add_set F s). min_card (add_set F s) < set_card sa}) = 
        (set_card ` sets F)"
    using assms
    unfolding min_card_above_def
    by (auto simp add: min_def add_set_def Family_inverse sets_inverse)
  thus ?thesis
    unfolding min_card_above_def min_card_def
    by simp
qed

(* card_sets *)

definition eq_card_sets :: "Family \<Rightarrow> nat \<Rightarrow> Set set" where
  "eq_card_sets F c = {s. s \<in> sets F \<and> set_card s = c}"

abbreviation min_card_sets :: "Family \<Rightarrow> nat \<Rightarrow> Set set" where
  "min_card_sets F c \<equiv> eq_card_sets F (min_card F)"

definition above_card_sets :: "Family \<Rightarrow> nat \<Rightarrow> Set set" where
  "above_card_sets F c = {s. s \<in> sets F \<and> set_card s > c}"

abbreviation above_min_card_sets :: "Family \<Rightarrow> Set set" where
  "above_min_card_sets F \<equiv> above_card_sets F (min_card F)" 


lemma permute_family_above_card_sets [simp]:
  assumes "p \<in> set (permute [0..<n])"
  assumes "finite (sets F)" "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}" 
  shows "permute_family p (Family (above_card_sets F c)) = 
         Family (above_card_sets (permute_family p F) c)"
proof (cases "sets F = {}")
  case True
  thus ?thesis
    by (simp add: above_card_sets_def permute_family_def Family_inverse)
next
  case False

  have "(permute_set p ` {s \<in> sets F. c < card (elements s)}) =
        {s \<in> permute_set p ` sets F. c < card (elements s)}" (is "?lhs = ?rhs")
  proof
    show "?lhs \<subseteq> ?rhs"
    proof
      fix x
      assume "x \<in> ?lhs"
      then obtain x' where "x = permute_set p x'" "x' \<in> sets F" "c < card (elements x')"
        by auto
      thus "x \<in> ?rhs"
        using assms permute_set_card[of p n x']
        by auto
    qed
  next
    show "?rhs \<subseteq> ?lhs"
    proof
      fix x
      assume "x \<in> ?rhs"
      then obtain x' where "x = permute_set p x'" "x' \<in> sets F" "c < card (elements x)"
        by auto
      thus "x \<in> ?lhs"
        using assms permute_set_card[of p n x']
        by auto
    qed
  qed
      
  thus ?thesis
    by (simp add: above_card_sets_def permute_family_def Family_inverse)
qed

lemma permute_family_above_min_card_sets [simp]:
  assumes "p \<in> set (permute [0..<n])" 
  assumes "finite (sets F)" "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}"
  shows "permute_family p (Family (above_min_card_sets F)) =
         Family (above_min_card_sets (permute_family p F))"
  using assms
proof (cases "sets F = {}")
  case True
  thus ?thesis
    by (simp add: above_card_sets_def permute_family_def Family_inverse)
next
  case False
  thus ?thesis
    using assms
    by simp
qed

lemma permute_family_card_sets [simp]:
  assumes "p \<in> set (permute [0..<n])" 
  assumes "finite (sets F)"
  assumes "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}"
  shows "permute_family p (Family (eq_card_sets F c)) =
         Family (eq_card_sets (permute_family p F) c)"
proof (cases "sets F = {}")
  case True
  thus ?thesis
    by (simp add: eq_card_sets_def permute_family_def Family_inverse)
next
  case False

  have "(permute_set p ` {s \<in> sets F. card (elements s) = c}) =
        {s \<in> permute_set p ` sets F. card (elements s) = c}" (is "?lhs = ?rhs")
  proof
    show "?lhs \<subseteq> ?rhs"
    proof
      fix x
      assume "x \<in> ?lhs"
      then obtain x' where "x = permute_set p x'" "x' \<in> sets F" "c = card (elements x')"
        by auto
      thus "x \<in> ?rhs"
        using assms permute_set_card[of p n x']
        by auto
    qed
  next
    show "?rhs \<subseteq> ?lhs"
    proof
      fix x
      assume "x \<in> ?rhs"
      then obtain x' where "x = permute_set p x'" "x' \<in> sets F" "c = card (elements x)"
        by auto
      thus "x \<in> ?lhs"
        using assms permute_set_card[of p n x']
        by auto
    qed
  qed

  thus ?thesis
    by (simp add: eq_card_sets_def permute_family_def Family_inverse)
qed

(* filter perms *)

definition perm_fixes_subset :: "perm \<Rightarrow> Set set \<Rightarrow> Set set \<Rightarrow> bool" where
  "perm_fixes_subset p F F' \<longleftrightarrow> (\<forall> s \<in> F'. permute_set p s \<in> F)"

abbreviation perm_fixes :: "perm \<Rightarrow> Set set \<Rightarrow> bool" where
  "perm_fixes p F \<equiv> perm_fixes_subset p F F"

definition filter_perms :: "perm list \<Rightarrow> Family \<Rightarrow> perm list" where
  "filter_perms ps F = (let FF = above_min_card_sets F
                         in filter (\<lambda> p. perm_fixes p FF) ps)"

(* is_canon_opt *)
definition is_canon_opt :: "nat \<Rightarrow> Family \<Rightarrow> bool" where
  "is_canon_opt n F \<longleftrightarrow> (\<forall> p \<in> set (filter_perms (permute [0..<n]) F). F \<le> permute_family p F)"

lemma is_canon_opt':
  assumes "is_canon n F"
  shows "is_canon_opt n F"
  using assms
  unfolding CanonFamily.is_canon_def is_canon_opt_def Let_def filter_perms_def
  by (smt Ball_set list.pred_map mem_Collect_eq set_filter isPermutation_permute)

lemma is_canon_opt:
  assumes "is_canon n (parent F)" "is_canon_opt n F" "F \<in> S n q"
  shows "is_canon n F"
proof-
  have setsF: "finite (sets F)" "sets F \<noteq> {}"
    using `F \<in> S n q`
    using S_finite S_not_empty by auto

  have elementsF: "\<forall> A \<in> sets F. elements A \<subseteq> {0..<n}"
    using `F \<in> S n q`
    by (simp add: S_def)

  let ?perms = "permute [0..<n]"
  let ?perms' = "filter_perms ?perms F"
  let ?Fperms = "map (\<lambda> p. permute_family p F) ?perms"
  let ?Fperms' = "map (\<lambda> p. permute_family p F) ?perms'"

  have *: "\<forall> F' \<in> set ?Fperms'. F \<le> F'"
    using assms
    unfolding is_canon_opt_def Let_def list_all_iff
    by simp

  have "\<forall> F' \<in> set ?Fperms. F \<le> F'"
  proof safe
    fix F'
    assume "F' \<in> set ?Fperms"
    then obtain p where **: "p \<in> set (permute [0..<n])" "F' = permute_family p F"
      by auto
    show "F \<le> F'"
    proof (cases "p \<in> set (filter_perms (permute [0..<n]) F)")
      case True
      then show ?thesis
        using * **
        by auto
    next

      let ?FX = "above_min_card_sets F"
      let ?P = "eq_card_sets F (min_card F)"
      let ?FX' = "permute_family p (Family ?FX)"

      have "finite ?FX"
        unfolding above_card_sets_def
        by auto

      have "finite ?P"
        unfolding eq_card_sets_def
        by simp

      case False
      then have "\<not> perm_fixes p ?FX"
        using **
        unfolding filter_perms_def Let_def
        by auto
      then obtain A where "A \<in> ?FX" "permute_set p A \<notin> ?FX"
        using `\<not> perm_fixes p ?FX`
        unfolding perm_fixes_subset_def
        by (auto simp add: list_all_iff)
      hence "sets ?FX' \<noteq> ?FX"
        using `finite ?FX`
        unfolding permute_family_def
        by (auto simp add: Family_inverse)
      hence "?FX' \<noteq> Family ?FX"
        using `finite ?FX`
        by (auto simp add: Family_inverse)
      have "?FX \<noteq> {}"
        using `A \<in> ?FX`
        by fastforce

      have P: "sets F = ?FX \<union> ?P"
        using min_card_min[of F] setsF
        unfolding above_card_sets_def eq_card_sets_def
        by auto

      show ?thesis
      proof-
        have "F < F'"
        proof (rule less_Family_add_sets)
          show "sets F = ?FX \<union> ?P"
            using P
            by auto
        next
          show "card ?FX = card (sets ?FX')"
            using "**"(1) S_def above_card_sets_def assms(3) permute_family_card
            using `finite ?FX`
            by (auto simp add: Family_inverse)
        next
          show "Family ?FX < Family (sets ?FX')"
          proof-
            have "Family ?FX \<le> Family (sets ?FX')"
            proof-
              let ?P' = "eq_card_sets (parent F) (min_card F)"
              have P': "sets (parent F) = ?FX \<union> ?P'"
                using min_card_min[of F]
                using min_card_Max[of F] setsF
                unfolding above_card_sets_def eq_card_sets_def
                by (auto simp add: parent_def setsF Family_inverse)
              have "is_canon n (Family ?FX)"
              proof (rule is_canon_parent_iterate)
                show "finite ?P'"
                  using setsF
                  by (simp add: eq_card_sets_def parent_def)
              next
                show "is_canon n (Family (?FX \<union> ?P'))"
                  using P' assms
                  by (metis sets_inverse)
              next
                show "Family (?FX \<union> ?P') \<in> S n (q-1)"
                proof-
                  have "?P \<noteq> {}"
                    using setsF 
                    using Max_in P P' Un_commute parent_def
                    by (smt Diff_iff Family_inverse Un_insert_right Un_left_absorb finite_Diff insert_Diff mem_Collect_eq singletonI sup_bot.right_neutral)
                  moreover
                  have "?P \<inter> ?FX = {}"
                    by (auto simp add: eq_card_sets_def above_card_sets_def)
                  ultimately
                  have "card (sets F) > 1"
                    using `?FX \<noteq> {}` P P' `finite ?FX`
                    by (metis finite_sets parent_card One_nat_def Suc_pred Un_commute add.commute card.empty card_union_gt finite.emptyI finite_Un not_add_less2 not_less_iff_gr_or_eq old.nat.inject plus_1_eq_Suc setsF(2) sup_bot.right_neutral)
                  hence "q > 0"
                    using `F \<in> S n q`
                    unfolding S_def
                    by simp
                  thus ?thesis
                    using P P' assms parent_dim[of F n "q-1"]
                    by (metis sets_inverse One_nat_def Suc_pred add.commute plus_1_eq_Suc)
                qed
              next
                show "?FX \<noteq> {}"
                  by fact
              next
                show "\<forall> x \<in> ?P'. \<forall> y \<in> ?FX. y < x"
                  by (simp add: above_card_sets_def eq_card_sets_def less_Set_def)
              next
                show "finite ?FX"
                  by fact
              qed

              thus ?thesis
                using `p \<in> set (permute [0..<n])`
                unfolding CanonFamily.is_canon_def Let_def
                by (simp add: isPermutation_permute sets_inverse)
            qed
            thus ?thesis
              using `?FX' \<noteq> Family ?FX`
              by (simp add: sets_inverse)
          qed
        next
          show "sets F' = sets ?FX' \<union> sets (permute_family p (Family ?P))"
            using `F' = permute_family p F` P `finite ?FX` `finite ?P`
            by (auto simp add: permute_family_def Family_inverse)
        next
          show "finite (sets F)"
            by simp
        next
          show "finite (sets F')"
            by simp
        next
          show "\<forall>x\<in>?FX. \<forall>y\<in>?P. x < y"
            using `p \<in> set (permute [0..<n])` setsF elementsF
            by (metis (mono_tags, lifting) above_card_sets_def eq_card_sets_def leq_Set_geq_card mem_Collect_eq not_le)
        next
          have "\<forall>x\<in>above_card_sets (permute_family p F) (min_card F).
                    \<forall>y\<in>eq_card_sets (permute_family p F) (min_card F). x \<le> y"
            by (metis (mono_tags, lifting) above_card_sets_def eq_card_sets_def less_Set_def less_imp_le mem_Collect_eq)
          thus "\<forall>x\<in>sets ?FX'.
                   \<forall>y\<in>sets (permute_family p (Family ?P)). x < y"
            using `p \<in> set (permute [0..<n])` setsF elementsF
            using "**"(2) above_card_sets_def eq_card_sets_def leD less_eq_Set_def mem_Collect_eq order_refl permute_family_above_card_sets permute_family_card_sets
            by (smt Family_inverse \<open>finite (above_min_card_sets F)\<close> \<open>finite (eq_card_sets F (min_card F))\<close> finite_imageI imageE less_Set_def permute_family_def permute_set_card)
        qed
        thus ?thesis
          by simp
      qed
    qed
  qed
  thus ?thesis
    unfolding CanonFamily.is_canon_def Let_def list_all_iff
    by (simp add: permute_isPermutation)
qed


(********************************************************************************** *)
definition N :: nat where "N = 6"

interpretation UnionClosed': FaradzevRead' "S N" "equiv N" "is_canon N" "is_canon_opt N" "augment N"
proof
  fix q
  show "equivp_on (S N q) (equiv N)"
    using EquivFamily.equivp_equiv[of N] equivp_on_subset[of "{a. \<forall>s\<in>sets a. elements s \<subseteq> {0..<N}}" "equiv N" "S N q"]
    unfolding S_def
    by auto
next
  fix s s' q
  assume "equiv N s s'" "s \<in> S N q"
  thus "s' \<in> S N q"
    by (rule equiv_S)
next
  fix q s
  assume "s \<in> S N q"
  thus "\<exists>! sc. equiv N s sc \<and> is_canon N sc"
    using CanonFamily.is_canon_unique[of s N]
    unfolding S_def
    by auto
next
  fix q s s'
  assume "s \<in> S N q" "is_canon N s" "s' \<in> set (augment N s)"
  thus "is_canon_opt N s' = is_canon N s'"
    using augment_dim augment_parent
    using is_canon_opt
    using is_canon_opt' S_finite S_finite_elements S_not_empty
    by metis
next
  fix s s' q
  assume "s' \<in> set (augment N s)"
  thus "s \<in> S N q \<longleftrightarrow> s' \<in> S N (q + 1)"
    by (rule augment_dim)
next
  fix q1 q2 :: nat
  assume "q1 \<noteq> q2"
  thus "S N q1 \<inter> S N q2 = {}"
    unfolding S_def
    by auto
qed


lemma parent_is_parent:
  assumes "X \<in> S N (q + 1)"  "UnionClosed'.parent Y X"
  shows "Y = parent X"
proof-
  have *: "X \<in> set (augment N Y)"
    using assms
    unfolding UnionClosed'.parent_def Let_def
    by (auto simp add: augment_set_finite)

  have **: "card (sets X) = q + 2" "sets X \<noteq> {}" "finite (sets X)"
    using `X \<in> S N (q + 1)`
    unfolding S_def
    using card_infinite 
    by force+

  have "finite (sets Y)" "sets Y \<noteq> {}"
    using * **
    unfolding augment_def
    by (auto simp add: augment_set_finite image_iff Family_inverse)

  thus "Y = parent X"
    using augment_parent *
    by auto
qed


global_interpretation UnionClosed: FaradzevReadStrict
where
 S = "S N" and 
 equiv = "equiv N" and
 is_canon = "is_canon N" and
 is_canon_test = "is_canon_opt N" and
 augment = "augment N" 
defines
 step_opt = "UnionClosed.step_opt" and
 steps_opt = "UnionClosed.steps_opt"
proof
  fix q s
  assume "s \<in> S N (q + 1)" "is_canon N s"
  show "\<exists>p\<in>S N q. UnionClosed'.parent p s"
  proof (rule_tac x="parent s" in bexI)
    show "UnionClosed'.parent (parent s) s"
      using augmenting_sets_complete is_canon_parent
      using `is_canon N s` `s \<in> S N (q + 1)`
      unfolding UnionClosed'.parent_def
      by simp
  next
    show "(parent s) \<in> S N q"
      using augmenting_sets_complete[OF `s \<in> S N (q + 1)`]
      using augment_dim `s \<in> S N (q + 1)`
      by simp
  qed
next
  fix s
  show "sorted (filter (is_canon N) (augment N s)) \<and> 
        distinct (filter (is_canon N) (augment N s))"
    unfolding augment_def
    by simp
next
  fix q p s p' s'
  assume *: "p \<in> S N q" "p' \<in> S N q" 
            "UnionClosed'.parent p s" "UnionClosed'.parent p' s'" "p < p'"
  obtain a a' where 
   a: "s = Family ((sets p) \<union> {a})" "s' = Family ((sets p') \<union> {a'})"
      "a \<in> augment_set N (Max (sets p))" "a' \<in> augment_set N (Max (sets p'))"
    using *
    unfolding UnionClosed'.parent_def augment_def
    by (auto simp add: augment_set_finite)
  show "s < s'"
  proof (rule FamilyAbs.less_Family_add_sets)
    show "finite (sets s)" "finite (sets s')" 
      using *
      using S_finite augment_dim UnionClosed'.parent_def
      by blast+
  next
    show "card (sets p) = card (sets p')"
      using * unfolding S_def UnionClosed'.parent_def
      by simp
  next
    show "Family (sets p) < Family (sets p')"
      using `p < p'`
      by (simp add: sets_inverse)
  next
    show "sets s = sets p \<union> {a}"  "sets s' = sets p' \<union> {a'}"
      using a
      by (simp_all add: Family_inverse)
  next
    show "\<forall>x\<in>sets p. \<forall>y\<in>{a}. x < y"
      by (metis "*"(1) Max_less_iff S_finite S_not_empty a(3) augment_set_gt singletonD)+
    show  "\<forall>x\<in>sets p'. \<forall>y\<in>{a'}. x < y"
      by (metis "*"(2) Max_less_iff S_finite S_not_empty a(4) augment_set_gt singletonD)+
  qed
qed

(********************************************* *)

(* above_card_sets *)
lemma above_card_sets_add_set:
  shows "above_card_sets (add_set F s) C = 
          (if set_card s > C then
              {s} \<union> above_card_sets F C
           else
              above_card_sets F C)"
  unfolding above_card_sets_def
  by (auto simp add: add_set_def Family_inverse)

lemma above_min_card_sets_add_set:
  assumes "finite (sets F)" "sets F \<noteq> {}"
  assumes "set_card s \<le> min_card F"
  shows "above_min_card_sets (add_set F s) = 
         (if set_card s < min_card F then
            sets F
          else
            above_min_card_sets F)" 
proof (cases "set_card s < min_card F")
  case True
  thus ?thesis
    using assms
    unfolding above_card_sets_def
    by (auto simp add: add_set_def min_def min_card_def Family_inverse)
next
  case False
  hence "set_card s \<ge> min_card F"
    by simp
  hence "min_card (add_set F s) = min_card F"
    using assms
    by (simp add: min_card_def add_set_def Family_inverse)
  thus ?thesis
    using assms
    unfolding above_card_sets_def Let_def
    by (auto simp add: add_set_def min_def min_card_def Family_inverse)
qed

(* perm_fixes *)

lemma perm_fixes_singleton:
  shows "perm_fixes p (above_min_card_sets (Family {s}))"
  unfolding above_card_sets_def
  by (simp add: min_card_def perm_fixes_subset_def Family_inverse)

lemma perm_fixes_add_set_1:
  assumes "finite (sets F)" "sets F \<noteq> {}" 
  assumes "perm_fixes p (above_min_card_sets F)"
  assumes "set_card s = min_card F"
  shows "perm_fixes p (above_min_card_sets (add_set F s))"
  unfolding perm_fixes_subset_def
proof
  fix s'
  assume "s' \<in> above_min_card_sets (add_set F s)"
  hence "s' \<in> sets (add_set F s)" 
    unfolding above_card_sets_def
    by auto
  thus "permute_set p s' \<in> above_min_card_sets (add_set F s)"
    using assms above_min_card_sets_add_set[OF `finite (sets F)` `sets F \<noteq> {}`, of s]
    using `s' \<in> above_min_card_sets (add_set F s)`
    by (simp add: perm_fixes_subset_def)
qed

lemma perm_fixes_add_set_2:
  assumes "p \<in> set (permute [0..<n])"
  assumes "finite (sets F)"  "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  assumes "set_card s \<le> min_card F"
  assumes "perm_fixes p (above_min_card_sets (add_set F s))" 
  shows "perm_fixes p (above_min_card_sets F)"
  unfolding perm_fixes_subset_def
proof safe
  fix s'
  assume "s' \<in> above_min_card_sets F"
  hence *: "s' \<in> sets F" "set_card s' > min_card F"
    unfolding above_card_sets_def
    by auto
  hence "sets F \<noteq> {}"
    by auto
  show "permute_set p s' \<in> above_min_card_sets F"
  proof (cases "set_card s < min_card F")
    case True
    hence "above_min_card_sets (add_set F s) = sets F"
      using assms above_min_card_sets_add_set[OF `finite (sets F)` `sets F \<noteq> {}`, of s]
      by simp
    hence "permute_set p s' \<in> sets F"
      using assms `s' \<in> sets F`
      by (simp add: perm_fixes_subset_def)
    moreover
    have "set_card (permute_set p s') = set_card s'"
      using assms `s' \<in> sets F`
      using permute_set_card by blast
    ultimately
    show ?thesis
      using * 
      unfolding above_card_sets_def
      by simp
  next
    case False
    hence "above_min_card_sets (add_set F s) = above_card_sets F (min_card F)"
      using * assms above_min_card_sets_add_set[OF `finite (sets F)` `sets F \<noteq> {}`, of s]
      by simp
    thus ?thesis
      using assms `s' \<in> above_card_sets F (min_card F)`
      unfolding perm_fixes_subset_def
      by simp
  qed
qed
  
lemma perm_fixes_above_eq:
  assumes "finite (sets F)" "p \<in> set (permute [0..<n])" "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  assumes "perm_fixes p (above_card_sets F (min_card_above_min_card F))"
  assumes "perm_fixes_subset p (sets F) (eq_card_sets F (min_card_above_min_card F))"
  shows "perm_fixes p (above_min_card_sets F)"
  unfolding perm_fixes_subset_def above_card_sets_def
proof safe
  fix s
  assume "s \<in> sets F" "min_card F < set_card s"

  show "min_card F < set_card (permute_set p s)"
    using `s \<in> sets F` `set_card s > min_card F` assms
    using permute_set_card
    by auto
  
  have *: "set_card s \<ge> min_card_above_min_card F"
  proof (rule min_card_above_leq)
    show "\<exists>sa\<in>sets F. set_card sa = set_card s"
      using `s \<in> sets F`
      by auto
  next
    show "min_card F < set_card s"
      by fact
  next
    show "finite (sets F)"
      by fact
  qed

  show "permute_set p s \<in> sets F"
  proof (cases "set_card s = min_card_above_min_card F")
    case True
    then show ?thesis
      using assms `s \<in> sets F`
      unfolding perm_fixes_subset_def eq_card_sets_def
      by simp
  next
    case False
    hence "set_card s > min_card_above_min_card F"
      using *
      by simp
    thus ?thesis
      using assms `s \<in> sets F`
      unfolding perm_fixes_subset_def above_card_sets_def
      by simp
  qed
qed

lemma perm_fixes_above_eq_iff:
  assumes "set_card s < min_card F" "finite (sets F)" "sets F \<noteq> {}" "p \<in> set (permute [0..<n])"
  assumes "\<forall>s\<in>sets F. elements s \<subseteq> {0..<n}" "elements s \<subseteq> {0..<n}"
  assumes "F' = add_set F s"
  shows "perm_fixes p (above_min_card_sets F) \<and>
         perm_fixes_subset p (sets F') (eq_card_sets F' (min_card_above_min_card F')) \<longleftrightarrow>
         perm_fixes p (above_min_card_sets F')" (is "?lhs1 \<and> ?lhs2 \<longleftrightarrow> ?rhs")
proof-
  have "min_card (add_set F s) = set_card s"
    using assms
    by simp
  hence *: "min_card_above (add_set F s) (min_card (add_set F s)) = min_card F"
    using assms
    by (metis (no_types, lifting) above_card_sets_def above_min_card_sets_add_set assms(3) less_imp_le min_card_above_def min_card_def)
  show ?thesis
  proof safe
    assume ?lhs1 ?lhs2
    show ?rhs
    proof (rule perm_fixes_above_eq)
      show "?lhs2"
        by fact
      show "perm_fixes p (above_card_sets F' (min_card_above_min_card F'))"
        using * `?lhs1` `set_card s < min_card F` `F' = add_set F s`
        by (auto simp add: perm_fixes_subset_def above_card_sets_def add_set_def Family_inverse)
    next
      show "finite (sets F')"
        using assms
        by (simp add: add_set_def)
    next
      show "p \<in> set (permute [0..<n])"
        by fact
    next
      show "\<forall>s\<in>sets F'. elements s \<subseteq> {0..<n}"
        using assms
        by (simp add: add_set_def Family_inverse)
    qed
  next
    assume ?rhs
    thus ?lhs1
      using assms less_imp_le perm_fixes_add_set_2 
      by blast
  next
    assume ?rhs
    thus ?lhs2
      using * `set_card s < min_card F`
      using above_min_card_sets_add_set add_set_def assms eq_card_sets_def perm_fixes_subset_def
      by (auto simp add: Family_inverse)
  qed
qed


lemma perm_fixes_subset_id:
  assumes "F' \<subseteq> F" "\<forall> s \<in> F. elements s \<subseteq> {0..<n}" 
  shows "perm_fixes_subset (perm_id n) F F'"
  using assms
  unfolding FamilyAbs.perm_fixes_subset_def
  using permute_set_id
  by auto


definition reduced where
  "reduced F = {s \<in> sets F. \<not> (\<exists> s' \<in> sets F. (elements s') \<subset> (elements s))}"

lemma union_with_gt:
  assumes "s > s'" "finite (elements s)" "finite (elements s')"
  shows "union s s' \<noteq> s"
  by (metis Un_iff assms(1) assms(2) card_seteq elements_inject elements_union leq_Set_geq_card less_imp_le neq_iff set_card_def subsetI)

lemma elements_subset_gt:
  assumes "elements a \<subset> elements b" "finite (elements b)"
  shows "a > b"
  using assms
  by (simp add: less_Set_def psubset_card_mono)

lemma ex_reduced:
  assumes "finite (sets F)" "a \<in> sets F" "finite (elements a)" "a \<notin> reduced F"
  shows "\<exists> a'. elements a' \<subset> elements a \<and> a' \<in> reduced F"
proof-
  let ?A = "{s \<in> sets F. elements s \<subset> elements a}"
  let ?M = "Max ?A"
  have "?A \<noteq> {}"
    using `a \<in> sets F` `a \<notin> reduced F`
    unfolding reduced_def
    by auto
  hence "?M \<in> ?A"
    using Max_in[of "?A"] `finite (sets F)`
    by simp
  hence "elements ?M \<subset> elements a" "?M \<in> sets F"
    by auto
  moreover
  have "?M \<in> reduced F"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    then obtain m where "m \<in> sets F" "elements m \<subset> elements ?M"
      using `?M \<in> sets F`
      unfolding reduced_def
      by auto
    hence "m \<in> ?A"
      using `elements ?M \<subset> elements a`
      by simp
    moreover
    have "m > ?M"
      using \<open>elements ?M \<subset> elements a\<close> \<open>elements m \<subset> elements ?M\<close> 
        assms(3) elements_subset_gt finite_subset
      by blast
    ultimately
    show False
      by (metis (no_types, lifting) Max_ge assms(1) finite_subset leD mem_Collect_eq subsetI)
  qed
  ultimately
  show ?thesis
    by blast
qed

lemma union_closed_iff_reduced:
  assumes "union_closed F" "\<forall> s' \<in> sets F. s > s'" "finite (sets F)" "\<forall> s \<in> sets F. finite (elements s)" "finite (elements s)"
  shows "union_closed (add_set F s) \<longleftrightarrow> (\<forall> s' \<in> (reduced F). union s' s \<in> sets F)" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?lhs"
  show ?rhs
  proof
    fix s'
    assume "s' \<in> reduced F"
    show "union s' s \<in> sets F"
    proof-
      have "union s' s \<in> sets F \<or> union s' s = s"
        using `?lhs` `s' \<in> reduced F`
        unfolding union_closed_def reduced_def add_set_def
        by (auto simp add: Family_inverse)
      moreover
      have "union s' s \<noteq> s"
        using assms `s' \<in> reduced F`
        unfolding reduced_def
        by (metis (no_types, lifting) Un_commute mem_Collect_eq union_def union_with_gt)
      ultimately
      show ?thesis
        by simp
    qed
  qed
next
  assume "?rhs"
  have "\<forall> a \<in> sets F. union a s \<in> sets F \<union> {s}"
  proof
    fix a
    assume "a \<in> sets F"
    show "union a s \<in> sets F \<union> {s}"
    proof (cases "a \<in> reduced F")
      case True
      then show ?thesis
        using `?rhs`
        by simp
    next
      case False
      then obtain a' where "elements a' \<subset> elements a" "a' \<in> reduced F"
        using ex_reduced[of F a]
        using \<open>a \<in> sets F\<close> assms(3) assms(4)
        by auto
      hence "union a' s \<in> sets F"
        using `?rhs`
        by auto
      hence "union (union a' s) a \<in> sets F"
        using `a \<in> sets F` `union_closed F`
        unfolding union_closed_def
        by auto
      moreover
      have "union (union a' s) a = union a s"
        using `elements a' \<subset> elements a`
        by (smt elements_union sup.strict_order_iff sup_commute sup_left_commute union_def)
      ultimately
      show ?thesis
        by simp  
    qed
  qed
  hence "\<forall>A\<in>sets F \<union> {s}. \<forall>B\<in>sets F \<union> {s}. FamilyAbs.union A B \<in> sets F \<union> {s}"
    using `union_closed F` 
    by (simp add: elements_inverse sup_commute union_closed_def union_def)
  thus "?lhs"
    unfolding union_closed_def add_set_def
    by (simp add: Family_inverse)
qed

lemma reduced_add_set: 
  assumes "\<forall> s' \<in> sets F. s' < s" "finite (sets F)" "finite (elements s)" "\<forall> s' \<in> sets F. finite (elements s')"
  shows "reduced (add_set F s) = {s' \<in> reduced F. \<not> elements s \<subseteq> elements s'} \<union> {s}" (is "?lhs = ?rhs")
proof-
  have *: "reduced (add_set F s) = {s' \<in> reduced F. \<not> elements s \<subset> elements s'} \<union> {s}"
    using assms elements_subset_gt[of _ s]
    unfolding reduced_def add_set_def
    using less_not_sym[of s]
    by (auto simp add: Family_inverse)
  thus ?thesis
    using elements_inject
    by auto
qed

lemma Family_singleton:
  assumes "card (sets s) = 1" and "Set A \<in> sets s"
  shows "s = Family {Set A}"
  using assms
  by (metis card_1_singletonE sets_inverse singletonD)


lemma sorted_list_of_set_less:
  assumes "card F1 = card F2" "finite F1" "finite F2"
  shows "sorted_list_of_set F1 < sorted_list_of_set F2 \<longleftrightarrow> 
        (\<exists> a b F F1' F2'. F1 = F \<union> {a} \<union> F1' \<and> F2 = F \<union> {b} \<union> F2' \<and> a < b \<and> 
                          (\<forall> x \<in> F. x < a) \<and> (\<forall> x \<in> F1'. a < x) \<and> (\<forall> x \<in> F2'. b < x))"
proof-
  from assms have l: "length (sorted_list_of_set F1) = length (sorted_list_of_set F2)"
    by (metis distinct_card distinct_sorted_list_of_set set_sorted_list_of_set)
  show ?thesis (is "?lhs \<longleftrightarrow> ?rhs")
  proof
    assume "?lhs"
    then obtain u a b v1 v2 where
    "sorted_list_of_set F1 = u @ a # v1" "sorted_list_of_set F2 = u @ b # v2" "a < b"
      using l
      unfolding list_less_def lexord_def
      by auto
    hence 
      "set (sorted_list_of_set F1) = set (u @ a # v1) \<and> sorted (u @ a # v1) \<and> distinct (u @ a # v1)"
      "set (sorted_list_of_set F2) = set (u @ b # v2) \<and> sorted (u @ b # v2) \<and> distinct (u @ b # v2)"
      using assms
      by (metis distinct_sorted_list_of_set sorted_sorted_list_of_set)+
    hence "F1 = set u \<union> {a} \<union> set v1 \<and>
          F2 = set u \<union> {b} \<union> set v2 \<and> a < b \<and> (\<forall>x\<in>set u. x < a) \<and> (\<forall>x\<in>set v1. a < x) \<and> (\<forall>x\<in>set v2. b < x)"
      using assms `a < b`
      using less_le
      by (simp add: sorted_append) blast
    thus ?rhs
      by blast
  next
    let ?sls = "sorted_list_of_set"
    assume ?rhs
    then obtain a b F F1' F2' where
      *: "F1 = F \<union> {a} \<union> F1' \<and> F2 = F \<union> {b} \<union> F2' \<and> a < b \<and> (\<forall>x\<in>F. x < a) \<and> (\<forall>x\<in>F1'. a < x) \<and> (\<forall>x\<in>F2'. b < x)"
      by auto
    have "?sls F1 = ?sls F @ a # ?sls F1'"
    proof (rule sorted_distinct_set_unique)
      show "sorted (?sls F1)" "distinct (?sls F1)"
        using assms
        by auto
    next                                      
      show "sorted (?sls F @ a # ?sls F1')"
        using * assms
        by (force simp add: sorted_append)
    next
      show "distinct (?sls F @ a # ?sls F1')"
        using * assms
        by force
    next
      have "a \<notin> F1'" "a \<notin> F"
        using * assms
        by auto
      thus "set (?sls F1) = set (?sls F @ a # ?sls F1')"
        using * assms
        by (metis Un_insert_left Un_insert_right finite_Un list.simps(15) set_append set_sorted_list_of_set sup_bot.right_neutral)
    qed

    moreover

    have "?sls F2 = ?sls F @ b # ?sls F2'"
    proof (rule sorted_distinct_set_unique)
      show "sorted (?sls F2)" "distinct (?sls F2)"
        using assms
        by auto
    next                                      
      show "sorted (?sls F @ b # ?sls F2')"
        using * assms
        by (force simp add: sorted_append)
    next
      show "distinct (?sls F @ b # ?sls F2')"
        using * assms
        by force
    next
      have "a \<notin> F2'" "a \<notin> F"
        using * assms
        by auto
      thus "set (?sls F2) = set (?sls F @ b # ?sls F2')"
        using * assms
        by (metis Un_insert_left Un_insert_right finite_Un list.simps(15) set_append set_sorted_list_of_set sup_bot.right_neutral)
    qed

    ultimately

    show ?lhs
      using *
      unfolding list_less_def lexord_def
      by blast
  qed
qed

lemma eq_card_sets_less_min_card [simp]:
  assumes "finite (sets F)" "sets F \<noteq> {}" "c < min_card F" 
  shows "eq_card_sets F c = {}"
  using assms
  by (smt Collect_empty_eq eq_card_sets_def leD min_card_min)


lemma permute_family_eq_card_sets:
  assumes "finite (sets F)" "p \<in> set (permute [0..<n])" "\<forall> s \<in> sets F. elements s \<subseteq> {0..<n}"
  shows "card (eq_card_sets (permute_family p F) c) = card (eq_card_sets F c)"
proof-
  let ?A = "{s \<in> sets F. card (elements s) = c}"
  let ?B = "{s \<in> permute_set p ` sets F. card (elements s) = c}"

  have "?B = permute_set p ` ?A"
    using permute_set_card assms
    by auto

  moreover

  have "inj_on (permute_set p) ?A"
    using assms permute_set_inj[of _ n]
    by (smt inj_onI mem_Collect_eq)

  ultimately

  show ?thesis
    unfolding eq_card_sets_def
    by (auto simp add: permute_family_def card_image Family_inverse)
qed

lemma sorted_list_of_set_by_card:
  assumes "finite F1" "\<forall> s \<in> F1. elements s \<subseteq> {0..<n}"
          "finite F2" "\<forall> s \<in> F2. elements s \<subseteq> {0..<n}" 
          "card F1 = card F2" "\<forall> c \<le> n. card (eq_card_sets (Family F1) c) = card (eq_card_sets (Family F2) c)"        
  shows "sorted_list_of_set F1 < sorted_list_of_set F2 \<longleftrightarrow> 
    (\<exists> c. c \<le> n \<and> 
         (\<forall> c'. c' > c \<and> c' \<le> n \<longrightarrow> 
              sorted_list_of_set (eq_card_sets (Family F1) c') =
              sorted_list_of_set (eq_card_sets (Family F2) c')) \<and>
         (sorted_list_of_set (eq_card_sets (Family F1) c) < (sorted_list_of_set (eq_card_sets (Family F2) c))))"
  (is "?lhs \<longleftrightarrow> ?rhs")
proof
  let ?sls = "sorted_list_of_set"
  let ?E = "\<lambda> F c. eq_card_sets (Family F) c"

  assume ?lhs
  hence "(\<exists> a b F F1' F2'. F1 = F \<union> {a} \<union> F1' \<and> F2 = F \<union> {b} \<union> F2' \<and> a < b \<and> 
                           (\<forall> x \<in> F. x < a) \<and> (\<forall> x \<in> F1'. a < x) \<and> (\<forall> x \<in> F2'. b < x))" 
    using assms
    by (simp add: sorted_list_of_set_less)
  then obtain a b F F1' F2' where
    *: "F1 = F \<union> {a} \<union> F1'" "F2 = F \<union> {b} \<union> F2'" "a < b" "\<forall>x\<in>F. x < a" "\<forall>x\<in>F1'. a < x" "\<forall>x\<in>F2'. b < x"
    by auto

  let ?a = "set_card a"

  have "?a \<le> n"
    using assms `F1 = F \<union> {a} \<union> F1'`
    by (metis Un_insert_right Un_upper1 insert_subset set_card_def subset_eq_atLeast0_lessThan_card)

  have "set_card b = ?a"
  proof-
    have "set_card b \<le> ?a"
      using `a < b`
      using leq_Set_geq_card by auto
    moreover
    have "set_card b \<ge> ?a"
    proof (rule ccontr)
      assume "\<not> ?thesis"
      hence "set_card b < ?a"
        by simp
      have "?E F ?a \<union> {a} \<subseteq> ?E F1 ?a"
        using *
        using Family_inverse Un_insert_right Un_upper1 assms(1) eq_card_sets_def finite_Un insert_subset mem_Collect_eq subset_iff sup_bot.right_neutral by auto
      moreover
      have "a \<notin> ?E F ?a"
        using *
        using eq_card_sets_def
        using Family_inverse assms(3) by auto
      hence "card (?E F ?a \<union> {a}) = card (?E F ?a) + 1"
        using card_insert[of "?E F ?a" a] `finite F1` *
        by (simp add: eq_card_sets_def)
      ultimately
      have "card (?E F1 ?a) \<ge> card (?E F ?a) + 1"
        using card_mono[of "?E F1 ?a" "?E F ?a \<union> {a}"] `finite F1`
        by (simp add: eq_card_sets_def)

      moreover

      have "\<forall> x \<in> F2'. set_card x < ?a"
        using * `set_card b < ?a`
        by (meson \<open>\<not> set_card a \<le> set_card b\<close> leI le_trans leq_Set_geq_card less_eq_Set_def)
      hence "?E F2 ?a = ?E F ?a"
        using * `set_card b < ?a` `finite F1` `finite F2`
        unfolding eq_card_sets_def
        by (auto simp add: Family_inverse)
      hence "card (?E F2 ?a) = card (?E F ?a)"
        by auto

      ultimately

      show False
        using assms `?a \<le> n`
        by simp
    qed
    ultimately
    show ?thesis
      by simp
  qed

  have E1: "?E F1 ?a = ?E F ?a \<union> {a} \<union> ?E F1' ?a"
    using * `finite F1`
    by (auto simp add: eq_card_sets_def Family_inverse)


  have E2: "?E F2 ?a = ?E F ?a  \<union> {b} \<union> ?E F2' ?a"
    using * `set_card b = ?a` `finite F2`
    by (auto simp add: eq_card_sets_def Family_inverse)

  show "?rhs"
  proof (rule_tac x="set_card a" in exI, safe)
    show "set_card a \<le> n"
      by fact

    show "?sls (eq_card_sets (Family F1) (set_card a))
         < ?sls (eq_card_sets (Family F2) (set_card a))" (is "sorted_list_of_set ?A < sorted_list_of_set ?B")
    proof (subst sorted_list_of_set_less)
      show "\<exists> a b F F1' F2'. ?A = F \<union> {a} \<union> F1' \<and> ?B = F \<union> {b} \<union> F2' \<and> 
                             a < b \<and> (\<forall> x \<in> F. x < a) \<and> (\<forall> x \<in> F1'. x > a) \<and> (\<forall> x \<in> F2'. x > b)"
        using * E1 E2 `finite F1` `finite F2`
        by (rule_tac x=a in exI, rule_tac x=b in exI,
            rule_tac x="eq_card_sets (Family F) (set_card a)" in exI,
            rule_tac x="eq_card_sets (Family F1') (set_card a)" in exI,
            rule_tac x="eq_card_sets (Family F2') (set_card a)" in exI)
           (simp add: eq_card_sets_def Family_inverse)
    next
      show "card ?A = card ?B"
        using assms `set_card a \<le> n`
        by simp
      show "finite ?A" "finite ?B"
        using assms
        by (auto simp add: eq_card_sets_def)
    qed

    fix c'
    assume "set_card a < c'"
    
    have "\<forall> x \<in> F1'. set_card x < c'"
      using * \<open>set_card a < c'\<close>
      by (meson leD leI le_trans leq_Set_geq_card less_imp_le)

    hence "eq_card_sets (Family F1) c' = eq_card_sets (Family F) c'"
      using * \<open>set_card a < c'\<close> `finite F1`
      by (auto simp add: eq_card_sets_def Family_inverse)

    moreover

    have "\<forall> x \<in> F2'. set_card x < c'"
      using * \<open>set_card a < c'\<close>
      by (meson leD leI le_trans leq_Set_geq_card less_imp_le)

    hence "eq_card_sets (Family F2) c' = eq_card_sets (Family F) c'"
      using * \<open>set_card a < c'\<close> \<open>set_card b = set_card a\<close> `finite F2`
      by (auto simp add: eq_card_sets_def Family_inverse)

    ultimately

    have "eq_card_sets (Family F1) c' = eq_card_sets (Family F2) c'"
      by simp

    thus "sorted_list_of_set (eq_card_sets (Family F1) c') = sorted_list_of_set (eq_card_sets (Family F2) c')"
      by simp
  qed
next
  let ?sls = "sorted_list_of_set"
  let ?E = "\<lambda> F c. eq_card_sets (Family F) c"
  let ?E1 = "?E F1"
  let ?E2 = "?E F2"

  assume ?rhs
  then obtain c where
  *: "c \<le> n" "\<forall>c'. c' > c \<and> c' \<le> n \<longrightarrow> ?sls (?E1 c') = ?sls (?E2 c')" "?sls (?E1 c) < ?sls (?E2 c)"
    by auto
  then obtain a b F F1' F2' where
    **: "?E1 c = F \<union> {a} \<union> F1'" "?E2 c = F \<union> {b} \<union> F2'" "a < b" "\<forall> x \<in> F. x < a" "\<forall> x \<in> F1'. x > a" "\<forall> x \<in> F2'. x > b"
    using sorted_list_of_set_less[of "?E1 c" "?E2 c"]
    using assms
    by (auto simp add: eq_card_sets_def)

  let ?a = "set_card a"

  have "?a \<le> n"
    using ** assms
    using "*"(1) Un_insert_right Un_upper1 eq_card_sets_def insert_subset mem_Collect_eq
    by fastforce

  let ?FF = "{s. s \<in> F1 \<and> set_card s > c}"
  let ?FF1 = "{s. s \<in> F1 \<and> set_card s < c}"
  let ?FF2 = "{s. s \<in> F2 \<and> set_card s < c}"

  have "set_card b = ?a"
  proof-
    have "set_card b \<le> ?a"
      using `a < b`
      using leq_Set_geq_card by auto
    moreover
    have "set_card b \<ge> ?a"
    proof (rule ccontr)
      assume "\<not> ?thesis"
      hence "set_card b < ?a"
        by simp
      have "?E F ?a \<union> {a} \<subseteq> ?E F1 ?a"
        using ** 
        by (metis (mono_tags, lifting) Un_insert_right \<open>set_card b < set_card a\<close> eq_card_sets_def insert_subset mem_Collect_eq sup.strict_order_iff sup_ge1)
      moreover
      have "a \<notin> ?E F ?a" 
        using **
        using eq_card_sets_def 
        by (metis (mono_tags, lifting) Un_insert_right \<open>set_card b < set_card a\<close> insert_subset mem_Collect_eq nat_less_le sup_ge1)
      have "finite (eq_card_sets (Family F) (set_card a))"
        using ** `finite F1`
        using eq_card_sets_def 
        by (metis (mono_tags, lifting) Un_insert_right Un_upper1 \<open>set_card b < set_card a\<close> insert_subset mem_Collect_eq nat_less_le)
      hence "card (?E F ?a \<union> {a}) = card (?E F ?a) + 1"
        using card_insert[of "?E F ?a" a] `finite F1` ** `a \<notin> ?E F ?a`
        by simp
      ultimately
      have "card (?E F1 ?a) \<ge> card (?E F ?a) + 1"
        using card_mono[of "?E F1 ?a" "?E F ?a \<union> {a}"] `finite F1`
        by (simp add: eq_card_sets_def)

      moreover

      have "\<forall> x \<in> F2'. set_card x < ?a"
        using ** `set_card b < ?a`
        by (metis leD le_trans leq_Set_geq_card less_eq_Set_def nat_less_le)
      hence "?E F2 ?a = ?E F ?a"
        using ** `set_card b < ?a`
        unfolding eq_card_sets_def
        by (metis (mono_tags, lifting) Un_insert_right Un_upper1 \<open>\<not> set_card a \<le> set_card b\<close> \<open>set_card b \<le> set_card a\<close> insert_subset mem_Collect_eq)
      hence "card (?E F2 ?a) = card (?E F ?a)"
        by auto

      ultimately

      show False
        using assms `?a \<le> n`
        by simp
    qed
    ultimately
    show ?thesis
      by simp
  qed


  have "F1 = ?FF \<union> (?E1 c) \<union> ?FF1"
    using `finite F1`
    unfolding eq_card_sets_def
    by (auto simp add: Family_inverse)
  hence "F1 = (?FF \<union> F) \<union> {a} \<union> (F1' \<union> ?FF1)"
    using **(1)
    by blast

  moreover

  have FF2: "?FF = {s. s \<in> F2 \<and> set_card s > c}"
  proof safe
    fix x
    assume "x \<in> F1" "c < set_card x"
    have "x \<in> set (?sls (?E1 (set_card x)))"
      using `x \<in> F1` `finite F1`
      by (simp add: eq_card_sets_def Family_inverse)   
    hence "x \<in> set (?sls (?E2 (set_card x)))"
      using *(2)[rule_format, of "set_card x"] `c < set_card x`
      using assms
      by (simp add: \<open>x \<in> F1\<close> subset_eq_atLeast0_lessThan_card)
      
    thus "x \<in> F2"
      using `x \<in> F1` `finite F2`
      by (simp add: eq_card_sets_def Family_inverse)   
  next
    fix x
    assume "x \<in> F2" "c < set_card x"
    have "x \<in> set (?sls (?E2 (set_card x)))"
      using `x \<in> F2` `finite F2`
      by (simp add: eq_card_sets_def Family_inverse)   
    hence "x \<in> set (?sls (?E1 (set_card x)))"
      using *(2)[rule_format, of "set_card x"] `c < set_card x`
      using assms
      by (simp add: \<open>x \<in> F2\<close> subset_eq_atLeast0_lessThan_card)
    thus "x \<in> F1"
      using `x \<in> F2` `finite F1`
      by (simp add: eq_card_sets_def Family_inverse)   
  qed

  have "F2 = ?FF \<union> (?E2 c) \<union> ?FF2"
    using `finite F2`
    unfolding eq_card_sets_def
    by (subst FF2, auto simp add: Family_inverse)

  hence "F2 = (?FF \<union> F) \<union> {b} \<union> (F2' \<union> ?FF2)"
    using **(2)
    by auto

  moreover
  have "\<forall> x \<in> ?FF. x < a"
    by (metis (mono_tags, lifting) "**"(1) Un_insert_right eq_card_sets_def insert_subset leD leI leq_Set_geq_card mem_Collect_eq sup_ge1)
  hence "\<forall> x \<in> ?FF \<union> F. x < a"
    using **
    by auto

  moreover
  have "\<forall> x \<in> ?FF1. x > a"
    by (metis (mono_tags, lifting) "**"(1) Un_iff eq_card_sets_def insert_iff leD leI leq_Set_geq_card mem_Collect_eq)
  hence "\<forall> x \<in>  F1' \<union> ?FF1. x > a"
    using **
    by auto

  moreover
  have "\<forall> x \<in> ?FF2. x > b"
    by (metis (mono_tags, lifting) Un_insert_right \<open>\<forall>x\<in>{s \<in> F1. c < set_card s}. x < a\<close> \<open>\<forall>x\<in>{s \<in> F1. set_card s < c}. a < x\<close> \<open>set_card b = set_card a\<close> calculation(1) insert_subset less_Set_def mem_Collect_eq neqE order_less_irrefl sup_ge1)
  hence "\<forall> x \<in> F2' \<union> ?FF2. x > b"
    using **
    by auto

  ultimately

  show ?lhs
    using assms `a < b`
    by (subst sorted_list_of_set_less, simp, simp, simp, blast)
qed


end