subsection \<open>Implementation\<close>

theory FamilyImpl
  imports Main "HOL-Library.Mapping"
          More_List Combinatorics FamilyAbs
begin


class SetCard = linorder +
  fixes card :: "'a \<Rightarrow> nat"

datatype 's SetOrder = SetOrder (Set : 's)

instantiation SetOrder :: (SetCard) linorder
begin
fun less_SetOrder :: "('a::SetCard) SetOrder \<Rightarrow> 'a SetOrder \<Rightarrow> bool" where 
  "less_SetOrder (SetOrder s1) (SetOrder s2) \<longleftrightarrow>
    (let n1 = card s1; n2 = card s2
      in n1 > n2 \<or> (n1 = n2 \<and> s1 > s2))"
definition less_eq_SetOrder :: "('a::SetCard) SetOrder \<Rightarrow> 'a SetOrder \<Rightarrow> bool" where
  "less_eq_SetOrder s1 s2 \<longleftrightarrow>
     s1 = s2 \<or> s1 < s2"
instance
proof
  fix x y :: "'a::SetCard SetOrder"
  show "x < y \<longleftrightarrow> x \<le> y \<and> \<not> y \<le> x"
    unfolding less_eq_SetOrder_def
    by (cases x, cases y) (auto simp add: Let_def)
next
  fix x :: "'a::SetCard SetOrder"
  show "x \<le> x"
    unfolding less_eq_SetOrder_def
    by simp
next
  fix x y z :: "'a::SetCard SetOrder"
  assume "x \<le> y" "y \<le> z"
  thus "x \<le> z"
    unfolding less_eq_SetOrder_def
    by (cases x, cases y, cases z) (auto simp add: Let_def)
next
  fix x y :: "'a::SetCard SetOrder"
  assume "x \<le> y" "y \<le> x" 
  thus "x = y"
    unfolding less_eq_SetOrder_def
    by (cases x, cases y) (auto simp add: Let_def)
next
  fix x y :: "'a::SetCard SetOrder"
  show "x \<le> y \<or> y \<le> x"
    unfolding less_eq_SetOrder_def
    by (cases x, cases y) (auto simp add: Let_def)
qed
end

locale SetImpl =
  fixes empty_set :: "'s::SetCard"
  fixes inv :: "nat \<Rightarrow> 's \<Rightarrow> bool"
  fixes elements :: "'s \<Rightarrow> nat list"
  fixes set_of :: "nat list \<Rightarrow> 's"
  fixes contains_element :: "'s \<Rightarrow> nat \<Rightarrow> bool"
  fixes is_subset :: "'s \<Rightarrow> 's \<Rightarrow> bool"
  fixes un :: "'s \<Rightarrow> 's \<Rightarrow> 's"
  fixes powerset :: "nat \<Rightarrow> 's list"
  fixes permute_set :: "nat \<Rightarrow> perm \<Rightarrow> 's \<Rightarrow> 's"
  fixes n_max :: "nat"
  assumes empty_set_elements [simp]: "elements empty_set = []"
  assumes empty_set_inv [simp]: "\<And> n. n \<le> n_max \<Longrightarrow> inv n empty_set"
  assumes set_elements: "\<And> n s. \<lbrakk>n \<le> n_max; inv n s\<rbrakk> \<Longrightarrow> set (elements s) \<subseteq> {0..<n}"
  assumes sorted_elements [simp]: "\<And> n s. \<lbrakk>n \<le> n_max; inv n s\<rbrakk> \<Longrightarrow> sorted (elements s)"
  assumes distinct_elements [simp]: "\<And> n s. \<lbrakk>n \<le> n_max; inv n s\<rbrakk> \<Longrightarrow> distinct (elements s)"
  assumes elements_inj: "\<And> n s1 s2. \<lbrakk>n \<le> n_max; inv n s1; inv n s2; elements s1 = elements s2\<rbrakk> \<Longrightarrow> s1 = s2"
  assumes card_length_elements: "\<And> n s. \<lbrakk>n \<le> n_max; inv n s\<rbrakk> \<Longrightarrow> card s = length (elements s)"
  assumes less_elements: "\<And> n s1 s2. \<lbrakk>n \<le> n_max; inv n s1; inv n s2; card s1 = card s2\<rbrakk> \<Longrightarrow> s1 < s2 \<longleftrightarrow> rev (elements s1) < rev (elements s2)"
  assumes set_of_elements [simp]: "\<And> n xs. \<lbrakk>n \<le> n_max; sorted xs; distinct xs; set xs \<subseteq> {0..<n} \<rbrakk> \<Longrightarrow> elements (set_of xs) = xs"
  assumes set_of_inv [simp]: "\<And> n xs. \<lbrakk>n \<le> n_max; sorted xs; distinct xs; set xs \<subseteq> {0..<n} \<rbrakk> \<Longrightarrow> inv n (set_of xs)"
  assumes un_inv [simp]: "\<And> n s1 s2. \<lbrakk>n \<le> n_max; inv n s1; inv n s2\<rbrakk> \<Longrightarrow> inv n (un s1 s2)"
  assumes un_elements [simp]: "\<And> n s1 s2. \<lbrakk>n \<le> n_max; inv n s1; inv n s2\<rbrakk> \<Longrightarrow> set (elements (un s1 s2)) = set (elements s1) \<union> set (elements s2)"
  assumes powerset_set: "\<And> n. n \<le> n_max \<Longrightarrow> set (powerset n) = {s. inv n s}"
  assumes powerset_distinct [simp]: "\<And> n. n \<le> n_max \<Longrightarrow> distinct (powerset n)"
  assumes permute_set_inv [simp]: "\<And> n s p. \<lbrakk>n \<le> n_max; inv n s; p \<in> set (permute [0..<n])\<rbrakk> \<Longrightarrow> inv n (permute_set n p s)"
  assumes permute_set_elements [simp]: "\<And> n s p. \<lbrakk>n \<le> n_max; inv n s; p \<in> set (permute [0..<n])\<rbrakk> \<Longrightarrow> set (elements (permute_set n p s)) = (list_to_fun p) ` set (elements s)"
  assumes is_subset: "\<And> n s s'. \<lbrakk>n \<le> n_max; inv n s; inv n s'\<rbrakk> \<Longrightarrow> is_subset s s' \<longleftrightarrow> set (elements s) \<subseteq> set (elements s')"
begin

(* abs_set *)
definition abs_set :: "'s \<Rightarrow> FamilyAbs.Set" where
  "abs_set s = FamilyAbs.Set (set (elements s))"

lemma abs_set_inj:
  assumes "n \<le> n_max" 
  assumes "abs_set s1 = abs_set s2"
  assumes "inv n s1" "inv n s2"
  shows "s1 = s2"
proof-
  have "set (elements s1) = set (elements s2)"
    using assms
    unfolding abs_set_def
    by (metis Set_inverse mem_Collect_eq set_elements subset_eq_atLeast0_lessThan_finite)
  hence "elements s1 = elements s2"
    using sorted_distinct_set_unique[of "elements s1" "elements s2"]
    using assms
    by auto
  thus ?thesis
    using assms elements_inj
    by blast
qed

(* card *)
lemma card_set_elements:
  assumes "n \<le> n_max" "inv n s"
  shows "card s = Finite_Set.card (set (elements s))"
  using card_length_elements[OF assms]
  using assms distinct_card distinct_elements
  by fastforce
  
lemma card_abs:
  assumes "n \<le> n_max" "inv n s"
  shows  "card s = FamilyAbs.set_card (abs_set s)"
  using assms
  using Set_inverse abs_set_def card_set_elements by auto

(* powerset *)
lemma powerset_not_empty:
  assumes "n \<le> n_max"
  shows "powerset n \<noteq> []"
proof-
  have "empty_set \<in> set (powerset n)"
    using assms powerset_set[of n]
    by simp
  thus ?thesis
    by auto
qed

lemma powerset_elements_card:
  assumes "n \<le> n_max"
  shows "\<forall> s \<in> set (powerset n). card s \<le> n"
  using assms powerset_set[of n]
  by (metis card_set_elements mem_Collect_eq set_elements subset_eq_atLeast0_lessThan_card)

(* less_set *)
lemma less_set_same_card_abs:
  fixes s1 s2 :: 's
  assumes "n \<le> n_max" "inv n s1" "inv n s2"
  assumes "card s1 = card s2"
  shows "abs_set s1 < abs_set s2 \<longleftrightarrow> s1 > s2"
proof-
  have "rev (sorted_list_of_set (set (elements s1))) = rev (elements s1)"
       "rev (sorted_list_of_set (set (elements s2))) = rev (elements s2)"
    using assms
    by (metis List.finite_set distinct_elements distinct_sorted_list_of_set set_sorted_list_of_set sorted_distinct_set_unique sorted_elements sorted_sorted_list_of_set)+
  thus ?thesis
    using assms
    unfolding abs_set_def
    by (simp add: less_elements less_Set_def Set_inverse card_set_elements Let_def)
qed

lemma less_set_abs:
  fixes s1 s2 :: 's
  assumes "n \<le> n_max" "inv n s1" "inv n s2"
  shows "SetOrder s1 < SetOrder s2 \<longleftrightarrow> abs_set s1 < abs_set s2"
proof-
  have "rev (sorted_list_of_set (set (elements s1))) = rev (elements s1)"
       "rev (sorted_list_of_set (set (elements s2))) = rev (elements s2)"
    using assms
    by (metis List.finite_set distinct_elements distinct_sorted_list_of_set set_sorted_list_of_set sorted_distinct_set_unique sorted_elements sorted_sorted_list_of_set)+
  thus ?thesis
    using assms
    unfolding abs_set_def
    by (auto simp add: Let_def card_set_elements less_elements less_Set_def Set_inverse)
qed

(* permute_set *)
lemma permute_set_abs [simp]:
  assumes "n \<le> n_max" "inv n s" "p \<in> set (permute [0..<n])"
  shows "abs_set (permute_set n p s) = FamilyAbs.permute_set p (abs_set s)"
  using assms
  unfolding abs_set_def FamilyAbs.permute_set_def
  by (simp add: Set_inverse)

end

type_synonym PermIndices = "nat list"
datatype ('f, 's) FamilyCache = 
   FamilyCache 
       (all_sets : 'f) 
       (reduced : 'f) 
       (last_set : 's)
       (perms : PermIndices)
type_synonym 'f SetsGroupedByCard = "(nat, 'f) mapping"
type_synonym 's SetPerms = "((nat \<times> 's), 's) mapping"
type_synonym 's Combinations = "(nat, 's list) mapping"
type_synonym 's AugmentingSets = "('s, 's list) mapping"


locale FamilyImpl = SetImpl empty_set for empty_set :: "'s::SetCard" + 
  fixes empty_family :: "'f :: linorder"
  fixes inv_f :: "nat \<Rightarrow> 'f \<Rightarrow> bool"
  fixes sets :: "'f \<Rightarrow> 's list"
  fixes contains :: "'f \<Rightarrow> 's \<Rightarrow> bool"
  fixes add :: "'f \<Rightarrow> 's \<Rightarrow> 'f"
  fixes remove :: "'f \<Rightarrow> 's \<Rightarrow> 'f"
  fixes inter :: "'f \<Rightarrow> 'f \<Rightarrow> 'f"

  assumes inv_f_inv [simp]: "\<And> n F s. \<lbrakk>n \<le> n_max; inv_f n F; s \<in> set (sets F)\<rbrakk> \<Longrightarrow> inv n s"
  assumes sorted_sets [simp]: "\<And> n F. \<lbrakk>n \<le> n_max; inv_f n F\<rbrakk> \<Longrightarrow> sorted (sets F)"
  assumes distinct_sets [simp]: "\<And> n F. \<lbrakk>n \<le> n_max; inv_f n F\<rbrakk> \<Longrightarrow> distinct (sets F)"
  assumes inj_sets: "\<And> n F1 F2. \<lbrakk>n \<le> n_max; inv_f n F1; inv_f n F2; sets F1 = sets F2\<rbrakk> \<Longrightarrow> F1 = F2"
  assumes empty_family_sets [simp]: "sets empty_family = []"
  assumes empty_family_inv [simp]: "inv_f n empty_family"
  assumes contains_sets: "\<And> F s. contains F s \<longleftrightarrow> s \<in> set (sets F)"
  assumes add_sets [simp]: "\<And> n F s. \<lbrakk>n \<le> n_max; inv_f n F; inv n s\<rbrakk> \<Longrightarrow> set (sets (add F s)) = set (sets F) \<union> {s}"
  assumes add_inv [simp]: "\<And> n F s. \<lbrakk>n \<le> n_max; inv_f n F; inv n s\<rbrakk> \<Longrightarrow> inv_f n (add F s)"
  assumes remove_sets[simp]: "\<And> F a. \<lbrakk>n \<le> n_max; inv_f n F; inv n a\<rbrakk> \<Longrightarrow> set (sets (remove F a)) = set (sets F) - {a}"
  assumes remove_inv[simp]: "\<And> F a. \<lbrakk>n \<le> n_max; inv_f n F; inv n a\<rbrakk> \<Longrightarrow> inv_f n (remove F a)"
  assumes inter_sets [simp]: "\<And> F1 F2. set (sets (inter F1 F2)) = set (sets F1) \<inter> set (sets F2)"
  assumes inter_inv [simp]: "\<And> F1 F2. \<lbrakk>n \<le> n_max; inv_f n F1; inv_f n F2\<rbrakk> \<Longrightarrow> inv_f n (inter F1 F2)"
  assumes less_family_sets: "\<And> n F1 F2. \<lbrakk>n \<le> n_max; inv_f n F1; inv_f n F2; length (sets F1) = length (sets F2)\<rbrakk> \<Longrightarrow> F1 < F2 \<longleftrightarrow> rev (sets F1) < rev (sets F2)"
begin

definition contains_set :: "('f, 's) FamilyCache \<Rightarrow> 's \<Rightarrow> bool" where
  "contains_set F s = contains (all_sets F) s"

definition update_reduced :: "'f \<Rightarrow> 's \<Rightarrow> 'f" where
  "update_reduced Rs s =
     (let Rs' = foldl (\<lambda> Rs s'. if is_subset s s' then remove Rs s' else Rs) Rs (sets Rs)
      in add Rs' s)"

definition add_set :: "('f, 's) FamilyCache \<Rightarrow> 's \<Rightarrow> ('f, 's) FamilyCache" where
  "add_set F s = FamilyCache (add (all_sets F) s) (update_reduced (reduced F) s) s (perms F)"

definition add_set_by_card :: "'f list \<Rightarrow> 's \<Rightarrow> 'f list" where
  "add_set_by_card Fs s = (let c = card s in list_update Fs c (add (Fs ! c) s))"

definition group_powerset_by_card :: "nat \<Rightarrow> 'f SetsGroupedByCard" where
  "group_powerset_by_card n = 
     (let ss = powerset n;
          Fs = replicate (n+1) empty_family
       in Mapping.bulkload (foldl add_set_by_card Fs ss))"

definition sets_by_card :: "'f SetsGroupedByCard \<Rightarrow> 'f \<Rightarrow> nat \<Rightarrow> 'f" where
  "sets_by_card powerset_by_card F c = 
     inter (the (Mapping.lookup powerset_by_card c)) F"

definition group_sets_by_card :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 'f \<Rightarrow> 'f list" where
  "group_sets_by_card n powerset_by_card F = 
     map (\<lambda> k. sets_by_card powerset_by_card F k) (rev [0..<n+1])"

definition group_sets_by_card' :: "nat \<Rightarrow> 'f list \<Rightarrow> 'f \<Rightarrow> 'f list" where
  "group_sets_by_card' n powerset_by_card F = 
     map (inter F) powerset_by_card" 
  
definition less_family :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> bool" where
  "less_family n powerset_by_card F1 F2 \<longleftrightarrow> 
     group_sets_by_card n powerset_by_card (all_sets F2) < 
     group_sets_by_card n powerset_by_card (all_sets F1)"

definition eq_family :: "('f, 's) FamilyCache \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> bool" where
  "eq_family F1 F2 \<longleftrightarrow> (all_sets F1) = (all_sets F2)"

definition less_eq_family :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> bool" where
  "less_eq_family n powerset_by_card F1 F2 \<longleftrightarrow> 
     eq_family F1 F2 \<or> less_family n powerset_by_card F1 F2"

definition init_set_perms :: "nat \<Rightarrow> 's SetPerms" where
  "init_set_perms n =
     (let ps = permute [0..<n];
          ss = powerset n;
          keys = concat (map (\<lambda> p. map (\<lambda> s. (p, s)) ss) [0..<length ps])
       in Mapping.tabulate keys (\<lambda> (p, s). permute_set n (ps ! p) s))"

definition permute_family :: "'s SetPerms \<Rightarrow> nat \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> ('f, 's) FamilyCache" where
  "permute_family set_perms p F =
    (let F' = foldl (\<lambda> F' s. add F' (the (Mapping.lookup set_perms (p, s)))) empty_family (sets (all_sets F))
      in FamilyCache F' empty_family empty_set [])"

definition is_canon :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> bool" where
  "is_canon n powerset_by_card set_perms F = 
     list_all (\<lambda> p. less_eq_family n powerset_by_card F (permute_family set_perms p F)) (tl (perms F))"

definition is_union_closed' :: "('f, 's) FamilyCache \<Rightarrow> 's \<Rightarrow> bool" where
  "is_union_closed' F s =
     list_all (\<lambda> s'. contains_set F (un s' s)) (sets (reduced F))"

definition k_sets :: "nat \<Rightarrow> nat \<Rightarrow> 's list" where
  "k_sets n k = map set_of (Combinatorics.combine [0..<n] k)"

definition init_combinations :: "nat \<Rightarrow> 's Combinations" where
  "init_combinations n = 
     (let ck = map (\<lambda> k. sort_key SetOrder (k_sets n k)) [0..<n+1]
       in Mapping.bulkload ck)"

definition augment_set :: "'s Combinations \<Rightarrow> nat \<Rightarrow> 's \<Rightarrow> 's list" where
  "augment_set combinations n s =
     (let c = card s
       in filter (\<lambda> s'. SetOrder s < SetOrder s') (the (Mapping.lookup combinations c)) @
          concat (map (\<lambda> k. the (Mapping.lookup combinations k)) (rev [1..<c])))"

definition init_augmenting_sets :: "nat \<Rightarrow> 's AugmentingSets" where
  "init_augmenting_sets n =
     (let combinations = init_combinations n
       in Mapping.tabulate (powerset n) (\<lambda> s. augment_set combinations n s))"

definition start_family :: "nat \<Rightarrow> ('f, 's) FamilyCache" where
  "start_family n =
     (let start = FamilyCache empty_family empty_family empty_set [0..<fact n]
       in add_set start (set_of [0..<n]))"

definition perm_fixes_subset :: "'s SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> 'f \<Rightarrow> nat \<Rightarrow> bool" where
  "perm_fixes_subset set_perms F F' p \<longleftrightarrow> 
     list_all (\<lambda> s. contains_set F (the (Mapping.lookup set_perms (p, s)))) (sets F')"

definition perm_fixes_card :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> bool" where
  "perm_fixes_card n powerset_by_card set_perms F c p =
     (let Sc = sets_by_card powerset_by_card (all_sets F) c 
       in perm_fixes_subset set_perms F Sc p)"

definition filter_perms :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> nat \<Rightarrow> ('f, 's) FamilyCache" where
  "filter_perms n powerset_by_card set_perms F c = 
     (let perms' = filter (perm_fixes_card n powerset_by_card set_perms F c) (perms F) 
       in FamilyCache (all_sets F) (reduced F) (last_set F) perms')"

definition extend_family :: "nat \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> 's \<Rightarrow> ('f, 's) FamilyCache" where
  "extend_family n powerset_by_card set_perms F s = 
     (let F' = add_set F s; cs = card s; cF = card (last_set F)
       in if cs \<noteq> cF then
            filter_perms n powerset_by_card set_perms F' cF
          else
            F')"

definition augment :: "nat \<Rightarrow> 's AugmentingSets \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache \<Rightarrow> ('f, 's) FamilyCache list" where 
  "augment n augmenting_sets powerset_by_card set_perms F =
     map (extend_family n powerset_by_card set_perms F) 
         (filter (is_union_closed' F) (the (Mapping.lookup augmenting_sets (last_set F))))"

definition step_fam :: "nat \<Rightarrow> 's AugmentingSets \<Rightarrow> 'f SetsGroupedByCard \<Rightarrow> 's SetPerms \<Rightarrow> ('f, 's) FamilyCache list \<Rightarrow> ('f, 's) FamilyCache list" where
  "step_fam n augmenting_sets powerset_by_card set_perms Fs =
    (let unionClosed = concat (map (augment n augmenting_sets powerset_by_card set_perms) Fs)
      in filter (is_canon n powerset_by_card set_perms) unionClosed)"

(* abs_fam *)
definition abs_fam :: "'f \<Rightarrow> FamilyAbs.Family" where
  "abs_fam F = FamilyAbs.Family (abs_set ` (set (sets F)))"

lemma abs_fam_inj:
  assumes "n \<le> n_max" "inv_f n F1" "inv_f n F2"
  assumes "abs_fam F1 = abs_fam F2"
  shows "F1 = F2"
proof-
  have "set (sets F1) = set (sets F2)"
    using assms
    using inj_on_Un_image_eq_iff[of abs_set "set (sets F1)" "set (sets F2)"]
    using inv_f_inv
    using abs_set_inj
    unfolding abs_fam_def
    by (smt Family_inverse List.finite_set Un_iff finite_imageI inj_on_def mem_Collect_eq)

  hence "sets F1 = sets F2"
    using assms 
    using sorted_distinct_set_unique[of "sets F1" "sets F2"]
    by auto

  thus ?thesis
    using assms inj_sets
    by auto
qed

lemma abs_fam_finite [simp]:
  shows "finite (Family.sets (abs_fam (all_sets F)))"
  by (simp add: abs_fam_def)

lemma abs_fam_not_empty [simp]:
  assumes "set (sets F) \<noteq> {}"
  shows "Family.sets (abs_fam F) \<noteq> {}"
  using assms
  by (simp add: abs_fam_def Family_inverse)

lemma abs_fam_member_not_empty [simp]:
  assumes "x \<in> set (sets F)"
  shows "Family.sets (abs_fam F) \<noteq> {}"
  using assms
  by (metis abs_fam_not_empty empty_iff)

lemma finite_elements_abs_set [simp]: 
  shows "finite (FamilyAbs.elements (abs_set s))"
  by (simp add: abs_set_def)

lemma finite_all_sets [simp]:
  shows "\<forall>s'\<in>FamilyAbs.sets (abs_fam (all_sets F)). finite (FamilyAbs.elements s')"
  by (simp add: abs_fam_def)

(* empty_family *)
lemma empty_family_abs [simp]:
  shows "abs_fam empty_family = Family {}"
  unfolding abs_fam_def
  by simp

(* contains *)  
lemma contains_set_sets:
  shows "contains_set F s \<longleftrightarrow> s \<in> set (sets (all_sets F))"
  unfolding contains_set_def
  by (simp add: contains_sets)

lemma contains_abs [simp]:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n s"
  shows "contains_set F s \<longleftrightarrow> abs_set s \<in> FamilyAbs.sets (abs_fam (all_sets F))"
  using assms contains_set_sets[of F s] abs_set_inj[of n s]
  by (auto simp add: abs_fam_def Family_inverse)


(* add_set *)

lemma add_abs [simp]: 
  assumes "n \<le> n_max" "inv_f n F" "inv n s"
  shows "abs_fam (add F s) = FamilyAbs.add_set (abs_fam F) (abs_set s)"
  using assms
  unfolding abs_fam_def FamilyAbs.add_set_def
  by (subst add_sets, simp_all add: Family_inverse)


(* update_reduced *)
lemma foldl_remove_inv:
  assumes "n \<le> n_max" "inv_f n F" "\<forall> x \<in> set xs. inv n x"
  shows "inv_f n (foldl (\<lambda>Rs s'. if is_subset s s' then remove Rs s' else Rs) F xs)"
  using assms
  by (induction xs rule: rev_induct) auto

lemma set_elements_inj:
  assumes "n \<le> n_max" "inv n s1" "inv n s2"
  assumes "set (elements s1) = set (elements s2)"
  shows "s1 = s2"
  using assms
proof-
  have "elements s1 = elements s2"
    using sorted_distinct_set_unique[of "elements s1" "elements s2"]
    using assms
    by simp
  thus ?thesis
    using assms elements_inj
    by blast
qed

lemma is_subset_abs:
  assumes "n \<le> n_max" "inv n s1" "inv n s2"
  shows "is_subset s1 s2 \<longleftrightarrow> FamilyAbs.elements (abs_set s1) \<subseteq> FamilyAbs.elements (abs_set s2)"
  using is_subset[OF assms]
  by (simp add: abs_set_def Set_inverse)

lemma remove_abs[simp]:
  assumes "n \<le> n_max" "inv_f n F" "inv n s"
  shows "FamilyAbs.sets (abs_fam (remove F s)) = FamilyAbs.sets (abs_fam F) - {abs_set s}"
  using assms set_elements_inj inv_f_inv[OF `n \<le> n_max`]
  by (auto simp add: abs_fam_def abs_set_def Family_inverse abs_set_inj)

lemma foldl_remove_abs':
  assumes "n \<le> n_max" "inv_f n F" "\<forall> x \<in> set xs. inv n x" "inv n s" "abs_set ` (set xs) \<subseteq> FamilyAbs.sets (abs_fam F)"
  shows "(FamilyAbs.sets (abs_fam (foldl (\<lambda>Rs s'. if is_subset s s' then remove Rs s' else Rs) F xs))) =
         {s'. s' \<in> FamilyAbs.sets (abs_fam F) - abs_set ` set xs} \<union> 
         {s' \<in> abs_set ` set xs. \<not> FamilyAbs.elements (abs_set s) \<subseteq> FamilyAbs.elements s'}"
  using assms
proof (induction xs rule: rev_induct)
  case Nil
  then show ?case
    by simp
next
  case (snoc a xs)
  then show ?case 
    using foldl_remove_inv[OF `n \<le> n_max`, of F xs s]
    by (auto simp add: is_subset_abs[of n s a])
qed

lemma foldl_remove_abs [simp]:
  assumes "n \<le> n_max" "inv_f n F" "inv n s"
  shows "(FamilyAbs.sets (abs_fam (foldl (\<lambda>Rs s'. if is_subset s s' then remove Rs s' else Rs) F (sets F)))) =
         {s' \<in> FamilyAbs.sets (abs_fam F). \<not> FamilyAbs.elements (abs_set s) \<subseteq> FamilyAbs.elements s'}"
  using foldl_remove_abs'[OF assms(1-2), of "sets F" s] inv_f_inv assms
  by (simp add: abs_fam_def Family_inverse)

lemma update_reduced_abs [simp]:
  assumes "n \<le> n_max" "inv_f n F" "inv n s"
  shows "FamilyAbs.sets (abs_fam (update_reduced F s)) = 
         {s' \<in> FamilyAbs.sets (abs_fam F). \<not> FamilyAbs.elements (abs_set s) \<subseteq> FamilyAbs.elements s'} \<union> {abs_set s}"
  using assms inv_f_inv[of n F]
  unfolding update_reduced_def Let_def
  by (simp add: foldl_remove_inv Family_inverse)

lemma update_reduced_inv_r [simp]:
  assumes "n \<le> n_max" "inv_f n (reduced F)" "inv n s"
  shows "inv_f n (update_reduced (reduced F) s)"
  unfolding update_reduced_def Let_def
  using assms
  by (simp add: foldl_remove_inv)

(* add_set *)
lemma add_set_sets [simp]:
  assumes "n \<le> n_max" "inv n s" "inv_f n (all_sets F)"
  shows "set (sets (all_sets (add_set F s))) = set (sets (all_sets F)) \<union> {s}"
  using assms
  by (simp add: add_set_def)

lemma add_set_abs [simp]:
  assumes "n \<le> n_max" "inv n s" "inv_f n (all_sets F)"
  shows "abs_fam (all_sets (add_set F s)) = FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s)"
  using assms
  by (simp add: abs_fam_def FamilyAbs.add_set_def Family_inverse)

lemma add_set_last_set [simp]:
  shows "last_set (add_set F s) = s"
  by (simp add: add_set_def)

lemma add_set_perms [simp]:
  shows "perms (add_set F s) = perms F"
  by (simp add: add_set_def)

lemma add_set_inv [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)" "inv n s"
  shows "inv_f n (all_sets (add_set F s))"
  using assms
  using FamilyCache.sel(1) FamilyImpl.add_set_def FamilyImpl_axioms 
  by fastforce

lemma add_set_reduced [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)" "inv n s"
  assumes "inv_f n (reduced F)"
  assumes "\<forall> s' \<in> FamilyAbs.sets (abs_fam (all_sets F)). s' < abs_set s"
  assumes "FamilyAbs.sets (abs_fam (reduced F)) = FamilyAbs.reduced (abs_fam (all_sets F))"
  shows "FamilyAbs.sets (abs_fam (reduced (add_set F s))) = FamilyAbs.reduced (abs_fam (all_sets (add_set F s)))"
  using assms
  by (simp add: add_set_def reduced_add_set)
  
lemma add_set_inv_r [simp]:
  assumes "n \<le> n_max" "inv_f n (reduced F)" "inv n s"
  shows "inv_f n (reduced (add_set F s))"
  using assms
  unfolding add_set_def
  by simp

(* foldl add *)
lemma foldl_add_inv_f:
  assumes "n \<le> n_max" "\<forall> s \<in> set F. inv n s" "\<forall> s. inv n s \<longrightarrow> inv n (f s)"
  shows "inv_f n (foldl (\<lambda> Fs s. add Fs (f s)) empty_family F)"
  using assms
  by (induction F rule: rev_induct) auto

(* add_set_by_card *)
lemma add_set_by_card [simp]:
  assumes "c < length Fs"
  shows  "add_set_by_card Fs s ! c = 
            (if c \<noteq> card s then Fs ! c else add (Fs ! c) s)"
  using assms
  by (simp add: add_set_by_card_def Let_def)

lemma add_set_by_card_length [simp]:
  shows "length (add_set_by_card Fs s) = length Fs"
  by (simp add: add_set_by_card_def Let_def)

(* foldl add_set_by_card *)
lemma foldl_add_set_by_card_length [simp]:
  shows "length (foldl add_set_by_card Fs ss) = length Fs"
  by (induction ss rule: rev_induct, auto)

lemma foldl_add_set_by_card_inv_f:
  assumes "n \<le> n_max"
  assumes "\<forall> F \<in> set Fs. inv_f n F" "\<forall> s \<in> set ss. inv n s"
  shows "\<forall> F \<in> set (foldl add_set_by_card Fs ss). inv_f n F"
  using assms
proof (induction ss rule: rev_induct)
  case Nil
  then show ?case
    by simp
next
  case (snoc s ss)
  show ?case 
  proof safe
    fix F
    let ?Fs = "foldl add_set_by_card Fs ss"
    assume "F \<in> set (foldl add_set_by_card Fs (ss @ [s]))"
    hence "F \<in> set (add_set_by_card ?Fs s)"
      by simp
    then obtain c where *: "c < length (add_set_by_card ?Fs s) \<and> F = add_set_by_card ?Fs s ! c"
      by (metis in_set_conv_nth)
    hence "F \<in> set ?Fs \<or> F = add (?Fs ! c) s"
      using add_set_by_card[of c "foldl add_set_by_card Fs ss" s]
      by simp
    thus "inv_f n F"
      using snoc add_inv[of n "?Fs ! c" s] *
      by (metis Un_insert_right append_Nil2 insert_iff add_set_by_card_length list.set(2) nth_mem set_append)
  qed
qed

lemma foldl_add_set_by_card_sets [simp]:
  assumes "n \<le> n_max" "\<forall> s \<in> set ss. inv n s"
  assumes "c \<le> n" 
  shows "set (sets (foldl add_set_by_card (replicate (n+1) empty_family) ss ! c)) =
         {s. s \<in> set ss \<and> card s = c}"
  using assms
proof (induction ss rule: rev_induct)
  case Nil
  thus ?case
    by (cases c) (auto simp add: nth_Cons)
next
  case (snoc p ps)
  let ?Fs = "foldl add_set_by_card (replicate (n + 1) empty_family) ps"
  have "\<forall>i\<in>set ?Fs. inv_f n i"
  proof (rule foldl_add_set_by_card_inv_f)
    show "n \<le> n_max"
      by fact
  next
    show "\<forall>i\<in>set (replicate (n + 1) empty_family). inv_f n i"
      by simp
  next
    show "\<forall>a\<in>set ps. inv n a"
      using snoc
      by simp
  qed
  hence "inv_f n (?Fs ! c)"
    using `c \<le> n`
    by auto
  thus ?case
    using snoc
    by auto
qed

(* group_powerset_by_card *)
definition group_powerset_by_card_OK where
  "group_powerset_by_card_OK powerset_by_card n \<longleftrightarrow> 
     (\<forall> c \<le> n. let Fo = Mapping.lookup powerset_by_card c 
                 in Fo \<noteq> None \<and> set (sets (the Fo)) = {s. inv n s \<and> card s = c})"

lemma group_powerset_by_card_OK [simp]:
  assumes "n \<le> n_max"
  shows  "group_powerset_by_card_OK (group_powerset_by_card n) n"
  using assms
  using foldl_add_set_by_card_sets
  unfolding group_powerset_by_card_def group_powerset_by_card_OK_def
  by (simp add: Let_def bulkload_tabulate Mapping.lookup_tabulate powerset_set)

definition group_powerset_by_card_inv where
  "group_powerset_by_card_inv powerset_by_card n \<longleftrightarrow> 
     (\<forall> c \<le> n. inv_f n (the (Mapping.lookup powerset_by_card c)))" 

lemma group_powerset_by_card_inv_f [simp]:
  assumes "n \<le> n_max" 
  shows "group_powerset_by_card_inv (group_powerset_by_card n) n"
  unfolding group_powerset_by_card_inv_def
proof safe
  fix c
  assume "c \<le> n"
  hence "inv_f n (foldl add_set_by_card (replicate (n + 1) empty_family) (powerset n) ! c)"
    using foldl_add_set_by_card_inv_f
    using foldl_add_set_by_card_length
    using assms 
    by (metis add.commute in_set_replicate empty_family_inv le_neq_implies_less length_replicate less_add_one local.powerset_set mem_Collect_eq nth_mem trans_less_add2)
  thus "inv_f n (the (Mapping.lookup (group_powerset_by_card n) c))"
    using assms `c \<le> n` 
    unfolding group_powerset_by_card_def
    by (auto simp add: Let_def Mapping.bulkload_tabulate Mapping.lookup_tabulate)
qed

(* inter *)
lemma inter_abs [simp]:
  assumes "n \<le> n_max" "inv_f n F1" "inv_f n F2"
  shows "abs_fam (inter F1 F2) = FamilyAbs.inter (abs_fam F1) (abs_fam F2)" 
  using assms abs_set_inj[of n]
  using inj_on_image_Int[of abs_set "set (sets F1) \<union> set (sets F2)" "set (sets F1)" "set (sets F2)"]
  by (force simp add: abs_fam_def FamilyAbs.inter_def Family_inverse inj_on_def)

(* sets_by_card *)
lemma sets_by_card_sets [simp]:
  assumes "group_powerset_by_card_OK powerset_by_card n" 
  assumes "cc \<le> n"
  shows "set (sets (sets_by_card powerset_by_card F cc)) = 
         {s \<in> set (sets F). inv n s \<and> card s = cc}"
  using assms
  unfolding sets_by_card_def group_powerset_by_card_OK_def
  by (auto simp add: Let_def)

lemma sets_by_card_abs [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n F"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "c \<le> n"
  shows "abs_fam (sets_by_card powerset_by_card F c) =
         Family {s. s \<in> FamilyAbs.sets (abs_fam F) \<and> Finite_Set.card (Set.elements s) = c}"
proof-
  have "abs_set ` {s \<in> set (sets F). inv n s \<and> SetCard_class.card s = c} =
        {s \<in> abs_set ` set (sets F). Finite_Set.card (Set.elements s) = c}" (is "?lhs = ?rhs")
  proof
    fix x
    show "?lhs \<subseteq> ?rhs"
      using card_abs[OF `n \<le> n_max`]
      by auto
  next
    show "?rhs \<subseteq> ?lhs"
    proof
      fix s
      assume "s \<in> ?rhs"
      then obtain x where "s = abs_set x" "x \<in> set (sets F)" "Finite_Set.card (Set.elements s) = c"
        by auto
      thus "s \<in> ?lhs"
        using inv_f_inv[OF `n \<le> n_max` `inv_f n F`]
        using card_abs[OF `n \<le> n_max`]
        by simp
    qed
  qed
  thus ?thesis
    using assms
    by (simp add: abs_fam_def Family_inverse)
qed

lemma sets_by_card_inv [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n F" 
  assumes "group_powerset_by_card_inv powerset_by_card n" 
  assumes "c \<le> n"
  shows "inv_f n (sets_by_card powerset_by_card F c)"
  using assms
  unfolding sets_by_card_def group_powerset_by_card_inv_def
  by simp

(* group_sets_by_card *)
definition group_sets_by_card_OK where
  "group_sets_by_card_OK by_card F n \<longleftrightarrow> 
   (\<forall> c \<le> n. let S = by_card ! c
              in set (sets S) = {s \<in> set (sets F). inv n s \<and> card s = n - c})"

lemma group_sets_by_card_OK:
  assumes "n \<le> n_max" 
  assumes "group_powerset_by_card_OK powerset_by_card n"
  shows "group_sets_by_card_OK (group_sets_by_card n powerset_by_card F) F n"
proof-
  let ?s = "sets_by_card powerset_by_card F"
  have "\<forall> c \<le> n. rev [0..<n + 1] ! c = n - c"
    by (smt One_nat_def add.left_neutral add.right_neutral add_Suc add_Suc_right add_diff_cancel_left' diff_le_self le_add_diff_inverse le_neq_implies_less length_upt less_add_one nth_upt rev_nth trans_less_add2)
  hence "\<forall> c \<le> n. map ?s (rev [0..<n + 1]) ! c = ?s (n-c)"
    using nth_map[of _ "rev [0..<n + 1]" ?s]
    by simp
  thus ?thesis
    using assms
    unfolding group_sets_by_card_OK_def group_sets_by_card_def
    by auto
qed

(* eq_family *)
lemma eq_family_abs [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F1)" "inv_f n (all_sets F2)"
  shows "eq_family F1 F2 \<longleftrightarrow> abs_fam (all_sets F1) = abs_fam (all_sets F2)"
  using assms abs_fam_inj
  unfolding eq_family_def
  by metis

(* less_family *)

lemma sets_by_card_eq:
  assumes "n \<le> n_max" "c \<le> n" 
  assumes "inv_f n (all_sets F1)" "inv_f n (all_sets F2)"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  shows "sets_by_card powerset_by_card (all_sets F1) c = sets_by_card powerset_by_card (all_sets F2) c \<longleftrightarrow> 
         sorted_list_of_set (eq_card_sets (abs_fam (all_sets F1)) c) =
         sorted_list_of_set (eq_card_sets (abs_fam (all_sets F2)) c)" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?rhs"
  have "eq_card_sets (abs_fam (all_sets F1)) c = eq_card_sets (abs_fam (all_sets F2)) c" (is "?A = ?B")
  proof (rule sorted_list_of_set_inj)
    show "finite ?A" "finite ?B"
      by (auto simp add: eq_card_sets_def)
    show "?rhs"
      by fact
  qed
  hence *: "abs_fam (sets_by_card powerset_by_card (all_sets F1) c) =
         abs_fam (sets_by_card powerset_by_card (all_sets F2) c)" (is "abs_fam ?A = abs_fam ?B")
    using assms
    by (simp add: eq_card_sets_def)
  show ?lhs
  proof (rule abs_fam_inj[OF `n \<le> n_max`])
    show "inv_f n ?A" "inv_f n ?B"
      using assms sets_by_card_inv[of n _ powerset_by_card]
      by auto
  next
    show "abs_fam ?A = abs_fam ?B"
      by fact
  qed
next
  assume ?lhs (is "?A = ?B")
  hence "abs_fam ?A = abs_fam ?B"
    by simp
  hence "eq_card_sets (abs_fam (all_sets F1)) c = eq_card_sets (abs_fam (all_sets F2)) c" (is "?A = ?B")
    using assms
    using Family_inject
    by (simp add: eq_card_sets_def)
  thus ?rhs
    by simp
qed

lemma sorted_list_of_set_abs:
  assumes "n \<le> n_max"
  assumes "inv_f n F1" "inv_f n F2"                     
  assumes "length (sets F1) = length (sets F2)" "\<forall> s \<in> set (sets F1). card s = c" "\<forall> s \<in> set (sets F2). card s = c"
  shows "F1 < F2 \<longleftrightarrow> 
         sorted_list_of_set (FamilyAbs.sets (abs_fam F1)) > sorted_list_of_set (FamilyAbs.sets (abs_fam F2))"
proof-
  let ?D = "set (sets F1) \<union> set (sets F2)"
  have *: "\<forall> x y. x \<in> ?D \<and> y \<in> ?D \<longrightarrow> 
            (x < y \<longleftrightarrow> abs_set x > abs_set y)"
  proof (rule allI, rule allI, rule impI, erule conjE)
    fix x y
    assume "x \<in> ?D" "y \<in> ?D"
    hence "inv n x" "inv n y" "card x = card y"
      using assms
      by auto
    thus "x < y \<longleftrightarrow> abs_set x > abs_set y"
      using assms less_set_same_card_abs
      by simp
  qed
  moreover
  hence "\<forall> x y. x \<in> set (sets F1) \<and> y \<in> set (sets F1) \<and> x < y \<longrightarrow> abs_set x > abs_set y"
        "\<forall> x y. x \<in> set (sets F2) \<and> y \<in> set (sets F2) \<and> x < y \<longrightarrow> abs_set x > abs_set y"
    by auto
  moreover
  have "inj_on abs_set (set (sets F1))" 
    using abs_set_inj
    by (metis (mono_tags, lifting) FamilyImpl.inv_f_inv FamilyImpl_axioms assms(1) assms(2) inj_onI)
  moreover
  have "inj_on abs_set (set (sets F2))"
    using abs_set_inj
    by (metis (mono_tags, lifting) FamilyImpl.inv_f_inv FamilyImpl_axioms assms(1) assms(3) inj_onI)
  moreover
  have "finite (set (sets F1))" "finite (set (sets F2))"
    by simp_all
  moreover
  have "sorted_list_of_set (set (sets F2)) = sets F2" "sorted_list_of_set (set (sets F1)) = sets F1"
    using assms
    by (meson calculation distinct_sets distinct_sorted_list_of_set set_sorted_list_of_set sorted_distinct_set_unique sorted_sets sorted_sorted_list_of_set)+
  moreover
  have "(rev (sets F1) < rev (sets F2)) = (map abs_set (rev (sets F2)) < map abs_set (rev (sets F1)))"
  proof (subst map_mono'[symmetric])
    let ?D = "set (rev (sets F1)) \<union> set (rev (sets F2))"
    show "\<forall>x\<in>?D. \<forall>y\<in>?D. (x < y) = (abs_set y < abs_set x)"
      by (metis "*" set_rev)
    show "\<forall>x\<in>?D. \<forall>y\<in>?D. abs_set x = abs_set y \<longrightarrow> x = y"
      using abs_set_inj
      using assms(1) assms(2) assms(3) by auto
    show "length (rev (sets F1)) = length (rev (sets F2))"
      using assms
      by simp
  qed simp
  ultimately
  show ?thesis
    using assms
    unfolding abs_fam_def
    by (simp add: sorted_list_of_set_image_rev less_family_sets rev_map Family_inverse)
qed

lemma sets_by_card_less:
  assumes "n \<le> n_max" "c \<le> n" 
  assumes "inv_f n (all_sets F1)" "inv_f n (all_sets F2)"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "\<forall>c\<le>n. Finite_Set.card (eq_card_sets (abs_fam (all_sets F1)) c) =
                 Finite_Set.card (eq_card_sets (abs_fam (all_sets F2)) c)"
  shows "sets_by_card powerset_by_card (all_sets F1) c < sets_by_card powerset_by_card (all_sets F2) c \<longleftrightarrow> 
         sorted_list_of_set (eq_card_sets (abs_fam (all_sets F1)) c) >
         sorted_list_of_set (eq_card_sets (abs_fam (all_sets F2)) c)" (is "?a < ?b \<longleftrightarrow> sorted_list_of_set ?a' > sorted_list_of_set ?b'")
proof-
  let ?sls = "sorted_list_of_set"
  have *: "\<forall>s\<in>set (sets ?a). card s = c" "\<forall>s\<in>set (sets ?b). card s = c" 
    using assms
    by simp_all

  have **: "length (sets ?a) = length (sets ?b)"
  proof-
    have 
      "length (sets ?a) = Finite_Set.card (set (sets ?a))"
      "length (sets ?b) = Finite_Set.card (set (sets ?b))"
      using assms
      by (metis distinct_card distinct_sets sets_by_card_inv)+
    moreover
    have "Finite_Set.card (set (sets ?a)) = Finite_Set.card (set (sets ?b))"
    proof-
      have "(eq_card_sets (abs_fam (all_sets F1)) c) =
            abs_set ` {s \<in> set (sets (all_sets F1)). inv n s \<and> SetCard_class.card s = c}"
        unfolding eq_card_sets_def
        using inv_f_inv[OF `n \<le> n_max` `inv_f n (all_sets F1)`]
        unfolding abs_fam_def
        using assms(1) card_abs
        by (auto simp add: Family_inverse)
      hence "Finite_Set.card (eq_card_sets (abs_fam (all_sets F1)) c) = 
             Finite_Set.card {s \<in> set (sets (all_sets F1)). inv n s \<and> SetCard_class.card s = c}"
        using card_image abs_set_inj assms
        by (smt inj_onI mem_Collect_eq)
      moreover
      have "(eq_card_sets (abs_fam (all_sets F2)) c) =
            abs_set ` {s \<in> set (sets (all_sets F2)). inv n s \<and> SetCard_class.card s = c}"
        unfolding eq_card_sets_def
        using inv_f_inv[OF `n \<le> n_max` `inv_f n (all_sets F2)`]
        unfolding abs_fam_def
        using assms(1) card_abs
        by (auto simp add: Family_inverse)
      hence "Finite_Set.card (eq_card_sets (abs_fam (all_sets F2)) c) = 
             Finite_Set.card {s \<in> set (sets (all_sets F2)). inv n s \<and> SetCard_class.card s = c}"
        using card_image abs_set_inj assms
        by (smt inj_onI mem_Collect_eq)
      ultimately
      show ?thesis
        using assms
        unfolding abs_fam_def
        by simp
    qed
    ultimately
    show ?thesis
      by simp
  qed

  show ?thesis
  proof
    assume "?a < ?b"

    have "?sls (FamilyAbs.sets (abs_fam ?a)) > ?sls (FamilyAbs.sets (abs_fam ?b))"
      using * ** assms `?a < ?b` 
      by (meson sets_by_card_inv sorted_list_of_set_abs)

    thus "?sls ?a' > ?sls ?b'"
      using assms
      by (simp add: eq_card_sets_def Family_inverse)
  next
    assume "?sls ?a' > ?sls ?b'"

    hence "?sls (FamilyAbs.sets (abs_fam ?a)) > ?sls (FamilyAbs.sets (abs_fam ?b))"
      using assms
      by (simp add: eq_card_sets_def Family_inverse)

    thus "?a < ?b"
      using * ** assms(1-5)
      by (meson sets_by_card_inv sorted_list_of_set_abs)
  qed
qed

lemma less_family_abs:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F1)" "inv_f n (all_sets F2)"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "FamilyAbs.family_card (abs_fam (all_sets F1)) = FamilyAbs.family_card (abs_fam (all_sets F2))"
  assumes "\<forall>c\<le>n. Finite_Set.card (eq_card_sets (abs_fam (all_sets F2)) c) =
                 Finite_Set.card (eq_card_sets (abs_fam (all_sets F1)) c)"
  shows "less_family n powerset_by_card F1 F2 \<longleftrightarrow>
         abs_fam (all_sets F1) < abs_fam (all_sets F2)"
proof-
  let ?sc = "sets_by_card powerset_by_card"
  let ?f = "\<lambda> F c. sorted_list_of_set (eq_card_sets (Family (FamilyAbs.sets (abs_fam (all_sets F)))) c)"

  have "less_family n powerset_by_card F1 F2 \<longleftrightarrow> 
        (\<exists> c. c \<le> n \<and> 
             ?sc (all_sets F2) c < ?sc (all_sets F1) c \<and> 
             (\<forall> c'. c < c' \<and> c' \<le> n \<longrightarrow> ?sc (all_sets F2) c' = ?sc (all_sets F1) c'))"
    unfolding less_family_def group_sets_by_card_def 
    apply (subst map_less)
    apply (auto simp del: upt_Suc simp add: rev_nth)
     apply (rule_tac x="n-i" in exI, simp)
     apply auto[1]
     apply (erule_tac x="n-c'" in allE)
     apply auto[1]
    apply (rule_tac x="n-c" in exI)
    apply (auto simp del: upt_Suc simp add: rev_nth)
    done
  also have "... \<longleftrightarrow> (\<exists>c\<le>n. (\<forall>c'. c < c' \<and> c' \<le> n \<longrightarrow> ?f F1 c' = ?f F2 c') \<and> 
                             ?f F1 c < ?f F2 c)"
    using assms
    using sets_by_card_less[OF `n \<le> n_max` _ `inv_f n (all_sets F2)` `inv_f n (all_sets F1)` _ _ assms(7)]
    using sets_by_card_eq[of n]
    by (smt sets_inverse)
  also have "... \<longleftrightarrow> sorted_list_of_set (FamilyAbs.sets (abs_fam (all_sets F2))) >
                     sorted_list_of_set (FamilyAbs.sets (abs_fam (all_sets F1)))"
    using assms
    by (subst sorted_list_of_set_by_card[symmetric])
       (simp_all add: abs_fam_def abs_set_def set_elements Family_inverse Set_inverse) 
  also have "... \<longleftrightarrow> abs_fam (all_sets F2) > abs_fam (all_sets F1)"
    by (simp add: less_Family_def)
  finally
  show ?thesis
    .
qed

(* set_perms *)
definition set_perms_OK where
  "set_perms_OK set_perms n \<longleftrightarrow> 
     (\<forall> p s. p < fact n \<and> inv n s \<longrightarrow> 
        (let so = Mapping.lookup set_perms (p, s) 
          in so \<noteq> None \<and> the so = permute_set n (permute[0..<n] ! p) s))"

lemma init_set_perms_OK:
  assumes "n \<le> n_max"
  shows "set_perms_OK (init_set_perms n) n"
proof-
  let ?D = "map (\<lambda>p. map (Pair p) (powerset n)) [0..<length (permute [0..<n])]"
  have "distinct (concat ?D)"
  proof (rule distinct_concat)
    have "inj_on (\<lambda>p. map (Pair p) (powerset n)) {0..<length (permute [0..<n])}"
      using powerset_not_empty[OF `n \<le> n_max`]
      unfolding inj_on_def
      by auto
    thus "distinct ?D"
      by (simp add: distinct_map)
  next
    fix ys
    assume "ys \<in> set ?D"
    thus "distinct ys"
      using `n \<le> n_max`
      by (auto simp add: distinct_map inj_on_def)
  next
    fix ys zs
    assume "ys \<in> set ?D" "zs \<in> set ?D" "ys \<noteq> zs"
    thus "set ys \<inter> set zs = {}"
      by auto
  qed
  thus ?thesis
    using assms
    unfolding init_set_perms_def set_perms_OK_def
    by (auto simp add: Mapping.lookup_tabulate powerset_set)
qed

(* permute_family *)
lemma foldl_add_set_perms_sets:
  assumes "n \<le> n_max"
  assumes "\<forall> s \<in> set F. inv n s"
  assumes "set_perms_OK set_perms n"
  assumes "p < fact n" 
  shows "set (sets
              (foldl (\<lambda>r s. add r (the (Mapping.lookup set_perms (p, s))))
                 empty_family F)) =
        permute_set n (permute [0..<n] ! p) ` set F"
  using assms
proof (induction F rule: rev_induct)
  case Nil
  thus ?case
    by simp
next
  case (snoc s F')
  have "the (Mapping.lookup set_perms (p, s)) = permute_set n (permute [0..<n] ! p) s"
    using snoc(2-5)    
    unfolding set_perms_OK_def
    by (simp add: Let_def)
  moreover
  have "inv n (permute_set n (permute [0..<n] ! p) s)"
    by (simp add: snoc.prems)
  moreover
  have "\<forall>s. inv n s \<longrightarrow> inv n (the (Mapping.lookup set_perms (p, s)))"
    using snoc
    unfolding set_perms_OK_def Let_def
    by simp
  ultimately
  show ?case
    using snoc
    by (simp add: foldl_add_inv_f)
qed

lemma permute_family_sets [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)" 
  assumes "set_perms_OK set_perms n"
  assumes "p < fact n"
  shows "set (sets (all_sets (permute_family set_perms p F))) = 
         permute_set n ((permute [0..<n]) ! p) ` (set (sets (all_sets F)))"
  using assms
  using foldl_add_set_perms_sets
  unfolding permute_family_def
  by simp

lemma permute_family_abs [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n"
  assumes "p < fact n" 
  shows "abs_fam (all_sets (permute_family set_perms p F)) =
         FamilyAbs.permute_family (permute [0..<n] ! p) (abs_fam (all_sets F))"
proof-
  have "abs_set ` permute_set n (permute [0..<n] ! p) ` set (sets (all_sets F)) =
        FamilyAbs.permute_set (permute [0..<n] ! p) ` abs_set ` set (sets (all_sets F))"
    using assms(1) assms(2) assms(4)
    by (auto simp add: image_iff)
  thus ?thesis
    using assms
    unfolding FamilyAbs.permute_family_def abs_fam_def
    by (auto simp add: image_iff Family_inverse)
qed

lemma permute_family_inv_f [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n"
  assumes "p < fact n"
  shows "inv_f n (all_sets (permute_family set_perms p F))"
  using assms
  unfolding permute_family_def set_perms_OK_def 
  by (simp add: foldl_add_inv_f Let_def)

lemma permute_family_0 [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n" 
  shows "all_sets (permute_family set_perms 0 F) = all_sets F" (is "?lhs = ?rhs")
proof-
  have "abs_fam ?lhs = abs_fam ?rhs"
    using assms permute_family_abs[of n F set_perms 0] 
    using set_elements[of n]
    using permute_family_id[of "abs_fam (all_sets F)" n]
    using hd_conv_nth[OF permute_not_empty, symmetric, of "[0..<n]"]
    by (simp add: perm_id_def abs_fam_def abs_set_def image_iff Family_inverse Set_inverse)
  thus ?thesis
    using abs_fam_inj
    using assms
    by auto
qed

(* perm_fixes_subset *)
lemma perm_fixes_subset_abs [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)" 
          "\<forall> s \<in> set (sets F'). inv n s" 
  assumes "set_perms_OK set_perms n"
  assumes "p < fact n" 
  shows "perm_fixes_subset set_perms F F' p \<longleftrightarrow> 
         FamilyAbs.perm_fixes_subset (permute [0..<n] ! p) 
                                     (FamilyAbs.sets (abs_fam (all_sets F))) 
                                     (FamilyAbs.sets (abs_fam F'))"
  using assms contains_abs[OF `n \<le> n_max` `inv_f n (all_sets F)`]
  unfolding perm_fixes_subset_def set_perms_OK_def Let_def FamilyAbs.perm_fixes_subset_def
  by (simp add: list_all_iff abs_fam_def Family_inverse)

(* perm_fixes_card *)

lemma last_set_Max_min_card:
  assumes "n \<le> n_max" 
  assumes "inv n (last_set F)"
  assumes "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "last_set F \<in> set (sets (all_sets F))"
  shows "card (last_set F) = min_card (abs_fam (all_sets F))"
proof (rule ccontr)
  assume "\<not> ?thesis"
  let ?C = "(Finite_Set.card \<circ> Set.elements) ` (Family.sets (abs_fam (all_sets F)))"
  
  have "finite ?C"
    by (simp add: abs_fam_def)

  have "card (last_set F) \<in> ?C" 
    using `n \<le> n_max` `last_set F \<in> set (sets (all_sets F))` `inv n (last_set F)`
    by (auto simp add: card_abs abs_fam_def Family_inverse)

  have "finite (Family.sets (abs_fam (all_sets F)))"
    by simp

  have "card (last_set F) > min_card (abs_fam (all_sets F))"
    using `card (last_set F) \<noteq> min_card (abs_fam (all_sets F))` Min_le[OF `finite ?C` `card (last_set F) \<in> ?C`]
    unfolding min_card_def
    by simp
  then obtain x where "x \<in> Family.sets (abs_fam (all_sets F))"
    "Finite_Set.card (Set.elements x) < card (last_set F)"
    using ex_min_card[of "abs_fam (all_sets F)"]
    using assms
    by auto
  hence "abs_set (last_set F) < x"
    using card_abs[OF `n \<le> n_max` `inv n (last_set F)`, symmetric]
    using \<open>SetCard_class.card (last_set F) \<noteq> min_card (abs_fam (all_sets F))\<close> 
    by (metis all_not_in_conv assms(3) finite_sets min_card_Max)

  thus False
    using `abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))`
    using Max_ge[OF `finite (Family.sets (abs_fam (all_sets F)))`, of x]
    using `x \<in> Family.sets (abs_fam (all_sets F))`
    by (simp add: leD)
qed

lemma min_card_above_last_nonempty [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
          "\<exists>s \<in> set (sets (all_sets F)). card (last_set F) < card s"
  shows "FamilyAbs.min_card_above (abs_fam (all_sets F)) (card (last_set F)) \<le> n"
proof-
  obtain s where *: "s \<in> set (sets (all_sets F))" "card (last_set F) < card s"
    using assms
    by auto
  hence "inv n s"
    using assms inv_f_inv
    by blast
  have "FamilyAbs.min_card_above (abs_fam (all_sets F)) (card (last_set F)) \<le> card s"
  proof (rule min_card_above_leq)
    show "\<exists>sa\<in>Family.sets (abs_fam (all_sets F)). FamilyAbs.set_card sa = card s"
      using * card_abs[OF `n \<le> n_max` `inv n s`]
      by (auto simp add: abs_fam_def Family_inverse)
  next
    show "card (last_set F) < card s"
      by fact
  next
    show "finite (Family.sets (abs_fam (all_sets F)))"
      by simp
  qed

  moreover

  have "card s \<le> n"
    by (simp add: \<open>inv n s\<close> `n \<le> n_max` powerset_set powerset_elements_card)

  ultimately

  show "FamilyAbs.min_card_above (abs_fam (all_sets F)) (card (last_set F)) \<le> n"
    by simp
qed

lemma perm_fixes_card_abs [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n" 
  assumes "group_powerset_by_card_OK powerset_by_card n"
          "group_powerset_by_card_inv powerset_by_card n"
  assumes "p < fact n"
  assumes "\<exists>s \<in> set (sets (all_sets F)). card s = c"
  shows "perm_fixes_card n powerset_by_card set_perms F c p \<longleftrightarrow> 
         (let F' = abs_fam (all_sets F)
           in FamilyAbs.perm_fixes_subset (permute [0..<n] ! p) 
                                          (FamilyAbs.sets F')
                                          (eq_card_sets F' c))"
proof-
  have "c \<le> n"
    using assms
    using powerset_elements_card powerset_set by auto
  thus ?thesis
    using assms 
    unfolding perm_fixes_card_def Let_def
    by (auto simp add: FamilyAbs.eq_card_sets_def Family_inverse)
qed

lemma perm_fixes_subset_id:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "\<exists>s \<in> set (sets (all_sets F)). card s = c"
  shows "perm_fixes_subset set_perms F (sets_by_card powerset_by_card (all_sets F) c) 0"
proof-
  let ?S = "sets_by_card powerset_by_card (all_sets F) c"

  have p: "permute [0..<n] ! 0 = perm_id n"
    by (metis perm_id_def hd_conv_nth permute_hd permute_not_empty)

  have "c \<le> n"
    using assms
    using powerset_elements_card powerset_set by auto

  moreover

  hence "inv_f n ?S"
    using sets_by_card_inv assms
    by auto

  moreover
  have "\<forall>s\<in>FamilyAbs.sets (abs_fam (all_sets F)). FamilyAbs.elements s \<subseteq> {0..<n}"
    using set_elements[OF `n \<le> n_max` inv_f_inv[OF `n \<le> n_max` `inv_f n (all_sets F)`]]
    using assms
    by (simp add: abs_fam_def abs_set_def Family_inverse Set_inverse)

  ultimately
  show ?thesis
    using assms
    apply (subst perm_fixes_subset_abs, simp_all)
    apply (subst p, subst perm_fixes_subset_id, simp_all add: Family_inverse)
    done
qed

definition hd_perms where
  "hd_perms F \<longleftrightarrow> perms F \<noteq> [] \<and> hd (perms F) = 0"

lemma hd_filter:
  assumes "xs \<noteq> [] \<and> P (hd xs)"
  shows "filter P xs \<noteq> [] \<and> hd (filter P xs) = hd xs"
  using assms
  by (cases xs, auto)

(* filter_perms *)
lemma filter_perms_all_sets [simp]:
  shows "all_sets (filter_perms n powerset_by_card set_perms F c) = all_sets F"
  unfolding filter_perms_def
  by simp

lemma filter_perms_reduced [simp]:
  shows "reduced (filter_perms n powerset_by_card set_perms F c) = reduced F"
  unfolding filter_perms_def
  by simp

lemma filter_perms_perms [simp]:
  shows "set (perms (filter_perms n powerset_by_card set_perms F c)) =  
         {pp \<in> set (perms F). perm_fixes_card n powerset_by_card set_perms F c pp}"
  unfolding filter_perms_def
  by simp

lemma filter_perms_hd_perms [simp]:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n (last_set F)"
     "set_perms_OK set_perms n"
     "group_powerset_by_card_OK powerset_by_card n"
     "group_powerset_by_card_inv powerset_by_card n"
     "\<exists>s\<in>set (sets (all_sets F)). card s = c"
  assumes "hd_perms F"
  shows "hd_perms (filter_perms n powerset_by_card set_perms F c)"
  using assms
  unfolding hd_perms_def filter_perms_def Let_def FamilyCache.sel(4)
  using hd_filter[of "perms F" "perm_fixes_card n powerset_by_card set_perms F c"]
  using perm_fixes_subset_id[of n F set_perms powerset_by_card c]
  unfolding perm_fixes_card_def Let_def
  by simp

(* TODO: express set = *)
lemma filter_perms_abs [simp]:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
          "\<exists>s\<in>set (sets (all_sets F)). card s = c"
  assumes "set_perms_OK set_perms n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
          "group_powerset_by_card_inv powerset_by_card n"
  assumes "p < fact n" 
  shows "let F' = abs_fam (all_sets F)
         in  p \<in> set (perms (filter_perms n powerset_by_card set_perms F c)) \<longleftrightarrow> 
             p \<in> set (perms F) \<and> 
               FamilyAbs.perm_fixes_subset (permute [0..<n] ! p) (Family.sets F')
                 (eq_card_sets F' c)"
  using assms
  unfolding filter_perms_def Let_def
  by (simp add: Let_def)

(* perms_filtered *)
definition perms_filtered where
  "perms_filtered F n \<longleftrightarrow> 
     set (perms F) = {pp. pp < fact n \<and> FamilyAbs.perm_fixes (permute [0..<n] ! pp) (above_min_card_sets (abs_fam (all_sets F)))}"

(* is_canon *)
lemma is_canon_abs [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "set_perms_OK set_perms n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "perms_filtered F n"
          "hd_perms F"
  shows "is_canon n powerset_by_card set_perms F \<longleftrightarrow>
         FamilyAbs.is_canon_opt n (abs_fam (all_sets F))" (is "?lhs = ?rhs")
proof
  assume ?lhs
  show ?rhs
    unfolding is_canon_opt_def
  proof safe
    fix p
    assume "p \<in> set (FamilyAbs.filter_perms (permute [0..<n]) (abs_fam (all_sets F)))"
    hence "p \<in> set (permute [0..<n])" and *: "FamilyAbs.perm_fixes p (above_min_card_sets (abs_fam (all_sets F)))"
      by (simp_all add: FamilyAbs.filter_perms_def Let_def)
    obtain pp where "p = permute [0..<n] ! pp" "pp < fact n"
      using `p \<in> set (permute [0..<n])`
      by (auto simp add: in_set_conv_nth)
    hence "pp \<in> set (perms F)"
      using `perms_filtered F n` *
      unfolding perms_filtered_def
      by simp

    have pp: "permute [0..<n] ! pp \<in> set (permute [0..<n])"
      using \<open>p = permute [0..<n] ! pp\<close> \<open>p \<in> set (permute [0..<n])\<close> by blast

    have n: "\<forall>s\<in>FamilyAbs.sets (abs_fam (all_sets F)). FamilyAbs.elements s \<subseteq> {0..<n}"
      using abs_fam_def abs_set_def assms(1) assms(2) set_elements
      by (auto simp add: Set_inverse Family_inverse)
    
    have "less_eq_family n powerset_by_card F (permute_family set_perms pp F)"
    proof (cases "pp = hd (perms F)")
      case True
      thus ?thesis
        using assms permute_family_0
        unfolding less_eq_family_def eq_family_def hd_perms_def
        by simp
    next
      case False
      hence "pp \<in> set (tl (perms F))"
        using `pp \<in> set (perms F)`
        by (metis hd_Cons_tl set_ConsD tl_Nil)
      thus ?thesis
        using `?lhs` 
        unfolding is_canon_def
        by (simp add: list_all_iff)
    qed

    moreover

    have "family_card (abs_fam (all_sets F)) =
          family_card (FamilyAbs.permute_family (permute [0..<n] ! pp) (abs_fam (all_sets F)))"
      using pp n
      using permute_family_card by auto

    moreover

    have "\<forall>c\<le>n. Finite_Set.card (eq_card_sets (abs_fam (all_sets F)) c) =
                Finite_Set.card (eq_card_sets (abs_fam (all_sets (permute_family set_perms pp F))) c)"
    proof safe
      fix c
      assume "c \<le> n"
      show "Finite_Set.card (eq_card_sets (abs_fam (all_sets F)) c) =
            Finite_Set.card (eq_card_sets (abs_fam (all_sets (local.permute_family set_perms pp F))) c)"
        using assms `pp < fact n` pp n
        by (metis abs_fam_finite permute_family_abs permute_family_eq_card_sets)        
    qed

    ultimately

    show "abs_fam (all_sets F) \<le> FamilyAbs.permute_family p (abs_fam (all_sets F))"
      unfolding less_eq_family_def less_eq_Family_def
      using assms \<open>pp < fact n\<close> \<open>p = permute [0..<n] ! pp\<close> pp
      by (subst (asm) less_family_abs, simp_all)
  qed                                                                                        
next
  assume ?rhs
  show ?lhs
    unfolding is_canon_def list_all_iff
  proof safe
    fix pp
    let ?p = "permute [0..<n] ! pp"
    assume "pp \<in> set (tl (perms F))"
    hence "pp \<in> set (perms F)"
      by (metis list.sel(2) list.set_sel(2))
    hence "pp < fact n" "perm_fixes (permute [0..<n] ! pp) (above_min_card_sets (abs_fam (all_sets F)))"
      using `perms_filtered F n`
      unfolding perms_filtered_def
      by auto
    hence "?p \<in> set (FamilyAbs.filter_perms (permute [0..<n]) (abs_fam (all_sets F)))"
      unfolding FamilyAbs.filter_perms_def
      by (simp add: Let_def)
    hence "abs_fam (all_sets F) \<le> FamilyAbs.permute_family ?p (abs_fam (all_sets F))"
      using `?rhs`
      unfolding is_canon_opt_def
      by simp
    moreover

    have pp: "permute [0..<n] ! pp \<in> set (permute [0..<n])"
      by (simp add: \<open>pp < fact n\<close>)

    have n: "\<forall>s\<in>FamilyAbs.sets (abs_fam (all_sets F)). FamilyAbs.elements s \<subseteq> {0..<n}"
      using abs_fam_def abs_set_def assms(1) assms(2) set_elements
      by (auto simp add: Family_inverse Set_inverse)

    have "family_card (abs_fam (all_sets F)) = family_card (abs_fam (all_sets (local.permute_family set_perms pp F)))"
      using assms pp n `pp < fact n`
      by (metis FamilyImpl.permute_family_abs FamilyImpl_axioms permute_family_card)

    moreover

    have "\<forall>c\<le>n. Finite_Set.card (eq_card_sets (abs_fam (all_sets F)) c) =
                Finite_Set.card (eq_card_sets (abs_fam (all_sets (permute_family set_perms pp F))) c)"
    proof safe
      fix c
      assume "c \<le> n"
      show "Finite_Set.card (eq_card_sets (abs_fam (all_sets F)) c) =
            Finite_Set.card (eq_card_sets (abs_fam (all_sets (local.permute_family set_perms pp F))) c)"
        using assms `pp < fact n` pp n
        by (metis abs_fam_finite permute_family_abs permute_family_eq_card_sets)        
    qed

    ultimately

    show "less_eq_family n powerset_by_card F (local.permute_family set_perms pp F)"
      unfolding less_eq_family_def less_eq_Family_def
      using assms \<open>pp < fact n\<close> 
      using eq_family_abs less_family_abs permute_family_abs permute_family_inv_f
      by presburger
  qed
qed

(* k_sets *)
lemma k_sets_sets [simp]: 
  assumes "n \<le> n_max"
  shows "set (map (set o elements) (k_sets n k)) =
         {A. Finite_Set.card A = k \<and> A \<subseteq> {0..<n}}"  (is "?lhs = ?rhs")
proof safe
  fix x
  assume "x \<in> ?lhs"
  thus "Finite_Set.card x = k"
    using assms
    unfolding k_sets_def
    by (smt SetImpl.set_of_elements SetImpl_axioms atLeastLessThan_upt combine_combines comp_apply distinct_card distinct_upt in_set_conv_nth k_sets_def length_map nth_map nth_mem sorted_upt)
next
  fix x x'
  assume "x \<in> ?lhs" "x' \<in> x"
  thus "x' \<in> {0..<n}"
    using assms
    unfolding k_sets_def
    by (smt SetImpl.set_of_elements SetImpl_axioms atLeastLessThan_upt combine_combines comp_apply distinct_upt in_set_conv_nth length_map nth_map nth_mem sorted_upt subsetD)
next
  fix x
  assume "x \<subseteq> {0..<n}" "k = Finite_Set.card x"
  thus "x \<in> set (map (set \<circ> elements) (k_sets n (Finite_Set.card x)))"
    using assms
    unfolding k_sets_def
    by (smt atLeastLessThan_upt combine_combines combine_sublist comp_apply distinct_upt in_set_conv_nth length_map local.set_of_elements nth_map sorted_upt)
qed

lemma k_sets_abs [simp]: 
  assumes "n \<le> n_max"
  shows "abs_set ` (set (k_sets n k)) = 
         FamilyAbs.Set ` {A. Finite_Set.card A = k \<and> A \<subseteq> {0..<n}}" 
  using assms k_sets_sets[of n k]
  unfolding abs_set_def
  by simp (metis image_image)

lemma set_of_inj:
  assumes "n \<le> n_max"
  assumes "sorted xs" "distinct xs" "set xs \<subseteq> {0..<n}"
  assumes "sorted ys" "distinct ys" "set ys \<subseteq> {0..<n}"
  assumes "set_of xs = set_of ys"
  shows "xs = ys"
  using assms
  using set_of_elements by fastforce

lemma k_sets_distinct [simp]:
  assumes "n \<le> n_max"
  shows "distinct (k_sets n k)"
  unfolding k_sets_def
proof (subst distinct_map, safe)
  show "distinct (combine [0..<n] k)"
    using assms
    by (simp add: distinct_combine)
next
  show "inj_on set_of (set (combine [0..<n] k))"
    unfolding inj_on_def
    using assms
    using set_of_inj combine_combines
    by (metis atLeastLessThan_upt distinct_upt sorted_upt)
qed

lemma k_sets_inv:
  assumes "n \<le> n_max"
  assumes "x \<in> set (k_sets n k)"
  shows "inv n x"
  using assms set_of_inv[OF `n \<le> n_max`]
  using combine_combines[of "[0..<n]" _ k]
  unfolding k_sets_def
  by auto

lemma k_sets_card:
  assumes "n \<le> n_max"
  assumes "x \<in> set (k_sets n k)"
  shows "card x = k"
proof-
  have "abs_set x \<in> abs_set ` set (k_sets n k)"
    using assms
    by blast
  thus ?thesis
    using assms
    by (auto simp add: card_abs k_sets_inv Set_inverse finite_subset)
qed

lemma k_sets_inj:
  assumes "n \<le> n_max"
  assumes "x \<in> set (k_sets n k1)" "x \<in> set (k_sets n k2)"
  shows "k1 = k2"
  using assms k_sets_card[of n]
  by auto

lemma k_sets_non_empty:
  assumes "k < n"
  shows "\<exists> a. a \<in> set (k_sets n k)"
  using assms
  unfolding k_sets_def
  by (simp add: combine_not_empty)

lemma k_sets_SetOrder:
  assumes "n \<le> n_max"
  assumes "x \<in> set (k_sets n i)" "y \<in> set (k_sets n j)" "i < j"
  shows "SetOrder y \<le> SetOrder x"
proof-
  have "abs_set x \<in> Set.Set ` {A. Finite_Set.card A = i \<and> A \<subseteq> {0..<n}}"
       "abs_set y \<in> Set.Set ` {A. Finite_Set.card A = j \<and> A \<subseteq> {0..<n}}"
    using assms k_sets_abs[of n i] k_sets_abs[of n j]
    by blast+
  hence "Finite_Set.card (FamilyAbs.elements (abs_set x)) = i"
        "Finite_Set.card (FamilyAbs.elements (abs_set y)) = j"
    unfolding abs_set_def
    using Set_inverse assms card_set_elements k_sets_card k_sets_inv
    by auto
  hence "card x = i" "card y = j"
    using assms k_sets_inv[of n x i] k_sets_inv[of n y j]
    by (auto simp add: card_abs)
  hence "card x < card y"
    using `i < j`
    by simp
  thus ?thesis
    unfolding less_eq_SetOrder_def 
    by (simp add: Let_def)
qed

(* combinations *)
definition combinations_OK where
  "combinations_OK combinations n \<longleftrightarrow> 
    (\<forall> k \<le> n. 
        let S = Mapping.lookup combinations k
         in S \<noteq> None \<and> the S = sort_key SetOrder (k_sets n k))"

lemma init_combinations_OK [simp]:
  shows "combinations_OK (init_combinations n) n"
  unfolding init_combinations_def combinations_OK_def
  by (auto simp add: Let_def Mapping.bulkload_tabulate Mapping.lookup_tabulate nth_append)

(* augment_set *)
lemma augment_set_inv:
  assumes "n \<le> n_max" "combinations_OK combinations n" "card s \<le> n"
  shows "\<forall> s' \<in> set (augment_set combinations n s). inv n s'"
  using assms
  unfolding augment_set_def combinations_OK_def
  by (auto simp add: Let_def k_sets_inv)

lemma augment_set_abs [simp]:
  assumes "n \<le> n_max" "inv n s" "combinations_OK combinations n" "card s \<le> n"
  shows "abs_set ` set (augment_set combinations n s) = 
         FamilyAbs.augment_set n (abs_set s)" (is "abs_set ` ?lhs = ?rhs")
proof safe
  fix x
  assume "x \<in> ?lhs"
  hence "x \<in> {x \<in> set (k_sets n (card s)). card x < card s \<or> card s = card x \<and> x < s} \<or>
         x \<in> (\<Union>x\<in>{1..<card s}. set (k_sets n x))" (is "x \<in> ?S1 \<or> x \<in> ?S2")
    using assms
    unfolding combinations_OK_def
    by (auto simp add: augment_set_def Let_def)
  thus "abs_set x \<in> ?rhs"
  proof
    assume "x \<in> ?S1"
    hence **: "abs_set x \<in> Set.Set ` {A. Finite_Set.card A = SetCard_class.card s \<and> A \<subseteq> {0..<n}}"
      using k_sets_abs[of n "card s"] `n \<le> n_max`
      by blast
    have "inv n x"
      using `x \<in> ?S1` k_sets_inv[OF `n \<le> n_max`]
      by auto
    show ?thesis
      unfolding FamilyAbs.augment_set_def
      using assms `inv n x` `x \<in> ?S1` **
      using less_set_same_card_abs[of n s x] k_sets_card
      using card_set_elements[symmetric] card_length_elements
      using Set_inverse subset_eq_atLeast0_lessThan_finite
      by (smt abs_set_def card_0_eq less_irrefl mem_Collect_eq set_elements)
  next
    assume "x \<in> ?S2"
    then obtain k where
      "1 \<le> k" "k < card s" "x \<in> set (k_sets n k)"
      by auto
    then obtain A where
      "abs_set x = Set.Set A" "Finite_Set.card A = k" "A \<subseteq> {0..<n}"
      using k_sets_abs[of n k] `n \<le> n_max`
      by blast
    thus "abs_set x \<in> FamilyAbs.augment_set n (abs_set s)"
      using `1 \<le> k` `k < card s` `inv n s` `n \<le> n_max`
      unfolding FamilyAbs.augment_set_def
      by (auto simp add: card_abs abs_set_def subset_eq_atLeast0_lessThan_finite Set_inverse finite_subset less_Set_def)
  qed
next
  fix x
  assume "x \<in> ?rhs"
  hence "Set.elements x \<noteq> {}" "Set.elements x \<subseteq> {0..<n}" and
        **: "card s = Finite_Set.card (Set.elements x) \<and> abs_set s < x \<or>
             Finite_Set.card (Set.elements x) < card s"
    using `n \<le> n_max` `inv n s`
    using leq_Set_geq_card[of "abs_set s"]
    unfolding FamilyAbs.augment_set_def less_eq_Set_def
    by (force simp add:  card_abs)+
  let ?k = "Finite_Set.card (Set.elements x)"
  have "x = Set.Set (Set.elements x)"
    by (simp add: elements_inverse)
  hence "x \<in> Set.Set ` {A. Finite_Set.card A = ?k \<and> A \<subseteq> {0..<n}}"
    using `Set.elements x \<subseteq> {0..<n}`
    by blast
  hence "x \<in> set (map abs_set (k_sets n ?k))"
    using k_sets_abs[of n] `n \<le> n_max`
    by simp
  then obtain x' where *: "x' \<in> set (k_sets n ?k)" "x = abs_set x'"
    by auto
  have "inv n x'"
    using *(1) k_sets_inv[OF `n \<le> n_max`]
    by simp
  show "x \<in> abs_set ` ?lhs"
    using **
  proof
    assume "card s = ?k \<and> abs_set s < x"
    thus ?thesis
      using assms * `inv n x'`
      unfolding combinations_OK_def
      using less_set_same_card_abs
      by (auto simp add: card_abs augment_set_def Let_def)
  next
    have "?k \<ge> 1"
      using `Set.elements x \<noteq> {}` `x = abs_set x'` `inv n x'` `n \<le> n_max`
      using abs_set_def not_less_eq_eq by fastforce
    assume "card s > ?k"
    hence "x' \<in> (\<Union>x\<in>{Suc 0..<SetCard_class.card s}. set (k_sets n x))"
      using `?k \<ge> 1` *
      by auto
    thus ?thesis
      using assms * `inv n x'` 
      unfolding combinations_OK_def
      by (simp add: augment_set_def Let_def)
  qed     
qed

lemma augment_set_sorted [simp]:
  assumes "n \<le> n_max" "combinations_OK combinations n" "card s \<le> n"
  shows "sorted (map SetOrder (augment_set combinations n s))"
proof-
  let ?C = "\<lambda> c. the (Mapping.lookup combinations c)"

  have "sorted (map SetOrder (filter (\<lambda>ss. SetOrder s < SetOrder ss) (?C (card s))))" (is "sorted ?L1")
    using assms
    unfolding combinations_OK_def
    by (simp add: Let_def linorder_class.sorted_filter)

  moreover

  have "sorted (concat (map (map SetOrder \<circ> (\<lambda>k. the (Mapping.lookup combinations k)))
                            (rev [Suc 0..<card s])))" (is "sorted (concat (map ?f ?I))")
  proof (rule sorted_concat)
    show "\<forall> xs \<in> set (map ?f ?I). sorted xs"
      using assms
      unfolding combinations_OK_def
      by (simp add: Let_def)
  next
    show "\<forall> i j. i < j \<and> j < length (map ?f ?I) \<longrightarrow> 
           (\<forall> x \<in> set (map ?f ?I ! i). \<forall> y \<in> set (map ?f ?I ! j). x \<le> y)" 
    proof safe
      fix i j x y
      assume "x \<in> set (map ?f ?I ! i)" "y \<in> set (map ?f ?I ! j)" "i < j" "j < length (map ?f ?I)"
      then obtain x' y' where
      "x = SetOrder x'" "y = SetOrder y'" 
      "x' \<in> set (k_sets n (Suc (SetCard_class.card s - Suc (Suc i))))"
      "y' \<in> set (k_sets n (Suc (SetCard_class.card s - Suc (Suc j))))"
        using assms k_sets_SetOrder[OF `n \<le> n_max`]
        unfolding combinations_OK_def
        by (auto simp add: rev_nth Let_def)
      thus "x \<le> y"
        using k_sets_SetOrder[OF `n \<le> n_max`, of y' "Suc (SetCard_class.card s - Suc (Suc j))"
                                                  x' "Suc (SetCard_class.card s - Suc (Suc i))"]
        using `i < j` `j < length (map ?f ?I)`
        by simp
    qed
  qed

  moreover

  have "\<forall> x \<in> set ?L1. \<forall> y \<in> set (concat (map ?f ?I)). x \<le> y"
  proof safe
    fix x y
    assume "x \<in> set ?L1"
    then obtain x' where "x' \<in> set (k_sets n (card s))" "x = SetOrder x'"
      using assms
      unfolding combinations_OK_def Let_def
      by auto
    moreover
    assume "y \<in> set (concat (map ?f ?I))"
    then obtain c y' where "c \<in>{1..<card s}" "y = SetOrder y'" "y' \<in> set (k_sets n c)"
      using assms
      unfolding combinations_OK_def Let_def
      by auto
    ultimately
    show "x \<le> y"
      using k_sets_card[OF `n \<le> n_max`, of x' "card s"]
      using k_sets_card[OF `n \<le> n_max`, of y' c]
      unfolding less_eq_SetOrder_def 
      by (simp add: Let_def)
  qed

  ultimately

  show ?thesis
    using assms
    unfolding augment_set_def
    by (simp add: Let_def sorted_append map_concat)
qed

lemma augment_set_distinct [simp]:
  assumes "n \<le> n_max" "combinations_OK combinations n" "card s \<le> n"
  shows "distinct (augment_set combinations n s)"
proof-
  let ?P = "(\<lambda>s'. SetOrder s < SetOrder s')"
  let ?l1 = "the (Mapping.lookup combinations (SetCard_class.card s))"
  let ?L1 = "filter ?P ?l1"
  let ?I = "rev [1..<SetCard_class.card s]"
  let ?f = "\<lambda>k. the (Mapping.lookup combinations k)"
  let ?L2 = "concat (map ?f ?I)"
  show ?thesis
    unfolding augment_set_def Let_def
  proof (subst distinct_append, safe)
    have "distinct (k_sets n (card s))"
      by (simp add: assms(1))
    thus "distinct ?L1"
      using `combinations_OK combinations n` `card s \<le> n`
      unfolding combinations_OK_def Let_def
      by (simp add: Let_def)
  next
  show "distinct ?L2"
  proof (rule distinct_concat)
    show "distinct (map ?f ?I)"
    proof (subst distinct_map, safe)
      show "distinct ?I"
        by simp
    next
      show "inj_on ?f (set ?I)"
        unfolding inj_on_def
      proof safe
        fix x y
        assume *: "x \<in> set ?I" "y \<in> set ?I"
        assume "?f x = ?f y"
        hence "sort_key SetOrder (k_sets n x) = sort_key SetOrder (k_sets n y)"
          using `combinations_OK combinations n` `card s \<le> n` *
          unfolding combinations_OK_def Let_def
          by auto
        then obtain a where
          "a \<in> set (k_sets n x)" "a \<in> set (k_sets n y)"
          using k_sets_non_empty[of x n] * `card s \<le> n`
          by (metis atLeastLessThan_iff atLeastLessThan_upt less_le_trans set_rev set_sort)
        thus "x = y"
          using `n \<le> n_max` k_sets_inj
          by simp
      qed
    qed
  next
    fix ys
    assume "ys \<in> set (map ?f ?I)"
    thus "distinct ys"
        using `combinations_OK combinations n` `card s \<le> n`
        unfolding combinations_OK_def Let_def
        by (auto simp add: assms)
    next
      fix xs ys
      assume "xs \<in> set (map ?f ?I)" "ys \<in> set (map ?f ?I)" "xs \<noteq> ys"
      thus "set xs \<inter> set ys = {}"
        using `combinations_OK combinations n` `card s \<le> n` `n \<le> n_max`
        using k_sets_inj[of n]
        unfolding combinations_OK_def Let_def
        by force
    qed
  next
    fix x
    assume "x \<in> set ?L1" "x \<in> set ?L2"
    thus "x \<in> {}"
      using `combinations_OK combinations n` `card s \<le> n` `n \<le> n_max`
      using k_sets_inj[of n]
      unfolding combinations_OK_def Let_def
      by (force simp add: Let_def)
  qed
qed

(* augmenting_sets *)
definition augmenting_sets_OK where
  "augmenting_sets_OK augmenting_sets n \<longleftrightarrow> 
    (\<forall> s \<in> set (powerset n). 
      (let S = Mapping.lookup augmenting_sets s
        in S \<noteq> None \<and> set (map abs_set (the S)) = FamilyAbs.augment_set n (abs_set s)))"

lemma init_augmenting_sets_OK [simp]:
  assumes "n \<le> n_max"
  shows "augmenting_sets_OK (init_augmenting_sets n) n"
  using assms                                                          
  unfolding augmenting_sets_OK_def init_augmenting_sets_def Let_def
  by (simp add: Mapping.lookup_tabulate powerset_elements_card powerset_set)

definition augmenting_sets_inv where
  "augmenting_sets_inv augmenting_sets n \<longleftrightarrow>
    (\<forall> s \<in> set (powerset n). (\<forall> s' \<in> set (the (Mapping.lookup augmenting_sets s)). inv n s'))"

lemma init_augmenting_sets_inv [simp]:
  assumes "n \<le> n_max"
  shows "augmenting_sets_inv (init_augmenting_sets n) n"
  unfolding augmenting_sets_inv_def
  unfolding init_augmenting_sets_def Let_def
  using assms init_combinations_OK[of n] powerset_elements_card[OF assms]
  using augment_set_inv[OF assms, of "init_combinations n"]
  by (auto simp add: Mapping.lookup_tabulate)

definition augmenting_sets_sorted_distinct where
 "augmenting_sets_sorted_distinct augmenting_sets n \<longleftrightarrow> 
    (\<forall> s \<in> set (powerset n). 
      let S = the (Mapping.lookup augmenting_sets s) 
       in sorted (map SetOrder S) \<and> distinct S)"

lemma augmenting_sets_sorted_distinct [simp]:
  assumes "n \<le> n_max" "combinations_OK (init_combinations n) n"
  shows "augmenting_sets_sorted_distinct (init_augmenting_sets n) n"
  using assms
  unfolding augmenting_sets_sorted_distinct_def Let_def
proof safe
  fix s
  assume "s \<in> set (powerset n)"
  thus "sorted (map SetOrder (the (Mapping.lookup (init_augmenting_sets n) s)))"
    using assms
    unfolding init_augmenting_sets_def
    using powerset_elements_card
    by (simp add: Mapping.lookup_tabulate)
next
  fix s
  assume "s \<in> set (powerset n)"
  thus "distinct (the (Mapping.lookup (init_augmenting_sets n) s))"
    using assms
    unfolding init_augmenting_sets_def
    using powerset_elements_card
    by (simp add: Mapping.lookup_tabulate)
qed

lemma augmenting_sets_less:
  assumes "n \<le> n_max"
  assumes "augmenting_sets_OK augmenting_sets n"
  assumes "augmenting_sets_inv augmenting_sets n"
  shows "\<forall> s \<in> set (powerset n). 
         (let S = Mapping.lookup augmenting_sets s
           in \<forall> s' \<in> set (the S). SetOrder s' > SetOrder s)" (is "\<forall> s \<in> ?p. ?P s")
proof safe
  fix s
  let ?S = "the (Mapping.lookup augmenting_sets s)"
  assume "s \<in> set (powerset n)"
  hence *: "FamilyAbs.augment_set n (abs_set s) = set (map abs_set ?S)"
            "\<forall> s' \<in> set ?S. inv n s'"
    using assms
    unfolding augmenting_sets_inv_def
    unfolding augmenting_sets_OK_def
    unfolding Let_def
    by blast+
  show "?P s"
    unfolding Let_def
  proof safe
    fix s'
    assume "s' \<in> set ?S"
    hence "abs_set s' \<in> FamilyAbs.augment_set n (abs_set s)"
      using *
      by simp
    hence "abs_set s' > abs_set s"
      unfolding FamilyAbs.augment_set_def 
      by (auto simp add: abs_set_def)
    thus "SetOrder s' > SetOrder s"
      using less_set_abs[OF `n \<le> n_max`, of s s'] * `s' \<in> set ?S` `s \<in> set (powerset n)`
      using powerset_set[OF `n \<le> n_max`]
      by (simp add: Let_def)
  qed
qed

lemma augmenting_sets_card_less_eq:
  assumes "n \<le> n_max"
  assumes "augmenting_sets_OK augmenting_sets n"
  assumes "augmenting_sets_inv augmenting_sets n"
  shows "\<forall> s \<in> set (powerset n). 
         (let S = Mapping.lookup augmenting_sets s
           in \<forall> s' \<in> set (the S). card s' \<le> card s)" (is "\<forall> s \<in> ?p. ?P s")
  using augmenting_sets_less[OF assms] 
  by (force simp add: Let_def)


(* extend_family *)
lemma extend_family_sets:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n s"
  shows "set (sets (all_sets (extend_family n powerset_by_card set_perms F s))) = 
         set (sets (all_sets F)) \<union> {s}"
  unfolding extend_family_def Let_def
  using assms
  by simp

lemma extend_family_abs [simp]:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n s"
  shows "abs_fam (all_sets (extend_family n powerset_by_card set_perms F s)) = 
         FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s)"
  using add_set_abs[of n s F] assms
  unfolding extend_family_def Let_def
  by auto

lemma extend_family_inv [simp]:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n s"
  shows "inv_f n (all_sets (extend_family n powerset_by_card set_perms F s))"
  using assms
  unfolding extend_family_def Let_def
  using FamilyImpl.add_inv FamilyImpl.add_set_def FamilyImpl_axioms
  by fastforce

lemma extend_family_inv_r [simp]:
  assumes "n \<le> n_max" "inv_f n (reduced F)" "inv n s"
  shows "inv_f n (reduced (extend_family n powerset_by_card set_perms F s))"
  using assms
  unfolding extend_family_def Let_def
  by simp

lemma extend_family_last_set [simp]:
  shows "last_set (extend_family n powerset_by_card set_perms F s) = s"
  unfolding extend_family_def Let_def
  by (auto simp add: filter_perms_def)

lemma extend_family_perms_filtered:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)" 
  assumes "inv n s" "card s \<le> card (last_set F)"
  assumes "inv n (last_set F)"
          "last_set F \<in> set (sets (all_sets F))"
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "set_perms_OK set_perms n"
          "group_powerset_by_card_OK powerset_by_card n"
          "group_powerset_by_card_inv powerset_by_card n"
  assumes "perms_filtered F n"
  shows "perms_filtered (extend_family n powerset_by_card set_perms F s) n"
proof (cases "card s = min_card (abs_fam (all_sets F))")
  case True
  thus ?thesis
    using assms
    unfolding perms_filtered_def extend_family_def Let_def
    using add_set_abs[of n s F] above_min_card_sets_add_set
    using perm_fixes_add_set_1[of "abs_fam (all_sets F)"]
    using last_set_Max_min_card
    by (simp add: card_abs powerset_set above_card_sets_add_set)
next
  case False

  {
    fix pp
    assume "pp < fact n"

    have "set_card (abs_set s) < min_card (abs_fam (all_sets F))"
      using False assms(1) assms(3) assms(4) assms(5) assms(6) assms(7) card_abs last_set_Max_min_card
      by auto
    moreover
    have "FamilyAbs.sets (abs_fam (all_sets F)) \<noteq> {}"
      using assms
      by auto
    moreover
    have "permute [0..<n] ! pp \<in> set (permute [0..<n])"
      using `pp < fact n`
      by auto
    moreover
    have "\<forall>s\<in>FamilyAbs.sets (abs_fam (all_sets F)). FamilyAbs.elements s \<subseteq> {0..<n}"
      using assms
      by (auto simp add: abs_fam_def abs_set_def set_elements Family_inverse Set_inverse)
    moreover
    have "FamilyAbs.elements (abs_set s) \<subseteq> {0..<n}"
      using assms
      by (simp add: abs_set_def set_elements Family_inverse Set_inverse)
    moreover
    have "abs_fam (all_sets (add_set F s)) = FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s)"
      using assms
      by simp
    moreover
    let ?p = "permute [0..<n] ! pp"
    let ?F = "abs_fam (all_sets (local.add_set F s))"
    have "perm_fixes_card n powerset_by_card set_perms (add_set F s) (card (last_set F)) pp \<longleftrightarrow>
          FamilyAbs.perm_fixes_subset ?p (FamilyAbs.sets ?F) (eq_card_sets ?F (min_card_above_min_card ?F))"
    proof-
      have "card (last_set F) = min_card (abs_fam (all_sets F))"
        using assms
        using last_set_Max_min_card
        by auto
      moreover
      hence "card s < card (last_set F)"
        using assms False
        by simp
      ultimately       
      have "card (last_set F) = min_card_above_min_card ?F"
        using \<open>?F = FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s)\<close>
        using \<open>set_card (abs_set s) < min_card (abs_fam (all_sets F))\<close>
        using \<open>FamilyAbs.sets (abs_fam (all_sets F)) \<noteq> {}\<close>
        using min_card_above_min_card_add_set[of _ "abs_set s"]
        using `card (last_set F) = min_card (abs_fam (all_sets F))`
        by (metis abs_fam_finite)
      thus ?thesis
        using assms `pp < fact n` False
        using perm_fixes_card_abs
        by (metis (full_types, hide_lams) FamilyImpl.add_set_inv FamilyImpl_axioms Un_iff add_set_sets)
    qed
    ultimately
    have "FamilyAbs.perm_fixes (permute [0..<n] ! pp) (above_min_card_sets (abs_fam (all_sets F))) \<and>
          perm_fixes_card n powerset_by_card set_perms (add_set F s) (card (last_set F)) pp \<longleftrightarrow>
          FamilyAbs.perm_fixes (permute [0..<n] ! pp) (above_min_card_sets (abs_fam (all_sets (add_set F s))))" 
      using perm_fixes_above_eq_iff
      by auto
  }
  thus ?thesis
    using assms False last_set_Max_min_card
    unfolding perms_filtered_def extend_family_def Let_def
    by auto
qed

lemma extend_family_hd_perms:
  assumes "n \<le> n_max"
  assumes "inv n s" "card s \<le> card (last_set F)"
  assumes "inv_f n (all_sets F)" "inv n (last_set F)"
     "last_set F \<in> set (sets (all_sets F))" 
     "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
     "set_perms_OK set_perms n"
     "group_powerset_by_card_OK powerset_by_card n"
     "group_powerset_by_card_inv powerset_by_card n"
  assumes "hd_perms F"
  shows "hd_perms (extend_family n powerset_by_card set_perms F s)"
proof (cases "card (last_set F) = card s")
  case True
  thus ?thesis
    unfolding extend_family_def Let_def
    using assms
    by (auto simp add: hd_perms_def)
next
  case False
  have "\<exists>s'\<in>set (sets (all_sets (add_set F s))). card s' = card (last_set F)"
    using assms False
    by force
  hence "hd_perms (filter_perms n powerset_by_card set_perms (local.add_set F s) (card (last_set F)))"
    using assms
    by (subst filter_perms_hd_perms, simp_all add: hd_perms_def)
  thus ?thesis
    using False assms
    unfolding extend_family_def Let_def
    by (auto simp add: Let_def)
qed

lemma extend_family_card_last_set:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv n s" "Family.sets (abs_fam (all_sets F)) \<noteq> {}"
  assumes "card (last_set F) = min_card (abs_fam (all_sets F))" "card s \<le> card (last_set F)"
  shows "card (last_set (extend_family n powerset_by_card set_perms F s)) = 
         min_card (abs_fam (all_sets (add_set F s)))"
  using assms 
  by (simp add: card_abs)

lemma extend_family_reduced [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)" "inv n s"
  assumes "\<forall> s' \<in> FamilyAbs.sets (abs_fam (all_sets F)). s' < abs_set s"
  assumes "inv_f n (reduced F)"
  assumes "FamilyAbs.sets (abs_fam (reduced F)) = FamilyAbs.reduced (abs_fam (all_sets F))"
  shows "FamilyAbs.sets (abs_fam (reduced (extend_family n powerset_by_card set_perms F s))) =
         FamilyAbs.reduced (FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s))"
  using assms
  unfolding extend_family_def Let_def
  by simp

(* augment *)                                 
lemma augment_inv:
  assumes "n \<le> n_max" "inv_f n (all_sets F)"
  assumes "augmenting_sets_inv augmenting_sets n"
  assumes "inv n (last_set F)"
  shows "\<forall> F \<in> set (augment n augmenting_sets powerset_by_card set_perms F). inv_f n (all_sets F)"
  using assms
  unfolding augment_def augmenting_sets_inv_def
  by (simp add: powerset_set)

lemma augment_inv_r:
  assumes "n \<le> n_max" "inv_f n (reduced F)"
  assumes "augmenting_sets_inv augmenting_sets n"
  assumes "inv n (last_set F)"
  shows "\<forall> F \<in> set (augment n augmenting_sets powerset_by_card set_perms F). inv_f n (reduced F)"
  using assms
  unfolding augment_def augmenting_sets_inv_def
  by (simp add: powerset_set)

lemma augment_perms_filtered:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
  assumes "last_set F \<in> set (sets (all_sets F))"
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "augmenting_sets_OK augmenting_sets n"
  assumes "augmenting_sets_inv augmenting_sets n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "set_perms_OK set_perms n"

  assumes "perms_filtered F n"
  shows "\<forall> F \<in> set (augment n augmenting_sets powerset_by_card set_perms F). perms_filtered F n"
proof-
  {
    fix Fa
    assume *: "Fa \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
           "is_union_closed' F Fa"
    have "last_set F \<in> set (powerset n)"
      by (simp add: assms powerset_set)
    hence "inv n Fa"
      using assms *
      unfolding augmenting_sets_inv_def
      by simp
    moreover
    have "card Fa \<le> card (last_set F)"
      using augmenting_sets_less
      using "*"(1) \<open>last_set F \<in> set (powerset n)\<close> assms augmenting_sets_card_less_eq
      by auto
    ultimately
    have "perms_filtered (extend_family n powerset_by_card set_perms F Fa) n"
      using assms 
      by (subst extend_family_perms_filtered, simp_all)
  }
  thus ?thesis
    unfolding augment_def
    by auto
qed

lemma augment_hd_perms:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
  assumes "last_set F \<in> set (sets (all_sets F))"
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "augmenting_sets_OK augmenting_sets n"
  assumes "augmenting_sets_inv augmenting_sets n"
  assumes "group_powerset_by_card_OK powerset_by_card n"
  assumes "group_powerset_by_card_inv powerset_by_card n"
  assumes "set_perms_OK set_perms n"

  assumes "hd_perms F"
  shows "\<forall> F \<in> set (augment n augmenting_sets powerset_by_card set_perms F). hd_perms F"
proof-
  {
    fix Fa
    assume *: "Fa \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
           "is_union_closed' F Fa"
    have "last_set F \<in> set (powerset n)"
      by (simp add: assms powerset_set)
    hence "inv n Fa"
      using assms *
      unfolding augmenting_sets_inv_def
      by simp
    moreover
    have "card Fa \<le> card (last_set F)"
      using augmenting_sets_less
      using "*"(1) \<open>last_set F \<in> set (powerset n)\<close> assms augmenting_sets_card_less_eq
      by auto
    ultimately
    have "hd_perms (extend_family n powerset_by_card set_perms F Fa)"
      using assms 
      by (subst extend_family_hd_perms, simp_all)
  }
  thus ?thesis
    unfolding augment_def
    by auto
qed

lemma augment_sorted:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
          "abs_set (last_set F) = Max (Family.sets (abs_fam (all_sets F)))"
  assumes "augmenting_sets_sorted_distinct augmenting_sets n"
          "augmenting_sets_inv augmenting_sets n"
          "augmenting_sets_OK augmenting_sets n"
  shows "sorted (map (abs_fam o all_sets) (augment n augmenting_sets powerset_by_card set_perms F))"
proof-
  have "sorted (map (abs_fam \<circ> all_sets \<circ> extend_family n powerset_by_card set_perms F \<circ> Set)
               (map SetOrder (the (Mapping.lookup augmenting_sets (last_set F)))))" (is "sorted (map ?f ?l)")
  proof (rule sorted_map_mono)
    show "sorted ?l"
      using `augmenting_sets_sorted_distinct augmenting_sets n`
      unfolding augmenting_sets_sorted_distinct_def Let_def
      by (simp add: assms powerset_set)

    show "\<forall> x \<in> set ?l. \<forall> y \<in> set ?l. x \<le> y \<longrightarrow> ?f x \<le> ?f y"
    proof safe
      fix x y
      assume "x \<in> set ?l" "y \<in> set ?l" "x \<le> y"
      then obtain x' y' where *:
        "x = SetOrder x'" "x' \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
        "y = SetOrder y'" "y' \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
        by auto
      hence "inv n x'" "inv n y'"
        using `augmenting_sets_inv augmenting_sets n`
        unfolding augmenting_sets_inv_def `inv n (last_set F)`
        using assms powerset_set 
        by fastforce+
      hence "abs_set x' \<le> abs_set y'"
        using `x \<le> y` less_set_abs[OF `n \<le> n_max`] `x = SetOrder x'` `y = SetOrder y'`
        unfolding less_eq_Set_def less_eq_SetOrder_def
        by auto
      have "FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set x') \<le>
             FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set y')"
      proof (rule less_eq_Family_add_set)
        show "finite (Family.sets (abs_fam (all_sets F)))"
          by (simp add: abs_fam_def)
      next
        have "SetOrder (last_set F) < SetOrder x'" "SetOrder (last_set F) < SetOrder y'"
          using augmenting_sets_less * `inv n (last_set F)`
          using assms powerset_set
          by auto
        hence "abs_set (last_set F) < abs_set x'" "abs_set (last_set F) < abs_set y'"
          using `n \<le> n_max` less_set_abs `inv n x'` `inv n y'` `inv n (last_set F)`
          by auto
        thus "\<forall> f \<in> Family.sets (abs_fam (all_sets F)). f < abs_set x'"
             "\<forall> f \<in> Family.sets (abs_fam (all_sets F)). f < abs_set y'"
          using `abs_set (last_set F) = Max (Family.sets (abs_fam (all_sets F)))`
          by (metis Max_less_iff empty_iff finite_sets)+
      next
        show "abs_set x' \<le> abs_set y'"
          by fact
      qed
      thus "?f x \<le> ?f y"
        using extend_family_abs[OF `n \<le> n_max` `inv_f n (all_sets F)` `inv n x'`]
        using extend_family_abs[OF `n \<le> n_max` `inv_f n (all_sets F)` `inv n y'`]
        using `x = SetOrder x'` `y = SetOrder y'`
        by simp
    qed                              
  qed
  thus ?thesis
    by (simp add: augment_def linorder_class.sorted_filter comp_def)
qed

lemma augment_distinct:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "augmenting_sets_sorted_distinct augmenting_sets n"
          "augmenting_sets_OK augmenting_sets n"
          "augmenting_sets_inv augmenting_sets n"
  shows "distinct (map (abs_fam o all_sets) (augment n augmenting_sets powerset_by_card set_perms F))" (is "distinct ?lhs")
proof-
  let ?A = "augment n augmenting_sets powerset_by_card set_perms F"
  have "?lhs = map abs_fam (map all_sets ?A)"
    by simp
  moreover
  have "distinct (map abs_fam (map all_sets ?A))"
  proof (subst distinct_map, safe)
    let ?S = "the (Mapping.lookup augmenting_sets (last_set F))"
    let ?f = "all_sets \<circ> extend_family n powerset_by_card set_perms F"
    have "map all_sets ?A = map ?f (filter (is_union_closed' F) ?S)"
      unfolding augment_def
      by simp

    moreover

    have "distinct (map ?f (filter (is_union_closed' F) ?S))"
    proof (subst distinct_map, safe)
      let ?S = "the (Mapping.lookup augmenting_sets (last_set F))"
      let ?f = "all_sets \<circ> extend_family n powerset_by_card set_perms F"
      show "inj_on ?f (set (filter (is_union_closed' F) ?S))"
        unfolding inj_on_def
      proof safe
        fix x y
        assume *: "x \<in> set (filter (is_union_closed' F) ?S)" "y \<in> set (filter (is_union_closed' F) ?S)"
        assume "?f x = ?f y"
        hence "abs_fam (?f x) = abs_fam (?f y)"
          by simp
        moreover
        have "last_set F \<in> set (powerset n)"
          using assms
          by (simp add: powerset_set)
        hence "inv n x" "inv n y"
          using `augmenting_sets_inv augmenting_sets n` * 
          unfolding augmenting_sets_inv_def
          by auto
        moreover
        have "x \<in> set ?S" "y \<in> set ?S"
          using *
          by auto
        hence "SetOrder x > SetOrder (last_set F)" "SetOrder y > SetOrder (last_set F)"
          using augmenting_sets_less[OF `n \<le> n_max` `augmenting_sets_OK augmenting_sets n` `augmenting_sets_inv augmenting_sets n`]
          using `last_set F \<in> set (powerset n)`
          by (simp_all add: Let_def)
        hence "abs_set x > abs_set (last_set F)" "abs_set y > abs_set (last_set F)"
          using less_set_abs[OF `n \<le> n_max` `inv n (last_set F)` `inv n x`]
          using less_set_abs[OF `n \<le> n_max` `inv n (last_set F)` `inv n y`]
          by simp_all

        have "finite (Family.sets (abs_fam (all_sets F)))"
          by (simp add: abs_fam_def)

        hence "abs_set x \<notin> FamilyAbs.sets (abs_fam (all_sets F))" "abs_set y \<notin> FamilyAbs.sets (abs_fam (all_sets F))"
          using `abs_set x > abs_set (last_set F)` `abs_set y > abs_set (last_set F)`
          using `abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))`
          using Max_ge[of "Family.sets (abs_fam (all_sets F))" "abs_set x"]
          using Max_ge[of "Family.sets (abs_fam (all_sets F))" "abs_set y"]
          by (metis leD)+
        ultimately
        have "abs_set x = abs_set y"
          using assms
          using extend_family_abs[of n F x powerset_by_card set_perms]
          using extend_family_abs[of n F y powerset_by_card set_perms]    
          by (auto simp add: FamilyAbs.add_set_def)
             (metis Family_inverse finite.insertI finite_sets insert_iff mem_Collect_eq)
        thus "x = y"
          using abs_set_inj[of n x y] `n \<le> n_max` `inv n x` `inv n y`
          by simp
      qed

      show "distinct (filter (is_union_closed' F) ?S)"
      proof (rule distinct_filter)   
        show "distinct ?S"
          using `augmenting_sets_sorted_distinct augmenting_sets n` 
          unfolding augmenting_sets_sorted_distinct_def
          by (metis assms(1) assms(3) mem_Collect_eq powerset_set)
      qed
    qed

    ultimately

    show "distinct (map all_sets ?A)"
      by simp

    show "inj_on abs_fam (set (map all_sets ?A))"
      unfolding inj_on_def
    proof safe
      fix x y
      assume "x \<in> set (map all_sets ?A)" "y \<in> set (map all_sets ?A)" "abs_fam x = abs_fam y"
      hence "inv_f n x" "inv_f n y"
        using assms
        using augment_inv[of n F augmenting_sets powerset_by_card]
        by auto
      thus "x = y"
        using abs_fam_inj[of n x y] `abs_fam x = abs_fam y` `n \<le> n_max`
        by simp
    qed
  qed
  ultimately
  show "distinct ?lhs"
    by simp
qed

lemma un_abs [simp]:
  assumes "n \<le> n_max" "inv n s1" "inv n s2"
  shows "abs_set (un s1 s2) = FamilyAbs.union (abs_set s1) (abs_set s2)"
  using assms
  unfolding abs_set_def
  by (simp add: FamilyAbs.union_def Set_inverse)

lemma is_union_closed'_abs:
  assumes "n \<le> n_max" "inv_f n (all_sets F)" "inv_f n (reduced F)" "inv n s"
  assumes *: "FamilyAbs.sets (abs_fam (reduced F)) =
              FamilyAbs.reduced (abs_fam (all_sets F))"
  shows "is_union_closed' F s \<longleftrightarrow> (\<forall>s' \<in> FamilyAbs.reduced (abs_fam (all_sets F)). (FamilyAbs.union s' (abs_set s)) \<in> FamilyAbs.sets (abs_fam (all_sets F)))"
proof-
  have "(\<forall>s'\<in>set (sets (FamilyCache.reduced F)). contains_set F (un s' s)) \<longleftrightarrow>
        (\<forall>s'\<in>FamilyAbs.sets (abs_fam (reduced F)). FamilyAbs.union s' (abs_set s) \<in> FamilyAbs.sets (abs_fam (all_sets F)))"
    using assms(1-4)
    by (simp add: abs_fam_def Family_inverse)
  thus ?thesis
    using *
    unfolding is_union_closed'_def list_all_iff
    by simp
qed


lemma augment_abs [simp]:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)" 
          "union_closed (abs_fam (all_sets F))"
  assumes "inv n (last_set F)" 
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
          "last_set F \<in> set (sets (all_sets F))"
  assumes "inv_f n (reduced F)"
          "FamilyAbs.sets (abs_fam (reduced F)) = FamilyAbs.reduced (abs_fam (all_sets F))"
  assumes "augmenting_sets_OK augmenting_sets n"
          "augmenting_sets_inv augmenting_sets n"
          "augmenting_sets_sorted_distinct augmenting_sets n"
  shows "map (abs_fam o all_sets) (augment n augmenting_sets powerset_by_card set_perms F) = 
         FamilyAbs.augment n (abs_fam (all_sets F))" (is "?lhs = ?rhs")
proof (rule sorted_distinct_set_unique)
  show "sorted ?lhs"
    using assms augment_sorted
    by simp
next
  show "distinct ?lhs"
    using assms augment_distinct
    by simp
next
  show "sorted ?rhs"
    unfolding FamilyAbs.augment_def Let_def
    by simp
next
  show "distinct ?rhs"
    unfolding FamilyAbs.augment_def Let_def
    by simp
next
  let ?A = "augment n augmenting_sets powerset_by_card set_perms F"
  let ?S = "the (Mapping.lookup augmenting_sets (last_set F))"

  have "last_set F \<in> set (powerset n)"
    using assms
    by (simp add: powerset_set)
  show "set ?lhs = set ?rhs"
  proof safe
    fix x
    assume "x \<in> set ?lhs"
    then obtain x' where
      "x' \<in> set ?A"
      "x = abs_fam (all_sets x')"
      by auto
    then obtain x'' where
      "x'' \<in> set ?S" "is_union_closed' F x''"
      "x' = extend_family n powerset_by_card set_perms F x''"
      unfolding augment_def
      by auto
    hence "abs_set x'' \<in> FamilyAbs.augment_set n (abs_set (last_set F))"
          "inv n x''"
      using `augmenting_sets_OK augmenting_sets n` `augmenting_sets_inv augmenting_sets n`
      using `last_set F \<in> set (powerset n)`
      unfolding augmenting_sets_OK_def augmenting_sets_inv_def
      by (auto simp add: Let_def)

    moreover

    have "\<forall>s'\<in>FamilyAbs.sets (abs_fam (all_sets F)). s' < abs_set x''"
      by (metis Max_less_iff abs_fam_finite assms(5) augment_set_gt calculation(1) empty_iff)

    hence "union_closed (FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set x''))"
      using `is_union_closed' F x''` `union_closed (abs_fam (all_sets F))`
      using assms `inv n x''` is_union_closed'_abs
      by (simp add: union_closed_iff_reduced)

    ultimately

    show "x \<in> set ?rhs"
      using assms
      using `x' = extend_family n powerset_by_card set_perms F x''`[symmetric]
      using `x = abs_fam (all_sets x')`[symmetric]
      using extend_family_abs[of n F x'' powerset_by_card set_perms]
      unfolding FamilyAbs.augment_def Let_def
      by (auto simp add: FamilyAbs.add_set_def augment_set_finite)
  next
    fix x
    assume "x \<in> set ?rhs"
    then obtain x' where
      "x' \<in> FamilyAbs.augment_set n (abs_set (last_set F))"
      "x = FamilyAbs.add_set (abs_fam (all_sets F)) x'"
      "union_closed x"
      using assms
      unfolding FamilyAbs.augment_def Let_def
      by (auto simp add: augment_set_finite FamilyAbs.add_set_def)
    then obtain x'' where 
      "x' = abs_set x''" 
      "x'' \<in> set ?S"
      using `augmenting_sets_OK augmenting_sets n` `last_set F \<in> set (powerset n)`
      unfolding augmenting_sets_OK_def 
      by (auto simp add: Let_def)

    moreover

    have "inv n x''"
      using FamilyImpl.augmenting_sets_inv_def FamilyImpl_axioms \<open>last_set F \<in> set (powerset n)\<close> assms(10) `x'' \<in> set ?S`
      by blast

    have "\<forall>s'\<in>FamilyAbs.sets (abs_fam (all_sets F)). s' < abs_set x''"
      by (metis Max_ge \<open>x' \<in> FamilyAbs.augment_set n (abs_set (last_set F))\<close> abs_fam_finite assms(5) augment_set_gt calculation(1) le_less_trans)

    hence "is_union_closed' F x''"
      using `union_closed x` `x = FamilyAbs.add_set (abs_fam (all_sets F)) x'` `x' = abs_set x''`
      using assms `inv n x''` is_union_closed'_abs
      by (simp add: union_closed_iff_reduced[symmetric])

    moreover

    have "inv n x''"
      using `x'' \<in> set ?S`
      using `augmenting_sets_inv augmenting_sets n` `last_set F \<in> set (powerset n)`
      unfolding augmenting_sets_inv_def
      by simp
    hence "x = abs_fam (all_sets (extend_family n powerset_by_card set_perms F x''))"
      using assms `x = FamilyAbs.add_set (abs_fam (all_sets F)) x'` `x' = abs_set x''`
      using extend_family_abs[of n F x'' powerset_by_card set_perms]
      by simp

    ultimately

    show "x \<in> set ?lhs"
      unfolding augment_def
      by simp
  qed
qed


lemma augment_reduced:
  assumes "n \<le> n_max" 
  assumes "inv_f n (all_sets F)" 
          "union_closed (abs_fam (all_sets F))"
  assumes "inv n (last_set F)" 
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
          "last_set F \<in> set (sets (all_sets F))"
  assumes "inv_f n (reduced F)"
          "FamilyAbs.sets (abs_fam (reduced F)) = FamilyAbs.reduced (abs_fam (all_sets F))"
  assumes "augmenting_sets_OK augmenting_sets n"
          "augmenting_sets_inv augmenting_sets n"
          "augmenting_sets_sorted_distinct augmenting_sets n"
  shows "\<forall> F' \<in> set (augment n augmenting_sets powerset_by_card set_perms F).
         FamilyAbs.sets (abs_fam (reduced F')) = FamilyAbs.reduced (abs_fam (all_sets F'))" (is "\<forall> F' \<in> set (?A F'). ?lhs F' = ?rhs F'")
proof
  have "last_set F \<in> set (powerset n)"
    using assms
    by (simp add: powerset_set)

  fix F'
  assume "F' \<in> set (?A F)"
  then obtain s where s:
    "F' = extend_family n powerset_by_card set_perms F s"
    "s \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))" "is_union_closed' F s"
    unfolding augment_def
    by auto
  moreover
  hence " \<forall>s'\<in>FamilyAbs.sets (abs_fam (all_sets F)). s' < abs_set s"
    using \<open>last_set F \<in> set (powerset n)\<close>
    by (smt Max_ge abs_fam_finite assms augmenting_sets_inv_def augmenting_sets_less less_set_abs not_le order_trans)
  moreover
  have "FamilyAbs.add_set (abs_fam (all_sets F)) (abs_set s) =
        abs_fam (all_sets (extend_family n powerset_by_card set_perms F s))"
    using \<open>last_set F \<in> set (powerset n)\<close> assms augmenting_sets_inv_def s
    by auto
  ultimately
  show "?lhs F' = ?rhs F'"
    using assms `last_set F \<in> set (powerset n)` s
    by (simp add: augmenting_sets_inv_def)
qed

lemma augment_last_set_inv:
  assumes "n \<le> n_max"
  assumes "inv n (last_set F)"
  assumes "augmenting_sets_inv augmenting_sets n"
  shows "\<forall> F \<in> set (augment n augmenting_sets powerset_by_card set_perms F). inv n (last_set F)"
proof-
  have "last_set F \<in> set (powerset n)"
    using assms powerset_set
    by auto
  thus ?thesis
    using assms
    unfolding augment_def augmenting_sets_inv_def
    by simp
qed

lemma augment_last_set_in_family:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
  assumes "augmenting_sets_inv augmenting_sets n"
  shows "\<forall> F' \<in> set (augment n augmenting_sets powerset_by_card set_perms F). last_set F' \<in> set (sets (all_sets F'))"
proof safe
  fix F'
  assume F': "F' \<in> set (augment n augmenting_sets powerset_by_card set_perms F)"
  then obtain s where
    "F' = extend_family n powerset_by_card set_perms F s"
    "s \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
    unfolding augment_def
    by auto
  moreover
  have "last_set F \<in> set (powerset n)"
    using assms powerset_set
    by auto
  ultimately
  show "last_set F' \<in> set (sets (all_sets F'))"
    using assms extend_family_sets[OF `n \<le> n_max`, of F s powerset_by_card set_perms]
    unfolding augment_def augmenting_sets_inv_def
    by auto
qed

lemma augment_last_set_Max:
  assumes "n \<le> n_max"
  assumes "inv_f n (all_sets F)"
  assumes "inv n (last_set F)"
          "last_set F \<in> set (sets (all_sets F))"
          "abs_set (last_set F) = Max (FamilyAbs.sets (abs_fam (all_sets F)))"
  assumes "augmenting_sets_inv augmenting_sets n"
  assumes "augmenting_sets_OK augmenting_sets n"
  shows "\<forall> F' \<in> set (augment n augmenting_sets powerset_by_card set_perms F).
           abs_set (last_set F') = Max (FamilyAbs.sets (abs_fam (all_sets F')))"
proof safe
  fix F'
  assume "F' \<in> set (augment n augmenting_sets powerset_by_card set_perms F)"
  then obtain s where
    F': "F' = extend_family n powerset_by_card set_perms F s"
    "s \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))"
    unfolding augment_def
    by auto
  moreover
  have "last_set F \<in> set (powerset n)"
    using assms powerset_set
    by auto
  ultimately
  have "card s < card (last_set F) \<or>
        card (last_set F) = card s \<and> s < last_set F"
    using assms
    using augmenting_sets_less[OF `n \<le> n_max`, of augmenting_sets]
    unfolding augment_def augmenting_sets_inv_def
    by (auto simp add: Let_def)
  hence "abs_set s > abs_set (last_set F)"
    using assms \<open>last_set F \<in> set (powerset n)\<close> \<open>s \<in> set (the (Mapping.lookup augmenting_sets (last_set F)))\<close> 
    by (meson augmenting_sets_inv_def augmenting_sets_less less_set_abs)
  thus "abs_set (last_set F') = Max (FamilyAbs.sets (abs_fam (all_sets F')))"
    using assms F' `last_set F \<in> set (powerset n)`
    unfolding augmenting_sets_inv_def
    using not_le
    by (simp add: FamilyAbs.add_set_def max_def)
       (smt Family_inverse Max_insert2 finite.insertI finite_sets less_eq_Set_def mem_Collect_eq)
qed

lemma map_filter:
  assumes "\<forall> x \<in> set xs. P x \<longleftrightarrow> P' (f x)"
  shows "map f (filter P xs) = filter P' (map f xs)"
  using assms
  by (induction xs) auto

lemma step_abs [simp]:
  assumes 
    "N \<le> n_max" 
    "augmenting_sets_OK augmenting_sets N"
    "augmenting_sets_inv augmenting_sets N"
    "augmenting_sets_sorted_distinct augmenting_sets N"
    "group_powerset_by_card_OK powerset_by_card N"
    "group_powerset_by_card_inv powerset_by_card N"
    "set_perms_OK set_perms N"
    "\<forall> x \<in> set L. inv_f N (all_sets x)"
    "\<forall> x \<in> set L. union_closed (abs_fam (all_sets x))"
    "\<forall> x \<in> set L. inv N (last_set x)"
    "\<forall> x \<in> set L. abs_set (last_set x) = Max (FamilyAbs.sets (abs_fam (all_sets x)))"
    "\<forall> x \<in> set L. last_set x \<in> set (sets (all_sets x))"
    "\<forall> x \<in> set L. perms_filtered x N"
    "\<forall> x \<in> set L. hd_perms x"
    "\<forall> x \<in> set L. inv_f N (reduced x)"
    "\<forall> x \<in> set L. FamilyAbs.sets (abs_fam (reduced x)) = FamilyAbs.reduced (abs_fam (all_sets x))"

    "distinct (map (abs_fam \<circ> all_sets) L)"
    "UnionClosed.catalogue (map (abs_fam \<circ> all_sets) L) q"

  shows "map (abs_fam \<circ> all_sets) (step_fam N augmenting_sets powerset_by_card set_perms L) =
         UnionClosed'.step (map (abs_fam \<circ> all_sets) L)"
proof-
  have "map (abs_fam \<circ> all_sets)
         (filter (is_canon N powerset_by_card set_perms)
         (concat (map (augment N augmenting_sets powerset_by_card set_perms)
           L))) = filter (is_canon_opt N)
         (concat (map (FamilyAbs.augment N \<circ> (abs_fam \<circ> all_sets)) L))" 
    (is "map ?f (filter ?c (concat (map ?a L))) = filter ?ca (concat (map (?aa \<circ> ?f) L))")
  proof-
    have "\<forall> x \<in> set L. filter ?ca ((?aa \<circ> ?f) x) = map ?f (filter ?c (?a x))"
    proof
      fix F
      assume "F \<in> set L"
      show "filter ?ca ((?aa \<circ> ?f) F) = map ?f (filter ?c (?a F))"
      proof (subst map_filter)
        show "\<forall> x \<in> set (?a F). ?c x = ?ca (?f x)"
        proof
          fix x
          assume "x \<in> set (?a F)"
          thus "?c x = ?ca (?f x)"
            using assms `F \<in> set L`
            by (subst is_canon_abs)
              (auto simp add: augment_perms_filtered augment_hd_perms augment_inv)
        qed
      next
        show "filter ?ca ((?aa \<circ> ?f) F) =
            filter ?ca (map ?f (?a F))"
          using assms `F \<in> set L`
          by simp
      qed
    qed
    thus ?thesis
      apply (subst filter_concat)
      apply (subst filter_concat)
      apply (subst map_concat)
      apply (subst concat_eq_concat_iff)
        apply (auto simp add: set_zip)
      done
  qed
  thus ?thesis
    using assms
    using UnionClosed.step_opt[of "map (abs_fam \<circ> all_sets) L" q] step_fam_def
    unfolding UnionClosed.catalogue_def 
    by (metis UnionClosed.step_opt_def map_map)
qed

lemma
  assumes
    "N \<le> n_max" 
    "augmenting_sets_OK augmenting_sets N"
    "augmenting_sets_inv augmenting_sets N"
    "augmenting_sets_sorted_distinct augmenting_sets N"
    "group_powerset_by_card_OK powerset_by_card N"
    "group_powerset_by_card_inv powerset_by_card N"
    "set_perms_OK set_perms N"
 
   "\<forall> x \<in> set L. inv_f N (all_sets x)"
    "\<forall> x \<in> set L. union_closed (abs_fam (all_sets x))"
    "\<forall> x \<in> set L. inv N (last_set x)"
    "\<forall> x \<in> set L. abs_set (last_set x) = Max (FamilyAbs.sets (abs_fam (all_sets x)))"
    "\<forall> x \<in> set L. last_set x \<in> set (sets (all_sets x))"
    "\<forall> x \<in> set L. perms_filtered x N"
    "\<forall> x \<in> set L. hd_perms x"
    "\<forall> x \<in> set L. inv_f N (reduced x)"
    "\<forall> x \<in> set L. FamilyAbs.sets (abs_fam (reduced x)) = FamilyAbs.reduced (abs_fam (all_sets x))"
  assumes  "distinct (map (abs_fam \<circ> all_sets) L)"
  assumes "UnionClosed.catalogue (map (abs_fam \<circ> all_sets) L) q"
  shows "UnionClosed.catalogue (map (abs_fam \<circ> all_sets) (step_fam N augmenting_sets powerset_by_card set_perms L)) (q+1)"
  using UnionClosed.step_catalogue[OF `UnionClosed.catalogue (map (abs_fam \<circ> all_sets) L) q`]
  using assms
  by simp

lemma
  assumes 
    "N \<le> n_max" 
    "augmenting_sets_OK augmenting_sets N"
    "augmenting_sets_inv augmenting_sets N"
    "augmenting_sets_sorted_distinct augmenting_sets N"
    "group_powerset_by_card_OK powerset_by_card N"
    "group_powerset_by_card_inv powerset_by_card N"
    "set_perms_OK set_perms N"
    "\<forall> x \<in> set L. inv_f N (all_sets x)"
    "\<forall> x \<in> set L. union_closed (abs_fam (all_sets x))"
    "\<forall> x \<in> set L. inv N (last_set x)"
    "\<forall> x \<in> set L. abs_set (last_set x) = Max (FamilyAbs.sets (abs_fam (all_sets x)))"
    "\<forall> x \<in> set L. last_set x \<in> set (sets (all_sets x))"
    "\<forall> x \<in> set L. perms_filtered x N"
    "\<forall> x \<in> set L. hd_perms x"
    "\<forall> x \<in> set L. inv_f N (reduced x)"
    "\<forall> x \<in> set L. FamilyAbs.sets (abs_fam (reduced x)) = FamilyAbs.reduced (abs_fam (all_sets x))"

    "distinct (map (abs_fam \<circ> all_sets) L)"
    "UnionClosed.catalogue (map (abs_fam \<circ> all_sets) L) q"

    "x \<in> set (step_fam N augmenting_sets powerset_by_card set_perms L)"
  shows 
    "inv_f N (all_sets x)"
    "union_closed (abs_fam (all_sets x))"
    "inv N (last_set x)"
    "abs_set (last_set x) = Max (FamilyAbs.sets (abs_fam (all_sets x)))"
    "last_set x \<in> set (sets (all_sets x))"
    "perms_filtered x N"
    "hd_perms x"
    "inv_f N (reduced x)"
    "FamilyAbs.sets (abs_fam (reduced x)) = FamilyAbs.reduced (abs_fam (all_sets x))"
proof-
  obtain x' where x': "x \<in> set (augment N augmenting_sets powerset_by_card set_perms x')" "x' \<in> set L" 
    using `x \<in> set (step_fam N augmenting_sets powerset_by_card set_perms L)`
    unfolding step_fam_def
    by auto
  show "inv_f N (all_sets x)"
    using x' assms 
    by (auto simp add: augment_inv)
  show "inv_f N (reduced x)"
    using x' assms 
    by (auto simp add: augment_inv_r)
  show "perms_filtered x N"
    using x' assms 
    by (auto simp add: augment_perms_filtered)
  show "hd_perms x"
    using x' assms 
    by (auto simp add: augment_hd_perms)

  show "inv N (last_set x)"
    using x' assms 
    by (auto simp add: augment_last_set_inv)

  show "last_set x \<in> set (sets (all_sets x))"
    using x' assms
    by (auto simp add: augment_last_set_in_family)

  show "abs_set (last_set x) = Max (FamilyAbs.sets (abs_fam (all_sets x)))"
    using x' assms
    by (auto simp add: augment_last_set_Max)

  have "(abs_fam (all_sets x)) \<in> set (map (abs_fam \<circ> all_sets)
         (local.augment N augmenting_sets powerset_by_card set_perms x'))"
    using x'
    by simp
  hence "(abs_fam (all_sets x)) \<in> set (FamilyAbs.augment N (abs_fam (all_sets x')))"
    using x' assms augment_abs[of N x' augmenting_sets powerset_by_card set_perms]
    by simp
  thus "union_closed (abs_fam (all_sets x))"
    by (simp add: FamilyAbs.augment_def augment_set_finite)

  show "FamilyAbs.sets (abs_fam (FamilyCache.reduced x)) = FamilyAbs.reduced (abs_fam (all_sets x))"
    using x' assms
    by (auto simp add: augment_reduced)
qed

(* start_family *)
lemma start_family_abs [simp]:
  assumes "n \<le> n_max"
  shows "abs_fam (all_sets (start_family n)) = Family {Set.Set {0..<n}}"
  using assms 
  using set_of_inv [OF assms, of "[0..<n]"]
  using set_of_elements[OF assms, of "[0..<n]"]
  using add_set_abs[OF assms]
  unfolding start_family_def Let_def
  by (simp add: abs_set_def FamilyAbs.add_set_def Set_inverse Family_inverse)

lemma start_family_inv_f [simp]:
  assumes "n \<le> n_max"
  shows "inv_f n (all_sets (start_family n))"
  unfolding start_family_def Let_def
  unfolding add_set_def
  using assms add_inv set_of_inv
  by auto

lemma start_family_union_closed [simp]:
  assumes "n \<le> n_max"
  shows "union_closed (abs_fam (all_sets (start_family n)))"
  using assms
  by (simp add: union_closed_def FamilyAbs.union_def Set_inverse Family_inverse)

lemma start_family_last [simp]:
  assumes "n \<le> n_max"
  shows "inv n (last_set (start_family n))"
        "abs_set (last_set (start_family n)) = Max (FamilyAbs.sets (abs_fam (all_sets (start_family n))))"
        "last_set (start_family n) \<in> set (sets (all_sets (start_family n)))"
  using assms
  by (auto simp add: start_family_def  FamilyAbs.add_set_def Family_inverse)

lemma start_family_perms [simp]: 
  shows "perms (start_family n) = [0..<Combinatorics.fact n]"
  unfolding start_family_def
  by simp

lemma start_family_perms_filtered:
  assumes "n \<le> n_max"
  shows "perms_filtered (start_family n) n"
        "hd_perms (start_family n)"
  using assms
  unfolding perms_filtered_def hd_perms_def
  by (auto simp add: perm_fixes_singleton)

lemma start_family_reduced [simp]:
  assumes "n \<le> n_max"
  shows "reduced (start_family n) = add empty_family (set_of [0..<n])"
  unfolding start_family_def
  by (simp add: add_set_def update_reduced_def)

lemma start_family_reduced_inv [simp]:
  assumes "n \<le> n_max"
  shows  "inv_f n (reduced (start_family n))"
         "FamilyAbs.sets (abs_fam (reduced (start_family n))) =
          FamilyAbs.reduced (abs_fam (all_sets (start_family n)))"
  using assms
  by (auto simp add: FamilyAbs.add_set_def abs_set_def FamilyAbs.reduced_def Family_inverse)

lemma start_family_S [simp]:
  assumes "n \<le> n_max" "n > 0"
  shows "abs_fam (all_sets (start_family n)) \<in> S n 0"
  using assms
  unfolding S_def
  by (simp add: union_closed_def FamilyAbs.union_def Set_inverse finite_subset Family_inverse Set_inject)


lemma start_family_catalogue [simp]:
  assumes "N \<le> n_max" "N > 0"
  shows "UnionClosed.catalogue [abs_fam (all_sets (start_family N))] 0"
  using assms start_family_S[OF assms] Family_singleton EquivFamily.equivp_equiv[of N]
  unfolding UnionClosed.catalogue_def
  by (auto simp add: CanonFamily.is_canon_def permute_family_upt_n S_def equivp_on_def reflp_on_def)

lemma start_family_distinct [simp]:
  shows "distinct (map (abs_fam \<circ> all_sets) [start_family N])"
  by simp

primrec steps_fam where
  "steps_fam 0 augmenting_sets powerset_by_card set_perms ss = []"
| "steps_fam (Suc limit) augmenting_sets powerset_by_card set_perms ss = ss # steps_fam limit augmenting_sets powerset_by_card set_perms (step_fam N augmenting_sets powerset_by_card set_perms ss)"

end

end