subsection \<open> Bitwise representation of sets and families \<close>

theory Family_Impl_Bits
  imports Main 
    "HOL-Library.Code_Target_Nat" 
    "HOL-Library.RBT_Mapping"
    "HOL-Library.Product_Lexorder"
    More_List_Lexord Bit_Operations FamilyImpl
begin

subsubsection \<open>Sets and their elements\<close>

type_synonym Elem = "uint8"
type_synonym Set = "uint8"

definition set_of :: "nat list \<Rightarrow> Set" where
  "set_of = of_pos"

definition elements :: "Set \<Rightarrow> nat list" where
  "elements = one_bits_pos"

lemma set_of_bits:
  assumes "set s \<subseteq> {0..<8}"
  shows "set s = {p. p < 8 \<and> set_of s !! p}"
  using assms
  unfolding set_of_def
  using bit_test_of_pos_uint8[of _ s]
  using bit_test_test_bit_uint8[of _ "of_pos s"]
  by auto

lemma elements_set_of [simp]:
  assumes "set xs \<subseteq> {0..<8}" "sorted xs" "distinct xs"
  shows "elements (set_of xs) = xs"
  using assms one_bits_pos_of_pos_uint8[of xs]
  unfolding elements_def set_of_def
  by simp

lemma set_of_elements [simp]:
  shows "set_of (elements s) = s"
  unfolding elements_def set_of_def
  by simp

lemma elements_inj:
  assumes "elements s = elements s'"
  shows "s = s'"
  using assms
  by (metis elements_def of_pos_one_bits_pos_uint8 set_of_def)

lemma elements_zero [simp]:
  shows "elements 0 = []"
proof-
  have "map (\<lambda>k. bit_test k (0::uint8)) [0..<8] = replicate 8 False" (is "?lhs = ?rhs")
  proof (rule nth_equalityI)
    show "length ?lhs = length ?rhs"
      by simp
  next
    fix i
    assume "i < length ?lhs"
    hence "i < 8"
      by simp
    hence "?rhs ! i = False"
      by (rule nth_replicate)
    moreover
    have "?lhs ! i = bit_test i (0::uint8)"
      using `i < 8` nth_map[of i "[0..<8]" "\<lambda> k. bit_test k (0::uint8)"]
      by simp
    ultimately
    show "?lhs ! i = ?rhs ! i"
      by (simp add: bit_test_test_bit_uint8)
  qed
  thus ?thesis
    unfolding elements_def one_bits_pos_def to_bl'_def
    by (simp add: positions_def)
qed

text \<open>@{text contains} - check if a set contains a given element\<close>

definition contains :: "Set \<Rightarrow> Elem \<Rightarrow> bool" where
 "contains s x \<longleftrightarrow> bit_test (nat_of_uint8 x) s"

lemma contains:
  "contains s x \<longleftrightarrow> nat_of_uint8 x \<in> set (elements s)"
  unfolding contains_def
  by (simp add: bit_test_bl_uint8 elements_def one_bits_pos_def set_positions to_bl'_uint8)

text \<open>@{text card} - cardinality of a set\<close>

definition card :: "Set \<Rightarrow> nat" where
  "card s = bit_pop_count s"

lemma card:
  shows "card s = length (elements s)"
  unfolding card_def
  by (simp add: bit_pop_count_uint8 elements_def)

instantiation uint8 :: SetCard
begin
definition card_uint8 :: "uint8 \<Rightarrow> nat" where
  "card_uint8 = Family_Impl_Bits.card"
instance ..
end

subsubsection \<open>Set operations\<close>

text \<open>@{text is_subset} - subsets\<close>

definition is_subset :: "Set \<Rightarrow> Set \<Rightarrow> bool" where
  "is_subset = bit_is_subset"

lemma is_subset:
  shows "is_subset s1 s2 \<longleftrightarrow> set (elements s1) \<subseteq> set (elements s2)"
proof safe
  assume "set (elements s1) \<subseteq> set (elements s2)"
  hence "\<forall> k < 8. bit_test k s1 \<longrightarrow> bit_test k s2"
    unfolding elements_def one_bits_uint8'
    by auto
  thus "is_subset s1 s2"
    unfolding is_subset_def bit_is_subset_uint8
    by (simp add: bit_test_bl_uint8 test_bit_bl_uint8)    
next
  fix x
  assume "is_subset s1 s2" "x \<in> set (elements s1)"
  thus "x \<in> set (elements s2)"
    using bit_is_subset_uint8
    unfolding is_subset_def elements_def one_bits_uint8'
    by (simp add: bit_test_bl_uint8 test_bit_bl_uint8)   
qed

text \<open>@{text union} - union of sets\<close>

definition union :: "Set \<Rightarrow> Set \<Rightarrow> Set" where
  "union = bit_union"

lemma union:
  shows "set (elements (union s1 s2)) = set (elements s1) \<union> set (elements s2)"
  using bit_union_uint8[of s1 s2]
  unfolding union_def elements_def one_bits_uint8'
  by (auto simp add: bit_test_bl_uint8 test_bit_bl_uint8)


subsubsection \<open>The ordering of sets\<close>

definition lessSet :: "Set \<Rightarrow> Set \<Rightarrow> bool" where
  "lessSet s1 s2 \<longleftrightarrow>
    (let n1 = card s1; n2 = card s2
      in n2 < n1 \<or> (n1 = n2 \<and> s1 > s2))"

definition lessEqSet :: "Set \<Rightarrow> Set \<Rightarrow> bool" where
  "lessEqSet s1 s2 \<longleftrightarrow> s1 = s2 \<or> lessSet s1 s2"


subsubsection \<open>@{text powerset} - the powerset of a set\<close>

definition powerset :: "nat \<Rightarrow> Set list" where
  "powerset n = map uint8_of_nat [0 ..< power 2 n]"

lemma set_elements_powerset:
  assumes "n < 8"
  shows "set (map (set o elements) (powerset n)) = {s. s \<subseteq> {0..<n}}"
proof-
  let ?f = "\<lambda>x. set (elements (uint8_of_nat x))" 
  have "?f ` {0..<2 ^ n} = {s. s \<subseteq> {0..<n}}"
  proof safe
    fix x e :: nat
    assume *: "x \<in> {0..<2 ^ n}" "e \<in> ?f x"
    have "x < 2 ^ n" "x < 2 ^ 8"
      using `n < 8` `x \<in> {0..<2^n}`
      using power_strict_increasing_iff[of 2 n 8]
      using order.strict_trans 
      by auto
    moreover
    have "e < 8" "uint8_of_nat x !! e"
      using *
      using nth_map[of e "[0..<8]" "\<lambda>k. bit_test k (uint8_of_nat x)"] `n < 8`
      unfolding elements_def one_bits_pos_def set_positions
      unfolding to_bl'_def
      by (simp_all add: bit_test_test_bit_uint8)
    ultimately
    show "e \<in> {0..<n}"                             
      using bang_is_le_nat[of x e]
      by (metis atLeast0LessThan le_less_trans lessThan_iff nat_power_less_imp_less zero_less_numeral)
  next
    fix x
    assume "x \<subseteq> {0..<n}"
    hence "finite x" 
      using subset_eq_atLeast0_lessThan_finite 
      by blast 
    have "x \<subseteq> {0..<8}"
      using `x \<subseteq> {0..<n}` `n < 8`
      by auto
    let ?x = "sorted_list_of_set x"
    let ?X = "set_of ?x"

    have "set ?x = x"
      using `finite x` `x \<subseteq> {0..<8}` 
      by simp
    hence "x = ?f (nat_of_uint8 ?X)"
      using elements_set_of[of ?x] `x \<subseteq> {0..<8}`
      by simp

    moreover

    have "nat_of_uint8 ?X < 2 ^ n"
    proof-
      have "?X < 2 ^ n"
        using of_pos_ub[of ?x n] `n < 8` `set ?x = x` `x \<subseteq> {0..<n}`
        unfolding set_of_def
        by simp
      thus ?thesis
        using nat_of_uint8_mono[of ?X "2 ^ n"]
        using nat_of_uint8_2pow[of n] `n < 8`
        by simp
    qed
    ultimately
    show "x \<in> ?f ` {0..<2^n}"
      using atLeast0LessThan
      by blast
  qed
  thus ?thesis
    unfolding powerset_def
    by simp
qed
  
lemma set_powerset:
  assumes "n < 8"
  shows "set (powerset n) = {s. set (elements s) \<subseteq> {0..<n}}" (is "?lhs = ?rhs")
proof safe
  fix s x
  assume "s \<in> set (powerset n)" "x \<in> set (elements s)"
  thus "x \<in> {0..<n}"
    using set_elements_powerset[OF assms]
    using atLeast0LessThan
    by auto
next
  fix s
  assume "set (elements s) \<subseteq> {0..<n}"
  hence "set (elements s) \<in> set (map (set o elements) (powerset n))"
    using set_elements_powerset[OF assms]
    by auto
  then obtain s' where "s' \<in> set (powerset n)" "set (elements s') = set (elements s)"
    by auto
  hence "elements s = elements s'"
    using sorted_distinct_set_unique[of "elements s" "elements s'"]
    by (metis distinct_positions elements_def one_bits_pos_def sorted_positions)
  hence "s = s'"
    using elements_inj
    by simp
  thus "s \<in> set (powerset n)"
    using `s' \<in> set (powerset n)`
    by simp
qed

lemma powerset_not_empty:
  assumes "n < 8"
  shows "powerset n \<noteq> []"
proof-
  have "0 \<in> set (powerset n)"
    using assms set_powerset[of n]
    by simp
  thus ?thesis
    by auto
qed

lemma distinct_powerset:
  assumes "n < 8"
  shows "distinct (powerset n)"
  unfolding powerset_def
proof (subst distinct_map, safe)
  show "distinct [0..<2^n]"
    by simp
next
  show "inj_on uint8_of_nat (set [0..<2^n])"
    using assms uint8_of_nat_inj
    unfolding inj_on_def
    by (smt atLeastLessThan_iff le_eq_less_or_eq nat_of_uint8_uint8_of_nat nat_power_less_imp_less not_le order.strict_trans set_upt zero_less_numeral)
qed

subsubsection \<open>Action of domain permutations on sets\<close>

definition permute_set :: "nat \<Rightarrow> perm \<Rightarrow> Set \<Rightarrow> Set" where
  "permute_set n p s = foldl (\<lambda> r (pi, i). bit_put pi r (bit_get i s)) 0 (zip p [0..<n])"

lemma permute_set_Nil [simp]:
  assumes "n = length p"
  shows "permute_set n [] s = 0"
  using assms
  by (simp add: permute_set_def)

lemma permute_set_snoc [simp]:
  assumes "n = length p"
  shows "permute_set (n + 1) (p @ [a]) s = bit_put a (permute_set n p s) (bit_get n s)"
  using assms
  by (simp add: permute_set_def)

lemma permute_set_test_bit:
  assumes "n = length p" "n < 8" "set p \<subseteq> {0..<8}"
  shows "permute_set n p s !! k \<longleftrightarrow> (\<exists> j < n. s !! j \<and> p ! j = k)"
  using assms
proof (induction p arbitrary: n rule: rev_induct)
  case Nil
  then show ?case 
    by simp
next
  case (snoc a p)
  have "n = length p + 1"
    using snoc(2)
    by simp

  show ?case
  proof (cases "bit_get (length p) s = 0")
    case True
    hence "\<not> (s !! length p)"
      using bit_get_uint8[of "length p" s] snoc(2-3)
      by (auto split: if_split_asm)
         (metis zero_neq_one_uint8)
    hence "(\<exists>j<length p. s !! j \<and> p ! j = k) = (\<exists>j<Suc (length p). s !! j \<and> (p @ [a]) ! j = k)"
      by (metis less_Suc_eq less_antisym nth_append)
    thus ?thesis
      using `n = length p + 1` `bit_get (length p) s = 0`
      using permute_set_snoc[of "length p" p a s]
      using snoc(1)[of "length p"] snoc(3) snoc(4)
      by (simp add: bit_put_0_uint8)
  next
    case False
    hence *: "s !! length p" "bit_get (length p) s = 1"
      using bit_get_uint8[of "length p" s] snoc(2-3)
      by (auto split: if_split_asm)
    show ?thesis
    proof (cases "k = a")
      case True
      thus ?thesis
        using False *
        using permute_set_snoc[of "length p" p a s]
        using `n = length p + 1` snoc(4)
        by (auto simp add: bit_put_1 bit_set_test_gen_uint8)
    next
      case False
      hence "(\<exists>j<length p. s !! j \<and> p ! j = k) = (\<exists>j<Suc (length p). s !! j \<and> (p @ [a]) ! j = k)"
        by (auto simp add: nth_append)
      thus ?thesis
        using False *
        using snoc(1)[of "length p"]
        using permute_set_snoc[of "length p" p a s]
        using `n = length p + 1` snoc(3) snoc(4)
        by (simp add: bit_put_1 bit_set_test_gen_uint8)
    qed
  qed
qed

definition permute_list :: "perm \<Rightarrow> nat list \<Rightarrow> nat list" where
  "permute_list p A = rev (sort (map (list_to_fun p) A))"

lemma elements_permute_set:
  fixes s :: uint8
  assumes "n = length p" "set (elements s) \<subseteq> {0..<n}" "distinct p" "set p = {0..<n}" "n < 8"
  shows "elements (permute_set n p s) = rev (permute_list p (elements s))" (is "?lhs = ?rhs")
proof (rule sorted_distinct_set_unique)
  show "set ?lhs = set ?rhs"
  proof safe
    fix x
    assume "x \<in> set ?lhs"
    hence "x < 8" "permute_set n p s !! x"
      unfolding elements_def permute_list_def
      by (simp_all add: set_one_bits_pos_uint8)
    then obtain j where "j<length p" "s !! j" "p ! j = x"
      using permute_set_test_bit[of n p s x] assms
      by auto
    hence "j \<in> set (elements s)"
      using assms
      by (simp add: elements_def set_one_bits_pos_uint8)
    thus "x \<in> set (rev (permute_list p (elements s)))"
      using `p ! j = x`
      unfolding permute_list_def
      by (auto simp add: list_to_fun_def)
  next
    fix x
    assume "x \<in> set ?rhs"
    then obtain j where "j \<in> set (elements s)" "x = p ! j"
      unfolding permute_list_def list_to_fun_def
      by auto

    hence "s !! j" "j < length p"
      using assms
      by (auto simp add: elements_def set_one_bits_pos_uint8)
    hence "permute_set n p s !! x"
      using permute_set_test_bit[of n p s x] assms
      using \<open>x = p ! j\<close>
      by auto
    thus "x \<in> set (elements (permute_set n p s))"
      by (simp add: elements_def set_one_bits_pos_uint8 test_bit_uint8_code)
  qed
next
  show "sorted ?lhs"
    by (simp add: elements_def one_bits_pos_def positions_sorted_list_of_set)
next
  show "distinct ?lhs"
    by (simp add: distinct_positions elements_def one_bits_pos_def)
next
  show "sorted ?rhs"
    unfolding permute_list_def
    by simp
next
  show "distinct ?rhs"
  proof-
    have "distinct (elements s)"
      by (simp add: distinct_positions elements_def one_bits_pos_def)
    moreover
    have "inj_on (list_to_fun p) (set (elements s))"
      unfolding inj_on_def list_to_fun_def
      by (metis assms(1) assms(2) assms(3) atLeastLessThan_iff index_of_list_element subsetD)
    ultimately
    show ?thesis
      unfolding permute_list_def
      by (simp add: distinct_map)
  qed
qed

subsubsection \<open>Interpreting the locale\<close>

global_interpretation SetImpl_Bit: SetImpl where
   empty_set = 0 and
   inv = "\<lambda> n s. s < 2^n" and
   elements = Family_Impl_Bits.elements and
   set_of = Family_Impl_Bits.set_of and
   powerset = Family_Impl_Bits.powerset and
   permute_set = Family_Impl_Bits.permute_set and
   un = "Family_Impl_Bits.union" and
   is_subset = "Family_Impl_Bits.is_subset" and
   n_max = 6
 defines
   abs_set = SetImpl_Bit.abs_set
proof
  show "Family_Impl_Bits.elements 0 = []"
    by simp
next
  fix n :: nat
  assume "n \<le> 6"
  show "0 < (2::uint8) ^ n"
  proof-
    have "0 < (2::nat) ^ n"
      using `n \<le> 6`
      by simp
    hence "uint8_of_nat 0 < uint8_of_nat ((2::nat) ^ n)"
      using power_increasing[of n 6 "2::nat"]
      using `n \<le> 6` uint8_of_nat_mono[of 0 "2^n"]
      by simp
    thus ?thesis
      using uint8_of_nat_2pow[of n] `n \<le> 6`
      by simp
  qed
next
  fix n :: nat and s :: uint8
  assume "n \<le> 6" "s < 2 ^ n"
  thus "set (Family_Impl_Bits.elements s) \<subseteq> {0..<n}"
    using Family_Impl_Bits.elements_def set_one_bits_pos_ub 
    by auto
next
  fix n :: nat and s :: uint8
  assume "n \<le> 6" "s < 2 ^ n"
  thus "sorted (Family_Impl_Bits.elements s)"
    by (simp add: Family_Impl_Bits.elements_def one_bits_pos_def sorted_positions)
next
  fix n :: nat and s :: uint8
  assume "n \<le> 6" "s < 2 ^ n"
  thus "distinct (Family_Impl_Bits.elements s)"
    by (simp add: Family_Impl_Bits.elements_def one_bits_pos_def distinct_positions)
next
  fix n :: nat and s1 s2 :: uint8
  assume "n \<le> 6" "s1 < 2 ^ n" "s2 < 2 ^ n" "Family_Impl_Bits.elements s1 = Family_Impl_Bits.elements s2"
  thus "s1 = s2"
    by (simp add: elements_inj)
next
  fix n :: nat and s :: uint8
  show "FamilyImpl.card s = length (Family_Impl_Bits.elements s)"
    by (simp add: card card_uint8_def)
next
  fix n s1 s2
  fix n :: nat and s1 s2 :: uint8
  assume "n \<le> 6" "s1 < 2 ^ n" "s2 < 2 ^ n" "FamilyImpl.card s1 = FamilyImpl.card s2"
  show "(s1 < s2) = (rev (Family_Impl_Bits.elements s1) < rev (Family_Impl_Bits.elements s2))"
    unfolding elements_def one_bits_uint8'
  proof (subst map_filter_less_list[symmetric])
    show "length (filter (\<lambda>k. bit_test k s1) [0..<8]) = length (filter (\<lambda>k. bit_test k s2) [0..<8])"
      using `FamilyImpl.card s1 = FamilyImpl.card s2`
      unfolding card_uint8_def card_def
      by (simp add: bit_pop_count_uint8 one_bits_uint8')
  next
    show "(s1 < s2) = (rev (map (\<lambda>k. bit_test k s1) [0..<8]) < rev (map (\<lambda>k. bit_test k s2) [0..<8]))"
      unfolding bit_test_test_bit_uint8
    proof transfer
      fix s1 s2 :: "8 word"
      have "0 \<le> uint s1" "uint s1 < 2 ^ 8" "0 \<le> uint s2" "uint s2 < 2 ^ 8"
        using uint_lt[of s1] uint_lt[of s2]
        by simp_all
      moreover
      have "bin_to_bl 8 (uint s1) = rev (map ((!!) s1) [0..<8])" (is "?lhs = ?rhs")
      proof (rule nth_equalityI)
        show "length ?lhs = length ?rhs"
          using size_bin_to_bl by auto
      next
        fix i
        assume "i < length ?lhs"
        thus "bin_to_bl 8 (uint s1) ! i = rev (map ((!!) s1) [0..<8]) ! i"
          using size_bin_to_bl nth_bin_to_bl[of i 8 "uint s1"]
          by (simp add: nth_rev  test_bit_def')
      qed

      moreover

      have "bin_to_bl 8 (uint s2) = rev (map ((!!) s2) [0..<8])" (is "?lhs = ?rhs")
      proof (rule nth_equalityI)
        show "length ?lhs = length ?rhs"
          using size_bin_to_bl by auto
      next
        fix i
        assume "i < length ?lhs"
        thus "bin_to_bl 8 (uint s2) ! i = rev (map ((!!) s2) [0..<8]) ! i"
          using size_bin_to_bl nth_bin_to_bl[of i 8 "uint s2"]
          by (simp add: nth_rev  test_bit_def')
      qed

      ultimately

      show "(s1 < s2) = (rev (map ((!!) s1) [0..<8]) < rev (map ((!!) s2) [0..<8]))"
        using bin_to_bl_lex[of "uint s1" 8 "uint s2"]
        by (simp add: word_less_def del: bin_to_bl_def)
    qed
  qed
next
  fix n :: nat and xs :: "nat list"
  assume "n \<le> 6" "sorted xs" "distinct xs" "set xs \<subseteq> {0..<n}"
  thus "Family_Impl_Bits.elements (set_of xs) = xs"
    by (smt elements_set_of ivl_subset le_cases3 le_numeral_extra(3) le_trans not_numeral_le_zero numeral_le_iff order_subst1 semiring_norm(69) semiring_norm(71) semiring_norm(72))
next
  fix n :: nat and xs :: "nat list"
  assume "n \<le> 6" "sorted xs" "distinct xs" "set xs \<subseteq> {0..<n}"
  thus "set_of xs < 2 ^ n"
    using of_pos_ub set_of_def
    by auto
next
  fix n :: nat
  assume "n \<le> 6"
  show "set (powerset n) = {s. s < 2 ^ n}"
    using `n \<le> 6` set_powerset[of n]
    using set_one_bits_pos_ub[of n]
    by (simp add: Family_Impl_Bits.elements_def)
next
  fix n::nat
  assume "n \<le> 6"
  thus "distinct (powerset n)"
    using distinct_powerset
    by simp
next
  fix n :: nat and s :: uint8 and p :: "nat list"
  assume "n \<le> 6" "s < 2 ^ n" "p \<in> set (permute [0..<n])"
  show "Family_Impl_Bits.permute_set n p s < 2 ^ n"
  proof-
    have "set (Family_Impl_Bits.elements (Family_Impl_Bits.permute_set n p s)) \<subseteq> {0..<n}"
    proof (subst elements_permute_set[of n p s])
      show "n = length p"
        using `p \<in> set (permute [0..<n])`
        by (simp_all add: permute_member_length)

      show "distinct p"
        using `p \<in> set (permute [0..<n])`
        by (metis card_atLeastLessThan card_distinct permute_member_length length_upt permute_member_set set_upt)

      show "set (Family_Impl_Bits.elements s) \<subseteq> {0..<n}"
        using `s < 2^n`
        by (metis (no_types, hide_lams) Family_Impl_Bits.elements_def atLeastLessThan_iff atLeastLessThan_upt less_le_trans not_less set_one_bits_pos_ub set_one_bits_pos_uint8 subset_code(1) zero_le)

      show "set p = {0..<n}"
        using `p \<in> set (permute [0..<n])`
        using permute_member_set set_upt
        by blast

      show "n < 8" 
        using `n \<le> 6` 
        by simp

      show "set (rev (Family_Impl_Bits.permute_list p (Family_Impl_Bits.elements s))) \<subseteq> {0..<n}"
        using `p \<in> set (permute [0..<n])` `set (Family_Impl_Bits.elements s) \<subseteq> {0..<n}`
        using \<open>n = length p\<close> permute_member_set[of p "[0..<n]"]
        unfolding Family_Impl_Bits.permute_list_def list_to_fun_def
        by force
    qed
    thus "Family_Impl_Bits.permute_set n p s < 2 ^ n"
      using \<open>n \<le> 6\<close> Family_Impl_Bits.elements_def le_trans not_less numeral_le_iff semiring_norm(69) semiring_norm(71) semiring_norm(72) set_one_bits_pos_ub 
      by auto
  qed
next
  fix n p and s :: uint8
  assume *: "n \<le> 6" "s < 2 ^ n" "p \<in> set (permute [0..<n])"
  have "distinct p"
    using  \<open>p \<in> set (permute [0..<n])\<close>
    by (metis atLeast0LessThan atLeastLessThan_upt card_atLeastLessThan card_distinct diff_zero perm_id_def length_upt permute_member_length permute_member_set)
  moreover
  have "set (elements s) \<subseteq> {0..<n}"
    using * Family_Impl_Bits.elements_def set_one_bits_pos_ub 
    by auto[1]
  ultimately
  show "set (Family_Impl_Bits.elements (Family_Impl_Bits.permute_set n p s)) =
        FamilyAbs.list_to_fun p ` set (Family_Impl_Bits.elements s)"
    using *
    by (subst elements_permute_set)
       (auto simp add: permute_member_length permute_member_set permute_list_def)
next
  fix n :: nat and s1 s2 :: uint8
  assume "n \<le> 6" "s1 < 2 ^ n" "s2 < 2 ^ n"
  hence "set (one_bits_pos s1) \<subseteq> {0..<n}" "set (one_bits_pos s2) \<subseteq> {0..<n}"
    using set_one_bits_pos_ub[of n "s1"]
    using set_one_bits_pos_ub[of n "s2"]
    by auto
  hence "set (one_bits_pos (union s1 s2)) \<subseteq> {0..<n}"
    unfolding union_def elements_def one_bits_uint8' bit_test_test_bit_uint8
    by (auto simp add: bit_union_uint8)
  thus "union s1 s2 < 2 ^ n"
    using set_one_bits_pos_ub[of n "union s1 s2"] `n \<le> 6`
    by simp
next
  fix n s1 s2
  show "\<lbrakk>n \<le> 6; s1 < 2 ^ n; s2 < 2 ^ n\<rbrakk>
       \<Longrightarrow> set (Family_Impl_Bits.elements (union s1 s2)) =
           set (Family_Impl_Bits.elements s1) \<union> set (Family_Impl_Bits.elements s2)"
    unfolding union_def elements_def one_bits_uint8' bit_test_test_bit_uint8
    by (auto simp add: bit_union_uint8)
next
  fix n s s'
  show "\<lbrakk>n \<le> 6; s < 2 ^ n; s' < 2 ^ n\<rbrakk>
       \<Longrightarrow> is_subset s s' = (set (Family_Impl_Bits.elements s) \<subseteq> set (Family_Impl_Bits.elements s'))"
    unfolding is_subset_def elements_def one_bits_uint8' bit_test_test_bit_uint8
    by (auto simp add: bit_is_subset_uint8)
qed


subsubsection \<open>Families of sets\<close>

type_synonym Family = "uint64"

text \<open>Sets contained in a given family\<close>

definition sets :: "Family \<Rightarrow> Set list" where
  "sets F = map uint8_of_nat (filter (\<lambda> k. bit_test k F) [0..<64])"

lemma sets_range:
  shows "set (sets F) \<subseteq> {0..<64}"
proof safe
  fix x :: Set
  assume "x \<in> set (sets F)"
  then obtain x' where "x = uint8_of_nat x'" "x' < 64"
    unfolding sets_def
    by auto
  thus "x \<in> {0..<64}"
    using uint8_of_nat_mono[of x' 64]
    by simp
qed

lemma sets:
  shows "s \<in> set (sets F) \<longleftrightarrow> F !! nat_of_uint8 s"
proof
  assume "s \<in> set (sets F)"
  thus "F !! nat_of_uint8 s"
    using bit_test_test_bit_uint64[of "nat_of_uint8 s" F, symmetric]
    by (auto simp add: sets_def)
next
  assume *: "F !! nat_of_uint8 s"
  hence "nat_of_uint8 s < 64"
    by (simp add: test_bit_uint64_code)
  thus "s \<in> set (sets F)"
    using *
    unfolding sets_def
    using atLeastLessThan_iff bit_test_test_bit_uint64 image_iff mem_Collect_eq not_le not_less_zero set_filter set_map set_upt uint8_of_nat_nat_of_uint8
    by fastforce
qed

lemma sets_sorted:
  shows "sorted (sets F)"
  unfolding sets_def
proof (rule sorted_map_mono)
  show "sorted (filter (\<lambda>k. bit_test k F) [0..<64])"
    by simp
next
  let ?F = "filter (\<lambda>k. bit_test k F) [0..<64]"
  show "\<forall>x\<in>set ?F.
          \<forall>y\<in>set ?F. x \<le> y \<longrightarrow> uint8_of_nat x \<le> uint8_of_nat y"
  proof safe
    fix x y
    assume "x \<in> set ?F" "y \<in> set ?F" "x \<le> y"
    thus "uint8_of_nat x \<le> uint8_of_nat y"
      using uint8_of_nat_mono_leq[of x y]
      by auto
  qed
qed

lemma sets_distinct:
  shows "distinct (sets F)"
  unfolding sets_def
proof (subst distinct_map, safe)
  show "distinct (filter (\<lambda>k. bit_test k F) [0..<64])"
    by simp
next
  let ?F = "filter (\<lambda>k. bit_test k F) [0..<64]"
  show "inj_on uint8_of_nat (set (filter (\<lambda>k. bit_test k F) [0..<64]))"
    unfolding inj_on_def 
  proof safe
    fix x y
    assume "x \<in> set ?F" "y \<in> set ?F" "uint8_of_nat x = uint8_of_nat y"
    thus "x = y"
      using uint8_of_nat_inj[of x y]
      by auto
  qed
qed

lemma sets_one_bits_pos:
  shows "sets F = map uint8_of_nat (one_bits_pos F)"
  unfolding sets_def one_bits_uint64'
  by simp

text \<open>Check if the family contains the given set\<close>

definition contains_set :: "Family \<Rightarrow> Set \<Rightarrow> bool" where
  "contains_set F s \<longleftrightarrow> bit_test (nat_of_uint8 s) F"

lemma contains_set:
  shows "contains_set F s \<longleftrightarrow> s \<in> set (sets F)"
  using sets
  unfolding contains_set_def
  by (simp add: bit_test_test_bit_uint64)


text \<open>Empty family\<close>

definition empty_family :: "Family" where
  "empty_family = 0"

lemma sets_0 [simp]:
  shows "sets 0 = []"
  unfolding sets_def
  by (simp add: bit_test_test_bit_uint64)

lemma empty_family [simp]:
  shows "sets (empty_family) = []"
  unfolding empty_family_def
  by simp

definition add_set :: "Family \<Rightarrow> Set \<Rightarrow> Family" where
  "add_set f s = bit_set (nat_of_uint8 s) f"

lemma sets_add_set:
  fixes s::uint8 and F :: uint64 and n::nat
  assumes "n \<le> 6" "\<forall>s\<in>set (sets F). s < 2 ^ n" "s < 2 ^ n"
  shows "set (sets (add_set F s)) = set (sets F) \<union> {s}"
proof-
  have "s < 64"
    using `s < 2^n` `n \<le> 6`
    using pow2_leq_64_uint8 by fastforce
  hence "nat_of_uint8 s < 64"
    using nat_of_uint8_mono[of s 64]
    by simp
  thus ?thesis
    using set_one_bit_pos_bit_set_uint64[of "nat_of_uint8 s" F]
    unfolding add_set_def sets_one_bits_pos
    by simp
qed

definition remove_set :: "Family \<Rightarrow> Set \<Rightarrow> Family" where
  "remove_set f s = bit_clear (nat_of_uint8 s) f"

lemma sets_remove_set:
  assumes "s < 64"
  shows "set (sets (remove_set f s)) = set (sets f) - {s}"
  unfolding sets_def remove_set_def bit_clear_set_bit_uint64
  unfolding bit_test_test_bit_uint64
  by (auto simp add: set_bit_test_gen_uint64)

subsubsection \<open>Set operations on families\<close>

definition inter :: "Family \<Rightarrow> Family \<Rightarrow> Family" where
  "inter = bit_inter"

lemma inter:
  shows "set (sets (inter F1 F2)) = set (sets F1) \<inter> set (sets F2)"
  using bit_int_uint64[of F1 F2]
  unfolding inter_def sets_one_bits_pos one_bits_uint64' bit_test_test_bit_uint64
  by (auto simp add: bit_int_uint64) (subgoal_tac "xa = xb", auto)

subsubsection \<open>Interpreting the locale\<close>

global_interpretation FamilyImpl_Bit: FamilyImpl where
  inv = "\<lambda> n s. s < 2^n" and
  elements = elements and
  set_of = set_of and
  powerset = powerset and
  permute_set = permute_set and
  n_max = 6 and
  empty_set = 0 and
  empty_family = 0 and
  inv_f = "\<lambda> n F. (\<forall> s \<in> set (sets F). s < 2 ^ n)" and
  sets = sets and
  contains = "\<lambda> f e. bit_test (nat_of_uint8 e) f" and
  add = add_set and
  inter = inter  and
  is_subset = is_subset and
  un = union and
  remove = remove_set
defines 
  update_reduced = "FamilyImpl_Bit.update_reduced" and
  add = "FamilyImpl_Bit.add_set" and
  containss = "FamilyImpl_Bit.contains_set" and
  filter_perms = "FamilyImpl_Bit.filter_perms" and
  perm_fixes_subset = "FamilyImpl_Bit.perm_fixes_subset" and
  perm_fixes_card = "FamilyImpl_Bit.perm_fixes_card" and
  extend_family = "FamilyImpl_Bit.extend_family" and
  start_family = "FamilyImpl_Bit.start_family" and
  step_fam = "FamilyImpl_Bit.step_fam" and
  steps_fam = "FamilyImpl_Bit.steps_fam" and
  init_set_perms = "FamilyImpl_Bit.init_set_perms" and
  sets_by_card = "FamilyImpl_Bit.sets_by_card" and
  group_sets_by_card = "FamilyImpl_Bit.group_sets_by_card" and
  group_powerset_by_card = "FamilyImpl_Bit.group_powerset_by_card" and
  augment_set = "FamilyImpl_Bit.augment_set" and
  init_augmenting_sets = "FamilyImpl_Bit.init_augmenting_sets" and
  augment = "FamilyImpl_Bit.augment" and
  init_combinations = "FamilyImpl_Bit.init_combinations" and
  is_union_closed' = "FamilyImpl_Bit.is_union_closed'" and
  is_canon = "FamilyImpl_Bit.is_canon" and
  add_set_by_card = "FamilyImpl_Bit.add_set_by_card" and
  permute_family = "FamilyImpl_Bit.permute_family" and
  less_eq_family = "FamilyImpl_Bit.less_eq_family" and
  less_family = "FamilyImpl_Bit.less_family" and
  eq_family = "FamilyImpl_Bit.eq_family" and
  k_sets = "FamilyImpl_Bit.k_sets" and
  abs_fam = "FamilyImpl_Bit.abs_fam" 
proof
  fix F s
  show "bit_test (nat_of_uint8 s) F = (s \<in> set (sets F))"
    using sets
    unfolding contains_set_def
    by (simp add: bit_test_test_bit_uint64)
next
  show "sets 0 = []"
    by simp
next
  fix s::uint8 and F :: uint64 and n::nat
  assume "n \<le> 6" "\<forall>s\<in>set (sets F). s < 2 ^ n" "s < 2 ^ n"
  thus "set (sets (add_set F s)) = set (sets F) \<union> {s}"
    by (simp add: sets_add_set)
next
  fix F1 F2
  show "set (sets (inter F1 F2)) = set (sets F1) \<inter> set (sets F2)"
    using bit_int_uint64[of F1 F2]
    unfolding inter_def sets_one_bits_pos one_bits_uint64' bit_test_test_bit_uint64
    by (auto simp add: bit_int_uint64) (subgoal_tac "xa = xb", auto)
next
  fix n
  show "\<forall> s \<in> set (sets 0). s < 2 ^ n"
    by simp
next
  fix n :: nat and F :: uint64 and s::uint8
  assume "n \<le> 6" "\<forall>s\<in>set (sets F). s < 2 ^ n" "s < 2 ^ n"
  thus "\<forall>s\<in>set (sets (add_set F s)). s < 2 ^ n"
    by (simp add: sets_add_set)
next
  fix n :: nat and s :: uint8 and F::uint64
  assume "n \<le> 6" "\<forall>s\<in>set (sets F). s < 2 ^ n" "s \<in> set (sets F)"
  thus "s < 2 ^ n"
    by simp
next
  fix n :: nat and s :: uint8 and F::uint64
  assume "n \<le> 6" "\<forall>s\<in>set (sets F). s < 2 ^ n"
  thus "sorted (sets F)"
    using sets_sorted by blast
next
  fix F
  show "distinct (sets F)"
    using sets_distinct by blast
next
  fix F1 F2 :: uint64 and n :: nat
  assume *: "sets F1 = sets F2" "n \<le> 6" "\<forall> s \<in> set (sets F1). s < 2 ^ n" "\<forall> s \<in> set (sets F2). s < 2 ^ n"
  have "map uint8_of_nat (one_bits_pos F1) = map uint8_of_nat (one_bits_pos F2)"
    using *
    unfolding sets_one_bits_pos
    by simp
  hence "one_bits_pos F1 = one_bits_pos F2"
  proof (rule map_inj_on)
    show "inj_on uint8_of_nat ((set (one_bits_pos F1) \<union> set (one_bits_pos F2)))"
      using uint8_of_nat_inj set_one_bits_pos_uint64
      by (auto simp add: inj_on_def)
  qed
  thus "F1 = F2"
    using of_pos_one_bits_pos_uint64[of F1]
    using of_pos_one_bits_pos_uint64[of F2]
    by metis
next
  fix F1 F2 :: uint64 and n :: nat
  assume *: "n \<le> 6" "\<forall>s\<in>set (Family_Impl_Bits.sets F1). s < 2 ^ n"
        "\<forall>s\<in>set (Family_Impl_Bits.sets F2). s < 2 ^ n"
  show "\<forall>s\<in>set (Family_Impl_Bits.sets (Family_Impl_Bits.inter F1 F2)). s < 2 ^ n"
  proof safe
    fix s 
    assume "s \<in> set (sets (inter F1 F2))"
    hence "s \<in> set (sets F1)"
      using bit_int_uint64[of F1 F2]
      unfolding inter_def sets_one_bits_pos one_bits_uint64' bit_test_test_bit_uint64
      by (auto simp add: bit_int_uint64) 
    thus "s < 2 ^ n"
      using *
      by simp
  qed
next
  fix n F a
  show "\<lbrakk>n \<le> 6; \<forall>s\<in>set (Family_Impl_Bits.sets F). s < 2 ^ n; a < 2 ^ n\<rbrakk>
       \<Longrightarrow> set (Family_Impl_Bits.sets (remove_set F a)) = set (Family_Impl_Bits.sets F) - {a}"
    using sets_remove_set[of a F]
    by (meson not_le order_trans pow2_leq_64_uint8)
next
  fix n F a
  show "\<lbrakk>n \<le> 6; \<forall>s\<in>set (Family_Impl_Bits.sets F). s < 2 ^ n; a < 2 ^ n\<rbrakk>
       \<Longrightarrow> \<forall>s\<in>set (Family_Impl_Bits.sets (remove_set F a)). s < 2 ^ n"
    using sets_remove_set[of a F]
    using pow2_leq_64_uint8
    by fastforce
next
  fix n F1 F2
  assume "\<forall>s\<in>set (Family_Impl_Bits.sets F1). s < 2 ^ n"
        "\<forall>s\<in>set (Family_Impl_Bits.sets F2). s < 2 ^ n"
        "length (Family_Impl_Bits.sets F1) = length (Family_Impl_Bits.sets F2)"
  show "(F1 < F2) = (rev (Family_Impl_Bits.sets F1) < rev (Family_Impl_Bits.sets F2))"
  proof-
    let ?r = "\<lambda> F. rev (filter (\<lambda>k. bit_test k F) [0..<64])"

    have "map uint8_of_nat (?r F1)  < map uint8_of_nat (?r F2) \<longleftrightarrow> ?r F1 < ?r F2"
    proof (rule map_mono[symmetric])
      show "length (?r F1) = length (?r F2)"
        using `length (Family_Impl_Bits.sets F1) = length (Family_Impl_Bits.sets F2)`
        by (simp add: sets_def)
    next
      show "\<forall> x \<in> set (?r F1) \<union> set (?r F2). \<forall> y \<in> set (?r F1) \<union> set (?r F2). 
              (x < y) = (uint8_of_nat x < uint8_of_nat y)"
      proof (rule, rule)
        fix x y
        assume "x \<in> set (?r F1) \<union> set (?r F2)" "y \<in> set (?r F1) \<union> set (?r F2)"
        thus "(x < y) = (uint8_of_nat x < uint8_of_nat y)"
          using uint8_of_nat_mono[of x y]
          by auto
      qed
    next
      show "\<forall> x \<in> set (?r F1) \<union> set (?r F2). \<forall> y \<in> set (?r F1) \<union> set (?r F2). 
              (uint8_of_nat x = uint8_of_nat y) \<longrightarrow> x = y"
      proof (rule, rule)
        fix x y
        assume "x \<in> set (?r F1) \<union> set (?r F2)" "y \<in> set (?r F1) \<union> set (?r F2)"
        thus "(uint8_of_nat x = uint8_of_nat y) \<longrightarrow> x = y"
          using uint8_of_nat_inj[of x y]
          by auto
      qed
    qed
    moreover
    have "F1 < F2 \<longleftrightarrow> rev (filter (\<lambda>k. bit_test k F1) [0..<64]) < rev (filter (\<lambda>k. bit_test k F2) [0..<64])"
    proof (subst map_filter_less_list[symmetric])
      show "length (filter (\<lambda>k. bit_test k F1) [0..<64]) = length (filter (\<lambda>k. bit_test k F2) [0..<64])"
        using `length (Family_Impl_Bits.sets F1) = length (Family_Impl_Bits.sets F2)`
        by (simp add: sets_def)
    next
      show "(F1 < F2) = (rev (map (\<lambda>k. bit_test k F1) [0..<64]) < rev (map (\<lambda>k. bit_test k F2) [0..<64]))"
        unfolding bit_test_test_bit_uint64
      proof transfer
        fix s1 s2 :: "64 word"
        have "0 \<le> uint s1" "uint s1 < 2 ^ 64" "0 \<le> uint s2" "uint s2 < 2 ^ 64"
          using uint_lt[of s1] uint_lt[of s2]
          by simp_all
        moreover
        have "bin_to_bl 64 (uint s1) = rev (map ((!!) s1) [0..<64])" (is "?lhs = ?rhs")
        proof (rule nth_equalityI)
          show "length ?lhs = length ?rhs"
            using size_bin_to_bl by auto
        next
          fix i
          assume "i < length ?lhs"
          thus "bin_to_bl 64 (uint s1) ! i = rev (map ((!!) s1) [0..<64]) ! i"
            using size_bin_to_bl nth_bin_to_bl[of i 64 "uint s1"]
            by (simp add: nth_rev  test_bit_def')
        qed

        moreover

        have "bin_to_bl 64 (uint s2) = rev (map ((!!) s2) [0..<64])" (is "?lhs = ?rhs")
        proof (rule nth_equalityI)
          show "length ?lhs = length ?rhs"
            using size_bin_to_bl by auto
        next
          fix i
          assume "i < length ?lhs"
          thus "bin_to_bl 64 (uint s2) ! i = rev (map ((!!) s2) [0..<64]) ! i"
            using size_bin_to_bl nth_bin_to_bl[of i 64 "uint s2"]
            by (simp add: nth_rev  test_bit_def')
        qed

        ultimately

        show "(s1 < s2) = (rev (map ((!!) s1) [0..<64]) < rev (map ((!!) s2) [0..<64]))"
          using bin_to_bl_lex[of "uint s1" 64 "uint s2"]
          by (simp add: word_less_def del: bin_to_bl_def)
      qed
    qed

    ultimately

    show ?thesis
      unfolding sets_def sets_def
      by (auto simp add: rev_map rev_filter)
  qed
qed 

definition "xxx = 
  (let n = 6;
       augmenting_sets = init_augmenting_sets n;
       powerset_by_card = group_powerset_by_card n;
       set_perms = init_set_perms n
    in map (\<lambda> Fs. length Fs) (steps_fam 5 augmenting_sets powerset_by_card set_perms [start_family n]))"

(*
lemma [code]:                   
  "less_family n powerset_by_card F1 F2 \<longleftrightarrow> (all_sets F1) > (all_sets F2)"
  oops
*)

value xxx

end
