section\<open>General Faradzev-Read Scheme\<close>

text \<open>This theory is the central contribution of this work. It formalizes the general Faradžev-Read
scheme that can be instantiated to enumerate various combinatorial objects.\<close>

theory Faradzev_Read
  imports Main 
          More_List More_Relation "HOL-Library.Multiset"
begin

subsection \<open>Basic ingredients\<close>

locale FaradzevRead' =
  fixes S :: "nat \<Rightarrow> ('s::linorder) set"
  fixes equiv :: "'s \<Rightarrow> 's \<Rightarrow> bool"
  fixes is_canon :: "'s \<Rightarrow> bool"
  fixes is_canon_test :: "'s \<Rightarrow> bool"
  fixes augment :: "'s \<Rightarrow> 's list"
  assumes S_disjunct: "\<And> q1 q2. q1 \<noteq> q2 \<Longrightarrow> S q1 \<inter> S q2 = {}"
  assumes equiv_equiv: "\<And> q. equivp_on (S q) equiv"
  assumes equiv_dim: "\<And> s s' q. \<lbrakk>equiv s s'; s \<in> S q\<rbrakk> \<Longrightarrow> s' \<in> S q"
  assumes ex1_is_canon: "\<And> s q. s \<in> S q \<Longrightarrow> \<exists>! sc. equiv s sc \<and> is_canon sc"
  assumes is_canon_test: "\<And> s s' q. \<lbrakk>s \<in> S q; is_canon s; s' \<in> set (augment s)\<rbrakk> \<Longrightarrow>  is_canon_test s' \<longleftrightarrow> is_canon s'"
  assumes augment_dim: "\<And> s s' q. \<lbrakk>is_canon s; s' \<in> set (augment s)\<rbrakk> \<Longrightarrow> (s \<in> S q \<longleftrightarrow> s' \<in> S (q + 1))"
begin

definition canon :: "'s \<Rightarrow> 's" where
  "canon s = (THE sc. equiv s sc \<and> is_canon sc)"

lemma is_canon_canon:
  assumes "s \<in> S q"
  shows "is_canon (canon s)"
  using theI'[OF ex1_is_canon[rule_format, OF assms]]
  unfolding canon_def
  by simp

lemma equiv_canon:
  assumes "s \<in> S q"
  shows "equiv s (canon s)"
  using theI'[OF ex1_is_canon[rule_format, OF assms]]
  unfolding canon_def
  by simp

lemma canon_dim: 
  assumes "s \<in> S q"
  shows "canon s \<in> S q"
  using assms
  using equiv_canon equiv_dim
  by blast

lemma is_canon: 
  assumes "s \<in> S q"
  shows "is_canon s \<longleftrightarrow> canon s = s"
  using assms
  using equiv_canon equiv_equiv equivp_on_def reflp_on_def ex1_is_canon is_canon_canon
  by metis

lemma filter_is_canon_test [simp]:
  assumes "s \<in> S q" "is_canon s" 
  shows "filter is_canon_test (augment s) = filter is_canon (augment s)"
  using assms
  by (auto simp add: is_canon_test intro: filter_cong)

lemma filter_is_canon_test_concat_map [simp]:
  assumes "set L \<subseteq> S q" "Ball (set L) is_canon"
  shows "filter is_canon_test (concat (map augment L)) = 
         filter is_canon (concat (map augment L))"
  using assms
proof (induction L)
  case Nil
  then show ?case 
    by simp
next
  case (Cons s L)
  then show ?case
    by auto    
qed

definition parent :: "'s \<Rightarrow> 's \<Rightarrow> bool" where 
  "parent p s \<longleftrightarrow> is_canon p \<and> s \<in> set (augment p)"

definition min_parent :: "'s \<Rightarrow> 's \<Rightarrow> bool" where 
  "min_parent p s \<longleftrightarrow> parent p s \<and> (\<forall> p'. parent p' s \<longrightarrow> p \<le> p')"

definition step_test :: "('a::ord) \<Rightarrow> 'a list \<Rightarrow> 'a list" where
  "step_test s res = (if res = [] \<or> s > hd res then s # res else res)"

definition step :: "'s list \<Rightarrow> 's list" where
  "step L = rev (fold step_test (filter is_canon_test (concat (map augment L))) [])"

end

subsection \<open>Correctness requirements\<close>

locale FaradzevRead = FaradzevRead' S for S :: "nat \<Rightarrow> 's::linorder set" +
  assumes cond1: "\<And> q s. \<lbrakk>s \<in> S (q + 1); is_canon s\<rbrakk> \<Longrightarrow> 
                            (\<exists> p \<in> S q. parent p s)"
  assumes cond2: "\<And> s s' p p' q. 
                     \<lbrakk>s \<in> S (q + 1); s' \<in> S (q + 1);
                      min_parent p s; min_parent p' s'; s < s'\<rbrakk> \<Longrightarrow> p \<le> p'"
  assumes cond3: "\<And> s q. \<lbrakk>s \<in> S q; is_canon s\<rbrakk> \<Longrightarrow> 
                      sorted (filter is_canon (augment s))"
begin

lemma step_canon:
  assumes "set L \<subseteq> S q" "Ball (set L) is_canon"
  shows "step L = rev (fold step_test (filter is_canon (concat (map augment L))) [])"
  using assms
  unfolding step_def
  by simp

definition catalogue where
  "catalogue L q \<longleftrightarrow>
     (sorted L) \<and> (distinct L) \<and> 
     set L \<subseteq> S q \<and> 
     (\<forall> s \<in> set L. is_canon s) \<and>
     (\<forall> s \<in> S q. \<exists> c \<in> set L. is_canon c \<and> equiv s c)"

lemma unique_catalogue:
  assumes "catalogue L1 q" "catalogue L2 q"
  shows "L1 = L2"
proof (rule sorted_distinct_set_unique)
  show "sorted L1" "sorted L2" "distinct L1" "distinct L2"
    using assms
    by (auto simp add: catalogue_def)
next
  show "set L1 = set L2"
  proof-
    {
      fix Lx Ly
      assume "catalogue Lx q" "catalogue Ly q"
      have "set Lx \<subseteq> set Ly"
      proof
        fix x
        assume "x \<in> set Lx"
        hence "is_canon x" "x \<in> S q"
          using `catalogue Lx q`
          unfolding catalogue_def
          by auto
        obtain y where "equiv x y" "is_canon y" "y \<in> set Ly"
          using `catalogue Ly q` `x \<in> S q`
          unfolding catalogue_def
          by auto
        have "x = y"
          using \<open>equiv x y\<close> \<open>is_canon x\<close> \<open>is_canon y\<close> \<open>x \<in> S q\<close> equiv_canon ex1_is_canon is_canon 
          by fastforce
        thus "x \<in> set Ly"
          using `y \<in> set Ly`
          by auto
      qed
    }
    thus ?thesis
      using assms
      by auto
  qed
qed

definition 
  "the_catalogue q = (THE L. catalogue L q)"

lemma the_catalogue:
  assumes "catalogue L q"
  shows "L = the_catalogue q"
proof-
  have "\<exists>! L. catalogue L q"
    using assms unique_catalogue
    by blast
  thus ?thesis
    using theI'[of "\<lambda> x. catalogue x q"] unique_catalogue[OF assms]
    unfolding the_catalogue_def
    by simp
qed

lemma fold_init_subset:
  "set init \<subseteq> set (fold step_test L init)"                                                   
  by (induction L rule: rev_induct) (auto simp add: step_test_def)
               
lemma fold_L_subset:
  "set (fold step_test L init) \<subseteq> set L \<union> set init"
  by (induction L rule: rev_induct) (auto simp add: step_test_def)

lemma sorted_distinct_step:
  "sorted (rev (fold step_test L [])) \<and> 
   distinct (rev (fold step_test L []))"
proof (induction L rule: rev_induct)
  case Nil
  thus ?case
    by simp
next
  case (snoc c L)
  thus ?case
    unfolding step_test_def
    using less_imp_le sorted_rev_cons
    by auto (metis (no_types, lifting) linorder_not_le list.set_intros(1) set_rev sorted_append)
qed

lemma step_augment [simp]:
  shows "set (step L) \<subseteq> set (concat (map augment L))"
  using step_def fold_L_subset 
  by fastforce

lemma step_all_canon [simp]:
  assumes "set L \<subseteq> S q" "Ball (set L) is_canon"
  shows "Ball (set (step L)) is_canon"
  using step_canon[OF assms] fold_L_subset
  by fastforce

theorem step_catalogue:
  assumes "catalogue L q"                                       
  shows "catalogue (step L) (q + 1)"
  unfolding catalogue_def
proof safe
  show "sorted (step L)" "distinct (step L)"
    unfolding step_def
    using sorted_distinct_step
    by auto
next
  fix s
  assume "s \<in> set (step L)"
  thus "s \<in> S (q + 1)"
    using step_augment[of L] step_all_canon[of L] augment_dim[of _ s q]
    using `catalogue L q`
    unfolding catalogue_def
    by auto
next
  fix s
  assume "s \<in> set (step L)" 
  thus "is_canon s"
    using step_all_canon[of L q]
    using `catalogue L q`
    unfolding catalogue_def
    by fastforce
next
  fix s
  assume "s \<in> S (q + 1)"
  have "canon s \<in> set (step L)"
  proof-
    let ?parent = "\<lambda> p s q. p \<in> S q \<and> parent p (canon s)"

    have "\<exists> p \<in> set L. ?parent p s q"
    proof-
      obtain p where "p \<in> S q" "parent p (canon s)"
        using `s \<in> S (q + 1)`
        using canon_dim[rule_format, of s "q + 1"]
        using cond1[rule_format, of "canon s" q] is_canon_canon
        by (auto simp add: parent_def)
      moreover
      have "p \<in> set L"
        using `p \<in> S q` `parent p (canon s)`
        using `catalogue L q`
        using is_canon equiv_canon ex1_is_canon
        unfolding catalogue_def parent_def
        by fastforce
      ultimately
      show ?thesis
        by auto
    qed

    have "\<exists> L'. catalogue (L @ L') q"
      using `catalogue L q`
      by (rule_tac x="[]" in exI, auto)

    show ?thesis
      using `\<exists> p \<in> set L. ?parent p s q` `\<exists> L'. catalogue (L @ L') q`
    proof (induction L rule: rev_induct)
      case Nil
      thus ?case
        by simp
    next
      case (snoc p L)

      obtain L' where rep: "catalogue (L @ [p] @ L') q"
        using `\<exists>L'. catalogue ((L @ [p]) @ L') q`
        by auto

      have "\<forall>s\<in>S q. \<exists>c\<in>set (L @ [p] @ L'). c = canon s"
        using `catalogue (L @ [p] @ L') q`
        unfolding catalogue_def
        by (metis equiv_canon ex1_is_canon is_canon_canon)

      show ?case
      proof (cases "\<exists> p \<in> set L. ?parent p s q")
        case True
        thus ?thesis
          using snoc
          using fold_init_subset
          by (auto simp add: step_def sorted_append)
      next
        case False
        hence "?parent p s q"
          using snoc(2)
          by auto

        have "min_parent p (canon s)"
          unfolding min_parent_def
        proof safe
          show "parent p (canon s)"
            using `?parent p s q` 
            by simp
        next
          fix p1
          assume "parent p1 (canon s)"
          hence "p1 \<in> S q" "p1 \<notin> set L"
            using `\<not> (\<exists> p \<in> set L. ?parent p s q)` `s \<in> S (q + 1)`
            using canon_dim[of s "q + 1"] 
            using is_canon_canon[of s "q + 1"]
            using augment_dim parent_def
            by blast+

          have "p1 \<in> set ([p] @ L')"
            using `\<forall>s\<in>S q. \<exists>c\<in>set (L @ [p] @ L'). c = canon s`
            using `p1 \<in> S q` `parent p1 (canon s)` `p1 \<notin> set L`
            using is_canon 
            unfolding parent_def
            by auto

          thus "p \<le> p1"
            using `catalogue (L @ [p] @ L') q`
            unfolding catalogue_def
            by (auto simp add: sorted_append)
        qed

        obtain cap where cap: "cap = filter is_canon (augment p)"
          by auto
        let ?aL = "concat (map augment L)"
        let ?caL = "filter is_canon ?aL"
        let ?stL = "fold step_test ?caL []"
        obtain stL where stL: "stL = ?stL"
          by auto

        have "sorted cap"
          using cond3 cap `?parent p s q`
          unfolding parent_def
          by blast
        moreover
        have "canon s \<in> set cap"
          using `?parent p s q` cap `s \<in> S (q + 1)` is_canon_canon
          unfolding parent_def
          by auto
        ultimately
        have "canon s \<in> set (fold step_test cap stL)"
        proof (induction cap rule: rev_induct)
          case Nil
          thus ?case
            by simp
        next
          case (snoc s' ss)
          let ?stss = "fold step_test ss stL"
          let ?sprev = "hd ?stss"
          show ?case
          proof (cases "canon s \<in> set ss")
            case True
            thus ?thesis
              using snoc
              unfolding step_test_def
              by (auto simp add: sorted_append)
          next
            case False
            hence "canon s = s'"
              using snoc(3)
              by auto
            show ?thesis
            proof (rule ccontr)
              assume "\<not> ?thesis"
              hence "?stss \<noteq> []" "\<not> canon s > ?sprev"
                using `canon s = s'`
                unfolding step_test_def
                by auto

              show False
              proof (cases "?sprev \<in> set ss")
                case True
                hence "\<not> sorted (ss @ [s'])"
                  using `canon s = s'` `canon s \<notin> set ss`
                  using `\<not> canon s > ?sprev`
                  by (force simp add: sorted_append)
                thus ?thesis
                  using snoc(2)
                  by simp
              next
                case False
                hence "?sprev \<in> set stL"
                  using fold_L_subset[of ss stL] hd_in_set[OF `?stss \<noteq> []`]
                  by auto
                hence "?sprev \<in> set ?aL" "is_canon ?sprev"
                  using stL
                  using fold_L_subset[of "?caL" "[]"]
                  by auto

                then obtain p' where
                  "p' \<in> set L"
                  "?sprev \<in> set (augment p')"
                  by auto

                hence "is_canon p'" "p' \<in> S q" "?sprev \<in> S (q + 1)"
                  using `catalogue (L @ [p] @ L') q`
                  using augment_dim[of p' ?sprev q]
                  unfolding catalogue_def    
                  by auto

                have "canon ?sprev = ?sprev"
                  using `is_canon ?sprev` `?sprev \<in> S (q + 1)`
                  using is_canon
                  by simp

                hence "?parent p' ?sprev q"
                  using `p' \<in> S q` `is_canon p'` `?sprev \<in> set (augment p')`
                  unfolding parent_def
                  by metis

                let ?Pi = "{ i. i < length L \<and> ?parent (L ! i) ?sprev q }"

                have "?Pi \<noteq> {}"
                  using `?parent p' ?sprev q` `p' \<in> set L`
                  by (auto simp add: in_set_conv_nth)

                have "finite ?Pi"
                  by simp

                let ?i = "Min ?Pi"
                let ?pmin = "L ! ?i"

                have "?i < length L" "?parent ?pmin ?sprev q"
                  using Min_in[OF `finite ?Pi` `?Pi \<noteq> {}`] `canon ?sprev = ?sprev`
                  by auto

                hence "?pmin \<in> set L"
                  by auto

                hence "canon s < ?sprev"
                  using `\<not> canon s > ?sprev` `canon ?sprev = ?sprev`
                  using  `?parent ?pmin ?sprev q` 
                  using `\<not> (\<exists> p \<in> set L. ?parent p s q)`
                  using linorder_cases
                  by force

                have "min_parent ?pmin ?sprev"
                  unfolding min_parent_def
                proof safe
                  show "parent ?pmin ?sprev"
                    using `?parent ?pmin ?sprev q`
                          `canon ?sprev = ?sprev` 
                    by simp
                next         
                  fix p1
                  assume "parent p1 ?sprev"

                  have "p1 \<in> S q"
                    using `?sprev \<in> S (q + 1)` `parent p1 ?sprev` 
                    using augment_dim parent_def
                    by blast

                  show "?pmin \<le> p1"
                  proof (cases "p1 \<in> set L")
                    case True
                    then obtain i' where "i' < length L \<and> p1 = L ! i'"
                      by (metis in_set_conv_nth)
                    hence "i' \<in> ?Pi"
                      using `parent p1 ?sprev` `p1 \<in> S q` `canon ?sprev = ?sprev`
                      by simp
                    hence "?i \<le> i'"
                      using Min_eq_iff `finite ?Pi`
                      by blast
                    thus ?thesis
                      using `i' < length L \<and> p1 = L ! i'` `?i < length L` 
                      using `catalogue (L @ [p] @ L') q`
                      unfolding catalogue_def
                      by (simp add: sorted_append sorted_nth_mono)
                  next
                    case False

                    hence "p1 \<in> set ([p] @ L')"
                      using `\<forall>s\<in>S q. \<exists>c\<in>set (L @ [p] @ L'). c = canon s`
                      using `p1 \<in> S q` `parent p1 ?sprev`
                      using is_canon parent_def
                      by auto
                    thus ?thesis
                      using `?pmin \<in> set L`
                      using `catalogue (L @ [p] @ L') q`
                      unfolding catalogue_def
                      by (auto simp add: sorted_append)
                  qed
                qed

                have "p \<le> ?pmin"
                  using cond2[OF canon_dim[rule_format, OF `s \<in> S (q + 1)`]
                                 `?sprev \<in> S (q + 1)`
                                 `min_parent p (canon s)`
                                 `min_parent?pmin ?sprev`]
                  using `canon s < ?sprev`
                  by simp

                moreover

                have "?pmin < p"   
                  using `?pmin \<in> set L` `catalogue (L @ [p] @ L') q`
                  unfolding catalogue_def
                  unfolding sorted_append
                  using calculation by fastforce

                ultimately
                show False
                  by simp
              qed          
            qed
          qed
        qed
        moreover
        have "set (L @ [p]) \<subseteq> S q" "Ball (set (L @ [p])) is_canon"
          using snoc.prems
          unfolding catalogue_def
          by auto
        ultimately
        show ?thesis
          using step_canon[of "L @ [p]" q]
          using cap stL
          by simp
      qed                                                                                 
    qed
  qed
  thus "\<exists>c\<in>set (step L). is_canon c \<and> equiv s c"
    using \<open>s \<in> S (q + 1)\<close> equiv_canon is_canon_canon
    by blast
qed

lemma step_sorted:
  shows "sorted (step L)"
  unfolding step_def
  using sorted_distinct_step by blast

lemma distinct_sorted:
  shows "distinct (step L)"
  unfolding step_def
  using sorted_distinct_step
  by blast

lemma step_equiv_inj:
  assumes "catalogue L q" "a \<in> set (step L)" "b \<in> set (step L)" "equiv a b" 
  shows "a = b"
proof-
  have "is_canon a" "is_canon b" "a \<in> S (q + 1)" "b \<in> S (q + 1)"
    using step_catalogue[OF `catalogue L q`] `a \<in> set (step L)` `b \<in> set (step L)`
    unfolding catalogue_def
    by auto
  thus ?thesis
    using ex1_is_canon `equiv a b`
    by (metis equiv_canon is_canon)
qed

lemma ex_catalogue:
  assumes "catalogue L q" "q' \<ge> q"
  shows "\<exists> L'. catalogue L' q'" 
  using assms
proof (induction q' arbitrary: q L)
  case 0
  then show ?case
    by auto
next
  case (Suc q')
  show ?case
  proof (cases "q = Suc q'")
    case True
    thus ?thesis
      using Suc(2)
      by auto
  next
    case False
    then obtain L' where "catalogue L' q'"
      using Suc(1)[of L "q"] Suc(2-3)
      by auto
    thus ?thesis
      using step_catalogue
      by auto
  qed
qed


lemma distinct_concat_catalogue:
  assumes "catalogue ss q"
  shows "distinct (concat (map the_catalogue [q..<q+limit]))" (is "distinct (concat ?L)")
proof (subst concat_filter_empty)
  show "distinct (concat (filter (\<lambda> x. x \<noteq> []) (map the_catalogue [q..<q + limit])))" (is "distinct (concat ?L)")
  proof (rule distinct_concat)
    show "distinct ?L"
    proof (subst filter_map, subst distinct_map, safe)
      show "distinct (filter ((\<lambda>x. x \<noteq> []) \<circ> the_catalogue) [q..<q + limit])"
        by simp
    next
      show "inj_on the_catalogue ((set (filter ((\<lambda>x. x \<noteq> []) \<circ> the_catalogue) [q..<q + limit])))"
        unfolding comp_def inj_on_def
      proof safe
        fix x y
        assume "x \<in> set (filter (\<lambda>x. the_catalogue x \<noteq> []) [q..<q + limit])"
               "y \<in> set (filter (\<lambda>x. the_catalogue x \<noteq> []) [q..<q + limit])"
        hence "x \<in> set [q..<q + limit]" "y \<in> set [q..<q + limit]"
          "the_catalogue x \<noteq> []" "the_catalogue y \<noteq> []"
          by auto
        assume *: "the_catalogue x = the_catalogue y"

        obtain k where k: "k \<in> set (the_catalogue x)"
          using `the_catalogue x \<noteq> []`
          by (cases "the_catalogue x", auto)

        have "catalogue (the_catalogue x) x"
          using ex_catalogue the_catalogue \<open>x \<in> set [q..<q + limit]\<close> assms 
          by fastforce
        hence "k \<in> S x"
          using k
          unfolding catalogue_def
          by auto
        moreover
        have "catalogue (the_catalogue y) y"
          using ex_catalogue the_catalogue \<open>y \<in> set [q..<q + limit]\<close> assms 
          by fastforce
        hence "k \<in> S y"
          using k *
          unfolding catalogue_def
          by auto
        ultimately
        show "x = y"
          using  S_disjunct[of x y]
          by auto
      qed
    qed
  next
    fix ys
    assume "ys \<in> set ?L"
    then obtain q' where "q \<le> q'" "q' < q + limit" "ys = the_catalogue q'"
      by auto
    hence "catalogue ys q'"
      using ex_catalogue the_catalogue assms 
      by fastforce
    thus "distinct ys"
      unfolding catalogue_def
      by simp
  next
    fix ys zs
    assume "ys \<in> set ?L" "zs \<in> set ?L"
    then obtain y z where *: "q \<le> z" "z < q + limit" "q \<le> y" "y < q + limit" 
      "ys = the_catalogue y" "zs = the_catalogue z"
      by auto
    assume "ys \<noteq> zs"
    hence "y \<noteq> z"
      using *
      by auto
    moreover
    have "catalogue ys y" "catalogue zs z"
      using * ex_catalogue the_catalogue assms 
      by fastforce+
    ultimately
    show "set ys \<inter> set zs = {}"
      using S_disjunct[of y z]
      unfolding catalogue_def
      by auto
  qed
qed

end

subsection \<open>Stricter correctness requirements\<close>

text \<open>If the following requirements are satisfied, then there is no need to perform the order test
after the objects are generated.\<close>

locale FaradzevReadStrict = FaradzevRead' S for S :: "nat \<Rightarrow> 's::linorder set" + 
  assumes strict1: "\<And> s q. \<lbrakk>s \<in> S (q + 1); is_canon s\<rbrakk> \<Longrightarrow> 
                            (\<exists> p \<in> S q. parent p s)"
  assumes strict2: "\<And> p s p' s' q. 
                       \<lbrakk>p \<in> S q; parent p s; p' \<in> S q; parent p' s';
                        p < p'\<rbrakk> \<Longrightarrow> s < s'"
  assumes strict3: "\<And> s q. \<lbrakk>s \<in> S q; is_canon s\<rbrakk> \<Longrightarrow> 
               sorted (filter is_canon (augment s)) \<and>
               distinct (filter is_canon (augment s))"
begin

text \<open>Optimized step, without the order test\<close>

definition step_opt where
  "step_opt L = filter is_canon_test (concat (map augment L))"

lemma strict_unique_parent:
  assumes "p \<in> S q" "p' \<in> S q" "parent p s" "parent p' s"
  shows "p = p'"
  using assms strict2
  by (meson less_irrefl neqE)

lemma strict_cond3:
  assumes "s \<in> S (q + 1)" "s' \<in> S (q + 1)"
          "min_parent p s" "min_parent p' s'" "s < s'" 
  shows "p \<le> p'"
  using assms strict2[of p' q s' p s] 
  using augment_dim[of p s q]
  using augment_dim[of p' s' q]
  unfolding min_parent_def parent_def
  by force

lemma distinct_step_opt:
  assumes "set L \<subseteq> S q \<and> Ball (set L) is_canon" "distinct L" "sorted L"
  shows "distinct (step_opt L)"
proof-
  have "distinct (filter is_canon (concat (map augment L)))"
  proof (subst filter_concat, subst map_map, subst concat_filter_empty, subst distinct_concat)
    show "distinct (filter (\<lambda>x. x \<noteq> []) (map (filter is_canon \<circ> augment) L))"
    proof (subst filter_map, subst distinct_map, safe)
      show "distinct (filter ((\<lambda>x. x \<noteq> []) \<circ> (filter is_canon \<circ> augment)) L)"
      proof (rule distinct_filter)
        show "distinct L"
          by fact
      qed
    next
      show "inj_on (filter is_canon \<circ> augment) (set (filter ((\<lambda>x. x \<noteq> []) \<circ> (filter is_canon \<circ> augment)) L))" (is "inj_on ?f ?S")
        unfolding inj_on_def
      proof safe
        fix x y
        assume *: "x \<in> ?S" "y \<in> ?S" "?f x = ?f y"
        then obtain a where
          "x \<in> set L" "a \<in> set (augment x)" "is_canon a"
          "y \<in> set L" "a \<in> set (augment y)"
          by (cases "filter is_canon (augment y)") 
             (simp, metis comp_def filter_set list.set_intros(1) member_filter)
        thus "x = y"
          using strict2 *
          using assms
          unfolding parent_def
          by (meson le_less not_le subsetD)
      qed
    qed
  next
    fix ys
    assume "ys \<in> set (filter (\<lambda>x. x \<noteq> []) (map (filter is_canon \<circ> augment) L))"
    thus "distinct ys"
      using strict3 assms
      by auto
  next
    fix ys zs
    let ?S = "set (filter (\<lambda>x. x \<noteq> []) (map (filter is_canon \<circ> augment) L))"
    assume *: "ys \<in> ?S" "zs \<in> ?S" "ys \<noteq> zs"
    thus "set ys \<inter> set zs = {}"
      using assms(1) *(3) strict2
      by auto (metis not_less_iff_gr_or_eq parent_def subset_code(1))
  next
    show True
      by simp
  qed
  thus ?thesis
    unfolding step_opt_def
    using assms
    by auto
qed

lemma sorted_step_opt:
  assumes  "set L \<subseteq> S q \<and> Ball (set L) is_canon" "distinct L" "sorted L"
  shows "sorted (step_opt L)"
  using assms
proof-
  have "sorted (filter is_canon (concat (map augment L)))"
    using assms
  proof (induction L)
    case Nil
    then show ?case
      by simp
  next
    case (Cons a L)
    have "sorted
            (filter is_canon (augment a) @
             filter is_canon (concat (map augment L)))" (is "sorted (?l1 @ ?l2)")
    proof (subst sorted_append, safe)
      show "sorted ?l1"
        using strict3 Cons.prems
        by auto
    next
      show "sorted ?l2"
        using Cons
        by simp
    next
      fix x y
      assume "x \<in> set ?l1" "y \<in> set ?l2"
      then obtain b where 
       *:"is_canon x" "x \<in> set (augment a)" 
          "a \<in> S q" "b \<in> S q" "is_canon a" "is_canon b"
          "b \<in> set L" "is_canon y" "y \<in> set (augment b)" "a \<le> b"
        using Cons.prems
        by auto
      thus "x \<le> y"
        using strict2[of a q x b y] `distinct (a # L)`
        unfolding parent_def
        by (cases "a = b", simp, force)
    qed
    thus ?case
      by simp
  qed
  thus ?thesis
    unfolding step_opt_def
    using assms
    by auto
qed


lemma step_test_distinct_sorted: 
  assumes "distinct L" "sorted L"
  shows "rev (fold step_test L []) = L"
  using assms
  by (induct L rule: rev_induct)
     (auto simp add: step_test_def sorted_append, metis less_le list.set_sel(1) set_rev)

lemma step_opt:
  assumes "distinct L" "sorted L" "set L \<subseteq> S q \<and> Ball (set L) is_canon"
  shows "step L = step_opt L"
  using assms distinct_step_opt sorted_step_opt
  unfolding step_def step_opt_def
  by (subst step_test_distinct_sorted) auto
end

sublocale FaradzevReadStrict \<subseteq> FaradzevRead
  using strict1 strict2 strict3 strict_cond3
  by unfold_locales auto

context FaradzevReadStrict
begin

primrec steps_opt where
  "steps_opt 0 ss = []"
| "steps_opt (Suc limit) ss = ss # steps_opt limit (step_opt ss)"

lemma steps_opt_catalogue:
  assumes "catalogue ss q"
  shows "\<forall> i < length (steps_opt limit ss). catalogue (steps_opt limit ss ! i) (q+i)"
  using assms
proof (induction limit arbitrary: ss q)
  case 0
  then show ?case
    by simp
next
  case (Suc limit)
  show ?case
  proof safe
    let ?s = "steps_opt (Suc limit) ss"
    fix i
    assume "i < length ?s"
    show "catalogue (?s ! i) (q + i)"
    proof (cases "i = 0")
      case True
      then show ?thesis
        using Suc
        by simp
    next
      case False
      have "catalogue (step ss) (q + 1)"
        using Suc(2) step_catalogue[of ss q]
        by simp
      moreover
      have "step ss = step_opt ss"
        using Suc(2) step_opt[of ss q]
        unfolding step_opt_def catalogue_def
        by simp
      ultimately
      have "catalogue (step_opt ss) (q + 1)"
        by simp
      show ?thesis
        using False `i < length ?s` 
        using Suc(1)[OF `catalogue (step_opt ss) (q + 1)`, rule_format, of "i-1"]
        by simp
    qed
  qed
qed

lemma steps_opt_length [simp]: 
  "length (steps_opt limit ss) = limit"
  by (induction limit arbitrary: ss) auto

end

subsection \<open>A depth-first version of the scheme\<close>

context FaradzevReadStrict
begin

fun fold_dfs :: "nat \<Rightarrow> ('s \<Rightarrow> 'a \<Rightarrow> 'a) \<Rightarrow> 'a \<Rightarrow> 's list \<Rightarrow> 'a" where
  "fold_dfs limit f i ss =
         (if limit = 0 then i
          else fold (\<lambda> s' x. f s' (fold_dfs (limit - 1) f x (filter is_canon_test (augment s')))) ss i)"

definition count_dfs :: "nat \<Rightarrow> 's list \<Rightarrow> nat" where
  "count_dfs limit ss = fold_dfs limit (\<lambda> s x. x + 1) 0 ss"

definition generate_dfs :: "nat \<Rightarrow> 's list \<Rightarrow> 's list" where
  "generate_dfs limit ss = fold_dfs limit (\<lambda> s x. s # x) [] ss"

declare fold_dfs.simps [simp del]

lemma count_generate_dfs_lemma:
  assumes "N = length xs" "inc = (\<lambda>s x. x + 1)"
  shows "fold_dfs Q inc N ss = length (fold_dfs Q (#) xs ss)"
  using assms
proof (induction Q inc N ss arbitrary: xs rule: fold_dfs.induct)
  case (1 Q inc N ss)
  show ?case
  proof (cases "Q = 0")
    case True
    thus ?thesis
      using `N = length xs`
      by (simp add: fold_dfs.simps)
  next
    case False
    let ?a = "\<lambda> s'. filter is_canon_test (augment s')"
    let ?f1 = "\<lambda>s' x. Suc (fold_dfs (Q - 1) inc x (?a s'))"
    let ?f2 = "\<lambda>s' x. s' # fold_dfs (Q - 1) (#) x (?a s')"
    obtain f1 f2 where f: "f1 = ?f1" "f2 = ?f2"
      by auto
    have "\<forall> s \<in> set ss. \<forall> ss'. f1 s (length (fold f2 ss' xs)) = length (f2 s (fold f2 ss' xs))"
      using f
      using 1(1) 1(3) False
      by simp
    hence "fold f1 ss N = length (fold f2 ss xs)"
      using `N = length xs`
      by (induction ss rule: rev_induct) auto
    thus ?thesis
      using fold_dfs.simps[of Q inc N ss] `inc = (\<lambda> s x. x + 1)`
      using fold_dfs.simps[of Q "(#)" xs ss] f `N = length xs`
      by simp
  qed
qed

lemma count_generate_dfs:
  shows "count_dfs Q ss = length (generate_dfs Q ss)"
  unfolding count_dfs_def generate_dfs_def
  using count_generate_dfs_lemma[of 0 "[]"]
  by simp

lemma fold_acc:
  assumes "\<forall> a \<in> set s. \<forall> acc. f a acc = f a [] @ acc"
  shows "fold f s acc = fold f s [] @ acc"
  using assms
proof (induction s rule: rev_induct)
  case Nil
  then show ?case
    by simp
next
  case (snoc x xs)
  have "fold f xs acc = fold f xs [] @ acc"
    using snoc
    by (metis butlast_snoc in_set_butlastD)
  then show ?case
    using snoc(2)[rule_format, of x "(fold f xs [] @ acc)"]
    using snoc(2)[rule_format, of x "fold f xs []"]
    by simp
qed

lemma fold_dfs_acc:
  assumes "app = (#)"
  shows "fold_dfs limit app acc ss = (fold_dfs limit app [] ss) @ acc"
  using assms
proof (induction limit app acc ss arbitrary: q rule: fold_dfs.induct)
  case (1 limit app acc ss)
  show ?case
  proof (cases "limit = 0")
    case True
    thus ?thesis
      using 1
      by (simp add: fold_dfs.simps)
  next
    case False
    let ?a = "\<lambda> s'. filter is_canon_test (augment s')"
    let ?f = "\<lambda> s' acc'. fold_dfs (limit - 1) app acc' (?a s')"
    obtain f where f: "f = ?f" 
      by auto
    let ?f' = "\<lambda>s' acc'. app s' (f s' acc')"

    have "fold ?f' ss acc = fold ?f' ss [] @ acc"
    proof (rule fold_acc, safe)
      fix a acc
      assume "a \<in> set ss"
      have "f a acc = f a [] @ acc"
      proof (subst f,  subst f, rule 1(1))
        show "limit \<noteq> 0" "a \<in> set ss" "app = (#)" by fact+
      qed
      thus "app a (f a acc) = app a (f a []) @ acc"
        using `app = (#)`
        by simp
    qed
    thus ?thesis
      by (simp add: fold_dfs.simps False f)
  qed
qed      

lemma fold_Cons_mset:
  assumes "mset ss = mset ss'" "\<forall> a \<in> set ss. \<forall> acc. f a acc = f a [] @ acc"
  shows "mset (fold f ss []) = mset (fold f ss' [])"
  using assms
proof (induction ss arbitrary: ss')
  case Nil
  then show ?case
    by simp
next
  case (Cons a ss)

  obtain i where "i < length ss'" "ss' ! i = a"
    using `mset (a # ss) = mset ss'`
    using in_set_conv_nth[of a ss']
    by (metis list.set_intros(1) mset_eq_setD)
  hence "ss' = take i ss' @ [a] @ drop (i+1) ss'"
    by (metis One_nat_def add.right_neutral add_Suc_right append.assoc append_take_drop_id hd_drop_conv_nth take_hd_drop)
  then obtain s1 s2 where ss': "ss' = s1 @ [a] @ s2"
    by auto
  have "set s1 \<subseteq> set (a # ss)" "set s2 \<subseteq> set (a # ss)"
    using ss' Cons(2)
    by (metis Un_iff mset_eq_setD set_append subsetI)+

  have "mset ss = mset s1 + mset s2"
    using ss' Cons(2)
    by auto
  hence "mset (fold f ss []) = mset (fold f (s1 @ s2) [])"
    using Cons(1)[of "s1 @ s2"] Cons(3)
    by (metis list.set_intros(2) mset_append)
  then show ?case
    using `ss' = s1 @ [a] @ s2`
    apply simp
    apply (subst fold_acc[of ss], meson Cons.prems(2) list.set_intros(2), simp)
    apply (subst fold_acc[of s1])
    using Cons.prems(2) \<open>set s1 \<subseteq> set (a # ss)\<close> apply blast
    apply simp
    apply (subst fold_acc[of s2])
    using Cons.prems(2) \<open>set s2 \<subseteq> set (a # ss)\<close> apply blast
    apply simp
    apply (subst fold_acc[of s2 f "(f a (fold f s1 []))"])
    using Cons.prems(2) \<open>set s2 \<subseteq> set (a # ss)\<close> apply blast
    apply simp
    apply (subst Cons.prems(2)[rule_format, of a "fold f s1 []"], simp, simp)
    done
qed

lemma fold_dfs_mset:
  assumes "app = (#)" "mset ss = mset ss'"
  shows "mset (fold_dfs limit app [] ss) = mset (fold_dfs limit app [] ss')"
proof (cases "limit = 0")
  case True
  thus ?thesis
    by (simp add: fold_dfs.simps)
next
  case False
  let ?f = "\<lambda>s' x. app s' (fold_dfs (limit - 1) app x (filter is_canon_test (augment s')))"
  have "mset (fold ?f ss []) = mset (fold ?f ss' [])"
  proof (rule fold_Cons_mset)
    show "mset ss = mset ss'"
      by fact
  next
    show "\<forall> a \<in> set ss. \<forall> acc. ?f a acc = ?f a [] @ acc"
    proof safe
      fix a acc
      show "?f a acc = ?f a [] @ acc"
        using `app = (#)` fold_dfs_acc[of app "limit - 1" acc]
        by simp
    qed
  qed
  thus ?thesis
    by (simp add: fold_dfs.simps)
qed

lemma fold_dfs_append:
  assumes "app = (#)"
  shows "fold_dfs limit app [] (ss1 @ ss2) = fold_dfs limit app (fold_dfs limit app [] ss1) ss2"
  by (induction ss1) (auto simp add: fold_dfs.simps)

lemma fold_dfs_rec:
  assumes "app = (#)" 
  shows "mset (fold_dfs limit app [] ss) = 
         (if limit = 0 then mset [] else mset ss + mset (fold_dfs (limit-1) app [] (step_opt ss)))"
proof (cases "limit = 0")
  case True
  thus ?thesis
    by (simp add: fold_dfs.simps)
next
  case False
  let ?a = "\<lambda> s. filter is_canon_test (augment s)"
  let ?ff = "\<lambda> s acc. fold_dfs (limit - 1) app acc s"
  let ?f = "\<lambda> s acc. ?ff (?a s) acc"
  let ?f' = "\<lambda> s acc. app s (?f s acc)"
  obtain ff where ff: "ff = ?ff" by auto
  obtain f where f: "f = ?f" by auto
  obtain f' where f': "f' = ?f'" by auto

  have "mset (fold f' ss []) =
        mset ss + mset (ff (step_opt ss) [])"
  proof (induction ss rule: rev_induct)
    case Nil
    then show ?case
      using ff
      by (simp add: step_opt_def fold_dfs.simps)
  next
    case (snoc a ss)
    have f'a: "f' a [] = a # (f a [])"
      using f f' `app = (#)`
      by simp

    have *: "\<forall> x \<in> set (a # ss). \<forall> acc. f' x acc = f' x [] @ acc"
    proof safe
      fix x acc
      assume "x \<in> set (a # ss)"
      show "f' x acc = f' x [] @ acc"
        by (subst f', subst f', subst fold_dfs_acc, simp_all add: `app = (#)`)
    qed

    have **: "mset (ff (step_opt (ss @ [a])) []) = mset (ff (step_opt ss @ ?a a) [])"
      unfolding step_opt_def ff
      by simp
    also have "... = mset (ff (?a a @ step_opt ss) [])"
    proof (subst ff, subst ff, subst fold_dfs_mset)
      show "mset (step_opt ss @ ?a a) = mset (?a a @ step_opt ss)"
        by simp
    next
      show "app = (#)" by fact
    qed simp
    also have "... = mset (f a []) + mset (ff (step_opt ss) [])"
      apply (subst ff, subst fold_dfs_append, simp add: `app = (#)`)
      apply (subst fold_dfs_acc[where acc="fold_dfs (limit - 1) app [] (filter is_canon_test (augment a))"])
      apply (simp add: `app = (#)`, simp add: f ff)
      done
    finally
    show ?case
      using snoc
      using * ** f'a
      by simp
  qed
  thus ?thesis
    using False
    by (simp add: f ff f' fold_dfs.simps)
qed

lemma generate_dfs_rec:
  "mset (generate_dfs limit ss) = 
    (if limit = 0 then mset []
     else mset ss + mset (generate_dfs (limit-1) (step_opt ss)))"
  unfolding generate_dfs_def
  by (simp add: fold_dfs_rec)

lemma generate_dfs_mset:
  shows "mset (generate_dfs limit ss) = sum_list (map mset (steps_opt limit ss))"
  by (induction limit arbitrary: ss) (simp_all add: generate_dfs_rec)


lemma generate_dfs_mset_catalogue:
  assumes "catalogue ss q"
  shows "mset (generate_dfs limit ss) = 
         sum_list (map (mset \<circ> the_catalogue) [q..<q+limit])"
proof-
  have "map mset (steps_opt limit ss) = map (mset \<circ> the_catalogue) [q..<q + limit]" (is "?lhs = ?rhs")
  proof (rule nth_equalityI)
    show "length ?lhs = length ?rhs"
      by simp
  next
    fix i
    assume "i < length ?lhs"
    hence "catalogue (steps_opt limit ss ! i) (q + i)"
      using steps_opt_catalogue[OF assms, of limit]
      by auto
    hence "steps_opt limit ss ! i = the_catalogue (q + i)"
      using the_catalogue
      by simp
    thus "?lhs ! i = ?rhs ! i"
      using `i < length ?lhs`
      by simp
  qed
  thus ?thesis
    using assms
    by (simp add: generate_dfs_mset)
qed

lemma sum_list_catalogue: 
  shows "sum_list (map (mset \<circ> the_catalogue) [q..<q+limit]) = 
       mset (concat (map the_catalogue [q..<q+limit]))"
  by (induction limit, auto)


lemma mset_eq_distinct:
  assumes "mset x = mset y" "distinct x"
  shows "distinct y"
  using assms
  by (metis distinct_count_atmost_1 mset_eq_setD)

lemma generate_dfs_distinct:
  assumes "catalogue ss q"
  shows "distinct (generate_dfs limit ss)"
proof (rule mset_eq_distinct)
  show "distinct (concat (map the_catalogue [q..<q+limit]))"
    using assms
    using distinct_concat_catalogue by blast
next
  show "mset (concat (map the_catalogue [q..<q + limit])) = mset (generate_dfs limit ss)"
    using generate_dfs_mset_catalogue sum_list_catalogue
    using assms
    by fastforce
qed


end

end