section \<open>Cataloguing undirected graphs\<close>

subsection \<open>Graph representation\<close>

theory Graph
  imports Main More_List More_Multiset Combinatorics "List_Lexorder_gt" Canon_Least_Perm
begin

subsubsection\<open>Adjacency matrices\<close>

type_synonym 'a matrix = "'a list list"
                                           
(* adjacency matrix made out of nodes and a list of edges *)
definition make_matrix :: "nat \<Rightarrow> (nat \<times> nat) list \<Rightarrow> bool matrix" where
 "make_matrix v edges = map (\<lambda> i. map (\<lambda> j. (i, j) \<in> set edges \<or> (j, i) \<in> set edges) [0..<v]) [0..<v]"

(* check for square matrix *)
definition is_square_matrix :: "nat \<Rightarrow> 'a matrix \<Rightarrow> bool" where
  "is_square_matrix n M \<longleftrightarrow> length M = n \<and> (\<forall> r \<in> set M. length r = n)"

(* list of all edges possible edges in a graph with v nodes *)
definition all_edges where
  "all_edges v = concat (map (\<lambda> i. map (\<lambda> j. (j, i)) [0..<i]) [1..<v])"

(* list of edges for a given adjacency matrix *)
definition read_matrix where
  "read_matrix M = 
     (let v = length M 
       in filter (\<lambda> (i, j). M ! i !  j) (all_edges v))"

(* diagonal elements of a matrix *)
definition diag where
  "diag M = map (\<lambda> (i, r). r ! i) (zip [0..<length M] M)"

lemma diag_Nil [simp]: "diag [] = []"
  unfolding diag_def
  by simp

lemma diag_snoc [simp]: "diag (M @ [r]) = diag M @ [r ! length M]"
  unfolding diag_def
  by simp

lemma ii_diag [simp]: 
  assumes "i < length M" "is_square_matrix n M"
  shows "M ! i ! i \<in> set (diag M)"
  using assms
  unfolding diag_def is_square_matrix_def
  by (auto simp add: set_zip) (rule image_eqI, auto)
  

(* multiset of matrix elements *)
definition mat_mset :: "'a matrix \<Rightarrow> 'a multiset" where
   "mat_mset M = sum_list (map mset M)"

lemma diag_subset:
  assumes "\<forall> r \<in> set M. length r \<ge> length M"
  shows "mset (diag M) \<subseteq># mat_mset M"
  using assms
  apply (induction M rule: rev_induct)
  apply (auto simp add: mat_mset_def diag_def)
  apply (metis Suc_leD add_mset_add_single le_simps(3) nth_mem set_mset_mset single_subset_iff subset_mset.add_mono)
  done

subsubsection\<open>Action of permutations on matrices\<close>

definition permute_matrix :: "perm \<Rightarrow> 'a matrix \<Rightarrow> 'a matrix" where
  "permute_matrix p M = permute_list p (map (permute_list p) M)"

lemma is_square_permute_matrix [simp]:
  assumes "is_square_matrix n M" "length p = n" "set p = {0..<n}"
  shows "is_square_matrix n (permute_matrix p M)"
  using assms                                                        
  unfolding permute_matrix_def is_square_matrix_def
  by auto

lemma permute_matrix_id [simp]:
  assumes "is_square_matrix n M"
  shows "permute_matrix [0..<n] M = M"
  using assms
  unfolding permute_matrix_def is_square_matrix_def
  by auto (metis map_idI map_nth permute_list_def)

lemma nth_permute_matrix [simp]:
  assumes "is_square_matrix n M" "i < n" "j < n" "set p = {0..<n}" "length p = n"
  shows "permute_matrix p M ! i ! j = M ! (p ! i) ! (p ! j)"
proof-
  have "p ! i < n"
    using assms
    by auto
  thus ?thesis
    using assms
    unfolding permute_matrix_def permute_list_def is_square_matrix_def
    by auto
qed
  
lemma mat_mset_permute_matrix [simp]:
  assumes "is_square_matrix n M" "p \<in> set (permute [0..<n])"  (* equiv to "length p = n" "set p = {0..<n}" *)
  shows "mat_mset (permute_matrix p M) = mat_mset M"
  unfolding permute_matrix_def mat_mset_def
proof (subst map_mset_permute_list)
  show "p \<in> set (permute [0..<length (map (permute_list p) M)])"
    using assms(1-2) 
    unfolding is_square_matrix_def
    by simp
next
  show "sum_list (permute_list p (map mset (map (permute_list p) M))) = sum_list (map mset M)"
  proof-
    have *: "map mset (map (permute_list p) M) = map mset M"
      using assms
      unfolding is_square_matrix_def     
      by auto
    thus ?thesis
      using assms(1-2)
      unfolding is_square_matrix_def
      by (metis length_map permute_list_mset sum_mset_sum_list)
  qed
qed

lemma diag_permute_matrix_lemma1:
  assumes "length M = length p" "p \<in> set (permute [0..<length p])"
  shows "map2 (\<lambda> i r. r ! i) [0..<length M] (permute_list p M) = 
         permute_list p (map2 (\<lambda> i r. r ! i) (perm_inv p) M)" (is "?lhs = ?rhs")
proof (subst list_eq_iff_nth_eq, safe)
  show "length ?lhs = length ?rhs"
    using assms
    by simp
next
  fix i
  assume "i < length ?lhs"

  hence "p ! i < length (perm_inv p)" "p ! i < length M"
    using assms
    using permute_member_set
    by fastforce+

  thus "?lhs ! i = ?rhs ! i"
    using `i < length ?lhs`
    using assms nth_zip[of "p ! i" "perm_inv p" "[0..<length M]"]
    using perm_perm_inv[of p p i]
    unfolding permute_list_def
    by auto
qed

lemma diag_permute_matrix_lemma2':
  assumes "p1 \<in> set (permute [0..<length p1])" "length p1 = length p2"
  shows "map2 (\<lambda>i r. r ! i) p1 (map (permute_list p2) M) = 
         map2 (\<lambda>i r. r ! i) (permute_list p1 p2) M" (is "?lhs = ?rhs")
proof (subst list_eq_iff_nth_eq, safe)
  show "length ?lhs = length ?rhs"
    by simp
next
  fix i
  assume "i < length ?lhs"
  hence *: "i < length p1" "i < length M"
    by auto
  hence "p1 ! i < length p2"
    using assms permute_member_set
    by fastforce
  thus "?lhs ! i = ?rhs ! i"
    using assms *
    unfolding permute_list_def
    by auto
qed

lemma diag_permute_matrix_lemma2:
  assumes "p \<in> set (permute [0..<length p])"
  shows "map2 (\<lambda>x y. y ! x) (perm_inv p) (map (permute_list p) M) = map2 (\<lambda>i r. r ! i) [0..<length p] M"
  using assms
  using diag_permute_matrix_lemma2'[of "perm_inv p" p M] permute_list_perm_inv2[of p]
  by (metis (full_types) perm_inv_length mset_eq_perm permute_list_mset permute_isPermutation permute_list_perm_inv1)

lemma diag_permute_matrix [simp]:
  assumes "is_square_matrix n M" "p \<in> set (permute [0..<n])"
  shows "diag (permute_matrix p M) = permute_list p (diag M)"
proof-
  have "length p = n" "set p = {0..<n}"
    using assms(2) permute_member_length[of p "[0..<n]"] permute_member_set[of p "[0..<n]"]
    by auto

  have "diag (permute_matrix p M) = map2 (\<lambda>i r. r ! i) [0..<length M] (permute_matrix p M)"
    using assms `length p = n`
    unfolding diag_def permute_matrix_def is_square_matrix_def
    by auto
  also have "... = map2 (\<lambda>i r. r ! i) [0..<length M] (permute_list p (map (permute_list p) M))"
    unfolding permute_matrix_def
    by simp
  also have "... = permute_list p (map2 (\<lambda>x y. y ! x) (perm_inv p) (map (permute_list p) M))"
    using diag_permute_matrix_lemma1[of "(map (permute_list p) M)" p] assms `length p = n`
    unfolding is_square_matrix_def
    by simp
  also have "... = permute_list p (map2 (\<lambda> x y. y ! x) [0..<length p] M)"
    using diag_permute_matrix_lemma2[of p M] assms `length p = n`
    by simp
  finally show ?thesis
    using assms(1) `length p = n`
    unfolding is_square_matrix_def
    by (simp add: diag_def)
qed

lemma permute_matrix_comp [simp]:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])" "is_square_matrix n M"
  shows "permute_matrix (perm_comp p1 p2) M = permute_matrix p2 (permute_matrix p1 M)" (is "?lhs = ?rhs")
proof (subst list_eq_iff_nth_eq, safe)
  have "length ?lhs = n"
    using is_square_permute_matrix[OF assms(3), of "perm_comp p1 p2"]
    unfolding is_square_matrix_def
    using assms(1) assms(2) permute_member_length permute_member_set
    by (force simp add: perm_comp_def)
  moreover
  have "length ?rhs = n"
    using is_square_permute_matrix[OF assms(3), of p1]
    using is_square_permute_matrix[of n "permute_matrix p1 M" p2]
    unfolding is_square_matrix_def
    by (metis assms(1) assms(2) diff_zero permute_member_length length_upt permute_member_set set_upt)
  ultimately
  show "length ?lhs = length ?rhs"
    by simp

  fix i
  assume "i < length ?lhs"

  hence "i < n"
    using `length ?lhs = n`
    by simp
  moreover
  hence "p2 ! i < n"
    using assms(2) permute_member_length permute_member_set by fastforce
  moreover
  hence "p1 ! (p2 ! i) < n"
    using assms(1) permute_member_length permute_member_set by fastforce
  moreover
  have "length (permute_list p1 (map (permute_list p1) M)) = n"
    using assms(1) permute_member_length by force
  moreover
  have "length M = n"
    using assms(3)
    unfolding is_square_matrix_def
    by simp
  ultimately
  show "?lhs ! i = ?rhs ! i"
    using `length ?lhs = n` `length ?rhs = n` 
    unfolding permute_matrix_def
    by (simp add: perm_comp_def)
       (smt assms(1) assms(2) assms(3) is_square_matrix_def nth_mem perm_comp_def permute_list_perm_comp)
qed

subsubsection\<open>Digraph lists\<close>

definition mat2dig :: "'a matrix \<Rightarrow> 'a list" where
  "mat2dig M = concat (map (\<lambda> (row, i). take i row @ drop (i+1) row) (zip M [0..<length M]))"

function dig2mat' :: "nat \<Rightarrow> nat \<Rightarrow> ('a::one) list \<Rightarrow> ('a::one) matrix" where
 "dig2mat' n i xs = 
    (if i \<ge> n then [] 
    else 
      let row = take (n-1) xs
       in (take i row @ [1] @ drop i row) # dig2mat' n (i+1) (drop (n-1) xs))"
  by pat_completeness auto
termination by (relation "measure (\<lambda> (n, i, xs). n + 1 - i)", auto)

definition dig2mat :: "nat \<Rightarrow> ('a::one) list \<Rightarrow> ('a::one) matrix" where
  "dig2mat n M = dig2mat' n 0 M"

declare dig2mat'.simps [simp del]

(* Properties of mat2dig *)

lemma length_mat2dig [simp]:
  assumes "is_square_matrix n M"
  shows "length (mat2dig M) = n*(n-1)"
proof-
  let ?f =  "\<lambda>(row, i). take i row @ drop (Suc i) row"
  let ?L = "zip M [0..<length M]"
  have "length (mat2dig M) = sum_list (map (length \<circ> ?f) ?L)"
    unfolding mat2dig_def                            
    by (auto simp add: length_concat)
  also have "... = (n - 1) * length ?L"
  proof (rule sum_list_const)
    show "\<forall> x \<in> set ?L. (length \<circ> ?f) x = n - 1"
      using assms
      unfolding is_square_matrix_def
      by (auto simp add: set_zip)
  qed
  finally
  show ?thesis
    using assms
    unfolding is_square_matrix_def
    by simp
qed

(* TODO: do we need induction *)
lemma mset_mat2dig' [simp]:
  assumes "\<forall> r \<in> set M. length r \<ge> length M"
  shows "mset (mat2dig M) = mat_mset M - mset (diag M)"
  using assms
proof (induction M rule: rev_induct)
  case Nil
  thus ?case
    by (simp add: mat2dig_def mat_mset_def)
next
  case (snoc x xs)
  have "mset (mat2dig (xs @ [x])) = mset (mat2dig xs) + (mset (take (length xs) x) + mset (drop (Suc (length xs)) x))"
    by (simp add: mat2dig_def)
  moreover
  have "length xs < length x"
    using snoc(2)
    by auto
  ultimately
  have "mset (mat2dig (xs @ [x])) = mset (mat2dig xs) + mset x - {#x ! length xs#}"
    using mset_take_drop[of "length xs" x]
    by simp
  moreover
  have "\<forall>r\<in>set xs. length xs \<le> length r"
    using snoc(2)
    by auto
  hence "mset (mat2dig xs) = mat_mset xs - mset (diag xs)"
    using snoc(1)
    by simp
  moreover
  have "\<forall>r\<in>set xs. length xs \<le> length r"
    using snoc(2)
    by auto
  hence "mset (diag xs) \<subseteq># mat_mset xs"
    using diag_subset[of xs]
    by auto
  ultimately
  show ?case
    by (simp add: mat_mset_def)
qed

lemma mset_mat2dig [simp]:
  assumes "is_square_matrix n M"
  shows "mset (mat2dig M) = mat_mset M - mset (diag M)"
  using assms
  using mset_mat2dig'[of M]
  unfolding is_square_matrix_def
  by auto

lemma hd_mat2dig:
  assumes "is_square_matrix n M" "n > 0"
  shows "take (n-1) (mat2dig M) = tl (hd M)"
proof-
  let ?f = "\<lambda>row i. take i row @ drop (Suc i) row"
  have "map2 ?f M [0..<length M] = map2 ?f (hd M # tl M) (0 # [1..<length M])"
    using assms
    unfolding is_square_matrix_def
    using upt_conv_Cons[of 0 "length M"] hd_Cons_tl[of M] 
    by fastforce
  hence "concat (map2 ?f M [0..<length M]) = tl (hd M) @ concat (map2 ?f (tl M) [1..<length M])"
    using assms
    by (simp add: drop_Suc)
  thus ?thesis
    unfolding mat2dig_def
    using assms
    unfolding is_square_matrix_def
    by auto
qed

lemma mat2dig_nth:
  assumes "is_square_matrix n M" "i < n * (n-1)"
  shows "mat2dig M ! i = 
               (if i mod (n-1) < i div (n-1) then 
                       M ! (i div (n-1)) ! (i mod (n-1))
                else
                       M ! (i div (n-1)) ! (i mod (n-1) + 1)
               )" (is "?lhs = ?rhs")
  unfolding mat2dig_def
proof (subst concat_nth)
  show " \<forall>x\<in>set (map2 (\<lambda>row i. take i row @ drop (i + 1) row) M [0..<length M]). length x = n - 1"
    using `is_square_matrix n M`
    unfolding is_square_matrix_def
    by (auto simp add: set_zip min_def)
next
  show "i < (n - 1) * length (map2 (\<lambda>row i. take i row @ drop (i + 1) row) M [0..<length M])"
    using `i < n * (n - 1)`
    using `is_square_matrix n M`
    unfolding is_square_matrix_def
    by (simp add: mult.commute)
next
  have "(i div (n - 1)) < length M"
    using `is_square_matrix n M`
    unfolding is_square_matrix_def
    using assms(2) less_mult_imp_div_less by auto
  thus "map2 (\<lambda>row i. take i row @ drop (i + 1) row) M [0..<length M] ! (i div (n - 1)) ! (i mod (n - 1)) = ?rhs"
    using `is_square_matrix n M`
    unfolding is_square_matrix_def
    by (auto simp add: nth_append min_def)
qed

(* Properties of dig2mat *)

lemma length_dig2mat':
  assumes "length l = (n-i) * (n-1)"
  shows "length (dig2mat' n i l) = (n-i) \<and> (\<forall> r \<in> set (dig2mat' n i l). length r = n)"
  using assms
proof (induction n i l rule: dig2mat'.induct)
  case (1 n i l)
  show ?case
  proof (cases "i \<ge> n")
    case True
    thus ?thesis
      using 1(2)
      by (simp add: dig2mat'.simps)
  next
    case False
    have "length (drop (n - 1) l) = (n - (i + 1)) * (n - 1)"
      using 1(2)
      by (simp add: False mult_eq_if)
    hence "length (dig2mat' n (i + 1) (drop (n - 1) l)) = n - (i + 1) \<and>
           (\<forall>r\<in>set (dig2mat' n (i + 1) (drop (n - 1) l)). length r = n)"
      using 1(1)[of "take (n-1) l"] False
      by blast
    moreover 
    have "dig2mat' n i l = (take i (take (n-1) l) @ [1] @ drop i (take (n-1) l)) # dig2mat' n (i + 1) (drop (n - 1) l)"
      using False dig2mat'.simps[of n i l] 
      by simp
    moreover
    have "length (take i (take (n - 1) l) @ [1] @ drop i (take (n - 1) l)) = n"
      using False 1(2)
      by (auto simp add: min_def) (simp add: mult_eq_if)+
    ultimately
    show ?thesis
      by (metis False One_nat_def Suc_diff_Suc add.commute leI list.size(4) plus_1_eq_Suc set_ConsD)
  qed
qed

lemma is_square_dig2mat [simp]:
  assumes "length l = n * (n-1)"
  shows "is_square_matrix n (dig2mat n l)"
  using assms length_dig2mat'[of l n 0]
  unfolding dig2mat_def is_square_matrix_def
  by simp

lemma length_dig2mat [simp]:
  assumes "length l = n * (n-1)"
  shows "length (dig2mat n l) = n"
  using assms is_square_dig2mat
  unfolding is_square_matrix_def
  by blast

lemma nth_dig2mat':
  assumes "k + i < n" "length M = (n-i) * (n-1)" 
  shows "dig2mat' n i M ! k = 
           (let elems = take (n - 1) (drop (k * (n - 1)) M)
             in take (k+i) elems @ [1] @ drop (k+i) elems)"
  using assms
proof (induction n i M arbitrary: k rule: dig2mat'.induct)
  case (1 n i xs)
  have "i < n"
    using `k + i < n`
    by auto
  hence *: "dig2mat' n i xs = (let row = take (n - 1) xs in (take i row @ [1] @ drop i row) # dig2mat' n (i + 1) (drop (n - 1) xs))"
    using dig2mat'.simps[of n i xs]
    by simp
  show ?case
  proof (cases "k = 0")
    case True
    thus ?thesis
      using *
      by simp
  next
    case False
    hence **: "k - 1 + (i + 1) < n" "length (drop (n - 1) xs) = (n - (i + 1)) * (n - 1)"   
      using 1(2) 1(3) `i < n`
      by (auto simp add: algebra_simps)
    hence "dig2mat' n (i + 1) (drop (n - 1) xs) ! (k - 1) =
        (let elems = take (n - 1) (drop ((k - 1) * (n - 1)) (drop (n - 1) xs))
          in take (k - 1 + (i + 1)) elems @ [1] @ drop (k - 1 + (i + 1)) elems)"
      using 1(1)[OF _ _ **, of "take (n-1) xs"] 1(2) 1(3) `k \<noteq> 0` `i < n`
      using linorder_not_le
      by blast
    thus ?thesis
      using * `k + i < n` `k \<noteq> 0`
      by (smt Nat.add_diff_assoc2 One_nat_def Suc_leI add.commute add_Suc_right add_diff_cancel_left' drop_drop mult_eq_if neq0_conv nth_Cons' plus_1_eq_Suc)
  qed
qed

lemma nth_dig2mat:
  assumes "k < n" "length M = n * (n - 1)" 
  shows "dig2mat n M ! k = 
           (let elems = take (n - 1) (drop (k * (n - 1)) M)
             in take k elems @ [1] @ drop k elems)"
  using assms nth_dig2mat'[of k 0 n M]
  unfolding dig2mat_def
  by auto

lemma hd_dig2mat:
  assumes "n > 0" "length M = n * (n - 1)"
  shows "hd (dig2mat n M) = 1 # take (n - 1) M"
  using assms nth_dig2mat[of 0 n M]
  by (auto simp add: dig2mat_def dig2mat'.simps)

lemma nth_dig2mat_special:
  assumes "i < n" "j < n" "i \<noteq> j"
  shows "dig2mat n [0..<n*(n-1)] ! i ! j = (if i < j then (n-1)*i + j - 1 else (n-1)*i + j)" (is "?lhs = ?rhs")
proof (subst nth_dig2mat)
  show "i < n"
    by fact
next
  let ?elems = "take (n - 1) (drop (i * (n - 1)) [0..<n * (n - 1)])"
  have "length ?elems = n - 1"
    using `i < n`
    apply (auto simp add: min_def)
    by (metis (no_types, hide_lams) One_nat_def Suc_le_mono add_0_right assms(1) diff_is_0_eq diff_mult_distrib le_antisym less_Suc0 less_not_refl  mult.commute mult_0_right mult_Suc_right mult_le_mono2 not_less_eq zero_less_Suc zero_less_diff)
  hence "(take i ?elems @ [1] @ drop i ?elems) ! j = ?rhs"
    using `i \<noteq> j` `i < n` `j < n`
    apply (cases "j < i", simp add: nth_append)
    apply (simp add: nth_append)
    apply (auto split: if_split_asm simp add: min_def mult.commute)
    done
  thus "(let elems = take (n - 1) (drop (i * (n - 1)) [0..<n * (n - 1)]) in take i elems @ [1] @ drop i elems) ! j = ?rhs"
    by simp
qed simp

lemma dig2mat_map:
  assumes "length M = n * (n - 1)"
  shows "dig2mat n M = 
         map (\<lambda> k. let elems = take (n-1) (drop (k*(n-1)) M) 
                    in take k elems @ [1] @ drop k elems) [0..<n]" (is "?l = ?r")
proof (rule nth_equalityI)
  show "length ?l = length ?r"
    using assms
    by simp
next
  fix k
  assume "k < length (dig2mat n M)"
  thus "?l ! k = ?r ! k"
    using length_dig2mat[OF assms]
    unfolding is_square_matrix_def
    using assms
    using nth_dig2mat[of k n M]
    by simp
qed

(* TODO: induction could be avoided using dig2mat_map *)
lemma mat_mset_dig2mat':
  assumes "length M = (n-i) * (n-1)" "i \<le> n"
  shows "mat_mset (dig2mat' n i M) = mset M + mset (replicate (n-i) 1)"
  using assms
proof (induction n i M rule: dig2mat'.induct)
  case (1 n i xs)
  show ?case
  proof (cases "i = n")
    case True
    thus ?thesis
      using 1(2)
      by (simp add: dig2mat'.simps mat_mset_def)
  next
    case False
    have "i \<le> length xs"
      using 1(2-3) `i \<noteq> n`
      by (auto simp add: algebra_simps) (metis Suc_leI add.commute le_neq_implies_less mult_le_mono order_refl ordered_cancel_comm_monoid_diff_class.le_add_diff plus_1_eq_Suc semiring_normalization_rules(2))

    have "i \<le> n-1"
      using 1(2-3) `i \<noteq> n`
      by auto

    have "length xs \<ge> n-1"
      using 1(2-3) `i \<noteq> n`
      by auto

    have "add_mset 1 (mset (take (n - Suc 0) xs) + (mset (drop (n - Suc 0) xs) + mset (replicate (n - Suc i) 1))) = 
          add_mset 1 (mset xs) + mset (replicate (n - Suc i) 1)"
      using mset_take_drop_id[of "n-1" xs] `length xs \<ge> n-1`
      by simp
    also have "... = mset xs + add_mset 1 (mset (replicate (n - Suc i) 1))"
      by simp
    also have "... = mset xs +  mset (replicate (n - i) 1)"
      using `i \<le> n` `i \<noteq> n`
      by (metis Suc_diff_Suc le_neq_implies_less mset.simps(2) replicate_Suc)
    finally have "add_mset 1 (mset (take (n - Suc 0) xs) + (mset (drop (n - Suc 0) xs) + mset (replicate (n - Suc i) 1))) = 
                  mset xs + mset (replicate (n - i) 1)"
      by simp

    moreover

    have "mat_mset (dig2mat' n (i + 1) (drop (n - 1) xs)) = mset (drop (n - 1) xs) + mset (replicate (n - (i + 1)) 1)"
      using 1(1)[of "take (n-1) xs"] 1(2-3) `i \<noteq> n`
      by (auto simp add: algebra_simps)

    ultimately

    show ?thesis
      using dig2mat'.simps[of n i xs] False 1(2-3)
      using mset_take_drop_id[of i "take (n-1) xs"] `i \<le> length xs` `i \<le> n-1`
      by (auto simp add: mat_mset_def)
  qed
qed

lemma mat_mset_dig2mat [simp]:
  assumes "length M = n * (n-1)"
  shows "mat_mset (dig2mat n M) = mset M + mset (replicate n 1)"
  using assms mat_mset_dig2mat'[of M n 0]
  unfolding dig2mat_def
  by simp

(* TODO: induction could be avoided using dig2mat_map *)
lemma diag_dig2mat':
  assumes "length M = (n-i)*(n-1)" "i \<le> n"
  shows "map2 (\<lambda> i r. r ! i) [i..<n] (dig2mat' n i M) = replicate (n-i) 1"
  using assms
proof (induction n i M rule: dig2mat'.induct)
  case (1 n i xs)
  show ?case
  proof (cases "i = n")
    case True
    thus ?thesis
      by simp
  next
    case False
    have "i \<le> length xs"
      using 1(2-3) `i \<noteq> n`
      by (auto simp add: algebra_simps) (metis Suc_leI add.commute le_neq_implies_less mult_le_mono order_refl ordered_cancel_comm_monoid_diff_class.le_add_diff plus_1_eq_Suc semiring_normalization_rules(2))

    have "i \<le> n-1"
      using 1(2-3) `i \<noteq> n`
      by auto

    hence "[i..<n] = i # [i+1..<n]"
      by (simp add: False le_neq_implies_less upt_conv_Cons)
    hence "map2 (\<lambda>x y. y ! x) [i..<n] (dig2mat' n i xs) = 
          1 # map2 (\<lambda> x y. y ! x) [i+1..<n] (dig2mat' n (i+1) (drop (n-1) xs))"
      using dig2mat'.simps[of n i xs] `i \<le> n - 1` `i \<noteq> n` `i \<le> length xs`
      by (auto simp add: min_def nth_append)
    thus ?thesis
      using 1(1)[of "take (n-1) xs"] 1(2-3) `i \<noteq> n` `i \<le> n - 1`
      by (simp add: algebra_simps min_def)
         (metis Suc_diff_Suc le_neq_implies_less replicate_Suc)
  qed
qed

lemma diag_dig2mat [simp]: 
  assumes "length M = n * (n-1)"
  shows "diag (dig2mat n M) = replicate n 1"
  using assms
  using diag_dig2mat'[of M n 0] length_dig2mat[of M n]
  unfolding diag_def dig2mat_def
  by simp

(* Connection between mat2dig and dig2mat *)

lemma mat2dig_dig2mat_lemma:
  assumes "length s = n * (n - 1)" "n > 0" "m \<le> n"
  shows "concat (map (\<lambda>k. take (n - 1) (drop (k * (n - 1)) s)) [0..<m]) = take (m * (n - 1)) s"
  using assms
proof (induction m)
  case 0
  thus ?case
    by simp
next
  case (Suc m)
  thus ?case
    by auto (metis Nat.add_diff_assoc2 Suc_leI add.commute assms(2) take_add)  
qed

lemma mat2dig_dig2mat [simp]:
  assumes "length s = n * (n - 1)" 
  shows "mat2dig (dig2mat n s) = s"
proof (cases "n = 0")
  case True
  thus ?thesis
    using assms
    by (simp add: dig2mat_def dig2mat'.simps mat2dig_def)
next
  case False

  let ?f = "\<lambda>row i. take i row @ drop (i + 1) row"
  let ?h = "\<lambda> k. take (n - 1) (drop (k * (n - 1)) s)"
  let ?g = "\<lambda> k. let elems = ?h k 
                  in take k elems @ [1] @ drop k elems"
  have *: "\<forall> k < n. ?f (?g k) k = ?h k"
  proof safe
    fix k
    assume "k < n"
    have "take k (?g k) = take k (?h k)"
      using `k < n` `length s = n * (n - 1)`
      by simp (smt Nat.le_diff_conv2 Suc_pred diff_le_self gr_implies_not0 le_SucE length_greater_0_conv less_imp_le_nat less_le list.size(3) list_eq_iff_nth_eq mult.commute mult_Suc_right mult_is_0 mult_le_cancel2 nat_0_less_mult_iff nat_neq_iff)
    moreover
    have "drop (k + 1) (?g k) = drop k (?h k)"
      using `k < n` `length s = n * (n - 1)`
      by (simp add: min_def)
    ultimately
    show "?f (?g k) k = ?h k"
      by (metis append_take_drop_id)
  qed
  have "mat2dig (dig2mat n s) = concat (map (\<lambda> k. take (n - 1) (drop (k * (n - 1)) s)) [0..<n])"
    using assms
    unfolding mat2dig_def
    apply (subst length_dig2mat[OF assms])
    apply (subst dig2mat_map[OF assms])
    apply (subst map2_map)
    using * map_cong[of "[0..<n]" "[0..<n]" "\<lambda> k. ?f (?g k) k" "?h"]
    by (metis (no_types, lifting) atLeastLessThan_iff set_upt)
  thus ?thesis
    using assms `n \<noteq> 0`
    using mat2dig_dig2mat_lemma[of s n n]
    by simp
qed

lemma dig2mat_mat2dig:
  assumes "is_square_matrix n M" "diag M = replicate n 1"
  shows "dig2mat n (mat2dig M) = M"
proof (rule nth_equalityI)
  show "length (dig2mat n (mat2dig M)) = length M"
    by (metis assms(1) is_square_matrix_def length_dig2mat length_mat2dig)
next
  fix i
  assume "i < length (dig2mat n (mat2dig M))"
  hence "i < n"
    using assms(1) length_mat2dig
    by fastforce
  show "dig2mat n (mat2dig M) ! i = M ! i"
  proof (subst nth_dig2mat)
    show "i < n"
      by fact
  next
    show "length (mat2dig M) = n * (n-1)"
      using assms(1) length_mat2dig
      by blast
  next
    let ?elems = "take (n - 1) (drop (i * (n - 1)) (mat2dig M))"
    let ?M = "map2 (\<lambda>row i. take i row @ drop (i + 1) row) M [0..<length M]"
    have "?elems = ?M ! i"
      unfolding mat2dig_def
    proof (subst take_drop_concat)
      show "\<forall> x \<in> set ?M. length x = n - 1"
        using assms(1)
        by (auto simp add: set_zip min_def is_square_matrix_def)

      show "i < length ?M"
        using `i < n` assms(1)
        by (simp add: is_square_matrix_def)
    qed simp

    moreover

    have "take i (M ! i) @ 1 # drop (Suc i) (M ! i) = M ! i"
    proof-
      have "M ! i ! i = 1"
        using `i < n` `is_square_matrix n M` `diag M = replicate n 1`
        by (metis ii_diag in_set_replicate is_square_matrix_def)
      thus ?thesis
        by (metis \<open>i < n\<close> assms(1) id_take_nth_drop is_square_matrix_def nth_mem)
    qed

    ultimately
        
    show "(let elems = take (n - 1) (drop (i * (n - 1)) (mat2dig M)) in take i elems @ [1] @ drop i elems) = M ! i"
      using `i < n` `is_square_matrix n M`
      by (simp add: Let_def is_square_matrix_def)
  qed
qed


subsubsection \<open>Permutations of digraphs\<close>

definition permute_dig where
  "permute_dig p n l \<equiv> mat2dig (permute_matrix p (dig2mat n l))"

lemma mset_permute_dig[simp]:
  assumes "length xs = n * (n-1)" "p \<in> set (permute [0..<n])" 
  shows "mset (permute_dig p n xs) = mset xs"
proof-
  let ?M = "dig2mat n xs"
  let ?pM = "permute_matrix p ?M"
  have "is_square_matrix n ?pM"
    using is_square_permute_matrix[OF is_square_dig2mat[OF assms(1)], of p]
    using assms(2) permute_member_length[of p "[0..<n]"] permute_member_set[of p "[0..<n]"]
    by simp
  hence "mset (permute_dig p n xs) = mat_mset ?pM - mset (diag ?pM)"
    using mset_mat2dig[of n ?pM]
    unfolding permute_dig_def
    by simp
  moreover
  have "mat_mset ?pM = mat_mset ?M"
    using mat_mset_permute_matrix[OF is_square_dig2mat[OF assms(1)], OF assms(2)] length_dig2mat[OF assms(1)]
    by simp
  moreover
  have "mset (diag ?pM) = mset (replicate n 1)"
    using diag_permute_matrix[OF is_square_dig2mat[OF assms(1)], OF assms(2)] 
    using diag_dig2mat[OF assms(1)] assms(2)
    by simp
  ultimately
  show ?thesis
    using mat_mset_dig2mat[OF assms(1)]
    by simp
qed

lemma set_permute_dig [simp]:
  assumes  "length l = n*(n-1)" "p \<in> set (permute [0..<n])"
  shows "set (permute_dig p n l) = set l"
  using mset_permute_dig[OF assms]
  using mset_eq_setD
  by auto

lemma sum_list_permute_dig [simp]:
  fixes l :: "'a::{comm_monoid_add, one} list"
  assumes "length l = n*(n-1)" "p \<in> set (permute [0..<n])"
  shows "sum_list (permute_dig p n l) = sum_list l"
  using mset_permute_dig[of l n p] assms
  using  mset_eq_sum_list_eq
  by blast

lemma permute_dig_id [simp]:
  assumes "length s = n * (n-1)"
  shows "permute_dig [0..<n] n s = s"
  using assms
  unfolding permute_dig_def
  by auto

lemma length_permute_dig [simp]:
  assumes "length l = n*(n-1)" "p \<in> set (permute [0..<n])"
  shows "length (permute_dig p n l) = length l"
proof-
  have "length p = n" "set p = {0..<n}"
    using `p \<in> set (permute [0..<n])`
    using atLeastLessThan_upt permute_member_length permute_member_set
    by (fastforce, blast)
  thus ?thesis
    using assms
    using length_dig2mat[OF assms(1)]
    using length_mat2dig[of n "permute_matrix p (dig2mat n l)"]
    using is_square_permute_matrix[of n "dig2mat n l" p]
    unfolding permute_dig_def
    by simp
qed

lemma nth_permute_dig:
  assumes "length M = n * (n-1)" "i < n * (n-1)" "length p = n" "set p = {0..<n}"
  shows "permute_dig p n M ! i = 
              (if i mod (n-1) < i div (n-1) then 
                       (dig2mat n M) ! (p ! (i div (n-1))) ! (p ! (i mod (n-1)))
                else
                       (dig2mat n M) ! (p ! (i div (n-1))) ! (p ! (i mod (n-1) + 1))
               )"
proof-
  have "n > 0"
    using `i < n * (n-1)`
    by simp

  have "is_square_matrix n (dig2mat n M)" "i div (n-1) < n" "i mod (n-1) < n" "i mod (n-1) + 1 < n" 
    using is_square_dig2mat[OF assms(1)] `i < n * (n-1)`
    using less_mult_imp_div_less
       apply auto
     apply (smt Suc_lessD Suc_lessI \<open>0 < n\<close> diff_Suc_less diff_self_eq_0 less_trans_Suc mod_less_divisor mult_0_right zero_less_diff)
    apply (metis Suc_lessI \<open>0 < n\<close> diff_Suc_less diff_self_eq_0 less_trans_Suc mod_less_divisor mult_0_right zero_less_diff)
    done
  thus ?thesis
    using assms
    unfolding permute_dig_def
    using mat2dig_nth[of n "permute_matrix p (dig2mat n M)" i]
    using is_square_permute_matrix[OF is_square_dig2mat[OF assms(1)], of p]
    using nth_permute_matrix[of n "dig2mat n M" "i div (n-1)" "i mod (n-1)" p]
    using nth_permute_matrix[of n "dig2mat n M" "i div (n-1)" "i mod (n-1) + 1" p]
    by simp
qed

lemma hd_dig2mat_permute:
  assumes "p \<in> set (permute [0..<n])" "n > 1" "length M = n * (n-1)"
  shows  "hd (dig2mat n (permute_dig p n M)) = permute_list p ((dig2mat n M) ! hd p)"
proof-
  have "n > 0" using `n > 1` by auto
  have "length p = n" "set p = {0..<n}" 
    using `p \<in> set (permute [0..<n])`
    using permute_member_set[of p "[0..<n]"] permute_member_length[of p "[0..<n]"]
    by auto
  hence "hd p < n"
    using hd_in_set[of p] `n > 0`
    by auto
  hence "min (hd p) (n-1) = hd p"
    by auto
  have "hd (dig2mat n (permute_dig p n M)) = 1 # take (n - 1) (permute_dig p n M)"
    using hd_dig2mat[OF `0 < n`, of "permute_dig p n M"]
    by (simp add: \<open>p \<in> set (permute [0..<n])\<close> \<open>length M = n * (n - 1)\<close>)
  also have "... = 1 # tl (hd (permute_matrix p (dig2mat n M)))"
    using hd_mat2dig[of n "permute_matrix p (dig2mat n M)"]
    using is_square_permute_matrix[of n "dig2mat n M" p] `length p = n`
    using is_square_dig2mat[of M n] `set p = {0..<n}` `n > 0`
    using `length M = n * (n - 1)`
    unfolding permute_dig_def
    by blast
  also have "... = 1 # (tl (map (permute_list p) (dig2mat n M) ! hd p))"
    using permute_list_nth[of 0 p "(map (permute_list p) (dig2mat n M))"] 
    unfolding permute_matrix_def 
    by (metis \<open>1 < n\<close> \<open>length p = n\<close> list.map_sel(1) list.size(3) not_one_less_zero permute_list_def)
  also have "... = 1 # (tl (permute_list p (dig2mat n M ! hd p)))"
    by (simp add: \<open>hd p < n\<close> \<open>length M = n * (n-1)\<close>)
  also have "... = permute_list p (dig2mat n M ! hd p)"
  proof-
    let ?M = "dig2mat n M"
    have "hd (permute_list p (?M ! hd p)) = (?M ! hd p) ! hd p"
      unfolding permute_list_def
      by (metis \<open>1 < n\<close> \<open>length p = n\<close> list.map_sel(1) list.size(3) not_one_less_zero)
    moreover
    have "length ?M = n"
      using \<open>length M = n * (n-1)\<close> length_dig2mat by blast
    hence "?M ! hd p ! hd p \<in> set (diag ?M)"
      using `hd p < n` `length p = n`
      using ii_diag[of "hd p" "?M"]
      using `length M = n * (n-1)` is_square_dig2mat
      by auto
    ultimately              
    have "hd (permute_list p (dig2mat n M ! hd p)) = 1"
      by (simp add: \<open>length M = n * (n-1)\<close>)
    moreover
    have "permute_list p (dig2mat n M ! hd p) \<noteq> []"
      by (metis \<open>1 < n\<close> \<open>length p = n\<close> permute_list_length list.size(3) not_less_zero)
    ultimately
    show ?thesis
      using hd_Cons_tl
      by fastforce
  qed
  finally
  show ?thesis
    .
qed

subsubsection\<open>Permutations of digraphs as permutations of lists\<close>

definition np2dp where "np2dp p n \<equiv> permute_dig p n [0..<n*(n-1)]"

lemma length_np2dp [simp]:
  assumes "p \<in> set (permute [0..<n])"
  shows "length (np2dp p n) = n*(n-1)"
  using assms
  unfolding np2dp_def
  using length_permute_dig[of "[0..<n*(n-1)]" n p]
  by simp

lemma np2dp_direct:
  assumes "p \<in> set (permute [0..<n])"
  shows "np2dp p n = map (\<lambda> i. (let idiv = i div (n - 1);
                                    imod = i mod (n - 1);
                                    pdiv = p ! idiv;
                                    pmod = p ! imod;
                                    pmod1 = p ! (imod + 1)
                                 in (n - 1) * pdiv + 
                                      (if imod < idiv then if pdiv < pmod then pmod - 1 else pmod
                                       else if pdiv < pmod1 then pmod1 - 1 else  pmod1))) [0..<n*(n-1)]" (is "?lhs = map ?f [0..<n*(n-1)]")
  unfolding np2dp_def
proof (rule nth_equalityI)
  fix i
  let ?PD = "permute_dig p n [0..<n * (n - 1)]"
  assume "i < length ?PD"
  hence "i < n * (n - 1)"
    using length_permute_dig[OF _ assms, of "[0..<n*(n-1)]"]
    by simp

  have p: "length p = n" "set p = {0..<n}"
    using permute_member_length[OF assms] permute_member_set[OF assms]
    by auto

  have "i div (n - 1) < n"
    using `i < n * (n - 1)`
    using less_mult_imp_div_less by blast
  moreover
  have "i mod (n - 1) < n"
    using `i < n * (n - 1)`
    by (metis diff_less gr_zeroI less_trans mod_less_divisor mult_is_0 not_less0 zero_less_one)
  moreover
  have "i mod (n - 1) + 1 < n"
    using `i < n * (n - 1)`
    by (metis less_diff_conv mod_less_divisor mult_is_0 nat_neq_iff not_less0)
  moreover
  have *: "p ! (i div (n - 1)) < n" "p ! (i mod (n - 1)) < n" "p ! (i mod (n - 1) + 1) < n"
    using `i div (n - 1) < n` `i mod (n - 1) < n` `i mod (n - 1) + 1 < n`
    using `length p = n` `set p = {0..<n}`
    by auto

  have "distinct p"
    using assms
    by (simp add: card_distinct p(1) p(2))

  have **: "i mod (n - 1) < i div (n - 1) \<Longrightarrow> p ! (i div (n - 1)) \<noteq> p ! (i mod (n - 1))"
    using `distinct p` 
    using `i mod (n - 1) < n` `i div (n - 1) < n` nat_neq_iff nth_eq_iff_index_eq p(1)
    by blast
  have ***: "i mod (n - 1) \<ge> i div (n - 1) \<Longrightarrow> p ! (i div (n - 1)) \<noteq> p ! (i mod (n - 1) + 1)"
    using `distinct p`
    using `i mod (n - 1) < n` `i mod (n - 1) + 1 < n` `i div (n - 1) < n`
    by (simp add: nth_eq_iff_index_eq p(1))

  show "?PD ! i = map ?f [0..<n*(n-1)] ! i"
  proof (cases "i mod (n - 1) < i div (n - 1)")
    case True
    thus ?thesis
      using `i < n * (n-1)` * ** p
      apply (subst nth_permute_dig, simp, simp, simp, simp)
      apply (subst nth_dig2mat_special, simp, simp, simp, simp)
      done
  next
    case False
    thus ?thesis
      using `i < n * (n-1)` * *** p
      apply (subst nth_permute_dig, simp, simp, simp, simp)
      apply (subst nth_dig2mat_special, simp, simp, simp, simp)
      done
  qed
next
  show "length (permute_dig p n [0..<n*(n-1)]) = length (map ?f [0..<n * (n-1)])"
    using length_permute_dig[OF _ assms, of "[0..<n*(n-1)]"]
    by simp
qed

lemma permute_dig_permute_list':
  assumes "length xs = n * (n - 1)" "i < n" "j < n" "i \<noteq> j"
  shows "dig2mat n xs ! i ! j = xs ! (dig2mat n [0..<n * (n - 1)] ! i ! j)"
proof-
  let ?elems1 = "take (n - 1) (drop (i * (n - 1)) xs)"
  let ?elems2 = "take (n - 1) (drop (i * (n - 1)) [0..<n * (n-1)])"

  have *: "?elems1 = map (\<lambda> p. xs ! p) ?elems2"
    using `i < n` `length xs = n * (n - 1)`
    by (subst take_map[symmetric], subst drop_map[symmetric], metis map_nth)
    
  have "dig2mat n xs ! i = take i ?elems1 @ [1] @ drop i ?elems1"
    using assms
    using nth_dig2mat[OF assms(2) assms(1)]
    by simp
  hence 1: "dig2mat n xs ! i = map (\<lambda> p. xs ! p) (take i ?elems2) @ [1] @ map (\<lambda> p. xs ! p) (drop i ?elems2)" (is "?lhs = ?rhsA @ [1] @ ?rhsB")
    using *
    by (subst take_map[symmetric], subst drop_map[symmetric], presburger)
  have 2: "dig2mat n [0..<n*(n-1)] ! i = take i ?elems2 @ [1] @ drop i ?elems2"
    using assms
    using nth_dig2mat[OF assms(2), of "[0..<n*(n-1)]"]
    by simp

  have "length ?elems2 = n-1"
  proof-
    have "length (drop (i * (n - 1)) [0..<n * (n - 1)]) \<ge> n - 1"
      using `i < n`
      by auto (metis One_nat_def diff_mult_distrib mult.left_neutral mult_le_cancel2 not_le not_less_eq zero_less_diff)
    thus ?thesis
      by (simp add: min_def)
  qed

  have "length (take i ?elems2) = i"
    using `i < n` `length ?elems2 = n-1`
    by (metis (no_types, lifting) One_nat_def Suc_diff_Suc diff_zero length_take linorder_not_le min_def nat_neq_iff not_less0 not_less_eq)
  hence "length ?rhsA = i"
    by simp

  have "length (drop i ?elems2) = n - (i + 1)"
    using `i < n` `length ?elems2 = n-1`
    by simp

  have AA: "dig2mat n xs ! i ! j =
            (if j < i then 
                (map (\<lambda> p. xs ! p) (take i ?elems2)) ! j
             else
                (map (\<lambda> p. xs ! p) (drop i ?elems2)) ! (j - (i + 1)))"
  proof (cases "j < i")
    case True
    hence "(?rhsA @ [1] @ ?rhsB) ! j = ?rhsA ! j"
      using `length ?rhsA = i`
      by (simp add: nth_append)
    thus ?thesis
      using True 1
      by simp
  next
    case False
    hence "j > i"
      using `i \<noteq> j`
      by simp
    hence "(?rhsA @ [1] @ ?rhsB) ! j = ?rhsB ! (j - (i + 1))"
      using `length ?rhsA = i`
      by (simp add: nth_append)
    thus ?thesis
      using False 1
      by simp
  qed

  have BB: "dig2mat n [0..<n * (n - 1)] ! i ! j = 
            (if j < i then 
                (take i ?elems2) ! j
             else
                (drop i ?elems2) ! (j - (i + 1)))"
  proof (cases "j < i")
    case True
    thus ?thesis
      using `length (take i ?elems2) = i` 2
      by (simp add: nth_append)
  next
    case False
    hence "j > i"
      using `i \<noteq> j`
      by simp
    thus ?thesis
      using `length (take i ?elems2) = i` 2
      by (simp add: nth_append)
  qed
  
  show ?thesis
  proof (cases "j < i")
    case True
    hence "(map (\<lambda> p. xs ! p) (take i ?elems2)) ! j = xs ! ((take i ?elems2) ! j)"
      using `length (take i ?elems2) = i`
      using nth_map[of j "take i ?elems2" "(!) xs"]
      by metis
    thus ?thesis
      using AA BB `j < i`
      by simp
  next
    case False
    hence "j > i"
      using `i \<noteq> j`
      by simp
    hence "j - (i + 1) < length (drop i ?elems2)"
      using `length (drop i ?elems2) = n - (i + 1)` `i < n` `j < n`
      by simp
    hence "(map (\<lambda> p. xs ! p) (drop i ?elems2)) ! (j - (i + 1)) = xs ! ((drop i ?elems2) ! (j - (i + 1)))"
      using `length (drop i ?elems2) = n - (i + 1)` `i < n` `j > i`
      using nth_map[of "j - (i + 1)" "drop i ?elems2" "(!) xs"]
      by simp      
    thus ?thesis
      using AA BB `j > i`
      by simp
  qed
qed


lemma permute_dig_permute_list:                     
  assumes "length xs = n * (n-1)" "p \<in> set (permute[0..<n])"
  shows "permute_dig p n xs = permute_list (np2dp p n) xs" (is "?lhs = ?rhs")
proof (rule nth_equalityI)
  show "length ?lhs = length ?rhs"
    using length_permute_dig[OF assms] permute_list_length[of "np2dp p n" xs]
    using length_np2dp[OF assms(2)] assms(1)
    by simp
next
  fix i
  have "length p = n" "set p = {0..<n}" "distinct p"
    using permute_member_length[OF assms(2)] permute_member_set[OF assms(2)]
    by auto (metis \<open>length p = length [0..<n]\<close> \<open>set p = set [0..<n]\<close> card_distinct distinct_card distinct_upt)

  assume "i < length (permute_dig p n xs)"
  hence "i < n * (n-1)"
    using length_permute_dig[OF assms] assms(1)
    by simp

  have "i div (n - 1) < n"
    using \<open>i < n * (n - 1)\<close> less_mult_imp_div_less by blast

  have "i mod (n - 1) < n"
    by (metis (no_types, lifting) Suc_lessD \<open>i < n * (n - 1)\<close> \<open>i div (n - 1) < n\<close> diff_less less_trans_Suc mod_less_divisor mult_0_right nat_neq_iff not_less0 zero_less_one)

  have "i mod (n - 1) + 1 < n"
    by (metis One_nat_def Suc_diff_Suc Suc_lessI \<open>i < n * (n - 1)\<close> \<open>i mod (n - 1) < n\<close> add.commute diff_Suc_Suc diff_zero less_Suc_eq_0_disj mod_if mod_mod_trivial mod_self nat.simps(3) not_gr_zero plus_1_eq_Suc right_diff_distrib' zero_diff zero_less_diff)


  have "i mod (n - 1) < i div (n - 1) \<Longrightarrow> dig2mat n xs ! (p ! (i div (n - 1))) ! (p ! (i mod (n - 1))) =
           xs ! (dig2mat n [0..<n * (n - 1)] ! (p ! (i div (n - 1))) ! (p ! (i mod (n - 1))))"
    using assms
  proof (subst permute_dig_permute_list', simp_all)
    show "p ! (i div (n - Suc 0)) < n"
      using `i div (n - 1) < n` `length p = n` `set p = {0..<n}`
      by auto
  next
    show "p ! (i mod (n - Suc 0)) < n"
      using `i mod (n - 1) < n` `length p = n` `set p = {0..<n}`
      by auto
  next
    assume "i mod (n - Suc 0) < i div (n - Suc 0)"
    thus "p ! (i div (n - Suc 0)) \<noteq> p ! (i mod (n - Suc 0))"
      using `distinct p` 
      by (metis One_nat_def \<open>i div (n - 1) < n\<close> \<open>i mod (n - 1) < n\<close> \<open>length p = n\<close> nat_neq_iff nth_eq_iff_index_eq)
  qed

  moreover
  have "\<not> i mod (n - 1) < i div (n - 1) \<Longrightarrow> dig2mat n xs ! (p ! (i div (n - 1))) ! (p ! (i mod (n - 1) + 1)) =
           xs ! (dig2mat n [0..<n * (n - 1)] ! (p ! (i div (n - 1))) ! (p ! (i mod (n - 1) + 1)))"
    using assms
  proof (subst permute_dig_permute_list', simp_all)
    show "p ! (i div (n - Suc 0)) < n"
      using `i div (n - 1) < n` `length p = n` `set p = {0..<n}`
      by auto
  next
    show "p ! Suc (i mod (n - Suc 0)) < n"
      using `i mod (n - 1) + 1 < n` `length p = n` `set p = {0..<n}`
      by auto
  next
    assume "\<not> i mod (n - Suc 0) < i div (n - Suc 0)"
    hence "i div (n - Suc 0) \<noteq> Suc (i mod (n - Suc 0))"
      by simp
    thus "p ! (i div (n - Suc 0)) \<noteq> p ! Suc (i mod (n - Suc 0))"
      using `distinct p`
      by (metis One_nat_def \<open>i div (n - 1) < n\<close> \<open>i mod (n - 1) + 1 < n\<close> \<open>length p = n\<close> add.commute nth_eq_iff_index_eq plus_1_eq_Suc)
  qed

  ultimately

  have "permute_dig p n xs ! i = xs ! (permute_dig p n [0..<n * (n - 1)] ! i)"
    using assms
    using nth_permute_dig[OF assms(1) `i < n * (n-1)` `length p = n` `set p = {0..<n}`]
    using nth_permute_dig[of "[0..<n * (n-1)]" n, OF _ `i < n * (n-1)` `length p = n` `set p = {0..<n}`]
    by auto

  thus "permute_dig p n xs ! i = permute_list (np2dp p n) xs ! i"
    using length_np2dp[OF assms(2)] assms(1) `i < n * (n-1)`
    unfolding permute_list_def
    unfolding np2dp_def
    by simp
qed

lemma permute_dig_permute_list_set:
  assumes "p \<in> set (permute [0..<n])" "n > 0"
  shows "np2dp p n \<in> set (permute [0..<n*(n-1)])" 
proof  (rule permute_isPermutation)
  show "np2dp p n <~~> [0..<n * (n - 1)]"
  proof (subst mset_eq_perm[symmetric])
    show "mset (np2dp p n) = mset [0..<n * (n - 1)]"
      using assms
      unfolding np2dp_def
      by simp
  qed
qed

lemma permute_dig_compose:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])" "length s = n * (n-1)" "n > 0"
  shows "permute_dig p2 n (permute_dig p1 n s) = permute_dig (perm_comp p1 p2) n s"
  using assms
  unfolding permute_dig_def
proof (subst permute_matrix_comp, simp_all)
  show "is_square_matrix n (dig2mat n s)"
    using assms(3) is_square_dig2mat 
    by blast
  have "dig2mat n (mat2dig (permute_matrix p1 (dig2mat n s))) = permute_matrix p1 (dig2mat n s)"
  proof (subst dig2mat_mat2dig)
    show "is_square_matrix n (permute_matrix p1 (dig2mat n s))"
      using assms(1) assms(3) permute_member_length permute_member_set
      by fastforce
  next
    show "diag (permute_matrix p1 (dig2mat n s)) = replicate n 1"
    proof (subst diag_permute_matrix)
      show "is_square_matrix n (dig2mat n s)"
        by fact
    next
      show "p1 \<in> set (permute [0..<n])"
        by fact
    next
      show "permute_list p1 (diag (dig2mat n s)) = replicate n 1"
        by (smt assms(1) assms(3) diag_dig2mat diff_zero in_set_replicate permute_member_length permute_list_length length_replicate length_upt replicate_eqI permute_member_set permute_list_set set_upt)
    qed
  qed simp
  thus "mat2dig (permute_matrix p2 (dig2mat n (mat2dig (permute_matrix p1 (dig2mat n s))))) = mat2dig (permute_matrix p2 (permute_matrix p1 (dig2mat n s)))"
    by simp
qed

lemma ex_permute_dig_compose:
  assumes "p1 \<in> set (permute [0..<n])" "p2 \<in> set (permute [0..<n])" "length s = n * (n-1)" "n > 0"
  shows "\<exists> p \<in> set (permute [0..<n]). permute_dig p2 n (permute_dig p1 n s) = permute_dig p n s"
  using assms(1) assms(2) assms(3) assms(4) perm_comp_permute permute_dig_compose by blast

subsubsection\<open>Check for digraph canonicity\<close>

(* canonical digraph of a given digraph (represented by a list) *)
global_interpretation CanonDig: CanonLeastPerm
  where 
    invar = "\<lambda> n l. n > 0 \<and> length l = n * (n - 1)" and
    permute = "\<lambda> n p. permute_dig p n"
  defines 
    is_canon_dig = "CanonDig.is_canon" and
    canon_dig = "CanonDig.canon" and
    equiv_dig = "CanonDig.equiv"
proof
  fix n :: nat and a :: "'a list" and p :: perm
  assume "n > 0 \<and> length a = n * (n - 1)" "p <~~> [0..<n]" 
  thus "n > 0 \<and> length (permute_dig p n a) = n * (n - 1)"
    by (simp add: permute_isPermutation)
next
  fix n p1 p2 and a :: "'a list"
  assume "n > 0 \<and> length a = n * (n - 1)" "p1 <~~> [0..<n]" "p2 <~~> [0..<n]"
  thus "\<exists> p. p <~~> [0..<n] \<and> permute_dig p n a = permute_dig p1 n (permute_dig p2 n a)"
    using permute_dig_compose[of p2 n p1 a]
    by (metis isPermutation_permute perm_comp_permute permute_isPermutation)
next
  fix n and a :: "'a list"
  assume "n > 0 \<and> length a = n * (n - 1)"
  thus "permute_dig (perm_id n) n a = a"
    by (simp add: permute_dig_def perm_id_def)
next
  fix n p' p and a :: "'a list"
  assume "n > 0 \<and> length a = n * (n - 1)" "p <~~> [0..<n]"
  moreover
  have "perm_inv p <~~> [0..<n]"
    using `p <~~> [0..<n]` perm_inv_permute isPermutation_permute permute_isPermutation
    by blast
  ultimately
  show "\<exists> p'. p' <~~> [0..<n] \<and> permute_dig p' n (permute_dig p n a) = a"
    using permute_dig_compose[of p n "perm_inv p" a]
    using permute_isPermutation[of p "[0..<n]"]
    by (rule_tac x="perm_inv p" in exI)
       (simp add: perm_comp_perm_inv_id_2)
qed

(* avoids repeating dig2mat *)
lemma canon_dig [code]:
 "canon_dig n M = 
      (let MM = dig2mat n M
        in min_list (map (\<lambda> p. mat2dig (permute_matrix p MM)) 
                         (permute [0..<n]))
      )"                                                                           
  unfolding CanonDig.canon_code permute_dig_def Let_def
  by simp

lemma length_canon_dig:
  assumes "length l = n * (n-1)" "n > 0"
  shows "length (canon_dig n l) = length l"
  using CanonDig.canon_invar[of n l] assms
  by simp

lemma set_canon_dig [simp]:
  assumes "length l = n*(n-1)"
  shows "set (canon_dig n l) = set l"
  using CanonDig.equiv_canon[of n l]
  unfolding CanonDig.equiv_def
  using assms permute_isPermutation[of _ "[0..<n]"] set_permute_dig[of l n]
  by fastforce
  
lemma sum_list_canon_dig:
  fixes l :: "('a::{one, linorder, comm_monoid_add}) list"
  assumes "length l = n*(n-1)"
  shows "sum_list (canon_dig n l) = sum_list l"
  using CanonDig.equiv_canon[of n l]
  unfolding CanonDig.equiv_def
  using assms
  using permute_isPermutation[of _ "[0..<n]"] 
  using isPermutation_permute[of _ "[0..<n]"]
  using sum_list_permute_dig [of l n]
  by metis
  
subsubsection\<open>Optimized check for digraph canonicity\<close>

(* this holds only for 0-1 matrices *)

definition out_degrees where
  "out_degrees M = map sum_list M"

definition positions where
  "positions p l = map fst (filter (\<lambda> (i, j). i \<noteq> p \<and> j = 1) (zip [0..<length l] l))"

definition is_canon_dig_opt :: "nat \<Rightarrow> nat list \<Rightarrow> bool" where
  "is_canon_dig_opt n M \<longleftrightarrow> 
       (let MM = dig2mat n M;
            od = out_degrees MM;
            md = max_list od;
            perms1 = filter (\<lambda> p. od ! hd p = md) (permute [0..<n]);
            ppp = map (\<lambda> (i, row). positions i row) (zip [0..<n] MM);
            perms2 = filter (\<lambda> p. let pp = ppp ! hd p 
                                   in set (take (length pp) (tl p)) = set pp)
                           perms1
         in list_all (\<lambda> p. M \<le> mat2dig (permute_matrix p MM)) perms2)"

lemma max_degree:
  fixes M :: "nat list"
  assumes "set M \<subseteq> {0, 1}" "n > 0" "length M = n * (n-1)"
  shows "max_list (out_degrees (dig2mat n M)) \<le> n" "max_list (out_degrees (dig2mat n M)) \<ge> 1"
proof-
  let ?M = "dig2mat n M"
  have *: "\<forall> r \<in> set ?M. set r \<subseteq> {0, 1} \<and> 1 \<in> set r"
  proof
    fix r
    assume "r \<in> set ?M" 
    obtain k where k: "k < n" "r = (let elems = take (n - 1) (drop (k * (n - 1)) M) in take k elems @ [1] @ drop k elems)"
      using `r \<in> set ?M`
      using dig2mat_map[OF assms(3)]
      by auto
    show "set r \<subseteq> {0, 1} \<and> 1 \<in> set r"
    proof safe
      fix x
      assume "x \<in> set r" "0 < x"
      have "x \<in> set M \<or> x = 1"
        using `x \<in> set r` k
        by (auto simp add: Let_def) (meson in_set_dropD in_set_takeD)+
      thus "x = 1"
        using `set M \<subseteq> {0, 1}` `x > 0`
        by auto
    next
      show "1 \<in> set r"
        using k
        by simp
    qed
  qed

  have *: "\<forall> r \<in> set ?M. sum_list r \<le> n \<and> sum_list r \<ge> 1"
  proof safe
    fix r
    assume "r \<in> set ?M"
    hence "set r \<subseteq> {0, 1}"
      using *
      by simp
    hence "\<forall> x \<in> set r. x \<le> 1" "\<forall> x \<in> set r. x \<ge> 0"
      by auto
    moreover
    have "length r = n"
      using `r \<in> set ?M`
      using is_square_dig2mat[OF assms(3)]
      unfolding is_square_matrix_def
      by simp
    ultimately
    show "sum_list r \<le> n"
      using sum_list_mono[of r id "\<lambda> x. 1"]
      using sum_list_triv[of "1::nat" r]
      by auto

    have "1 \<in> set r"
      using * `r \<in> set ?M`
      by auto

    show "1 \<le> sum_list r"
      using sum_list_nonneg_eq_0_iff[of r] `1 \<in> set r`
      by (simp add: member_le_sum_list)
  qed

  show "max_list (out_degrees ?M) \<ge> 1"
    using * max_list_max[of "out_degrees ?M"] assms(2-3)
    by (metis (no_types, lifting) One_nat_def Suc_leI gr0I image_eqI le_0_eq length_dig2mat length_greater_0_conv list.set_sel(1) out_degrees_def set_map)

  show "max_list (out_degrees ?M) \<le> n"
    using * max_list_ubound[of "out_degrees ?M" n] `n > 0` length_dig2mat[OF `length M = n * (n-1)`] 
    unfolding out_degrees_def
    by fastforce
qed
    

lemma max_out_degrees:
  fixes M :: "nat list list"
  shows "\<forall> row \<in> set M. sum_list row \<le> max_list (out_degrees M)"
  unfolding out_degrees_def
  using max_list_max[of "map sum_list M"]
  by auto

lemma positions_snoc:
  shows "positions i (xs @ [x]) = positions i xs @ (if i \<noteq> length xs \<and> x = 1 then [length xs] else [])"
  unfolding positions_def
  by simp

lemma length_positions_outofbounds:
  assumes "set xs \<subseteq> {0, 1}" "i \<ge> length xs"
  shows "length (positions i xs) = sum_list xs"
  using assms
  by (induction xs rule: rev_induct)
     (auto simp add: positions_snoc positions_def)

lemma length_positions:
  assumes "i < length M" "M ! i = 1" "set M \<subseteq> {0, 1}"
  shows "length (positions i M) =  sum_list M - 1"
  using assms
proof (induction M rule: rev_induct)
  case Nil
  then show ?case by simp
next
  case (snoc x xs)
  show ?case
  proof (cases "i = length xs")
    case True
    thus ?thesis
      using snoc(3) snoc(4) length_positions_outofbounds[of "xs" i]
      by (auto simp add: positions_snoc)
  next
    case False
    hence "i < length xs"
      using snoc(2)
      by simp
    hence "length (positions i xs) = sum_list xs - 1"
      using snoc(1) snoc(3) snoc(4)
      by (simp add: nth_append)
    moreover
    have "sum_list xs > 0"
      using snoc(3) `i < length xs`
      by (simp add: nth_append)
         (metis One_nat_def elem_le_sum_list gr0I not_one_le_zero)
    ultimately
    show ?thesis
      using `i < length xs` snoc(4)
      by (auto simp add: positions_snoc nth_append)
  qed
qed


(* Equivalence with the unoptimized version  *)
lemma is_canon_dig_opt:
  fixes M :: "nat list"
  assumes "set M \<subseteq> {0, 1}" "length M = n * (n-1)" "n > 0"
  shows "is_canon_dig_opt n M \<longleftrightarrow> is_canon_dig n M"
proof
  assume "is_canon_dig n M"
  thus "is_canon_dig_opt n M"
    unfolding CanonDig.is_canon_code is_canon_dig_opt_def permute_dig_def Let_def list_all_iff
    by auto
next
  assume "is_canon_dig_opt n M"
  show "is_canon_dig n M"
  proof (cases "n = 1")
    case True
    thus ?thesis
      using assms permute_dig_id
      unfolding CanonDig.is_canon_code
      by fastforce
  next
    case False
    hence "n > 1"
      using `n > 0`
      by simp
    show ?thesis
      unfolding CanonDig.is_canon_code list_all_iff
    proof safe
      fix p
      assume "p \<in> set (permute [0..<n])"
      let ?MM = "dig2mat n M"
      let ?od = "out_degrees ?MM"
      let ?md = "max_list ?od"
      let ?perms1 = "filter (\<lambda> p. ?od ! hd p = ?md) (permute [0..<n])"
      let ?ppp = "map (\<lambda> (i, row). positions i row) (zip [0..<n] ?MM)"
      let ?perms2 = "filter (\<lambda> p. let pp = ?ppp ! hd p in set (take (length pp) (tl p)) = set pp) ?perms1"

      let ?pM = "permute_dig p n M"

      obtain md where "md = ?md - 1"
        by auto

      have "md < n" "?md \<ge> 1" "?md \<le> n"
        using max_degree[OF `set M \<subseteq> {0, 1}` `n > 0` `length M = n * (n-1)`]
        using `md = ?md - 1`
        by simp_all

      have "n \<le> length M"
        using `n > 0` `n \<noteq> 1` `length M = n * (n - 1)`
        by auto

      have "length p = n"
        using `p \<in> set (permute [0..<n])`
        using permute_member_length by fastforce

      have "hd p < n"
        using `p \<in> set (permute [0..<n])` `n > 0`
        using permute_member_length permute_member_set by fastforce


      have "length ?pM = length M"
        using length_permute_dig[OF `length M = n * (n - 1)` `p \<in> set  (permute [0..<n])`]
        by simp

      have "set ?pM = set M"
        using set_permute_dig[OF `length M = n * (n - 1)` `p \<in> set (permute [0..<n])` ]
        by simp

      have "hd (dig2mat n ?pM) = permute_list p (?MM ! hd p)"
        using hd_dig2mat_permute[of p n M] \<open>p \<in> set (permute [0..<n])\<close> \<open>1 < n\<close> `length M = n * (n-1)`
        by simp

      have "length (?MM ! hd p) = n"
        by (metis \<open>hd p < n\<close> assms(2) diff_zero dig2mat_def length_dig2mat' nth_mem)

      {
        fix p0
        assume "p0 < n"
        have "set (?MM ! p0) \<subseteq> {0, 1}"
        proof-
          let ?elems = "take (n - 1) (drop (p0 * (n - 1)) M)"
          have "set ?elems \<subseteq> {0, 1}"
            using `set M \<subseteq> {0, 1}`
            using set_take_subset set_drop_subset
            by force
          hence "set (take p0 ?elems @ [1] @ drop p0 ?elems) \<subseteq> {0, 1}"
            using set_take_subset[of p0 ?elems] set_drop_subset[of p0 ?elems]
            by simp blast
          thus ?thesis
            using nth_dig2mat[of p0 n M] `p0 < n` `length M = n * (n-1)` 
            by simp
        qed
      } note set_nth_MM = this

      {
        fix p0
        assume "p0 < n" "?od ! p0 = ?md"
        let ?pos = "positions p0 (?MM ! p0)"
        have "length ?pos = md"
        proof-
          have "length ?pos = sum_list (?MM ! p0) - 1"
          proof (rule length_positions)
            show "p0 < length (?MM ! p0)"
              using `p0 < n`
              by (metis assms(2) diff_zero dig2mat_def length_dig2mat' nth_mem)
          next
            show "?MM ! p0 ! p0 = 1"
              by (metis \<open>p0 < n\<close> assms(2) diag_dig2mat ii_diag in_set_replicate is_square_dig2mat length_dig2mat)
          next
            show "set (?MM ! p0) \<subseteq> {0, 1}"
              using set_nth_MM[OF `p0 < n`]
              by simp
          qed
          moreover 
          have "sum_list (?MM ! p0) - 1 = md"
            using `?od ! p0 = ?md` `md = ?md - 1`
            unfolding out_degrees_def
            by (simp add: \<open>p0 < n\<close> assms(2))
          ultimately
          show ?thesis
            by simp
        qed
      } note length_max_positions = this

      {
        fix P
        let ?PM = "permute_dig P n M"
        assume "P \<in> set (permute [0..<n])"
        have "take md ?PM = map (\<lambda> i. ?MM ! (hd P) ! i) (take md (tl P))"
        proof-
          have "hd P < n"
            using `P \<in> set (permute [0..<n])` `n > 0`
            using permute_member_length permute_member_set
            by fastforce
          have "length ?PM = n * (n - 1)"
            using length_permute_dig[of M n P] `P \<in> set (permute [0..<n])` `length M = n * (n-1)`
            by simp
          have "take md ?PM = take md (tl (hd (dig2mat n ?PM)))"
          proof-
            have "take md ?PM = take md (take (n-1) ?PM)"
              using `md < n` `n > 0`
              by (auto simp add: min_def)
            moreover
            have "take md (tl (hd (dig2mat n ?PM))) = take md (take (n-1) (tl (hd (dig2mat n ?PM))))"
              using `md < n` `n > 0`
              by (auto simp add: min_def)
            moreover
            have "take (n-1) ?PM = take (n-1) (tl (hd (dig2mat n ?PM)))"
              using `length ?PM = n * (n-1)`
              using hd_dig2mat[of n ?PM] `n > 0`
              by simp
            ultimately
            show ?thesis
              by metis
          qed
          also have "... = take md (tl (permute_list P (?MM ! (hd P))))"
            using `n > 1` `length M = n * (n-1)` `P \<in> set (permute [0..<n])`
            by (subst hd_dig2mat_permute, simp_all)
          also have "... = map (\<lambda> i. ?MM ! (hd P) ! i) (take md (tl P))"
            unfolding permute_list_def
            by (simp add: map_tl[symmetric] take_map)
          finally show ?thesis
            .
        qed
      } note take_md_permute_dig = this

      have "\<exists> P \<in> set ?perms2. take md (permute_dig P n M) = replicate md 1"
      proof-
        obtain p0 where "p0 < n" "?od ! p0 = ?md"
          using max_list_is_nth[of ?od]
          unfolding out_degrees_def
          using assms(2) assms(3) length_dig2mat
          by force

        let ?pos = "positions p0 (?MM ! p0)"
        let ?rest = "sorted_list_of_set ({0..<n} - set (p0 # ?pos))"
        let ?P = "p0 # ?pos @ ?rest"

        let ?PM = "permute_dig ?P n M"

        have "length ?pos = md"
          using length_max_positions[of p0] `p0 < n` `?od ! p0 = ?md`
          by simp

        have "?P \<in> set (permute [0..<n])"
        proof-
          have "distinct ?P"
          proof-
            have "p0 \<notin> set ?pos"
              unfolding positions_def
              by auto
            moreover
            have "distinct ?pos"
              unfolding positions_def
              by (auto simp add: inj_on_def distinct_map distinct_zipI1)
            ultimately
            show ?thesis
              by auto
          qed
          moreover
          have "set ?P = {0..<n}"
          proof-
            have "set ?pos \<subseteq> {0..<n}"
              unfolding positions_def
              by (auto simp add: set_zip) (metis \<open>p0 < n\<close> assms(2) diff_zero dig2mat_def length_dig2mat' nth_mem)
            thus ?thesis
              using `p0 < n`
              by auto
          qed
          ultimately
          have "mset ?P = mset [0..<n]"
            by (metis atLeastLessThan_upt distinct_upt set_eq_iff_mset_eq_distinct)
          thus ?thesis
            using permute_isPermutation[of ?P "[0..<n]"]
            using mset_eq_perm by blast
        qed

        moreover

        have "?od ! hd ?P = ?md"
          using `?od ! p0 = ?md`
          by simp

        moreover

        let ?pp = "map2 positions [0..<n] (dig2mat n M) ! hd ?P"
        have *: "?pp = ?pos"
          by (simp add: \<open>p0 < n\<close> assms(2))
        hence "set (take (length ?pp) (tl ?P)) = set ?pp"
          using * `length ?pos = md`
          by simp
        ultimately
        have "?P \<in> set ?perms2"
          by simp

        moreover

        have "take md ?PM = replicate md 1"
        proof-
          have "take md ?PM = map (\<lambda> i. ?MM ! (hd ?P) ! i) (take md (tl ?P))"
            using take_md_permute_dig[OF `?P \<in> set (permute [0..<n])`]
            by simp
          also have "... = map (\<lambda> i. ?MM ! p0 ! i) ?pos"
            using `length ?pos = md`
            by simp
          also have "... = replicate md 1" (is "?lhs = ?rhs")
          proof-
            have "\<forall>p\<in>set ?pos. ?MM ! p0 ! p = 1"
              unfolding positions_def
              by (auto simp add: set_zip)
            thus ?thesis
              using list_eq_replicate[of ?lhs 1]
              using `length ?pos = md`
              by simp
          qed
          finally
          show ?thesis
            .
        qed

        ultimately
        show ?thesis
          by blast
      qed

      then obtain P where
        "P \<in> set ?perms2"
        "take md (permute_dig P n M) = replicate md 1"
        by auto

      let ?PM = "permute_dig P n M"

      have "length ?PM = length M"
        using `P \<in> set ?perms2` length_permute_dig[OF `length M = n * (n - 1)`, of P] `n > 0`
        by simp

      have "M \<le> ?PM"
        using `is_canon_dig_opt n M` `P \<in> set ?perms2` 
        unfolding is_canon_dig_opt_def permute_dig_def
        by (simp add: Let_def list_all_iff)

      have "take md M = replicate md 1"
      proof (rule ccontr)
        assume "\<not> ?thesis"
        thus False
          using earlier_zero_less[of md "permute_dig P n M" M]
          using `take md (permute_dig P n M) = replicate md 1`
          using `md < n` `n \<le> length M`
          using `set M \<subseteq> {0, 1}` `M \<le> permute_dig P n M`
          using `length (permute_dig P n M) = length M`
          by simp
      qed

      show "M \<le> ?pM"
      proof (cases "p \<in> set ?perms2")
        case True
        thus ?thesis
          using `is_canon_dig_opt n M`
          unfolding is_canon_dig_opt_def permute_dig_def
          by (simp add: Let_def list_all_iff)
      next
        case False

        have "take md ?pM \<noteq> replicate md 1"
        proof (cases "p \<in> set ?perms1")
          case False
          show ?thesis
          proof (rule ccontr)
            assume "\<not> ?thesis"
            hence "sum_list (take md ?pM) = md"
              by simp
            hence "sum_list (take (n-1) ?pM) \<ge> md"
              using `md < n`
              using take_add[of "md" "n-1-md" ?pM]
              by auto

            moreover

            have "sum_list (take (n-1) ?pM) < md"
            proof-
              have "out_degrees ?MM ! hd p \<noteq> ?md"
                using `p \<in> set (permute [0..<n])` `p \<notin> set ?perms1`
                by auto
              hence "sum_list (?MM ! hd p) \<noteq> ?md"
                unfolding out_degrees_def
                using `p \<in> set (permute [0..<n])` permute_member_set[of p "[0..<n]"] `n > 0` hd_in_set[of p] permute_member_length[of p "[0..<n]"]
                using `length M = n * (n-1)` length_dig2mat
                by fastforce
              hence "sum_list (?MM ! hd p) < ?md"
                using max_out_degrees[of "dig2mat n M"]
                using nth_mem[of "hd p" "dig2mat n M"]
                by (metis \<open>p \<in> set (permute [0..<n])\<close> assms(2) assms(3) atLeastLessThan_iff diff_zero le_neq_implies_less length_dig2mat permute_member_length length_upt list.set_sel(1) list.size(3) neq0_conv permute_member_set set_upt)
              hence "sum_list (hd (dig2mat n ?pM)) < ?md"
                using \<open>p \<in> set (permute [0..<n])\<close> assms(2-3)
                using `hd (dig2mat n (permute_dig p n M)) = permute_list p (?MM ! hd p)`
                using sum_list_permute_list[of "dig2mat n M ! hd p" n p]
                by (metis atLeastLessThan_iff diff_zero dig2mat_def length_dig2mat' length_upt list.set_sel(1) not_le nth_mem permute_member_set set_upt)
              thus ?thesis
                using `n > 0` `md = ?md - 1` `?md \<ge> 1` hd_dig2mat[of n ?pM]
                using length_permute_dig[of M n p] `p \<in> set (permute [0..<n])` `length M = n * (n - 1)`
                by (simp add: hd_dig2mat)
            qed
            
            ultimately
            show False
              by simp 
          qed
        next
          case True

          hence "?od ! hd p = ?md"
            by simp
          hence "sum_list (?MM ! hd p) = ?md"
            unfolding out_degrees_def
            using `hd p < n` `length M = n * (n-1)`
            by simp

          let ?pp = "?ppp ! hd p"
          let ?pos = "positions (hd p) (?MM ! hd p)"
          have pp: "?pp = ?pos"
            by (simp add: \<open>hd p < n\<close> assms(2))
          hence "length ?pp = md"
            using length_max_positions[of "hd p"]
            using `hd p < n` `?od ! hd p = ?md`
            by simp

          have "length (take md (tl p)) = md"
            using `md < n` `p \<in> set (permute [0..<n])`
            using permute_member_length[of p "[0..<n]"]
            by (auto simp add: min_def)

          have "distinct p"
            using `p \<in> set (permute [0..<n])`
            using distinct_upt isPermutation_permute perm_distinct_iff
            by blast

          have "set p = {0..<n}"
            using `p \<in> set (permute [0..<n])`
            using permute_member_set set_upt
            by blast

          have "set (take (length ?pp) (tl p)) \<noteq> set ?pp"
            using `p \<in> set ?perms1` `p \<notin> set ?perms2`
            by (auto simp add: Let_def)

          have "\<exists> x. x \<in> set (take md (tl p)) \<and> x \<notin> set ?pos"
          proof (rule ccontr)
            assume "\<not> ?thesis"
            hence "set (take md (tl p)) \<subseteq> set ?pos"
              by auto
            moreover
            have "card (set (take md (tl p))) = card (set ?pos)"
            proof-
              have "card (set (take md (tl p))) = md"
                using `distinct p`
                by (metis \<open>length (take md (tl p)) = md\<close> distinct_card distinct_take distinct_tl)
              moreover
              have "card (set ?pos) = md"
              proof-                 
                have "distinct (zip [0..<length (dig2mat n M ! hd p)] (dig2mat n M ! hd p))"
                  by (simp add: distinct_zipI1)
                hence "distinct ?pos"
                  unfolding positions_def
                  by (auto simp add: distinct_map inj_on_def)
                thus ?thesis
                  using `length ?pp = md` `?pp = ?pos`
                  by (simp add: distinct_card)
              qed
              ultimately
              show ?thesis
                by simp
            qed
            moreover
            have "finite (set (take md (tl p)))" "finite (set ?pos)"
              by auto
            ultimately
            have "set (take md (tl p)) = set ?pos"
              using card_subset_eq
              by blast
            thus False
              using `set (take (length ?pp) (tl p)) \<noteq> set ?pp` `length ?pp = md` `?pp = ?pos`
              by simp
          qed
          then obtain x where "x \<in> set (take md (tl p))" "x \<notin> set ?pos"
            by metis

          have "x \<noteq> hd p"
          proof (rule ccontr)
            assume "\<not> x \<noteq> hd p"
            hence "hd p \<in> set (take md (tl p))"
              using `x \<in> set (take md (tl p))`
              by simp
            hence "hd p \<in> set (tl p)"
              using set_take_subset[of md "tl p"]
              by auto
            thus False
              using `distinct p`
              using hd_Cons_tl[of p] `length p = n` `n > 0`
              using distinct.simps(2) by fastforce
          qed

          have "x < n"
          proof-
            have "x \<in> set (tl p)"
              using `x \<in> set (take md (tl p))` 
              using set_take_subset[of md "tl p"]
              by auto
            hence "x \<in> set p"
              using `length p = n` `n > 0`
              using list.set_sel(2) by fastforce
            thus ?thesis
              using `set p = {0..<n}`
              by simp
          qed

          have "?MM ! hd p ! x = 0"
          proof-
            have "?MM ! hd p ! x \<noteq> 1"
            proof (rule ccontr)
              assume "\<not> ?thesis"
              moreover
              have "(x, ?MM ! hd p ! x) \<in> set (zip [0..<length (dig2mat n M ! hd p)] (dig2mat n M ! hd p))"
                using `length (?MM ! hd p) = n` `hd p < n` `x < n`
                by (force simp add: set_zip)
              ultimately
              have "(x, ?MM ! hd p ! x) \<in> set (filter (\<lambda>(i, j). i \<noteq> hd p \<and> j = 1) (zip [0..<length (dig2mat n M ! hd p)] (dig2mat n M ! hd p)))"
                using `x \<noteq> hd p`
                by simp
              thus False
                using `x \<notin> set ?pos`
                unfolding positions_def
                by force
            qed
            moreover
            have "set (?MM ! hd p) \<subseteq> {0, 1}"
              using set_nth_MM[OF `hd p < n`]
              by simp
            ultimately
            show ?thesis
              using `x < n` `length (?MM ! hd p) = n`
              by (smt insertE insert_subset mk_disjoint_insert nth_mem singleton_iff)
          qed

          thus ?thesis
            using take_md_permute_dig[OF `p \<in> set (permute [0..<n])`]
            using `x \<in> set (take md (tl p))`
            by (metis \<open>length (take md (tl p)) = md\<close> in_set_conv_nth nth_map nth_replicate zero_neq_one)
        qed
        thus ?thesis
          using earlier_zero_less[of md M "permute_dig p n M"]
          using `length (permute_dig p n M) = length M` `set (permute_dig p n M) = set M`
          using `take md M = replicate md 1` `set M \<subseteq> {0, 1}`
          using `md < n` `n \<le> length M`
          by (smt le_trans less_imp_le)
      qed
    qed
  qed
qed

end