header{* Executable implementation of set families *}

theory FamilyImpl
imports Main Rat 
        "~~/src/HOL/Library/Mapping" "~~/src/HOL/Library/RBT_Mapping"
        "~~/src/HOL/Library/Efficient_Nat"
        MoreList ListNat Frankl
begin

abbreviation map where "map \<equiv> List.map"

(* -------------------------------------------------------------------------- *)
text{* Set family representation should support the operations specified in the
following locale. *}

locale FamilyImpl =
  fixes toList :: "'s \<Rightarrow> 'a list"
  fixes toSet :: "'s \<Rightarrow> 'a set"
  fixes inv :: "'s \<Rightarrow> bool"
  assumes toSet_toList: "toSet s = set (toList s)"
  assumes inv_set_unique: "\<lbrakk>inv s1; inv s2; toSet s1 = toSet s2\<rbrakk> \<Longrightarrow> s1 = s2"

  fixes empty :: "'s"
  assumes empty_set: "toSet empty = {}"
  assumes empty_inv: "inv empty"

  fixes union :: "'s \<Rightarrow> 's \<Rightarrow> 's" (infixl "\<squnion>" 100)
  assumes union_set: "toSet (s1 \<squnion> s2) = toSet s1 \<union> toSet s2"
  assumes union_inv: "\<lbrakk>inv s1; inv s2\<rbrakk> \<Longrightarrow> inv (s1 \<squnion> s2)"

  fixes pow :: "'s \<Rightarrow> 's list"
  assumes pow_set: "set (map toSet (pow s)) = Pow (toSet s)"
  assumes pow_inv: "inv s \<Longrightarrow> \<forall> A \<in> set (pow s). inv A"
  assumes pow_distinct: "inv s \<Longrightarrow> distinct (pow s)"

  fixes set_weight :: "('a \<Rightarrow> nat) \<Rightarrow> 's \<Rightarrow> nat"
  assumes set_weight: "inv x \<Longrightarrow> set_weight w x = w \<rhd> toSet x"
  fixes set_weight_map :: "('a, nat) mapping \<Rightarrow> 's \<Rightarrow> nat"
  assumes set_weight_map: 
  "\<lbrakk>inv A; \<forall> x \<in> toSet A. Mapping.lookup wm x = Some (w x)\<rbrakk> \<Longrightarrow> 
      set_weight_map wm A = set_weight w A"
begin

(* -------------------------------------------------------------------------- *)

abbreviation f_toSet :: "'s list \<Rightarrow> 'a set set" where
  "f_toSet F \<equiv> set (map toSet F)"

lemma inv_inj: "\<forall>a\<in>set A. inv a \<Longrightarrow> inj_on toSet (set A)"
unfolding inj_on_def
by (simp add: inv_set_unique)

lemma set_set:  "\<lbrakk>\<forall>a\<in>set A. inv a; inv h\<rbrakk> \<Longrightarrow> (h \<in> set A) = (toSet h \<in> f_toSet A)"
using inv_set_unique
by auto

(* -------------------------------------------------------------------------- *)

lemma in_set_pow:
  assumes "inv A" "inv S" "toSet A \<subseteq> toSet S"
  shows "A \<in> set (pow S)"
proof-
  have "toSet A \<in> f_toSet (pow S)"
    using `toSet A \<subseteq> toSet S` pow_set[of S]
    by simp
  thus ?thesis
    using `inv A` `inv S`
    using pow_inv inv_set_unique
    by auto
qed

(* -------------------------------------------------------------------------- *)

definition Union :: "'s list \<Rightarrow> 's" where
"Union F = foldl union empty F"

lemma Union_set: "toSet (Union F) = \<Union> (f_toSet F)"
by (induct F rule: rev_induct) (auto simp add: Union_def empty_set union_set)

lemma Union_inv: "\<forall> A \<in> set F. inv A \<Longrightarrow> inv (Union F)"
unfolding Union_def
by (induct F rule: rev_induct) (auto simp add: empty_inv union_inv)

(* -------------------------------------------------------------------------- *)

abbreviation union_set :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  "union_set A F \<equiv> (map (op \<squnion> A) F)"

abbreviation insert_set :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_set A F \<equiv> (if A \<in> set F then F else A # F)"

definition insert_sets :: "'s list \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_sets l F = foldl (\<lambda> F' a. insert_set a F') F l"

abbreviation insert_and_close :: "'s \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_and_close A F \<equiv> insert_sets ([A] @ union_set A F) F"

abbreviation insert_and_close_additional :: "'s \<Rightarrow> 's list \<Rightarrow> 's list \<Rightarrow> 's list" where
  "insert_and_close_additional A F I \<equiv> insert_sets ([A] @ union_set A F @ union_set A I) F"

definition close where
"close F = foldl (\<lambda> F A. insert_and_close A F) [] F"

lemma toSet_union_set: "map toSet (union_set A F) = map (op \<union> (toSet A) \<circ> toSet) F"
by (simp add: comp_def union_set)

lemma set_toSet_union_set: "(toSet ` (op \<squnion> A ` set F)) = (op \<union> (toSet A) ` toSet ` set F)"
proof-
  have "set (map toSet (union_set A F)) = set (map (op \<union> (toSet A) \<circ> toSet) F)"
    using toSet_union_set[of A F]
    by (auto simp add: comp_def)
  thus ?thesis
    by (simp add: image_compose)
qed

lemma insert_sets_remdups: "distinct F \<Longrightarrow> insert_sets l F = remdups (rev l @ F)"
unfolding insert_sets_def
by (induct l rule: rev_induct) (auto simp add: distinct_remdups_id)

lemma distinct_insert_sets: "distinct F \<Longrightarrow> distinct (insert_sets l F)"
using insert_sets_remdups[of F l]
by auto

lemma insert_and_close:
  assumes "distinct F"
  shows "f_toSet (insert_and_close h F) = 
         Frankl.insert_and_close (toSet h) (f_toSet F)"
  using assms
  using insert_sets_remdups
  using set_toSet_union_set[of h F, THEN sym]
  by auto

lemma  insert_and_close_inv:
  assumes "\<forall> l \<in> set F. inv l" "inv h" "distinct F"
  shows "\<forall> l \<in> set (insert_and_close h F). inv l"
using assms
by (auto simp add: insert_sets_remdups union_inv)

lemma  insert_and_close_subset:
  assumes "\<forall> l \<in> set F. toSet l \<subseteq> S" "toSet h \<subseteq> S" "distinct F"
  shows "\<forall> l \<in> set (insert_and_close h F). toSet l \<subseteq> S"
proof (safe)
  fix x l
  assume "l \<in> set (insert_and_close h F)" "x \<in> toSet l"
  hence "x \<in> \<Union> (f_toSet (insert_and_close h F))"
    by auto
  hence "x \<in> \<Union> (Frankl.insert_and_close (toSet h) (f_toSet F))"
    using `distinct F`
    apply (subst insert_and_close[of F h, THEN sym])
    by simp_all
  thus "x \<in> S"
    using assms
    by (auto split: split_if_asm)
qed

lemma close_distinct: 
  "distinct (close A)"
  unfolding close_def
by (induct A rule: rev_induct) (auto simp add: distinct_insert_sets)

lemma close_set: 
  "f_toSet (close A) = closure (f_toSet A)"
proof (induct A rule: rev_induct)
  case (snoc l a)
  let ?L = "\<lambda> x. foldl (\<lambda>F A. local.insert_and_close A F) [] x"
  show ?case
    using close_distinct[of a]
    using insert_and_close[of "?L a"]
    using closure_insert[of "f_toSet a" "toSet l"]
    using snoc(1)
    unfolding close_def
    by simp
next
  case Nil
  thus ?case
    by (simp add: close_def closure_def)
qed

lemma close_inv:
  assumes "\<forall> l \<in> set F. inv l"
  shows "\<forall> l \<in> set (close F). inv l"
using assms
proof (induct F rule: rev_induct)
  case (snoc a l)
  thus ?case
    using close_distinct[of l]
    by (auto simp add: close_def insert_sets_remdups union_inv)
qed (simp add: close_def)

lemma close_subset:
  assumes "\<forall> l \<in> set F. toSet l \<subseteq> S"
  shows "\<forall> l \<in> set (close F). toSet l \<subseteq> S"
using assms
proof (induct F rule: rev_induct)
  case (snoc a l)
  thus ?case
    using insert_and_close_subset[of "close l" S a]
    using close_distinct[of l]
    unfolding close_def
    by simp
qed (simp add: close_def)

lemma insert_and_close_additional:
  assumes "distinct F"
  shows "f_toSet (insert_and_close_additional h F Init) = 
         Frankl.insert_and_close_additional (toSet h) (f_toSet F) (f_toSet Init)"
  using assms
  using insert_sets_remdups
  using set_toSet_union_set[of h F, THEN sym]
  using set_toSet_union_set[of h Init, THEN sym]
  by auto

lemma  insert_and_close_additional_subset:
  assumes "\<forall> l \<in> set F. toSet l \<subseteq> S" "toSet h \<subseteq> S" "\<forall> l \<in> set I. toSet l \<subseteq> S" "distinct F"
  shows "\<forall> l \<in> set (insert_and_close_additional h F I). toSet l \<subseteq> S"
proof (safe)
  fix x l
  assume "l \<in> set (insert_and_close_additional h F I)" "x \<in> toSet l"
  hence "x \<in> \<Union> (f_toSet (insert_and_close_additional h F I))"
    by auto
  hence "x \<in> \<Union> (Frankl.insert_and_close_additional (toSet h) (f_toSet F) (f_toSet I))"
    using `distinct F`
    apply (subst insert_and_close_additional[of F h I, THEN sym])
    by simp_all
  thus "x \<in> S"
    using assms
    by (auto split: split_if_asm)
qed

lemma  insert_and_close_additional_inv:
  assumes "\<forall> l \<in> set F. inv l" "inv h" "\<forall> l \<in> set I. inv l" "distinct F"
  shows "\<forall> l \<in> set (insert_and_close_additional h F I). inv l"
using assms
by (auto simp add: insert_sets_remdups union_inv)

lemma insert_and_close_additional_equal_set:
  assumes "set F = set F'" "distinct F" "distinct F'"
  shows "set (insert_and_close_additional A F I) = set (insert_and_close_additional A F' I)"
  using assms
  by (simp add: insert_sets_remdups)

(* -------------------------------------------------------------------------- *)
definition set_share :: "'s \<Rightarrow> ('a \<Rightarrow> nat) \<Rightarrow> 's \<Rightarrow> int" where
 "set_share A w X = 2 * int (set_weight w A) - int (set_weight w X)"

definition Family_share :: "'s list \<Rightarrow> ('a \<Rightarrow> nat) \<Rightarrow> 's \<Rightarrow> int" where
  "Family_share F w X = listsum (map (\<lambda> A. set_share A w X) F)"

lemma set_share:
  assumes "inv x" "inv X"
  shows "set_share x w X = Frankl.set_share (toSet x) w (toSet X)"
unfolding Frankl.set_share_def set_share_def
using assms
by (auto simp add: set_weight)

lemma Family_share:
  assumes "distinct F" "\<forall> l \<in> set F. inv l" "inv X"
  shows "Family_share F w X = Frankl.Family_share (f_toSet F) w (toSet X)"
unfolding Family_share_def Frankl.Family_share_def
apply (subst listsum_distinct_conv_setsum_set)
using assms
apply simp_all
apply (subst setsum_reindex)
using assms
apply (auto simp add: inv_inj)
apply (rule setsum_cong)
using assms
by (auto simp add: set_share)

abbreviation set_share_map :: "'s \<Rightarrow> ('a, nat) mapping \<Rightarrow> 's \<Rightarrow> int" where
 "set_share_map A w X \<equiv> 2 * int(set_weight_map w A) - int (set_weight_map w X)"

(* -------------------------------------------------------------------------- *)

abbreviation Family_share_cached :: "'s list \<Rightarrow> ('s, int) mapping \<Rightarrow> int" where
  "Family_share_cached F shares \<equiv> listsum (map (the \<circ> (\<lambda> A. Mapping.lookup shares A)) F)"

lemma Family_share_cached_Family_share:
  assumes "\<forall> A. inv A \<and> toSet A \<subseteq> S \<longrightarrow> Mapping.lookup shares A = Some (set_share A w X)"
          "\<forall> A \<in> set F. inv A \<and> toSet A \<subseteq> S"
  shows "Family_share_cached F shares = Family_share F w X"
proof-
  have "map (\<lambda> x. the (Mapping.lookup shares x)) F = map (\<lambda> A. local.set_share A w X) F"
    using assms
    by auto
  thus ?thesis
    unfolding Family_share_def comp_def
    by metis
qed

lemma Family_share_cached_equal_set:
  assumes "set F = set F'" "distinct F" "distinct F'"
  shows "Family_share_cached F shares = Family_share_cached F' shares"
using assms
using listsum_distinct_conv_setsum_set[of F "the \<circ> (\<lambda> A. Mapping.lookup shares A)"]
using listsum_distinct_conv_setsum_set[of F' "the \<circ> (\<lambda> A. Mapping.lookup shares A)"]
by simp

(* -------------------------------------------------------------------------- *)

abbreviation insert_and_close_additional' where
  "insert_and_close_additional' A FS I shares \<equiv>
      let add = [A] @ union_set A (fst FS) @ union_set A I;
          add = filter (\<lambda> x. x \<notin> set (fst FS)) (remdups add) in 
        (add @ (fst FS), (snd FS) + Family_share_cached add shares)"

lemma insert_and_close_additional'_set:
  assumes "distinct F"
  shows
  "let add = [A] @ union_set A F @ union_set A I;
       add = filter (\<lambda> x. x \<notin> set F) (remdups add) in
  set (add @ F) = set (insert_and_close_additional A F I) \<and> distinct (add @ F)"
unfolding Let_def
using assms
by (auto simp add: insert_sets_remdups)

lemma insert_and_close_additional'_Family_share_cached:
  shows "let (F', s') = insert_and_close_additional' A (F, Family_share_cached F shares) I shares in
  s' = Family_share_cached F' shares"
by (auto simp add: Let_def)

lemma insert_and_close_additional':
assumes
  "distinct F"
  "insert_and_close_additional' A (F, Family_share_cached F shares) I shares = (F1, s1)"
  "let F' = insert_and_close_additional A F I;
       s' = Family_share_cached F' shares in (F2, s2) = (F', s')"
shows
  "set F1 = set F2 \<and> s1 = s2"
proof-
  let ?add = "[A] @ union_set A F @ union_set A I"
  let ?add = "filter (\<lambda> x. x \<notin> set F) (remdups ?add)"

  have "F1 = ?add @ F" "distinct F1" "s1 = Family_share_cached F shares + Family_share_cached ?add shares"
    using `insert_and_close_additional' A (F, Family_share_cached F shares) I shares = (F1, s1)`
    unfolding Let_def
    using `distinct F`
    by auto
  moreover
  have "set F2 = set (?add @ F)" "distinct F2" "s2 = Family_share_cached F2 shares"
    using `let F' = local.insert_and_close_additional A F I;
               s' = Family_share_cached F' shares in (F2, s2) = (F', s')` `distinct F`
    using insert_and_close_additional'_set[of F A I]
    unfolding Let_def
    using distinct_insert_sets
    by auto

  ultimately
  show "set F1 = set F2 \<and> s1 = s2"
    using Family_share_cached_equal_set[of F1 F2 shares]
    by auto
qed

(* -------------------------------------------------------------------------- *)

end (* Locale *)

(* ----------------------------------------------------------------------- *)
subsection{* Implementation by sets represented by (sorted and distinct) lists *}
(* ----------------------------------------------------------------------- *)

text{* For example, the family @{text "{{1, 2, 3}, {2, 3, 4}"} is represented
by @{text "[[1, 2, 3], [2, 3, 4]]"}. *}

definition set_weight_l :: "('a \<Rightarrow> nat) \<Rightarrow> 'a list \<Rightarrow> nat"where
 "set_weight_l w S = listsum (map w S)"

lemma set_weight_l:
  assumes "distinct x"
  shows "set_weight_l w x = w \<rhd> set x"
unfolding set_weight_l_def set_weight_def
apply (subst listsum_distinct_conv_setsum_set)
using assms
by simp_all

definition set_weight_map_l :: "('b, nat) mapping \<Rightarrow> 'b list \<Rightarrow> nat" where
  "set_weight_map_l w S = listsum (map (the \<circ> (Mapping.lookup w)) S)"

lemma set_weight_map_l: 
  "\<lbrakk>distinct A; \<forall> x \<in> set A. Mapping.lookup wm x = Some (w x)\<rbrakk> \<Longrightarrow> 
  set_weight_map_l wm A = set_weight_l w A"
unfolding set_weight_l_def set_weight_map_l_def
using listsum_distinct_conv_setsum_set[of A w]
using listsum_distinct_conv_setsum_set[of A "the \<circ> (Mapping.lookup wm)"]
by auto

fun merge :: "nat list \<Rightarrow> nat list \<Rightarrow> nat list" (infixl "\<squnion>" 100) where
  "merge [] l = l"
| "merge l [] = l"
| "merge (h1 # t1) (h2 # t2) = 
     (if h1 < h2 then 
          h1 # merge t1 (h2 # t2)
      else if h1 > h2 then
          h2 # merge (h1 # t1) t2
      else
          h1 # merge t1 t2)"

lemma set_merge: "set (l1 \<squnion> l2) = set l1 \<union> set l2"
by (induct l1 l2 rule: merge.induct) auto

lemma sorted_merge:
  assumes "sorted l1" "sorted l2"
  shows "sorted (l1 \<squnion> l2)"
using assms
by (induct l1 l2 rule: merge.induct) (auto simp add: sorted_Cons set_merge)

lemma distinct_merge:
  assumes "distinct l1" "distinct l2" "sorted l1" "sorted l2"
  shows "distinct (l1 \<squnion> l2)"
using assms
by (induct l1 l2 rule: merge.induct) (auto simp add: sorted_Cons set_merge)

interpretation FamilyImpl_lists:
  FamilyImpl id "List.set" "\<lambda> l. sorted l \<and> distinct l" "[]" merge ListPow set_weight_l set_weight_map_l
proof
  fix s
  show "set (map set (ListPow s)) = Pow (set s)"
    by (rule setListPow)
next
  fix s
  assume "sorted s \<and> distinct s"
  thus "\<forall>A\<in>set (ListPow s). sorted A \<and> distinct A"
    using sortedListPowElems distinctListPowElems
    by auto
next
  fix A and wm :: "('a::linorder, nat) mapping" and w :: "'a \<Rightarrow> nat"
  assume "sorted A \<and> distinct A" "\<forall>x\<in>set A. Mapping.lookup wm x = Some (w x)"
  thus "set_weight_map_l wm A = set_weight_l w A"
    by (simp add: set_weight_map_l)
qed (auto simp add: set_weight_l set_merge sorted_merge distinct_merge sorted_distinct_set_unique distinctListPow)

(* ----------------------------------------------------------------------- *)
subsection{* Implementation by sets represented as natural numbers *}
(* ----------------------------------------------------------------------- *)

text{* For example, the family @{text "{{0, 1, 2}, {1, 2, 3}}"} is represented
as @{text "[7, 14]"}. *}

function bitor :: "nat \<Rightarrow> nat \<Rightarrow> nat" where
 "bitor x y = 
  (if x = 0 then y
   else if y = 0 then x else
   let (xd, xm) = divmod_nat x 2;
       (yd, ym) = divmod_nat y 2 in
   if (xm = 1 | ym = 1)
   then 1 + 2*bitor xd yd
   else 2*bitor xd yd)"
by pat_completeness auto
termination
by  (relation "measure (\<lambda> (x, _). x)", auto simp add: divmod_nat_div_mod)
declare bitor.simps[simp del]

definition set_weight_n :: "(nat \<Rightarrow> nat) \<Rightarrow> nat \<Rightarrow> nat" where
  "set_weight_n w n = listsum (map w (nat2list n))"

definition set_weight_map_n :: "(nat, nat) Mapping.mapping \<Rightarrow> nat \<Rightarrow> nat" where
  "set_weight_map_n w n = listsum (map (the \<circ> (Mapping.lookup w)) (nat2list n))"

definition pow_n :: "nat \<Rightarrow> nat list" where
"pow_n n = map list2nat (ListPow (nat2list n))"

definition close_n where
 [simp, code del]: "close_n = FamilyImpl.close bitor"
definition insert_sets_n :: "nat list \<Rightarrow> nat list \<Rightarrow> nat list" where
 [simp, code del]: "insert_sets_n = FamilyImpl.insert_sets"
definition insert_and_close_n where
 [simp, code del]: "insert_and_close_n = FamilyImpl.insert_and_close bitor"
definition Union_n where
 [simp, code del]: "Union_n = FamilyImpl.Union 0 bitor"

lemma nat2list_bitor: "set (nat2list (bitor x y)) = set (nat2list x) \<union> set (nat2list y)"
proof (induct x y rule: bitor.induct)
  case (1 x' y')
  show ?case
  proof (cases "x' = 0")
    case True
    thus ?thesis
      by (simp add: bitor.simps nat2list_def)
  next
    case False
    show ?thesis
    proof (cases "y' = 0")
      case True
      thus ?thesis
        using `x' \<noteq> 0`
        by (simp add: bitor.simps nat2list_def)
    next
      case False
      show ?thesis
      proof (cases "x' mod 2 \<noteq> Suc 0 \<and> y' mod 2 \<noteq> Suc 0")
        case True
        hence "bitor x' y' = 2 * bitor (x' div 2) (y' div 2)"
          using `x' \<noteq> 0` `y' \<noteq> 0`
          using bitor.simps[of x' y']
          by (simp add: split_def Let_def div_nat_def[THEN sym] mod_nat_def[THEN sym] split: split_if_asm)
        hence "nat2list (bitor x' y') = map (op + 1) (nat2list (bitor (x' div 2) (y' div 2)))"
          by (simp add: nat2list_even)
        moreover
        have "set (nat2list (bitor (x' div 2) (y' div 2))) = set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))"
          using 1(2)[of "divmod_nat x' 2" "x' div 2" "x' mod 2"
                        "divmod_nat y' 2" "y' div 2" "y' mod 2"]
          using `x' \<noteq> 0` `y' \<noteq> 0` True
          by (simp add: div_nat_def mod_nat_def)
        moreover
        from True
        have "x' mod 2 = 0" "y' mod 2 = 0"
          by auto
        hence "x' = 2 * (x' div 2)" "y' = 2 * (y' div 2)"
          by  auto
        hence "nat2list x' = map (op +1) (nat2list (x' div 2))"
              "nat2list y' = map (op +1) (nat2list (y' div 2))"
          using nat2list_even[of "x' div 2"] nat2list_even[of "y' div 2"]
          by auto
        ultimately
        show ?thesis
          by auto
      next
        case False
        hence "bitor x' y' = 2 * bitor (x' div 2) (y' div 2) + 1"
          using `x' \<noteq> 0` `y' \<noteq> 0`
          using bitor.simps[of x' y']
          by (simp add: split_def Let_def div_nat_def[THEN sym] mod_nat_def[THEN sym] split: split_if_asm)
        hence "nat2list (bitor x' y') = 0 # map (op + 1) (nat2list (bitor (x' div 2) (y' div 2)))"
          using nat2list_odd
          by simp
        moreover
        have "set (nat2list (bitor (x' div 2) (y' div 2))) = set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))"
          using 1(1)[of "divmod_nat x' 2" "x' div 2" "x' mod 2"
                        "divmod_nat y' 2" "y' div 2" "y' mod 2"]
          using `x' \<noteq> 0` `y' \<noteq> 0` False
          by (simp add: div_nat_def mod_nat_def)
        moreover
        from False
        have *: "x' mod 2 = 0 \<and> y' mod 2 = 1 \<or> x' mod 2 = 1 \<and> y' mod 2 = 0 \<or> x' mod 2 = 1 \<and> y' mod 2 = 1"
          by auto
        have "set (nat2list x') \<union> set (nat2list y') = {0} \<union> (op +1 ` (set (nat2list (x' div 2)) \<union> set (nat2list (y' div 2))))"
        proof-
          {
            assume "x' mod 2 = 0 \<and> y' mod 2 = 1"
            hence "x' = 2 * (x' div 2) \<and> y' = 2 * (y' div 2) + 1"
              using mod_div_equality[of y' 2, THEN sym] mod_div_equality[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_even[of "x' div 2"] nat2list_odd[of "y' div 2"]
              by auto
          }
          moreover
          {
            assume "x' mod 2 = 1 \<and> y' mod 2 = 0"
            hence "x' = 2 * (x' div 2) + 1 \<and> y' = 2 * (y' div 2)"
              using mod_div_equality[of y' 2, THEN sym] mod_div_equality[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_even[of "y' div 2"] nat2list_odd[of "x' div 2"]
              by auto
          }
          moreover
          {
            assume "x' mod 2 = 1 \<and> y' mod 2 = 1"
            hence "x' = 2 * (x' div 2) + 1 \<and> y' = 2 * (y' div 2) + 1"
              using mod_div_equality[of y' 2, THEN sym] mod_div_equality[of x' 2, THEN sym]
              by simp
            hence ?thesis
              using nat2list_odd[of "x' div 2"] nat2list_odd[of "y' div 2"]
              by auto
          }
          ultimately
          show ?thesis
            using *
            by blast
        qed
        ultimately
        show ?thesis
          by simp
      qed
    qed
  qed
qed

interpretation FamilyImpl_nats: FamilyImpl nat2list "set \<circ> nat2list" "\<lambda> n. True" 0 bitor pow_n set_weight_n set_weight_map_n where
"FamilyImpl.close bitor = close_n" and
"FamilyImpl.insert_sets = insert_sets_n" and
"FamilyImpl.insert_and_close bitor = insert_and_close_n" and
"FamilyImpl.Union 0 bitor = Union_n"
proof (unfold_locales)
  fix s1 s2
  assume "(set \<circ> nat2list) s1 = (set \<circ> nat2list) s2"
  thus "s1 = s2"
    using sorted_nat2list[of s1] sorted_nat2list[of s2]
    using distinct_nat2list[of s1] distinct_nat2list[of s2]
    using sorted_distinct_set_unique[of "nat2list s1" "nat2list s2"]
    using inj_nat2list
    by (simp add: inj_on_def)
next
  fix x w
  show "set_weight_n w x = w \<rhd> (set \<circ> nat2list) x"
    using distinct_nat2list[of x]
    unfolding set_weight_n_def set_weight_def
    by (simp add: listsum_distinct_conv_setsum_set)
next
  fix A wm and w :: "nat \<Rightarrow> nat"
  assume "\<forall>x\<in>(set \<circ> nat2list) A. Mapping.lookup wm x = Some (w x)"
  thus "set_weight_map_n wm A = set_weight_n w A"
    unfolding set_weight_n_def set_weight_map_n_def
    using distinct_nat2list[of A]
    by (auto simp add: listsum_distinct_conv_setsum_set)
next
  show "(set \<circ> nat2list) 0 = {}"
    by (simp add: nat2list_def)
next
  fix s1 s2
  show "(set \<circ> nat2list) (bitor s1 s2) = (set \<circ> nat2list) s1 \<union> (set \<circ> nat2list) s2"
    using nat2list_bitor
    by simp
next
  fix s
  show "set (map (set \<circ> nat2list) (pow_n s)) = Pow ((set \<circ> nat2list) s)"
  proof-
    have "(set \<circ> nat2list \<circ> list2nat) ` set (ListPow (nat2list s)) = set ` set (ListPow (nat2list s))"
    proof (rule image_cong, simp)
      fix x
      assume "x \<in> set (ListPow (nat2list s))"
      thus "(set \<circ> nat2list \<circ> list2nat) x = set x"
        using sorted_nat2list[of s] distinct_nat2list[of s]
        using nat2list_list2nat[of x] sortedListPowElems[of "nat2list s" x] distinctListPowElems[of "nat2list s" x]
        by simp
    qed
    thus ?thesis
      unfolding pow_n_def
      using setListPow[of "nat2list s"]
      by simp
  qed
next
  fix s
  show "distinct (pow_n s)"
    unfolding pow_n_def
    by (auto simp add: distinct_map distinct_nat2list distinctListPow 
        inj_on_def  inj_list2nat 
        sortedListPowElems[of "nat2list s"] sorted_nat2list
        distinctListPowElems[of "nat2list s"])
qed simp_all

end