header{* Frankl's conjecture *}

theory Frankl
imports Main Rat
        MoreSet MoreFun MoreBig_Operators
begin

(* ************************************************************************** *)
subsection{* Union closed families *}
(* ************************************************************************** *)

definition union_closed :: "'a set set \<Rightarrow> bool" where
  "union_closed F \<equiv> \<forall> A B. A \<in> F \<and> B \<in> F \<longrightarrow> A \<union> B \<in> F"

abbreviation finite_union_closed where
  "finite_union_closed F \<equiv> union_closed F \<and> finite (\<Union> F)"

text{* Union of each @{text "n"} (@{text "n > 0"}) family members
belongs to the family. *}
lemma union_closed_n:
  assumes "finite F'" and "F' \<noteq> {}" and "F' \<subseteq> F"
  assumes "union_closed F"
  shows "\<Union> F' \<in> F"
using assms
proof (induct F')
  case empty
  thus ?case
    by simp
next
  case (insert a F'')
  thus ?case
    using insert
    by (cases "F'' = {}") (auto simp add: union_closed_def)
qed

(* ************************************************************************** *)
subsubsection{* Closure -- minimal union closed family containing F *}
(* ************************************************************************** *)

definition closure where
  "closure F = Union ` (Pow F - {{}})"

lemma closure_closed: 
  "union_closed (closure F)"
unfolding union_closed_def
proof ((rule allI)+ , rule impI, erule conjE)
  fix A B :: "'a set"
  assume "A \<in> closure F" "B \<in> closure F"
  then obtain xa xb where
    "A = \<Union> xa" "xa \<in> Pow F - {{}}"
    "B = \<Union> xb" "xb \<in> Pow F - {{}}"
    by (auto simp add: closure_def)
  have "\<Union>xa \<union> \<Union>xb = \<Union> (xa \<union> xb)"
    by auto
  moreover
  have "xa \<union> xb \<in> Pow F - {{}}"
    using `xa \<in> Pow F - {{}}` `xb \<in> Pow F - {{}}`
    by auto
  ultimately
  show "A \<union> B \<in> closure F"
    using `A = \<Union> xa` `B = \<Union> xb`
    unfolding closure_def
    by blast
qed

lemma closure_min_closed:
  assumes "finite F" and "F \<subseteq> F'" and "union_closed F'"
  shows "closure F \<subseteq> F'"
proof
  fix x
  assume "x \<in> closure F"
  then obtain S where "x = \<Union> S" "S \<in> Pow F - {{}}"
    unfolding closure_def
    by auto
  thus "x \<in> F'"
    using union_closed_n[of S F'] `union_closed F'` `finite F` `F \<subseteq> F'`
    using finite_subset
    by auto
qed

lemma closure_union_closed_id: 
  "\<lbrakk>finite F; union_closed F\<rbrakk> \<Longrightarrow> closure F = F"
using closure_min_closed[of F F]
by (auto simp add: closure_def)

lemma finite_closure:
  assumes "finite (\<Union> F)"
  shows "finite (\<Union> (closure F))"
proof (subst finiteUn_iff, rule conjI)
  show "finite (closure F)"
    using assms
    unfolding closure_def
    by (auto simp add: finiteUn_iff)
next
  show "\<forall>X\<in>closure F. finite X"
  proof
    fix X
    assume "X \<in> closure F"
    then obtain z where "X = \<Union> z" "z \<noteq> {}" "z \<subseteq> F"
      unfolding closure_def
      by auto
    hence "finite z" "\<forall> X \<in> z. finite X"
      using `finite (\<Union> F)`
      by (auto simp add: finite_subset finiteUn_iff)
    thus "finite X"
      using `X = \<Union> z`
      by auto
  qed
qed

text{* Iterative closure: insert set to closed family and close. *}

abbreviation insert_and_close :: "'a set \<Rightarrow> 'a set set \<Rightarrow> 'a set set" where
  "insert_and_close A F \<equiv> F \<union> {A} \<union> (op \<union> A) ` F"

lemma union_closed_insert_and_close:
  assumes "union_closed F"
  shows "union_closed (insert_and_close A F)"
using assms
unfolding union_closed_def
by auto

lemma  closure_insert:
  assumes "finite F"
  shows "closure (F \<union> {A}) = insert_and_close A (closure F)" (is "?lhs = ?rhs")
proof
  show "?lhs \<subseteq> ?rhs"
    using assms union_closed_insert_and_close[of "closure F" A]
    using closure_closed[of F]
    using closure_min_closed[of "F \<union> {A}" ?rhs]
    by (auto simp add: closure_def)
next
  show "?rhs \<subseteq> ?lhs"
  proof
    fix x
    assume "x \<in> ?rhs"
    hence "x \<in> closure F \<union> {A} \<or> x \<in> op \<union> A ` (closure F)"
      by simp
    thus "x \<in> ?lhs"
    proof
      assume "x \<in> closure F \<union> {A}"
      thus "x \<in> closure (F \<union> {A})"
        by (auto simp add: closure_def)
    next
      assume "x \<in> op \<union> A ` (closure F)"
      then obtain y where "x = y \<union> A" "y \<in> closure F"
        by auto
      then obtain k where "k \<in> Pow F" "y = \<Union> k" "k \<noteq> {}"
        unfolding closure_def
        by auto

      show ?thesis
        unfolding closure_def
        apply (rule rev_image_eqI[of "k \<union> {A}"])
        using `x = y \<union> A` `y = \<Union> k` `k \<noteq> {}` `k \<in> Pow F`
        by auto
    qed
  qed
qed

lemma insert_and_close_closure:
  assumes "finite F" and "union_closed F"
  shows "closure (F \<union> {A}) = insert_and_close A F" (is "?lhs = ?rhs")
using assms
using closure_insert[of F A]
using closure_union_closed_id[of F]
by simp

(* ************************************************************************** *)
subsection{* Union closed families closed to unions with an additional family *}
(* ************************************************************************** *)

abbreviation union_closed_additional where 
  "union_closed_additional F I \<equiv> union_closed F \<and> (\<forall> A \<in> F. (op \<union> A) ` I \<subseteq> F)"

lemma union_closed_additional: 
  "(\<forall> A \<in> F. \<forall> Ai \<in> I. A \<union> Ai \<in> F) \<longleftrightarrow> (\<forall> A \<in> F. (op \<union> A) ` I \<subseteq> F)"
by auto

abbreviation insert_and_close_additional where
  "insert_and_close_additional A F I \<equiv> F \<union> {A} \<union> ((op \<union> A) ` F) \<union> ((op \<union> A) ` I)"

lemma closure_additional_set:
  assumes "finite F" and "A \<in> closure F" and
  "A' \<in> F'" and "union_closed F'" and "\<forall> A' \<in> F'. op \<union> A' ` F \<subseteq> F'"
  shows "A \<union> A' \<in> F'"
proof-
  have "op \<union> A' ` F \<subseteq> F'"
    using `\<forall>A'\<in>F'. image (op \<union> A') F \<subseteq> F'` `A' \<in> F'`
    by simp
  obtain X where "A = \<Union> X" "X \<subseteq> F" "X \<noteq> {}"
    using `A \<in> closure F`
    unfolding closure_def
    by auto
  hence "finite X"
    using `finite F`
    by (auto simp add: finite_subset)
  
  have "A \<union> A' = \<Union> (op \<union> A') ` X"
    using `A = \<Union> X` `X \<noteq> {}`
    by auto

  show ?thesis
  proof (subst `A \<union> A' = \<Union> image (op \<union> A') X`, rule union_closed_n)
    show "finite (op \<union> A' ` X)"
      using `finite X`
      by simp
  next
    show "op \<union> A' ` X \<noteq> {}"
      using `X \<noteq> {}`
      by simp
  next
    show "op \<union> A' ` X \<subseteq> F'"
      using `image (op \<union> A') F \<subseteq> F'`
      using `X \<subseteq> F`
      by auto
  next
    show "union_closed F'"
      by fact
  qed
qed

lemma union_closed_additional_closure:
  assumes "finite A" and "union_closed_additional F A"
  shows "union_closed_additional F (closure A)"
proof (safe)
  fix x y
  assume "x \<in> F" "y \<in> closure A"
  thus "x \<union> y \<in> F"
    using closure_additional_set[of A y x F]
    using `union_closed_additional F A` `finite A`
    by (auto simp add: Un_commute)
qed (simp add: `union_closed_additional F A`)


(* ************************************************************************** *)
subsection{* Frankl's condition *}
(* ************************************************************************** *)
  
definition count :: "'a \<Rightarrow> 'a set set \<Rightarrow> nat" where 
 "count a F \<equiv> card {S \<in> F. a \<in> S}"

definition frankl :: "'a set set \<Rightarrow> bool" where
 "frankl F \<equiv> finite (\<Union> F) \<and> (\<exists> x. x \<in> \<Union> F \<and> 2 * count x F \<ge> card F)"

text{* Note that, in order to avoid using rational numbers, the
condition is not formulated as @{text "count x F \<ge> (card F) / 2"}, what
would be more expected. *}

lemma franklFinite1:
  assumes "frankl F" 
  obtains "F \<noteq> {}" and "finite F" and "finite (\<Union> F)"
  using assms
  unfolding frankl_def
  by (auto simp add: finiteUn_iff)

lemma franklFinite2:
  assumes "frankl F" and "S \<in> F" 
  obtains "finite S"
using assms
unfolding frankl_def
by (auto simp add: finiteUn_iff)

lemma franklElement:
  assumes "frankl F"
  obtains x where "x \<in> \<Union> F" and " 2 * count x F \<ge> card F"
using assms
unfolding frankl_def
by auto

(* ************************************************************************** *)
subsection{* Isomorphisms of set families *}
(* ************************************************************************** *)

text{* Each injective function @{text "f"} on the domain @{text "\<Union> F"}
maps the set family @{text "F"} to its isomorphic image @{text "op ` f
` F"}. Elements of @{text "\<Union> F"} are mapped using @{text "f"}, while
elements of @{text "F"} are mapped using @{text "op ` f"}. *}

lemma empty_iso: "F = {} \<longleftrightarrow> (op ` f ` F) = {}"
  by auto

lemma inj_on_iso:
  assumes "inj_on f (\<Union> F)"
  shows "inj_on (op ` f) F"
using assms
unfolding inj_on_def
by blast

lemma inj_on_iso_strong:
  assumes "inj_on f (\<Union> F)"
  shows "inj_on (op ` f) (Pow (\<Union> F))"
using assms
unfolding inj_on_def
by blast

lemma finite_iso:
  assumes "inj_on f (\<Union> F)"
  shows "finite F \<longleftrightarrow> finite (op ` f ` F)"
  using assms
  using inj_on_iso[of f F]
  by (auto dest: finite_imageD)

lemma card_iso:
  assumes "inj_on f (\<Union> F)"
  shows "card (op ` f ` F) = card F"
proof (subst card_image)
  let ?f = "op ` f"
  show "inj_on ?f F"
    using assms
    by (simp add: inj_on_iso)
qed simp

lemma finite_elems_iso:
  assumes "inj_on f (\<Union> F)"
  shows "(\<forall>S\<in>F. finite S) \<longleftrightarrow> (\<forall>S\<in>(op ` f ` F). finite S)"
proof (safe)
  let ?F = "op ` f ` F"
  fix S
  assume "\<forall> S \<in> ?F. finite S" "S \<in> F"
  
  show "finite S"
  proof (rule finite_imageD)
    show "finite (f ` S)"
      using `\<forall> S \<in> ?F. finite S` `S \<in> F`
      by simp
  next
    show "inj_on f S"
    proof (rule subset_inj_on)
      show "S \<subseteq> \<Union> F"
        using `S \<in> F`
        by auto
    qed (simp add: assms)
  qed
qed auto

lemma card_elem_iso:
  assumes "inj_on f (\<Union> F)" "A \<in> F" "card A > 0"
  shows "card (f ` A) = card A"
using assms
using inj_on_iff_eq_card[of A f]
using subset_inj_on[of f "\<Union> F" A]
using card_ge_0_finite[of A]
by blast

lemma finite_Union_iso:
  assumes "inj_on f (\<Union> F)"
  shows "finite (\<Union> F) = finite (\<Union> (op ` f ` F))"
using assms
by ((subst finiteUn_iff)+, simp add: finite_iso finite_elems_iso)

lemma union_closed_iso:
  assumes "inj_on f (\<Union> F)"
  shows "union_closed F \<longleftrightarrow> union_closed (op ` f ` F)"
proof
  assume "union_closed F"
  show "union_closed (op ` f ` F)"
    unfolding union_closed_def
  proof (safe)
    fix A B x y
    assume *: "x \<in> F" "y \<in> F"
    show "f ` x \<union> f ` y \<in> op ` f ` F"
    proof (rule rev_image_eqI[of "x \<union> y"])
      show "x \<union> y \<in> F"
        using `union_closed F` `x \<in> F` `y \<in> F`
        unfolding union_closed_def
        by simp
    qed auto
  qed
next
  assume *: "union_closed (op ` f ` F)"
  show "union_closed F"
    unfolding union_closed_def
  proof (safe)
    fix A B
    assume "A \<in> F" "B \<in> F"
    hence "f ` A \<union> f ` B \<in> op ` f ` F"
      using *
      unfolding union_closed_def
      by auto
    then obtain X where "f ` X = f ` A \<union> f ` B" "X \<in> F"
      by auto
    hence "f ` X = f ` (A \<union> B)"
      by auto
    hence "X = A \<union> B"
      using assms `X \<in> F` `A \<in> F` `B \<in> F`
      using inj_on_iso_strong[of f F]
      unfolding inj_on_def
      by blast
    thus "A \<union> B \<in> F"
      using `X \<in> F`
      by simp
  qed
qed

lemma finite_union_closed_iso:
  assumes "inj_on f (\<Union> F)"
  shows "finite_union_closed F \<longleftrightarrow> finite_union_closed (op ` f ` F)"
  using assms
  using finite_Union_iso[of f F] union_closed_iso[of f F]
  by (simp add: finiteUn_iff)

lemma count_iso:
  assumes "inj_on f (\<Union> F)" and "a \<in> \<Union> F"
  shows "count a F = count (f a) (op ` f ` F)"
using assms
unfolding count_def
proof (subst card_image[THEN sym])
  let ?f = "op ` f"
  let ?aF = "{S \<in> F. a \<in> S}" 
  let ?fafF = "{S \<in> op ` f ` F. f a \<in> S}"
  show "inj_on ?f ?aF"
    using assms 
    using inj_on_iso[of f F]
    using subset_inj_on[of "op ` f" F ?aF]
    by auto

  show "card (?f ` ?aF) = card ?fafF"
  proof-
    have "?f ` ?aF = ?fafF"
    proof (auto)
      fix A a'
      assume "A \<in> F" "a' \<in> A" "f a = f a'"
      hence "a = a'" using assms
        unfolding inj_on_def
        by auto
      thus "f ` A \<in> ?f ` ?aF"
        using rev_image_eqI[of A ?aF "f ` A" "op ` f"] `A \<in> F` `a' \<in> A`
        by simp
    qed
    thus ?thesis
      by simp
  qed
qed

lemma Frankl_iso:
  assumes "inj_on f (\<Union> F)"
  shows "frankl F \<longleftrightarrow> frankl (op ` f ` F)"
proof-
  let ?F = "op ` f ` F"
  have "(\<exists>x. x \<in> \<Union>F \<and> card F \<le> 2 * count x F) \<longleftrightarrow> (\<exists>x. x \<in> \<Union>?F \<and> card ?F \<le> 2 * count x ?F)"
  proof (safe)
    fix X x
    assume *: "X \<in> F" "x \<in> X" "card F \<le> 2 * count x F"
    show "\<exists>x. x \<in> \<Union>op ` f ` F \<and> card (op ` f ` F) \<le> 2 * count x (op ` f ` F)"
      apply (rule_tac x="f x" in exI)
      using * `inj_on f (\<Union> F)`
      by (subst count_iso[THEN sym]) (auto simp add: card_iso)
  next
    fix X x
    assume *: "card ?F \<le> 2 * count (f x) ?F" "X \<in> F" "x \<in> X"
    show "\<exists>x. x \<in> \<Union>F \<and> card F \<le> 2 * count x F"
      apply (rule_tac x="x" in exI)
      using *  `inj_on f (\<Union> F)`
      by (subst count_iso) (auto simp add: card_iso)
  qed
  thus ?thesis
    unfolding frankl_def
    using `inj_on f (\<Union> F)`
    using empty_iso[of F f] finite_Union_iso[of f F]
    by blast
qed

(* ************************************************************************** *)
subsection{* Weight functions *}
(* ************************************************************************** *)

text{* 
  A technique that is used for analyzing Frankl's conjecture is based on the
concept of weights. 
*}

definition weight_fun :: "('a \<Rightarrow> nat) \<Rightarrow> 'a set \<Rightarrow> bool" where
  "weight_fun w X \<equiv> (\<exists> a \<in> X. w a > 0)"

definition set_weight :: "('a \<Rightarrow> nat) \<Rightarrow> 'a set \<Rightarrow> nat" (infixl "\<rhd>" 100) where 
  "w \<rhd> S = (\<Sum>x\<in>S. w x)"

definition Family_weight :: "('a \<Rightarrow> nat) \<Rightarrow>'a set set \<Rightarrow>  nat" (infixl "\<triangleright>" 100) where 
  "w \<triangleright> F = (\<Sum>S\<in>F. w \<rhd> S)"

text{* Frankl's characterization using weight *}
theorem Frankl_weight:
  assumes "F \<noteq> {}" and "finite (\<Union> F)"
  shows "frankl F \<longleftrightarrow> (\<exists> w. weight_fun w (\<Union> F) \<and> 2 * (w \<triangleright> F) \<ge> (w \<rhd> (\<Union> F)) * card F)" (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?lhs"
  then obtain a where "a \<in> \<Union> F" "finite (\<Union> F)"
    "2 * count a F \<ge> card F"
    unfolding frankl_def
    by auto

  let ?w = "\<lambda> x. if x = a then (1::nat) else 0"
  have "card (\<Union>F \<inter> {a}) = 1"
    using `a \<in> \<Union> F`
    by auto
  hence "?w \<rhd> (\<Union> F) = 1"
    using `a \<in> \<Union> F` `finite (\<Union> F)`
    unfolding set_weight_def
    by (auto simp add: setsum_cases)

  have "\<forall> S \<in> F. ?w \<rhd> S = (if a \<in> S then 1 else 0)"
    using `finite (\<Union> F)`
    unfolding set_weight_def
    by (auto simp add: setsum_cases finiteUn_iff)
  hence "?w \<triangleright> F = count a F"
    unfolding count_def Family_weight_def
    using `finite (\<Union> F)`
    by (auto simp add: setsum_cases Collect_conj_eq finiteUn_iff)
  hence "2 * (?w \<triangleright> F) \<ge> card(F)"
    using `2 * count a F \<ge> card F`
    by auto
  hence "2 * (?w \<triangleright> F) \<ge> (?w \<rhd> (\<Union> F)) * card F"
    by (subst `?w \<rhd> (\<Union> F) = 1`) simp

  moreover
  have "weight_fun ?w (\<Union> F)"
    using `a \<in> \<Union> F`
    unfolding weight_fun_def
    by auto
  ultimately
  show ?rhs
    by blast
next
  assume "?rhs"
  then obtain w where
    "weight_fun w (\<Union> F)" "2 * (w \<triangleright> F) \<ge> (w \<rhd> (\<Union> F)) * card F"
    by auto
  show ?lhs
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence *: "\<forall> a. a \<in> \<Union> F \<longrightarrow> 2* (count a F) < card F"
      using assms
      unfolding frankl_def
      by auto
    
    obtain a0 where "a0 \<in> \<Union> F" "w a0 > 0" "\<forall>a\<in>\<Union>F. 0 \<le> w a"
      using `weight_fun w (\<Union> F)`
      unfolding weight_fun_def
      by auto

    have "w \<triangleright> F = (\<Sum>S\<in>F. w \<rhd> S)"
      unfolding Family_weight_def
      by simp
    also have "... = (\<Sum>S\<in>F. \<Sum>a\<in>S. w a)"
      unfolding set_weight_def
      by simp
    also have "... = (\<Sum>a\<in>\<Union> F. (w a)* (\<Sum>_\<in>{S \<in> F. a \<in> S}. 1))"
    proof-
      let ?S = "\<lambda> a. {S \<in> F. a \<in> S} \<times> {a}"
      let ?S' = "\<lambda> x. {(S, a). S \<in> F \<and> a \<in> S}"
      let ?CS = "?S ` (\<Union> F)"

      have "(\<Sum>S\<in>F. \<Sum>a\<in>S. w a) = (\<Sum>(S, a)\<in>(SIGMA S:F. S). w a)" 
        using assms
        by (simp add: setsum_Sigma finiteUn_iff)
      also have "... = (\<Sum>(S, a) \<in> ?S' a. w a)"
        using SetCompr_Sigma_eq[of F id, THEN sym]
        unfolding Collect_def mem_def
        by auto
      also have "... = (\<Sum>(S, a)\<in>\<Union>?CS. w a)"
        by (rule setsum_cong) auto
      also have "... = (\<Sum>A\<in>?CS. (\<Sum>x\<in>A. w(snd x)))"
        using assms
        by (subst setsum_Union_disjoint) (auto simp add: split_def finiteUn_iff)
      also have "... = (\<Sum>a\<in>\<Union> F. (\<Sum>x\<in>?S a. w (snd x)))"
      proof (subst setsum_reindex)
        show "setsum (setsum (\<lambda>x. w (snd x)) \<circ> ?S) (\<Union>F) = (\<Sum>a\<in>\<Union> F. setsum (\<lambda> x. w (snd x)) (?S a))"
        proof (rule setsum_cong)
          fix a
          assume "a \<in> \<Union> F"
          have "setsum (\<lambda> x. w (snd x)) (?S a) = setsum (\<lambda> x. (w a)) (?S a)"
            by (rule setsum_cong) auto
          thus "(setsum (\<lambda>x. w (snd x)) \<circ> ?S) a = (\<Sum>x\<in>{S \<in> F. a \<in> S} \<times> {a}. w (snd x))"
            using assms `a \<in> \<Union> F`
            by auto
        qed simp
      next
        show "inj_on ?S (\<Union> F)"
          unfolding inj_on_def
          by auto
      qed
      also have "... = (\<Sum>a\<in>\<Union>F. w a * (\<Sum>x\<in>{S \<in> F. a \<in> S}. 1))"
      proof (rule setsum_cong)
        fix a
        assume "a \<in> \<Union> F"
        have "setsum (\<lambda> x. w (snd x)) (?S a) = setsum (\<lambda> x. (w a)) (?S a)"
          by (rule setsum_cong) auto
        thus "(\<Sum>x\<in>{S \<in> F. a \<in> S} \<times> {a}. w (snd x)) = w a * (\<Sum>x\<in>{S \<in> F. a \<in> S}. 1)"
          unfolding count_def
          using assms `a \<in> \<Union> F`
          by (auto simp add: finiteUn_iff)
      qed simp
      finally show ?thesis
        .
    qed
    also have "... = (\<Sum>a\<in>\<Union> F. (w a)* (count a F))"
      unfolding count_def
      by simp
    finally
    have "2 * (w \<triangleright> F) = 2 * (\<Sum>a\<in>\<Union> F. (w a)* (count a F))"
      by simp
    
    have "(\<Sum>a\<in>\<Union> F. 2 * (w a)* (count a F)) < (\<Sum>a\<in>\<Union> F. (w a) * card F)"
    proof (rule setsum_mono_single_lt_nat)
      show "finite (\<Union>F)" using assms by simp
    next
      fix a
      assume "a \<in> \<Union>F" 
      thus "2 * w a * (count a F) \<le> w a * (card F)"
        using *
        by auto
    next
      show "a0 \<in> \<Union>F" using `a0 \<in> \<Union>F`  .
    next
      show "2 * w a0 * (count a0 F) < w a0 * (card F)"
        using * `a0 \<in> \<Union>F` `w a0 > 0`
        by auto
    qed
    also have "... = (w \<rhd> (\<Union> F)) * card F"
      unfolding set_weight_def
      by (auto simp add: setsum_left_distrib)
    finally have "2 * (\<Sum>a\<in>\<Union> F. (w a)* (count a F)) < (w \<rhd> (\<Union>F)) * (card F)"
      by (auto simp add: setsum_right_distrib mult_assoc)

    show False
      using `2 * (\<Sum>a\<in>\<Union> F. (w a)* (count a F)) < (w \<rhd> (\<Union>F)) * (card F)`
      using `2 * w \<triangleright> F = 2 * (\<Sum>a\<in>\<Union> F. (w a)* (count a F))`
      using `2 * w \<triangleright> F \<ge> (w \<rhd> (\<Union> F)) * (card F)`
      by simp
  qed
qed

lemma set_weight_weights_equal:
  assumes "\<forall> v \<in> S. w v = w' v"
  shows "w \<rhd> S = w' \<rhd> S"
using assms
unfolding set_weight_def
by (subst setsum_cong) auto

lemma set_weight_w_0_on_K:
  assumes "finite K" and "finite S"
  assumes "K \<inter> S = {}" and "\<forall> x \<in> K. w x = 0" and "S' \<subseteq> S"
  shows "w \<rhd> (K \<union> S') = w \<rhd> S'"
  unfolding set_weight_def
  using assms
  using setsum_Un_disjoint[of K "S'" w]
  by (auto simp add: finite_subset)

(* ************************************************************************** *)
subsection{* Share *}
(* ************************************************************************** *)

definition set_share :: "'a set \<Rightarrow> ('a \<Rightarrow> nat) \<Rightarrow> 'a set \<Rightarrow> int" ("\<Join>") where 
  "\<Join> S w X \<equiv> 2 * int (w \<rhd> S) - int (w \<rhd> X)"

definition Family_share :: "'a set set \<Rightarrow> ('a \<Rightarrow> nat) \<Rightarrow> 'a set \<Rightarrow> int" ("\<bowtie>")  where 
  "\<bowtie> F w X \<equiv> (\<Sum>S \<in> F. \<Join> S w X)"

lemma Pow_Family_share_zero:
  assumes "finite (\<Union> F)"
  shows "\<bowtie> (Pow (\<Union> F)) w (\<Union> F) = 0"
proof-
  let ?X = "\<Union> F"
  let ?P = "Pow ?X"
  have "\<forall> S \<in> ?P. \<Join> S w ?X + \<Join> (?X - S) w ?X = 0"
  proof
    fix S
    assume "S \<in> ?P"
    hence "finite S" "finite (?X - S)" "S \<union> \<Union> F = \<Union> F"
      using assms
      by (auto simp add: finite_subset)
    thus "\<Join> S w ?X + \<Join> (?X - S) w ?X = 0"
      unfolding set_share_def set_weight_def
      using setsum_Un_disjoint[of "S" "?X - S" w]
      by (auto simp add: field_simps)
  qed
  moreover
  have "(\<Sum>S\<in>?P. \<Join> S w ?X) + (\<Sum>S\<in>?P. \<Join> (?X - S) w ?X) = 
            (\<Sum>S\<in>?P. \<Join> S w ?X + \<Join> (?X - S) w ?X)"
    using `finite (\<Union> F)`
    by (auto simp add: setsum.distrib)
  ultimately
  have "(\<Sum>S\<in>?P. \<Join> S w ?X) + (\<Sum>S\<in>?P. \<Join> (?X - S) w ?X) = 0"
    by auto
  moreover
  have "(\<Sum>S\<in>?P. \<Join> (?X - S) w ?X) = (\<Sum>S\<in>Pow ?X. \<Join> S w ?X)"
  proof-
    have "inj_on (op - (\<Union>F)) ?P"
      unfolding inj_on_def
      by auto
    moreover
    have "op - (\<Union>F) ` ?P = ?P"
      by auto
    ultimately
    show ?thesis
      using setsum_reindex[of "\<lambda> S. ?X - S" "Pow ?X" "\<lambda> S. \<Join> S w ?X", THEN sym]
      by auto
  qed
  hence "(\<Sum>S\<in>?P. \<Join> S w ?X) + (\<Sum>S\<in>?P. \<Join> (\<Union>F - S) w ?X)  = 2 * (\<Sum>S\<in>?P. \<Join> S w ?X)"
    by auto
  ultimately
  show ?thesis
    unfolding Family_share_def
    by simp
qed

lemma Family_share_lemma:
  shows "\<bowtie> F w X = int (2 * w \<triangleright> F) - int (w \<rhd> X * (card F))"
unfolding Family_share_def set_share_def Family_weight_def
unfolding diff_int_def
by (subst comm_monoid_add_class.setsum_addf, subst setsum_right_distrib[THEN sym]) (auto simp add: int_mult int_setsum)

text{* Frankl's characterization using shares *}
theorem Frankl_Family_share_ge_0:
  assumes "F \<noteq> {}" and "finite (\<Union> F)"
  shows "frankl F \<longleftrightarrow> (\<exists> w. weight_fun w (\<Union> F) \<and> \<bowtie> F w (\<Union> F) \<ge> 0)"
using assms
by (subst Frankl_weight) (auto simp add: Family_share_lemma)

lemma Family_share_weights_equal_on_Union:
  assumes "\<forall> v \<in> X. w v = w' v" "\<Union> F \<subseteq> X"
  shows "\<bowtie> F w X = \<bowtie> F w' X"
proof-
  have "w \<rhd> X = w' \<rhd> X"
    using set_weight_weights_equal[of X w w']
    using assms
    by auto
  moreover 
  {
    fix S
    assume "S \<in> F"
    hence "w \<rhd> S =  w' \<rhd> S"
      using set_weight_weights_equal[of S w w']
      using assms
      by auto
  }
  ultimately
  show ?thesis
    unfolding Family_share_def set_share_def
    by simp
qed

(* ************************************************************************** *)
subsection{* Hypercube construction *}
(* ************************************************************************** *)
definition Hypercube where 
  "Hypercube K S \<equiv> {L. (K \<subseteq> L) \<and> L \<subseteq> (K \<union> S)}"

lemma Hypercube_Pow: "Hypercube K S = (op \<union> K) ` Pow S" (is "?H = ?KP")
proof
  show "?H \<subseteq> ?KP"
  proof
    fix M
    assume "M \<in> Hypercube K S"
    then obtain L where "L \<subseteq> S" "M = K \<union> L"
      unfolding Hypercube_def
      by auto
    thus "M \<in> ?KP" 
      by auto
  qed
  show "?KP \<subseteq> ?H"
    unfolding Hypercube_def
    by auto
qed

lemma hypercubes_inter:
  assumes "K1 \<noteq> K2" and "K1 \<inter> S = {}" and "K2 \<inter> S = {}"
  shows "Hypercube K1 S \<inter> Hypercube K2 S = {}"
using assms
unfolding Hypercube_def
by auto

lemma hypercube_union:
  shows "\<Union> ((\<lambda> K. Hypercube K S) ` (Pow K)) = Pow (K \<union> S)" (is "?lhs = ?rhs")
proof
  show "?lhs \<subseteq> ?rhs"
    unfolding Hypercube_def
    by auto
next
  show "?rhs \<subseteq> ?lhs"
  proof
    fix X
    assume "X \<in> ?rhs"
    show "X \<in> ?lhs"
    proof-
      let ?Kx = "X \<inter> K"
      let ?Cx = "Hypercube ?Kx S"
      have "?Kx \<in> Pow K"
        by auto
      moreover
      have "X \<in> ?Cx"
        using `X \<in> ?rhs`
        unfolding Hypercube_def
        by auto
      ultimately
      show ?thesis
        by blast
    qed
  qed
qed

definition Hyper_share where 
  "Hyper_share K S F w X \<equiv> (\<Sum>L \<in> Hypercube K S \<inter> F. \<Join> L w X)"

lemma Family_share_Hyper_share:
  assumes "finite (\<Union> F)"
  assumes "K' \<union> S = (\<Union> F)" and "K' \<inter> S = {}"
  shows "\<bowtie> F w (\<Union> F) = (\<Sum>K \<in> Pow K'. Hyper_share K S F w (\<Union> F))"
proof-
  let ?w = "\<lambda> S. \<Join> S w (\<Union> F)"
  let ?H = "\<lambda>K. Hypercube K S \<inter> F"
  let ?K = "Pow K'"
  let ?C = "?H ` ?K"

  have "\<Union> ?C = F"
    using hypercube_union[of S K'] assms
    by auto
  moreover
  have "(\<Sum>K \<in> ?K. Hyper_share K S F w (\<Union> F)) = (\<Sum>L \<in> \<Union> ?C. ?w L)"
  proof-
    have "(\<Sum>L \<in> \<Union> ?C. ?w L) = setsum (setsum ?w) ?C"
    proof (subst setsum_Union_disjoint)
      show "\<forall> L \<in> ?C. finite L"
        using `finite (\<Union> F)`
        by (simp add: finiteUn_iff)
    next
      show "\<forall>A\<in>?C. \<forall>B\<in>?C. A \<noteq> B \<longrightarrow> A \<inter> B = {}"
      proof
        fix A assume "A \<in> ?C"
        then obtain Ka where *: "Ka \<in> ?K" "?H Ka = A"
          by auto
        
        show "\<forall>B\<in>?C. A \<noteq> B \<longrightarrow> A \<inter> B = {}"
        proof
          fix B assume  "B \<in> ?C"
          then obtain Kb where **: "Kb \<in> ?K" "?H Kb = B"
            by auto
          show "A \<noteq> B \<longrightarrow> A \<inter> B = {}"
          proof
            assume "A \<noteq> B"
            hence "Ka \<noteq> Kb"
              using * **
              by auto
            moreover
            have "Ka \<inter> S = {}" "Kb \<inter> S = {}"
              using `Ka \<in> ?K` `Kb \<in> ?K`
              using assms
              by auto
            ultimately
            have "Hypercube Ka S \<inter> Hypercube Kb S = {}"
              using hypercubes_inter[of Ka Kb S]
              by simp
            thus "A \<inter> B = {}"
              using * **
              by auto
          qed
        qed
      qed
    qed simp
    also have "... = (\<Sum>K\<in>?K. setsum ?w (?H K))"
    proof (subst setsum_reindex_nonzero)
      show "finite ?K"
        using assms
        using finite_subset[of K' "\<Union> F"]
        by auto
    next
      fix Kx Ky
      assume "Kx \<in> ?K" "Ky \<in> ?K" "Kx \<noteq> Ky" "?H Kx = ?H Ky"
      hence "Hypercube Kx S \<inter> Hypercube Ky S = {}"
        using hypercubes_inter[of Kx Ky S] assms
        by force
      with `?H Kx = ?H Ky`
      have "?H Kx = {}"
        by auto
      thus "setsum ?w (?H Kx) = 0"
        by auto
    qed simp
    finally show ?thesis
      unfolding Hyper_share_def
      by simp
  qed
  ultimately
  show ?thesis
    unfolding Family_share_def Hyper_share_def
    by simp
qed

lemma Frankl_All_Hyper_share_ge_0:
  assumes "F \<noteq> {}" and "finite (\<Union> F)"
  assumes "weight_fun w (\<Union>F)"
  assumes "K' \<union> S = \<Union> F" and "K' \<inter> S = {}"
  assumes "\<forall> K \<in> Pow K'. Hyper_share K S F w (\<Union> F) \<ge> 0"
  shows "frankl F"
using assms
by (auto simp add: Frankl_Family_share_ge_0 Family_share_Hyper_share setsum_nonneg)

subsubsection{* HyperCube projection *}

abbreviation HyperCube_prj where
 "HyperCube_prj K S F \<equiv> (\<lambda>S. S - K) ` (F \<inter> op \<union> K ` Pow S)"

lemma union_closed_HyperCube_prj:
  assumes "union_closed F"
  shows "union_closed (HyperCube_prj K S F)"
unfolding union_closed_def
proof ((rule allI)+, rule impI, erule conjE)
  fix A B
  let ?F = "HyperCube_prj K S F"
  let ?HC = "op \<union> K ` Pow S"
  assume "A \<in> ?F" "B \<in> ?F"
  then obtain a b where 
    *: "a \<in> F" "a \<in> ?HC" "A = a - K" "A \<in> Pow S"
       "b \<in> F" "b \<in> ?HC" "B = b - K" "B \<in> Pow S"
    by auto

  show "A \<union> B \<in> ?F"
  proof (rule rev_image_eqI[of "a \<union> b"])
    show   "a \<union> b \<in> F \<inter> ?HC"
    proof
      show "a \<union> b \<in> F"
        using `a \<in> F` `b \<in> F` `union_closed F`
        unfolding union_closed_def
        by auto
    next
      show "a \<union> b \<in> ?HC"
        apply (rule rev_image_eqI[of "A \<union> B"])
        using *
        by auto
    qed
  next
    show "A \<union> B = a \<union> b - K"
      using *
      by auto
  qed
qed

lemma union_closed_additional_HyperCube_prj:
  assumes "union_closed F" and "I \<subseteq> F" and "S = \<Union> I" and "K \<inter> S = {}" 
  shows "union_closed_additional (HyperCube_prj K S F) I"
proof (rule conjI)
  let ?F = "HyperCube_prj K S F"
  show "union_closed ?F"
    using assms by (simp add: union_closed_HyperCube_prj)

  show "\<forall> A' \<in> ?F. (op \<union> A') ` I \<subseteq> ?F"
  proof (rule ballI, rule subsetI)
    fix A' x
    assume "A' \<in> ?F" "x \<in> op \<union> A' ` I"
    then obtain y where "x = y \<union> A'" "y \<in> I"
      by auto
    hence "x \<subseteq> S"
      using `S = \<Union> I` `A' \<in> ?F`
      by auto
    show "x \<in> ?F"
    proof (rule rev_image_eqI[of "K \<union> x"])
      show "x = K \<union> x - K"
        using `K \<inter> S = {}` `x \<subseteq> S`
        by auto
    next
      show "K \<union> x \<in> F \<inter> op \<union> K ` Pow S"
      proof
        show "K \<union> x \<in> op \<union> K ` Pow S"
          using `x \<subseteq> S`
          by auto
      next
        have "y \<in> F"
          using `y \<in> I` `I \<subseteq> F`
          by auto
        moreover
        have "K \<union> A' \<in> F"
          using `A' \<in> ?F`
          by auto
        ultimately
        have "K \<union> A' \<union> y \<in> F"
          using assms
          unfolding union_closed_def
          by auto
        thus "K \<union> x \<in> F"
          by (subst `x = y \<union> A'`) (subst Un_commute[of y A'], subst Un_assoc[THEN sym])
      qed
    qed
  qed
qed

lemma Hyper_share_Family_share_prj_w_0_on_K:
  assumes "finite S" and "finite K" and "\<forall> x \<in> K. w x = 0"
  shows "Hyper_share K S F w X = \<bowtie> (HyperCube_prj K S F) w X"
unfolding Hyper_share_def Family_share_def
proof (subst setsum_reindex)
  let ?HCF = "Hypercube K S \<inter> F"
  let ?HCF' = "F \<inter> op \<union> K ` Pow S"
  let ?mK = "\<lambda> S. S - K"
  let ?s = "\<lambda>S. \<Join> S w X"
  show "inj_on ?mK ?HCF'"
    by (auto simp add: inj_on_def)

  show "(\<Sum>L\<in>?HCF. ?s L) = (\<Sum>L\<in>?HCF'. (?s \<circ> ?mK) L)"
  proof (rule setsum_cong)
    show "?HCF = ?HCF'"
      by (auto simp add: Hypercube_Pow)
  next
    fix L
    assume "L \<in> ?HCF'"
    thus "?s L = (?s \<circ> ?mK) L"
    proof (auto simp add: set_share_def)
      fix x
      assume "x \<subseteq> S"
      hence "finite x"
        using `finite S`
        by (auto simp add: finite_subset)
      thus "w \<rhd> (K \<union> x) = w \<rhd> (K \<union> x - K)"
        using set_weight_w_0_on_K[of K "K \<union> x - K" w "K \<union> x - K"]
        using `finite K` `\<forall> x \<in> K. w x = 0`
        by simp
    qed
  qed
qed


(* ************************************************************************** *)
subsection{* Union closed extension *}
(* ************************************************************************** *)

abbreviation union_closed_extension where
  "union_closed_extension F \<equiv> {F'. F' \<subseteq> Pow (\<Union> F) \<and> union_closed_additional F' F}"

syntax
  "_union_closed_extension" :: "'a set set \<Rightarrow> 'a set set set"    ("\<lbrace>_\<rbrace>")
translations 
  "\<lbrace>F\<rbrace>" == "CONST union_closed_extension F"

abbreviation uce_shares_nonneg where
  "uce_shares_nonneg A w \<equiv> \<forall> F' \<in> \<lbrace>A\<rbrace>. \<bowtie> F' w (\<Union> A) \<ge> 0"

theorem Frankl_uce_shares_nonneg:
  assumes "F \<noteq> {}" and "finite_union_closed F" and "weight_fun w (\<Union>F)"
  assumes  "A \<subseteq> F" and "\<forall> x \<in> (\<Union> F) - (\<Union> A). w x = 0"
  assumes "uce_shares_nonneg A w"
  shows "frankl F"
proof (rule Frankl_All_Hyper_share_ge_0)
  show "\<forall> K\<in>Pow (\<Union> F - \<Union> A). Hyper_share K (\<Union> A) F w (\<Union> F) \<ge> 0"
  proof
    fix K
    assume "K \<in> Pow (\<Union>F - \<Union>A)"
    show "0 \<le> Hyper_share K (\<Union> A) F w (\<Union> F)"
    proof (subst Hyper_share_Family_share_prj_w_0_on_K)
      show "finite (\<Union>A)" 
        using `finite_union_closed F` `A \<subseteq> F`
        using finite_subset[of "\<Union> A" "\<Union> F"]
        by auto
      show "finite K" 
        using `K \<in> Pow (\<Union>F - \<Union>A)` `finite (\<Union> A)` `finite_union_closed F`
        by (auto simp add: finite_subset)
      show "\<forall> x \<in> K. w x = 0"
        using `K \<in> Pow (\<Union>F - \<Union>A)`
        using `\<forall>x\<in>\<Union>F - \<Union>A. w x = 0`
        by blast
      show "0 \<le> \<bowtie> (HyperCube_prj K (\<Union>A) F) w (\<Union>F)"
      proof-
        let ?P = "HyperCube_prj K (\<Union>A) F"
        have "0 \<le> \<bowtie> ?P w (\<Union>A)"
        proof-
          have "?P \<subseteq> Pow (\<Union> A)"
            by auto
          moreover
          have "K \<inter> \<Union> A = {}"
            using `K \<in> Pow (\<Union>F - \<Union>A)`
            by auto
          hence "union_closed_additional ?P A"
            using union_closed_additional_HyperCube_prj[of F A "\<Union> A" K]
            using `A \<subseteq> F` `K \<in> Pow (\<Union>F - \<Union>A)` `finite_union_closed F`
            by simp
          ultimately
          show ?thesis
            using `finite (\<Union> A)`
            using `\<forall> F' \<in> union_closed_extension A. \<bowtie> F' w (\<Union> A) \<ge> 0`
            by simp
        qed
        moreover
        have "w \<rhd> (\<Union> A) = w \<rhd> (\<Union> F)"
        proof-
          have "\<Union> F - \<Union> A \<union> \<Union> A = \<Union> F"
            using `A \<subseteq> F` 
            by auto
          thus ?thesis
            using set_weight_w_0_on_K[of "\<Union> F - \<Union> A" "\<Union> A" w "\<Union> A"]
            using `\<forall> x \<in> (\<Union> F) - (\<Union> A). w x = 0` `finite_union_closed F` `finite (\<Union> A)`
            by force
        qed
        ultimately
        show ?thesis
          unfolding Family_share_def set_share_def
          by simp
      qed
    qed
  qed
next
  show "\<Union>F - \<Union>A \<union> \<Union>A = \<Union>F"
    using `A \<subseteq> F`
    by auto
qed (auto simp add: assms)

subsection{* FC Families *}

definition FC_family where
 "FC_family Fc \<equiv> \<forall> F. F \<supseteq> Fc \<and> finite_union_closed F \<longrightarrow> frankl F"

lemma FC_family_iso:
  fixes f :: "'a \<Rightarrow> nat"
  assumes "inj_on f (\<Union> Fc)" "FC_family (op ` f ` Fc)"
  shows "FC_family Fc"
using assms
unfolding FC_family_def
proof (safe)
  fix F
  assume *: "\<forall>F. op ` f ` Fc \<subseteq> F \<and> finite_union_closed F \<longrightarrow> frankl F"
  assume "inj_on f (\<Union>Fc)" "Fc \<subseteq> F" "union_closed F" "finite (\<Union>F)"
  have "\<Union>Fc \<subseteq> \<Union>F" "finite (\<Union> Fc)" "\<Union>Fc \<union> \<Union>F = \<Union> F"
    using `Fc \<subseteq> F` `finite (\<Union> F)` finite_subset[of "\<Union> Fc" "\<Union> F"]
    by auto
  then obtain f' where "\<forall> x \<in> \<Union> Fc. f' x = f x" "inj_on f' (\<Union> F)"
    using bij_betw_inj_extend[of f "\<Union> Fc" "f ` (\<Union> Fc)" "\<Union> F - \<Union> Fc"]
    using `inj_on f (\<Union>Fc)` `finite (\<Union> F)` `Fc \<subseteq> F`
    unfolding bij_betw_def
    by auto
  let ?F' = "op ` f' ` F"
  have "op ` f ` Fc \<subseteq> ?F'"
  proof (safe)
    fix x
    assume "x \<in> Fc"
    show "f ` x \<in> op ` f' ` F"
    proof (rule rev_image_eqI)
      show "x \<in> F" using `x \<in> Fc` `Fc \<subseteq> F`
        by auto
    next
      show "f ` x = f' ` x"
        using `x \<in> Fc` `\<forall> x \<in> \<Union> Fc. f' x = f x`
        by force
    qed
  qed
  moreover
  have "finite_union_closed ?F'"
    using finite_union_closed_iso
    using `inj_on f' (\<Union> F)` `finite (\<Union> F)` `union_closed F`
    by auto
  ultimately
  have "frankl ?F'"
    using *
    by auto
  thus "frankl F"
    using `inj_on f' (\<Union> F)` Frankl_iso
    by auto
qed

theorem FC_family_uce_shares_nonneg:
  assumes "weight_fun w (\<Union> Fc)" "uce_shares_nonneg Fc w"
  shows "FC_family Fc"
using assms
unfolding FC_family_def
proof (safe)
  fix F
  assume "uce_shares_nonneg Fc w" "Fc \<subseteq> F" "union_closed F" "finite (\<Union> F)"
  let ?w'  = "\<lambda> a. if a \<in> \<Union> Fc then w a else 0"
  show "frankl F"
  proof (rule Frankl_uce_shares_nonneg)
    have "\<Union> Fc \<subseteq> \<Union> F"
      using `Fc \<subseteq> F`
      by auto
    show "weight_fun ?w' (\<Union> F)"
      using `weight_fun w (\<Union> Fc)` `Fc \<subseteq> F`
      unfolding weight_fun_def
      by auto 
  next
    show "\<forall>x\<in>\<Union>F - \<Union>Fc. ?w' x = 0"
      by auto
  next
    show "uce_shares_nonneg Fc ?w'"
    proof
      fix F'
      assume "F' \<in> \<lbrace>Fc\<rbrace>"
      hence "\<Union> F' \<subseteq> \<Union> Fc"
        by auto
      hence "\<bowtie> F' ?w' (\<Union>Fc) = \<bowtie> F' w (\<Union>Fc)"
        using Family_share_weights_equal_on_Union[of "\<Union> Fc" w ?w' F']
        by auto
      thus "0 \<le> \<bowtie> F' ?w' (\<Union>Fc)"
        using `uce_shares_nonneg Fc w` `F' \<in> \<lbrace>Fc\<rbrace>`
        by simp
    qed
  next
    show "Fc \<subseteq> F"
      by fact
  next
    have "Fc \<noteq> {}"
      using `weight_fun w (\<Union> Fc)`
      unfolding weight_fun_def
      by auto
    thus "F \<noteq> {}"
      using `Fc \<subseteq> F`
      by auto
  next
    show "finite_union_closed F"
      using `union_closed F` `finite (\<Union> F)`
      by simp
  qed
qed

end
