header{* Non equivalent uniform families of sets *}

theory NonEquivalentFamilies
imports Main MoreSet MoreList Combinatorics
  "~~/src/HOL/Library/Efficient_Nat"
begin

(* TODO: Move to Permutation.thy *)
lemma perm_sym:  "A <~~> B \<longleftrightarrow> B <~~> A"
unfolding multiset_of_eq_perm[THEN sym]
by auto

(* TODO: Move to MoreList.thy *)
lemma length_filter_less [simp]: "length [l\<leftarrow>list . P l] < Suc (length list)"
using length_filter_le[of P list]
by arith

text{* In this section we define a combinatorial algorithm
(implemented by a function @{text "non_equivalent_families"}) that
computes non equivalent uniform families of sets (a family is called
uniform if all its members contain the same number of elements). Two
uniform families (over the same domain) are called equivalent if one
can be obtained from the other by permuting elements of the
domain. This function is used to remove symmetric cases in
verification of uniform FC families.*}


text{* The implementation uses a canonical (sorted and distinct) list
representation of uniform set families (containing @{text "m"} sets,
each with @{text "k"} elements over a domain of @{text "n"}
elements). The function @{text "f_toSet"} converts the list back to
the original family, and the function @{text "isNKMFamily"} checks
wheter the list representation is valid. *}

definition f_toSet :: "'a list list \<Rightarrow> 'a set set" where 
 "f_toSet Fl = List.set (map List.set Fl)"

definition isNKMFamily where
  "isNKMFamily F n k m \<equiv> 
      length F = m \<and> sorted F \<and> distinct F \<and> 
      (\<forall> A \<in> List.set F. sorted A \<and> distinct A \<and> length A = k \<and> List.set A \<subseteq> {0..<n})"

text{* The function @{text "families"} computes all uniform families
for given parameters. *}
definition families :: "nat \<Rightarrow> nat \<Rightarrow> nat \<Rightarrow> nat list list list " where 
  "families n m k = combine (combine [0..<n] m) k"

lemma families_NKM:
 "F \<in> List.set (families n k m) \<longleftrightarrow> isNKMFamily F n k m"
unfolding families_def
unfolding isNKMFamily_def
using combine_combines[of "combine [0..<n] k" F m]
using sorted_combine[of "[0..<n]" k]
using distinct_combine[of "[0..<n]" k]
using combine_combines[of "[0..<n]" _ k]
by auto

lemma NKM_family_of_set: 
  assumes "k > 0" and "m > 0"
  assumes "\<Union> F \<subseteq> {0::nat..<n}" and "card F = m" and "\<forall> A \<in> F. card A = k"
  shows "\<exists> Fl. isNKMFamily Fl n k m \<and> f_toSet Fl = F "
proof-
  let ?sF = "sorted_list_of_set ` F"
  let ?Fl = "sorted_list_of_set ?sF"
  have "finite F" "finite ?sF" 
    using `m > 0` `card F = m`
    by (auto simp add: card_ge_0_finite)

  show ?thesis
  proof (rule_tac x="?Fl" in exI, rule conjI)
    have "sorted ?Fl" "distinct ?Fl" "List.set ?Fl = ?sF"
      using sorted_list_of_set[of ?sF]
      using `finite ?sF`
      by auto
    moreover
    have "length ?Fl = m"
    proof-
      have "inj_on sorted_list_of_set F"
        unfolding inj_on_def
        using `\<forall> A \<in> F. card A = k` `k > 0`
        by (auto simp add: sorted_list_of_set_inj card_ge_0_finite)
      thus ?thesis
        using `finite ?sF` `finite F` `card F = m`
        using length_sorted_list_of_set[of "sorted_list_of_set ` F"]
        using inj_on_iff_eq_card[of F sorted_list_of_set]
        by auto
    qed
    moreover
    have "\<forall> A \<in> F. sorted (sorted_list_of_set A) \<and> distinct (sorted_list_of_set A) \<and> List.set (sorted_list_of_set A) = A \<and> length (sorted_list_of_set A) = k"
    proof
      fix A
      assume "A \<in> F"
      hence "finite A" "card A = k"
        using `\<forall> A \<in> F. card A = k` `k > 0`
        by (simp_all add: card_ge_0_finite)
      let ?A = "sorted_list_of_set A"
      show "sorted ?A \<and> distinct ?A \<and> List.set ?A = A \<and> length ?A = k"
        using sorted_list_of_set[of A] `finite A` length_sorted_list_of_set[of A] `card A = k`
        by auto
    qed
    ultimately
    show "isNKMFamily ?Fl n k m" "f_toSet ?Fl = F"
      unfolding isNKMFamily_def
      using `\<Union> F \<subseteq> {0::nat..<n}`
      unfolding f_toSet_def
      by auto
  qed
qed

lemma canonical_family_list:
assumes
  "card l = m" and "m > 0" and "k > 0" and
  "\<forall> A \<in> l. card A = k" and
  "\<Union> l \<subseteq> {0..<n}"
shows "\<exists> F \<in> set (families n k m).  isNKMFamily F n k m \<and> f_toSet F = l"
proof-
  from assms
  obtain Fl where "isNKMFamily Fl n k m \<and> f_toSet Fl = l"
    using NKM_family_of_set[of k m l n]
    by auto
  thus ?thesis
    using families_NKM[of Fl n k m]
    by auto
qed

(* ************************************************************************** *)
text{* There is a bijective mapping between equivalent uniform families. *}
abbreviation inj_embed where
  "inj_embed F F' \<equiv> \<exists> f. inj_on f (\<Union> (f_toSet F)) \<and> f_toSet F' = (op ` f) ` f_toSet F"

lemma [simp]: "inj_embed F F"
by (rule_tac x="id" in exI) auto

lemma inj_embed_bij_betw:
  assumes "inj_embed F F'"
  shows "\<exists> h. bij_betw h (\<Union> (f_toSet F)) (\<Union> (f_toSet F')) \<and> 
         f_toSet F' = (op ` h) ` f_toSet F"
proof-
  obtain f where *: "inj_on f (\<Union> (f_toSet F))"  "f_toSet F' = (op ` f) ` f_toSet F"
    using assms
    by auto
  show ?thesis
    unfolding bij_betw_def
    by (rule_tac x="f" in exI, rule conjI) (auto simp add: *)
qed

(* ************************************************************************** *)
text{* The function @{text "permute_family"} applies the given permutation 
to a given family, retaining the valid (sorted) list representation. *}

definition permute_family :: "nat list list \<Rightarrow> nat list \<Rightarrow> nat list list" where
 "permute_family F p = sort (map (\<lambda> l. sort (map (\<lambda> x. p ! x) l)) F)"

lemma set_permute_family:
  assumes "p <~~> [0..<n]" and "\<Union> f_toSet F \<subseteq> {0..<n}"
  shows "\<Union> f_toSet (permute_family F p) \<subseteq> {0..<n}"
  unfolding f_toSet_def permute_family_def
proof (auto)
  fix x a
  assume "a \<in> set F" "x \<in> set a"
  hence "x < n"
    using `\<Union> f_toSet F \<subseteq> {0..<n}`
    unfolding f_toSet_def
    by auto
  thus "p ! x < n"
    using `p <~~> [0..<n]`
    using perm_length[of p "[0..<n]"]
    using perm_set_eq[of p "[0..<n]"]
    by auto
qed

lemma permute_family_inj_embed:
  assumes "p <~~> [0..<n]" and "\<Union> (f_toSet Fl) \<subseteq> {0..<n}"
  shows "inj_embed (permute_family Fl p) Fl"
proof-
  obtain f where "bij_betw f {..<n} {..<n}"
    "\<forall>i<n. p ! f i = i"
    using `p <~~> [0..<n]`
    using permutation_Ex_bij[of "[0..<n]" p]
    using BIJ[of "{..<n}" "{..<length p}"]
    using perm_sym[of "[0..<n]" p]
    by auto

  show ?thesis
  proof (rule_tac x="f" in exI, rule conjI)
    show "inj_on f (\<Union> (f_toSet (permute_family Fl p)))"
      using `bij_betw f {..<n} {..<n}`
      unfolding bij_betw_def
      using set_permute_family[of p n Fl] 
      using `p <~~> [0..<n]` `\<Union>f_toSet Fl \<subseteq> {0..<n}`
      using subset_inj_on[of f "{0..<n}" "\<Union> f_toSet (permute_family Fl p) "]
      by (auto simp add: atLeast0LessThan)
  next
    show "f_toSet Fl = op ` f ` f_toSet (permute_family Fl p)"
      unfolding f_toSet_def permute_family_def
    proof (auto)
      fix l
      assume "l \<in> set Fl"

      have "\<forall> x \<in> set l. f (p ! x) = x"
      proof
        fix x
        assume "x \<in> set l"
        hence "x < n"
          using `\<Union> (f_toSet Fl) \<subseteq> {0..<n}` `l \<in> set Fl`
          unfolding f_toSet_def
          by auto
        show "f (p ! x) = x"
        proof-
          from `x < n` `bij_betw f {..<n} {..<n}`
          obtain x' where "x' < n"  "x = f x'"
            unfolding bij_betw_def
            by auto
          thus "f (p ! x) = x"
            using `\<forall> i<n. p ! f i = i` `x < n`
            by simp
        qed
      qed

      show "f ` op ! p ` set l \<in> set ` set Fl"
      proof (rule rev_image_eqI[of l])
        show "l \<in> set Fl"
          using `l \<in> set Fl`
          .
        show "f ` op ! p ` set l = set l"
          using `\<forall> x \<in> set l. f (p ! x) = x`
          by force
      qed

      let ?l = "set (sort (map (op ! p) l))"
      show "set l \<in> op ` f ` set ` (\<lambda>l. sort (map (op ! p) l)) ` set Fl"
      proof (rule rev_image_eqI[of ?l])
        show "?l \<in> set ` (\<lambda> l. sort (map (op ! p) l)) ` set Fl"
          apply (rule rev_image_eqI[of "sort (map (op ! p) l)"])
          using `l \<in> set Fl`
          by auto
      next
        show "set l = f ` set (sort (map (op ! p) l))"
          using `\<forall> x \<in> set l. f (p ! x) = x`
          by force
      qed
    qed
  qed
qed

(* ************************************************************************** *)
function non_equivalent_families_aux where
 "non_equivalent_families_aux fams perms res =
  (case fams of 
    [] \<Rightarrow> res
  | h # t \<Rightarrow> 
    let hp = remdups (h # (map (\<lambda> p. permute_family h p) perms)) in
    non_equivalent_families_aux (filter (\<lambda> l. l \<notin> set hp) fams) perms (h # res))"
by pat_completeness auto
termination
by (relation "measure (\<lambda> (f, p, r). length f)") auto
declare non_equivalent_families_aux.simps [simp del]

definition non_equivalent_families where
[simp]: "non_equivalent_families fams perms = non_equivalent_families_aux fams perms []"

abbreviation nef where
 "nef n k m \<equiv> non_equivalent_families (families n k m) (permute [0..<n])"

lemma non_equivalent_families_aux_res_mono: 
  "set res \<subseteq> set (non_equivalent_families_aux fams perms res)"
proof (induct fams perms res rule: non_equivalent_families_aux.induct)
  case (1 fams perms res)
  show ?case
  proof (cases fams)
    case Nil
    thus ?thesis
      by (simp add: non_equivalent_families_aux.simps)
  next
    case (Cons a fams')
    thus ?thesis
      using non_equivalent_families_aux.simps[of fams perms res]
      using 1[of a fams' "remdups (a # (map (permute_family a) perms))"]
      by (auto simp add: Let_def)
  qed
qed

lemma non_equivalent_families_aux:
  assumes
  "\<forall> p \<in> set perms. p <~~> [0..<n]" and
  "\<forall> F \<in> set fams. \<Union> (f_toSet F) \<subseteq> {0..<n}" and
  "nefams = non_equivalent_families_aux fams perms res"
  shows "\<forall> F. F \<in> set fams \<longrightarrow> (\<exists> F' \<in> set nefams. inj_embed F F')"
using assms
proof (induct fams perms res arbitrary: nefams rule: non_equivalent_families_aux.induct)
  case (1 fams perms res)
  show ?case
  proof (cases "fams")
    case Nil
    thus ?thesis
      by simp
  next
    case (Cons F fams')

    let ?hp = "remdups (F # (map (\<lambda> p. permute_family F p) perms))"
    let ?filt = "filter (\<lambda> l. l \<notin> List.set ?hp) fams"
    let ?nef = "non_equivalent_families_aux ?filt perms (F # res)"

    from Cons have
      "nefams = ?nef"
      using `nefams = non_equivalent_families_aux fams perms res`
      using non_equivalent_families_aux.simps[of fams perms res]
      by simp

    show ?thesis
    proof (rule allI, rule impI)
      fix Fa
      assume "Fa \<in> set fams"
      hence "Fa = F \<or> Fa \<in> set fams'"
        using Cons
        by simp
      thus "\<exists>a\<in>set nefams. inj_embed Fa a"
      proof
        assume "Fa = F"
        hence "Fa \<in> set nefams"
          using non_equivalent_families_aux_res_mono[of "F # res" ?filt perms]
          using `nefams = ?nef`
          by simp
        thus ?thesis
          by force
      next
        assume "Fa \<in> set fams'"
        show ?thesis
        proof (cases "Fa \<in> set ?hp")
          case True
          have "inj_embed Fa F"
            using `\<forall>F\<in>set fams. \<Union>f_toSet F \<subseteq> {0..<n}`
            using Cons
            using True
            using `\<forall>p\<in>set perms. p <~~> [0..<n]`
            using permute_family_inj_embed
            by (auto split: split_if_asm)
          moreover
          have "F \<in> set nefams"
            using `nefams = ?nef`
            using non_equivalent_families_aux_res_mono[of "F # res" ?filt perms]
            by simp
          ultimately
          show ?thesis
            by auto
        next
          case False
          thus ?thesis
            using Cons `Fa \<in> set fams'`
            using 1(1)[of F fams' ?hp ?nef] 1(2) 1(3) 1(4)
            using `nefams = ?nef`
            by auto
        qed
      qed
    qed
  qed
qed

lemma non_equivalent_families':
  assumes
  "\<forall> p \<in> set perms. p <~~> [0..<n]" and
  "\<forall> F \<in> set fams. \<Union> (f_toSet F) \<subseteq> {0..<n}" and
  "nefams = non_equivalent_families fams perms"
  shows "\<forall> F. F \<in> set fams \<longrightarrow> (\<exists> F' \<in> set nefams. inj_embed F F')"
using assms
unfolding non_equivalent_families_def
using non_equivalent_families_aux
by simp

lemma non_equivalent_families: 
  assumes "set F \<subseteq> set (families n k m)" "set P \<subseteq> set (permute [0..<n])"
  shows "\<forall>l \<in> set F. \<exists>l' \<in> set (non_equivalent_families F P). inj_embed l l'"
proof-
  have "\<forall>p\<in>set P. p <~~> [0..<n]"
    using `set P \<subseteq> set (permute [0..<n])`
    by (auto simp add: isPermutation_permute)
  moreover
  have "\<forall>F\<in>set F. \<Union>f_toSet F \<subseteq> {0..<n}"
    using `set F \<subseteq> set (families n k m)`
    using families_NKM[of _ n k m]
    unfolding isNKMFamily_def
    by (auto simp add: f_toSet_def)
  ultimately
  show ?thesis
    using non_equivalent_families'[of P n F "non_equivalent_families F P"]
    by simp
qed


end
