header{* General chess rules *}
theory ChessRules
imports Main
begin

section{* Chess *}
text{* In this section, we formalize chess rules as given by Hurd et al. *}

datatype side = White | Black
primrec opponent where
  "opponent White = Black"
| "opponent Black = White"

datatype piece = King | Queen | Rook | Bishop | Knight

type_synonym square = "int \<times> int"

(* 
definition files :: int where 
  "files = 8"
definition ranks :: int where
  "ranks = 8"
lemma files_eq_ranks: "files = ranks"
  by (simp add: files_def ranks_def)
lemma files_ranks_geq4: "files \<ge> 4" "ranks \<ge> 4"
  by (simp_all add: files_def ranks_def)
*)

consts files :: int
consts ranks :: int
axiomatization where
 files_eq_ranks: "files = ranks" and
 files_ranks_geq6: "files \<ge> 6" "ranks \<ge> 6"

primrec "file" :: "square \<Rightarrow> int" where
  "file (f, r) = f"
primrec rank :: "square \<Rightarrow> int" where 
  "rank (f, r) = r"

definition board where 
  "board sq \<longleftrightarrow> 0 \<le> file sq \<and> file sq < files \<and> 0 \<le> rank sq \<and> rank sq < ranks"

definition same_file :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_file sq sq' \<longleftrightarrow> file sq = file sq'"
definition same_rank :: "square \<Rightarrow> square \<Rightarrow> bool" where 
  "same_rank sq sq' \<longleftrightarrow> rank sq = rank sq'"
definition same_diag1 :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag1 sq sq' \<longleftrightarrow> file sq + rank sq = file sq' + rank sq'"
definition same_diag2 :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag2 sq sq' \<longleftrightarrow> file sq + rank sq' = file sq' + rank sq"
definition same_diag :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag sq1 sq2 \<longleftrightarrow> same_diag1 sq1 sq2 \<or> same_diag2 sq1 sq2"

definition diff :: "int \<Rightarrow> int \<Rightarrow> int" where 
  "diff m n = (if m \<le> n then n - m else m - n)"
definition file_diff :: "square \<Rightarrow> square \<Rightarrow> int" where
  "file_diff sq sq' = diff (file sq) (file sq')"
definition rank_diff :: "square \<Rightarrow> square \<Rightarrow> int" where
  "rank_diff sq sq' = diff (rank sq) (rank sq')"

text{* Scopes of pieces on empty board *}
definition bishop_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "bishop_scope sq1 sq2 \<longleftrightarrow> same_diag sq1 sq2 \<and> sq1 \<noteq> sq2"
definition rook_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "rook_scope sq1 sq2 \<longleftrightarrow> (same_file sq1 sq2 \<or> same_rank sq1 sq2) \<and> sq1 \<noteq> sq2"
definition queen_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "queen_scope sq1 sq2 \<longleftrightarrow> bishop_scope sq1 sq2 \<or> rook_scope sq1 sq2"
definition knight_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "knight_scope sq1 sq2 \<longleftrightarrow> (file_diff sq1 sq2 = 1 \<and> rank_diff sq1 sq2 = 2) \<or> (file_diff sq1 sq2 = 2 \<and> rank_diff sq1 sq2 = 1)"
definition king_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "king_scope sq1 sq2 \<longleftrightarrow> file_diff sq1 sq2 \<le> 1 \<and> rank_diff sq1 sq2 \<le> 1 \<and> sq1 \<noteq> sq2"

definition between where
  "between n1 n n2 \<longleftrightarrow> (n1 < n \<and> n < n2) \<or> (n2 < n \<and> n < n1)"
definition square_between where
  "square_between sq1 sq sq2 \<longleftrightarrow> 
     (if same_file sq1 sq2 then same_file sq sq1 \<and> between (rank sq1) (rank sq) (rank sq2)
      else if same_rank sq1 sq2 then same_rank sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else if same_diag1 sq1 sq2 then same_diag1 sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else if same_diag2 sq1 sq2 then same_diag2 sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else False)"

locale Position = 
  fixes to_move :: "'p \<Rightarrow> side"
  fixes on_square :: "'p \<Rightarrow> square \<Rightarrow> (side \<times> piece) option"
  fixes invar :: "'p \<Rightarrow> bool"
begin

definition empty :: "'p \<Rightarrow> square \<Rightarrow> bool" where
  "empty p sq \<longleftrightarrow> on_square p sq = None"

definition occupies :: "'p \<Rightarrow> side \<Rightarrow> square \<Rightarrow> bool" where
  "occupies p s sq \<longleftrightarrow> (\<exists> v. on_square p sq = Some (s, v))"

definition clear_line :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "clear_line p sq1 sq2 \<longleftrightarrow> (\<forall> sq. square_between sq1 sq sq2 \<longrightarrow> empty p sq)"

definition attacks :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "attacks p sq sq' \<longleftrightarrow> clear_line p sq sq' \<and>
     (case on_square p sq of
          None \<Rightarrow> False
       | Some (_, King) \<Rightarrow> king_scope sq sq'
       | Some (_, Queen) \<Rightarrow> queen_scope sq sq'
       | Some (_, Rook) \<Rightarrow> rook_scope sq sq'
       | Some (_, Bishop) \<Rightarrow> bishop_scope sq sq'
       | Some (_, Knight) \<Rightarrow> knight_scope sq sq'
     )"

definition in_check :: "side \<Rightarrow> 'p \<Rightarrow> bool" where
  "in_check s p \<longleftrightarrow> 
     (\<exists> sq1 sq2. board sq1 \<and> board sq2 \<and> on_square p sq1 = Some (s, King) \<and> occupies p (opponent s) sq2 \<and> attacks p sq2 sq1)"

definition all_on_board :: "'p \<Rightarrow> bool" where
  "all_on_board p \<longleftrightarrow> (\<forall> sq. \<not> empty p sq \<longrightarrow> board sq)"

definition legal_position :: "'p \<Rightarrow> bool" where
  "legal_position p \<longleftrightarrow> invar p \<and> all_on_board p \<and> \<not> in_check (opponent (to_move p)) p"

definition sorties :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "sorties p sq sq' \<longleftrightarrow> board sq \<and> board sq' \<and> attacks p sq sq' \<and> empty p sq'"

definition captures  :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "captures p sq sq' \<longleftrightarrow> board sq \<and> board sq' \<and> attacks p sq sq' \<and> occupies p (opponent (to_move p)) sq'"

definition simple_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "simple_move p p' \<longleftrightarrow> 
    (\<exists> sq1 sq2. occupies p (to_move p) sq1 \<and> sorties p sq1 sq2 \<and>
       (\<forall> sq. on_square p' sq = 
              (if sq = sq1 then None
               else if sq = sq2 then on_square p sq1
               else on_square p sq)))"

definition capture_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
 "capture_move p p' \<longleftrightarrow> 
  (\<exists> sq1 sq2. occupies p (to_move p) sq1 \<and> captures p sq1 sq2 \<and>
     (\<forall> sq. on_square p' sq = 
            (if sq = sq1 then None
             else if sq = sq2 then on_square p sq1
             else on_square p sq)))"

definition legal_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "legal_move p p' \<longleftrightarrow>
     legal_position p \<and> legal_position p' \<and>
     (to_move p' = opponent (to_move p)) \<and>
     (simple_move p p' \<or> capture_move p p')"

definition game_over :: "'p \<Rightarrow> bool" where
  "game_over p \<longleftrightarrow> legal_position p \<and> \<not> (\<exists> p'. legal_move p p')"
definition checkmated :: "'p \<Rightarrow> bool" where
  "checkmated p \<longleftrightarrow> game_over p \<and> in_check (to_move p) p"
definition stalemate :: "'p \<Rightarrow> bool" where
  "stalemate p \<longleftrightarrow>  game_over p \<and> \<not> in_check (to_move p) p"

inductive_set play for p0 :: "'p" where
  init: "p0 \<in> play p0"
| step: "\<lbrakk>p \<in> play p0; legal_move p p'\<rbrakk> \<Longrightarrow> p' \<in> play p0"

definition draw :: "'p \<Rightarrow> bool" where
  "draw p \<longleftrightarrow> (\<not> (\<exists> p'. p' \<in> play p \<and> checkmated p'))"
end

subsection{* Some properties of chess rules *}

lemma king_scope_sym: 
  "king_scope sq1 sq2 \<longleftrightarrow> king_scope sq2 sq1"
unfolding king_scope_def
by (auto simp add: file_diff_def rank_diff_def diff_def)

definition all_king_pos where
  "all_king_pos P sq \<longleftrightarrow> 
      (let (x, y) = sq in 
         P (x-(1::int), y-(1::int)) \<and>
         P (x-(1::int), y) \<and>
         P (x-(1::int), y+(1::int)) \<and>
         P (x, y-(1::int)) \<and>
         P (x, y+(1::int)) \<and>
         P (x+(1::int), y-(1::int)) \<and>
         P (x+(1::int), y) \<and>
         P (x+(1::int), y+(1::int))
      )"

definition some_king_pos where
  [simp]: "some_king_pos P sq \<longleftrightarrow> \<not> all_king_pos (\<lambda> x. \<not> P x) sq"

lemma some_king_pos:
"some_king_pos P sq \<longleftrightarrow> 
      (let (x, y) = sq in 
         P (x-(1::int), y-(1::int)) \<or>
         P (x-(1::int), y) \<or>
         P (x-(1::int), y+(1::int)) \<or>
         P (x, y-(1::int)) \<or>
         P (x, y+(1::int)) \<or>
         P (x+(1::int), y-(1::int)) \<or>
         P (x+(1::int), y) \<or>
         P (x+(1::int), y+(1::int))
      )"
unfolding some_king_pos_def all_king_pos_def Let_def
by auto

lemma king_scope_iff: 
  "king_scope s1 s2 \<longleftrightarrow> some_king_pos (\<lambda> x. x = s2) s1"
unfolding some_king_pos
apply (cases s1, cases s2, simp)
unfolding king_scope_def file_diff_def rank_diff_def diff_def
by auto

lemma ex_king_scope:
  shows "(\<exists>a b. king_scope (x, y) (a, b) \<and> P (a, b)) \<longleftrightarrow> 
         (P (x - 1, y - 1) \<or> P (x - 1, y) \<or> P (x - 1, y + 1) \<or>
          P (x, y - 1) \<or> P (x, y + 1) \<or>
          P (x + 1, y - 1) \<or> P (x + 1, y) \<or> P (x + 1, y + 1))"
using assms
by (auto simp add: king_scope_iff all_king_pos_def)

lemma [simp]: "\<not> square_between sq sq sq'"
by (simp add: square_between_def between_def)

lemma [simp]: "\<not> square_between sq' sq sq"
by (simp add: square_between_def between_def)

lemma (in Position) king_scope_clear_line:
  assumes "king_scope sq1 sq2"
  shows "clear_line p sq1 sq2"
using assms
unfolding king_scope_def Position.clear_line_def square_between_def between_def
by (cases sq1, cases sq2) (auto simp add: file_diff_def rank_diff_def diff_def)

lemma (in Position) play_trans:
  assumes "p2 \<in> play p1" "p1 \<in> play p0"
  shows "p2 \<in> play p0"
using assms
proof (induct rule: play.induct)
  case (step p p')
  thus ?case
    by (auto simp add: play.step)
qed

subsection{* Strategies for White *}
locale Strategy = Position to_move for to_move :: "'p \<Rightarrow> side" + 
  (* Move of white, according to some strategy *)
  fixes strategy_white_move_rel :: "'p \<Rightarrow> 'p \<Rightarrow> bool"
  (* Invariant that strategy moves must preserve *)
  fixes strategy_invar :: "'p \<Rightarrow> bool"
  (* Strategy must make only legal moves *)
  assumes strategy_white_move_legal_move:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_invar p0; strategy_white_move_rel p0 p1\<rbrakk> \<Longrightarrow> legal_move p0 p1"
  (* Strategy preserves the invariant *)
  assumes strategy_white_move_invar:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_invar p0; strategy_white_move_rel p0 p1\<rbrakk> \<Longrightarrow> strategy_invar p1"
  assumes strategy_white_move_black_move_invar:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_invar p0; strategy_white_move_rel p0 p1; legal_move p1 p2\<rbrakk> \<Longrightarrow> strategy_invar p2"
begin

(* Strategy move (ply): move of white, according to some strategy, or any legal move of black *)
definition strategy_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "strategy_move p p' \<longleftrightarrow> (to_move p = White \<and> strategy_white_move_rel p p') \<or> 
                          (to_move p = Black \<and> legal_move p p')"

(* Play is a sequence of strategy moves by white/legal moves by black. *)
inductive_set strategy_play for p0 :: 'p
where
  init: "p0 \<in> strategy_play p0"
| step: "\<lbrakk>p \<in> strategy_play p0; strategy_move p p'\<rbrakk> \<Longrightarrow> p' \<in> strategy_play p0"

end

text{* Winning strategy *}
locale WinningStrategy = Strategy on_square for on_square :: "'p \<Rightarrow> int \<times> int \<Rightarrow> (side \<times> piece) option" + 
  fixes start :: "'p \<Rightarrow> bool"
  assumes strategy_terminates: "start p0 \<Longrightarrow> \<not> (\<exists> Q. p0 \<in> Q \<and> (\<forall> p \<in> Q. \<exists> p' \<in> Q. strategy_move p p'))"
  assumes white_wins: "\<lbrakk>start p0; p \<in> strategy_play p0; \<not> (\<exists> p'. strategy_move p p')\<rbrakk> \<Longrightarrow> to_move p = Black \<and> checkmated p"

text{* Some properties *}
context Strategy
begin
lemma strategy_move_white:
  assumes "strategy_move p p'" "to_move p = White"
  shows "strategy_white_move_rel p p'"
using assms
by (auto simp add: strategy_move_def)


lemma strategy_move_black:
  assumes "strategy_move p p'" "to_move p = Black"
  shows "legal_move p p'"
using assms
by (auto simp add: strategy_move_def)

lemma strategy_play:
  assumes "p \<in> strategy_play p0" "legal_position p0" "strategy_invar p0" "to_move p0 = White"
  shows "legal_position p \<and> strategy_invar p \<and> (to_move p = Black \<longrightarrow> (\<exists> px. px \<in> strategy_play p0 \<and> legal_position px \<and> strategy_invar px \<and> to_move px = White \<and> strategy_white_move_rel px p))"
using assms
proof (induct rule: strategy_play.induct)
  case (step p p')
  show ?case
  proof (cases "to_move p = White")
    case True
    thus ?thesis
      using step strategy_move_white[of p p'] strategy_white_move_legal_move[of p p'] strategy_white_move_invar[of p p'] strategy_move_white[of p p']
      by (auto simp add: legal_move_def)
  next
    case False
    hence "to_move p = Black"
      by (metis side.exhaust)
    then obtain px where "px \<in> strategy_play p0" "legal_position px" "strategy_invar px" "to_move px = White"  "strategy_white_move_rel px p" 
      using step
      by auto
    thus ?thesis
      using step `to_move p = Black`
      using `strategy_move p p'` strategy_move_black[of p p'] strategy_white_move_black_move_invar[of px p p']
      by (simp add: legal_move_def)
  qed
qed simp

lemmas strategy_play_legal = strategy_play[THEN conjunct1]
lemmas strategy_play_invar = strategy_play[THEN conjunct2, THEN conjunct1]
lemmas strategy_play_black = strategy_play[THEN conjunct2, THEN conjunct2, rule_format]

end

locale WinningStrategy' = Strategy on_square for on_square :: "'p \<Rightarrow> int \<times> int \<Rightarrow> (side \<times> piece) option" + 
  fixes start :: "'p \<Rightarrow> bool" and ordering :: "'p \<Rightarrow> ('p \<times> 'p) set" 
  assumes start: "\<And> p0. start p0 \<Longrightarrow> to_move p0 = White \<and> strategy_invar p0 \<and> legal_position p0"
  assumes wf_ordering: "\<And> p0. start p0 \<Longrightarrow> wf (ordering p0)"
  assumes ordering: "\<And> p0 p p' p''. \<lbrakk>start p0; p \<in> strategy_play p0; to_move p = White; strategy_white_move_rel p p'; legal_move p' p''\<rbrakk> \<Longrightarrow> (p'', p) \<in> ordering p0"
  assumes white_can_move: "\<And> p0 p. \<lbrakk> start p0; p \<in> strategy_play p0; to_move p = White \<rbrakk> \<Longrightarrow> (\<exists> p'. strategy_white_move_rel p p')"
  assumes strategy_not_stalemate: "\<And> p0 p p'. \<lbrakk>start p0; p \<in> strategy_play p0; to_move p = White; strategy_white_move_rel p p'\<rbrakk> \<Longrightarrow> \<not> stalemate p'"

sublocale WinningStrategy' < WinningStrategy invar to_move strategy_white_move_rel strategy_invar on_square start
proof
  fix p0 p
  assume *: "start p0" "p \<in> strategy_play p0" "\<not> (\<exists>p'. strategy_move p p')"
  have "to_move p = Black"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "to_move p = White"
      by (metis side.exhaust)
    thus False
      using * white_can_move[of p0 p]
      unfolding strategy_move_def
      by auto
  qed
  moreover
  have "checkmated p"
  proof-
    have "\<not> (\<exists> p'. legal_move p p')"
      using *(3) `to_move p = Black`
      unfolding strategy_move_def
      by auto
    obtain px where "px \<in> strategy_play p0"  "legal_position px"  "strategy_invar px"  "to_move px = White"  "strategy_white_move_rel px p"
      using strategy_play_black[OF *(2)] *(1) start `to_move p = Black`
      by auto
    thus ?thesis
      using strategy_not_stalemate[rule_format, of p0 px] `\<not> (\<exists> p'. legal_move p p')` `start p0` strategy_play_legal[OF `p \<in> strategy_play p0`]  start
      by (auto simp add: checkmated_def stalemate_def game_over_def)
  qed
  ultimately
  show "to_move p = Black \<and> checkmated p"
    by simp
next
  fix p0
  assume "start p0"
  show "\<not> (\<exists>Q. p0 \<in> Q \<and> (\<forall>p\<in>Q. \<exists>p'\<in>Q. strategy_move p p'))"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    then obtain Q where "p0 \<in> Q" and *: "\<forall> p\<in>Q. \<exists>p'\<in>Q. strategy_move p p'"
      by auto
    let ?Q = "Q \<inter> strategy_play p0"
    let ?Qw = "?Q \<inter> {p. to_move p = White}"
    have "p0 \<in> ?Qw"
      using  `p0 \<in> Q` start[of p0] `start p0` strategy_play.init[of p0]
      by auto
    then obtain pmin where "pmin\<in>?Qw" and **: "\<forall>p. (p, pmin) \<in> ordering p0 \<longrightarrow> p \<notin> ?Qw"
      using wf_ordering[of p0] `start p0`
      unfolding wf_eq_minimal
      by metis
    obtain p' where "p' \<in> ?Q" "strategy_move pmin p'"
      using * `pmin \<in> ?Qw`
      by (metis Int_iff strategy_play.step)
    have "strategy_white_move_rel pmin p'"
      using `strategy_move pmin p'` `pmin \<in> ?Qw`
      using strategy_move_white
      by auto
    hence "legal_move pmin p'"
      using strategy_white_move_legal_move[of pmin p'] strategy_play[of pmin p0]
      using start[OF `start p0`] `pmin \<in> ?Qw`
      by simp
    hence "to_move p' = Black"
      using `pmin \<in> ?Qw`
      by (simp add: legal_move_def)
    obtain p'' where "p'' \<in> ?Q" "strategy_move p' p''"
      using * `p' \<in> ?Q`
      by (metis Int_iff strategy_play.step)
    hence "legal_move p' p''"
      using strategy_move_black `to_move p' = Black`
      by simp
    hence "p'' \<in> ?Qw"
      using `to_move p' = Black` `p'' \<in> ?Q`
      by (simp add: legal_move_def)
    have "(p'', pmin) \<in> ordering p0"
      using `strategy_white_move_rel pmin p'` `legal_move p' p''`
      using ordering[of p0 pmin p' p''] `start p0`
      using `pmin \<in> ?Qw`
      by simp
    hence "p'' \<notin> ?Qw"
      using **
      by auto
    thus False
      using `p'' \<in> ?Qw`
      by simp
  qed
qed

end
