header{* KRK - chess with two kings and a white rook *}
theory KRK
imports ChessRules
begin

(* ************************************************************************** *)
section{* KRK chess *}
text{* 
  In this section we formalize chess positions with only Black
  King, White King and possibly White rook on the board *}
(* ************************************************************************** *)


(* ************************************************************************** *)
subsection{* Definitions *}
(* ************************************************************************** *)

record KRKPosition = 
  WK :: "square" (* position of White King *)
  BK :: "square" (* position of Black King *)
  WRopt :: "square option" (* possition of White Rook (None if captured) *)
  WhiteOnTurn :: "bool" (* Is white on turn? *)

abbreviation BlackOnTurn :: "KRKPosition \<Rightarrow> bool" where
 "BlackOnTurn p \<equiv> \<not> WhiteOnTurn p"

definition WR :: "KRKPosition \<Rightarrow> square" where
 "WR p = the (WRopt  p)"

definition WRcaptured :: "KRKPosition \<Rightarrow> bool" where
 "WRcaptured p \<longleftrightarrow> WRopt p = None"

definition invar :: "KRKPosition \<Rightarrow> bool" where (* Positions of all pieces are different *)
  "invar p \<longleftrightarrow> WK p \<noteq> BK p \<and> WRopt p \<noteq> Some (WK p) \<and> WRopt p \<noteq> Some (BK p)"

definition to_move :: "KRKPosition \<Rightarrow> side" where 
  "to_move p = (if WhiteOnTurn p then White else Black)"

definition on_square :: "KRKPosition \<Rightarrow> square \<Rightarrow> (side \<times> piece) option" where
  "on_square p sq = 
     (if WK p = sq then Some (White, King) 
      else if BK p = sq then Some (Black, King)
      else if WRopt p = Some sq then Some (White, Rook)
      else None)"

interpretation Position_KRK: Position to_move on_square invar
done

definition all_on_board where
  "all_on_board p \<longleftrightarrow> board (WK p) \<and> board (BK p) \<and> (\<not> WRcaptured p \<longrightarrow> board (WR p))"

definition kings_separated :: "KRKPosition \<Rightarrow> bool" where
  "kings_separated p \<longleftrightarrow> \<not> king_scope (WK p) (BK p)"

(* In legal positions when white is on turn then BK cannot block WR (as BK would be in check) so there is no need to check that condition *)
definition WR_attacks_nobk :: "KRKPosition \<Rightarrow> square \<Rightarrow> bool" where
  "WR_attacks_nobk p sq \<longleftrightarrow> \<not> WRcaptured p \<and> rook_scope (WR p) sq \<and> \<not> square_between (WR p) (WK p) sq"

definition WR_attacks_BK :: "KRKPosition \<Rightarrow> bool" where
  [simp]: "WR_attacks_BK p \<longleftrightarrow> WR_attacks_nobk p (BK p)"

definition WK_attacks :: "KRKPosition \<Rightarrow> square \<Rightarrow> bool" where
  [simp]: "WK_attacks p sq \<longleftrightarrow> king_scope (WK p) sq"

definition BK_attacks :: "KRKPosition \<Rightarrow> square \<Rightarrow> bool" where
  [simp]: "BK_attacks p sq \<longleftrightarrow> king_scope (BK p) sq"

definition white_attacks_nobk :: "KRKPosition \<Rightarrow> square \<Rightarrow> bool" where
  "white_attacks_nobk p sq \<longleftrightarrow> WK_attacks p sq \<or> WR_attacks_nobk p sq"

definition legal_position :: "KRKPosition \<Rightarrow> bool" where
  "legal_position p \<longleftrightarrow>
      invar p \<and> all_on_board p \<and> kings_separated p \<and> (WhiteOnTurn p \<longrightarrow> \<not> WR_attacks_BK p)"

definition BK_cannot_move :: "KRKPosition \<Rightarrow> bool" where
  "BK_cannot_move p \<longleftrightarrow> BlackOnTurn p \<and> all_king_pos (\<lambda> sq. board sq \<longrightarrow> white_attacks_nobk p sq) (BK p)"

definition draw :: "KRKPosition \<Rightarrow> bool" where
  "draw p \<longleftrightarrow> WRcaptured p"
definition stalemate :: "KRKPosition \<Rightarrow> bool"  where
  "stalemate p \<longleftrightarrow> legal_position p \<and> BK_cannot_move p \<and> \<not> WR_attacks_BK p"
definition checkmated :: "KRKPosition \<Rightarrow> bool"  where
  "checkmated p \<longleftrightarrow> legal_position p \<and> BK_cannot_move p \<and> WR_attacks_BK p"

text{* Check move functions *}
definition moveWK :: "KRKPosition \<Rightarrow> square \<Rightarrow> KRKPosition"  where
  "moveWK p sq = p \<lparr> WK := sq, WhiteOnTurn := False\<rparr>"
definition moveWR :: "KRKPosition \<Rightarrow> square \<Rightarrow> KRKPosition"  where
  "moveWR p sq = p \<lparr> WRopt := Some sq, WhiteOnTurn := False\<rparr>"
definition moveBK :: "KRKPosition \<Rightarrow> square \<Rightarrow> KRKPosition"  where
  "moveBK p s = 
      (let p' = p \<lparr> BK := s, WhiteOnTurn := True \<rparr> 
        in if WR p = s then p' \<lparr> WRopt := None \<rparr> else p')"

text{* Legal check moves - relations *}
definition legal_move_WK :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
"legal_move_WK p1 p2 \<longleftrightarrow> legal_position p1 \<and> WhiteOnTurn p1 \<and> WK_attacks p1 (WK p2) \<and> p2 = moveWK p1 (WK p2) \<and> legal_position p2"
definition legal_move_WR :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "legal_move_WR p1 p2 \<longleftrightarrow> legal_position p1 \<and> WhiteOnTurn p1 \<and> WR_attacks_nobk p1 (WR p2) \<and> p2 = moveWR p1 (WR p2) \<and> legal_position p2"
definition legal_move_BK :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
 "legal_move_BK p1 p2 \<longleftrightarrow> legal_position p1 \<and> BlackOnTurn p1 \<and> BK_attacks p1 (BK p2) \<and> p2 = moveBK p1 (BK p2) \<and> legal_position p2"
definition legal_move_white :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "legal_move_white p p' \<longleftrightarrow> legal_move_WR p p' \<or> legal_move_WK p p'"

text{* These check if the move to the square sq will maintain a legal position *}
definition BK_can_move_to where
  "BK_can_move_to p sq \<longleftrightarrow> king_scope (BK p) sq \<and> board sq \<and> \<not> white_attacks_nobk p sq"
definition WK_can_move_to where
  "WK_can_move_to p sq \<longleftrightarrow> king_scope (WK p) sq \<and> board sq \<and> \<not> king_scope (BK p) sq \<and> (\<not> WRcaptured p \<longrightarrow> WR p \<noteq> sq)"
definition WR_can_move_to where
  "WR_can_move_to p sq \<longleftrightarrow> WR_attacks_nobk p sq \<and> board sq \<and> WK p \<noteq> sq \<and> BK p \<noteq> sq"


(* ************************************************************************** *)
subsection{* Properties *}
(* ************************************************************************** *)

lemma WRopt:
  assumes "\<not> WRcaptured p" "WR p = (WRx, WRy)"
  shows "WRopt p \<noteq> Some (a, b) \<longleftrightarrow> (WRx \<noteq> a \<or> WRy \<noteq> b)"
using assms
unfolding WRcaptured_def WR_def
by auto


(* ************************************************************************** *)
text{* Properties of on_square *}
(* ************************************************************************** *)

lemma on_square_WK:
  assumes "invar p"
  shows "on_square p (WK p) = Some (White, King)"
  unfolding on_square_def
  by simp

lemma on_square_WK':
  assumes "invar p" "on_square p sq = Some (White, King)"
  shows "sq = WK p"
  using assms
  unfolding invar_def on_square_def
  by (simp split: split_if_asm)

lemma on_square_BK:
  assumes "invar p"
  shows "on_square p (BK p) = Some (Black, King)"
  using assms
  unfolding invar_def on_square_def
  by simp

lemma on_square_BK':
  assumes "invar p" "on_square p sq = Some (Black, King)"
  shows "sq = BK p"
  using assms
  unfolding invar_def on_square_def
  by (simp split: split_if_asm)

lemma on_square_WR:
  assumes "invar p" "\<not> WRcaptured p"
  shows "on_square p (WR p) = Some (White, Rook)"
  using assms
  unfolding on_square_def invar_def
  by (force simp add: WR_def WRcaptured_def)

lemma on_square_WR':
  assumes "invar p" "on_square p sq = Some (White, Rook)"
  shows "sq = WR p" "\<not> WRcaptured p"
  using assms
  unfolding invar_def on_square_def
  by (auto simp add: WRcaptured_def WR_def split: split_if_asm)

lemma all_pieces:
  assumes "invar p"
  shows "(\<forall>sq. on_square p sq \<noteq> None \<longrightarrow> P sq) = (P (WK p) \<and> P (BK p) \<and> (\<not> WRcaptured p \<longrightarrow> P (WR p)))"
using assms
unfolding on_square_def invar_def
by (metis WR_def WRcaptured_def option.collapse option.distinct(1) option.sel)
(* by (metis WR_def WRcaptured_def option.distinct(1) option.exhaust the.simps) *)

lemma occupies_white:
  assumes "invar p"
  assumes "Position_KRK.occupies p White sq"
  shows "(\<not> WRcaptured p \<and> sq = WR p) \<or> sq = WK p"
using assms
unfolding Position_KRK.occupies_def
by (auto simp add: on_square_def WR_def WRcaptured_def split: split_if_asm)

lemma occupies_black:
  assumes "invar p"
  assumes "Position_KRK.occupies p Black sq"
  shows "sq = BK p"
using assms
unfolding Position_KRK.occupies_def
by (auto simp add: on_square_def split: split_if_asm)

(* ************************************************************************** *)
text{* Properites of all_on_board *}
(* ************************************************************************** *)

(* --------------------------------------------------------------- *)
theorem all_on_board:
  assumes "invar p"
  shows "Position_KRK.all_on_board p \<longleftrightarrow> all_on_board p"
(* --------------------------------------------------------------- *)
using assms
unfolding Position_KRK.all_on_board_def all_on_board_def
unfolding Position.empty_def
by (rule all_pieces)

(* ************************************************************************** *)
text{* Properies of kings_separated *}
(* ************************************************************************** *)

lemma kings_separated_iff:
  "kings_separated p \<and> WK p \<noteq> BK p  \<longleftrightarrow> 
     (let (WKx, WKy) = WK p; (BKx, BKy) = BK p in
       WKx > BKx + 1 \<or> BKx > WKx + 1 \<or> WKy > BKy + 1 \<or> BKy > WKy + 1)"
unfolding kings_separated_def king_scope_def file_diff_def rank_diff_def diff_def
by (cases "WK p", cases "BK p")  auto

(* --------------------------------------------------------------- *)
theorem kings_separated_attacks_WKBK:
  assumes "invar p"
  shows "kings_separated p \<longleftrightarrow> \<not> Position_KRK.attacks p (WK p) (BK p)"
(* --------------------------------------------------------------- *)
using assms
using on_square_WK[of p]
unfolding Position_KRK.attacks_def kings_separated_def
by (auto simp add: Position.king_scope_clear_line)

(* --------------------------------------------------------------- *)
theorem kings_separated_attacks_BKWK:
  assumes "invar p"
  shows "kings_separated p \<longleftrightarrow> \<not> Position_KRK.attacks p (BK p) (WK p)"
(* --------------------------------------------------------------- *)
using assms on_square_WK[of p] on_square_BK[of p]
unfolding Position_KRK.attacks_def kings_separated_def
by (auto simp add: Position.king_scope_clear_line king_scope_sym)

(* ************************************************************************** *)
text{* Properties of clear line *}
(* ************************************************************************** *)

lemma clear_line_WR:
  assumes "invar p"
  shows "Position_KRK.clear_line p (WR p) sq \<longleftrightarrow> \<not> square_between (WR p) (WK p) sq \<and> \<not> square_between (WR p) (BK p) sq"
using assms
using on_square_WK[of p] on_square_BK[of p]
unfolding Position.clear_line_def Position.empty_def
using all_pieces[of p "\<lambda> x. \<not> square_between (WR p) x sq"]
by auto (metis option.exhaust prod.exhaust)

(* ************************************************************************** *)
text{* Definition and properties of WR_attacks *}
(* ************************************************************************** *)

definition WR_attacks :: "KRKPosition \<Rightarrow> square \<Rightarrow> bool" where
  "WR_attacks p sq \<longleftrightarrow> \<not> WRcaptured p \<and> rook_scope (WR p) sq \<and> \<not> square_between (WR p) (WK p) sq  \<and> \<not> square_between (WR p) (BK p) sq"

(* --------------------------------------------------------------- *)
theorem WR_attacks:
  assumes "invar p"
  shows "WR_attacks p sq \<longleftrightarrow> \<not> WRcaptured p \<and> Position_KRK.attacks p (WR p) sq"
(* --------------------------------------------------------------- *)
using assms clear_line_WR[of p]
unfolding Position_KRK.attacks_def WR_attacks_def
by (auto simp add: on_square_WR)

(* --------------------------------------------------------------- *)
theorem WR_attacks_nobk:
  assumes "legal_position p" "WhiteOnTurn p"
  shows "WR_attacks_nobk p sq \<longleftrightarrow> WR_attacks p sq"
(* --------------------------------------------------------------- *)
using assms 
by (cases "WR p", cases "BK p", cases "WK p", cases sq, simp) (auto simp add:  WR_attacks_nobk_def WR_attacks_def legal_position_def square_between_def same_file_def same_rank_def rook_scope_def between_def)


(* ************************************************************************** *)
text{* Properties of WR_attacks_BK *}
(* ************************************************************************** *)

(* --------------------------------------------------------------- *)
theorem WR_attacks_BK': "WR_attacks_BK p \<longleftrightarrow> WR_attacks p (BK p)"
(* ------------------------------------------------------------- *)
by (simp add: WR_attacks_def WR_attacks_nobk_def)

(* --------------------------------------------------------------- *)
theorem WR_attacks_BK:
  assumes "invar p"
  shows "WR_attacks_BK p \<longleftrightarrow> \<not> WRcaptured p \<and> Position_KRK.attacks p (WR p) (BK p)"
(* --------------------------------------------------------------- *)
using assms WR_attacks_BK'[of p] WR_attacks[of p]
by simp

(* ************************************************************************** *)
text{* Properties of in_check  *}
(* ************************************************************************** *)

lemma not_kings_separated_black_in_check:
  assumes "invar p" "board (WK p)" "board (BK p)"
  assumes "\<not> kings_separated p"
  shows "Position_KRK.in_check Black p"
using assms
using kings_separated_attacks_WKBK[of p]  on_square_WK[of p] on_square_BK[of p]
unfolding Position_KRK.in_check_def Position_KRK.occupies_def
by auto (metis prod.exhaust)

lemma not_kings_separated_white_in_check:
  assumes "invar p" "board (WK p)" "board (BK p)"
  assumes "\<not> kings_separated p"
  shows "Position_KRK.in_check White p"
using assms
using kings_separated_attacks_BKWK[of p]  on_square_WK[of p] on_square_BK[of p]
unfolding Position_KRK.in_check_def Position_KRK.occupies_def
by auto (metis prod.exhaust)

lemma white_in_check:
  assumes "invar p" "Position_KRK.in_check White p"
  shows "\<not> kings_separated p"
using assms on_square_WK'[of p] occupies_black[of p] kings_separated_attacks_BKWK[of p]
unfolding Position_KRK.in_check_def
by auto

lemma WR_attacks_BK_black_in_check:
  assumes "invar p" "board (WR p)" "board (BK p)"
  assumes "WR_attacks_BK p"
  shows "Position_KRK.in_check Black p"
using assms
using WR_attacks_BK[of p]
unfolding WR_attacks_BK_def
unfolding Position_KRK.in_check_def Position_KRK.occupies_def
using on_square_BK[of p] on_square_WR[of p]
by auto (metis surj_pair)

lemma black_in_check:
  assumes "invar p" "Position_KRK.in_check Black p"
  shows "\<not> kings_separated p \<or> WR_attacks_BK p"
proof-
  obtain sq2 where "Position_KRK.occupies p White sq2" "Position_KRK.attacks p sq2 (BK p)" "board sq2"
    using assms on_square_BK'[of p]
    unfolding Position_KRK.in_check_def
    by auto
  hence "Position_KRK.attacks p (WK p) (BK p) \<or> (\<not> WRcaptured p \<and> Position_KRK.attacks p (WR p) (BK p))"
    using `invar p` occupies_white[of p sq2]
    by auto
  thus ?thesis
    using WR_attacks_BK[of p] `invar p` kings_separated_attacks_WKBK[of p]
    by auto
qed

(* --------------------------------------------------------------- *)
theorem black_in_check_iff:
  assumes "invar p" "all_on_board p"
  shows "Position_KRK.in_check Black p \<longleftrightarrow> \<not> kings_separated p \<or> WR_attacks_BK p"
(* --------------------------------------------------------------- *)
using assms black_in_check[of p] WR_attacks_BK_black_in_check[of p] not_kings_separated_black_in_check[of p]
unfolding all_on_board_def 
by auto (simp add: WR_attacks_nobk_def)

(* --------------------------------------------------------------- *)
theorem white_in_check_iff:
  assumes "invar p" "board (WK p)" "board (BK p)"
  shows "Position_KRK.in_check White p \<longleftrightarrow> \<not> kings_separated p"
(* --------------------------------------------------------------- *)
using assms white_in_check[of p] not_kings_separated_white_in_check[of p]
by auto

(* ************************************************************************** *)
text{* Properites of legal_position *}
(* ************************************************************************** *)

(* --------------------------------------------------------------- *)
theorem legal_position: "Position_KRK.legal_position p \<longleftrightarrow> legal_position p"
(* --------------------------------------------------------------- *)
proof (cases "to_move p = White")
  case True
  thus ?thesis
    unfolding Position_KRK.legal_position_def legal_position_def
    using black_in_check_iff[of p]
    by (auto simp add: all_on_board to_move_def)
next
  case False
  hence "to_move p = Black"
    by (metis side.exhaust)
  thus ?thesis
    unfolding Position_KRK.legal_position_def legal_position_def
    using white_in_check_iff[of p]    
    by (auto simp add: all_on_board all_on_board_def to_move_def split: split_if_asm)
qed

(* ************************************************************************** *)
text{* Alternative formulations for legal moves - technicallity 
       (might be eliminated when proofs are polished) *}
(* ************************************************************************** *)

lemma legal_move_WK_def':
  "legal_move_WK p1 p2 \<longleftrightarrow> legal_position p1 \<and> legal_position p2 \<and> WhiteOnTurn p1 \<and> (\<exists> sq. king_scope (WK p1) sq \<and> p2 = moveWK p1 sq )"
unfolding legal_move_WK_def
by (cases "WK p2") (auto simp add: moveWK_def)

lemma legal_move_WK_def'':
  "legal_move_WK p1 p2 \<longleftrightarrow> (\<exists> sq. king_scope (WK p1) sq \<and> p2 = moveWK p1 sq \<and> WhiteOnTurn p1 \<and> legal_position p1 \<and> legal_position p2)"
unfolding legal_move_WK_def'
by auto

lemma legal_move_WR_def':
  "legal_move_WR p1 p2 \<longleftrightarrow> legal_position p1 \<and> legal_position p2 \<and> WhiteOnTurn p1 \<and> (\<exists> sq. WR_attacks p1 sq \<and> p2 = moveWR p1 sq)"
using WR_attacks_nobk[of p1]
unfolding legal_move_WR_def
by (cases "WR p2") (auto simp add: moveWR_def WR_def)

lemma legal_move_WR_def'':
  "legal_move_WR p1 p2 \<longleftrightarrow> (\<exists> sq. WR_attacks p1 sq \<and> p2 = moveWR p1 sq \<and> WhiteOnTurn p1 \<and> legal_position p1 \<and> legal_position p2)"
unfolding legal_move_WR_def'
by auto

lemma legal_move_BK_def':
  "legal_move_BK p1 p2 \<longleftrightarrow> legal_position p1 \<and> legal_position p2 \<and> BlackOnTurn p1 \<and> (\<exists> sq. king_scope (BK p1) sq \<and> p2 = moveBK p1 sq)"
unfolding legal_move_BK_def
by (cases "BK p2") (auto simp add: moveBK_def Let_def)

(* ************************************************************************** *)
text{* Properties of moveWK *}
(* ************************************************************************** *)

lemma moveWK_eqI:
  assumes "WK p2 = sq" "BK p2 = BK p1" "WRopt p2 = WRopt p1" "BlackOnTurn p2"
  shows "p2 = moveWK p1 sq"
using assms
unfolding moveWK_def
by (subst KRKPosition.equality) simp_all

lemma moveWK_fields [simp]: "WK (moveWK p sq) = sq" "BK (moveWK p sq) = BK p" "WRopt (moveWK p sq) = WRopt p" "WR (moveWK p sq) = WR p" "WRcaptured (moveWK p sq) = WRcaptured p"
by (simp_all add: moveWK_def WR_def WRcaptured_def)

lemma WhiteOnTurn_MoveWK [simp]: "WhiteOnTurn (moveWK p sq) = False"
by (simp add: moveWK_def)

lemma [simp]:
  assumes "invar p"
  shows "on_square (moveWK p sq) sq = Some (White, King)"
by (simp add: on_square_def)

lemma [simp]: 
  assumes "invar p" "sq \<noteq> WK p"
  shows "on_square (moveWK p sq) (WK p) = None"
using assms
by (auto simp add: on_square_def invar_def)

lemma [simp]: 
  assumes "invar p" "sq' \<noteq> sq" "sq' \<noteq> WK p"
  shows "on_square (moveWK p sq) sq' = on_square p sq'"
using assms
by (auto simp add: on_square_def)


(* --------------------------------------------------------------- *)
theorem legal_move_WK_legal_move:
  assumes "legal_move_WK p1 p2"
  shows "Position_KRK.legal_move p1 p2"
(* --------------------------------------------------------------- *)
proof-
  obtain sq where "WhiteOnTurn p1" "king_scope (WK p1) sq" "p2 = moveWK p1 sq" "legal_position p1" "legal_position p2"
    using `legal_move_WK p1 p2`
    unfolding legal_move_WK_def'
    by auto

  have "on_square p1 sq = None"
  proof-
    have "on_square p1 sq \<noteq> Some (White, King)"
      using on_square_WK'[of p1 sq]
      using `king_scope (WK p1) sq`
      by (simp add: on_square_def king_scope_def)
    moreover
    have "on_square p1 sq \<noteq> Some (Black, King)"
      using `legal_position p1` `WhiteOnTurn p1` `king_scope (WK p1) sq`
      unfolding legal_position_def kings_separated_def
      using on_square_BK'[of p1 sq]
      by auto
    moreover
    have "on_square p1 sq \<noteq> Some (White, Rook)"
      using `legal_position p1` `legal_position p2` `p2 = moveWK p1 sq`
      using on_square_WR'[of p1 sq]
      unfolding legal_position_def moveWK_def invar_def
      by (auto simp add: WR_def WRcaptured_def)
    ultimately
    show ?thesis
      unfolding on_square_def
      by (auto split: split_if_asm)
  qed

  have "board sq" "board (WK p1)" "invar p1" "invar (moveWK p1 sq)"
    using `legal_position p1` `legal_position p2` `p2 = moveWK p1 sq`
    unfolding legal_position_def all_on_board_def
    by auto

  hence "Position_KRK.simple_move p1 (moveWK p1 sq)"
    using `WhiteOnTurn p1` `on_square p1 sq = None` `king_scope (WK p1) sq`
    using on_square_WK[of p1] on_square_WK[of "moveWK p1 sq"]
    unfolding Position_KRK.simple_move_def
    by (rule_tac x="WK p1" in exI, rule_tac x="sq" in exI)
       (auto simp add: to_move_def Position_KRK.occupies_def Position_KRK.sorties_def Position.empty_def Position_KRK.attacks_def Position.king_scope_clear_line)
  thus ?thesis
    using `p2 = moveWK p1 sq` `legal_position p1` `legal_position p2` `WhiteOnTurn p1`
    by (simp add: Position_KRK.legal_move_def legal_position  to_move_def)
qed

lemma legal_move_WK_notWRcaptured:
  "\<lbrakk>legal_move_WK p p'; \<not> WRcaptured p\<rbrakk> \<Longrightarrow> \<not> WRcaptured p'"
unfolding legal_move_WK_def
by auto (metis moveWK_fields(5))

lemma legal_move_WK_BlackOnTurn: 
  "\<lbrakk> WhiteOnTurn p; legal_move_WK p p' \<rbrakk> \<Longrightarrow> BlackOnTurn p'"
by (metis KRK.legal_move_WK_def WhiteOnTurn_MoveWK)

(* ************************************************************************** *)
text{* Properties of moveWR *}
(* ************************************************************************** *)

lemma moveWR_eqI:
  assumes "WRopt p2 = Some sq" "BK p2 = BK p1" "WK p2 = WK p1" "BlackOnTurn p2"
  shows "p2 = moveWR p1 sq"
using assms
unfolding moveWR_def
by (subst KRKPosition.equality) simp_all

lemma moveWR_fields [simp]: "WK (moveWR p sq) = WK p" "BK (moveWR p sq) = BK p" "WRopt (moveWR p sq) = Some sq" "WR (moveWR p sq) = sq" "WRcaptured (moveWR p sq) = False" "WhiteOnTurn (moveWR p sq) = False"
by (simp_all add: moveWR_def WR_def WRcaptured_def)

lemma [simp]:
  assumes "invar p" "\<not> WRcaptured p" "sq \<noteq> WK p" "sq \<noteq> BK p"
  shows "on_square (moveWR p sq) sq = Some (White, Rook)"
using assms
by (simp add: on_square_def)

lemma [simp]: 
  assumes "invar p" "\<not> WRcaptured p" "sq \<noteq> WR p"
  shows "on_square (moveWR p sq) (WR p) = None"
using assms
unfolding on_square_def invar_def WR_def WRcaptured_def
by auto

lemma [simp]: 
  assumes "invar p" "sq' \<noteq> sq" "sq' \<noteq> WR p"
  shows "on_square (moveWR p sq) sq' = on_square p sq'"
using assms
by (auto simp add: on_square_def moveWR_def WR_def)

(* ------------------------------------------------------------- *)
theorem legal_move_WR_legal_move:
  assumes "legal_move_WR p1 p2"
  shows "Position_KRK.legal_move p1 p2"
(* ------------------------------------------------------------- *)
proof-
  obtain sq where "WhiteOnTurn p1" "\<not> WRcaptured p1" "rook_scope (WR p1) sq" "\<not> square_between (WR p1) (WK p1) sq" "\<not> square_between (WR p1) (BK p1) sq"  "p2 = moveWR p1 sq" "legal_position p1" "legal_position p2" "WR_attacks p1 sq"

    using `legal_move_WR p1 p2`
    unfolding legal_move_WR_def' WR_attacks_def
    by auto
  have "WR (p1) \<noteq> sq"
    using `rook_scope (WR p1) sq`
    unfolding rook_scope_def
    by auto

  have "on_square p1 sq = None"
  proof-
    have "on_square p1 sq \<noteq> Some (White, King)"
      using `legal_position p1` `legal_position p2` `p2 = moveWR p1 sq`
      using on_square_WK'[of p1 sq]
      unfolding legal_position_def invar_def
      by auto
    moreover
    have "on_square p1 sq \<noteq> Some (Black, King)"
      using `legal_position p1` `legal_position p2` `p2 = moveWR p1 sq`
      using on_square_BK'[of p1 sq]
      unfolding legal_position_def invar_def
      by auto
    moreover
    have "on_square p1 sq \<noteq> Some (White, Rook)"
      using `legal_position p1` on_square_WR'[of p1 sq] `WR p1 \<noteq> sq`
      by (auto simp add: legal_position_def)
    ultimately
    show ?thesis
      unfolding on_square_def
      by (auto split: split_if_asm)
  qed
  have "board sq" "board (WR p1)" "invar p1" "invar (moveWR p1 sq)" "WK p1 \<noteq> sq" "BK p1 \<noteq> sq"
    using `legal_position p1` `legal_position p2` `p2 = moveWR p1 sq``\<not> WRcaptured p1`
    using `on_square p1 sq = None` on_square_WK[of p1] on_square_BK[of p1]
    unfolding legal_position_def all_on_board_def
    by auto

  hence "Position_KRK.simple_move p1 (moveWR p1 sq)"
    using `WhiteOnTurn p1` `on_square p1 sq = None` `WR_attacks p1 sq` `\<not> WRcaptured p1`
    using on_square_WR[of p1]
    unfolding Position_KRK.simple_move_def
    by (rule_tac x="WR p1" in exI, rule_tac x="sq" in exI)
       (auto simp add: to_move_def Position_KRK.occupies_def Position_KRK.sorties_def Position.empty_def WR_attacks)
  thus ?thesis
    using `p2 = moveWR p1 sq` `legal_position p1` `legal_position p2` `WhiteOnTurn p1`
    by (simp add: Position_KRK.legal_move_def legal_position to_move_def)
qed

lemma legal_move_WR_notWRcaptured:
  "legal_move_WR p p' \<Longrightarrow> \<not> WRcaptured p'"
unfolding legal_move_WR_def
by auto (metis moveWR_fields(5))

lemma legal_move_WR_BlackOnTurn: "\<lbrakk> WhiteOnTurn p; legal_move_WR p p' \<rbrakk> \<Longrightarrow> BlackOnTurn p'"
by (metis legal_move_WR_def' moveWR_fields(6))

(* -------------------------------------------------------------------------- *)

lemma legal_move_white_notWRcaptured:
  "\<lbrakk>\<not> WRcaptured p; legal_move_white p p'\<rbrakk> \<Longrightarrow> \<not> WRcaptured p'"
by (metis legal_move_WK_notWRcaptured legal_move_WR_notWRcaptured legal_move_white_def)

lemma legal_move_white_BlackOnTurn: "\<lbrakk> WhiteOnTurn p; legal_move_white p p' \<rbrakk> \<Longrightarrow> BlackOnTurn p'"
by (metis legal_move_WK_BlackOnTurn legal_move_WR_BlackOnTurn legal_move_white_def)


(* ************************************************************************** *)
text{* Completeness of moveWK and moveWR for White *}
(* ************************************************************************** *)

lemma no_capture_move_white:
  assumes "to_move p1 = White" "legal_position p1"
  shows "\<not> Position_KRK.capture_move p1 p2"
proof (rule ccontr)
  assume "\<not> ?thesis"
  then obtain sq1 sq2 where "board sq1" "board sq2" "Position_KRK.occupies p1 White sq1" "Position_KRK.attacks p1 sq1 sq2" "Position_KRK.occupies p1 Black sq2"
    using assms
    unfolding Position_KRK.capture_move_def Position_KRK.captures_def
    by auto
  moreover
  have "on_square p1 sq2 = Some (Black, King)"
    using `Position_KRK.occupies p1 Black sq2`
    unfolding Position_KRK.occupies_def
    by (auto simp add: on_square_def split: split_if_asm)
  ultimately
  have "Position_KRK.in_check Black p1"
    unfolding Position_KRK.in_check_def
    by (rule_tac x="sq2" in exI, rule_tac x="sq1" in exI) auto
  thus False
    using `legal_position p1` `to_move p1 = White`
    unfolding legal_position[symmetric]
    unfolding Position_KRK.legal_position_def
    by simp
qed

lemma simple_move_white:
  assumes "Position_KRK.simple_move p1 p2" "to_move p1 = White" "to_move p2 = Black" "legal_position p1"  "legal_position p2"
  shows "legal_move_WK p1 p2 \<or> legal_move_WR p1 p2"
proof-
  have "invar p1" "invar p2"
    using `legal_position p1` `legal_position p2`
    unfolding legal_position_def
    by simp_all

  obtain sq1 sq2 where "Position_KRK.occupies p1 White sq1" "board sq1" "board sq2" "Position_KRK.attacks p1 sq1 sq2" "on_square p1 sq2 = None" and
    *: "\<forall>sq. on_square p2 sq = (if sq = sq1 then None else if sq = sq2 then on_square p1 sq1 else on_square p1 sq)"
    using assms
    unfolding Position_KRK.simple_move_def Position_KRK.sorties_def Position.empty_def
    by auto

  have "BK p1 \<noteq> sq1"
    using `invar p1` `Position_KRK.occupies p1 White sq1` occupies_white[of p1 sq1]
    unfolding invar_def
    by (auto simp add: WR_def WRcaptured_def)

  have "WK p1 \<noteq> sq2" "BK p1 \<noteq> sq2" "\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq2"
    using `on_square p1 sq2 = None` `invar p1` on_square_WK[of p1] on_square_BK[of p1] on_square_WR[of p1]
    by auto

  have "sq1 \<noteq> sq2"
    using `Position_KRK.attacks p1 sq1 sq2`
    unfolding Position_KRK.attacks_def
    by (metis `on_square p1 sq2 = None` option.simps(4))

  have "on_square p2 (BK p1) = Some (Black, King)"
    using *[rule_format, of "BK p1"] `BK p1 \<noteq> sq1`  `BK p1 \<noteq> sq2` on_square_BK[of p1] `invar p1`
    by (simp split: split_if_asm)
  hence "BK p2 = BK p1"
    using on_square_BK'[of p2 "BK p1"] `invar p2`
    by simp

  have "sq1 = WK p1 \<or> (\<not> WRcaptured p1 \<and> sq1 = WR p1)"
    using `Position_KRK.occupies p1 White sq1` `invar p1`
    using occupies_white[of p1 sq1]
    by auto

  thus ?thesis
  proof
    assume "sq1 = WK p1"

    have  "on_square p2 sq2 = Some (White, King)"
      using *[rule_format, of sq2] `sq1 \<noteq> sq2` `sq1 = WK p1` on_square_WK[of p1] `invar p1`
      by simp
    hence "WK p2 = sq2"
      using on_square_WK'[of p2 sq2] `invar p2`
      by auto

    have "\<not> WRcaptured p1 \<longrightarrow> \<not> WRcaptured p2 \<and> on_square p2 (WR p1) = Some (White, Rook)"
      using `sq1 = WK p1` on_square_WK[of p1] on_square_WR[of p1] `invar p1`  `on_square p1 sq2 = None`
      using *[rule_format, of "WR p1"]
      unfolding invar_def
      by (auto, force, force) (simp split: split_if_asm add: on_square_def WRcaptured_def)
    hence "\<not> WRcaptured p1 \<longrightarrow> \<not> WRcaptured p2 \<and> WR p2 = WR p1"
      by (simp add: on_square_def WR_def WRcaptured_def split: split_if_asm)

    moreover

    have "WRcaptured p1 \<longrightarrow> WRcaptured p2"
      using *[rule_format, of "WR p2"] on_square_WR[of p2] `invar p2` `sq1 = WK p1` on_square_WK[of p1] `invar p1` on_square_WR'[of p1 "WR p2"]
      by (auto split: split_if_asm)

    have "p2 = moveWK p1 sq2"
    proof (rule moveWK_eqI)
      show "WK p2 = sq2" by fact
    next
      show "BK p2 = BK p1" by fact
    next
      show "WRopt p2 = WRopt p1"
        using `WRcaptured p1 \<longrightarrow> WRcaptured p2` `\<not> WRcaptured p1 \<longrightarrow> \<not> WRcaptured p2 \<and> WR p2 = WR p1`
        by (auto simp add: WRcaptured_def WR_def) (metis option.exhaust prod.exhaust)
    next
      show "BlackOnTurn p2" 
        using `to_move p2 = Black`
        by (auto simp add: to_move_def)
    qed
      
    hence "legal_move_WK p1 p2"
      using `to_move p1 = White` `legal_position p1` `legal_position p2`
      using `Position_KRK.attacks p1 sq1 sq2` `sq1 = WK p1` on_square_WK[of p1] `invar p1`
      unfolding legal_move_WK_def' Position_KRK.attacks_def
      by (simp add: to_move_def split: split_if_asm) (rule_tac x="fst sq2" in exI, rule_tac x="snd sq2" in exI, auto)
    thus ?thesis
      by simp
  next
    assume "\<not> WRcaptured p1 \<and> sq1 = WR p1"
    hence "WK p1 \<noteq> sq1"
      using `invar p1`
      unfolding invar_def
      by (auto simp add: WRcaptured_def WR_def)

    have "on_square p2 (WK p1) = Some (White, King)"
      using *[rule_format, of "WK p1"] `WK p1 \<noteq> sq2` `WK p1 \<noteq> sq1` on_square_WK[of p1] `invar p1`
      by simp
    hence "WK p2 = WK p1"
      using on_square_WK'[of p2 "WK p1"] `invar p2`
      by auto

    have "on_square p2 sq2 = Some (White, Rook)"
      using *[rule_format, of sq2] `\<not> WRcaptured p1 \<and> sq1 = WR p1` `sq1 \<noteq> sq2` on_square_WR[of p1] `invar p1`
      by (simp split: split_if_asm)
    hence "\<not> WRcaptured p2" "sq2 = WR p2"
      using on_square_WR'[of p2 sq2] `invar p2`
      by auto

    have "p2 = moveWR p1 sq2"
    proof (rule moveWR_eqI)
      show "BK p2 = BK p1" by fact
    next
      show "WK p2 = WK p1" by fact
    next
      show "BlackOnTurn p2"
        using `to_move p2 = Black`
        by (auto simp add: to_move_def)
    next
      show "WRopt p2 = Some sq2"
        using `\<not> WRcaptured p2` `sq2 = WR p2`
        unfolding WRcaptured_def WR_def
        by auto
    qed
    moreover
    have "WR_attacks p1 sq2"
      using `Position_KRK.attacks p1 sq1 sq2` `invar p1` `\<not> WRcaptured p1 \<and> sq1 = WR p1`
      using WR_attacks[of p1 sq2]
      by simp
    ultimately
    have "legal_move_WR p1 p2"
      using `to_move p1 = White` `legal_position p1` `legal_position p2`
      unfolding legal_move_WR_def'
      by (simp add: to_move_def split: split_if_asm) (rule_tac x="fst sq2" in exI, rule_tac x="snd sq2" in exI, auto)
    thus ?thesis
      by simp
  qed
qed

(* ------------------------------------------------------------- *)
theorem legal_move_white:
  assumes "Position_KRK.legal_move p1 p2" "to_move p1 = White"
  shows "legal_move_white p1 p2"
(* ------------------------------------------------------------- *)
using assms no_capture_move_white[of p1 p2] simple_move_white[of p1 p2]
unfolding Position_KRK.legal_move_def
by (auto simp add: legal_position legal_move_white_def)


(* ************************************************************************** *)
text{* Properties of moveBK *}
(* ************************************************************************** *)

lemma moveBK_eqI:
  assumes "BK p2 = sq" "WK p2 = WK p1" "WRopt p2 = (if sq = WR p1 then None else WRopt p1)" "WhiteOnTurn p2"
  shows "p2 = moveBK p1 sq"
using assms
unfolding moveBK_def
by (subst KRKPosition.equality) (simp_all split: split_if_asm)

lemma moveBK_fields [simp]: 
  "BK (moveBK p sq) = sq"
  "WK (moveBK p sq) = WK p"
  "WRopt (moveBK p sq) = (if Some sq = WRopt p then None else WRopt p)" 
  "WR (moveBK p sq) = (if Some sq = WRopt p then the (None) else WR p)" 
  "WRcaptured (moveBK p sq) = (if Some sq = WRopt p then True else WRcaptured p)"
by (auto simp add: moveBK_def Let_def WR_def WRcaptured_def) 
(metis option.collapse option.sel)+
(* (metis not_Some_eq the.simps)+ *)

lemma WhiteOnTurn_MoveBK[simp]: "WhiteOnTurn (moveBK p sq) = True"
by (simp add: moveBK_def Let_def)

lemma [simp]:
  assumes "invar p" "sq \<noteq> WK p"
  shows "on_square (moveBK p sq) sq = Some (Black, King)"
using assms
by (simp add: on_square_def)

lemma [simp]: 
  assumes "invar p" "sq \<noteq> BK p" "sq \<noteq> WK p"
  shows "on_square (moveBK p sq) (BK p) = None"
using assms
unfolding on_square_def
by (auto split: split_if_asm simp add: invar_def)

lemma [simp]: 
  assumes "invar p" "sq' \<noteq> sq" "sq' \<noteq> BK p"
  shows "on_square (moveBK p sq) sq' = on_square p sq'"
using assms
by (auto simp add: on_square_def)

lemma legal_move_BK_BlackOnTurn: 
  "\<lbrakk> WhiteOnTurn p; legal_move_BK p p' \<rbrakk> \<Longrightarrow> BlackOnTurn p'"
by (metis KRK.legal_move_BK_def)

(* ------------------------------------------------------------- *)
theorem legal_move_BK_legal_move:
  assumes "legal_move_BK p1 p2"
  shows "Position_KRK.legal_move p1 p2"
(* ------------------------------------------------------------- *)
unfolding Position_KRK.legal_move_def
proof (safe)
  show "Position_KRK.legal_position p1"
    using `legal_move_BK p1 p2` 
    by (simp add: legal_move_BK_def' legal_position)
next
  show "Position_KRK.legal_position p2"
    using `legal_move_BK p1 p2` 
    by (auto simp add: legal_move_BK_def legal_position)
next
  show "to_move p2 = opponent (to_move p1)"
    using `legal_move_BK p1 p2` 
    by (auto simp add: legal_move_BK_def' to_move_def)
next
  have "legal_position p1"
    using `legal_move_BK p1 p2`
    by (simp add: legal_move_BK_def)

  have "invar p1"
    using `legal_position p1` 
    by (simp add: legal_position_def)

  obtain sq where "BlackOnTurn p1" "p2 = moveBK p1 sq" "king_scope (BK p1) sq" "legal_position (moveBK p1 sq)"
    using `legal_move_BK p1 p2` 
    unfolding legal_move_BK_def'
    by auto
  have "Position_KRK.capture_move p1 p2 \<or> Position_KRK.simple_move p1 p2"
  proof (cases "\<not> WRcaptured p1 \<and> sq = WR p1")
    case True
    hence "sq \<noteq> WK p1" "sq \<noteq> BK p1"
      using `invar p1`
      by (auto simp add: invar_def WRcaptured_def WR_def)
    hence "Position_KRK.capture_move p1 p2"
      using `BlackOnTurn p1` on_square_BK[of p1] on_square_WR[of p1] `p2 = moveBK p1 sq` `\<not> WRcaptured p1 \<and> sq = WR p1` `legal_position p1` `king_scope (BK p1) sq`
      unfolding Position_KRK.capture_move_def Position_KRK.captures_def legal_position_def all_on_board_def
      by (rule_tac x="BK p1" in exI, rule_tac x="sq" in exI) (auto simp add: Position_KRK.occupies_def to_move_def Position_KRK.attacks_def Position_KRK.king_scope_clear_line)
    thus ?thesis
      by simp
  next
    case False
    have "on_square p1 sq = None"
    proof-
      have "on_square p1 sq \<noteq> Some (Black, King)"
        using on_square_BK'[of p1 sq]
        using `king_scope (BK p1) sq`
        by (simp add: on_square_def king_scope_def)
      moreover
      have "on_square p1 sq \<noteq> Some (White, King)"
        using `legal_position p1` `BlackOnTurn p1` `king_scope (BK p1) sq`
        unfolding legal_position_def kings_separated_def
        using on_square_WK'[of p1 sq]
        by (auto simp add: king_scope_sym)
      moreover
      have "on_square p1 sq \<noteq> Some (White, Rook)"
        using False `invar p1`
        unfolding WRcaptured_def WR_def
        by (cases sq) (auto simp add: on_square_def)
      ultimately
      show ?thesis
        unfolding on_square_def
        by (auto split: split_if_asm)
    qed

    have "board sq" "board (BK p1)" "invar p1" "invar (moveBK p1 sq)" "WK p1 \<noteq> sq" "BK p1 \<noteq> sq"
      using `legal_position p1` `legal_position (moveBK p1 sq)`
      using `on_square p1 sq = None` on_square_WK[of p1] on_square_BK[of p1]
      unfolding legal_position_def all_on_board_def
      by auto

    hence "Position_KRK.simple_move p1 p2"
      using `p2 = moveBK p1 sq` `BlackOnTurn p1` `on_square p1 sq = None` `king_scope (BK p1) sq` on_square_BK[of p1]
      unfolding Position_KRK.simple_move_def Position_KRK.occupies_def
      by (rule_tac x="BK p1" in exI, rule_tac x="sq" in exI) (auto simp add: to_move_def Position_KRK.sorties_def Position_KRK.attacks_def Position_KRK.empty_def Position_KRK.king_scope_clear_line)
    thus ?thesis
      by simp
  qed
  moreover
  assume "\<not> Position_KRK.capture_move p1 p2"
  ultimately
  show "Position_KRK.simple_move p1 p2"
    by simp
qed

lemma legal_move_BK_white_on_turn:
  assumes "legal_move_BK p p'"
  shows "WhiteOnTurn p'"
using assms
unfolding legal_move_BK_def'
by auto

(* ************************************************************************** *)
text{* Completeness of moveBK *}
(* ************************************************************************** *)

lemma simple_move_black:
  assumes "WhiteOnTurn p2" "BlackOnTurn p1" "legal_position p1" "legal_position p2" "Position_KRK.simple_move p1 p2"
  shows "legal_move_BK p1 p2"
proof-
  have "invar p1" "invar p2"
    using `legal_position p1` `legal_position p2`
    unfolding legal_position_def
    by simp_all

  obtain sq1 sq2 where "Position_KRK.occupies p1 Black sq1" "board sq1" "board sq2" "Position_KRK.attacks p1 sq1 sq2" "on_square p1 sq2 = None" and
    *: "\<forall>sq. on_square p2 sq = (if sq = sq1 then None else if sq = sq2 then on_square p1 sq1 else on_square p1 sq)"
    using assms
    unfolding Position_KRK.simple_move_def Position_KRK.sorties_def Position.empty_def
    by (auto simp add: to_move_def)

  have "WK p1 \<noteq> sq1" "\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq1"
    using `invar p1` `Position_KRK.occupies p1 Black sq1` occupies_black[of p1 sq1]
    unfolding invar_def
    by (auto simp add: WR_def WRcaptured_def)

  have "WK p1 \<noteq> sq2" "BK p1 \<noteq> sq2" "\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq2"
    using `on_square p1 sq2 = None` `invar p1` on_square_WK[of p1] on_square_BK[of p1] on_square_WR[of p1]
    by auto

  have "sq1 \<noteq> sq2"
    using `Position_KRK.attacks p1 sq1 sq2`
    unfolding Position_KRK.attacks_def
    by (metis `on_square p1 sq2 = None` option.simps(4))

  have "on_square p2 (WK p1) = Some (White, King)"
    using *[rule_format, of "WK p1"] `WK p1 \<noteq> sq1`  `WK p1 \<noteq> sq2` on_square_WK[of p1] `invar p1`
    by (simp split: split_if_asm)
  hence "WK p2 = WK p1"
    using on_square_WK'[of p2 "WK p1"] `invar p2`
    by simp

  have "sq1 = BK p1"
    using `Position_KRK.occupies p1 Black sq1` `invar p1`
    using occupies_black[of p1 sq1]
    by auto

  have  "on_square p2 sq2 = Some (Black, King)"
    using *[rule_format, of sq2] `sq1 \<noteq> sq2` `sq1 = BK p1` on_square_BK[of p1] `invar p1`
    by simp
  hence "BK p2 = sq2"
    using on_square_BK'[of p2 sq2] `invar p2`
    by auto

  have "WRcaptured p1 \<longrightarrow> WRcaptured p2"
    using *[rule_format, of "WR p2"] on_square_WR[of p2] `invar p2` `sq1 = BK p1` on_square_BK[of p1] `invar p1` on_square_WR'[of p1 "WR p2"]
    by (auto split: split_if_asm)

  have "\<not> WRcaptured p1 \<longrightarrow> \<not> WRcaptured p2 \<and> WR p2 = WR p1"
    using *[rule_format, of "WR p1"] `\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq1`  `\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq2` on_square_WR[of p1] `invar p1` on_square_WR'[of p2 "WR p1"] `invar p2`
    by auto

  have "p2 = moveBK p1 sq2"
  proof (rule moveBK_eqI)
    show "BK p2 = sq2" by fact
  next
    show "WK p2 = WK p1" by fact
  next
    show "WRopt p2 = (if sq2 = WR p1 then None else WRopt p1)"
      using `\<not> WRcaptured p1 \<longrightarrow> WR p1 \<noteq> sq2` `WRcaptured p1 \<longrightarrow> WRcaptured p2` `\<not> WRcaptured p1 \<longrightarrow> \<not> WRcaptured p2 \<and> WR p2 = WR p1`
      by (auto simp add: WRcaptured_def WR_def) (metis option.exhaust surj_pair)+
  next
    show "WhiteOnTurn p2" by fact
  qed
  
  hence "legal_move_BK p1 p2"
    using `BlackOnTurn p1` `legal_position p1` `legal_position p2`
    using `Position_KRK.attacks p1 sq1 sq2` `sq1 = BK p1` on_square_BK[of p1] `invar p1`
    unfolding legal_move_BK_def' Position_KRK.attacks_def
    by (simp split: split_if_asm) (rule_tac x="fst sq2" in exI, rule_tac x="snd sq2" in exI, simp)
  thus ?thesis
    by simp
qed

lemma capture_move_black:
  assumes "WhiteOnTurn p2" "BlackOnTurn p1" "legal_position p1" "legal_position p2" "Position_KRK.capture_move p1 p2"
  shows "legal_move_BK p1 p2"
proof-
  have "invar p1" "invar p2"
    using `legal_position p1` `legal_position p2`
    unfolding legal_position_def
    by simp_all

  obtain sq1 sq2 where "Position_KRK.occupies p1 Black sq1" "board sq1" "board sq2" "Position_KRK.attacks p1 sq1 sq2" "Position_KRK.occupies p1 White sq2" and
    *: "\<forall>sq. on_square p2 sq = (if sq = sq1 then None else if sq = sq2 then on_square p1 sq1 else on_square p1 sq)"
    using assms
    unfolding Position_KRK.capture_move_def Position_KRK.captures_def Position.empty_def
    by (auto simp add: to_move_def)

  have "sq1 = BK p1"
    using `Position_KRK.occupies p1 Black sq1`
    using occupies_black[of p1 sq1] `invar p1`
    by simp
    
  hence "sq1 \<noteq> sq2" "WK p1 \<noteq> sq2" "BK p1 \<noteq> sq2"
    using `sq1 = BK p1` `Position_KRK.attacks p1 sq1 sq2` on_square_BK[of p1] `legal_position p1` kings_separated_attacks_BKWK[of p1]
    unfolding Position_KRK.attacks_def legal_position_def
    by (auto simp add: king_scope_def)
  
  hence "\<not> WRcaptured p1 \<and> sq2 = WR p1"
    using occupies_white[of p1 sq2] `invar p1` `Position_KRK.occupies p1 White sq2`
    by auto

  hence "WK p1 \<noteq> sq1" "WR p1 \<noteq> sq1"
    using `invar p1` `sq1 = BK p1`
    unfolding invar_def
    by (auto simp add: WR_def WRcaptured_def)

  have "on_square p2 (WK p1) = Some (White, King)"
    using *[rule_format, of "WK p1"] `WK p1 \<noteq> sq1`  `WK p1 \<noteq> sq2` on_square_WK[of p1] `invar p1`
    by (simp split: split_if_asm)
  hence "WK p2 = WK p1"
    using on_square_WK'[of p2 "WK p1"] `invar p2`
    by simp

  have  "on_square p2 sq2 = Some (Black, King)"
    using *[rule_format, of sq2] `sq1 \<noteq> sq2` `sq1 = BK p1` on_square_BK[of p1] `invar p1`
    by simp
  hence "BK p2 = sq2"
    using on_square_BK'[of p2 sq2] `invar p2`
    by auto

  have "WRcaptured p1 \<longrightarrow> WRcaptured p2"
    using *[rule_format, of "WR p2"] on_square_WR[of p2] `invar p2` `sq1 = BK p1` on_square_BK[of p1] `invar p1` on_square_WR'[of p1 "WR p2"]
    by (auto split: split_if_asm)

  have "WRcaptured p2"
    using *[rule_format, of "WR p2"] `\<not> WRcaptured p1 \<and> sq2 = WR p1` on_square_WR[of p2] `invar p2` `sq1 = BK p1` on_square_BK[of p1] `invar p1` on_square_WR'[of p1 "WR p2"]
    by (auto split: split_if_asm)

  have "p2 = moveBK p1 sq2"
  proof (rule moveBK_eqI)
    show "BK p2 = sq2" by fact
  next
    show "WK p2 = WK p1" by fact
  next
    show "WRopt p2 = (if sq2 = WR p1 then None else WRopt p1)"
      using `WRcaptured p2` `\<not> WRcaptured p1 \<and> sq2 = WR p1`
      by (auto simp add: WRcaptured_def)
  next
    show "WhiteOnTurn p2" by fact
  qed
  
  hence "legal_move_BK p1 p2"
    using `BlackOnTurn p1` `legal_position p1` `legal_position p2`
    using `Position_KRK.attacks p1 sq1 sq2` `sq1 = BK p1` on_square_BK[of p1] `invar p1`
    unfolding legal_move_BK_def' Position_KRK.attacks_def
    by (simp split: split_if_asm) (rule_tac x="fst sq2" in exI, rule_tac x="snd sq2" in exI, simp)
  thus ?thesis
    by simp
qed

(* ------------------------------------------------------------- *)
theorem legal_move_black:
  assumes "Position_KRK.legal_move p1 p2" "to_move p1 = Black"
  shows "legal_move_BK p1 p2"
(* ------------------------------------------------------------- *)
using assms
unfolding Position_KRK.legal_move_def
by (auto simp add: to_move_def legal_position  simple_move_black capture_move_black split: split_if_asm)


(* ************************************************************************** *)
text{* _can_move_to: check if a move to the given square will be legal*}
(* ************************************************************************** *)

lemma legal_position_moveBK:
  assumes "legal_position p" "BlackOnTurn p" "BK_can_move_to p sq"
  shows "legal_position (moveBK p sq)"
using assms
unfolding legal_position_def white_attacks_nobk_def invar_def BK_can_move_to_def
by (auto simp add: kings_separated_def king_scope_sym all_on_board_def WR_attacks_nobk_def)

lemma BK_can_move_to:
  assumes "legal_position p" "BlackOnTurn p"
  shows "legal_move_BK p (moveBK p sq) \<longleftrightarrow> BK_can_move_to p sq"
proof
  assume "legal_move_BK p (moveBK p sq)"
  thus "BK_can_move_to p sq"
    using assms
    unfolding BK_can_move_to_def legal_move_BK_def
    apply (cases sq)
    apply (auto simp add: legal_position_def all_on_board_def kings_separated_def WRcaptured_def king_scope_sym invar_def split: split_if_asm)
    apply metis
    apply (simp add: white_attacks_nobk_def WR_attacks_nobk_def WRcaptured_def WR_def rook_scope_def, 
          metis option.sel (* the.simps*) )
    apply (simp_all add: white_attacks_nobk_def WR_attacks_nobk_def)
    done
next
  assume "BK_can_move_to p sq"
  thus "legal_move_BK p (moveBK p sq)"
    using assms legal_position_moveBK[of p sq]
    unfolding legal_move_BK_def
    by (simp add: BK_can_move_to_def)
qed

lemma no_legal_move_BK_iff':
  assumes "legal_position p" "BlackOnTurn p"
  shows "\<not> (\<exists> p'. legal_move_BK p p') \<longleftrightarrow> (\<forall> sq. king_scope (BK p) sq \<and> board sq \<longrightarrow> white_attacks_nobk p sq)" (is "?lhs = ?rhs")
proof
  assume "?rhs"
  thus "?lhs"
    using assms
    unfolding legal_move_BK_def'
    apply auto
    apply (erule_tac x="a" in allE, erule_tac x="b" in allE)
    apply (auto simp add: legal_position_def all_on_board_def WR_def WRcaptured_def WR_attacks_nobk_def invar_def kings_separated_def white_attacks_nobk_def king_scope_sym rook_scope_def split: split_if_asm)
    done
next
  assume "?lhs"
  thus "?rhs"
    using assms legal_position_moveBK[of p]
    by (auto simp add: legal_move_BK_def' BK_can_move_to_def)
qed

lemma  no_legal_move_BK_iff:
  assumes "legal_position p" "BlackOnTurn p"
  shows "\<not> (\<exists> p'. legal_move_BK p p') \<longleftrightarrow> all_king_pos (\<lambda> sq. board sq \<longrightarrow> white_attacks_nobk p sq) (BK p)"
using assms
apply (subst no_legal_move_BK_iff', simp_all add: all_king_pos_def king_scope_iff)
apply (cases "BK p", simp, smt)
done

lemma legal_move_BK_iff':
  shows "(\<exists> p'. legal_move_BK p p') \<longleftrightarrow> legal_position p \<and> BlackOnTurn p \<and> (\<exists> sq. BK_can_move_to p sq)"
using no_legal_move_BK_iff'[of p]
by (auto simp add: legal_move_BK_def BK_can_move_to_def)

lemma WK_can_move_to:
  assumes "legal_position p" "WhiteOnTurn p"
  shows "legal_move_WK p (moveWK p sq) \<longleftrightarrow> WK_can_move_to p sq"
using assms
unfolding legal_move_WK_def
by (auto simp add: legal_position_def invar_def kings_separated_def all_on_board_def WRcaptured_def WR_def king_scope_sym WK_can_move_to_def)

lemma WR_can_move_to:
  assumes "legal_position p" "WhiteOnTurn p" "\<not> WRcaptured p"
  shows "legal_move_WR p (moveWR p sq) \<longleftrightarrow> WR_can_move_to p sq"
using assms
unfolding legal_move_WR_def
by (auto simp add: legal_position_def invar_def all_on_board_def kings_separated_def WR_can_move_to_def)

(* ************************************************************************** *)
text{* Properties of checkmated *}
(* ************************************************************************** *)

(* ------------------------------------------------------------- *)
theorem checkmated:
  "Position_KRK.checkmated p \<longleftrightarrow> checkmated p"
(* ------------------------------------------------------------- *)
unfolding checkmated_def BK_cannot_move_def
using white_in_check_iff[of p] black_in_check_iff[of p] legal_move_BK_legal_move[of p] no_legal_move_BK_iff[of p]
unfolding Position_KRK.checkmated_def Position_KRK.game_over_def legal_position legal_position_def all_on_board_def
apply (auto simp add: to_move_def)
using legal_move_black[of p]
apply (auto simp add: to_move_def)
done

lemma checkmated_no_legal_move:
  assumes "checkmated p" "legal_move_BK p p'"
  shows "False"
using assms
using KRK.checkmated[of p] KRK.legal_move_BK_legal_move[of p p']
unfolding Position_KRK.checkmated_def Position.game_over_def
by simp


(* ************************************************************************** *)
text{* Properties of stalemate *}
(* ************************************************************************** *)

lemma white_can_move:
  assumes "WhiteOnTurn p1" "legal_position p1" "files \<ge> 2" "ranks \<ge> 2"
  shows "\<exists> p2. legal_move_WK p1 p2 \<or> legal_move_WR p1 p2"
proof(cases "WRcaptured p1")
  case True
  hence "\<exists> p2. legal_move_WK p1 p2"
    using assms
    unfolding legal_move_WK_def'' legal_position_def all_on_board_def WRcaptured_def invar_def kings_separated_def
    apply (cases "WK p1", cases "BK p1")
    apply simp
    apply (case_tac "let (a', b') = (a-(1::int), b-(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a-(1::int)" in exI, rule_tac x="b-(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a-(1::int), b) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a-(1::int)" in exI, rule_tac x="b" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a-(1::int), b+(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a-(1::int)" in exI, rule_tac x="b+(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a, b-(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a" in exI, rule_tac x="b-(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a, b+(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a" in exI, rule_tac x="b+(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a+(1::int), b-(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a+(1::int)" in exI, rule_tac x="b-(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a+(1::int), b) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a+(1::int)" in exI, rule_tac x="b" in exI, force simp add: Let_def, simp add: Let_def)
    apply (case_tac "let (a', b') = (a+(1::int), b+(1::int)) in king_scope (a, b) (a', b') \<and> (a', b) \<noteq> (aa, ba) \<and> board (a', b') \<and> \<not> king_scope (a', b') (aa, ba)")
    apply (rule_tac x="a+(1::int)" in exI, rule_tac x="b+(1::int)" in exI, force simp add: Let_def, simp add: Let_def)
    apply (simp add: board_def king_scope_iff some_king_pos del: some_king_pos_def)
    apply smt
    done
  thus ?thesis
    by auto
next
  case False
  have "\<exists> p2. legal_move_WR p1 p2"
  proof-
    obtain WRx WRy BKx BKy WKx WKy where "WR p1 = (WRx, WRy)" "BK p1 = (BKx, BKy)" "WK p1 = (WKx, WKy)"
      by (cases "WR p1", cases "BK p1", cases "WK p1", force)
    thus ?thesis
      using assms `\<not> WRcaptured p1`
      unfolding legal_move_WR_def'' legal_position_def all_on_board_def WRcaptured_def WR_def kings_separated_def invar_def
      apply simp
      apply (case_tac "let (a', b') = (WRx-(1::int), WRy) in (a', b') \<noteq> WK p1 \<and> board (a', b')")
      apply (rule_tac x="WRx-(1::int)" in exI, rule_tac x="WRy" in exI)
      apply (simp add: Let_def WR_attacks_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (force simp add: WR_attacks_nobk_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (case_tac "let (a', b') = (WRx, WRy-(1::int)) in (a', b') \<noteq> WK p1 \<and> board (a', b')")
      apply (rule_tac x="WRx" in exI, rule_tac x="WRy-(1::int)" in exI)
      apply (simp add: Let_def WR_attacks_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (force simp add: WR_attacks_nobk_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (case_tac "let (a', b') = (WRx+(1::int), WRy) in (a', b') \<noteq> WK p1 \<and> board (a', b')")
      apply (rule_tac x="WRx+(1::int)" in exI, rule_tac x="WRy" in exI)
      apply (simp add: Let_def WR_attacks_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (force simp add: WR_attacks_nobk_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (case_tac "let (a', b') = (WRx, WRy+(1::int)) in (a', b') \<noteq> WK p1 \<and> board (a', b')")
      apply (rule_tac x="WRx" in exI, rule_tac x="WRy+(1::int)" in exI)
      apply (simp add: Let_def WR_attacks_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (force simp add: WR_attacks_nobk_def WRcaptured_def WR_def rook_scope_def same_file_def same_rank_def square_between_def between_def board_def)
      apply (simp add: Let_def board_def)
      apply smt
      done
  qed
  thus ?thesis
    by auto
qed

(* ------------------------------------------------------------- *)
theorem stalemate:
  assumes "files \<ge> 2" "ranks \<ge> 2"
  shows "Position_KRK.stalemate p \<longleftrightarrow> stalemate p"
(* ------------------------------------------------------------- *)
using assms white_can_move[of p] legal_move_WK_legal_move[of p]  legal_move_WR_legal_move[of p] legal_move_BK_legal_move[of p]
black_in_check_iff[of p] no_legal_move_BK_iff[of p]
unfolding Position_KRK.stalemate_def stalemate_def Position_KRK.game_over_def legal_position legal_position_def all_on_board_def BK_cannot_move_def
apply (auto simp add: to_move_def)
using legal_move_black[of p]
apply (auto simp add: to_move_def, force+)
done

(* ************************************************************************** *)
text{* Properties of draw *}
(* ************************************************************************** *)

lemma legal_move_WRcaptured:
  assumes "Position_KRK.legal_move p p'" "WRcaptured p"
  shows "WRcaptured p'"
using assms
using legal_move_white[of p p'] legal_move_black[of p p']
by (cases "to_move p") (auto simp add: legal_move_white_def legal_move_WK_def' legal_move_WR_def' legal_move_BK_def' WR_attacks_def)

lemma legal_move_chain_WRcaptured:
  assumes "p' \<in> Position_KRK.play p" "WRcaptured p"
  shows "WRcaptured p'"
using assms
by (induct rule: Position_KRK.play.induct) (auto simp add: legal_move_WRcaptured)

(* ------------------------------------------------------------- *)
theorem WRcaptured_draw:
  assumes "WRcaptured p" 
  shows "Position_KRK.draw p"
(* ------------------------------------------------------------- *)
proof (rule ccontr)
  assume "\<not> ?thesis"
  then obtain p' where "p' \<in> Position_KRK.play p" "Position_KRK.checkmated p'"
    unfolding Position_KRK.draw_def
    by auto
  hence "WRcaptured p'"
    using legal_move_chain_WRcaptured assms
    by simp
  thus False
    using `Position_KRK.checkmated p'`
    using checkmated[of p']
    unfolding checkmated_def WR_attacks_BK_def WR_attacks_nobk_def
    by simp
qed

end
