(* ************************************************************************** *)
section{* General chess rules *}
(* ************************************************************************** *)

theory ChessRules
imports Main
begin

(* -------------------------------------------------------------------------- *)
subsection{* Formalization of pawnless chess rules *}
(* -------------------------------------------------------------------------- *)

text{*
In this section, we formalize chess rules as given in:
J. Hurd. Formal verification of chess endgame databases. In Theorem Proving in Higher Order Logics:
Emerging Trends, Oxford University CLR Report, 2005.
We consider pawnless endgames without castling.
*}

text{* Chess is a game between two players: black and white*}
datatype side = White | Black
primrec opponent where
  "opponent White = Black"
| "opponent Black = White"

text{* Pieces on the board - pawns are not considered *}
datatype piece = King | Queen | Rook | Bishop | Knight

text{* Board is a collection of squares, each given by its integer coordinates *}
type_synonym square = "int \<times> int"
text{* The first coordinate is called the file, and the second one is called the rank *}
primrec "file" :: "square \<Rightarrow> int" where
  "file (f, r) = f"
primrec rank :: "square \<Rightarrow> int" where 
  "rank (f, r) = r"

text{* Dimensions of the chessboard *}

(*
text{* 8\<times>8 chessboard *}
definition files :: int where 
  "files = 8"
definition ranks :: int where
  "ranks = 8"
lemma files_eq_ranks: "files = ranks"
  by (simp add: files_def ranks_def)
lemma files_ranks_geq6: "files \<ge> 6" "ranks \<ge> 6"
  by (simp_all add: files_def ranks_def)
*)

text{* n\<times>n chessboard - we consider only square shaped boards greater than 6\<times>6 *}
consts files :: int
consts ranks :: int
axiomatization where
 files_eq_ranks: "files = ranks" and           (* board is square shaped *)
 files_ranks_geq6: "files \<ge> 6 \<and> ranks \<ge> 6"    (* dimension is at least 6 *)

text{* Test if the given square is within the chessboard bounds *}
definition board :: "square \<Rightarrow> bool" where 
  "board sq \<longleftrightarrow> 0 \<le> file sq \<and> file sq < files \<and> 0 \<le> rank sq \<and> rank sq < ranks"

text{* Tests if two given squares are colinear (horizontally, vertically, or diagonally) *}

definition same_file :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_file sq sq' \<longleftrightarrow> file sq = file sq'"
definition same_rank :: "square \<Rightarrow> square \<Rightarrow> bool" where 
  "same_rank sq sq' \<longleftrightarrow> rank sq = rank sq'"
definition same_diag1 :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag1 sq sq' \<longleftrightarrow> file sq + rank sq = file sq' + rank sq'"
definition same_diag2 :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag2 sq sq' \<longleftrightarrow> file sq + rank sq' = file sq' + rank sq"
definition same_diag :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "same_diag sq1 sq2 \<longleftrightarrow> same_diag1 sq1 sq2 \<or> same_diag2 sq1 sq2"

text{* Absolute difference of square coordinates -
       auxiliary functions used to define piece scopes *}

definition diff :: "int \<Rightarrow> int \<Rightarrow> int" where 
  "diff m n = (if m \<le> n then n - m else m - n)"
definition file_diff :: "square \<Rightarrow> square \<Rightarrow> int" where
  "file_diff sq sq' = diff (file sq) (file sq')"
definition rank_diff :: "square \<Rightarrow> square \<Rightarrow> int" where
  "rank_diff sq sq' = diff (rank sq) (rank sq')"

text{* Scopes of pieces on an empty board *}

definition bishop_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "bishop_scope sq1 sq2 \<longleftrightarrow> same_diag sq1 sq2 \<and> sq1 \<noteq> sq2"
definition rook_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "rook_scope sq1 sq2 \<longleftrightarrow> (same_file sq1 sq2 \<or> same_rank sq1 sq2) \<and> sq1 \<noteq> sq2"
definition queen_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "queen_scope sq1 sq2 \<longleftrightarrow> bishop_scope sq1 sq2 \<or> rook_scope sq1 sq2"
definition knight_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "knight_scope sq1 sq2 \<longleftrightarrow> (file_diff sq1 sq2 = 1 \<and> rank_diff sq1 sq2 = 2) \<or>
                            (file_diff sq1 sq2 = 2 \<and> rank_diff sq1 sq2 = 1)"
definition king_scope :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "king_scope sq1 sq2 \<longleftrightarrow> file_diff sq1 sq2 \<le> 1 \<and> rank_diff sq1 sq2 \<le> 1 \<and> sq1 \<noteq> sq2"

text{* Betweenness of numbers and betweenness of squares *} 
definition between where
  "between n1 n n2 \<longleftrightarrow> (n1 < n \<and> n < n2) \<or> (n2 < n \<and> n < n1)"
definition square_between where
  "square_between sq1 sq sq2 \<longleftrightarrow> 
     (if same_file sq1 sq2 then same_file sq sq1 \<and> between (rank sq1) (rank sq) (rank sq2)
      else if same_rank sq1 sq2 then same_rank sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else if same_diag1 sq1 sq2 then same_diag1 sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else if same_diag2 sq1 sq2 then same_diag2 sq sq1 \<and> between (file sq1) (file sq) (file sq2)
      else False)"

text{* A chess position is determined by knowing:
       the side that makes the next move and 
       the arrangement of pieces on the board. *}
locale Position = 
  (* the side that makes the next move *)
  fixes to_move :: "'p \<Rightarrow> side"
  (* a piece on the given square (or None if the square is empty) *)
  fixes on_square :: "'p \<Rightarrow> square \<Rightarrow> (side \<times> piece) option"
  (* an invariant on the datatype that represents the chessboard position *)
  fixes inv :: "'p \<Rightarrow> bool"
begin

(* Test if the given square is empty *)
definition empty :: "'p \<Rightarrow> square \<Rightarrow> bool" where
  "empty p sq \<longleftrightarrow> on_square p sq = None"

(* Test if the given square is occupied by a piece of the given side *)
definition occupies :: "'p \<Rightarrow> side \<Rightarrow> square \<Rightarrow> bool" where
  "occupies p s sq \<longleftrightarrow> (\<exists> v. on_square p sq = Some (s, v))"

(* Test if the line between the two given squares is empty - 
   always true if the squares are not collinear *)
definition clear_line :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "clear_line p sq1 sq2 \<longleftrightarrow> (\<forall> sq. square_between sq1 sq sq2 \<longrightarrow> empty p sq)"

(* Test if a piece on the first given square attacks the piece on the second given square *)
definition attacks :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "attacks p sq sq' \<longleftrightarrow> clear_line p sq sq' \<and>
     (case on_square p sq of
          None \<Rightarrow> False
       | Some (_, King) \<Rightarrow> king_scope sq sq'
       | Some (_, Queen) \<Rightarrow> queen_scope sq sq'
       | Some (_, Rook) \<Rightarrow> rook_scope sq sq'
       | Some (_, Bishop) \<Rightarrow> bishop_scope sq sq'
       | Some (_, Knight) \<Rightarrow> knight_scope sq sq'
     )"

(* Test if the given side is in check *)
definition in_check :: "side \<Rightarrow> 'p \<Rightarrow> bool" where
  "in_check s p \<longleftrightarrow> 
     (\<exists> sq1 sq2. on_square p sq1 = Some (s, King) \<and> occupies p (opponent s) sq2 \<and> attacks p sq2 sq1)"

(* Test if all pieces are within the board bounds *)
definition all_on_board :: "'p \<Rightarrow> bool" where
  "all_on_board p \<longleftrightarrow> (\<forall> sq. \<not> empty p sq \<longrightarrow> board sq)"

(* Test if the given position is legal *)
definition legal_position :: "'p \<Rightarrow> bool" where
  "legal_position p \<longleftrightarrow> inv p \<and> all_on_board p \<and> \<not> in_check (opponent (to_move p)) p"

(* Test if a non-capturing move can be made by a piece on the first given square to the second given square *)
definition sorties :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "sorties p sq sq' \<longleftrightarrow> attacks p sq sq' \<and> empty p sq'"

(* Test if a capturing move can be made by a piece on the first given square to the second given square *)
definition captures  :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> bool" where
  "captures p sq sq' \<longleftrightarrow> attacks p sq sq' \<and> occupies p (opponent (to_move p)) sq'"

(* Test if the second given position is obtained from the first one by moving the piece on the first given
   square to the second given square *)
definition moved :: "'p \<Rightarrow> square \<Rightarrow> square \<Rightarrow> 'p \<Rightarrow> bool" where
  "moved p sq1 sq2 p' \<longleftrightarrow> 
      (\<forall> sq. on_square p' sq = 
              (if sq = sq1 then None
               else if sq = sq2 then on_square p sq1
               else on_square p sq))"

(* Test if the second given position is obtained from the first one by a non-capturing move *)
definition simple_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "simple_move p p' \<longleftrightarrow> 
    (\<exists> sq1 sq2. occupies p (to_move p) sq1 \<and> sorties p sq1 sq2 \<and> moved p sq1 sq2 p')"

(* Test if the second given position is obtained from the first one by a capturing move *)
definition capture_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
 "capture_move p p' \<longleftrightarrow> 
  (\<exists> sq1 sq2. occupies p (to_move p) sq1 \<and> captures p sq1 sq2 \<and> moved p sq1 sq2 p')"

(* Test if the second given position is obtained from the first one by a legal move *)
definition legal_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "legal_move p p' \<longleftrightarrow>
     legal_position p \<and> legal_position p' \<and>
     (to_move p' = opponent (to_move p)) \<and>
     (simple_move p p' \<or> capture_move p p')"

(* The game is over if no legal move can be made from a legal position *)
definition game_over :: "'p \<Rightarrow> bool" where
  "game_over p \<longleftrightarrow> legal_position p \<and> \<not> (\<exists> p'. legal_move p p')"
(* Test if the player on the move is checkmated - in check and cannot move *)
definition checkmated :: "'p \<Rightarrow> bool" where
  "checkmated p \<longleftrightarrow> game_over p \<and> in_check (to_move p) p"
(* Test if the player on the move is stalemated - not in check and cannot move *)
definition stalemate :: "'p \<Rightarrow> bool" where
  "stalemate p \<longleftrightarrow>  game_over p \<and> \<not> in_check (to_move p) p"

(* A play is a sequence of legal moves starting from some position p0 *)
inductive_set play for p0 :: "'p" where
  init: "p0 \<in> play p0"
| step: "\<lbrakk>p \<in> play p0; legal_move p p'\<rbrakk> \<Longrightarrow> p' \<in> play p0"

(* The game is drawn if no player can checkmate the other one *)
definition draw :: "'p \<Rightarrow> bool" where
  "draw p \<longleftrightarrow> (\<not> (\<exists> p'. p' \<in> play p \<and> checkmated p'))"
end

(* -------------------------------------------------------------------------- *)
subsection{* Some properties of chess rules *}
(* -------------------------------------------------------------------------- *)

lemma king_scope_sym: 
  "king_scope sq1 sq2 \<longleftrightarrow> king_scope sq2 sq1"
unfolding king_scope_def
by (auto simp add: file_diff_def rank_diff_def diff_def)

(* Test if all squares that king can move to satisfy the given predicate *)
definition all_king_pos :: "(square \<Rightarrow> bool) \<Rightarrow> square \<Rightarrow> bool" where
  "all_king_pos P sq \<longleftrightarrow> 
      (let (x, y) = sq in 
         P (x-(1::int), y-(1::int)) \<and>
         P (x-(1::int), y) \<and>
         P (x-(1::int), y+(1::int)) \<and>
         P (x, y-(1::int)) \<and>
         P (x, y+(1::int)) \<and>
         P (x+(1::int), y-(1::int)) \<and>
         P (x+(1::int), y) \<and>
         P (x+(1::int), y+(1::int))
      )"

(* Test if some square that king can move to satisfies the given predicate *)
definition some_king_pos where
  [simp]: "some_king_pos P sq \<longleftrightarrow> \<not> all_king_pos (\<lambda> x. \<not> P x) sq"

(* Explicit characterization of the previous test *)
lemma some_king_pos:
"some_king_pos P sq \<longleftrightarrow> 
      (let (x, y) = sq in 
         P (x-(1::int), y-(1::int)) \<or>
         P (x-(1::int), y) \<or>
         P (x-(1::int), y+(1::int)) \<or>
         P (x, y-(1::int)) \<or>
         P (x, y+(1::int)) \<or>
         P (x+(1::int), y-(1::int)) \<or>
         P (x+(1::int), y) \<or>
         P (x+(1::int), y+(1::int))
      )"
unfolding some_king_pos_def all_king_pos_def Let_def
by auto

lemma king_scope_iff: 
  "king_scope s1 s2 \<longleftrightarrow> some_king_pos (\<lambda> x. x = s2) s1"
unfolding some_king_pos
apply (cases s1, cases s2, simp)
unfolding king_scope_def file_diff_def rank_diff_def diff_def
by auto

lemma ex_king_scope:
  shows "(\<exists>a b. king_scope (x, y) (a, b) \<and> P (a, b)) \<longleftrightarrow> 
         (P (x - 1, y - 1) \<or> P (x - 1, y) \<or> P (x - 1, y + 1) \<or>
          P (x, y - 1) \<or> P (x, y + 1) \<or>
          P (x + 1, y - 1) \<or> P (x + 1, y) \<or> P (x + 1, y + 1))"
by (auto simp add: king_scope_iff all_king_pos_def)

lemma [simp]: "\<not> square_between sq sq sq'"
by (simp add: square_between_def between_def)

lemma [simp]: "\<not> square_between sq' sq sq"
by (simp add: square_between_def between_def)

lemma (in Position) king_scope_clear_line:
  assumes "king_scope sq1 sq2"
  shows "clear_line p sq1 sq2"
using assms
unfolding king_scope_def Position.clear_line_def square_between_def between_def
by (cases sq1, cases sq2) (auto simp add: file_diff_def rank_diff_def diff_def)

lemma (in Position) play_trans:
  assumes "p2 \<in> play p1" "p1 \<in> play p0"
  shows "p2 \<in> play p0"
using assms
proof (induct rule: play.induct)
  case (step p p')
  thus ?case
    by (auto simp add: play.step)
qed

lemma (in Position) play_step_inverse:
  assumes
   "q \<in> play p'" "legal_move p p'"
  shows "q \<in> play p"
using assms
by (induct rule: play.induct) (metis play.init play.step, metis play.step)

(* -------------------------------------------------------------------------- *)
subsection{* Strategies for the white player *}
(* -------------------------------------------------------------------------- *)

text{*
A strategy for the white player is a relation determining its possible "good" moves
in a given position. Positions where the strategy is applicable are characterized by
some invariant that strategy must preserve. It must also be preserved after a strategy
move by white followed by an arbitrary legal move of black.
*}

locale Strategy = Position to_move for to_move :: "'p \<Rightarrow> side" + 
  (* Move of white, according to the strategy *)
  fixes strategy_white_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool"
  (* Invariant that strategy moves must preserve *)
  fixes strategy_inv :: "'p \<Rightarrow> bool"
  (* Strategy must make only legal moves *)
  assumes strategy_white_move_legal_move:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_inv p0; strategy_white_move p0 p1\<rbrakk> \<Longrightarrow> legal_move p0 p1"
  (* Strategy preserves the invariant *)
  assumes strategy_white_move_inv:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_inv p0; strategy_white_move p0 p1\<rbrakk> \<Longrightarrow> strategy_inv p1"
  assumes strategy_white_move_black_move_inv:
      "\<lbrakk>legal_position p0; to_move p0 = White; strategy_inv p0; strategy_white_move p0 p1; legal_move p1 p2\<rbrakk> \<Longrightarrow> strategy_inv p2"
begin

(* Legal positions that satisfy the strategy invariant *)
definition strategy_white_pos where 
  "strategy_white_pos p \<longleftrightarrow> to_move p = White \<and> legal_position p \<and> strategy_inv p"
definition strategy_black_pos where 
  "strategy_black_pos p \<longleftrightarrow> to_move p = Black \<and> legal_position p \<and> strategy_inv p"

(* Strategy move (ply): move of white, according to some strategy, or any legal move of black *)
definition strategy_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "strategy_move p p' \<longleftrightarrow> (to_move p = White \<and> strategy_white_move p p') \<or> 
                          (to_move p = Black \<and> legal_move p p')"

(* Strategy move of white, followed by any legal move of black *)
definition strategy_full_move :: "'p \<Rightarrow> 'p \<Rightarrow> bool" where
  "strategy_full_move p p' \<longleftrightarrow> (\<exists> p''. to_move p = White \<and> strategy_white_move p p'' \<and> legal_move p'' p')"

(* A play is a sequence of strategy moves by white/legal moves by black. *)
inductive_set strategy_play for p0 :: 'p
where
  init: "p0 \<in> strategy_play p0"
| step: "\<lbrakk>p \<in> strategy_play p0; strategy_move p p'\<rbrakk> \<Longrightarrow> p' \<in> strategy_play p0"

end

(* -------------------------------------------------------------------------- *)
subsubsection{* Winning strategies for white player *}
(* -------------------------------------------------------------------------- *)

text{* A strategy for white is a winning strategy if it always terminates in a position where the black is checkmated *}

locale WinningStrategy = Strategy on_square for on_square :: "'p \<Rightarrow> int \<times> int \<Rightarrow> (side \<times> piece) option" + 
  (* Strategy must always terminate - there is no infinite set of strategy moves *)
  assumes strategy_terminates: 
    "strategy_white_pos p0 \<Longrightarrow> \<not> (\<exists> Q. p0 \<in> Q \<and> (\<forall> p \<in> Q. \<exists> p' \<in> Q. strategy_move p p'))"
  (* When the strategy terminates, it is in a position where the white wins *)
  assumes white_wins:
     "\<lbrakk>strategy_white_pos p0; p \<in> strategy_play p0; \<not> (\<exists> p'. strategy_move p p')\<rbrakk> \<Longrightarrow> to_move p = Black \<and> checkmated p"


(* -------------------------------------------------------------------------- *)
subsubsection{* Some properties of strategies for white *}
(* -------------------------------------------------------------------------- *)

context Strategy
begin

lemma strategy_move_white:
  assumes "strategy_move p p'" "to_move p = White"
  shows "strategy_white_move p p'"
using assms
by (auto simp add: strategy_move_def)

lemma strategy_move_black:
  assumes "strategy_move p p'" "to_move p = Black"
  shows "legal_move p p'"
using assms
by (auto simp add: strategy_move_def)

lemma strategy_play_step_inverse:
  assumes
   "q \<in> strategy_play p'" "strategy_move p p'"
  shows "q \<in> strategy_play p"
using assms
by (induct rule: strategy_play.induct) (metis strategy_play.init strategy_play.step, metis strategy_play.step)

lemma strategy_play_trans:
  assumes  "p2 \<in> strategy_play p1" "p1 \<in> strategy_play p0"
  shows "p2 \<in> strategy_play p0"
using assms
proof (induct rule: strategy_play.induct)
  case (step p p')
  thus ?case
    by (metis strategy_play.step)
qed simp

thm strategy_white_pos_def

(* Any play from a legal white starting position (legal position, satisfying the invariant, with the white on turn) 
   results in:
    - a legal position
    - that satisfies the invariant, and
    - if the black is on turn, the last move was a strategy move by white
*)
lemma strategy_play:
  assumes "p \<in> strategy_play p0" "strategy_white_pos p0"
  shows "legal_position p \<and> 
         strategy_inv p \<and>
         (to_move p = Black \<longrightarrow> (\<exists> px. px \<in> strategy_play p0 \<and> strategy_white_pos px \<and> strategy_white_move px p))"
using assms
proof (induct rule: strategy_play.induct)
  case (step p p')
  show ?case
  proof (cases "to_move p = White")
    case True
    thus ?thesis
      using step strategy_move_white[of p p'] strategy_white_move_legal_move[of p p'] strategy_white_move_inv[of p p'] strategy_move_white[of p p']
      by (auto simp add: legal_move_def strategy_white_pos_def)
  next
    case False
    hence "to_move p = Black"
      by (metis side.exhaust)
    then obtain px where "px \<in> strategy_play p0" "legal_position px" "strategy_inv px" "to_move px = White"  "strategy_white_move px p" 
      using step
      by (auto simp add: strategy_white_pos_def)
    thus ?thesis
      using step `to_move p = Black`
      using `strategy_move p p'` strategy_move_black[of p p'] strategy_white_move_black_move_inv[of px p p']
      by (simp add: legal_move_def)
  qed
qed (simp add: strategy_white_pos_def)

(* strategy play results in a legal position *)
lemmas strategy_play_legal = strategy_play[THEN conjunct1]
(* strategy play results in a position that satisfies the invariant *)
lemmas strategy_play_inv = strategy_play[THEN conjunct2, THEN conjunct1]
(* if strategy play results in a position such that black is on turn, the last move was a strategy move by white *)
lemmas strategy_play_black = strategy_play[THEN conjunct2, THEN conjunct2, rule_format]

(* play of strategic moves is also a play of legal moves *)
lemma strategy_play_play:
  assumes "strategy_white_pos p0" "p \<in> strategy_play p0"
  shows "strategy_play p \<subseteq> play p"
proof
  fix x
  assume "x \<in> strategy_play p"
  thus "x \<in> play p"
    using assms
  proof (induct rule: strategy_play.induct)
    case init
    thus ?case
      by (simp add: play.init)
  next
    case (step p1 p2)
    thus ?case
      using play.step[of p1 p p2]
      using strategy_white_move_legal_move[of p1 p2]
      using strategy_move_black[of p1 p2]
      apply (cases "to_move p1")
      apply auto
      apply (subgoal_tac "p1 \<in> strategy_play p0")
      apply (metis strategy_move_def strategy_play)
      apply (metis strategy_play_trans)
      done
  qed
qed

end

(* -------------------------------------------------------------------------- *)
subsubsection{* Some properites of winning strategies for white *}
(* -------------------------------------------------------------------------- *)

context WinningStrategy
begin

(* A winning strategic play always ends *)
theorem ex_game_over:
  assumes "strategy_white_pos p0"
  shows "\<exists> p \<in> strategy_play p0. game_over p"
proof (rule ccontr)
  assume *: "\<not> ?thesis"
  hence "\<forall>p\<in>strategy_play p0. Ex (legal_move p)"
    using assms
    unfolding game_over_def
    by (metis strategy_play)
  let ?Q = "strategy_play p0"
  have "p0 \<in> ?Q \<and> (\<forall>p\<in>?Q. \<exists>p'\<in>?Q. strategy_move p p')"
    using assms *
    by (metis checkmated_def strategy_play.init strategy_play.step white_wins)
  thus False
    using strategy_terminates[OF assms]
    by blast
qed

(* Strategic play always leads to a position where the black is checkmated *)
lemma ex_mate':
  assumes "strategy_white_pos p0"
  shows "\<exists> p. p \<in> strategy_play p0 \<and> to_move p = Black \<and> checkmated p"
proof-
  from assms obtain p where
    "p \<in> strategy_play p0" "game_over p"
    using ex_game_over
    by auto
  thus ?thesis
    apply (rule_tac x="p" in exI)
    using white_wins[of p0 p] assms
    by (metis game_over_def strategy_move_def strategy_play_inv strategy_white_move_legal_move)
qed

theorem ex_mate:
  assumes "strategy_white_pos p0" "p \<in> strategy_play p0"
  shows "\<exists> p'. p' \<in> strategy_play p \<and> to_move p' = Black \<and> checkmated p'"
proof (cases "to_move p = White")
  case True
  hence "strategy_white_pos p"
    using assms
    by (metis strategy_play strategy_white_pos_def)
  thus ?thesis
    using ex_mate'
    by auto
next
  case False
  hence "to_move p = Black"
    by (cases "to_move p") auto
  show ?thesis
  proof (cases "game_over p")
    case True
    thus ?thesis
      using white_wins[of p0 p]
      by (metis False Position.game_over_def assms(1) assms(2) strategy_move_def strategy_play.init)
  next
    case False
    obtain px where
      *: "px \<in> strategy_play p0 \<and> strategy_white_pos px \<and> strategy_white_move px p"
      using strategy_play_black[of p p0]
      using assms `to_move p = Black`
      by (auto simp add: strategy_white_pos_def)
    obtain p1 where "legal_move p p1"
      using assms `\<not> game_over p`
      unfolding game_over_def
      by (metis strategy_play_legal)
    hence "strategy_white_pos p1"
      unfolding strategy_white_pos_def
      using * `to_move p = Black`
      by (metis legal_move_def opponent.simps(2) strategy_white_move_black_move_inv strategy_white_pos_def)
    then obtain py where "py \<in> strategy_play p1 \<and> to_move py = Black \<and> checkmated py"
      using ex_mate'[of p1] 
      by auto
    thus ?thesis   
      using `legal_move p p1` `to_move p = Black`
      by (rule_tac x="py" in exI) (metis strategy_move_def strategy_play_step_inverse)
  qed
qed

(* A winning strategy cannot result in a draw *)

theorem no_draw:
  assumes "strategy_white_pos p0" "p \<in> strategy_play p0"
  shows "\<not> draw p"
using ex_mate[OF assms] assms strategy_play_play[of p0]
unfolding draw_def 
by auto

end

(* -------------------------------------------------------------------------- *)
subsubsection{* Some sufficient conditions for a strategy to be winning *}
(* -------------------------------------------------------------------------- *)

text{*
A strategy is winning if there is a well founded ordering on positions so that each 
*}

locale WinningStrategy' = Strategy on_square for on_square :: "'p \<Rightarrow> int \<times> int \<Rightarrow> (side \<times> piece) option" + 
  (* ordering of positions - that can depend on a starting position *)
  fixes ordering :: "'p \<Rightarrow> ('p \<times> 'p) set"
  (* the ordering is well founded *)
  assumes wf_ordering:
     "strategy_white_pos p0 \<Longrightarrow> wf (ordering p0)"
  (* the position is decreased (wrt. the ordering) after every full strategy move
     (strategic white move, followed by a legal black move) *)
  assumes ordering:
     "\<lbrakk>strategy_white_pos p0; p \<in> strategy_play p0; to_move p = White; strategy_white_move p p'; legal_move p' p''\<rbrakk> \<Longrightarrow> 
        (p'', p) \<in> ordering p0"
  (* white can always make a move in a strategic play *)
  assumes white_can_move:
     "\<lbrakk>strategy_white_pos p0; p \<in> strategy_play p0; to_move p = White \<rbrakk> \<Longrightarrow> (\<exists> p'. strategy_white_move p p')"
  (* black is not stalemated in a strategic play *)
  assumes strategy_not_stalemate:
     "\<lbrakk>strategy_white_pos p0; p \<in> strategy_play p0; to_move p = White; strategy_white_move p p'\<rbrakk> \<Longrightarrow> \<not> stalemate p'"

(* The previous conditions are sufficient for a strategy to be winning *)
sublocale WinningStrategy' < WinningStrategy inv to_move strategy_white_move strategy_inv on_square
proof
  fix p0 p
  assume *: "strategy_white_pos p0" "p \<in> strategy_play p0" "\<not> (\<exists>p'. strategy_move p p')"
  have "to_move p = Black"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    hence "to_move p = White"
      by (metis side.exhaust)
    thus False
      using * white_can_move[of p0 p]
      unfolding strategy_move_def
      by auto
  qed
  moreover
  have "checkmated p"
  proof-
    have "\<not> (\<exists> p'. legal_move p p')"
      using *(3) `to_move p = Black`
      unfolding strategy_move_def
      by auto
    obtain px where "px \<in> strategy_play p0"  "legal_position px"  "strategy_inv px"  "to_move px = White"  "strategy_white_move px p"
      using strategy_play_black[OF *(2)] *(1) `to_move p = Black`
      by (auto simp add: strategy_white_pos_def)
    thus ?thesis
      using strategy_not_stalemate[rule_format, of p0 px] `\<not> (\<exists> p'. legal_move p p')` `strategy_white_pos p0` strategy_play_legal[OF `p \<in> strategy_play p0`]  
      by (auto simp add: checkmated_def stalemate_def game_over_def strategy_white_pos_def)
  qed
  ultimately
  show "to_move p = Black \<and> checkmated p"
    by simp
next
  fix p0
  assume "strategy_white_pos p0"
  show "\<not> (\<exists>Q. p0 \<in> Q \<and> (\<forall>p\<in>Q. \<exists>p'\<in>Q. strategy_move p p'))"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    then obtain Q where "p0 \<in> Q" and *: "\<forall> p\<in>Q. \<exists>p'\<in>Q. strategy_move p p'"
      by auto
    let ?Q = "Q \<inter> strategy_play p0"
    let ?Qw = "?Q \<inter> {p. to_move p = White}"
    have "p0 \<in> ?Qw"
      using  `p0 \<in> Q` `strategy_white_pos p0` strategy_play.init[of p0]
      by (auto simp add: strategy_white_pos_def)
    then obtain pmin where "pmin\<in>?Qw" and **: "\<forall>p. (p, pmin) \<in> ordering p0 \<longrightarrow> p \<notin> ?Qw"
      using wf_ordering[of p0] `strategy_white_pos p0`
      unfolding wf_eq_minimal
      by metis
    obtain p' where "p' \<in> ?Q" "strategy_move pmin p'"
      using * `pmin \<in> ?Qw`
      by (metis Int_iff strategy_play.step)
    have "strategy_white_move pmin p'"
      using `strategy_move pmin p'` `pmin \<in> ?Qw`
      using strategy_move_white
      by auto
    hence "legal_move pmin p'"
      using strategy_white_move_legal_move[of pmin p'] strategy_play[of pmin p0]
      using `strategy_white_pos p0` `pmin \<in> ?Qw`
      by (simp add: strategy_white_pos_def)
    hence "to_move p' = Black"
      using `pmin \<in> ?Qw`
      by (simp add: legal_move_def)
    obtain p'' where "p'' \<in> ?Q" "strategy_move p' p''"
      using * `p' \<in> ?Q`
      by (metis Int_iff strategy_play.step)
    hence "legal_move p' p''"
      using strategy_move_black `to_move p' = Black`
      by simp
    hence "p'' \<in> ?Qw"
      using `to_move p' = Black` `p'' \<in> ?Q`
      by (simp add: legal_move_def)
    have "(p'', pmin) \<in> ordering p0"
      using `strategy_white_move pmin p'` `legal_move p' p''`
      using ordering[of p0 pmin p' p''] `strategy_white_pos p0`
      using `pmin \<in> ?Qw`
      by simp
    hence "p'' \<notin> ?Qw"
      using **
      by auto
    thus False
      using `p'' \<in> ?Qw`
      by simp
  qed
qed

end
