(* ************************************************************************** *)
section{* Strategy for KRK game *}
(* ************************************************************************** *)
theory KRKStrategy
imports KRK Auxiliary Symmetry
begin

(* -------------------------------------------------------------------------- *)
subsection{* Auxiliary notions for strategy definition *}
(* -------------------------------------------------------------------------- *)

(* Chebyshev distance - minimal number of kings moves from square to square *)
fun chebyshev_dist :: "square \<Rightarrow> square \<Rightarrow> int" where
  "chebyshev_dist (x1, y1) (x2, y2) = max (abs (x1 - x2)) (abs (y1 - y2))"
declare chebyshev_dist.simps [simp del]

lemma chebyshev_dist:
  "chebyshev_dist s1 s2 = 
     (let (s1x, s1y) = s1; 
          (s2x, s2y) = s2
       in chebyshev_dist (s1x, s1y) (s2x, s2y))"
by (cases s1, cases s2, simp add: chebyshev_dist.simps)

lemma chebyshev_dist_cases: 
  "chebyshev_dist (x1, y1) (x2, y2) = 
       (if x2 \<ge> x1 then 
          if y2 \<ge> y1 then max (x2-x1) (y2-y1) else max (x2-x1) (y1-y2)
        else
          if y2 \<ge> y1 then max (x1-x2) (y2-y1) else max (x1-x2) (y1-y2))"
by (simp add: chebyshev_dist.simps)

lemma king_scope_chebyshev_dist:
 "ChessRules.king_scope sq1 sq2 \<longleftrightarrow> chebyshev_dist sq1 sq2 = 1"
by (cases "sq1", cases "sq2") (simp add: king_scope_iff all_king_pos_def chebyshev_dist_cases, smt)

lemma chebyshev_dist_pos:
  assumes "sq1 \<noteq> sq2"
  shows "chebyshev_dist sq1 sq2 > 0"
  using assms
  by (cases sq1, cases sq2) (auto simp add: chebyshev_dist.simps max_def)

(* --------------------------------------------------------------- *)

(* Manhattan distance - only horizontal and vertical kings moves are allowed *)

fun manhattan_dist :: "square \<Rightarrow> square \<Rightarrow> int" where
 "manhattan_dist (x1, y1) (x2, y2) = abs (x1 - x2) + abs (y1 - y2)"
declare manhattan_dist.simps[simp del]

lemma manhattan_dist_cases: 
  "manhattan_dist (x1, y1) (x2, y2) = 
     (if x2 \<ge> x1 then
        if y2 \<ge> y1 then (x2-x1)+(y2-y1) else (x2-x1)+(y1-y2)
      else
        if y2 \<ge> y1 then (x1-x2)+(y2-y1) else (x1-x2)+(y1-y2))"
by (simp add: manhattan_dist.simps)

(* --------------------------------------------------------------- *)

(* Room - size of the space in which BK is constrained by the WR
   The size is measured by its half-perimeter instead of area.  *)

fun room' :: "square \<Rightarrow> square \<Rightarrow> int" where 
"room' (xr, yr) (xk, yk) = 
   (if xr = xk \<or> yr = yk then files + ranks - 1
    else let x = if xr > xk then xr else files - 1 - xr;
             y = if yr > yk then yr else ranks - 1 - yr
          in x + y)"

lemma room':
 "room' R K = 
    (let (xr, yr) = R; (xk, yk) = K in
     (if xr = xk \<or> yr = yk then files + ranks - 1
      else let x = if xr > xk then xr else files - 1 - xr;
               y = if yr > yk then yr else ranks - 1 - yr
            in x + y))"
by (cases R, cases K, simp)

lemma 
  "\<lbrakk>xr > xk; yr > yk\<rbrakk> \<Longrightarrow> room' (xr, yr) (xk, yk) = xr + yr"
  "\<lbrakk>xr > xk; yr < yk\<rbrakk> \<Longrightarrow> room' (xr, yr) (xk, yk) = xr + ranks - 1 - yr"
  "\<lbrakk>xr < xk; yr > yk\<rbrakk> \<Longrightarrow> room' (xr, yr) (xk, yk) = files - 1 - xr + yr"
  "\<lbrakk>xr < xk; yr < yk\<rbrakk> \<Longrightarrow> room' (xr, yr) (xk, yk) = files - 1 - xr + ranks - 1 - yr"
  "\<lbrakk>xr = xk \<or> yr = yk\<rbrakk> \<Longrightarrow> room' (xr, yr) (xk, yk) = files + ranks - 1"
unfolding room'
by simp_all

definition room :: "KRKPosition \<Rightarrow> int"  where 
  [simp]: "room p = room' (WR p) (BK p)"

(* --------------------------------------------------------------- *)
definition critical_square :: "KRKPosition \<Rightarrow> square" where
  "critical_square p =
   (let (xr, yr) = WR p; (xk, yk) = BK p;
         x = (if xr = xk then xr else if xr > xk then xr - 1 else xr + 1);
         y = (if yr = yk then yr else if yr > yk then yr - 1 else yr + 1)
      in (x, y))"

(* --------------------------------------------------------------- *)

(* WR is exposed if BK can reach it before WK can arrive to protect it *)

definition WR_exposed :: "KRKPosition \<Rightarrow> bool" where
  "WR_exposed p \<longleftrightarrow> 
      (let cdwkwr = chebyshev_dist (WK p) (WR p);
           cdbkwr = chebyshev_dist (BK p) (WR p)
        in (WhiteOnTurn p \<and> cdwkwr > cdbkwr + 1) \<or> (BlackOnTurn p \<and> cdwkwr > cdbkwr))"

lemma not_WR_exposed_not_WRcaptured':
  assumes "\<not> WR_exposed p" "BlackOnTurn p" "WK p \<noteq> WR p"
  shows "ChessRules.king_scope (BK p) (WR p) \<longrightarrow> ChessRules.king_scope (WK p) (WR p)"
using assms chebyshev_dist_pos[of "WK p" "WR p"]
unfolding WR_exposed_def
by (auto simp add: Let_def king_scope_chebyshev_dist)

lemma not_WR_exposed_not_WRcaptured:
  assumes "\<not> WR_exposed p" "legal_move_BK p p'" "\<not> WRcaptured p"
  shows "\<not> WRcaptured p'"
using assms not_WR_exposed_not_WRcaptured'[of p]
unfolding legal_move_BK_def'
by (auto split: if_split_asm simp add: KRK.legal_position_def KRK.invar_def WR_def WRcaptured_def KRK.kings_separated_def)

(* --------------------------------------------------------------- *)

(* WR divides BK and WK (either horizontally or vertically) *)

definition WR_divides :: "KRKPosition \<Rightarrow> bool" where
  "WR_divides p \<longleftrightarrow> 
     (let (wrx, wry) = WR p; (bkx, bky) = BK p; (wkx, wky) = WK p 
       in \<not> WRcaptured p \<and> (between bkx wrx wkx \<or> between bky wry wky))"

(* --------------------------------------------------------------- *)

(* WK, BK, and WR form an L-shaped pattern *)

definition Lpattern :: "KRKPosition \<Rightarrow> bool" where
  "Lpattern p \<longleftrightarrow> 
     (let (WKx, WKy) = WK p; (BKx, BKy) = BK p; (WRx, WRy) = WR p
       in (WKy=BKy \<and> abs(WKx-BKx)=2 \<and> WRx=WKx \<and> abs (WRy-WKy) = 1) \<or>
          (WKx=BKx \<and> abs(WKy-BKy)=2 \<and> WRy=WKy \<and> abs (WRx-WKx) = 1))"

(* --------------------------------------------------------------- *)

(* two squares are on a same edge of the chessboard *)

fun on_same_edge :: "square \<Rightarrow> square \<Rightarrow> bool" where
  "on_same_edge (x, y) (x', y') \<longleftrightarrow> (x = 0 \<and> x' = 0) \<or> (x = files - 1 \<and> x' = files - 1) \<or> (y = 0 \<and> y' = 0) \<or> (y = ranks - 1 \<and> y' = ranks - 1)"

(* --------------------------------------------------------------- *)
(* White can checkmate in exactly two moves *)
definition ready_to_mate :: "KRKPosition \<Rightarrow> bool" where
  "ready_to_mate p \<longleftrightarrow> 
     (let p = canon p; (WKx, WKy) = WK p; (BKx, BKy) = BK p; (WRx, WRy) = WR p 
       in 
          (BKx = 0 \<and> BKy = 0 \<and> WKx = 1 \<and> WKy = 2 \<and> WRx > 2 \<and> WRy > 0) \<or>
          (BKx = 0 \<and> BKy = 1 \<and> WKx = 2 \<and> WKy = 1 \<and> WRx \<ge> 1 \<and> WRy = 2) \<or>
          (BKx = 0 \<and> WKx = 2 \<and> WRx \<ge> 2 \<and> abs (WKy - BKy) = 1 \<and> abs (WRy - BKy) = 1 \<and> WKy \<noteq> WRy)
     )"

(* --------------------------------------------------------------- *)

(* Manhattan distance to the critical square *)
definition mdcs where
 "mdcs p = manhattan_dist (WK p) (critical_square p)"

definition approach_critical_square where
 "approach_critical_square p1 p2 \<longleftrightarrow> mdcs p1 > mdcs p2"

(* --------------------------------------------------------------- *)

definition WRandWKdiverging where
  "WRandWKdiverging p p' \<longleftrightarrow> 
      chebyshev_dist (WK p) (WR p) < chebyshev_dist (WK p') (WR p')"

(* --------------------------------------------------------------- *)

definition divide_attempt where
  "divide_attempt p p' \<longleftrightarrow>
      (let (WKx, WKy) = WK p; (BKx, BKy) = BK p; (WRx, WRy) = WR p; 
           (WKx', WKy') = WK p'; (BKx', BKy') = BK p'; (WRx', WRy') = WR p'
        in (BKx < WKx \<and> WRx \<noteq> WKx - 1 \<and> WRx' = WKx - 1) \<or> 
           (BKx > WKx \<and> WRx \<noteq> WKx + 1 \<and> WRx' = WKx + 1) \<or> 
           (BKy < WKy \<and> WRy \<noteq> WKy - 1 \<and> WRy' = WKy - 1) \<or> 
           (BKy > WKy \<and> WRy \<noteq> WKy + 1 \<and> WRy' = WKy + 1) \<or> 
           (WRx = WKx \<and> WKx = BKx \<and> (WRx' = WKx+1 \<or> WRx' = WKx - 1)) \<or> 
           (WRy = WKy \<and> WKy = BKy \<and> (WRy' = WKy+1 \<or> WRy' = WKy - 1)))"

(* --------------------------------------------------------------- *)

definition move_WR_to_edge  where
  "move_WR_to_edge p p' \<longleftrightarrow> 
    (let (WKx, WKy) = WK p; (BKx, BKy) = BK p; (WRx, WRy) = WR p; 
         (WKx', WKy') = WK p'; (BKx', BKy') = BK p'; (WRx', WRy') = WR p'
      in (WRx \<noteq> 0 \<and> WRx' = 0) \<or> (WRx \<noteq> files - 1 \<and> WRx' = files - 1) \<or> (WRy \<noteq> 0 \<and> WRy' = 0) \<or> (WRy \<noteq> ranks - 1 \<and> WRy' = ranks - 1))"

(* --------------------------------------------------------------- *)

definition WK_protects_WR where
  "WK_protects_WR p \<longleftrightarrow> (ChessRules.king_scope (BK p) (WR p) \<longrightarrow> ChessRules.king_scope (WK p) (WR p))"

definition back_move :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "back_move p p'  \<longleftrightarrow>
      (let (BKx, BKy) = BK p; (WRx, WRy) = WR p; (WKx, WKy) = WK p; (WKx', WKy') = WK p'
        in BKx = 0 \<and> WRx = 1 \<and> WKx' < WKx \<or>
           BKx = files - 1 \<and> WRx = files - 2 \<and> WKx' > WKx \<or>
           BKy = 0 \<and> WRy = 1 \<and> WKy' < WKy \<or>
           BKy = ranks - 1 \<and> WRy = ranks - 2 \<and> WKy' > WKy)"


(* -------------------------------------------------------------------------- *)
subsection{* Move encoding *}
(* -------------------------------------------------------------------------- *)

(* All king's moves encoded by numbers from 1 to 8 *)
definition kings_square :: "square \<Rightarrow> int \<Rightarrow> square" where
 "kings_square sq k = 
  (let (x, y) = sq 
    in   if k = 1 then (x-(1::int), y-(1::int))
    else if k = 2 then (x-(1::int), y+1)
    else if k = 3 then (x+1, y-(1::int))
    else if k = 4 then (x+1, y+1)
    else if k = 5 then (x-(1::int), y)
    else if k = 6 then (x, y-(1::int))
    else if k = 7 then (x, y+1)
    else if k = 8 then (x+1, y)
    else undefined
  )"

lemma kings_square_fstsnd:
  "fst (kings_square (x, y) 1) = x - 1" "snd (kings_square (x, y) 1) = y - 1"
  "fst (kings_square (x, y) 2) = x - 1" "snd (kings_square (x, y) 2) = y + 1"
  "fst (kings_square (x, y) 3) = x + 1" "snd (kings_square (x, y) 3) = y - 1"
  "fst (kings_square (x, y) 4) = x + 1" "snd (kings_square (x, y) 4) = y + 1"
  "fst (kings_square (x, y) 5) = x - 1" "snd (kings_square (x, y) 5) = y"
  "fst (kings_square (x, y) 6) = x" "snd (kings_square (x, y) 6) = y - 1"
  "fst (kings_square (x, y) 7) = x" "snd (kings_square (x, y) 7) = y + 1"
  "fst (kings_square (x, y) 8) = x + 1" "snd (kings_square (x, y) 8) = y"
unfolding kings_square_def
by simp_all

(* All rook's moves encoded by numbers from 1 to files+ranks *)
definition rooks_square :: "square \<Rightarrow> int \<Rightarrow> square" where
 "rooks_square sq mv = 
   (let (x, y) = sq in 
      (if 1 \<le> mv \<and> mv \<le> files then (mv - 1, y) 
       else if files + 1 \<le> mv \<and> mv \<le> files + ranks then (x, mv - files - 1) 
       else undefined))"

definition move_white :: "KRKPosition \<Rightarrow> int \<Rightarrow> KRKPosition" where
 "move_white p k = 
   (if 1 \<le> k \<and> k \<le> 8 then moveWK p (kings_square (WK p) k)
    else if 9 \<le> k \<and> k \<le> 8 + files + ranks then moveWR p (rooks_square (WR p) (k - 8))
    else undefined)"

lemma BK_move_white [simp]:
  assumes "1 \<le> k \<and> k \<le> 8 + files + ranks"
  shows "BK (move_white p k) = BK p"
using assms
unfolding move_white_def
by auto

lemma WK_move_white [simp]: 
  assumes "1 \<le> k \<and> k \<le> 8 + files + ranks"
  shows "WK (move_white p k) = 
    (if 1 \<le> k \<and> k \<le> 8 then kings_square (WK p) k
     else WK p)"
using assms
unfolding move_white_def
by auto

lemma WR_move_white [simp]: 
  assumes "1 \<le> k \<and> k \<le> 8 + files + ranks" "\<not> WRcaptured p"
  shows "WR (move_white p k) = 
    (if 1 \<le> k \<and> k \<le> 8 then (WR p)
     else (rooks_square (WR p) (k - 8)))"
using assms
unfolding move_white_def
by auto

(* -------------------------------------------------------------------------- *)
subsection{* Strategy definition *}
(* -------------------------------------------------------------------------- *)

datatype MoveType = ImmediateMateMove | ReadyToMateMove | SqueezeMove | ApproachDiagMove | ApproachNonDiagMove | KeepRoomDiagMove | KeepRoomNonDiagMove | RookHomeMove | RookSafeMove | ErrorMove

definition mate_moves where
  "mate_moves = {ImmediateMateMove, ReadyToMateMove}"
definition basic_moves where
  "basic_moves = {SqueezeMove, ApproachDiagMove, ApproachNonDiagMove, KeepRoomDiagMove, KeepRoomNonDiagMove}"
definition no_mate_moves where 
  "no_mate_moves = basic_moves \<union> {RookHomeMove, RookSafeMove}"
definition basic_mate_moves where 
  "basic_mate_moves = mate_moves \<union> basic_moves"
definition all_moves where
  "all_moves = basic_mate_moves \<union> {RookHomeMove, RookSafeMove}"
definition keep_room_moves where
  "keep_room_moves = {KeepRoomNonDiagMove, KeepRoomDiagMove}"
definition approach_moves where
  "approach_moves = {ApproachNonDiagMove, ApproachDiagMove}"

definition immediate_mate_cond :: "KRKPosition \<Rightarrow> bool" where
  "immediate_mate_cond p \<longleftrightarrow> BK_cannot_move p \<and> WR_attacks_BK p"

definition ready_to_mate_cond :: "KRKPosition \<Rightarrow> bool" where
  [simp]: "ready_to_mate_cond p \<longleftrightarrow> ready_to_mate p"

definition squeeze_cond :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "squeeze_cond p p' \<longleftrightarrow>
     room p' < room p \<and> 
      \<not> WR_exposed p' \<and> 
      WR_divides p' \<and> 
      (BK_cannot_move p' \<longrightarrow> WR_attacks_BK p')"

definition approach_cond :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "approach_cond p p' \<longleftrightarrow> 
         approach_critical_square p p' \<and> 
         \<not> WR_exposed p' \<and> 
         (WR_divides p' \<or> Lpattern p') \<and>
         (room p' \<le> 3 \<longrightarrow> (\<not> on_same_edge (WK p') (BK p') \<and> \<not> back_move p p')) \<and> 
         (BK_cannot_move p' \<longrightarrow> WR_attacks_BK p')"

definition keep_room_cond :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "keep_room_cond p p' \<longleftrightarrow> 
         \<not> WR_exposed p' \<and> 
         WR_divides p' \<and> 
         \<not> WRandWKdiverging p p' \<and> 
         (room p' \<le> 3 \<longrightarrow> \<not> on_same_edge (WK p') (BK p')) \<and> 
         (BK_cannot_move p' \<longrightarrow> WR_attacks_BK p')"

definition rook_home_cond :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "rook_home_cond p p' \<longleftrightarrow> 
        divide_attempt p p' \<and> 
        WK_protects_WR p' \<and> 
        (BK_cannot_move p' \<longrightarrow> WR_attacks_BK p')"

definition rook_safe_cond :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> bool" where
  "rook_safe_cond p p' \<longleftrightarrow> 
        move_WR_to_edge p p' \<and> 
        (chebyshev_dist (BK p') (WR p') > 2) \<and> 
        (BK_cannot_move p' \<longrightarrow> WR_attacks_BK p')"

definition no_immediate_mate_WK where
 "no_immediate_mate_WK p \<longleftrightarrow> all8 (\<lambda>x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> immediate_mate_cond (moveWK p sq))"
definition no_immediate_mate_WR where
 "no_immediate_mate_WR p \<longleftrightarrow> all_n (files + ranks) (\<lambda> x. let sq = rooks_square (WR p) x in WR_can_move_to p sq \<longrightarrow> \<not> immediate_mate_cond (moveWR p sq))"
abbreviation "no_immediate_mate \<equiv> no_immediate_mate_WR"
definition no_ready_to_mate_WR where
  "no_ready_to_mate_WR p \<longleftrightarrow> all_n (files + ranks) (\<lambda> x. let sq = rooks_square (WR p) x in WR_can_move_to p sq \<longrightarrow> \<not> ready_to_mate_cond (moveWR p sq))"
definition no_ready_to_mate_WK where
  "no_ready_to_mate_WK p \<longleftrightarrow> all8 (\<lambda> x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> ready_to_mate_cond (moveWK p sq))"
definition no_ready_to_mate where
"no_ready_to_mate p \<longleftrightarrow> no_ready_to_mate_WK p \<and> no_ready_to_mate_WR p"
definition no_squeeze where
  "no_squeeze p \<longleftrightarrow> all_n (files + ranks) (\<lambda> x. let sq = rooks_square (WR p) x in WR_can_move_to p sq \<longrightarrow> \<not> squeeze_cond p (moveWR p sq))"
definition no_approach where
 "no_approach p \<longleftrightarrow> all8 (\<lambda> x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> approach_cond p (moveWK p sq))"
definition no_approach_diag where
 "no_approach_diag p \<longleftrightarrow> all4 (\<lambda> x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> approach_cond p (moveWK p sq))"
definition no_keep_room where
  "no_keep_room p \<longleftrightarrow> all8 (\<lambda> x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> keep_room_cond p (moveWK p sq))"
definition no_keep_room_diag where
 "no_keep_room_diag p \<longleftrightarrow> all4 (\<lambda> x. let sq = kings_square (WK p) x in WK_can_move_to p sq \<longrightarrow> \<not> keep_room_cond p (moveWK p sq))"
definition no_rook_home where
  "no_rook_home p \<longleftrightarrow> all_n (files + ranks) (\<lambda> x. let sq = rooks_square (WR p) x in WR_can_move_to p sq \<longrightarrow> \<not> rook_home_cond p (moveWR p sq))"
definition no_rook_safe where
 "no_rook_safe p \<longleftrightarrow> all_n (files + ranks) (\<lambda> x. let sq = rooks_square (WR p) x in WR_can_move_to p sq \<longrightarrow> \<not> rook_safe_cond p (moveWR p sq))"

lemma no_approach_no_approach_diag:
  "KRKStrategy.no_approach p \<Longrightarrow> KRKStrategy.no_approach_diag p"
unfolding KRKStrategy.no_approach_def KRKStrategy.no_approach_diag_def all8_def all4_def
by auto

lemma no_keep_room_no_keep_room_diag:
  "KRKStrategy.no_keep_room p \<Longrightarrow> KRKStrategy.no_keep_room_diag p"
unfolding KRKStrategy.no_keep_room_def KRKStrategy.no_keep_room_diag_def all8_def all4_def
by auto

definition strategy_white_move :: "KRKPosition \<Rightarrow> KRKPosition \<Rightarrow> MoveType \<Rightarrow> bool" where
  "strategy_white_move p p' t \<longleftrightarrow>
      (if      t = ImmediateMateMove \<and> 
               legal_move_WR p p' \<and> 
               immediate_mate_cond p' then True
       else if t = ReadyToMateMove \<and> 
               legal_move_white p p' \<and> 
               no_immediate_mate p \<and> 
               ready_to_mate_cond p' then True
       else if t = SqueezeMove \<and> 
               legal_move_WR p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               squeeze_cond p p' then True
       else if t = ApproachDiagMove \<and> 
               legal_move_WK p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               approach_cond p p' \<and>
               same_diag (WK p) (WK p') then True 
       else if t = ApproachNonDiagMove \<and> 
               legal_move_WK p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               no_approach_diag p \<and>
               approach_cond p p' then True 
       else if t = KeepRoomDiagMove \<and> 
               legal_move_WK p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               no_approach p \<and>
               keep_room_cond p p' \<and>
               same_diag (WK p) (WK p') then True 
       else if t = KeepRoomNonDiagMove \<and> 
               legal_move_WK p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               no_approach p \<and>
               no_keep_room_diag p \<and>
               keep_room_cond p p' then True 
       else if t = RookHomeMove \<and> 
               legal_move_WR p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               no_approach p \<and>
               no_keep_room p \<and>
               rook_home_cond p p' then True
       else if t = RookSafeMove \<and> 
               legal_move_WR p p' \<and> 
               no_immediate_mate p \<and> 
               no_ready_to_mate p \<and> 
               no_squeeze p \<and> 
               no_approach p \<and>
               no_keep_room p \<and>
               no_rook_home p \<and>
               rook_safe_cond p p' then True
       else False)"


(* -------------------------------------------------------------------------- *)
subsection{* Postconditions of strategic moves *}
(* -------------------------------------------------------------------------- *)

(* After ImmediateMateMove black is checkmated. It is a legal move of WR. *)

lemma ImmediateMateMove_postcond:
  assumes "strategy_white_move p0 p1 ImmediateMateMove"
  shows "KRKStrategy.immediate_mate_cond p1" "legal_move_WR p0 p1"
using assms
by (auto simp add: strategy_white_move_def legal_move_white_def split: if_split_asm)

lemma ImmediateMateMove_checkmated:
  assumes "\<not> WRcaptured p0" "strategy_white_move p0 p1 t1" "t1 = ImmediateMateMove"
  shows "checkmated p1"
using assms
by (simp add: strategy_white_move_def KRKStrategy.immediate_mate_cond_def KRK.checkmated_def KRK.legal_move_WR_def split: if_split_asm)

(* After ReadyToMateMove white can mate in the next move. It is a legal move of a white piece. *)

lemma ReadyToMateMove_postcond:
  assumes "strategy_white_move p0 p1 ReadyToMateMove"
  shows "KRKStrategy.ready_to_mate_cond p1" "legal_move_white p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* SqueezeMove satisfies the squeeze_cond. It is a legal move of WR. *)

lemma SqueezeMove_postcond:
  assumes "strategy_white_move p0 p1 SqueezeMove"
  shows "KRKStrategy.squeeze_cond p0 p1" "legal_move_WR p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* ApproachDiagMove satisfies the approach_cond. It is a legal, diagonal move of WK. *)

lemma ApproachDiagMove_postcond:
  assumes "strategy_white_move p0 p1 ApproachDiagMove"
  shows "KRKStrategy.approach_cond p0 p1" "legal_move_WK p0 p1" "same_diag (WK p0) (WK p1)"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* ApproachNonDiagMove satisfies the approach_cond. It is a legal move of WK. *)

lemma ApproachNonDiagMove_postcond:
  assumes "strategy_white_move p0 p1 ApproachNonDiagMove"
  shows "KRKStrategy.approach_cond p0 p1" "legal_move_WK p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* KeepRoomDiagMove satisfies the keep_room_cond. It is a legal, diagonal move of WK. *)

lemma KeepRoomDiagMove_postcond:
  assumes "strategy_white_move p0 p1 KeepRoomDiagMove"
  shows "KRKStrategy.keep_room_cond p0 p1" "legal_move_WK p0 p1" "same_diag (WK p0) (WK p1)"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* KeepRoomNonDiagMove satisfies the keep_room_cond. It is a legal move of WK. *)

lemma KeepRoomNonDiagMove_postcond:
  assumes "strategy_white_move p0 p1 KeepRoomNonDiagMove"
  shows "KRKStrategy.keep_room_cond p0 p1" "legal_move_WK p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* RookHomeMove satisfies the rook_home_cond. It is a legal move of WR. *)

lemma RookHomeMove_postcond:
  assumes "strategy_white_move p0 p1 RookHomeMove"
  shows "KRKStrategy.rook_home_cond p0 p1" "legal_move_WR p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* RookHomeMove satisfies the rook_safe_cond. It is a legal move of WR. *)

lemma RookSafeMove_postcond:
  assumes "strategy_white_move p0 p1 RookSafeMove"
  shows "KRKStrategy.rook_safe_cond p0 p1" "legal_move_WR p0 p1"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

(* -------------------------------------------------------------------------- *)
subsection{* Some properties of strategic moves *}
(* -------------------------------------------------------------------------- *)

(* -------------------------------------------------------------------------- *)
(* Each type of move is played whenever it is possible *)

lemma notImmediateMateMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove"
  shows "KRKStrategy.no_immediate_mate p0"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notReadyToMateMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove"
  shows "KRKStrategy.no_ready_to_mate_WK p0 \<and> KRKStrategy.no_ready_to_mate_WR p0"
using assms
by (auto simp add: strategy_white_move_def no_ready_to_mate_def split: if_split_asm)

lemma notSqueezeMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove"
  shows "KRKStrategy.no_squeeze p0"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notApproachDiagMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove" "t1 \<noteq> ApproachDiagMove"
  shows "KRKStrategy.no_approach_diag p0"
using assms no_approach_no_approach_diag
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notApproachNonDiagMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove" "t1 \<noteq> ApproachDiagMove" "t1 \<noteq> ApproachNonDiagMove"
  shows "KRKStrategy.no_approach p0"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notKeepRoomDiagMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove" "t1 \<noteq> ApproachDiagMove" "t1 \<noteq> ApproachNonDiagMove" "t1 \<noteq> KeepRoomDiagMove"
  shows "KRKStrategy.no_keep_room_diag p0"
using assms no_keep_room_no_keep_room_diag
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notKeepRoomNonDiagMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove" "t1 \<noteq> ApproachDiagMove" "t1 \<noteq> ApproachNonDiagMove" "t1 \<noteq> KeepRoomDiagMove" "t1 \<noteq> KeepRoomNonDiagMove"
  shows "KRKStrategy.no_keep_room p0"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemma notRookHomeMove:
  assumes "strategy_white_move p0 p1 t1" "t1 \<noteq> ImmediateMateMove" "t1 \<noteq> ReadyToMateMove" "t1 \<noteq> SqueezeMove" "t1 \<noteq> ApproachDiagMove" "t1 \<noteq> ApproachNonDiagMove" "t1 \<noteq> KeepRoomDiagMove" "t1 \<noteq> RookHomeMove" "t1 \<noteq> KeepRoomNonDiagMove"
  shows "KRKStrategy.no_rook_home p0"
using assms
by (auto simp add: strategy_white_move_def split: if_split_asm)

lemmas postcond = ImmediateMateMove_postcond ReadyToMateMove_postcond SqueezeMove_postcond ApproachDiagMove_postcond ApproachNonDiagMove_postcond KeepRoomDiagMove_postcond KeepRoomNonDiagMove_postcond RookHomeMove_postcond RookSafeMove_postcond


(* ---------------------------------------------------------------- *)
(* Strategy always suggests one of the supported move types *)
lemma strategy_white_move_moves:
  assumes "strategy_white_move p0 p1 t1"
  shows "t1 = ImmediateMateMove \<or> t1 = ReadyToMateMove \<or> t1 = SqueezeMove \<or> t1 = ApproachDiagMove \<or> t1 = ApproachNonDiagMove \<or> t1 = KeepRoomDiagMove \<or> t1 = KeepRoomNonDiagMove \<or> t1 = RookHomeMove \<or> t1 = RookSafeMove"
using assms
by (simp add: strategy_white_move_def Let_def split: if_split_asm)

(* Strategy move is always a legal move of white *)
lemma strategy_white_move_legal_move:
  assumes "strategy_white_move p0 p1 t1"  
  shows "legal_move_white p0 p1"
using assms
by (auto simp add: strategy_white_move_def legal_move_white_def split: if_split_asm)

(* Strategy move is always played from a legal position to a legal position *)
lemma strategy_white_move_legal_position:
  assumes "strategy_white_move p0 p1 t1"  
  shows "legal_position p0" "legal_position p1"
using strategy_white_move_legal_move[OF assms]
unfolding legal_move_white_def KRK.legal_move_WR_def KRK.legal_move_WK_def
by auto

(* Strategy move is always played from a position where the white is on turn *)
lemma strategy_white_move_white_on_turn: 
  assumes "strategy_white_move p0 p1 t"
  shows "WhiteOnTurn p0"
using strategy_white_move_legal_move[OF assms]
unfolding legal_move_white_def legal_move_WR_def' legal_move_WK_def'
by auto

(* Strategy move always leads to a position where black is on turn *)
lemma strategy_white_move_black_on_turn: 
  assumes "strategy_white_move p0 p1 t"
  shows "BlackOnTurn p1"
using strategy_white_move_white_on_turn[OF assms] strategy_white_move_legal_move[OF assms]
using legal_move_white_BlackOnTurn[of p0 p1]
by simp

(* After a strategy move WR remains uncaptured *)
lemma strategy_white_move_notWRCaptured:
  assumes "\<not> WRcaptured p0" "strategy_white_move p0 p1 t1"
  shows "\<not> WRcaptured p1"
using strategy_white_move_legal_move[OF assms(2)] assms(1)
by (metis legal_move_white_notWRcaptured)

(* ---------------------------------------------------------------- *)

(* If no_immediate_mate condition is not satisfied, then ImmediateMateMove can be played *)
lemma not_no_immediate_mate:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  "\<not> KRKStrategy.no_immediate_mate p" 
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove"
using assms
unfolding KRKStrategy.no_immediate_mate_WR_def all_n_def Let_def
using KRK.WR_can_move_to[OF p(3) p(1-2)]
unfolding strategy_white_move_def
by metis

(* If no_ready_to_mate condition is not satisfied, then ReadyToMateMove (or ImmediateMateMove) can be played *)
lemma not_no_ready_to_mate:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_ready_to_mate p" 
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove"
proof (cases "\<not> KRKStrategy.no_immediate_mate p")
  case True
  thus ?thesis
    using not_no_immediate_mate[OF p]
    by metis
next
  case False
  hence "\<exists> p'. strategy_white_move p p' ReadyToMateMove"
    using assms
    unfolding KRKStrategy.no_ready_to_mate_def KRKStrategy.no_ready_to_mate_WR_def KRKStrategy.no_ready_to_mate_WK_def
    using KRK.WR_can_move_to[OF p(3) p(1-2)]  KRK.WK_can_move_to[OF p(3) p(1)]
    by (simp add: strategy_white_move_def legal_move_white_def all8_def all_n_def Let_def) metis
  thus ?thesis
    by auto
qed

(* If no_squeeze condition is not satisfied, then SqueezeMove (or some of its previous moves) can be played *)
lemma not_no_squeeze:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_squeeze p"
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove \<or> strategy_white_move p p' SqueezeMove"
proof (cases "\<not> KRKStrategy.no_ready_to_mate p \<or> \<not> KRKStrategy.no_immediate_mate p")
  case True
  thus ?thesis
    using not_no_ready_to_mate[OF p]  not_no_immediate_mate[OF p]
    by metis
next
  case False
  hence "\<exists> p'. strategy_white_move p p' SqueezeMove"
    using assms
    using KRK.WR_can_move_to[OF p(3) p(1-2)]
    unfolding KRKStrategy.no_squeeze_def
    by (simp add: strategy_white_move_def all_n_def Let_def) metis
  thus ?thesis
    by auto
qed

(* If no_approach condition is not satisfied, then ApproachDiagMove or ApproachNotDiagMove
   (or some of their previous moves) can be played *)

lemma same_diag_kings_square:
  assumes "1 \<le> x \<and> x \<le> 8"
  shows "same_diag sq (kings_square sq x) \<longleftrightarrow> 1 \<le> x \<and> x \<le> 4"
using assms
by (cases sq) (simp add: kings_square_def same_diag_def same_diag1_def same_diag2_def)

lemma not_no_approach:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_approach p"
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove \<or> strategy_white_move p p' SqueezeMove \<or> strategy_white_move p p' ApproachDiagMove \<or> strategy_white_move p p' ApproachNonDiagMove"
proof (cases "\<not> KRKStrategy.no_ready_to_mate p \<or> \<not> KRKStrategy.no_immediate_mate p \<or> \<not> KRKStrategy.no_squeeze p")
  case True
  thus ?thesis
    using not_no_ready_to_mate[OF p]  not_no_immediate_mate[OF p] not_no_squeeze[OF p]
    by metis
next
  case False
  have "\<exists> p'. strategy_white_move p p' ApproachNonDiagMove \<or> strategy_white_move p p' ApproachDiagMove"
  proof (cases "no_approach_diag p")
    case False
    thus ?thesis
      using `\<not> (\<not> no_ready_to_mate p \<or> \<not> no_immediate_mate p \<or> \<not> no_squeeze p)`
      unfolding no_approach_diag_def all4_def
      by (auto simp add: Let_def strategy_white_move_def KRK.WK_can_move_to[OF p(3) p(1)] same_diag_kings_square)
  next
    case True
    thus ?thesis
      using assms
      using `\<not> (\<not> no_ready_to_mate p \<or> \<not> no_immediate_mate p \<or> \<not> no_squeeze p)`
      unfolding no_approach_def all8_def
      by (auto simp add: Let_def strategy_white_move_def KRK.WK_can_move_to[OF p(3) p(1)])
  qed
  thus ?thesis
    by auto
qed

(* If no_keep_room condition is not satisfied, then ApproachDiagMove or ApproachNotDiagMove
   (or some of their previous moves) can be played *)

lemma not_no_keep_room:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_keep_room p"
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove \<or> strategy_white_move p p' SqueezeMove \<or> strategy_white_move p p' ApproachDiagMove \<or> strategy_white_move p p' ApproachNonDiagMove \<or> strategy_white_move p p' KeepRoomDiagMove \<or> strategy_white_move p p' KeepRoomNonDiagMove"
proof (cases "\<not> KRKStrategy.no_ready_to_mate p \<or> \<not> KRKStrategy.no_immediate_mate p \<or> \<not> KRKStrategy.no_squeeze p \<or> \<not> KRKStrategy.no_approach p")
  case True
  thus ?thesis
    using not_no_ready_to_mate[OF p]  not_no_immediate_mate[OF p] not_no_squeeze[OF p] not_no_approach[OF p]
    by metis
next
  case False
  hence "\<exists> p'. strategy_white_move p p' KeepRoomNonDiagMove \<or> strategy_white_move p p' KeepRoomDiagMove"
  proof (cases "no_keep_room_diag p")
    case False
    thus ?thesis
      using `\<not> (\<not> no_ready_to_mate p \<or> \<not> no_immediate_mate p \<or> \<not> no_squeeze p \<or> \<not> no_approach p)`
      unfolding no_keep_room_diag_def all4_def
      by (auto simp add: Let_def strategy_white_move_def KRK.WK_can_move_to[OF p(3) p(1)] same_diag_kings_square)
  next
    case True
    thus ?thesis
      using assms
      using `\<not> (\<not> no_ready_to_mate p \<or> \<not> no_immediate_mate p \<or> \<not> no_squeeze p \<or> \<not> no_approach p)`
      unfolding no_keep_room_def all8_def
      by (auto simp add: Let_def strategy_white_move_def KRK.WK_can_move_to[OF p(3) p(1)])
  qed
  thus ?thesis
    by auto
qed

(* If no_rook_home condition is not satisfied, then RookHomeMove (or some of its previous moves) can be played *)
lemma not_no_rook_home:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_rook_home p"
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove \<or> strategy_white_move p p' SqueezeMove \<or> strategy_white_move p p' ApproachDiagMove \<or> strategy_white_move p p' ApproachNonDiagMove \<or> strategy_white_move p p' KeepRoomDiagMove \<or> strategy_white_move p p' KeepRoomNonDiagMove \<or> strategy_white_move p p' RookHomeMove"
proof (cases "\<not> KRKStrategy.no_ready_to_mate p \<or> \<not> KRKStrategy.no_immediate_mate p \<or> \<not> KRKStrategy.no_squeeze p \<or> \<not> KRKStrategy.no_approach p \<or> \<not> KRKStrategy.no_keep_room p")
  case True
  thus ?thesis
    using not_no_ready_to_mate[OF p]  not_no_immediate_mate[OF p] not_no_squeeze[OF p] not_no_approach[OF p] not_no_keep_room[OF p]
    by metis
next
  case False
  hence "\<exists> p'. strategy_white_move p p' RookHomeMove"
    using assms
    using KRK.WR_can_move_to[OF p(3) p(1-2)] 
    unfolding KRKStrategy.no_rook_home_def strategy_white_move_def all_n_def Let_def
    by metis
  thus ?thesis
    by auto
qed

(* If no_rook_safe condition is not satisfied, then RookSafeMove (or some of its previous moves) can be played *)
lemma not_no_rook_safe:
  assumes p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" and
  "\<not> KRKStrategy.no_rook_safe p"
  shows "\<exists> p'. strategy_white_move p p' ImmediateMateMove \<or> strategy_white_move p p' ReadyToMateMove \<or> strategy_white_move p p' SqueezeMove \<or> strategy_white_move p p' ApproachDiagMove \<or> strategy_white_move p p' ApproachNonDiagMove \<or> strategy_white_move p p' KeepRoomDiagMove \<or> strategy_white_move p p' KeepRoomNonDiagMove \<or> strategy_white_move p p' RookHomeMove \<or> strategy_white_move p p' RookSafeMove"
proof (cases "\<not> KRKStrategy.no_ready_to_mate p \<or> \<not> KRKStrategy.no_immediate_mate p \<or> \<not> KRKStrategy.no_squeeze p \<or> \<not> KRKStrategy.no_approach p \<or> \<not> KRKStrategy.no_keep_room p \<or> \<not> KRKStrategy.no_rook_home p")
  case True
  thus ?thesis
    using not_no_ready_to_mate[OF p]  not_no_immediate_mate[OF p] not_no_squeeze[OF p] not_no_approach[OF p] not_no_keep_room[OF p] not_no_rook_home[OF p]
    by metis
next
  case False
  hence "\<exists> p'. strategy_white_move p p' RookSafeMove"
    using assms
    using KRK.WR_can_move_to[OF p(3) p(1-2)] 
    unfolding KRKStrategy.no_rook_safe_def strategy_white_move_def all_n_def Let_def
    by metis
  thus ?thesis
    by auto
qed

end
