(* ************************************************************************** *)
section{* Linear Integer Arithmetic formulations of some notions *}
(* ************************************************************************** *)
theory LIA
imports KRK KRKStrategy KRKSymmetry
begin

(* In this section we shall define all KRK notions in terms of pure Linear integer arithmetics *)

lemma [z3_rule]:
  "(\<not> (A \<longleftrightarrow> \<not> B)) \<longleftrightarrow> (A \<longleftrightarrow> B)"
by simp

lemma [z3_rule]:
  "(\<not> ( \<not> A \<longleftrightarrow> B)) \<longleftrightarrow> (A \<longleftrightarrow> B)"
by simp

(* -------------------------------------------------------------------------- *)
subsection{* Canonical positions (Symmetry.is_canon, Symmetry.is_canon_xy) *}
(* -------------------------------------------------------------------------- *)

definition is_canon where
"is_canon WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
     (2 * BKx + 1 < files \<or> 2 * BKx + 1 = files \<and> (2 * WKx + 1 < files \<or> 2 * WKx + 1 = files \<and> 2 * WRx + 1 \<le> files)) \<and>
     (2 * BKy + 1 < ranks \<or> 2 * BKy + 1 = ranks \<and> (2 * WKy + 1 < ranks \<or> 2 * WKy + 1 = ranks \<and> 2 * WRy + 1 \<le> ranks)) \<and>
     (BKx < BKy \<or> BKx \<le> BKy \<and> (WKx < WKy \<or> WKx \<le> WKy \<and> WRx \<le> WRy))"

lemma is_canon:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "Symmetry.is_canon p \<longleftrightarrow> LIA.is_canon WKx WKy BKx BKy WRx WRy"
using assms
unfolding Symmetry.is_canon_def LIA.is_canon_def
by (auto simp add: is_canon_x_def is_canon_y_def is_canon_diag_def)

definition is_canon_xy where
"is_canon_xy WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
     (2 * BKx + 1 < files \<or> 2 * BKx + 1 = files \<and> (2 * WKx + 1 < files \<or> 2 * WKx + 1 = files \<and> 2 * WRx + 1 \<le> files)) \<and>
     (2 * BKy + 1 < ranks \<or> 2 * BKy + 1 = ranks \<and> (2 * WKy + 1 < ranks \<or> 2 * WKy + 1 = ranks \<and> 2 * WRy + 1 \<le> ranks))"

lemma is_canon_xy:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "Symmetry.is_canon_xy p \<longleftrightarrow> LIA.is_canon_xy WKx WKy BKx BKy WRx WRy"
using assms
unfolding Symmetry.is_canon_xy_def LIA.is_canon_xy_def
by (auto simp add: is_canon_x_def is_canon_y_def is_canon_diag_def)

(* -------------------------------------------------------------------------- *)
subsection{* General chess rules notions *}
(* -------------------------------------------------------------------------- *)

(* ChessRules.board *)
definition board where
  "board sqx sqy \<longleftrightarrow> 0 \<le> sqx \<and> sqx < files \<and> 0 \<le> sqy \<and> sqy < ranks"

lemma board: "ChessRules.board (x, y) \<longleftrightarrow> LIA.board x y"
unfolding LIA.board_def ChessRules.board_def
by simp

(* ChessRules.king_scope *)
definition king_scope :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
  "king_scope Kx Ky Kx' Ky' \<longleftrightarrow> 
    (Kx - 1 = Kx' \<or> Kx = Kx' \<or> Kx + 1 = Kx') \<and>
    (Ky - 1 = Ky' \<or> Ky = Ky' \<or> Ky + 1 = Ky') \<and>
    (Kx \<noteq> Kx' \<or> Ky \<noteq> Ky')"

lemma king_scope:
  "ChessRules.king_scope (Kx, Ky) (Kx', Ky') \<longleftrightarrow> LIA.king_scope Kx Ky Kx' Ky'"
unfolding king_scope_iff some_king_pos_def all_king_pos_def Let_def king_scope_def
by auto

(* ChessRules.rook_scope *)
definition rook_scope :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
  "rook_scope sq1x sq1y sq2x sq2y \<longleftrightarrow> (sq1x = sq2x \<or> sq1y = sq2y) \<and> (sq1x \<noteq> sq2x \<or> sq1y \<noteq> sq2y)"

lemma rook_scope:
  "ChessRules.rook_scope (sq1x, sq1y) (sq2x, sq2y) \<longleftrightarrow> LIA.rook_scope sq1x sq1y sq2x sq2y"
unfolding ChessRules.rook_scope_def LIA.rook_scope_def same_file_def same_rank_def
by simp

(* -------------------------------------------------------------------------- *)
subsection{* KRK notions *}
(* -------------------------------------------------------------------------- *)

(* KRK.invar *)
definition invar :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
 "invar WKx WKy BKx BKy WRx WRy \<longleftrightarrow> (WKx \<noteq> BKx \<or> WKy \<noteq> BKy) \<and> (WRx \<noteq> WKx \<or> WRy \<noteq> WKy) \<and> (WRx \<noteq> BKx \<or> WRy \<noteq> BKy)"

lemma invar:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "KRK.invar p \<longleftrightarrow> LIA.invar WKx WKy BKx BKy WRx WRy"
using assms WRopt[of p WRx WRy WKx WKy] WRopt[of p WRx WRy BKx BKy]
unfolding KRK.invar_def invar_def
by simp

(* KRK.all_on_board *)
definition all_on_board :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
  "all_on_board WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
   board WKx WKy \<and> board BKx BKy \<and> board WRx WRy"

lemma all_on_board:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "KRK.all_on_board p \<longleftrightarrow> LIA.all_on_board WKx WKy BKx BKy WRx WRy"
using assms
unfolding KRK.all_on_board_def all_on_board_def
by (simp add: LIA.board)

(* KRK.kings_separated *)
definition kings_separated :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
"kings_separated WKx WKy BKx BKy \<longleftrightarrow> \<not> LIA.king_scope WKx WKy BKx BKy"

lemma kings_separated:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)"
  shows "KRK.kings_separated p \<longleftrightarrow> LIA.kings_separated WKx WKy BKx BKy"
using assms
unfolding KRK.kings_separated_def kings_separated_def
by (simp add: LIA.king_scope)

(* KRK.square_between_hv *)
definition square_between_hv :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
 "square_between_hv sq1x sq1y sqx sqy sq2x sq2y \<longleftrightarrow> 
    (sq1x = sqx \<and> sqx = sq2x \<and> between sq1y sqy sq2y) \<or>
    (sq1y = sqy \<and> sqy = sq2y \<and> between sq1x sqx sq2x)"

lemma square_between_hv:
  "KRK.square_between_hv (sq1x, sq1y) (sqx, sqy) (sq2x, sq2y) \<longleftrightarrow> LIA.square_between_hv sq1x sq1y sqx sqy sq2x sq2y"
by (simp add: KRK.square_between_hv_def LIA.square_between_hv_def same_file_def same_rank_def)

(* KRK.WR_attacks_square *)
definition WR_attacks_square :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
  "WR_attacks_square rx ry kx ky x y \<longleftrightarrow> 
     LIA.rook_scope rx ry x y \<and> \<not> LIA.square_between_hv rx ry kx ky x y"

lemma WR_attacks_square:
  assumes "WK p = (WKx, WKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "KRK.WR_attacks_square p (a, b) \<longleftrightarrow> LIA.WR_attacks_square WRx WRy WKx WKy a b"
using assms
unfolding KRK.WR_attacks_square_def LIA.WR_attacks_square_def
by (simp add: LIA.rook_scope LIA.square_between_hv)

(* KRK.legal_position *)

definition legal_position_white_on_turn where
  "legal_position_white_on_turn WKx WKy BKx BKy WRx WRy \<longleftrightarrow>  
      LIA.invar WKx WKy BKx BKy WRx WRy \<and> 
      LIA.all_on_board WKx WKy BKx BKy WRx WRy \<and> 
      LIA.kings_separated WKx WKy BKx BKy \<and> 
      \<not> LIA.WR_attacks_square WRx WRy WKx WKy BKx BKy"

lemma legal_position_white_on_turn:
assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p"
shows "KRK.legal_position p \<longleftrightarrow> LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy"
using assms invar[of p WKx WKy BKx BKy WRx WRy] all_on_board[of p WKx WKy BKx BKy WRx WRy]  kings_separated[of p WKx WKy BKx BKy] WR_attacks_square[of p WKx WKy WRx WRy BKx BKy]
unfolding legal_position_def legal_position_white_on_turn_def
by simp

definition legal_position_black_on_turn where
  "legal_position_black_on_turn WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
     LIA.invar WKx WKy BKx BKy WRx WRy \<and>
     LIA.all_on_board WKx WKy BKx BKy WRx WRy \<and>
     LIA.kings_separated WKx WKy BKx BKy"

lemma legal_position_black_on_turn:                                                                           
assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "BlackOnTurn p" "\<not> WRcaptured p"
shows "KRK.legal_position p \<longleftrightarrow> LIA.legal_position_black_on_turn WKx WKy BKx BKy WRx WRy"
using assms invar[of p WKx WKy BKx BKy WRx WRy] all_on_board[of p WKx WKy BKx BKy WRx WRy]  kings_separated[of p WKx WKy BKx BKy]
unfolding legal_position_def legal_position_black_on_turn_def
by simp

lemmas LIA_unfold = LIA.board_def LIA.king_scope_def LIA.rook_scope_def LIA.invar_def LIA.all_on_board_def LIA.kings_separated_def LIA.square_between_hv_def LIA.WR_attacks_square_def LIA.legal_position_white_on_turn_def LIA.legal_position_black_on_turn_def

(* KRK.legal_move_WK *)
definition legal_move_WK where
  "legal_move_WK WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow> 
       (LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy \<and>
        LIA.king_scope WKx WKy WKx' WKy' \<and>
        LIA.legal_position_black_on_turn WKx' WKy' BKx' BKy' WRx' WRy' \<and> 
        BKx' = BKx \<and> BKy' = BKy \<and> WRx' = WRx \<and> WRy' = WRy)"

lemma legal_move_WK:
assumes
    "WK p1 = (WKx, WKy)" "BK p1 = (BKx, BKy)" "WR p1 = (WRx, WRy)"
    "WK p1' = (WKx', WKy')" "BK p1' = (BKx', BKy')" "WR p1' = (WRx', WRy')"
    "WhiteOnTurn p1" "\<not> WRcaptured p1" "BlackOnTurn p1'" "\<not> WRcaptured p1'"
shows
  "KRK.legal_move_WK p1 p1' \<longleftrightarrow> 
   LIA.legal_move_WK WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms
unfolding KRK.legal_move_WK_def legal_move_WK_def
using legal_position_white_on_turn[of p1 WKx WKy BKx BKy WRx WRy] legal_position_black_on_turn[of p1' WKx' WKy' BKx' BKy' WRx' WRy']
by (auto split: if_split_asm simp add: king_scope) (rule moveWK_eqI, auto simp add: WRcaptured_def WR_def)

(* KRK.legal_move_BK *)
definition legal_move_BK where
  "legal_move_BK WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow> 
       LIA.legal_position_black_on_turn WKx WKy BKx BKy WRx WRy \<and>
       LIA.king_scope BKx BKy BKx' BKy' \<and>
       LIA.legal_position_white_on_turn WKx' WKy' BKx' BKy' WRx' WRy' \<and> 
       WKx' = WKx \<and> WKy' = WKy \<and> WRx' = WRx \<and> WRy' = WRy"

lemma legal_move_BK:
assumes
    "WK p1 = (WKx, WKy)" "BK p1 = (BKx, BKy)" "WR p1 = (WRx, WRy)"
    "WK p1' = (WKx', WKy')" "BK p1' = (BKx', BKy')" "WR p1' = (WRx', WRy')"
     "BlackOnTurn p1" "\<not> WRcaptured p1" "WhiteOnTurn p1'" "\<not> WRcaptured p1'"
shows
"KRK.legal_move_BK p1 p1' \<longleftrightarrow> LIA.legal_move_BK WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms
unfolding KRK.legal_move_BK_def legal_move_BK_def
using legal_position_black_on_turn[of p1 WKx WKy BKx BKy WRx WRy] legal_position_white_on_turn[of p1' WKx' WKy' BKx' BKy' WRx' WRy']
by (auto split: if_split_asm simp add: king_scope) (rule moveBK_eqI, auto simp add: WRcaptured_def WR_def legal_position_white_on_turn_def LIA.invar_def)

(* KRK.legal_move_WR *)
definition legal_move_WR where
  "legal_move_WR WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow> 
       (LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy \<and>
        LIA.WR_attacks_square WRx WRy WKx WKy WRx' WRy' \<and>
        LIA.legal_position_black_on_turn WKx' WKy' BKx' BKy' WRx' WRy' \<and> 
        WKx' = WKx \<and> WKy' = WKy \<and> BKx' = BKx \<and> BKy' = BKy)"

lemma legal_move_WR:
assumes
    "WK p1 = (WKx, WKy)" "BK p1 = (BKx, BKy)" "WR p1 = (WRx, WRy)"
    "WK p1' = (WKx', WKy')" "BK p1' = (BKx', BKy')" "WR p1' = (WRx', WRy')"
     "WhiteOnTurn p1" "\<not> WRcaptured p1" "BlackOnTurn p1'" "\<not> WRcaptured p1'"
shows
  "KRK.legal_move_WR p1 p1' \<longleftrightarrow> LIA.legal_move_WR WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms
unfolding KRK.legal_move_WR_def legal_move_WR_def
using legal_position_white_on_turn[of p1 WKx WKy BKx BKy WRx WRy] legal_position_black_on_turn[of p1' WKx' WKy' BKx' BKy' WRx' WRy']
using WR_attacks_square[of p1 WKx WKy WRx WRy WRx' WRy']
by auto (rule moveWR_eqI, auto simp add: WRcaptured_def WR_def)

text{* KRK.white_attacks *}
definition white_attacks_square where
  "white_attacks_square WKx WKy WRx WRy sqx sqy \<longleftrightarrow> king_scope WKx WKy sqx sqy \<or> WR_attacks_square WRx WRy WKx WKy sqx sqy"

lemma white_attacks_square:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "KRK.white_attacks_square p (sqx, sqy) \<longleftrightarrow> LIA.white_attacks_square WKx WKy WRx WRy sqx sqy"
using assms
unfolding KRK.white_attacks_square_def LIA.white_attacks_square_def KRK.WK_attacks_square_def
by (simp add: king_scope LIA.WR_attacks_square[of p WKx WKy WRx WRy sqx sqy])

(* KRK.BK_cannot_move *)
definition BK_cannot_move where
 "BK_cannot_move WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
  (board (BKx - 1) (BKy - 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy (BKx - 1) (BKy - 1)) \<and> 
  (board (BKx - 1) BKy \<longrightarrow> white_attacks_square WKx WKy WRx WRy (BKx - 1) BKy) \<and>
  (board (BKx - 1) (BKy + 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy (BKx - 1) (BKy + 1)) \<and> 
  (board BKx (BKy - 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy BKx (BKy - 1)) \<and> 
  (board BKx (BKy + 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy BKx (BKy + 1)) \<and> 
  (board (BKx + 1) (BKy - 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy (BKx + 1) (BKy - 1)) \<and> 
  (board (BKx + 1) BKy \<longrightarrow> white_attacks_square WKx WKy WRx WRy (BKx + 1) BKy) \<and>
  (board (BKx + 1) (BKy + 1) \<longrightarrow> white_attacks_square WKx WKy WRx WRy  (BKx + 1) (BKy + 1))"

lemma BK_cannot_move:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "BlackOnTurn p" "\<not> WRcaptured p"
  shows "KRK.BK_cannot_move p \<longleftrightarrow> LIA.BK_cannot_move WKx WKy BKx BKy WRx WRy"
using assms
unfolding KRK.BK_cannot_move_def LIA.BK_cannot_move_def
by (simp add: all_king_pos_def board) (metis LIA.white_attacks_square)

(* KRK.WR_attacks_BK *)
definition WR_attacks_BK where
   "WR_attacks_BK WKx WKy BKx BKy WRx WRy \<longleftrightarrow> WR_attacks_square WRx WRy WKx WKy BKx BKy"

lemma WR_attacks_BK:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "\<not> WRcaptured p"
  shows "KRK.WR_attacks_BK p \<longleftrightarrow> LIA.WR_attacks_BK WKx WKy BKx BKy WRx WRy"
using assms
unfolding KRK.WR_attacks_BK_def LIA.WR_attacks_BK_def
by (simp add: LIA.WR_attacks_square)
  

(* KRK.stalemate *)
definition stalemate where
  "stalemate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> legal_position_black_on_turn WKx WKy BKx BKy WRx WRy \<and> BK_cannot_move WKx WKy BKx BKy WRx WRy \<and> \<not> WR_attacks_BK WKx WKy BKx BKy WRx WRy"

lemma stalemate:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "BlackOnTurn p" "\<not> WRcaptured p"
  shows "KRK.stalemate p \<longleftrightarrow> LIA.stalemate WKx WKy BKx BKy WRx WRy"
using assms                                                                            
unfolding KRK.stalemate_def LIA.stalemate_def
by (metis BK_cannot_move LIA.WR_attacks_BK legal_position_black_on_turn)

(* Optimized definition of stalemate. Note: requires files \<ge> 2 and ranks \<ge> 2 *)
definition stalemate_opt where "stalemate_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
  (BKx = 0 \<and> BKy = 0 \<or> BKx = files-(1::int) \<and> BKy = 0 \<or> BKx = 0 \<and> BKy = ranks-(1::int) \<or> BKx = files-(1::int) \<and> BKy = ranks-(1::int)) \<and> 
  ((abs(BKx - WRx) = 1 \<and> BKy \<noteq> WRy \<and> ((BKx = WKx \<and> abs(BKy - WKy) = 2) \<or> (abs(BKy - WRy) = 1 \<and> LIA.king_scope WKx WKy WRx WRy ))) \<or> 
  (abs(BKy - WRy) = 1 \<and> BKx \<noteq> WRx \<and> ((BKy = WKy \<and> abs(BKx - WKx) = 2) \<or> (abs(BKx - WRx) = 1 \<and> LIA.king_scope WKx WKy WRx WRy ))))"

lemma stalemate_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRK.stalemate p \<longleftrightarrow> LIA.stalemate_opt WKx WKy BKx BKy WRx WRy"
proof-
  have "legal_position_black_on_turn WKx WKy BKx BKy WRx WRy"
    using p legal_position_black_on_turn[OF * p(1-2)]
    by simp
  hence "\<not> stalemate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> \<not> stalemate_opt WKx WKy BKx BKy WRx WRy"
    using `files \<ge> 2` `ranks \<ge> 2`
    unfolding LIA_unfold LIA.BK_cannot_move_def LIA.stalemate_def LIA.white_attacks_square_def LIA.WR_attacks_BK_def stalemate_opt_def between_def
    by - (rule iffI, smt+)
  thus ?thesis
    using stalemate[OF * p(1-2)] assms
    by auto
qed


(* KRK.checkmated *)
definition checkmated where 
  "checkmated WKx WKy BKx BKy WRx WRy \<longleftrightarrow> legal_position_black_on_turn WKx WKy BKx BKy WRx WRy \<and> BK_cannot_move WKx WKy BKx BKy WRx WRy \<and> WR_attacks_BK WKx WKy BKx BKy WRx WRy"

lemma checkmated:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "BlackOnTurn p" "\<not> WRcaptured p"
  shows "KRK.checkmated p \<longleftrightarrow> LIA.checkmated WKx WKy BKx BKy WRx WRy"
using assms
unfolding KRK.checkmated_def LIA.checkmated_def
by (metis BK_cannot_move LIA.WR_attacks_BK legal_position_black_on_turn)

(* Optimized definition of checkmated. Note: requires files \<ge> 2 and ranks \<ge> 2 *)
definition checkmated_opt :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
  "checkmated_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
  (BKx = 0 \<and> WRx = 0 \<and> WKx = 2 \<and> abs(BKy - WRy) > 1 \<and> (BKy = 0 \<or> BKy = ranks - 1 \<longrightarrow> abs(WKy - BKy) \<le> 1) \<and> (BKy \<noteq> 0 \<and> BKy \<noteq> ranks - 1 \<longrightarrow> WKy = BKy)) \<or> 
  (BKy = 0 \<and> WRy = 0 \<and> WKy = 2 \<and> abs(BKx - WRx) > 1 \<and> (BKx = 0 \<or> BKx = files - 1 \<longrightarrow> abs(WKx - BKx) \<le> 1) \<and> (BKx \<noteq> 0 \<and> BKx \<noteq> files - 1 \<longrightarrow> WKx = BKx)) \<or> 
  (BKx = files - 1 \<and> WRx = files - 1 \<and> WKx = files - 3 \<and> abs(BKy - WRy) > 1 \<and> (BKy = 0 \<or> BKy = ranks - 1 \<longrightarrow> abs(WKy - BKy) \<le> 1) \<and> (BKy \<noteq> 0 \<and> BKy \<noteq> ranks - 1 \<longrightarrow> WKy = BKy)) \<or> 
  (BKy = ranks - 1 \<and> WRy = ranks - 1 \<and> WKy = ranks - 3 \<and> abs(BKx - WRx) > 1 \<and> (BKx = 0 \<or> BKx = files - 1 \<longrightarrow> abs(WKx - BKx) \<le> 1) \<and> (BKx \<noteq> 0 \<and> BKx \<noteq> files - 1 \<longrightarrow> WKx = BKx))"
  
lemma checkmated_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p" 
  shows "KRK.checkmated p \<longleftrightarrow> LIA.checkmated_opt WKx WKy BKx BKy WRx WRy"
proof-
  have "legal_position_black_on_turn WKx WKy BKx BKy WRx WRy"
    using p legal_position_black_on_turn[OF * p(1-2)]
    by simp
  thus ?thesis
    using assms
    unfolding KRK.checkmated_def checkmated_opt_def KRK.WR_attacks_BK_def KRK.BK_cannot_move_def KRK.white_attacks_square_def all_king_pos_def
    using WR_attacks_square[OF *(1) *(3) p(2)]
    apply (simp add: ChessRules.board_def king_scope)
    unfolding LIA_unfold
    by (rule iffI, smt+)
qed

(* KRK.WK_can_move_to *)
definition WK_can_move_to where
  "WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longleftrightarrow>
     LIA.king_scope WKx WKy sqx sqy \<and> 
     0 \<le> sqx \<and> sqx < files \<and> 0 \<le> sqy \<and> sqy < ranks \<and> 
     \<not> LIA.king_scope BKx BKy sqx sqy \<and> 
     (WRx = sqx \<longrightarrow> WRy \<noteq> sqy)"

lemma WK_can_move_to:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p"
  shows "KRK.WK_can_move_to p (sqx, sqy) \<longleftrightarrow> LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy"
using assms
by (simp add: LIA.king_scope ChessRules.board_def KRK.WK_can_move_to_def LIA.WK_can_move_to_def)

(* KRK.WR_can_move_to *)
definition WR_can_move_to where
  "WR_can_move_to WKx WKy BKx BKy WRx WRy x y \<longleftrightarrow> 
     LIA.WR_attacks_square WRx WRy WKx WKy x y \<and>
     0 \<le> x \<and> x < files \<and> 0 \<le> y \<and> y < ranks \<and> 
     (WKx = x \<longrightarrow> WKy \<noteq> y) \<and> (BKx = x \<longrightarrow> BKy \<noteq> y)"

lemma WR_can_move_to:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRK.WR_can_move_to p (x, y) \<longleftrightarrow> LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy x y"
using assms legal_position_white_on_turn[of p WKx WKy BKx BKy WRx WRy] WR_attacks_square[of p WKx WKy WRx WRy x y]
by (simp add: ChessRules.board_def legal_position_white_on_turn_def LIA.all_on_board_def LIA.WR_can_move_to_def KRK.WR_can_move_to_def)

(* -------------------------------------------------------------------------- *)
subsection{* KRK strategy notions *}
(* -------------------------------------------------------------------------- *)

(* KRKStrategy.chebyshev_dist *)
definition chebyshev_dist where
"chebyshev_dist x1 y1 x2 y2 = (if (if x1 - x2 < 0 then - (x1 - x2) else x1 - x2) \<le> (if y1 - y2 < 0 then - (y1 - y2) else y1 - y2)
     then if y1 - y2 < 0 then - (y1 - y2) else y1 - y2 else if x1 - x2 < 0 then - (x1 - x2) else x1 - x2)"

lemma chebyshev_dist: "KRKStrategy.chebyshev_dist (x1, y1) (x2, y2) = LIA.chebyshev_dist x1 y1 x2 y2"
by (simp add: chebyshev_dist.simps chebyshev_dist_def abs_if max_def)

(* KRKStrategy.manhattan_dist *)
definition manhattan_dist where
  "manhattan_dist x1 y1 x2 y2 = (if x1 \<le> x2 then if y1 \<le> y2 then x2 - x1 + (y2 - y1) else x2 - x1 + (y1 - y2) else if y1 \<le> y2 then x1 - x2 + (y2 - y1) else x1 - x2 + (y1 - y2))"

lemma manhattan_dist: "KRKStrategy.manhattan_dist (x1, y1) (x2, y2) = LIA.manhattan_dist x1 y1 x2 y2"
unfolding manhattan_dist_cases manhattan_dist_def
by simp

(* KRKStrategy.Lpattern *)
definition Lpattern :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where 
  "Lpattern WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
       WKy = BKy \<and> (WKx = BKx + 2 \<or> BKx = WKx + 2) \<and> WRx = WKx \<and> (WRy = WKy + 1 \<or> WKy = WRy + 1) \<or>
       WKx = BKx \<and> (WKy = BKy + 2 \<or> BKy = WKy + 2) \<and> WRy = WKy \<and> (WRx = WKx + 1 \<or> WKx = WRx + 1)"

lemma Lpattern:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"
  shows "KRKStrategy.Lpattern p \<longleftrightarrow> LIA.Lpattern WKx WKy BKx BKy WRx WRy"
using assms
unfolding KRKStrategy.Lpattern_def Lpattern_def Let_def
by (smt split_conv)


(* KRKStrategy.room *)
declare KRKStrategy.room'.simps [simp del]
definition room where
"room x1 y1 x2 y2 = (if x1 = x2 \<or> y1 = y2 then files + ranks - 1 else (if x2 < x1 then x1 else files - 1 - x1) + (if y2 < y1 then y1 else ranks - 1 - y1))"

lemma room: 
  "KRKStrategy.room' (x1, y1) (x2, y2) = LIA.room x1 y1 x2 y2"
by (simp add: room' room_def)

(* KRKStrategy.critical_square *)
definition critical_square where
 "critical_square WRx WRy BKx BKy = 
    (if WRx = BKx then WRx else if BKx < WRx then WRx - 1 else WRx + 1,
     if WRy = BKy then WRy else if BKy < WRy then WRy - 1 else WRy + 1)"

lemma critical_square: 
  assumes "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"
  shows "KRKStrategy.critical_square p = LIA.critical_square WRx WRy BKx BKy"
using assms
by (simp add: KRKStrategy.critical_square_def critical_square_def Let_def split_def)

(* KRKStrategy.mdcs *)
definition mdcs :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int" where
 "mdcs WKx WKy BKx BKy WRx WRy = 
  manhattan_dist WKx WKy (if WRx = BKx then WRx else if BKx < WRx then WRx - 1 else WRx + 1) (if WRy = BKy then WRy else if BKy < WRy then WRy - 1 else WRy + 1)"

lemma mdcs:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"
  shows "KRKStrategy.mdcs p = LIA.mdcs WKx WKy BKx BKy WRx WRy"
using assms manhattan_dist critical_square
unfolding KRKStrategy.mdcs_def mdcs_def
by (metis critical_square_def)

(* KRKStrategy.approach_critical_square *)
definition approach_critical_square where
  "approach_critical_square WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow>
     mdcs WKx' WKy' BKx' BKy' WRx' WRy' < mdcs WKx WKy BKx BKy WRx WRy"

lemma approach_critical_square:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
  shows "KRKStrategy.approach_critical_square p p' \<longleftrightarrow> LIA.approach_critical_square WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms mdcs
unfolding KRKStrategy.approach_critical_square_def approach_critical_square_def
by metis

lemma approach_critical_square_moveWR:
  assumes "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "sq = (sqx, sqy)"
  shows "KRKStrategy.approach_critical_square p (moveWK p sq) \<longleftrightarrow> LIA.approach_critical_square WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy"
using assms approach_critical_square[OF assms(1-3), of "moveWK p sq" sqx sqy BKx BKy WRx WRy]
by simp

(* KRKStrategy.back_move *)
definition back_move :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where
"back_move WKx WKy BKx BKy WRx WRy WKx' WKy'  \<longleftrightarrow> 
         BKx = 0 \<and> WRx = 1 \<and> WKx' < WKx \<or>
         BKx = files - 1 \<and> WRx = files - 2 \<and> WKx' > WKx \<or>
         BKy = 0 \<and> WRy = 1 \<and> WKy' < WKy \<or>
         BKy = ranks - 1 \<and> WRy = ranks - 2 \<and> WKy' > WKy"

definition not_back_move :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where 
  "not_back_move WKx WKy BKx BKy WRx WRy WKx' WKy' \<longleftrightarrow> 
  (BKx \<noteq> 0 \<or> WRx \<noteq> 1 \<or> WKx' \<ge> WKx) \<and> 
  (BKx \<noteq> files - 1 \<or> WRx \<noteq> files - 2 \<or> WKx' \<le> WKx) \<and>
  (BKy \<noteq> 0 \<or> WRy \<noteq> 1 \<or> WKy' \<ge> WKy) \<and>
  (BKy \<noteq> ranks - 1 \<or> WRy \<noteq> ranks - 2 \<or> WKy' \<le> WKy)"

lemma not_back_move: "LIA.not_back_move WKx WKy BKx BKy WRx WRy WKx' WKy' \<longleftrightarrow> \<not> LIA.back_move WKx WKy BKx BKy WRx WRy WKx' WKy'"
unfolding not_back_move_def back_move_def
by smt

(* KRKStrategy.no_immediate_mate *)
(* NOTE: Requires files \<ge> 2 and ranks \<ge> 2 when checkmated_opt is used *)
definition no_immediate_mate where
  "no_immediate_mate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all_n (files + ranks) (\<lambda>x. let (sqx, sqy) = rooks_square (WRx, WRy) x in WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> checkmated_opt WKx WKy BKx BKy sqx sqy)"

lemma no_immediate_mate:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows 
  "KRKStrategy.no_immediate_mate p \<longleftrightarrow> LIA.no_immediate_mate WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_immediate_mate_WR_def LIA.no_immediate_mate_def split_def Let_def
proof (rule all_n_eq, rule allI, rule impI)
  fix i::int
  assume "1 \<le> i \<and> i \<le> files + ranks"
  let ?sq = "rooks_square (WRx, WRy) i"
  show "(KRK.WR_can_move_to p (rooks_square (WR p) i) \<longrightarrow> \<not> immediate_mate_cond (moveWR p (rooks_square (WR p) i))) =
        (WR_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow> \<not> checkmated_opt WKx WKy BKx BKy (fst ?sq) (snd ?sq))"
    using assms KRK.WR_can_move_to[of p ?sq] LIA.WR_can_move_to[OF assms(3-)]
    using checkmated_opt[of "moveWR p (rooks_square (WR p) i)" WKx WKy BKx BKy "fst ?sq" "snd ?sq"] 
    unfolding immediate_mate_cond_def KRK.checkmated_def
    by (force simp add: KRK.legal_move_WR_def)
qed

(* Optimized definition of no_immediate mate. NOTE: Requires files \<ge> 2 and ranks \<ge> 2 *)
definition no_immediate_mate_opt where
  "no_immediate_mate_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
      (let (sqx, sqy) = (0, WRy) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.checkmated_opt WKx WKy BKx BKy sqx sqy) \<and> 
      (let (sqx, sqy) = (files - 1, WRy) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.checkmated_opt WKx WKy BKx BKy sqx sqy) \<and>   
      (let (sqx, sqy) = (WRx, 0) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.checkmated_opt WKx WKy BKx BKy sqx sqy) \<and>  
      (let (sqx, sqy) = (WRx, ranks - 1) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.checkmated_opt WKx WKy BKx BKy sqx sqy)"

lemma no_immediate_mate_opt':
  assumes "files \<ge> 2" "ranks \<ge> 2" "LIA.no_immediate_mate WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_immediate_mate_opt WKx WKy BKx BKy WRx WRy"
proof-
  have "rooks_square (WRx, WRy) 1 = (0, WRy)"
       "rooks_square (WRx, WRy) files = (files - 1, WRy)"
       "rooks_square (WRx, WRy) (files+1) = (WRx, 0)"
       "rooks_square (WRx, WRy) (files+ranks) = (WRx, ranks - 1)"
    using assms(1-2)
    by (simp_all add: rooks_square_def)
  thus ?thesis
    using assms
    unfolding no_immediate_mate_def no_immediate_mate_opt_def all_n_def Let_def
    by smt
qed

(*
lemma
  assumes "LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy" 
          "LIA.no_immediate_mate_opt WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_immediate_mate WKx WKy BKx BKy WRx WRy"
using assms
unfolding legal_position_white_on_turn_def all_on_board_def no_immediate_mate_def no_immediate_mate_opt_def rooks_square_def LIA.invar_def checkmated_opt_def WR_attacks_square_def Let_def
apply simp
unfolding LIA_unfold LIA.WR_can_move_to_def
unfolding  files_def ranks_def all_n_8
using [[smt_oracle = true]]
by smt
*)

lemma no_immediate_mate_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows 
  "KRKStrategy.no_immediate_mate p \<longrightarrow> LIA.no_immediate_mate_opt WKx WKy BKx BKy WRx WRy"
using no_immediate_mate[OF assms] no_immediate_mate_opt'[OF assms(1-2)] legal_position_white_on_turn[OF assms(3-7)] assms(8)
by auto


(* KRKStrategy.ready_to_mate *)

(* NOTE: Requires files \<ge> 4 and ranks \<ge> 4 *)
definition ready_to_mate :: "int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> int \<Rightarrow> bool" where 
  "ready_to_mate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
(BKx = 0 \<and> BKy = 0 \<and> WKx = 1 \<and> WKy = 2 \<and> 2 < WRx \<and> 0 < WRy \<or>
 BKx = 0 \<and> BKy = 1 \<and> WKx = 2 \<and> WKy = 1 \<and> 1 \<le> WRx \<and> WRy = 2 \<or>
 BKx = 0 \<and> WKx = 2 \<and> 2 \<le> WRx \<and> \<bar>WKy - BKy\<bar> = 1 \<and> \<bar>WRy - BKy\<bar> = 1 \<and> WKy \<noteq> WRy) \<or>
(files - 1 = BKx \<and> ranks - 1 = BKy \<and> files - WKx = 2 \<and> ranks - WKy = 3 \<and> 3 < files - WRx \<and> 0 < ranks - 1 - WRy \<or>
 files - 1 = BKx \<and> ranks - BKy = 2 \<and> files - WKx = 3 \<and> ranks - WKy = 2 \<and> 2 \<le> files - WRx \<and> ranks - WRy = 3 \<or>
 files - 1 = BKx \<and> files - WKx = 3 \<and> 3 \<le> files - WRx \<and>  \<bar>- WKy + BKy\<bar> = 1 \<and> \<bar>- WRy + BKy\<bar> = 1 \<and> WKy \<noteq> WRy) \<or>
(files - 1 = BKx \<and> BKy = 0 \<and> files - WKx = 2 \<and> WKy = 2 \<and> 3 < files - WRx \<and> 0 < WRy \<or>
 files - 1 = BKx \<and> BKy = 1 \<and> files - WKx = 3 \<and> WKy = 1 \<and> 2 \<le> files - WRx \<and> WRy = 2 \<or>
 files - 1 = BKx \<and> files - WKx = 3 \<and> 3 \<le> files - WRx \<and> \<bar>WKy - BKy\<bar> = 1 \<and> \<bar>WRy - BKy\<bar> = 1 \<and> WKy \<noteq> WRy) \<or>
(BKx = 0 \<and> ranks - 1 = BKy \<and> WKx = 1 \<and> ranks - WKy = 3 \<and> 2 < WRx \<and> 0 < ranks - 1 - WRy \<or>
 BKx = 0 \<and> ranks - BKy = 2 \<and> WKx = 2 \<and> ranks - WKy = 2 \<and> 1 \<le> WRx \<and> ranks - WRy = 3 \<or>
 BKx = 0 \<and> WKx = 2 \<and> 2 \<le> WRx \<and> \<bar>- WKy + BKy\<bar> = 1 \<and> \<bar>- WRy + BKy\<bar> = 1 \<and> WKy \<noteq> WRy) \<or>
(ranks - 1 = BKy \<and> files - 1 = BKx \<and> ranks - WKy = 2 \<and> files - WKx = 3 \<and> 3 < ranks - WRy \<and> 0 < files - 1 - WRx \<or>
 ranks - 1 = BKy \<and> files - BKx = 2 \<and> ranks - WKy = 3 \<and> files - WKx = 2 \<and> 2 \<le> ranks - WRy \<and> files - WRx = 3 \<or>
 ranks - 1 = BKy \<and> ranks - WKy = 3 \<and> 3 \<le> ranks - WRy \<and> \<bar>- WKx + BKx\<bar> = 1 \<and> \<bar>- WRx + BKx\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(ranks - 1 = BKy \<and> files - 1 = BKx \<and> ranks - WKy = 2 \<and> files - WKx = 3 \<and> 3 < ranks - WRy \<and> 0 < files - 1 - WRx \<or>
 ranks - 1 = BKy \<and> ranks - WKy = 3 \<and> 3 \<le> ranks - WRy \<and> \<bar>- WKx + BKx\<bar> = 1 \<and> \<bar>- WRx + BKx\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(BKy = 0 \<and> files - 1 = BKx \<and> WKy = 1 \<and> files - WKx = 3 \<and> 2 < WRy \<and> 0 < files - 1 - WRx \<or>
 BKy = 0 \<and> files - BKx = 2 \<and> WKy = 2 \<and> files - WKx = 2 \<and> 1 \<le> WRy \<and> files - WRx = 3 \<or>
 BKy = 0 \<and> WKy = 2 \<and> 2 \<le> WRy \<and> \<bar>- WKx + BKx\<bar> = 1 \<and> \<bar>- WRx + BKx\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(ranks - 1 = BKy \<and> BKx = 0 \<and> ranks - WKy = 2 \<and> WKx = 2 \<and> 3 < ranks - WRy \<and> 0 < WRx \<or>
 ranks - 1 = BKy \<and> BKx = 1 \<and> ranks - WKy = 3 \<and> WKx = 1 \<and> 2 \<le> ranks - WRy \<and> WRx = 2 \<or>
 ranks - 1 = BKy \<and> ranks - WKy = 3 \<and> 3 \<le> ranks - WRy \<and> \<bar>WKx - BKx\<bar> = 1 \<and> \<bar>WRx - BKx\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(ranks - 1 = BKy \<and> 0 = BKx \<and> ranks - WKy = 2 \<and> WKx = 2 \<and> 3 < ranks - WRy \<and> 0 < WRx \<or>
 ranks - 1 = BKy \<and> ranks - WKy = 3 \<and> 3 \<le> ranks - WRy \<and> \<bar>WKx - (ranks - 1 - BKy)\<bar> = 1 \<and> \<bar>WRx - (ranks - 1 - BKy)\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(BKy = 0 \<and> BKx = files - 1 \<and> WKy = 1 \<and> files - WKx = 3 \<and> 2 < WRy \<and> 0 < files - 1 - WRx \<or>
 BKy = 0 \<and> BKx = files - 2 \<and> WKy = 2 \<and> files - WKx = 2 \<and> 1 \<le> WRy \<and> files - WRx = 3 \<or>
 BKy = 0 \<and> WKy = 2 \<and> 2 \<le> WRy \<and> \<bar>files - 1 - WKx - BKy\<bar> = 1 \<and> \<bar>files - 1 - WRx - BKy\<bar> = 1 \<and> WKx \<noteq> WRx) \<or>
(BKy = 0 \<and> BKx = 0 \<and> WKy = 1 \<and> WKx = 2 \<and> 2 < WRy \<and> 0 < WRx \<or>
 BKy = 0 \<and> BKx = 1 \<and> WKy = 2 \<and> WKx = 1 \<and> 1 \<le> WRy \<and> WRx = 2 \<or>
 BKy = 0 \<and> WKy = 2 \<and> 2 \<le> WRy \<and> \<bar>WKx - BKx\<bar> = 1 \<and> \<bar>WRx - BKx\<bar> = 1 \<and> WKx \<noteq> WRx)"

(* COMMENT: the original ready_to_mate definition is based on symmetries, so the proof should also be based on symmetries *)

lemma reflectx_ready_to_mate:
  assumes p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
  assumes *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and
  "p' = reflectx_p p" and
  "(WKx', WKy') = WK p'" "(BKx', BKy') = BK p'" "(WRx', WRy') = WR p'"
  shows "LIA.ready_to_mate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> LIA.ready_to_mate WKx' WKy' BKx' BKy' WRx' WRy'"
using assms legal_position_black_on_turn[OF * p(1-2)]
by (simp add: ready_to_mate_def legal_position_black_on_turn_def all_on_board_def board_def invar_def) (rule, smt+)

lemma reflecty_ready_to_mate:
  assumes p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
  assumes *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and
  "p' = reflecty_p p" and
  "(WKx', WKy') = WK p'" "(BKx', BKy') = BK p'" "(WRx', WRy') = WR p'"
  shows "LIA.ready_to_mate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> LIA.ready_to_mate WKx' WKy' BKx' BKy' WRx' WRy'"
using assms legal_position_black_on_turn[OF * p(1-2)]
by (simp add: ready_to_mate_def legal_position_black_on_turn_def all_on_board_def board_def invar_def) (rule, smt+)

lemma reflectdiag_ready_to_mate:
  assumes "files = ranks"
  assumes p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
  assumes *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and
  "p' = reflectdiag_p p" and
  "(WKx', WKy') = WK p'" "(BKx', BKy') = BK p'" "(WRx', WRy') = WR p'"
  shows "LIA.ready_to_mate WKx WKy BKx BKy WRx WRy \<longleftrightarrow> LIA.ready_to_mate WKx' WKy' BKx' BKy' WRx' WRy'"
using assms legal_position_black_on_turn[OF * p(1-2)]
by (simp add: ready_to_mate_def legal_position_black_on_turn_def all_on_board_def board_def invar_def) (rule, smt+)

(* COMMENT: a variant of the original definition, but without the diagonal reflection - an intermediate step that makes the proof easier *)
definition ready_to_mate' :: "KRKPosition \<Rightarrow> bool" where
  "ready_to_mate' p \<longleftrightarrow> 
     (let p = canon_xy p; (WKx, WKy) = WK p; (BKx, BKy) = BK p; (WRx, WRy) = WR p 
       in 
          (BKx = 0 \<and> BKy = 0 \<and> WKx = 1 \<and> WKy = 2 \<and> WRx > 2 \<and> WRy > 0) \<or>
          (BKx = 0 \<and> BKy = 1 \<and> WKx = 2 \<and> WKy = 1 \<and> WRx \<ge> 1 \<and> WRy = 2) \<or>
          (BKx = 0 \<and> WKx = 2 \<and> WRx \<ge> 2 \<and> abs (WKy - BKy) = 1 \<and> abs (WRy - BKy) = 1 \<and> WKy \<noteq> WRy) \<or>
          (BKy = 0 \<and> BKx = 0 \<and> WKy = 1 \<and> WKx = 2 \<and> WRy > 2 \<and> WRx > 0) \<or>
          (BKy = 0 \<and> BKx = 1 \<and> WKy = 2 \<and> WKx = 1 \<and> WRy \<ge> 1 \<and> WRx = 2) \<or>
          (BKy = 0 \<and> WKy = 2 \<and> WRy \<ge> 2 \<and> abs (WKx - BKx) = 1 \<and> abs (WRx - BKx) = 1 \<and> WKx \<noteq> WRx)
     )"

lemma [simp]: "WRcaptured (maybe_reflectx p) \<longleftrightarrow> WRcaptured p"  "WRcaptured (maybe_reflecty p) \<longleftrightarrow> WRcaptured p"
by (simp_all add: maybe_reflectx_def maybe_reflecty_def)

lemma [simp]: "KRK.all_on_board (maybe_reflectx p) \<longleftrightarrow> KRK.all_on_board p"
unfolding all_on_board_def maybe_reflectx_def
by (cases "WK p", cases "BK p", cases "WR p") (auto simp add: board_def)

lemma [simp]: "KRK.all_on_board (maybe_reflecty p) \<longleftrightarrow> KRK.all_on_board p"
unfolding all_on_board_def maybe_reflecty_def
by (cases "WK p", cases "BK p", cases "WR p") (auto simp add: board_def)

lemma ready_to_mate':
  assumes "\<not> WRcaptured p" "KRK.all_on_board p"
  shows  "ready_to_mate' p \<longleftrightarrow> KRKStrategy.ready_to_mate p"
proof-
  let ?p = "maybe_reflecty (maybe_reflectx p)"
  have "KRK.all_on_board ?p"
    using `KRK.all_on_board p`
    by simp
  thus ?thesis
    using is_canon_xy_maybe_reflectxy[of p] `\<not> WRcaptured p`
    unfolding KRKStrategy.ready_to_mate_def ready_to_mate'_def canon_def canon_xy_def maybe_reflectdiag_def
    by (cases "BK ?p", cases "WK ?p", cases "WR ?p")
       (simp add: Let_def is_canon_diag_def is_canon_x_def is_canon_y_def KRK.all_on_board_def ChessRules.board_def, smt)
qed

lemma reflectx_ready_to_mate' [simp]:
  shows "ready_to_mate' (reflectx_p p) \<longleftrightarrow> ready_to_mate' p"
unfolding ready_to_mate'_def
by (subst reflectx_canon_xy, simp)

lemma reflecty_ready_to_mate' [simp]:
  shows "ready_to_mate' (reflecty_p p) \<longleftrightarrow> ready_to_mate' p"
unfolding ready_to_mate'_def
by (subst reflecty_canon_xy, simp)

lemma ready_to_mate:
  assumes "files \<ge> 4" "ranks \<ge> 4" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.ready_to_mate p \<longleftrightarrow> LIA.ready_to_mate WKx WKy BKx BKy WRx WRy"
proof-
  have "\<forall> p WKx WKy BKx BKy WRx WRy. WK p = (WKx, WKy) \<and> BK p = (BKx, BKy) \<and> WR p = (WRx, WRy) \<and> BlackOnTurn p \<and> \<not> WRcaptured p \<and> legal_position p \<longrightarrow> (ready_to_mate' p \<longleftrightarrow> LIA.ready_to_mate WKx WKy BKx BKy WRx WRy)" (is "\<forall> p. ?P p")
  proof (rule symmetry_xy)
    show "\<forall> p. ?P (reflectx_p p) \<longrightarrow> ?P p"
      using reflectx_ready_to_mate
      by auto (* metis+ *)
  next
    show "\<forall> p. ?P (reflecty_p p) \<longrightarrow> ?P p"
      using reflecty_ready_to_mate
      by auto (* metis+ *)
  next
    show "\<forall> p. Symmetry.is_canon_xy p \<longrightarrow> ?P p"
    proof (rule allI, rule impI, (rule allI)+, rule impI, (erule conjE)+)
      fix p WKx WKy WRx WRy BKx BKy
      assume "Symmetry.is_canon_xy p"
      assume *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"
      assume p: "BlackOnTurn p" "\<not> WRcaptured p" "legal_position p"
      show "ready_to_mate' p \<longleftrightarrow> LIA.ready_to_mate WKx WKy BKx BKy WRx WRy"
        unfolding ready_to_mate_cond_def ready_to_mate'_def
        using is_canon_xy_canon_xy_id[of p] `Symmetry.is_canon_xy p` * p `files \<ge> 4` `ranks \<ge> 4`
        apply (subst (asm) legal_position_black_on_turn[OF * p(1-2)])
        apply (simp add: Symmetry.is_canon_xy_def Symmetry.is_canon_x_def Symmetry.is_canon_y_def ready_to_mate_def legal_position_black_on_turn_def invar_def all_on_board_def board_def kings_separated_def)
        by  (rule, smt+)
    qed
  qed
  thus ?thesis
    using assms ready_to_mate'
    by (simp add: legal_position_def)
qed

(* KRKStrategy.no_ready_to_mate_WR *)
(* Note: Requires files \<ge> 4 and ranks \<ge> 4 because of ready_to_mate *)
definition no_ready_to_mate_WR where
  "no_ready_to_mate_WR WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all_n (files + ranks) (\<lambda>x. let (sqx, sqy) = rooks_square (WRx, WRy) x in WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> ready_to_mate WKx WKy BKx BKy sqx sqy)"

lemma no_ready_to_mate_WR:
  assumes "files \<ge> 4" "ranks \<ge> 4" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_ready_to_mate_WR p \<longleftrightarrow> LIA.no_ready_to_mate_WR WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_ready_to_mate_WR_def no_ready_to_mate_WR_def Let_def split_def
proof (rule all_n_eq, rule allI, rule impI)
  fix i::int
  let ?sq = "rooks_square (WR p) i"
  let ?sqxy = "rooks_square (WRx, WRy) i"
  obtain sqx sqy where "(sqx, sqy) = ?sq"
    by (cases ?sq, auto)
  assume "1 \<le> i \<and> i \<le> files + ranks"
  thus "(KRK.WR_can_move_to p ?sq \<longrightarrow> \<not> ready_to_mate_cond (moveWR p ?sq)) =
        (WR_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sqxy) (snd ?sqxy) \<longrightarrow> \<not> ready_to_mate WKx WKy BKx BKy (fst ?sqxy) (snd ?sqxy))"
    using KRK.WR_can_move_to[of p ?sq, OF p(3) p(1-2)] WR_can_move_to[OF assms(3-), of sqx sqy]
    unfolding KRK.legal_move_WR_def
    using ready_to_mate[where p="moveWR p ?sq", of WKx WKy BKx BKy sqx sqy] `(sqx, sqy) = ?sq` assms
    by (auto simp add: split_def Let_def rooks_square_def)
qed

(* no_ready_to_mate - optimized definition. Note: requires files, ranks \<ge> 3 *)
definition no_ready_to_mate_WR_opt where
  "no_ready_to_mate_WR_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy 0 WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy 0 WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy (files - 1) WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy (files - 1) WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx 0 \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx 0) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (ranks - 1) \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx (ranks - 1)) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy 2 WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy 2 WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy (files - 3) WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy (files - 3) WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx 2 \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx 2) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (ranks - 3) \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx (ranks - 3)) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx - 1) WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy (BKx - 1) WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx + 1) WRy \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy (BKx + 1) WRy) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy - 1) \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx (BKy - 1)) \<and>
     (LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy + 1) \<longrightarrow> \<not> LIA.ready_to_mate WKx WKy BKx BKy WRx (BKy + 1))"

lemma no_ready_to_mate_WR_opt':
  assumes "files \<ge> 3" "ranks \<ge> 3" "LIA.no_ready_to_mate_WR WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_ready_to_mate_WR_opt WKx WKy BKx BKy WRx WRy"
proof-
  have *:
       "rooks_square (WRx, WRy) 1 = (0, WRy)"
       "rooks_square (WRx, WRy) files = (files - 1, WRy)"
       "rooks_square (WRx, WRy) (files + 1) = (WRx, 0)"
       "rooks_square (WRx, WRy) (files + ranks) = (WRx, ranks - 1)"
       "rooks_square (WRx, WRy) 3 = (2, WRy)"
       "rooks_square (WRx, WRy) (files - 2) = (files - 3, WRy)"
       "rooks_square (WRx, WRy) (files + 3) = (WRx, 2)"
       "rooks_square (WRx, WRy) (files + ranks - 2) = (WRx, ranks - 3)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx - 1) WRy \<longrightarrow> rooks_square (WRx, WRy) BKx = (BKx - 1, WRy)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx + 1) WRy \<longrightarrow> rooks_square (WRx, WRy) (BKx + 2) = (BKx + 1, WRy)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy - 1) \<longrightarrow> rooks_square (WRx, WRy) (files + BKy) = (WRx, BKy - 1)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy + 1) \<longrightarrow> rooks_square (WRx, WRy) (files + BKy + 2) = (WRx, BKy + 1)"
    using `files \<ge> 3` `ranks \<ge> 3`
    unfolding rooks_square_def WR_can_move_to_def
    by simp_all
  show ?thesis
    unfolding no_ready_to_mate_WR_opt_def Let_def
    apply (rule conjI)
    using assms *(1) `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="1" in allE)
    apply simp
    apply (rule conjI)
    using assms *(2) `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files" in allE)
    apply simp
    apply (rule conjI)
    using assms *(3) `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files+1" in allE)
    apply simp
    apply (rule conjI)
    using assms *(4) `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files+ranks" in allE)
    apply simp
    apply (rule conjI)
    using assms *(5)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="3" in allE)
    apply simp
    apply (rule conjI)
    using assms *(6)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files - 2" in allE)
    apply simp
    apply (rule conjI)
    using assms *(7)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files + 3" in allE)
    apply simp
    apply (rule conjI)
    using assms *(8)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files + ranks - 2" in allE)
    apply simp
    apply (rule conjI)
    using assms *(9)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="BKx" in allE)
    apply (rule impI)
    apply (simp add: LIA.WR_can_move_to_def)
    apply (rule conjI)
    using assms *(10)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="BKx + 2" in allE)
    apply (rule impI)
    apply (simp add: LIA.WR_can_move_to_def)
    apply (rule conjI)
    using assms *(11)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files + BKy" in allE)
    apply (rule impI)
    apply (simp add: LIA.WR_can_move_to_def)
    using assms *(12)  `files \<ge> 3` `ranks \<ge> 3`
    unfolding no_ready_to_mate_WR_def all_n_def
    apply (erule_tac x="files + BKy + 2" in allE)
    apply (rule impI)
    apply (simp add: LIA.WR_can_move_to_def)
    done
qed

lemma no_ready_to_mate_WR_opt:
  assumes "files \<ge> 4" "ranks \<ge> 4" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_ready_to_mate_WR p \<longrightarrow> no_ready_to_mate_WR_opt WKx WKy BKx BKy WRx WRy"
using assms
using no_ready_to_mate_WR[OF assms] no_ready_to_mate_WR_opt'
by simp


(* KRKStrategy.no_ready_to_mate_WK *)

(* Note: requires files \<ge> 4 and ranks \<ge> 4 because of ready_to_mate *)
definition no_ready_to_mate_WK where
  "no_ready_to_mate_WK WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all8 (\<lambda>x. let (sqx, sqy) = kings_square (WKx, WKy) x in WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> ready_to_mate sqx sqy BKx BKy WRx WRy)"

lemma no_ready_to_mate_WK:
  assumes "4 \<le> files" "4 \<le> ranks" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_ready_to_mate_WK p \<longleftrightarrow> LIA.no_ready_to_mate_WK WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_ready_to_mate_WK_def no_ready_to_mate_WK_def Let_def split_def
proof (rule all8_eq, rule allI, rule impI)
  fix i::int
  let ?sq = "kings_square (WK p) i"
  let ?sqxy = "kings_square (WKx, WKy) i"
  obtain sqx sqy where "(sqx, sqy) = ?sq"
    by (cases ?sq, auto)
  assume "1 \<le> i \<and> i \<le> 8"
  thus "(KRK.WK_can_move_to p ?sq \<longrightarrow> \<not> ready_to_mate_cond (moveWK p ?sq)) =
        (WK_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sqxy) (snd ?sqxy) \<longrightarrow> \<not> ready_to_mate  (fst ?sqxy) (snd ?sqxy) BKx BKy WRx WRy)"
    using KRK.WK_can_move_to[of p ?sq, OF p(3) p(1)] WK_can_move_to[OF * p(1-2), of sqx sqy]
    unfolding KRK.legal_move_WK_def
    using ready_to_mate[where p="moveWK p ?sq", of  sqx sqy BKx BKy WRx WRy] `(sqx, sqy) = ?sq` assms
    by (metis WhiteOnTurn_MoveWK moveWK_fields fst_conv snd_conv ready_to_mate_cond_def)
qed

(* KRKStrategy.squeeze_cond *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used *)
definition squeeze_cond where
 "squeeze_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow>
    (LIA.room WRx' WRy' BKx' BKy' < room WRx WRy BKx BKy \<and>
     LIA.chebyshev_dist WKx' WKy' WRx' WRy' \<le> LIA.chebyshev_dist BKx' BKy' WRx' WRy' \<and>
     (between BKx' WRx' WKx' \<or> between BKy' WRy' WKy') \<and> \<not> LIA.stalemate_opt WKx' WKy' BKx' BKy' WRx' WRy')"

lemma squeeze_cond:
  assumes "files \<ge> 2" "ranks \<ge> 2"
          "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" 
          "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
          "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" 
          "BlackOnTurn p'" "\<not> WRcaptured p'" "legal_position p'"
  shows "KRKStrategy.squeeze_cond p p' \<longleftrightarrow> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms stalemate_opt[of p' WKx' WKy' BKx' BKy' WRx' WRy']
by (simp add: KRKStrategy.squeeze_cond_def Let_def KRK.stalemate_def WR_exposed_def WR_divides_def chebyshev_dist room squeeze_cond_def) (metis linorder_not_le)

lemma squeeze_cond_moveWR:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" "sq = (sqx, sqy)"
  "KRK.WR_can_move_to p sq"
  shows "KRKStrategy.squeeze_cond p (moveWR p sq) \<longleftrightarrow> LIA.squeeze_cond  WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy"
using assms squeeze_cond[OF assms(1-5), of "moveWR p sq" WKx WKy BKx BKy sqx sqy] KRK.WR_can_move_to[of p sq]
by (simp add: KRK.legal_move_WR_def squeeze_cond_def)

(* KRKStrategy.no_squeeze *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used in squeeze_cond *)
definition no_squeeze where
  "no_squeeze WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all_n (files + ranks) (\<lambda>x. let sq = rooks_square (WRx, WRy) x; (sqx, sqy) = sq in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_squeeze:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_squeeze p \<longleftrightarrow> LIA.no_squeeze WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_squeeze_def LIA.no_squeeze_def Let_def split_def
proof (rule all_n_eq, rule allI, rule impI)
  fix i::int
  let ?sq = "rooks_square (WRx, WRy) i"
  assume "1 \<le> i \<and> i \<le> files + ranks"
  thus "(KRK.WR_can_move_to p (rooks_square (WR p) i) \<longrightarrow> \<not> (KRKStrategy.squeeze_cond p (moveWR p (rooks_square (WR p) i)))) =
        (WR_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow>
         \<not> squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy (fst ?sq) (snd ?sq))"
    using assms WR_can_move_to[OF assms(3-)] squeeze_cond_moveWR[OF assms]
    by (simp add: rooks_square_def)
qed

(* Optimized definition of no squeeze *)
definition no_squeeze_opt where
 "no_squeeze_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
  (let sqx = WRx; sqy = (WKy + BKy) div 2 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = (WKy + BKy + 1) div 2 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = (WKx + BKx) div 2; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy)  \<and>
  (let sqx = (WKx + BKx + 1) div 2; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and> 
  (let sqx = WRx; sqy = BKy - WKx + WRx in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WKx - BKy + WRy; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = WKx + BKy - WRx in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WKx + BKy - WRy; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = WKy - BKx + WRx in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = BKx - WKy + WRy; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = BKx + WKy - WRx in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = BKx + WKy - WRy; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = BKx + 1; sqy =  WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = BKx - 1; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = BKy + 1 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let sqx = WRx; sqy = BKy - 1 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.squeeze_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_squeeze_opt':
  assumes "LIA.all_on_board WKx WKy BKx BKy WRx WRy" "LIA.no_squeeze WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_squeeze_opt WKx WKy BKx BKy WRx WRy"
proof-
  have  *:
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx ((WKy + BKy) div 2) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (WKy + BKy) div 2) = (WRx, (WKy + BKy) div 2)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx ((WKy + BKy + 1) div 2) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (WKy + BKy + 1) div 2) = (WRx, (WKy + BKy + 1) div 2)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy ((WKx + BKx) div 2) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (WKx + BKx) div 2) = ((WKx + BKx) div 2, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy ((WKx + BKx + 1) div 2) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (WKx + BKx + 1) div 2) = ((WKx + BKx + 1) div 2, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy - WKx + WRx) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (BKy - WKx + WRx)) = (WRx, BKy - WKx + WRx)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (WKx - BKy + WRy) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (WKx - BKy + WRy)) = (WKx - BKy + WRy, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (WKx + BKy - WRx) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (WKx + BKy - WRx)) = (WRx, WKx + BKy - WRx)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (WKx + BKy - WRy) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (WKx + BKy - WRy)) = (WKx + BKy - WRy, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (WKy - BKx + WRx) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (WKy - BKx + WRx)) = (WRx, WKy - BKx + WRx)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx - WKy + WRy) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (BKx - WKy + WRy)) = (BKx - WKy + WRy, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKx + WKy - WRx) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (BKx + WKy - WRx)) = (WRx, BKx + WKy - WRx)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx + WKy - WRy) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (BKx + WKy - WRy)) = (BKx + WKy - WRy, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx + 1) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (BKx + 1)) = (BKx + 1, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy (BKx - 1) WRy \<longrightarrow> rooks_square (WRx, WRy) (1 + (BKx - 1)) = (BKx - 1, WRy)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy + 1) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (BKy + 1)) = (WRx, BKy + 1)"
    "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (BKy - 1) \<longrightarrow> rooks_square (WRx, WRy) (files + 1 + (BKy - 1)) = (WRx, BKy - 1)"
    unfolding rooks_square_def Let_def WR_can_move_to_def
    by simp_all
  show ?thesis
    using assms(2)
    unfolding no_squeeze_def all_n_def no_squeeze_opt_def Let_def
    apply -
    apply (rule conjI, rule impI)
    using *(1)
    apply (erule_tac x="files + 1 + (WKy + BKy) div 2" in allE, simp add: WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(2)
    apply (erule_tac x="files + 1 + (WKy + BKy + 1) div 2" in allE, simp add: WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(3)
    apply (erule_tac x="1 + (WKx + BKx) div 2" in allE, simp add: WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(4)
    apply (erule_tac x="1 + (WKx + BKx + 1) div 2" in allE, simp add: WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(5)
    apply (erule_tac x="files + 1 + (BKy - WKx + WRx)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(6)
    apply (erule_tac x="1 + (WKx - BKy + WRy)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(7)
    apply (erule_tac x="files + 1 + (WKx + BKy - WRx)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(8)
    apply (erule_tac x="1 + (WKx + BKy - WRy)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(9)
    apply (erule_tac x="files + 1 + (WKy - BKx + WRx)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(10)
    apply (erule_tac x="1 + (BKx - WKy + WRy)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(11)
    apply (erule_tac x="files + 1 + (BKx + WKy - WRx)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(12)
    apply (erule_tac x="1 + (BKx + WKy - WRy)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(13)
    apply (erule_tac x="1 + (BKx + 1)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(14)
    apply (erule_tac x="1 + (BKx - 1)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule conjI, rule impI)
    using *(15)
    apply (erule_tac x="files + 1 + (BKy + 1)" in allE, simp add: LIA.WR_can_move_to_def)
    apply (rule impI)
    using *(16)
    apply (erule_tac x="files + 1 + (BKy - 1)" in allE, simp add: LIA.WR_can_move_to_def)
    done
qed

(*
lemma 
  assumes "LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy"
  "LIA.no_immediate_mate_opt WKx WKy BKx BKy WRx WRy \<and>
   LIA.no_ready_to_mate_WK WKx WKy BKx BKy WRx WRy \<and> 
   LIA.no_ready_to_mate_WR_opt WKx WKy BKx BKy WRx WRy"
  "LIA.no_squeeze_opt WKx WKy BKx BKy WRx WRy"
  shows "no_squeeze WKx WKy BKx BKy WRx WRy"
using assms
unfolding legal_position_white_on_turn_def no_immediate_mate_opt_def no_ready_to_mate_WK_def no_ready_to_mate_WR_opt_def
apply (simp add: all8_def kings_square_def)
unfolding LIA.no_squeeze_def all_n_def no_squeeze_opt_def
apply (simp add: Let_def rooks_square_def)
unfolding LIA_full_unfold
using [[ z3_with_extensions ]]
using [[ z3_new_extensions ]]
apply smt
done
*)

lemma no_squeeze_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" and *: "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" and p: "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRK.all_on_board p \<and> KRKStrategy.no_squeeze p \<longrightarrow> no_squeeze_opt WKx WKy BKx BKy WRx WRy"
using assms
using LIA.all_on_board[OF *] no_squeeze[OF assms] no_squeeze_opt'
by simp


(* KRKStrategy.approach_cond *)

(* Note: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used *)
definition approach_cond where
"approach_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow> 
   (LIA.approach_critical_square WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<and>
    LIA.chebyshev_dist WKx' WKy' WRx' WRy' \<le> LIA.chebyshev_dist BKx' BKy' WRx' WRy' \<and>
    (between BKx' WRx' WKx' \<or> between BKy' WRy' WKy' \<or> LIA.Lpattern WKx' WKy' BKx' BKy' WRx' WRy') \<and>
    (LIA.room WRx' WRy' BKx' BKy' \<le> 3 \<longrightarrow> (WKx' \<noteq> 0 \<or> BKx' \<noteq> 0) \<and> (WKx' \<noteq> files - 1 \<or> BKx' \<noteq> files - 1) \<and> (WKy' \<noteq> 0 \<or> BKy' \<noteq> 0) \<and> (WKy' \<noteq> ranks - 1 \<or> BKy' \<noteq> ranks - 1) \<and> LIA.not_back_move WKx WKy BKx BKy WRx WRy WKx' WKy') \<and>
    \<not> LIA.stalemate_opt WKx' WKy' BKx' BKy' WRx' WRy')"

lemma approach_cond:
  assumes "files \<ge> 2" "ranks \<ge> 2"
          "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" 
          "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
  "BlackOnTurn p'" "\<not> WRcaptured p'" "legal_position p'"
  shows "KRKStrategy.approach_cond p p' \<longleftrightarrow> LIA.approach_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms approach_critical_square[OF assms(3-8)] stalemate_opt[OF assms(1-2) assms(6-8)] Lpattern[OF assms(6-8)]
unfolding KRKStrategy.approach_cond_def LIA.approach_cond_def
by (subst not_back_move, simp add: KRKStrategy.approach_cond_def KRK.stalemate_def WR_exposed_def WR_divides_def chebyshev_dist room approach_cond_def KRKStrategy.back_move_def back_move_def)
   (rule iffI, metis le_less_linear, simp)

lemma approach_cond_moveWK:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" "sq = (sqx, sqy)" "KRK.WK_can_move_to p sq"
  shows "KRKStrategy.approach_cond p (moveWK p sq) \<longleftrightarrow> LIA.approach_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy"
using assms approach_cond[OF assms(1-5), of "moveWK p sq" sqx sqy BKx BKy WRx WRy] KRK.WK_can_move_to[of p sq]
by (simp add: KRK.legal_move_WK_def)

(* KRKStrategy.no_approach *)

(* Note: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used in approach_cond *)
definition no_approach where
 "no_approach WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all8 (\<lambda>x. let sq = kings_square (WKx, WKy) x; (sqx, sqy) = sq in LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.approach_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy)"

lemma no_approach:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_approach p \<longleftrightarrow> LIA.no_approach WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_approach_def LIA.no_approach_def Let_def split_def
proof (rule all8_eq, rule allI, rule impI)
  fix i::int
  let ?sq = "kings_square (WKx, WKy) i"
  assume "1 \<le> i \<and> i \<le> 8"
  show "(KRK.WK_can_move_to p (kings_square (WK p) i) \<longrightarrow> \<not> KRKStrategy.approach_cond p (moveWK p (kings_square (WK p) i))) =
        (WK_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow>
         \<not> approach_cond WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) BKx BKy WRx WRy)"
    using assms KRK.WK_can_move_to[of p ?sq]
    using approach_cond_moveWK[OF assms] WK_can_move_to[OF assms(3-7)]
    by (auto, force, force, fastforce, force, force)
qed

(* KRKStrategy.no_approach_diag *)

definition no_approach_diag where
 "no_approach_diag WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all4 (\<lambda>x. let sq = kings_square (WKx, WKy) x; (sqx, sqy) = sq in LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.approach_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy)"

lemma no_approach_diag:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows 
  "KRKStrategy.no_approach_diag p \<longleftrightarrow> LIA.no_approach_diag WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_approach_diag_def _no_approach_diag_def Let_def split_def
proof (rule all4_eq, rule allI, rule impI)
  fix i::int
  assume "1 \<le> i \<and> i \<le> 4"
  let ?sq = "kings_square (WKx, WKy) i"
  show "(KRK.WK_can_move_to p (kings_square (WK p) i) \<longrightarrow> \<not> KRKStrategy.approach_cond p (moveWK p (kings_square (WK p) i))) =
        (WK_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow>
         \<not> approach_cond WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) BKx BKy WRx WRy)"
    using assms KRK.WK_can_move_to[of p ?sq]
    using approach_cond_moveWK[OF assms] WK_can_move_to[OF assms(3-7)]
    by (auto, force, force, fastforce, force, force)
qed


(* KRKStrategy.keep_room_cond *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used *)
definition keep_room_cond where
 "keep_room_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow>
       (LIA.chebyshev_dist WKx' WKy' WRx' WRy' \<le> LIA.chebyshev_dist BKx' BKy' WRx' WRy' \<and>
        (between BKx' WRx' WKx' \<or> between BKy' WRy' WKy') \<and>
        \<not> LIA.chebyshev_dist WKx WKy WRx WRy < LIA.chebyshev_dist WKx' WKy' WRx' WRy' \<and>
        (LIA.room WRx' WRy' BKx' BKy' \<le> 3 \<longrightarrow> (WKx' \<noteq> 0 \<or> BKx' \<noteq> 0) \<and> (WKx' \<noteq> files - 1 \<or> BKx' \<noteq> files - 1) \<and> (WKy' \<noteq> 0 \<or> BKy' \<noteq> 0) \<and> (WKy' \<noteq> ranks - 1 \<or> BKy' \<noteq> ranks - 1)) \<and>
        \<not> LIA.stalemate_opt WKx' WKy' BKx' BKy' WRx' WRy')"

lemma keep_room_cond:
  assumes "files \<ge> 2" "ranks \<ge> 2"
  "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" 
  "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
  "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  "BlackOnTurn p'" "\<not> WRcaptured p'" "legal_position p'"
  shows "KRKStrategy.keep_room_cond p p' \<longleftrightarrow> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
using assms stalemate_opt[OF assms(1-2) assms(6-8)]
by (simp add: KRKStrategy.keep_room_cond_def WR_exposed_def WR_divides_def WRandWKdiverging_def KRK.stalemate_def chebyshev_dist room keep_room_cond_def) (rule iffI, metis le_less_linear, simp)

lemma keep_room_cond_moveWK:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" "sq = (sqx, sqy)" "KRK.WK_can_move_to p sq"
  shows "KRKStrategy.keep_room_cond p (moveWK p sq) \<longleftrightarrow> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy"
using assms keep_room_cond[OF assms(1-5), of "moveWK p sq" sqx sqy BKx BKy WRx WRy] KRK.WK_can_move_to[of p sq]
by (simp add: KRK.legal_move_WK_def)


(* KRKStrategy.no_keep_room *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used in keep_room_cond *)
definition no_keep_room where
  "no_keep_room WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all8 (\<lambda>x. let sq = kings_square (WKx, WKy) x; (sqx, sqy) = sq in LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy)"

lemma no_keep_room:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows 
  "KRKStrategy.no_keep_room p \<longleftrightarrow> LIA.no_keep_room WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_keep_room_def no_keep_room_def Let_def split_def
proof (rule all8_eq, rule allI, rule impI)
  fix i::int
  assume "1 \<le> i \<and> i \<le> 8"
  let ?sq = "kings_square (WKx, WKy) i"
  show "(KRK.WK_can_move_to p (kings_square (WK p) i) \<longrightarrow> \<not> KRKStrategy.keep_room_cond p (moveWK p (kings_square (WK p) i))) \<longleftrightarrow>
        (LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow>
         \<not> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) BKx BKy WRx WRy)"
    using assms KRK.WK_can_move_to[of p ?sq]
    using keep_room_cond_moveWK[OF assms] WK_can_move_to[OF assms(3-7)]
    by (auto, force, force, fastforce, force, force)
qed


(* KRKStrategy.no_keep_room_diag *)

definition no_keep_room_diag where
 "no_keep_room_diag WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all4 (\<lambda>x. let sq = kings_square (WKx, WKy) x; (sqx, sqy) = sq in LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy sqx sqy BKx BKy WRx WRy)"

lemma no_keep_room_diag:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows 
  "KRKStrategy.no_keep_room_diag p \<longleftrightarrow> LIA.no_keep_room_diag WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_keep_room_diag_def no_keep_room_diag_def Let_def split_def
proof (rule all4_eq, rule allI, rule impI)
  fix i::int
  assume "1 \<le> i \<and> i \<le> 4"
  let ?sq = "kings_square (WKx, WKy) i"
  show "(KRK.WK_can_move_to p (kings_square (WK p) i) \<longrightarrow> \<not> KRKStrategy.keep_room_cond p (moveWK p (kings_square (WK p) i))) =
        (LIA.WK_can_move_to WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) \<longrightarrow>
         \<not> LIA.keep_room_cond WKx WKy BKx BKy WRx WRy (fst ?sq) (snd ?sq) BKx BKy WRx WRy)"
    using assms KRK.WK_can_move_to[of p ?sq]
    using keep_room_cond_moveWK[OF assms] WK_can_move_to[OF assms(3-7)]
    by (auto, force, force, fastforce, force, force)
qed


(* KRKStrategy.rook_home_cond *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used *)
definition rook_home_cond where
"rook_home_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow>
  (BKx < WKx \<and> WRx \<noteq> WKx - 1 \<and> WRx' = WKx - 1 \<or>
   WKx < BKx \<and> WRx \<noteq> WKx + 1 \<and> WRx' = WKx + 1 \<or>
   BKy < WKy \<and> WRy \<noteq> WKy - 1 \<and> WRy' = WKy - 1 \<or>
   WKy < BKy \<and> WRy \<noteq> WKy + 1 \<and> WRy' = WKy + 1 \<or>
   WRx = WKx \<and> WKx = BKx \<and> (WRx' = WKx + 1 \<or> WRx' = WKx - 1) \<or> WRy = WKy \<and> WKy = BKy \<and> (WRy' = WKy + 1 \<or> WRy' = WKy - 1)) \<and>
  (king_scope BKx' BKy' WRx' WRy' \<longrightarrow> king_scope WKx' WKy' WRx' WRy') \<and>
  \<not> LIA.stalemate_opt WKx' WKy' BKx' BKy' WRx' WRy'"

lemma rook_home_cond:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
  "BlackOnTurn p'" "\<not> WRcaptured p'" "legal_position p'"
  shows "KRKStrategy.rook_home_cond p p' \<longleftrightarrow> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
  unfolding KRKStrategy.rook_home_cond_def KRKStrategy.divide_attempt_def
  using assms LIA.stalemate_opt[OF assms(1-2) assms(6-8)]
  by (simp add: Let_def split_def WK_protects_WR_def KRK.stalemate_def chebyshev_dist rook_home_cond_def king_scope) (rule iffI, metis, simp)

lemma rook_home_cond_moveWR:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"  "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" "sq = (sqx, sqy)" "KRK.WR_can_move_to p sq"
  shows "KRKStrategy.rook_home_cond p (moveWR p sq) \<longleftrightarrow> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy"
using assms rook_home_cond[OF assms(1-5), of "moveWR p sq" WKx WKy BKx BKy sqx sqy] KRK.WR_can_move_to[of p sq]
by (simp add: KRK.legal_move_WR_def)


(* KRKStrategy.no_rook_home *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used in rook_home_cond *)
definition no_rook_home where
  "no_rook_home WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all_n (files + ranks) (\<lambda>x. let sq = rooks_square (WRx, WRy) x; (sqx, sqy) = sq in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_rook_home:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_rook_home p \<longleftrightarrow> LIA.no_rook_home WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_rook_home_def no_rook_home_def Let_def split_def
proof (rule all_n_eq, rule allI, rule impI)
  fix i::int
  let ?sq = "rooks_square (WRx, WRy) i"
  show "(KRK.WR_can_move_to p (rooks_square (WR p) i) \<longrightarrow> \<not> KRKStrategy.rook_home_cond p (moveWR p (rooks_square (WR p) i))) =
        (WR_can_move_to WKx WKy BKx BKy WRx WRy (fst (rooks_square (WRx, WRy) i)) (snd (rooks_square (WRx, WRy) i)) \<longrightarrow>
         \<not> rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy (fst (rooks_square (WRx, WRy) i)) (snd (rooks_square (WRx, WRy) i)))"
    using assms WR_can_move_to[OF assms(3-), of "fst ?sq" "snd ?sq"] rook_home_cond_moveWR[OF assms, of ?sq "fst ?sq" "snd ?sq"]
    by auto
qed


(* Opimized definition of no_rook_home. Note: requires files \<ge> 0 and ranks \<ge> 0 *)
definition no_rook_home_opt where
  "no_rook_home_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
     (let sqx = WRx; sqy = WKy - 1 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and> 
     (let sqx = WRx; sqy = WKy + 1 in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and> 
     (let sqx = WKx - 1; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
     (let sqx = WKx + 1; sqy = WRy in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_home_cond WKx WKy BKx BKy WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_rook_home_opt':
  assumes "files \<ge> 0" "ranks \<ge> 0" "LIA.no_rook_home WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_rook_home_opt WKx WKy BKx BKy WRx WRy"
proof-
  have *:
       "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (WKy - 1) \<longrightarrow> rooks_square (WRx, WRy) (files + WKy) = (WRx, WKy - 1)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy WRx (WKy + 1) \<longrightarrow> rooks_square (WRx, WRy) (files + WKy + 2) = (WRx, WKy + 1)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy (WKx - 1) WRy \<longrightarrow> rooks_square (WRx, WRy) WKx = (WKx - 1, WRy)"
       "WR_can_move_to WKx WKy BKx BKy WRx WRy (WKx + 1) WRy \<longrightarrow> rooks_square (WRx, WRy) (WKx + 2) = (WKx + 1, WRy)"
    using assms
    unfolding rooks_square_def WR_can_move_to_def
    by simp_all
  show ?thesis
    unfolding no_rook_home_opt_def Let_def
    apply (rule conjI)
    using assms *(1)
    unfolding no_rook_home_def all_n_def
    apply (erule_tac x="files + WKy" in allE)
    apply (rule impI)
    unfolding WR_can_move_to_def
    apply simp
    apply (rule conjI)
    using assms *(2)
    unfolding no_rook_home_def all_n_def
    apply (erule_tac x="files + WKy + 2" in allE)
    apply (rule impI)
    unfolding WR_can_move_to_def
    apply simp
    apply (rule conjI)
    using assms *(3)
    unfolding no_rook_home_def all_n_def
    apply (erule_tac x="WKx" in allE)
    apply (rule impI)
    unfolding WR_can_move_to_def
    apply simp
    using assms *(4)
    unfolding no_rook_home_def all_n_def
    apply (erule_tac x="WKx + 2" in allE)
    apply (rule impI)
    unfolding WR_can_move_to_def
    apply simp
    done
qed

(*
lemma 
  assumes "LIA.no_rook_home_opt WKx WKy BKx BKy WRx WRy" "LIA.legal_position_white_on_turn WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_rook_home WKx WKy BKx BKy WRx WRy"
  using assms
  unfolding LIA.no_rook_home_opt_def LIA.no_rook_home_def all_n_def Let_def
  unfolding legal_position_white_on_turn_def LIA.rook_home_cond_def
  by (simp add: rooks_square_def) smt
*)

lemma no_rook_home_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_rook_home p \<longrightarrow> LIA.no_rook_home_opt WKx WKy BKx BKy WRx WRy"
using no_rook_home[OF assms] no_rook_home_opt' assms(1-2)
by simp


(* KRKStrategy.rook_safe_cond *)
(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used *)
definition rook_safe_cond where
"rook_safe_cond WRx WRy WKx' WKy' BKx' BKy' WRx' WRy' \<longleftrightarrow>
  ((WRx \<noteq> 0 \<and> WRx' = 0 \<or> WRx \<noteq> files - 1 \<and> WRx' = files - 1 \<or> WRy \<noteq> 0 \<and> WRy' = 0 \<or> WRy \<noteq> ranks - 1 \<and> WRy' = ranks - 1) \<and>
    2 < LIA.chebyshev_dist BKx' BKy' WRx' WRy' \<and> \<not> LIA.stalemate_opt WKx' WKy' BKx' BKy' WRx' WRy')"

lemma rook_safe_cond:
  assumes  "files \<ge> 2" "ranks \<ge> 2" "WR p = (WRx, WRy)" "WK p' = (WKx', WKy')" "BK p' = (BKx', BKy')" "WR p' = (WRx', WRy')"
  "BlackOnTurn p'" "\<not> WRcaptured p'" "legal_position p'"
  shows "KRKStrategy.rook_safe_cond p p' \<longleftrightarrow> LIA.rook_safe_cond WRx WRy WKx' WKy' BKx' BKy' WRx' WRy'"
  unfolding KRKStrategy.rook_safe_cond_def LIA.rook_safe_cond_def move_WR_to_edge_def
using assms stalemate_opt[OF assms(1-2) assms(4-6)]
by (simp add: Let_def split_def KRK.stalemate_def chebyshev_dist) (rule iffI, metis, simp)

lemma rook_safe_cond_moveWR:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)"  "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p" "sq = (sqx, sqy)" "KRK.WR_can_move_to p sq"
  shows "KRKStrategy.rook_safe_cond p (moveWR p sq) \<longleftrightarrow> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy"
using assms rook_safe_cond[OF assms(1-2) assms(5), of "moveWR p sq" WKx WKy BKx BKy sqx sqy] KRK.WR_can_move_to[of p sq]
by (simp add: KRK.legal_move_WR_def)

(* KRKStrategy.no_rook_safe *)

(* NOTE: requires riles \<ge> 2 and ranks \<ge> 2 if stalemate_opt is used in rook_safe_cond *)
definition no_rook_safe where
 "no_rook_safe WKx WKy BKx BKy WRx WRy \<longleftrightarrow> all_n (files + ranks) (\<lambda>x. let sq = rooks_square (WRx, WRy) x; (sqx, sqy) = sq in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_rook_safe:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_rook_safe p \<longleftrightarrow> LIA.no_rook_safe WKx WKy BKx BKy WRx WRy"
unfolding KRKStrategy.no_rook_safe_def LIA.no_rook_safe_def Let_def split_def
proof (rule all_n_eq, rule allI, rule impI)
    fix i::int
    assume "1 \<le> i \<and> i \<le> files + ranks"
    thus "(KRK.WR_can_move_to p (rooks_square (WR p) i) \<longrightarrow> \<not> KRKStrategy.rook_safe_cond p (moveWR p (rooks_square (WR p) i))) =
        (WR_can_move_to WKx WKy BKx BKy WRx WRy (fst (rooks_square (WRx, WRy) i)) (snd (rooks_square (WRx, WRy) i)) \<longrightarrow>
         \<not> rook_safe_cond WRx WRy WKx WKy BKx BKy (fst (rooks_square (WRx, WRy) i))  (snd (rooks_square (WRx, WRy) i)))"
      using assms WR_can_move_to[OF assms(3-)] rook_safe_cond_moveWR[OF assms]
      by (simp add: rooks_square_def)
qed

(* Optimized definition of no_rook_safe *)
definition no_rook_safe_opt where
 "no_rook_safe_opt WKx WKy BKx BKy WRx WRy \<longleftrightarrow> 
  (let (sqx, sqy) = (0, WRy) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let (sqx, sqy) = (files - 1, WRy) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let (sqx, sqy) = (WRx, 0) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy) \<and>
  (let (sqx, sqy) = (WRx, ranks - 1) in LIA.WR_can_move_to WKx WKy BKx BKy WRx WRy sqx sqy \<longrightarrow> \<not> LIA.rook_safe_cond WRx WRy WKx WKy BKx BKy sqx sqy)"

lemma no_rook_safe_opt':
  assumes  "files \<ge> 1" "ranks \<ge> 1" "LIA.no_rook_safe WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_rook_safe_opt WKx WKy BKx BKy WRx WRy"
proof-
  have *:
    "rooks_square (WRx, WRy) 1 = (0, WRy)"
    "rooks_square (WRx, WRy) files = (files - 1, WRy)"
    "rooks_square (WRx, WRy) (files + 1) = (WRx, 0)"
    "rooks_square (WRx, WRy) (files + ranks) = (WRx, ranks - 1)"
    using `files \<ge> 1` `ranks \<ge> 1`
    by (simp_all add: rooks_square_def)
  show ?thesis
    using assms *
    unfolding no_rook_safe_def no_rook_safe_opt_def all_n_def Let_def
    apply simp
    apply (rule conjI, rule impI, erule_tac x="1" in allE, simp)
    apply (rule conjI, rule impI, erule_tac x="files" in allE, simp)
    apply (rule conjI, rule impI, erule_tac x="files+1" in allE, simp)
    apply (rule impI, erule_tac x="files+ranks" in allE, simp)
    done
qed

(*
lemma 
  assumes "LIA.no_rook_safe_opt WKx WKy BKx BKy WRx WRy"
  shows "LIA.no_rook_safe WKx WKy BKx BKy WRx WRy"
using assms
unfolding no_rook_safe_def no_rook_safe_opt_def Let_def
unfolding rook_safe_cond_def all_n_def
by (simp add: rooks_square_def) smt
*)

lemma no_rook_safe_opt:
  assumes "files \<ge> 2" "ranks \<ge> 2" "WK p = (WKx, WKy)" "BK p = (BKx, BKy)" "WR p = (WRx, WRy)" "WhiteOnTurn p" "\<not> WRcaptured p" "legal_position p"
  shows "KRKStrategy.no_rook_safe p \<longrightarrow> LIA.no_rook_safe_opt WKx WKy BKx BKy WRx WRy"
using assms
using no_rook_safe[OF assms] no_rook_safe_opt' 
by simp


lemmas LIA_full_unfold = legal_move_BK_def legal_move_WK_def legal_move_WR_def squeeze_cond_def approach_cond_def keep_room_cond_def rook_safe_cond_def rook_home_cond_def room_def mdcs_def chebyshev_dist_def manhattan_dist_def between_def approach_critical_square_def Lpattern_def checkmated_opt_def stalemate_opt_def WR_can_move_to_def WK_can_move_to_def ready_to_mate_def LIA_unfold


(* ---------------------------- Full postcond --------------------------------- *)
lemma ImmediateMateMove_LIA_fullpostcond:
  assumes "files \<ge> 2" "ranks \<ge> 2"  and
   "t1 = ImmediateMateMove" and wm: "strategy_white_move p0 p1 t1" and 
   "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>  
         LIA.checkmated_opt WKx1 WKy1 BKx1 BKy1 WRx1 WRy1"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(3-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using ImmediateMateMove_postcond[of p0 p1] LIA.legal_move_WR[OF * p0(1-2) p1(1-2)] ImmediateMateMove_checkmated[of p0 p1 t1] LIA.checkmated_opt[OF assms(1-2) *(4-6) p1]
    using assms(3-5) p0
    by auto
qed

lemma ReadyToMateMove_LIA_fullpostcond:
  assumes  
  "files \<ge> 4" "ranks \<ge> 4" and
  "t1 = ReadyToMateMove" "strategy_white_move p0 p1 t1" and  "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "(LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<or> 
          LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1) \<and>  
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.ready_to_mate WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 "
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(4-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using ReadyToMateMove_postcond[of p0 p1] LIA.legal_move_WR[OF * p0(1-2) p1(1-2)] LIA.legal_move_WK[OF * p0(1-2) p1(1-2)] LIA.ready_to_mate[OF assms(1-2) *(4-6)] notImmediateMateMove[of p0 p1 t1]  LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] assms(1-2) assms(3-4) p0 p1
    by (simp add: legal_move_white_def)
qed

lemma SqueezeMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
  "t1 = SqueezeMove" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and> 
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.squeeze_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(4-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using LIA.squeeze_cond[OF _ _ * p0 p1] SqueezeMove_postcond[of p0 p1] LIA.legal_move_WR[OF * p0(1-2) p1(1-2)]
      notReadyToMateMove[of p0 p1 t1] notImmediateMateMove[of p0 p1 t1] LIA.no_ready_to_mate_WR_opt[OF assms(1-2) *(1-3) p0] LIA.no_ready_to_mate_WK[OF assms(1-2) *(1-3) p0] LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] assms(1-3)
      assms(3-4)
    by (simp add: split_def)
qed

lemma ApproachDiagMove_diagmove:
  assumes *: "WK p0 = (WKx0, WKy0)" "WK p1 = (WKx1, WKy1)" and
  wm: "strategy_white_move p0 p1 ApproachDiagMove"
  shows "WKx0 \<noteq> WKx1 \<and> WKy0 \<noteq> WKy1"
using ApproachDiagMove_postcond[OF wm] *
by (simp add: same_diag_def same_diag1_def same_diag2_def KRK.legal_move_WK_def king_scope_chebyshev_dist chebyshev_dist.simps) smt

lemma ApproachMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
         "t1 \<in> approach_moves" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_squeeze WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
         (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(4-6) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using LIA.approach_cond[OF _ _ * p1] ApproachDiagMove_postcond[of p0 p1]  ApproachNonDiagMove_postcond[of p0 p1]
      LIA.legal_move_WK[OF * p0(1-2) p1(1-2)]
      notReadyToMateMove[of p0 p1 t1] notImmediateMateMove[of p0 p1 t1] notSqueezeMove[of p0 p1 t1] LIA.no_ready_to_mate_WR_opt[OF assms(1-2) *(1-3) p0] LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] LIA.no_squeeze[OF _ _ *(1-3) p0] LIA.no_ready_to_mate_WK[OF assms(1-2) *(1-3) p0]
      notApproachDiagMove[of p0 p1 t1] ApproachDiagMove_diagmove[OF *(1) *(4)] LIA.no_approach_diag[OF _ _ *(1-3) p0]
      assms(3-4) assms(1-2)
    unfolding approach_moves_def
    by simp (erule disjE, simp_all)
qed

lemma KeepRoomDiagMove_diagmove:
  assumes *: "WK p0 = (WKx0, WKy0)" "WK p1 = (WKx1, WKy1)" and
  wm: "strategy_white_move p0 p1 KeepRoomDiagMove"
  shows "WKx0 \<noteq> WKx1 \<and> WKy0 \<noteq> WKy1"
using KeepRoomDiagMove_postcond[OF wm] * 
by (simp add: same_diag_def same_diag1_def same_diag2_def KRK.legal_move_WK_def king_scope_chebyshev_dist chebyshev_dist.simps) smt

lemma KeepRoomMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 \<in> keep_room_moves" "strategy_white_move p0 p1 t1" and  "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_squeeze WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
         (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(3-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using LIA.keep_room_cond[OF _ _ * p0 p1] KeepRoomDiagMove_postcond[of p0 p1]  KeepRoomNonDiagMove_postcond[of p0 p1]
      LIA.legal_move_WK[OF * p0(1-2) p1(1-2)]
      notReadyToMateMove[of p0 p1 t1] notImmediateMateMove[of p0 p1 t1] notSqueezeMove[of p0 p1 t1] notApproachNonDiagMove[of p0 p1 t1]
      LIA.no_ready_to_mate_WR_opt[OF assms(1-2) *(1-3) p0] LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] LIA.no_squeeze[OF _ _  *(1-3) p0] LIA.no_approach[OF _ _  *(1-3) p0]  LIA.no_ready_to_mate_WK[OF assms(1-2) *(1-3) p0]
      notKeepRoomDiagMove[of p0 p1 t1] KeepRoomDiagMove_diagmove[OF *(1) *(4)] LIA.no_keep_room_diag[OF _ _ *(1-3) p0]
      assms(3-4) assms(1-2)
    unfolding keep_room_moves_def
    by simp (erule disjE, simp_all)
qed

lemma RookHomeMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 = RookHomeMove" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_squeeze WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_keep_room WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.rook_home_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(3-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using LIA.rook_home_cond[OF _ _ * p1] RookHomeMove_postcond[of p0 p1] 
      LIA.legal_move_WR[OF * p0(1-2) p1(1-2)]
      notReadyToMateMove[of p0 p1 t1] notImmediateMateMove[of p0 p1 t1] notSqueezeMove[of p0 p1 t1] notApproachNonDiagMove[of p0 p1 t1] notKeepRoomNonDiagMove[of p0 p1 t1]
      LIA.no_ready_to_mate_WR_opt[OF assms(1-2) *(1-3) p0] LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] LIA.no_squeeze[OF _ _ *(1-3) p0] LIA.no_approach[OF _ _ *(1-3) p0] LIA.no_keep_room[OF _ _ *(1-3) p0] LIA.no_ready_to_mate_WK[OF assms(1-2) *(1-3) p0]
      assms(3-4) assms(1-2)
    by simp
qed

lemma RookSafeMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 = RookSafeMove" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_squeeze WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_keep_room WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_rook_home_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.rook_safe_cond WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1"
proof-
  have 
    p0: "WhiteOnTurn p0" "\<not> WRcaptured p0" "legal_position p0" and
    p1: "BlackOnTurn p1" "\<not> WRcaptured p1" "legal_position p1"
    using strategy_white_move_black_on_turn[of p0 p1 t1] strategy_white_move_notWRCaptured[of p0 p1 t1] strategy_white_move_legal_position[of p0 p1 t1] assms(3-5) strategy_white_move_white_on_turn[of p0 p1 t1]
    by auto
  show ?thesis
    using LIA.rook_safe_cond[OF _ _ *(3-6) p1] RookSafeMove_postcond[of p0 p1] 
      LIA.legal_move_WR[OF * p0(1-2) p1(1-2)]
      notReadyToMateMove[of p0 p1 t1] notImmediateMateMove[of p0 p1 t1] notSqueezeMove[of p0 p1 t1] notApproachNonDiagMove[of p0 p1 t1] notKeepRoomNonDiagMove[of p0 p1 t1] notRookHomeMove[of p0 p1 t1]
      LIA.no_ready_to_mate_WR_opt[OF assms(1-2) *(1-3) p0] LIA.no_immediate_mate_opt[OF _ _ *(1-3) p0] LIA.no_squeeze[OF _ _ *(1-3) p0] LIA.no_approach[OF _ _ *(1-3) p0] LIA.no_keep_room[OF _ _ *(1-3) p0] LIA.no_rook_home_opt[OF _ _ *(1-3) p0] LIA.no_ready_to_mate_WK[OF assms(1-2) *(1-3) p0]
      assms(3-4) assms(1-2)
    by simp
qed


lemma ApproachKeepRoomMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 \<in> approach_moves \<or> t1 \<in> keep_room_moves" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_squeeze WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         ((LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
          (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)) \<or>
          (LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
          LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
           (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
          )
         )"
using ApproachMove_LIA_fullpostcond[OF assms(1-2) _ assms(4-)] KeepRoomMove_LIA_fullpostcond[OF assms(1-2) _ assms(4-)] assms(3)
by auto

lemma BasicMove_LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 \<in> basic_moves" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         (if t1 = SqueezeMove then
             LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
             LIA.squeeze_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
          else
             LIA.no_squeeze_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
             LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
             (if t1 = ApproachDiagMove \<or> t1 = ApproachNonDiagMove then
                 LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
                 (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
              else 
                 LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                 (if t1 = KeepRoomDiagMove \<or> t1 = KeepRoomNonDiagMove then
                     LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                    (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
                  else 
                     False)))"
using SqueezeMove_LIA_fullpostcond[OF assms(1-2) _ assms(4-)] ApproachMove_LIA_fullpostcond[OF assms(1-2) _ assms(4-)] KeepRoomMove_LIA_fullpostcond[OF assms(1-2) _ assms(4-)]
using strategy_white_move_moves[OF assms(4)]
using LIA.no_squeeze_opt'[of WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 ]
using strategy_white_move_legal_position[OF assms(4)]
using LIA.all_on_board[OF *(1-3) assms(5)] assms(1-3)
unfolding legal_position_def 
by (auto simp add: basic_moves_def approach_moves_def keep_room_moves_def)

lemma BasicMove_LIA_fullpostcond_opt:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "t1 \<in> basic_moves" "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.legal_position_white_on_turn WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.legal_position_black_on_turn WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         (if t1 = SqueezeMove then
             LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and>
             WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0 \<and>
             LIA.squeeze_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
          else
             LIA.no_squeeze_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
             LIA.king_scope WKx0 WKy0 WKx1 WKy1 \<and>
             BKx1 = BKx0 \<and> BKy1 = BKy0 \<and> WRx1 = WRx0 \<and> WRy1 = WRy0 \<and>
             (if t1 = ApproachDiagMove \<or> t1 = ApproachNonDiagMove then
                 LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
                 (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
              else 
                 LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                 LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                 (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)))"
using BasicMove_LIA_fullpostcond[OF assms]
unfolding LIA.legal_move_WK_def LIA.legal_move_WR_def
by smt

lemma LIA_fullpostcond:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "
        (if t1 = ImmediateMateMove then
          LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
          LIA.checkmated_opt WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
        else
          LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
          (if t1 = ReadyToMateMove then
             (LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<or> 
              LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1) \<and>  
             LIA.ready_to_mate WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 
          else
             LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and> LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
             (if t1 = SqueezeMove then
                LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                LIA.squeeze_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
              else
                LIA.no_squeeze_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                (if t1 = ApproachDiagMove \<or> t1 = ApproachNonDiagMove then
                   LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                   LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and> 
                   (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
                else 
                   LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                   (if t1 = KeepRoomDiagMove \<or> t1 = KeepRoomNonDiagMove then
                     LIA.legal_move_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                     LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                    (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
                   else
                     LIA.no_keep_room WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                     (if t1 = RookHomeMove then
                        LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                        LIA.rook_home_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                     else
                        LIA.no_rook_home_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                        (if t1 = RookSafeMove then
                           LIA.legal_move_WR WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                           LIA.rook_safe_cond WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                        else
                           False
          )))))))"
using ImmediateMateMove_LIA_fullpostcond[OF _ _ _ assms(3-)]
using ReadyToMateMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using SqueezeMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using ApproachMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using KeepRoomMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using RookHomeMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using RookSafeMove_LIA_fullpostcond[OF assms(1-2) _ assms(3-)]
using strategy_white_move_moves[OF assms(3)]
using LIA.no_squeeze_opt'[of WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 ]
using strategy_white_move_legal_position[OF assms(3)]
using LIA.all_on_board[OF *(1-3) assms(4)] assms(1-2)
unfolding legal_position_def approach_moves_def keep_room_moves_def
by auto

lemma LIA_fullpostcond_opt:
  assumes "files \<ge> 4" "ranks \<ge> 4" and 
          "strategy_white_move p0 p1 t1" and "\<not> WRcaptured p0" and
  *: "WK p0 = (WKx0, WKy0)" "BK p0 = (BKx0, BKy0)" "WR p0 = (WRx0, WRy0)"
     "WK p1 = (WKx1, WKy1)" "BK p1 = (BKx1, BKy1)" "WR p1 = (WRx1, WRy1)"
  shows "LIA.legal_position_white_on_turn WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
         LIA.legal_position_black_on_turn WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
         (if t1 = ImmediateMateMove then (
             LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and>
             WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0) \<and>
             LIA.checkmated_opt WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
          else LIA.no_immediate_mate_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
               (if t1 = ReadyToMateMove then (
                   LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and>
                   WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0 \<or>
                   LIA.king_scope WKx0 WKy0 WKx1 WKy1 \<and>
                   BKx1 = BKx0 \<and> BKy1 = BKy0 \<and> WRx1 = WRx0 \<and> WRy1 = WRy0) \<and>
                   LIA.ready_to_mate WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                else LIA.no_ready_to_mate_WR_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                   LIA.no_ready_to_mate_WK WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                   (if t1 = SqueezeMove then (
                       LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and>
                       WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0) \<and>
                       LIA.squeeze_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                    else LIA.no_squeeze_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                       (if t1 = ApproachDiagMove \<or> t1 = ApproachNonDiagMove then (
                           LIA.king_scope WKx0 WKy0 WKx1 WKy1 \<and>
                           BKx1 = BKx0 \<and> BKy1 = BKy0 \<and> WRx1 = WRx0 \<and> WRy1 = WRy0) \<and>
                           LIA.approach_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                           (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_approach_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
                        else LIA.no_approach WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                           (if t1 = KeepRoomDiagMove \<or> t1 = KeepRoomNonDiagMove then (
                               LIA.king_scope WKx0 WKy0 WKx1 WKy1 \<and>
                               BKx1 = BKx0 \<and> BKy1 = BKy0 \<and> WRx1 = WRx0 \<and> WRy1 = WRy0) \<and>
                               LIA.keep_room_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1 \<and>
                               (WKx1 = WKx0 \<or> WKy1 = WKy0 \<longrightarrow> LIA.no_keep_room_diag WKx0 WKy0 BKx0 BKy0 WRx0 WRy0)
                            else LIA.no_keep_room WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                               (if t1 = RookHomeMove then (
                                   LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and> 
                                   WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0) \<and>
                                   LIA.rook_home_cond WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                                else LIA.no_rook_home_opt WKx0 WKy0 BKx0 BKy0 WRx0 WRy0 \<and>
                                   (if t1 = RookSafeMove then (
                                       LIA.WR_attacks_square WRx0 WRy0 WKx0 WKy0 WRx1 WRy1 \<and>
                                       WKx1 = WKx0 \<and> WKy1 = WKy0 \<and> BKx1 = BKx0 \<and> BKy1 = BKy0) \<and>
                                       LIA.rook_safe_cond WRx0 WRy0 WKx1 WKy1 BKx1 BKy1 WRx1 WRy1
                                    else False)))))))"
using LIA_fullpostcond[OF assms]
unfolding LIA.legal_move_WR_def LIA.legal_move_WK_def
by smt

end
