(* ************************************************************************** *)
section{* Symmetries of the KRK board *}
(* ************************************************************************** *)
theory Symmetry
imports KRK 
       "HOL-Library.Product_Lexorder"
begin

(* -------------------------------------------------------------------------- *)
subsection{* Reflections of squares *}
(* -------------------------------------------------------------------------- *)

(* Three types of square reflections: horizontal, vertical, and diagonal *)
fun reflectx :: "square \<Rightarrow> square" where
  "reflectx (x, y) = (files - 1 - x, y)"
fun reflecty :: "square \<Rightarrow> square" where
  "reflecty (x, y) = (x, ranks - 1 - y)"
fun reflectdiag :: "square \<Rightarrow> square" where
  "reflectdiag (x, y) = (y, x)"

(* Reflections are involutions *)
lemma reflectx_involution [simp]: "reflectx (reflectx sq) = sq"
  by (cases sq) auto
lemma reflecty_involution [simp]: "reflecty (reflecty sq) = sq"
  by (cases sq) auto
lemma reflectdiag_involution [simp]: "reflectdiag (reflectdiag sq) = sq"
  by (cases sq) auto

(* Reflections are injective *)
lemma reflectx_inj: "reflectx a = reflectx b \<Longrightarrow> a = b"
by (cases a, cases b) simp
lemma reflecty_inj: "reflecty a = reflecty b \<Longrightarrow> a = b"
by (cases a, cases b) simp
lemma reflectdiag_inj: "reflectdiag a = reflectdiag b \<Longrightarrow> a = b"
by (cases a, cases b) simp


(* -------------------------------------------------------------------------- *)
subsection{* Transformations of the KRK board *}
(* -------------------------------------------------------------------------- *)

(* Applies a given transformation of squares to the given KRK board,
   by reflecting each of its pieces *)
definition reflect_p :: "(square \<Rightarrow> square) \<Rightarrow> KRKPosition \<Rightarrow> KRKPosition" where
 "reflect_p f p = 
    p \<lparr>BK := f (BK p), 
       WK := f (WK p), 
       WRopt := (case WRopt p of None \<Rightarrow> None | Some sq \<Rightarrow> Some (f sq)) \<rparr>"

(* Getters for the transformed KRK board *)

lemma reflect_p_BK [simp]: "BK (reflect_p f p) = f (BK p)"
by (simp add: reflect_p_def)

lemma reflect_p_WK [simp]: "WK (reflect_p f p) = f (WK p)"
by (simp add: reflect_p_def)

lemma reflect_p_WhiteOnTurn [simp]: "WhiteOnTurn (reflect_p f p) = WhiteOnTurn p"
by (simp add: reflect_p_def)

lemma reflect_p_WRcaptured [simp]: "WRcaptured (reflect_p f p) \<longleftrightarrow> WRcaptured p"
unfolding reflect_p_def
by (cases "WRopt p") (auto simp add: WRcaptured_def)

lemma reflect_p_WR [simp]: "\<not> WRcaptured p \<Longrightarrow> WR (reflect_p f p) = f (WR p)"
by (auto simp add: reflect_p_def WRcaptured_def WR_def)

(* Transformation of KRK board is involution if transformation of squares is *)
lemma reflect_p_involution [simp]:
  assumes "\<And> x. f (f x) = x"
  shows "reflect_p f (reflect_p f p) = p"
by (rule KRKPosition.equality) (simp_all add: assms,  (cases "WRopt p", simp_all add: reflect_p_def assms))

lemma reflect_p_solve: "\<lbrakk>\<And> x. f (f x) = x; reflect_p f p = p'\<rbrakk> \<Longrightarrow> p = reflect_p f p'"
using reflect_p_involution[of f p]
by simp

(* Transformation of KRK board is injective if transformation of squares is *)
lemma reflect_p_inj:
  assumes inj: "\<And> a b. f a = f b \<Longrightarrow> a = b"
  shows "reflect_p f a = reflect_p f b \<Longrightarrow> a = b"
proof-
  assume "reflect_p f a = reflect_p f b"
  hence "f (WK a) = f (WK b)" "f (BK a) = f (BK b)" and *: "(case WRopt a of None \<Rightarrow> None | Some sq \<Rightarrow> Some (f sq)) = (case WRopt b of None \<Rightarrow> None | Some sq \<Rightarrow> Some (f sq))"
    unfolding reflect_p_def
    by (metis (lifting, no_types) ext_inject surjective update_convs)+
  hence "WK a = WK b" "BK a = BK b"
    by (force simp add: inj)+
  moreover
  have "WRopt a = WRopt b"
    using *
    by (case_tac[!] "WRopt a", case_tac[!] "WRopt b")  (auto simp add: inj)
  moreover
  have "WhiteOnTurn a = WhiteOnTurn b"
    using `reflect_p f a = reflect_p f b`
    by (metis (lifting, no_types) ext_inject reflect_p_def surjective update_convs)
  moreover
  have "more a = more b"
    using `reflect_p f a = reflect_p f b`
    by (metis (lifting, mono_tags) ext_inject reflect_p_def surjective update_convs)
  ultimately
  show "a = b"
    by (rule KRKPosition.equality)
qed

(* -------------------------------------------------------------------------- *)
subsection{* Reflections of KRK board *}
(* -------------------------------------------------------------------------- *)

(* Horizontal, vectical, and diagonal reflection of the KRK board *)
definition reflectx_p :: "KRKPosition \<Rightarrow> KRKPosition" where
 "reflectx_p = reflect_p reflectx"
definition reflecty_p :: "KRKPosition \<Rightarrow> KRKPosition" where
  "reflecty_p = reflect_p reflecty"
definition reflectdiag_p :: "KRKPosition \<Rightarrow> KRKPosition" where
  "reflectdiag_p = reflect_p reflectdiag"

(* Getters for the transformed KRK board *)

lemma reflectx_fields [simp]: "BK (reflectx_p p) = reflectx (BK p)" "WK (reflectx_p p) = reflectx (WK p)"  "WhiteOnTurn (reflectx_p p) = WhiteOnTurn p" "WRcaptured (reflectx_p p) = WRcaptured p" "\<not> WRcaptured p \<Longrightarrow> WR (reflectx_p p) = reflectx (WR p)"
by (auto simp add: reflectx_p_def)

lemma reflecty_fields  [simp]: "BK (reflecty_p p) = reflecty (BK p)" "WK (reflecty_p p) = reflecty (WK p)"  "WhiteOnTurn (reflecty_p p) = WhiteOnTurn p" "WRcaptured (reflecty_p p) = WRcaptured p" "\<not> WRcaptured p \<Longrightarrow> WR (reflecty_p p) = reflecty (WR p)"
by (auto simp add: reflecty_p_def)

lemma reflectdiag_fields [simp]: "BK (reflectdiag_p p) = reflectdiag (BK p)" "WK (reflectdiag_p p) = reflectdiag (WK p)"  "WhiteOnTurn (reflectdiag_p p) = WhiteOnTurn p" "WRcaptured (reflectdiag_p p) = WRcaptured p" "\<not> WRcaptured p \<Longrightarrow> WR (reflectdiag_p p) = reflectdiag (WR p)"
by (auto simp add: reflectdiag_p_def)

lemma WRcaptured_WRopt_reflectx_p: "WRcaptured p \<Longrightarrow> WRopt (reflectx_p p) = None"
unfolding reflectx_p_def reflect_p_def
by (simp add: WRcaptured_def)

lemma notWRcaptured_WRopt_reflectx_p: "\<not> WRcaptured p \<Longrightarrow> WRopt (reflectx_p p) = Some (reflectx (the (WRopt p)))"
unfolding reflectx_p_def reflect_p_def
by (cases "WRopt p") (auto simp add: WRcaptured_def)

lemma WRcaptured_WRopt_reflecty_p: "WRcaptured p \<Longrightarrow> WRopt (reflecty_p p) = None"
unfolding reflecty_p_def reflect_p_def
by (simp add: WRcaptured_def)

lemma notWRcaptured_WRopt_reflecty_p: "\<not> WRcaptured p \<Longrightarrow> WRopt (reflecty_p p) = Some (reflecty (the (WRopt p)))"
unfolding reflecty_p_def reflect_p_def
by (cases "WRopt p") (auto simp add: WRcaptured_def)

lemma WRcaptured_WRopt_reflectdiag_p: "WRcaptured p \<Longrightarrow> WRopt (reflectdiag_p p) = None"
unfolding reflectdiag_p_def reflect_p_def
by (simp add: WRcaptured_def)

lemma notWRcaptured_WRopt_reflectdiag_p: "\<not> WRcaptured p \<Longrightarrow> WRopt (reflectdiag_p p) = Some (reflectdiag (the (WRopt p)))"
unfolding reflectdiag_p_def reflect_p_def
by (cases "WRopt p") (auto simp add: WRcaptured_def)

(* Reflections of KRK board are involutions *)

lemma reflect_involution [simp]:
  shows "reflectx_p (reflectx_p p) = p" "reflecty_p (reflecty_p p) = p" "reflectdiag_p (reflectdiag_p p) = p"
using reflect_p_involution[of reflectx p] reflect_p_involution[of reflecty p] reflect_p_involution[of reflectdiag p]
unfolding reflectx_p_def reflecty_p_def reflectdiag_p_def
by simp_all

lemma reflect_solve: 
  "reflectx_p p = p' \<Longrightarrow> p = reflectx_p p'"
  "reflecty_p p = p' \<Longrightarrow> p = reflecty_p p'"
  "reflectdiag_p p = p' \<Longrightarrow> p = reflectdiag_p p'"
using reflect_p_solve[OF reflectx_involution, of p p']
using reflect_p_solve[OF reflecty_involution, of p p']
using reflect_p_solve[OF reflectdiag_involution, of p p']
unfolding reflectx_p_def reflecty_p_def reflectdiag_p_def
by simp_all

(* Reflections of KRK board are injective *)

lemma reflectx_p_inj: "reflectx_p a = reflectx_p b \<Longrightarrow> a = b"
using reflect_p_inj[of reflectx, OF reflectx_inj] 
unfolding reflectx_p_def
by blast

lemma reflecty_p_inj: "reflecty_p a = reflecty_p b \<Longrightarrow> a = b"
using reflect_p_inj[of reflecty, OF reflecty_inj] 
unfolding reflecty_p_def
by blast

lemma reflectdiag_p_inj: "reflectdiag_p a = reflectdiag_p b \<Longrightarrow> a = b"
using reflect_p_inj[of reflectdiag, OF reflectdiag_inj] 
unfolding reflectdiag_p_def
by blast

(* Compositions of reflections *)

lemma reflectx_p_reflecty_p: "reflectx_p (reflecty_p p) = reflecty_p (reflectx_p p)"
apply (rule KRKPosition.equality)
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all)+
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp add: WRcaptured_WRopt_reflectx_p WRcaptured_WRopt_reflecty_p)
apply (simp add: notWRcaptured_WRopt_reflecty_p notWRcaptured_WRopt_reflectx_p WR_def)
done

lemma reflectx_p_reflectdiag_p: "files = ranks \<Longrightarrow> reflectx_p (reflectdiag_p p) = reflectdiag_p (reflecty_p p)"
apply (rule KRKPosition.equality)
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all)+
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp add: WRcaptured_WRopt_reflectx_p WRcaptured_WRopt_reflecty_p WRcaptured_WRopt_reflectdiag_p)
apply (simp add: notWRcaptured_WRopt_reflecty_p notWRcaptured_WRopt_reflectx_p notWRcaptured_WRopt_reflectdiag_p WR_def)
done

lemma reflecty_p_reflectdiag_p: "files = ranks \<Longrightarrow> reflecty_p (reflectdiag_p p) = reflectdiag_p (reflectx_p p)"
apply (rule KRKPosition.equality)
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all)+
apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp add: WRcaptured_WRopt_reflectx_p WRcaptured_WRopt_reflecty_p WRcaptured_WRopt_reflectdiag_p)
apply (simp add: notWRcaptured_WRopt_reflecty_p notWRcaptured_WRopt_reflectx_p notWRcaptured_WRopt_reflectdiag_p WR_def)
done

(* -------------------------------------------------------------------------- *)
subsection{* KRK boards in canonical positions *}                                                   
(* -------------------------------------------------------------------------- *)

(*
Chessboard can be divided into 8 equal triangles. KRK board is canonical if BK is
in the triangle shown below. 

+.......
++......
+++.....
++++....
........
........
........
........

If board has odd size other two pieces must also be considered.
*)

(* Definition of canonical positions *)                                                              

(* BK must be on the left half of the board. 
   If it is on the middle column, then WK must be on the left half of the board. 
   If it is on the middle column and WR is not captured, then it must be on the left half of the board. *)
definition is_canon_x where
 "is_canon_x p \<longleftrightarrow> 
      (let (Bx, By) = BK p; (Wx, Wy) = WK p; (Rx, Ry) = WR p 
        in if WRcaptured p then (2*Bx+1, 2*Wx+1) \<le> (files, files) else (2*Bx+1, 2*Wx+1, 2*Rx+1) \<le> (files, files, files))"

(* BK must be on the upper half of the board. 
   If it is on the middle row, then WK must be on the upper half of the board. 
   If it is on the middle row and WR is not captured, then it must be on the upper half of the board. *)
definition is_canon_y where
 "is_canon_y p \<longleftrightarrow> 
      (let (Bx, By) = BK p; (Wx, Wy) = WK p; (Rx, Ry) = WR p 
        in if WRcaptured p then (2*By+1, 2*Wy+1) \<le> (ranks, ranks) else (2*By+1, 2*Wy+1, 2*Ry+1) \<le> (ranks, ranks, ranks))"

(* BK must be below the main diagonal. 
   If it is on the main diagonal, then WK must be below the main diagonal. 
   If it is on the main diagonal and WR is not captured, then it must be below the main diagonal. *)
definition is_canon_diag where
 "is_canon_diag p \<longleftrightarrow> 
     (let (Bx, By) = BK p; (Wx, Wy) = WK p; (Rx, Ry) = WR p 
       in if WRcaptured p then (Bx, Wx) \<le> (By, Wy) else (Bx, Wx, Rx) \<le> (By, Wy, Ry))"

(* Weak canonical form - only horizontal and vertical 
   BK is on the upper left quarter of the board. *)
definition is_canon_xy where
  "is_canon_xy p \<longleftrightarrow> is_canon_x p \<and> is_canon_y p"

(* Fully canonical board - horizontal, vertical, and diagonal - 
   requires square board i.e. files = ranks.
   BK is on the canonical triangle. *)
definition is_canon where
  "is_canon p \<longleftrightarrow> is_canon_x p \<and> is_canon_y p \<and> is_canon_diag p"

(* -------------------------------------------------------------------------- *)
subsection{* Canonization of KRK boards *}
(* -------------------------------------------------------------------------- *)

(* Boards can be transformed to canonical positions by 
   horizontal, vertical and diagonal reflections *)

(* Reflections are applied only when necessary *)

definition maybe_reflectx where
  "maybe_reflectx p = (if \<not> is_canon_x p then reflectx_p p else p)"

definition maybe_reflecty where
  "maybe_reflecty p = (if \<not> is_canon_y p then reflecty_p p else p)"

definition maybe_reflectdiag where
 "maybe_reflectdiag p = (if \<not> is_canon_diag p then reflectdiag_p p else p)"

(* Reflections do canonize positions *)

lemma is_canon_x_reflecty_p: "is_canon_x (reflecty_p p) \<longleftrightarrow> is_canon_x p"
  unfolding is_canon_x_def
  by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_y_reflectx_p: "is_canon_y (reflectx_p p) \<longleftrightarrow> is_canon_y p"
  unfolding is_canon_y_def
  by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_x_reflectdiag_p:
  assumes "files = ranks" 
  shows "is_canon_x (reflectdiag_p p) \<longleftrightarrow> is_canon_y p"
using assms
unfolding is_canon_x_def is_canon_y_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_y_reflectdiag_p:
  assumes "files = ranks" 
  shows "is_canon_y (reflectdiag_p p) \<longleftrightarrow> is_canon_x p"
using assms
unfolding is_canon_x_def is_canon_y_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_x_maybe_reflectx: "is_canon_x (maybe_reflectx p)"
unfolding maybe_reflectx_def is_canon_x_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_y_maybereflecty: "is_canon_y (maybe_reflecty p)"
unfolding maybe_reflecty_def is_canon_y_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_x_maybereflecty: "is_canon_x p \<Longrightarrow> is_canon_x (maybe_reflecty p)"
unfolding is_canon_x_def maybe_reflecty_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_xy_maybe_reflectxy: "let p' = maybe_reflecty (maybe_reflectx p) in is_canon_x p' \<and> is_canon_y p'"
using is_canon_y_maybereflecty[of "maybe_reflectx p"]  is_canon_x_maybereflecty[OF is_canon_x_maybe_reflectx[of p]]
by simp

lemma is_canon_diag_maybe_reflectdiag: "is_canon_diag (maybe_reflectdiag p)"
unfolding maybe_reflectdiag_def is_canon_diag_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p") auto

lemma is_canon_xy_maybe_reflectdiag:
  assumes "is_canon_x p" "is_canon_y p" "files = ranks"
  shows "is_canon_x (maybe_reflectdiag p)" "is_canon_y (maybe_reflectdiag p)"
using assms
unfolding is_canon_x_def is_canon_y_def maybe_reflectdiag_def
by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", auto)

(* Canonization procedures are compositions of individual reflections *)

(* Weak canonization - only horizontal and vertical *)
definition canon_xy where
 "canon_xy p = (
      let p = maybe_reflectx p;
          p = maybe_reflecty p
       in
      p
  )"

(* Full canonization - horizontal, vertical and diagonal - works only for square boards *)
definition canon where
 "canon p = (
      let p = maybe_reflectx p;
          p = maybe_reflecty p;
          p = maybe_reflectdiag p
       in
      p
  )"

(* Possible compositions of reflections that lead to canonization *)

lemma canon_xy_cases:
  "canon_xy p = p \<or> 
   canon_xy p = reflectx_p p \<or> 
   canon_xy p = reflecty_p p \<or> 
   canon_xy p = reflecty_p (reflectx_p p)"
unfolding canon_xy_def maybe_reflectx_def maybe_reflecty_def
by (simp add: Let_def)

lemma canon_cases:
  "canon p = p \<or> 
   canon p = reflectx_p p \<or> 
   canon p = reflecty_p p \<or> 
   canon p = reflecty_p (reflectx_p p) \<or> 
   canon p = reflectdiag_p p \<or> 
   canon p = reflectdiag_p (reflectx_p p) \<or> 
   canon p = reflectdiag_p (reflecty_p p) \<or> 
   canon p = reflectdiag_p (reflecty_p (reflectx_p p))"
unfolding canon_def maybe_reflectx_def maybe_reflecty_def maybe_reflectdiag_def
by (simp add: Let_def)

(* Canonization procedures lead to canonical boards *)

lemma is_canon_xy_canon_xy:
  shows "is_canon_xy (canon_xy p)"
unfolding is_canon_xy_def canon_xy_def
by (metis is_canon_xy_maybe_reflectxy)

lemma is_canon_canon:
  assumes "files = ranks"
  shows "is_canon (canon p)"
using assms
unfolding is_canon_def canon_def
by (metis is_canon_diag_maybe_reflectdiag is_canon_xy_maybe_reflectdiag is_canon_xy_maybe_reflectxy)

(* Canonization of canonical boards does not make any effect *)

lemma is_canon_xy_canon_xy_id: "is_canon_xy p \<Longrightarrow> canon_xy p = p"
unfolding is_canon_xy_def canon_xy_def maybe_reflectx_def maybe_reflecty_def
by simp

lemma is_canon_canon_id: "is_canon p \<Longrightarrow> canon p = p"
unfolding is_canon_def canon_def maybe_reflectx_def maybe_reflecty_def maybe_reflectdiag_def
by simp

(* 
  Without loss of generality principle using canonical boards.
  Each property invariant under appropriate reflections can be shown only for boards in 
  canonical positions.
*)

(* .......................................................................... *)
theorem symmetry_xy:
  fixes P :: "KRKPosition \<Rightarrow> bool"
  assumes invariant: "\<forall> p. P (reflectx_p p) \<longrightarrow> P p"  "\<forall> p. P (reflecty_p p) \<longrightarrow> P p"
  assumes "\<forall> p. is_canon_xy p \<longrightarrow> P p" 
  shows "\<forall> p. P p"
(* .......................................................................... *)
proof
  fix p
  have "P (canon_xy p)"
    using assms(3) is_canon_xy_canon_xy[of p]
    by simp

  thus "P p"
    using invariant
    using canon_xy_cases[of p]
    by smt
qed

(* .......................................................................... *)
theorem symmetry:
  fixes P :: "KRKPosition \<Rightarrow> bool"
  assumes "files = ranks"
  assumes invariant: "\<forall> p. P (reflectx_p p) \<longrightarrow> P p"  "\<forall> p. P (reflecty_p p) \<longrightarrow> P p"   "\<forall> p. P (reflectdiag_p p) \<longrightarrow> P p"
  assumes "\<forall> p. is_canon p \<longrightarrow> P p" 
  shows "\<forall> p. P p"
(* .......................................................................... *)
proof
  fix p
  have "P (canon p)"
    using assms(5) is_canon_canon[of p, OF `files = ranks`]
    by simp

  thus "P p"
    using invariant
    using canon_cases[of p]
    by smt
qed

(* Some other properties of reflections *)

lemma reflectx_maybe_reflectx:
  shows "maybe_reflectx p = maybe_reflectx (reflectx_p p)"
proof-
  have *: "\<And> BKx BKy WKx WKy. \<lbrakk>(BKx, BKy) = BK p; (WKx, WKy) = WK p; 2*BKx + 1 = files \<and> 2*WKx + 1 = files \<and> WRcaptured p\<rbrakk> \<Longrightarrow>  p = reflectx_p p"
    apply (cases "WK p", cases "BK p",  auto)
    apply (rule KRKPosition.equality, simp_all)
    apply (subst WRcaptured_WRopt_reflectx_p, simp_all add: WRcaptured_def)
    done
  have **: "\<And> BKx BKy WKx WKy WRx WRy. \<lbrakk>(BKx, BKy) = BK p; (WKx, WKy) = WK p; (WRx, WRy) = WR p; 2*BKx + 1 = files \<and> 2*WKx + 1 = files \<and> 2*WRx + 1 = files \<and> \<not> WRcaptured p\<rbrakk> \<Longrightarrow>  p = reflectx_p p"
    apply (cases "WK p", cases "BK p",  cases "WR p", auto)
    apply (rule KRKPosition.equality, auto)
    apply (subst notWRcaptured_WRopt_reflectx_p, simp, cases "WRopt p", simp_all add: WRcaptured_def WR_def)
    done
  have ++: "is_canon_x p \<or> is_canon_x (reflectx_p p)"
    unfolding is_canon_x_def
    by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", auto)
  have +++: "is_canon_x p \<and> is_canon_x (reflectx_p p) \<Longrightarrow> p = reflectx_p p"
    apply (cases "BK p", cases "WK p", cases "WR p")
    apply (cases "WRcaptured p")
    apply (rule *, (force simp add: is_canon_x_def)+)
    apply (rule **, (force simp add: is_canon_x_def)+)
    done
  thus ?thesis
    unfolding maybe_reflectx_def
    using ++ +++
    by auto
qed

lemma reflecty_maybe_reflecty:
  shows "maybe_reflecty p = maybe_reflecty(reflecty_p p)"
proof-
  have *: "\<And> BKx BKy WKx WKy. \<lbrakk>(BKx, BKy) = BK p; (WKx, WKy) = WK p; 2*BKy + 1 = ranks \<and> 2*WKy + 1 = ranks \<and> WRcaptured p\<rbrakk> \<Longrightarrow>  p = reflecty_p p"
    apply (cases "WK p", cases "BK p",  auto)
    apply (rule KRKPosition.equality, simp_all)
    apply (subst WRcaptured_WRopt_reflecty_p, simp_all add: WRcaptured_def)
    done
  have **: "\<And> BKx BKy WKx WKy WRx WRy. \<lbrakk>(BKx, BKy) = BK p; (WKx, WKy) = WK p; (WRx, WRy) = WR p; 2*BKy + 1 = ranks \<and> 2*WKy + 1 = ranks \<and> 2*WRy + 1 = ranks \<and> \<not> WRcaptured p\<rbrakk> \<Longrightarrow>  p = reflecty_p p"
    apply (cases "WK p", cases "BK p",  cases "WR p", auto)
    apply (rule KRKPosition.equality, auto)
    apply (subst notWRcaptured_WRopt_reflecty_p, simp, cases "WRopt p", simp_all add: WRcaptured_def WR_def)
    done
  have ++: "is_canon_y p \<or> is_canon_y (reflecty_p p)"
    unfolding is_canon_y_def
    by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", auto)
  have +++: "is_canon_y p \<and> is_canon_y (reflecty_p p) \<Longrightarrow> p = reflecty_p p"
    apply (cases "BK p", cases "WK p", cases "WR p")
    apply (cases "WRcaptured p")
    apply (rule *, (force simp add: is_canon_y_def)+)
    apply (rule **, (force simp add: is_canon_y_def)+)
    done
  thus ?thesis
    unfolding maybe_reflecty_def
    using ++ +++
    by auto
qed

lemma reflecty_maybe_reflectx: 
  "maybe_reflectx (reflecty_p p) = reflecty_p (maybe_reflectx p)"
unfolding maybe_reflectx_def
by (auto simp add: reflectx_p_reflecty_p is_canon_x_reflecty_p[of p])

lemma reflectdiag_maybe_reflectx: 
  assumes "files = ranks"
  shows "maybe_reflectx (reflectdiag_p p) = reflectdiag_p (maybe_reflecty p)"
using assms
unfolding maybe_reflectx_def maybe_reflecty_def
by (auto simp add: reflectx_p_reflectdiag_p is_canon_x_reflectdiag_p)

lemma reflectdiag_maybe_reflecty: 
  assumes "files = ranks"
  shows "maybe_reflecty (reflectdiag_p p) = reflectdiag_p (maybe_reflectx p)"
using assms
unfolding maybe_reflectx_def maybe_reflecty_def
by (auto simp add: reflecty_p_reflectdiag_p is_canon_y_reflectdiag_p)

lemma reflectdiag_maybe_reflectdiag: 
  "maybe_reflectdiag (reflectdiag_p p) = maybe_reflectdiag p"
proof-
  have ++: "is_canon_diag (reflectdiag_p p) \<or> is_canon_diag p"
    unfolding is_canon_diag_def
    by (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", auto)
  
  have +++: "is_canon_diag p \<and> is_canon_diag (reflectdiag_p p) \<Longrightarrow> p = reflectdiag_p p"
    apply (rule KRKPosition.equality)
    apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all add: is_canon_diag_def, force+)+
    apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all add: is_canon_diag_def WRcaptured_WRopt_reflectdiag_p WRcaptured_def)
    apply (cases "BK p", cases "WK p", cases "WR p", cases "WRcaptured p", simp_all add: is_canon_diag_def notWRcaptured_WRopt_reflectdiag_p WRcaptured_def WR_def, force)
    done
  thus ?thesis
    unfolding maybe_reflectdiag_def
    using ++ +++
    by auto
qed

lemma maybe_reflectx_maybe_reflecty: "maybe_reflectx (maybe_reflecty p) = maybe_reflecty (maybe_reflectx p)"
unfolding maybe_reflectx_def maybe_reflecty_def
by (auto simp add: is_canon_y_reflectx_p is_canon_x_reflecty_p reflectx_p_reflecty_p)

(* Canonization of reflected boards is the same as canonization of original boards. *)

lemma reflectx_canon_xy: "canon_xy (reflectx_p p) = canon_xy p"
using reflectx_maybe_reflectx[of p]
unfolding canon_xy_def
by simp

lemma reflecty_canon_xy: "canon_xy (reflecty_p p) = canon_xy p"
using reflecty_maybe_reflecty reflecty_maybe_reflectx
unfolding canon_xy_def
by simp


lemma reflectx_canon: "canon (reflectx_p p) = canon p"
using reflectx_maybe_reflectx[of p]
unfolding canon_def
by simp

lemma reflecty_canon: "canon (reflecty_p p) = canon p"
using reflecty_maybe_reflecty reflecty_maybe_reflectx
unfolding canon_def
by simp

lemma reflectdiag_canon: 
  assumes "files = ranks"
  shows "canon (reflectdiag_p p) = canon p"
using assms
using reflectdiag_maybe_reflectx reflectdiag_maybe_reflecty reflectdiag_maybe_reflectdiag maybe_reflectx_maybe_reflecty
unfolding canon_def
by auto

end
