/* ***************************** URSA-krk *********************************** */

/* ************************************************************************** */
/* ************************************************************************** */
/* ************* Specification of Chess Rules for KRK ending **************** */
/* ************************************************************************** */
/* ************************************************************************** */

/* ** Written by Predrag Janicic, Filip Maric, and Marko Malikovic (2016) *** */

/* ************************************************************************** */
/* For convenience, a KRK 8x8 position is represented as a sequence of 20 bits:
   -------------------------------------------------------------
   |19|18|17|16|15|14|13|12|11|10| 9| 8| 7| 6| 5| 4| 3| 2| 1| 0|
   -------------------------------------------------------------
   |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |  |
   -------------------------------------------------------------
   |bW|bR|  nWRy  |  nWRx  |  nBKy  |  nBKx  |  nWKy  |  nWKx  |    
   -------------------------------------------------------------  */
/* ************************************************************************** */


/* ******************************************************************** */
/* Miscellanous                                                         */ 
/* ******************************************************************** */ 

/* ------------------------------------------------------------------------- */

/* log_2 - for arguments <=32 */
procedure Log2(n,nLog) {
  nLog=1;
  nLog = ite(n<=4,2,nLog);
  nLog = ite(n>4  && n<=8,3,nLog);
  nLog = ite(n>8  && n<=16,4,nLog);
  nLog = ite(n>16 && n<=32,5,nLog);
}

procedure Max(nx, ny, nMax) {
  nMax = ite(nx>=ny, nx, ny);
}

procedure AbsDiff(nA,nB,nAbsDiff) {
nAbsDiff = ite(nA>=nB,nA-nB,nB-nA);
}

procedure NumberBetween(n1, n2, n3, bBetween) {
   bBetween = ((n1<n2 && n2<n3) || (n3<n2 && n2<n1));
}

/* -------------------------------------------------------------------------- */
/* Lexicographic comparison of tuples */

/* returns (na1,nb1) <lex (na2,nb2) */
procedure lexLt2(na1, nb1, na2, nb2, blex2) {
  blex2 = na1 < na2 || (na1 == na2 && nb1 < nb2);
}

/* returns (na1,nb1) <=lex (na2,nb2) */
procedure lexLe2(na1, nb1, na2, nb2, blex2) {
  blex2 = na1 < na2 || (na1 == na2 && nb1 <= nb2);
}

/* returns (na1,nb1,nc1) <=lex (na2,nb2,nc2) */
procedure lexLe3(na1, nb1, nc1, na2, nb2, nc2, blex3) {
  blex3 = na1 < na2 || (na1 == na2 && nb1 < nb2) || (na1 == na2 && nb1 == nb2 && nc1 <= nc2);
}


/* ------------------------------------------------------------------------- */

/* Compressing and uncompressing procedures */
procedure Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn, nPos) {
  call Log2(nFiles,nFilesLog);
  call Log2(nRanks,nRanksLog);
  nFilesMask = (1<<nFilesLog)-1;
  nRanksMask = (1<<nRanksLog)-1;

  nPos = ite(bWhiteOnTurn, 1, 0);
  nPos = (nPos << 1) | ite(bWRCaptured, 1, 0);
  nPos = (nPos << nRanksLog) | (nWRy & nRanksMask);
  nPos = (nPos << nFilesLog) | (nWRx & nFilesMask);
  nPos = (nPos << nRanksLog) | (nBKy & nRanksMask);
  nPos = (nPos << nFilesLog) | (nBKx & nFilesMask);
  nPos = (nPos << nRanksLog) | (nWKy & nRanksMask);
  nPos = (nPos << nFilesLog) | (nWKx & nFilesMask);
}


procedure Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn) {
  call Log2(nFiles,nFilesLog);
  call Log2(nRanks,nRanksLog);
  nFilesMask = (1<<nFilesLog)-1;
  nRanksMask = (1<<nRanksLog)-1;

  nWKx = nPos & nFilesMask;
  nWKy = (nPos >> (1*nRanksLog+0*nFilesLog)) & nRanksMask;
  nBKx = (nPos >> (1*nRanksLog+1*nFilesLog)) & nFilesMask;
  nBKy = (nPos >> (2*nRanksLog+1*nFilesLog)) & nRanksMask;
  nWRx = (nPos >> (2*nRanksLog+2*nFilesLog)) & nFilesMask;
  nWRy = (nPos >> (3*nRanksLog+2*nFilesLog)) & nRanksMask;
  bWRCaptured = num2bool((nPos >> (3*nRanksLog+3*nFilesLog)) & 1);  
  bWhiteOnTurn  = num2bool(nPos >> ((3*nRanksLog+3*nFilesLog)+1));
}

/* -------------------------------------------------------------------------- */

procedure IsWRCaptured(nFiles, nRanks, nPos, bWRCaptured) {
  call Log2(nFiles,nFilesLog);
  call Log2(nRanks,nRanksLog);
  bWRCaptured = num2bool(nPos >> ((3*nRanksLog+3*nFilesLog)) & 1);
}

/* -------------------------------------------------------------------------- */

procedure IsWhiteOnTurn(nFiles, nRanks, nPos, bWhiteOnTurn) {
  call Log2(nFiles,nFilesLog);
  call Log2(nRanks,nRanksLog);
  bWhiteOnTurn = num2bool(nPos >> ((3*nRanksLog+3*nFilesLog)+1));
}

/* -------------------------------------------------------------------------- */

procedure ManhattanDistance(nx1, ny1, nx2, ny2, nMD) {
  nMD = ite(nx2>=nx1, 
            ite(ny2>=ny1, (nx2-nx1)+(ny2-ny1), (nx2-nx1)+(ny1-ny2)),
            ite(ny2>=ny1, (nx1-nx2)+(ny2-ny1), (nx1-nx2)+(ny1-ny2)));
}

/* -------------------------------------------------------------------------- */

procedure ChebyshevDistance(nx1, ny1, nx2, ny2, nCD) {
call Max(nx2-nx1, ny2-ny1, nMax1);
call Max(nx2-nx1, ny1-ny2, nMax2);
call Max(nx1-nx2, ny2-ny1, nMax3);
call Max(nx1-nx2, ny1-ny2, nMax4);
nCD = ite(nx2>=nx1,
          ite(ny2>=ny1, nMax1, nMax2),
          ite(ny2>=ny1, nMax3, nMax4));
}

/* -------------------------------------------------------------------------- */

/* Square (x2,y2) is between squares (x1,y1) and (x3,y3): */
procedure SquareBetween(nx1, ny1, nx2, ny2, nx3, ny3, bSquareBetween) {
  call NumberBetween(ny1, ny2, ny3, bBetween_y); 
  call NumberBetween(nx1, nx2, nx3, bBetween_x);
  bSquareBetween = ((nx1==nx2 && nx2==nx3 && bBetween_y) || (ny1==ny2 && ny2==ny3 && bBetween_x));
}

/* -------------------------------------------------------------------------- */

procedure SquareDivides(nx1, ny1, nx2, ny2, nx3, ny3, bSquareBetween) {
  call NumberBetween(ny1, ny2, ny3, bBetween_y); 
  call NumberBetween(nx1, nx2, nx3, bBetween_x);
  bSquareBetween = (bBetween_y || bBetween_x);
}


/* -------------------------------------------------------------------------- */
/* Canonical positions */


procedure IsCanonX(nFiles, nRanks, nPos, bIsCanonX) {
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   call lexLe2(2*nBKx+1, 2*nWKx+1, nFiles, nFiles, b1);
   call lexLe3(2*nBKx+1, 2*nWKx+1, 2*nWRx+1, nFiles, nFiles, nFiles, b2);
   bIsCanonX = ite(bWRCaptured, b1, b2);
}

procedure IsCanonY(nFiles, nRanks, nPos, bIsCanonY) {
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   call lexLe2(2*nBKy+1, 2*nWKy+1, nRanks, nRanks, b1);
   call lexLe3(2*nBKy+1, 2*nWKy+1, 2*nWRy+1, nRanks, nRanks, nRanks, b2);
   bIsCanonY = ite(bWRCaptured, b1, b2);
}

procedure IsCanonDiag(nFiles, nRanks, nPos, bIsCanonD) {
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   call lexLe2(nBKx, nWKx, nBKy, nWKy, b1);
   call lexLe3(nBKx, nWKx, nWRx, nBKy, nWKy, nWRy, b2);
   bIsCanonD = ite(bWRCaptured, b1, b2);
}

procedure IsCanon(nFiles, nRanks, nPos, bIsCanon) {
   call IsCanonX(nFiles, nRanks, nPos, bIsCanonX);
   call IsCanonY(nFiles, nRanks, nPos, bIsCanonY);
   call IsCanonDiag(nFiles, nRanks, nPos, bIsCanonDiag);
   bIsCanon = bIsCanonX && bIsCanonY && bIsCanonDiag;
}

/* -------------------------------------------------------------------------- */
/* Reflections of the board (used to find the canonical position for the given board) */

procedure ReflectX(nFiles, nRanks, nPos1, nPos2) {
   call IsCanonX(nFiles, nRanks, nPos1, b);
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   nWKx = ite(b, nWKx, nFiles-1-nWKx);
   nWRx = ite(b, nWRx, nFiles-1-nWRx);
   nBKx = ite(b, nBKx, nFiles-1-nBKx);
   call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn, nPos2);
}

procedure ReflectY(nFiles, nRanks, nPos1, nPos2) {
   call IsCanonY(nFiles, nRanks, nPos1, b);
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   nWKy = ite(b, nWKy, nRanks-1-nWKy);
   nWRy = ite(b, nWRy, nRanks-1-nWRy);
   nBKy = ite(b, nBKy, nRanks-1-nBKy);
   call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn, nPos2);
}

procedure ReflectDiag(nFiles, nRanks, nPos1, nPos2) {
   call IsCanonDiag(nFiles, nRanks, nPos1, b);
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   nWKx2 = ite(b, nWKx, nWKy);
   nWKy2 = ite(b, nWKy, nWKx);
   nWRx2 = ite(b, nWRx, nWRy);
   nWRy2 = ite(b, nWRy, nWRx);
   nBKx2 = ite(b, nBKx, nBKy);
   nBKy2 = ite(b, nBKy, nBKx);
   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured, bWhiteOnTurn, nPos2);
}

procedure Canonize(nFiles, nRanks, nPos1, nPos2) {
  call ReflectX(nFiles, nRanks, nPos1, nPos1x);
  call ReflectY(nFiles, nRanks, nPos1x, nPos1xy);
  call ReflectDiag(nFiles, nRanks, nPos1xy, nPos2);
}


/* -------------------------------------------------------------------------- */

procedure KingScope(nKx, nKy, nx, ny, bKingScope) {
  bKingScope = (nKx - 1 == nx || nKx == nx || nKx + 1 == nx) &&
               (nKy - 1 == ny || nKy == ny || nKy + 1 == ny) &&
               (nKx != nx || nKy != ny);
}

/* -------------------------------------------------------------------------- */

procedure RookScope(nRx, nRy, nx, ny, bRookScope) {
  bRookScope = (nRx == nx || nRy == ny) && (nRx != nx || nRy != ny);
}

/* -------------------------------------------------------------------------- */

procedure WRAttacksSquareIgnoringBK(nWRx, nWRy, nWKx, nWKy, nx, ny, bWRAttacksSquareIgnoringBK) {
  call RookScope(nWRx, nWRy, nx, ny, bRookScope);
  call SquareBetween(nWRx, nWRy, nWKx, nWKy, nx, ny, bSquareBetween);
  bWRAttacksSquareIgnoringBK = bRookScope && !bSquareBetween;
}

procedure WRAttacksBK(nFiles, nRanks, nPos, bBKAttacked) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call WRAttacksSquareIgnoringBK(nWRx, nWRy, nWKx, nWKy, nBKx, nBKy, bBKAttacked);
}

procedure WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nx, ny, bWhiteAttacksSquareIgnoringBK) {
  call KingScope(nWKx, nWKy, nx, ny, bKingScope);
  call WRAttacksSquareIgnoringBK(nWRx, nWRy, nWKx, nWKy, nx, ny, bWRAttacksSquareIgnoringBK);
  bWhiteAttacksSquareIgnoringBK = bKingScope || bWRAttacksSquareIgnoringBK;
}

/* -------------------------------------------------------------------------- */

procedure WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo) {
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   call KingScope(nWKx, nWKy, nx, ny, bKingScope);
   call KingScope(nBKx, nBKy, nx, ny, bBKScope);
   bWKCanMoveTo =  bKingScope && nx < nFiles && ny < nRanks && !bBKScope && (nWRx != nx || nWRy != ny);
}

/* -------------------------------------------------------------------------- */

procedure WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo) {
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
   call WRAttacksSquareIgnoringBK(nWRx, nWRy, nWKx, nWKy, nx, ny, bWRAttacksSquareIgnoringBK);
   bWRCanMoveTo =  bWRAttacksSquareIgnoringBK && nx < nFiles && ny < nRanks  
                          && (nWKx != nx || nWKy != ny)
                          && (nBKx != nx || nBKy != ny);
}

/* ******************************************************************** */
/* Chess rules for legal positions                                      */      
/* ******************************************************************** */

procedure WithinRange(nFiles, nRanks, nPos, bWithinRange) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bWithinRange = (nWKx<nFiles && nWKy<nRanks &&
                  nBKx<nFiles && nBKy<nRanks &&
                  (bWRCaptured || (nWRx<nFiles && nWRy<nRanks)));
} 

/* -------------------------------------------------------------------------- */

procedure TwoPiecesOnSameSquare(nFiles, nRanks, nPos, bOnSameSquare) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bOnSameSquare = (nWKx == nBKx && nWKy == nBKy) || 
                  (nWKx == nWRx && nWKy == nWRy && !bWRCaptured) || 
                  (nWRx == nBKx && nWRy == nBKy && !bWRCaptured);
}

/* -------------------------------------------------------------------------- */

procedure KingsSeparated(nFiles, nRanks, nPos, bKingsSeparated) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bKingsSeparated = nWKx>nBKx+1 || nBKx>nWKx+1 || nWKy>nBKy+1 || nBKy>nWKy+1;
} 

/* -------------------------------------------------------------------------- */
procedure LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegalPositionWhite) {
  call WithinRange(nFiles, nRanks, nPos, bWithinRange);
  call IsWhiteOnTurn(nFiles, nRanks, nPos, bWhiteOnTurn);
  call TwoPiecesOnSameSquare(nFiles, nRanks, nPos, bOnSameSquare);
  call KingsSeparated(nFiles, nRanks, nPos, bKingsSeparated);
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn1);
  call WRAttacksBK(nFiles, nRanks, nPos, bWRAttacksBK);
  bLegalPositionWhite = !bOnSameSquare && bWithinRange && bKingsSeparated && !bWRAttacksBK && bWhiteOnTurn;
}

/* -------------------------------------------------------------------------- */

procedure LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegalPositionWhite) {
  call WithinRange(nFiles, nRanks, nPos, bWithinRange);
  call IsWhiteOnTurn(nFiles, nRanks, nPos, bWhiteOnTurn);
  call TwoPiecesOnSameSquare(nFiles, nRanks, nPos, bOnSameSquare);
  call KingsSeparated(nFiles, nRanks, nPos, bKingsSeparated);
  bLegalPositionWhite = !bOnSameSquare && bWithinRange && bKingsSeparated && !bWhiteOnTurn;
}


/* ******************************************************************** */
/* Chess rules for legal moves                                          */      
/* ******************************************************************** */

procedure LegalMoveWK(nFiles, nRanks, nPos1, nPos2, bLegalMoveWK) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalPositionWhiteToMove(nFiles, nRanks, nPos1, bLegalPositionWhiteToMove1);
  call LegalPositionBlackToMove(nFiles, nRanks, nPos2, bLegalPositionBlackToMove2);
  call KingScope(nWKx1, nWKy1, nWKx2, nWKy2, bKingScope);
  bOtherAfterMoveWK = (nBKx2==nBKx1 && nBKy2==nBKy1 && nWRx2==nWRx1 && nWRy2==nWRy1 && 
                             !(bWRCaptured1 ^^ bWRCaptured2));
  bLegalMoveWK = bLegalPositionWhiteToMove1 && bLegalPositionBlackToMove2 &&
                        bKingScope && bOtherAfterMoveWK;
}

/* -------------------------------------------------------------------------- */
      
procedure LegalMoveBK(nFiles, nRanks, nPos1, nPos2, bLegalMoveBK) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalPositionBlackToMove(nFiles, nRanks, nPos1, bLegalPositionBlackToMove1);
  call LegalPositionWhiteToMove(nFiles, nRanks, nPos2, bLegalPositionWhiteToMove2);
  call KingScope(nBKx1, nBKy1, nBKx2, nBKy2, bKingScope);
  bOtherAfterMoveBK = (nWKx2==nWKx1 && nWKy2==nWKy1 && nWRx2==nWRx1 && nWRy2==nWRy1) &&
                             ((bWRCaptured1 && bWRCaptured2) ||
                              (!bWRCaptured1 && nBKx2==nWRx1 && nBKy2==nWRy1 && bWRCaptured2) ||
                              (!bWRCaptured1 && (nBKx2!=nWRx1 || nBKy2!=nWRy1) && !bWRCaptured2));
  bLegalMoveBK = bLegalPositionBlackToMove1 && bLegalPositionWhiteToMove2 &&
                        bKingScope && bOtherAfterMoveBK;
}

/* -------------------------------------------------------------------------- */

procedure LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalPositionWhiteToMove(nFiles, nRanks, nPos1, bLegalPositionWhiteToMove1);
  call LegalPositionBlackToMove(nFiles, nRanks, nPos2, bLegalPositionBlackToMove2);
  call WRAttacksSquareIgnoringBK(nWRx1, nWRy1, nWKx1, nWKy1, nWRx2, nWRy2, bWRAttacksSquare);
  bMoveWR = ((nWRx2==nWRx1 && nWRy2!=nWRy1) || (nWRx2!=nWRx1 && nWRy2==nWRy1)) && 
                   !bWRCaptured1 && !bWRCaptured2;
  bOtherAfterMoveWR = (nBKx2==nBKx1 && nBKy2==nBKy1 && nWKx2==nWKx1 && nWKy2==nWKy1);
  bLegalMoveWR = bLegalPositionWhiteToMove1 && bLegalPositionBlackToMove2 &&
                        bMoveWR && bOtherAfterMoveWR &&
                        bWRAttacksSquare;
}

/* -------------------------------------------------------------------------- */

/* All possible legal moves of players: */
procedure LegalMoveWhite(nFiles, nRanks, nPos1, nPos2, bLegalMoveWhite) {
  call LegalMoveWK(nFiles, nRanks, nPos1, nPos2, bLegalMoveWK);
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  bLegalMoveWhite = bLegalMoveWK || bLegalMoveWR;
}

/* -------------------------------------------------------------------------- */

procedure LegalMoveBlack(nFiles, nRanks, nPos1, nPos2, bLegalMoveBlack) {
  call LegalMoveBK(nFiles, nRanks, nPos1, nPos2, bLegalMoveBK);
  bLegalMoveBlack = bLegalMoveBK;
}


/* ******************************************************************** */ 
/* Mate, Stalemate                                                      */
/* ******************************************************************** */

procedure BKCannotMove(nFiles, nRanks, nPos, bBKCannotMove) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nBKxInc =   nBKx+1;
  nBKyInc =   nBKy+1;
  nBKxDec =   nBKx-1;
  nBKyDec =   nBKy-1;
  bBKCannotMove = !bWhiteOnTurn;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxInc, nBKy, bSquareAttacked);
  bBKCannotMove &&= nBKx==nFiles-1 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxInc, nBKyInc, bSquareAttacked);
  bBKCannotMove &&= nBKx==nFiles-1 || nBKy==nRanks-1 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxInc, nBKyDec, bSquareAttacked);
  bBKCannotMove &&= nBKx==nFiles-1 || nBKy==0 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxDec, nBKy, bSquareAttacked);
  bBKCannotMove &&= nBKx==0 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxDec, nBKyInc, bSquareAttacked);
  bBKCannotMove &&= nBKx==0 || nBKy==nRanks-1 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKxDec, nBKyDec, bSquareAttacked);
  bBKCannotMove &&= nBKx==0 || nBKy==0 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKx, nBKyInc, bSquareAttacked);
  bBKCannotMove &&= nBKy==nRanks-1 || bSquareAttacked;

  call WhiteAttacksSquareIgnoringBK(nWKx, nWKy, nWRx, nWRy, nBKx, nBKyDec, bSquareAttacked);
  bBKCannotMove &&= nBKy==0 || bSquareAttacked;
}

/* -------------------------------------------------------------------------- */
procedure Mate(nFiles, nRanks, nPos, bMate) {
/*  call IsWhiteOnTurn(nFiles, nRanks, nPos, bIsWhiteOnTurn);*/
  call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegalPositionBlackToMove);
  call BKCannotMove(nFiles, nRanks, nPos, bBKCannotMove);
  call WRAttacksBK(nFiles, nRanks, nPos, bBKAttacked);
/*  bMate = !bIsWhiteOnTurn && bBKCannotMove && bBKAttacked;*/
  bMate = bLegalPositionBlackToMove && bBKCannotMove && bBKAttacked;
}

/* -------------------------------------------------------------------------- */
procedure MateOpt(nFiles, nRanks, nPos, bMate) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call AbsDiff(nBKy,nWRy,nBKynWRy);
  call AbsDiff(nWKy,nBKy,nWKynBKy);
  call AbsDiff(nBKx,nWRx,nBKxnWRx);
  call AbsDiff(nWKx,nBKx,nWKxnBKx);
  bMate = !bWhiteOnTurn && 
   (nBKx==0 && nWRx==0 && nWKx==2 && nBKynWRy>1 && 
    ((nBKy!=0 && nBKy!=nRanks-1) || nWKynBKy <= 1) && 
    ((nBKy==0 || nBKy==nRanks-1) || nWKy==nBKy)) || 
   (nBKy==0 && nWRy==0 && nWKy==2 && nBKxnWRx > 1 && 
    ((nBKx!=0 && nBKx!=nFiles-1) || nWKxnBKx <= 1) && 
    ((nBKx==0 || nBKx==nFiles-1) || nWKx==nBKx)) || 
   (nBKx==nFiles-1 && nWRx==nFiles-1 && nWKx==nFiles-3 && nBKynWRy>1 && 
    ((nBKy!=0 && nBKy!=nRanks-1) || nWKynBKy <= 1)   && 
    ((nBKy == 0 || nBKy == nRanks-1) || nWKy==nBKy)) || 
   (nBKy == nRanks-1 && nWRy == nRanks-1 && nWKy == nRanks-3 && nBKxnWRx>1 && 
    ((nBKx!= 0 && nBKx!=nFiles-1) || nWKxnBKx <= 1) && 
    ((nBKx== 0 || nBKx==nFiles-1) || nWKx==nBKx));
}

procedure MateOptSym(nFiles, nRanks, nPos, bMate) {
  call Canonize(nFiles, nRanks, nPos, nPosC);
  call Bitvector2Pos(nFiles, nRanks, nPosC, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call AbsDiff(nBKy,nWRy,nBKynWRy);
  call AbsDiff(nWKy,nBKy,nWKynBKy);
  call AbsDiff(nBKx,nWRx,nBKxnWRx);
  call AbsDiff(nWKx,nBKx,nWKxnBKx);
    bMate = !bWhiteOnTurn &&
      (nBKx==0 && nWRx==0 && nWKx==2 && nBKynWRy > 1 && (nBKy!=0 || nWKynBKy <= 1) && (nBKy==0 || nWKy==nBKy)) ||
      (nBKy==0 && nWRy==0 && nWKy==2 && nBKxnWRx > 1 && (nBKx!=0 || nWKxnBKx <= 1) && (nBKx==0 || nWKx==nBKx));
}


/* -------------------------------------------------------------------------- */
procedure Stalemate(nFiles, nRanks, nPos, bStalemate) {
/*  call IsWhiteOnTurn(nFiles, nRanks, nPos, bIsWhiteOnTurn);*/
  call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegalPositionBlackToMove);
  call BKCannotMove(nFiles, nRanks, nPos, bBKCannotMove);
  call WRAttacksBK(nFiles, nRanks, nPos, bBKAttacked);
/*  bStalemate = bBKCannotMove && !bBKAttacked && !bIsWhiteOnTurn;*/
  bStalemate = bLegalPositionBlackToMove && bBKCannotMove && !bBKAttacked;
}

/* -------------------------------------------------------------------------- */
procedure StalemateOpt(nFiles, nRanks, nPos, bStalemate) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call AbsDiff(nBKy, nWKy, nDiffBKyWKy);
  call AbsDiff(nBKx, nWKx, nDiffBKxWKx);
  call AbsDiff(nBKy, nWRy, nDiffBKyWRy);
  call AbsDiff(nBKx, nWRx, nDiffBKxWRx);
  bKingScope = (nWKx - 1 == nWRx || nWKx == nWRx || nWKx + 1 == nWRx) &&
               (nWKy - 1 == nWRy || nWKy == nWRy || nWKy + 1 == nWRy) &&
               (nWKx != nWRx || nWKy != nWRy);
  bStalemate = !bWhiteOnTurn &&
                 ((nBKx == 0 && nBKy == 0) || 
                   (nBKx == nFiles-1 && nBKy == 0) || 
                   (nBKx == 0 && nBKy == nRanks-1) || 
                   (nBKx == nFiles-1 && nBKy == nRanks-1)) &&
                 ((nDiffBKxWRx == 1 && nBKy != nWRy && 
                   ((nBKx == nWKx && nDiffBKyWKy == 2) || (nDiffBKyWRy == 1 && bKingScope))) 
                 ||
                  (nDiffBKyWRy == 1 && nBKx != nWRx && 
                   ((nBKy == nWKy && nDiffBKxWKx == 2) || (nDiffBKxWRx == 1 && bKingScope))));
}

procedure StalemateOptSym(nFiles, nRanks, nPos, bStalemate) {
  call Canonize(nFiles, nRanks, nPos, nPosC);
  call Bitvector2Pos(nFiles, nRanks, nPosC, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
    bStalemate = !bWhiteOnTurn &&
      (nBKx == 0 && nBKy == 0) && (nWRx == 1 && 0 != nWRy) &&
      nWKy == 2 && (0 == nWKx || (nWRy == 1 && ((nWKx == 1 || nWKx == 2))));
}


/* ******************************************************************** */
/* Definitions for Strategy                                             */
/* ******************************************************************** */

procedure Room(nFiles, nRanks, nPos, nRoom) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nx = ite(nBKx < nWRx, nWRx, nFiles-1-nWRx);
  ny = ite(nBKy < nWRy, nWRy, nRanks-1-nWRy);
  nRoom = ite (nWRx==nBKx || nWRy==nBKy, nFiles+nRanks-1, nx+ny);
}

/* -------------------------------------------------------------------------- */

/* White rook is exposed to attack of the black king: */
procedure WRExposed(nFiles, nRanks, nPos, bWRExposed) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call ChebyshevDistance(nWKx, nWKy, nWRx, nWRy, nCDWKWR);
  call ChebyshevDistance(nBKx, nBKy, nWRx, nWRy, nCDBKWR);
  bWRExposed = (!bWhiteOnTurn || nCDWKWR>nCDBKWR+1) && (bWhiteOnTurn || nCDWKWR>nCDBKWR);
}


/* -------------------------------------------------------------------------- */

/* The kings are separated by a rook (either horizontally or vertically): */
procedure WRDivides(nFiles, nRanks, nPos, bWRDivides) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call SquareDivides(nWKx, nWKy, nWRx, nWRy, nBKx, nBKy, bSquareDivides);
  bWRDivides = bSquareDivides;
}

/* -------------------------------------------------------------------------- */

/* Critical square: */
procedure CriticalSquare(nFiles, nRanks, nPos, nCSx, nCSy) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nCSx=ite(nWRx==nBKx,nWRx,ite(nWRx>nBKx,nWRx-1,nWRx+1));
  nCSy=ite(nWRy==nBKy,nWRy,ite(nWRy>nBKy,nWRy-1,nWRy+1));
}

/* -------------------------------------------------------------------------- */

procedure MDCS(nFiles, nRanks, nPos, nMD) {
  call CriticalSquare(nFiles, nRanks, nPos, nCSx, nCSy);
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  call ManhattanDistance(nWKx, nWKy, nCSx, nCSy, nMD);
}

/* -------------------------------------------------------------------------- */

/* White king (after his moves) is closer to the critical square in sence of Manhattan distance: */
procedure ApproachCriticalSquare(nFiles, nRanks, nPos1, nPos2, bApproachCriticalSquare) {
  call MDCS(nFiles, nRanks, nPos1, nMD1);
  call MDCS(nFiles, nRanks, nPos2, nMD2);
  bApproachCriticalSquare = nMD1>nMD2;
}

/* -------------------------------------------------------------------------- */

/* Pieces forming the so-called L-shaped pattern: */
procedure LPattern(nFiles, nRanks, nPos, bLPattern) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bLPattern = (nWKy==nBKy && (nWKx==nBKx+2 || nBKx==nWKx+2) && nWRx==nWKx && (nWRy==nWKy+1 || nWKy==nWRy+1))
              ||
              (nWKx==nBKx && (nWKy==nBKy+2 || nBKy==nWKy+2) && nWRy==nWKy && (nWRx==nWKx+1 || nWKx==nWRx+1));
}


procedure LPatternBlack(nFiles, nRanks, nPos, bLPattern) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bLPattern =
    (nWKy==nBKy && (nWKx==nBKx+3 || nBKx==nWKx+3) && nWRx==nWKx && (nWRy==nWKy+1 || nWKy==nWRy+1)) ||
    (nWKx==nBKx && (nWKy==nBKy+3 || nBKy==nWKy+3) && nWRy==nWKy && (nWRx==nWKx+1 || nWKx==nWRx+1)) ||

    (nWKy==nBKy+1 && nWKx==nBKx+2 && nWRx==nWKx && nWRy==nWKy+1) ||
    (nWKy+1==nBKy && nWKx==nBKx+2 && nWRx==nWKx && nWKy==nWRy+1) ||
    (nWKy==nBKy+1 && nBKx==nWKx+2 && nWRx==nWKx && nWRy==nWKy+1) ||
    (nWKy+1==nBKy && nBKx==nWKx+2 && nWRx==nWKx && nWKy==nWRy+1) ||
    (nWKx==nBKx+1 && nWKy==nBKy+2 && nWRy==nWKy && nWRx==nWKx+1) ||
    (nWKx+1==nBKx && nWKy==nBKy+2 && nWRy==nWKy && nWKx==nWRx+1) ||
    (nWKx==nBKx+1 && nBKy==nWKy+2 && nWRy==nWKy && nWRx==nWKx+1) ||
    (nWKx+1==nBKx && nBKy==nWKy+2 && nWRy==nWKy && nWKx==nWRx+1) ||

    (nWKy==nBKy+1 && nWKx==nBKx+3 && nWRx==nWKx && nWRy==nWKy+1) ||
    (nWKy+1==nBKy && nWKx==nBKx+3 && nWRx==nWKx && nWKy==nWRy+1) ||
    (nWKy==nBKy+1 && nBKx==nWKx+3 && nWRx==nWKx && nWRy==nWKy+1) ||
    (nWKy+1==nBKy && nBKx==nWKx+3 && nWRx==nWKx && nWKy==nWRy+1) ||
    (nWKx==nBKx+1 && nWKy==nBKy+3 && nWRy==nWKy && nWRx==nWKx+1) ||
    (nWKx+1==nBKx && nWKy==nBKy+3 && nWRy==nWKy && nWKx==nWRx+1) ||
    (nWKx==nBKx+1 && nBKy==nWKy+3 && nWRy==nWKy && nWRx==nWKx+1) ||
    (nWKx+1==nBKx && nBKy==nWKy+3 && nWRy==nWKy && nWKx==nWRx+1);
}
	  

/* -------------------------------------------------------------------------- */

/* Distance of the white king and white rook has not decreased after move in sense of Chebyshev distance: */
procedure WKAndWRNotDiverging(nFiles, nRanks, nPos1, nPos2, bWKAndWRNotDiverging) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call ChebyshevDistance(nWKx1, nWKy1, nWRx1, nWRy1, nCDWKWR1);
  call ChebyshevDistance(nWKx2, nWKy2, nWRx2, nWRy2, nCDWKWR2);
  bWKAndWRNotDiverging = nCDWKWR1>=nCDWKWR2;
}

/* -------------------------------------------------------------------------- */

/* Diagonal and non-diagonal moves of white king: */
procedure KingDiag(nFiles, nRanks, nPos1, nPos2, bKingDiag) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  bKingDiag = nWKx2!=nWKx1 && nWKy2!=nWKy1;
}

/* -------------------------------------------------------------------------- */

procedure KingNotDiag(nFiles, nRanks, nPos1, nPos2, bKingNotDiag) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  bKingNotDiag = nWKx2==nWKx1 || nWKy2==nWKy1;
}

/* -------------------------------------------------------------------------- */

/* Black king and white rook are not placed next to each other: */
procedure BKNextWR(nFiles, nRanks, nPos, bBKNextWR) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bBKNextWR = nWRx<=nBKx+1 && nBKx<=nWRx+1 && nWRy<=nBKy+1 && nBKy<=nWRy+1;
}

/* -------------------------------------------------------------------------- */

/* White king and white rook are placed next to each other: */
procedure WKNextWR(nFiles, nRanks, nPos, bWKNextWR) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bWKNextWR = !(nWRx>nWKx+1 || nWKx>nWRx+1 || nWRy>nWKy+1 || nWKy>nWRy+1);
}

/* -------------------------------------------------------------------------- */

/* Attempt to divide kings: */
procedure DivideAttempt(nFiles, nRanks, nPos1, nPos2, bDivideAttempt) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  bDivideAttempt = ((nBKx1<nWKx1 && nWRx1!=nWKx1-1 && nWRx2==nWKx1-1) || 
                    (nBKx1>nWKx1 && nWRx1!=nWKx1+1 && nWRx2==nWKx1+1) || 
                    (nBKy1<nWKy1 && nWRy1!=nWKy1-1 && nWRy2==nWKy1-1) || 
                    (nBKy1>nWKy1 && nWRy1!=nWKy1+1 && nWRy2==nWKy1+1) || 
                    (nWRx1==nWKx1 && nWKx1==nBKx1 && (nWRx2==nWKx1+1 || nWRx2==nWKx1-1)) || 
                    (nWRy1==nWKy1 && nWKy1==nBKy1 && (nWRy2==nWKy1+1 || nWRy2==nWKy1-1)));
}

/* -------------------------------------------------------------------------- */

procedure BackMove(nFiles, nRanks, nPos1, nPos2, bBackMove) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  bBackMove = ((nBKx1 == 0 &&        nWRx1 == 1 &&          nWKx2 < nWKx1) ||
               (nBKx1 == nFiles-1 && nWRx1 == nFiles-2 && nWKx2 > nWKx1) ||
               (nBKy1 == 0 &&        nWRy1 == 1 &&          nWKy2 < nWKy1) ||
               (nBKy1 == nRanks-1 && nWRy1 == nRanks-2 && nWKy2 > nWKy1));
}

/* -------------------------------------------------------------------------- */

procedure KingsOnSameEdge(nFiles, nRanks, nPos, bKingsOnSameEdge) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bKingsOnSameEdge = ((nBKx == 0 && nWKx == 0) ||
                      (nBKy == 0 && nWKy == 0) ||
                      (nBKx == nFiles-1 && nWKx == nFiles-1) ||
                      (nBKy == nRanks-1 && nWKy == nRanks-1));
}



/* ******************************************************************** */
/* Strategy (for White)                                                 */
/* ******************************************************************** */

procedure ImmediateMateCond(nFiles, nRanks, nPos1, nPos2, bImmediateMateCond) {
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  call MateOptSym(nFiles, nRanks, nPos2, bMate);
  bImmediateMateCond = bLegalMoveWR && bMate;
}

procedure ReadyToMateCond(nFiles, nRanks, nPos1, nPos2, bReadyToMateCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalMoveWhite(nFiles, nRanks, nPos1, nPos2, bLegalMoveWhite);
  bReadyToMateCond = bLegalMoveWhite &&
      ((nBKx2 == 0 && nBKy2 == 0 && nWKx2 == 1 && nWKy2 == 2 && nWRx2 > 2 && nWRy2 > 0) ||
       (nBKx2 == 0 && nBKy2 == 1 && nWKx2 == 2 && nWKy2 == 1 && nWRx2 >= 1 && nWRy2 == 2) ||
       (nBKx2 == 0 && nWKx2 == 2 && nWRx2 >= 2 && (nWKy2==nBKy2+1 || nWKy2+1==nBKy2) 
        && (nWRy2==nBKy2+1 || nWRy2+1==nBKy2) && nWKy2!=nWRy2) ||
       (nBKx2 == nFiles-1 && nBKy2 == 0 && nWKx2 == nFiles-2 && nWKy2 == 2 && nWRx2 < nFiles-3 && nWRy2 > 0) ||
       (nBKx2 == nFiles-1 && nBKy2 == 1 && nWKx2 == nFiles-3 && nWKy2 == 1 && nWRx2 <= nFiles-2 && nWRy2 == 2) ||
       (nBKx2 == nFiles-1 && nWKx2 == nFiles-3 && nWRx2 <= nFiles-3 && (nWKy2==nBKy2+1 || nWKy2+1==nBKy2) 
        && (nWRy2==nBKy2+1 || nWRy2+1==nBKy2) && nWKy2!=nWRy2) ||
       (nBKx2 == 0 && nBKy2 == nRanks-1 && nWKx2 == 1 && nWKy2 == nRanks-3 && nWRx2 > 2 && nWRy2 < nRanks-1) ||
       (nBKx2 == 0 && nBKy2 == nRanks-2 && nWKx2 == 2 && nWKy2 == nRanks-2 && nWRx2 >= 1 && nWRy2 == nRanks-3) ||
       (nBKx2 == 0 && nWKx2 == 2 && nWRx2 >= 2 && (nWKy2==nBKy2+1 || nWKy2+1==nBKy2) 
        && (nWRy2==nBKy2+1 || nWRy2+1==nBKy2) && nWKy2!=nWRy2) ||
       (nBKx2 == nFiles-1 && nBKy2 == nRanks-1 && nWKx2 == nFiles-2 && nWKy2 == nRanks-3 && nWRx2 < nFiles-3 && nWRy2 < nRanks-1) ||
       (nBKx2 == nFiles-1 && nBKy2 == nRanks-2 && nWKx2 == nFiles-3 && nWKy2 == nRanks-2 && nWRx2 <= nFiles-2 && nWRy2 == nRanks-3) ||
       (nBKx2 == nFiles-1 && nWKx2 == nFiles-3 && nWRx2 <= nFiles-3 && (nWKy2==nBKy2+1 || nWKy2+1==nBKy2) 
        && (nWRy2==nBKy2+1 || nWRy2+1==nBKy2) && nWKy2!=nWRy2) ||
       (nBKy2 == 0 && nBKx2 == 0 && nWKy2 == 1 && nWKx2 == 2 && nWRy2 > 2 && nWRx2 > 0) ||
       (nBKy2 == 0 && nBKx2 == 1 && nWKy2 == 2 && nWKx2 == 1 && nWRy2 >= 1 && nWRx2 == 2) ||
       (nBKy2 == 0 && nWKy2 == 2 && nWRy2 >= 2 && (nWKx2==nBKx2+1 || nWKx2+1==nBKx2) 
        && (nWRx2==nBKx2+1 || nWRx2+1==nBKx2) && nWKx2!=nWRx2) ||
       (nBKy2 == nRanks-1 && nBKx2 == 0 && nWKy2 == nRanks-2 && nWKx2 == 2 && nWRy2 < nRanks-3 && nWRx2 > 0) ||
       (nBKy2 == nRanks-1 && nBKx2 == 1 && nWKy2 == nRanks-3 && nWKx2 == 1 && nWRy2 <= nRanks-2 && nWRx2 == 2) ||
       (nBKy2 == nRanks-1 && nWKy2 == nRanks-3 && nWRy2 <= nRanks-3 && (nWKx2==nBKx2+1 || nWKx2+1==nBKx2) 
        && (nWRx2==nBKx2+1 || nWRx2+1==nBKx2) && nWKx2!=nWRx2) ||
       (nBKy2 == 0 && nBKx2 == nFiles-1 && nWKy2 == 1 && nWKx2 == nFiles-3 && nWRy2 > 2 && nWRx2 < nFiles-1) ||
       (nBKy2 == 0 && nBKx2 == nFiles-2 && nWKy2 == 2 && nWKx2 == nFiles-2 && nWRy2 >= 1 && nWRx2 == nFiles-3) ||
       (nBKy2 == 0 && nWKy2 == 2 && nWRy2 >= 2 && (nWKx2==nBKx2+1 || nWKx2+1==nBKx2) 
        && (nWRx2==nBKx2+1 || nWRx2+1==nBKx2) && nWKx2!=nWRx2) ||
       (nBKy2 == nRanks-1 && nBKx2 == nFiles-1 && nWKy2 == nRanks-2 && nWKx2 == nFiles-3 && nWRy2 < nRanks-3 && nWRx2 < nFiles-1) ||
       (nBKy2 == nRanks-1 && nBKx2 == nFiles-2 && nWKy2 == nRanks-3 && nWKx2 == nFiles-2 && nWRy2 <= nRanks-2 && nWRx2 == nFiles-3) ||
       (nBKy2 == nRanks-1 && nWKy2 == nRanks-3 && nWRy2 <= nRanks-3 && (nWKx2==nBKx2+1 || nWKx2+1==nBKx2) 
        && (nWRx2==nBKx2+1 || nWRx2+1==nBKx2) && nWKx2!=nWRx2));
}


/* -------------------------------------------------------------------------- */
procedure ReadyToMateCanonCond(nFiles, nRanks, nPos, bReadyToMateCanonCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bReadyToMateCanonCond = 
      ((nBKx == 0 && nBKy == 0 && nWKx == 1 && nWKy == 2 && nWRx > 2  && nWRy > 0) ||
       (nBKx == 0 && nBKy == 1 && nWKx == 2 && nWKy == 1 && nWRx >= 1 && nWRy == 2) ||
       (nBKx == 0 && nWKx == 2 && nWRx >= 2 && ( nWKy==nBKy+1 || nWKy+1==nBKy)  && (nWRy==nBKy+1 || nWRy+1==nBKy) && nWKy!=nWRy));
}

/* -------------------------------------------------------------------------- */

/* fixme - how to perform this sort of symmetry, involving diag-reflection?! */
procedure ReadyToMateCondSym(nFiles, nRanks, nPos1, nPos2, bReadyToMateCond) {
  call LegalMoveWhite(nFiles, nRanks, nPos1, nPos2, bLegalMoveWhite);
  call Canonize(nFiles, nRanks, nPos2, nPos2c);
  call ReadyToMateCanonCond(nFiles, nRanks, nPos2c, bReadyToMateCanonCond);
  bReadyToMateCond = bLegalMoveWhite && bReadyToMateCanonCond;
}

/* -------------------------------------------------------------------------- */

procedure SqueezeCond(nFiles, nRanks, nPos1,nPos2,b)  {
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  call WRExposed(nFiles, nRanks, nPos2, bWRExposed);
  call WRDivides(nFiles, nRanks, nPos2, bWRDivides);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  call Room(nFiles, nRanks, nPos1, nRoom1);
  call Room(nFiles, nRanks, nPos2, nRoom2);
  b = (bLegalMoveWR && !bWRExposed && bWRDivides && !bStalemate && nRoom1>nRoom2);
}

/* -------------------------------------------------------------------------- */

procedure ApproachCond(nFiles, nRanks, nPos1, nPos2, bApproachCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call LegalMoveWK(nFiles, nRanks, nPos1, nPos2, bLegalMoveWK);
  call ApproachCriticalSquare(nFiles, nRanks, nPos1, nPos2, bApproachCriticalSquare);
  call WRExposed(nFiles, nRanks, nPos2, bWRExposed);
  call WRDivides(nFiles, nRanks, nPos2, bWRDivides);
  call LPattern(nFiles, nRanks, nPos2, bLPattern);
  call Room(nFiles, nRanks, nPos2, nRoom);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  call KingsOnSameEdge(nFiles, nRanks, nPos2, bKingsOnSameEdge);
  call BackMove(nFiles, nRanks, nPos1, nPos2, bBackMove);
  call ChebyshevDistance(nWKx1, nWKy1, nWRx1, nWRy1, nCDWKWR);

  bApproachCond = bLegalMoveWK && bApproachCriticalSquare && !bWRExposed && 
    (bWRDivides || bLPattern) && 
    (nRoom > 3 || (!bKingsOnSameEdge && (nCDWKWR!=1 || !bBackMove))) && !bStalemate;
}

/* -------------------------------------------------------------------------- */

procedure ApproachDiagCond(nFiles, nRanks, nPos1, nPos2, bApproachDiagCond) {
  call ApproachCond(nFiles, nRanks, nPos1, nPos2, bApproachCond);
  call KingDiag(nFiles, nRanks, nPos1, nPos2, bKingDiag);
  bApproachDiagCond = bApproachCond && bKingDiag;
}

/* -------------------------------------------------------------------------- */

procedure ApproachNotDiagCond(nFiles, nRanks, nPos1, nPos2, bApproachNotDiagCond) {
  call ApproachCond(nFiles, nRanks, nPos1, nPos2, bApproachCond);
  call KingNotDiag(nFiles, nRanks, nPos1, nPos2, bKingNotDiag);
  bApproachNotDiagCond = bApproachCond && bKingNotDiag;
}

/* -------------------------------------------------------------------------- */

procedure KeepRoomCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call LegalMoveWK(nFiles, nRanks, nPos1, nPos2, bLegalMoveWK);
  call WRExposed(nFiles, nRanks, nPos2, bWRExposed);
  call WRDivides(nFiles, nRanks, nPos2, bWRDivides);
  call WKAndWRNotDiverging(nFiles, nRanks, nPos1, nPos2, bWKAndWRNotDiverging);
  call Room(nFiles, nRanks, nPos2, nRoom);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  call KingsOnSameEdge(nFiles, nRanks, nPos2, bKingsOnSameEdge);
  call ChebyshevDistance(nWKx1, nWKy1, nWRx1, nWRy1, nCDWKWR);
  call BackMove(nFiles, nRanks, nPos1, nPos2, bBackMove);
  bKeepRoomCond = bLegalMoveWK && !bWRExposed && bWRDivides &&
                  bWKAndWRNotDiverging && (nRoom > 3 || (!bKingsOnSameEdge && (nCDWKWR!=1 || !bBackMove))) && !bStalemate;
}

/* -------------------------------------------------------------------------- */
procedure KeepRoomDiagCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomDiagCond) {
  call KeepRoomCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomCond); 
  call KingDiag(nFiles, nRanks, nPos1, nPos2, bKingDiag);
  bKeepRoomDiagCond = bKeepRoomCond && bKingDiag;
}

/* -------------------------------------------------------------------------- */
procedure KeepRoomNotDiagCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomNotDiagCond) {
  call KeepRoomCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomCond);
  call KingNotDiag(nFiles, nRanks, nPos1, nPos2, bKingNotDiag);
  bKeepRoomNotDiagCond = bKeepRoomCond && bKingNotDiag;
}

/* -------------------------------------------------------------------------- */

procedure RookHomeCond(nFiles, nRanks, nPos1, nPos2, bRookHomeCond) {
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  call DivideAttempt(nFiles, nRanks, nPos1, nPos2, bDivideAttempt);
  call BKNextWR(nFiles, nRanks, nPos2, bBKNextWR);
  call WKNextWR(nFiles, nRanks, nPos2, bWKNextWR);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  bRookHomeCond = bLegalMoveWR && bDivideAttempt && (!bBKNextWR || bWKNextWR) && !bStalemate;
}

/* -------------------------------------------------------------------------- */

procedure RookSafeCond(nFiles, nRanks, nPos1, nPos2, bRookSafeCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  call ChebyshevDistance(nBKx2, nBKy2, nWRx2, nWRy2, nCDBKWR2);
  call BKNextWR(nFiles, nRanks, nPos2, bBKNextWR);
  call WKNextWR(nFiles, nRanks, nPos2, bWKNextWR);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  bRookSafeCond = bLegalMoveWR && 
                  ((nWRx1!=0 && nWRx2==0) || (nWRx1!=nFiles-1 && nWRx2==nFiles-1) || 
                   (nWRy1!=0 && nWRy2==0) || (nWRy1!=nRanks-1 && nWRy2==nRanks-1)) && 
    (nCDBKWR2>2 || (bBKNextWR && bWKNextWR)) && !bStalemate;
}

procedure RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos2, bRookSafeCond) {
  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWRCaptured2, bWhiteOnTurn2);
  call LegalMoveWR(nFiles, nRanks, nPos1, nPos2, bLegalMoveWR);
  call ChebyshevDistance(nBKx2, nBKy2, nWRx2, nWRy2, nCDBKWR2);
  call StalemateOptSym(nFiles, nRanks, nPos2, bStalemate);
  bRookSafeCond = bLegalMoveWR && 
                  ((nWRx1!=0 && nWRx2==0) || (nWRx1!=nFiles-1 && nWRx2==nFiles-1) || 
                   (nWRy1!=0 && nWRy2==0) || (nWRy1!=nRanks-1 && nWRy2==nRanks-1)) &&
                   nCDBKWR2 == 2 && (nWRx2 == nWKx2 || nWRy2 == nWKy2) && !bStalemate;
}

/* -------------------------------------------------------------------------- */

/* 
NO MOVE CONDITIONS
*/

procedure RooksSquare(nFiles, nRanks, nRx, nRy, ni, nx, ny) {
  nx = ite(1 <= ni && ni <= nFiles, ni - 1, nRx);
  ny = ite(1 <= ni && ni <= nFiles, nRy, ni - nFiles - 1);
}

/*
procedure KingsSquare(nKx, nKy, ni, nx, ny) {
  nx = ite(ni == 1 || ni == 2 || ni == 5, nKx - 1, ite(ni == 6 || ni == 7, nKx, nKx + 1));
  ny = ite(ni == 1 || ni == 3 || ni == 6, nKy - 1, ite(ni == 5 || ni == 8, nKy, nKy + 1));
}
*/

procedure KingsSquare(nX, nY, nK, nX1, nY1) {
   nX1 = ite(nK == 1, nX - 1,
           ite (nK == 2, nX - 1,
             ite (nK == 3, nX + 1, 
               ite (nK == 4, nX + 1, 
                 ite (nK == 5, nX - 1,
                   ite (nK == 6, nX, 
                     ite (nK == 7, nX,
                       ite (nK == 8, nX + 1, nX))))))));
   nY1 = ite(nK == 1, nY - 1,
           ite (nK == 2, nY + 1,
             ite (nK == 3, nY - 1, 
               ite (nK == 4, nY + 1, 
                 ite (nK == 5, nY,
                   ite (nK == 6, nY - 1, 
                     ite (nK == 7, nY + 1,
                       ite (nK == 8, nY, nY))))))));
}


/* -------------------------------------------------------------------------- */
procedure NoImmediateMate(nFiles, nRanks, nPos, bNoImmediateMate) {
  bNoImmediateMate = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles + nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call MateOptSym(nFiles, nRanks, nPos1, bCond);
    bNoImmediateMate &&= (!bWRCanMoveTo || !bCond);
  }
}


procedure NoImmediateMateOpt(nFiles, nRanks, nPos, bNoImmediateMate) {
  bNoImmediateMate = true;
  bBlackOnTurn = false;

  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nx[0] = 0; ny[0] = nWRy;
  nx[1] = nFiles - 1; ny[1] = nWRy;
  nx[2] = nWRx; ny[2] = 0;
  nx[3] = nWRx; ny[3] = nRanks - 1;

  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos1);
     call MateOptSym(nFiles, nRanks, nPos1, bCond);
     bNoImmediateMate &&= (!bWRCanMoveTo || !bCond);
  }
}


/* -------------------------------------------------------------------------- */

procedure NoReadyToMateWROpt(nFiles, nRanks, nPos, bNoReadyToMateWR) {
   bBlackOnTurn = false;
   bNoReadyToMateWR = true;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   nx[0] = 0; ny[0] = nWRy;
   nx[1] = nFiles - 1; ny[1] = nWRy;
   nx[2] = nWRx; ny[2] = 0;
   nx[3] = nWRx; ny[3] = nRanks - 1;
   nx[4] = 2; ny[4] = nWRy;
   nx[5] = nFiles - 3; ny[5] = nWRy;
   nx[6] = nWRx; ny[6] = 2;
   nx[7] = nWRx; ny[7] = nRanks - 3;
   nx[8] = nBKx - 1; ny[8] = nWRy;
   nx[9] = nBKx + 1; ny[9] = nWRy;
   nx[10] = nWRx; ny[10] = nBKy - 1;
   nx[11] = nWRx; ny[11] = nBKy + 1;

   for (ni = 0; ni < 12; ni++) {
      nxe = nx[ni]; nye = ny[ni];
      call WRCanMoveTo(nFiles, nRanks, nPos, nxe, nye, bWRCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos1);
      call ReadyToMateCondSym(nFiles, nRanks, nPos, nPos1, bCond);
      bNoReadyToMateWR &&= (!bWRCanMoveTo || !bCond);
   }
}


procedure NoReadyToMateWR(nFiles, nRanks, nPos, bNoReadyToMateWR) {
  bNoReadyToMateWR = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles+nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call ReadyToMateCondSym(nFiles, nRanks, nPos, nPos1, bCond);
    bNoReadyToMateWR &&= (!bWRCanMoveTo || !bCond);
  }
}

/* -------------------------------------------------------------------------- */

procedure NoReadyToMateWK(nFiles, nRanks, nPos, bNoReadyToMateWK) {
   bNoReadyToMateWK = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   for (ni=1; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos1);
      call ReadyToMateCondSym(nFiles, nRanks, nPos /* fix me- delete this arg*/, nPos1, bCond);
      bNoReadyToMateWK &&= (!bWKCanMoveTo || !bCond);
   }
}

/* -------------------------------------------------------------------------- */

/*
procedure NoSqueezeOpt(nFiles, nRanks, nPos, bNoSqueeze) {
   bNoSqueeze = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   nx[0] = nWRx; ny[0] = (nWKy + nBKy) >> 1;
   nx[1] = nWRx; ny[1] = (nWKy + nBKy + 1) >> 1;
   nx[2] = (nWKx + nBKx) >> 1; ny[2] = nWRy;
   nx[3] = (nWKx + nBKx + 1) >> 1; ny[3] = nWRy;
   nx[4] = nWRx; ny[4] = nBKy - nWKx + nWRx;
   nx[5] = nWKx - nBKy + nWRy; ny[5] = nWRy;
   nx[6] = nWRx; ny[6] = nWKx + nBKy - nWRx;
   nx[7] = nWKx + nBKy - nWRy; ny[7] = nWRy;
   nx[8] = nWRx; ny[8] = nWKy - nBKx + nWRx;
   nx[9] = nBKx - nWKy + nWRy; ny[9] = nWRy;
   nx[10] = nWRx; ny[10] = nBKx + nWKy - nWRx;
   nx[11] = nBKx + nWKy - nWRy; ny[11] = nWRy;
   nx[12] = nBKx + 1; ny[12] =  nWRy;
   nx[13] = nBKx - 1; ny[13] = nWRy;
   nx[14] = nWRx; ny[14] = nBKy + 1;
   nx[15] = nWRx; ny[15] = nBKy - 1;
   
   for (ni = 0; ni < 16; ni++) {
      nxe = nx[ni]; nye = ny[ni];
      call WRCanMoveTo(nFiles, nRanks, nPos, nxe, nye, bWRCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos1);
      call SqueezeCond(nFiles, nRanks, nPos, nPos1, bCond);
      bNoSqueeze &&= (!bWRCanMoveTo || !bCond);
   }
}
*/

procedure NoSqueeze(nFiles, nRanks, nPos, bNoSqueeze) {
  bNoSqueeze = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles+nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call SqueezeCond(nFiles, nRanks, nPos, nPos1, bCond);
    bNoSqueeze &&= (!bWRCanMoveTo || !bCond);
  }
}

/* -------------------------------------------------------------------------- */

procedure NoApproachDiag(nFiles, nRanks, nPos, bNoApproachDiag) {
   bNoApproachDiag = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   for (ni=1; ni<=4; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos1);
      call ApproachCond(nFiles, nRanks, nPos, nPos1, bCond);
      bNoApproachDiag &&= (!bWKCanMoveTo || !bCond);
   }
}

/* -------------------------------------------------------------------------- */

procedure NoApproachNotDiag(nFiles, nRanks, nPos, bNoApproachNotDiag) {
   bNoApproachNotDiag = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   for (ni=5; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos1);
      call ApproachCond(nFiles, nRanks, nPos, nPos1, bCond);
      bNoApproachNotDiag &&= (!bWKCanMoveTo || !bCond);
   }
}

/* -------------------------------------------------------------------------- */

procedure NoKeepRoomDiag(nFiles, nRanks, nPos, bNoKeepRoomDiag) {
   bNoKeepRoomDiag = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   for (ni=1; ni<=4; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos1);
      call KeepRoomCond(nFiles, nRanks, nPos, nPos1, bCond);
      bNoKeepRoomDiag &&= (!bWKCanMoveTo || !bCond);
   }
}


/* -------------------------------------------------------------------------- */

procedure NoKeepRoomNotDiag(nFiles, nRanks, nPos, bNoKeepRoomNotDiag) {
   bNoKeepRoomNotDiag = true;
   bBlackOnTurn = false;
   call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);

   for (ni=5; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos1);
      call KeepRoomCond(nFiles, nRanks, nPos, nPos1, bCond);
      bNoKeepRoomNotDiag &&= (!bWKCanMoveTo || !bCond);
   }
}

/* -------------------------------------------------------------------------- */
procedure NoRookHomeOpt(nFiles, nRanks, nPos, bNoRookHome) {
  bNoRookHome = true;
  bBlackOnTurn = false;

  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nx[0] = nWRx; ny[0] = nWKy-1;
  nx[1] = nWRx; ny[1] = nWKy+1;
  nx[2] = nWKx-1; ny[2] = nWRy;
  nx[3] = nWKx+1; ny[3] = nWRy;

  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos1);
     call RookHomeCond(nFiles, nRanks, nPos, nPos1, bRookHome);
     bNoRookHome &&= (!bWRCanMoveTo || !bRookHome);
  }
}

procedure NoRookHome(nFiles, nRanks, nPos, bNoRookHome) {
  bNoRookHome = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles+nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call RookHomeCond(nFiles, nRanks, nPos, nPos1, bCond);
    bNoRookHome &&= (!bWRCanMoveTo || !bCond);
  }
}


/* -------------------------------------------------------------------------- */

procedure NoRookSafeOpt(nFiles, nRanks, nPos, bNoRookSafe) {
  bNoRookSafe = true;
  bBlackOnTurn = false;

  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  nx[0] = 0; ny[0] = nWRy;
  nx[1] = nFiles - 1; ny[1] = nWRy;
  nx[2] = nWRx; ny[2] = 0;
  nx[3] = nWRx; ny[3] = nRanks - 1;

  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos1);
     call RookSafeCond(nFiles, nRanks, nPos, nPos1, bRookSafe);
     bNoRookSafe &&= (!bWRCanMoveTo || !bRookSafe);
  }
}

procedure NoRookSafe(nFiles, nRanks, nPos, bNoRookSafe) {
  bNoRookSafe = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles+nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call RookSafeCond(nFiles, nRanks, nPos, nPos1, bCond);
    bNoRookSafe &&= (!bWRCanMoveTo || !bCond);
  }
}

procedure NoRookSafeSmallBoards(nFiles, nRanks, nPos, bNoRookSafeSB) {
  bNoRookSafeSB = true;
  bBlackOnTurn = false;
  call Bitvector2Pos(nFiles, nRanks, nPos, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy,
		     bWRCaptured, bWhiteOnTurn);
  for (ni=1; ni<=nFiles+nRanks; ni++) {
    call RooksSquare(nFiles, nRanks, nWRx, nWRy, ni, nx, ny);
    call WRCanMoveTo(nFiles, nRanks, nPos, nx, ny, bWRCanMoveTo);
    call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nx, ny,
		       bWRCaptured, bBlackOnTurn, nPos1);
    call RookSafeSmallBoardsCond(nFiles, nRanks, nPos, nPos1, bCond);
    bNoRookSafeSB &&= (!bWRCanMoveTo || !bCond);
  }
}

/* -------------------------------------------------------------------------- */
procedure StrategyRelation(nFiles, nRanks, nPos1, nPos2, b, nStep) {
  nsImmediateMate  =1;
  nsReadyToMate    =2;
  nsSqueeze        =3;
  nsApproachDiag   =4;
  nsApproachNotDiag=5;
  nsKeepRoomDiag   =6;
  nsKeepRoomNotDiag=7;
  nsRookHome       =8;
  nsRookSafe       =9;
  nsRookSafeSB  =10;
  
  call ImmediateMateCond(nFiles, nRanks, nPos1, nPos2, bImmediateMateCond);
  call NoImmediateMate(nFiles, nRanks, nPos1, bNoImmediateMateCond);

  call ReadyToMateCond(nFiles, nRanks, nPos1, nPos2, bReadyToMateCond);
  call NoReadyToMateWR(nFiles, nRanks, nPos1, bNoReadyToMateCondWR);
  call NoReadyToMateWK(nFiles, nRanks, nPos1, bNoReadyToMateCondWK);
  bNoReadyToMateCond = bNoReadyToMateCondWR && bNoReadyToMateCondWK;

  call SqueezeCond(nFiles, nRanks, nPos1, nPos2, bSqueezeCond);
  call NoSqueeze(nFiles, nRanks, nPos1, bNoSqueezeCond);

  call ApproachCond(nFiles, nRanks, nPos1, nPos2, bApproachCond);
  call KingDiag(nFiles, nRanks, nPos1, nPos2, bKingDiag);
  bApproachDiagCond = bApproachCond && bKingDiag;
  bApproachNotDiagCond = bApproachCond && !bKingDiag;
  call NoApproachDiag(nFiles, nRanks, nPos1, bNoApproachDiagCond);
  call NoApproachNotDiag(nFiles, nRanks, nPos1, bNoApproachNotDiagCond);

  call KeepRoomCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomCond);
  bKeepRoomDiagCond = bKeepRoomCond && bKingDiag;
  bKeepRoomNotDiagCond = bKeepRoomCond && !bKingDiag;
  call NoKeepRoomDiag(nFiles, nRanks, nPos1, bNoKeepRoomDiagCond);
  call NoKeepRoomNotDiag(nFiles, nRanks, nPos1, bNoKeepRoomNotDiagCond);

  call RookHomeCond(nFiles, nRanks, nPos1, nPos2, bRookHomeCond);
  call NoRookHome(nFiles, nRanks, nPos1, bNoRookHomeCond);

  call RookSafeCond(nFiles, nRanks, nPos1, nPos2, bRookSafeCond);
  call NoRookSafe(nFiles, nRanks, nPos1, bNoRookSafeCond);

  call RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos2, bRookSafeSBCond);
  
  b = ite(nStep == nsImmediateMate, 
         bImmediateMateCond,
         bNoImmediateMateCond &&
         ite(nStep == nsReadyToMate, 
            bReadyToMateCond,
            bNoReadyToMateCond && 
            ite(nStep == nsSqueeze, 
               bSqueezeCond,
               bNoSqueezeCond &&
               ite(nStep == nsApproachDiag, 
                  bApproachDiagCond,
                  bNoApproachDiagCond && 
                  ite(nStep == nsApproachNotDiag, 
                     bApproachNotDiagCond,
                     bNoApproachNotDiagCond && 
                     ite(nStep == nsKeepRoomDiag,
                        bKeepRoomDiagCond,
                        bNoKeepRoomDiagCond && 
                        ite(nStep == nsKeepRoomNotDiag,
                           bKeepRoomNotDiagCond,
                           bNoKeepRoomNotDiagCond && 
                              ite(nStep == nsRookHome, 
                                 bRookHomeCond,
                                 bNoRookHomeCond && 
                                 ite(nStep == nsRookSafe, 
                                    bRookSafeCond,
                                    bNoRookSafeCond &&
                                       ite(nStep == nsRookSafeSB,
                                       bRookSafeSBCond,
	false))))))))));
}

/* -------------------------------------------------------------------------- */
procedure StrategyRelationOpt(nFiles, nRanks, nPos1, nPos2, b, nStep) {
  nsImmediateMate   = 1;
  nsReadyToMate     = 2;
  nsSqueeze         = 3;
  nsApproachDiag    = 4;
  nsApproachNotDiag = 5;
  nsKeepRoomDiag    = 6;
  nsKeepRoomNotDiag = 7;
  nsRookHome        = 8;
  nsRookSafe        = 9;
  nsRookSafeSB      = 10;
  
  call ImmediateMateCond(nFiles, nRanks, nPos1, nPos2, bImmediateMateCond);
  call NoImmediateMateOpt(nFiles, nRanks, nPos1, bNoImmediateMateCond);

  call ReadyToMateCondSym(nFiles, nRanks, nPos1, nPos2, bReadyToMateCond);
  call NoReadyToMateWROpt(nFiles, nRanks, nPos1, bNoReadyToMateCondWR);
  call NoReadyToMateWK(nFiles, nRanks, nPos1, bNoReadyToMateCondWK);
  bNoReadyToMateCond = bNoReadyToMateCondWR && bNoReadyToMateCondWK;

  call SqueezeCond(nFiles, nRanks, nPos1, nPos2, bSqueezeCond);
  call NoSqueeze(nFiles, nRanks, nPos1, bNoSqueezeCond);

  call ApproachCond(nFiles, nRanks, nPos1, nPos2, bApproachCond);
  call KingDiag(nFiles, nRanks, nPos1, nPos2, bKingDiag);
  bApproachDiagCond = bApproachCond && bKingDiag;
  bApproachNotDiagCond = bApproachCond && !bKingDiag;
  call NoApproachDiag(nFiles, nRanks, nPos1, bNoApproachDiagCond);
  call NoApproachNotDiag(nFiles, nRanks, nPos1, bNoApproachNotDiagCond);

  call KeepRoomCond(nFiles, nRanks, nPos1, nPos2, bKeepRoomCond);
  bKeepRoomDiagCond = bKeepRoomCond && bKingDiag;
  bKeepRoomNotDiagCond = bKeepRoomCond && !bKingDiag;
  call NoKeepRoomDiag(nFiles, nRanks, nPos1, bNoKeepRoomDiagCond);
  call NoKeepRoomNotDiag(nFiles, nRanks, nPos1, bNoKeepRoomNotDiagCond);

  call RookHomeCond(nFiles, nRanks, nPos1, nPos2, bRookHomeCond);
  call NoRookHomeOpt(nFiles, nRanks, nPos1, bNoRookHomeCond);

  call RookSafeCond(nFiles, nRanks, nPos1, nPos2, bRookSafeCond);
  call NoRookSafeOpt(nFiles, nRanks, nPos1, bNoRookSafeCond);

  call RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos2, bRookSafeSBCond);
  
  b = ite(nStep == nsImmediateMate, 
         bImmediateMateCond,
         bNoImmediateMateCond &&
         ite(nStep == nsReadyToMate, 
            bReadyToMateCond,
            bNoReadyToMateCond && 
            ite(nStep == nsSqueeze, 
               bSqueezeCond,
               bNoSqueezeCond &&
               ite(nStep == nsApproachDiag, 
                  bApproachDiagCond,
                  bNoApproachDiagCond && 
                  ite(nStep == nsApproachNotDiag, 
                     bApproachNotDiagCond,
                     bNoApproachNotDiagCond && 
                     ite(nStep == nsKeepRoomDiag,
                        bKeepRoomDiagCond,
                        bNoKeepRoomDiagCond && 
                        ite(nStep == nsKeepRoomNotDiag,
                           bKeepRoomNotDiagCond,
                           bNoKeepRoomNotDiagCond && 
                              ite(nStep == nsRookHome, 
                                 bRookHomeCond,
                                 bNoRookHomeCond && 
                                 ite(nStep == nsRookSafe, 
                                    bRookSafeCond,
				    bNoRookSafeCond &&
				    ite(nStep == nsRookSafeSB,
				      bRookSafeSBCond,
	false))))))))));
  
}

/* -------------------------------------------------------------------------- */

procedure NextRookMove(nFiles, nRanks, nPos1,nI,nPos2,b)
{
   b1 = (nI>=1) && (nI<=(nFiles+nRanks));

   nPos2n=nPos1;
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
   nWKx2 = nWKx1; 
   nWKy2 = nWKy1;
   nWRx2 = nWRx1;
   nWRy2 = nWRy1;

   nWRx2 = ite(nI>=1 && nI<=nFiles && nWRx1!=nI-1, nI-1, nWRx2);
   nWRy2 = ite(nI>=nFiles+1 && nI<=(nFiles+nRanks) && nWRy1!=nI-(nFiles+1), nI-(nFiles+1), nWRy2);

   nBKx2 = nBKx1;
   nBKy2 = nBKy1;
   bWhiteOnTurn2 = !bWhiteOnTurn1;

   b2=false;
   b2 ||= (nI>=1 && nI<=nFiles && nWRx1!=nI-1);
   b2 ||= (nI>=nFiles+1 && nI<=(nFiles+nRanks) && nWRy1!=nI-(nFiles+1));

   b3 = bWhiteOnTurn1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2n);    
   nPos2=nPos2n;

   b = b1 && b2 && b3 && (nPos2n==nPos2);
}

/* -------------------------------------------------------------------------- */

procedure NextKingMove(nFiles, nRanks, nPos1,nI,nPos2,b)
{
   b1 = (nI>=1) && (nI<=8);

   nPos2n=nPos1;
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);

   nWKx2 = nWKx1; 
   nWKx2 = ite((nI==1 || nI==4 || nI==6) && nWKx1>0, nWKx1-1, nWKx2);
   nWKx2 = ite((nI==3 || nI==5 || nI==8) && nWKx1<nFiles-1, nWKx1+1, nWKx2);
   nWKy2 = nWKy1;
   nWKy2 = ite((nI==1 || nI==2 || nI==3) && nWKy1<nRanks-1, nWKy1+1, nWKy2);
   nWKy2 = ite((nI==6 || nI==7 || nI==8) && nWKy1>0, nWKy1-1, nWKy2);

   nWRx2 = nWRx1;
   nWRy2 = nWRy1;
   nBKx2 = nBKx1;
   nBKy2 = nBKy1;
   bWhiteOnTurn2 = !bWhiteOnTurn1;

   b2=false;
   b2 ||= (nI==1 && nWKx1>0 && nWKy1<nRanks-1);
   b2 ||= (nI==2 && nWKy1<nRanks-1);
   b2 ||= (nI==3 && nWKx1<nFiles-1 && nWKy1<nRanks-1);
   b2 ||= (nI==4 && nWKx1>0);
   b2 ||= (nI==5 && nWKx1<nFiles-1);
   b2 ||= (nI==6 && nWKx1>0 && nWKy1>0);
   b2 ||= (nI==7 && nWKy1>0);
   b2 ||= (nI==8 && nWKx1<nFiles-1 && nWKy1>0);

   b3 = bWhiteOnTurn1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2n);    
   nPos2=nPos2n;

   b = b1 && b2 && b3 && (nPos2n==nPos2);
}

/* -------------------------------------------------------------------------- */

procedure NextMove(nFiles, nRanks, nPos1,nI,nPos2,b)
{
   b1 = (nI>=1) && (nI<=8+(nFiles+nRanks));

   nPos2n=nPos1;
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);

   nWKx2 = nWKx1; 
   nWKx2 = ite((nI==1 || nI==4 || nI==6) && nWKx1>0, nWKx1-1, nWKx2);
   nWKx2 = ite((nI==3 || nI==5 || nI==8) && nWKx1<nFiles-1, nWKx1+1, nWKx2);
   nWKy2 = nWKy1;
   nWKy2 = ite((nI==1 || nI==2 || nI==3) && nWKy1<nRanks-1, nWKy1+1, nWKy2);
   nWKy2 = ite((nI==6 || nI==7 || nI==8) && nWKy1>0, nWKy1-1, nWKy2);

   nWRx2 = nWRx1;
   nWRy2 = nWRy1;

   nWRx2 = ite(nI>=9 && nI<=8+nFiles && nWRx1!=nI-9, nI-9, nWRx2);
   nWRy2 = ite(nI>=9+nFiles && nI<=8+nFiles+nRanks && nWRy1!=nI-(9+nFiles), nI-(9+nFiles), nWRy2);

   nBKx2 = nBKx1;
   nBKy2 = nBKy1;
   bWhiteOnTurn2 = !bWhiteOnTurn1;

   b2=false;
   b2 ||= (nI==1 && nWKx1>0 && nWKy1<nRanks-1);
   b2 ||= (nI==2 && nWKy1<nRanks-1);
   b2 ||= (nI==3 && nWKx1<nFiles-1 && nWKy1<nRanks-1);
   b2 ||= (nI==4 && nWKx1>0);
   b2 ||= (nI==5 && nWKx1<nFiles-1);
   b2 ||= (nI==6 && nWKx1>0 && nWKy1>0);
   b2 ||= (nI==7 && nWKy1>0);
   b2 ||= (nI==8 && nWKx1<nFiles-1 && nWKy1>0);

   b2 ||= (nI>=9 && nI<=8+nFiles && nWRx1!=nI-9);
   b2 ||= (nI>=9+nFiles && nI<=8+nFiles+nRanks && nWRy1!=nI-(9+nFiles));

   b3 = bWhiteOnTurn1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2n);    
   nPos2=nPos2n;

   b = b1 && b2 && b3 && (nPos2n==nPos2);
}

/* -------------------------------------------------------------------------- */

procedure IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos2)
{
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
   nWKx2 = nWKx1; nWKy2 = nWKy1;
   nWRx2 = nWRx1; nWRy2 = nWRy1;
   nBKx2 = nBKx1; nBKy2 = nBKy1;

   nWRx2 = ite(nMove >= 1 && nMove <= nFiles && nWRx1 != nMove-1, nMove-1, nWRx2);
   nWRy2 = ite(nMove >= nFiles+1 && nMove <= nFiles+nRanks && nWRy1 != nMove-(nFiles+1), nMove-(nFiles+1), nWRy2);
   bWhiteOnTurn2 = !bWhiteOnTurn1;
   bWRCaptured2 = bWRCaptured1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2);    
}

/* -------------------------------------------------------------------------- */

procedure IndexedKingMove(nFiles, nRanks, nPos1, nI, nPos2)
{
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);

   nWKx2 = nWKx1; 
   nWKx2 = ite((nI==1 || nI==4 || nI==6) && nWKx1>0, nWKx1-1, nWKx2);
   nWKx2 = ite((nI==3 || nI==5 || nI==8) && nWKx1<nFiles-1, nWKx1+1, nWKx2);
   nWKy2 = nWKy1;
   nWKy2 = ite((nI==1 || nI==2 || nI==3) && nWKy1<nRanks-1, nWKy1+1, nWKy2);
   nWKy2 = ite((nI==6 || nI==7 || nI==8) && nWKy1>0, nWKy1-1, nWKy2);

   nWRx2 = nWRx1;
   nWRy2 = nWRy1;
   nBKx2 = nBKx1;
   nBKy2 = nBKy1;
   bWhiteOnTurn2 = !bWhiteOnTurn1;
   bWRCaptured2 = bWRCaptured1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2);    
}

/* -------------------------------------------------------------------------- */

procedure IndexedMove(nFiles, nRanks, nPos1, nI, nPos2)
{
   call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);

   nWKx2 = nWKx1; 
   nWKx2 = ite((nI==1 || nI==4 || nI==6) && nWKx1>0, nWKx1-1, nWKx2);
   nWKx2 = ite((nI==3 || nI==5 || nI==8) && nWKx1<nFiles-1, nWKx1+1, nWKx2);
   nWKy2 = nWKy1;
   nWKy2 = ite((nI==1 || nI==2 || nI==3) && nWKy1<nRanks-1, nWKy1+1, nWKy2);
   nWKy2 = ite((nI==6 || nI==7 || nI==8) && nWKy1>0, nWKy1-1, nWKy2);

   nWRx2 = nWRx1;
   nWRy2 = nWRy1;

   nWRx2 = ite(nI>=9 && nI<=8+nFiles && nWRx1!=nI-9, nI-9, nWRx2);
   nWRy2 = ite(nI>=9+nFiles && nI<=8+nFiles+nRanks && nWRy1!=nI-(9+nFiles), nI-(9+nFiles), nWRy2);

   nBKx2 = nBKx1;
   nBKy2 = nBKy1;
   bWhiteOnTurn2 = !bWhiteOnTurn1;
   bWRCaptured2 = bWRCaptured1;

   call Pos2Bitvector(nFiles, nRanks, nWKx2, nWKy2, nBKx2, nBKy2, nWRx2, nWRy2, bWhiteOnTurn2, bWRCaptured2, nPos2);    
}

/* -------------------------------------------------------------------------- */

procedure StrategyFunction(nFiles, nRanks, nPos1, nPos2, nStep)
{

  b = true; 
  nStep=0;

  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ImmediateMateCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 1, nStep);
  }
  for(nMove=1;nMove<=8+nFiles+nRanks;nMove++) {
    call IndexedMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ReadyToMateCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 2, nStep);
  }

/*  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call SqueezeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 3, nStep);
  }*/
  nRoomMax = 0;
  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call SqueezeCond(nFiles, nRanks, nPos1, nPos0, b2);
    call Room(nFiles, nRanks, nPos1, nRoom);
    nPos2 = ite(b2 && (nStep==0 || (nStep==3 && nRoom>nRoomMax)), nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 3, nStep);
    nRoomMax = ite(b2 && (nStep==0 || (nStep==3 && nRoom>nRoomMax)), nRoom, nRoomMax);
  }

  for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ApproachDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 4, nStep);
  }
  for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ApproachNotDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 5, nStep);
  }
  for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call KeepRoomDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 6, nStep);
  }
  for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call KeepRoomNotDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 7, nStep);
  }
  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookHomeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 8, nStep);
  }
  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookSafeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 9, nStep);
  }
  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 10, nStep);
  }
}


/* -------------------------------------------------------------------------- */

procedure StrategyFunctionOptimized(nFiles, nRanks, nPos1, nPos2, nStep)
{
  b = true; 
  nStep=0;

  call Bitvector2Pos(nFiles, nRanks, nPos1, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
  bBlackOnTurn = false;

  /* *************** ImmediateMateCond ***************** */
/*  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ImmediateMateCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 1, nStep);
  }*/
  nx[0] = 0; ny[0] = nWRy;
  nx[1] = nFiles - 1; ny[1] = nWRy;
  nx[2] = nWRx; ny[2] = 0;
  nx[3] = nWRx; ny[3] = nRanks - 1;
  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
     call MateOptSym(nFiles, nRanks, nPos0, bCond);
     b2 = (bWRCanMoveTo && bCond);
     nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
     nStep = ite(b2 && nStep==0, 1, nStep);
  }

  /* *************** ReadyToMateCond ***************** */
/*  for(nMove=1;nMove<=8+nFiles+nRanks;nMove++) {
    call IndexedMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ReadyToMateCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 2, nStep);
  }*/

   nx[0] = 0; ny[0] = nWRy;
   nx[1] = nFiles - 1; ny[1] = nWRy;
   nx[2] = nWRx; ny[2] = 0;
   nx[3] = nWRx; ny[3] = nRanks - 1;
   nx[4] = 2; ny[4] = nWRy;
   nx[5] = nFiles - 3; ny[5] = nWRy;
   nx[6] = nWRx; ny[6] = 2;
   nx[7] = nWRx; ny[7] = nRanks - 3;
   nx[8] = nBKx - 1; ny[8] = nWRy;
   nx[9] = nBKx + 1; ny[9] = nWRy;
   nx[10] = nWRx; ny[10] = nBKy - 1;
   nx[11] = nWRx; ny[11] = nBKy + 1;
   for (ni = 0; ni < 12; ni++) {
      nxe = nx[ni]; nye = ny[ni];
      call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
      call ReadyToMateCondSym(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWRCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 2, nStep);
   }

   for (ni=1; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos1, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos0);
      call ReadyToMateCondSym(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWKCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 2, nStep);
   }

  /* *************** SqueezeCond ***************** */
  /*for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call SqueezeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 3, nStep);
  }*/
   nx[0] = nWRx; ny[0] = (nWKy + nBKy) >> 1;
   nx[1] = nWRx; ny[1] = (nWKy + nBKy + 1) >> 1;
   nx[2] = (nWKx + nBKx) >> 1; ny[2] = nWRy;
   nx[3] = (nWKx + nBKx + 1) >> 1; ny[3] = nWRy;
   nx[4] = nWRx; ny[4] = nBKy - nWKx + nWRx;
   nx[5] = nWKx - nBKy + nWRy; ny[5] = nWRy;
   nx[6] = nWRx; ny[6] = nWKx + nBKy - nWRx;
   nx[7] = nWKx + nBKy - nWRy; ny[7] = nWRy;
   nx[8] = nWRx; ny[8] = nWKy - nBKx + nWRx;
   nx[9] = nBKx - nWKy + nWRy; ny[9] = nWRy;
   nx[10] = nWRx; ny[10] = nBKx + nWKy - nWRx;
   nx[11] = nBKx + nWKy - nWRy; ny[11] = nWRy;
   nx[12] = nBKx + 1; ny[12] =  nWRy;
   nx[13] = nBKx - 1; ny[13] = nWRy;
   nx[14] = nWRx; ny[14] = nBKy + 1;
   nx[15] = nWRx; ny[15] = nBKy - 1;
   for (ni = 0; ni < 16; ni++) {
      nxe = nx[ni]; nye = ny[ni];
      call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
      call SqueezeCond(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWRCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 3, nStep);
   }

  /* *************** ApproachDiagCond ***************** */
/*  for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ApproachDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 4, nStep);
  }*/
   for (ni=1; ni<=4; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos1, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos0);
      call ApproachCond(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWKCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 4, nStep);

   }

  /* *************** ApproachNotDiagCond ***************** */
  /*for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call ApproachNotDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 5, nStep);
  }*/
   for (ni=5; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos1, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos0);
      call ApproachCond(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWKCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 5, nStep);
   }

  /* *************** KeepRoomDiagCond ***************** */
  /*for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call KeepRoomDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 6, nStep);
  }*/
   for (ni=1; ni<=4; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos1, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos0);
      call KeepRoomCond(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWKCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 6, nStep);
   }

  /* *************** KeepRoomNotDiagCond ***************** */
  /*for(nMove=1;nMove<=8;nMove++) {
    call IndexedKingMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call KeepRoomNotDiagCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 7, nStep);
  }*/
   for (ni=5; ni<=8; ni++) {
      call KingsSquare(nWKx, nWKy, ni, nx, ny); 
      call WKCanMoveTo(nFiles, nRanks, nPos1, nx, ny, bWKCanMoveTo);
      call Pos2Bitvector(nFiles, nRanks, nx, ny, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bBlackOnTurn, nPos0);
      call KeepRoomCond(nFiles, nRanks, nPos1, nPos0, bCond);
      b2 = (bWKCanMoveTo && bCond);
      nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
      nStep = ite(b2 && nStep==0, 7, nStep);
   }

  /* *************** RookHomeCond ***************** */
/*  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookHomeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 8, nStep);
  }*/
  nx[0] = nWRx; ny[0] = nWKy-1;
  nx[1] = nWRx; ny[1] = nWKy+1;
  nx[2] = nWKx-1; ny[2] = nWRy;
  nx[3] = nWKx+1; ny[3] = nWRy;
  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
     call RookHomeCond(nFiles, nRanks, nPos1, nPos0, bCond);
     b2 = (bWRCanMoveTo && bCond);
     nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
     nStep = ite(b2 && nStep==0, 8, nStep);
  }

  /* *************** RookSafeCond ***************** */
/*  for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookSafeCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 9, nStep);
  }*/
  nx[0] = 0; ny[0] = nWRy;
  nx[1] = nFiles - 1; ny[1] = nWRy;
  nx[2] = nWRx; ny[2] = 0;
  nx[3] = nWRx; ny[3] = nRanks - 1;
  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
     call RookSafeCond(nFiles, nRanks, nPos1, nPos0, bCond);
     b2 = (bWRCanMoveTo && bCond);
     nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
     nStep = ite(b2 && nStep==0, 9, nStep);
  }

  /* *************** RookSafeSmallBoardsCond ***************** */
  /*for(nMove=1;nMove<=nFiles+nRanks;nMove++) {
    call IndexedRookMove(nFiles, nRanks, nPos1, nMove, nPos0);
    call RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos0, b2);
    nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
    nStep = ite(b2 && nStep==0, 10, nStep);
  }*/
  nx[0] = 0; ny[0] = nWRy;
  nx[1] = nFiles - 1; ny[1] = nWRy;
  nx[2] = nWRx; ny[2] = 0;
  nx[3] = nWRx; ny[3] = nRanks - 1;
  for (ni = 0; ni < 4; ni++) {
     nxe = nx[ni]; nye = ny[ni];
     call WRCanMoveTo(nFiles, nRanks, nPos1, nxe, nye, bWRCanMoveTo);
     call Pos2Bitvector(nFiles, nRanks, nWKx, nWKy, nBKx, nBKy, nxe, nye, bWRCaptured, bBlackOnTurn, nPos0);
     call RookSafeSmallBoardsCond(nFiles, nRanks, nPos1, nPos0, bCond);
     b2 = (bWRCanMoveTo && bCond);
     nPos2 = ite(b2 && nStep==0, nPos0, nPos2);
     nStep = ite(b2 && nStep==0, 10, nStep);
  }
}


/* -------------------------------------------------------------------------- */

procedure IsBasicMove(nMove, bBasic) {
  nsImmediateMate  =1; 
  nsReadyToMate    =2; 
  nsSqueeze        =3; 
  nsApproachDiag   =4; 
  nsApproachNotDiag=5; 
  nsKeepRoomDiag   =6; 
  nsKeepRoomNotDiag=7; 
  nsRookHome       =8; 
  nsRookSafe       =9; 
  bBasic = nMove == nsSqueeze || nMove == nsApproachDiag || nMove == nsApproachNotDiag || nMove == nsKeepRoomDiag || nMove == nsKeepRoomNotDiag;
}

procedure BasicMoves(nStep,bBasicMoves)
{
        nsSqueeze = 3;
        nsApproachDiag = 4;
        nsApproachNotDiag = 5;
        nsKeepRoomDiag = 6;
        nsKeepRoomNotDiag = 7;
        bBasicMoves = (nStep==nsSqueeze || nStep==nsApproachDiag || nStep==nsApproachNotDiag || nStep==nsKeepRoomDiag || nStep==nsKeepRoomNotDiag);
}

procedure MeasureDecreases(nFiles, nRanks, nPos1,nPos2,bMeasureDecreases) {
  call Room(nFiles, nRanks, nPos1,nRoom1);
  call Room(nFiles, nRanks, nPos2,nRoom2);
  call CriticalSquare(nFiles, nRanks, nPos1,nCSx1,nCSy1);
  call CriticalSquare(nFiles, nRanks, nPos2,nCSx2,nCSy2);
  call Bitvector2Pos(nFiles, nRanks, nPos1,nWKx1,nWKy1,nBKx1,nBKy1,nWRx1,nWRy1,bWRCaptured1,bWhiteOnTurn1);
  call Bitvector2Pos(nFiles, nRanks, nPos2,nWKx2,nWKy2,nBKx2,nBKy2,nWRx2,nWRy2,bWRCaptured2,bWhiteOnTurn2);
  call ManhattanDistance(nWKx1,nWKy1,nCSx1,nCSy1,nMDWKCS1);
  call ManhattanDistance(nWKx2,nWKy2,nCSx2,nCSy2,nMDWKCS2);
  call lexLe2(nRoom2,nMDWKCS2,nRoom1,nMDWKCS1,bLexOrderingRoomMD);
  bMeasureDecreases = bLexOrderingRoomMD;
}

/* ---------------------------------------------------------------------------- 

Broj poteza relacije
--------------------

                  4x4         5x5            6x6        7x7          8x8    
total:            1312        7040          25440                    175168     
ImmediateMate:   120 120    288   288    560    560    960    960   1512   1512  
ReadyToMate:     404 544    968  1344   1836   2616   3056   4456   4676   6960   
Squeeze:         436 560   3660  6288  15376  31576  46300 113016 116504 325168
ApproachDiag:     72  72    576   576   2124   2124   5524   5524  12160  12160
ApproachNotDiag: 124 136    472   536   1140   1280   2320   2560   4020   4384
KeepRoomDiag:     24  24    136   136    520    520   1448   1448   3160   3160
KeepRoomNotDiag:  24  24     64    64     88     88    152    152    184    184
RookHome:         60  80    768   976   3604   5064  12320  18232  32520  50792
RookSafe:         28  48     96   152    192    376    300    696    432   1112
RookSafeSB:       20  24     12    16      0      0      0      0      0      0

---------------------------------------------------------------------------- */
nFiles = 8;
nRanks = 8;

nsImmediateMate   = 1; 
nsReadyToMate     = 2; 
nsSqueeze         = 3; 
nsApproachDiag    = 4; 
nsApproachNotDiag = 5; 
nsKeepRoomDiag    = 6; 
nsKeepRoomNotDiag = 7; 
nsRookHome        = 8; 
nsRookSafe        = 9; 
nsRookSafeSB      = 10;

/*
nPos=n0;
call IsCanon(nFiles, nRanks, nPos, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos,bLegal);
call IsWRCaptured(nFiles,nRanks,nPos,bWRCaptured);
call NoSqueeze(nFiles,nRanks,nPos,bNoSqueeze);
call NoApproachDiag(nFiles,nRanks,nPos,bNoApproachDiag);
call NoApproachNotDiag(nFiles,nRanks,nPos,bNoApproachNotDiag);
call NoKeepRoomDiag(nFiles,nRanks,nPos,bNoKeepRoomDiag);
call NoKeepRoomNotDiag(nFiles,nRanks,nPos,bNoKeepRoomNotDiag);
call NoRookHome(nFiles,nRanks,nPos,bNoRookHome);
call NoRookSafe(nFiles,nRanks,nPos,bNoRookSafe);
call NoRookSafeSmallBoards(nFiles,nRanks,nPos,bNoRookSafeSB);
bNoMove = bLegal && !bWRCaptured && bCanon &&
  bNoSqueeze && bNoApproachDiag && bNoApproachNotDiag && bNoKeepRoomDiag && bNoKeepRoomNotDiag &&
  bNoRookHome && bNoRookSafe && bNoRookSafeSB;
assert_all(bNoMove);
*/


/* Count moves */
nPos0 = n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0, bCaptured);
call StrategyFunction(nFiles, nRanks, nPos0, nPos1, ns);
assert_all(bLegal && !bCaptured && ns == nsReadyToMate);

/*nPos0 = n0; nPos1=n1; nsStep = ns;
call IsCanon(nFiles, nRanks, nPos0, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0, bCaptured);
call StrategyRelationOpt(nFiles, nRanks, nPos0, nPos1, bRel, nsRookSafe);
assert_all(bLegal && !bCaptured && bRel);*/


/* LEMMAS: */

/* RookSafe can only be played in a first move */
/*
nPos0w=n0w; nPos0b=n0b; nPos1w=n1w; nPos1b=n1b; nStep0w = ns0; nStep1w = ns1;
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nStep0w);
call LegalMoveBlack(nFiles, nRanks, nPos0b, nPos1w, bBlack0);
call StrategyRelation(nFiles, nRanks, nPos1w, nPos1b, bRel1, nStep1w);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && bBlack0 && bRel1 &&
	   nStep1w == nsRookSafeSB);
*/

/* RookHome can be played only after a RookHome or a RookSafe move */
/*
nPos0w=n0w; nPos0b=n0b; nPos1w=n1w; nPos1b=n1b; nPos2w=n2w; nPos2b=n2b;
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nStep0w);
call LegalMoveBlack(nFiles, nRanks, nPos0b, nPos1w, bBlack0);
call StrategyRelation(nFiles, nRanks, nPos1w, nPos1b, bRel1, nStep1w);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && bBlack0 && bRel1 &&
	   (nStep0w != nsRookSafeSB && nStep0w != nsRookSafe && nStep0w != nsRookHome) && nStep1w == nsRookHome);
*/

/* After two RookHome moves, RookHome cannot be played */
/*
nPos0w=n0w; nPos0b=n0b; nPos1w=n1w; nPos1b=n1b; nPos2w=n2w; nPos2b=n2b;
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nStep0w);
call LegalMoveBlack(nFiles, nRanks, nPos0b, nPos1w, bBlack0);
call StrategyRelation(nFiles, nRanks, nPos1w, nPos1b, bRel1, nStep1w);
call LegalMoveBlack(nFiles, nRanks, nPos1b, nPos2w, bBlack1);
call StrategyRelation(nFiles, nRanks, nPos2w, nPos2b, bRel2, nStep2w);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && bBlack0 && bRel1 && bBlack1 && bRel2 &&
	   nStep0w == nsRookHome && nstep1w == nsRookHome && nStep2w == nsRookHome);
*/



/* NoRookSafe is equvivalent to NoRookSafeOpt */
/*
nPos=n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call NoRookSafe(nFiles, nRanks, nPos, bNo);
call NoRookSafeOpt(nFiles, nRanks, nPos, bNoOpt);
assert(bLegal && !bCaptured && (bNo ^^ bNoOpt));
*/

/* NoRookHome is equvivalent to NoRookHomeOpt */
/*
nPos=n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call NoRookHome(nFiles, nRanks, nPos, bNo);
call NoRookHomeOpt(nFiles, nRanks, nPos, bNoOpt);
assert(bLegal && !bCaptured && (bNo ^^ bNoOpt));
*/


/* NoReadyToMateWR is equvivalent to NoReadyToMateWROpt */
/*
nPos=n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call NoReadyToMateWR(nFiles, nRanks, nPos, bNo);
call NoReadyToMateWROpt(nFiles, nRanks, nPos, bNoOpt);
assert(bLegal && !bCaptured && (bNo ^^ bNoOpt));
*/


/* NoImmediateMate is equvivalent to NoImmediateMateOpt */
/*
nPos=n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call NoImmediateMate(nFiles, nRanks, nPos, bNo);
call NoImmediateMateOpt(nFiles, nRanks, nPos, bNoOpt);
assert(bLegal && !bCaptured && (bNo ^^ bNoOpt));
*/

/* ReadyToMateCond is equvivalent to ReadyToMateCondSym */
/*
nPos=n0;
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call ReadyToMateCond(nFiles, nRanks, nPos, nPos1, bCond);
call ReadyToMateCondSym(nFiles, nRanks, nPos, nPos1, bCondSym);
assert(bLegal && !bCaptured && (bCond ^^ bCondSym));
*/

/* Stalemate is equvivalent to StalemateOpt */
/*
nPos=n0;
call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call Stalemate(nFiles, nRanks, nPos, bMate);
call StalemateOpt(nFiles, nRanks, nPos, bMateOpt);
assert(bLegal && !bCaptured && (bMate ^^ bMateOpt));
*/


/* StalemateOpt is equvivalent to StalemateOptSym */
/*
nPos=n0;
call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call StalemateOptSym(nFiles, nRanks, nPos, bMateOptSym);
call StalemateOpt(nFiles, nRanks, nPos, bMateOpt);
assert(bLegal && !bCaptured && (bMateOptSym ^^ bMateOpt));
*/


/* MateOpt is equvivalent to MateOptSym */
/* 
nPos=n0;
call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call MateOptSym(nFiles, nRanks, nPos, bMateOptSym);
call MateOpt(nFiles, nRanks, nPos, bMateOpt);
assert(bLegal && !bCaptured && (bMateOptSym ^^ bMateOpt));
*/

/* Mate is equvivalent to MateOpt */
/* 
nPos=n0;
call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call Mate(nFiles, nRanks, nPos, bMate);
call MateOpt(nFiles, nRanks, nPos, bMateOpt);
assert(bLegal && !bCaptured && (bMate ^^ bMateOpt));
*/


/* After a strategy move by white, black cannot capture the white rook. */
/*
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call LegalMoveBlack(nFiles,nRanks,nPos2,nPos3,b2);
call IsWRCaptured(nFiles,nRanks,nPos3,bRookIsCaptured3);
bRookIsNotCaptured = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && b1 && b2 && bRookIsCaptured3);
assert_all(bRookIsNotCaptured);
*/

/* After an ImmediateMate move, the black is checkmated. */
/*
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call MateOptSym(nFiles,nRanks,nPos2,bMate2);
bAfterImmediateMateBlackIsCheckmated = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && b1 && nStep1==nsImmediateMate && !bMate2);
assert_all(bAfterImmediateMateBlackIsCheckmated);
*/

/* ReadyToMate move leads to checkmate in the next move. */
/*
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call LegalMoveBlack(nFiles,nRanks,nPos2,nPos3,b2);
call StrategyRelationOpt(nFiles,nRanks,nPos3,nPos4,b3,nStep3);
call MateOptSym(nFiles,nRanks,nPos4,bMate4);
bNotMateAfterReadyToMate = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && b1 && nStep1==nsReadyToMate && b2 && b3 && !bMate4);
assert_all(bNotMateAfterReadyToMate);
*/

/* RookHome and RookSafe moves are played only in the first three moves. */
/*
nPos1 = n1; nPos2 = n2; nPos3 = n3; nPos4 = n4; nPos5 = n5; nPos6 = n6; nPos7 = n7; nPos8 = n8; nStep1=ns1; nStep3=ns3; nStep5=ns5; nStep7=ns7;
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call LegalMoveBlack(nFiles,nRanks,nPos2,nPos3,b2);
call StrategyRelationOpt(nFiles,nRanks,nPos3,nPos4,b3,nStep3);
call LegalMoveBlack(nFiles,nRanks,nPos4,nPos5,b4);
call StrategyRelationOpt(nFiles,nRanks,nPos5,nPos6,b5,nStep5);
call LegalMoveBlack(nFiles,nRanks,nPos6,nPos7,b6);
call StrategyRelationOpt(nFiles,nRanks,nPos7,nPos8,b7,nStep7);
bRookHomeOrRookSafeAfterThreeMoves = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && b1 && b2 && b3 && b4 && b5 && b6 && b7 && (nStep7==nsRookHome || nStep7==nsRookSafe || nStep7==nsRookSafeSB));
assert_all(bRookHomeOrRookSafeAfterThreeMoves);
*/

/* Starting from a position with a room greater then 3, playing three full moves where the white performs a basic strategy move (Squeeze, Approach or KeepRoom) reduces the room or leaves room the same, but decreases the Manhattan distance to the critical square. */
/*
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call Room(nFiles,nRanks,nPos1,nRoom);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call LegalMoveBlack(nFiles,nRanks,nPos2,nPos3,b2);
call StrategyRelationOpt(nFiles,nRanks,nPos3,nPos4,b3,nStep3);
call LegalMoveBlack(nFiles,nRanks,nPos4,nPos5,b4);
call StrategyRelationOpt(nFiles,nRanks,nPos5,nPos6,b5,nStep5);
call LegalMoveBlack(nFiles,nRanks,nPos6,nPos7,b6);
call BasicMoves(nStep1,bBasicMoves1);
call BasicMoves(nStep3,bBasicMoves3);
call BasicMoves(nStep5,bBasicMoves5);
call MeasureDecreases(nFiles,nRanks,nPos1,nPos7,bMeasureDecreases);
bTerminationLemma2 = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && nRoom > 3 && b1 && b2 && b3 && b4 && b5 && b6 && bBasicMoves1 && bBasicMoves3 && bBasicMoves5 && !bMeasureDecreases);
assert(bTerminationLemma2);
*/


/* When the room is less or equal to 3, after a three full moves where the white performs a basic strategy move, the next move must be a mating move (ReadyToMove or ImmediateMate). */
/*
call IsCanon(nFiles, nRanks, nPos1, bCanon);
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegalPositionWhiteToMove1);
call IsWRCaptured(nFiles,nRanks,nPos1,bRookIsCaptured1);
call nRoom(nFiles,nRanks,nPos1,nRoom);
call StrategyRelationOpt(nFiles,nRanks,nPos1,nPos2,b1,nStep1);
call LegalMoveBlack(nFiles,nRanks,nPos2,nPos3,b2);
call StrategyRelationOpt(nFiles,nRanks,nPos3,nPos4,b3,nStep3);
call LegalMoveBlack(nFiles,nRanks,nPos4,nPos5,b4);
call StrategyRelationOpt(nFiles,nRanks,nPos5,nPos6,b5,nStep5);
call LegalMoveBlack(nFiles,nRanks,nPos6,nPos7,b6);
call StrategyRelationOpt(nFiles,nRanks,nPos7,nPos8,b7,nStep7);
call BasicMoves(nStep1,bBasicMoves1);
call BasicMoves(nStep3,bBasicMoves3);
call BasicMoves(nStep5,bBasicMoves5);
bMateRoomGe3 = bCanon && (bLegalPositionWhiteToMove1 && !bRookIsCaptured1 && !nRoom > 3 && b1 && b2 && b3 && b4 && b5 && b6 && b7 && bBasicMoves1 && bBasicMoves3 && bBasicMoves5 && nStep7!=nsImmediateMate && nStep7!=nsReadyToMate);
assert_all(bMateRoomGe3);
*/

/* ************************************************************************** */
/* ********************** 5.2 Explicit invariants *************************** */
/* ************************************************************************** */

/* Exact precondition for RookSafe move */
/*
nPos0w=n0;
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nStep);
call WRDivides(nFiles, nRanks, nPos0w, bDivides);
call Bitvector2Pos(nFiles, nRanks, nPos0w, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
call ChebyshevDistance(nBKx, nBKy, nWRx, nWRy, nCDBKWR);
call ChebyshevDistance(nBKx, nBKy, nWKx, nWKy, nCDBKWK);
call AbsDiff(nWKx, nWRx, nDx);
call AbsDiff(nWKy, nWRy, nDy);
call Room(nFiles, nRanks, nPos0w, nRoom);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && nStep == nsRookSafe &&
	   ((nCDBKWR == 1 && nCDBKWK == 2 && !bDivides && nWRx != nWKx && nWRy != nWKy && nWKx != nBKx && nWKy != nBKy) ||
	    (nBKx == 0 && nBKx == 0 && nWKx == 0 && nWKy == 2 && nWRx < 2 && nWRy > 2) ||
	    (nRoom == 2 && nWKy == 2 && (nWKx == 0 || nWKx == 2))));
*/

/* No RookSafe precondition after a full move */
/*
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nStep);
call LegalMoveBK(nFiles, nRanks, nPos0b, nPos1w, bBlack0);
call WRDivides(nFiles, nRanks, nPos1w, bDivides);
call Bitvector2Pos(nFiles, nRanks, nPos1w, nWKx, nWKy, nBKx, nBKy, nWRx, nWRy, bWRCaptured, bWhiteOnTurn);
call ChebyshevDistance(nBKx, nBKy, nWRx, nWRy, nCDBKWR);
call ChebyshevDistance(nBKx, nBKy, nWKx, nWKy, nCDBKWK);
call AbsDiff(nWKx, nWRx, nDx);
call AbsDiff(nWKy, nWRy, nDy);
call Room(nFiles, nRanks, nPos1w, nRoom);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && bBlack0 &&
	   ((nCDBKWR == 1 && nCDBKWK == 2 && !bDivides && nWRx != nWKx && nWRy != nWKy && nWKx != nBKx && nWKy != nBKy) ||
	    (nBKx == 0 && nBKx == 0 && nWKx == 0 && nWKy == 2 && nWRx < 2 && nWRy > 2) ||
	    (nRoom == 2 && nWKy == 2 && (nWKx == 0 || nWKx == 2))));
*/


/* Two RookHome moves ensures that bDivides && !bExposed and !bCheck and  nRoom > 2 */
/*
call IsCanon(nFiles, nRanks, nPos0w, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos0w, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos0w, bCaptured);
call StrategyRelation(nFiles, nRanks, nPos0w, nPos0b, bRel0, nsRookHome);
call LegalMoveBlack(nFiles, nRanks, nPos0b, nPos1w, bBlack0);
call StrategyRelation(nFiles, nRanks, nPos1w, nPos1b, bRel1, nsRookHome);
call WRDivides(nFiles, nRanks, nPos1b, bDivides);
call LPattern(nFiles, nRanks, nPos1b, bLPattern);
call WRExposed(nFiles, nRanks, nPos1b, bExposed);
call Bitvector2Pos(nFiles, nRanks, nPos1b, nWKx1, nWKy1, nBKx1, nBKy1, nWRx1, nWRy1, bWRCaptured1, bWhiteOnTurn1);
call Room(nFiles, nRanks, nPos1b, nRoom);
call WRAttacksBK(nFiles, nRanks, nPos1b, bCheck);
assert_all(bCanon && bLegal && !bCaptured && bRel0 && bBlack0 && bRel1 && !((bDivides && !bExposed) && !bCheck && nRoom > 2));
*/

/* The condition (bDivides && !bExposed || bLPattern) && !bCheck && nRoom > 2 is
   preserved by a legal move of the black king */
/*
nPos=n0;nPos1=n1;
call IsCanon(nFiles, nRanks, nPos, bCanon);
call LegalPositionBlackToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call LegalMoveBlack(nFiles, nRanks, nPos, nPos1, bBlack);
call WRExposed(nFiles, nRanks, nPos, bExposed);
call WRDivides(nFiles, nRanks, nPos, bDivides);
call LPattern(nFiles, nRanks, nPos, bLPattern);
call Room(nFiles, nRanks, nPos, nRoom);
call WRAttacksBK(nFiles, nRanks, nPos, bCheck);
call WRExposed(nFiles, nRanks, nPos1, bExposed1);
call WRDivides(nFiles, nRanks, nPos1, bDivides1);
call LPatternBlack(nFiles, nRanks, nPos1, bLPattern1);
call Room(nFiles, nRanks, nPos1, nRoom1);
call WRAttacksBK(nFiles, nRanks, nPos, bCheck1);
assert_all(bCanon && bLegal && !bCaptured && bBlack &&
		  (!bExposed && (bDivides || bLPattern) && !bCheck && nRoom > 2) &&
		 !(!bExposed1 && (bDivides1 || bLPattern1) && !bCheck1 && nRoom1 > 2));
*/

/* The condition (!bExposed && bDivides || bLPattern) && !bCheck && nRoom > 2 is
   preserved by all no mate moves */
/*
call IsCanon(nFiles, nRanks, nPos, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call WRExposed(nFiles, nRanks, nPos, bExposed);
call WRDivides(nFiles, nRanks, nPos, bDivides);
call LPatternBlack(nFiles, nRanks, nPos, bLPattern);
call Room(nFiles, nRanks, nPos, nRoom);
call WRAttacksBK(nFiles, nRanks, nPos, bCheck);
call StrategyRelation(nFiles, nRanks, nPos, nPos1, bRel, nStep);
call WRExposed(nFiles, nRanks, nPos1, bExposed1);
call WRDivides(nFiles, nRanks, nPos1, bDivides1);
call LPattern(nFiles, nRanks, nPos1, bLPattern1);
call Room(nFiles, nRanks, nPos1, nRoom1);
call WRAttacksBK(nFiles, nRanks, nPos, bCheck1);
assert_all(bCanon && bLegal && !bCaptured && bRel && nStep != nsImmediateMate && nStep != nsReadyToMate &&
	   ((!bExposed && bDivides || bLPattern) && !bCheck && nRoom > 2) &&
	   !((!bExposed1 && bDivides1 || bLPattern1) && !bCheck1 && nRoom1 > 2));
*/

/* The condition !bExposed && (bDivides || bLPattern) && nRoom > 2 enables a basic or a mate move */
/*
nPos=n0;
call IsCanon(nFiles, nRanks, nPos, bCanon);
call LegalPositionWhiteToMove(nFiles, nRanks, nPos, bLegal);
call IsWRCaptured(nFiles, nRanks, nPos, bCaptured);
call WRExposed(nFiles, nRanks, nPos, bExposed);
call WRDivides(nFiles, nRanks, nPos, bDivides);
call LPatternBlack(nFiles, nRanks, nPos, bLPattern);
call Room(nFiles, nRanks, nPos, nRoom);
call StrategyRelation(nFiles, nRanks, nPos, nPos1, bRel, nStep);
call BasicMoves(nStep, bBasic);
assert_all(bCanon && bLegal && !bCaptured && bRel && nStep != nsImmediateMate && nStep != nsReadyToMate && !bBasic &&
	   (!bExposed && bDivides || bLPattern) && nRoom > 2);
*/

/*
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegal);
call IsWRCaptured(nFiles, nRanks, nPos1, bWRCaptured);
call StrategyFunction(nFiles, nRanks, nPos1, nPos2, nStep1);
call StrategyRelation(nFiles, nRanks, nPos1, nPos2, b, nStep1);
bb = bLegal && !bWRCaptured && !b && nStep1!=0;
assert_all(bb);*/


/*
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegal);
call IsWRCaptured(nFiles, nRanks, nPos1, bWRCaptured);
call StrategyFunction(nFiles, nRanks, nPos1, nPos2, nStep1);
call StrategyRelation(nFiles, nRanks, nPos1, nPos2, b, nStep1);
bb = bLegal && !bWRCaptured && !b && nStep1!=0;
assert_all(bb);
*/

/*
call LegalPositionWhiteToMove(nFiles,nRanks,nPos1,bLegal);
call IsWRCaptured(nFiles, nRanks, nPos1, bWRCaptured);
call StrategyFunction(nFiles, nRanks, nPos1, nPos2, nStep1);
call StrategyFunctionOptimized(nFiles, nRanks, nPos1, nPos2a, nStep1a);
bb = bLegal && !bWRCaptured && nStep1!=0 && nStep1!=nStep1a;
assert_all(bb);
*/


