#ifndef __BACKTRACKABLE_STACK_H__
#define __BACKTRACKABLE_STACK_H__

#include <vector>
#include <string>
#include <cassert>
#include <iostream>

#define UNDEFINED_LEVEL (unsigned)(-1)

std::string toString_(int t);
std::string toString_(unsigned t);
std::string toString_(const std::string& t);

template <class ELEMENT_TYPE>
std::string toString(const ELEMENT_TYPE& t) {
	return t->toString();
}

template <class ELEMENT_TYPE>
class BacktrackableStack {
 private:
    struct Element {
	Element(const ELEMENT_TYPE& element, size_t in_level) 
	    : _element(element), _in_level(in_level){
	}
	ELEMENT_TYPE _element;
	unsigned _in_level;
    };

    std::vector<Element> _elements;
    size_t _cachedSize;
    typename std::vector<Element>::iterator _cachedLast;
    const unsigned int& _currentLevel;
 public:
    BacktrackableStack(const unsigned int& currentLevel)
	: _currentLevel(currentLevel) {
	_cachedSize = 0;
    }

    BacktrackableStack(const BacktrackableStack& s) 
	: _elements(s._elements), _cachedSize(s._cachedSize), _currentLevel(s._currentLevel) {
	_cachedLast = _elements.end() - 1;
    }

    void push(const ELEMENT_TYPE& item) {
	_elements.push_back(Element(item, _currentLevel));
	_cachedLast = _elements.end() - 1;
	_cachedSize++;
	assert(empty() || _cachedLast == _elements.end()-1);
    }

    ELEMENT_TYPE pop() {
	assert(!empty());
        ELEMENT_TYPE lastElement = _cachedLast->element;
	_elements.pop_back();
	_cachedLast = _elements.end() - 1;
	_cachedSize--;
	assert(empty() || _cachedLast == _elements.end()-1);
	return lastElement;
    }

    bool lastLevelEmpty() const {
	return empty() || _cachedLast->_in_level < _currentLevel;
    }

    const ELEMENT_TYPE& peek() const {
	assert(!empty());
	assert(_cachedLast == _elements.end()-1);
	return _cachedLast->_element;
    }

    void newLevel() {
	assert(empty() || _cachedLast == _elements.end()-1);
//	_currentLevel++;
    }


    void backtrack() {
	assert(empty() || _cachedLast == _elements.end()-1);
	while (_cachedSize > 0 &&  _cachedLast->_in_level > _currentLevel) {
	    _elements.pop_back();
	    _cachedLast = _elements.end() - 1;
	    _cachedSize--;
	}

	assert(empty() || _cachedLast == _elements.end()-1);
    }
	
    void clear() {
	_elements.clear();
	_cachedSize = 0;
    }

    std::string toString() const {
	std::string result = "[ ";
	typename std::vector<Element>::const_iterator it =
	    _elements.begin();
	for (unsigned level = 0; level <= _currentLevel; level++) {
	    while (it != _elements.end() && it->_in_level == level) {
		result += toString_(it->_element);
		if (it->_out_level != UNDEFINED_LEVEL)
		    result += "*";
		result += " ";
		it++;
	    }
	    if (level < _currentLevel)
		result += "| ";
	}
	result += "]";
//	for (size_t i = 0; i < _elements.size(); i++)
//	    cout << _elements[i]._element << "." << _elements[i]._in_level << " ";
//	cout << endl;
	return result;
    }

    void deactivate(ELEMENT_TYPE element) {
	Element* pelement = findElement(element);
	if (pelement)
	    pelement->_out_level = _currentLevel;
    }

    class const_iterator {
    public:
	const_iterator() {}

	const_iterator(typename std::vector<Element>::const_iterator it, 
		       const std::vector<Element>* elements) 
	    : _it(it), _elements(elements) {
	}

	void operator++() {
	    _it++;
	}

	bool operator!= (const const_iterator& i) {
	    return !(*this == i);
	}

	bool operator== (const const_iterator& i) {
	    return _it == i._it;
	}

	const ELEMENT_TYPE& operator*() {
	    return _it->_element;
	}

    private:
	typename std::vector<Element>::const_iterator _it;
	const std::vector<Element>* _elements;
    };

    const_iterator begin() const {
	return const_iterator(_elements.begin(), &_elements);
    }

    const_iterator end() const {
	return const_iterator(_elements.end(), &_elements);
    }

    const_iterator beginLastLevel() const {
	if (empty())
	    return end();
	typename std::vector<Element>::const_iterator it; 
	for ( it = _elements.end()-1; it >= _elements.begin() && it->_in_level > _currentLevel; it--)
	    ;
	return const_iterator(it+1, &_elements);
    }

    const_iterator endLastLevel() const {
	return end();
    }

    bool empty() const {
	return _cachedSize == 0;
    }

    const_iterator find(ELEMENT_TYPE element) const {
	typename std::vector<Element>::const_iterator it;
	for (it = _elements.begin(); it != _elements.end(); it++)
	    if (it->_element == element)
		return const_iterator(it, _elements);
	return end();
    }

    bool contains(ELEMENT_TYPE element) const {
	const Element* e = findElement(element);
	return  e != 0;
    }

    size_t size() const {
	return _cachedSize;
    }

    const ELEMENT_TYPE& operator[](unsigned n) const {
	return _elements[n]._element;
    }

 private:
    Element* findElement(ELEMENT_TYPE element) {
	typename std::vector<Element>::iterator it;
	for (it = _elements.begin(); it != _elements.end(); it++)
	    if (it->_element == element)
		return &(*it);
	return 0;
    }

    const Element* findElement(ELEMENT_TYPE element) const {
	typename std::vector<Element>::const_iterator it;
	for (it = _elements.begin(); it != _elements.end(); it++) {
	    if (it->_element == element)
		return &(*it);
	}
	return 0;
    }
};
#endif

