#include <cassert>
#include <vector>
#include <iostream>
#include <sstream>

#include "expressions/Expression.h"
#include "parsing/ExpressionParser.h"
#include "expressions/ExpressionFactory.h"

#include "expressions/OperatorNode.h"

//Reading from a file
bool Expression::Read(std::istream& istr, Expression& expr)
{   
    std::vector<Expression> assumptions;
    ExpressionParser::yyin=&istr;
    bool eof = false;
    while(!eof) {
	int token = ExpressionParser::yylex();
	switch(token) {
	    case FORMULA: {
		if (ExpressionParser::yyparse()!=0) {   
		    ExpressionParser::yyin=NULL;
		    return false;
		}
		expr = *ExpressionParser::p_parsed_expression;
		delete ExpressionParser::p_parsed_expression;
		break;
	    }
	    case ASSUMPTION: {
		if (ExpressionParser::yyparse()!=0) {   
		    ExpressionParser::yyin=NULL;
		    return false;
		}
		Expression assumption = *ExpressionParser::p_parsed_expression;
		delete ExpressionParser::p_parsed_expression;
//		coutput << "Assumption: " << assumption << endl;
		assumptions.push_back(assumption);
		break;
	    }
	    case 0:
		eof = true;
	}
    }
    if (!assumptions.empty()) {
	expr = expr && Expression::AND(assumptions);
    }
    ExpressionParser::yyin=NULL;
    return !(expr.IsNULL());
}

//Reading from a string
bool Expression::Read(const std::string& str, Expression& expr)
{   
     std::istringstream istr(str.c_str());
     return  Read(istr, expr);
}

std::string Expression::toString() const {
	std::ostringstream str_stream;
	Output str_output(str_stream);
	str_output << *this;
	return str_stream.str();
}

//Expression construction functions
Expression Expression::Constant(const std::string name)
{
	 return Expression(new ExpressionNode(EXPR_CONSTANT,name));
}

Expression Expression::Numeral(const INT& value)
{
     return Expression(new IntNumeralNode(value));
}

Expression Expression::Numeral(const std::string name)
{
     return Expression(new IntNumeralNode(name));
}

Expression Expression::RationalNumeral(const RATIONAL& value)
{
     return Expression(new RationalNumeralNode(value));
}

Expression Expression::RationalNumeral(const std::string name)
{
     return Expression(new RationalNumeralNode(name));
}

Expression Expression::Variable(const std::string name)
{
     return Expression(new ExpressionNode(EXPR_VARIABLE,name));
}

Expression Expression::FormulaVariable(const std::string name)
{
     return Expression(new ExpressionNode(EXPR_FORMULA_VARIABLE,name));
}

Expression Expression::MetaVariable(const std::string name)
{
     return Expression(new ExpressionNode(EXPR_METAVARIABLE,name));
}

Expression Expression::Function(const std::string name, const Expression& arg1)
{
     std::vector<Expression> operands;
     operands.push_back(arg1);
     return Expression::Function(name, operands);
}

Expression Expression::Function(const std::string name, const Expression& arg1, const Expression& arg2)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     return Expression::Function(name,operands);
}

Expression Expression::Function(const std::string name, const Expression& arg1, const Expression& arg2, const Expression& arg3)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     operands.push_back(arg3);
     return Expression::Function(name,operands);
}

Expression Expression::Function(const std::string name, const Expression& arg1, const Expression& arg2, const Expression& arg3, const Expression& arg4)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     operands.push_back(arg3);
     operands.push_back(arg4);
     return Expression::Function(name,operands);
}

Expression Expression::Function(const std::string name, const std::vector<Expression>& operands)
{

#ifndef NDEBUG
     unsigned int i;
     for (i = 0; i < operands.size(); i++)
	  assert(operands[i].IsTerm());
#endif

     return Expression(new OperatorNode(EXPR_FUNCTION, name, operands));
}

Expression Expression::IfThenElse(const Expression& formula, const Expression& arg1, const Expression& arg2) {
	assert(formula.IsFormula());
	assert(arg1.IsTerm());
	assert(arg2.IsTerm());

	std::vector<Expression> operands;
	operands.push_back(formula);
	operands.push_back(arg1);
	operands.push_back(arg2);

	return Expression(new OperatorNode(EXPR_TERM_ITE, "ITE", operands));	
}


Expression Expression::Predicate(const std::string name, const Expression& arg1)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     return Expression::Predicate(name, operands);
}

Expression Expression::Predicate(const std::string name, const Expression& arg1, const Expression& arg2)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     return Expression::Predicate(name,operands);
}

Expression Expression::Predicate(const std::string name, const Expression& arg1, const Expression& arg2, const Expression& arg3)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     operands.push_back(arg3);
     return Expression::Predicate(name,operands);
}

Expression Expression::Predicate(const std::string name, const std::vector<Expression>& operands)
{
#ifndef NDEBUG
     unsigned int i;
     for (i = 0; i < operands.size(); i++)
	  assert(operands[i].IsTerm());
#endif

     return Expression(new OperatorNode(EXPR_PREDICATE,name,operands));
}


Expression Expression::Equality(const Expression& l, const Expression& r)
{
     assert(l.IsTerm() && r.IsTerm());
     std::vector<Expression> operands;
     operands.push_back(l);
     operands.push_back(r);
     return Expression(new OperatorNode(EXPR_EQUALITY, "=", operands));
}

Expression Expression::Disequality(const Expression& l, const Expression& r)
{	
     assert(l.IsTerm() && r.IsTerm());
     std::vector<Expression> operands;
     operands.push_back(l);
     operands.push_back(r);
     return Expression(new OperatorNode(EXPR_DISEQUALITY, "!=", operands));
}


Expression Expression::AND(const Expression& arg1, const Expression& arg2)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     return AND(operands);
	
}

Expression Expression::AND(const Expression& arg1)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     return AND(operands);
	
}

Expression Expression::AND(const std::vector<Expression>& operands)
{	
#ifndef NDEBUG
     unsigned int iarg;
     for (iarg = 0; iarg < operands.size(); iarg++)
	  assert(operands[iarg].IsFormula());
#endif


	 return Expression(ANDNode::Simplify(operands));
}

Expression Expression::OR(const Expression& arg1, const Expression& arg2)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
     return OR(operands);
}

Expression Expression::OR(const Expression& arg1)
{	
     std::vector<Expression> operands;
     operands.push_back(arg1);
     return OR(operands);
}


Expression Expression::OR(const std::vector<Expression>& operands)
{
#ifndef NDEBUG
     unsigned int iarg;
     for (iarg = 0; iarg < operands.size(); iarg++)
	  assert(operands[iarg].IsFormula());
#endif
	 return Expression(ORNode::Simplify(operands));
}


Expression Expression::IMPL(const Expression& pre, const Expression& ant)
{	
     assert(pre.IsFormula() && ant.IsFormula());

     std::vector<Expression> operands;
     operands.push_back(pre);
     operands.push_back(ant);
     return Expression(new OperatorNode(EXPR_IMPL,"impl",operands));
}

Expression Expression::IFF(const Expression& e1, const Expression& e2)
{
     assert(e1.IsFormula() && e2.IsFormula());

     std::vector<Expression> operands;
     operands.push_back(e1);
     operands.push_back(e2);
     return Expression(new OperatorNode(EXPR_IFF,"iff",operands));
}


Expression Expression::NOT(const Expression& e)
{	
     assert(e.IsFormula());

     if (e.IsNOT())
		 return e[0];
     if (e.IsBOT())
		return Expression::TOP();
     if (e.IsTOP())
		return Expression::BOT();

     std::vector<Expression> operands;
     operands.push_back(e);
     return Expression(new OperatorNode(EXPR_NOT,"not",operands));
}


Expression Expression::XOR(const Expression& arg1, const Expression& arg2)
{
	assert(arg1.IsFormula());
	assert(arg2.IsFormula());

     std::vector<Expression> operands;
     operands.push_back(arg1);
     operands.push_back(arg2);
	
     return Expression(new OperatorNode(EXPR_XOR, "xor", operands));
}

Expression Expression::TOP()
{	
     return Expression(new ExpressionNode(EXPR_TOP,"top"));
}

Expression Expression::BOT()
{	
     return Expression(new ExpressionNode(EXPR_BOT,"bot"));
}


// Expression building operators
Expression Expression::operator&& (const Expression& e) const
{	
     if (IsNULL())
	  return e;

     if (e.IsNULL())
	  return *this;

     return Expression::AND(*this,e);
}

Expression Expression::operator|| (const Expression& e) const
{	
     if (IsNULL())
	  return e;

     if (e.IsNULL())
	  return *this;

     return Expression::OR(*this,e);
}


Expression Expression::operator!() const {
	return Expression::NOT(*this);
}

//Subexrpression finding
bool Expression::ContainsSubexpression(const Expression& subexpression) const
{	
    if (*this == subexpression)
	return true;

    if (hasOperands()) {	
	Expression::operands_iterator i, iend = end();
	for (i=begin(); i!=iend; i++) {
	    if (i->ContainsSubexpression(subexpression))
		return true;
	}
    }
    return false;
}

bool Expression::ContainsSubexpression(const Expression& subexpression, Position& position) const
{	
    if (*this == subexpression)
	return true;

    if (hasOperands()) {	
	Expression::operands_iterator i, iend = end();
	for (i=begin(); i!=iend; i++) {
	    position.push_back(i - begin());
	    if (i->ContainsSubexpression(subexpression, position))
		return true;
	    position.pop_back();
	}
    }
    return false;
}

