/***************************************************************************
  Copyright (C) 2007 Filip Maric, Predrag Janicic

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------------
  This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
  It uses Bliss (C) Tommi Junttila
*****************************************************************************/

#ifndef __BACKTRACKABLE_STACK_H__
#define __BACKTRACKABLE_STACK_H__

#include <vector>
#include <list>
#include <string>
#include <cassert>

std::string toString(int t);
std::string toString(unsigned t);
std::string toString(unsigned short t);
std::string toString(std::string t);

template <class T>
std::string toString(T t) {
	return t->toString();
}


class BacktrackableLevels {
 private:
    std::vector<unsigned> levels;
 public:
    int getLastLevelCount() const {
	return levels.back();
    }
	
    void setLastLevelCount(int num) {
	levels.back() = num;
    }

    BacktrackableLevels() {
	newLevel();
    }

    void retractLevel() {
	if (getLevel() > 0) {
	    assert(getLastLevelCount() == 0);
	    levels.pop_back();
	}
    }

    unsigned getLevel() const {
	return size() - 1;
    }

    unsigned operator[] (unsigned n) const {
	return levels[n];
    }

    void push() {
	setLastLevelCount(getLastLevelCount() + 1);
    }
	
    void pop() {
	if (getLastLevelCount() > 0)
	    setLastLevelCount(getLastLevelCount() - 1);
    }

    bool lastLevelEmpty() {
	return getLastLevelCount() == 0;
    }

    void newLevel() {
	levels.push_back(0);
    }
	
    void clear() {
	levels.clear();
	newLevel();
    }

    unsigned size() const {
	return levels.size();
    }

    unsigned sumLastNLevels(unsigned n) const {
	unsigned sum = 0;
	for (unsigned k = (levels.size() >= n ? levels.size() - n : 0); k < levels.size(); k++) {
	    sum += levels[k];
	}
	return sum;
    }
};

template <class T>
class BacktrackableStack : public std::list<T> {
 private:
    BacktrackableLevels levels;
    
    // Make users use push instead of push_back
    void push_back(const T& item) {
	std::list<T>::push_back(item);
    }


    void pop_back() {
	std::list<T>::pop_back();
    }
    
 public:
    const T& back() const {
	return std::list<T>::back();
    }

    typename std::list<T>::const_iterator begin() const {
	return std::list<T>::begin();
    }

    typename std::list<T>::const_iterator end() const {
	return std::list<T>::end();
    }

    void push(const T& item) {
	levels.push();
	push_back(item);
    }
	
    T pop() {
	levels.pop();
	T last = back();
	pop_back();
	return last;
    }

    bool lastLevelEmpty() {
	return levels.lastLevelEmpty();
    }

    T peek() {
	return back();
    }

    void newLevel() {
	levels.newLevel();
    }

    void retractLevel() {
	levels.retractLevel();
    }


    void popLastN(unsigned n) {
	for (unsigned i = 0; i < n; i++)
	    pop();
    }
	
    void backtrackLastLevel() {
	popLastN(levels.getLastLevelCount());
	if (levels.lastLevelEmpty())
	    levels.retractLevel();
    }
	
    int getLevel() {
	return levels.getLevel();
    }
	
    int getLowestLevel(T item) {		
	int level = 0; 
	unsigned sum = levels[0];
	unsigned currentPosition = 0;
	typename std::list<T>::const_iterator i; 
	for (i = begin(); i != end(); i++) {
	    if (currentPosition++ == sum) {
		level++;
		sum += levels[level];
	    }
	    if ((*i) == item)
		return level;
	}
	return -1;
    }
	
    void clear() {
	levels.clear();
	std::vector<T>::clear();
    }

    const T& get(int j) const {
	int i = 0;
	typename std::list<T>::const_iterator it;
	for (it = begin(); i < j; it++, i++)
	    ;
	return *it;
    }



    std::string toString() const;

    unsigned sumLastNLevels(unsigned n) const {
	return levels.sumLastNLevels(n);
    }
    
};

#endif

