/***************************************************************************
  Copyright (C) 2007 Filip Maric, Predrag Janicic

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------------
  This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
  It uses Bliss (C) Tommi Junttila
*****************************************************************************/

#ifndef __CLAUSE_H__
#define __CLAUSE_H__

#include "Literal.h"

class Clause {
 public:
    typedef std::vector<Literal>::const_iterator const_iterator;
    typedef std::vector<Literal>::iterator iterator;

    Clause(std::vector<Literal> literals, bool initial=false) 
	: _literals(literals),
	_initial(initial),
	_activity(0.0) {
    }

    const_iterator begin() const {
	return _literals.begin();
    }

    size_t size() const {
	return _literals.size();
    }

    const_iterator end() const {
	return _literals.end();
    }

    iterator begin() {
	return _literals.begin();
    }

    Literal operator[] (size_t i) const {
	return _literals[i];
    }

    Literal& operator[] (size_t i) {
	return _literals[i];
    }

    void swapLiterals(size_t i, size_t j) {
	Literal tmp = _literals[i];
	_literals[i] = _literals[j];
	_literals[j] = tmp;
    }

    bool containsLiteral(Literal literal) const {
	return std::find(_literals.begin(), _literals.end(), literal) != _literals.end();
    }

    bool removeLiteral(Literal literal) {
	std::vector<Literal>::iterator it = std::find(_literals.begin(), _literals.end(), literal);
	if (it == _literals.end())
	    return false;
	_literals.erase(it);
	return true;
    }

    bool isTrue() const;
    bool isFalse() const;
    
    bool isTrue(const std::vector<Literal>& model);
    
    bool isUnit() const;

    /**
     *   A literal is unit iff it is the only unassigned literal
     */
    const Literal* getUnitLiteral() const;

    std::string toString() const;
    std::string valuationString() const;

    Clause* getOpposite() const;

    double& getActivity() {
	return _activity;
    }

    bool isInitial() const {
	return _initial;
    }

    void canonicalForm();
    Clause* cannonicalForm() const;

    bool subsumes(Clause* candidate) const {
	const_iterator it, beg = candidate->begin(), en = candidate->end();
	for (it = beg; it != en; it++)
	    if (!containsLiteral(*it)) {
//		cout << "Not found: " << Literals::toString(*it) << endl;
		return false;
	    }
	return true;
    }
    
    
 private:
    std::vector<Literal> _literals;
    bool _initial;
    double _activity;
};
#endif
