/***************************************************************************
  Copyright (C) 2007 Filip Maric, Predrag Janicic

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------------
  This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
  It uses Bliss (C) Tommi Junttila
*****************************************************************************/

#include "Solver.h"
#include "SolverStatistics.h"

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <string>
using std::cout;
using std::cerr;
using std::endl;

std::ofstream logger_file("log.txt");
std::ostream& logger = logger_file;

class SolverTest {
private:
    Solver solver;
	
    int offset;
	
    int getIndex(int i, int j, int n) {
	return offset + i*n + j;
    }
	
    void generateUniqueElementClauses(const std::vector<int>& indices) {
	size_t i, j;
	std::vector<Literal> literals;
	for (i = 0; i < indices.size(); i++) {
	    for (j = i+1; j < indices.size(); j++) {
		literals.clear();
		literals.push_back(Literals::getLiteral(indices[i], false));
		literals.push_back(Literals::getLiteral(indices[j], false));
		solver.addInitialClause(new Clause(literals, true));
	    }
	}
    }
	
    void generateQueens(int n) {
	solver.setNumberOfVariables(n*n);
	std::vector<Literal> literals;
	for (int i = 0; i < n; i++) {
	    literals.clear();
	    for (int j = 0; j < n; j++) {
		literals.push_back(Literals::getLiteral(getIndex(i, j, n), true));
	    }
	    solver.addInitialClause(new Clause(literals, true));
	}
		
	for (int i = 0; i < n; i++) {
	    std::vector<int> row;
	    for (int j = 0; j < n; j++)
		row.push_back(getIndex(i, j, n));
	    generateUniqueElementClauses(row);
	}

	for (int j = 0; j < n; j++) {
	    std::vector<int> col;
	    for (int i = 0; i < n; i++)
		col.push_back(getIndex(i, j, n));
	    generateUniqueElementClauses(col);
	}
		
	for (int k = -n+2; k < n-1; k++) {
	    std::vector<int> diag;
	    for (int i = std::max(0, k); i <= std::min(n-1, k+n-1); i++) {
		diag.push_back(getIndex(i, i-k, n));
	    }
	    generateUniqueElementClauses(diag);
	}
		
	for (int k = 1; k < 2*n - 2; k++) {
	    std::vector<int> diag;
	    for (int i = std::max(0, k-n+1); i <= std::min(n-1, k); i++) {
		diag.push_back(getIndex(i, k-i, n));
	    }
	    generateUniqueElementClauses(diag);
	}

    }
	
    void generatePigeons(int n) {
	cout << "Generating pigeons: " << n << endl;
	solver.setNumberOfVariables(n*(n+1));
	std::vector<Literal> literals;
	for (int i = 0; i < n+1; i++) {
	    literals.clear();
	    for (int j = 0; j < n; j++) {
		literals.push_back(Literals::getLiteral(getIndex(i, j, n), true));			
	    }
	    solver.addInitialClause(new Clause(literals, true));
	}
		
	for (int j = 0; j < n; j++) {
	    std::vector<int> col;
	    for (int i = 0; i < n + 1; i++)
		col.push_back(getIndex(i, j, n));
	    generateUniqueElementClauses(col);
	}
		
    }

    
    void parseDIMACS(string fileName) {
	cout << "Parsing file: " << fileName << endl;
	std::ifstream file(fileName.c_str());
	if (!file.is_open()) {
	    cerr << "Error opening file: " << fileName << endl;
	    exit(EXIT_FAILURE);
	}
	string line;
	while(getline(file, line)) {
	    if (line[0] == 'c')
		continue;
	    if (line[0] == 'p') {
		std::istringstream stream(line.c_str() + 5);
		int numVars;
		stream >> numVars;
		int numClauses;
		stream >> numClauses;
		
		solver.setNumberOfVariables(numVars);
		continue;
	    }
	    if (line == "")
		continue;
	    
	    std::istringstream stream(line.c_str());
	    std::vector<Literal> clauseLiterals;
	    int num;
	    stream >> num;
	    while (num != 0) {
		clauseLiterals.push_back(Literals::fromInt(num));
		stream >> num;
	    }
	    solver.addInitialClause(new Clause(clauseLiterals, true));
	}
    }

    static std::string copyrightString, usageString;
public:	
    SolverTest() 
	: offset(0) {
    }

    void generateAndSolveProblem(int argc, char* argv[]) {
	cout << copyrightString << endl;

	if (argc < 2) {
	    cerr << usageString << endl;
	    exit(EXIT_FAILURE);
	}

	if (strcmp(argv[1], "pigeons") == 0) {
	    generatePigeons(atoi(argv[2]));
	} else 	if (strcmp(argv[1], "queens") == 0) {
	    generateQueens(atoi(argv[2]));
	} else
	    parseDIMACS(argv[1]);

	for (int i = 1; i < argc; i++) {
	    if (strcasecmp(argv[i], "--literal_selection_polarity") == 0 ||
		strcasecmp(argv[i], "--polarity") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Polarity not specified. Expected one of the following {true, false, random}" << endl;
		    exit(EXIT_FAILURE);
		}
		if (strcasecmp(argv[i], "true") == 0) {
		    solver.setOptionLiteralSelectionPolarity(Solver::LITERAL_SELECTION_POLARITY_TRUE);
		} else if (strcasecmp(argv[i], "false") == 0) {
		    solver.setOptionLiteralSelectionPolarity(Solver::LITERAL_SELECTION_POLARITY_FALSE);
		} else if (strcasecmp(argv[i], "random") == 0 ||
			   strcasecmp(argv[i], "rand") == 0) {
		    solver.setOptionLiteralSelectionPolarity(Solver::LITERAL_SELECTION_POLARITY_RANDOM);
		} else {
		    cerr << "Wrong polarity. Expected one of the following {true, false, random}" << endl;
		    exit(EXIT_FAILURE);
		}
	    } else if (strcasecmp(argv[i], "--variable_selection_strategy") == 0 ||
		strcasecmp(argv[i], "--selection") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Selection strategy not specified. Expected one of the following {first, max}" << endl;
		    exit(EXIT_FAILURE);
		}
		if (strcasecmp(argv[i], "first") == 0 ||
		    strcasecmp(argv[i], "first_undefined") == 0) {
		    solver.setOptionVariableSelectionStrategy(Solver::VARIABLE_SELECTION_FIRST_UNDEFINED);
		} else if ((strcasecmp(argv[i], "max") == 0) ||
			   (strcasecmp(argv[i], "max_active") == 0)) {
		    solver.setOptionVariableSelectionStrategy(Solver::VARIABLE_SELECTION_MAX_ACTIVE);
		} else {
		    cerr << "Wrong selection strategy. Expected one of the following {first, max}" << endl;
		    exit(EXIT_FAILURE);
		}
	    } else if (strcasecmp(argv[i], "--variable_activity_decay_factor") == 0 ||
		       strcasecmp(argv[i], "--variable_decay") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Variable activity decay factor expected" << endl;
		}
		float factor = atof(argv[i]);
		solver.setOptionVariableActivityDecayFactor(factor);
	    } else if (strcasecmp(argv[i], "--restart_first") == 0 ||
		       strcasecmp(argv[i], "--restart_first_number_of_conflicts") == 0 ) {
		i++;
		if (i == argc) {
		    cerr << "Number of conflicts for first restart expected" << endl;
		}
		size_t numConflicts = atoi(argv[i]);
		solver.setOptionFirstRestart(numConflicts);
	    } else if (strcasecmp(argv[i], "--restart_increase_factor") == 0 ||
		strcasecmp(argv[i], "--restart_inc") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Restart increase factor expected" << endl;
		}
		float factor = atof(argv[i]);
		solver.setOptionRestartIncreaseFactor(factor);
	    } else if (strcasecmp(argv[i], "--forget_increase_factor") == 0 ||
		strcasecmp(argv[i], "--forget_inc") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Forget increase factor  expected" << endl;
		}
		float coefficient = atof(argv[i]);
		solver.setOptionForgetIncreaseFactor(coefficient);
	    } else if (strcasecmp(argv[i], "--clause_activity_decay_factor") == 0 ||
		       strcasecmp(argv[i], "--clause_decay") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Clause activity decay factor expected" << endl;
		}
		float factor = atof(argv[i]);
		solver.setOptionClauseActivityDecayFactor(factor);
	    } else if (strcasecmp(argv[i], "--seed") == 0) {
		i++;
		if (i == argc) {
		    cerr << "Random seed expected" << endl;
		}
		int seed = atoi(argv[i]);
		cout << "Randomized with seed: " << seed << endl;
		srand(seed);
	    } else if (argv[i][0] == '-') {
		cerr << "Unknown option " << argv[i] << endl;
		exit(EXIT_FAILURE);
	    }
	}

	int numOfModels = 0;

	solver.solve();
	while (solver.isSatisfiable()) {
	    std::vector<Literal> model = solver.getModel();
	    cout << "Model: " << ++numOfModels << endl;
	    cout << "------------------------------------------------------------------------------" << endl;

	    Clause* modelClause = new Clause(model, true);
	    cout << modelClause->toString() << endl;
	    cout << "------------------------------------------------------------------------------" << endl;


#ifndef NDEBUG
	    solver.verifyModel();
#endif

	    Clause* oppositeClause = modelClause->getOpposite();
	    delete modelClause;
	    solver.addInitialClause(oppositeClause);
	    solver.solve();
	}
		
	cout << "------------------------------------------------------------------------------" << endl;
	cout << numOfModels << " models found" << endl;
	cout << "------------------------------------------------------------------------------" << endl;

    }
};

int main(int argc, char* argv[]) {
    SolverTest test;
    test.generateAndSolveProblem(argc, argv);
    cout << "----------------------------- S U M M A R Y ------------------------------------" << endl;
    SolverStatistics::printHeader();
    SolverStatistics::printStatistics();
    SolverStatistics::printFooter();
    return 0;
}

std::string SolverTest::copyrightString = 
"------------------------------------------------------------------------\n"
"ArgoSat v. 1.0\n"
"Copyright (C) 2007\n"
"Filip Maric, Predrag Janicic, Faculty of Mathematics, Belgrade\n"
"------------------------------------------------------------------------\n"
"Inspired by MiniSat (C) Een, Sorensson, 2003-2006\n"
"Uses Bliss (C) Tommi Junttila, 2006\n";

std::string SolverTest::usageString = 
"Usage: argosat problem [options]\n"
"    problem:\n"
"          pigeons num - pigeonhole problem\n"
"          queens  num - n queens problem\n"
"          filename    - file that contains uncompressed DIMACS format input\n"
"    options:\n"
"          --literal_selection_polarity {true | false | random}\n"
"                 true   - decision literals are always positive\n"
"                 false  - decision literals are always negative (default)\n"
"                 random - decision literals are uniformly positive or negative\n"
"          --variable_selection_strategy {first | max}\n"
"                 first - first undefined variable is used for a decision variable\n"
"                 max   - VSIDS heuristic - maximally acitive variable is used for a decision variable (default)\n"
"          --variable_activity_decay_factor num - (default 1.05)\n"
"                 used as a factor in max variable selection strategy\n"
"                 larger number gives more preference to selecting variables that occured in newer confilcts\n"
"                 smaller number (closer to 1.0) does not distinguish between newer and older conflicts\n"
"          --restart_first_number_of_conflicts num - (default 100)\n"
"                 number of conflicts needed for the first restart\n"
"          --restart_increase_factor num - (default 1.5)\n"
"                 restart is applied with periodicity increased exponentially with this factor\n"
"          --forget_increase_factor num - (default 1.1)\n"
"                 forget is applied when the number of clauses exceeds numbers that are\n"
"                 increased exponentially with this factor\n"
"          --clause_activity_decay_factor num - (default 1.01)\n"
"                 used as a factor in forget clause strategy\n"
"                 larger number gives more preference to clauses that participated in newer confilcts\n"
"                 smaller number (closer to 1.0) does not distinguish between newer and older conflicts\n"
"          --symmetry [num]\n"
"                 symplify formula by exploiting its symmetries\n"
"                 num  - degree of symmetry exploatation (default -1 exploit all symmetries)\n"
"          --seed num\n"
"                 set random seed\n";

