#ifndef __LEARNT_CLAUSE_H__
#define __LEARNT_CLAUSE_H__


#include <vector>
#include <algorithm>
#include <sstream>
#include <cassert>
#include "Clause.h"
#include "Literal.h"
#include "Variable.h"

class Solver;

/**
 * Helper data structure used in firstUIP conflict analysis.
 * Class represents a clause that is resolved with other clauses in
 * order to compute firstUIP clause. Current decision level literals
 * are all going to be resolved during process and they are not explicitly
 * stored for efficiency reasons. Only previous decision level literals
 * that are going to be included in the resulting clause are stored.
 * @author Filip Maric
 */
class ResolutionClause {
 private:
    /**
     * Literals from previous decision levels. These literals are not
     * resolved during resolution process and they all participate
     * in the resulting clause.
     */
    std::vector<Literal> previousLevelLiterals;

    /**
     * Literals from the initial decision level
     */
    std::vector<Literal> initialLevelLiterals;

    /**
     * Number of literals from the current decision level. For efficiency
     * reasons, the literals are not explicitly stored, since they are
     * all going to be resolved away.
     */
    int numCurrentLevelLiterals;

    /**
     * Literal on which the clause resolution is performed. This
     * literal has to be current decision level literal.
     */
    Literal resolutionLiteral;

    Clause* clause;
	
    /**
     * Sat solver which tells decision levels of the literals
     */
    Solver* solver;

 public:
    /**
     * Clause is initialized to the empty clause.
     * @param solver
     */
    ResolutionClause(Solver* solver);

    /**
     * Starts the resolution process by adding all the literals from
     * the conflict clause.
     * @param conflictClause
     */
    void setConflictClause(Clause* conflictClause) {
	Clause::const_iterator i, b = conflictClause->begin(), e = conflictClause->end();
	for (i = b; i != e; i++)
	    addLiteral(*i);
    }

    /**
     * Specifies the next resolution literal which is going to be used
     * for the next resolution step. Resolution literal has to be one
     * of the current decision level literals.
     * @param resolutionLiteral
     */
    void setResolutionLiteral(Literal resolutionLiteral) {
	assert(getDecisionLevel(resolutionLiteral) == getCurrentDecisionLevel());
	this->resolutionLiteral = resolutionLiteral;
    }

    /**
     * @return resolution literal
     */
    Literal getResolutionLiteral() const {
	return resolutionLiteral;
    }

    /**
     * Removes resolution literal from the clause
     */
    void removeResolutionLiteral();
	
    /**
     * Resolves this clause with the given clause, storing the result in
     * this clause. Resolution is done on the resolution literal which has
     * to be set using the setResolutionLiteral().
     * @param clause 
     */
    void resolve(Clause* clause) {
	Clause::const_iterator i, b = clause->begin(), e = clause->end();
	for (i = b; i != e; i++) {
	    if (*i != resolutionLiteral) {
		addLiteral(*i);
	    }
	}
	
	removeResolutionLiteral();
    }

    /**
     * Convert this optimized data structure to the ordinary clause
     * @return clause that represents the firstUIP
     */
    Clause* getClause() {
	if (clause != 0)
	    return clause;
		
	std::vector<Literal> clauseLiterals = previousLevelLiterals;
	clauseLiterals.insert(clauseLiterals.begin(), resolutionLiteral);
	clause = new Clause(clauseLiterals);
	return clause;
    }
	
    /**
     * @return number of current decision level literals in the clause
     */
    int getNumberOfCurrentLevelLiterals() const {
	return numCurrentLevelLiterals;
    }

    /**
     * @return Maximal decision level of a literal (excluding current decision level). 
     * If there were no previous level literals in the clause, decision level 0 is returned.
     */
    int getMaxPreviousLevel() const {
	int maxLevel = 0;

	std::vector<Literal>::const_iterator i, 
	    b = previousLevelLiterals.begin(), 
	    e = previousLevelLiterals.end();
	for (i = b; i != e; i++) {
	    int currLevel = getDecisionLevel(Literals::getOpposite(*i)); 
	    if (maxLevel < currLevel) {
		maxLevel = currLevel;
	    }
	}
		
	assert(maxLevel != -1);
	return maxLevel;
    }	


    /**
     * Textual representation of the clause
     */
    std::string toString() const {
	std::ostringstream stream;
	stream << numCurrentLevelLiterals;
	return (resolutionLiteral != 0 ? Literals::toString(resolutionLiteral) : string("null")) + 
	    string(" ") + 
	    previousLevelLiteralsToString() + 
	    string(" ") + 
	    stream.str();
    }
	
    const std::vector<Literal>& getInitialLevelLiterals() const {
	return initialLevelLiterals;
    }

    void getPreviousLevelLiterals(std::vector<Literal>& literals) const {
	literals = previousLevelLiterals;
    }

 
    /**
     * @param variable
     * @return true if the clause contains the given variable
     */
    bool containsVariable(Variable variable) const {
	return _containsVariableCache[variable];
    }


    /* Literal polarity is not checked !!! */
    bool subsumes(Clause* clause) {
	Clause::const_iterator it,
	    beg = clause->begin(), en = clause->end();
	for (it = beg; it != en; it++)
	    if (!containsVariable(Literals::getVariable(*it))) {
		return false;
	    }
	return true;
    }
    
    
 private: 
    /**
     * Adds literal to the clause
     * @param literal
     */
    void addLiteral(Literal literal) {
	Variable variable = Literals::getVariable(literal);
	// Clause contains the literal iff it contains its variable
	if(containsVariable(variable))
	    return;
	
	size_t decisionLevel = getDecisionLevel(Literals::getOpposite(literal));
	if (decisionLevel == getCurrentDecisionLevel()) {
	    // cout << "Current level: " << Literals::toString(literal) << endl;
	    // No need to store current level literals, we just count them
	    numCurrentLevelLiterals++;
	} else if (decisionLevel == 0) {
	    // cout << "Initial level: " << Literals::toString(literal) << endl;
	    initialLevelLiterals.push_back(literal);
	} else {
	    // Add literals from previous levels except initial
	    // cout << "Previous level: " << Literals::toString(literal) << endl;
	    previousLevelLiterals.push_back(literal);
	}
		
	// Mark that the clause contains the given variable
	setContainsVariable(Literals::getVariable(literal), true);
    }

 public:
    void removeLiteral(Literal literal) {
	size_t decisionLevel = getDecisionLevel(Literals::getOpposite(literal));
	if (decisionLevel == getCurrentDecisionLevel()) {
	    numCurrentLevelLiterals--;
	} else if (decisionLevel != 0) {
	    previousLevelLiterals.erase(
		std::find(previousLevelLiterals.begin(), 
			  previousLevelLiterals.end(), literal));
	}
	// Mark that the variable is not contained in the clause anymore
	setContainsVariable(Literals::getVariable(literal), false);
    }

 private:		
    /**
     * List of variables that are currently in the clause.
     * This is just an optimization feature. Since we assume that
     * no conflicting literals are going to be added to the clause,
     * it is sufficent to check just if the clause contain the
     * variable in order to check if the clause contais the literal
     */
    static std::vector<bool> _containsVariableCache;
    static size_t _nVars;
	
    /**
     * Add the variable to the clause variable array.
     * @param variable
     * @param contains if true, variable is included, and if false, variable is excluded.
     */
    void setContainsVariable(Variable variable, bool contains) {
	_containsVariableCache[variable] = contains;
    }	
	
    std::string previousLevelLiteralsToString() const {
	std::string result = "[";
	unsigned n = previousLevelLiterals.size();
	for (unsigned i = 0; i < n; i++) {
	    result += Literals::toString(previousLevelLiterals[i]);
	    if (i != n - 1)
		result += ", ";
	}
	result += " | " ;

	n = initialLevelLiterals.size();
	for (unsigned i = 0; i < n; i++) {
	    result += Literals::toString(initialLevelLiterals[i]);
	    if (i != n - 1)
		result += ", ";
	}

	result += "]";
	return result;
    }

    /**
     * Checks if the literal was asserted in the current decision level
     */
    size_t getDecisionLevel(Literal literal) const;
	
    /**
     * Checks if the literal was asserted in the initial decision level
     */
    size_t getCurrentDecisionLevel() const;
};

#endif

