/***************************************************************************
  Copyright (C) 2007 Filip Maric, Predrag Janicic

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------------
  This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
  It uses Bliss (C) Tommi Junttila
*****************************************************************************/
#ifndef __SOLVER_H__
#define __SOLVER_H__

#include "BacktrackableStack.h"
#include "Activity.h"

#include "Clause.h"
#include "ResolutionClause.h"

#include "Heap.h"

#include <vector>
#include <deque>

/**
   Example usage:

      Solver solver;
      solver.setOption...(...);
      solver.setNumberOfVariables(...);
      solver.addInitialClause(...);
      ...
      solver.addInitialClause(...);
      solver.solve();
      if (solver.isUnsatisfiable())
          ... report unsatisfiability ...
      else if (solver.isSatisfiable()) {
          model = solver.getModel();
          ... report satifiabilitiy ...
      }
 */
class Solver {
    public: Solver();
    public: ~Solver();
    
    /****************************************************
     *      SOLVE
     *****************************************************/
    public: void solve();

    public: bool isSatisfiable() {
	return _satisfiable == TRUE;
    }
    public: bool isUnsatisfiable() {
	return _satisfiable == FALSE;
    }
    private: ExtendedBoolean _satisfiable;

    /****************************************************
     *       MODEL
     ****************************************************/
    public: const std::vector<Literal>& getModel() const {
	return _model;
    }
    public: bool verifyModel() const;
    private: void generateModel();
    private: std::vector<Literal> _model;

    /****************************************************
     *       VARIABLE SET
     ****************************************************/
    public: Variable newVariable(bool useAsDecisionVariable = true);
    public: int getNumberOfVariables() const {
	return Variables::size();
    }
    public: void setNumberOfVariables(size_t n) {
	for (size_t i = 0; i < n; i++)
	    newVariable();
    }

    /****************************************************
     *      INITIAL CLAUSE SET
     ****************************************************/
    public: void addInitialClause(Clause* clause);
    private: size_t getNumberOfInitialClauses() const;
    private: bool isInitialClause(Clause* clause) const;
    private: std::vector<Clause*> _initialClauses;
    public: void printClauses();

    /****************************************************
     *       ASSERTION TRAIL  -  M
     ****************************************************/
    private: void assertLiteral(Literal l);
    private: Literal getLastAssertedLiteral();
    private: size_t getCurrentDecisionLevel();
    private: size_t getLiteralsDecisionLevel(Literal literal);
    private: Literal backtrackLiteral();
    private: Literal backtrackLevel();
    private: void backtrackToLevel(size_t level);
    private: BacktrackableStack<Literal> _assertionTrail;

    /***************************************************
     *     CONFLICT CLAUSE DETECTION
     ***************************************************/
    private: bool isConflicting() const {
	return getConflictClause() != 0;
    }
    private: Clause* getConflictClause() const {
	return _conflictClause;
    }
    private: void setConflictClause(Clause* clause);
    Clause* _conflictClause;
    
    /*****************************************************
     *         UNIT PROPAGATE
     *****************************************************/
    private: bool applicableUnitPropagate();
    private: void applyUnitPropagate();

    private: Literal getUnitLiteral(Clause* clause);
    private: Clause* getUnitClause();
    private: void findNextUnitClause();

    // Unit clause queue
    private: std::deque<Clause*> _unitClauseQueue;
    private: void enqueueUnitClause(Clause* unitClause);

    // Two watch literal scheme
    private: void setWatch(char watchNumber, Clause* clause, size_t position);
    private: Literal getWatch(char watchNumber, Clause* clause);
    private: void removeWatch(char watchNumber, Clause* clause);
    private: void swapWatches(Clause* clause);
    private: void initializeWatches(Clause* clause);
    private: void findAndSetWatch(char watchNumber, Clause* clause);
    private: int findUnfalsifiedLiteralPosition(Clause* clause, char watchNumber);
    private: int findLatestFalsifiedLiteralPosition(Clause* clause, char watchNumber);    
    private: std::vector< std::vector<Clause*> > _watchClauses;

    private: void addWatchClause(Literal literal, Clause* clause);
    private: void removeWatchClause(Literal literal, Clause* clause);
    private: void notifyWatchClauses(Literal literal);

    // Propagation graph
    private: Clause* getReason(Literal literal);
    private: void setReason(Literal literal, Clause* clause);
    private: std::vector<Clause*> _reason;

    /*****************************************************
     *        BACKTRACK
     *****************************************************/
    private: void applyBacktrack();

    /*****************************************************
     *        BACKJUMP
     *****************************************************/
    private: void applyBackjump();

    private: void findBackjumpClause();
    private: ResolutionClause _backjumpClause;

    private: void findFirstUIPClause();
    private: void performSubsumptionResolution();
    
    /*****************************************************
     *        DECIDE
     *****************************************************/
    private: void applyDecide();
    private: bool allVariablesAssigned();
    private: Literal selectDecisionLiteral();

    private: Variable getFirstUndefinedVariable();

    // Activity measurment
    private: Activity _variableActivity;
    private: void bumpVariableActivity(Variable variable);
    private: void bumpClauseVariablesActivity(Clause* clause);

    // Max activity variable selection
    private: Variable getMaxActiveUndefinedVariable();
    struct VariableComparator {
	bool operator() (Variable x, Variable y) {
	    return Variables::getActivity(x) > Variables::getActivity(y);
	}

	void print(Variable x) {
	    printf("%4.2f ", Variables::getActivity(x));
	}
    };
    private: ArgoSat::Heap<Variable, VariableComparator> _variableActivityHeap;

    // Options
    public: enum LiteralSelectionPolarity {
	LITERAL_SELECTION_POLARITY_TRUE,
	LITERAL_SELECTION_POLARITY_FALSE,
	LITERAL_SELECTION_POLARITY_RANDOM};
    private: LiteralSelectionPolarity _literalSelectionPolarity;

    public: enum VariableSelectionStrategy {
	VARIABLE_SELECTION_FIRST_UNDEFINED,
	VARIABLE_SELECTION_MAX_ACTIVE};
    private: VariableSelectionStrategy _variableSelectionStrategy;

    /*****************************************************
     *        LEARN
     *****************************************************/
    private: void applyLearn(Clause* clause);

    private: void addLearntClause(Clause* clause);
    private: bool isLearntClause(Clause* clause) const;
    private: size_t getNumberOfLearntClauses() const;
    private: size_t getTotalNumberOfClauses() const;
    private: std::vector<Clause*> _learntClauses;

    /*****************************************************
     *        FORGET
     *****************************************************/
    private: void applyForget();
    private: void removeClause(Clause* clause);

    private: size_t _numClausesForForget;
    // Option
    private: float  _forgetIncreaseFactor;

    // Activity based Forget clause selection
    private: Activity _clauseActivity;
    private: void bumpClauseActivity(Clause* clause);
    private: bool isLockedByCurrentTrail(Clause* clause);
    struct ClauseComparator {
	Solver* solver;

	ClauseComparator(Solver* s) 
	    : solver(s) {}

	bool operator() (Clause* c1, Clause* c2);
    };

    // Forget satisfied clauses
    private: void forgetSatisfiedClauses();
    private: bool _shouldRemoveSatisfiedInitialClauses;

    /*****************************************************
     *        RESTART
     *****************************************************/
    private: void applyRestart();

    private: size_t _numConflictsSinceLastRestart;
    private: size_t _numConflictsForRestart;
    // Options
    private: size_t _numConflictsForFirstRestart;
    private: float  _restartIncreaseFactor;

    /*****************************************************
     *        THEORY PROPAGATE
     *****************************************************/
    private: bool applyTheoryPropagate();
    private: void addTheoryLemma();

    /****************************************************
     *       SOLVER OPTIONS API
     ****************************************************/
    public: void setOptionLiteralSelectionPolarity(LiteralSelectionPolarity polarity) {
	_literalSelectionPolarity = polarity;
    }
    public: void setOptionVariableSelectionStrategy(VariableSelectionStrategy strategy) {
	_variableSelectionStrategy = strategy;
    }
    public: void setOptionForgetIncreaseFactor(float factor) {
	_forgetIncreaseFactor = factor;
    }

    public: void setOptionRestartIncreaseFactor(float factor) {
	_restartIncreaseFactor = factor;
    }

    public: void setOptionFirstRestart(size_t numConflicts) {
        _numConflictsForFirstRestart = numConflicts;
    }

    public: void setOptionVariableActivityDecayFactor(float factor) {
	_variableActivity.setDecayFactor(factor);
    }
    
    public: void setOptionClauseActivityDecayFactor(float factor) {
	_clauseActivity.setDecayFactor(factor);
    }
    
    friend class ResolutionClause;

};


#endif
