#include <vector>
#include <map>
#include <set>
#include "expressions/Expression.h"
#include "theory/pra/VariableOrdering.h"

class Matrix {
 public:
    Matrix(const std::vector<std::string>& variables) 
	: _rows(0), _cols(0), _variables(variables),
	_basicVariableOrder(_variables, _basicVariables),
	_nonBasicVariableOrder(_variables),
	_orderedBasicVariables(_basicVariableOrder) {
    }

    void print();

    void pivot(unsigned i, unsigned x);


    unsigned cols() const {
	return _cols;
    }

    unsigned rows() const {
	return _rows;
    }

    void addRow() {
	_rows++;
	std::vector<RATIONAL> row(cols());
	_matrix.push_back(row);
	_nonBasicVariables.push_back(new std::set<unsigned, NonBasicVariableOrder>(_nonBasicVariableOrder));
    }

    void addCol() {
	_cols++;
	for (unsigned i = 0; i < rows(); i++) {
	    _matrix[i].push_back(0);
	}
    }

    void setCoefficient(unsigned i, unsigned j, const RATIONAL& c) {
	_matrix[i][j] = c;
    }

    void setBasicVariable(unsigned eq, unsigned v);

    bool isBasicVariable(unsigned v) {
	return _basicVariableEquations.find(v) != _basicVariableEquations.end();
    }

    unsigned getBasicVariable(unsigned i) {
	return _basicVariables[i];
    }

    unsigned getEqualityForVariable(unsigned v) {
	return _basicVariableEquations[v];
    }

    void normalize(unsigned i, RATIONAL a);


    void addNonBasicVariable(unsigned i, unsigned variable) {
	_nonBasicVariables[i]->insert(variable);
    }

    void check(const std::map<std::string, RATIONAL>& valuation) const {
	cout << "CHECK MATRIX" << endl;
	for (unsigned i = 0; i < rows(); i++) {
	    RATIONAL sum = 0;
	    for (unsigned j = 0; j < cols(); j++) {
		const RATIONAL& val = valuation.find(_variables[j])->second;
		sum += _matrix[i][j]*val;
	    }
	    if (sum != 0) {
		cout << "ERROR in row : " << i << endl;
		exit(EXIT_FAILURE);
	    }
	}
    }

 private:
    unsigned _rows;
    unsigned _cols;

    std::vector< std::vector<RATIONAL> > _matrix;
    const std::vector<std::string>& _variables;
    
 public:
    struct BasicVariableOrder {
	BasicVariableOrder(const std::vector<std::string>& variables, const std::vector<unsigned>& basic_variables) 
	    : _variables(&variables), _basicVariables(&basic_variables) {
	}

	int operator() (unsigned e1, unsigned e2) {
	    return _variableOrdering.compare((*_basicVariables)[e1], (*_basicVariables)[e2]) > 0;
	}

	const std::vector<std::string>* _variables;
	const std::vector<unsigned>* _basicVariables;
    };
 private:
    BasicVariableOrder _basicVariableOrder;

 public:
    struct NonBasicVariableOrder {
	NonBasicVariableOrder(const std::vector<std::string>& variables) 
	    : _variables(&variables) {
	}

	int operator() (unsigned v1, unsigned v2) {
	    return _variableOrdering.compare(v1, v2) > 0;
	}

	const std::vector<std::string>* _variables;
    };
 private:
    NonBasicVariableOrder _nonBasicVariableOrder;

 private:
    void allocate(unsigned m, unsigned n) {
	_matrix.resize(m);
	for (unsigned i = 0; i < n; i++)
	    _matrix[i].resize(n);
    }

    // eq -> var
    std::vector<unsigned> _basicVariables;
    // eqs ordered by var
    std::set<unsigned, BasicVariableOrder> _orderedBasicVariables;
    // var -> eq
    std::map<unsigned, unsigned> _basicVariableEquations;

    // eq -> var set
    std::vector< std::set<unsigned, NonBasicVariableOrder>* > _nonBasicVariables;


 public:
    const std::set<unsigned, NonBasicVariableOrder>& getNonBasicVariables(unsigned i) {
	return *_nonBasicVariables[i];
    }

    const std::vector<RATIONAL>& operator[] (unsigned i) const {
	return _matrix[i];
    }

    typedef std::set<unsigned, BasicVariableOrder>::const_iterator const_iterator;

    const_iterator begin() {
	return _orderedBasicVariables.begin();
    }

    const_iterator end() {
	return _orderedBasicVariables.end();
    }

    void sort();
};
