#include "PRALiteral.h"
#include "PRAPolyform.h"
#include "PRASignature.h"
#include <iterator>

Expression PRALiteral::Cannonize(const Expression& e) {
//    coutput << "Can: " << e << endl;
    if (IsCannonized(e))
	return e;

    Expression cannonized_left  = e.IsNOT() ? 
	PRAPolyform::Cannonize(e[0][0]) : 
	PRAPolyform::Cannonize(e[0]);

    Expression cannonized_right  = e.IsNOT() ? 
	PRAPolyform::Cannonize(e[0][1]) : 
	PRAPolyform::Cannonize(e[1]);

//    coutput << "CL: " << cannonized_left << endl;
//    coutput << "CR: " << cannonized_right << endl;

    std::string pred = e.IsNOT() ? PRASignature::NegatedPredicate(e[0].GetName()) : e.GetName();

    Expression minus = PRAPolyform::Minus(cannonized_left, cannonized_right);

//    coutput << "Minus: " << minus << endl;

    if (PRAPolyform::IsGround(minus)) {
	return FormExpression(pred, minus, Expression::RationalNumeral(RATIONAL(0)));
    }

    const RATIONAL& c0 = PRAPolyform::GetMaximalVariableCoefficient(minus);
    minus = PRAPolyform::Divide(minus, c0);
    return FormExpression(c0 > 0 ? pred : PRASignature::OppositePredicate(pred) , 
			  minus, Expression::RationalNumeral(RATIONAL(0)));
}

bool PRALiteral::IsCannonized(const Expression& e) {
    if (!e.IsPredicate() && !e.IsEquality() && !e.IsDisequality())
	return false;

    if (e.GetArity() != 2)
	return false;

    if (!e[1].IsNumeral() || e[1].GetValueRational() != 0)
	return false;

    if (!PRAPolyform::IsCannonizedPRAPolyform(e[0], false))
	return false;

    if (GetMaximalVariableCoefficient(e) != 1)
	return false;

    return true;
}


bool PRALiteral::IsGround(const Expression& e)
{
     return PRAPolyform::IsGround(e[0]);
}


Expression PRALiteral::Negate(const Expression& e)
{
     return FormExpression(PRASignature::NegatedPredicate(e.GetName()),
			   e[0],
			   e[1]);
}

Expression PRALiteral::Opposite(const Expression& e)
{
     return FormExpression(PRASignature::OppositePredicate(e.GetName()), 
			   PRAPolyform::Times(e[0],-1), 
			   e[1]);
}

bool PRALiteral::TruthValue(const Expression& e)
{
     assert(IsGround(e));

     std::string rel = e.GetName();
     RATIONAL l = e[0].IsNumeral() ? e[0].GetValueRational() : e[0][0].GetValueRational();
     RATIONAL r = 0;

     if (rel == GT)
	  return l>r;
     else if (rel == GEQ)
	  return l>=r;
     else if (rel == LT)
	  return l<r;
     else if (rel == LEQ)
	  return l<=r;
     else if (rel == EQ)
	  return l==r;
     else if (rel == DISEQ)
	  return l!=r;

     return false;
}


Expression PRALiteral::Simplify(const Expression& e)
{
     // Handle negation
     Expression cannonized = Cannonize(e);

     // Simplify ground expressions
     if (IsGround(cannonized))
	  return TruthValue(cannonized) ? Expression::TOP() : Expression::BOT();
     return FormExpression(cannonized.GetName(), PRAPolyform::Simplify(cannonized[0]), cannonized[1]);
}

const std::string& PRALiteral::GetMaximalVariableName(const Expression& e)
{
     return PRAPolyform::GetMaximalVariableName(e[0]);
}

RATIONAL PRALiteral::GetMaximalVariableCoefficient(const Expression& e)
{
     return PRAPolyform::GetMaximalVariableCoefficient(e[0]);
}

const Expression& PRALiteral::GetMaximalVariable(const Expression& e)
{
     return e[0][1][1];
}

RATIONAL PRALiteral::GetVariableCoefficient(const Expression& e, std::string variable) {
    return PRAPolyform::GetVariableCoefficient(e[0], variable);
}

RATIONAL PRALiteral::GetVariableCoefficient(const Expression& e, size_t i) {
    return PRAPolyform::GetVariableCoefficient(e[0], i);    
}

RATIONAL PRALiteral::GetConstantCoefficient(const Expression& e) {
    return PRAPolyform::GetConstantCoefficient(e[0]);
}

Expression PRALiteral::FormExpression(const std::string& rel, const Expression& left, const Expression& right)
{
     if (rel == EQ)
	  return Expression::Equality(left, right);
     if (rel == DISEQ)
	  return Expression::Disequality(left, right);

     return  Expression::Predicate(rel, left, right);
}

bool PRALiteral::IsVariableEquality(const Expression& e) {
    return e.IsEquality() && 
	PRAPolyform::IsVariableEquality(e[0]) && 
	e[1].GetValueRational() == 0;
}

bool PRALiteral::IsVariableDisequality(const Expression& e) {
    return e.IsDisequality() && 
	PRAPolyform::IsVariableEquality(e[0]) && 
	e[1].GetValueRational() == 0;
}

bool PRALiteral::IsSingleVariableConstraint(const Expression& e) {
    return  PRAPolyform::IsSingleVariableConstraint(e[0]);
}

size_t PRALiteral::GetNumberOfVariables(const Expression& e) {
    assert(IsCannonized(e));
    return PRAPolyform::GetNumberOfVariables(e[0]);
}

Expression PRALiteral::GetVariable(const Expression& e, int i) {
    assert(IsCannonized(e));
    return PRAPolyform::GetVariable(e[0], i);
}

void PRALiteral::GetVariables(const Expression& e, std::set<std::string>& variables) {
    size_t n = GetNumberOfVariables(e);
    for (size_t k = 0; k < n;  k++)
	variables.insert(GetVariable(e, k).GetName());
}

Expression PRALiteral::GetEquality(const Expression& e) {
    assert(IsCannonized(e));
    return Expression::Equality(e[0], e[1]);
}

