#ifndef __PRA_LITERAL_H__
#define __PRA_LITERAL_H__

#include <set>
#include <string>
#include "expressions/Expression.h"

/**
 * Collection of function for manipulation with cannonized PRA literals.
 * $$1 * v_1 + c_2 * v_2 + ... + c_n * v_n \cdot c_0$$
 * $$ \cdot\in\{>, \geq, =, \neq, \leq, <\}$$ 
 */
class PRALiteral
{
public:
     /**
      *  Cannonization of literal to the form 
      */
     static Expression Cannonize (const Expression& e);

     /**
      *  Checks if expression is cannonized PRA literal
      */
     static bool IsCannonized (const Expression& e);


     /**
      * Cheks if cannonized literal is ground (ie. contains no variables)
      */
     static bool IsGround(const Expression& e);


     /**
      *  Truth value of cannonized ground literal
      */
     static bool TruthValue(const Expression& e);


     /**
      *  Simpification of arbitrary PRA literals.
      *  Ground literals are replaced with their truth value, 
      *  x+0 -> x
      *  x*1 -> x
      */
     static Expression Simplify (const Expression& e);

     /**
      *  Negation of arbitrary PRA literal by changing its predicate. eg. 
      *     NegateLiteral(x+y>3) = x+y<=3
      */
     static Expression Negate (const Expression& e);          

     /**
      *  Opposite literal, eg.
      *   OppositeLiteral(x+y>3) = -x - y < -3
      */
     static Expression Opposite (const Expression& e);


     /**
      *  Name of the maximal variable in cannonized literal 
      */
     static const std::string& GetMaximalVariableName(const Expression& e);

     static const Expression& GetMaximalVariable(const Expression& e);

     /**
      *  Coefficient of the maximal variable in cannonized literal 
      */
     static RATIONAL GetMaximalVariableCoefficient(const Expression& e);
     static RATIONAL GetVariableCoefficient(const Expression& e, size_t i);
     static RATIONAL GetVariableCoefficient(const Expression& e, std::string variable);
     static RATIONAL GetConstantCoefficient(const Expression& e);


     static size_t GetNumberOfVariables(const Expression& e);

     static Expression GetVariable(const Expression& e, int i);

     /**
      *   Check if this literal represents variable equality (e.g. x = y) 
      */
     static bool IsVariableEquality(const Expression& e);

     /**
      *   Check if this literal represents variable disequality (e.g. x != y) 
      */
     static bool IsVariableDisequality(const Expression& e);

     /**
      *   Check if this literal represents single variable constraint
      */
     static bool IsSingleVariableConstraint(const Expression& e);

     /**
      *   Isolates maximal variable on the left side with possitive coefficient
      */
     static Expression SolveMaximalVariable (const Expression& e);

     static bool VariableEliminationPossible(const Expression& e1, const Expression& e2);

     /**
      *   Eliminates common maximal variable from two cannonized PRA literals
      */
     static Expression EliminateMaximalVariable (const Expression& e1, const Expression& e2);
     /**
      *   Eliminates common given variable from two cannonized PRA literals
      */ 
     static Expression EliminateVariable (const Expression& e1, const Expression& e2, const std::string& variable);

     static Expression CrossMultiplyAndAdd(const Expression& e1, const Expression& e2, INT a, INT b);

     static Expression GetEquality(const Expression& e);

     static void GetVariables(const Expression& e, std::set<std::string>& variables);

     static Expression FormExpression(const std::string& rel, const Expression& left, const Expression& right);
     

};


#endif
