#ifndef __PRA_POLYFORM_H__
#define __PRA_POLYFORM_H__


#include "expressions/Expression.h"

/**
 * Collection of functions for manipulation with cannonized PRA terms (polyforms)
 * $$c_0 + c_1v_1 + c_2v_2 + \ldots + c_nv_n$$, where 
 * $$v_1\succ v_2\succ \ldots \succ v_n$$ in variable ordering $$\succ$$
 */
class PRAPolyform
{
 public:
    /**
     * Checks if Expression e is PRA polyform
     */
    static bool IsPRAPolyform(const Expression& e, bool homogenous);

    static bool IsCannonizedPRAPolyform(const Expression& e, bool homogenous);
     
    /**
     * Converts arbitrary Expression e to polyform 
     * Assumes that e is PRA term
     */
    static Expression Cannonize(const Expression& e);


    /**
     *  Checks if polyform is ground ie. contains no variables
     */
    static bool IsGround(const Expression& pf);


    /**
     *  The following simplifications are performed : 
     *  0 + x -> x
     *  1*x -> x
     */
    static Expression  Simplify  (const Expression& pf);

    /** 
     *  Maximal variable of the polyform (in the variable ordering) 
     *  v_1
     */
    static const std::string& GetMaximalVariableName(const Expression& pf)
	{
	    assert(IsPRAPolyform(pf));
	    assert(!IsGround(pf));
	    return pf[1][1].GetName();
	}

    /**
     *  Maximal variable coefficient c_1
     */
    static RATIONAL GetMaximalVariableCoefficient(const Expression& pf) {
	assert(IsPRAPolyform(pf));
	assert(!IsGround(pf));
	return pf[1][0].GetValueRational();
    }

    /**
     *  Constant coefficient c_0
     */
    static const RATIONAL& GetConstantCoefficient(const Expression& pf) {
	assert(IsPRAPolyform(pf));
	return pf[0].GetValueRational();
    }

    static size_t GetNumberOfVariables(const Expression& pf) {
	return pf.GetArity() - 1;
    }

    static const RATIONAL& GetVariableCoefficient(const Expression& pf, std::string variable) {
	for (size_t i = 0; i < GetNumberOfVariables(pf); i++) {
	    if (GetVariable(pf, i).GetName() == variable) {
		return GetVariableCoefficient(pf, i);
	    }
	}
	throw "No such var";
    }

    static const RATIONAL& GetVariableCoefficient(const Expression& pf, int i) {
	return pf[i+1][0].GetValueRational();
    }

    static const Expression& GetVariable(const Expression& pf, int i) {
	return pf[i+1][1];
    }

    /**
     *   Check if this polyform represents variable equality (e.g. x = y) 
     */
    static bool IsVariableEquality(const Expression& pf) {
	assert(IsPRAPolyform(pf));
	return GetNumberOfVariables(pf) == 2 && 
	    GetConstantCoefficient(pf) == 0 &&
	    ((GetVariableCoefficient(pf, 0) == 1 &&
	      GetVariableCoefficient(pf, 1) == -1)) || 
	    ((GetVariableCoefficient(pf, 0) == -1 &&
	      GetVariableCoefficient(pf, 1) == 1));
    }

    static bool IsSingleVariableConstraint(const Expression& pf) {
	assert(IsPRAPolyform(pf));
	return GetNumberOfVariables(pf) == 1;
    }

    static bool isNumeric(const Expression& e, RATIONAL& value);

    /**
     *  Multiplies polyform with constant coefficient
     */
    static Expression Times(const Expression& pf, const RATIONAL& coefficient);

    /**
     *  Divides polyform with constant coefficient
     */     
    static Expression Divide(const Expression& pf, const RATIONAL& coefficient);


    /**
     *  Adds two polyforms
     */
    static Expression Plus(const Expression& pf_1, const Expression& pf_2);


    /**
     *  Subtracts two polyforms
     */
    static Expression Minus(const Expression& pf_1, const Expression& pf_2);

    /* Variable ordering */
    static int VariableCompare(const std::string& var1, const std::string& var2) {
	return var1.compare(var2);
    }

 private:
    /* Adds int constant to polyform */
    static void AddConstant  (const RATIONAL& coefficient, std::vector<Expression>& pfv);

    /* Adds coeff*var to polyform */
    static void   AddVariable  (const RATIONAL& coefficient, const Expression& variable, std::vector<Expression>& pfv);

    /* Adds arbitrary PRA term to polyform */
    static void   AddTerm  (const Expression& term, std::vector<Expression>& pfv);



    /* Forms num*var expression */
    static Expression CoefficientMultVariable(const RATIONAL& coefficient, const Expression& variable)
	{
	    return Expression::Function("*",
					Expression::RationalNumeral(coefficient),
					variable);
	}
};

#endif
