#include "Simplex.h"
#include "PRASignature.h"
#include "PRAPolyform.h"
#include "satsolver/Literal.h"
#include "auxiliary/BacktrackableStack_.h"
#include <algorithm>


Expression Simplex::cannonize(Expression e) {
//    coutput << "E: " << e << endl;
    Expression c = PRALiteral::Cannonize(e);
//    coutput << "C: " << c << endl;

    if (PRALiteral::IsGround(c))
	return c;

    if (isBound(c)) {
	Expression var = PRALiteral::GetMaximalVariable(c);
	_variableOrdering.add(var.GetName(), getVariableIndex(var.GetName()));
//	coutput << "RETURN: " << c << endl;
	return c;
    }
    
    Expression free = c[0][0];
    Expression left = c[0];
    left.RemoveOperand(0);

    Expression s = getFreshVariable(left);
    _variableOrdering.add(s.GetName(), getVariableIndex(s.GetName()));
    Expression eq = Expression::Equality(s, left);
    addEquality(eq);

    if (PRALiteral::IsVariableEquality(c) || 
	PRALiteral::IsVariableDisequality(c))
	addVariableEquality(s, c);

    Expression result = PRALiteral::FormExpression(
	c.GetName(),
	Expression::Function("+", free, 
			     Expression::Function("*", Expression::RationalNumeral(RATIONAL(1)), s)),
	Expression::RationalNumeral(RATIONAL(0)));

//    coutput << "RETURN: " << result << endl;
    return result;
}

void Simplex::addVariableEquality(const Expression& s, const Expression& e) {
    unsigned varIndex = getVariableIndex(s.GetName());
    if (_variableEqualitiesVarToPair.find(varIndex) == _variableEqualitiesVarToPair.end()) {
	unsigned x = getVariableIndex(PRALiteral::GetVariable(e, 0).GetName());
	unsigned y = getVariableIndex(PRALiteral::GetVariable(e, 1).GetName());
	_variableEqualitiesVarToPair[varIndex] =  std::pair<unsigned, unsigned>(x, y);
    }   
}

bool Simplex::isVariableEquality(unsigned varIndex, const RATIONAL& bound, Expression& var1, Expression& var2) {
    if (bound != 0)
	return false;

    std::map<unsigned, std::pair<unsigned, unsigned> >::const_iterator it = 
	_variableEqualitiesVarToPair.find(varIndex);
    if (it == _variableEqualitiesVarToPair.end())
	return false;
    
    var1 = Expression::Variable(getVariable(it->second.first));
    var2 = Expression::Variable(getVariable(it->second.second));
    return true;
}

void Simplex::convertEUFExplanation(std::vector<Expression>& explanation) {
    std::vector<Expression>::iterator it;
    for (it = explanation.begin(); it != explanation.end(); it++) {
	if ((*it)[1].IsNumeral()) {
	    (*it) = PRALiteral::Cannonize(*it);
	} else {
	    unsigned indexX = getVariableIndex((*it)[0].GetName());
	    unsigned indexY = getVariableIndex((*it)[1].GetName());
	    std::map<unsigned, std::pair<unsigned, unsigned> >::const_iterator jt;
	    for (jt = _variableEqualitiesVarToPair.begin(); jt != _variableEqualitiesVarToPair.end(); jt++) {
		if ((jt->second.first == indexX &&
		     jt->second.second == indexY) ||
		    (jt->second.first == indexY &&
		     jt->second.second == indexX)) {
		    static Expression zero = Expression::RationalNumeral(RATIONAL(0));
		    static Expression one = Expression::RationalNumeral(RATIONAL(1));
		    Expression left = Expression::Function(
			"+", zero, Expression::Function(
			    "*", one, Expression::Variable(getVariable(jt->first))));
		    if (it->IsEquality())
			(*it) = Expression::Equality(left, zero);
		    else
			(*it) = Expression::Disequality(left, zero);			
		    break;
		}
	    }
	}
    }
}

void Simplex::addExpression(Expression e) {
//    coutput << "ADD: " << e << endl;
    const RATIONAL& bound = -PRALiteral::GetConstantCoefficient(e);
    const std::string& var = PRALiteral::GetMaximalVariableName(e);
    unsigned varIndex = getVariableIndex(var);
    if (e.GetName() == LT) {
	_initialAtoms.push_back(CompiledConstraint(e, QDelta(bound, RATIONAL(-1)), BOUND_TYPE_LT, varIndex));
    } else if (e.GetName() == GT) {
	_initialAtoms.push_back(CompiledConstraint(e, QDelta(bound, RATIONAL(1)), BOUND_TYPE_GT, varIndex));
    } else if (e.GetName() == LEQ) {
	_initialAtoms.push_back(CompiledConstraint(e, bound, BOUND_TYPE_LEQ, varIndex));
    } else if (e.GetName() == GEQ) {
	_initialAtoms.push_back(CompiledConstraint(e, bound, BOUND_TYPE_GEQ, varIndex));
    } else if (e.GetName() == EQ) {
	_initialAtoms.push_back(CompiledConstraint(e, bound, BOUND_TYPE_EQ, varIndex));
	_euf.addExpression(Expression::Equality(Expression::Variable(var), Expression::RationalNumeral(bound)));
	Expression x, y;
	if (isVariableEquality(varIndex, bound, x, y))
	    _euf.addExpression(Expression::Equality(x, y));
    } else if (e.GetName() == DISEQ) {
	_initialAtoms.push_back(CompiledConstraint(e, bound, BOUND_TYPE_DISEQ, varIndex));
	_euf.addExpression(Expression::Disequality(Expression::Variable(var), Expression::RationalNumeral(bound)));
	Expression x, y;
	if (isVariableEquality(varIndex, bound, x, y))
	    _euf.addExpression(Expression::Disequality(x, y));
    } else {
	throw "Unexpected expression type";
    }
//    cout << "Bound: " << _initialAtoms.back()._bound << endl;
}

void Simplex::addEquality(const Expression& equality) {
    if (std::find(_initialEqualities.begin(), _initialEqualities.end(), equality) == _initialEqualities.end()) {
//	coutput << "Add equality: " << equality << endl;
	_initialEqualities.push_back(equality);
	_matrix.addRow();
	size_t i = _matrix.rows() - 1;
	const std::string& var = equality[0].GetName();
	_matrix.setCoefficient(i, getVariableIndex(var),  1);
	_matrix.setBasicVariable(i, getVariableIndex(var));
	for (size_t j = 0; j < equality[1].GetArity(); j++) {
	    const RATIONAL& b = equality[1][j][0].GetValueRational();
	    const std::string& var = equality[1][j][1].GetName();
	    _matrix.setCoefficient(i, getVariableIndex(var),  -b);
	    _matrix.addNonBasicVariable(i, getVariableIndex(var));
	}
    }
}

void Simplex::assertExpression(Expression e) {
//    coutput << "========================================" << endl;
//    coutput << "PRAAssert: " << e << endl;

    _currentLevel++;
    _euf.newLevel();
    
    const std::string& var = PRALiteral::GetMaximalVariableName(e);
    unsigned varIndex = getVariableIndex(var);

    RATIONAL new_bound = -PRALiteral::GetConstantCoefficient(e);

    if (e.GetName() == EQ) {
	if (strncmp(var.c_str(), _FRESH_VAR_NAME_, 4) != 0)
	    _euf.assertExpression(Expression::Equality(Expression::Variable(var), 
						       Expression::RationalNumeral(new_bound)));
	Expression x, y;
	if (isVariableEquality(varIndex, new_bound, x, y))
	    _euf.assertExpression(Expression::Equality(x, y));
    }

    if (e.GetName() == DISEQ) {
	if (strncmp(var.c_str(), _FRESH_VAR_NAME_, 4) != 0)
	    _euf.assertExpression(Expression::Disequality(Expression::Variable(var), 
							  Expression::RationalNumeral(new_bound)));	
	Expression x, y;
	if (isVariableEquality(varIndex, new_bound, x, y))
	    _euf.assertExpression(Expression::Disequality(x, y));

	_disequalities.push(e);
    }

    if (e.GetName() == LT ||  e.GetName() == LEQ || e.GetName() == EQ) {
	QDelta qd_new_bound = e.GetName() == LT ? QDelta(new_bound, RATIONAL(-1)) : new_bound;
//	cout << "ASSERT UPPER BOUND: " << var << " <= " << qd_new_bound << endl;
	if (GEQUpperBound(varIndex, qd_new_bound)) {
//	    coutput << "SAT" << endl;
	} else {
	    if (!GEQLowerBound(varIndex, qd_new_bound)) {
//		coutput << "UNSAT" << endl;
		_explanation.clear();
		_explanation.push_back(e);
		_explanation.push_back(_lowerBounds[varIndex]->peek()._expression);
		_unsat = true;
		return;
	    }
	
	    changeUpperBound(varIndex, qd_new_bound, e);
	    if (getValue(varIndex) > qd_new_bound && !isBasicVariable(varIndex)) {
		setValue(varIndex, qd_new_bound);
	    }
//	    fixValuation();
	    if (_unsat)
		return;
	}
    }

    if (e.GetName() == GT || e.GetName() == GEQ || e.GetName() == EQ) {
	QDelta qd_new_bound = e.GetName() == GT ? QDelta(new_bound, RATIONAL(1)) : new_bound;
//	cout << "ASSERT LOWER BOUND: " << var << " >= " << qd_new_bound << endl;

	if (LEQLowerBound(varIndex, qd_new_bound)) {
//	    coutput << "SAT" << endl;
	} else {
	    if (!LEQUpperBound(varIndex, qd_new_bound)) {
//		cout << "UNSAT" << endl;
		_explanation.clear();
		_explanation.push_back(e);
		_explanation.push_back(_upperBounds[varIndex]->peek()._expression);
		_unsat = true;
		return;
	    }

	    changeLowerBound(varIndex, qd_new_bound, e);
	    if (getValue(varIndex) < qd_new_bound && !isBasicVariable(varIndex)) {
		setValue(varIndex, qd_new_bound);
	    }
//	    fixValuation();
	    if (_unsat)
		return;
	}
    }

    if (_euf.isUnsat()) {
//	cout << "EUF unsat" << endl;
	_euf.explain(_explanation);
	convertEUFExplanation(_explanation);
	_unsat = true;
	return;
    }

}


bool Simplex::LEQLowerBound(unsigned var, const QDelta& value) {
    // hasLowerBound call inlined for efficency
    const Bound& bound = _lowerBounds[var]->peek();
    if (bound._infinite)
	return false;

    return value <= bound._value;
}

bool Simplex::GEQUpperBound(unsigned var, const QDelta& value) {
    // hasUpperBound call inlined for efficency
    const Bound& bound = _upperBounds[var]->peek();
    if (bound._infinite)
	return false;

    return value >= bound._value;
}

bool Simplex::LEQUpperBound(unsigned var, const QDelta& value) {
    if (!hasUpperBound(var))
	return true;

    return value <= _upperBounds[var]->peek()._value;
}

bool Simplex::GEQLowerBound(unsigned var, const QDelta& value) {
    if (!hasLowerBound(var))
	return true;

    return value >= _lowerBounds[var]->peek()._value;
}

bool Simplex::LTUpperBound(unsigned var, const QDelta& value) {
    if (!hasUpperBound(var))
	return true;

    return value < _upperBounds[var]->peek()._value;
}

bool Simplex::GTLowerBound(unsigned var, const QDelta& value) {
    if (!hasLowerBound(var))
	return true;

    return value > _lowerBounds[var]->peek()._value;
}

void Simplex::setValue(unsigned variableIndex, const QDelta& value) {
//    cout << "SET: " << variable << "  " << value << endl;
    /* 
       1*s0 + aij*x0
       1*s1 + aij*x1
       s1 - s0 + aij*(x1 - x0) - 0
       s1 = s0 - aij*(x1 - x0)
    */
    for (size_t i = 0; i < _matrix.rows(); i++) {
		const RATIONAL& aij = _matrix[i][variableIndex];
		unsigned basicVariableIndex = _matrix.getBasicVariable(i);
		if (aij == 0)
		    continue;
		if (aij == 1) {
		    _valuation[basicVariableIndex] += (_valuation[variableIndex] - value);
		} else if (aij == -1) {
		    _valuation[basicVariableIndex] += (value - _valuation[variableIndex]);
		} else {
		    _valuation[basicVariableIndex] += (_valuation[variableIndex] - value)*aij;
		}
		_boundChecked[basicVariableIndex] = false;
    }
    _valuation[variableIndex] = value;
    _boundChecked[variableIndex] = false;
}

void Simplex::fixValuation() {
//    cout << "FIX VALUATION" << endl;
    while(1) {
	Matrix::const_iterator i;
	for (i = _matrix.begin(); i != _matrix.end(); i++) {
	    unsigned variIndex = _matrix.getBasicVariable(*i);

	    if (_boundChecked[variIndex])
		continue;

	    std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j,
		beg  = _matrix.getNonBasicVariables(*i).begin(),
		en = _matrix.getNonBasicVariables(*i).end();
	    const QDelta& valuei = getValue(variIndex);

	    if (!GEQLowerBound(variIndex, valuei)) {
		const QDelta& li = getLowerBound(variIndex);
//		cout << "SHOULD FIX LOWER: " << getVariable(variIndex) << endl;
//		cout << "Value: " << valuei << "  Lower: " << li << endl;
		for (j = beg; j != en; j++) {
		    const RATIONAL&  aij = _matrix[*i][*j];
//		    cout << "aij: " << aij << " varj: " << getVariable(*j) << endl;
		    if ( aij < 0 && LTUpperBound(*j, getValue(*j)) ||
			 aij > 0 && GTLowerBound(*j, getValue(*j))) {
			pivot(variIndex, *j, li);
			break;
		    }
		}
		if (j == en) {
		    _explanation.clear();
		    _explanation.push_back(_lowerBounds[variIndex]->peek()._expression);
		    for (j = beg; j != en; j++) {
			const RATIONAL& aij = _matrix[*i][*j];
//			cout << "aij: " << aij << " varj: " << getVariable(*j) << endl;
			if (aij < 0) {
			    _explanation.push_back(_upperBounds[*j]->peek()._expression);
			} else if (aij > 0) {
			    _explanation.push_back(_lowerBounds[*j]->peek()._expression);
			}
		    }
		    _unsat = true;
//		    coutput << "UNSAT" << endl;
		    return;
		}
		break;
	    }

	    if (!LEQUpperBound(variIndex, valuei)) {
		const QDelta& ui = getUpperBound(variIndex);
//		cout << "SHOULD FIX UPPER: " << getVariable(variIndex) << endl;
//		cout << "Value: " << valuei << "  Upper: " << ui << endl;
		for (j = beg; j != en; j++) {
		    const RATIONAL& aij = _matrix[*i][*j];
//		    cout << "aij: " << aij << " varj: " << getVariable(*j) << endl;
		    if (aij > 0 && LTUpperBound(*j, getValue(*j)) ||
			aij < 0 && GTLowerBound(*j, getValue(*j))) {
			pivot(variIndex, *j, ui);
			break;
		    }
		}
		if (j == en) {
		    _explanation.clear();
		    _explanation.push_back(_upperBounds[variIndex]->peek()._expression);
		    for (j = beg; j != en; j++) {
			const RATIONAL& aij = _matrix[*i][*j];
//			cout << "aij: " << aij << " varj: " << getVariable(*j) << endl;
			if (aij > 0) {
			    _explanation.push_back(_upperBounds[*j]->peek()._expression);
			} else if (aij < 0) {
			    _explanation.push_back(_lowerBounds[*j]->peek()._expression);
			}
		    }
		    _unsat = true;
//		    coutput << "UNSAT" << endl;
		    return;
		}
		break;
	    }

	    _boundChecked[variIndex] = true;
	}
	if (i == _matrix.end())
	    break;
    }
}

void Simplex::forceCheck(bool strong) {
//    cout << "CHECK FORCED" << endl;
    _euf.forceCheck(strong);
    if (_euf.isUnsat()) {
	_euf.explain(_explanation);
	convertEUFExplanation(_explanation);
	_unsat = true;
	return;
    }

    fixValuation();
    if (_unsat)
	return;

    if (!strong)
	return;

    BacktrackableStack<Expression>::const_iterator i;
    for (i = _disequalities.begin(); i != _disequalities.end(); ++i) {
//	coutput << *i << endl;
	std::vector<Expression> explanation;
	Expression lt = Expression::Predicate("<", (*i)[0], (*i)[1]);
	bool ltUnsat, gtUnsat;
//	coutput << "Asert: " << lt << endl;
	assertExpression(lt);
	fixValuation();
	ltUnsat = _unsat;
	backtrack();
	commitBacktrack();
	if (ltUnsat) {
//	    cout << "LT UNSAT" << endl;
	    explanation.insert(explanation.end(), _explanation.begin(), _explanation.end());
	}

	Expression gt = Expression::Predicate(">", (*i)[0], (*i)[1]);
//	coutput << "Asert: " << gt << endl;
	assertExpression(gt);
	fixValuation();
	gtUnsat = _unsat;
	backtrack();
	commitBacktrack();
	if (gtUnsat) {
//	    cout << "GT UNSAT" << endl;
	    explanation.insert(explanation.end(), _explanation.begin(), _explanation.end());
	}

	if (ltUnsat && gtUnsat) {
//	    coutput << "diseq: " << *i << endl;
	    _explanation.clear();
	    _explanation.push_back(*i);
	    std::vector<Expression>::const_iterator i;
	    for (i = explanation.begin(); i != explanation.end(); i++) {
		if (*i != lt && *i != gt) {
		    if (find(_explanation.begin(), _explanation.end(), *i) == _explanation.end()) {
//			coutput << *i << endl;
			_explanation.push_back(*i);
		    }
		}
	    }
	    _unsat = true;
//	    cout << "UNSAT" << endl;
	    break;
	}
    }
}

void Simplex::backtrack() {
//    coutput << "Backtrack..." << endl;
    _currentLevel--;

    _euf.backtrack();
    _unsat = false;
}

std::string toString_(const Expression& e) {
    return e.toString();
}

void Simplex::commitBacktrack() {
    std::vector< BacktrackableStack<Bound>* >::iterator i, 
	beg = _upperBounds.begin(), en = _upperBounds.end();
    for (i = beg; i != en; i++)
	(*i)->backtrack();
    beg = _lowerBounds.begin(), en = _lowerBounds.end();
    for (i = beg; i != en; i++)
	(*i)->backtrack();
    _disequalities.backtrack();

    _euf.commitBacktrack();
}

void Simplex::pivot(unsigned varsIndex, unsigned varxIndex, const QDelta& new_s_value) {
//    coutput << "PIVOT: " << getVariable(varsIndex) << " " << getVariable(varxIndex) << " " << new_s_value << endl;
    unsigned eq = _matrix.getEqualityForVariable(varsIndex);

/*
  1*s0 + aij*x0
  1*s1 + aij*x1
  (s1-s0) + aij*(x1 - x0) = 0
  x1 = x0 + 1/aij*(s0 - s1)
  theta = x1 - x0

  1*sk0 + akj * x0
  1*sk1 + akj * x1
  (sk1 - sk0) + akj*(x1 - x0)
  sk1 = sk0 + akj(x0 - x1)
  sk1 = sk0 - akj*theta
*/

    const RATIONAL& aij = _matrix[eq][varxIndex];
    QDelta theta;
    if (aij == 1) {
	theta = (_valuation[varsIndex] - new_s_value);
    } else if (aij == -1) {
	theta = (new_s_value - _valuation[varsIndex]);
    } else {
	RATIONAL c = 1/aij;
	c.canonicalize();
	theta = (_valuation[varsIndex] - new_s_value)*c;
    }
    _valuation[varsIndex] = new_s_value;
    _valuation[varxIndex] += theta;
    _boundChecked[varsIndex] = false;
    _boundChecked[varxIndex] = false;

    for (size_t eqIndex = 0; eqIndex < _matrix.rows(); eqIndex++) {
	if (eqIndex == eq)
	    continue;
	
	const RATIONAL& akj = _matrix[eqIndex][varxIndex];
	if (akj == 0)
	    continue;
	
	unsigned basicVariable = _matrix.getBasicVariable(eqIndex);
	if (akj == 1) {
	    _valuation[basicVariable] -= theta;
	} else if (akj == -1) {
	    _valuation[basicVariable] += theta;
	} else {
	    _valuation[basicVariable] -= theta*akj;
	}
	_boundChecked[basicVariable] = false;
    }

    _matrix.pivot(eq, varxIndex);
//    _matrix.print();
}


bool Simplex::refineBound(unsigned eq, Bound& lower, Bound& upper) {
/*
  1*s = -a * y
  a>0
  y >= l
  -a*y <= -a*l
  y <= u
  -a*y >= -a*u
  a<0
  y >= l
  -a*y >= -a*l
  y <= u
  -a*y <= -a*u
*/
    std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j, beg, en;
    lower._infinite = false; upper._infinite = false;
    bool old = true;

    beg  = _matrix.getNonBasicVariables(eq).begin();
    en = _matrix.getNonBasicVariables(eq).end();
    for (j = beg; j != en; j++)
	if (!_boundChecked[*j])
	    old = false;
    if (old)
	return false;

    beg  = _matrix.getNonBasicVariables(eq).begin();
    en = _matrix.getNonBasicVariables(eq).end();
    for (j = beg; j != en; j++) {
	const RATIONAL& aij = _matrix[eq][*j];
	if (aij < 0) {
	    if (!hasUpperBound(*j)) {
		upper._infinite = true;
		break;
	    }
	} else {
	    if (!hasLowerBound(*j)) {
		upper._infinite = true;
		break;
	    }
	}
    }

    beg  = _matrix.getNonBasicVariables(eq).begin();
    en = _matrix.getNonBasicVariables(eq).end();
    for (j = beg; j != en; j++) {
	const RATIONAL& aij = _matrix[eq][*j];
	if (aij < 0) {
	    if (!hasLowerBound(*j)) {
		lower._infinite = true;
		break;
	    }
	} else {
	    if (!hasUpperBound(*j)) {
		lower._infinite = true;
		break;
	    }
	}
    }

    if (lower._infinite && upper._infinite)
	return false;

    lower._value.reset(); upper._value.reset();
    for (j = beg; j != en; j++) {
	const RATIONAL& aij = _matrix[eq][*j];
	if (aij < 0) {
	    if (!upper._infinite) {
		assert(hasUpperBound(*j));
		if (aij == -1)
		    upper._value += getUpperBound(*j);
		else
		    upper._value -= getUpperBound(*j)*aij;
	    }

	    if (!lower._infinite) {
		assert(hasLowerBound(*j));
		if (aij == -1)
		    lower._value += getLowerBound(*j);
		else
		    lower._value -= getLowerBound(*j)*aij;
	    }
	} else {
	    if (!lower._infinite) {
		assert(hasUpperBound(*j));
		if (aij == 1)
		    lower._value -= getUpperBound(*j);
		else
		    lower._value -= getUpperBound(*j)*aij;
	    }
	    if (!upper._infinite) {
		assert (hasLowerBound(*j));
		if (aij == 1)
		    upper._value -= getLowerBound(*j);
		else
		    upper._value -= getLowerBound(*j)*aij;
	    }
	}
    }

    bool result = false;
    unsigned varIndex = _matrix.getBasicVariable(eq);
    if (!upper._infinite && LTUpperBound(varIndex, upper._value)) {
/*
	if (hasUpperBound(variIndex))
	    cout << "OLD BOUND: " << getVariable(variIndex) << " <= " << getUpperBound(variIndex) << endl;
	else
	    cout << "OLD BOUND: " << getVariable(variIndex) << " <= " << "+Inf" << endl;

	cout << "NEW BOUND: " << getVariable(variIndex) << " <= " << upper << endl;	
*/
	result = true;
    }
    if (!lower._infinite && GTLowerBound(varIndex, lower._value)) {
/*
	if (hasLowerBound(variIndex))
	    cout << "OLD BOUND: " << getVariable(variIndex) << " >= " << getLowerBound(variIndex) << endl;
	else
	    cout << "OLD BOUND: " << getVariable(variIndex) << " >= " << "-Inf" << endl;
	cout << "NEW BOUND: " << getVariable(variIndex) << " >= " << lower << endl;
*/
	result = true;
    }

    return result;
}

bool Simplex::theoryPropagate(std::vector<Expression>& explanation) {
//    print();
//    cout << "Theory propagate" << endl;
    if (_euf.theoryPropagate(explanation)) {
	convertEUFExplanation(explanation);
	return true;
    }

    std::vector<CompiledConstraint>::const_iterator i,
	beg = _initialAtoms.begin(), en = _initialAtoms.end();
    for (i = beg; i != en; i++) {
	if (i->_expression.assigned()) {
//	    coutput << "ASSIGNED: " << i->_expression << endl;
	    continue;
	}
//	coutput << "CHECK: " << i->_expression << endl;

	//-----------------------------------------------------
	if (i->_boundType == BOUND_TYPE_LT || 
	    i->_boundType == BOUND_TYPE_LEQ) {
	    if (GEQUpperBound(i->_variable, i->_bound)) {
//		coutput << "PRA: " << _upperBounds[i->_variable]->peek()._expression << " --> " << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		explanation.push_back(_upperBounds[i->_variable]->peek()._expression);
		return true;
	    }
	    //-----------------------------------------------------
	} else if (i->_boundType == BOUND_TYPE_GT || 
		   i->_boundType == BOUND_TYPE_GEQ) {
	    if (LEQLowerBound(i->_variable, i->_bound)) {
//		coutput << "PRA: " << _lowerBounds[i->_variable]->peek()._expression << " --> " << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		explanation.push_back(_lowerBounds[i->_variable]->peek()._expression);
		return true;
	    }
	    //-----------------------------------------------------
	} else if (i->_boundType == BOUND_TYPE_DISEQ) {
	    //-----------------------------------------------------
//	    x >= l > a
	    if (hasLowerBound(i->_variable) && i->_bound < getLowerBound(i->_variable)) {
//		coutput << "PRA: " << _lowerBounds[i->_variable]->peek()._expression << " --> " << i->_expression << endl;

		explanation.clear();
		explanation.push_back(i->_expression);
		explanation.push_back(_lowerBounds[i->_variable]->peek()._expression);
		return true;
	    }
//          x <= u < a
	    if (hasUpperBound(i->_variable) && i->_bound > getUpperBound(i->_variable)) {
//		coutput << "PRA: " << _upperBounds[i->_variable]->peek()._expression << " --> " << i->_expression << endl;

		explanation.clear();
		explanation.push_back(i->_expression);
		explanation.push_back(_upperBounds[i->_variable]->peek()._expression);
		return true;
	    }
	} else if (i->_boundType == BOUND_TYPE_EQ) {
	    // -------------------------------------------------------------
	    if (isFixed(i->_variable)) {
		const QDelta& b = getLowerBound(i->_variable);
		if (i->_bound == b) {
		    explanation.clear();
		    explanation.push_back(i->_expression);
		    explanation.push_back(_lowerBounds[i->_variable]->peek()._expression);
		    explanation.push_back(_upperBounds[i->_variable]->peek()._expression);

//		    coutput << "PRA: " << _lowerBounds[i->_variable]->peek()._expression << " & " 
//			    << _upperBounds[i->_variable]->peek()._expression << " --> " << i->_expression << endl;
		    return true;
		}
	    }
	}
    }


    _refined.assign(_refined.size(), false);

    Bound refinedLower, refinedUpper;
    beg = _initialAtoms.begin(), en = _initialAtoms.end();
    for (i = beg; i != en; i++) {
	if (i->_expression.assigned()) {
//	    coutput << "ASSIGNED: " << i->_expression << endl;
	    continue;
	}

	if (!_matrix.isBasicVariable(i->_variable))
	    continue;

	if (_refined[i->_variable]) {
	    refinedLower = _refinedLowerBounds[i->_variable];
	    refinedUpper = _refinedUpperBounds[i->_variable];
	} else {
	    if (!refineBound(_matrix.getEqualityForVariable(i->_variable), refinedLower, refinedUpper)) {
		continue;
	    }
	    _refined[i->_variable] = true;
	    _refinedLowerBounds[i->_variable] = refinedLower;
	    _refinedUpperBounds[i->_variable] = refinedUpper;
	}

	
	if (i->_boundType == BOUND_TYPE_LT || 
	    i->_boundType == BOUND_TYPE_LEQ) {
	    if (!refinedUpper._infinite && i->_bound >= refinedUpper._value) {
//		cout << "PRA REFINE: " <<  getVariable(i->_variable) << " <= " << refinedUpper._value << " --> ";
//		coutput << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		unsigned eq = _matrix.getEqualityForVariable(i->_variable);
		std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j,
		    beg  = _matrix.getNonBasicVariables(eq).begin(),
		    en = _matrix.getNonBasicVariables(eq).end();
		for (j = beg; j != en; j++) {
		    const RATIONAL& aij = _matrix[eq][*j];
		    if (aij < 0)
			explanation.push_back(_upperBounds[*j]->peek()._expression);
		    else
			explanation.push_back(_lowerBounds[*j]->peek()._expression);
		}
		return true;
	    }
	}	    
	else if (i->_boundType == BOUND_TYPE_GT || 
		 i->_boundType == BOUND_TYPE_GEQ) {
	    if (!refinedLower._infinite && i->_bound <= refinedLower._value) {
//		cout << "PRA REFINE: " <<  getVariable(i->_variable) << " >= " << refinedLower._value << " --> ";
//		coutput << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		unsigned eq = _matrix.getEqualityForVariable(i->_variable);
		std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j,
		    beg  = _matrix.getNonBasicVariables(eq).begin(),
		    en = _matrix.getNonBasicVariables(eq).end();
		for (j = beg; j != en; j++) {
		    const RATIONAL& aij = _matrix[eq][*j];
		    if (aij < 0)
			explanation.push_back(_lowerBounds[*j]->peek()._expression);
		    else
			explanation.push_back(_upperBounds[*j]->peek()._expression);
		}
		return true;
	    }
	} else if (i->_boundType == BOUND_TYPE_DISEQ) {
	    if (!refinedLower._infinite && i->_bound < refinedLower._value) {
//		cout << "PRA REFINE: " <<  getVariable(i->_variable) << " >= " << refinedLower._value << " --> ";
//		coutput << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		unsigned eq = _matrix.getEqualityForVariable(i->_variable);
		std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j,
		    beg  = _matrix.getNonBasicVariables(eq).begin(),
		    en = _matrix.getNonBasicVariables(eq).end();
		for (j = beg; j != en; j++) {
		    const RATIONAL& aij = _matrix[eq][*j];
		    if (aij < 0)
			explanation.push_back(_lowerBounds[*j]->peek()._expression);
		    else
			explanation.push_back(_upperBounds[*j]->peek()._expression);
		}
		return true;
	    }
	    if (!refinedUpper._infinite && i->_bound > refinedUpper._value) {
//		cout << "PRA REFINE: " <<  getVariable(i->_variable) << " <= " << refinedUpper._value << " --> ";
//		coutput << i->_expression << endl;
		explanation.clear();
		explanation.push_back(i->_expression);
		unsigned eq = _matrix.getEqualityForVariable(i->_variable);
		std::set<unsigned, Matrix::NonBasicVariableOrder>::const_iterator j,
		    beg  = _matrix.getNonBasicVariables(eq).begin(),
		    en = _matrix.getNonBasicVariables(eq).end();
		for (j = beg; j != en; j++) {
		    const RATIONAL& aij = _matrix[eq][*j];
		    if (aij < 0)
			explanation.push_back(_upperBounds[*j]->peek()._expression);
		    else
			explanation.push_back(_lowerBounds[*j]->peek()._expression);
		}
		return true;
	    }
	}
    }

    return false;
}

bool Simplex::generateModelDisequalities(unsigned n) {
    if (n >= _disequalities.size())
	return true;
    const Expression& disequality = _disequalities[n];
    Expression lt = Expression::Predicate("<", disequality[0], disequality[1]);
    assertExpression(lt);
    fixValuation();
    if (_unsat) {
	backtrack();
	commitBacktrack();
	Expression gt = Expression::Predicate(">", disequality[0], disequality[1]);
	assertExpression(gt);
	fixValuation();
	if (_unsat)
	    return false;
    }

    return generateModelDisequalities(n+1);
}

bool Simplex::generateModel() {
//    print();
//    _matrix.print();

    if (!generateModelDisequalities(0))
	return false;
    RATIONAL delta0 = 1;
    for (size_t i = 0; i < _valuation.size(); i++) {
//	cout << "VAL: " << i->first << " : " << i->second << endl;
	const RATIONAL& pj = _valuation[i].getA();
	const RATIONAL& qj = _valuation[i].getB();
	if (hasLowerBound(i)) {
	    const QDelta& lower = getLowerBound(i);
//	    coutput << "Lower: " <<  lower << endl;
	    const RATIONAL& cj = lower.getA();
	    const RATIONAL& kj = lower.getB();
//	    cout << "Compare: " << cj << ", " << kj << " vs " << pj << ", " << qj << endl;
	    if (cj < pj && kj > qj) {
		RATIONAL delta = (pj - cj)/(kj - qj);
		delta.canonicalize();
		if (delta0 > delta)
		    delta0 = delta;
	    }
	}
	if (hasUpperBound(i)) {
	    const QDelta& upper = getUpperBound(i);
	    
//	    coutput << "Upper: " << upper << endl;
	    const RATIONAL& dj = upper.getA();
	    const RATIONAL& hj = upper.getB();
//	    cout << "Compare: " << pj << ", " << qj << " vs " << dj << ", " << hj << endl;
	    if (pj < dj && qj > hj) {
		RATIONAL delta = (dj - pj)/(qj - hj);
		delta.canonicalize();
		if (delta0 > delta)
		    delta0 = delta;
	    }
	}
    }
//    coutput << "DELTA0: " << delta0 << endl;

    for (size_t i = 0; i < _valuation.size(); i++) {
	_model[getVariable(i)] = _valuation[i].toRational(delta0);
//	cout << getVariable(i) << " : " << _model[getVariable(i)] << endl;
    }

    std::vector<Expression>::const_iterator ei;
    for (ei = _initialEqualities.begin(); ei != _initialEqualities.end(); ei++) {
//	coutput << "Checking: " << *ei << endl;
	std::map<std::string, RATIONAL>::const_iterator it = 
	    _model.find((*ei)[0].GetName());
        RATIONAL el = it == _model.end() ? 0 : it->second;
	RATIONAL er = 0;
	for (size_t i = 0; i < (*ei)[1].GetArity(); i++) {
	    std::map<std::string, RATIONAL>::const_iterator it = 
		_model.find((*ei)[1][i][1].GetName());
	    er += (*ei)[1][i][0].GetValueRational() * (it == _model.end() ? 0 : it->second);
	}
	if (el != er) {
	    coutput << "WRONG EQUALITY: " << *ei << endl;
	    return false;
	}
    }
//    _matrix.check(_model);
	
    return true;
}

bool Simplex::checkAgainstModel(Expression e) {
//    coutput << "Checking: " << e << endl;
    const Expression& left = e[0];
    RATIONAL rl = PRAPolyform::GetConstantCoefficient(left);
    for (size_t i = 0; i < PRAPolyform::GetNumberOfVariables(left); i++) {
	std::map<std::string, RATIONAL>::const_iterator it = 
	    _model.find(PRAPolyform::GetVariable(left, i).GetName());
	rl += PRAPolyform::GetVariableCoefficient(left, i) * (it == _model.end() ? 0 : it->second);
    }
    RATIONAL rr = 0;
//    cout << "RL: " << rl << endl;
//    cout << "RR: " << rr << endl;
    if (e.GetName() == EQ) {
	return rl == rr;
    } else if (e.GetName() == LT) {
	return rl < rr; 
    } else if (e.GetName() == LEQ) {
	return rl <= rr; 
    } else if (e.GetName() == GT) {
	return rl > rr; 
    } else if (e.GetName() == GEQ) {
	return rl >= rr; 
    } else if (e.GetName() == DISEQ) {
	return rl != rr; 
    }

    return false;
}
