#ifndef __SIMPLEX_H__
#define __SIMPLEX_H__

#include "auxiliary/BacktrackableStack_.h"
#include "auxiliary/BacktrackableUnionFind.h"
#include "theory/Theory.h"
#include "theory/pra/PRALiteral.h"
#include "theory/pra/QDelta.h"
#include "theory/pra/Matrix.h"
#include "theory/euf/EUF.h"
#include <string>
#include <map>
#include <set>
#include <sstream>

#define _FRESH_VAR_NAME_ "#_s_"

class Simplex : public Theory {
 public:
    Simplex()
	: _currentLevel(0), 
	_freshVariableIndex(0), 
	_matrix(_variables),
	_disequalities(_currentLevel) {
	_unsat = false;
	registerTheory(this);
    }

    void init() {
	_matrix.sort();
    }

    void addExpression(Expression e);

    void newDecision() {
    }

    void backtrackDecision() {
    }

    void assertExpression(Expression e);

    void backtrack();
    void commitBacktrack();
    void explain(std::vector<Expression>& explanation) {
	explanation = _explanation;
    }

    bool theoryPropagate(std::vector<Expression>& explanation);

    bool isUnsat() {
	return _unsat;
    }


    std::map<std::string, RATIONAL> _model;
    bool generateModel();
    bool generateModelDisequalities(unsigned n);

    bool checkAgainstModel(Expression e);

    Expression cannonize(Expression e);

    Expression negate(Expression e) {
	return PRALiteral::Negate(e);
    }

    bool isTrue(Expression e) {
	return PRALiteral::IsGround(e) && PRALiteral::TruthValue(e) == true;
    }

    bool isFalse(Expression e) {
	return PRALiteral::IsGround(e) && PRALiteral::TruthValue(e) == false;
    }

    void print() {
//	_matrix.print();
	printEqualities();
	printBounds();
	printValuation();
    }
 private:
    /***************************************************************/
    void pivot(unsigned varsIndex, unsigned varxIndex, const QDelta& new_value);
    std::vector<Expression> _initialEqualities;

    bool isBasicVariable(unsigned varIndex) {
	return _matrix.isBasicVariable(varIndex);
    }

    void addEquality(const Expression& equality);

    void printEqualities() {
	coutput << "-----------EQ------------" << endl;
	std::vector<Expression>::const_iterator i;
	for (i = _initialEqualities.begin(); i != _initialEqualities.end(); i++)
	    coutput << *i << endl;
	coutput << "-------------------------" << endl;
    }
    /***************************************************************/

 public:
    struct Bound {
	Bound() {}
	Bound(QDelta value, Expression expression, bool infinite)
	    : _value(value), _expression(expression), _infinite(infinite) {
	}
	bool operator==(const Bound& b) {
	    return _value == b._value && _infinite == b._infinite;
	}
	bool operator!=(const Bound& b) {
	    return !(*this == b);
	}
	QDelta _value;
	Expression _expression;
	bool _infinite;
    };

 private:
    unsigned _currentLevel;
    std::vector< BacktrackableStack<Bound>* > _lowerBounds;
    std::vector< BacktrackableStack<Bound>* > _upperBounds;
    std::vector<bool> _boundChecked;
    

    bool GEQLowerBound(unsigned var, const QDelta& value);
    bool LEQLowerBound(unsigned var, const QDelta& value);
    bool GTLowerBound(unsigned var, const QDelta& value);

    bool LEQUpperBound(unsigned var, const QDelta& value);
    bool GEQUpperBound(unsigned var, const QDelta& value);
    bool LTUpperBound(unsigned var, const QDelta& value);



    bool hasLowerBound(unsigned var) {
	return !_lowerBounds[var]->peek()._infinite;
    }

    const QDelta& getLowerBound(unsigned var) {
	return _lowerBounds[var]->peek()._value;
    }

    bool hasUpperBound(unsigned var) {
	return !_upperBounds[var]->peek()._infinite;
    }

    const QDelta& getUpperBound(unsigned var) {
	return _upperBounds[var]->peek()._value;
    }

    void changeLowerBound(unsigned var, const QDelta& bound, const Expression& boundExpression) {
	_lowerBounds[var]->push(Bound(bound, boundExpression, false));
	_boundChecked[var] = false;
    }

    void changeUpperBound(unsigned var, const QDelta& bound, const Expression& boundExpression) {
	_upperBounds[var]->push(Bound(bound, boundExpression, false));
	_boundChecked[var] = false;
    }

    bool isFixed(unsigned var) {
	return hasLowerBound(var) && hasUpperBound(var) && getLowerBound(var) == getUpperBound(var);
    }

    void printBounds() {
	size_t i;
	for (i = 0; i < _valuation.size(); i++) {
	    if (_lowerBounds[i]->peek()._infinite && _upperBounds[i]->peek()._infinite)
		continue;

	    if (_lowerBounds[i]->peek()._infinite) {
		coutput << "-Inf";
	    } else {
		coutput << _lowerBounds[i]->peek()._value;
//		coutput << _lowerBoundsExpressions[i->first]->peek();
	    }
	    coutput << "\t" << getVariable(i) << "\t";
	    if (_upperBounds[i]->peek()._infinite) {
		coutput << "+Inf";
	    } else {
		coutput << _upperBounds[i]->peek()._value;
//		coutput << _upperBoundsExpressions[i->first]->peek();
	    }
	    coutput << endl;
	}
    }

    
    bool refineBound(unsigned var, Bound& l, Bound& u);
    std::vector<bool> _refined;
    std::vector<Bound> _refinedLowerBounds;
    std::vector<Bound> _refinedUpperBounds;

    /***************************************************************/
    const QDelta& getValue(unsigned variableIndex) {
	return _valuation[variableIndex];
    }

    void setValue(unsigned variableIndex, const QDelta& value);

    void fixValuation();

    void forceCheck(bool total);

    void printValuation() {
	std::vector<QDelta>::const_iterator i;
	for (i = _valuation.begin(); i != _valuation.end(); i++)
	    cout << getVariable(i-_valuation.begin()) << " : " << *i << endl;
    }

    std::vector<QDelta> _valuation;

    /***************************************************************/
    size_t _freshVariableIndex;
    std::map<Expression, std::string> _freshVariables;
    Expression getFreshVariable(Expression e) {
	std::map<Expression, std::string>::const_iterator i = _freshVariables.find(e);
	if (i == _freshVariables.end()) {
	    std::ostringstream stream;
	    stream << _FRESH_VAR_NAME_ << _freshVariableIndex++;
	    _freshVariables[e] = stream.str();
	    return Expression::Variable(_freshVariables[e]);
	} else {
	    return Expression::Variable(i->second);
	}
    }

    /***************************************************************/
    bool isBound(Expression e) {
//	assert(PRALiteral::isCannonized(e));
	return PRALiteral::IsSingleVariableConstraint(e);
    }

    Expression formBoundsExpression(const std::string& type, const std::string& var, const RATIONAL& value);

    /***************************************************************/
    std::vector<Expression> _explanation;
    bool _unsat;

    /**************************************************************/
    struct CompiledConstraint {
	CompiledConstraint(Expression expression, QDelta bound, char boundType, unsigned variable) 
	    : _expression(expression), _bound(bound), _boundType(boundType), _variable(variable)  {
	}

	Expression _expression;
	QDelta _bound;
	char _boundType;
	unsigned _variable;
    };

    std::vector<CompiledConstraint> _initialAtoms;
    /*****************************************************************/
    Matrix _matrix;

    std::vector<std::string> _variables;
    std::map<std::string, unsigned> _variableIndex;


    void addVariable(const std::string& var) {
	size_t varIndex = _variables.size();
	_variableIndex[var] = varIndex;
	_variables.push_back(var);
	_valuation.push_back(QDelta(0));
	_lowerBounds.push_back(new BacktrackableStack<Bound> (_currentLevel));
	_upperBounds.push_back(new BacktrackableStack<Bound> (_currentLevel));
	_lowerBounds[varIndex]->push(Bound(QDelta(0), Expression(), true));
	_upperBounds[varIndex]->push(Bound(QDelta(0), Expression(), true));
	_boundChecked.push_back(true);
	_refined.push_back(true);
	_refinedLowerBounds.push_back(Bound(QDelta(0), Expression(), true));
	_refinedUpperBounds.push_back(Bound(QDelta(0), Expression(), true));
	_matrix.addCol();
    }

    unsigned getVariableIndex(const std::string& var) {
	std::map<std::string, unsigned>::const_iterator i = 
	    _variableIndex.find(var);
	if (i == _variableIndex.end()) {
	    addVariable(var);
	    i = _variableIndex.find(var);
	}
	return i->second;
    }

    bool variableDefined(const std::string& var) {
	return _variableIndex.find(var) != _variableIndex.end();
    }

    const std::string& getVariable(unsigned i) {
	return _variables[i];
    }

    //     s0 -> <x, y>
    std::map< unsigned, std::pair<unsigned, unsigned> > _variableEqualitiesVarToPair;
    //     <x, y> -> s0
    std::map< std::pair<unsigned, unsigned>, unsigned > _variableEqualitiesPairToVar;
    void addVariableEquality(const Expression& s, const Expression& e);
    bool isVariableEquality(unsigned varIndex, const RATIONAL& bound, Expression& var1, Expression& var2);
    void convertEUFExplanation(std::vector<Expression>& explanation);

    BacktrackableStack<Expression> _disequalities;

    EUF _euf;
};

#define BOUND_TYPE_LT  0
#define BOUND_TYPE_LEQ 1
#define BOUND_TYPE_GT  2
#define BOUND_TYPE_GEQ 3
#define BOUND_TYPE_EQ  4
#define BOUND_TYPE_DISEQ  5

#endif
