/******************************************************************************
 * Copyright (C) 2007-2009. Filip Maric, Predrag Janicic
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
 ******************************************************************************/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstring>

#include <iostream>
#include <fstream>

#include "Auxiliary.hpp"
#include "Clock.hpp"
#include "Solver.hpp"
#include "SolverStatistics.hpp"
#include "ProblemGenerator.hpp"

#include "LoggingSolverListener.hpp"
#include "TimerSolverListener.hpp"
#include "TimeoutSolverListener.hpp"

using namespace ArgoSat;


std::string copyrightString = 
"------------------------------------------------------------------------\n"
"ArgoSat v. 1.0\n"
"Build date: " BUILDDATE "\n" 
"Copyright (C) 2007-2009\n"
"Filip Maric, Predrag Janicic, Faculty of Mathematics, Belgrade\n"
"------------------------------------------------------------------------\n"
"Inspired by MiniSat (C) Een, Sorensson, 2003-2006\n\n"
"This is free software.  You may redistribute copies of it under the terms of\n"
"the GNU General Public License <http://www.gnu.org/licenses/gpl.html>.\n"
"There is NO WARRANTY, to the extent permitted by law.\n";



std::string usageString = 
"Usage: argosat problem [options]\n"
"    problem:\n"
"          pigeons num - pigeonhole problem\n"
"          queens  num - n queens problem\n"
"          filename    - file that contains uncompressed DIMACS format input\n"
"    options:\n";

std::ostream* modelStream = &cout;
bool singleModel = true;
std::string decisionVariablesFileName = "";
bool verifyModel = false;


void parseCmdLine(int argc, char* argv[], Solver* solver) {
  for (int curr = 0; curr < argc; curr++) {
    if (strcmp(argv[curr], "--seed") == 0) {
      if (!(curr+1 < argc) || !isInt(argv[curr+1]))
	throw std::string("Seed number expected");
      srand(atoi(argv[curr+1]));
    } else if (strcmp(argv[curr], "--model") == 0) {
      if (!(curr+1 < argc))
	throw std::string("File name expected");
      modelStream = new std::ofstream(argv[curr+1]);
    } else if (strcmp(argv[curr], "--decision_variables") == 0) {
      if (!(curr+1 < argc))
	throw std::string("File name expected");
      decisionVariablesFileName = argv[curr+1];
    } else if (strcmp(argv[curr], "--all_models") == 0) {
      singleModel = false;
    } else if (strcmp(argv[curr], "--verify") == 0) {
      verifyModel = true;
    } else if (strcmp(argv[curr], "--timeout") == 0) {
      curr++;
      if (curr >= argc || !isInt(argv[curr]))
	throw std::string("Timout in seconds expected");
      size_t t = abs(atoi(argv[curr++]));
      solver->addListener(new TimeoutSolverListener(t));
    } else if (strcmp(argv[curr], "--verbose") == 0) {
      curr++;
      solver->addListener(new LoggingSolverListener(*solver, curr < argc && strcmp(argv[curr], "propagation") == 0));

    }
  }
}

  /* Entry point of the solver */
int main(int argc, char* argv[]) {
  Clock totalTime;
  cout << copyrightString << endl;
  srand(0);

  if (argc < 2) {
    cerr << usageString << endl;
    exit(EXIT_FAILURE);
  }

  Clock creationTime;
  Solver* solver;
  ProblemGenerator* generator;
  try {
    solver = Solver::createFromCmdLine(argc, argv);
    parseCmdLine(argc, argv, solver);
    generator = ProblemGenerator::createFromCmdLine(argc, argv);
  } catch (std::string s) {
    cerr << s << endl;
    exit(EXIT_FAILURE);
  }
  generator->generateProblem(solver);
  solver->addListener(new DotLoggingSolverListener(*solver));

  /*
  // Allow that some variables are not decision variables
  if (decisionVariablesFileName != "") {
  cout << "\nReading decision variables from file: " << decisionVariablesFileName << endl;
  cout << "Warning: this could affect the soundness" << endl;
  solver->getVariables().setDecisionVariablesFromFile(decisionVariablesFileName);
  }
  */

  cout << endl << "Problem created in " << creationTime.elapsed() << "s" << endl;

  SolverStatistics* stats = new SolverStatistics(*solver);
  Clock solveTime;
  int numOfModels = 0;

  while(true) {
    stats->printHeader();
    stats->printStatistics();
    solver->solve();
    stats->printFooter();

    if (solver->isUnsatisfiable()) {
      cout << "Formula found unsatisfiable" << endl;
      break;
    }

    Valuation model = solver->getModel();
    cout << "Model: " << ++numOfModels << endl;
    cout << "Found in: " << solveTime.elapsed() << "s" << endl;
    cout << "------------------------------------------------------------------------------" << endl;
    model.print(*modelStream);
    cout << "------------------------------------------------------------------------------" << endl;

    if (verifyModel)
      solver->verifyModel();

    if (singleModel)
      break;

    cout << "Looking for another model ..." << endl;
    std::vector<Literal> oppositeClause;
    for (Variable var = 0; var < solver->getNumberOfVariables(); var++) {
      if (model.isTrueVariable(var)) {
	oppositeClause.push_back(Literals::literal(var, false));
      } else {
	oppositeClause.push_back(Literals::literal(var, true));
      }
    }

    solver->addInitialClause(oppositeClause);
  }

  cout << endl;		
  cout << "----------------------------------------------------------------------------------------------" << endl;
  cout << numOfModels << " models found" << endl;
  cout << "----------------------------------------------------------------------------------------------" << endl;

  cout << "--------------------------------------- S U M M A R Y ----------------------------------------" << endl;
  stats->printReport();
  cout << "Total time: " << totalTime.elapsed() << endl;

#ifdef ENABLE_SUMMARY_FILE
  std::ofstream summary_file;
  summary_file.open("summary", std::ios_base::app);
  summary_file << argv[1] << "\t" 
	       << numOfModels << "\t" 
	       << statistics.getNumberOfDecisions() << "\t" 
	       << totalTime.elapsed() << endl;
  summary_file.close();
#endif
  
  delete stats;
  delete solver;
  delete generator;
  
  return 0;
}

