/******************************************************************************
 * Copyright (C) 2007-2009. Filip Maric, Predrag Janicic
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * This program is inspired by MiniSat solver (C) Een, Sorensson 2003-2006.
 ******************************************************************************/
#ifndef __PROBLEM_GENERATOR_H__
#define __PROBLEM_GENERATOR_H__

#include "Solver.hpp"
#include <fstream>
#include <string>
#include <iostream>
using std::cout;
using std::cerr;
using std::endl;

namespace ArgoSat {
/******************************************************************************/
class ProblemGenerator {
 public:
  virtual ~ProblemGenerator() {}
  virtual void generateProblem(Solver* solver) = 0;

  static ProblemGenerator* createFromCmdLine(int argc, char* argv[]);
};

class ProblemGeneratorDIMACS : public ProblemGenerator {
 public:
  ProblemGeneratorDIMACS(const std::string& fileName)
    : _fileName(fileName) {
  }

  void generateProblem(Solver* solver) {
    cout << "Parsing file: " << _fileName << endl;
    std::ifstream file(_fileName.c_str());
    if (!file.is_open()) {
      cerr << "Error opening file: " << _fileName << endl;
      exit(EXIT_FAILURE);
    }
    std::string line;
    while(getline(file, line)) {
      if (line[0] == 'c')
	continue;
      if (line[0] == 'p') {
	std::istringstream stream(line.c_str() + 5);
	int numVars;
	stream >> numVars;
	int numClauses;
	stream >> numClauses;
		
	solver->setNumberOfVariables(numVars);
	continue;
      }
      if (line == "")
	continue;
	    
      std::istringstream stream(line.c_str());
      std::vector<Literal> clauseLiterals;
      int num;
      stream >> num;
      while (num != 0) {
	clauseLiterals.push_back(Literals::fromInt(num));
	stream >> num;
      }
      solver->addInitialClause(clauseLiterals);
    }
  }
  
 private:
  std::string _fileName;
};


/******************************************************************************/
class ProblemGeneratorDIMACSMiniSat : public ProblemGenerator {
public:
  ProblemGeneratorDIMACSMiniSat(const std::string& fileName)
    : _fileName(fileName) {
  }

  void generateProblem(Solver* solver) {
    cout << "Parsing file: " << _fileName << endl;
    FILE* file = fopen(_fileName.c_str(), "r");
    if (!file) {
      cerr << "Error opening file: " << _fileName << endl;
      exit(EXIT_FAILURE);
    }
    _buffer = new StreamBuffer(file);

    std::vector<Literal> clauseLiterals;

    while (true) {
      skipWhitespace(*_buffer);
      if (**_buffer == EOF)
	break;
      else if (**_buffer == 'p') {
	if (match(*_buffer, "p cnf")){
	  int numVars    = parseInt(*_buffer);
	  int numClauses = parseInt(*_buffer);
	  cout << "CNF: " << numVars << " variables and " << numClauses << " clauses" << endl;
	  solver->setNumberOfVariables(numVars);
	}else{
	  cerr << "Error reading file: " << _fileName << endl;
	  exit(EXIT_FAILURE);
	}
      } else if (**_buffer == 'c' || **_buffer == 'p')
	skipLine(*_buffer);
      else {
	clauseLiterals.clear();
	readClause(*_buffer, clauseLiterals);
	solver->addInitialClause(clauseLiterals);
      }
    }

    delete _buffer;
  }

private:
  class StreamBuffer {
  public:
    StreamBuffer(FILE* file)
      : _file(file), _position(0), _size(0) {
      assureLookahead();
    }

    int operator* () {
      return (_position >= _size) ? EOF : _buffer[_position];
    }

    void operator++ () {
      _position++; assureLookahead();
    }

  private:

    FILE* _file;
    static const unsigned CHUNK_LIMIT = 1024*1024;
    char _buffer[CHUNK_LIMIT];
    unsigned _position;
    unsigned _size;

    void assureLookahead() {
      if (_position >= _size) {
	_position = 0;
	_size = fread(_buffer, 1, CHUNK_LIMIT, _file);
	cout << "Read: " << _size << " chars" << endl;
      }
    }
  };

  int parseInt(StreamBuffer& buffer) {
    int     val = 0;
    bool    neg = false;
    skipWhitespace(buffer);
    int c;
    c = *buffer;
    if (c == '-') {
      neg = true;
      ++buffer; 
      c = *buffer;
    }
    else if (c == '+') {
      ++buffer; 
      c = *buffer;
    }
    if (!isdigit(c)) {
      cerr << "Error reading file: " << _fileName << endl;
      exit(EXIT_FAILURE);
    }
      
    while (isdigit(c)) {
      val = val*10 + (c - '0');
      ++buffer; 
      c = *buffer;
    }

    return neg ? -val : val; 
  }


  void skipWhitespace(StreamBuffer& buffer) {
    int c = *buffer;
    while (isspace(c)) {
      ++buffer; 
      c = *buffer;
    }
  }

  void skipLine(StreamBuffer& buffer) {
    int c = *buffer;
    while (true){
      if (c == EOF || c == '\0') 
	return;
      if (c == '\n') {
	++buffer; 
	return; 
      }
      ++buffer; 
      c = *buffer;
    }
  }

  bool match(StreamBuffer& buffer, const char* str) {
    for (; *str != 0; ++str, ++buffer)
      if (*str != *buffer)
	return false;
    return true;
  }

  void readClause(StreamBuffer& buffer, std::vector<Literal>& literals) {
    int num;
    while(true) {
      num = parseInt(buffer);
      if (num == 0) 
	break;
      literals.push_back(Literals::fromInt(num));
    }
  }

  StreamBuffer* _buffer;
  std::string _fileName;
};



/******************************************************************************/
class ProblemGeneratorMatrixProblems : public ProblemGenerator {
protected:
  static int getIndex(int i, int j, int n) {
    return i*n + j;
  }

  static void generateAllDistinctClauses(const std::vector<int>& indices, Solver* solver) {
    size_t i, j;
    std::vector<Literal> literals;
    for (i = 0; i < indices.size(); i++) {
      for (j = i+1; j < indices.size(); j++) {
	literals.clear();
	literals.push_back(Literals::literal(indices[i], false));
	literals.push_back(Literals::literal(indices[j], false));
	solver->addInitialClause(literals);
      }
    }
  }
};

/******************************************************************************/
class ProblemGeneratorPigeons : public ProblemGeneratorMatrixProblems {
public:
  ProblemGeneratorPigeons(int n) 
    : _n(n) {
  }

  void generateProblem(Solver* solver) {
    solver->setNumberOfVariables(_n*(_n+1));
    std::vector<Literal> literals;
    for (int i = 0; i < _n+1; i++) {
      literals.clear();
      for (int j = 0; j < _n; j++) {
	literals.push_back(Literals::literal(getIndex(i, j, _n), true));
      }
      solver->addInitialClause(literals);
    }
		
    for (int j = 0; j < _n; j++) {
      std::vector<int> col;
      for (int i = 0; i < _n + 1; i++)
	col.push_back(getIndex(i, j, _n));
      generateAllDistinctClauses(col, solver);
    }
  }

private:
  int _n;
};


/******************************************************************************/
class ProblemGeneratorQueens : public ProblemGeneratorMatrixProblems {
public:
  ProblemGeneratorQueens(int n) 
    : _n(n) {
  }

  void generateProblem(Solver* solver) {
    solver->setNumberOfVariables(_n*_n);
    std::vector<Literal> literals;
    for (int i = 0; i < _n; i++) {
      literals.clear();
      for (int j = 0; j < _n; j++) {
	literals.push_back(Literals::literal(getIndex(i, j, _n), true));
      }
      solver->addInitialClause(literals);
    }
		
    for (int i = 0; i < _n; i++) {
      std::vector<int> row;
      for (int j = 0; j < _n; j++)
	row.push_back(getIndex(i, j, _n));
      generateAllDistinctClauses(row, solver);
    }

    for (int j = 0; j < _n; j++) {
      std::vector<int> col;
      for (int i = 0; i < _n; i++)
	col.push_back(getIndex(i, j, _n));
      generateAllDistinctClauses(col, solver);
    }
		
    for (int k = -_n+2; k < _n-1; k++) {
      std::vector<int> diag;
      for (int i = std::max(0, k); i <= std::min(_n-1, k+_n-1); i++) {
	diag.push_back(getIndex(i, i-k, _n));
      }
      generateAllDistinctClauses(diag, solver);
    }
		
    for (int k = 1; k < 2*_n - 2; k++) {
      std::vector<int> diag;
      for (int i = std::max(0, k-_n+1); i <= std::min(_n-1, k); i++) {
	diag.push_back(getIndex(i, k-i, _n));
      }
      generateAllDistinctClauses(diag, solver);
    }
  }

private:
  int _n;
};

/******************************************************************************/
class ProblemGeneratorUnitPropagationSpeedTest : public ProblemGenerator {
public:
  ProblemGeneratorUnitPropagationSpeedTest(int n) 
    :_n(n) {
  }

  void generateProblem(Solver* solver) {
    /*
      1
      1 => 2
      2 => 3
      3 => 4
      ...
      9 => 10
      1 & 2 & 3 & 4 & 5 & 6 & 7 & 8 & 9 & 10 => 11
      ...
      n-10 & n-9 & n-8 & n-7 & n-6 & n-5 & n-4 & n-3 & n-2 & n-1 => n
    */
    std::vector<Literal> literals;
    solver->setNumberOfVariables(_n);
    const int k = 10;
    for (int i = 0; i < k; i++) {
      literals.clear();
      literals.push_back(Literals::literal(i, false));
      literals.push_back(Literals::literal(i + 1, true));
      solver->addInitialClause(literals);
    }
    
    for (int i = k+1; i < _n; i++) {
      literals.clear();
      for (int j = i - k; j < i; j++) {
	literals.push_back(Literals::literal(j, false));
      }
      literals.push_back(Literals::literal(i, true));
      Literals::shuffleVector(literals);
      solver->addInitialClause(literals);
    }

    literals.clear();
    literals.push_back(Literals::literal(0, true));
    solver->addInitialClause(literals);
  }

private:
  int _n;
};

}//namespace ArgoSat
#endif
