#include "clang/Driver/Options.h"
#include "clang/AST/AST.h"
#include "clang/AST/ASTContext.h"
#include "clang/AST/ASTConsumer.h"
#include "clang/AST/RecursiveASTVisitor.h"
#include "clang/Frontend/ASTConsumers.h"
#include "clang/Frontend/FrontendActions.h"
#include "clang/Frontend/CompilerInstance.h"
#include "clang/Tooling/CommonOptionsParser.h"
#include "clang/Tooling/Tooling.h"
#include "clang/Rewrite/Core/Rewriter.h"
#include "clang/Lex/Lexer.h"
#include <ctime>
#include <iostream>

using namespace std;
using namespace clang;
using namespace clang::driver;
using namespace clang::tooling;
using namespace llvm;

Rewriter rewriter;

typedef std::set< std::pair<std::string, std::string> > VarTypeMap;

std::string _preSource;
std::string _postSource;
// _preSource = "#include \"../check.h\"\n";


class CommentMainVisitor : public RecursiveASTVisitor<CommentMainVisitor> {
private:
  ASTContext *astContext; // used for getting additional AST info
  std::string _var;

public:
  explicit CommentMainVisitor(CompilerInstance *CI) 
    : astContext(&(CI->getASTContext()))
  {
    rewriter.setSourceMgr(astContext->getSourceManager(), astContext->getLangOpts());
  }


  virtual bool VisitDecl(Decl* d) {
    FunctionDecl* f = dyn_cast<FunctionDecl>(d);
    if (f != 0 && f->isMain()) {
      SourceRange range = f->getSourceRange();
      range.setEnd(clang::Lexer::getLocForEndOfToken(range.getEnd(), 0, astContext->getSourceManager(), astContext->getLangOpts()));


      rewriter.InsertText(range.getBegin(), "\n#if 0\n");
      rewriter.InsertText(range.getEnd(), "\n#endif\n");
    }
    
    return true;
  }
};

class CommentMainASTConsumer : public ASTConsumer {
private:
  CommentMainVisitor *comment_main_visitor; // doesn't have to be private

public:
    // override the constructor in order to pass CI
  explicit CommentMainASTConsumer(CompilerInstance *CI)
    : comment_main_visitor(new CommentMainVisitor(CI))
  { }

  // override this to call our RecurserVisitor on each top-level Decl
  virtual bool HandleTopLevelDecl(DeclGroupRef DG) {
    // a DeclGroupRef may have multiple Decls, so we iterate through each one
    for (DeclGroupRef::iterator i = DG.begin(), e = DG.end(); i != e; i++) {
      Decl *D = *i;
      comment_main_visitor->TraverseDecl(D); // recursively visit each AST node in Decl "D"
    }
    return true;
  }  
};



class CommentMainFrontendAction : public ASTFrontendAction {
public:
  virtual unique_ptr<ASTConsumer> CreateASTConsumer(CompilerInstance &CI, StringRef file) {
    return unique_ptr<ASTConsumer>(new CommentMainASTConsumer(&CI)); // pass CI pointer to ASTConsumer
  }
};


static cl::OptionCategory CommentMainCategory("CommentMainTool options");

int main(int argc, const char **argv) {
  srand(time(NULL));
  
  // parse the command-line args passed to your code
  CommonOptionsParser op(argc, argv, CommentMainCategory);
  // create a new Clang Tool instance (a LibTooling environment)
  ClangTool Tool(op.getCompilations(), op.getSourcePathList());

  // run the Clang Tool, creating a new FrontendAction (explained below)
  int result = Tool.run(newFrontendActionFactory<CommentMainFrontendAction>().get());

  // print out the rewritten source code ("rewriter" is a global var.)
  outs() << _preSource << "\n";
  rewriter.getEditBuffer(rewriter.getSourceMgr().getMainFileID()).write(outs());
  outs() << _postSource << "\n";
  return result;
}
