#ifndef __QUERY_HPP__
#define __QUERY_HPP__ 1

#include <vector>
#include <string>
#include <map>
#include <set>
using namespace std;

#include <iostream>
using namespace std;


extern void yyerror(string s);

string n_tabs(unsigned tabs);
bool attributeOf(string a, string t);
string number2bitvec(unsigned rank);
string inputVar2var(string a);
string removePrefix(string x);

string generateCommon(set<string> tables, int max_rank);
string generateFunctions(set<string> columns, bool la = false, int max_rank = -1);
string keys2formulas(vector<vector<string> > froms, map<string, set<string> > table_keys, bool la = false);
string keys2formulas1(vector<vector<string> > froms, vector<vector<set<string> > > table_keys_v, vector<string> tts, bool la = false);
string key2formula(string table, set<string> keys, bool la = false);
void popuni(string &result, vector<string> v, vector<vector<set<string > > > table_keys_v, vector<string> tts, bool la, set<string> &keys, unsigned n, unsigned k);
string cross_product_axioms(vector<vector<string> > froms, vector<set<string> > cols, bool la = false);
string prefix2table(string x);
string c_function_declaration(string fn_name, unsigned len);
string add_rank(string v, int jj);
string froms2table(vector<string> froms);
void writeCommon(ofstream &out1, set<string> tables, int max_rank, bool la, set<string> functions, set<string> uninterpreted_functions, set<string> variables_in);


class Expression {
public:
  virtual ~Expression() {
  
  }
  virtual string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const = 0;
  virtual bool isCorellated(vector<string> tables) const = 0;
  virtual string lav_exists(string table, int rank = -1, bool la = true) const = 0;
  virtual vector<string> differentVars() const = 0;
  virtual bool isExpression() const {
    return true;
  }
  virtual bool isConst() const {
    return false;
  }
  virtual string toFormula(map<string, string> cs, bool la = false) const = 0;
  virtual vector<string> differentInputs() const = 0;
  virtual bool equal(Expression *e) const = 0;
  virtual Expression* substitute(string s, Expression *e) {
    return this;
  }
  virtual void setQID(int i) { }
  virtual void disableAggs() { }
};

class True : public Expression {
public:
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const {
    return "$true";
  }
  bool isCorellated(vector<string> tables) const {
    return false;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const {
    return vector<string>();
  }
  bool isExpression() const {
    return false;
  }
  bool isConst() const {
    return true;
  }
  string toFormula(map<string, string> cs, bool la) const {
    return "true ";
  }
  vector<string> differentInputs() const {
    return vector<string>();
  }
  bool equal(Expression *e) const;
};

class False : public Expression {
public:
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const {
    return "$false";
  }
  bool isCorellated(vector<string> tables) const {
    return false;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const {
    return vector<string>();
  }
  bool isExpression() const {
    return false;
  }
  bool isConst() const {
    return true;
  }
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  vector<string> differentInputs() const {
    return vector<string>();
  }
  bool equal(Expression *e) const;
};

class Var : public Expression {
public:
  Var (string s) 
    :_s(s)
  {} 
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const;
  string getString() const {
    return _s;
  }
  bool isCorellated(vector<string> tables) const {
    for(auto t : tables)
      if (attributeOf(_s, t))
	return false;
    return true;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const;
  bool isExpression() const {
    return false;
  }
  string toFormula(map<string, string> cs, bool la) const;
  vector<string> differentInputs() const {
    return vector<string>();
  }
  bool equal(Expression *e) const;
  Expression* substitute(string s, Expression *e) {
    unsigned i = _s.find_first_of(".");
    if (i == string::npos)
      i = -1;
    string s1 = _s.substr(i+1);
    if (s == s1) {
      return e;
    } 
    return this;
  }
protected:
  string _s;
};

class InputVar : public Var {
public:
  InputVar (string s) 
    :Var(s)
  {} 
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const {
    return vector<string>();
  }
  bool isExpression() const {
    return false;
  }
  bool isConst() const {
    return true;
  }
  string toFormula(map<string, string> cs, bool la) const {
    return cs[_s];
  }
  vector<string> differentInputs() const {
    vector<string> tmp;
    tmp.push_back(_s);
    return tmp;
  }
};

class Num : public Expression {
public:
  Num(int broj) 
    : _broj(broj)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  bool isCorellated(vector<string> tables) const {
    return false;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const {
    return vector<string>();
  }
  bool isExpression() const {
    return false;
  }
  bool isConst() const {
    return true;
  }
  string toFormula(map<string, string> cs, bool la) const {
    if (la)
      return to_string(_broj);
    else
      return number2bitvec(_broj);
  }
  vector<string> differentInputs() const {
    return vector<string>();
  }
  bool equal(Expression *e) const;
private:
  int _broj;
};

class BinaryOperator : public Expression {
public:
  BinaryOperator(Expression *first, Expression *second)
    :_first(first), _second(second)
  {}
  ~BinaryOperator() {
    delete _first;
    delete _second;
  }
  bool equal(Expression *e) const;
  bool isConst() const;
  void setQID(int i) {
    _first->setQID(i);
    _second->setQID(i);
  }
  void disableAggs() {
    _first->disableAggs();
    _second->disableAggs();
  }
private:
  // Posto nam je potreban destruktor,
  // moramo da napravimo i konstruktor koprije i operator dodele
  // Ako smo sigurni da ih nigde ne pozivamo, niti se bilo gde 
  // implicitno pozivaju, mozemo da ih stavimo kao private
  // i da ih nigde ne definisemo (samo ih deklarisemo)
  // Ruzan nacin, ali prihvatljiv ako nesto treba uraditi za malo vremena.
  BinaryOperator(const BinaryOperator& b);
  BinaryOperator& operator=(const BinaryOperator& b);
  bool isCorellated(vector<string> tables) const {
    return _first->isCorellated(tables) || _second->isCorellated(tables);
  }
  vector<string> differentVars() const;
  vector<string> differentInputs() const;
protected:
  Expression *_first;
  Expression *_second;
};

class UnaryOperator : public Expression {
public:
  UnaryOperator(Expression *first)
    :_first(first)
  {}
  ~UnaryOperator() {
    delete _first;
  }
  bool isCorellated(vector<string> tables) const {
    return _first->isCorellated(tables);
  }
  vector<string> differentVars() const;
  vector<string> differentInputs() const;
  bool equal(Expression *e) const;
  bool isConst() const;
  void setQID(int i) {
    _first->setQID(i);
  }
  void disableAggs() {
    _first->disableAggs();
  }
private:
  // Posto nam je potreban destruktor,
  // moramo da napravimo i konstruktor koprije i operator dodele
  // Ako smo sigurni da ih nigde ne pozivamo, niti se bilo gde 
  // implicitno pozivaju, mozemo da ih stavimo kao private
  // i da ih nigde ne definisemo (samo ih deklarisemo)
  // Ruzan nacin, ali prihvatljiv ako nesto treba uraditi za malo vremena.
  UnaryOperator(const UnaryOperator& b);
  UnaryOperator& operator=(const UnaryOperator& b);
protected:
  Expression *_first;
};

class Greater : public BinaryOperator {
public:
  Greater(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Greater(_first, _second);
  }
private:
  Greater operator = (const Greater& p);
};

class Less : public BinaryOperator {
public:
  Less(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Less(_first, _second);
  }

private:
  Less operator = (const Less& p);
};

class GreaterEq : public BinaryOperator {
public:
  GreaterEq(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new GreaterEq(_first, _second);
  }
private:
  GreaterEq operator = (const GreaterEq& p);
};

class LessEq : public BinaryOperator {
public:
  LessEq(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new LessEq(_first, _second);
  }
private:
  LessEq operator = (const LessEq& p);
};

class Eq : public BinaryOperator {
public:
  Eq(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Eq(_first, _second);
  }
private:
  Eq operator = (const Eq& p);
};

class Dis : public BinaryOperator {
public:
  Dis(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Dis(_first, _second);
  }
private:
  Dis operator = (const Dis& p);
};

class Or : public BinaryOperator {
public:
  Or(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Or(_first, _second);
  }
private:
  Or operator = (const Or& p);
};

class And : public BinaryOperator {
public:
  And(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new And(_first, _second);
  }
private:
  And operator = (const And& p);
};

class Add : public BinaryOperator {
public:
  Add(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const;
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Add(_first, _second);
  }
private:
  Add operator = (const Add& p);
};

class Sub : public BinaryOperator {
public:
  Sub(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const;
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Sub(_first, _second);
  }
private:
  Sub operator = (const Sub& p);
};

class Mul : public BinaryOperator {
public:
  Mul(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const;
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Mul(_first, _second);
  }
private:
  Mul operator = (const Mul& p);
};

class Div : public BinaryOperator {
public:
  Div(Expression *first, Expression *second)
    :BinaryOperator(first, second)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const;
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    _second = _second->substitute(s, e);
    return new Div(_first, _second);
  }
private:
  Div operator = (const Div& p);
};


class Not : public UnaryOperator {
public:
  Not(Expression *first)
    :UnaryOperator(first)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    return new Not(_first);
  }
private:
  Not operator = (const Not& p);
};

class Function : public UnaryOperator {
public:
  Function(string name, Expression *first)
    :UnaryOperator(first), _name(name)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "(" + _name + " " + _first->toFormula(cs, la) + ")";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    return new Function(_name, _first);
  }
private:
  Function operator = (const Function& p);
protected:
  string _name;
};

class AggFunction : public Function {
public:
  AggFunction(string name, Expression *first, bool disabled = false);
  string lav_exists(string table, int rank, bool la) const;
  string toFormula(map<string, string> cs, bool la) const;
  bool isExpression() const {
    return false;
  }
  void setQID(int i) {
    _q_id = i;
  }
  static unsigned getTotal() {
    return _total;
  }
  unsigned getFreshNew() const {
    return _fresh_new;
  }
  Expression* getArg() const {
    return _first;
  }
  void disableAggs() {
    _disabled = true;
  }
private:
  int _q_id;
  unsigned _fresh_new;
  static unsigned _total;
  bool _disabled;
};


class Group : public UnaryOperator {
public:
  Group(Expression *first)
    :UnaryOperator(first)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  string lav_exists(string table, int rank = -1, bool la = true) const;
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  Expression* substitute(string s, Expression *e) {
    _first = _first->substitute(s, e);
    return new Group(_first);
  }
private:
  Group operator = (const Group& p);
};


class Query;


class Exists : public Expression {
public:
  Exists(Query *q, int i)
    :_q(q), _i(i)
  {}
  ~Exists();
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  bool isCorellated(vector<string> tables) const {
    return false;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  vector<string> differentVars() const {
    vector<string> tmp;
    tmp.push_back("TODO");
    return tmp;
  }
  vector<string> differentInputs() const {
    vector<string> tmp;
    tmp.push_back("TODO");
    return tmp;
  }
  string toFormula(map<string, string> cs, bool la) const {
    return "TODO";
  }
  bool equal(Expression *e) const;
  Expression* substitute(string s, Expression *e) {
    //TODO:
    return this;
  }
protected:
  Query *_q;
  int _i;
};

class In : public Exists {
public:
  In(string s, int num, Query *q, int i)
    :Exists(q, i), _s(s), _num(num)
  {}
  string translate(unsigned tabs, string varname, vector<string> tables, bool cardinality = false) const ;
  bool isCorellated(vector<string> tables) const {
    return false;
  }
  string lav_exists(string table, int rank = -1, bool la = true) const;
  Expression* substitute(string s, Expression *e) {
    //TODO:
    return this;
  }
private:
  string _s;
  int _num;
};

class Selection {
public:
  Selection(Expression *e, string name)
    :_e(e), _name(name)
  {}
  string getSelectionString() const {
    Var *v = dynamic_cast<Var*>(_e);
    if (v) {
      return v->getString();
    }
    else
      throw "TODO: add expressions to projections in select clause";
  }
  vector<string> differentVars() const {
    return _e->differentVars();
  }
  ~Selection() {
    delete _e;
  }
  bool isExpression() const {
    return _e->isExpression();
  }
  string toFormula(map<string, string> cs, bool la = false) const {
    return _e->toFormula(cs, la);
  }
  Expression* getExpression() const {
    return _e;
  }
  void setExpression(Expression *e) {
    _e = e;
  }
  string getName() const {
    return _name;
  }
  void setName(const string &name) {
    _name = name;
  }
private:
  Selection(const Selection&);
  Expression *_e;
  string _name;
};

class QEx {
public:
  virtual string translate(unsigned tabs, bool cardinality) const = 0;
  virtual bool isDistinct() = 0;
  virtual ~QEx() {

  }
  virtual string lav_predicate_feature(bool la = false, int max_rank = -1) const = 0;
  virtual string lav_unique_existence(bool c = false, bool la = false) const = 0;
  virtual string lav_existence(bool c = false, bool la = false, int rank = 0) const = 0;
  virtual string lav_result(bool c = false, bool la = false, int rank = -1) const = 0;
  virtual vector<string> differentVars(unsigned i) const = 0;
  virtual vector<string> differentInputs() const = 0;
  virtual set<string> tables() const = 0;
  virtual vector<string> vectables() const = 0;
  virtual set<string> columns() const = 0;
  virtual void preproccess(map<string, vector<string> > table_columns, vector<vector<set<string> > > table_keys_v) = 0;
  virtual string lav_position(bool la = false, int rank = -1) const = 0;
};

class BinQex : public QEx {
public:
  BinQex(QEx *q1, QEx *q2)
    :_q1(q1), _q2(q2)
  {}
  virtual ~BinQex() {
    delete _q1;
    delete _q2;
  }
  virtual bool isDistinct() {
    return true;
  }
  vector<string> differentVars(unsigned i) const {
    return _q1->differentVars(i);
  }
  vector<string> differentInputs() const {
    vector<string> tmp1 = _q1->differentInputs();
    vector<string> tmp2 = _q2->differentInputs();
    tmp1.insert(tmp1.end(), tmp2.begin(), tmp2.end());
    return tmp1;
  }
  set<string> tables() const {
    set<string> tmp = _q1->tables();
    for (auto t :_q2->tables())
      tmp.insert(t);
    return tmp;
  }
  vector<string> vectables() const {
    vector<string> tmp = _q1->vectables();
    for (auto t :_q2->vectables())
      tmp.push_back(t);
    return tmp;
  }
  set<string> columns() const {
    set<string> tmp = _q1->columns();
    for (auto t :_q2->columns())
      tmp.insert(t);
    return tmp;
  }
private:
  BinQex(const BinQex& b);
  BinQex& operator=(const BinQex& b);
protected:
  QEx *_q1, *_q2;
};

class Union : public BinQex {
public:
  Union(QEx *q1, QEx *q2, bool all)
    :BinQex(q1, q2), _all(all)
  {}
  string translate(unsigned tabs, bool cardinality) const;
  bool isDistinct() {
    return !_all;
  }
  string lav_predicate_feature(bool la = false, int max_rank = -1) const;
  string lav_unique_existence(bool c = false, bool la = false) const;
  string lav_existence(bool c = false, bool la = false, int rank = 0) const;
  string lav_result(bool c = false, bool la = false, int rank = -1) const;
  void preproccess(map<string, vector<string> > table_columns, vector<vector<set<string> > > table_keys_v);
  string lav_position(bool la = false, int rank = -1) const;
private:
  bool _all;
};

class Except : public BinQex {
public:
  Except(QEx *q1, QEx *q2)
    :BinQex(q1, q2)
  {}
  string translate(unsigned tabs, bool cardinality) const;
  string lav_predicate_feature(bool la = false, int max_rank = -1) const;
  string lav_unique_existence(bool c = false, bool la = false) const;
  string lav_existence(bool c = false, bool la = false, int rank = 0) const;
  string lav_result(bool c = false, bool la = false, int rank = -1) const;
  void preproccess(map<string, vector<string> > table_columns, vector<vector<set<string> > > table_keys_v);
  string lav_position(bool la = false, int rank = -1) const;
};

class Intersect : public BinQex {
public:
  Intersect(QEx *q1, QEx *q2)
    :BinQex(q1, q2)
  {}
  string translate(unsigned tabs, bool cardinality) const;
  string lav_predicate_feature(bool la = false, int max_rank = -1) const;
  string lav_unique_existence(bool c = false, bool la = false) const;
  string lav_existence(bool c = false, bool la = false, int rank = 0) const;
  string lav_result(bool c = false, bool la = false, int rank = -1) const;
  void preproccess(map<string, vector<string> > table_columns, vector<vector<set<string> > > table_keys_v);
  string lav_position(bool la = false, int rank = -1) const;
};

extern map<string, QEx*> views;
#include <algorithm>

class Query : public QEx {
public:
  Query(vector<string> froms, vector<pair<string, string> > aliases1, Expression* where, vector<pair<Expression*, bool> > crts, int limit, vector<Expression*> groups)
    :_froms(froms), _where(where), _distinct(false), _id(_total++ - views.size()), _crts(crts), _limit(limit), _groups(groups)
  {
    for(vector<pair<string, string> >::iterator i = aliases1.begin(); i != aliases1.end(); i++) {
      _aliases1[(*i).second] = (*i).first;
    }

    for (auto cc : _crts)
      cc.first->setQID(_id);

  }
  virtual ~Query() {
    delete _where;
    for (auto s : _selections)
      delete s;
    for (auto g : _groups)
      delete g;
  }
  string translate(unsigned tabs, bool cardinality) const;
  vector<Selection*> getSelections() const {
    return _selections;
  }
  vector<Expression*> getGroups() const {
    return _groups;
  }
  void disableGroups() {
    _groups.clear();
    vector<Selection*> tmp;
    for (auto &a : _selections) {
      Expression *e = a->getExpression();
      e->disableAggs();
      tmp.push_back(new Selection(e, a->getName()));
    }
    _selections = tmp;
    for (auto cc : _crts)
      cc.first->disableAggs(); 
  }
  void setSelections(vector<Selection*> s) {
    vector<Selection*> tmp;
    for (auto &a : s) {
      Expression *e = a->getExpression();
      e->setQID(_id);
      tmp.push_back(new Selection(e, a->getName()));
    }
    _selections = tmp;
  }
  void setIntos(vector<string> s) {
    _intos = s;
  }
  vector<string> getIntos() const {
    return _intos;
  }
  void setDistinct() {
    _distinct = true;
  }
  virtual bool isDistinct() {
    return _distinct;
  }
  string getFirstSelectionString() const {
    return _selections[0]->getSelectionString();
  }
  bool isCorellated() const;
  string lav_predicate_feature(bool la = false, int max_rank = -1) const;
  string lav_unique_existence(bool c = false, bool la = false) const;
  string lav_existence(bool c = false, bool la = false, int rank = 0) const;
  string lav_result(bool c = false, bool la = false, int rank = -1) const;
  vector<string> differentVars(unsigned i) const {
    return _selections[i]->differentVars();
  }
  vector<string> differentInputs() const {
    return _where->differentInputs();
  }
  set<string> tables() const {
    set<string> r;
    for (auto t : _froms)
      r.insert(t);
    return r;
  }
  vector<string> vectables() const {
    return _froms;
  }
  set<string> columns() const {
    set<string> r;
    for (auto c : _where->differentVars())
      r.insert(removePrefix(c));
    for (auto s : _selections)
      for (auto c : s->differentVars())
	r.insert(removePrefix(c));
    for (auto c : _crts)
      for (auto d : c.first->differentVars())
	r.insert(removePrefix(d));
    for (auto c : _groups)
      for (auto d : c->differentVars())
	r.insert(removePrefix(d));
    return r;
  }
  bool isCompleteOrder(map<string, set<set<string> > > table_keys) const;
  bool isCompleteOrderByPK(map<string, set<set<string> > > table_keys) const;
  bool isGroupByPK(map<string, set<set<string> > > table_keys) const;
  vector<pair<Expression*, bool> > getOrderBy() const {
    return _crts;
  }
  void preproccess(map<string, vector<string> > table_columns, vector<vector<set<string> > > table_keys_v);
  string lav_position(bool la = false, int rank = -1) const;
  map<string, string> getAliases1() const {
    return _aliases1;
  }
  Expression* getWhere() const {
    return _where;
  }
  int getLimit() const {
    return _limit;
  }
  string positionInjectivness(bool la = false) const;
  string detOrderBy(bool la = false) const;
private:
  vector<Selection*> _selections;
  vector<string> _froms;
  map<string, string> _aliases1;
  Expression* _where;
  bool _distinct;
  static unsigned _total;
  unsigned _id;
  vector<string> _intos;
  vector<pair<Expression*, bool> > _crts;
  int _limit;
  vector<Expression*> _groups;
};

string position_criteria_equality(Query *q1, Query *q2, bool la = false, int rank = 0);
string position_range(Query *q1, Query *q2, bool la = false, int rank = 0);

#endif

